/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file fritzbox.c
 * \brief This file contains FRITZ!Box router related functions
 */

#include <ffgtk.h>
#include <router/common.h>

#define NAME_COUNT  27
#define BLOCKED_SECONDS	"var loginBlocked = parseInt\\(\"([^\"]*)\",10\\)"

extern GList *psDiversityList;

struct sVoiceData {
	/* 0 */
	int nHeader;
	/* 4 */
	int nIndex;
	/* 8 (2=own message, 3=remote message) */
	int nType;
	/* 12 */
	unsigned int nSubType;
	/* 16 */
	unsigned int nSize;
	/* 20 */
	unsigned int nDuration;
	/* 24 */
	unsigned int nStatus;
	/* 28 */
	unsigned char anTmp0[ 24 ];
	/* 52 */
	char anRemoteNumber[ 54 ];
	/* 106 */
	char anTmp1[ 18 ];
	/* 124 */
	char anFile[ 32 ];
	/* 151 */
	char anPath[ 128 ];
	/* 279 */
	unsigned char nDay;
	unsigned char nMon;
	unsigned char nYear;
	unsigned char nHour;
	unsigned char nMin;
	unsigned char anTmp2[ 31 ];
	char anLocalNumber[ 24 ];
	char anTmp3[ 4 ];
};

enum {
	COL_CM_TYPE,
	COL_CM_DATETIME,
	COL_CM_NAME,
	COL_CM_COMPANY,
	COL_CM_NUMBER,
	COL_CM_LOCALNAME,
	COL_CM_LOCALNUMBER,
	COL_CM_DURATION
};

struct sVoiceBox {
	gchar *pnData;
	gint nLen;
};

static gchar *pnVoiceBoxPath = NULL;
static gchar *pnFaxBoxPath = NULL;
static struct sVoiceBox asVoiceBox[ 5 ];

struct sPorts {
	char *pnName;
	int nType;
	int nNumber;
};

/** Port names */
static struct sPorts asPortNames[ NAME_COUNT ] = {
	/* Analog */
	{ "telcfg:settings/MSN/Port0/Name", PORT_ANALOG1, 1 },
	{ "telcfg:settings/MSN/Port1/Name", PORT_ANALOG2, 2 },
	{ "telcfg:settings/MSN/Port2/Name", PORT_ANALOG3, 3 },
	/* ISDN */
	{ "telcfg:settings/NTHotDialList/Name1", PORT_ISDN1, 51 },
	{ "telcfg:settings/NTHotDialList/Name2", PORT_ISDN2, 52 },
	{ "telcfg:settings/NTHotDialList/Name3", PORT_ISDN3, 53 },
	{ "telcfg:settings/NTHotDialList/Name4", PORT_ISDN4, 54 },
	{ "telcfg:settings/NTHotDialList/Name5", PORT_ISDN5, 55 },
	{ "telcfg:settings/NTHotDialList/Name6", PORT_ISDN6, 56 },
	{ "telcfg:settings/NTHotDialList/Name7", PORT_ISDN7, 57 },
	{ "telcfg:settings/NTHotDialList/Name8", PORT_ISDN8, 58 },
	/* DECT */
	{ "telcfg:settings/Foncontrol/User1/Name", PORT_DECT1, 60 },
	{ "telcfg:settings/Foncontrol/User2/Name", PORT_DECT2, 61 },
	{ "telcfg:settings/Foncontrol/User3/Name", PORT_DECT3, 62 },
	{ "telcfg:settings/Foncontrol/User4/Name", PORT_DECT4, 63 },
	{ "telcfg:settings/Foncontrol/User5/Name", PORT_DECT5, 64 },
	{ "telcfg:settings/Foncontrol/User6/Name", PORT_DECT6, 65 },
	/* IP-Phone */
	{ "telcfg:settings/VoipExtension0/Name", PORT_IP1, 620 },
	{ "telcfg:settings/VoipExtension1/Name", PORT_IP2, 621 },
	{ "telcfg:settings/VoipExtension2/Name", PORT_IP3, 622 },
	{ "telcfg:settings/VoipExtension3/Name", PORT_IP4, 623 },
	{ "telcfg:settings/VoipExtension4/Name", PORT_IP5, 624 },
	{ "telcfg:settings/VoipExtension5/Name", PORT_IP6, 625 },
	{ "telcfg:settings/VoipExtension6/Name", PORT_IP7, 626 },
	{ "telcfg:settings/VoipExtension7/Name", PORT_IP8, 627 },
	{ "telcfg:settings/VoipExtension8/Name", PORT_IP9, 628 },
	{ "telcfg:settings/VoipExtension9/Name", PORT_IP10, 629 },
};

/** logged in counter */
static gchar nLoggedIn = 0;
static GMutex *psLogMutex = NULL;

/**
 * \brief Detect fritzbox
 * \param psProfile profile structure
 * \return error code
 */
static int fritzBoxDetect( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gint nError;
	gchar anUrl[ BUF_SIZE ];

	Debug( KERN_INFO, "Starting detection\n" );

	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		freeHandler( psHandler );
		Debug( KERN_DEBUG, "No FRITZ!Box\n" );
		return nError;
	}

	if ( g_strcasestr( psHandler -> pnData, "fritz!box" ) ) {
		nError = 0;
		Debug( KERN_DEBUG, "FRITZ!Box found!\n" );
	} else {
		nError = 1;
		Debug( KERN_DEBUG, "No FRITZ!Box - no indication in site found\n" );
	}
	freeHandler( psHandler );

	return nError;
}

/**
 * \brief Replace special chars with dots
 * \param pnString input string
 * \return output string
 */
static gchar *makeDots( const gchar *pnString ) {
	/*GString *psNewString = g_string_new( "" );
	gint nIndex = 0;
	gint nLen = strlen( pnString );

	for ( nIndex = 0; nIndex < nLen; nIndex++ ) {
		if ( pnString[ nIndex ] > 255 ) {
			psNewString = g_string_append_c( psNewString, '.' );
		} else {
			psNewString = g_string_append_c( psNewString, pnString[ nIndex ] );
		}
	}

	return g_string_free( psNewString, FALSE );*/
	return g_strdup( pnString );
}

/**
 * \brief Convert string to an array of little-endian words
 * \param pnStr input string
 * \return array of words
 */
uint8_t *str2binl( gchar *pnStr ) {
        gint nChrSize = 16;
        gint nLen = strlen( pnStr ) + 1;
        gint nSize =  ( ( ( nLen * nChrSize ) >> 5 ) + 1 );
        gint *pnBin = g_malloc0( nSize * sizeof( gint ) );
        gint nMask = ( 1 << nChrSize ) - 1;
        gint nIndex = 0;

        for ( nIndex = 0; nIndex < nLen * nChrSize; nIndex += nChrSize ) {
                pnBin[ nIndex >> 5 ] |= ( pnStr[ nIndex / nChrSize ] & nMask ) << ( nIndex % 32 );
        }

        uint8_t *pnBinL =  g_malloc0 ( nSize * 4 );
        for (nIndex = 0; nIndex < nSize; nIndex++) {
                pnBinL [ 4*nIndex ] = ( uint8_t ) ( pnBin [ nIndex ] & 0xff );
                pnBinL [ 4*nIndex + 1] = ( uint8_t ) ( (pnBin [ nIndex ] >> 8) & 0xff);
                pnBinL [ 4*nIndex + 2] = ( uint8_t ) ( (pnBin [ nIndex ] >> 16) & 0xff);
                pnBinL [ 4*nIndex + 3] = ( uint8_t ) ( (pnBin [ nIndex ] >> 24) & 0xff);
        }
        g_free ( pnBin );
        return pnBinL;
}


/**
 * \brief Compute md5 sum of input string
 * \param pnInput input string
 * \return md5 in hex
 */
static gchar *md5( gchar *pnInput ) {
	uint8_t *pnBin = str2binl( pnInput );
	gchar *pnRet = g_compute_checksum_for_string( G_CHECKSUM_MD5, ( gchar *) pnBin, strlen( pnInput ) * 2 );
	g_free( pnBin );

	return pnRet;
}

/**
 * \brief Get value of key
 * \param pnData data pointer
 * \param pnKey key
 * \return requested key value
 */
static gchar *fritzboxGetValue( gchar *pnData, gchar *pnKey ) {
	gint nStart = -1;
	gint nEnd = -1;
	gchar *pnReturn = NULL;

	nStart = findString( pnData, 0, pnKey );
	if ( nStart != -1 ) {
		nStart += strlen( pnKey ) + 1;
		nStart = findString( pnData, nStart, "value=\"" );
		if ( nStart != -1 ) {
			nStart += 7;
			nEnd = findString( pnData, nStart, "\"" );
			pnReturn = getSubString( pnData, nStart, nEnd - nStart );
		}
	}

	return pnReturn;
}

/**
 * \brief Login to fritzbox
 * \param psProfile profile structure
 * \return 0 on success, negative values (seconds to sleep) on error
 */
static int fritzBoxLogin( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler = NULL;
	gint nError = -1;
	gchar anUrl[ BUF_SIZE ];
	const gchar *pnPassword = routerGetPassword( psProfile );
	gboolean bSecure = FALSE;
	gint nStart = -1;
	gint nEnd = -1;
	gchar *pnChallenge = NULL;
	gchar *pnResponse = NULL;

	g_mutex_lock( psLogMutex );
	if ( nLoggedIn != 0 ) {
		nLoggedIn++;
		Debug( KERN_DEBUG, "Already LoggedIn: %d\n", nLoggedIn );
		g_mutex_unlock( psLogMutex );
		return 0;
	}

	if ( pnPassword == NULL ) {
		goto end;
	}

again:
	psProfile -> sRouterInfo.pnSessionId = "";

	/* Check for secure login */
	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );
	setPostData( psHandler,	"getpage=../html/login_sid.xml&sid=0000000000000000" );

	/* set timeout to 3sec */
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_TIMEOUT, 3 );
	nError = readUrl( psHandler, psProfile );
	if ( nError == 0 ) {
		saveDebugData( "fritzbox-check.html", psHandler -> pnData, psHandler -> nSize );

		nStart = findString( psHandler -> pnData, 0, "<iswriteaccess>" );
		if ( nStart != -1 ) {
			nStart += 15;
			nEnd = findString( psHandler -> pnData, nStart, "<" );
			if ( nEnd != -1 ) {
				gchar *pnAccess = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
				int nAccess = atoi( pnAccess );
				g_free( pnAccess );
				//Debug( KERN_DEBUG, "access: %d\n", nAccess );

				if ( nAccess == 0 ) {
					/* Secure login */
					bSecure = TRUE;
					nStart = findString( psHandler -> pnData, 0, "<Challenge>" );
					if ( nStart != -1 ) {
						nStart += 11;
						nEnd = findString( psHandler -> pnData, nStart, "<" );
						if ( nEnd != -1 ) {
							pnChallenge = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
							Debug( KERN_DEBUG, "pnChallenge '%s'\n", pnChallenge );
							gchar *pnDots = makeDots( pnPassword );
							gchar *pnStr = g_strdup_printf( "%s-%s", pnChallenge, pnDots );
							gchar *pnMd5 = md5( pnStr );
							pnResponse = g_strdup_printf( "%s-%s", pnChallenge, pnMd5 );
							g_free( pnMd5 );
							g_free( pnStr );
							g_free( pnDots );
							g_free( pnChallenge );
							Debug( KERN_DEBUG, "Response: '%s'\n", pnResponse );
							bSecure = TRUE;
						}
					}
				} else {
					nStart = findString( psHandler -> pnData, 0, "<SID>" );
					if ( nStart != -1 ) {
						nStart += 5;
						nEnd = findString( psHandler -> pnData, nStart, "<" );
						if ( nEnd != -1 ) {
							psProfile -> sRouterInfo.pnSessionId = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
						}
					}

					nError = 0;
					goto end;
				}
			}
		}
		freeHandler( psHandler );
	}

	/* login */
	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );

	if ( bSecure == TRUE ) {
		setPostData( psHandler, "&getpage=../html/de/menus/menu2.html&login:command/response=%s&sid=0000000000000000 ", pnResponse );
		g_free( pnResponse );
	} else {
		setPostData( psHandler, "&login:command/password=%s&var:loginDone=1", pnPassword );
	}

	/* set timeout to 3sec */
	//curl_easy_setopt( psHandler -> psHandle, CURLOPT_TIMEOUT, 3 );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		goto end;
	}

	saveDebugData( "fritzbox-login.html", psHandler -> pnData, psHandler -> nSize );

	if ( findString( psHandler -> pnData, 0, "FRITZ!Box Anmeldung" ) != -1 ) {
		GRegex *psReg;
		GMatchInfo *psInfo;
		gint nSleep = 1;

		saveDebugData( "fritzbox-login-failed.html", psHandler -> pnData, psHandler -> nSize );

		psReg = g_regex_new( BLOCKED_SECONDS, 0, 0, NULL );
		g_regex_match( psReg, psHandler -> pnData, 0, &psInfo );
		if ( g_match_info_matches( psInfo ) ) {
			gchar *pnWord = g_match_info_fetch( psInfo, 0 );
			if ( pnWord != NULL ) {
				gint nStart, nEnd;

				nStart = findString( pnWord, 0, "\"" );
				if ( nStart != -1 ) {
					nStart++;
					nEnd = findString( pnWord, nStart, "\"" );
					if ( nEnd != -1 ) {
						gchar *pnTmp = getSubString( pnWord, nStart, nEnd - nStart );
						nSleep = atoi( pnTmp );
						g_free( pnTmp );
					}
				}
			}
			g_free( pnWord );
		}

		/* Set error code to -sleep seconds */
		nError = -nSleep;

		g_match_info_free( psInfo );

		g_regex_unref( psReg );

		if ( nError == 0 ) {
			freeHandler( psHandler );
			Debug( KERN_WARNING, "Seconds zero, try again\n" );
			goto again;
		}

		Debug( KERN_WARNING, "Login failed, wrong password! Wait for %d seconds\n", nSleep );
	} else {
		if ( bSecure == TRUE ) {
			psProfile -> sRouterInfo.pnSessionId = fritzboxGetValue( psHandler -> pnData, "name=\"sid\"" );
		}
		nError = 0;
		Debug( KERN_DEBUG, "Login successful!\n" );
		nLoggedIn++;
	}
	freeHandler( psHandler );

	Debug( KERN_DEBUG, "LoggedIn: %d\n", nLoggedIn );

end:
	/* If we failed to login due to a wrong password, lock router structure */
	if ( nError < 0 ) {
		routerLock( psProfile, TRUE );
	}

	g_mutex_unlock( psLogMutex );

	return nError;
}

/**
 * \brief Logout from fritzbox
 * \param psProfile profile structure
 * \return error code
 */
static int fritzBoxLogout( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ BUF_SIZE ];
	gint nError = -1;

	g_mutex_lock( psLogMutex );
	nLoggedIn--;

	if ( nLoggedIn > 0 ) {
		g_mutex_unlock( psLogMutex );
		return 0;
	}
	if ( nLoggedIn < 0 ) {
		nLoggedIn = 0;
		g_mutex_unlock( psLogMutex );
		return 0;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );
	setPostData( psHandler, "&sid=%s&security:command/logout=&getpage=../html/confirm_logout.html", psProfile -> sRouterInfo.pnSessionId );
	nError = readUrl( psHandler, psProfile );

	saveDebugData( "fritzbox-logout.html", psHandler -> pnData, psHandler -> nSize );

	freeHandler( psHandler );
	g_mutex_unlock( psLogMutex );

	return nError;
}

/**
 * \brief Get language of fritzbox needed for further calls
 * \param psProfile profile structure
 * \return language string, "de" or "en"
 */
static char *fritzBoxGetLanguage( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nError = -1;

	if ( psProfile -> sRouterInfo.pnLanguage == NULL || strlen( psProfile -> sRouterInfo.pnLanguage ) == 0 ) {
		Debug( KERN_DEBUG, "Detecting fritzbox language...\n" );

		snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
		psHandler = urlHandler( anUrl, 80 );
		setPostData( psHandler,	"getpage=../html/index_inhalt.html" );
		nError = readUrl( psHandler, psProfile );
		if ( nError == 0 ) {
			int nTmp = findString( psHandler -> pnData, 0, "name=\"var:lang\"" );
			if ( nTmp != -1 ) {
				nTmp += 16;
				int nStart, nStop;
				nStart = findString( psHandler -> pnData, nTmp, "\"" );
				nStart++;
				nStop = findString( psHandler -> pnData, nStart, "\"" );
				if ( nStart != -1 && nStop != -1 ) {
					psProfile -> sRouterInfo.pnLanguage = getSubString( psHandler -> pnData, nStart, nStop - nStart );
					Debug( KERN_DEBUG, "Language is '%s'\n", psProfile -> sRouterInfo.pnLanguage );
				}
			}
		}
		freeHandler( psHandler );

		if ( psProfile -> sRouterInfo.pnLanguage == NULL || strlen( psProfile -> sRouterInfo.pnLanguage ) == 0 ) {
			psProfile -> sRouterInfo.pnLanguage = strdup( "de" );
			Debug( KERN_DEBUG, "No fritzbox language defined, set to 'de'\n" );
		}
	}

	return psProfile -> sRouterInfo.pnLanguage;
}

/**
 * \brief Detect fritzbox firmware
 * \param psProfile profile structure
 * \return error code
 */
static int fritzBoxDetectFirmware( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ 1024 ];
	gint nError = 1;

	Debug( KERN_DEBUG, "Detecting fritzbox firmware...\n" );

	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "Could not login\n" );
		return nError;
	}

	psProfile -> sRouterInfo.pnFirmware = NULL;

	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm?getpage=../html/%s/menus/menu2.html&var:lang=%s&var:menu=home&var:pagename=home&sid=%s", routerGetHost( psProfile ), fritzBoxGetLanguage( psProfile ), fritzBoxGetLanguage( psProfile ), psProfile -> sRouterInfo.pnSessionId );
	psHandler = urlHandler( anUrl, 80 );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "Could not read start page\n" );
		goto end;
	}

	saveDebugData( "fritzbox-firmware.html", psHandler -> pnData, psHandler -> nSize );

	GRegex *psReg;
	GMatchInfo *psInfo;
	gchar *pnRegex = "(Firmware|Labor)(-| )(V|v)ersion[^\\d]*(\\d\\d\\d*).(\\d\\d).(\\d\\d\\d*)([^<]*)";
	psReg = g_regex_new( pnRegex, 0, 0, NULL );

	g_regex_match( psReg, psHandler -> pnData, 0, &psInfo );

	if ( g_match_info_matches( psInfo ) ) {
		gchar *pnWord = g_match_info_fetch( psInfo, 0 );
		Debug( KERN_DEBUG, "pnWord: %s\n", pnWord );

		if ( pnWord[ strlen( pnWord ) - 1 ] == '\n' ) {
			pnWord[ strlen( pnWord ) - 1 ] = '\0';
		}
		psProfile -> sRouterInfo.pnFirmware = convertEntities( pnWord );
		g_free( pnWord );
	}
	g_match_info_free( psInfo );

	g_regex_unref( psReg );

	routerSetFirmware( psProfile, psProfile -> sRouterInfo.pnFirmware );

end:
	commonParseFirmware( psProfile );

	if ( psProfile -> sRouterInfo.pnFirmware == NULL || strlen( psProfile -> sRouterInfo.pnFirmware ) == 0 ) {
		Debug( KERN_WARNING, "Could not parse fritzbox firmware. set type to 0 and enabled all ports as fallback\n" );
		nError = 1;
	} else {
		nError = 0;
	}

	freeHandler( psHandler );
	routerLogout( psProfile );

	return nError;
}

/**
 * \brief Get fon msn
 * \param pnData data buffer
 * \param nType what kind of fon do we want
 * \return msn number
 */
static char *getFon( char *pnData, int nType ) {
	int nSkip = 0, nEnd;

	switch ( nType ) {
		case 0 ... 9: {
			gchar *pnTmp = g_strdup_printf( "telcfg:settings/MSN/MSN%d", nType );
			nSkip = findString( pnData, 0, pnTmp );
			g_free( pnTmp );
			break;
		}
		case 10:
			nSkip = findString( pnData, 0, "telcfg:settings/MSN/POTS" );
			break;
		case 11 ... 14: {
			gchar *pnTmp = g_strdup_printf( "sip:settings/sip%d/displayname", nType - 11 );
			nSkip = findString( pnData, 0, pnTmp );
			g_free( pnTmp );
			break;
		}
	}

	if ( nSkip == -1 ) {
		return NULL;
	}

	nSkip = findString( pnData, nSkip, "value=\"" );
	if ( nSkip == -1 ) {
		return NULL;
	}

	nSkip += 7;
	nEnd = findString( pnData, nSkip, "\"" );
	if ( nEnd - nSkip <= 0 ) {
		return NULL;
	}

	return getSubString( pnData, nSkip, nEnd - nSkip );
}

/**
 * \brief Read msn information from webpage data
 * \param psProfile profile structure
 * \param pnData webpage data
 */
static void readMsn( struct sProfile *psProfile, gchar *pnData ) {
	const gchar *pnFon = NULL;
	int nIndex = 0;
	int nSkip = 0, nStart = 0, nEnd = 0;

	routerClearNumbers( psProfile );

	nSkip = findString( pnData, 0, "readFonNumbers()" );

	if ( nSkip != -1 ) {
		/* POTS */
		nSkip = findString( pnData, nSkip, "nrs.pots" );
		if ( nSkip != -1 ) {
			nStart = findString( pnData, nSkip, "\"" );
			nEnd = findString( pnData, nStart + 1, "\"" );
			if ( nEnd - nStart - 1 > 0 ) {
				pnFon = getSubString( pnData, nStart + 1, nEnd - nStart - 1 );
				routerSetFonNumber( psProfile, 10, pnFon );
				Debug( KERN_DEBUG, "MSN10 %s\n", pnFon );
			}
		} else {
			nSkip = 0;
		}

		/* MSN */
		for ( nIndex = 0; nIndex < 10; nIndex++ ) {
			nSkip = findString( pnData, nSkip, "nrs.msn.push" );
			if ( nSkip != -1 ) {
				nStart = findString( pnData, nSkip, "\"" );
				nEnd = findString( pnData, nStart + 1, "\"" );
				if ( nEnd - nStart - 1 > 0 ) {
					pnFon = getSubString( pnData, nStart + 1, nEnd - nStart - 1 );
					routerSetFonNumber( psProfile, nIndex, pnFon );
					Debug( KERN_DEBUG, "MSN%d %s\n", nIndex, pnFon );
				}
				nSkip = nEnd;
			}
		}

		/* SIP */
		for ( nIndex = 0; nIndex < 19; nIndex++ ) {
			nSkip = findString( pnData, nSkip, "nrs.sip.push" );
			if ( nSkip != -1 ) {
				nStart = findString( pnData, nSkip, "\"" );
				nEnd = findString( pnData, nStart + 1, "\"" );
				if ( nEnd - nStart - 1 > 0 ) {
					pnFon = getSubString( pnData, nStart + 1, nEnd - nStart - 1 );
					routerSetFonNumber( psProfile, 11 + nIndex, pnFon );
					Debug( KERN_DEBUG, "MSN%d %s\n", 11 + nIndex, pnFon );
				}
				nSkip = nEnd;
			}
		}
	} else {
		for ( nIndex = 0; nIndex < 15; nIndex++ ) {
			pnFon = getFon( pnData, nIndex );
			if ( pnFon != NULL ) {
				routerSetFonNumber( psProfile, nIndex, pnFon );
				Debug( KERN_DEBUG, "MSN%d %s\n", nIndex, pnFon );
			}
		}
	}
}

/**
 * \brief Get active telephone ports
 * \param psProfile profile structure
 * \return error code
 */
static int fritzBoxGetActivePorts( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nPos = -1, nStart, nEnd, nIndex;
	gchar *pnName;
	gint nError = -1;

	Debug( KERN_DEBUG, "Get active ports...\n" );

	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return nError;
	}

	switch ( psProfile -> sRouterInfo.nType ) {
		case FRITZBOX_FON_WLAN:
		case FRITZBOX_5050:
			snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm?getpage=../html/%s/menus/menu2.html&var:lang=%s&var:pagename=isdn&var:menu=fon&sid=%s", routerGetHost( psProfile ), fritzBoxGetLanguage( psProfile ), fritzBoxGetLanguage( psProfile ), psProfile -> sRouterInfo.pnSessionId );
			break;
		case FRITZBOX_7050:
			snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm?getpage=../html/%s/menus/menu2.html&var:lang=%s&var:pagename=fondevices&var:menu=fon&sid=%s", routerGetHost( psProfile ), fritzBoxGetLanguage( psProfile ), fritzBoxGetLanguage( psProfile ), psProfile -> sRouterInfo.pnSessionId );
			break;
		default:
			snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm?getpage=../html/%s/menus/menu2.html&var:lang=%s&var:pagename=fondevices&var:menu=home&sid=%s", routerGetHost( psProfile ), fritzBoxGetLanguage( psProfile ), fritzBoxGetLanguage( psProfile ), psProfile -> sRouterInfo.pnSessionId );
			break;
	}

	psHandler = urlHandler( anUrl, 80 );
	nError = readUrl( psHandler, psProfile );
	if ( nError == 0 ) {
		saveDebugData( "./fritzbox-getactiveports.html", psHandler -> pnData, psHandler -> nSize );

		/* Extract msn */
		readMsn( psProfile, psHandler -> pnData );

		/* Extract port names */
		for ( nIndex = 0; nIndex < NAME_COUNT; nIndex++ ) {
			nPos = findString( psHandler -> pnData, 0, asPortNames[ nIndex ].pnName );
			if ( nPos == -1 ) {
				nPos = 0;
				continue;
			}
			nStart = findString( psHandler -> pnData, nPos, "value=\"" );
			if ( nStart == -1 ) {
				nPos = 0;
				continue;
			}
			nStart += 7;
			nEnd = findString( psHandler -> pnData, nStart, "\"" );
			if ( nEnd == -1 ) {
				nPos = 0;
				continue;
			}
			if ( nEnd - nStart == 0 ) {
				continue;
			}
			pnName = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
			gchar *pnConvertName = convertEntities( pnName );
			Debug( KERN_DEBUG, "Adding to port %d, name '%s'\n", asPortNames[ nIndex ].nNumber, pnConvertName );
			routerSetPortName( psProfile, asPortNames[ nIndex ].nType, pnConvertName );
			g_free( pnConvertName );
			g_free( pnName );
		}
	} else {
		Debug( KERN_WARNING, "No data!\n" );
	}

	freeHandler( psHandler );

	routerLogout( psProfile );

	return 0;
}

/**
 * \brief Get phone settings of fritzbox
 * \param psProfile profile structure
 * \return error code
 */
static int fritzBoxGetPhoneSettings( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nError = -1;

	if ( psProfile -> sRouterInfo.pnCountryCode != NULL || psProfile -> sRouterInfo.pnAreaCode != NULL ) {
		return 0;
	}

	Debug( KERN_DEBUG, "Get phone settings...\n" );
	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "Login failed\n" );
		return nError;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm?getpage=../html/%s/menus/menu2.html&var:lang=%s&var:pagename=sipoptionen&var:menu=fon&sid=%s", routerGetHost( psProfile ), fritzBoxGetLanguage( psProfile ), fritzBoxGetLanguage( psProfile ), psProfile -> sRouterInfo.pnSessionId );
	psHandler = urlHandler( anUrl, 80 );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		routerLogout( psProfile );
		return nError;
	}

	gint nCountryPrefixStart = findString( psHandler -> pnData, 0, "telcfg:settings/Location/LKZ" );
	if ( nCountryPrefixStart == -1 ) {
		Debug( KERN_DEBUG, "error parsing country prefix start\n" );
		goto end;
	}
	nCountryPrefixStart += 37;

	gint nCountryPrefixStop = findString( psHandler -> pnData, nCountryPrefixStart, "\"" );
	if ( nCountryPrefixStop == -1 ) {
		Debug( KERN_DEBUG, "error parsing country prefix stop\n" );
		goto end;
	}

	gint nCityPrefixStart = findString( psHandler -> pnData, 0, "telcfg:settings/Location/OKZ" );
	if ( nCityPrefixStart == -1 ) {
		Debug( KERN_DEBUG, "error parsing city prefix start\n" );
		goto end;
	}
	nCityPrefixStart += 37;

	gint nCityPrefixStop = findString( psHandler -> pnData, nCityPrefixStart, "\"" );
	if ( nCityPrefixStop == -1 ) {
		Debug( KERN_DEBUG, "error parsing city prefix stop\n" );
		goto end;
	}

	gint nInternationalPrefixStart = findString( psHandler -> pnData, 0, "telcfg:settings/Location/LKZPrefix" );
	if ( nInternationalPrefixStart == -1 ) {
		Debug( KERN_DEBUG, "error parsing international prefix start\n" );
		goto end;
	}
	nInternationalPrefixStart += 43;

	gint nInternationalPrefixStop = findString( psHandler -> pnData, nInternationalPrefixStart, "\"" );
	if ( nInternationalPrefixStop == -1 ) {
		Debug( KERN_DEBUG, "error parsing international prefix stop\n" );
		goto end;
	}

	gint nNationalPrefixStart = findString( psHandler -> pnData, 0, "telcfg:settings/Location/OKZPrefix" );
	if ( nNationalPrefixStart == -1 ) {
		Debug( KERN_DEBUG, "error parsing national prefix start\n" );
		goto end;
	}
	nNationalPrefixStart += 43;

	gint nNationalPrefixStop = findString( psHandler -> pnData, nNationalPrefixStart, "\"" );
	if ( nNationalPrefixStop == -1 ) {
		Debug( KERN_DEBUG, "error parsing national prefix stop\n" );
		goto end;
	}

	psProfile -> sRouterInfo.pnInternationalPrefix = getSubString( psHandler -> pnData, nInternationalPrefixStart, nInternationalPrefixStop - nInternationalPrefixStart );
	psProfile -> sRouterInfo.pnCountryCode = getSubString( psHandler -> pnData, nCountryPrefixStart, nCountryPrefixStop - nCountryPrefixStart );
	psProfile -> sRouterInfo.pnNationalPrefix = getSubString( psHandler -> pnData, nNationalPrefixStart, nNationalPrefixStop - nNationalPrefixStart );
	psProfile -> sRouterInfo.pnAreaCode = getSubString( psHandler -> pnData, nCityPrefixStart, nCityPrefixStop - nCityPrefixStart );

	if ( psProfile -> sRouterInfo.pnCountryCode == NULL ) {
		Debug( KERN_DEBUG, "Country code not found! Set to germany (49)\n" );
		psProfile -> sRouterInfo.pnCountryCode = g_strdup( "49" );
	}
	if ( psProfile -> sRouterInfo.pnAreaCode == NULL ) {
		Debug( KERN_DEBUG, "Area code not found!\n" );
	}
	if ( psProfile -> sRouterInfo.pnInternationalPrefix == NULL ) {
		Debug( KERN_DEBUG, "International prefix not found!\n" );
	}
	if ( psProfile -> sRouterInfo.pnNationalPrefix == NULL ) {
		Debug( KERN_DEBUG, "National prefix not found!\n" );
	}

	Debug( KERN_DEBUG, "CountryCode: %s\n", psProfile -> sRouterInfo.pnCountryCode );
	Debug( KERN_DEBUG, "AreaCode: %s\n", psProfile -> sRouterInfo.pnAreaCode );
	Debug( KERN_DEBUG, "National prefix: %s\n", psProfile -> sRouterInfo.pnNationalPrefix );
	Debug( KERN_DEBUG, "International prefix: %s\n", psProfile -> sRouterInfo.pnInternationalPrefix );

end:

	freeHandler( psHandler );
	routerLogout( psProfile );

	return nError;
}

/**
 * \brief Read call list from fritzbox
 * \param psProfile profile structure
 * \param pnCallerList caller list file
 * \return error code
 */
static int fritzBoxGetCallList( struct sProfile *psProfile, const gchar *pnCallerList ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nError = -1;

	Debug( KERN_DEBUG, "Get calllist...\n" );
	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return nError;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );
	setPostData( psHandler,
		"getpage=../html/%s/menus/menu2.html&var:lang=%s&var:pagename=foncalls&var:menu=fon&sid=%s", fritzBoxGetLanguage( psProfile ), fritzBoxGetLanguage( psProfile ), psProfile -> sRouterInfo.pnSessionId );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		routerLogout( psProfile );
		return nError;
	}
	saveDebugData( "fritzbox-list1.html", psHandler -> pnData, psHandler -> nSize );

	gint nUrlPos = findString( psHandler -> pnData, 0, ".csv" );
	if ( nUrlPos == -1 ) {
		freeHandler( psHandler );
		return -1;
	}

	gint nUrlStop = findString( psHandler -> pnData, nUrlPos, "\"" );
	if ( nUrlStop == -1 ) {
		freeHandler( psHandler );
		return -1;
	}

	gint nUrlStart = findStringReverse( psHandler -> pnData, nUrlPos, "\"" ) + 1;
	if ( nUrlStart == -1 ) {
		freeHandler( psHandler );
		return -1;
	}

	gchar *pnUrl = getSubString( psHandler -> pnData, nUrlStart, nUrlStop - nUrlStart );
	if ( pnUrl == NULL ) {
		freeHandler( psHandler );
		return -1;
	}

	freeHandler( psHandler );

	psHandler = urlHandler( anUrl, 80 );
	setPostData( psHandler, "getpage=%s&sid=%s", pnUrl, psProfile -> sRouterInfo.pnSessionId );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		routerLogout( psProfile );
		return nError;
	}
	saveDebugData( "fritzbox-list2.html", psHandler -> pnData, psHandler -> nSize );

	if ( strlen( pnCallerList ) ) {
		gint nSkip = 0;

		if ( strncmp( psHandler -> pnData, "HTTP", 4 ) == 0 ) {
			nSkip = findString( psHandler -> pnData, 0, "\r\n\r\n" );
			nSkip += 4;
		}
		parseCsvData( psHandler -> pnData + nSkip, psProfile -> pnName );
		saveCallerList();
		Debug( KERN_DEBUG, "saved callerlist\n" );
		freeHandler( psHandler );
		routerLogout( psProfile );
		return 0;
	}

	Debug( KERN_DEBUG, "no callerlist\n" );
	freeHandler( psHandler );
	routerLogout( psProfile );
	return -4;
}

/**
 * \brief Clear callerlist on fritzbox
 * \param psProfile profile structure
 * \return error code
 */
static int fritzBoxClearCallList( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nError = -1;

	Debug( KERN_DEBUG, "Clear calllist...\n" );
	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return nError;
	}

	Debug( KERN_DEBUG, "Delete call list on fritzbox...\n" );

	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );
	setPostData( psHandler, "getpage=../html/%s/menus/menu2.html&var:lang=%s&var:pagename=foncalls&var:menu=fon&telcfg:settings/ClearJournal=1&sid=%s", fritzBoxGetLanguage( psProfile ), fritzBoxGetLanguage( psProfile ), psProfile -> sRouterInfo.pnSessionId );
	nError = readUrl( psHandler, psProfile );
	if ( nError == 0 ) {
		Debug( KERN_DEBUG, "Success (%d)\n", psHandler -> nSize );
		saveDebugData( "fritzbox-clearlist.html", psHandler -> pnData, psHandler -> nSize );
	} else {
		Debug( KERN_DEBUG, "Failed\n" );
		saveDebugData( "fritzbox-clearlisterror.html", psHandler -> pnData, psHandler -> nSize );
	}

	freeHandler( psHandler );

	routerLogout( psProfile );

	return nError;
}

/**
 * \brief Disconnect and quick reconnect
 * \param psProfile profile structure
 * \return error code
 */
static int fritzBoxReconnect( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	struct curl_slist *psList = NULL;
	gint nError = -1;

	snprintf( anUrl, sizeof( anUrl ), "http://%s:49000/upnp/control/WANIPConn1", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 49000 );

	psList = curl_slist_append( psList, "Content-Type: text/xml; charset=\"utf-8\"" );
	psList = curl_slist_append( psList, "SoapAction: urn:schemas-upnp-org:service:WANIPConnection:1#ForceTermination" );

	curl_easy_setopt( psHandler -> psHandle, CURLOPT_HTTPHEADER, psList );
	setPostData( psHandler,
		"<?xml version=\"1.0\" encoding=\"utf-8\"?>"
		" <s:Envelope s:encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" xmlns:s=\"http://schemas.xmlsoap.org/soap/envelope/\">"
		" <s:Body>"
		" <u:ForceTermination xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:1\" />"
		" </s:Body>"
		" </s:Envelope>" );

	nError = readUrl( psHandler, psProfile );

	curl_slist_free_all( psList );
	freeHandler( psHandler );

	return nError;
}

/**
 * \brief Get external ip
 * \param psProfile profile structure
 * \return external ip address or NULL on error
 */
static gchar *fritzBoxGetIp( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	struct curl_slist *psList = NULL;
	gchar *pnIp = NULL;
	gint nError = -1;

	if ( psProfile == NULL ) {
		return NULL;
	}

	snprintf( anUrl, sizeof( anUrl ), "http://%s:49000/upnp/control/WANCommonIFC1", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 49000 );

	psList = curl_slist_append( psList, "Content-Type: text/xml; charset=\"utf-8\"" );
	psList = curl_slist_append( psList, "SoapAction: urn:schemas-upnp-org:service:WANIPConnection:1#GetExternalIPAddress" );

	curl_easy_setopt( psHandler -> psHandle, CURLOPT_HTTPHEADER, psList );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_IPRESOLVE, 1 );
	setPostData( psHandler,
		"<?xml version=\"1.0\" encoding=\"utf-8\"?>"
		" <s:Envelope s:encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" xmlns:s=\"http://schemas.xmlsoap.org/soap/envelope/\">"
		" <s:Body>"
		" <u:GetExternalIPAddress xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:1\" />"
		" </s:Body>"
		" </s:Envelope>" );

	nError = readUrl( psHandler, psProfile );
	if ( nError == 0 ) {
		gint nStart = findString( psHandler -> pnData, 0, "<NewExternalIPAddress>" );

		if ( nStart != -1 ) {
			gint nEnd = findString( psHandler -> pnData, nStart + 22, "</NewExternalIPAddress>" );
			if ( nEnd != -1 ) {
				pnIp = getSubString( psHandler -> pnData, nStart + 22, nEnd - nStart - 22 );
			}
		}
	}

	curl_slist_free_all( psList );
	freeHandler( psHandler );

	return pnIp;
}

/**
 * \brief Get maximal speed
 * \param psProfile profile structure
 * \param pnMaxUp maximal up speed
 * \param pnMaxDown maximual down speed
 */
static void fritzBoxGetSpeed( struct sProfile *psProfile, gint *pnMaxUp, gint *pnMaxDown ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	struct curl_slist *psList = NULL;
	gchar *pnTmp = NULL;
	gint nError = -1;

	if ( psProfile == NULL ) {
		return;
	}

	snprintf( anUrl, sizeof( anUrl ), "http://%s:49000/upnp/control/WANCommonIFC1", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 49000 );

	psList = curl_slist_append( psList, "Content-Type: text/xml; charset=\"utf-8\"" );
	psList = curl_slist_append( psList, "SoapAction: urn:schemas-upnp-org:service:WANCommonInterfaceConfig:1#GetCommonLinkProperties" );

	curl_easy_setopt( psHandler -> psHandle, CURLOPT_HTTPHEADER, psList );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_IPRESOLVE, 1 );
	setPostData( psHandler,
		"<?xml version=\"1.0\" encoding=\"utf-8\"?>"
		" <s:Envelope s:encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" xmlns:s=\"http://schemas.xmlsoap.org/soap/envelope/\">"
		" <s:Body>"
		" <u:GetCommonLinkProperties xmlns:u=\"urn:schemas-upnp-org:service:WANCommonInterfaceConfig:1\" />"
		" </s:Body>"
		" </s:Envelope>" );

	nError = readUrl( psHandler, psProfile );
	if ( nError == 0 ) {
		int nStart = findString( psHandler -> pnData, 0, "<NewLayer1UpstreamMaxBitRate>" );

		if ( nStart != -1 ) {
			int nEnd = findString( psHandler -> pnData, nStart + 29, "</NewLayer1UpstreamMaxBitRate>" );
			if ( nEnd != -1 ) {
				pnTmp = getSubString( psHandler -> pnData, nStart + 29, nEnd - nStart - 29 );
				//if ( pnTmp != NULL ) {
					*pnMaxUp = atoi( pnTmp );
					g_free( pnTmp );
				//}
			}
		}

		nStart = findString( psHandler -> pnData, 0, "<NewLayer1DownstreamMaxBitRate>" );

		if ( nStart != -1 ) {
			int nEnd = findString( psHandler -> pnData, nStart + 31, "</NewLayer1DownstreamMaxBitRate>" );
			if ( nEnd != -1 ) {
				pnTmp = getSubString( psHandler -> pnData, nStart + 31, nEnd - nStart - 31 );
				//if ( pnTmp != NULL ) {
					*pnMaxDown = atoi( pnTmp );
					g_free( pnTmp );
				//}
			}
		}
	}

	curl_slist_free_all( psList );
	freeHandler( psHandler );
}

/**
 * \brief Get fritzbox internal port number by port name
 * \param nType port type
 * \return internal port number
 */
static int fritzBoxGetPortNumber( int nType ) {
	int nIndex;

	for ( nIndex = 0; nIndex < NAME_COUNT; nIndex++ ) {
		if ( asPortNames[ nIndex ].nType == nType ) {
			return asPortNames[ nIndex ].nNumber;
		}
	}

	return -1;
}

/**
 * \brief Replace # and *
 * \param pnNumber input number
 * \return replaced number
 */
gchar *urlEncode( const gchar *pnNumber ) {
	GString *psNewNumber;

	if ( pnNumber == NULL ) {
		return NULL;
	}

	psNewNumber = g_string_new( "" );

	while ( *pnNumber != 0 ) {
		if ( *pnNumber == '*' ) {
			g_string_append( psNewNumber, "%2A" );
		} else if ( *pnNumber == '#' ) {
			g_string_append( psNewNumber, "%23" );
		} else {
			g_string_append_c( psNewNumber, *pnNumber );
		}
		pnNumber++;
	}

	return g_string_free( psNewNumber, FALSE );
}

/**
 * \brief Call a number via fritzbox
 * \param psProfile profile structure
 * \param pnNumber telephone number
 * \param nType port type
 * \return error code
 */
static int fritzBoxCallNumber( struct sProfile *psProfile, const gchar *pnNumber, int nType ) {
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ BUF_SIZE ];
	gint nPort = 0;
	gint nError = -1;

	Debug( KERN_DEBUG, "Call number...\n" );

	nPort = fritzBoxGetPortNumber( nType );
	if ( nPort == -1 ) {
		return nError;
	}

	nError = routerLogin( psProfile );
	if ( nError == 0 ) {
		gchar *pnTmp1 = urlEncode( pnNumber );

		snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
		psHandler = urlHandler( anUrl, 80 );

		Debug( KERN_INFO, "pnTmp1: '%s'\n", pnTmp1 );
		setPostData( psHandler,
					"&telcfg:settings/UseClickToDial=1&telcfg:settings/DialPort=%d&telcfg:command/Dial=%s&sid=%s", nPort, pnTmp1, psProfile -> sRouterInfo.pnSessionId );
		nError = readUrl( psHandler, psProfile );
		g_free( pnTmp1 );

		freeHandler( psHandler );
		routerLogout( psProfile );
	}

	return nError;
}

/**
 * \brief Hangup call via fritzbox
 * \param psProfile profile structure
 * \param nPort port number
 * \return error code
 */
static int fritzBoxHangup( struct sProfile *psProfile, int nType ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nPort = 0;
	gint nError = -1;

	Debug( KERN_DEBUG, "Hang number...\n" );

	nPort = fritzBoxGetPortNumber( nType );
	if ( nPort == -1 ) {
		return nError;
	}

	nError = routerLogin( psProfile );
	if ( nError == 0 ) {
		snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
		psHandler = urlHandler( anUrl, 80 );
		setPostData( psHandler, "&telcfg:settings/UseClickToDial=1&telcfg:settings/DialPort=%d&telcfg:command/Hangup&sid=%s", nPort, psProfile -> sRouterInfo.pnSessionId );
		nError = readUrl( psHandler, psProfile );

		freeHandler( psHandler );
		routerLogout( psProfile );
	}

	return nError;
}

/**
 * \brief Get fritzbox name
 * \param psProfile profile structure
 * \return fritzbox name or NULL for unknown
 */
static const char *fritzBoxGetName( struct sProfile *psProfile ) {
	switch ( psProfile -> sRouterInfo.nType ) {
		case FRITZBOX_FON:
			return "FRITZ!Box Fon";
		case FRITZBOX_FON_WLAN:
			return "FRITZ!Box Fon WLAN";
		case FRITZBOX_ATA:
			return "FRITZ!Box ata";
		case FRITZBOX_5050:
			return "FRITZ!Box 5050";
		case FRITZBOX_7050:
			return "FRITZ!Box 7050";
		case EUMEX_300:
			return "Eumex 300ip";
		case FRITZBOX_5010:
			return "FRITZ!Box 5010";
		case FRITZBOX_5012:
			return "FRITZ!Box 5012";
		case FRITZBOX_SPEEDPORT_W501V:
			return "T-Com Speedport W 501V";
		case FRITZBOX_7170:
			return "FRITZ!Box 7170";
		case FRITZBOX_7170A:
			return "FRITZ!Box 7170A";
		case FRITZBOX_7140:
			return "FRITZ!Box 7140";
		case FRITZBOX_SPEEDPORT_W900V:
			return "T-Com Speedport W 900V";
		case FRITZBOX_7141:
			return "FRITZ!Box 7141";
		case FRITZBOX_5140:
			return "FRITZ!Box 5140";
		case FRITZBOX_7270:
			return "FRITZ!Box 7270";
		case FRITZBOX_7270V3:
			return "FRITZ!Box 7270V3";
		case FRITZBOX_7113:
			return "FRITZ!Box 7113";
		case FRITZBOX_7150:
			return "FRITZ!Box 7150";
		case FRITZBOX_7240:
			return "FRITZ!Box 7240";
		case FRITZBOX_7320:
			return "FRITZ!Box 7320";
		case FRITZBOX_7340:
			return "FRITZ!Box 7340";
		case FRITZBOX_7390:
			return "FRITZ!Box 7390";
		case FRITZBOX_7570:
			return "FRITZ!Box 7570 vDSL";
		default:
			break;
	}

	return NULL;
}

/**
 * \brief Get fritzbox version
 * \param psProfile profile structure
 * \return fritzbox version
 */
static const gchar *fritzBoxGetVersion( struct sProfile *psProfile ) {
	if ( psProfile -> sRouterInfo.pnFirmware != NULL ) {
		return psProfile -> sRouterInfo.pnFirmware;
	}

	return "Unknown";
}

/**
 * \brief Init fritz box info structure
 * \param psProfile profile structure
 * \return error code 0
 */
static int fritzBoxInit( struct sProfile *psProfile ) {
	memset( &psProfile -> sRouterInfo, 0, sizeof( psProfile -> sRouterInfo ) );
	psProfile -> sRouterInfo.pnFirmware = ( gchar * ) routerGetFirmware( psProfile );
	commonParseFirmware( psProfile );

	psLogMutex = g_mutex_new();

	return 0;
}

/**
 * \brief Read diversity information from fritzbox
 * \param psProfile active profile structure
 * \return error code
 */
static gint fritzBoxReadDiversity( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nStart;
	gint nEnd;
	gint nCount = 0;
	gint nIndex;
	gint nFromStart = -1;
	gint nFromEnd;
	gint nToStart;
	gint nToEnd;
	gchar *pnFrom = NULL;
	gchar *pnTo = NULL;
	gchar *pnName = NULL;
	gchar *pnTmp = NULL;
	struct sDiversity *psDiversity = NULL;
	GList *psList = NULL;
	gint nError = -1;
	gint nRulCount = 0;
	gint nRubCount = 0;
	gint nNsCount = 0;

	Debug( KERN_DEBUG, "Read diversity\n" );
	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return nError;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm?getpage=../html/%s/menus/menu2.html&var:lang=%s&var:pagename=rulall&var:menu=fon&sid=%s", routerGetHost( psProfile ), fritzBoxGetLanguage( psProfile ), fritzBoxGetLanguage( psProfile ), psProfile -> sRouterInfo.pnSessionId );
	psHandler = urlHandler( anUrl, 80 );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		routerLogout( psProfile );
		return nError;
	}
	saveDebugData( "fritzbox-diversity.html", psHandler -> pnData, psHandler -> nSize );

	nStart = 0;
	while ( ( nStart = findString( psHandler -> pnData, nStart, "document.write(TrRul(" ) ) ) {
		if ( nStart == -1 ) {
			break;
		}
		nRulCount++;
		nStart += 20;

		nFromStart = findString( psHandler -> pnData, nStart, "," );
		if ( nFromStart == -1 ) {
			continue;
		}

		nFromStart = findString( psHandler -> pnData, nFromStart, "\"" );
		if ( nFromStart == -1 ) {
			continue;
		}
		nFromStart++;

		nFromEnd = findString( psHandler -> pnData, nFromStart, "\"" );

		pnFrom = getSubString( psHandler -> pnData, nFromStart, nFromEnd - nFromStart );

		nToStart = findString( psHandler -> pnData, nFromEnd + 2, "," );
		if ( nToStart == -1 ) {
			g_free( pnFrom );
			continue;
		}

		nToStart = findString( psHandler -> pnData, nToStart, "\"" );
		if ( nToStart == -1 ) {
			g_free( pnFrom );
			continue;
		}
		nToStart++;

		nToEnd = findString( psHandler -> pnData, nToStart, "\"" );

		pnTo = getSubString( psHandler -> pnData, nToStart, nToEnd - nToStart );

		psDiversity = g_malloc( sizeof( struct sDiversity ) );
		psDiversity -> nIndex = nRulCount - 1;
		if ( isdigit( pnFrom[ 0 ] ) != 0 ) {
			/* MSN? */
			psDiversity -> pnName = g_strdup_printf( "%s -> %s", routerGetFonNumber( psProfile, pnFrom[ 0 ] - '0' ), pnTo );
		} else if ( pnFrom[ 0 ] == 'P' ) {
			/* POTS */
			psDiversity -> pnName = g_strdup_printf( "%s -> %s", routerGetFonNumber( psProfile, 10 ), pnTo );
		} else if ( pnFrom[ 0 ] == 'S' ) {
			/* SIP */
			psDiversity -> pnName = g_strdup_printf( "%s -> %s", routerGetFonNumber( psProfile, 11 + pnFrom[ 3 ] - '0' ), pnTo );
		} else {
			g_free( pnFrom );
			pnFrom = NULL;
			g_free( psDiversity );
			psDiversity = NULL;
			continue;
		}

		psDiversity -> bSet = FALSE;

		psDiversityList = g_list_append( psDiversityList, psDiversity );

		g_free( pnTo );
		g_free( pnFrom );

		nCount++;
	}

	nStart = 0;
	while ( ( nStart = findString( psHandler -> pnData, nStart, "document.write(TrRub(" ) ) ) {
		if ( nStart == -1 ) {
			break;
		}
		nRubCount++;
		nStart += 20;

		nFromStart = findString( psHandler -> pnData, nStart, "," );
		if ( nFromStart == -1 ) {
			continue;
		}

		nFromStart = findString( psHandler -> pnData, nFromStart, "\"" );
		if ( nFromStart == -1 ) {
			continue;
		}
		nFromStart++;

		nFromEnd = findString( psHandler -> pnData, nFromStart, "\"" );

		if ( nFromEnd - nFromStart <= 0 ) {
			continue;
		}

		pnFrom = getSubString( psHandler -> pnData, nFromStart, nFromEnd - nFromStart );

		nToStart = findString( psHandler -> pnData, nFromEnd + 2, "," );
		if ( nToStart == -1 ) {
			g_free( pnFrom );
			continue;
		}

		nToStart = findString( psHandler -> pnData, nToStart, "\"" );
		if ( nToStart == -1 ) {
			g_free( pnFrom );
			continue;
		}
		nToStart++;

		nToEnd = findString( psHandler -> pnData, nToStart, "\"" );

		pnTo = getSubString( psHandler -> pnData, nToStart, nToEnd - nToStart );

		psDiversity = g_malloc( sizeof( struct sDiversity ) );
		psDiversity -> nIndex = 64 + nRubCount - 1;
		if ( pnFrom != NULL && strlen( pnFrom ) > 0 ) {
			if ( strcmp( pnFrom, "0" ) == 0 ) {
				psDiversity -> pnName = g_strdup_printf( _( "All incoming calls -> %s" ), pnTo );
			} else if ( pnFrom[ 0 ] == '#' ) {
				psDiversity -> pnName = g_strdup_printf( _( "Unknown on %s -> %s" ), pnFrom + 1, pnTo );
			} else {
				if ( pnTo == NULL || strlen( pnTo ) <= 0 ) {
					psDiversity -> pnName = g_strdup_printf( _( "%s (BLOCKED)" ), pnFrom );
				} else {
					psDiversity -> pnName = g_strdup_printf( _( "%s -> %s" ), pnFrom, pnTo );
				}
			}
		} else {
			psDiversity -> pnName = g_strdup_printf( _( "Unknown -> %s" ), pnTo );
		}
		psDiversity -> bSet = FALSE;

		psDiversityList = g_list_append( psDiversityList, psDiversity );

		g_free( pnTo );
		g_free( pnFrom );

		nCount++;
	}

	nStart = 0;
	while ( ( nStart = findString( psHandler -> pnData, nStart, "document.write(TrNs(" ) ) ) {
		if ( nStart == -1 ) {
			break;
		}
		nNsCount++;
		nStart += 19;

		nFromStart = findString( psHandler -> pnData, nStart, "\"" );
		if ( nFromStart == -1 ) {
			continue;
		}
		nFromStart++;

		nFromEnd = findString( psHandler -> pnData, nFromStart, "\"" );

		if ( nFromEnd - nFromStart <= 0 ) {
			continue;
		}

		pnFrom = getSubString( psHandler -> pnData, nFromStart, nFromEnd - nFromStart );

		nToStart = findString( psHandler -> pnData, nFromEnd + 2, "\"" );
		if ( nToStart == -1 ) {
			g_free( pnFrom );
			continue;
		}
		nToStart++;

		nToEnd = findString( psHandler -> pnData, nToStart, "\"" );
		if ( nToEnd - nToStart <= 0 ) {
			g_free( pnFrom );
			continue;
		}

		pnTo = getSubString( psHandler -> pnData, nToStart, nToEnd - nToStart );

		psDiversity = g_malloc( sizeof( struct sDiversity ) );
		psDiversity -> nIndex = 128 + nNsCount - 1;
		if ( pnFrom != NULL && strlen( pnFrom ) > 0 ) {
			psDiversity -> pnName = g_strdup_printf( _( "Tel %s -> %s" ), pnFrom, pnTo );
		} else {
			psDiversity -> pnName = g_strdup_printf( _( "Unknown -> %s" ), pnTo );
		}
		psDiversity -> bSet = FALSE;

		psDiversityList = g_list_append( psDiversityList, psDiversity );

		g_free( pnTo );
		g_free( pnFrom );

		nCount++;
	}


again:
	for ( nIndex = 0, psList = psDiversityList; nIndex < nCount; nIndex++, psList = psList -> next ) {
		psDiversity = psList -> data;
		gboolean bFound = FALSE;

		if ( psDiversity -> nIndex < 64 ) {
			pnName = g_strdup_printf( "telcfg:settings/Diversity%d/Active", psDiversity -> nIndex );
		} else if ( psDiversity -> nIndex < 128 ) {
			pnName = g_strdup_printf( "telcfg:settings/CallerIDActions%d/Active", psDiversity -> nIndex - 64 );
		} else {
			pnName = g_strdup_printf( "telcfg:settings/MSN/Port%d/Diversion", psDiversity -> nIndex - 128 );
		}

		nStart = findString( psHandler -> pnData, 0, pnName );
		if ( nStart != -1 ) {
			nStart += 6;
			nStart = findString( psHandler -> pnData, nStart, "value=\"" );
			if ( nStart != -1 ) {
				nStart += 7;
				nEnd = findString( psHandler -> pnData, nStart, "\"" );
				if ( nEnd != -1 && nEnd - nStart > 0 ) {
					pnTmp = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
					if ( *pnTmp == '1' ) {
						psDiversity -> bSet = TRUE;
					}
					bFound = TRUE;
					g_free( pnTmp );
				}
			}
		}
		g_free( pnName );

		if ( bFound == FALSE ) {
			psDiversityList = g_list_remove_link( psDiversityList, psList );
			nCount--;
			goto again;
		}
	}


	freeHandler( psHandler );
	routerLogout( psProfile );

	return nError;
}

/**
 * \brief Set diversity state
 * \param psProfile profile structure
 * \param nIndex diversity index
 * \param bSet diversity state toggle
 * \return error code
 */
static int fritzBoxSetDiversity( struct sProfile *psProfile, gint nIndex, gboolean bSet ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gchar *pnName = NULL;
	gint nError = -1;

	Debug( KERN_DEBUG, "Set diversity\n" );
	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return nError;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s/cgi-bin/webcm", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );

	Debug( KERN_DEBUG, "nIndex %d\n", nIndex );
	if ( nIndex < 64 ) {
		pnName = g_strdup_printf( "telcfg:settings/Diversity%d/Active", nIndex );
	} else if ( nIndex < 128 ) {
		pnName = g_strdup_printf( "telcfg:settings/CallerIDActions%d/Active", nIndex - 64 );
	} else {
		pnName = g_strdup_printf( "telcfg:settings/MSN/Port%d/Diversion", nIndex - 128 );
	}

	setPostData( psHandler, "&%s=%s&sid=%s", pnName, bSet == TRUE ? "1" : "0", psProfile -> sRouterInfo.pnSessionId );
	g_free( pnName );
	nError = readUrl( psHandler, psProfile );
	if ( nError == 0 ) {
		routerLogout( psProfile );
	}

	freeHandler( psHandler );

	return nError;
}

/**
 * \brief Swap encoding
 * \param nValue number
 * \return swapped number
 */
static inline unsigned int swap( unsigned int nValue ) {
	unsigned int nResult = 0;
	unsigned char *pnTemp = ( unsigned char * )( &nResult );

	*pnTemp = (unsigned char)( ( nValue >> 24 ) & 0x000000ff );
	pnTemp++;
	*pnTemp = (unsigned char)( ( nValue >> 16 ) & 0x000000ff );
	pnTemp++;
	*pnTemp = (unsigned char)( ( nValue >>  8 ) & 0x000000ff );
	pnTemp++;
	*pnTemp = (unsigned char)( nValue & 0x000000ff );

	return nResult;
}

/**
 * \brief parse voice data file and add to list store
 * \param pnData file data pointer
 * \param nLen len of data
 */
static void fritzBoxParseVoiceData( gchar *pnData, gint nLen ) {
	struct sVoiceData *psVoiceData = NULL;
	gint nCount = 0;
	gint nIndex;
	struct sCaller sCaller;

	if ( nLen % sizeof( struct sVoiceData ) ) {
		return;
	}
	ffgtkLock();
	nCount = nLen / sizeof( struct sVoiceData );
	for ( nIndex = 0; nIndex < nCount; nIndex++ ) {
		psVoiceData = ( struct sVoiceData * ) ( pnData + nIndex * sizeof( struct sVoiceData ) );

		if ( psVoiceData -> nHeader == 0x5C010000 ) {
			psVoiceData -> nHeader = swap( psVoiceData -> nHeader );
			psVoiceData -> nType = swap( psVoiceData -> nType );
			psVoiceData -> nSubType = swap( psVoiceData -> nSubType );
			psVoiceData -> nSize = swap( psVoiceData -> nSize );
			psVoiceData -> nDuration = swap( psVoiceData -> nDuration );
			psVoiceData -> nStatus = swap( psVoiceData -> nStatus );
		}
		if ( strncmp( psVoiceData -> anFile, "uvp", 3 ) == 0 ) {
			continue;
		}

		/* Create caller structure */
		memset( &sCaller, 0, sizeof( struct sCaller ) );
		sCaller.nType = 4;
		sCaller.pnDateTime = g_strdup_printf( "%2.2d.%2.2d.%2.2d %2.2d:%2.2d", psVoiceData -> nDay, psVoiceData -> nMon, psVoiceData -> nYear, psVoiceData -> nHour, psVoiceData -> nMin );
		sCaller.pnNumber = g_strdup( psVoiceData -> anRemoteNumber );
		sCaller.pnLocalName = g_strdup( psVoiceData -> anFile );
		sCaller.pnLocalNumber = g_strdup( psVoiceData -> anLocalNumber );
		sCaller.pnDuration = g_strdup_printf( "%ds", psVoiceData -> nDuration );

		/* Add caller to list */
		AddCall( "4", sCaller.pnDateTime, "", sCaller.pnNumber, sCaller.pnLocalName, sCaller.pnLocalNumber, sCaller.pnDuration, getActiveProfile() -> pnName );

		callMonitorAddCaller( &sCaller, NULL, NULL );
	}
	ffgtkUnlock();
}

/**
 * \brief Get attached volume name
 * \param psProfile current profile structure
 * \return volume name or NULL on error
 */
static gchar *getVolumeName( struct sProfile *psProfile ) {
	gchar *pnUrl;
	struct sUrlHandler *psHandler;
	gint nError;
	gint nPos = -1;
	gint nEnd = -1;
	gint nStart = -1;
	gchar *pnVolume = NULL;

	pnUrl = g_strdup_printf( "%s/cgi-bin/webcm?getpage=../html/%s/menus/menu2.html&var:lang=%s&var:pagename=status&var:menu=usb&sid=%s", routerGetHost( psProfile ), fritzBoxGetLanguage( psProfile ), fritzBoxGetLanguage( psProfile ), psProfile -> sRouterInfo.pnSessionId );
	psHandler = urlHandler( pnUrl, 80 );
	nError = readUrl( psHandler, psProfile );
	g_free( pnUrl );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "1. Error: %s, data %p\n", curl_easy_strerror( nError ), psHandler -> pnData );
		freeHandler( psHandler );
		return pnVolume;
	}
	saveDebugData( "volume1.html", psHandler -> pnData, psHandler -> nSize );

	nPos = findString( psHandler -> pnData, 0, "idx: \"logvol" );
	if ( nPos != -1 ) {
		nPos = findString( psHandler -> pnData, nPos, "name: \"" );
		if ( nPos != -1 ) {
			nPos += 7;
			nEnd = findString( psHandler -> pnData, nPos, "\"" );
			if ( nEnd != -1 ) {
				pnVolume = getSubString( psHandler -> pnData, nPos, nEnd - nPos );
				if ( pnVolume != NULL ) {
					Debug( KERN_DEBUG, "Found volume: %s\n", pnVolume );
				}
			}
		}
	} else {
		Debug( KERN_DEBUG, "Could not find logvol tag in current page, trying new method:\n" );
		freeHandler( psHandler );

		pnUrl = g_strdup_printf( "%s/usb/show_usb_devices.lua?sid=%s", routerGetHost( psProfile ), psProfile -> sRouterInfo.pnSessionId );
		psHandler = urlHandler( pnUrl, 80 );
		nError = readUrl( psHandler, psProfile );
		g_free( pnUrl );
		if ( nError != 0 ) {
			Debug( KERN_DEBUG, "1. Error: %s, data %p\n", curl_easy_strerror( nError ), psHandler -> pnData );
			freeHandler( psHandler );
			return pnVolume;
		}
		saveDebugData( "volume2.html", psHandler -> pnData, psHandler -> nSize );

		nStart = findString( psHandler -> pnData, 0, "logvol0" );
		if ( nStart != -1 ) {
			nPos = findString( psHandler -> pnData, nStart, "name = " );
			if ( nPos != -1 ) {
				nPos += 7;
				nEnd = findString( psHandler -> pnData, nPos, "\n" );
				if ( nEnd != -1 ) {
					pnVolume = getSubString( psHandler -> pnData, nPos, nEnd - nPos );
					if ( pnVolume != NULL ) {
						Debug( KERN_DEBUG, "Found volume: %s\n", pnVolume );
					}
				}
			} else {
				nPos = findString( psHandler -> pnData, nStart, "[\"name\"] = \"" );
				if ( nPos != -1 ) {
					nPos += 12;
					nEnd = findString( psHandler -> pnData, nPos, "\"" );
					if ( nEnd != -1 ) {
						pnVolume = getSubString( psHandler -> pnData, nPos, nEnd - nPos );
						if ( pnVolume != NULL ) {
							Debug( KERN_DEBUG, "Found volume: %s\n", pnVolume );
						}
					}
				}
			}
		}

		if ( pnVolume == NULL ) {
			Debug( KERN_DEBUG, "Could not find logv in current page, trying new method:\n" );

		}
	}
	freeHandler( psHandler );

	return pnVolume;
}

/**
 * \brief Load voicebox content from box and add them to liststore
 * \param psProfile profile structure
 * \return error code (0=success, otherwise error)
 */
static int fritzBoxLoadVoiceBox( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler = NULL;
	gchar *pnUrl = NULL;
	gint nError = -1;
	gint nIndex = 0;
	gchar *pnVolume = NULL;
	const gchar *pnUser = NULL;
	const gchar *pnPassword = NULL;
	gchar *pnUsrPwd = NULL;

	if ( routerLogin( psProfile ) == -1 ) {
		return -1;
	}

	pnVolume = getVolumeName( psProfile );

	if ( pnVolume != NULL ) {
		if ( pnVoiceBoxPath != NULL ) {
			g_free( pnVoiceBoxPath );
		}
		pnVoiceBoxPath = pnVolume;
	} else {
		pnVoiceBoxPath = g_strdup( "" );
	}

	pnUser = voiceBoxGetUser( psProfile );
	pnPassword = voiceBoxGetPassword( psProfile );
	if ( pnUser != NULL && strlen( pnUser ) > 0 ) {
		pnUsrPwd = g_strdup_printf( "%s:%s", pnUser, pnPassword );
	}

	for ( nIndex = 0; nIndex < 5; nIndex++ ) {
again:
		pnUrl = g_strdup_printf( "ftp://%s/%s/FRITZ/voicebox/meta%d", routerGetHost( psProfile ), pnVoiceBoxPath, nIndex );
		Debug( KERN_DEBUG, "URL: %s\n", pnUrl );

		/* create url handler */
		psHandler = urlHandler( pnUrl, 21 );

		if ( pnUsrPwd != NULL ) {
			curl_easy_setopt( psHandler -> psHandle, CURLOPT_USERPWD, pnUsrPwd );
		}

		/* set timeout to 3sec */
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_TIMEOUT, 3 );
		/* try to get data */
		nError = readUrl( psHandler, psProfile );
		g_free( pnUrl );
		switch ( nError ) {
		case CURLE_FAILED_INIT:
			/* check if file is empty */
			nError = CURLE_OK;
			break;
		case CURLE_REMOTE_ACCESS_DENIED:
			if ( pnVoiceBoxPath != NULL && strcmp( pnVoiceBoxPath, "" ) ) {
				free( pnVoiceBoxPath );
				pnVoiceBoxPath = g_strdup( "" );
				goto again;
			}
			break;
		case CURLE_OK:
			/* Copy data to internal structure */
			asVoiceBox[ nIndex ].nLen = psHandler -> nSize;
			asVoiceBox[ nIndex ].pnData = g_malloc( asVoiceBox[ nIndex ].nLen );
			memcpy( asVoiceBox[ nIndex ].pnData, psHandler -> pnData, asVoiceBox[ nIndex ].nLen );

			/* Parse data */
			fritzBoxParseVoiceData( asVoiceBox[ nIndex ].pnData, asVoiceBox[ nIndex ].nLen );
		default:
			/* error */
			Debug( KERN_DEBUG, "Error: %s\n", curl_easy_strerror( nError ) );
			freeHandler( psHandler );
			goto error1;
		}
		freeHandler( psHandler );
	}

error1:

	if ( pnUsrPwd != NULL ) {
		g_free( pnUsrPwd );
	}

	routerLogout( psProfile );
	return nError;
}

/**
 * \brief Load voice file from box
 * \param psProfile profile structure
 * \param pnFile voice file we want to load
 * \return error code (0=success, otherwise error)
 */
int fritzBoxLoadVoiceFile( struct sProfile *psProfile, const gchar *pnFile ) {
	struct sUrlHandler *psHandler = NULL;
	gchar *pnUrl = NULL;
	const gchar *pnUser = NULL;
	const gchar *pnPassword = NULL;
	gint nError = -1;
	FILE *psFile;
	gchar *pnUsrPwd = NULL;
	gchar *pnStorePath = NULL;
	DIR *psDir = NULL;

	gchar *pnDir = g_strdup_printf( "%s%cffgtk-%s", g_get_tmp_dir(), G_DIR_SEPARATOR, g_get_user_name() );

	/* Check if spooler is present, create directory if needed */
	psDir = opendir( pnDir );
	if ( psDir == NULL ) {
		g_mkdir( pnDir, 0777 );
	} else {
		closedir( psDir );
	}

	pnUser = voiceBoxGetUser( psProfile );
	pnPassword = voiceBoxGetPassword( psProfile );
	if ( pnUser != NULL && strlen( pnUser ) > 0 ) {
		pnUsrPwd = g_strdup_printf( "%s:%s", pnUser, pnPassword );
	}

	if ( strncmp( pnFile, "uvp", 3 ) == 0 ) {
		pnUrl = g_strdup_printf( "ftp://%s/%s/FRITZ/voicebox/%s", routerGetHost( psProfile ), pnVoiceBoxPath, pnFile );
	} else {
		pnUrl = g_strdup_printf( "ftp://%s/%s/FRITZ/voicebox/rec/%s", routerGetHost( psProfile ), pnVoiceBoxPath, pnFile );
	}
	Debug( KERN_DEBUG, "Loading '%s'\n", pnUrl );
	/* create url handler */
	psHandler = urlHandler( pnUrl, 21 );
	if ( pnUsrPwd != NULL ) {
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_USERPWD, pnUsrPwd );
	}

	/* set timeout to 3sec */
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_TIMEOUT, 3 );
	/* try to get data */
	nError = readUrl( psHandler, psProfile );
	g_free( pnUrl );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "Error: %s\n", curl_easy_strerror( nError ) );
		goto error1;
	}

	pnStorePath = g_strdup_printf( "%s%cffgtk-%s%cvoice", g_get_tmp_dir(), G_DIR_SEPARATOR, g_get_user_name(), G_DIR_SEPARATOR );
	if ( pnStorePath != NULL ) {
		psFile = fopen( pnStorePath, "wb+" );

		if ( psFile != NULL ) {
			gint nResult = 0;

			nResult = fwrite( psHandler -> pnData, 1, psHandler -> nSize, psFile );
			if ( nResult != psHandler -> nSize ) {
				Debug( KERN_DEBUG, "Warning: Could not save voice file\n" );
			}
			fclose( psFile );
			nError = 0;
		} else {
			nError = -1;
		}
	}

error1:
	freeHandler( psHandler );

	if ( pnUsrPwd != NULL ) {
		g_free( pnUsrPwd );
	}

	return nError;
}

/**
 * \brief Delete voice file from box
 * \param psProfile profile structure
 * \param pnFile voice file we want to delete
 * \return error code (0=success, otherwise error)
 */
int fritzBoxDeleteVoiceFile( struct sProfile *psProfile, const gchar *pnFile ) {
	struct sUrlHandler *psHandler = NULL;
	gchar *pnUrl = NULL;
	const gchar *pnUser = NULL;
	const gchar *pnPassword = NULL;
	gchar *pnDelCommand = NULL;
	gint nError = -1;
	struct curl_slist *psCommands = NULL ;
	gint nCount = 0;
	gint nIndex = 0;
	struct sVoiceData *psVoiceData = NULL;
	gchar *pnNewData = NULL;
	gint nOffset = 0;
	FILE *psFile = NULL;
	gchar *pnUsrPwd = NULL;
	gint nNr = 0;
	gchar *pnTmp = NULL;

	/* Remove voice file */
	pnUser = voiceBoxGetUser( psProfile );
	pnPassword = voiceBoxGetPassword( psProfile );

	if ( pnUser != NULL && strlen( pnUser ) > 0 ) {
		pnUsrPwd = g_strdup_printf( "%s:%s", pnUser, pnPassword );
	}

	if ( strncmp( pnFile, "uvp", 3 ) == 0 ) {
		pnUrl = g_strdup_printf( "ftp://%s/%s/FRITZ/voicebox/", routerGetHost( psProfile ), pnVoiceBoxPath );
	} else {
		pnUrl = g_strdup_printf( "ftp://%s/%s/FRITZ/voicebox/rec/", routerGetHost( psProfile ), pnVoiceBoxPath );
	}

	Debug( KERN_DEBUG, "Using '%s'\n", pnUrl );
	/* create url handler */
	psHandler = urlHandler( pnUrl, 21 );

	pnDelCommand = g_strdup_printf( "dele /%s/FRITZ/voicebox/rec/%s", pnVoiceBoxPath, pnFile );
	psCommands = curl_slist_append( psCommands , pnDelCommand ) ;
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_QUOTE, psCommands );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_VERBOSE, 1 );

	if ( pnUsrPwd != NULL ) {
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_USERPWD, pnUsrPwd );
	}

	/* try to get data */
	nError = readUrl( psHandler, psProfile );
	g_free( pnUrl );
	curl_slist_free_all( psCommands );
	g_free( pnDelCommand );
	freeHandler( psHandler );

	nNr = pnFile[ 4 ] - '0';
	Debug( KERN_DEBUG, "VoiceBox: %d\n", nNr );

	/* Update meta data */
	if ( asVoiceBox[ nNr ].pnData != NULL && asVoiceBox[ nNr ].nLen != 0 ) {
		nCount = asVoiceBox[ nNr ].nLen / sizeof( struct sVoiceData );
		pnNewData = g_malloc( ( nCount - 1 ) * sizeof( struct sVoiceData ) );

		for ( nIndex = 0; nIndex < nCount; nIndex++ ) {
			psVoiceData = ( struct sVoiceData * ) ( asVoiceBox[ nNr ].pnData + nIndex * sizeof( struct sVoiceData ) );
			if ( strncmp( psVoiceData -> anFile, pnFile, strlen( pnFile ) ) != 0 ) {
				memcpy( pnNewData + nOffset, asVoiceBox[ nNr ].pnData + nIndex * sizeof( struct sVoiceData ), sizeof( struct sVoiceData ) );
				nOffset += sizeof( struct sVoiceData );
			}
		}

		Debug( KERN_DEBUG, "NewData %p, size: %d\n", pnNewData, nOffset );
		pnTmp = g_strdup_printf( "%s%cmeta", g_get_tmp_dir(), G_DIR_SEPARATOR );
		psFile = fopen( pnTmp, "wb+" );

		if ( psFile != NULL ) {
			if ( pnNewData != NULL ) {
				gint nResult = 0;
				nResult = fwrite( pnNewData, 1, nOffset, psFile );
				if ( nResult != nOffset ) {
					Debug( KERN_DEBUG, "Warning: Could not save voice file\n" );
				}
			}
			fclose( psFile );
			nError = 0;
		}
		FILE *psMetaData = fopen( pnTmp, "rb" );
		g_free( pnTmp );

		/* create url handler */
		pnUrl = g_strdup_printf( "ftp://%s/%s/FRITZ/voicebox/meta%d", routerGetHost( psProfile ), pnVoiceBoxPath, nNr );

		psHandler = urlHandler( pnUrl, 21 );
		/* set timeout to 3sec */
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_TIMEOUT, 3 );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_VERBOSE, 1 );

		if ( pnUsrPwd != NULL ) {
			curl_easy_setopt( psHandler -> psHandle, CURLOPT_USERPWD, pnUsrPwd );
		}

		curl_easy_setopt( psHandler -> psHandle, CURLOPT_UPLOAD, 1 );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_INFILE, psMetaData );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_INFILESIZE, nOffset );

		/* try to get data */
		nError = readUrl( psHandler, psProfile );
		g_free( pnUrl );
		freeHandler( psHandler );

		g_free( asVoiceBox[ nNr ].pnData );
		asVoiceBox[ nNr ].pnData = pnNewData;
		asVoiceBox[ nNr ].nLen = nOffset;
	}

	if ( pnUsrPwd != NULL ) {
		g_free( pnUsrPwd );
		pnUsrPwd = NULL;
	}

	return nError;
}

/**
 * \brief Load faxbox content from box and add them to liststore
 * \param psProfile profile structure
 * \return error code (0=success, otherwise error)
 */
static int fritzBoxLoadFaxBox( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler = NULL;
	gchar *pnUrl = NULL;
	gint nError = -1;
	gchar *pnVolume = NULL;
	const gchar *pnUser = NULL;
	const gchar *pnPassword = NULL;
	gchar *pnUsrPwd = NULL;
	gchar *pnCommand = NULL;
	gchar **ppnSplit = NULL;
	gint nCount = 0;

	if ( routerLogin( psProfile ) == -1 ) {
		return -1;
	}

	pnVolume = getVolumeName( psProfile );

	if ( pnVolume != NULL ) {
		if ( pnFaxBoxPath != NULL ) {
			g_free( pnFaxBoxPath );
		}
		pnFaxBoxPath = pnVolume;
	} else {
		pnFaxBoxPath = g_strdup( "" );
	}

	pnUser = voiceBoxGetUser( psProfile );
	pnPassword = voiceBoxGetPassword( psProfile );
	if ( pnUser != NULL && strlen( pnUser ) > 0 ) {
		pnUsrPwd = g_strdup_printf( "%s:%s", pnUser, pnPassword );
	}

again:
	/* create url handler */
	pnUrl = g_strdup_printf( "ftp://%s/%s/FRITZ/faxbox/", routerGetHost( psProfile ), pnFaxBoxPath );
	Debug( KERN_DEBUG, "URL: %s\n", pnUrl );
	psHandler = urlHandler( pnUrl, 21 );

	curl_easy_setopt( psHandler -> psHandle, CURLOPT_DIRLISTONLY, 1 );

	if ( pnUsrPwd != NULL ) {
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_USERPWD, pnUsrPwd );
	}

	/* try to get data */
	nError = readUrl( psHandler, psProfile );
	g_free( pnUrl );
	g_free( pnCommand );
	if ( nError != 0 ) {
		if ( nError == CURLE_REMOTE_ACCESS_DENIED ) {
			if ( pnFaxBoxPath != NULL && strcmp( pnFaxBoxPath, "" ) ) {
				free( pnFaxBoxPath );
				pnFaxBoxPath = g_strdup( "" );
				freeHandler( psHandler );
				goto again;
			}
		}

		Debug( KERN_DEBUG, "2. Error: %s\n", curl_easy_strerror( nError ) );
		freeHandler( psHandler );
		goto error1;
	}


	ppnSplit = g_strsplit( psHandler -> pnData, "\n", -1 );
	freeHandler( psHandler );

	ffgtkLock();
	while ( ppnSplit[ nCount ] != NULL ) {
		gchar anDate[9];
		gchar anTime[6];
		gchar anRemoteNumber[ 32 ];
		gchar *pnStart = NULL;
		gint nPos = -1;
		struct sCaller sCaller;

		strncpy( anDate, ppnSplit[ nCount ], 8 );
		anDate[ 8 ] = '\0';
		strncpy( anTime, ppnSplit[ nCount ] + 9, 5 );
		anTime[ 2 ] = ':';
		anTime[ 5 ] = '\0';
		pnStart = strstr( ppnSplit[ nCount ], "Telefax" );
		if ( pnStart == NULL ) {
			nCount++;
			continue;
		}
		nPos = findString( pnStart, 8, "." );
		strncpy( anRemoteNumber, pnStart + 8, nPos - 8 );
		anRemoteNumber[ nPos - 8 ] = '\0';

		memset( &sCaller, 0, sizeof( struct sCaller ) );
		sCaller.nType = 5;
		sCaller.pnDateTime = g_strdup_printf( "%s %s", anDate, anTime );
		sCaller.pnName = g_strdup( "" );
		sCaller.pnNumber = g_strdup( anRemoteNumber );
		sCaller.pnLocalName = g_strdup( _( "Telefax" ) );
		sCaller.pnLocalNumber = g_strdup( "" );
		sCaller.pnDuration = g_strdup( "" );
		sCaller.pnProfile = g_strdup( getActiveProfile() -> pnName );

		/* Add caller to list */
		AddCall( "5", sCaller.pnDateTime, "", sCaller.pnNumber, sCaller.pnLocalName, sCaller.pnLocalNumber, sCaller.pnDuration, getActiveProfile() -> pnName );

		callMonitorAddCaller( &sCaller, NULL, NULL );

		nCount++;
	}
	ffgtkUnlock();

error1:

	if ( pnUsrPwd != NULL ) {
		g_free( pnUsrPwd );
	}

	routerLogout( psProfile );
	return nError;
}

/**
 * \brief Load fax file from box
 * \param psProfile profile structure
 * \param pnFile voice file we want to load
 * \return error code (0=success, otherwise error)
 */
int fritzBoxLoadFaxFile( struct sProfile *psProfile, const gchar *pnFile ) {
	struct sUrlHandler *psHandler = NULL;
	gchar *pnUrl = NULL;
	const gchar *pnUser = NULL;
	const gchar *pnPassword = NULL;
	gint nError = -1;
	FILE *psFile;
	gchar *pnUsrPwd = NULL;
	gchar *pnStorePath = NULL;
	DIR *psDir = NULL;

	gchar *pnDir = g_strdup_printf( "%s%cffgtk-%s", g_get_tmp_dir(), G_DIR_SEPARATOR, g_get_user_name() );
	Debug( KERN_DEBUG, "pnDir: %s\n", pnDir );

	/* Check if spooler is present, create directory if needed */
	psDir = opendir( pnDir );
	if ( psDir == NULL ) {
		g_mkdir( pnDir, 0777 );
	} else {
		closedir( psDir );
	}

	pnUser = voiceBoxGetUser( psProfile );
	pnPassword = voiceBoxGetPassword( psProfile );
	if ( pnUser != NULL && strlen( pnUser ) > 0 ) {
		pnUsrPwd = g_strdup_printf( "%s:%s", pnUser, pnPassword );
	}

	pnUrl = g_strdup_printf( "ftp://%s/%s/FRITZ/faxbox/%s", routerGetHost( psProfile ), pnFaxBoxPath, pnFile );
	/* create url handler */
	psHandler = urlHandler( pnUrl, 21 );
	if ( pnUsrPwd != NULL ) {
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_USERPWD, pnUsrPwd );
	}

	/* try to get data */
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "Error: %s (Loading '%s')\n", curl_easy_strerror( nError ), pnUrl );
		g_free( pnUrl );
		goto error1;
	}
	g_free( pnUrl );

	Debug( KERN_DEBUG, "Size: %d\n", psHandler -> nSize );
	pnStorePath = g_strdup_printf( "%s%cffgtk-%s%cfax.pdf", g_get_tmp_dir(), G_DIR_SEPARATOR, g_get_user_name(), G_DIR_SEPARATOR );
	if ( pnStorePath != NULL ) {
		psFile = fopen( pnStorePath, "wb+" );
		g_free( pnStorePath );

		if ( psFile != NULL ) {
			gint nResult;
			nResult = fwrite( psHandler -> pnData, 1, psHandler -> nSize, psFile );
			if ( nResult != psHandler -> nSize ) {
				Debug( KERN_DEBUG, "Could not save fax file\n" );
			}
			fclose( psFile );
			nError = 0;
		} else {
			nError = -1;
		}
	}

error1:
	freeHandler( psHandler );

	if ( pnUsrPwd != NULL ) {
		g_free( pnUsrPwd );
	}

	return nError;
}

/**
 * \brief Delete FAX file from box
 * \param psProfile profile structure
 * \param pnFile FAX file we want to delete
 * \return error code (0=success, otherwise error)
 */
int fritzBoxDeleteFaxFile( struct sProfile *psProfile, const gchar *pnFile ) {
	const gchar *pnUser;
	const gchar *pnPassword;
	gchar *pnUrl;
	struct sUrlHandler *psHandler;
	gchar *pnDelCommand;
	struct curl_slist *psCommands = NULL;
	int nError = 0;
	gchar *pnUsrPwd = NULL;

	/* Remove voice file */
	pnUser = voiceBoxGetUser( psProfile );
	pnPassword = voiceBoxGetPassword( psProfile );

	if ( pnUser != NULL && strlen( pnUser ) > 0 ) {
		pnUsrPwd = g_strdup_printf( "%s:%s", pnUser, pnPassword );
	}

	pnUrl = g_strdup_printf( "ftp://%s/%s/FRITZ/faxbox/%s", routerGetHost( psProfile ), pnFaxBoxPath, pnFile );

	Debug( KERN_DEBUG, "Using '%s'\n", pnUrl );

	/* create url handler */
	psHandler = urlHandler( pnUrl, 21 );

	pnDelCommand = g_strdup_printf( "dele /%s/FRITZ/faxbox/%s", pnFaxBoxPath, pnFile );
	psCommands = curl_slist_append( psCommands , pnDelCommand ) ;
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_QUOTE, psCommands );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_VERBOSE, 1 );

	if ( pnUsrPwd != NULL ) {
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_USERPWD, pnUsrPwd );
	}

	/* try to get data */
	nError = readUrl( psHandler, psProfile );
	g_free( pnUrl );
	curl_slist_free_all( psCommands );
	g_free( pnDelCommand );
	freeHandler( psHandler );

	return nError;
}

/** FritzBox router structure */
struct sRouter sFritzBox = {
	"AVM FRITZ!Box",
	fritzBoxDetect,
	fritzBoxInit,
	fritzBoxLogin,
	fritzBoxLogout,
	fritzBoxDetectFirmware,
	fritzBoxGetPhoneSettings,
	fritzBoxGetActivePorts,
	fritzBoxGetCallList,
	fritzBoxClearCallList,
	fritzBoxCallNumber,
	fritzBoxHangup,
	fritzBoxReconnect,
	fritzBoxGetIp,
	fritzBoxGetSpeed,
	fritzBoxReadDiversity,
	fritzBoxSetDiversity,
	fritzBoxGetName,
	fritzBoxGetVersion,
	fritzBoxLoadVoiceBox,
	fritzBoxLoadVoiceFile,
	fritzBoxDeleteVoiceFile,
	fritzBoxLoadFaxBox,
	fritzBoxLoadFaxFile,
	fritzBoxDeleteFaxFile,
};
