/*******************************************************************************
 * Copyright (c) 2011-2012 Ericsson, Ecole Polytechnique de Montreal and others
 *
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Matthew Khouzam - Initial API and implementation
 * Contributors: Simon Marchi - Initial API and implementation
 *******************************************************************************/

package org.eclipse.linuxtools.ctf.core.event;

import java.util.HashMap;
import java.util.List;

import org.eclipse.linuxtools.ctf.core.event.types.Definition;
import org.eclipse.linuxtools.ctf.core.event.types.IDefinitionScope;
import org.eclipse.linuxtools.ctf.core.event.types.StructDefinition;
import org.eclipse.linuxtools.ctf.core.trace.StreamInputReader;

/**
 * Representation of a particular instance of an event.
 */
public class EventDefinition implements IDefinitionScope {

    // ------------------------------------------------------------------------
    // Attributes
    // ------------------------------------------------------------------------

    /**
     * The corresponding event declaration.
     */
    private final EventDeclaration declaration;

    /**
     * The timestamp of the current event.
     */
    private long timestamp;

    /**
     * The event context structure definition.
     */
    private StructDefinition context;

    /**
     * The event fields structure definition.
     */
    private StructDefinition fields;

    /**
     * The StreamInputReader that reads this event definition.
     */
    private final StreamInputReader streamInputReader;

    // ------------------------------------------------------------------------
    // Constructors
    // ------------------------------------------------------------------------

    /**
     * Constructs an event definition.
     *
     * @param declaration
     *            The corresponding event declaration
     * @param streamInputReader
     *            The SIR from where this EventDef was read
     */
    public EventDefinition(EventDeclaration declaration,
            StreamInputReader streamInputReader) {
        this.declaration = declaration;
        this.streamInputReader = streamInputReader;
    }

    // ------------------------------------------------------------------------
    // Getters/Setters/Predicates
    // ------------------------------------------------------------------------

    @Override
    public String getPath() {
        return "event"; //$NON-NLS-1$
    }

    /**
     * Gets the declaration (the form) of the data
     *
     * @return the event declaration
     */
    public EventDeclaration getDeclaration() {
        return declaration;
    }

    /**
     * Gets the fields of a definition
     *
     * @return the fields of a definition in struct form. Can be null.
     */
    public StructDefinition getFields() {
        return fields;
    }

    /**
     * Gets the context of this event
     *
     * @return the context in struct form
     */
    public StructDefinition getContext() {
        return context;
    }

    /**
     * Gets the stream input reader that this event was made by
     *
     * @return the parent
     */
    public StreamInputReader getStreamInputReader() {
        return streamInputReader;
    }

    /**
     * Gets the context of packet the event is in.
     *
     * @return the packet context
     */
    public StructDefinition getPacketContext() {
        return streamInputReader.getCurrentPacketContext();
    }

    /**
     * gets the CPU the event was generated by. Slightly LTTng specific
     *
     * @return The CPU the event was generated by
     */
    public int getCPU() {
        return streamInputReader.getCPU();
    }

    /**
     * @return the timestamp
     */
    public long getTimestamp() {
        return timestamp;
    }

    /**
     * @param timestamp
     *            the timestamp to set
     */
    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    /**
     * @param context
     *            the context to set
     */
    public void setContext(StructDefinition context) {
        this.context = context;
    }

    /**
     * @param fields
     *            the fields to set
     */
    public void setFields(StructDefinition fields) {
        this.fields = fields;
    }

    // ------------------------------------------------------------------------
    // Operations
    // ------------------------------------------------------------------------

    @Override
    public Definition lookupDefinition(String lookupPath) {
        if (lookupPath.equals("context")) { //$NON-NLS-1$
            return context;
        } else if (lookupPath.equals("fields")) { //$NON-NLS-1$
            return fields;
        } else {
            return null;
        }
    }

    @Override
    public String toString() {
        HashMap<String, Definition> definitions;
        List<String> list;
        StringBuilder retString = new StringBuilder();
        final String cr = System.getProperty("line.separator");//$NON-NLS-1$

        retString.append("Event type: " + declaration.getName() + cr); //$NON-NLS-1$
        retString.append("Timestamp: " + Long.toString(timestamp) + cr); //$NON-NLS-1$

        if (context != null) {
            definitions = context.getDefinitions();
            list = context.getDeclaration().getFieldsList();

            for (String field : list) {
                retString.append(field
                        + " : " + definitions.get(field).toString() + cr); //$NON-NLS-1$
            }
        }

        if (fields != null) {
            definitions = fields.getDefinitions();
            list = fields.getDeclaration().getFieldsList();

            for (String field : list) {
                retString.append(field
                        + " : " + definitions.get(field).toString() + cr); //$NON-NLS-1$
            }
        }

        return retString.toString();
    }

}
