/***************************************************************************
                          dlgviewbookmarks.cpp  -  description
                             -------------------
    begin                : Sat Nov 16 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

#include "dlgviewbookmarks.h"
#include "browserlist.h"

#include <QMessageBox>
#include <QClipboard>

/***************************************************************************
 *                                                                         *
 *   The BkListViewItem class (derived from QTreeWidgetItem) represents an *
 *   item in the bookmark tree.  All it really does is hold a pointer to   *
 *   the BkNode representing the folder or bookmark.                       *
 *                                                                         *
 ***************************************************************************/

class BkListViewItem : public QTreeWidgetItem
{
public:
	BkListViewItem(QTreeWidget *list) : QTreeWidgetItem(list)
		{ m_node = NULL; }
	BkListViewItem(QTreeWidgetItem *item) : QTreeWidgetItem(item)
		{ m_node = NULL; }

	BkNode	*m_node;
};

/***************************************************************************
 *                                                                         *
 *   DlgViewBookmarks::DlgViewBookmarks()                                  *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &                                                  *
 *      BkFolder *root                                                     *
 *      QWidget *parent                                                    *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      Class constructor.  The first parameter is the root of the         *
 *      bookmark tree to display.                                          *
 *                                                                         *
 ***************************************************************************/

DlgViewBookmarks::DlgViewBookmarks(const BridgeCfg &cfg,
								   BkFolder *root,
								   QWidget *parent)
	: QDialog(parent), m_cfg(cfg)
{
	setupUi(this);

	listBk->setHeaderLabels(QStringList("Bookmarks"));

	fillList(root);
}

/***************************************************************************
 *                                                                         *
 *   DlgViewBookmarks::~DlgViewBookmarks()                                 *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      Class destructor                                                   *
 *                                                                         *
 ***************************************************************************/

DlgViewBookmarks::~DlgViewBookmarks()
{
}

/***************************************************************************
 *                                                                         *
 *   DlgViewBookmarks::on_buttonCopy_clicked()                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the copy button.                                *
 *                                                                         *
 ***************************************************************************/

void DlgViewBookmarks::on_buttonCopy_clicked(void)
{
	BkListViewItem *bk;

	QList<QTreeWidgetItem *> items = listBk->selectedItems();
	if(items.count() != 1)
		return;
	else
		bk = static_cast<BkListViewItem *>(items[0]);

	if(bk->m_node->type() != NODETYPE_BOOKMARK)
		return;

	BkBookmark *bmark = (BkBookmark *)bk->m_node;

	QClipboard *cp = QApplication::clipboard();
	cp->setText(bmark->url());
}

/***************************************************************************
 *                                                                         *
 *   DlgViewBookmarks::on_checkIgnore_clicked()                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the ignore checkbox.                            *
 *                                                                         *
 ***************************************************************************/

void DlgViewBookmarks::on_checkIgnore_clicked(void)
{
	bool checked = checkIgnore->isChecked();

	// make sure a bookmark or folder is actually selected

	BkListViewItem *bk;

	QList<QTreeWidgetItem *> items = listBk->selectedItems();

	if(items.count() != 1)
		return;
	else
		bk = static_cast<BkListViewItem *>(items[0]);
	
	// save the user's change
	
	bk->m_node->setIgnore(checked);

	if(bk->m_node->type() == NODETYPE_FOLDER)
	{
		SetFolderIgnore(static_cast<BkFolder &>(*bk->m_node), checked);

		QString msg;

		msg.sprintf(
			"You have asked BookmarkBridge to %signore this folder.\n"
			"All bookmarks and folders contained within this folder\n"
			"are also %signored.  You can optionally override this\n"
			"setting by changing the ignore checkbox for any of the\n"
			"folders or bookmarks contained below this folder.",
			checked ? "" : "no longer ",
			checked ? "" : "no longer ");
		QMessageBox::information(this, "Notice", msg);
	}

	// if the user asked to no longer ignore a bookmark or folder,
	// turn off each of its ancestors' ignore bits.

	if(checked == false)
	{
		BkListViewItem *bkParent;
		BkListViewItem *current = bk;
		
		for(;;)
		{
			bkParent = (BkListViewItem *)current->parent();
			if(bkParent == NULL)
				break;
			bkParent->m_node->setIgnore(false);
			current = bkParent;
		}
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgViewBookmarks::on_listBk_itemSelectionChanged()                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks anywhere within the bookmark tree.                 *
 *                                                                         *
 ***************************************************************************/

void DlgViewBookmarks::on_listBk_itemSelectionChanged(void)
{
	BkListViewItem	*bk;

	// clear the dialog widgets displaying information about the current selection
	
	labelTitle->setText("");
	labelUrl->setText("");
	labelPath->setText("");
	listBrowsers->clear();
	checkIgnore->setChecked(false);

	QList<QTreeWidgetItem *> items = listBk->selectedItems();
	if(items.count() != 1)
		bk = NULL;
	else
		bk = static_cast<BkListViewItem *>(items[0]);

	if(bk == NULL)
	{
		// the user didn't click on an item
		buttonCopy->setEnabled(false);
		checkIgnore->setEnabled(false);
		return;
	}
	else
	{
		buttonCopy->setEnabled(true);
		checkIgnore->setEnabled(true);
	}

	if(bk->m_node == NULL)
	{
		// this item doesn't have an associated node
		return;
	}

	// put information about the current selection into
	// the dialog's widgets

	displayText(labelTitle, bk->m_node->title());
	displayText(labelPath, bk->m_node->path());
	checkIgnore->setChecked(bk->m_node->ignore());

	if(bk->m_node->type() == NODETYPE_BOOKMARK)
	{
		BkBookmark *bmark = (BkBookmark *)bk->m_node;
		displayText(labelUrl, bmark->url());
	}
	else
		buttonCopy->setEnabled(false);

	BrowserOrdLst ordlst = bk->m_node->browserFound();
	BrowserOrdLst::iterator it;
	std::vector<BrowserCfg>::const_iterator bit;
	int i = 0;
	for(it=ordlst.begin(); it!=ordlst.end(); ++it)
	{
		for(bit=m_cfg.browsers.begin(); bit!=m_cfg.browsers.end(); ++bit)
		{
			if(bit->ordinal() == *it)
				break;
		}

		if(bit != m_cfg.browsers.end())
			listBrowsers->insertItem(i++, bit->Descrip());
	}
}

void DlgViewBookmarks::displayText(QLabel *label, const QString &text)
{
	QString	str(text);
	int		width;
	bool	reduced = false;

	QFontMetrics fm = label->fontMetrics();
	QSize size = label->size();

	for(;;)
	{
		width = fm.width(str);
		if(width >= size.width())
		{
			str = str.left(str.length()-1);
			reduced = true;
		}
		else
			break;
	}

	if(reduced)
		str = str.left(str.length()-3) + "...";
	label->setText(str);
}

/***************************************************************************
 *                                                                         *
 *   DlgViewBookmarks::fillList()                                          *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkFolder *root                                                     *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function is called to populate the bookmark tree.             *
 *                                                                         *
 ***************************************************************************/

void DlgViewBookmarks::fillList(BkFolder *root)
{
	BookmarkLst::iterator bit;
	FolderLst::iterator fit;
	BkListViewItem	*item;
	QString		title;

	// empty the list
	
	listBk->clear();

	// fill in the bookmarks in the root folder
	
	for(bit = root->BookmarksBegin();
		bit != root->BookmarksEnd();
		++bit)
	{
		item = new BkListViewItem(listBk);
		item->m_node = &(*bit);
		title = bit->title();
		if(title.length() < 1)
			title = "No Title Shown";
		item->setText(0, title);
	}

	// fill in the folders in the root folder
	
	for(fit = root->FoldersBegin();
		fit != root->FoldersEnd();
		++fit)
	{
		item = new BkListViewItem(listBk);
		item->m_node = &(*fit);
		title = fit->title();
		if(title.length() < 1)
			title = "No Title Shown";
		item->setText(0, title);
		fillFolder(*fit, item);
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgViewBookmarks::fillFolder()                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkFolder &parent                                                   *
 *      QListViewItem *parentItem                                          *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function is called to populate a sub-folder within the        *
 *      bookmark tree.                                                     *
 *                                                                         *
 ***************************************************************************/

void DlgViewBookmarks::fillFolder(BkFolder &parent, QTreeWidgetItem *parentItem)
{
	BookmarkLst::iterator bit;
	FolderLst::iterator fit;
	BkListViewItem	*item;

	// fill in the bookmarks in this folder
	
	for(bit = parent.BookmarksBegin();
		bit != parent.BookmarksEnd();
		++bit)
	{
		item = new BkListViewItem(parentItem);
		item->m_node = static_cast<BkNode *>(&(*bit));
		item->setText(0, bit->title());
		QString	href;
		href = bit->url();
		if(href.length() < 1)
			href = "No Href Shown";
		item->setText(1, href);
	}

	// fill in the sub-folders in this folder
	
	for(fit = parent.FoldersBegin();
		fit != parent.FoldersEnd();
		++fit)
	{
		item = new BkListViewItem(parentItem);
		item->m_node = static_cast<BkNode *>(&(*fit));
		item->setText(0, fit->title());
		fillFolder(*fit, item);
	}
}

void DlgViewBookmarks::resizeEvent(QResizeEvent *)
{
	on_listBk_itemSelectionChanged();
}

/***************************************************************************
 *                                                                         *
 *   DlgViewBookmarks::SetFolderIgnore()                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkFolder &folder                                                   *
 *      bool checked                                                       *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function is called when the user clicks on the ignore         *
 *      checkbox for a folder.  It recursively sets the ignore attribute   *
 *      on all children (both folders and bookmarks).                      *
 *                                                                         *
 ***************************************************************************/

void DlgViewBookmarks::SetFolderIgnore(BkFolder &folder, bool checked)
{
	BookmarkLst::iterator bit;
	FolderLst::iterator fit;

	for(bit = folder.BookmarksBegin();
		bit != folder.BookmarksEnd();
		++bit)
	{
		bit->setIgnore(checked);
	}

	for(fit = folder.FoldersBegin();
		fit != folder.FoldersEnd();
		++fit)
	{
		fit->setIgnore(checked);
		SetFolderIgnore(*fit, checked);
	}
}
