#!/usr/bin/env python

# Copyright (C) Andrew Mitchell 2006

#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.

#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
############################################################################

## Configure /etc/pam.d/common-* ##


# Format of files in /etc/pam.d:

# auth    required        pam_unix.so nullok_secure
# |       |               |           |  
# |       |               |           options
# |       |               shared library in /lib/security
# |       security 'level' - required,optional,requisite,sufficient
# class of pam module (auth,account,password,session)

# Ordering of the module lines in PAM configuration matters very much, as
# modules are tested & loaded in order, like iptables. The required/optional
# portion controls what happens if a module fails or succeeds

import sys, string, preserve

class PAMConfig:

    cfgs = ['account','auth','password','session']
    # raw lines to write out
    pam={}
    # list format
    pam_config={}
    
    def __init__(self):
        fh={}
        for cfg in self.cfgs:
            fh[cfg] = open("/etc/pam.d/common-%s" % cfg)
            self.pam[cfg] = fh[cfg].readlines()
            self.pam_config[cfg]=[]
            for line in self.pam[cfg]:
                # test for comment
                if not (line.strip() == "" or line.strip()[0] == "#"):
                    # 3 or more components to a non-comment line
                    parts = line.split()
                    self.pam_config[cfg].append(parts)
                self.pam[cfg][self.pam[cfg].index(line)] = self.pam[cfg][self.pam[cfg].index(line)].strip()

                
            fh[cfg].close()

    def dump(self):
        for x in self.cfgs:
            print x
            print self.pam_config[x]
            print self.pam[x]

    def find(self, cfg, module, verbose=False):
        # Search for module in config - return true if found, false otherwise
        #print self.pam_config[cfg]
        for mod in self.pam_config[cfg]:
            if mod[2] == module:
                if verbose:
                    print "Found %s in %s" % (module, cfg)
                return True

    def insert(self, cfg, level, module, order, options=""):
        #insert pam module in the correct /etc/pam.d/common-cfg file
        
        #insert into list structure
        self.pam_config[cfg].insert( order, [cfg, level, module ])

        #insert into files
        #find which module to stick it after
        if order > 0: # not first
            prev = self.pam_config[cfg][order-1][2]
            #print "Inserting after %s" % prev
            for line in self.pam[cfg]:
                if line.find(prev) != -1:
                    self.pam[cfg].insert(self.pam[cfg].index(line)+1, "%s\t%s\t%s\t%s" % (cfg, level, module, options))
                    #insert only once - doesn't work very well
                    return

    def insert_after(self, cfg, level, module, previous, options=""):
        #find the order in the current file
        for line in self.pam_config[cfg]:
            #print line
            if line[2] == previous:
                order = self.pam_config[cfg].index(line)+1
                self.insert(cfg, level, module, order, options)
                #insert only once
                return

    def disable(self, cfg, module):
        # comment out module
        for line in self.pam[cfg]:
            if line.find(module) != -1 and line.strip()[0] != '#':
                self.pam[cfg][self.pam[cfg].index(line)] = '# ' + line

    def change_level(self, cfg, level, module ):
        # change list
        for item in self.pam_config[cfg]:
            if item[2] == module:
                item[1] = level

        # change lines
        for line in self.pam[cfg]:
            if line.find(module) != -1 and line[0] != '#':
                comps = line.split()
                #print "Found %s in %s" % (module,line), comps
                comps[1] = level
                #print comps.join()
                self.pam[cfg][self.pam[cfg].index(line)] = string.join(comps, '\t')
                
        
        
    def save(self):
        for cfg in self.cfgs:
            name = "/etc/pam.d/common-%s" % cfg
            preserve.preserve(name)
            try:
                fh = open(name, "w")
                close = True
                for line in self.pam[cfg]:
                    line += '\n'
                    fh.write(line)

            except IOError:
                fh = sys.stderr
                close = False
                # Don't spit to stderr now
                #for line in self.pam[cfg]:
                #    line += '\n'
                #    fh.write(line)
            if close:
                fh.close()


if __name__ == '__main__':
    pam = PAMConfig()
    pam.dump()
    pam.insert('auth', 'required', 'pam_krb5.so', 1)
    print "Post-insert #1"
    pam.dump()
    pam.insert_after('password', 'sufficient', 'pam_krb5.so', 'pam_unix.so')
    print "Post-insert #2"
    pam.dump()
    pam.insert_after('session', 'sufficient', 'pam_krb5.so', 'pam_unix.so')
    print "Post-insert #3"
    pam.dump()
    print "Changing level"
    pam.change_level('session', 'required', 'pam_krb5.so')
    pam.dump()
    print "Disabling"
    pam.disable('session', 'pam_krb5.so')
    pam.dump()

    pam.save()

