; ACL2 Version 4.1 -- A Computational Logic for Applicative Common Lisp
; Copyright (C) 2010  University of Texas at Austin

; This version of ACL2 is a descendent of ACL2 Version 1.9, Copyright
; (C) 1997 Computational Logic, Inc.  See the documentation topic NOTE-2-0.

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

; Written by:  Matt Kaufmann               and J Strother Moore
; email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
; Department of Computer Sciences
; University of Texas at Austin
; Austin, TX 78712-1188 U.S.A.

; This file, axioms.lisp, serves two purposes.  First, it describes
; the theory of ACL2 by enumerating the axioms and definitions.
; Second, it implements in Common Lisp those functions of the theory
; which are not already provided in Common Lisp.  In some cases, the
; implementation of a function is identical to its axiomatization (cf.
; implies).  In other cases, we provide functions whose semantics are
; applicative but whose implementations are decidely ``von
; Neumann-esque''.  For example, we implement the array, property
; list, and io primitives with non-applicative techniques.

; This file is read by Common Lisp in two ways.  First, we bring ACL2
; into its initial state with the function boot-strap, which loads
; this file.  Second, this file is read and compiled in the
; implementation of ACL2 itself.  To support these two readings, we
; use the #+ and #- read macro feature of Common Lisp.  While we are
; loading this file in boot-strap, we arrange for *features* to
; contain the symbol :acl2-loop-only; otherwise, *features* does not
; contain :acl2-loop-only.  Thus, during boot-strap, forms immediately
; preceded by #+acl2-loop-only are ``seen'', whereas those
; immediately preceded by #-acl2-loop-only are invisible.  The
; converse is true when we are compiling and loading the code for
; ACL2.

; If a symbol described in CLTL is axiomatized here, then we give it
; exactly the same semantics as it has in CLTL, under restrictions for
; which we check.  (Actually, this is currently a lie about DEFUN,
; DEFMACRO, and PROGN, but we will provide someday a check that that
; those are only used in files in ways such that their ACL2 and Common
; Lisp meanings are prefectly consistent.)  Thus, when we talk about
; +, we really mean the Common Lisp +.  However, our + does not handle
; floating point numbers, so there is a guard on + that checks that
; its args are rationals.  The symbols in the list
; acl2::*common-lisp-symbols-from-main-lisp-package* are the symbols
; that we take as having a meaning in Common Lisp.  If a user wishes
; access to these in a package, then he can use the permanent value of
; the global *common-lisp-symbols-from-main-lisp-package* as an import
; list for defpkg.

; If we use a symbol that has a $ suffix, it is a symbol we have
; defined with a meaning that it is similar to the Common Lisp symbol
; without the $ suffix, but different in some way, e.g. princ$ takes a
; state arg and returns a state.

(in-package "ACL2")

; Leave the following as the second form in axioms.lisp.  It is read
; by acl2.lisp.  Leave the acl2:: prefix there, too.

; We are aware that as of this writing, various Lisp implementations deviate
; from the dpANS specification of the external symbols of the main Lisp
; package.  However, we will guarantee that variable names and logical names
; that lie in the main Lisp package will all come from this list, and in the
; case of variables, we will guarantee that they are not special variables.
; Note that however we handle this constant, it is crucial that its value be
; independent of the implementation, lest we can prove something about its
; length (say) in one Lisp that is false in another.  Our requirement on this
; list is that it allow the compiler to deal correctly with Common Lisp functions
; such as CAR that we are bringing into the ACL2 environment, and the dpANS list
; certainly satisfies that requirement.

(acl2::defconst acl2::*common-lisp-symbols-from-main-lisp-package*

; From the info page for dpANS, node "Symbols in the COMMON-LISP Package."
; The comments are from that page as well, though we have inserted "; "
; in front of each.

 '(

; The figures on the next twelve pages contain a complete enumeration of the
; 978 external symbols in the COMMON-LISP package.

  &allow-other-keys            *print-miser-width*          
  &aux                         *print-pprint-dispatch*      
  &body                        *print-pretty*               
  &environment                 *print-radix*                
  &key                         *print-readably*             
  &optional                    *print-right-margin*         
  &rest                        *query-io*                   
  &whole                       *random-state*               
  *                            *read-base*                  
  **                           *read-default-float-format*  
  ***                          *read-eval*                  
  *break-on-signals*           *read-suppress*              
  *compile-file-pathname*      *readtable*                  
  *compile-file-truename*      *standard-input*             
  *compile-print*              *standard-output*            
  *compile-verbose*            *terminal-io*                
  *debug-io*                   *trace-output*               
  *debugger-hook*              +                            
  *default-pathname-defaults*  ++                           
  *error-output*               +++                          
  *features*                   -                            
  *gensym-counter*             /                            
  *load-pathname*              //                           
  *load-print*                 ///                          
  *load-truename*              /=                           
  *load-verbose*               1+                           
  *macroexpand-hook*           1-                           
  *modules*                    <                            
  *package*                    <=                           
  *print-array*                =                            
  *print-base*                 >                            
  *print-case*                 >=                           
  *print-circle*               abort                        
  *print-escape*               abs                          
  *print-gensym*               acons                        
  *print-length*               acos                         
  *print-level*                acosh                        
  *print-lines*                add-method                   

;   Figure 1-4: Symbols in the COMMON-LISP package (part one of twelve).


  adjoin                      atom          boundp                    
  adjust-array                base-char     break                     
  adjustable-array-p          base-string   broadcast-stream          
  allocate-instance           bignum        broadcast-stream-streams  
  alpha-char-p                bit           built-in-class            
  alphanumericp               bit-and       butlast                   
  and                         bit-andc1     byte                      
  append                      bit-andc2     byte-position             
  apply                       bit-eqv       byte-size                 
  apropos                     bit-ior       caaaar                    
  apropos-list                bit-nand      caaadr                    
  aref                        bit-nor       caaar                     
  arithmetic-error            bit-not       caadar                    
  arithmetic-error-operands   bit-orc1      caaddr                    
  arithmetic-error-operation  bit-orc2      caadr                     
  array                       bit-vector    caar                      
  array-dimension             bit-vector-p  cadaar                    
  array-dimension-limit       bit-xor       cadadr                    
  array-dimensions            block         cadar                     
  array-displacement          boole         caddar                    
  array-element-type          boole-1       cadddr                    
  array-has-fill-pointer-p    boole-2       caddr                     
  array-in-bounds-p           boole-and     cadr                      
  array-rank                  boole-andc1   call-arguments-limit      
  array-rank-limit            boole-andc2   call-method               
  array-row-major-index       boole-c1      call-next-method          
  array-total-size            boole-c2      car                       
  array-total-size-limit      boole-clr     case                      
  arrayp                      boole-eqv     catch                     
  ash                         boole-ior     ccase                     
  asin                        boole-nand    cdaaar                    
  asinh                       boole-nor     cdaadr                    
  assert                      boole-orc1    cdaar                     
  assoc                       boole-orc2    cdadar                    
  assoc-if                    boole-set     cdaddr                    
  assoc-if-not                boole-xor     cdadr                     
  atan                        boolean       cdar                      
  atanh                       both-case-p   cddaar                    

;   Figure 1-5: Symbols in the COMMON-LISP package (part two of twelve).


  cddadr             clear-input                  copy-tree                  
  cddar              clear-output                 cos                        
  cdddar             close                        cosh                       
  cddddr             clrhash                      count                      
  cdddr              code-char                    count-if                   
  cddr               coerce                       count-if-not               
  cdr                compilation-speed            ctypecase                  
  ceiling            compile                      debug                      
  cell-error         compile-file                 decf                       
  cell-error-name    compile-file-pathname        declaim                    
  cerror             compiled-function            declaration                
  change-class       compiled-function-p          declare                    
  char               compiler-macro               decode-float               
  char-code          compiler-macro-function      decode-universal-time      
  char-code-limit    complement                   defclass                   
  char-downcase      complex                      defconstant                
  char-equal         complexp                     defgeneric                 
  char-greaterp      compute-applicable-methods   define-compiler-macro      
  char-int           compute-restarts             define-condition           
  char-lessp         concatenate                  define-method-combination  
  char-name          concatenated-stream          define-modify-macro        
  char-not-equal     concatenated-stream-streams  define-setf-expander       
  char-not-greaterp  cond                         define-symbol-macro        
  char-not-lessp     condition                    defmacro                   
  char-upcase        conjugate                    defmethod                  
  char/=             cons                         defpackage                 
  char<              consp                        defparameter               
  char<=             constantly                   defsetf                    
  char=              constantp                    defstruct                  
  char>              continue                     deftype                    
  char>=             control-error                defun                      
  character          copy-alist                   defvar                     
  characterp         copy-list                    delete                     
  check-type         copy-pprint-dispatch         delete-duplicates          
  cis                copy-readtable               delete-file                
  class              copy-seq                     delete-if                  
  class-name         copy-structure               delete-if-not              
  class-of           copy-symbol                  delete-package             

;     Figure 1-6: Symbols in the COMMON-LISP package (part three of twelve).  


  denominator                    eq                   
  deposit-field                  eql                  
  describe                       equal                
  describe-object                equalp               
  destructuring-bind             error                
  digit-char                     etypecase            
  digit-char-p                   eval                 
  directory                      eval-when            
  directory-namestring           evenp                
  disassemble                    every                
  division-by-zero               exp                  
  do                             export               
  do*                            expt                 
  do-all-symbols                 extended-char        
  do-external-symbols            fboundp              
  do-symbols                     fceiling             
  documentation                  fdefinition          
  dolist                         ffloor               
  dotimes                        fifth                
  double-float                   file-author          
  double-float-epsilon           file-error           
  double-float-negative-epsilon  file-error-pathname  
  dpb                            file-length          
  dribble                        file-namestring      
  dynamic-extent                 file-position        
  ecase                          file-stream          
  echo-stream                    file-string-length   
  echo-stream-input-stream       file-write-date      
  echo-stream-output-stream      fill                 
  ed                             fill-pointer         
  eighth                         find                 
  elt                            find-all-symbols     
  encode-universal-time          find-class           
  end-of-file                    find-if              
  endp                           find-if-not          
  enough-namestring              find-method          
  ensure-directories-exist       find-package         
  ensure-generic-function        find-restart         

;   Figure 1-7: Symbols in the COMMON-LISP package (part four of twelve).


  find-symbol                       get-internal-run-time        
  finish-output                     get-macro-character          
  first                             get-output-stream-string     
  fixnum                            get-properties               
  flet                              get-setf-expansion           
  float                             get-universal-time           
  float-digits                      getf                         
  float-precision                   gethash                      
  float-radix                       go                           
  float-sign                        graphic-char-p               
  floating-point-inexact            handler-bind                 
  floating-point-invalid-operation  handler-case                 
  floating-point-overflow           hash-table                   
  floating-point-underflow          hash-table-count             
  floatp                            hash-table-p                 
  floor                             hash-table-rehash-size       
  fmakunbound                       hash-table-rehash-threshold  
  force-output                      hash-table-size              
  format                            hash-table-test              
  formatter                         host-namestring              
  fourth                            identity                     
  fresh-line                        if                           
  fround                            ignorable                    
  ftruncate                         ignore                       
  ftype                             ignore-errors                
  funcall                           imagpart                     
  function                          import                       
  function-keywords                 in-package                   
  function-lambda-expression        incf                         
  functionp                         initialize-instance          
  gcd                               inline                       
  generic-function                  input-stream-p               
  gensym                            inspect                      
  gentemp                           integer                      
  get                               integer-decode-float         
  get-decoded-time                  integer-length               
  get-dispatch-macro-character      integerp                     
  get-internal-real-time            interactive-stream-p         

;   Figure 1-8: Symbols in the COMMON-LISP package (part five of twelve).


  intern                                  lisp-implementation-type            
  internal-time-units-per-second          lisp-implementation-version         
  intersection                            list                                
  invalid-method-error                    list*                               
  invoke-debugger                         list-all-packages                   
  invoke-restart                          list-length                         
  invoke-restart-interactively            listen                              
  isqrt                                   listp                               
  keyword                                 load                                
  keywordp                                load-logical-pathname-translations  
  labels                                  load-time-value                     
  lambda                                  locally                             
  lambda-list-keywords                    log                                 
  lambda-parameters-limit                 logand                              
  last                                    logandc1                            
  lcm                                     logandc2                            
  ldb                                     logbitp                             
  ldb-test                                logcount                            
  ldiff                                   logeqv                              
  least-negative-double-float             logical-pathname                    
  least-negative-long-float               logical-pathname-translations       
  least-negative-normalized-double-float  logior                              
  least-negative-normalized-long-float    lognand                             
  least-negative-normalized-short-float   lognor                              
  least-negative-normalized-single-float  lognot                              
  least-negative-short-float              logorc1                             
  least-negative-single-float             logorc2                             
  least-positive-double-float             logtest                             
  least-positive-long-float               logxor                              
  least-positive-normalized-double-float  long-float                          
  least-positive-normalized-long-float    long-float-epsilon                  
  least-positive-normalized-short-float   long-float-negative-epsilon         
  least-positive-normalized-single-float  long-site-name                      
  least-positive-short-float              loop                                
  least-positive-single-float             loop-finish                         
  length                                  lower-case-p                        
  let                                     machine-instance                    
  let*                                    machine-type                        

;      Figure 1-9: Symbols in the COMMON-LISP package (part six of twelve).    


  machine-version                mask-field                  
  macro-function                 max                         
  macroexpand                    member                      
  macroexpand-1                  member-if                   
  macrolet                       member-if-not               
  make-array                     merge                       
  make-broadcast-stream          merge-pathnames             
  make-concatenated-stream       method                      
  make-condition                 method-combination          
  make-dispatch-macro-character  method-combination-error    
  make-echo-stream               method-qualifiers           
  make-hash-table                min                         
  make-instance                  minusp                      
  make-instances-obsolete        mismatch                    
  make-list                      mod                         
  make-load-form                 most-negative-double-float  
  make-load-form-saving-slots    most-negative-fixnum        
  make-method                    most-negative-long-float    
  make-package                   most-negative-short-float   
  make-pathname                  most-negative-single-float  
  make-random-state              most-positive-double-float  
  make-sequence                  most-positive-fixnum        
  make-string                    most-positive-long-float    
  make-string-input-stream       most-positive-short-float   
  make-string-output-stream      most-positive-single-float  
  make-symbol                    muffle-warning              
  make-synonym-stream            multiple-value-bind         
  make-two-way-stream            multiple-value-call         
  makunbound                     multiple-value-list         
  map                            multiple-value-prog1        
  map-into                       multiple-value-setq         
  mapc                           multiple-values-limit       
  mapcan                         name-char                   
  mapcar                         namestring                  
  mapcon                         nbutlast                    
  maphash                        nconc                       
  mapl                           next-method-p               
  maplist                        nil                         

;   Figure 1-10: Symbols in the COMMON-LISP package (part seven of twelve).


  nintersection         package-error                  
  ninth                 package-error-package          
  no-applicable-method  package-name                   
  no-next-method        package-nicknames              
  not                   package-shadowing-symbols      
  notany                package-use-list               
  notevery              package-used-by-list           
  notinline             packagep                       
  nreconc               pairlis                        
  nreverse              parse-error                    
  nset-difference       parse-integer                  
  nset-exclusive-or     parse-namestring               
  nstring-capitalize    pathname                       
  nstring-downcase      pathname-device                
  nstring-upcase        pathname-directory             
  nsublis               pathname-host                  
  nsubst                pathname-match-p               
  nsubst-if             pathname-name                  
  nsubst-if-not         pathname-type                  
  nsubstitute           pathname-version               
  nsubstitute-if        pathnamep                      
  nsubstitute-if-not    peek-char                      
  nth                   phase                          
  nth-value             pi                             
  nthcdr                plusp                          
  null                  pop                            
  number                position                       
  numberp               position-if                    
  numerator             position-if-not                
  nunion                pprint                         
  oddp                  pprint-dispatch                
  open                  pprint-exit-if-list-exhausted  
  open-stream-p         pprint-fill                    
  optimize              pprint-indent                  
  or                    pprint-linear                  
  otherwise             pprint-logical-block           
  output-stream-p       pprint-newline                 
  package               pprint-pop                     

;   Figure 1-11: Symbols in the COMMON-LISP package (part eight of twelve).


  pprint-tab                 read-char                   
  pprint-tabular             read-char-no-hang           
  prin1                      read-delimited-list         
  prin1-to-string            read-from-string            
  princ                      read-line                   
  princ-to-string            read-preserving-whitespace  
  print                      read-sequence               
  print-not-readable         reader-error                
  print-not-readable-object  readtable                   
  print-object               readtable-case              
  print-unreadable-object    readtablep                  
  probe-file                 real                        
  proclaim                   realp                       
  prog                       realpart                    
  prog*                      reduce                      
  prog1                      reinitialize-instance       
  prog2                      rem                         
  progn                      remf                        
  program-error              remhash                     
  progv                      remove                      
  provide                    remove-duplicates           
  psetf                      remove-if                   
  psetq                      remove-if-not               
  push                       remove-method               
  pushnew                    remprop                     
  quote                      rename-file                 
  random                     rename-package              
  random-state               replace                     
  random-state-p             require                     
  rassoc                     rest                        
  rassoc-if                  restart                     
  rassoc-if-not              restart-bind                
  ratio                      restart-case                
  rational                   restart-name                
  rationalize                return                      
  rationalp                  return-from                 
  read                       revappend                   
  read-byte                  reverse                     

;   Figure 1-12: Symbols in the COMMON-LISP package (part nine of twelve).


  room                          simple-bit-vector                  
  rotatef                       simple-bit-vector-p                
  round                         simple-condition                   
  row-major-aref                simple-condition-format-arguments  
  rplaca                        simple-condition-format-control    
  rplacd                        simple-error                       
  safety                        simple-string                      
  satisfies                     simple-string-p                    
  sbit                          simple-type-error                  
  scale-float                   simple-vector                      
  schar                         simple-vector-p                    
  search                        simple-warning                     
  second                        sin                                
  sequence                      single-float                       
  serious-condition             single-float-epsilon               
  set                           single-float-negative-epsilon      
  set-difference                sinh                               
  set-dispatch-macro-character  sixth                              
  set-exclusive-or              sleep                              
  set-macro-character           slot-boundp                        
  set-pprint-dispatch           slot-exists-p                      
  set-syntax-from-char          slot-makunbound                    
  setf                          slot-missing                       
  setq                          slot-unbound                       
  seventh                       slot-value                         
  shadow                        software-type                      
  shadowing-import              software-version                   
  shared-initialize             some                               
  shiftf                        sort                               
  short-float                   space                              
  short-float-epsilon           special                            
  short-float-negative-epsilon  special-operator-p                 
  short-site-name               speed                              
  signal                        sqrt                               
  signed-byte                   stable-sort                        
  signum                        standard                           
  simple-array                  standard-char                      
  simple-base-string            standard-char-p                    

;   Figure 1-13: Symbols in the COMMON-LISP package (part ten of twelve).


  standard-class             sublis                      
  standard-generic-function  subseq                      
  standard-method            subsetp                     
  standard-object            subst                       
  step                       subst-if                    
  storage-condition          subst-if-not                
  store-value                substitute                  
  stream                     substitute-if               
  stream-element-type        substitute-if-not           
  stream-error               subtypep                    
  stream-error-stream        svref                       
  stream-external-format     sxhash                      
  streamp                    symbol                      
  string                     symbol-function             
  string-capitalize          symbol-macrolet             
  string-downcase            symbol-name                 
  string-equal               symbol-package              
  string-greaterp            symbol-plist                
  string-left-trim           symbol-value                
  string-lessp               symbolp                     
  string-not-equal           synonym-stream              
  string-not-greaterp        synonym-stream-symbol       
  string-not-lessp           t                           
  string-right-trim          tagbody                     
  string-stream              tailp                       
  string-trim                tan                         
  string-upcase              tanh                        
  string/=                   tenth                       
  string<                    terpri                      
  string<=                   the                         
  string=                    third                       
  string>                    throw                       
  string>=                   time                        
  stringp                    trace                       
  structure                  translate-logical-pathname  
  structure-class            translate-pathname          
  structure-object           tree-equal                  
  style-warning              truename                    

;   Figure 1-14: Symbols in the COMMON-LISP package (part eleven of twelve).


  truncate                             values-list               
  two-way-stream                       variable                  
  two-way-stream-input-stream          vector                    
  two-way-stream-output-stream         vector-pop                
  type                                 vector-push               
  type-error                           vector-push-extend        
  type-error-datum                     vectorp                   
  type-error-expected-type             warn                      
  type-of                              warning                   
  typecase                             when                      
  typep                                wild-pathname-p           
  unbound-slot                         with-accessors            
  unbound-slot-instance                with-compilation-unit     
  unbound-variable                     with-condition-restarts   
  undefined-function                   with-hash-table-iterator  
  unexport                             with-input-from-string    
  unintern                             with-open-file            
  union                                with-open-stream          
  unless                               with-output-to-string     
  unread-char                          with-package-iterator     
  unsigned-byte                        with-simple-restart       
  untrace                              with-slots                
  unuse-package                        with-standard-io-syntax   
  unwind-protect                       write                     
  update-instance-for-different-class  write-byte                
  update-instance-for-redefined-class  write-char                
  upgraded-array-element-type          write-line                
  upgraded-complex-part-type           write-sequence            
  upper-case-p                         write-string              
  use-package                          write-to-string           
  use-value                            y-or-n-p                  
  user-homedir-pathname                yes-or-no-p               
  values                               zerop                     

;   Figure 1-15: Symbols in the COMMON-LISP package (part twelve of twelve).
))

; Leave this here.  It is read when loading acl2.lisp.

(defconst *common-lisp-specials-and-constants*

; In acl2-check.lisp we ensure that this constant is consistent with the
; underlying Common Lisp.  The draft proposed ANSI standard for Common Lisp
; specifies (see "The COMMON-LISP Package") exactly which symbols are external
; symbols of the Common Lisp package (not just initially, but always).  It also
; states, in "Constraints on the COMMON-LISP Package for Conforming
; Implementations," that: "conforming programs can use external symbols of the
; COMMON-LISP package as the names of local lexical variables with confidence
; that those names have not been proclaimed special by the implementation
; unless those symbols are names of standardized global variables."
; Unfortunately, we cannot seem to find out in a direct fashion just which
; variables are standardized global variables, i.e., global variables defined
; in the standard.  Our check handles this.

; Shortly before releasing Version  2.5 (6/00), we have checked that the above
; form returns NIL on Unix systems running Allegro 5.0 and 5.0.1 and GCL 2.2.1
; and 2.2.2, on a Windows 98 system (via John Cowles) running Allegro 5.0.1,
; and (after defining the requisite constants) on CMU Common Lisp 18a on a Unix
; system at UT.

; It is completely acceptable to add symbols to this list.  If one certifies a
; book in such an ACL2, it will be a legal certification in an ACL2 in which
; the following list has not been modified.  The only potential source of
; concern here is if one certifies a book in an ACL2 where this list has not
; been modified and then includes it, without recertification, in an ACL2 where
; this list has been added to.  At this point we have not checked that such an
; include-book would catch an inappropriate use of one of those added symbols.
; But that seems a relatively minor concern.

  '(* ** *** *BREAK-ON-SIGNALS* *COMPILE-FILE-PATHNAME*
      *COMPILE-FILE-TRUENAME* *COMPILE-PRINT* *COMPILE-VERBOSE* *DEBUG-IO*
      *DEBUGGER-HOOK* *DEFAULT-PATHNAME-DEFAULTS* *ERROR-OUTPUT*
      *FEATURES* *GENSYM-COUNTER* *LOAD-PATHNAME* *LOAD-PRINT*
      *LOAD-TRUENAME* *LOAD-VERBOSE* *MACROEXPAND-HOOK* *MODULES*
      *PACKAGE* *PRINT-ARRAY* *PRINT-BASE* *PRINT-CASE* *PRINT-CIRCLE*
      *PRINT-ESCAPE* *PRINT-GENSYM* *PRINT-LENGTH* *PRINT-LEVEL*
      *PRINT-LINES* *PRINT-MISER-WIDTH* *PRINT-PPRINT-DISPATCH*
      *PRINT-PRETTY* *PRINT-RADIX* *PRINT-READABLY* *PRINT-RIGHT-MARGIN*
      *QUERY-IO* *RANDOM-STATE* *READ-BASE* *READ-DEFAULT-FLOAT-FORMAT*
      *READ-EVAL* *READ-SUPPRESS* *READTABLE* *STANDARD-INPUT*
      *STANDARD-OUTPUT* *TERMINAL-IO* *TRACE-OUTPUT* + ++ +++ - / // ///
      ARRAY-DIMENSION-LIMIT ARRAY-RANK-LIMIT ARRAY-TOTAL-SIZE-LIMIT
      BOOLE-1 BOOLE-2 BOOLE-AND BOOLE-ANDC1 BOOLE-ANDC2 BOOLE-C1 BOOLE-C2
      BOOLE-CLR BOOLE-EQV BOOLE-IOR BOOLE-NAND BOOLE-NOR BOOLE-ORC1
      BOOLE-ORC2 BOOLE-SET BOOLE-XOR CALL-ARGUMENTS-LIMIT CHAR-CODE-LIMIT
      DOUBLE-FLOAT-EPSILON DOUBLE-FLOAT-NEGATIVE-EPSILON
      INTERNAL-TIME-UNITS-PER-SECOND LAMBDA-LIST-KEYWORDS
      LAMBDA-PARAMETERS-LIMIT LEAST-NEGATIVE-DOUBLE-FLOAT
      LEAST-NEGATIVE-LONG-FLOAT LEAST-NEGATIVE-NORMALIZED-DOUBLE-FLOAT
      LEAST-NEGATIVE-NORMALIZED-LONG-FLOAT
      LEAST-NEGATIVE-NORMALIZED-SHORT-FLOAT
      LEAST-NEGATIVE-NORMALIZED-SINGLE-FLOAT LEAST-NEGATIVE-SHORT-FLOAT
      LEAST-NEGATIVE-SINGLE-FLOAT LEAST-POSITIVE-DOUBLE-FLOAT
      LEAST-POSITIVE-LONG-FLOAT LEAST-POSITIVE-NORMALIZED-DOUBLE-FLOAT
      LEAST-POSITIVE-NORMALIZED-LONG-FLOAT
      LEAST-POSITIVE-NORMALIZED-SHORT-FLOAT
      LEAST-POSITIVE-NORMALIZED-SINGLE-FLOAT LEAST-POSITIVE-SHORT-FLOAT
      LEAST-POSITIVE-SINGLE-FLOAT LONG-FLOAT-EPSILON
      LONG-FLOAT-NEGATIVE-EPSILON MOST-NEGATIVE-DOUBLE-FLOAT
      MOST-NEGATIVE-FIXNUM MOST-NEGATIVE-LONG-FLOAT
      MOST-NEGATIVE-SHORT-FLOAT MOST-NEGATIVE-SINGLE-FLOAT
      MOST-POSITIVE-DOUBLE-FLOAT MOST-POSITIVE-FIXNUM
      MOST-POSITIVE-LONG-FLOAT MOST-POSITIVE-SHORT-FLOAT
      MOST-POSITIVE-SINGLE-FLOAT MULTIPLE-VALUES-LIMIT NIL PI
      SHORT-FLOAT-EPSILON SHORT-FLOAT-NEGATIVE-EPSILON
      SINGLE-FLOAT-EPSILON SINGLE-FLOAT-NEGATIVE-EPSILON T

; Added in Version  2.6 to support Allegro 6.0 on Windows 2000:

      REPLACE FILL CHARACTER =

; Added in Version  2.6 to support GCL on Windows:

      BREAK PRIN1

      ))

(defconst *stobj-inline-declare*

; This constant is being introduced in v2-8.  In this file it is only used in
; raw Lisp, specifically in the progn just below.  But it is also used in
; defstobj-field-fns-raw-defs so we define it in the ACL2 loop.

  '(declare (stobj-inline-fn t)))

; Essay on Hidden Packages

; Before Version_2.8, ACL2 was unsound because of a hole in its handling of
; packages.  The books in the example below can all be certified in
; Version_2.7, including the top-level book top.lisp, which concludes with a
; proof of nil.  The details are slightly tricky, but the basic idea is simple:
; it was possible for traces of a defpkg event, including the axiom it added
; about symbol-package-name, to disappear by making include-books local.  And
; thus, it was possible to prove contradictory theorems, using contradictory
; defpkg events in different locally included books, about the
; symbol-package-name of a symbol.  One solution would be to disallow defpkg
; events in the context of a local include-book (much as we do for defaxiom),
; but that is too restrictive to be practical, especially since non-local
; include-book forms are prohibited inside encapsulate.  So instead we track
; such "hidden" defpkg events; more on that below.

; Here is the example promised above.  The idea is to define a package "FOO"
; that does not import any symbol of name "A", so that the symbol FOO::A has
; symbol-package-name "FOO".  But we do this twice, where one time package
; "FOO" imports ACL2::B and the other time it does not.  The two cases
; introduce symbols (wit1) and (wit2), which we can prove are equal, basically
; because both are FOO::A.  But the result of interning "B" in the package of
; (wit1) or (wit2) is "FOO" in one case and "ACL2" in the other, which allows
; us to prove nil.  We have tried simpler approaches but ACL2 caught us in
; those cases.  We use local include-books below in order to avoid some of
; those catches by avoiding the use of FOO:: in wit1.lisp and wit2.lisp.

#|
;;; file top.lisp

  (in-package "ACL2")

  (include-book "wit1")
  (include-book "wit2")

  ; The idea:
  ; (wit1) = (wit2) by symbol-equality
  ; But by evaluation (see wit1-prop and wit2-prop in the included books):
  ;   (symbol-package-name (intern-in-package-of-symbol "B" (wit1))) = "FOO"
  ;   (symbol-package-name (intern-in-package-of-symbol "B" (wit2))) = "ACL2"

  (defthm bug
    nil
    :hints (("Goal" :use (wit1-prop
                          wit2-prop
                          (:instance symbol-equality
                                     (s1 (wit1))
                                     (s2 (wit2))))))
    :rule-classes nil)

;;; file wit1.lisp

  (in-package "ACL2")

  (local (include-book "sub1"))

  (encapsulate
   ((wit1 () t))
   (local (defun wit1 () (sub1)))
   (local (in-theory (disable (wit1))))
   (defthm wit1-prop
     (and (symbolp (wit1))
          (equal (symbol-name (wit1)) "A")
          (equal (symbol-package-name (wit1)) "FOO")
          (equal (symbol-package-name
                  (intern-in-package-of-symbol "B" (wit1)))
                 "FOO"))
     :rule-classes nil))

;;; file sub1.lisp

  (in-package "ACL2")

  ; Portcullis:
  ; (defpkg "FOO" nil)

  (encapsulate
   ((sub1 () t))
   (local (defun sub1 () 'foo::a))
   (defthm sub1-prop
     (and (symbolp (sub1))
          (equal (symbol-name (sub1)) "A")
          (equal (symbol-package-name (sub1)) "FOO")
          (equal (symbol-package-name
                  (intern-in-package-of-symbol "B" (sub1)))
                 "FOO"))))

;;; file wit2.lisp

  (in-package "ACL2")

  (local (include-book "sub2"))

  (encapsulate
   ((wit2 () t))
   (local (defun wit2 () (sub2)))
   (local (in-theory (disable (wit2))))
   (defthm wit2-prop
     (and (symbolp (wit2))
          (equal (symbol-name (wit2)) "A")
          (equal (symbol-package-name (wit2)) "FOO")
          (equal (symbol-package-name
                  (intern-in-package-of-symbol "B" (wit2)))
                 "ACL2"))
     :rule-classes nil))

;;; file sub2.lisp

  (in-package "ACL2")

  ; Portcullis:
  ; (defpkg "FOO" '(b))

  (encapsulate
   ((sub2 () t))
   (local (defun sub2 () 'foo::a))
   (defthm sub2-prop
     (and (symbolp (sub2))
          (equal (symbol-name (sub2)) "A")
          (equal (symbol-package-name (sub2)) "FOO")
          (equal (symbol-package-name
                  (intern-in-package-of-symbol "B" (sub2)))
                 "ACL2"))))

;;; file sub1.acl2 (portcullis for sub1.lisp)

  (value :q)
  (lp)
  (defpkg "FOO" nil)
  (certify-book "sub1" 1)

;;; file sub2.acl2 (portcullis for sub2.lisp)

  (value :q)
  (lp)
  (defpkg "FOO" '(b))
  (certify-book "sub2" 1)

|#

; The key to disallowing this unfortunate exploitation of defpkg axioms is to
; maintain an invariant, which we call "the package invariant on logical
; worlds."  Roughly put, this invariant states that if the world depends in any
; way on a defpkg event, then that defpkg event occurs explicitly in that
; world.  (This invariant, like many others, depends on not having executed any
; event in the world when state global ld-skip-proofsp has a non-nil value.
; Note that we guarantee that this property holds for any certification world;
; see chk-acceptable-certify-book.)  Let us say that a defpkg event "supports"
; a world if it is either in that world or it is in some book (including its
; portcullis) that is hereditarily included in the current world via a chain of
; include-book events, some of which may be local to books or to encapsulate
; events.  Then we can be more precise by stating the package invariant on
; logical worlds as follows: Every defpkg event that supports a logical world
; is present in the known-package-alist of that world.

; It is convenient to introduce the notion of a "hidden" defpkg event in a
; logical world as one that supports that world but is not present as an event
; in that world.  The discussion below relies on the presence of several fields
; in a known-package-alist entry; see make-package-entry.

; We guarantee the (above) package invariant on logical worlds starting with
; Version_2.8 by way of the following two actions, which allow include-book and
; encapsulate (respectively) to preserve this invariant.  Roughly speaking:
; action (1) extends a book's portcullis by any hidden defpkg supporting the
; book, so that the defpkg will not be missing from the world (thus violating
; the invariant) when we include the book; and action (2) puts a
; known-package-alist entry for each (hidden) defpkg introduced by a given
; encapsulate.

;   (1) Recall that when a book is successfully certified in an existing
;   certification world, we write the commands of that world to the book's
;   certificate, as so-called "portcullis commands."  We extend those
;   portcullis commands with defpkg events in two ways.  First, we add a defpkg
;   at the end of the portcullis commands for every known-package-alist entry
;   that has hidden-p fields equal to t (for example, because of a local
;   include-book in a top-level encapsulate), and hence is not an event in the
;   certification world.  We will of course not count these extra defpkgs when
;   checking against a numeric argument given to certify-book.  Second, for
;   each package entry present in the known-package-alist at the end of the
;   proof pass of certify-book that is not present at the end of the
;   include-book pass, we add a corresponding defpkg event to the end of the
;   portcullis commands.

;   Each defpkg event added to the portcullis as described above will have a
;   :book-path argument derived from the book-path field of a package-entry in
;   the known-package-alist, intended to represent the list of full book names
;   leading from the innermost book actually containing the corresponding
;   defpkg (in the car), up to the top-level such include-book (at the end of
;   the list).  Thus, when we evaluate that defpkg, the new package-entry in
;   known-package-alist is obtained by appending the current world's
;   include-book-path to the event's book-path.  The book-path field in the
;   package-entry can be used later when reporting an error during a package
;   conflict, so that the user can see the source of the defpkg that was added
;   to the portcullis under the hood.  Documentation topic hidden-death-package
;   explains hidden defpkgs in detail, and is referenced during such errors.

;   In order to keep the certificate size under control, we will check whether
;   the body of a hidden defpkg event to be added to the portcullis is a term
;   in the world where it will be evaluated, and that this term's value is
;   indeed the list of symbols associated with that package in the
;   known-package-alist (a necessary check for a hidden defpkg since that term
;   may have a different value in the world present at the time of the
;   executing of the defpkg).  If so, then we leave that term in place.
;   Otherwise, we replace it by the appropriate quoted list of symbols, though
;   we might still optimize by removing subsets that are commonly-used
;   constants (e.g. *acl2-exports* and
;   *common-lisp-symbols-from-main-lisp-package*), in favor of suitable calls
;   of append or union-eq.  Note that for hidden defpkg events encountered in a
;   book during its certification, our decision to put them at the end of the
;   certificate's portcullis commands, rather than the beginning, increases the
;   chances that the original defpkg's term can be retained.

;   (2) At the end of any encapsulate, the known-package-alist will be extended
;   with an entry for each introduced defpkg.  (We do this for every package in
;   the known-package-alist at the end of the first pass of the encapsulate
;   that was not there in the beginning, since these must all have been
;   introduced by include-book, and only local include-books are allowed by
;   encapsulate.)  Each such entry will have appropriate package-entry fields,
;   including hidden-p = t.

; Note that when we evaluate a defpkg in a world where that package exists but
; is hidden, the event will not be redundant, and we will change the hidden-p
; field to nil in the known-package-alist entry.  Other fields can be used for
; error reporting.  For example, if we attempt to introduce a defpkg when there
; is already a hidden defpkg conflicting with it, we can report the
; include-book path to the defpkg.

; Finally, we discuss how to ensure that :puff preserves the package invariant.
; Recall that the basic idea behind the implementation of :puff is the
; execution of function puffed-command-sequence to obtain a sequence of
; commands to execute after backing up through the given command.  It is
; straightforward to find the hidden defpkg events that occur in the
; known-package-alist of the world just after the command but not just before,
; and add corresponding defpkg events to the front of the
; puffed-command-sequence.  This preserves the invariant.

; End of Essay on Hidden Packages

(defmacro make-package-entry (&key name imports hidden-p book-path
                                   defpkg-event-form tterm)

; Normally we would use defrec here.  But defrec is defined in basis.lisp.
; Rather than move all the code relevant to defrec into axioms.lisp, we make
; our lives easy for now and simply define the relevant macros directly.  For
; the record (pun intended), here is the defrecord:

; (defrec package-entry
;   (name imports hidden-p book-path defpkg-event-form . tterm)
;   t)

; WARNING: We allow assoc-equal (actually its macro form, find-package-entry)
; to look up names in the known-package-alist, so keep the name as the car.
; Also note that name, imports, and hidden-p are accessed much more frequently
; than the rest, so these should all get relatively fast access.

  `(list* ,name      ; the package name
          ,imports   ; the list of imported symbols
          ,hidden-p  ; t if the introducing defpkg is hidden, else nil

 ; The remaining fields are used for messages only; they have no logical import.

          ,book-path ; a true list of full book names, where the path
                     ; from the first to the last in the list is intended to
                     ; give the location of the introducing defpkg, starting
                     ; with the innermost book

; The final fields are def and tterm, where def is the defpkg event that
; introduced this package and tterm is the translation of the body of that
; defpkg.  If this package-entry becomes hidden, we may use these fields to
; extend the portcullis commands in a book's certificate file.  In doing so, we
; use tterm if it is a term in the world w that is present at the point of
; insertion into the portcullis commands, except that better yet, we will use
; the originating untranslated term from the defpkg if that is the result of
; untranslating tterm in w.

          ,defpkg-event-form
          ,tterm
          ))

(defmacro find-package-entry (name known-package-alist)
  `(assoc-equal ,name ,known-package-alist))

(defmacro package-entry-name (package-entry)
  `(car ,package-entry))

(defmacro package-entry-imports (package-entry)
  `(cadr ,package-entry))

(defmacro package-entry-hidden-p (package-entry)
  `(caddr ,package-entry))

(defmacro package-entry-book-path (package-entry)
  `(cadddr ,package-entry))

(defmacro package-entry-defpkg-event-form (package-entry)
  `(car (cddddr ,package-entry)))

(defmacro package-entry-tterm (package-entry)
  `(cdr (cddddr ,package-entry)))

(defmacro find-non-hidden-package-entry (name known-package-alist)
  `(let ((entry (assoc-equal ,name ,known-package-alist)))
     (and (not (package-entry-hidden-p entry))
          entry)))

(defmacro remove-package-entry (name known-package-alist)
  `(delete-assoc-equal ,name ,known-package-alist))

(defmacro change-package-entry-hidden-p (entry value)
  `(let ((entry ,entry))
     (make-package-entry
      :name (package-entry-name entry)
      :imports (package-entry-imports entry)
      :hidden-p ,value
      :book-path (package-entry-book-path entry)
      :defpkg-event-form (package-entry-defpkg-event-form entry)
      :tterm (package-entry-tterm entry))))

(defmacro getprop (symb key default world-name world-alist)

; This definition formerly occurred after fgetprop and sgetprop, but since
; getprop is used in defpkg-raw we move it before defpkg-raw.  This move would
; not be necessary if we were always to load a source file before we load the
; corresponding compiled file, but with *suppress-compile-build-time* we do not
; load the latter (nor do we re-load the source file, as of this writing, for
; efficiency).

; We avoid cond here because it hasn't been defined yet!

  (if (equal world-name ''current-acl2-world)
      `(fgetprop ,symb ,key ,default ,world-alist)
    `(sgetprop ,symb ,key ,default ,world-name ,world-alist)))

#-acl2-loop-only
(progn

(defvar *user-stobj-alist* nil)

; The value of the above variable is an alist that pairs user-defined
; single-threaded object names with their live ones.  It does NOT
; contain an entry for STATE, which is not user-defined.

; The following SPECIAL VARIABLE, *wormholep*, when non-nil, means that we
; are within a wormhole and are obliged to undo every change visited upon
; *the-live-state*.  Clearly, we can undo some of them, e.g., f-put-globals, by
; remembering the first time we make a change to some component.  But other
; changes, e.g., printing to a file, we can't undo and so must simply disallow.
; We disallow all modifications to user stobjs.

; This feature is implemented so that we can permit the "wormhole window" to
; manipulate a "copy" of state without changing it.  The story is that wormhole,
; which does not take state as an arg and which always returns nil, is
; "actually" implemented by calling the familiar LD on a near image of the
; current state.  That near image is like the current state except that certain
; state globals have been set for wormhole.  In addition, we assume that the
; physical map between ACL2 channels and the outside world has been altered so
; that *standard-co*, *standard-ci*, and *standard-oi* now actually interact
; with the "wormhole window" streams.  Thus, even when *wormholep* is non-nil, we
; can allow i/o to those standard channels because it causes no change to the
; streams normally identified with those channels.  If, while *wormholep* is
; non-nil we are asked to make a change that would undoably alter the state, we
; print a soft-looking error message and abort.  If the requested change can be
; undone, we make the change after remembering enough to undo it.  When we exit
; the wormhole we undo the changes.

(defparameter *wormholep* nil)

; Below we define the function that generates the error message when
; non-undoable state changes are attempted within wormholes.  It throws
; to a tag that is set up within LP.  We do all that later.  Right now
; we just define the error handler so we can code the primitives.

(defun-one-output replace-bad-lisp-object (x)
  (if (bad-lisp-objectp x)
      (let ((pair (rassoc x *user-stobj-alist*)))
        (if pair
            (car pair)

; The following will be printed if we are looking at a local stobj value.

          '|<Unknown value>|))
    x))

(defun-one-output replace-bad-lisp-object-list (x)
  (if (null x)
      nil
    (cons (replace-bad-lisp-object (car x))
          (replace-bad-lisp-object-list (cdr x)))))

(defun-one-output wormhole-er (fn args)
  (error-fms nil 'wormhole
             "It is not possible to apply ~x0~#1~[~/ to ~&2~] in the current ~
              context because we are in a wormhole state."
             (list (cons #\0 fn)
                   (cons #\1 (if args 1 0))
                   (cons #\2 (replace-bad-lisp-object-list args)))
             *the-live-state*)
  (throw 'local-top-level :wormhole-er))

; The following parameter is where we will accumulate changes to
; state components that we will undo.

(defparameter *wormhole-cleanup-form* nil)

; The value of *wormhole-cleanup-form* is a lisp (but not ACL2) form that will
; be executed to cleanup the live state.  This form is built up incrementally
; by certain state changing primitives (e.g., f-put-global) so as to enable us
; to "undo" the effects of those primitives.  We store this undo information
; as an executable form (rather than, say, a list of "undo tuples") because of
; the interaction between this mechanism and our acl2-unwind-protect
; mechanism.  In particular, it will just happen to be the case that the
; *wormhole-cleanup-form* is always on the unwind protection stack (a true
; lisp global variable) so that if an abort happens while executing in a
; wormhole and we get ripped all the way out because of perfectly timed
; aborts, the undo cleanup form(s) will be at their proper places on the stack
; of cleanup forms and it will just look certain acl2-unwind-protects were
; interrupted.  See the discussion in and around LD-FN.  The value of
; *wormhole-cleanup-form* is (PROGN save-globals undo-form1 ... undo-formk
; safety-set STATE).  The individual undo-formi are created and added to the
; *wormhole-cleanup-form* by push-wormhole-undo- formi, below.  The initial
; value of the cleanup form is (PROGN save-globals safety-set STATE) and new
; formis are added immediately after save-globals, making the final form a
; stack with save-globals always on top and the formi succeeding it in reverse
; order of their storage.  The save-globals form will save into a lisp special
; the final values of the global variables that are available only in the
; wormhole.  The save-globals form is complicated because it also contains a
; check that the cleanup form has never been completely executed.  It does
; this by checking the car of a cons that ``belongs'' to this incarnation of
; the form.  The safety-set at the end of the form sets the car of that cons
; to t.  We cannot prevent the possible partial re-execution of the unwind
; protection form in the face of repeated ill-timed ctrl-c's and we cannot
; really guarantee that a ctrl-c doesn't prevent the execution of the
; safety-set even though the ``real'' cleanup work has been successfully done.
; But the re-execution of the cleanup form can confuse the tracking of the
; brr-stack gstack and we installed this check just for an increased sense of
; sanity.  See the comment after wormhole1.

; We introduce a CLTL structure for the sole purpose of preventing the
; accidental printing of huge objects like the world.  If, in raw lisp, you
; write (make-cloaking-device :hint "world" :obj (w *the-live-state*)) then you
; get an object, x, that CLTL will print as <cloaked world> and from which the
; actual world can be recovered via (cloaking-device-obj x).

(defstruct (cloaking-device
            (:print-function
             (lambda (x stream k)
               (declare (ignore k))
               (format stream "<cloaked ~a>" (cloaking-device-hint x)))))
  hint obj)

(defun-one-output cloaked-set-w! (x state)

; We invented this function, which is merely set-w! but takes a cloaked world,
; just so we can print the *acl2-unwind-protect-stack* during debugging without
; getting the world printed.

  (set-w! (cloaking-device-obj x) state))

(defun-one-output assoc-eq-butlast-2 (x alist)

; This variant of assoc-eq is used in push-wormhole-undo-formi, for which alist
; is not a true alist but rather has two final elements that we do not want to
; consider.  It is run only in raw Lisp on "alists" of the form mentioned
; above.

  (cond ((endp (cddr alist)) nil)
        ((eq x (car (car alist))) (car alist))
        (t (assoc-eq-butlast-2 x (cdr alist)))))

(defun-one-output assoc-eq-equal-butlast-2 (x y alist)

; This variant of assoc-eq-equal is used in push-wormhole-undo-formi, for which
; alist is not a true alist but rather has two final elements that we do not
; want to consider.  It is run only in raw Lisp on "alists" of the form
; mentioned above.

  (cond ((endp (cddr alist)) nil)
        ((and (eq (car (car alist)) x)
              (equal (car (cdr (car alist))) y))
         (car alist))
        (t (assoc-eq-equal-butlast-2 x y (cdr alist)))))

(defun-one-output push-wormhole-undo-formi (op arg1 arg2)

; When a primitive state changing function is called while *wormholep*
; is non-nil it actually carries out the change (in many cases) but
; saves some undo information on the special *wormhole-cleanup-form*.
; The value of that special is (PROGN save-globals form1 ... formk
; safety-set STATE).  In response to this call we will add a new form,
; say form0, and will destructively modify *wormhole-cleanup-form* so
; that it becomes (PROGN save-globals form0 form1 ...  formk
; safety-set STATE).

; We modify *wormhole-cleanup-form* destructively because it shares
; structure with the *acl2-unwind-protect-stack* as described above.

; The convention is that the primitive state changer calls this function before
; making any change.  It passes us the essential information about the
; operation that must be performed to undo what it is about to do.  Thus, if we
; store a new value for a global var, v, whose old value was x, then op will be
; 'put-global, arg1 will be v, and arg2 will be x.  The formi we create will be
; (put-global 'v 'x *the-live-state*) and when that is executed it will undo
; the primitive state change.  Note that we do not know what the primitive
; actually was, e.g., it might have been a put-global but it might also have
; been a makunbound-global.  The point is that the 'put-global in our note is
; the operation that must be done at undo-time, not the operation that we are
; undoing.

; Furthermore, we need not save undo information after the first time
; we smash v.  So we don't necessarily store a formi.  But to implement this we
; have to know every possible formi and what its effects are.  That is why we
; insist that this function (rather than our callers) create the forms.

; To think about the avoidance of formi saving, consider the fact that the
; cleanup form, being a PROGN, will be executed sequentially -- -- undoing the
; state changes in the reverse order of their original execution.  Imagine that
; we in fact added a new formi at the front of the PROGN for each state change.
; Now think about it: if later on down the PROGN there is a form that will
; overwrite the effects of the form we are about to add, then there is no need
; to add it.  In particular, the result of evaluating all the forms is the same
; whether we add the redundant one or not.

  (cond ((null *wormhole-cleanup-form*)
         (interface-er
          "push-wormhole-undo-formi was called with an empty ~
           *wormhole-cleanup-form*.  Supposedly, push-wormhole-undo-formi is ~
           only called when *wormholep* is non-nil and, supposedly, when ~
           *wormholep* is non-nil, the *wormhole-cleanup-form* is too.")))
  (let ((qarg1 (list 'quote arg1))
        (undo-forms-and-last-two (cddr *wormhole-cleanup-form*)))
    (case op
      (put-global

; So we want to push (put-global 'arg1 'arg2 state).  But if there is already a
; form that will set arg1 or one that unbinds arg1, there is no point.

       (or (assoc-eq-equal-butlast-2 'put-global qarg1
                                     undo-forms-and-last-two)
           (assoc-eq-equal-butlast-2 'makunbound-global qarg1
                                     undo-forms-and-last-two)
           (and (eq arg1 'current-acl2-world)
                (assoc-eq-butlast-2 'cloaked-set-w!
                                    undo-forms-and-last-two))
           (setf (cddr *wormhole-cleanup-form*)
                 (cons (let ((put-global-form
                              `(put-global ,qarg1 (quote ,arg2)
                                           *the-live-state*)))

; We compress arrays for side-effect only, to ensure that we do not install a
; different global value than was there before.  Fortunately, we know that the
; arrays in question are already in compressed form, i.e., they satisfy
; array1p; so we believe that these side-effects do not change the array's
; alist (in the sense of eq), and hence the restored global value will be
; installed as an ACL2 array.  (If we're wrong, it's not a soundness issue --
; rather, we will see slow-array-warning messages.)

                         (cond ((eq arg1 'global-enabled-structure)
                                `(progn (let ((qarg2 (quote ,arg2)))
                                          (compress1 (access enabled-structure
                                                             qarg2
                                                             :array-name)
                                                     (access enabled-structure
                                                             qarg2
                                                             :theory-array)))
                                        ,put-global-form))
                               ((and (eq arg1 'iprint-ar)
                                     arg2)
                                `(progn (let ((qarg2 (quote ,arg2)))
                                          (compress1 'iprint-ar qarg2))
                                        ,put-global-form))
                               ((eq arg1 'trace-specs)
                                nil) ; handled by fix-trace-specs
                               (t put-global-form)))
                       (cddr *wormhole-cleanup-form*)))))
      (makunbound-global

; We want to push (makunbound-global 'arg1 state).  But if there is already
; a form that will make arg1 unbound or if there is a form that will
; give it a binding, this is redundant.

       (or (assoc-eq-equal-butlast-2 'put-global qarg1
                                     undo-forms-and-last-two)
           (assoc-eq-equal-butlast-2 'makunbound-global qarg1
                                     undo-forms-and-last-two)
           (and (eq arg1 'current-acl2-world)
                (assoc-eq-butlast-2 'cloaked-set-w!
                                    undo-forms-and-last-two))
           (setf (cddr *wormhole-cleanup-form*)
                 (cons `(makunbound-global ,qarg1 *the-live-state*)
                       (cddr *wormhole-cleanup-form*)))))
      (cloaked-set-w!
       (or (assoc-eq-butlast-2 'cloaked-set-w! undo-forms-and-last-two)
           (setf (cddr *wormhole-cleanup-form*)
                 (cons `(cloaked-set-w!
                         ,(make-cloaking-device
                           :hint "world"
                           :obj arg1)
                         *the-live-state*)
                       (cddr *wormhole-cleanup-form*)))))
      (otherwise
       (interface-er "Unrecognized op in push-wormhole-undo-formi,~
                          ~x0." op)))))

; The following symbol is the property under which we store Common
; Lisp streams on the property lists of channels.

(defconstant *open-input-channel-key*
  'acl2_invisible::|Open Input Channel Key|)

; The following symbol is the property under which we store the types
; of Common Lisp streams on the property lists of channels.

(defconstant *open-input-channel-type-key*
  'acl2_invisible::|Open Input Channel Type Key|)

(defconstant *open-output-channel-key*
  'acl2_invisible::|Open Output Channel Key|)

(defconstant *open-output-channel-type-key*
  'acl2_invisible::|Open Output Channel Type Key|)

(defconstant *non-existent-stream*
  'acl2_invisible::|A Non-Existent Stream|)

(defmacro live-state-p (x)
  (list 'eq x '*the-live-state*))

; We get ready to handle errors in such a way that they return to the
; top level logic loop if we are under it.

(defun interface-er (&rest args)

; This function can conceivably be called before ACL2 has been fully
; compiled and loaded, so we check whether the usual error handler is
; around.

  (cond
   ((macro-function 'er)
    (eval
     `(let ((state *the-live-state*))
        (er soft 'acl2-interface
            ,@(let (ans)
                (dolist (a args)
                        (push (list 'quote a) ans))
                (reverse ans)))
        (error "ACL2 Halted"))))
   (t (error "ACL2 error:  ~a." args))))

; We provide here ``raw'' implementations of basic functions that we
; ``wish'' were already in Common Lisp, to support primitives of the
; ACL2 logic.

; Some of the Common Lisp arithmetic primitives are n-ary functions.
; However, ACL2 supports only functions of fixed arity, to keep the
; logic simple.  But in practice we find we want to use the n-ary
; arithmetic symbols ourselves.  So in the logic we have binary-+ as
; the primitive binary addition function symbol, but we also have the
; macro +, which expands into a suitable number of uses of binary-+.
; Similarly for *, -, and /.  (The ACL2 user cannot invoke
; symbol-function, fboundp, macro-function or macroexpand, so it is no
; concern to the user whether we implement + as a macro or a
; function.)

(defun-one-output acl2-numberp (x)
  (numberp x))

(defun-one-output binary-+ (x y) (+ x y))

(defun-one-output binary-* (x y) (* x y))

(defun-one-output unary-- (x) (- x))

(defun-one-output unary-/ (x) (/ x))

; Below we define our top-level events as seen by the Common Lisp
; compiler.  For example, (defuns a b c) expands into a progn of defun
; forms, (defthm ...) is a no-op, etc.

(defparameter *in-recover-world-flg* nil)

; Warning:  Keep the initial value of the following defparameter identical to
; that of the ACL2 constant *initial-known-package-alist* below.

(defparameter *ever-known-package-alist*
  (list (make-package-entry :name "ACL2-INPUT-CHANNEL"
                            :imports nil)
        (make-package-entry :name "ACL2-OUTPUT-CHANNEL"
                            :imports nil)
        (make-package-entry :name "ACL2"
                            :imports *common-lisp-symbols-from-main-lisp-package*)
        (make-package-entry :name

; Warning: The following is just *main-lisp-package-name* but that is not
; defined yet.  If you change the following line, change the defconst of
; *main-lisp-package-name* below.

                            "COMMON-LISP"
                            :imports nil)
        (make-package-entry :name "KEYWORD"
                            :imports nil)))

; The known-package-alist of the state will grow and shrink as packages are
; defined and undone.  But *ever-known-package-alist* will just grow.  A
; package can be redefined only if its imports list is identical to that in its
; old definition.

; The following alist associates package names with Common Lisp packages, and
; is used in function find-package-fast, which is used by princ$ in place of
; find-package in order to save perhaps 15% of the print time.
(defparameter *package-alist* nil)

(defun-one-output find-package-fast (string)
  (or (cdr (assoc-equal string *package-alist*))
      (let ((pkg (find-package string)))
        (push (cons string pkg) *package-alist*)
        pkg)))

(defvar **1*-symbol-key* (make-symbol "**1*-SYMBOL-KEY*"))

(defvar *global-symbol-key* (make-symbol "*GLOBAL-SYMBOL-KEY*"))

(defun global-symbol (x)
  (or (get x *global-symbol-key*)
      (setf (get x *global-symbol-key*)
            (intern (symbol-name x)
                    (find-package-fast
                     (concatenate 'string
                                  *global-package-prefix*
                                  (symbol-package-name x)))))))

(defun *1*-symbol (x)
; Keep this in sync with *1*-symbol?.
  (or (get x **1*-symbol-key*)
      (setf (get x **1*-symbol-key*)
            (intern (symbol-name x)
                    (find-package-fast
                     (concatenate 'string
                                  *1*-package-prefix*
                                  (symbol-package-name x)))))))

(defun *1*-symbol? (x)
; Keep this in sync with *1*-symbol.  Returns nil if the *1* package doesn't
; exist.
  (let ((pack (find-package-fast (concatenate 'string
                                              *1*-package-prefix*
                                              (symbol-package-name x)))))
    (and pack
         (or (get x **1*-symbol-key*)
             (setf (get x **1*-symbol-key*)
                   (intern (symbol-name x)
                           pack))))))

(defun our-import (syms pkg)

; We have seen a case in which Allegro CL 8.0 spent about 20% of the time in
; IMPORT, on an include-book (with lots of nested include-books, and 20 defpkg
; forms executed altogether).  That time was reduced to near 0 by using the
; present function, OUR-IMPORT, in place of IMPORT, presumably because
; (according to the profiler) calls to EXCL::INTERNAL-STRING= were avoided,
; probably in favor of hashing.  We saw no significant change in time in GCL,
; however, so we exclude GCL and any non-ANSI (hence maybe no LOOP) Common Lisp
; from this enhancement.  It might be worthwhile to consider other Common Lisp
; implementations besides Allegro CL and GCL.  Perhaps Allegro CL will speed up
; its handling of IMPORT in future implementations (we have sent email to Franz
; Inc. about this), in which case we might consider deleting this function.

  #+(and (not gcl) cltl2)
  (loop for sym in syms do (import (or sym (list sym)) pkg))
  #-(and (not gcl) cltl2)
  (import syms pkg))

(defvar *defpkg-virgins* nil)

(defun-one-output defpkg-raw1 (name imports event-form)
  (let ((package-entry (find-package-entry name *ever-known-package-alist*)))
    (cond
     ((raw-mode-p *the-live-state*)
      (interface-er
       "It is illegal to execute defpkg in raw mode.  See :DOC set-raw-mode."))
     (t
      (let ((pkg (find-package name))
            (global-name (concatenate 'string
                                      acl2::*global-package-prefix*
                                      name))
            (*1*-name (concatenate 'string
                                   acl2::*1*-package-prefix*
                                   name))
            (proposed-imports (sort-symbol-listp imports)))
        (assert pkg) ; see defpkg-raw

; We bind proposed-imports to the value of the imports argument.  We do not
; want to evaluate it more than once below.  We DO reference, and hence
; evaluate, name more than once below.  But name must be an explicit string
; constant.

        (cond
         (package-entry
          (cond
           ((equal proposed-imports (package-entry-imports package-entry))

; The package has already been built in Common Lisp and the imports are
; identical.  There is nothing for us to do.

            name)
           (t

; The package has already been built in Common Lisp but with the wrong imports.
; There is nothing we can do.  We do not want to unintern any symbols in it
; because we may thus render bad some saved logical worlds.  See :DOC
; package-reincarnation-import-restrictions.  In addition, see the Lisp comment
; that is part of that deflabel (but which is not actually part of the
; ACL2 documentation).

            (error
             "~%We cannot reincarnate the package ~s with imports ~s~%because ~
              it was previously defined with imports ~s.  See~%:DOC ~
              package-reincarnation-import-restrictions."
             name
             proposed-imports
             (package-entry-imports package-entry)))))
         ((not (member-equal name *defpkg-virgins*))

; The package has been built in this Common Lisp but not by defpkg-raw1.  It
; may be new because of the defpackage form in defpkg-raw, in which case it is
; an element of *defpkg-virgins*.  Otherwise, it was defined in Common Lisp
; outside ACL2, and we should cause an error.

          (error
           "~%It is illegal to defpkg ~s because a package of that name ~
            already exists in this lisp.~%"
           name))
         (t
          (assert (not (assoc-equal name *package-alist*)))
          (let ((incomplete-p t)
                (saved-ever-known-package-alist *ever-known-package-alist*))
            (setq *defpkg-virgins*
                  (remove1-equal name *defpkg-virgins*))
            (unwind-protect
                (progn
                  (setq *ever-known-package-alist*
                        (cons (make-package-entry :name name
                                                  :imports proposed-imports
                                                  :defpkg-event-form event-form)
                              *ever-known-package-alist*))
                  (when proposed-imports

; Without the qualifier above, clisp imports nil if proposed-imports = nil.

                    (our-import proposed-imports (find-package name)))

; So at this point we have set the package's imports appropriately.  We now
; handle the dual packages in which the state globals and executable
; counterparts of symbols from pkg will reside.  We do not reinitialize these
; hidden variables if we are recovering from an error or booting.

                  (cond
                   ((and (not *in-recover-world-flg*)
                         (not (getprop 'boot-strap-flg 'global-value nil
                                       'current-acl2-world
                                       (w *the-live-state*))))
                    (cond ((find-package global-name)
                           (do-symbols (sym (find-package global-name))
                             (makunbound sym)))
                          (t (make-package global-name :use nil)))
                    (cond ((find-package *1*-name)
                           nil)
                          (t (make-package *1*-name :use nil)))))
                  (setq incomplete-p nil)
                  name)
              (when incomplete-p
                (setq *ever-known-package-alist*
                      saved-ever-known-package-alist)
                (do-symbols (sym pkg)
                  (unintern sym))
                (delete-package (find-package name))))))))))))

(defun package-has-no-imports (name)
  (let ((pkg (find-package name)))
    (do-symbols (sym pkg)
                (when (not (eq (symbol-package sym) pkg))
                  (return-from package-has-no-imports nil))))
  t)

(defmacro maybe-introduce-empty-pkg-1 (name)
  (let ((defp #+cltl2 'defpackage #-cltl2 'user::defpackage))
    `(progn
       (,defp ,name
         (:use))
       (,defp ,(concatenate 'string
                            acl2::*global-package-prefix*
                            name)
         (:use))
       (,defp ,(concatenate 'string
                            acl2::*1*-package-prefix*
                            name)
         (:use)))))

(defmacro maybe-introduce-empty-pkg-2 (name)
  `(when (and (not (member-equal ,name *defpkg-virgins*))
              (not (member-equal ,name *ever-known-package-alist*))
              (package-has-no-imports ,name))
     (push ,name *defpkg-virgins*)))

(defmacro defpkg-raw (name imports event-form)

; Defpkg checks that name is a string.  Event-form is a cons.  So we don't need
; to worry about capture below.

  `(let ((package-entry
          (find-package-entry ,name *ever-known-package-alist*)))
     (cond
      ((and package-entry
            (let ((old-event-form
                   (package-entry-defpkg-event-form package-entry)))
              (and (equal (cadr old-event-form) (cadr ,event-form))
                   (equal (caddr old-event-form) (caddr ,event-form)))))

; This shorcut is potentially a big concern!  We are checking that the name and
; term of the defpkg form agrees with an old defpkg form.  But these two forms
; may have been evaluated in different worlds!  Nevertheless, for now we trust
; that they really are equivalent, for efficiency's sake.  Defpkg-fn will call
; chk-acceptable-defpkg, which will call
; chk-package-reincarnation-import-restrictions, and if there is a discrepancy
; between the current and old package, we'll find out then.

       ,name)
      (t
       (maybe-introduce-empty-pkg-1 ,name)
       (maybe-introduce-empty-pkg-2 ,name)
       (defpkg-raw1 ,name ,imports ,event-form)))))

(defmacro defpkg (&whole event-form name imports &optional doc book-path)
  (declare (ignore doc book-path))
  (or (stringp name)
      (interface-er "Attempt to call defpkg on a non-string, ~x0."
                    name))
  `(defpkg-raw ,name ,imports ',event-form))

(defmacro defuns (&rest lst)
  `(progn ,@(mapcar #'(lambda (x) `(defun ,@x))
                    lst)))

#+:non-standard-analysis
(defmacro defun-std (name formals &rest args)
  (list* 'defun
         name
         formals
         (append (butlast args 1)
                 (list (non-std-body name formals (car (last args)))))))

#+:non-standard-analysis
(defmacro defuns-std (&rest args)
  `(defuns ,@args))

(defmacro defthm (&rest args)
  (declare (ignore args))
  nil)

(defmacro defthmd (&rest args)
  (declare (ignore args))
  nil)

#+:non-standard-analysis
(defmacro defthm-std (&rest args)
  (declare (ignore args))
  nil)

(defmacro defaxiom (&rest args)
  (declare (ignore args))
  nil)

(defmacro skip-proofs (arg)
  arg)

(defmacro deflabel (&rest args)
  (declare (ignore args))
  nil)

(defmacro defdoc (&rest args)
  (declare (ignore args))
  nil)

(defmacro deftheory (&rest args)
  (declare (ignore args))
  nil)

(defun-one-output stobj-initial-statep-arr (n i arr init)
  (or (zp n)
      (and (equal (aref arr i) init)
           (stobj-initial-statep-arr (1- n) (1+ i) arr init))))

(defun-one-output stobj-initial-statep-entry (temp entry)

; Keep this function in sync with defstobj-raw-init-fields.  (See the comments
; about this function, below.)

  (let ((type (cadr temp))
        (init (caddr temp)))
    (cond
     ((and (consp type)
           (eq (car type) 'ARRAY))

; For stobj array fields, we need to check each entry in the array to make sure
; it is the initial value and we also need to check that the array has not been
; resized to a size different than the initial size.

      (let ((size (car (caddr type))))
        (and (equal (length entry) size)
             (stobj-initial-statep-arr size 0 entry init))))
     ((equal type t)

; For type "T", the stobj field is not "boxed" by defstobj-raw-init-fields.

      (equal entry init))
     (t

; For other types, the value is "boxed" by defstobj-raw-init-fields in a single
; entry array.

      (equal (aref entry 0) init)))))

(defun-one-output stobj-initial-statep1 (field-templates ndx stobj)
  (cond ((endp field-templates) t)
        (t (and (stobj-initial-statep-entry (car field-templates)
                                            (aref stobj ndx))
                (stobj-initial-statep1 (cdr field-templates)
                                       (1+ ndx)
                                       stobj)))))

(defun-one-output stobj-initial-statep (stobj field-templates)

; Stobj is the live object corresponding to some defstobj and
; field-templates is the field templates for the defstobj.  We return
; t or nil according to whether the live object is in the initial
; state.

; Each element of field-templates is of the form (recog-fn type
; init-val acc-fn upd-fn ...).  If type indicates an array, then it
; has the form (ARRAY typ (max)), and the indices of the array range
; from 0 to max-1, i.e., max is the first illegal index.

  (stobj-initial-statep1 field-templates 0 stobj))

(defun remove-stobj-inline-declare (x)
  (cond ((atom x) x)
        ((equal (car x) *stobj-inline-declare*)
         (cdr x))
        (t (cons (car x)
                 (remove-stobj-inline-declare (cdr x))))))

; Note: The code below bothers me a little because of its impact on
; the toothbrush model.  In particular, it uses defstobj-raw-defs,
; which is defined far away in other-events.lisp.

(defmacro defstobj (name &rest args)

; This function is run when we evaluate (defstobj name . args) in raw lisp.
; A typical such form is

; (defstobj $st
;   (flag :type t :initially run)
;   (pc   :type (integer 0 255) :initially 128)
;   (mem  :type (array (integer 0 255) (256)) :initially 0))

; This function must contend with a problem analogous to the one addressed by
; acl2::defconst in acl2.lisp: the need to avoid re-declaration of the same
; stobj.  We use redundant-raw-lisp-discriminator in much the same way as in
; the raw lisp defmacro of acl2::defconst.

  (let* ((template (defstobj-template name args))
         (init (defstobj-raw-init template))
         (the-live-name (the-live-var name)))
    `(progn

; We place the defvar above the subsequent let*, in order to avoid
; warnings in Lisps such as CCL that compile on-the-fly.

       (defvar ,the-live-name)
       (let* ((template ',template)
              (boundp (boundp ',the-live-name))
              (d (and boundp
                      (get ',the-live-name
                           'redundant-raw-lisp-discriminator)))

; d is expected to be of the form (DEFSTOBJ namep creator . field-templates)

              (ok-p (and boundp
                         (consp d)
                         (eq (car d) 'defstobj)
                         (consp (cdr d))
                         (eq (cadr d) (car template))
                         (consp (cddr d))
                         (eq (caddr d) (cadr template))
                         (equal (cdddr d) (caddr template))

; We also formerly required:

;                        (stobj-initial-statep (symbol-value ',the-live-name)
;                                              (caddr template))

; However, the stobj need not have its initial value; consider a redundant
; defstobj in a book whose certification world has already modified the stobj,
; or a defstobj in a book whose value is modified in a make-event later in that
; book.  Either way, ok-p would be false when this code is executed by loading
; the compiled file.

                         )))
         (cond
          (ok-p ',name)
          ((and boundp (not (raw-mode-p *the-live-state*)))
           (interface-er
            "Illegal attempt to redeclare the single-threaded object ~s0."
            ',name))
          (t

; Now we lay down the defuns of the recognizers, accessors and updaters as
; generated by defstob-raw-defs.  The boilerplate below just adds the DEFUN to
; the front of each def generated, preserving the order of the defs as
; generated.  We deal here with the :inline case; note that
; *stobj-inline-declare* was added in defstobj-field-fns-raw-defs.

           ,@(mapcar (function (lambda (def)
                                 (if (member-equal *stobj-inline-declare* def)
                                     (cons 'DEFABBREV
                                           (remove-stobj-inline-declare def))
                                   (cons 'DEFUN def))))
                     (defstobj-raw-defs name template (w *the-live-state*)))

           ,@(defstobj-defconsts (strip-accessor-names (caddr template)) 0)
           (defparameter ,the-live-name ,init)
           (setf (get ',the-live-name 'redundant-raw-lisp-discriminator)
                 (list* 'defstobj (car template) (cadr template)
                        (caddr template)))
           (let ((old (and boundp

; Since boundp, we also know (raw-mode-p state).  This boundp test could be
; omitted, since otherwise we know that the assoc below will return nil; it is
; just an optimization.

                           (assoc-eq ',name *user-stobj-alist*))))
             (cond
              (old ; hence raw-mode
               (fms "Note:  Redefining and reinitializing stobj ~x0 in raw ~
                   mode.~%"
                    (list (cons #\0 ',name))
                    (standard-co *the-live-state*) *the-live-state* nil)
               (setf (cdr old)
                     (symbol-value ',the-live-name)))
              (t
               (assert$
                (not (assoc-eq ',name *user-stobj-alist*))
                (setq *user-stobj-alist*
                      (cons (cons ',name (symbol-value ',the-live-name))
                            *user-stobj-alist*))))))
           ',name))))))

(defmacro value-triple (&rest args)
  (declare (ignore args))
  nil)

(defmacro verify-termination-boot-strap (&rest args)
  (declare (ignore args))
  nil)

(defmacro verify-guards (&rest args)
  (declare (ignore args))
  nil)

(defmacro in-theory (&rest args)
  (declare (ignore args))
  nil)

(defmacro in-arithmetic-theory (&rest args)
  (declare (ignore args))
  nil)

(defmacro push-untouchable (&rest args)
  (declare (ignore args))
  nil)

(defmacro remove-untouchable (&rest args)
  (declare (ignore args))
  nil)

(defmacro set-body (&rest args)
  (declare (ignore args))
  nil)

(defmacro table (&rest args)

; Note: The decision to make table a no-op in compiled files was not
; taken lightly.  But table, like defthm, has no effect on the logic.
; Indeed, like defthm, table merely modifies the world and if it is
; permitted in compiled code to ignore defthm's effects on the world
; then so too the effects of table.

  (declare (ignore args))
  nil)

(defmacro encapsulate (signatures &rest lst)

; The code we generate for the constrained functions in signatures is
; the same (except, possibly, for the formals) as executed in
; extend-world1 when we introduce an undefined function.

; Sig below may take on any of several forms, illustrated by
; the examples:

; ((fn * * $S * STATE) => (MV * STATE))
; (fn (x y $S z STATE)    (MV t STATE))
; (fn (x y $S z STATE)    (MV t STATE) :stobjs ($S))

; Because the first form above does not provide explicit formals, we
; generate them with gen-formals-from-pretty-flags when we process
; ENCAPSULATE in the logic.  So what do we do here in raw Lisp when an
; encapsulate is loaded?  We ignore all but the arity and generate (x1
; x2 ... xn).  We did not want to have to include
; gen-formals-from-pretty-flags in the toothbrush model.

  `(progn ,@(mapcar
             (function
              (lambda (sig)
                (let* ((fn (if (consp (car sig)) (caar sig) (car sig)))
                       (formals
                        (if (consp (car sig))
                            (let ((i 0))
                              (mapcar (function
                                       (lambda (v)
                                         (declare (ignore v))
                                         (setq i (+ 1 i))
                                         (intern (format nil "X~a" i)
                                                 "ACL2")))
                                      (cdar sig)))
                          (cadr sig))))
                  (list 'defun fn formals
                        (null-body-er fn formals t)))))
             signatures)
          ,@lst))

(defparameter *inside-include-book-fn*

; We trust include-book-fn and certify-book-fn to take care of all include-book
; processing without any need to call the raw Lisp include-book.  It seems that
; the only way this could be a bad idea is if include-book or certify-book
; could be called from a user utility, rather than at the top level, while
; inside a call of include-book-fn or certify-book-fn.  We disallow this in
; translate11.

  nil)

(defmacro include-book (user-book-name
                        &key
                        (load-compiled-file ':default)
                        uncertified-okp
                        defaxioms-okp
                        skip-proofs-okp
                        ttags
                        dir
                        doc)
  (declare (ignore uncertified-okp defaxioms-okp skip-proofs-okp ttags doc))
  `(include-book-raw ,user-book-name nil ,load-compiled-file ,dir
                     '(include-book . ,user-book-name)
                     *the-live-state*))

(defmacro certify-book (&rest args)
  (declare (ignore args))

; Unlike the embedded event forms such as DEFTHM, it is safe to cause an error
; here.  We want embedded event forms such as DEFTHM to be quietly ignored
; when books are included, but CERTIFY-BOOK is not an embedded event form, so
; it has no business being called from raw Lisp.

  (interface-er "Apparently you have called CERTIFY-BOOK from outside the ~
                 top-level ACL2 loop.  Perhaps you need to call (LP) first."))

(defmacro local (x)
  (declare (ignore x))
  nil)

(defmacro defchoose (&rest args)
  (let ((free-vars (caddr args)))
    `(defun ,(car args) ,free-vars
       ,(null-body-er (car args) free-vars nil))))

; Although defuns provides us conceptually with the right function for
; packaging together mutually recursive functions, we never use it
; because it hides things from standard Lisp editor indexing programs
; such as etags.  Instead, we use mutual-recursion.

(defmacro mutual-recursion (&rest lst)
  (cons 'progn lst))

(defmacro make-event (&whole event-form
                             form
                             &key check-expansion on-behalf-of)
  (declare (ignore form on-behalf-of))
  (cond ((consp check-expansion)
         check-expansion)
        (t `(error ; not er; so certify-book and include-book fail
             "It is illegal to execute make-event in raw Lisp (including ~%~
              raw mode) unless :check-expansion is a cons, which represents ~%~
              the expected expansion.  If this error occurs when executing ~%~
              an include-book form in raw mode or raw Lisp, consider loading a ~%~
              corresponding file *@expansion.lsp instead; see :DOC ~%~
              certify-book.  If you are not in raw Lisp, then this is an ~%~
              ACL2 bug; please contact the ACL2 implementors and report the ~%~
              offending form:~%~%~s~%"
             ',event-form))))
)

;                          STANDARD CHANNELS

(deflabel programming

; Be sure to include documentation for all functions in
; primitive-formals-and-guards.

  :doc
  ":Doc-Section Programming

  built-in ACL2 functions~/

  The built-in ACL2 functions that one typically uses in writing programs are
  listed below.  See their individual ~il[documentation]s.  We do not bother to
  document some of the more obscure functions provided by ACL2 that do not
  correspond to functions of Common Lisp.~/

  See any documentation for Common Lisp for more details on many of these
  functions.~/")

(deflabel miscellaneous
  :doc
  ":Doc-Section Miscellaneous

  a miscellany of documented functions and concepts
                 (often cited in more accessible ~il[documentation])~/~/

  Perhaps as the system matures this section will become more
  structured.~/")

(defconst *standard-co* 'acl2-output-channel::standard-character-output-0
  ":Doc-Section ACL2::Programming

  the ACL2 analogue of CLTL's ~c[*standard-output*]~/

  The value of the ACL2 constant ~c[*standard-co*] is an open character
  output channel that is synonymous to Common Lisp's
  ~c[*standard-output*].~/

  ACL2 character output to ~c[*standard-co*] will go to the stream named
  by Common Lisp's ~c[*standard-output*].  That is, by changing the
  setting of ~c[*standard-output*] in raw Common Lisp you can change the
  actual destination of ACL2 output on the channel named by
  ~c[*standard-co*].  Observe that this happens without changing the
  logical value of ~c[*standard-co*] (which is some channel symbol).
  Changing the setting of ~c[*standard-output*] in raw Common Lisp
  essentially just changes the map that relates ACL2 to the physical
  world of terminals, files, etc.

  To see the value of this observation, consider the following.
  Suppose you write an ACL2 function which does character output to
  the constant channel ~c[*standard-co*].  During testing you see that the
  output actually goes to your terminal.  Can you use the function to
  output to a file?  Yes, if you are willing to do a little work in
  raw Common Lisp: open a stream to the file in question, set
  ~c[*standard-output*] to that stream, call your ACL2 function, and then
  close the stream and restore ~c[*standard-output*] to its nominal value.
  Similar observations can be made about the two ACL2 input channels,
  ~ilc[*standard-oi*] and ~ilc[*standard-ci*], which are analogues of
  ~c[*standard-input*].

  Another reason you might have for wanting to change the actual
  streams associated with ~ilc[*standard-oi*] and ~c[*standard-co*] is to drive
  the ACL2 top-level loop, ~ilc[ld], on alternative input and output
  streams.  This end can be accomplished easily within ACL2 by either
  calling ~ilc[ld] on the desired channels or file names or by resetting the
  ACL2 ~ilc[state] global variables ~c[']~ilc[standard-oi] and ~c[']~ilc[standard-co] which are
  used by ~ilc[ld].  ~l[standard-oi] and ~pl[standard-co].")

(defconst *standard-oi* 'acl2-input-channel::standard-object-input-0
  ":Doc-Section ACL2::Programming

  an ACL2 object-based analogue of CLTL's ~c[*standard-input*]~/

  The value of the ACL2 constant ~c[*standard-oi*] is an open object input
  channel that is synonymous to Common Lisp's ~c[*standard-input*].~/

  ACL2 object input from ~c[*standard-oi*] is actually obtained by reading
  from the stream named by Common Lisp's ~c[*standard-input*].  That is,
  by changing the setting of ~c[*standard-input*] in raw Common Lisp you
  can change the source from which ACL2 reads on the channel
  ~c[*standard-oi*].  ~l[*standard-co*].")

(defconst *standard-ci* 'acl2-input-channel::standard-character-input-0
  ":Doc-Section ACL2::Programming

  an ACL2 character-based analogue of CLTL's ~c[*standard-input*]~/

  The value of the ACL2 constant ~c[*standard-ci*] is an open character
  input channel that is synonymous to Common Lisp's
  ~c[*standard-input*].~/

  ACL2 character input from ~c[*standard-ci*] is actually obtained by
  reading ~il[characters] from the stream named by Common Lisp's
  ~c[*standard-input*].  That is, by changing the setting of
  ~c[*standard-input*] in raw Common Lisp you can change the source from
  which ACL2 reads on the channel ~c[*standard-ci*].
  ~l[*standard-co*].")


;                            IF and EQUAL

; Convention:  when a term t is used as a formula it means
; (not (equal t nil))

; The following four axioms define if and equal but are not expressed
; in the ACL2 language.

;         (if NIL y z) = z

; x/=NIL -> (if x y z) = y

; (equal x x) = T

; x/=y -> (equal x y) = NIL


;                               LOGIC

#+acl2-loop-only
(defconst nil 'nil

; We cannot document a NIL symbol.

 " NIL, a symbol, represents in Common Lisp both the false truth value
 and the empty list.")

#+acl2-loop-only
(defconst t 't

; We cannot document a NIL symbol.  So, we do not document T either.

  "T, a symbol, represents the true truth value in Common Lisp.")

(defun iff (p q)

  ":Doc-Section ACL2::Programming

  logical ``if and only if''~/

  ~c[Iff] is the ACL2 biconditional, ``if and only if''.  ~c[(iff P Q)]
  means that either ~c[P] and ~c[Q] are both false (i.e., ~c[nil]) or both true
  (i.e., not ~c[nil]).~/~/"

  (declare (xargs :guard t))
  (if p (if q t nil) (if q nil t)))

(defun xor (p q)

  ":Doc-Section ACL2::Programming

  logical ``exclusive or''~/

  ~c[Xor] is the ACL2 exclusive-or function.  ~c[(xor P Q)] means that either
  ~c[P] or ~c[Q], but not both, is false (i.e., ~c[nil]).~/~/"

  (declare (xargs :guard t))
  (if p (if q nil t) (if q t nil)))

(defun booleanp (x)

  ":Doc-Section ACL2::Programming

  recognizer for booleans~/

  ~c[(Booleanp x)] is ~c[t] if ~c[x] is ~c[t] or ~c[nil], and is ~c[nil] otherwise.~/

  ~l[generalized-booleans] for a discussion of a potential
  soundness problem for ACL2 related to the question:  Which Common
  Lisp functions are known to return Boolean values?~/"

  (declare (xargs :guard t))
  (if (equal x t)
      t
    (equal x nil)))

; We do not want to try to define defequiv at this point, so we use the
; expansion of (defequiv iff).

(defthm iff-is-an-equivalence
  (and (booleanp (iff x y))
       (iff x x)
       (implies (iff x y) (iff y x))
       (implies (and (iff x y) (iff y z))
                (iff x z)))
  :rule-classes (:equivalence))

(defun implies (p q)

  ":Doc-Section ACL2::Programming

  logical implication~/

  ~c[Implies] is the ACL2 implication function.  ~c[(implies P Q)] means
  that either ~c[P] is false (i.e., ~c[nil]) or ~c[Q] is true (i.e., not ~c[nil]).~/~/"

  (declare (xargs :mode :logic :guard t))
  (if p (if q t nil) t))

(defthm iff-implies-equal-implies-1
  (implies (iff y y-equiv)
           (equal (implies x y) (implies x y-equiv)))
  :rule-classes (:congruence))

(defthm iff-implies-equal-implies-2
  (implies (iff x x-equiv)
           (equal (implies x y) (implies x-equiv y)))
  :rule-classes (:congruence))

#+acl2-loop-only
(defun not (p)

  ":Doc-Section ACL2::Programming

  logical negation~/

  ~c[Not] is the ACL2 negation function.  The negation of ~c[nil] is ~c[t] and
  the negation of anything else is ~c[nil].~/

  ~c[Not] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

 (declare (xargs :mode :logic :guard t))
 (if p nil t))

(defthm iff-implies-equal-not
  (implies (iff x x-equiv)
           (equal (not x) (not x-equiv)))
  :rule-classes (:congruence))

(defun hide (x)

  ":Doc-Section Miscellaneous

  hide a term from the rewriter~/

  ~c[Hide] is actually the ~il[identity] function:  ~c[(hide x) = x] for
  all ~c[x].  However, terms of the form ~c[(hide x)] are ignored by the
  ACL2 rewriter, except when explicit ~c[:expand] ~il[hints] are given
  for such terms (~pl[hints]) or when rewrite rules explicitly
  about ~c[hide] are available.  An ~c[:expand] hint that removes all
  calls of ~c[hide] is:
  ~bv[]
  :expand ((:free (x) (hide x)))
  ~ev[]
  The above hint can be particularly useful when ACL2's equality heuristics
  apply ~c[hide] to an equality after substituting it into the rest of the
  goal, if that goal (or a subgoal of it) fails to be proved.

  ~c[Hide] terms are also ignored by the induction heuristics.~/

  Sometimes the ACL2 simplifier inserts ~c[hide] terms into a proof
  attempt out of the blue, as it were.  Why and what can you do about
  it?  Suppose you have a constrained function, say ~c[constrained-fn], and
  you define another function, say ~c[another-fn], in terms of it, as in:
  ~bv[]
  (defun another-fn (x y z)
    (if (big-hairy-test x y z)
        (constrained-fn x y z)
        t))
  ~ev[]
  Suppose the term ~c[(another-fn 'a 'b 'c)] arises in a proof.  Since
  the arguments are all constants, ACL2 will try to reduce such a term
  to a constant by executing the definition of ~c[another-fn].
  However, after a possibly extensive computation (because of
  ~c[big-hairy-test]) the execution fails because of the unevaluable
  call of ~c[constrained-fn].  To avoid subsequent attempts to evaluate
  the term, ACL2 embeds it in a ~c[hide] expression, i.e., rewrites it
  to ~c[(hide (another-fn 'a 'b 'c))].

  You might think this rarely occurs since all the arguments of
  ~c[another-fn] must be constants.  You would be right except for one
  special case:  if ~c[another-fn] takes no arguments, i.e., is a
  constant function, then every call of it fits this case.  Thus, if
  you define a function of no arguments in terms of a constrained
  function, you will often see ~c[(another-fn)] rewrite to
  ~c[(hide (another-fn))].

  We do not hide the term if the executable counterpart of the
  function is disabled -- because we do not try to evaluate it in the
  first place.  Thus, to prevent the insertion of a ~c[hide] term into
  the proof attempt, you can globally disable the executable
  counterpart of the offending defined function, e.g.,
  ~bv[]
  (in-theory (disable (:executable-counterpart another-fn))).
  ~ev[]

  It is conceivable that you cannot afford to do this:  perhaps some
  calls of the offending function must be computed while others cannot
  be.  One way to handle this situation is to leave the executable
  counterpart enabled, so that ~c[hide] terms are introduced on the
  calls that cannot be computed, but prove explicit :~ilc[rewrite]
  rules for each of those ~c[hide] terms.  For example, suppose that in
  the proof of some theorem, thm, it is necessary to leave the
  executable counterpart of ~c[another-fn] enabled but that the call
  ~c[(another-fn 1 2 3)] arises in the proof and cannot be computed.
  Thus the proof attempt will introduce the term
  ~c[(hide (another-fn 1 2 3))].  Suppose that you can show that
  ~c[(another-fn 1 2 3)] is ~c[(contrained-fn 1 2 3)] and that such
  a step is necessary to the proof.  Unfortunately, proving the rewrite
  rule
  ~bv[]
  (defthm thm-helper
    (equal (another-fn 1 2 3) (constrained-fn 1 2 3)))
  ~ev[]
  would not help the proof of thm because the target term is hidden
  inside the ~c[hide].  However,
  ~bv[]
  (defthm thm-helper
    (equal (hide (another-fn 1 2 3)) (constrained-fn 1 2 3)))
  ~ev[]
  would be applied in the proof of thm and is the rule you should
  prove.

  Now to prove ~c[thm-helper] you need to use the two ``tricks'' which
  have already been discussed.  First, to eliminate the ~c[hide] term
  in the proof of ~c[thm-helper] you should include the hint
  ~c[:expand] ~c[(hide (another-fn 1 2 3))].  Second, to prevent the
  ~c[hide] term from being reintroduced when the system tries and fails
  to evaluate ~c[(another-fn 1 2 3)] you should include the hint
  ~c[:in-theory] ~c[(disable (:executable-counterpart another-fn))].
  Thus, ~c[thm-helper] will actually be:
  ~bv[]
  (defthm thm-helper
    (equal (hide (another-fn 1 2 3)) (constrained-fn 1 2 3))
    :hints
    ((\"Goal\" :expand (hide (another-fn 1 2 3))
             :in-theory (disable (:executable-counterpart another-fn)))))
  ~ev[]

  ~l[eviscerate-hide-terms] for how to affect the printing of ~c[hide]
  terms."

  (declare (xargs :guard t))
  x)

(defun rewrite-equiv (x)

; Documentation to be written.  This is experimental for Version_3.1, to be
; tried out by Dave Greve.

  (declare (xargs :mode :logic :guard t))
  x)

; As of ACL2 Version_2.5, we can compile with or without support for
; non-standard analysis.  To make maintenance of the two versions simpler,
; we define the macro "real/rationalp" which is defined as either realp or
; rationalp depending on whether the reals exist in the current ACL2
; universe or not.

(defmacro real/rationalp (x)

  ":Doc-Section ACL2::Programming

  recognizer for rational numbers (including real number in ACL2(r))~/

  For most ACL2 users, this is a macro abbreviating ~ilc[rationalp].  In
  ACL2(r) (~pl[real]), this macro abbreviates the predicate ~c[realp],
  which holds for real numbers as well (including rationals).  Most
  ACL2 users can ignore this macro and use ~ilc[rationalp] instead,
  but many books in the ACL2 distribution use ~c[real/rationalp] so that
  these books will be suitable for ACL2(r) as well.~/~/"

  #+:non-standard-analysis
  `(realp ,x)
  #-:non-standard-analysis
  `(rationalp ,x))

(defmacro complex/complex-rationalp (x)

  ":Doc-Section ACL2::Programming

  recognizer for complex numbers~/

  For most ACL2 users, this is a macro abbreviating ~c[complex-rationalp];
  ~pl[complex-rationalp].  In  ACL2(r) (~pl[real]), a complex number ~c[x]
  may have irrational real and imaginary parts.  This macro
  abbreviates the predicate ~c[complexp] in ACL2(r), which holds for such
  ~c[x].  Most ACL2 users can ignore this macro and use ~ilc[complex-rationalp]
  instead.  Some books in the ACL2 distribution use
  ~c[complex/complex-rationalp] so that they are suitable for ACL2(r) as
  well.~/~/"

  #+:non-standard-analysis
  `(complexp ,x)
  #-:non-standard-analysis
  `(complex-rationalp ,x))

; Comments labeled "RAG" are from Ruben Gamboa, pertaining to his work
; in creating ACL2(r) (see :doc real).

#+acl2-loop-only
(defun eq (x y)

  ":Doc-Section ACL2::Programming

  equality of symbols~/

  ~c[Eq] is the function for determining whether two objects are
  identical (i.e., have the exact same store address in the current
  von Neumann implementation of Common Lisp).  It is the same as
  ~ilc[equal] in the ACL2 logic.~/

  ~c[Eq] is a Common Lisp function.  In order to ensure conformance
  with Common Lisp, the ACL2 ~il[guard] on ~c[eq] requires at least one of
  the arguments to ~c[eq] to be a symbol.  Common Lisp guarantees that
  if ~c[x] is a symbol, then ~c[x] is ~c[eq] to ~c[y] if and only if ~c[x]
  is ~ilc[equal] to ~c[y].  Thus, the ACL2 user should think of ~c[eq] as
  nothing besides a fast means for checking ~ilc[equal] when one argument
  is known to be a symbol.  In particular, it is possible that an
  ~c[eq] test will not even require the cost of a function call but
  will be as fast as a single machine instruction.~/"

  (declare (xargs :guard (if (symbolp x)
                             t
                           (symbolp y))
                  :mode :logic :verify-guards t))
  (equal x y))

(defun true-listp (x)

  ":Doc-Section ACL2::Programming

  recognizer for proper (null-terminated) lists~/

  ~c[True-listp] is the function that checks whether its argument is a
  list that ends in, or equals, ~c[nil].~/~/"

  (declare (xargs :guard t :mode :logic))
  (if (consp x)
      (true-listp (cdr x))
    (eq x nil)))

(defun list-macro (lst)
  (declare (xargs :guard t))
  (if (consp lst)
      (cons 'cons
            (cons (car lst)
                  (cons (list-macro (cdr lst)) nil)))
      nil))

#+acl2-loop-only
(defmacro list (&rest args)

  ":Doc-Section ACL2::Programming

  build a list~/

  ~c[List] is the macro for building a list of objects.  For example,
  ~c[(list 5 6 7)] returns a list of length 3 whose elements are ~c[5],
  ~c[6], and ~c[7] respectively.  Also ~pl[list*].~/

  ~c[List] is defined in Common Lisp.  See any Common Lisp documentation
  for more information.~/"

  (list-macro args))

(defun and-macro (lst)
  (declare (xargs :guard t))
  (if (consp lst)
      (if (consp (cdr lst))
          (list 'if (car lst)
                (and-macro (cdr lst))
                nil)
        (car lst))
    t))

#+acl2-loop-only
(defmacro and (&rest args)

  ":Doc-Section ACL2::Programming

  conjunction~/

  ~c[And] is the macro for conjunctions.  ~c[And] takes any number of
  arguments.  ~c[And] returns ~c[nil] if one of the arguments is ~c[nil],
  but otherwise returns the last argument.  If there are no arguments,
  ~c[and] returns ~c[t].~/

  ~c[And] is a Common Lisp macro.  See any Common Lisp documentation
  for more information.~/"

 (and-macro args))

(defun or-macro (lst)
  (declare (xargs :guard t))
  (if (consp lst)
      (if (consp (cdr lst))
          (list 'if
                (car lst)
                (car lst)
                (or-macro (cdr lst)))
        (car lst))
    nil))

#+acl2-loop-only
(defmacro or (&rest args)

  ":Doc-Section ACL2::Programming

  disjunction~/

  ~c[Or] is the macro for disjunctions.  ~c[Or] takes any number of
  arguments and returns the first that is non-~c[nil], or ~c[nil] if
  there is no non-~c[nil] element.~/

  In the ACL2 logic, the macroexpansion of ~c[(or x y)] is an ~c[IF] term that
  appears to cause ~c[x] to be evaluated twice:
  ~bv[]
  ACL2 !>:trans (or x y)

  (IF X X Y)

  => *

  ACL2 !>
  ~ev[]
  If ~c[x] were replaced by an expression whose evaluation takes a long time,
  then such an expansion would be ineffecient.  However, don't be fooled: you
  can expect Common Lisp implementations to avoid this problem, say by
  generating a new variable, for example:
  ~bv[]
  ACL2 !>:q ; Exit the ACL2 loop and go into raw Common Lisp

  Exiting the ACL2 read-eval-print loop.  To re-enter, execute (LP).
  ACL2>(macroexpand '(or x y))

  (LET ((#:G5374 X)) (IF #:G5374 #:G5374 Y))
  T

  ACL2>
  ~ev[]

  ~c[Or] is a Common Lisp macro.  See any Common Lisp documentation
  for more information.~/"

   (or-macro args))

#+acl2-loop-only
(defmacro - (x &optional (y 'nil binary-casep))

; In the general case, (- x y) expands to (binary-+ x (unary-- y)).  But in the
; special case that y is a numeric constant we go ahead and run the unary--
; and we put it in front of x in the binary-+ expression so that it is in the
; expected "normal" form.  Thus, (- x 1) expands to (binary-+ -1 x).  Two forms
; of y allow this "constant folding": explicit numbers and the quotations of
; explicit numbers.

; Constant folding is important in processing definitions.  If the user has
; written (1- x), we translate that to (binary-+ -1 x) instead of to the more
; mechanical (binary-+ (unary-- 1) x).  Note that the type of the former is
; easier to determine that the latter because type-set knows about the effect
; of adding the constant -1 to a positive, but not about adding the term (- 1).

  (if binary-casep

; First we map 'n to n so we don't have so many cases.

      (let ((y (if (and (consp y)
                        (eq (car y) 'quote)
                        (consp (cdr y))
                        (acl2-numberp (car (cdr y)))
                        (eq (cdr (cdr y)) nil))
                   (car (cdr y))
                   y)))
        (if (acl2-numberp y)
            (cons 'binary-+
                  (cons (unary-- y)
                        (cons x nil)))
            (cons 'binary-+
                  (cons x
                        (cons (cons 'unary-- (cons y nil))
                              nil)))))
      (let ((x (if (and (consp x)
                        (eq (car x) 'quote)
                        (consp (cdr x))
                        (acl2-numberp (car (cdr x)))
                        (eq (cdr (cdr x)) nil))
                   (car (cdr x))
                   x)))
        (if (acl2-numberp x)
            (unary-- x)
            (cons 'unary-- (cons x nil))))))

(defthm booleanp-compound-recognizer
  (equal (booleanp x)
         (or (equal x t)
             (equal x nil)))
  :rule-classes :compound-recognizer)

(in-theory (disable booleanp))

; integer-abs is just abs if x is an integer and is 0 otherwise.
; integer-abs is used because we don't know that that (abs x) is a
; nonnegative integer when x is an integer.  By using integer-abs in
; the defun of acl2-count below we get that the type-prescription for
; acl2-count is a nonnegative integer.

(defun integer-abs (x)
  (declare (xargs :guard t))
  (if (integerp x)
      (if (< x 0) (- x) x)
      0))

(defun xxxjoin (fn args)

 " (xxxjoin fn args) spreads the binary function symbol fn over args, a list
 of arguments.  For example,

      (xxxjoin '+ '(1 2 3)) = '(+ 1 (+ 2 3)))."

  (declare (xargs :guard (if (true-listp args)
                             (cdr args)
                           nil)
                  :mode :program))
  (if (cdr (cdr args))
      (cons fn
            (cons (car args)
                  (cons (xxxjoin fn (cdr args))
                        nil)))
    (cons fn args)))

#+acl2-loop-only
(defmacro + (&rest rst)
  (if rst
      (if (cdr rst)
          (xxxjoin 'binary-+ rst)
          (cons 'binary-+ (cons 0 (cons (car rst) nil))))
      0))

; We now define length (and its subroutine len) so we can use them in
; acl2-count.

#-acl2-loop-only
(defun-one-output len2 (x acc)
  (cond ((atom x) acc)
        (t (len2 (cdr x) (1+ acc)))))

#-acl2-loop-only
(defun len1 (x acc)

; This function is an optimized version of len2 above, which is a simple
; tail-recursive implementation of len.

   (declare (type fixnum acc))
   (the fixnum ; to assist in ACL2's proclaiming
        (cond ((atom x) acc)
              ((eql (the fixnum acc) most-positive-fixnum)
               #+(or gcl ccl allegro sbcl cmu)

; The error below is entirely optional, and can be safely removed from the
; code.  Here is the story.

; We cause an error for the Lisps listed above in order to highlight the
; violation of the following expectation for those Lisps: the length of a list
; is always bounded by most-positive-fixnum.  To be safe, we omit CLISP and
; Lispworks (where most-positive-fixnum is only 16777215 and 8388607,
; respectively; see the Essay on Fixnum Declarations).  But for 32-bit versions
; of the Lisps in the above readtime conditional, we believe the above
; expectation because a cons takes at least 8 bytes and each of the lisps below
; has most-positive-fixnum of at least approximately 2^29.  This may need to be
; re-thought for 64-bit Lisps; however we are hopeful that most-positive-fixnum
; will still be sufficiently large to accommodate long lists.

               (error "We have encountered a list whose length exceeds ~
                       most-positive-fixnum!")
               -1)
              (t (len1 (cdr x) (the fixnum (+ (the fixnum acc) 1)))))))

(defun len (x)

  ":Doc-Section ACL2::Programming

  length of a list~/

  ~c[Len] returns the length of a list.~/

  A Common Lisp function that is appropriate for both strings and
  proper lists is ~c[length]; ~pl[length].  The guard for ~c[len] is ~c[t].

  (Low-level implementation note.  ACL2 provides a highly-optimized
  implementation of ~c[len], which is tail-recursive and fixnum-aware, that
  differs from its simple ACL2 definition.)~/"

  (declare (xargs :guard t :mode :logic))
  #-acl2-loop-only
  (return-from len
               (let ((val (len1 x 0)))
                 (if (eql val -1)
                     (len2 x 0)
                   val)))
  (if (consp x)
      (+ 1 (len (cdr x)))
      0))

#+acl2-loop-only
(defun length (x)

  ":Doc-Section ACL2::Programming

  length of a string or proper list~/

  ~c[Length] is the function for determining the length of a sequence.
  In ACL2, the argument is required to be either a ~ilc[true-listp] or a
  string.~/

  ~c[Length] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (if (true-listp x)
                             t
                             (stringp x))
                  :mode :logic))
  (if (stringp x)
      (len (coerce x 'list))
      (len x)))

#-acl2-loop-only
(defun-one-output complex-rationalp (x)
  (complexp x))

(defun acl2-count (x)

 ":Doc-Section Miscellaneous

  a commonly used measure for justifying recursion~/

  ~c[(Acl2-count x)] returns a nonnegative integer that indicates the
  ``size'' of its argument ~c[x].~/

  All ~il[characters] and symbols have ~c[acl2-count 0].  The ~c[acl2-count] of a
  string is the number of ~il[characters] in it, i.e., its length.  The
  ~c[acl2-count] of a ~ilc[cons] is one greater than the sum of the ~c[acl2-count]s
  of the ~ilc[car] and ~ilc[cdr].  The ~c[acl2-count] of an integer is its absolute
  value.  The ~c[acl2-count] of a rational is the sum of the ~c[acl2-count]s
  of the numerator and denominator.  The ~c[acl2-count] of a complex
  rational is one greater than the sum of the ~c[acl2-count]s of the real
  and imaginary parts."

; We used to define the acl2-count of symbols to be (+ 1 (length
; (symbol-name x))) but then found it useful to make the acl2-count of
; NIL be 0 so that certain normalizations didn't explode the count.
; We then made the count of all symbols 0.  This broad stroke was not
; strictly necessary, as far as we can see, it just simplifies the
; definition of acl2-count and does not seem to affect the common
; recursions and inductions.

  (declare (xargs :guard t))
  (if (consp x)
      (+ 1
         (acl2-count (car x))
         (acl2-count (cdr x)))
      (if (rationalp x)
          (if (integerp x)
              (integer-abs x)
              (+ (integer-abs (numerator x))
                 (denominator x)))
          (if (complex/complex-rationalp x)
              (+ 1
                 (acl2-count (realpart x))
                 (acl2-count (imagpart x)))
              (if (stringp x)
                  (length x)
                  0)))))

; The following rewrite rule may be useful for termination proofs, but
; at this point it seems premature to claim any kind of understanding
; of how to integrate such rules with appropriate linear rules.

#||
(defthm acl2-count-consp
  (implies (consp x)
           (equal (acl2-count x)
                  (+ 1
                     (acl2-count (car x))
                     (acl2-count (cdr x))))))
||#

(defun cond-clausesp (clauses)
  (declare (xargs :guard t))
  (if (consp clauses)
      (and (consp (car clauses))
           (true-listp (car clauses))
           (< (len (car clauses)) 3)
           (cond-clausesp (cdr clauses)))
    (eq clauses nil)))

(defun cond-macro (clauses)
  (declare (xargs :guard (cond-clausesp clauses)))
  (if (consp clauses)
      (if (and (eq (car (car clauses)) t)
               (eq (cdr clauses) nil))
          (if (cdr (car clauses))
              (car (cdr (car clauses)))
            (car (car clauses)))
        (if (cdr (car clauses))
            (list 'if
                  (car (car clauses))
                  (car (cdr (car clauses)))
                  (cond-macro (cdr clauses)))

; We could instead generate the IF term corresponding to the expansion of the
; following OR term, and that is what we did through Version_3.3.  But the
; extra cost of further expanding this OR call is perhaps outweighed by the
; advantage that tools using macroexpand1 can see the OR, which is an odd macro
; in that its logical expansion can result in evaluating the first argument
; twice.

          (list 'or
                (car (car clauses))
                (cond-macro (cdr clauses)))))
    nil))

#+acl2-loop-only
(defmacro cond (&rest clauses)

  ":Doc-Section ACL2::Programming

  conditional based on if-then-else~/

  ~c[Cond] is the construct for IF, THEN, ELSE IF, ...  The test is
  against ~c[nil].  The argument list for ~c[cond] is a list of
  ``clauses'', each of which is a list.  In ACL2, clauses must have
  length 1 or 2.~/

  ~c[Cond] is a Common Lisp macro.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (cond-clausesp clauses)))
  (cond-macro clauses))

; The function eqlablep is :common-lisp-compliant even during the first pass,
; in order to support the definition of eql, which is in
; *expandable-boot-strap-non-rec-fns* and hence needs to be
; :common-lisp-compliant.

(defun eqlablep (x)

  ":Doc-Section ACL2::Programming

  the ~il[guard] for the function ~ilc[eql]~/

  The predicate ~c[eqlablep] tests whether its argument is suitable for
  ~ilc[eql], at least one of whose arguments must satisfy this predicate
  in Common Lisp.  ~c[(Eqlablep x)] is true if and only if its argument
  is a number, a symbol, or a character.~/~/"

  (declare (xargs :mode :logic :guard t))
  (or (acl2-numberp x)
      (symbolp x)
      (characterp x)))

; Note: Eqlablep is the guard on the function eql.  Eql is on *expandable-boot-
; strap-non-rec-fns* and is hence expanded by type-set and assume-true-false
; when its guard is established.  Thus, the system works best if eqlablep is
; known to be a compound recognizer so that type-set can work with it when it
; sees it in the guard of eql.

(defthm eqlablep-recog
  (equal (eqlablep x)
         (or (acl2-numberp x)
             (symbolp x)
             (characterp x)))
  :rule-classes :compound-recognizer)

(in-theory (disable eqlablep))

(defun eqlable-listp (l)

  ":Doc-Section ACL2::Programming

  recognizer for a true list of objects each suitable for ~ilc[eql]~/

  The predicate ~c[eqlable-listp] tests whether its argument is a
  ~ilc[true-listp] of objects satisfying ~ilc[eqlablep].~/~/"

  (declare (xargs :guard t))
  (if (consp l)
      (and (eqlablep (car l))
           (eqlable-listp (cdr l)))
    (equal l nil)))

#+acl2-loop-only
(defun atom (x)

  ":Doc-Section ACL2::Programming

  recognizer for atoms~/

  ~c[(atom x)] is true if and only if ~c[x] is an atom, i.e., not a
  ~ilc[cons] pair.~/

  ~c[Atom] has a ~il[guard] of ~c[t], and is a Common Lisp function.  See any
  Common Lisp documentation for more information.~/"

 (declare (xargs :mode :logic :guard t))
 (not (consp x)))

; We use this in the *1* code for coerce.

(defun make-character-list (x)

  ":Doc-Section ACL2::Programming

  ~il[coerce] to a list of characters~/

  Non-characters in the given list are ~il[coerce]d to the character with
  code 0.~/~/"

  (declare (xargs :guard t))
  (cond ((atom x) nil)
        ((characterp (car x))
         (cons (car x) (make-character-list (cdr x))))
        (t

; There's nothing special about (code-char 0), but at least it will look
; strange when people come across it.

         (cons (code-char 0) (make-character-list (cdr x))))))

(defun eqlable-alistp (x)

  ":Doc-Section ACL2::Programming

  recognizer for a true list of pairs whose ~ilc[car]s are suitable for ~ilc[eql]~/

  The predicate ~c[eqlable-alistp] tests whether its argument is a
  ~ilc[true-listp] of ~ilc[consp] objects whose ~ilc[car]s all satisfy
  ~ilc[eqlablep].~/~/"

  (declare (xargs :guard t))
  (cond ((atom x) (equal x nil))
        (t (and (consp (car x))
                (eqlablep (car (car x)))
                (eqlable-alistp (cdr x))))))

(defun alistp (l)

  ":Doc-Section ACL2::Programming

  recognizer for association lists~/

  ~c[(alistp x)] is true if and only if ~c[x] is a list of ~ilc[cons] pairs.~/

  ~c[(alistp x)] has a ~il[guard] of ~c[t].~/"

  (declare (xargs :guard t))
  (cond ((atom l) (eq l nil))
        (t (and (consp (car l)) (alistp (cdr l))))))

(defthm alistp-forward-to-true-listp
  (implies (alistp x)
           (true-listp x))
  :rule-classes :forward-chaining)

(defthm eqlable-alistp-forward-to-alistp
  (implies (eqlable-alistp x)
           (alistp x))
  :rule-classes :forward-chaining)

#+acl2-loop-only
(defun acons (key datum alist)

  ":Doc-Section ACL2::Programming

  constructor for association lists~/

  ~c[(Acons key datum alist)] equals the result of consing the pair
  ~c[(cons key datum)] to the front of the association list ~c[alist].~/

  ~c[(Acons key datum alist)] has a ~il[guard] of ~c[(alistp alist)].
  ~c[Acons] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (alistp alist)))
  (cons (cons key datum) alist))

#+acl2-loop-only
(defun endp (x)

  ":Doc-Section ACL2::Programming

  recognizer for empty lists~/

  In the ACL2 logic, ~c[(endp x)] is the same as ~c[(atom x)].
  ~l[atom].~/

  Unlike ~ilc[atom], the ~il[guard] for ~c[endp] requires that ~c[x] is a
  ~ilc[cons] pair or is ~c[nil].  Thus, ~c[endp] is typically used as a
  termination test for functions that recur on a ~ilc[true-listp]
  argument.  ~l[guard] for general information about ~il[guard]s.

  ~c[Endp] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :mode :logic
                  :guard (or (consp x) (equal x nil))))
  (atom x))

#-acl2-loop-only
(defmacro must-be-equal (logic exec)

; We rely on this macroexpansion in raw Common Lisp.  See in particular the
; code and comment on must-be-equal in guard-clauses.

  (declare (ignore logic))
  exec)

#+acl2-loop-only
(defun must-be-equal (logic exec)

  ":Doc-Section ACL2::Programming

  attach code for execution~/~/

  The form ~c[(must-be-equal logic exec)] evaluates to ~c[logic] in the ACL2
  logic but evaluates to ~c[exec] in raw Lisp.  The point is to be able to
  write one definition to reason about logically but another for evaluation.
  Please ~pl[mbe] and ~pl[mbt] for appropriate macros to use, rather than
  calling ~c[must-be-equal] directly, since it is easy to commute the arguments
  of ~c[must-be-equal] by accident.

  The guard for ~c[(must-be-equal x y)] is ~c[(equal x y)]."

  (declare (ignore exec)
           (xargs :guard (equal logic exec)

; By setting the :mode to :logic, we ensure that this definition will be
; redundant in pass 2 of initialization.  That probably isn't important, but it
; could conceivably avoid difficulties in add-trip.

                  :mode :logic))
  logic)

(defmacro mbe (&key (exec 'nil exec-p) (logic 'nil logic-p))

  ":Doc-Section ACL2::Programming

  attach code for execution~/

  The macro ~c[mbe] (``must be equal'') can be used in function definitions in
  order to cause evaluation to use alternate code to that provided for the
  logic.  An example is given below.  However, the use of ~c[mbe] can lead to
  non-terminating computations.  ~l[defexec], perhaps after reading the present
  documentation, for a way to guarantee termination (at least theoretically).

  In the ACL2 logic, ~c[(mbe :exec exec-code :logic logic-code)] equals
  ~c[logic-code]; the value of ~c[exec-code] is ignored.  However, in raw Lisp
  it is the other way around:  this form macroexpands simply to ~c[exec-code].
  ACL2's ~il[guard] verification mechanism ensures that the raw Lisp code is
  only evaluated when appropriate, since the guard proof obligations generated
  for this call of ~c[mbe] are ~c[(equal exec-code logic-code)] together with
  the guard proof obligations from ~c[exec-code].  ~l[verify-guards] and, for
  general discussion of guards, ~pl[guard].

  Warning for nested ~ilc[mbe] calls: The equality of ~c[:exec] and ~c[:logic]
  code is not checked in the scope of superior ~c[:logic] code, as for the
  equality of ~c[x] and ~c[y] in the following.
  ~bv[]
  (mbe :logic 
       (mbe :logic x :exec y)
       :exec
       z)
  ~ev[]

  The form ~c[(mbe :exec exec-code :logic logic-code)] expands in the logic to
  the function call ~c[(]~ilc[must-be-equal]~c[ logic-code exec-code)].  The guard for
  ~c[(must-be-equal logic exec)] is ~c[(equal logic exec)].  We recommend that
  you use ~c[mbe] instead of ~ilc[must-be-equal] because the use of keywords
  eliminates errors caused by unintentially reversing the order of arguments.
  The ~c[:exec] and the ~c[:logic] code in an ~c[mbe] call must have the same
  return type; for example, one cannot return ~c[(]~ilc[mv]~c[ * *)] while the
  other returns just a single value.

  Also ~pl[mbt], which stands for ``must be true.''  You may find it more
  natural to use ~ilc[mbt] for certain applications, as described in its
  ~il[documentation].~/

  Here is an example of the use of ~c[mbe].  Suppose that you want to define
  factorial in the usual recursive manner, as follows.
  ~bv[]
  (defun fact (n)
    (if (zp n)
        1
      (* n (fact (1- n)))))
  ~ev[]
  But perhaps you want to be able to execute calls of ~c[fact] on large
  arguments that cause stack overflows, perhaps during proofs.  (This isn't a
  particularly realistic example, but it should serve.)  So, instead you can
  define this tail-recursive version of factorial:
  ~bv[]
  (defun fact1 (n acc)
    (declare (xargs :guard (and (integerp n) (>= n 0) (integerp acc))))
    (if (zp n)
        acc
      (fact1 (1- n) (* n acc))))
  ~ev[]
  We are now ready to define ~c[fact] using ~c[mbe].  Our intention is that
  logically, ~c[fact] is as shown in the first definition above, but that
  ~c[fact] should be executed by calling ~c[fact1].  Notice that we defer
  ~il[guard] verification, since we are not ready to prove the correspondence
  between ~c[fact1] and ~c[fact].
  ~bv[]
  (defun fact (n)
    (declare (xargs :guard (and (integerp n) (>= n 0))
                    :verify-guards nil))
    (mbe :exec  (fact1 n 1)
         :logic (if (zp n)
                    1
                  (* n (fact (1- n))))))
  ~ev[]
  Next, we prove the necessary correspondence lemmas.  Notice the inclusion of
  a standard book to help with the arithmetic reasoning.
  ~bv[]
  (include-book \"books/arithmetic/top-with-meta\")

  (defthm fact1-fact
    (implies (integerp acc)
             (equal (fact1 n acc)
                    (* acc (fact n)))))
  ~ev[]
  We may now do guard verification for ~c[fact], which will allow the execution
  of the raw Lisp ~c[fact] function, where the above ~c[mbe] call expands
  simply to ~c[(fact1 n 1)].
  ~bv[]
  (verify-guards fact)
  ~ev[]
  Now that guards have been verified, a trace of function calls illustrates
  that the evaluation of calls of ~c[fact] is passed to evaluation of calls of
  ~c[fact1].  The outermost call below is of the logical function stored for
  the definition of ~c[fact]; all the others are of actual raw Common Lisp
  functions.
  ~bv[]
  ACL2 !>(trace$ fact fact1)
  NIL
  ACL2 !>(fact 3)
  1> (ACL2_*1*_ACL2::FACT 3)
    2> (FACT 3)
      3> (FACT1 3 1)
        4> (FACT1 2 3)
          5> (FACT1 1 6)
            6> (FACT1 0 6)
            <6 (FACT1 6)
          <5 (FACT1 6)
        <4 (FACT1 6)
      <3 (FACT1 6)
    <2 (FACT 6)
  <1 (ACL2_*1*_ACL2::FACT 6)
  6
  ACL2 !>
  ~ev[]

  You may occasionally get warnings when you compile functions defined using
  ~c[mbe].  (For commands that invoke the compiler, ~pl[compilation].)  These
  can be inhibited by using an ~c[ignorable] ~ilc[declare] form.  Here is a
  simple but illustrative example.  Note that the declarations can optionally
  be separated into two ~ilc[declare] forms.
  ~bv[]
  (defun foo (x y)
    (declare (ignorable x)
             (xargs :guard (equal x y)))
    (mbe :logic x :exec y))
  ~ev[]

  Finally, we observe that when the body of a function contains a term of the
  form ~c[(mbe :exec exec-code :logic logic-code)], the user is very unlikely
  to see any logical difference than if this were replaced by ~c[logic-code].
  ACL2 takes various steps to ensure this.  For example, the proof obligations
  generated for admitting a function treat the above ~c[mbe] term simply as
  ~c[logic-code].  Function expansion, ~c[:use] ~il[hints],
  ~c[:]~ilc[definition] rules, generation of ~il[constraint]s for functional
  instantiation, and creation of rules of class ~c[:]~ilc[rewrite] and
  ~c[:]~ilc[forward-chaining] also treat ~c[mbe] calls as their ~c[:logic]
  code."

  (declare (xargs :guard (and exec-p logic-p))

; OpenMCL Versions 14.2 and 14.3 (CCL), CMUCL Version 19b, and SBCL Version
; 0.9.8 (other versions of these too, most likely) produce a warning with the
; following ignore declaration, but we don't know why.  We tried using
; "ignorable" instead of "ignore" for CMUCL, but that didn't help.  So we
; eliminate the ignore declaration in these cases with the #- directive below.

           #-(and (or ccl cmu sbcl) (not acl2-loop-only))
           (ignore exec-p logic-p))
  `(must-be-equal ,logic ,exec))

(defmacro mbt (x)

  ":Doc-Section ACL2::Programming

  introduce a test not to be evaluated~/

  The macro ~c[mbt] (``must be true'') can be used in order to add code in
  order to admit function definitions in ~c[:]~ilc[logic] mode, without paying
  a cost in execution efficiency.  Examples below illustrate its intended use.

  Semantically, ~c[(mbt x)] equals ~c[x].  However, in raw Lisp ~c[(mbt x)]
  ignores ~c[x] entirely, and macroexpands to ~c[t].  ACL2's ~il[guard]
  verification mechanism ensures that the raw Lisp code is only evaluated when
  appropriate, since a guard proof obligation ~c[(equal x t)] is generated.
  ~l[verify-guards] and, for general discussion of guards, ~pl[guard].

  Also ~pl[mbe], which stands for ``must be equal.''  Although ~c[mbt] is more
  natural in many cases, ~c[mbe] has more general applicability.  In fact,
  ~c[(mbt x)] is essentially defined to be ~c[(mbe :logic x :exec t)].~/

  We can illustrate the use of ~c[mbt] on the following generic example, where
  ~c[<g>], ~c[<test>], ~c[<rec-x>], and ~c[<base>] are intended to be terms
  involving only the variable ~c[x].
  ~bv[]
  (defun foo (x)
    (declare (xargs :guard <g>))
    (if <test>
        (foo <rec-x>)
      <base>))
  ~ev[]
  In order to admit this function, ACL2 needs to discharge the proof obligation
  that ~c[<rec-x>] is smaller than ~c[x], namely:
  ~bv[]
  (implies <test>
           (o< (acl2-count ~c[<rec-x>])
                (acl2-count x)))
  ~ev[]
  But suppose we need to know that ~c[<g>] is true in order to prove this.
  Since ~c[<g>] is only the ~il[guard], it is not part of the logical
  definition of ~c[foo].  A solution is to add the guard to the definition of
  ~c[foo], as follows.
  ~bv[]
  (defun foo (x)
    (declare (xargs :guard <g>))
    (if (mbt <g>)
        (if <test>
            (foo <rec-x>)
          <base>)
      nil))
  ~ev[]
  If we do this using ~c[<g>] rather than ~c[(mbt <g>)], then evaluation of
  every recursive call of ~c[foo] will cause the evaluation of (the appropriate
  instance of) ~c[<g>].  But since ~c[(mbt <g>)] expands to ~c[t] in raw Lisp,
  then once we verify the guards of ~c[foo], the evaluations of ~c[<g>] will be
  avoided (except at the top level, when we check the guard before allowing
  evaluation to take place in Common Lisp).

  Other times, the guard isn't the issue, but rather, the problem is that a
  recursive call has an argument that itself is a recursive call.  For example,
  suppose that ~c[<rec-x>] is of the form ~c[(foo <expr>)].  There is no way we
  can hope to discharge the termination proof obligation shown above.  A
  standard solution is to add some version of this test:
  ~bv[]
  (mbt (o< (acl2-count ~c[<rec-x>]) (acl2-count x)))
  ~ev[]
  Here is a specific example based on one sent by Vernon Austel.
  ~bv[]
  (defun recurX2 (n)
    (declare (xargs :guard (and (integerp n) (<= 0 n))
                    :verify-guards nil))
    (cond ((zp n) 0)
          (t (let ((call (recurX2 (1- n))))
               (if (mbt (< (acl2-count call) n))
                   (recurX2 call)
                 1 ;; this branch is never actually taken
                 )))))

  (defthm recurX2-0
   (equal (recurX2 n) 0))

  (verify-guards recurX2)
  ~ev[]
  If you ~c[(]~ilc[trace$]~c[ acl2-count)], you will see that evaluation of
  ~c[(recurX2 2)] causes several calls of ~ilc[acl2-count] before the
  ~ilc[verify-guards].  But this evaluation does not call ~c[acl2-count] after
  the ~c[verify-guards], because the ACL2 evaluation mechanism uses raw Lisp to
  do the evaluation, and the form ~c[(mbt (< (acl2-count call) n))]
  macroexpands to ~c[t] in Common Lisp.

  You may occasionally get warnings when you compile functions defined using
  ~c[mbt].  (For commands that invoke the compiler, ~pl[compilation].)  These
  can be inhibited by using an ~c[ignorable] ~ilc[declare] form.  Here is a
  simple but illustrative example.  Note that the declarations can optionally
  be separated into two ~ilc[declare] forms.
  ~bv[]
  (defun foo (x y)
    (declare (ignorable x)
             (xargs :guard (equal x t)))
    (and (mbt x) y))
  ~ev[]"

  `(must-be-equal ,x t))

(defun member-equal (x lst)

  ":Doc-Section ACL2::Programming

  membership predicate~/

  ~c[(Member-equal x lst)] equals the longest tail of ~c[lst] that
  begins with ~c[x], or else ~c[nil] if no such tail exists.~/

  ~c[(Member-equal x lst)] has a ~il[guard] of ~c[(true-listp lst)].
  ~c[Member-equal] has the same functionality as the Common Lisp
  function ~ilc[member], except that it uses the ~ilc[equal] function to
  test whether ~c[x] is the same as each successive element of ~c[lst].
  ~l[member] and ~pl[member-eq].~/"

  (declare (xargs :guard (true-listp lst)))
  #-acl2-loop-only ; for assoc-equal, Jared Davis found native assoc efficient
  (member x lst :test #'equal)
  #+acl2-loop-only
  (cond ((endp lst) nil)
        ((equal x (car lst)) lst)
        (t (member-equal x (cdr lst)))))

(defun union-equal (x y)

  ":Doc-Section ACL2::Programming

  union of two lists~/

  ~c[(Union-equal x y)] equals a list whose members
  (~pl[member-equal]) contains the members of ~c[x] and the members
  of ~c[y].  More precisely, the resulting list is the same as one
  would get by first deleting the members of ~c[y] from ~c[x], and then
  concatenating the result to the front of ~c[y].~/

  The ~il[guard] for ~c[union-equal] requires both arguments to be true
  lists.  Essentially, ~c[union-equal] has the same functionality as
  the Common Lisp function ~c[union], except that it uses the ~ilc[equal]
  function to test membership rather than ~ilc[eql].  However, we do not
  include the function ~c[union] in ACL2, because the Common Lisp
  language does not specify the order of the elements in the list that
  it returns.~/"

  (declare (xargs :guard (and (true-listp x) (true-listp y))))
  (cond ((endp x) y)
        ((member-equal (car x) y) (union-equal (cdr x) y))
        (t (cons (car x) (union-equal (cdr x) y)))))

(defun subsetp-equal (x y)

  ":Doc-Section ACL2::Programming

  check if all members of one list are members of the other~/

  ~c[(Subsetp-equal x y)] returns ~c[t] if every member of ~c[x] is a
  member of ~c[y], where membership is tested using ~ilc[member-equal].~/

  The ~il[guard] for ~c[subsetp-equal] requires both arguments to be true
  lists.  ~c[Subsetp-equal] has the same functionality as the Common Lisp
  function ~ilc[subsetp], except that it uses the ~ilc[equal] function to
  test membership rather than ~ilc[eql].~/"

  (declare (xargs :guard (and (true-listp y)
                              (true-listp x))))
  #-acl2-loop-only ; for assoc-eq, Jared Davis found native assoc efficient
  (subsetp x y :test #'equal)
  #+acl2-loop-only
  (cond ((endp x) t)
        ((member-equal (car x) y)
         (subsetp-equal (cdr x) y))
        (t nil)))

(defun symbol-listp (lst)

  ":Doc-Section ACL2::Programming

  recognizer for a true list of symbols~/

  The predicate ~c[symbol-listp] tests whether its argument is a
  true list of symbols.~/~/"

  (declare (xargs :guard t :mode :logic))
  (cond ((atom lst) (eq lst nil))
        (t (and (symbolp (car lst))
                (symbol-listp (cdr lst))))))

(defthm symbol-listp-forward-to-true-listp
  (implies (symbol-listp x)
           (true-listp x))
  :rule-classes :forward-chaining)

#+acl2-loop-only
(defun null (x)

  ":Doc-Section ACL2::Programming

  recognizer for the empty list~/

  ~c[Null] is the function that checks whether its argument is ~c[nil].
  For recursive definitions it is often preferable to test for the end
  of a list using ~ilc[endp] instead of ~c[null]; ~pl[endp].~/

  ~c[Null] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :mode :logic :guard t))
  (eq x nil))

(defun member-eq (x lst)

  ":Doc-Section ACL2::Programming

  membership predicate, using ~ilc[eq] as test~/

  ~c[(Member-eq x lst)] equals the longest tail of ~c[lst] that
  begins with ~c[x], or else ~c[nil] if no such tail exists.~/

  ~c[(Member-eq x lst)] is provably the same in the ACL2 logic as
  ~c[(member x lst)] and ~c[(member-equal x lst)], but it has a stronger
  ~il[guard] because it uses ~ilc[eq] for a more efficient test for whether
  ~c[x] is equal to a given member of ~c[lst].  Its ~il[guard] requires that
  ~c[lst] is a true list, and moreover, either ~c[x] is a symbol or
  ~c[lst] is a list of symbols.  ~l[member-equal] and
  ~pl[member].~/"

  (declare (xargs :guard (if (symbolp x)
                             (true-listp lst)
                           (symbol-listp lst))))
  #-acl2-loop-only ; for assoc-eq, Jared Davis found native assoc efficient
  (member x lst :test #'eq)
  #+acl2-loop-only
  (cond ((endp lst) nil)
        ((eq x (car lst)) lst)
        (t (member-eq x (cdr lst)))))

(defun subsetp-eq (x y)
  (declare (xargs :guard (and (true-listp x)
                              (true-listp y)
                              (or (symbol-listp x)
                                  (symbol-listp y)))))
  #-acl2-loop-only ; for assoc-eq, Jared Davis found native assoc efficient
  (subsetp x y :test #'eq)
  #+acl2-loop-only
  (cond ((endp x) t)
        ((member-eq (car x) y)
         (subsetp-eq (cdr x) y))
        (t nil)))

(defun symbol-alistp (x)

  ":Doc-Section ACL2::Programming

  recognizer for association lists with symbols as keys~/

  ~c[(Symbol-alistp x)] is true if and only if ~c[x] is a list of pairs
  of the form ~c[(cons key val)] where ~c[key] is a ~ilc[symbolp].~/~/"

  (declare (xargs :guard t))
  (cond ((atom x) (eq x nil))
        (t (and (consp (car x))
                (symbolp (car (car x)))
                (symbol-alistp (cdr x))))))

(defthm symbol-alistp-forward-to-eqlable-alistp
  (implies (symbol-alistp x)
           (eqlable-alistp x))
  :rule-classes :forward-chaining)

(defun assoc-eq (x alist)

  ":Doc-Section ACL2::Programming

  look up key in association list, using ~ilc[eq] as test~/

  ~c[(Assoc-eq x alist)] is the first member of ~c[alist] whose ~ilc[car]
  is ~c[x], or ~c[nil] if no such member exists.~/

  ~c[(Assoc-eq x alist)] is provably the same in the ACL2 logic as
  ~c[(assoc x alist)] and ~c[(assoc-equal x alist)], but it has a
  stronger ~il[guard] because it uses ~ilc[eq] for a more efficient test for
  whether ~c[x] is equal to a given key of ~c[alist].  Its ~il[guard]
  requires that ~c[alist] is an association list (~pl[alistp]), and
  moreover, either ~c[x] is a symbol or all keys of ~c[alist] are
  symbols, i.e., ~c[alist] is a ~ilc[symbol-alistp].~/"

  (declare (xargs :guard (if (symbolp x)
                             (alistp alist)
                           (symbol-alistp alist))))
  #-acl2-loop-only ; Jared Davis found efficiencies in using native assoc
  (assoc x alist :test #'eq)
  #+acl2-loop-only
  (cond ((endp alist) nil)
        ((eq x (car (car alist))) (car alist))
        (t (assoc-eq x (cdr alist)))))

(defun assoc-equal (x alist)

  ":Doc-Section ACL2::Programming

  look up key in association list~/

  ~c[(Assoc-equal x alist)] is the first member of ~c[alist] whose ~ilc[car]
  is ~c[x], or ~c[nil] if no such member exists.~/

  ~c[(Assoc-equal x alist)] has a ~il[guard] of ~c[(alistp alist)], and
  returns the first member of alist whose ~ilc[car] is ~c[x], or ~c[nil] if
  no such member exists.  Thus, ~c[assoc-equal] has the same
  functionality as the Common Lisp function ~ilc[assoc], except that it
  uses the ~ilc[equal] function to test whether ~c[x] is the same as each
  successive `key' of ~c[alist].  ~l[assoc] and ~pl[assoc-eq].~/"

  (declare (xargs :guard (alistp alist)))
  #-acl2-loop-only ; Jared Davis found efficiencies in using native assoc
  (assoc x alist :test #'equal)
  #+acl2-loop-only
  (cond ((endp alist) nil)
        ((equal x (car (car alist))) (car alist))
        (t (assoc-equal x (cdr alist)))))

(defun assoc-eq-equal-alistp (x)
  (declare (xargs :guard t))
  (cond ((atom x) (eq x nil))
        (t (and (consp (car x))
                (symbolp (car (car x)))
                (consp (cdr (car x)))
                (assoc-eq-equal-alistp (cdr x))))))

(defun assoc-eq-equal (x y alist)

; We look for a pair on alist of the form (x y . val) where we compare the
; first key using eq and the second using equal.  We return the pair or nil.
; The guard could be weakened so that if x is a symbol, then alist need only be
; a true-listp whose elements are of the form (x y . val).  But there seems to
; be little advantage in having such a guard, considering the case splits that
; it could induce.

  (declare (xargs :guard (assoc-eq-equal-alistp alist)))
  (cond ((endp alist) nil)
        ((and (eq (car (car alist)) x)
              (equal (car (cdr (car alist))) y))
         (car alist))
        (t (assoc-eq-equal x y (cdr alist)))))

(defun no-duplicatesp-equal (l)

  ":Doc-Section ACL2::Programming

  check for duplicates in a list (using ~c[equal] for equality)~/

  ~c[(no-duplicatesp-equal l)] is true if and only if no member of ~c[l]
  occurs twice in ~c[l].~/

  ~c[(no-duplicatesp-equal l)] has a ~il[guard] of ~c[(true-listp l)].
  Membership is tested using ~ilc[member-equal], hence using ~ilc[equal] as
  the test.~/"

  (declare (xargs :guard (true-listp l)))
  (cond ((endp l) t)
        ((member-equal (car l) (cdr l)) nil)
        (t (no-duplicatesp-equal (cdr l)))))


; Essay on Strip-cars -- To Tail Recur or not to Tail Recur?

; We have seen instances where strip-cdrs causes a segmentation fault
; because it overflows the stack.  We therefore decided to recode
; strip-cdrs in a tail-recursive way.  We therefore decided to do the
; same thing to strip-cars.  This essay is about strip-cars but the
; issues are the same for strip-cdrs, we believe.

; First, what is the longest list you can strip-cars without a
; segmentation fault.  The answer for

; GCL (GNU Common Lisp)  Version(2.2.1) Wed Mar 12 00:47:19 CST 1997

; is 74790, when the test form is (length (strip-cars test-lst)).
; Because our test forms below are a little more elaborate, we will do
; our tests on a list of length 74000:

#|
(defvar test-lst
  (loop for i from 1 to 74000 collect (cons i i)))
|#

; Just for the record, how long does it take to do strip-cars 30 times on
; this test-lst?  Answer: 6.190 seconds.

#|
(proclaim-form
 (defun test1 (n)
   (loop for i from 1 to n do (strip-cars test-lst))))

(compile 'test1)

(time (test1 30))
|#

; Now the obvious tail recursive version of strip-cars is:

#|
(proclaim-form
 (defun strip-cars2 (x a)
   (if (endp x)
       (reverse a)
     (strip-cars2 (cdr x) (cons (car (car x)) a)))))

(compile 'strip-cars2)

(proclaim-form
 (defun test2 (n)
   (loop for i from 1 to n do (strip-cars2 test-lst))))

(compile 'test2)

(time (test2 30))
|#

; This function is actually faster than strip-cars: 5.530 seconds!
; That is surprising because this function does TWICE as many conses,
; since it conses up the final answer from the accumulated partial
; one.  The reason this function beats strip-cars can only be that
; that the tail-recursive jump is quite a lot faster than a function
; call.

; But Common Lisp allows to avoid consing to do a reverse if we are
; willing to smash the existing spine.  And in this case we are, since
; we have just consed it up.  So here is a revised function that only
; does as many conses as strip-cars:

#|
(proclaim-form
 (defun strip-cars3 (x a)
   (if (endp x)
       (nreverse a)   ;;; Note destructive reverse!
     (strip-cars3 (cdr x) (cons (car (car x)) a)))))

(compile 'strip-cars3)

(proclaim-form
 (defun test3 (n)
   (loop for i from 1 to n do (strip-cars3 test-lst))))

(compile 'test3)

(time (test3 30))
|#

; This function takes 2.490 seconds.

; Therefore, we decided to code strip-cars (and strip-cdrs) in the
; style of strip-cars3 above.  Henceforth, we call that function
; strip-cars1.  (Sorry for the confusion.)

; However, we did not want to do
; (defun strip-cars (x) (strip-cars1 x nil))
; because proofs about strip-cars -- both in our system build and in
; user theorems about strip-cars -- would have to use the
; accumulator-style generalization.  So we decided to keep strip-cars
; defined, logically, just as it was and to make its #-acl2-loop-only
; executable code be that for strip-cars1.

; The next paragraph is bogus!  But it used to read:

;  Furthermore, we decided that strip-cars1 is a perfectly nice
;  function the user might want, so we added it to the logic first --
;  changing the nreverse to a reverse for logical purposes but leaving
;  the nreverse in for execution.  This way, if the user wants an
;  accumulator-version of strip-cars, he can have it and it will be
;  very fast.  But if he wants a simple recursive version he can have
;  it too.

; That is unsound because we don't know that the accumulator is all
; new conses and so we can't smash it!  So strip-cars1 is hidden
; from the user.

#-acl2-loop-only
(defun-one-output strip-cars1 (x a)

; WARNING: THIS PROGRAM IS DESTRUCTIVE.  DO NOT USE THIS PROGRAM
; UNLESS YOU KNOW WHAT YOU'RE DOING!  IT SMASHES THE CONSES IN THE
; SECOND ARGUMENT!

  (cond ((endp x) (nreverse a))
        (t (strip-cars1 (cdr x) (cons (car (car x)) a)))))

(defun strip-cars (x)

  ":Doc-Section ACL2::Programming

  collect up all first components of pairs in a list~/

  ~c[(strip-cars x)] is the list obtained by walking through the list
  ~c[x] and collecting up all first components (~ilc[car]s).
  This function is implemented in a tail-recursive way, despite its
  logical definition.~/

  ~c[(strip-cars x)] has a ~il[guard] of ~c[(alistp x)].~/"

  (declare (xargs :guard (alistp x)))

; See the Essay on Strip-cars -- To Tail Recur or not to Tail Recur?
; above.

  #-acl2-loop-only
  (strip-cars1 x nil)
  #+acl2-loop-only
  (cond ((endp x) nil)
        (t (cons (car (car x))
                 (strip-cars (cdr x))))))

#+acl2-loop-only
(defun eql (x y)
  (declare (xargs :mode :logic
                  :guard (or (eqlablep x)
                             (eqlablep y))))
  ":Doc-Section ACL2::Programming

  test equality (of two numbers, symbols, or ~il[characters])~/

  ~c[(eql x y)] is logically equivalent to ~c[(equal x y)].~/

  Unlike ~ilc[equal], ~c[eql] has a ~il[guard] requiring at least one of its
  arguments to be a number, a symbol, or a character.  Generally,
  ~c[eql] is executed more efficiently than ~ilc[equal].

  For a discussion of the various ways to test against 0,
  ~l[zero-test-idioms].

  ~c[Eql] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (equal x y))


;                             DATA TYPES

#+acl2-loop-only
(defmacro <= (x y)

  ":Doc-Section ACL2::Programming

  less-than-or-equal test~/

  ~c[<=] is a macro, and ~c[(<= x y)] expands to the same thing as
  ~c[(not (< y x))].  ~l[<].~/

  ~c[<=] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (List 'not (list '< y x)))

#+acl2-loop-only
(defun = (x y)

  ":Doc-Section ACL2::Programming

  test equality of two numbers~/

  ~c[(= x y)] is logically equivalent to ~c[(equal x y)].~/

  Unlike ~ilc[equal], ~c[=] has a ~il[guard] requiring both of its arguments
  to be numbers.  Generally, ~c[=] is executed more efficiently than
  ~ilc[equal].

  For a discussion of the various ways to test against 0,
  ~l[zero-test-idioms].

  ~c[=] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (declare (xargs :mode :logic
                  :guard (and (acl2-numberp x)
                              (acl2-numberp y))))

  (equal x y))

#+acl2-loop-only
(defun /= (x y)

  ":Doc-Section ACL2::Programming

  test inequality of two numbers~/

  ~c[(/= x y)] is logically equivalent to ~c[(not (equal x y))].~/

  Unlike ~ilc[equal], ~c[/=] has a ~il[guard] requiring both of its arguments
  to be numbers.  Generally, ~c[/=] is executed more efficiently than
  a combination of ~ilc[not] and ~ilc[equal].

  For a discussion of the various ways to test against 0,
  ~l[zero-test-idioms].

  ~c[/=] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (Declare (xargs :mode :logic
                  :guard (and (acl2-numberp x)
                              (acl2-numberp y))))
  (not (equal x y)))

#+acl2-loop-only
(defmacro > (x y)

  ":Doc-Section ACL2::Programming

  greater-than test~/

  ~c[>] is a macro, and ~c[(> x y)] expands to the same thing as
  ~c[(< y x)].  ~l[<].~/

  ~c[>] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (list '< y x))

#+acl2-loop-only
(defmacro >= (x y)

  ":Doc-Section ACL2::Programming

  greater-than-or-equal test~/

  ~c[>=] is a macro, and ~c[(>= x y)] expands to the same thing as
  ~c[(not (< x y))].  ~l[<].~/

  ~c[>=] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (list 'not (list '< x y)))

(deflabel zero-test-idioms
  :doc
  ":Doc-Section ACL2::Programming

  how to test for 0~/

  Below are six commonly used idioms for testing whether ~c[x] is ~c[0].
  ~ilc[Zip] and ~ilc[zp] are the preferred termination tests for recursions
  down the integers and naturals, respectively.
  ~bv[]
  idiom       logical              guard                 primary
              meaning                                 compiled code*

  (equal x 0) (equal x 0)          t                   (equal x 0)

  (eql x 0)   (equal x 0)          t                   (eql x 0)

  (zerop x)   (equal x 0)          x is a number       (= x 0)

  (= x 0)     (equal x 0)          x is a number       (= x 0)

  (zip x)     (equal (ifix x) 0)   x is an integer     (= x 0)

  (zp x)      (equal (nfix x) 0)   x is a natural      (int= x 0)

  (zpf x)     (equal (nfix x) 0)   x is a fixnum >= 0  (eql (the-fixnum x) 0)
  ~ev[]
  *~l[guards-and-evaluation], especially the subsection titled
  ``Guards and evaluation V: efficiency issues''.  Primary code is
  relevant only if ~il[guard]s are verified.  The ``compiled code'' shown
  is only suggestive.~/

  The first four idioms all have the same logical meaning and differ
  only with respect to their executability and efficiency.  In the
  absence of compiler optimizing, ~c[(= x 0)] is probably the most
  efficient, ~c[(equal x 0)] is probably the least efficient, and
  ~c[(eql x 0)] is in between.  However, an optimizing compiler could
  always choose to compile ~c[(equal x 0)] as ~c[(eql x 0)] and, in
  situations where ~c[x] is known at compile-time to be numeric,
  ~c[(eql x 0)] as ~c[(= x 0)].  So efficiency considerations must, of
  course, be made in the context of the host compiler.

  Note also that ~c[(zerop x)] and ~c[(= x 0)] are indistinguishable.
  They have the same meaning and the same ~il[guard], and can reasonably be
  expected to generate equally efficient code.

  Note that ~c[(zip x)] and ~c[(zp x)] do not have the same logical
  meanings as the others or each other.  They are not simple tests for
  equality to ~c[0].  They each coerce ~c[x] into a restricted domain,
  ~ilc[zip] to the integers and ~ilc[zp] to the natural numbers, choosing
  ~c[0] for ~c[x] when ~c[x] is outside the domain.  Thus, ~c[1/2], ~c[#c(1 3)],
  and ~c['abc], for example, are all ``recognized'' as zero by both
  ~ilc[zip] and ~ilc[zp].  But ~ilc[zip] reports that ~c[-1] is different from
  ~c[0] while ~ilc[zp] reports that ~c[-1] ``is'' ~c[0].  More precisely,
  ~c[(zip -1)] is ~c[nil] while ~c[(zp -1)] is ~c[t].

  Note that the last five idioms all have ~il[guard]s that restrict their
  Common Lisp executability.  If these last five are used in
  situations in which ~il[guard]s are to be verified, then proof
  obligations are incurred as the price of using them.  If guard
  verification is not involved in your project, then the first five
  can be thought of as synonymous.

  ~ilc[Zip] and ~ilc[zp] are not provided by Common Lisp but are
  ACL2-specific functions.  Why does ACL2 provide these functions?
  The answer has to do with the admission of recursively defined
  functions and efficiency.  ~ilc[Zp] is provided as the zero-test in
  situations where the controlling formal parameter is understood to
  be a natural number.  ~ilc[Zip] is analogously provided for the integer
  case.  We illustrate below.

  Here is an admissible definition of factorial
  ~bv[]
  (defun fact (n) (if (zp n) 1 (* n (fact (1- n)))))
  ~ev[]
  Observe the classic recursion scheme: a test against ~c[0] and recursion
  by ~ilc[1-].  Note however that the test against ~c[0] is expressed with the
  ~ilc[zp] idiom.  Note also the absence of a ~il[guard] making explicit our
  intention that ~c[n] is a natural number.

  This definition of factorial is readily admitted because when ~c[(zp n)]

  is false (i.e., ~c[nil]) then ~c[n] is a natural number other than
  ~c[0] and so ~c[(1- n)] is less than ~c[n].  The base case, where ~c[(zp n)]
  is true, handles all the ``unexpected'' inputs, such as arise with
  ~c[(fact -1)] and ~c[(fact 'abc)].  When calls of ~c[fact] are
  evaluated, ~c[(zp n)] checks ~c[(integerp n)] and ~c[(> n 0)].  ~il[Guard]
  verification is unsuccessful for this definition of ~c[fact] because
  ~ilc[zp] requires its argument to be a natural number and there is no
  ~il[guard] on ~c[fact], above.  Thus the primary raw lisp for ~c[fact] is
  inaccessible and only the ~c[:]~ilc[logic] definition (which does runtime
  ``type'' checking) is used in computation.  In summary, this
  definition of factorial is easily admitted and easily manipulated by
  the prover but is not executed as efficiently as it could be.

  Runtime efficiency can be improved by adding a ~il[guard] to the definition.
  ~bv[]
  (defun fact (n)
    (declare (xargs :guard (and (integerp n) (>= n 0))))
    (if (zp n) 1 (* n (fact (1- n)))))
  ~ev[]
  This ~il[guard]ed definition has the same termination conditions as
  before -- termination is not sensitive to the ~il[guard].  But the ~il[guard]s
  can be verified.  This makes the primary raw lisp definition
  accessible during execution.  In that definition, the ~c[(zp n)] above
  is compiled as ~c[(= n 0)], because ~c[n] will always be a natural number
  when the primary code is executed.  Thus, by adding a ~il[guard] and
  verifying it, the elegant and easily used definition of factorial is
  also efficiently executed on natural numbers.

  Now let us consider an alternative definition of factorial in which
  ~c[(= n 0)] is used in place of ~c[(zp n)].
  ~bv[]
  (defun fact (n) (if (= n 0) 1 (* n (fact (1- n)))))
  ~ev[]
  This definition does not terminate.  For example ~c[(fact -1)] gives
  rise to a call of ~c[(fact -2)], etc.  Hence, this alternative is
  inadmissible.  A plausible response is the addition of a ~il[guard]
  restricting ~c[n] to the naturals:
  ~bv[]
  (defun fact (n)
   (declare (xargs :guard (and (integerp n) (>= n 0))))
   (if (= n 0) 1 (* n (fact (1- n)))))
  ~ev[]
  But because the termination argument is not sensitive to the ~il[guard],
  it is still impossible to admit this definition.  To influence the
  termination argument one must change the conditions tested.  Adding
  a runtime test that ~c[n] is a natural number would suffice and allow
  both admission and ~il[guard] verification.  But such a test would slow
  down the execution of the compiled function.

  The use of ~c[(zp n)] as the test avoids this dilemma.  ~ilc[Zp]
  provides the logical equivalent of a runtime test that ~c[n] is a
  natural number but the execution efficiency of a direct ~ilc[=]
  comparison with ~c[0], at the expense of a ~il[guard] conjecture to prove.
  In addition, if ~il[guard] verification and most-efficient execution are
  not needed, then the use of ~c[(zp n)] allows the admission of the
  function without a ~il[guard] or other extraneous verbiage.

  While general rules are made to be broken, it is probably a good
  idea to get into the habit of using ~c[(zp n)] as your terminating
  ``~c[0] test'' idiom when recursing down the natural numbers.  It
  provides the logical power of testing that ~c[n] is a non-~c[0]
  natural number and allows efficient execution.

  We now turn to the analogous function, ~ilc[zip].  ~ilc[Zip] is the
  preferred ~c[0]-test idiom when recursing through the integers toward
  ~c[0].  ~ilc[Zip] considers any non-integer to be ~c[0] and otherwise
  just recognizes ~c[0].  A typical use of ~ilc[zip] is in the definition
  of ~ilc[integer-length], shown below.  (ACL2 can actually accept this
  definition, but only after appropriate lemmas have been proved.)
  ~bv[]
  (defun integer-length (i)
    (declare (xargs :guard (integerp i)))
    (if (zip i)
        0
      (if (= i -1)
        0
        (+ 1 (integer-length (floor i 2))))))
  ~ev[]
  Observe that the function recurses by ~c[(floor i 2)].  Hence,
  calling the function on ~c[25] causes calls on ~c[12], ~c[6], ~c[3],
  ~c[1], and ~c[0], while calling it on ~c[-25] generates calls on
  ~c[-13], ~c[-7], ~c[-4], ~c[-2], and ~c[-1].  By making ~c[(zip i)] the
  first test, we terminate the recursion immediately on non-integers.
  The ~il[guard], if present, can be verified and allows the primary raw
  lisp definition to check ~c[(= i 0)] as the first terminating
  condition (because the primary code is executed only on integers).")

(defmacro int= (i j)

  ":Doc-Section ACL2::Programming

  test equality of two integers~/

  ~c[(int= x y)] is logically equivalent to ~c[(equal x y)].~/

  Unlike ~ilc[equal], ~c[int=] requires its arguments to be numbers (or
  else causes a ~il[guard] violation; ~pl[guard]).  Generally, ~c[int=]
  is executed more efficiently than ~ilc[equal] or ~ilc[=] on integers."

  (list 'eql

; The extra care taken below not to wrap (the integer ...) around integers is
; there to overcome an inefficiency in Allegro 5.0.1 (and probably other
; Allegro releases).  Rob Sumners has reported this problem (6/25/00) to Franz.

        (if (integerp i) i (list 'the 'integer i))
        (if (integerp j) j (list 'the 'integer j))))

#+acl2-loop-only
(defun zp (x)
  (declare (xargs :mode :logic
                  :guard (and (integerp x) (<= 0 x))))

  ":Doc-Section ACL2::Programming

  testing a ``natural'' against 0~/

  ~c[(Zp n)] is logically equivalent to ~c[(equal (nfix n) 0)] and is
  the preferred termination test for recursion down the natural
  numbers. ~c[(Zp n)] returns ~c[t] if ~c[n] is ~c[0] or not a natural
  number; it returns ~c[nil] otherwise.  Thus, in the ACL2 logic
  (ignoring the issue of ~il[guard]s):
  ~bv[]
      n       (zp n)
     3         nil
     0         t
     -1        t
     5/2       t
     #c(1 3)   t
     'abc      t
  ~ev[]~/

  ~c[(Zp n)] has a ~il[guard] requiring ~c[n] to be a natural number.

  For a discussion of the various idioms for testing against ~c[0],
  ~pl[zero-test-idioms].

  ~c[Zp] is typically used as the termination test in recursions down
  the natural numbers.  It has the advantage of ``coercing'' its
  argument to a natural and hence allows the definition to be admitted
  without an explicit type check in the body.  ~il[Guard] verification
  allows ~c[zp] to be compiled as a direct ~ilc[=]-comparision with ~c[0]."

  (if (integerp x)
      (<= x 0)
    t))

#-acl2-loop-only
; Consider using mbe to avoid this cheat.
(defun-one-output zp (x)
  (declare (type integer x))
  (int= x 0))

(defthm zp-compound-recognizer

; This rule improves the ability of ACL2 to compute useful type prescriptions
; for functions.  For example, the following function is typed using
; acl2-numberp instead of integerp unless we have this rule:
; (defun foo (index lst)
;   (if (zp index)
;       nil
;     (let ((i (1- index))) (or (foo i lst) (and (not (bar i lst)) i)))))

  (equal (zp x)
         (or (not (integerp x))
             (<= x 0)))
  :rule-classes :compound-recognizer)

(defthm zp-open

; The preceding event avoids some case-splitting when the
; zp-compound-recognizer (above) provides all the information needed about an
; argument of zp.  However, the following example illustrates the need to open
; up zp on some non-variable terms:

; (thm (implies (and (zp (+ (- k) n))
;                   (integerp k)
;                   (integerp n)
;                   (<= k j))
;               (<= n j)))

; The present rule allows the theorem above to go through.  This example
; theorem was distilled from the failure (without this rule) of event
; compress11-assoc-property-1 in books/data-structures/array1.lisp.

  (implies (syntaxp (not (variablep x)))
           (equal (zp x)
                  (if (integerp x)
                      (<= x 0)
                    t))))

(in-theory (disable zp))

#+acl2-loop-only
(defun zip (x)
  (declare (xargs :mode :logic
                  :guard (integerp x)))

  ":Doc-Section ACL2::Programming

  testing an ``integer'' against 0~/

  ~c[(Zip i)] is logically equivalent to ~c[(equal (ifix i) 0)] and is
  the preferred termination test for recursion through the integers.
  ~c[(Zip i)] returns ~c[t] if ~c[i] is ~c[0] or not an integer; it
  returns ~c[nil] otherwise.  Thus,
  ~bv[]
     i         (zip i)
     3         nil
     0         t
     -2        nil
     5/2       t
     #c(1 3)   t
     'abc      t
  ~ev[]~/

  ~c[(Zip i)] has a ~il[guard] requiring ~c[i] to be an integer.

  For a discussion of the various idioms for testing against ~c[0],
  ~pl[zero-test-idioms].

  ~c[Zip] is typically used as the termination test in recursions
  through the integers.  It has the advantage of ``coercing'' its
  argument to an integer and hence allows the definition to be
  admitted without an explicit type check in the body.  ~il[Guard]
  verification allows ~c[zip] to be compiled as a direct
  ~ilc[=]-comparision with ~c[0]."

  (if (integerp x)
      (= x 0)
    t))

#-acl2-loop-only
; If we had :body we wouldn't need this cheat.
(defun-one-output zip (x) (= x 0))

(defthm zip-compound-recognizer

; See the comment for zp-compound-recognizer.

  (equal (zip x)
         (or (not (integerp x))
             (equal x 0)))
  :rule-classes :compound-recognizer)

(defthm zip-open
  (implies (syntaxp (not (variablep x)))
           (equal (zip x)
                  (or (not (integerp x))
                      (equal x 0)))))

(in-theory (disable zip))

#+acl2-loop-only
(defun nth (n l)

  ":Doc-Section ACL2::Programming

  the nth element (zero-based) of a list~/

  ~c[(Nth n l)] is the ~c[n]th element of ~c[l], zero-based.  If ~c[n] is
  greater than or equal to the length of ~c[l], then ~c[nth] returns ~c[nil].~/

  ~c[(Nth n l)] has a ~il[guard] that ~c[n] is a non-negative integer and
  ~c[l] is a ~ilc[true-listp].

  ~c[Nth] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (declare (xargs :guard (and (integerp n)
                              (>= n 0)
                              (true-listp l))))
  (if (endp l)
      nil
    (if (zp n)
        (car l)
      (nth (- n 1) (cdr l)))))

#+acl2-loop-only
(defun char (s n)

  ":Doc-Section ACL2::Programming

  the ~il[nth] element (zero-based) of a string~/

  ~c[(Char s n)] is the ~c[n]th element of ~c[s], zero-based.  If ~c[n] is
  greater than or equal to the length of ~c[s], then ~c[char] returns
  ~c[nil].~/

  ~c[(Char s n)] has a ~il[guard] that ~c[n] is a non-negative integer and
  ~c[s] is a ~ilc[stringp].

  ~c[Char] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (declare (xargs :guard (and (stringp s)
                              (integerp n)
                              (>= n 0)
                              (< n (length s)))))
  (nth n (coerce s 'list)))

(defun proper-consp (x)

  ":Doc-Section ACL2::Programming

  recognizer for proper (null-terminated) non-empty lists~/

  ~c[Proper-consp] is the function that checks whether its argument is
  a non-empty list that ends in ~c[nil].  Also ~pl[true-listp].~/~/"

  (declare (xargs :guard t))
  (and (consp x)
       (true-listp x)))

(defun improper-consp (x)

  ":Doc-Section ACL2::Programming

  recognizer for improper (non-null-terminated) non-empty lists~/

  ~c[Improper-consp] is the function that checks whether its argument
  is a non-empty list that ends in other than ~c[nil].
  ~l[proper-consp] and also ~pl[true-listp].~/~/"

  (declare (xargs :guard t))
  (and (consp x)
       (not (true-listp x))))

#+acl2-loop-only
(defmacro * (&rest rst)

  ":Doc-Section ACL2::Programming

  multiplication macro~/

  ~c[*] is really a macro that expands to calls of the function
  ~ilc[binary-*].  So for example
  ~bv[]
  (* x y 4 z)
  ~ev[]
  represents the same term as
  ~bv[]
  (binary-* x (binary-* y (binary-* 4 z))).
  ~ev[]~/

  ~l[binary-*].

  ~c[*] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (cond ((null rst) 1)
        ((null (cdr rst)) (list 'binary-* 1 (car rst)))
        (t (xxxjoin 'binary-* rst))))

;; RAG - This function was modified to accept all complex arguments,
;; not just the complex-rationalps

#+acl2-loop-only
(defun conjugate (x)

  ":Doc-Section ACL2::Programming

  complex number conjugate~/

  ~c[Conjugate] takes an ACL2 number as an argument, and returns its
  complex conjugate (i.e., the result of negating its imaginary
  part.).~/

  ~c[Conjugate] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (acl2-numberp x)))
  (if (complex/complex-rationalp x)
      (complex (realpart x)
               (- (imagpart x)))
      x))

#+acl2-loop-only
(defun prog2$ (x y)

; This odd little duck is not as useless as it seems.  The original
; purpose of this function was to serve as a messenger for translate
; to use to send a message to the guard checker.  Guards that are
; created by declarations in lets and other places are put into the
; first arg of a prog2$.  Once the guards required by x have been
; noted, x's value may be ignored.  If this function is changed,
; consider the places it is mentioned, including the mention of
; 'prog2$ in distribute-first- if.

; We have since found other uses for this function, which are
; documented in the doc string below.

  ":Doc-Section ACL2::Programming

  execute two forms and return the value of the second one~/

  ~l[hard-error], ~pl[illegal], and ~pl[cw] for examples of functions
  to call in the first argument of ~c[prog2$].~/

  Semantically, ~c[(Prog2$ x y)] equals ~c[y]; the value of ~c[x] is ignored.
  However, ~c[x] is first evaluated for side effect.  Since the ACL2
  ~il[programming] language is applicative, there can be no logical impact
  of evaluating ~c[x].  However, ~c[x] may involve a call of a function such
  as ~ilc[hard-error] or ~ilc[illegal], which can cause so-called ``hard errors'',
  or a call of ~ilc[cw] to perform output.

  Here is a simple, contrived example using ~ilc[hard-error].  The intention
  is to check at run-time that the input is appropriate before calling
  function ~c[bar].
  ~bv[]
  (defun foo-a (x)
    (declare (xargs :guard (consp x)))
    (prog2$
     (or (good-car-p (car x))
         (hard-error 'foo-a
                     \"Bad value for x: ~~p0\"
                     (list (cons #\\0 x))))
     (bar x)))
  ~ev[]
  The following similar function uses ~ilc[illegal] instead of ~c[hard-error].
  Since ~c[illegal] has a guard of ~c[nil], ~il[guard] verification would
  guarantee that the call of ~c[illegal] below will never be made (at
  least when guard checking is on; ~pl[set-guard-checking]).
  ~bv[]
  (defun foo-b (x)
    (declare (xargs :guard (and (consp x) (good-car-p (car x)))))
    (prog2$
     (or (good-car-p (car x))
         (illegal 'foo-b
                  \"Bad value for x: ~~p0\"
                  (list (cons #\\0 x))))
     (bar x)))
  ~ev[]

  We conclude with a simple example using ~ilc[cw] from the ACL2 sources.

  ~bv[]
  (defun print-terms (terms iff-flg wrld)

  ; Print untranslations of the given terms with respect to iff-flg, following
  ; each with a newline.

  ; We use cw instead of the fmt functions because we want to be able to use this
  ; function in print-type-alist-segments (used in brkpt1), which does not return
  ; state.

    (if (endp terms)
        terms
      (prog2$
       (cw \"~~q0\" (untranslate (car terms) iff-flg wrld))
       (print-terms (cdr terms) iff-flg wrld))))
  ~ev[]~/"

  (declare (ignore x)
           (xargs :mode :logic :guard t))
  y)

#-acl2-loop-only
(defmacro prog2$ (x y)

; We need to make prog2$ a macro in raw Lisp when feature acl2-mv-as-values is
; set, because otherwise we will lose multiple values returned during the
; computation of y.  So we might as well make prog2$ a macro in raw Lisp
; unconditionally, for consistency.

  (list 'progn x y))

(deflabel Other
  :doc
  ":Doc-Section Other

  other commonly used top-level functions~/~/

  This section contains an assortment of top-level functions that fit into none
  of the other categories and yet are suffiently useful as to merit
  ``~c[advertisement]'' in the ~c[:]~ilc[help] command.~/")

(deflabel acl2-help
  :doc
  ":Doc-Section Other

  the acl2-help mailing list~/

  You can email questions about ACL2 usage to the acl2-help mailing list:
  ~c[acl2-help@utlists.utexas.edu].  If you have more general questions about
  ACL2, for example, about projects completed using ACL2, you may prefer the
  acl2 mailing list, ~c[acl2@utlists.utexas.edu], which tends to have wider
  distribution.~/~/")

#+acl2-loop-only
(defun ec-call (x)

; It is instructive to consider (ec-call (must-be-equal term1 term2)).  This
; call of must-be-equal is treated as an ordinary function in the logic whose
; guard requires that term1 and term2 have equal values.  Without the
; surrounding ec-call, a special case in guard-clauses would avoid considering
; the guards for term1.  However, since the raw Lisp definition of bar calls
; the function *1*must-be-equal rather than the macro must-be-equal, the guards
; must be verified for term1 as well.  Hence, we do not give any special
; treatment to the ec-call term above.  You can see what goes wrong if we
; ignore the guards for term1 by running the following example.

; (skip-proofs
;  (defun bar (x)
;    (declare (xargs :guard t))
;    (ec-call (must-be-equal (consp (append x x))
;                            (consp x)))))
; (bar 3)

; The above discussion is a comment because it seems too obscure to put in the
; documentation, especially since (ec-call (mbe ...)) would much more likely
; occur instead.  (And since mbe is a macro, that is not an issue.)

  ":Doc-Section ACL2::Programming

  execute a call in the ACL2 logic instead of raw Lisp~/

  The name ``~c[ec-call]'' represents ``executable-counterpart call.''  This
  utility is intended for users who are familiar with guards.  ~l[guard] for a
  general discussion of guards.

  Logically, ~c[ec-call] is the identity function; indeed, during proofs it
  behaves more like the identity macro, in the sense that ~c[(ec-call TERM)] is
  typically replaced quickly by ~c[TERM] during a proof attempt.  However,
  ~c[ec-call] causes function calls to be evaluated in the ACL2 logic rather
  than raw Lisp, as explained below.~/

  ~bv[]
  General Form:
  (ec-call (fn term1 ... termk))
  ~ev[]
  where ~c[fn] is a known function symbol other than those in the list that is
  the value of the constant ~c[*ec-call-bad-ops*].  Semantically,
  ~c[(ec-call (fn term1 ... termk))] equals ~c[(fn term1 ... termk)].  However,
  this use of ~c[ec-call] has two effects.
  ~bq[]

  (1) ~il[Guard] verification generates no proof obligations from the guard of
  ~c[fn] for this call.  Indeed, guards need not have been verified for
  ~c[fn].

  (2) During evaluation, after the arguments of ~c[fn] are evaluated as usual,
  the executable counterpart of ~c[fn] is called, rather than ~c[fn] as defined
  in raw Lisp.  That is, the call of ~c[fn] is made on its evaluated arguments
  as though this call is being made in the ACL2 top-level loop, rather than in
  raw Lisp.  In particular, the ~il[guard] of ~c[fn] is checked, at least by
  default (~pl[set-guard-checking].~eq[]

  Note that in the term ~c[(ec-call (fn term1 ... termk))], only the indicated
  call of ~c[fn] is made in the logic; each ~c[termi] is evaluated in the
  normal manner.  If you want an entire term evaluated in the logic, wrap
  ~c[ec-call] around each function call in the term (other than calls of ~c[if]
  and ~c[ec-call]).

  Here is a small example.  We define ~c[foo] recursively but with guard
  verification inhibited on the recursive call, which is to be evaluated in the
  ACL2 logic.
  ~bv[]
  ACL2 !>(defun foo (x y)
          (declare (xargs :guard (consp y)))
          (if (consp x)
              (cons (car x) (ec-call (foo (cdr x) (cdr y))))
            (car y)))

  The admission of FOO is trivial, using the relation O< (which is known
  to be well-founded on the domain recognized by O-P) and the measure
  (ACL2-COUNT X).  We could deduce no constraints on the type of FOO.

  Computing the guard conjecture for FOO....

  The guard conjecture for FOO is trivial to prove.  FOO is compliant
  with Common Lisp.

  Summary
  Form:  ( DEFUN FOO ...)
  Rules: NIL
  Warnings:  None
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)
   FOO
  ACL2 !>(foo '(2 3 4 5) '(6 7))


  ACL2 Error in TOP-LEVEL:  The guard for the function symbol FOO, which
  is (CONSP Y), is violated by the arguments in the call (FOO '(4 5) NIL).
  See :DOC trace for a useful debugging utility.  See :DOC set-guard-
  checking for information about suppressing this check with (set-guard-
  checking :none), as recommended for new users.
  
  ~ev[]
  The error above arises because eventually, foo recurs down to a value of
  parameter ~c[y] that violates the guard.  This is clear from tracing
  (~pl[trace$] and ~pl[trace]).  Each call of the executable counterpart of
  ~c[foo] (the so-called ``*1*'' function for ~c[foo]) checks the guard and
  then invokes the raw Lisp version of ~c[foo].  The raw Lisp version calls
  the executable counterpart on the recursive call.  When the guard check fails
  we get a violation.
  ~bv[]
  ACL2 !>(trace$ foo)
   ((FOO))
  ACL2 !>(foo '(2 3 4 5) '(6 7))
  1> (ACL2_*1*_ACL2::FOO (2 3 4 5) (6 7))
    2> (FOO (2 3 4 5) (6 7))
      3> (ACL2_*1*_ACL2::FOO (3 4 5) (7))
        4> (FOO (3 4 5) (7))
          5> (ACL2_*1*_ACL2::FOO (4 5) NIL)


  ACL2 Error in TOP-LEVEL:  The guard for the function symbol FOO, which
  is (CONSP Y), is violated by the arguments in the call (FOO '(4 5) NIL).
  See :DOC trace for a useful debugging utility.  See :DOC set-guard-
  checking for information about suppressing this check with (set-guard-
  checking :none), as recommended for new users.

  ACL2 !>
  ~ev[]
  If we turn off guard errors then we can see the trace as above, but where we
  avoid calling the raw Lisp function when the guard fails to hold.
  ~bv[]
  ACL2 !>:set-guard-checking nil

  Masking guard violations but still checking guards except for self-
  recursive calls.  To avoid guard checking entirely, :SET-GUARD-CHECKING
  :NONE.  See :DOC set-guard-checking.

  ACL2 >(foo '(2 3 4 5) '(6 7))
  1> (ACL2_*1*_ACL2::FOO (2 3 4 5) (6 7))
    2> (FOO (2 3 4 5) (6 7))
      3> (ACL2_*1*_ACL2::FOO (3 4 5) (7))
        4> (FOO (3 4 5) (7))
          5> (ACL2_*1*_ACL2::FOO (4 5) NIL)
            6> (ACL2_*1*_ACL2::FOO (5) NIL)
              7> (ACL2_*1*_ACL2::FOO NIL NIL)
              <7 (ACL2_*1*_ACL2::FOO NIL)
            <6 (ACL2_*1*_ACL2::FOO (5))
          <5 (ACL2_*1*_ACL2::FOO (4 5))
        <4 (FOO (3 4 5))
      <3 (ACL2_*1*_ACL2::FOO (3 4 5))
    <2 (FOO (2 3 4 5))
  <1 (ACL2_*1*_ACL2::FOO (2 3 4 5))
  (2 3 4 5)
  ACL2 >
  ~ev[]
  ~/"

  (declare (xargs :guard t))
  x)

#-acl2-loop-only
(defmacro ec-call (x)
  (if (and (consp x)
           (symbolp (car x)))
      (cons (*1*-symbol (car x))
            (cdr x))

; We might want to cause an error in the following case.  But then we would
; have to take special measures to avoid that error when defining the *1*
; function for ec-call during the boot-strap.  The case below should never
; arise in any other case, anyhow; translate insists that ec-call be applied
; only to function calls.

    x))

#+acl2-loop-only
(defmacro / (x &optional (y 'nil binary-casep))

  ":Doc-Section ACL2::Programming

  macro for division and reciprocal~/

  ~l[binary-*] for multiplication and ~pl[unary-/] for reciprocal.~/

  Note that ~c[/] represents division as follows:
  ~bv[]
  (/ x y)
  ~ev[]
  represents the same term as
  ~bv[]
  (* x (/ y))
  ~ev[]
  which is really
  ~bv[]
  (binary-* x (unary-/ y)).
  ~ev[]
  Also note that ~c[/] represents reciprocal as follows:
  ~bv[]
  (/ x)
  ~ev[]
  expands to
  ~bv[]
  (unary-/ x).
  ~ev[]
  ~c[/] is a Common Lisp macro.  See any Common Lisp documentation
  for more information.~/"

  (cond (binary-casep (list 'binary-* x (list 'unary-/ y)))
        (t (list 'unary-/ x))))

; This, and many of the axioms that follow, could be defthms.  However, we want
; to make explicit what our axioms are, rather than relying on (e.g.) linear
; arithmetic.  This is a start.

(defaxiom closure
  (and (acl2-numberp (+ x y))
       (acl2-numberp (* x y))
       (acl2-numberp (- x))
       (acl2-numberp (/ x)))
  :rule-classes nil)

(defaxiom Associativity-of-+
  (equal (+ (+ x y) z) (+ x (+ y z))))

(defaxiom Commutativity-of-+
  (equal (+ x y) (+ y x)))

(defun fix (x)

  ":Doc-Section ACL2::Programming

  coerce to a number~/

  ~c[Fix] simply returns any numeric argument unchanged, returning ~c[0]
  on a non-numeric argument.  Also ~pl[nfix], ~pl[ifix], and
  ~pl[rfix] for analogous functions that coerce to a natural
  number, an integer, and a rational number, respectively.~/

  ~c[Fix] has a ~il[guard] of ~c[t].~/"

  (declare (xargs :guard t))
  (if (acl2-numberp x)
      x
    0))

(defaxiom Unicity-of-0
  (equal (+ 0 x)
         (fix x)))

(defaxiom Inverse-of-+
  (equal (+ x (- x)) 0))

(defaxiom Associativity-of-*
  (equal (* (* x y) z) (* x (* y z))))

(defaxiom Commutativity-of-*
  (equal (* x y) (* y x)))

(defaxiom Unicity-of-1
  (equal (* 1 x)
         (fix x)))

(defaxiom Inverse-of-*
  (implies (and (acl2-numberp x)
                (not (equal x 0)))
           (equal (* x (/ x)) 1)))

(defaxiom Distributivity
  (equal (* x (+ y z))
         (+ (* x y) (* x z))))

(defaxiom <-on-others
  (equal (< x y)
         (< (+ x (- y)) 0))
  :rule-classes nil)

(defaxiom Zero
  (not (< 0 0))
  :rule-classes nil)

(defaxiom Trichotomy
  (and
   (implies (acl2-numberp x)
            (or (< 0 x)
                (equal x 0)
                (< 0 (- x))))
   (or (not (< 0 x))
       (not (< 0 (- x)))))
  :rule-classes nil)

;; RAG - This axiom was weakened to accomodate real x and y

(defaxiom Positive
  (and (implies (and (< 0 x) (< 0 y))
                (< 0 (+ x y)))
       (implies (and (real/rationalp x)
                     (real/rationalp y)
                     (< 0 x)
                     (< 0 y))
                (< 0 (* x y))))
  :rule-classes nil)

(defaxiom Rational-implies1
  (implies (rationalp x)
           (and (integerp (denominator x))
                (integerp (numerator x))
                (< 0 (denominator x))))
  :rule-classes nil)

(defaxiom Rational-implies2
  (implies (rationalp x)

; We use the left-hand side below out of respect for the fact that
; unary-/ is invisible with respect to binary-*.

           (equal (* (/ (denominator x)) (numerator x)) x)))

(defaxiom integer-implies-rational
  (implies (integerp x) (rationalp x))
  :rule-classes nil)

#+:non-standard-analysis
(defaxiom rational-implies-real
  (implies (rationalp x) (realp x))
  :rule-classes nil)

;; RAG - This axiom was weakened to accomodate the reals.

(defaxiom complex-implies1
  (and (real/rationalp (realpart x))
       (real/rationalp (imagpart x)))
  :rule-classes nil)

;; RAG - This axiom was strengthened to include the reals.

(defaxiom complex-definition
  (implies (and (real/rationalp x)
                (real/rationalp y))
           (equal (complex x y)
                  (+ x (* #c(0 1) y))))
  :rule-classes nil)

;; RAG - This axiom was weakened to accomodate the reals.

; This rule was called complex-rationalp-has-nonzero-imagpart before
; Version_2.5.
(defaxiom nonzero-imagpart
  (implies (complex/complex-rationalp x)
           (not (equal 0 (imagpart x))))
  :rule-classes nil)

(defaxiom realpart-imagpart-elim
  (implies (acl2-numberp x)
           (equal (complex (realpart x) (imagpart x)) x))
  :rule-classes (:REWRITE :ELIM))

; We think that the following two axioms can be proved from the others.

;; RAG - This axiom was strengthened to include the reals.

(defaxiom realpart-complex
  (implies (and (real/rationalp x)
                (real/rationalp y))
           (equal (realpart (complex x y))
                  x)))

;; RAG - This axiom was also strengthened to include the reals.

(defaxiom imagpart-complex
  (implies (and (real/rationalp x)
                (real/rationalp y))
           (equal (imagpart (complex x y))
                  y)))

;; RAG - Another axiom strengthened to include the reals.

(defthm complex-equal
  (implies (and (real/rationalp x1)
                (real/rationalp y1)
                (real/rationalp x2)
                (real/rationalp y2))
           (equal (equal (complex x1 y1) (complex x2 y2))
                  (and (equal x1 x2)
                       (equal y1 y2))))
  :hints (("Goal" :use
           ((:instance imagpart-complex
                       (x x1) (y y1))
            (:instance imagpart-complex
                       (x x2) (y y2))
            (:instance realpart-complex
                       (x x1) (y y1))
            (:instance realpart-complex
                       (x x2) (y y2)))
           :in-theory (disable imagpart-complex realpart-complex))))

(defun force (x)

  ":Doc-Section Miscellaneous

  identity function used to force a hypothesis~/

  When a hypothesis of a conditional rule has the form ~c[(force hyp)] it
  is logically equivalent to ~c[hyp] but has a pragmatic effect.  In
  particular, when the rule is considered, the needed instance of the
  hypothesis, ~c[hyp'], is assumed and a special case is generated,
  requiring the system to prove that ~c[hyp'] is true in the current
  context.  The proofs of all such ``forced assumptions'' are delayed
  until the successful completion of the main goal.
  ~l[forcing-round].

  Forcing should only be used on hypotheses that are always expected
  to be true, such as the ~il[guard]s of functions.  All the power of the
  theorem prover is brought to bear on a forced hypothesis and no
  backtracking is possible.  If the ~c[:]~ilc[executable-counterpart] of the
  function ~c[force] is ~il[disable]d, then no hypothesis is forced.
  ~l[enable-forcing] and ~pl[disable-forcing].  Forced goals can be
  attacked immediately (~pl[immediate-force-modep]) or in a subsequent
  forcing round (~pl[forcing-round]).  Also ~pl[case-split].~/

  It sometimes happens that a conditional rule is not applied because
  some hypothesis, ~c[hyp], could not be relieved, even though the
  required instance of ~c[hyp], ~c[hyp'], can be shown true in the context.
  This happens when insufficient resources are brought to bear on ~c[hyp']
  at the time we try to relieve it.  A sometimes desirable alternative
  behavior is for the system to assume ~c[hyp'], apply the rule, and to
  generate explicitly a special case to show that ~c[hyp'] is true in the
  context.  This is called ``forcing'' ~c[hyp].  It can be arranged by
  restating the rule so that the offending hypothesis, ~c[hyp], is
  embedded in a call of ~c[force], as in ~c[(force hyp)].  By using the
  ~c[:]~ilc[corollary] field of the ~ilc[rule-classes] entry, a hypothesis
  can be forced without changing the statement of the theorem from
  which the rule is derived.

  Technically, ~c[force] is just a function of one argument that returns
  that argument.  It is generally ~il[enable]d and hence evaporates during
  simplification.  But its presence among the hypotheses of a
  conditional rule causes case splitting to occur if the hypothesis
  cannot be conventionally relieved.

  Since a forced hypothesis must be provable whenever the rule is
  otherwise applicable, forcing should be used only on hypotheses that
  are expected always to be true.

  A particularly common situation in which some hypotheses should be
  forced is in ``most general'' ~il[type-prescription] lemmas.  If a single
  lemma describes the ``expected'' type of a function, for all
  ``expected'' arguments, then it is probably a good idea to force the
  hypotheses of the lemma.  Thus, every time a term involving the
  function arises, the term will be given the expected type and its
  arguments will be required to be of the expected type.  In applying
  this advice it might be wise to avoid forcing those hypotheses that
  are in fact just type predicates on the arguments, since the routine
  that applies ~il[type-prescription] lemmas has fairly thorough knowledge
  of the types of all terms.

  ~c[Force] can have the additional benefit of causing the ACL2 typing
  mechanism to interact with the ACL2 rewriter to establish the
  hypotheses of ~il[type-prescription] rules.  To understand this remark,
  think of the ACL2 type reasoning system as a rather primitive
  rule-based theorem prover for questions about Common Lisp types,
  e.g., ``does this expression produce a ~ilc[consp]?''  ``does this
  expression produce some kind of ACL2 number, e.g., an ~ilc[integerp], a
  ~ilc[rationalp], or a ~ilc[complex-rationalp]?'' etc.  It is driven by
  ~il[type-prescription] rules.  To relieve the hypotheses of such rules,
  the type system recursively invokes itself.  This can be done for
  any hypothesis, whether it is ``type-like'' or not, since any
  proposition, ~c[p], can be phrased as the type-like question ``does ~c[p]
  produce an object of type ~c[nil]?''  However, as you might expect, the
  type system is not very good at establishing hypotheses that are not
  type-like, unless they happen to be assumed explicitly in the
  context in which the question is posed, e.g., ``If ~c[p] produces a
  ~ilc[consp] then does ~c[p] produce ~c[nil]?''  If type reasoning alone is
  insufficient to prove some instance of a hypothesis, then the
  instance will not be proved by the type system and a
  ~il[type-prescription] rule with that hypothesis will be inapplicable in
  that case.  But by embedding such hypotheses in ~c[force] expressions
  you can effectively cause the type system to ``punt'' them to the
  rest of the theorem prover.  Of course, as already noted, this
  should only be done on hypotheses that are ``always true.''  In
  particular, if rewriting is required to establish some hypothesis of
  a ~il[type-prescription] rule, then the rule will be found inapplicable
  because the hypothesis will not be established by type reasoning
  alone.

  The ACL2 rewriter uses the type reasoning system as a subsystem.  It
  is therefore possible that the type system will force a hypothesis
  that the rewriter could establish.  Before a forced hypothesis is
  reported out of the rewriter, we try to establish it by rewriting.

  This makes the following surprising behavior possible: A
  ~il[type-prescription] rule fails to apply because some true hypothesis
  is not being relieved.  The user changes the rule so as to ~st[force] the
  hypothesis.  The system then applies the rule but reports no
  forcing.  How can this happen?  The type system ``punted'' the
  forced hypothesis to the rewriter, which established it.

  Finally, we should mention that the rewriter is never willing to force when
  there is an ~ilc[if] term present in the goal being simplified.  Since
  ~ilc[and] terms and ~ilc[or] terms are merely abbreviations for ~ilc[if]
  terms, they also prevent forcing.  Note that ~ilc[if] terms are ultimately
  eliminated using the ordinary flow of the proof (but
  ~pl[set-case-split-limitations]), allowing ~c[force] ultimately to function
  as intended.  Moreover, forcing can be disabled, as described above; also
  ~pl[disable-forcing].~/"

; We define this function in :logic mode on the first pass so that it gets a
; nume.  See the comment in check-built-in-constants.

  (declare (xargs :mode :logic :guard t))

  x)

; See the comment in check-built-in-constants.

;; RAG - As promised by the comment above, this number had to be
;; changed to get ACL2 to compile.  The number "104" is magical.  I
;; figured it out by compiling ACL2, getting the error message that
;; said *force-xnume* should be "104" but wasn't, and then changed the
;; definition here.  The comment in check-built-in-constants explains
;; why we need to play this (apparently silly) game.

;; RAG - After adding the non-standard predicates, this number grew to 110.

(defconst *force-xnume*
  (let ((x 123))
    #+:non-standard-analysis
    (+ x 12)
    #-:non-standard-analysis
    x))

(defun immediate-force-modep ()

  ":Doc-Section Miscellaneous

  when executable counterpart is ~il[enable]d,
   ~il[force]d hypotheses are attacked immediately~/

  Also ~pl[disable-immediate-force-modep] and
  ~pl[enable-immediate-force-modep].

  This function symbol is defined simply to provide a ~il[rune] which can
  be ~il[enable]d and ~il[disable]d.  Enabling
  ~bv[]
  (:executable-counterpart immediate-force-modep)
  ~ev[]
  causes ACL2 to attack ~il[force]d hypotheses immediately instead of
  delaying them to the next forcing round.
  ~bv[]
  Example Hints
  :in-theory (disable (:executable-counterpart immediate-force-modep))
             ; delay forced hyps to forcing round
  :in-theory (enable (:executable-counterpart immediate-force-modep))
             ; split on forced hyps immediately~/
  ~ev[]
  ~l[force] for background information.  When a ~ilc[force]d
  hypothesis cannot be established a record is made of that fact and
  the proof continues.  When the proof succeeds a ``forcing round'' is
  undertaken in which the system attempts to prove each of the ~il[force]d
  hypotheses explicitly.  However, if the ~il[rune]
  ~c[(:executable-counterpart immediate-force-modep)] is ~il[enable]d at the
  time the hypothesis is ~il[force]d, then ACL2 does not delay the attempt
  to prove that hypothesis but undertakes the attempt more or less
  immediately."

; We make this function :common-lisp-compliant so that it gets a nume on pass 1
; of initialization.  See the comment in check-built-in-constants.

  (declare (xargs :mode :logic :guard t))

  "See :DOC immediate-force-modep.")

; See the comment in check-built-in-constants.

;; RAG - The value of "107" was modified as suggested during the
;; compilation of ACL2.  It's magic.  See the comment in
;; check-built-in-constants to find out more.

;; RAG - After adding the non-standard predicates, this changed to 113.

(defconst *immediate-force-modep-xnume*
  (+ *force-xnume* 3))

(defun case-split (x)

  ":Doc-Section Miscellaneous

  like force but immediately splits the top-level goal on the hypothesis~/

  When a hypothesis of a conditional rule has the form ~c[(case-split hyp)]
  it is logically equivalent to ~c[hyp].  However it affects the
  application of the rule generated as follows:  if ACL2
  attempts to apply the rule but cannot establish that the required
  instance of ~c[hyp] holds in the current context, it considers the
  hypothesis true anyhow, but (assuming all hypotheses are seen to be true and
  the rule is applied) creates a subgoal in which that instance of ~c[hyp] is
  assumed false.  (There are exceptions, noted below.)~/

  For example, given the rule
  ~bv[]
  (defthm p1->p2
    (implies (case-split (p1 x))
             (p2 x)))
  ~ev[]
  then an attempt to prove
  ~bv[]
  (implies (p3 x) (p2 (car x)))
  ~ev[]
  can give rise to a single subgoal:
  ~bv[]
  (IMPLIES (AND (NOT (P1 (CAR X))) (P3 X))
           (P2 (CAR X))).
  ~ev[]
  Unlike ~ilc[force], ~c[case-split] does not delay the ``false case'' to
  a forcing round but tackles it more or less immediately.

  The special ``split'' treatment of ~c[case-split] can be disabled by
  disabling forcing:  ~pl[force] for a discussion of disabling forcing, and
  also ~pl[disable-forcing].  Finally, we should mention that the rewriter is
  never willing to split when there is an ~ilc[if] term present in the goal
  being simplified.  Since ~ilc[and] terms and ~ilc[or] terms are merely
  abbreviations for ~ilc[if] terms, they also prevent splitting.  Note that
  ~ilc[if] terms are ultimately eliminated using the ordinary flow of the proof
  (but ~pl[set-case-split-limitations]), so ~c[case-split] will ultimately
  function as intended.

  When in the proof checker, ~c[case-split] behaves like ~c[force].~/"

; We define this function in :logic mode on the first pass so that it gets a
; nume.  See the comment in check-built-in-constants.

  (declare (xargs :mode :logic :guard t))

  x)

(in-theory (disable (:executable-counterpart immediate-force-modep)))

(defmacro disable-forcing nil

  ":Doc-Section Miscellaneous

  to disallow ~ilc[force]d ~ilc[case-split]s~/
  ~bv[]
  General Form:
  ACL2 !>:disable-forcing   ; disallow forced case splits
  ~ev[]
  ~l[force] and ~pl[case-split] for a discussion of forced case splits,
  which are inhibited by this command.~/

  ~c[Disable-forcing] is actually a macro that ~il[disable]s the executable
  counterpart of the function symbol ~c[force]; ~pl[force].  When
  you want to use ~il[hints] to turn off forced case splits, use a form such
  as:
  ~bv[]
  :in-theory (disable (:executable-counterpart force))
  ~ev[]
  "
  '(in-theory (disable (:executable-counterpart force))))

(defmacro enable-forcing nil

  ":Doc-Section Miscellaneous

  to allow ~ilc[force]d ~ilc[case split]s~/
  ~bv[]
  General Form:
  ACL2 !>:enable-forcing    ; allowed forced case splits
  ~ev[]
  ~l[force] and ~pl[case-split] for a discussion of ~il[force]d case splits,
  which are turned back on by this command.  (~l[disable-forcing] for how to
  turn them off.)~/

  ~c[Enable-forcing] is actually a macro that ~il[enable]s the executable
  counterpart of the function symbol ~c[force]; ~pl[force].  When
  you want to use ~il[hints] to turn on forced case splits, use a form such as:
  ~bv[]
  :in-theory (enable (:executable-counterpart force))
  ~ev[]
  "

  '(in-theory (enable (:executable-counterpart force))))

(defmacro disable-immediate-force-modep ()

  ":Doc-Section Miscellaneous

  ~il[force]d hypotheses are not attacked immediately~/
  ~bv[]
  General Form:
  ACL2 !>:disable-immediate-force-modep
  ~ev[]
  This event causes ACL2 to delay ~il[force]d hypotheses to the next forcing
  round, rather than attacking them immediately.  ~l[immediate-force-modep].
  Or for more basic information, first ~pl[force] for a discussion of
  ~il[force]d case splits.~/

  Disable-immediate-force-modep is a macro that ~il[disable]s the executable
  counterpart of the function symbol ~ilc[immediate-force-modep].  When
  you want to ~il[disable] this mode in ~il[hints], use a form such as:
  ~bv[]
  :in-theory (disable (:executable-counterpart immediate-force-modep))
  ~ev[]
  "

  '(in-theory (disable (:executable-counterpart immediate-force-modep))))

(defmacro enable-immediate-force-modep ()

  ":Doc-Section Miscellaneous

  ~il[force]d hypotheses are attacked immediately~/
  ~bv[]
  General Form:
  ACL2 !>:enable-immediate-force-modep
  ~ev[]
  This event causes ACL2 to attack ~il[force]d hypotheses immediately
  instead of delaying them to the next forcing round.
  ~l[immediate-force-modep].  Or for more basic information, first
  ~pl[force] for a discussion of ~il[force]d case splits.~/

  Enable-immediate-force-modep is a macro that ~il[enable]s the executable
  counterpart of the function symbol ~ilc[immediate-force-modep].  When
  you want to ~il[enable] this mode in ~il[hints], use a form such as:
  ~bv[]
  :in-theory (enable (:executable-counterpart immediate-force-modep))
  ~ev[]
  "

  '(in-theory (enable (:executable-counterpart immediate-force-modep))))

(defun synp (vars form term)

; Top-level calls of this function in the hypothesis of a linear or
; rewrite rule are given special treatment when relieving the rule's
; hypotheses.  (When the rule class gives such special treatment, it
; is an error to use synp in other than at the top-level.)  The
; special treatment is as follows.  Term is evaluated, binding state
; to the live state and mfc to the current metafunction context, as
; with meta rules.  The result of this evaluation should be either t,
; nil, or an alist binding variables to terms, else we get a hard
; error.  Moreover, if we get an alist then either (1) vars should be
; t, representing the set of all possible vars, and none of the keys
; in the alist should already be bound; or else (2) vars should be of
; the form (var1 ... vark), the keys of alist should all be among the
; vari, and none of vari should already be bound (actually this is
; checked when the rule is submitted) -- otherwise we get a hard
; error.

; As of Version_2.7 there are two macros that expand into calls to synp:

; (syntaxp form) ==>
; `(synp (quote nil) (quote (syntaxp ,form)) (quote (and ,form t)))

; (bind-free form &optional (vars 't)) ==>
;  (if vars
;      `(synp (quote ,vars) (quote (bind-free ,form ,vars)) (quote ,form))
;    `(synp (quote t) (quote (bind-free ,form)) (quote ,form))))

; Warning: This function must be defined to always return t in order
; for our treatment of it (in particular, in translate) to be sound.
; The special treatment referred to above happens within relieve-hyp.

  (declare (xargs :mode :logic :guard t)
           (ignore vars form term))
  t)

(defmacro syntaxp (form)
  (declare (xargs :guard t))
  ":Doc-Section Miscellaneous

  attach a heuristic filter on a ~c[:]~ilc[rewrite], ~c[:]~ilc[meta], or ~c[:]~ilc[linear] rule~/
  ~bv[]
  Example:
  Consider the :REWRITE rule created from

  (IMPLIES (SYNTAXP (NOT (AND (CONSP X)
                              (EQ (CAR X) 'NORM))))
           (EQUAL (LXD X)
                  (LXD (NORM X)))).
  ~ev[]
  The ~c[syntaxp] hypothesis in this rule will allow the rule to be
  applied to ~c[(lxd (trn a b))] but will not allow it to be applied to
  ~c[(lxd (norm a))].~/
  ~bv[]
  General Form:
  (SYNTAXP test)
  ~ev[]
  ~c[Syntaxp] always returns ~c[t] and so may be added as a vacuous
  hypothesis.  However, when relieving the hypothesis, the test
  ``inside'' the ~c[syntaxp] form is actually treated as a meta-level
  proposition about the proposed instantiation of the rule's variables
  and that proposition must evaluate to true (non-~c[nil]) to
  ``establish'' the ~c[syntaxp] hypothesis.

  Note that the test of a ~c[syntaxp] hypothesis does not, in general,
  deal with the meaning or semantics or values of the terms, but rather
  with their syntactic forms.  In the example above, the ~c[syntaxp]
  hypothesis allows the rule to be applied to every target of the form
  ~c[(lxd u)], provided ~c[u] is not of the form ~c[(norm v)].
  Observe that without this syntactic restriction the rule above could
  loop producing a sequence of increasingly complex targets ~c[(lxd a)],
  ~c[(lxd (norm a))], ~c[(lxd (norm (norm a)))], etc.  An intuitive reading
  of the rule might be ``~c[norm] the argument of ~c[lxd] unless it has
  already been ~c[norm]ed.''

  Note also that a ~c[syntaxp] hypothesis deals with the syntactic
  form used internally by ACL2, rather than that seen by the user.  In
  some cases these are the same, but there can be subtle differences
  with which the writer of a ~c[syntaxp] hypothesis must be aware.
  You can use ~c[:]~ilc[trans] to display this internal representation.

  There are two types of ~c[syntaxp] hypotheses.  The simpler type of
  ~c[syntaxp] hypothesis may be used as the nth hypothesis in a
  ~c[:]~ilc[rewrite] or ~c[:]~ilc[linear] rule whose ~c[:]~ilc[corollary] is
  ~c[(implies (and hyp1 ... hypn ... hypk) (equiv lhs rhs))]
  provided ~c[test] is a term, ~c[test] contains at least one variable, and
  every variable occuring freely in ~c[test] occurs freely in ~c[lhs] or in
  some ~c[hypi], ~c[i<n].  In addition, ~c[test] must not use any
  stobjs.  The case of ~c[:]~ilc[meta] rules is similar to the above, except
  that it applies to the result of applying the hypothesis metafunction.
  (Later below we will describe the second type, an ~em[extended] ~c[syntaxp]
  hypothesis, which may use ~ilc[state].)

  We illustrate the use of simple ~c[syntaxp] hypotheses by slightly
  elaborating the example given above.  Consider a ~c[:]~ilc[rewrite]
  rule whose ~c[:]~ilc[corollary] is:
  ~bv[]
  (IMPLIES (AND (RATIONALP X)
                (SYNTAXP (NOT (AND (CONSP X)
                                   (EQ (CAR X) 'NORM)))))
           (EQUAL (LXD X)
                  (LXD (NORM X))))
  ~ev[]
  How is this rule applied to ~c[(lxd (trn a b))]?  First, we form a
  substitution that instantiates the left-hand side of the conclusion
  of the rule so that it is identical to the target term.  In the
  present case, the substitution replaces ~c[x] with ~c[(trn a b)].
  ~bv[]
  (LXD X) ==> (LXD (trn a b)).
  ~ev[]
  Then we backchain to establish the hypotheses, in order.  Ordinarily this
  means that we instantiate each hypothesis with our substitution and
  then attempt to rewrite the resulting instance to true.
  Thus, in order to relieve the first hypothesis above, we rewrite
  ~bv[]
  (RATIONALP (trn a b)).
  ~ev[]
  If this rewrites to true, we continue.

  Of course, most users are aware of some exceptions to this general
  description of the way we relieve hypotheses.  For
  example, if a hypothesis contains a ``free-variable'' ~-[] one not
  bound by the current substitution ~-[] we attempt to extend the
  substitution by searching for an instance of the hypothesis among
  known truths.  ~l[free-variables].  ~ilc[Force]d hypotheses are another exception to the
  general rule of how hypotheses are relieved.

  Hypotheses marked with ~c[syntaxp], as in ~c[(syntaxp test)], are
  also exceptions.  We instantiate such a hypothesis; but instead of
  rewriting the instantiated instance, we evaluate the instantiated
  ~c[test].  More precisely, we evaluate ~c[test] in an environment in
  which its variable symbols are bound to the quotations of the terms
  to which those variables are bound in the instantiating
  substitution.  So in the case in point, we (in essence) evaluate
  ~bv[]
  (NOT (AND (CONSP '(trn a b)) (EQ (CAR '(trn a b)) 'NORM))).
  ~ev[]
  This clearly evaluates to ~c[t].  When a ~c[syntaxp] test evaluates
  to true, we consider the ~c[syntaxp] hypothesis to have been
  established; this is sound because logically ~c[(syntaxp test)] is
  ~c[t] regardless of ~c[test].  If the test evaluates to ~c[nil] (or
  fails to evaluate because of ~il[guard] violations) we act as though
  we cannot establish the hypothesis and abandon the attempt to apply
  the rule; it is always sound to give up.

  The acute reader will have noticed something odd about the form
  ~bv[]
  (NOT (AND (CONSP '(trn a b)) (EQ (CAR '(trn a b)) 'NORM))).
  ~ev[]
  When relieving the first hypothesis, ~c[(RATIONALP X)], we substituted
  ~c[(trn a b)] for ~c[X]; but when relieving the second hypothesis,
  ~c[(SYNTAXP (NOT (AND (CONSP X) (EQ (CAR X) 'NORM))))], we substituted the
  quotation of ~c[(trn a b)] for ~c[X].  Why the difference?  Remember
  that in the first hypothesis we are talking about the value of
  ~c[(trn a b)] ~-[] is it rational ~-[] while in the second one we are
  talking about its syntactic form.  Remember also that Lisp, and hence
  ACL2, evaluates the arguments to a function before applying the function
  to the resulting values. Thus, we are asking ``Is the list ~c[(trn a b)]
  a ~ilc[consp] and if so, is its ~ilc[car] the symbol ~c[NORM]?''  The
  ~c[quote]s on both ~c[(trn a b)] and ~c[NORM] are therefore necessary.
  One can verify this by defining ~c[trn] to be, say ~ilc[cons], and then
  evaluating forms such as
  ~bv[]
  (AND (CONSP '(trn a b)) (EQ (CAR '(trn a b)) 'NORM))
  (AND (CONSP (trn a b)) (EQ (CAR (trn a b)) NORM))
  (AND (CONSP (trn 'a 'b)) (EQ (CAR (trn 'a 'b)) NORM))
  (AND (CONSP '(trn a b)) (EQ '(CAR (trn a b)) ''NORM))
  ~ev[]
  at the top-level ACL2 prompt.

  ~l[syntaxp-examples] for more examples of the use of ~c[syntaxp].

  An extended ~c[syntaxp] hypothesis is similar to the simple type
  described above, but it uses two additional variables, ~c[mfc] and ~c[state],
  which must not be bound by the left hand side or an earlier hypothesis
  of the rule.  They must be the last two variables mentioned by ~c[form];
  first ~c[mfc], then ~c[state].  These two variables give access to
  the functions ~c[mfc-]xxx; ~pl[extended-metafunctions].  As
  described there, ~c[mfc] is bound to the so-called
  metafunction-context and ~c[state] to ACL2's ~ilc[state].
  ~l[syntaxp-examples] for an example of the use of these extended
  ~c[syntaxp] hypotheses."

  `(synp (quote nil) (quote (syntaxp ,form)) (quote (and ,form t))))

(deflabel syntaxp-examples

  :doc
  ":Doc-Section Syntaxp

  examples pertaining to syntaxp hypotheses~/

  ~l[syntaxp] for a basic discussion of the use of ~c[syntaxp] to control
  rewriting.~/

  A common syntactic restriction is
  ~bv[]
  (SYNTAXP (AND (CONSP X) (EQ (CAR X) 'QUOTE)))
  ~ev[]
  or, equivalently,
  ~bv[]
  (SYNTAXP (QUOTEP X)).
  ~ev[]
  A rule with such a hypothesis can be applied only if ~c[x] is bound to
  a specific constant.  Thus, if ~c[x] is ~c[23] (which is actually
  represented internally as ~c[(quote 23)]), the test evaluates to ~c[t]; but
  if ~c[x] prints as ~c[(+ 11 12)] then the test evaluates to ~c[nil]
  (because ~c[(car x)] is the symbol ~ilc[binary-+]).  We see the use
  of this restriction in the rule
  ~bv[]
  (implies (and (syntaxp (quotep c))
                (syntaxp (quotep d)))
           (equal (+ c d x)
                  (+ (+ c d) x))).
  ~ev[]
  If ~c[c] and ~c[d] are constants, then the
  ~ilc[executable-counterpart] of ~ilc[binary-+] will evaluate the sum
  of ~c[c] and ~c[d].  For instance, under the influence of this rule
  ~bv[]
  (+ 11 12 foo)
  ~ev[]
  rewrites to
  ~bv[]
  (+ (+ 11 12) foo)
  ~ev[]
  which in turn rewrites to ~c[(+ 23 foo)].  Without the syntactic
  restriction, this rule would loop with the built-in rules
  ~c[ASSOCIATIVITY-OF-+] or ~c[COMMUTATIVITY-OF-+].

  We here recommend that the reader try the affects of entering expressions
  such as the following at the top level ACL2 prompt.
  ~bv[]
  (+ 11 23)
  (+ '11 23)
  (+ '11 '23)
  (+ ''11 ''23)
  :trans (+ 11 23)
  :trans (+ '11 23)
  :trans (+ ''11 23)
  :trans (+ c d x)
  :trans (+ (+ c d) x)
  ~ev[]
  We also recommend that the reader verify our claim above about looping
  by trying the affect of each of the following rules individually.
  ~bv[]
  (defthm good
     (implies (and (syntaxp (quotep c))
                   (syntaxp (quotep d)))
              (equal (+ c d x)
                     (+ (+ c d) x))))

  (defthm bad
     (implies (and (acl2-numberp c)
                   (acl2-numberp d))
              (equal (+ c d x)
                     (+ (+ c d) x))))
  ~ev[]
  on (the false) theorems:
  ~bv[]
  (thm
    (equal (+ 11 12 x) y))

  (thm
    (implies (and (acl2-numberp c)
                  (acl2-numberp d)
                  (acl2-numberp x))
             (equal (+ c d x) y))).
  ~ev[]
  One can use ~c[:]~ilc[brr], perhaps in conjunction with
  ~ilc[cw-gstack], to investigate any looping.

  Here is a simple example showing the value of rule ~c[good] above.  Without
  ~c[good], the ~c[thm] form below fails.
  ~bv[]
  (defstub foo (x) t)

  (thm (equal (foo (+ 3 4 x)) (foo (+ 7 x))))
  ~ev[]

  The next three examples further explore the use of ~c[quote] in
  ~ilc[syntaxp] hypotheses.

  We continue the examples of ~ilc[syntaxp] hypotheses with a rule from
  ~c[books/finite-set-theory/set-theory.lisp].  We will not discuss
  here the meaning of this rule, but it is necessary to point out that
  ~c[(ur-elementp nil)] is true in this book.
  ~bv[]
  (defthm scons-nil
    (implies (and (syntaxp (not (equal a ''nil)))
                  (ur-elementp a))
             (= (scons e a)
                (scons e nil)))).
  ~ev[]
  Here also, ~ilc[syntaxp] is used to prevent looping.  Without the
  restriction, ~c[(scons e nil)] would be rewritten to itself since
  ~c[(ur-elementp nil)] is true.~nl[]
  Question: Why the use of two quotes in ~c[''nil]?~nl[]
  Hints: ~c[Nil] is a constant just as 23 is.  Try ~c[:trans (cons a nil)],
  ~c[:trans (cons 'a 'nil)], and ~c[:trans (cons ''a ''nil)].
  Also, don't forget that the arguments to a function are evaluated before
  the function is applied.

  The next two rules move negative constants to the other side of an
  inequality.
  ~bv[]
  (defthm |(< (+ (- c) x) y)|
    (implies (and (syntaxp (quotep c))
                  (syntaxp (< (cadr c) 0))
                  (acl2-numberp y))
             (equal (< (+ c x) y)
                    (< (fix x) (+ (- c) y)))))

  (defthm |(< y (+ (- c) x))|
    (implies (and (syntaxp (quotep c))
                  (syntaxp (< (cadr c) 0))
                  (acl2-numberp y))
             (equal (< y (+ c x))
                    (< (+ (- c) y) (fix x)))))
  ~ev[]
  Questions: What would happen if ~c[(< (cadr c) '0)] were used?
  What about ~c[(< (cadr c) ''0)]?

  One can also use ~c[syntaxp] to restrict the application of a rule
  to a particular set of variable bindings as in the following taken from
  ~c[books/ihs/quotient-remainder-lemmas.lisp].
  ~bv[]
  (encapsulate ()

    (local
     (defthm floor-+-crock
       (implies
        (and (real/rationalp x)
             (real/rationalp y)
             (real/rationalp z)
             (syntaxp (and (eq x 'x) (eq y 'y) (eq z 'z))))
        (equal (floor (+ x y) z)
               (floor (+ (+ (mod x z) (mod y z))
                         (* (+ (floor x z) (floor y z)) z)) z)))))

    (defthm floor-+
      (implies
       (and (force (real/rationalp x))
            (force (real/rationalp y))
            (force (real/rationalp z))
            (force (not (equal z 0))))
       (equal (floor (+ x y) z)
              (+ (floor (+ (mod x z) (mod y z)) z)
                 (+ (floor x z) (floor y z))))))

    )
  ~ev[]
  We recommend the use of ~c[:]~c[brr] to investigate the use of
  ~c[floor-+-crock].

  Another useful restriction is defined by
  ~bv[]
  (defun rewriting-goal-literal (x mfc state)

    ;; Are we rewriting a top-level goal literal, rather than rewriting
    ;; to establish a hypothesis from a rewrite (or other) rule?

    (declare (ignore x state))
    (null (access metafunction-context mfc :ancestors))).
  ~ev[]
  We use this restriction in the rule
  ~bv[]
  (defthm |(< (* x y) 0)|
      (implies (and (syntaxp (rewriting-goal-literal x mfc state))
                    (rationalp x)
                    (rationalp y))
               (equal (< (* x y) 0)
                      (cond ((equal x 0)
                             nil)
                            ((equal y 0)
                             nil)
                            ((< x 0)
                             (< 0 y))
                            ((< 0 x)
                             (< y 0))))))
  ~ev[]
  which has been found to be useful, but which also leads to excessive
  thrashing in the linear arithmetic package if used indiscriminately.

  ~l[extended-metafunctions] for information on the use of ~c[mfc]
  and ~c[metafunction-context].

  ~/")

(defmacro bind-free (form &optional (vars))
  (declare (xargs :guard (or (eq vars nil)
                             (eq vars t)
                             (and (symbol-listp vars)
                                  (not (member-eq t vars))
                                  (not (member-eq nil vars))))))
  ":Doc-Section Miscellaneous

  to bind free variables of a rewrite or linear rule~/
  ~bv[]
  Examples:
  (IMPLIES (AND (RATIONALP LHS)
                (RATIONALP RHS)
                (BIND-FREE (FIND-MATCH-IN-PLUS-NESTS LHS RHS) (X)))
           (EQUAL (EQUAL LHS RHS)
                  (EQUAL (+ (- X) LHS) (+ (- X) RHS))))

  (IMPLIES (AND (BIND-FREE
                  (FIND-RATIONAL-MATCH-IN-TIMES-NESTS LHS RHS MFC STATE)
                  (X))
                (RATIONALP X)
                (CASE-SPLIT (NOT (EQUAL X 0))))
           (EQUAL (< LHS RHS)
                  (IF (< 0 X)
                      (< (* (/ X) LHS) (* (/ X) RHS))
                     (< (* (/ X) RHS) (* (/ X) LHS)))))
  ~ev[]~/
  General Forms:
  ~bv[]
  (BIND-FREE term var-list)
  (BIND-FREE term t)
  (BIND-FREE term)
  ~ev[]
  A rule which uses a ~c[bind-free] hypothesis has similarities to both a
  rule which uses a ~ilc[syntaxp] hypothesis and to a ~c[:]~ilc[meta] rule.
  ~c[Bind-free] is like ~ilc[syntaxp], in that it logically always
  returns ~c[t] but may affect the application of a ~c[:]~ilc[rewrite]
  or ~c[:]~ilc[linear] rule when it is called at the top-level of a
  hypothesis.  It is like a ~c[:]~ilc[meta] rule, in that it allows the
  user to perform transformations of terms under progammatic control.

  Note that a ~c[bind-free] hypothesis does not, in general, deal with the
  meaning or semantics or values of the terms, but rather with their
  syntactic forms.  Before attempting to write a rule which uses
  ~c[bind-free], the user should be familiar with ~ilc[syntaxp] and the
  internal form that ACL2 uses for terms.  This internal form is
  similar to what the user sees, but there are subtle and important
  differences.  ~ilc[Trans] can be used to view this internal form.

  Just as for a ~ilc[syntaxp] hypothesis, there are two types of
  ~c[bind-free] hypotheses.  The simpler type of ~c[bind-free]
  hypothesis may be used as the nth hypothesis in a ~c[:]~ilc[rewrite]
  or ~c[:]~ilc[linear] rule whose ~c[:]~ilc[corollary] is
  ~c[(implies (and hyp1 ... hypn ... hypk) (equiv lhs rhs))] provided ~c[term]
  is a term, ~c[term] contains at least one variable, and every variable
  occuring freely in ~c[term] occurs freely in ~c[lhs] or in some ~c[hypi],
  ~c[i<n].  In addition, ~c[term] must not use any stobjs.
  (Later below we will describe the second type, an ~em[extended]
  ~c[bind-free] hypothesis, which may use ~ilc[state].)

  We begin our description of ~c[bind-free] by examining the
  first example above in some detail.

  We wish to write a rule which will cancel ``like'' addends from both
  sides of an equality.  Clearly, one could write a series of rules such
  as
  ~bv[]
  (DEFTHM THE-HARD-WAY-2-1
     (EQUAL (EQUAL (+ A X B)
                   (+ X C))
            (EQUAL (+ A B)
                   (FIX C))))
  ~ev[]
  with one rule for each combination of positions the matching addends
  might be found in (if one knew before-hand the maximum number of
  addends that would appear in a sum).  But there is a better way.
  (In what follows, we assume the presence of an appropriate set of rules
  for simplifying sums.)

  Consider the following definitions and theorem:
  ~bv[]
  (DEFUN INTERSECTION-EQUAL (X Y)
    (COND ((ENDP X)
           NIL)
          ((MEMBER-EQUAL (CAR X) Y)
           (CONS (CAR X) (INTERSECTION-EQUAL (CDR X) Y)))
          (T
           (INTERSECTION-EQUAL (CDR X) Y))))

  (DEFUN PLUS-LEAVES (TERM)
    (IF (EQ (FN-SYMB TERM) 'BINARY-+)
        (CONS (FARGN TERM 1)
              (PLUS-LEAVES (FARGN TERM 2)))
      (LIST TERM)))

  (DEFUN FIND-MATCH-IN-PLUS-NESTS (LHS RHS)
    (IF (AND (EQ (FN-SYMB LHS) 'BINARY-+)
             (EQ (FN-SYMB RHS) 'BINARY-+))
        (LET ((COMMON-ADDENDS (INTERSECTION-EQUAL (PLUS-LEAVES LHS)
                                                  (PLUS-LEAVES RHS))))
          (IF COMMON-ADDENDS
              (LIST (CONS 'X (CAR COMMON-ADDENDS)))
            NIL))
      NIL))

  (DEFTHM CANCEL-MATCHING-ADDENDS-EQUAL
    (IMPLIES (AND (RATIONALP LHS)
                  (RATIONALP RHS)
                  (BIND-FREE (FIND-MATCH-IN-PLUS-NESTS LHS RHS) (X)))
             (EQUAL (EQUAL LHS RHS)
                    (EQUAL (+ (- X) LHS) (+ (- X) RHS)))))
  ~ev[]

  How is this rule applied to the following term?
  ~bv[]
  (equal (+ 3 (expt a n) (foo a c))
         (+ (bar b) (expt a n)))
  ~ev[]
  As mentioned above, the internal form of an ACL2 term is not always
  what one sees printed out by ACL2.  In this case, by using ~c[:]~ilc[trans]
  one can see that the term is stored internally as
  ~bv[]
  (equal (binary-+ '3
                   (binary-+ (expt a n) (foo a c)))
         (binary-+ (bar b) (expt a n))).
  ~ev[]

  When ACL2 attempts to apply ~c[cancel-matching-addends-equal] to the
  term under discussion, it first forms a substitution that instantiates
  the left-hand side of the conclusion so that it is identical to the
  target term.  This substitution is kept track of by the substitution
  alist:
  ~bv[]
  ((LHS . (binary-+ '3
                     (binary-+ (expt a n) (foo a c))))
   (RHS . (binary-+ (bar b) (expt a n)))).
  ~ev[]
  ACL2 then attempts to relieve the hypotheses in the order they were
  given. Ordinarily this means that we instantiate each hypothesis
  with our substitution and then attempt to rewrite the resulting
  instance to true.  Thus, in order to relieve the first hypothesis,
  we rewrite:
  ~bv[]
  (RATIONALP (binary-+ '3
                        (binary-+ (expt a n) (foo a c)))).
  ~ev[]
  Let us assume that the first two hypotheses rewrite to ~c[t].  How
  do we relieve the ~c[bind-free] hypothesis?  Just as for a ~ilc[syntaxp]
  hypothesis, ACL2 evaluates ~c[(find-match-in-plus-nests lhs rhs)]
  in an environment where ~c[lhs] and ~c[rhs] are instantiated as determined
  by the substitution.  In this case we evaluate
  ~bv[]
  (FIND-MATCH-IN-PLUS-NESTS '(binary-+ '3
                                        (binary-+ (expt a n) (foo a c)))
                            '(binary-+ (bar b) (expt a n))).
  ~ev[]
  Observe that, just as in the case of a ~ilc[syntaxp] hypothesis, we
  substitute the quotation of the variables bindings into the term to be
  evaluated.  ~l[syntaxp] for the reasons for this.  The result of this
  evaluation, ~c[((X . (EXPT A N)))], is then used to extend the
  substitution alist:
  ~bv[]
  ((X . (EXPT A N))
   (LHS . (binary-+ '3
                     (binary-+ (expt a n) (foo a c))))
   (RHS . (binary-+ (bar b) (expt a n)))),
  ~ev[]
  and this extended substitution determines
  ~c[cancel-matching-addends-equal]'s result:
  ~bv[]
  (EQUAL (+ (- X) LHS) (+ (- X) RHS))
  ==>
  (EQUAL (+ (- (EXPT A N)) 3 (EXPT A N) (FOO A C))
         (+ (- (EXPT A N)) (BAR B) (EXPT A N))).
  ~ev[]
  Question: What is the internal form of this result?~nl[]
  Hint: Use ~c[:]~ilc[trans].

  When this rule fires, it adds the negation of a common term
  to both sides of the equality by selecting a binding for the
  otherwise-free variable ~c[x], under programmatic control.  Note
  that other mechanisms such as the binding of ~il[free-variables]
  may also extend the substitution alist.

  Just as for a ~ilc[syntaxp] test, a ~c[bind-free] form signals
  failure by returning ~c[nil].  However, while a ~ilc[syntaxp] test
  signals success by returning true, a ~c[bind-free] form signals
  success by returning an alist which is used to extend the current
  substitution alist.  Because of this use of the alist, there are
  several restrictions on it ~-[] in particular the alist must only
  bind variables, these variables must not be already bound by the
  substitution alist, and the variables must be bound to ACL2 terms.
  If ~c[term] returns an alist and the alist meets these restrictions,
  we append the alist to the substitution alist and use the result as
  the new current substitution alist.  This new current
  substitution alist is then used when we attempt to relieve the next
  hypothesis or, if there are no more, instantiate the right hand side
  of the rule.

  There is also a second, optional, ~c[var-list] argument to a ~c[bind-free]
  hypothesis.  If provided, it must be either ~c[t] or a list of variables.  If
  it is not provided, it defaults to ~c[t].  If it is a list of variables, this
  second argument is used to place a further restriction on the possible values
  of the alist to be returned by ~c[term]: any variables bound in the alist
  must be present in the list of variables.  We strongly recommend the use of
  this list of variables, as it allows some consistency checks to be performed
  at the time of the rule's admittance which are not possible otherwise.

  An extended ~c[bind-free] hypothesis is similar to the simple type
  described above, but it uses two additional variables, ~c[mfc] and ~c[state],
  which must not be bound by the left hand side or an earlier hypothesis
  of the rule.  They must be the last two variables mentioned by ~c[term]:
  first ~c[mfc], then ~c[state].  These two variables give access to
  the functions ~c[mfc-]xxx; ~pl[extended-metafunctions].  As
  described there, ~c[mfc] is bound to the so-called
  metafunction-context and ~c[state] to ACL2's ~ilc[state].  ~l[bind-free-examples]
  for examples of the use of these extended ~c[bind-free] hypotheses.~/"

  (if vars
      `(synp (quote ,vars) (quote (bind-free ,form ,vars)) (quote ,form))
    `(synp (quote t) (quote (bind-free ,form)) (quote ,form))))

(deflabel bind-free-examples

  :doc
  ":Doc-Section Bind-free

  examples pertaining to ~ilc[bind-free] hypotheses~/

  ~l[bind-free] for a basic discussion of the use of ~c[bind-free] to control
  rewriting.~/

  We give examples of the use of ~ilc[bind-free] hypotheses from the
  perspective of a user interested in reasoning about arithmetic, but
  it should be clear that ~ilc[bind-free] can be used for many other
  purposes also.

  EXAMPLE 1:  Cancel a common factor.

  ~bv[]
  (defun bind-divisor (a b)

  ; If a and b are polynomials with a common factor c, we return a
  ; binding for x.  We could imagine writing get-factor to compute the
  ; gcd, or simply to return a single non-invertible factor.

    (let ((c (get-factor a b)))
      (and c (list (cons 'x c)))))

  (defthm cancel-factor
    ;; We use case-split here to ensure that, once we have selected
    ;; a binding for x, the rest of the hypotheses will be relieved.
    (implies (and (acl2-numberp a)
                  (acl2-numberp b)
                  (bind-free (bind-divisor a b) (x))
                  (case-split (not (equal x 0)))
                  (case-split (acl2-numberp x)))
             (iff (equal a b)
                  (equal (/ a x) (/ b x)))))
  ~ev[]

  EXAMPLE 2:  Pull integer summand out of floor.  Note:  This example
  has an ~em[extended] ~ilc[bind-free] hypothesis, which uses the term
  ~c[(find-int-in-sum sum mfc state)].

  ~bv[]
  (defun fl (x)
    ;; This function is defined, and used, in the IHS books.
    (floor x 1))

  (defun int-binding (term mfc state)
    ;; The call to mfc-ts returns the encoded type of term.
    ;; Thus, we are asking if term is known by type reasoning to
    ;; be an integer.
    (declare (xargs :stobjs (state) :mode :program))
    (if (ts-subsetp (mfc-ts term mfc state)
                    *ts-integer*)
          (list (cons 'int term))
    nil))

  (defun find-int-in-sum (sum mfc state)
    (declare (xargs :stobjs (state) :mode :program))
    (if (and (nvariablep sum)
             (not (fquotep sum))
             (eq (ffn-symb sum) 'binary-+))
        (or (int-binding (fargn sum 1) mfc state)
            (find-int-in-sum (fargn sum 2) mfc state))
      (int-binding sum mfc state)))

  ; Some additional work is required to prove the following.  So for
  ; purposes of illustration, we wrap skip-proofs around the defthm.

  (skip-proofs
   (defthm cancel-fl-int
    ;; The use of case-split is probably not needed, since we should
    ;; know that int is an integer by the way we selected it.  But this
    ;; is safer.
     (implies (and (acl2-numberp sum)
                   (bind-free (find-int-in-sum sum mfc state) (int))
                   (case-split (integerp int)))
              (equal (fl sum)
                     (+ int (fl (- sum int)))))
     :rule-classes ((:rewrite :match-free :all)))
  )

  ; Arithmetic libraries will have this sort of lemma.
  (defthm hack (equal (+ (- x) x y) (fix y)))

  (in-theory (disable fl))

  (thm (implies (and (integerp x) (acl2-numberp y))
                (equal (fl (+ x y)) (+ x (fl y)))))

  ~ev[]

  EXAMPLE 3:  Simplify terms such as (equal (+ a (* a b)) 0)

  ~bv[]
  (defun factors (product)
    ;; We return a list of all the factors of product.  We do not
    ;; require that product actually be a product.
    (if (eq (fn-symb product) 'BINARY-*)
        (cons (fargn product 1)
              (factors (fargn product 2)))
      (list product)))

  (defun make-product (factors)
    ;; Factors is assumed to be a list of ACL2 terms.  We return an
    ;; ACL2 term which is the product of all the ellements of the
    ;; list factors.
    (cond ((atom factors)
           ''1)
          ((null (cdr factors))
           (car factors))
          ((null (cddr factors))
           (list 'BINARY-* (car factors) (cadr factors)))
          (t
           (list 'BINARY-* (car factors) (make-product (cdr factors))))))

  (defun quotient (common-factors sum)
    ;; Common-factors is a list of ACL2 terms.   Sum is an ACL2 term each
    ;; of whose addends have common-factors as factors.  We return
    ;; (/ sum (make-product common-factors)).
    (if (eq (fn-symb sum) 'BINARY-+)
        (let ((first (make-product (set-difference-equal (factors (fargn sum 1))
                                                         common-factors))))
          (list 'BINARY-+ first (quotient common-factors (fargn sum 2))))
      (make-product (set-difference-equal (factors sum)
                                          common-factors))))

  (defun intersection-equal (x y)
    (cond ((endp x)
           nil)
          ((member-equal (car x) y)
           (cons (car x) (intersection-equal (cdr x) y)))
          (t
           (intersection-equal (cdr x) y))))

  (defun common-factors (factors sum)
    ;; Factors is a list of the factors common to all of the addends
    ;; examined so far.  On entry, factors is a list of the factors in
    ;; the first addend of the original sum, and sum is the rest of the
    ;; addends.  We sweep through sum, trying to find a set of factors
    ;; common to all the addends of sum.
    (declare (xargs :measure (acl2-count sum)))
    (cond ((null factors)
           nil)
          ((eq (fn-symb sum) 'BINARY-+)
           (common-factors (intersection-equal factors (factors (fargn sum 1)))
                           (fargn sum 2)))
          (t
           (intersection-equal factors (factors sum)))))

  (defun simplify-terms-such-as-a+ab-rel-0-fn (sum)
    ;; If we can find a set of factors common to all the addends of sum,
    ;; we return an alist binding common to the product of these common
    ;; factors and binding quotient to (/ sum common).
    (if (eq (fn-symb sum) 'BINARY-+)
        (let ((common-factors (common-factors (factors (fargn sum 1))
                                              (fargn sum 2))))
          (if common-factors
              (let ((common (make-product common-factors))
                    (quotient (quotient common-factors sum)))
                (list (cons 'common common)
                      (cons 'quotient quotient)))
            nil))
      nil))

  (defthm simplify-terms-such-as-a+ab-=-0
    (implies (and (bind-free
                   (simplify-terms-such-as-a+ab-rel-0-fn sum)
                   (common quotient))
                  (case-split (acl2-numberp common))
                  (case-split (acl2-numberp quotient))
                  (case-split (equal sum
                                     (* common quotient))))
             (equal (equal sum 0)
                    (or (equal common 0)
                        (equal quotient 0)))))

  (thm (equal (equal (+ u (* u v)) 0)
        (or (equal u 0) (equal v -1))))
  ~ev[]")

(defun extra-info (x y)
  (declare (ignore x y)
           (xargs :guard t))
  t)

(in-theory (disable extra-info (extra-info) (:type-prescription extra-info)))

(defconst *extra-info-fn*

; If this symbol changes, then change *acl2-exports* and the documentation for
; xargs and verify-guards accordingly.

  'extra-info)

; The following axiom can be proved.  John Cowles has proved some of these and
; we have proved others in our efforts to verify the guards in our code.
; Eventually we will replace some of these axioms by theorems.  But not now
; because things are too fluid.

;; RAG - This axiom was strengthened to include the reals.  Amusingly,
;; it was also weakened, since it leaves open the possibility that for
;; rational x, x*x is irrational.  Luckily, the type-system knows this
;; isn't the case, so hopefully we have not weakened ACL2.

(defaxiom nonnegative-product

; Note that in (* x x), x might be complex.  So, we do not want to force the
; hypothesis below.

  (implies (real/rationalp x)
           (and (real/rationalp (* x x))
                (<= 0 (* x x))))

; We need the :type-prescription rule class below.  Without it, ACL2 cannot
; prove (implies (rationalp x) (<= 0 (* x x))); primitive type-set reasoning
; will not notice that both arguments of * are identical.

  :rule-classes ((:type-prescription
                  :typed-term (* x x))))

#|
(add-schema Induction Schema
            (and (implies (not (integerp x)) (p x))
                 (p 0)
                 (implies (and (integerp x)
                               (< 0 x)
                               (p (- x 1)))
                          (p x))
                 (implies (and (integerp x)
                               (< x 0)
                               (p (+ x 1)))
                          (p x)))
            (p x))

|#

(defaxiom Integer-0
  (integerp 0)
  :rule-classes nil)

(defaxiom Integer-1
  (integerp 1)
  :rule-classes nil)

(defaxiom Integer-step
  (implies (integerp x)
           (and (integerp (+ x 1))
                (integerp (+ x -1))))
  :rule-classes nil)

(defaxiom Lowest-Terms
  (implies (and (integerp n)
                (rationalp x)
                (integerp r)
                (integerp q)
                (< 0 n)
                (equal (numerator x) (* n r))
                (equal (denominator x) (* n q)))
           (equal n 1))
  :rule-classes nil)

#|

the following predicates are disjoint:
  (((acl2-numberp x)
    (complex-rationalp x)
    ((rationalp x)
     ((integerp x) (< 0 x) (equal x 0) (< x 0))
     ((not (integerp x)) (< 0 x) (< x 0))))
   ((consp x) (proper-consp x) (improper-consp x))
   ((symbolp x) (equal x nil) (equal x T) (not (or (equal x T)
                                                   (equal x NIL))))
   (stringp x)
   (characterp x)
   (other-kinds-of-objects))


; For each of the primitives we have the axiom that when their guards
; are unhappy, the result is given by apply.  This is what permits us
; to replace unguarded terms by apply's.  E.g.,

(defaxiom +-guard
  (implies (or (not (rationalp x))
               (not (rationalp y)))
           (equal (+ x y)
                  (apply '+ (list x y)))))
|#

(defaxiom car-cdr-elim
  (implies (consp x)
           (equal (cons (car x) (cdr x)) x))
  :rule-classes :elim)

(defaxiom car-cons (equal (car (cons x y)) x))

(defaxiom cdr-cons (equal (cdr (cons x y)) y))

(defaxiom cons-equal
  (equal (equal (cons x1 y1) (cons x2 y2))
         (and (equal x1 x2)
              (equal y1 y2))))

#|
Induction Schema:   (and (implies (not (consp x)) (p x))
                         (implies (and (consp x) (p (car x)) (p (cdr x)))
                                  (p x)))
                    ----------------------------------------------
                    (p x)


|#

(defaxiom booleanp-characterp
  (booleanp (characterp x))
  :rule-classes nil)

(defaxiom characterp-page
  (characterp #\Page)
  :rule-classes nil)

(defaxiom characterp-tab
  (characterp #\Tab)
  :rule-classes nil)

(defaxiom characterp-rubout
  (characterp #\Rubout)
  :rule-classes nil)

#+acl2-loop-only
(defun member (x l)

  ":Doc-Section ACL2::Programming

  membership predicate, using ~ilc[eql] as test~/

  ~c[(Member x l)] equals the longest tail of ~c[l] that begins with
  ~c[x], or else ~c[nil] if no such tail exists.~/

  ~c[(Member x l)] is provably the same in the ACL2 logic as
  ~c[(member-equal x l)].  It has a stronger ~il[guard] than ~ilc[member-equal]
  because uses ~ilc[eql] to test for whether ~c[x] is equal to a given
  member of ~c[l].  Its ~il[guard] requires that ~c[l] is a true list, and
  moreover, either ~c[(eqlablep x)] or all members of ~c[l] are
  ~ilc[eqlablep].  ~l[member-equal] and ~pl[member-eq].

  ~c[Member] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  Since ACL2 functions cannot
  take keyword arguments (though macros can), the ACL2 functions
  ~ilc[member-equal] and ~ilc[member-eq] are defined to correspond to calls
  of the Common Lisp function ~c[member] whose keyword argument
  ~c[:test] is ~ilc[equal] or ~ilc[eq], respectively.~/"

  (declare (xargs :guard (if (eqlablep x)
                             (true-listp l)
                           (eqlable-listp l))))
  (cond ((endp l)

; We return nil rather than l in this case, because we want to be able to use
; member as a predicate.

         nil)
        ((eql x (car l)) l)
        (t (member x (cdr l)))))

(defun no-duplicatesp (l)

  ":Doc-Section ACL2::Programming

  check for duplicates in a list (using ~c[eql] for equality)~/

  ~c[(no-duplicatesp l)] is true if and only if no member of ~c[l]
  occurs twice in ~c[l].~/

  ~c[(no-duplicatesp l)] has a ~il[guard] of ~c[(eqlable-listp l)].
  Membership is tested using ~ilc[member], hence using ~ilc[eql] as
  the test.~/"

  (declare (xargs :guard (eqlable-listp l)))
  (cond ((endp l) t)
        ((member (car l) (cdr l)) nil)
        (t (no-duplicatesp (cdr l)))))

#+acl2-loop-only
(defun assoc (x alist)

  ":Doc-Section ACL2::Programming

  look up key in association list, using ~ilc[eql] as test~/

  ~c[(Assoc x alist)] is the first member of ~c[alist] whose ~ilc[car]
  is ~c[x], or ~c[nil] if no such member exists.~/

  ~c[(Assoc x alist)] is provably the same in the ACL2 logic as
  ~c[(assoc-equal x alist)].  It has a stronger ~il[guard] than
  ~ilc[assoc-equal] because it uses ~ilc[eql] to test whether ~c[x] is equal
  to the ~ilc[car] of a given member of ~c[alist].  Its ~il[guard]
  requires that ~c[alist] is an ~ilc[alistp], and moreover, either
  ~c[(eqlablep x)] or all ~ilc[car]s of members of ~c[alist] are
  ~ilc[eqlablep].  ~l[assoc-equal] and ~pl[assoc-eq].

  ~c[Assoc] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  Since ACL2 functions cannot
  take keyword arguments (though macros can), the ACL2 functions
  ~ilc[assoc-equal] and ~ilc[assoc-eq] are defined to correspond to calls
  of the Common Lisp function ~c[assoc] whose keyword argument
  ~c[:test] is ~ilc[equal] or ~ilc[eq], respectively.~/"

  (declare (xargs :guard (if (eqlablep x)
                             (alistp alist)
                           (eqlable-alistp alist))))
  (cond ((endp alist) nil)
        ((eql x (car (car alist))) (car alist))
        (t (assoc x (cdr alist)))))

(defun r-eqlable-alistp (x)

; For guard to rassoc.

  (declare (xargs :guard t))
  (cond ((atom x) (equal x nil))
        (t (and (consp (car x))
                (eqlablep (cdr (car x)))
                (r-eqlable-alistp (cdr x))))))

#+acl2-loop-only
(defun rassoc (x alist)

  ":Doc-Section ACL2::Programming

  look up value in association list, using ~ilc[eql] as test~/

  ~c[(Rassoc x alist)] is similar to ~c[(assoc x alist)], the difference
  being that it looks for the first pair in the given alist whose
  ~ilc[cdr], rather than ~ilc[car], is ~ilc[eql] to ~c[x].  ~l[assoc].~/

  The ~il[guard] of ~c[rassoc] requires its second argument to be an alist,
  and in addition, that either its first argument is ~ilc[eqlablep] or
  else all second components of pairs belonging to the second argument
  are ~ilc[eqlablep].  ~l[eqlablep].

  ~c[Rassoc] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (if (eqlablep x)
                             (alistp alist)
                           (r-eqlable-alistp alist))))
  (cond ((endp alist) nil)
        ((eql x (cdr (car alist))) (car alist))
        (t (rassoc x (cdr alist)))))

(defun rassoc-equal (x alist)

  ":Doc-Section ACL2::Programming

  look up value in association list, using ~ilc[equal] as test~/

  ~c[(Rassoc-equal x alist)] is similar to ~c[(assoc-equal x alist)], the
  difference being that it looks for the first pair in the given alist whose
  ~ilc[cdr], rather than ~ilc[car], is ~ilc[equal] to ~c[x].
  ~l[assoc-equal].~/

  The ~il[guard] of ~c[rassoc-equal] requires its second argument to be an
  alist.  ~l[rassoc] and ~pl[rassoc-eq].~/"

  (declare (xargs :guard (alistp alist)))
  (cond ((endp alist) nil)
        ((equal x (cdr (car alist))) (car alist))
        (t (rassoc-equal x (cdr alist)))))

(defun r-symbol-alistp (x)

; For guard to rassoc-eq.

  (declare (xargs :guard t))
  (cond ((atom x) (equal x nil))
        (t (and (consp (car x))
                (symbolp (cdr (car x)))
                (r-symbol-alistp (cdr x))))))

(defun rassoc-eq (x alist)

  ":Doc-Section ACL2::Programming

  look up value in association list, using ~ilc[eq] as test~/

  ~c[(Rassoc-eq x alist)] is similar to ~c[(assoc-eq x alist)], the difference
  being that it looks for the first pair in the given alist whose ~ilc[cdr],
  rather than ~ilc[car], is ~ilc[eq] to ~c[x].  ~l[assoc].~/

  The ~il[guard] of ~c[rassoc-eq] requires its second argument to be an alist,
  and in addition, that either its first argument is a ~ilc[symbolp] or
  else all second components of pairs belonging to the second argument
  are ~ilc[symbolp]s.

  ~c[Rassoc] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (if (symbolp x)
                             (alistp alist)
                           (r-symbol-alistp alist))))
  (cond ((endp alist) nil)
        ((eq x (cdr (car alist))) (car alist))
        (t (rassoc-eq x (cdr alist)))))

(defconst *standard-chars*
  '(#\Newline #\Space
    #\! #\" #\# #\$ #\% #\& #\' #\( #\) #\* #\+ #\, #\- #\. #\/ #\0 #\1
    #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\: #\; #\< #\= #\> #\? #\@ #\A #\B
    #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S
    #\T #\U #\V #\W #\X #\Y #\Z #\[ #\\ #\] #\^ #\_ #\` #\a #\b #\c #\d
    #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u
    #\v #\w #\x #\y #\z #\{ #\| #\} #\~))

#+acl2-loop-only
(defun standard-char-p (x)

  ":Doc-Section ACL2::Programming

  recognizer for standard characters~/

  ~c[(Standard-char-p x)] is true if and only if ~c[x] is a ``standard''
  character, i.e., a member of the list ~c[*standard-chars*].  This
  list includes ~c[#\Newline] and ~c[#\Space] ~il[characters], as well as the usual
  punctuation and alphanumeric ~il[characters].~/

  ~c[Standard-char-p] has a ~il[guard] requiring its argument to be a
  character.

  ~c[Standard-char-p] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

; The following guard is required by p. 234 of CLtL.

  (declare (xargs :guard (characterp x)))
  (if (member x *standard-chars*)
      t
    nil))

(defun standard-char-listp (l)

  ":Doc-Section ACL2::Programming

  recognizer for a true list of standard characters~/

  ~c[(standard-char-listp x)] is true if and only if ~c[x] is a
  null-terminated list all of whose members are standard ~il[characters].
  ~l[standard-char-p].~/

  ~c[Standard-char-listp] has a ~il[guard] of ~c[t].~/"

  (declare (xargs :guard t))
  (cond ((consp l)
         (and (characterp (car l))
              (standard-char-p (car l))
              (standard-char-listp (cdr l))))
        (t (equal l nil))))


(defun character-listp (l)

  ":Doc-Section ACL2::Programming

  recognizer for a true list of characters~/

  The predicate ~c[character-listp] tests whether its argument is a
  true list of ~il[characters].~/~/"

  (declare (xargs :guard t))
  (cond ((atom l) (equal l nil))
        (t (and (characterp (car l))
                (character-listp (cdr l))))))

(defthm character-listp-forward-to-eqlable-listp
  (implies (character-listp x)
           (eqlable-listp x))
  :rule-classes :forward-chaining)

(defthm standard-char-listp-forward-to-character-listp
  (implies (standard-char-listp x)
           (character-listp x))
  :rule-classes :forward-chaining)

(defaxiom coerce-inverse-1
  (implies (character-listp x)
           (equal (coerce (coerce x 'string) 'list) x)))

#| A "historical document" regarding standard characters:
To: Kaufmann
Subject: over strong axiom
FCC: ~moore/old-mail
--text follows this line--
Axioms.lisp currently contains

(defaxiom coerce-inverse-2
  (implies (stringp x)
           (equal (coerce (coerce x 'list) 'string) x)))

But the guard for coerce (when the second argument is 'string) requires the first
argument to be a standard-char-listp.  Thus, unless we know that (coerce x 'list)
returns a standard-char-listp when (stringp x), the guard on the outer coerce is
violated.

If we are really serious that ACL2 strings may contain nonstandard chars, then
this axiom is too strong.  I will leave this note in axioms.lisp and just go
on.  But when the guard question is settled I would like to return to this and
make explicit our occasional implicit assumption that strings are composed of
standard chars.

J
|#

(defaxiom coerce-inverse-2
  (implies (stringp x)
           (equal (coerce (coerce x 'list) 'string) x)))

; Once upon a time, Moore (working alone) added the following axiom.

#|
(defaxiom standard-char-listp-coerce
  (implies (stringp str)
           (standard-char-listp (coerce str 'list))))
|#

(defaxiom character-listp-coerce
  (character-listp (coerce str 'list))
  :rule-classes
  (:rewrite
   (:forward-chaining :trigger-terms
                      ((coerce str 'list)))))

; In AKCL the nonstandard character #\Page prints as ^L and may be included in
; strings, as in "^L".  Now if you try to type that string in ACL2, you get an
; error.  And ACL2 does not let you use coerce to produce the string, e.g.,
; with (coerce (list #\Page) 'string), because the guard for coerce is
; violated.  So here we have a situation in which no ACL2 function in LP will
; ever see a nonstandard char in a string, but CLTL permits it.  However, we
; consider the axiom to be appropriate, because ACL2 strings contain only
; standard characters.

(in-theory (disable standard-char-listp standard-char-p))

#|
(defthm standard-char-listp-coerce-forward-chaining

; If (stringp str) is in the context, we want to make a "note" that
; (coerce str 'list) is a standard-char-listp in case this fact is
; needed during later backchaining.  We see no need to forward chain
; from (standard-char-listp (coerce str 'list)), however; the rewrite
; rule generated here should suffice for relieving any such hypothesis.

  (implies (stringp str)
           (standard-char-listp (coerce str 'list)))
  :rule-classes ((:forward-chaining :trigger-terms
                                    ((coerce str 'list)))))
|#

#+acl2-loop-only
(defun string (x)

  ":Doc-Section ACL2::Programming

  ~il[coerce] to a string~/

  ~c[(String x)] ~il[coerce]s ~c[x] to a string.  If ~c[x] is already a
  string, then it is returned unchanged; if ~c[x] is a symbol, then its
  ~ilc[symbol-name] is returned; and if ~c[x] is a character, the
  corresponding one-character string is returned.~/

  The ~il[guard] for ~c[string] requires its argument to be a string, a
  symbol, or a character.

  ~c[String] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard

; NOTE:  When we finally get hold of a definitive Common Lisp
; reference, let's clarify the statement near the bottom of p. 466 of
; CLtL2, which says:  "Presumably converting a character to a string
; always works according to this vote."  But we'll plunge ahead as
; follows, in part because we want to remain compliant with CLtL1,
; which isn't as complete as one might wish regarding which characters
; can go into strings.

                  (or (stringp x)
                      (symbolp x)
                      (characterp x))))
  (cond
   ((stringp x) x)
   ((symbolp x) (symbol-name x))
   (t (coerce (list x) 'string))))

#+acl2-loop-only
(defun alpha-char-p (x)

  ":Doc-Section ACL2::Programming

  recognizer for alphabetic characters~/

  ~c[(Alpha-char-p x)] is true if and only if ~c[x] is a alphabetic
  character, i.e., one of the ~il[characters] ~c[#\a], ~c[#\b], ..., ~c[#\z], ~c[#\A], ~c[#\B],
  ..., ~c[#\Z].~/

  The ~il[guard] for ~c[alpha-char-p] requires its argument to be a character.

  ~c[Alpha-char-p] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

; The following guard is required by p. 235 of CLtL.

  (declare (xargs :guard (characterp x)))
  (and (member x
               '(#\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m
                 #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z
                 #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M
                 #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
       t))

#+acl2-loop-only
(defun upper-case-p (x)

  ":Doc-Section ACL2::Programming

  recognizer for upper case characters~/

  ~c[(Upper-case-p x)] is true if and only if ~c[x] is an upper case
  character, i.e., a member of the list ~c[#\A], ~c[#\B], ..., ~c[#\Z].~/

  The ~il[guard] for ~c[upper-case-p] requires its argument to be a standard
  character (~pl[standard-char-p]).

  ~c[Upper-case-p] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

; The guard characterp is required by p. 235 of CLtL.  However, In Allegro 6.0
; we see characters other than standard characters that are treated as upper
; case, such as (code-char (+ 128 65)).  So we strengthen that guard.

  (declare (xargs :guard (and (characterp x)
                              (standard-char-p x))))
  (and (member x
               '(#\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M
                 #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
       t))

#+acl2-loop-only
(defun lower-case-p (x)

  ":Doc-Section ACL2::Programming

  recognizer for lower case characters~/

  ~c[(Lower-case-p x)] is true if and only if ~c[x] is a lower case
  character, i.e., a member of the list ~c[#\A], ~c[#\B], ..., ~c[#\Z].~/

  The ~il[guard] for ~c[lower-case-p] requires its argument to be a standard
  character (~pl[standard-char-p]).

  ~c[Lower-case-p] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

; The guard characterp is required by p. 235 of CLtL.  However, In Allegro 6.0
; we see characters other than standard characters that are treated as upper
; case, such as (code-char (+ 128 65)).  So we strengthen that guard.

  (declare (xargs :guard (and (characterp x)
                              (standard-char-p x))))
  (and (member x
               '(#\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m
                 #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))
       t))

#+acl2-loop-only
(defun char-upcase (x)

  ":Doc-Section ACL2::Programming

  turn lower-case ~il[characters] into upper-case ~il[characters]~/

  ~c[(Char-upcase x)] is equal to ~c[#\A] when ~c[x] is ~c[#\a], ~c[#\B] when ~c[x] is
  ~c[#\b], ..., and ~c[#\Z] when ~c[x] is ~c[#\z], and is ~c[x] for any other character.~/

  The ~il[guard] for ~c[char-upcase] requires its argument to be a standard
  character (~pl[standard-char-p]).

  ~c[Char-upcase] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

; The guard characterp is required by p. 231 of CLtL.  However, In Allegro 6.0
; we see characters other than standard characters that are treated as upper
; case, such as (code-char (+ 128 65)).  So we strengthen that guard.

  (declare (xargs :guard (and (characterp x)
                              (standard-char-p x))))
  (let ((pair (assoc x
                     '((#\a . #\A)
                       (#\b . #\B)
                       (#\c . #\C)
                       (#\d . #\D)
                       (#\e . #\E)
                       (#\f . #\F)
                       (#\g . #\G)
                       (#\h . #\H)
                       (#\i . #\I)
                       (#\j . #\J)
                       (#\k . #\K)
                       (#\l . #\L)
                       (#\m . #\M)
                       (#\n . #\N)
                       (#\o . #\O)
                       (#\p . #\P)
                       (#\q . #\Q)
                       (#\r . #\R)
                       (#\s . #\S)
                       (#\t . #\T)
                       (#\u . #\U)
                       (#\v . #\V)
                       (#\w . #\W)
                       (#\x . #\X)
                       (#\y . #\Y)
                       (#\z . #\Z)))))
    (cond (pair (cdr pair))
          ((characterp x) x)
          (t (code-char 0)))))

#+acl2-loop-only
(defun char-downcase (x)

  ":Doc-Section ACL2::Programming

  turn upper-case ~il[characters] into lower-case ~il[characters]~/

  ~c[(Char-downcase x)] is equal to ~c[#\a] when ~c[x] is ~c[#\A], ~c[#\b] when ~c[x] is
  ~c[#\B], ..., and ~c[#\z] when ~c[x] is ~c[#\Z], and is ~c[x] for any other character.~/

  The ~il[guard] for ~c[char-downcase] requires its argument to be a standard
  character (~pl[standard-char-p]).

  ~c[Char-downcase] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

; The guard characterp is required by p. 231 of CLtL.  However, In Allegro 6.0
; we see characters other than standard characters that are treated as upper
; case, such as (code-char (+ 128 65)).  So we strengthen that guard.

  (declare (xargs :guard (and (characterp x)
                              (standard-char-p x))))
    (let ((pair (assoc x
                       '((#\A . #\a)
                         (#\B . #\b)
                         (#\C . #\c)
                         (#\D . #\d)
                         (#\E . #\e)
                         (#\F . #\f)
                         (#\G . #\g)
                         (#\H . #\h)
                         (#\I . #\i)
                         (#\J . #\j)
                         (#\K . #\k)
                         (#\L . #\l)
                         (#\M . #\m)
                         (#\N . #\n)
                         (#\O . #\o)
                         (#\P . #\p)
                         (#\Q . #\q)
                         (#\R . #\r)
                         (#\S . #\s)
                         (#\T . #\t)
                         (#\U . #\u)
                         (#\V . #\v)
                         (#\W . #\w)
                         (#\X . #\x)
                         (#\Y . #\y)
                         (#\Z . #\z)))))
      (cond (pair (cdr pair))
            ((characterp x) x)
            (t (code-char 0)))))

(defthm lower-case-p-char-downcase
  (implies (and (upper-case-p x)
                (characterp x))
           (lower-case-p (char-downcase x))))

(defthm upper-case-p-char-upcase
  (implies (and (lower-case-p x)
                (characterp x))
           (upper-case-p (char-upcase x))))

(defthm lower-case-p-forward-to-alpha-char-p
  (implies (and (lower-case-p x)
                (characterp x))
           (alpha-char-p x))
  :rule-classes :forward-chaining)

(defthm upper-case-p-forward-to-alpha-char-p
  (implies (and (upper-case-p x)
                (characterp x))
           (alpha-char-p x))
  :rule-classes :forward-chaining)

(defthm alpha-char-p-forward-to-characterp
  (implies (alpha-char-p x)
           (characterp x))
  :rule-classes :forward-chaining)

(defthm characterp-char-downcase
  (characterp (char-downcase x))
  :rule-classes :type-prescription)

(defthm characterp-char-upcase
  (characterp (char-upcase x))
  :rule-classes :type-prescription)

; We disable the following functions in order to protect people from getting
; burned by their explosive definitions.
(in-theory (disable alpha-char-p upper-case-p lower-case-p
                    char-upcase char-downcase))

(defun string-downcase1 (l)
  (declare (xargs :guard (standard-char-listp l)
                  :guard-hints
                  (("Goal" :in-theory (enable standard-char-listp)))))
  (if (atom l)
      nil
    (cons (char-downcase (car l))
          (string-downcase1 (cdr l)))))

(defthm character-listp-string-downcase-1
  (character-listp (string-downcase1 x)))

#+acl2-loop-only
(defun string-downcase (x)

  ":Doc-Section ACL2::Programming

  in a given string, turn upper-case ~il[characters] into lower-case~/

  For a string ~c[x], ~c[(string-downcase x)] is the result of applying
  ~ilc[char-downcase] to each character in ~c[x].~/

  The ~il[guard] for ~c[string-downcase] requires its argument to be a string
  containing only standard characters.

  ~c[String-downcase] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (stringp x)
                              (standard-char-listp (coerce x 'list)))))

; As with other functions, e.g., reverse, the guards on this function
; can't currently be proved because the outer coerce below requires
; its argument to be made of standard characters.  We don't know that
; the string x is made of standard characters.

    (coerce (string-downcase1 (coerce x 'list)) 'string))

(defun string-upcase1 (l)
  (declare (xargs :guard (standard-char-listp l)
                  :guard-hints
                  (("Goal" :in-theory (enable standard-char-listp)))))
  (if (atom l)
      nil
    (cons (char-upcase (car l))
          (string-upcase1 (cdr l)))))

(defthm character-listp-string-upcase1-1
  (character-listp (string-upcase1 x)))

#+acl2-loop-only
(defun string-upcase (x)

  ":Doc-Section ACL2::Programming

  in a given string, turn lower-case ~il[characters] into upper-case~/

  For a string ~c[x], ~c[(string-upcase x)] is the result of applying
  ~ilc[char-upcase] to each character in ~c[x].~/

  The ~il[guard] for ~c[string-upcase] requires its argument to be a string
  containing only standard characters.

  ~c[String-upcase] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

    (declare (xargs :guard (and (stringp x)
                                (standard-char-listp (coerce x 'list)))))
    (coerce (string-upcase1 (coerce x 'list)) 'string))

(defun our-digit-char-p (ch radix)
  (declare (xargs :guard (and (characterp ch)
                              (integerp radix)
                              (<= 2 radix)
                              (<= radix 36))))
  (let ((l (assoc ch
                  '((#\0 . 0)
                    (#\1 . 1)
                    (#\2 . 2)
                    (#\3 . 3)
                    (#\4 . 4)
                    (#\5 . 5)
                    (#\6 . 6)
                    (#\7 . 7)
                    (#\8 . 8)
                    (#\9 . 9)
                    (#\a . 10)
                    (#\b . 11)
                    (#\c . 12)
                    (#\d . 13)
                    (#\e . 14)
                    (#\f . 15)
                    (#\g . 16)
                    (#\h . 17)
                    (#\i . 18)
                    (#\j . 19)
                    (#\k . 20)
                    (#\l . 21)
                    (#\m . 22)
                    (#\n . 23)
                    (#\o . 24)
                    (#\p . 25)
                    (#\q . 26)
                    (#\r . 27)
                    (#\s . 28)
                    (#\t . 29)
                    (#\u . 30)
                    (#\v . 31)
                    (#\w . 32)
                    (#\x . 33)
                    (#\y . 34)
                    (#\z . 35)
                    (#\A . 10)
                    (#\B . 11)
                    (#\C . 12)
                    (#\D . 13)
                    (#\E . 14)
                    (#\F . 15)
                    (#\G . 16)
                    (#\H . 17)
                    (#\I . 18)
                    (#\J . 19)
                    (#\K . 20)
                    (#\L . 21)
                    (#\M . 22)
                    (#\N . 23)
                    (#\O . 24)
                    (#\P . 25)
                    (#\Q . 26)
                    (#\R . 27)
                    (#\S . 28)
                    (#\T . 29)
                    (#\U . 30)
                    (#\V . 31)
                    (#\W . 32)
                    (#\X . 33)
                    (#\Y . 34)
                    (#\Z . 35)))))
    (cond ((and l (< (cdr l) radix))
           (cdr l))
          (t nil))))

#+acl2-loop-only
(defmacro digit-char-p (ch &optional (radix '10))

  ":Doc-Section ACL2::Programming

  the number, if any, corresponding to a given character~/

  ~c[(digit-char-p ch)] is the integer corresponding to the character
  ~c[ch] in base ~c[10].  For example, ~c[(digit-char-p #\\3)] is equal to
  the integer ~c[3].  More generally, an optional second argument
  specifies the radix (default ~c[10], as indicated above).~/

  The ~il[guard] for ~c[digit-char-p] (more precisely, for the function
  ~c[our-digit-char-p] that calls of this macro expand to) requires its
  second argument to be an integer between 2 and 36, inclusive, and
  its first argument to be a character.

  ~c[Digit-char-p] is a Common Lisp function, though it is implemented
  in the ACL2 logic as an ACL2 macro.  See any Common Lisp
  documentation for more information.~/"

  `(our-digit-char-p ,ch ,radix))

#+acl2-loop-only
(defun char-equal (x y)

  ":Doc-Section ACL2::Programming

  character equality without regard to case~/

  For ~il[characters] ~c[x] and ~c[y], ~c[(char-equal x y)] is true if and only if ~c[x]
  and ~c[y] are the same except perhaps for their case.~/

  The ~il[guard] on ~c[char-equal] requires that its arguments are both
  standard ~il[characters] (~pl[standard-char-p]).

  ~c[Char-equal] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (characterp x)
                              (standard-char-p x)
                              (characterp y)
                              (standard-char-p y))))
  (eql (char-downcase x)
       (char-downcase y)))

(defun atom-listp (lst)

  ":Doc-Section ACL2::Programming

  recognizer for a true list of ~il[atom]s~/

  The predicate ~c[atom-listp] tests whether its argument is a
  ~ilc[true-listp] of ~il[atom]s, i.e., of non-conses.~/~/"

  (declare (xargs :guard t))
  (cond ((atom lst) (eq lst nil))
        (t (and (atom (car lst))
                (atom-listp (cdr lst))))))

(defthm atom-listp-forward-to-true-listp
  (implies (atom-listp x)
           (true-listp x))
  :rule-classes :forward-chaining)

(defthm eqlable-listp-forward-to-atom-listp
  (implies (eqlable-listp x)
           (atom-listp x))
  :rule-classes :forward-chaining)

(defthm characterp-nth
  (implies (and (character-listp x)
                (integerp i)
                (<= 0 i)
                (< i (len x)))
           (characterp (nth i x))))

(defun ifix (x)

  ":Doc-Section ACL2::Programming

  coerce to an integer~/

  ~c[Ifix] simply returns any integer argument unchanged, returning ~c[0]
  on a non-integer argument.  Also ~pl[nfix], ~pl[rfix],
  ~pl[realfix] and ~pl[fix] for analogous functions that coerce to
  a natural number, a rational number, a real, and a number,
  respectively.~/

  ~c[Ifix] has a ~il[guard] of ~c[t].~/"

  (declare (xargs :guard t))
  (if (integerp x) x 0))

(defun rfix (x)

  ":Doc-Section ACL2::Programming

  coerce to a rational number~/

  ~c[Rfix] simply returns any rational number argument unchanged,
  returning ~c[0] on a non-rational argument.  Also ~pl[nfix],
  ~pl[ifix], ~pl[realfix], and ~pl[fix] for analogous
  functions that coerce to a natural number, an integer, a real, and a
  number, respectively.~/

  ~c[Rfix] has a ~il[guard] of ~c[t].~/"

  (declare (xargs :guard t))
  (if (rationalp x) x 0))

;; RAG - I added "realfix" to coerce numbers into reals.  I would have
;; liked to use "rfix" for it, but "rfix" was taken for the
;; rationals.  "ifix" as in "irrational-fix" would be a misnomer,
;; since it's the identity functions for rationals as well as
;; irrationals.  In desperation, we called it realfix, even though
;; that makes it more awkward to use than the other "fix" functions.

; Since the next function, realfix, is referred to by other :doc topics, do not
; make it conditional upon #+:non-standard-analysis.

(defun realfix (x)

  ":Doc-Section ACL2::Programming

  coerce to a real number~/

  ~c[Realfix] simply returns any real number argument unchanged,
  returning ~c[0] on a non-real argument.  Also ~pl[nfix],
  ~pl[ifix], ~pl[rfix], and ~pl[fix] for analogous functions
  that coerce to a natural number, an integer, a rational, and a
  number, respectively.~/

  ~c[Realfix] has a ~il[guard] of ~c[t].~/"

  (declare (xargs :guard t))
  (if (real/rationalp x) x 0))

(defun nfix (x)

  ":Doc-Section ACL2::Programming

  coerce to a natural number~/

  ~c[Nfix] simply returns any natural number argument unchanged,
  returning ~c[0] on an argument that is not a natural number.  Also
  ~pl[ifix], ~pl[rfix], ~pl[realfix], and ~pl[fix] for
  analogous functions that coerce to an integer, a rational number, a
  real, and a number, respectively.~/

  ~c[Nfix] has a ~il[guard] of ~c[t].~/"

  (declare (xargs :guard t))
  (if (and (integerp x) (>= x 0))
      x
    0))

(defun string-equal1 (str1 str2 i maximum)
  (declare (xargs :guard (and (stringp str1)
                              (standard-char-listp (coerce str1 'list))
                              (stringp str2)
                              (standard-char-listp (coerce str2 'list))
                              (integerp i)
                              (integerp maximum)
                              (<= maximum (length str1))
                              (<= maximum (length str2))
                              (<= 0 i)
                              (<= i maximum))

; We make this function :program until we know enough about o-p
; to prove its termination.

                  :mode :program))
  (let ((i (nfix i)))
    (cond
     ((>= i (ifix maximum))
      t)
     (t (and (char-equal (char str1 i)
                         (char str2 i))
             (string-equal1 str1 str2 (+ 1 i) maximum))))))

#+acl2-loop-only
(defun string-equal (str1 str2)

  ":Doc-Section ACL2::Programming

  string equality without regard to case~/

  For strings ~c[str1] and ~c[str2], ~c[(string-equal str1 str2)] is true if
  and only ~c[str1] and ~c[str2] are the same except perhaps for the cases of
  their ~il[characters].~/

  The ~il[guard] on ~c[string-equal] requires that its arguments are strings
  consisting of standard characters (~pl[standard-char-listp]).

  ~c[String-equal] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (stringp str1)
                              (standard-char-listp (coerce str1 'list))
                              (stringp str2)
                              (standard-char-listp (coerce str2 'list)))
                  :mode :program))
  (let ((len1 (length str1)))
    (and (= len1 (length str2))
         (string-equal1 str1 str2 0 len1))))

(defun standard-string-alistp (x)

  ":Doc-Section ACL2::Programming

  recognizer for association lists with standard strings as keys~/

  ~c[(Standard-string-alistp x)] is true if and only if ~c[x] is a list of
  pairs of the form ~c[(cons key val)] where ~c[key] is a string all of whose
  characters are standard (~pl[standard-char-p]).~/

  ~c[Standard-string-alistp] has a ~il[guard] of ~c[t].~/"

  (declare (xargs :guard t))
  (cond ((atom x) (eq x nil))
        (t (and (consp (car x))
                (stringp (car (car x)))
                (standard-char-listp (coerce (car (car x)) 'list))
                (standard-string-alistp (cdr x))))))

(defthm standard-string-alistp-forward-to-alistp
  (implies (standard-string-alistp x)
           (alistp x))
  :rule-classes :forward-chaining)

(defun assoc-string-equal (str alist)

  ":Doc-Section ACL2::Programming

  look up key, a string, in association list~/

  ~c[(Assoc-string-equal x alist)] is similar to ~ilc[assoc-equal].
  However, for string ~c[x] and alist ~c[alist], the comparison of ~c[x]
  with successive keys in ~c[alist] is done using ~ilc[string-equal]
  rather than ~ilc[equal].~/

  The ~il[guard] for ~c[assoc-string-equal] requires that ~c[x] is a string
  and ~c[alist] is an alist.~/"

  (declare (xargs :guard (and (stringp str)
                              (standard-char-listp (coerce str 'list))
                              (standard-string-alistp alist))
                  :mode :program))
  (cond
   ((endp alist)
    nil)
   ((string-equal str (car (car alist)))
    (car alist))
   (t (assoc-string-equal str (cdr alist)))))

#+acl2-loop-only
(defmacro caar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[car]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'car x)))

#+acl2-loop-only
(defmacro cadr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[cdr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cdr x)))

#+acl2-loop-only
(defmacro cdar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[car]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'car x)))

#+acl2-loop-only
(defmacro cddr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[cdr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'cdr x)))

#+acl2-loop-only
(defmacro caaar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[caar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'caar x)))

#+acl2-loop-only
(defmacro caadr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[cadr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cadr x)))

#+acl2-loop-only
(defmacro cadar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[cdar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cdar x)))

#+acl2-loop-only
(defmacro caddr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[cddr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cddr x)))

#+acl2-loop-only
(defmacro cdaar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[caar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'caar x)))

#+acl2-loop-only
(defmacro cdadr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[cadr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'cadr x)))

#+acl2-loop-only
(defmacro cddar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[cdar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'cdar x)))

#+acl2-loop-only
(defmacro cdddr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[cddr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'cddr x)))

#+acl2-loop-only
(defmacro caaaar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[caaar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'caaar x)))

#+acl2-loop-only
(defmacro caaadr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[caadr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'caadr x)))

#+acl2-loop-only
(defmacro caadar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[cadar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cadar x)))

#+acl2-loop-only
(defmacro caaddr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[caddr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'caddr x)))

#+acl2-loop-only
(defmacro cadaar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[cdaar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cdaar x)))

#+acl2-loop-only
(defmacro cadadr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[cdadr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cdadr x)))

#+acl2-loop-only
(defmacro caddar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[cddar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cddar x)))

#+acl2-loop-only
(defmacro cadddr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[car] of the ~ilc[cdddr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cdddr x)))

#+acl2-loop-only
(defmacro cdaaar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[caaar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'caaar x)))

#+acl2-loop-only
(defmacro cdaadr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[caadr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'caadr x)))

#+acl2-loop-only
(defmacro cdadar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[cadar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'cadar x)))

#+acl2-loop-only
(defmacro cdaddr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[caddr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'caddr x)))

#+acl2-loop-only
(defmacro cddaar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[cdaar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'cdaar x)))

#+acl2-loop-only
(defmacro cddadr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[cdadr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'cdadr x)))

#+acl2-loop-only
(defmacro cdddar (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[cddar]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'cddar x)))

#+acl2-loop-only
(defmacro cddddr (x)
  ":Doc-Section ACL2::Programming

  ~ilc[cdr] of the ~ilc[cdddr]~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cdr (list 'cdddr x)))

; Ordinal stuff.  It seems more or less impossible to get o<g and o< admitted
; during boot-strapping unless we cheat by declaring them explicitly :mode
; :logic so that they will be admitted in the first pass of the build.  But
; then we also need to declare functions on which they depend to be :mode
; :logic as well (since :logic mode functions cannot have :program mode
; functions in their bodies).

;first: we mention the old ordinals:

(defdoc e0-ordinalp
  ":Doc-Section ACL2::Programming

   the old recognizer for ACL2 ordinals~/

   ~l[o-p] for the current recognizer for ACL2 ordinals.~/

   The functions ~c[e0-ordinalp] and ~ilc[e0-ord-<] were replaced in ACL2
   Version_2.8 by ~ilc[o-p] and ~ilc[o<], respectively.  However, books created
   before that version used the earlier functions for termination proofs; the
   old functions might be of use in these cases.  To use the old functions in
   termination proofs, include the book ~c[books/ordinals/e0-ordinal] and
   execute the event ~c[(set-well-founded-relation e0-ord-<)]
   (~pl[set-well-founded-relation]).  For a more thorough discussion of
   these functions, see the documentation at the end of
   ~c[books/ordinals/e0-ordinal.lisp].")

(defdoc e0-ord-<
  ":Doc-Section ACL2::Programming

   the old ordering function for ACL2 ordinals~/

   ~l[o<] for the current new ordering function for ACL2 ordinals.~/

   The functions ~c[e0-ordinalp] and ~ilc[e0-ord-<] were replaced in ACL2
   Version_2.8 by ~ilc[o-p] and ~ilc[o<], respectively.  However, books created
   before that version used the earlier functions for termination proofs; the
   old functions might be of use in these cases.  To use the old functions in
   termination proofs, include the book ~c[books/ordinals/e0-ordinal] and
   execute the event ~c[(set-well-founded-relation e0-ord-<)]
   (~pl[set-well-founded-relation]).  For a more thorough discussion of
   these functions, see the documentation at the end of
   ~c[books/ordinals/e0-ordinal.lisp].")

(defun natp (x)

  ":Doc-Section ACL2::Programming

   a recognizer for the natural numbers~/~/

  The natural numbers is the set of all non-negative integers,
  ~c[{0,1,2,3,...}].  ~c[Natp] returns ~c[t] if and only its argument is a
  natural number, and ~c[nil] otherwise.  We recommend the file
  ~c[books/arithmetic/natp-posp.lisp] as a book for reasoning about ~c[posp]
  and ~c[natp].  This book is included in ~c[books/arithmetic/top] and
  ~c[books/arithmetic/top-with-meta]."

  (declare (xargs :guard t :mode :logic))
  (and (integerp x)
       (<= 0 x)))

(defthm natp-compound-recognizer
  (equal (natp x)
         (and (integerp x)
              (<= 0 x)))
  :rule-classes :compound-recognizer)

(defun posp (x)
  ":Doc-Section ACL2::Programming

   a recognizer for the positive integers~/~/

  ~c[(posp x)] is logically equivalent to ~c[(not (zp x))] (~pl[zp]) and also
  to ~c[(and (natp x) (not (equal x 0)))].  We recommend the file
  ~c[books/ordinals/natp-posp] as a book for reasoning about ~c[posp] and
  ~c[natp].  This book is included in ~c[books/arithmetic/top] and
  ~c[books/arithmetic/top-with-meta]."

  (declare (xargs :guard t :mode :logic))
  (and (integerp x)
       (< 0 x)))

(defthm posp-compound-recognizer
  (equal (posp x)
         (and (integerp x)
              (< 0 x)))
  :rule-classes :compound-recognizer)

(defun o-finp (x)
  ":Doc-Section ACL2::Programming

  recognizes if an ordinal is finite~/~/

  We introduce the function ~c[o-finp] which returns ~c[t] for any ordinal that
  is finite, else ~c[nil].  This function is equivalent to the function
  ~ilc[atom], and is introduced so that we can ~ilc[disable] its definition
  when dealing with ordinals (also ~pl[make-ord])."

  (declare (xargs :guard t :mode :logic))
  (atom x))

(defmacro o-infp (x)
  ":Doc-Section ACL2::Programming

  recognizes if an ordinal is infinite~/~/

  ~c[O-infp] is a macro.  ~c[(O-infp x)] opens up to ~c[(not (o-finp x))]
  (~pl[o-finp])."

  `(not (o-finp ,x)))

(defun o-first-expt (x)
  ":Doc-Section ACL2::Programming

  the first exponent of an ordinal~/~/

  An ACL2 ordinal is either a natural number or, for an infinite ordinal, a
  list whose elements are exponent-coefficient pairs (~pl[o-p]).  In the latter
  case, this function returns the ~ilc[car] of the first pair in the list.  In
  the case of a natural number, the value returned is 0 (since a natural
  number, ~c[n], can be thought of as (w^0)n).

  For the corresponding coefficient, ~pl[o-first-coeff]."

  (declare (xargs :guard (or (o-finp x) (consp (car x))) :mode :logic))
  (if (o-finp x)
      0
    (caar x)))

(defun o-first-coeff (x)
  ":Doc-Section ACL2::Programming

  returns the first coefficient of an ordinal~/~/

  An ACL2 ordinal is either a natural number or, for an infinite ordinal, a
  list whose elements are exponent-coefficient pairs (~pl[o-p]).  In the latter
  case, this function returns the ~ilc[cdr] of the first pair in the list.  In
  the case of a natural number, this function returns the ordinal itself
  (since a natural number, n, can be thought of as (w^0)n).

  For the corresponding exponent, ~pl[o-first-expt]."

  (declare (xargs :guard (or (o-finp x) (consp (car x))) :mode :logic))
  (if (o-finp x)
      x
    (cdar x)))

(defun o-rst (x)
  ":Doc-Section ACL2::Programming

  returns the rest of an infinite ordinal~/~/

  An ACL2 infinite ordinal is a list whose elements are exponent-coefficient
  pairs (~pl[o-p] and ~pl[o-infp]).  The first exponent and first coefficient
  of an ordinal can be obtained by using ~ilc[o-first-expt] and
  ~ilc[o-first-coeff] respectively.  To obtain the rest of the ordinal (for
  recursive analysis), use the ~c[o-rst] function. It returns the rest of the
  ordinal after the first exponent and coefficient are removed."

  (declare (xargs :guard (consp x) :mode :logic))
  (cdr x))

(defun o<g (x)

; This function is used only for guard proofs.

  (declare (xargs :guard t :mode :logic))
  (if (atom x)
      (rationalp x)
    (and (consp (car x))
         (rationalp (o-first-coeff x))
         (o<g (o-first-expt x))
         (o<g (o-rst x)))))

(defun o< (x y)

  ":Doc-Section ACL2::Programming

   the well-founded less-than relation on ordinals up to ~c[epsilon-0]~/

   If ~c[x] and ~c[y] are both ~c[o-p]s (~pl[o-p]) then
   ~c[(o< x y)] is true iff ~c[x] is strictly less than ~c[y].  ~c[o<] is
   well-founded on the ~ilc[o-p]s.  When ~c[x] and ~c[y] are both nonnegative
   integers, ~c[o<] is just the familiar ``less than'' relation (~ilc[<]).~/

   ~c[o<] plays a key role in the formal underpinnings of the ACL2
   logic.  In order for a recursive definition to be admissible it must
   be proved to ``terminate.''  By terminate we mean that the arguments to
   the function ``get smaller'' as the function recurses and this sense
   of size comparison must be such that there is no ``infinitely
   descending'' sequence of ever smaller arguments.  That is, the
   relation used to compare successive arguments must be well-founded
   on the domain being measured.

   The most basic way ACL2 provides to prove termination requires the
   user to supply (perhaps implicitly) a mapping of the argument tuples
   into the ordinals with some ``measure'' expression in such a way
   that the measures of the successive argument tuples produced by
   recursion decrease according to the relation ~c[o<].  The validity
   of this method rests on the well-foundedness of ~c[o<] on the
   ~ilc[o-p]s.

   Without loss of generality, suppose the definition in question
   introduces the function ~c[f], with one formal parameter ~c[x] (which might
   be a list of objects).  Then we require that there exist a measure
   expression, ~c[(m x)], that always produces an ~ilc[o-p].
   Furthermore, consider any recursive call, ~c[(f (d x))], in the body of
   the definition.  Let ~c[hyps] be the conjunction of terms, each of which is
   either the test of an ~ilc[if] in the body or else the negation of such a
   test, describing the path through the body to the recursive call in
   question.  Then it must be a theorem that
   ~bv[]
     (IMPLIES hyps (O< (m (d x)) (m x))).
   ~ev[]
   When we say ~c[o<] is ``well-founded'' on the ~ilc[o-p]s we
   mean that there is no infinite sequence of ~ilc[o-p]s such that
   each is smaller than its predecessor in the sequence.  Thus, the
   theorems that must be proved about ~c[f] when it is introduced establish
   that it cannot recur forever because each time a recursive call is
   taken ~c[(m x)] gets smaller.  From this, and the syntactic restrictions
   on definitions, it can be shown (as on page 44 in ``A Computational
   Logic'', Boyer and Moore, Academic Press, 1979) that there exists a
   function satisfying the definition; intuitively, the value assigned
   to any given ~c[x] by the alleged function is that computed by a
   sufficiently large machine.  Hence, the logic is consistent if the
   axiom defining ~c[f] is added.

   ~l[o-p] for a discussion of the ordinals and how to
   compare two ordinals.

   The definitional principle permits the use of relations other than
   ~c[o<] but they must first be proved to be well-founded on some
   domain.  ~l[well-founded-relation].  Roughly put, alternative
   relations are shown well-founded by providing an order-preserving
   mapping from their domain into the ordinals.  ~l[defun] for
   details on how to specify which well-founded relation is to be
   used."
  (declare (xargs :guard (and (o<g x) (o<g y)) :mode :logic))
  (cond ((o-finp x)
         (or (o-infp y) (< x y)))
        ((o-finp y) nil)
        ((not (equal (o-first-expt x) (o-first-expt y)))
         (o< (o-first-expt x) (o-first-expt y)))
        ((not (= (o-first-coeff x) (o-first-coeff y)))
         (< (o-first-coeff x) (o-first-coeff y)))
        (t (o< (o-rst x) (o-rst y)))))

(defmacro o> (x y)
  ":Doc-Section ACL2::Programming

  the greater-than relation for the ordinals~/~/

  ~c[O>] is a macro and ~c[(o> x y)] expands to ~c[(o< y x)].  ~l[o<]."

  `(o< ,y ,x))

(defmacro o<= (x y)
  ":Doc-Section ACL2::Programming

  the less-than-or-equal relation for the ordinals~/~/

  ~c[o<=] is a macro and ~c[(o<= x y)] expands to ~c[(not (o< y x))].  ~l[o<]."

  `(not (o< ,y ,x)))

(defmacro o>= (x y)
  ":Doc-Section ACL2::Programming

  the greater-than-or-equal relation for the ordinals~/~/

  ~c[O>=] is a macro and ~c[(o>= x y)] expands to ~c[(not (o< x y))].  ~l[o<]."

  `(not (o< ,x ,y)))

(defun o-p (x)

  ":Doc-Section ACL2::Programming

   a recognizer for the ordinals up to epsilon-0~/

   Using the nonnegative integers and lists we can represent the ordinals up to
   ~c[epsilon-0]. The ordinal representation used in ACL2 has changed as
   of Version_2.8 from that of Nqthm-1992, courtesy of Pete Manolios and Daron
   Vroon; additional discussion may be found in ``Ordinal Arithmetic in ACL2'',
   proceedings of ACL2 Workshop 2003,
   ~url[http://www.cs.utexas.edu/users/moore/acl2/workshop-2003/].  Previously,
   ACL2's notion of ordinal was very similar to the development given in ``New
   Version of the Consistency Proof for Elementary Number Theory'' in The
   Collected Papers of Gerhard Gentzen, ed. M.E. Szabo, North-Holland
   Publishing Company, Amsterdam, 1969, pp 132-213.~/

   The following essay is intended to provide intuition about ordinals.
   The truth, of course, lies simply in the ACL2 definitions of
   ~c[o-p] and ~ilc[o<].

   Very intuitively, think of each non-zero natural number as by being
   denoted by a series of the appropriate number of strokes, i.e.,
   ~bv[]
   0             0
   1             |
   2             ||
   3             |||
   4             ||||
   ...           ...
   ~ev[]
   Then ``~c[omega],'' here written as ~c[w], is the ordinal that might be
   written as
   ~bv[]
   w             |||||...,
   ~ev[]
   i.e., an infinite number of strokes.  Addition here is just
   concatenation.  Observe that adding one to the front of ~c[w] in the
   picture above produces ~c[w] again, which gives rise to a standard
   definition of ~c[w]:  ~c[w] is the least ordinal such that adding another
   stroke at the beginning does not change the ordinal.

   We denote by ~c[w+w] or ~c[w*2] the ``~c[doubly infinite]'' sequence that we
   might write as follows.
   ~bv[]
   w*2           |||||... |||||...
   ~ev[]
   One way to think of ~c[w*2] is that it is obtained by replacing each
   stroke in ~c[2] ~c[(||)] by ~c[w].  Thus, one can imagine ~c[w*3], ~c[w*4], etc., which
   leads ultimately to the idea of ``~c[w*w],'' the ordinal obtained by
   replacing each stroke in ~c[w] by ~c[w].  This is also written as ``~c[omega]
   squared'' or ~c[w^2], or:
   ~bv[]
    2
   w             |||||... |||||... |||||... |||||... |||||... ...
   ~ev[]
   We can analogously construct ~c[w^3] by replacing each stroke in ~c[w] by
   ~c[w^2] (which, it turns out, is the same as replacing each stroke in
   ~c[w^2] by ~c[w]).  That is, we can construct ~c[w^3] as ~c[w] copies of ~c[w^2],
   ~bv[]
    3              2       2       2       2
   w              w  ...  w  ...  w  ...  w ... ...
   ~ev[]
   Then we can construct ~c[w^4] as ~c[w] copies of ~c[w^3], ~c[w^5] as ~c[w] copies of
   ~c[w^4], etc., ultimately suggesting ~c[w^w].  We can then stack ~c[omega]s,
   i.e., ~c[(w^w)^w] etc.  Consider the ``limit'' of all of those stacks,
   which we might display as follows.
   ~bv[]
          .
         .
        .
       w
      w
     w
    w
   w
   ~ev[]
   That is epsilon-0.

   Below we begin listing some ordinals up to ~c[epsilon-0]; the reader can
   fill in the gaps at his or her leisure.  We show in the left column
   the conventional notation, using ~c[w] as ``~c[omega],'' and in the right
   column the ACL2 object representing the corresponding ordinal.
   ~bv[]
     ordinal            ACL2 representation

     0                  0
     1                  1
     2                  2
     3                  3
     ...                ...
     w                 '((1 . 1) . 0)
     w+1               '((1 . 1) . 1)
     w+2               '((1 . 1) . 2)
     ...                ...
     w*2               '((1 . 2) . 0)
     (w*2)+1           '((1 . 2) . 1)
     ...                ...
     w*3               '((1 . 3) . 0)
     (w*3)+1           '((1 . 3) . 1)
     ...                ...

      2
     w                 '((2 . 1) . 0)
     ...                ...

      2
     w +w*4+3          '((2 . 1) (1 . 4) . 3)
     ...                ...

      3
     w                 '((3 . 1) . 0)
     ...                ...


      w
     w                 '((((1 . 1) . 0) . 1) . 0)
     ...                ...

      w  99
     w +w  +w4+3       '((((1 . 1) . 0) . 1) (99 . 1) (1 . 4) . 3)
     ...                ...

       2
      w
     w                 '((((2 . 1) . 0) . 1) . 0)

     ...                ...

       w
      w
     w                 '((((((1 . 1) . 0) . 1) . 0) . 1) . 0)
     ...               ...
   ~ev[]
   Observe that the sequence of ~c[o-p]s starts with the natural
   numbers (which are recognized by ~ilc[natp]). This is convenient
   because it means that if a term, such as a measure expression for
   justifying a recursive function (~pl[o<]) must produce an ~c[o-p],
   it suffices for it to produce a natural number.

   The ordinals listed above are listed in ascending order.  This is
   the ordering tested by ~ilc[o<].

   The ``~c[epsilon-0] ordinals'' of ACL2 are recognized by the recursively
   defined function ~c[o-p].  The base case of the recursion tells us that
   natural numbers are ~c[epsilon-0] ordinals.  Otherwise, an ~c[epsilon-0]
   ordinal is a list of ~ilc[cons] pairs whose final ~ilc[cdr] is a natural
   number, ~c[((a1 . x1) (a2 . x2) ... (an . xn) . p)].  This corresponds to
   the ordinal ~c[(w^a1)x1 + (w^a2)x2 + ... + (w^an)xn + p].  Each ~c[ai] is an
   ordinal in the ACL2 representation that is not equal to 0.  The sequence of
   the ~c[ai]'s is strictly decreasing (as defined by ~ilc[o<]). Each ~c[xi]
   is a positive integer (as recognized by ~ilc[posp]).

   Note that infinite ordinals should generally be created using the ordinal
   constructor, ~ilc[make-ord], rather than ~ilc[cons]. The functions
   ~ilc[o-first-expt], ~ilc[o-first-coeff], and ~ilc[o-rst] are ordinals
   destructors.  Finally, the function ~ilc[o-finp] and the macro ~ilc[o-infp]
   tell whether an ordinal is finite or infinite, respectively.

   The function ~ilc[o<] compares two ~c[epsilon-0] ordinals, ~c[x] and ~c[y].
   If both are integers, ~c[(o< x y)] is just ~c[x<y].  If one is an integer
   and the other is a ~ilc[cons], the integer is the smaller.  Otherwise,
   ~ilc[o<] recursively compares the ~ilc[o-first-expt]s of the ordinals to
   determine which is smaller.  If they are the same, the ~ilc[o-first-coeff]s
   of the ordinals are compared.  If they are equal, the ~ilc[o-rst]s of the
   ordinals are recursively compared.

   Fundamental to ACL2 is the fact that ~ilc[o<] is well-founded on
   ~c[epsilon-0] ordinals.  That is, there is no ``infinitely descending
   chain'' of such ordinals.  ~l[proof-of-well-foundedness]."

  (declare (xargs :guard t
                  :verify-guards nil))
  (if (o-finp x)
      (natp x)
    (and (consp (car x))
         (o-p (o-first-expt x))
         (not (eql 0 (o-first-expt x)))
         (posp (o-first-coeff x))
         (o-p (o-rst x))
         (o< (o-first-expt (o-rst x))
             (o-first-expt x)))))

(defthm o-p-implies-o<g
  (implies (o-p a)
           (o<g a)))

(verify-guards o-p)

(defun make-ord (fe fco rst)

  ":Doc-Section Programming

  a constructor for ordinals.~/

  ~c[Make-ord] is the ordinal constructor.  Its use is recommended instead of
  using ~ilc[cons] to make ordinals.  For a discussion of ordinals,
  ~pl[ordinals].~/

  For any ordinal, ~c[alpha < epsilon-0], there exist natural numbers ~c[p] and
  ~c[n], positive integers ~c[x1, x2, ..., xn] and ordinals
  ~c[a1 > a2 > ... > an > 0] such that ~c[alpha > a1] and
  ~c[alpha = w^(a1)x1 + w^(a2)x2 + ... + w^(an)xn + p].  We call ~c[a1] the ``first
  exponent'', ~c[x1] the ``first coefficient'', and the remainder
  ~c[(w^(a2)x2 + ... + w^(an)xn + p)] the ``rest'' of alpha.

  ~c[(Make-ord fe fco rst)] corresponds to the ordinal
  ~c[(w^fe)fco + rst].  Thus the first infinite ordinal, ~c[w] (~c[omega]), is
  constructed by
  ~bv[]
  (make-ord 1 1 0)
  ~ev[]
  and, for example, the ordinal ~c[(w^2)5 + w2 + 7] is constructed by:
  ~bv[]
  (make-ord 2 5 (make-ord 1 2 7)) .
  ~ev[]

  The reason ~c[make-ord] is used rather than ~ilc[cons] is that it
  allows us to reason more abstractly about the ordinals, without
  having to worry about the underlying representation."

  (declare (xargs :guard (and (posp fco)
                              (o-p fe)
                              (o-p rst))))
  (cons (cons fe fco) rst))

(defun list*-macro (lst)
  (declare (xargs :guard (and (true-listp lst)
                              (consp lst))))
  (if (endp (cdr lst))
      (car lst)
      (cons 'cons
            (cons (car lst)
                  (cons (list*-macro (cdr lst)) nil)))))

#+acl2-loop-only
(defmacro list* (&rest args)

  ":Doc-Section ACL2::Programming

  build a list~/

  ~c[List*] is the Common Lisp macro for building a list of objects from
  given elements and a tail.  For example, ~c[(list* 5 6 '(7 8 9))] equals
  the list ~c['(5 6 7 8 9)].  Also ~pl[list].~/

  ~c[List*] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (consp args)))
  (list*-macro args))

#-acl2-loop-only
(progn

(defmacro throw-without-attach (fn formals)
  `(throw-raw-ev-fncall
    (list* 'ev-fncall-null-body-er
           ',fn
           (list ,@formals))))

(defvar *aokp*

; We set *aokp* to t simply so that we can use attachments in raw Lisp.  It
; will be bound suitably inside the ACL2 loop by calls of raw-ev-fncall.

  t)

(defmacro aokp ()
  '*aokp*)

(defvar *attached-fn-called*)

(defmacro throw-or-attach (fn formals alt-formals &optional *1*-p)

; If alt-formals is non-nil, then it is to be used in place of formals when
; reporting an undefined-function error.

  (let ((at-fn (attachment-symbol fn)))
    `(let ()
       (declare (special ,at-fn))
       (cond ((and (boundp ',at-fn)
                   ,at-fn
                   (aokp))
              #+hons
              (when (and (boundp '*attached-fn-called*)
                         (null *attached-fn-called*))
                (setq *attached-fn-called* ',fn))
              (funcall ,(if *1*-p
                            `(*1*-symbol ,at-fn)
                          at-fn)
                       ,@formals))
             (t (throw-without-attach
                 ,fn
                 ,(or alt-formals
                      formals)))))))

)

(defun null-body-er (fn formals maybe-attach)
  (declare (xargs :guard t))
  (if maybe-attach
      (list 'throw-or-attach fn formals nil)
    (list 'throw-without-attach fn formals)))

; CLTL2 and the ANSI standard have made the main Lisp package name be
; COMMON-LISP rather than the older LISP.  Before Version_2.6 we
; handled this discrepancy in a way that could be said to be unsound.
; For example, one could prove (equal (symbol-package-name 'car)
; "LISP") in an ACL2 built on top of GCL, then prove (equal
; (symbol-package-name 'car) "COMMON-LISP")) in an ACL2 built on top
; of Allegro CL.  Thus, one could certify a book with the former
; theorem in a GCL-based ACL2, then include that book in an
; Allegro-based ACL2 and prove NIL.  Our solution is to make the
; "LISP" package look like "COMMON-LISP" from the perspective of ACL2,
; for example: (symbol-package-name 'car) = "COMMON-LISP".

; Warning: If you change the following, change the corresponding line in the
; defparameter for *ever-known-package-alist* above, consider changing
; symbol-package-name, and perhaps adjust the check for "LISP" in defpkg-fn.

(defconst *main-lisp-package-name* "COMMON-LISP")

; Warning: If you add primitive packages to this list, be sure to add
; the defaxioms that would be done by defpkg.  For example, below you
; will find a defaxiom for ACL2-INPUT-CHANNEL-PACKAGE and any new
; package should have an analogous axiom added.  Each of the primitive
; packages below has such an axiom explicitly added in axioms.lisp
; (except for the main lisp package name, whose import list is
; essentially unknown).

; Warning:  Keep the initial value of the following constant identical to
; that of the raw lisp defparameter *ever-known-package-alist* above.

(defconst *initial-known-package-alist*
  (list (make-package-entry :name "ACL2-INPUT-CHANNEL"
                            :imports nil)
        (make-package-entry :name "ACL2-OUTPUT-CHANNEL"
                            :imports nil)
        (make-package-entry :name "ACL2"
                            :imports *common-lisp-symbols-from-main-lisp-package*)
        (make-package-entry :name *main-lisp-package-name*
                            :imports nil)
        (make-package-entry :name "KEYWORD"
                            :imports nil)))

(defaxiom stringp-symbol-package-name
  (stringp (symbol-package-name x))
  :rule-classes :type-prescription)

(defaxiom symbolp-intern-in-package-of-symbol
  (symbolp (intern-in-package-of-symbol x y))
  :rule-classes :type-prescription)

(defaxiom symbolp-pkg-witness
  (symbolp (pkg-witness x))
  :rule-classes :type-prescription)

#-acl2-loop-only
(defparameter *ld-level*

; This parameter will always be equal to the number of recursive calls of LD
; and/or WORMHOLE we are in.  Since each pushes a new frame on
; *acl2-unwind-protect-stack* the value of *ld-level* should always be the
; length of the stack.  But *ld-level* is maintained as a special, i.e., it is
; always bound when we enter LD while the stack is a global.  An abort may
; possibly rip us out of a call of LD, causing *ld-level* to decrease but not
; affecting the stack.  It is this violation of the "invariant" between the two
; that indicates that the stack must be unwound some (to cleanup after an
; aborted inferior).

; Warning: This variable is let-bound in ld-fn.  This could present a problem
; if parallelism is enabled and the theorem prover uses parallelism
; primitives.  We can imagine (and we may have seen) a case in which there are
; two threads doing rewriting, and one does a throw (say, because time has
; expired), which puts the two threads temporarily out of sync in their values
; of *ld-level*.  So think about this sort of issue before parallelizing the
; theorem prover!

  0)

; For an explanation of the next defvar, see the comment in
; hard-error, below.

#-acl2-loop-only
(defvar *hard-error-returns-nilp* nil)

#-acl2-loop-only
(defun-one-output throw-raw-ev-fncall (val)

; This function just throws to raw-ev-fncall (or causes an
; interface-er if there is no raw-ev-fncall).  The coding below
; actually assumes that we are in a raw-ev-fncall if *ld-level* > 0.

; This assumption may not be entirely true.  If we have a bug in our
; LD code, e.g., in printing the prompt, we could throw to a
; nonexistent tag.  We might get the GCL

; Error: The tag RAW-EV-FNCALL is undefined.

  (if (or (= *ld-level* 0)
          (raw-mode-p *the-live-state*))
      (interface-er "~@0"
                    (ev-fncall-msg val (w *the-live-state*)))
    (throw 'raw-ev-fncall val)))

(defun hard-error (ctx str alist)

; Logically, this function just returns nil.  The implementation
; usually signals a hard error, which is sound since it is akin to
; running out of stack or some other resource problem.

; But if this function is called as part of a proof, e.g.,
; (thm (equal (car (cons (hard-error 'ctx "Test" nil) y)) nil))
; we do not want to cause an error!  (Note:  the simpler example
; (thm (equal (hard-error 'ctx "Test" nil) nil)) can be proved
; without any special handling of the executable counterpart of
; hard-error, because we know its type-set is *ts-nil*.  So to cause
; an error, you have to have the hard-error term used in a place
; where type-reasoning alone won't do the job.)

; Sometimes hard-error is used in the guard of a function, e.g.,
; illegal.  Generally evaluating that guard is to signal an error.
; But if guard-checking-on is nil, then we want to cause no error and
; just let the guard return nil.  We evaluate the guard even when
; guard-checking-on is nil (though not for user-defined functions when
; it is :none) so we know whether to call the raw Lisp version or the
; ACL2_*1*_ACL2 version of a function.

; Logically speaking the two behaviors of hard-error, nil or error,
; are indistinguishable.  So we can choose which behavior we want
; without soundness concerns.  Therefore, we have a raw Lisp special
; variable, named *hard-error-returns-nilp*, and if it is true, we
; return nil.  It is up to the environment to somehow set that special
; variable.

; In ev-fncall we provide the argument hard-error-returns-nilp which
; is used as the binding of *hard-error-returns-nil* when we invoke
; the raw code.  This also infects ev and the other functions in the
; ev-fncall clique, namely ev-lst and ev-acl2-unwind-protect.  It is
; up to the user of ev-fncall to specify which behavior is desired.
; Generally speaking, that argument of ev-fncall is set to t in those
; calls of ev-fncall that are from within the theorem prover and on
; terms from the conjecture being proved.  Secondly, (up to
; Version_2.5) in oneify-cltl-code and oneify-cltl-code, when we
; generated the ACL2_*1*_ACL2 code for a function, we laid down a
; binding for *hard-error-returns-nil*.  That binding is in effect
; just when we evaluate the guard of the function.  The binding is t
; if either it was already (meaning somebody above us has asked for
; hard-error to be treated this way) or if guard checking is turned
; off.

; See the comment after ILLEGAL (below) for a discussion of an
; earlier, inadequate handling of these issues.

  ":Doc-Section ACL2::Programming

  print an error message and stop execution~/

  ~c[(Hard-error ctx str alist)] causes evaluation to halt with a short
  message using the ``context'' ~c[ctx].  An error message is first printed
  using the string ~c[str] and alist ~c[alist] that are of the same kind
  as expected by ~ilc[fmt].  ~l[fmt].  Also ~pl[er] for a macro that provides a
  unified way of signaling errors.~/

  ~c[Hard-error] has a guard of ~c[t].  Also ~pl[illegal] for a
  similar capability which however has a guard of ~c[nil] that supports
  static checking using ~ilc[guard] verification, rather than using dynamic
  (run-time) checking.   This distinction is illustrated elsewhere:
  ~pl[prog2$] for examples.

  Semantically, ~c[hard-error] ignores its arguments and always returns
  ~c[nil].  But if a call ~c[(hard-error ctx str alist)] is encountered
  during evaluation, then the string ~c[str] is printed using the
  association list ~c[alist] (as in ~ilc[fmt]), after which evaluation halts
  immediately.  Here is a trivial, contrived example.
  ~bv[]
  ACL2 !>(cons 3 (hard-error 'my-context
                              \"Printing 4: ~~n0\"
                              (list (cons #\\0 4))))


  HARD ACL2 ERROR in MY-CONTEXT:  Printing 4: four



  ACL2 Error in TOP-LEVEL:  Evaluation aborted.

  ACL2 !>
  ~ev[]~/"

  (declare (xargs :guard t))
  #-acl2-loop-only
  (cond
   ((not *hard-error-returns-nilp*)

; We are going to ``cause an error.''

    (cond ((fboundp 'acl2::error-fms)                        ;;; Print a msg
           (let ((*standard-output* *error-output*)          ;;; one way ...
                 (fn 'acl2::error-fms))
             (funcall fn t ctx str alist *the-live-state*)))
          (t (print (list ctx str alist) *error-output*)))   ;;; or another.

; Once upon a time hard-error took a throw-flg argument and did the
; following throw-raw-ev-fncall only if the throw-flg was t.  Otherwise,
; it signalled an interface-er.  Note that in either case it behaved like
; an error -- interface-er's are rougher because they do not leave you in
; the ACL2 command loop.  I think this aspect of the old code was a vestige
; of the pre-*ld-level* days when we didn't know if we could throw or not.

      (throw-raw-ev-fncall 'illegal)))
  #+acl2-loop-only
  (declare (ignore ctx str alist))
  nil)

(defun illegal (ctx str alist)

  ":Doc-Section ACL2::Programming

  print an error message and stop execution~/

  ~c[(Illegal ctx str alist)] causes evaluation to halt with a short
  message using the ``context'' ~c[ctx].  An error message is first printed
  using the string ~c[str] and alist ~c[alist] that are of the same kind
  as expected by ~ilc[fmt].  ~l[fmt], and ~pl[prog2$] for an
  example of how to use a related function, ~ilc[hard-error]
  (~pl[hard-error]).  Also ~pl[er] for a macro that provides a unified
  way of signaling errors.~/

  The difference between ~c[illegal] and ~ilc[hard-error] is that the former
  has a guard of ~c[nil] while the latter has a ~ilc[guard] of ~c[t].  Thus,
  you may want to use ~c[illegal] rather than ~c[hard-error] when you intend
  to do ~ilc[guard] verification at some point, and you expect the guard
  to guarantee that the ~c[illegal] call is never executed.
  ~l[prog2$] for an example.~/"

; We would like to use this function in :common-lisp-compliant function
; definitions, but prove that it's never called.  Thus we have to make this
; function :common-lisp-compliant, and its guard is then nil.

  (declare (xargs :guard (hard-error ctx str alist)))
  (hard-error ctx str alist))

; Note on Inadequate Handling of Illegal.

; Once upon a time (pre-Version  2.4) we had hard-error take an additional
; argument and the programmer used that argument to indicate whether the
; function was to cause an error or return nil.  When hard-error was used
; in the :guard of ILLEGAL it was called so as not to cause an error (if
; guard checking was off) and when it was called in the body of ILLEGAL it
; was programmed to cause an error.  However, the Rockwell folks, using
; LETs in support of stobjs, discovered that we caused hard errors on
; some guard verifications.  Here is a simple example distilled from theirs:

#|
 (defun foo (i)
   (declare (xargs :guard (integerp i)))
   (+ 1
      (car
       (let ((j i))
         (declare (type integer j))
         (cons j nil)))))
|#

; This function caused a hard error during guard verification.  The
; troublesome guard conjecture is:
#|
 (IMPLIES
  (INTEGERP I)
  (ACL2-NUMBERP
   (CAR (LET ((J I))
          (PROG2$ (IF (INTEGERP J)
                      T
                      (ILLEGAL 'VERIFY-GUARDS
                               "Some TYPE declaration is violated."
                               NIL))
                  (LIST J))))))
|#
; The problem was that we eval'd the ILLEGAL during the course of trying
; to prove this.  A similar challenge is the above mentioned
; (thm (equal (car (cons (hard-error 'ctx "Test" nil) y)) nil))
; We leave this note simply in case the current handling of
; hard errors is found still to be inadequate.

#+acl2-loop-only
(defmacro intern (x y)
  (declare (xargs :guard (member-equal y
                                       (cons *main-lisp-package-name*
                                             '("ACL2"
                                               *main-lisp-package-name*
                                               "ACL2-INPUT-CHANNEL"
                                               "ACL2-OUTPUT-CHANNEL"
                                               "KEYWORD")))))
  ":Doc-Section ACL2::Programming

  create a new symbol in a given package~/

  ~c[(intern symbol-name symbol-package-name)] returns a symbol with
  the given ~ilc[symbol-name] and the given ~ilc[symbol-package-name].  We
  restrict Common Lisp's ~c[intern] so that the second argument is
  either the symbol *main-lisp-package-name*, the value of that
  constant, or is one of \"ACL2\", \"ACL2-INPUT-CHANNEL\",
  \"ACL2-OUTPUT-CHANNEL\", or \"KEYWORD\".  To avoid that restriction,
  ~pl[intern$].~/

  In ACL2 ~c[intern] is actually implemented as a macro that expands to
  a call of a similar function whose second argument is a symbol.
  Invoke ~c[:pe intern] to see the definition, or
  ~pl[intern-in-package-of-symbol].

  To see why is ~c[intern] so restricted consider
  ~c[(intern \"X\" \"P\")].  In particular, is it a symbol and if so,
  what is its ~ilc[symbol-package-name]?  One is tempted to say ``yes, it
  is a symbol in the package ~c[\"P\"].''  But if package ~c[\"P\"] has
  not yet been defined, that would be premature because the imports to
  the package are unknown.  For example, if ~c[\"P\"] were introduced
  with
  ~bv[]
  (defpkg \"P\" '(LISP::X))
  ~ev[]
  then in Common Lisp ~c[(symbol-package-name (intern \"X\" \"P\"))] returns
  ~C[\"LISP\"].

  The obvious restriction on ~c[intern] is that its second argument be
  the name of a package known to ACL2.  We cannot express such a
  restriction (except, for example, by limiting it to those packages
  known at some fixed time, as we do).  Instead, we provide
  ~ilc[intern-in-package-of-symbol] which requires a ``witness symbol''
  for the package instead of the package.  The witness symbol is any
  symbol (expressible in ACL2) and uniquely specifies a package
  necessarily known to ACL2."

  (list 'intern-in-package-of-symbol
        x
        (cond
         ((equal y "ACL2")
          ''rewrite)
         ((equal y "ACL2-INPUT-CHANNEL")
          ''acl2-input-channel::a-random-symbol-for-intern)
         ((equal y "ACL2-OUTPUT-CHANNEL")
          ''acl2-output-channel::a-random-symbol-for-intern)
         ((equal y "KEYWORD")
          ':a-random-symbol-for-intern)
         ((or (equal y *main-lisp-package-name*)
              (eq y '*main-lisp-package-name*))
          ''car)
         (t (illegal 'intern
                     "The guard for INTERN is out of sync with its ~
                      definition.~%Consider adding a case for a second ~
                      argument of ~x0."
                     (list (cons #\0 y)))))))

(defmacro intern$ (x y)

  ":Doc-Section ACL2::Programming

  create a new symbol in a given package~/

  ~c[Intern$] is a macro that behaves the same as the macro ~ilc[intern],
  except for weakening the restriction to a fixed set of package names so that
  any package name other than ~c[\"\"] is legal.  ~l[intern].  Note that if you
  evaluate a call ~c[(intern$ x y)] for which there is no package with name
  ~c[y] that is known to ACL2, you will get an error.~/

  ~c[(Intern$ x y)] expands to:
  ~bv[]
  (intern-in-package-of-symbol x (pkg-witness y))
  ~ev[]
  ~l[intern-in-package-of-symbol] and ~pl[pkg-witness].~/"

  `(intern-in-package-of-symbol ,x (pkg-witness ,y)))

#+acl2-loop-only
(defun keywordp (x)

  ":Doc-Section ACL2::Programming

  recognizer for keywords~/

  ~c[(Keywordp x)] is true if and only if ~c[x] is a keyword, i.e., a symbol in
  the \"KEYWORD\" package.  Such symbols are typically printed using a colon
  (:) followed by the ~ilc[symbol-name] of the symbol.~/

  ~c[Keywordp] has a ~il[guard] of ~c[t].

  ~c[Keywordp] is a Common Lisp function.  See any Common Lisp documentation
  for more information.  The following log may be illuminating.
  ~bv[]
  ACL2 !>(intern \"ABC\" \"KEYWORD\")
  :ABC
  ACL2 !>(symbol-name ':ABC)
  \"ABC\"
  ACL2 !>(symbol-package-name ':ABC)
  \"KEYWORD\"
  ACL2 !>
  ~ev[]~/"

  (declare (xargs :guard t))
  (and (symbolp x) (equal (symbol-package-name x) "KEYWORD")))

(defthm keywordp-forward-to-symbolp
  (implies (keywordp x)
           (symbolp x))
  :rule-classes :forward-chaining)

(defaxiom intern-in-package-of-symbol-symbol-name

; This axiom assumes that "" is not the name of any package, but is instead
; used as a default value when symbol-package-name is applied to a non-symbol.
; So, the hypotheses below imply (symbolp y).  See also the lemma
; symbol-package-name-of-symbol-is-not-empty-string, below.  See also
; chk-acceptable-defpkg for a related comment, in which a proof of nil is shown
; using this axiom when "" is not disallowed as a package name.

  (implies (and (symbolp x)
                (equal (symbol-package-name x) (symbol-package-name y)))
           (equal (intern-in-package-of-symbol (symbol-name x) y) x)))

(defthm symbol-package-name-of-symbol-is-not-empty-string

; This rule became necessary for the proof of lemma nice->simple-inverse in
; books/workshops/2003/sumners/support/n2n.lisp, after axiom
; symbol-package-name-pkg-witness-name (below) was modified after Version_3.0.1
; by adding the condition (not (equal pkg-name "")).  We make it a
; :forward-chaining rule in order to avoid hanging a rewrite rule on 'equal.

  (implies (symbolp x)
           (not (equal (symbol-package-name x) "")))
  :hints (("Goal"
           :use ((:instance intern-in-package-of-symbol-symbol-name
                            (x x) (y 3)))
           :in-theory (disable intern-in-package-of-symbol-symbol-name)))
  :rule-classes ((:forward-chaining :trigger-terms ((symbol-package-name x)))))

(defconst *pkg-witness-name* "ACL2-PKG-WITNESS")

(defaxiom symbol-name-pkg-witness
  (equal (symbol-name (pkg-witness pkg-name))
         *pkg-witness-name*))

(defaxiom symbol-package-name-pkg-witness-name
  (equal (symbol-package-name (pkg-witness pkg-name))
         (if (and (stringp pkg-name)
                  (not (equal pkg-name "")))
             pkg-name

; See the comment in intern-in-package-of-symbol-symbol-name for why we do not
; use "" below.  We avoid questions about names of built-in Lisp and keyword
; packages by using our own package name.

           "ACL2")))

;  member-symbol-name is used in the defpkg axiom.

(defun member-symbol-name (str l)
  (declare (xargs :guard (symbol-listp l)))
  (cond ((endp l) nil)
        ((equal str (symbol-name (car l))) l)
        (t (member-symbol-name str (cdr l)))))

(defthm symbol-equality

; This formula is provable using intern-in-package-of-symbol-symbol-name.

   (implies (and (symbolp s1)
                 (symbolp s2)
                 (equal (symbol-name s1) (symbol-name s2))
                 (equal (symbol-package-name s1) (symbol-package-name s2)))
            (equal s1 s2))
   :rule-classes nil
   :hints (("Goal"
            :in-theory (disable intern-in-package-of-symbol-symbol-name)
            :use
            ((:instance
              intern-in-package-of-symbol-symbol-name
              (x s1) (y s2))
             (:instance
              intern-in-package-of-symbol-symbol-name
              (x s2) (y s2))))))

(defaxiom symbol-name-intern-in-package-of-symbol
  (implies (and (stringp s)
                (symbolp any-symbol))
           (equal (symbol-name (intern-in-package-of-symbol s any-symbol)) s)))

; These axioms are just the ones that would be added by defpkg had the packages
; in question been introduced that way.

; Warning: If the forms of these axioms are changed, you should
; probably visit the same change to the rules added by defpkg.

(defaxiom acl2-input-channel-package
  (implies (and (stringp x)
                (symbolp y)
                (equal (symbol-package-name y)
                       "ACL2-INPUT-CHANNEL"))
           (equal (symbol-package-name (intern-in-package-of-symbol x y))
                  "ACL2-INPUT-CHANNEL")))

(defaxiom acl2-output-channel-package
  (implies (and (stringp x)
                (symbolp y)
                (equal (symbol-package-name y)
                       "ACL2-OUTPUT-CHANNEL"))
           (equal (symbol-package-name (intern-in-package-of-symbol x y))
                  "ACL2-OUTPUT-CHANNEL")))

(defaxiom acl2-package
  (implies (and (stringp x)
                (not (member-symbol-name
                      x
                      *common-lisp-symbols-from-main-lisp-package*))
                (symbolp y)
                (equal (symbol-package-name y)
                       "ACL2"))
           (equal (symbol-package-name (intern-in-package-of-symbol x y))
                  "ACL2")))

(defaxiom keyword-package
  (implies (and (stringp x)
                (symbolp y)
                (equal (symbol-package-name y)
                       "KEYWORD"))
           (equal (symbol-package-name (intern-in-package-of-symbol x y))
                  "KEYWORD")))

; Adding a similar axiom for pkg "COMMON-LISP" would be wrong.  We do not
; know what the imports to "COMMON-LISP" are, they differ from lisp to lisp.

; The following two axioms are probably silly.  But at least they may provide
; steps towards building up the ACL2 objects constructively from a few
; primitives.

(defaxiom string-is-not-circular
  (equal 'string
         (intern-in-package-of-symbol
          (coerce (cons #\S (cons #\T (cons #\R (cons #\I (cons #\N (cons #\G 0))))))
                  (cons #\S (cons #\T (cons #\R (cons #\I (cons #\N (cons #\G 0)))))))
          (intern-in-package-of-symbol 0 0)))
  :rule-classes nil)

(defaxiom nil-is-not-circular
  (equal nil
         (intern-in-package-of-symbol
          (coerce (cons #\N (cons #\I (cons #\L 0))) 'string)
          'string))
  :rule-classes nil)

; Essay on Symbols and Packages

; A symbol may be viewed as a pair consisting of two strings: its symbol-name
; and its symbol-package-name, where the symbol-package-name is not "".  (A
; comment in intern-in-package-of-symbol-symbol-name discusses why we disallow
; "".)  However, some such pairs are not symbols because of the import
; structure (represented in world global 'known-package-alist).  For example,
; the "ACL2" package imports a symbol with symbol-name "CAR" from the
; "COMMON-LISP" package, so the symbol-package-name of ACL2::CAR is
; "COMMON-LISP".  Thus there is no symbol with a symbol-name of "CAR" and a
; symbol-package-name of "ACL2".

; The package system has one additional requirement: No package is allowed to
; import any symbol named *pkg-witness-name* from any other package.  The
; function pkg-witness returns a symbol with that name; moreover, the
; symbol-package-name of (pkg-witness p) is p if p is a string other than "",
; else is "ACL2".

; Logically, we imagine that a package exists for every string (serving as the
; symbol-package-name of its symbols) except "".  Of course, at any given time
; only finite many packages have been specified (either being built-in, or
; specified with defpkg); and, ACL2 will prohibit explicit specification of
; packages for certain strings, such as "ACL2_INVISIBLE".

; Finally, we specify that the symbol-name and symbol-package-name of any
; non-symbol are "".

#-acl2-loop-only
(defun-one-output intern-in-package-of-symbol (str sym)

; In general we require that intern be given an explicit string constant
; that names a package known at translate time.  This avoids the run-time
; check that the package is known -- which would require passing state down
; to intern everywhere.  However, we would like a more general intern
; mechanism and hence define the following, which is admitted by special
; decree in translate.  The beauty of this use of intern is that the user
; supplies a symbol which establishes the existence of the desired package.

  (declare (type string str)
           (type symbol sym))
  (let ((ans

; Bob Boyer notes that intern returns more than one value, so we may get better
; efficiency (at least in GCL) by this let-binding trick, which throws away all
; but the first value.

         (intern str

; We do not use symbol-package-name below because for the main Lisp package, it
; may return the name of a non-existent package.  In particular, for GCL we
; have (symbol-package-name 'car) = "COMMON-LISP", which is not the name of a
; package, while (symbol-name (symbol-package 'car)) = "LISP", which is.

                 (symbol-package sym))))
    ans))

(defdoc pkg-witness
  ":Doc-Section ACL2::Programming

  return a specific symbol in the indicated package~/

  For any string ~c[pkg] that names a package currently known to ACL2,
  ~c[(pkg-witness pkg)] is a symbol in that package whose ~ilc[symbol-name] is
  the value of constant ~c[*pkg-witness-name*].  Logically, this is the case
  even if the package is not currently known to ACL2.  However, if
  ~c[pkg-witness] is called on a string that is not the name of a package known
  to ACL2, a hard Lisp error will result.~/

  ~c[(Pkg-witness pkg)] has a guard of
  ~c[(and (stringp pkg) (not (equal pkg \"\")))].  If ~c[pkg] is not a string,
  then ~c[(pkg-witness pkg)] is equal to ~c[(pkg-witness \"ACL2\")]~/")

#-acl2-loop-only
(defun-one-output pkg-witness (pkg)
  (declare (type string pkg))
  (cond ((find-non-hidden-package-entry pkg
                                        (known-package-alist *the-live-state*))
         (let ((ans (intern *pkg-witness-name* pkg)))
; See comment in intern-in-package-of-symbol for an explanation of this trick.
           ans))
        (t

; We use error rather than illegal, because we want to throw an error even when
; *hard-error-returns-nilp* is true.

         (error "The argument supplied to PKG-WITNESS, ~s, is not the name of ~
                 a package currently known to ACL2."
                pkg))))

;  UTILITIES - definitions of the rest of applicative Common Lisp.

(defun binary-append (x y)

  ":Doc-Section ACL2::Programming

  ~il[concatenate] two lists~/

  This binary function implements ~ilc[append], which is a macro in ACL2.
  ~l[append]~/

  The ~il[guard] for ~c[binary-append] requires the first argument to be a
  ~ilc[true-listp].~/"

  (declare (xargs :guard (true-listp x)))
  (cond ((endp x) y)
        (t (cons (car x) (binary-append (cdr x) y)))))

#+acl2-loop-only
(defmacro append (x y &rest rst)

  ":Doc-Section ACL2::Programming

  ~il[concatenate] two or more lists~/

  ~c[Append], which takes two or more arguments, expects all the
  arguments except perhaps the last to be true (null-terminated)
  lists.  It returns the result of concatenating all the elements of
  all the given lists into a single list.  Actually, in ACL2 ~c[append]
  is a macro that expands into calls of the binary function
  ~ilc[binary-append].~/

  ~c[Append] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (xxxjoin 'binary-append (cons x (cons y rst))))

(defthm true-listp-append

; This rule has the effect of making the system automatically realize that (rev
; x) is a true-list, for example, where:

;   (defun rev (x)
;     (if (endp x)
;         nil
;       (append (rev (cdr x))
;               (list (car x)))))

; That in turn means that when it generalizes (rev x) to z it adds (true-listp
; z).

; That in turn means it can prove

;   (defthm rev-append
;     (equal (rev (append a b))
;            (append (rev b) (rev a))))
;   
; automatically, doing several generalizations and inductions.

  (implies (true-listp b)
           (true-listp (append a b)))
  :rule-classes :type-prescription)

; The following lemma originally appeared to be useful for accepting the
; definition of make-input-channel.  Then it became useful for accepting the
; definition of string-append, though that's changed a bit.

(defthm standard-char-listp-append
  (implies (true-listp x)
           (equal (standard-char-listp (append x y))
                  (and (standard-char-listp x)
                       (standard-char-listp y))))
  :hints (("Goal" :in-theory (enable standard-char-listp))))

(defthm character-listp-append
  (implies (true-listp x)
           (equal (character-listp (append x y))
                  (and (character-listp x)
                       (character-listp y)))))

(defthm append-to-nil
  (implies (true-listp x)
           (equal (append x nil)
                  x)))

#+acl2-loop-only
(defmacro concatenate (result-type &rest sequences)

  ":Doc-Section ACL2::Programming

  concatenate lists or strings together~/
  ~bv[]
  Examples:
  (concatenate 'string \"ab\" \"cd\" \"ef\")     ; equals \"abcdef\"
  (concatenate 'string \"ab\")               ; equals \"ab\"
  (concatenate 'list '(a b) '(c d) '(e f)) ; equals '(a b c d e f)
  (concatenate 'list)                      ; equals nil~/

  General Form:
  (concatenate result-type x1 x2 ... xn)
  ~ev[]
  where ~c[n >= 0] and either:  ~c[result-type] is ~c[']~ilc[string] and each ~c[xi] is a
  string; or ~c[result-type] is ~c[']~ilc[list] and each ~c[xi] is a true list.
  ~c[Concatenate] simply concatenates its arguments to form the result
  string or list.  Also ~pl[append] and ~pl[string-append].  (The latter
  immediately generates a call to ~c[concatenate] when applied to strings.)

  Note:  We do *not* try to comply with the Lisp language's insistence
  that ~c[concatenate] copies its arguments.  Not only are we in an
  applicative setting, where this issue shouldn't matter for the
  logic, but also we do not actually modify the underlying lisp
  implementation of ~c[concatenate]; we merely provide a definition for
  it.

  ~c[Concatenate] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (member-equal result-type
                                       '('string 'list))))
  (cond
   ((equal result-type ''string)
    (cond ((and sequences (cdr sequences) (null (cddr sequences)))

; Here we optimize for a common case, but more importantly, we avoid expanding
; to a call of string-append-lst for the call of concatenate in the definition
; of string-append.

           (list 'string-append (car sequences) (cadr sequences)))
          (t
           (list 'string-append-lst (cons 'list sequences)))))
   ((endp sequences) nil)
   (t

; Consider the call (concatenate 'list .... '(a . b)).  At one time we tested
; for (endp (cdr sequences)) here, returning (car sequences) in that case.  And
; otherwise, we returned (cons 'append sequences).  However, these are both
; errors, because the last member of sequences might be a non-true-listp, in
; which case append signals no guard violation but Common Lisp breaks.

    (cons 'append (append sequences (list nil))))))

(defun string-append (str1 str2)

  ":Doc-Section ACL2::Programming

  ~il[concatenate] two strings~/

  ~c[String-append] takes two arguments, which are both strings (if the
  ~il[guard] is to be met), and returns a string obtained by concatenating
  together the ~il[characters] in the first string followed by those in the
  second.  Also ~pl[concatenate], noting that the macro call
  ~bv[]
  (concatenate 'string str1 str2).
  ~ev[]
  expands to the call
  ~bv[]
  (string-append str1 str2).
  ~ev[]~/~/"

  (declare (xargs :guard (and (stringp str1)
                              (stringp str2))))
  (mbe :logic
       (coerce (append (coerce str1 'list)
                       (coerce str2 'list))
               'string)
       :exec

; This code may seem circular, since string-append calls the concatenate macro,
; which expands here into a call of string-append.  However, the :exec case is
; only called if we are executing the raw Lisp code for string-append, in which
; case we will be executing the raw Lisp code for concatenate, which of course
; does not call the ACL2 function string-append.

       (concatenate 'string str1 str2)))

(defun string-listp (x)

  ":Doc-Section ACL2::Programming

  recognizer for a true list of strings~/

  The predicate ~c[string-listp] tests whether its argument is a
  ~ilc[true-listp] of strings.~/~/"

  (declare (xargs :guard t))
  (cond
   ((atom x)
    (eq x nil))
   (t
    (and (stringp (car x))
         (string-listp (cdr x))))))

(defun string-append-lst (x)
  (declare (xargs :guard (string-listp x)))
  (cond
   ((endp x)
    "")
   (t
    (string-append (car x)
                   (string-append-lst (cdr x))))))

; We make 1+ and 1- macros in order to head off the potentially common error of
; using these as nonrecursive functions on left-hand sides of rewrite rules.

#+acl2-loop-only
(defmacro 1+ (x)

  ":Doc-Section ACL2::Programming

  increment by 1~/

  ~c[(1+ x)] is the same as ~c[(+ 1 x)].  ~l[+].~/

  ~c[1+] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (list '+ 1 x))

#+acl2-loop-only
(defmacro 1- (x)

  ":Doc-Section ACL2::Programming

  decrement by 1~/

  ~c[(1- x)] is the same as ~c[(- x 1)].  ~l[-].~/

  ~c[1-] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (list '- x 1))

#+acl2-loop-only
(defun remove (x l)

  ":Doc-Section ACL2::Programming

  remove all occurrences, testing using ~ilc[eql]~/

  ~c[(remove x l)] is ~c[l] if ~c[x] is not a member of ~c[l], else is the
  result of removing all occurrences of ~c[x] from ~c[l].~/

  The ~il[guard] for ~c[(remove x l)] requires ~c[l] to be a true list and
  moreover, either ~c[x] is ~ilc[eqlablep] or all elements of ~c[l] are
  ~ilc[eqlablep].

  ~c[Remove] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  Note that we do not allow
  keyword arguments (such as ~c[test]) in ACL2 functions, in
  particular, in ~c[remove].

  Also ~pl[remove1], ~pl[remove-equal], and ~pl[remove-eq].~/"

  (declare (xargs :guard (if (eqlablep x)
                             (true-listp l)
                           (eqlable-listp l))))
  (cond ((endp l) nil)
        ((eql x (car l))
         (remove x (cdr l)))
        (t (cons (car l) (remove x (cdr l))))))

(defun remove-eq (x l)

  ":Doc-Section ACL2::Programming

  remove all occurrences, testing using ~ilc[eq]~/

  ~c[(remove-eq x l)] is ~c[l] if ~c[x] is not a member of ~c[l] as tested with
  ~ilc[member-eq], else is the result of removing all occurrences of ~c[x] from
  ~c[l].~/

  The ~il[guard] for ~c[(remove-eq x l)] requires ~c[l] to be a true list and
  moreover, either ~c[x] is a ~ilc[symbolp] or all elements of ~c[l] are
  symbols (i.e., ~c[l] is a ~ilc[symbol-listp]).

  Also ~pl[remove1-eq], ~pl[remove], and ~pl[remove-equal].~/"

  (declare (xargs :guard (if (symbolp x)
                             (true-listp l)
                           (symbol-listp l))))
  #-acl2-loop-only ; for assoc-eq, Jared Davis found native assoc efficient
  (remove x l :test #'eq)
  #+acl2-loop-only
  (cond ((endp l) nil)
        ((eq x (car l))
         (remove-eq x (cdr l)))
        (t (cons (car l) (remove-eq x (cdr l))))))

(defun remove-equal (x l)

  ":Doc-Section ACL2::Programming

  remove all occurrences, testing using ~ilc[equal]~/

  ~c[(remove-equal x l)] is ~c[l] if ~c[x] is not a member of ~c[l] as tested with
  ~ilc[member-equal], else is the result of removing all occurrences of ~c[x] from
  ~c[l].~/

  The ~il[guard] for ~c[(remove-equal x l)] requires ~c[l] to be a true list.

  Also ~pl[remove1-equal], ~pl[remove], and ~pl[remove-eq].~/"

  (declare (xargs :guard (true-listp l)))
  #-acl2-loop-only ; for assoc-eq, Jared Davis found native assoc efficient
  (remove x l :test #'equal)
  #+acl2-loop-only
  (cond ((endp l) nil)
        ((equal x (car l))
         (remove-equal x (cdr l)))
        (t (cons (car l) (remove-equal x (cdr l))))))

(defun remove1 (x l)

  ":Doc-Section ACL2::Programming

  remove first occurrences, testing using ~ilc[eql]~/

  ~c[(remove1 x l)] is ~c[l] if ~c[x] is not a member of ~c[l], else is the
  result of removing the first occurrence of ~c[x] from ~c[l].~/

  The ~il[guard] for ~c[(remove1 x l)] requires ~c[l] to be a true list and
  moreover, either ~c[x] is ~ilc[eqlablep] or all elements of ~c[l] are
  ~ilc[eqlablep].

  Also ~pl[remove], ~pl[remove1-equal], and ~pl[remove1-eq].~/"

  (declare (xargs :guard (if (eqlablep x)
                             (true-listp l)
                           (eqlable-listp l))))
  (cond ((endp l) nil)
        ((eql x (car l))
         (cdr l))
        (t (cons (car l) (remove1 x (cdr l))))))

(defun remove1-eq (x l)

  ":Doc-Section ACL2::Programming

  remove first occurrences, testing using ~ilc[eq]~/

  ~c[(remove1-eq x l)] is ~c[l] if ~c[x] is not a member of ~c[l] as tested with
  ~ilc[member-eq], else is the result of removing the first occurrences of
  ~c[x] from ~c[l].~/

  The ~il[guard] for ~c[(remove1-eq x l)] requires ~c[l] to be a true list and
  moreover, either ~c[x] is a ~ilc[symbolp] or all elements of ~c[l] are
  symbols (i.e., ~c[l] is a ~ilc[symbol-listp]).

  Also ~pl[remove-eq], ~pl[remove1], and ~pl[remove1-equal].~/"

  (declare (xargs :guard (if (symbolp x)
                             (true-listp l)
                           (symbol-listp l))))
  (cond ((endp l) nil)
        ((eq x (car l))
         (cdr l))
        (t (cons (car l) (remove1-eq x (cdr l))))))

(defun remove1-equal (x l)

  ":Doc-Section ACL2::Programming

  remove first occurrences, testing using ~ilc[equal]~/

  ~c[(remove1-equal x l)] is ~c[l] if ~c[x] is not a member of ~c[l] as tested with
  ~ilc[member-equal], else is the result of removing the first occurrence of
  ~c[x] from ~c[l].~/

  The ~il[guard] for ~c[(remove1-equal x l)] requires ~c[l] to be a true list.

  Also ~pl[remove-equal], ~pl[remove1], and ~pl[remove1-eq].~/"

  (declare (xargs :guard (true-listp l)))
  (cond ((endp l) nil)
        ((equal x (car l))
         (cdr l))
        (t (cons (car l) (remove1-equal x (cdr l))))))

(deflabel pairlis
  :doc
  ":Doc-Section ACL2::Programming

  ~l[pairlis$]~/

  The Common Lisp language allows its ~c[pairlis] function to construct
  an alist in any order!  So we have to define our own version:
  ~l[pairlis$].~/~/")

(defun pairlis$ (x y)

; CLTL allows its pairlis to construct an alist in any order!  So we
; have to give this function a different name.

  ":Doc-Section ACL2::Programming

  zipper together two lists~/

  The Common Lisp language allows its ~ilc[pairlis] function to construct
  an alist in any order!  So we have to define our own version,
  ~c[pairlis$].  It returns the list of pairs obtained by ~ilc[cons]ing
  together successive respective members of the given lists until the
  first list runs out.  (Hence in particular, if the second argument
  is ~c[nil] then each element of the first argument is paired with ~c[nil].)~/

  The ~il[guard] for ~c[pairlis$] requires that its arguments are true lists.~/"

  (declare (xargs :guard (and (true-listp x)
                              (true-listp y))))
  (cond ((endp x) nil)
        (t (cons (cons (car x) (car y))
                 (pairlis$ (cdr x) (cdr y))))))

(defun remove-duplicates-eql (l)
  (declare (xargs :guard (eqlable-listp l)))
  (cond
   ((endp l) nil)
   ((member (car l) (cdr l)) (remove-duplicates-eql (cdr l)))
   (t (cons (car l) (remove-duplicates-eql (cdr l))))))

(defthm character-listp-remove-duplicates-eql
  (implies (character-listp x)
           (character-listp (remove-duplicates-eql x))))

#+acl2-loop-only
(defun remove-duplicates (l)

  ":Doc-Section ACL2::Programming

  remove duplicates from a string or (using ~ilc[eql]) a list~/

  ~c[Remove-duplicates] returns the result of deleting duplicate
  elements from the beginning of the given string or true list, i.e.,
  leaving the last element in place.  For example,
  ~bv[]
  (remove-duplicates '(1 2 3 2 4))
  ~ev[]
  is equal to ~c['(1 3 2 4)].~/

  The ~il[guard] for ~c[Remove-duplicates] requires that its argument is a
  string or a true-list of ~ilc[eqlablep] objects.  It uses the function
  ~ilc[eql] to test for equality between elements of its argument.

  ~c[Remove-duplicates] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  Note that we do not allow
  keyword arguments (such as ~c[test]) in ACL2 functions, in
  particular, in ~c[remove-duplicates].  But
  ~pl[remove-duplicates-equal], which is similar but uses the
  function ~ilc[equal] to test for duplicate elements.~/"

  (declare (xargs :guard (or (stringp l)
                             (eqlable-listp l))))
  (cond
   ((stringp l)
    (coerce (remove-duplicates-eql (coerce l 'list)) 'string))
   (t (remove-duplicates-eql l))))

(defun remove-duplicates-equal (l)

  ":Doc-Section ACL2::Programming

  remove duplicates from a list~/

  ~c[Remove-duplicates-equal] is the same as ~ilc[remove-duplicates],
  except that its argument must be a true list (not a string), and
  ~ilc[equal] is used to check membership rather than ~ilc[eql].
  ~l[remove-duplicates].~/

  The ~il[guard] for ~c[Remove-duplicates-equal] requires that its argument
  is a true list.  Note that unlike ~ilc[remove-duplicates], it does not
  allow string arguments.~/"

  (declare (xargs :guard (true-listp l)))
  (cond
   ((endp l) nil)
   ((member-equal (car l) (cdr l)) (remove-duplicates-equal (cdr l)))
   (t (cons (car l) (remove-duplicates-equal (cdr l))))))

; We now define the first five documentation sections: Events,
; Documentation, History, Other, and Miscellaneous.  These
; are defined here simply so we can use them freely throughout.  The
; first four are advertised in :help.

(deflabel events
  :doc
  ":Doc-Section Events

  functions that extend the logic~/~/

  Any extension of the syntax of ACL2 (i.e., the definition of a new
  constant or macro), the axioms (i.e., the definition of a function),
  or the rule data base (i.e., the proof of a theorem), constitutes a
  logical ``event.''  Events change the ACL2 logical world
  (~pl[world]).  Indeed, the only way to change the ACL2
  ~il[world] is via the successful evaluation of an event function.
  Every time the ~il[world] is changed by an event, a landmark is left
  on the ~il[world] and it is thus possible to identify the ~il[world]
  ``as of'' the evaluation of a given event.  An event may introduce
  new logical names.  Some events introduce no new names (e.g.,
  ~ilc[verify-guards]), some introduce exactly one (e.g., ~ilc[defmacro] and
  ~ilc[defthm]), and some may introduce many (e.g., ~ilc[encapsulate] ).

  ACL2 typically completes processing of an event by printing a summary that
  includes a breakdown of time used and, unless proofs are skipped
  (~pl[ld-skip-proofsp]) or summary output is inhibited
  (~pl[set-inhibit-output-lst]), information about the proof attempt
  (if any) including a list of rules used and a summary of warnings.  A detail:
  The time is calculated using Common Lisp function ~c[get-internal-run-time],
  which may ignore calls to external tools (~pl[sys-call] and
  ~pl[clause-processor]).~/")

(deflabel documentation
  :doc
  ":Doc-Section Documentation

  functions that display documentation~/

  This section explains the ACL2 online documentation system.  Thus,
  most of it assumes that you are typing at the terminal, inside an ACL2
  session.  If you are reading this description in another setting
  (for example, in a web browser, in Emacs info, or on paper), simply
  ignore the parts of this description that involve typing at the
  terminal.

  ACL2 users are welcome to contribute additional documentation.  See
  the web page ~url[http://www.cs.utexas.edu/users/moore/acl2/contrib/].

  For an introduction to the ACL2 online documentation system, type
  ~c[:]~ilc[more] below.  Whenever the documentation system concludes with
  ``(type :more for more, :more! for the rest)'' you may type ~c[:]~ilc[more]
  to see the next block of documentation.

  Topics related to documentation are documented individually:~/

  To view the documentation in a web browser, open a browser to file
  ~c[doc/HTML/acl2-doc.html] under your ACL2 source directory, or just go to
  the ACL2 home page at ~url[http://www.cs.utexas.edu/users/moore/acl2/].

  To use Emacs Info (inside Emacs), first load distributed file
  ~c[emacs/emacs-acl2.el] (perhaps inside your ~c[.emacs] file) and then
  execute ~c[meta-x acl2-info].  In order to see true links to external web
  pages, you may find the following addition to your ~c[.emacs] file to be
  helpful.
  ~bv[]
  ; For emacs-version 22 or (presumably) later, you can probably set
  ; arrange that in Emacs Info, URLs become links, in the sense that
  ; if you hit ~c[<RETURN>] while standing on a URL, then you will be
  ; taken to that location in a web browser.  If this does not happen
  ; automatically, then evaluating the `setq' form below might work
  ; if you have firefox.  If that does not work, then you can probably
  ; figure out what to do as follows.  First type
  ;   control-h v browse-url-browser-function
  ; and then from the resulting help page,
  ; hit <return> on the link ``customize'' in:
  ; ``You can customize this variable''
  ; and then follow instructions.
  (setq browse-url-browser-function (quote browse-url-firefox))
  ~ev[]

  There is a print version of the documentation, though we recommend using one
  of the other methods (web, Emacs Info, or online) to browse it.  If you
  really want the print version, you can find it here:
  ~url[http://www.cs.utexas.edu/users/moore/publications/acl2-book.ps.gz].

  Below we focus on how to access the online documentation, but some of the
  discussion is relevant to other formats.

  The ACL2 online documentation feature allows you to see extensive
  documentation on many ACL2 functions and ideas.  You may use the
  documentation facilities to document your own ACL2 functions and
  theorems.

  If there is some name you wish to know more about, then type
  ~bv[]
  ACL2 !>:doc name
  ~ev[]
  in the top-level loop.  If the name is documented, a brief blurb
  will be printed.  If the name is not documented, but is ``similar''
  to some documented names, they will be listed.  Otherwise, ~c[nil] is
  returned.

  Every name that is documented contains a one-line description, a few
  notes, and some details.  ~c[:]~ilc[Doc] will print the one-liner and the
  notes.  When ~c[:]~ilc[doc] has finished it stops with the message
  ``(type :more for more, :more! for the rest)'' to remind you that details are
  available.  If you then type
  ~bv[]
  ACL2 !>:more
  ~ev[]
  a block of the continued text will be printed, again concluding
  with ``(type :more for more, :more! for the rest)'' if the text continues
  further, or concluding with ``~c[*-]'' if the text has been exhausted.  By
  continuing to type ~c[:]~ilc[more] until exhausting the text you can read
  successive blocks.  Alternatively, you can type ~c[:]~ilc[more!] to get all
  the remaining blocks.

  If you want to get the details and don't want to see the elementary
  stuff typed by ~c[:]~ilc[doc] name, type:
  ~bv[]
  ACL2 !>:MORE-DOC name
  ~ev[]
  We have documented not just function names but names of certain
  important ideas too.  For example, ~pl[rewrite] and
  ~pl[meta] to learn about ~c[:]~ilc[rewrite] rules and ~c[:]~ilc[meta] rules,
  respectively.  ~l[hints] to learn about the structure of the
  ~c[:]~ilc[hints] argument to the prover.  The ~ilc[deflabel] event
  (~pl[deflabel]) is a way to introduce a logical name for no
  reason other than to attach documentation to it; also
  ~pl[defdoc].

  How do you know what names are documented?  There is a documentation
  data base which is querried with the ~c[:]~ilc[docs] command.

  The documentation data base is divided into sections.  The sections
  are listed by
  ~bv[]
  ACL2 !>:docs *
  ~ev[]
  Each section has a name, ~c[sect], and by typing
  ~bv[]
  ACL2 !>:docs sect
  ~ev[]
  or equivalently
  ~bv[]
  ACL2 !>:doc sect
  ~ev[]
  you will get an enumeration of the topics within that section.
  Those topics can be further explored by using ~c[:]~ilc[doc] (and ~c[:]~ilc[more]) on
  them.  In fact the section name itself is just a documented name.
  ~c[:]~ilc[more] generally gives an informal overview of the general subject of
  the section.
  ~bv[]
  ACL2 !>:docs **
  ~ev[]
  will list all documented topics, by section.  This fills several
  pages but might be a good place to start.

  If you want documentation on some topic, but none of our names or
  brief descriptions seem to deal with that topic, you can invoke a
  command to search the text in the data base for a given string.
  This is like the GNU Emacs ``~ilc[apropos]'' command.
  ~bv[]
  ACL2 !>:docs \"functional inst\"
  ~ev[]
  will list every documented topic whose ~c[:]~ilc[doc] or ~c[:]~ilc[more-doc] text
  includes the substring ~c[\"functional inst\"], where case and the exact
  number of spaces are irrelevant.

  If you want documentation on an ACL2 function or macro and the
  documentation data base does not contain any entries for it, there
  are still several alternatives.
  ~bv[]
  ACL2 !>:args fn
  ~ev[]
  will print the arguments and some other relevant information about
  the named function or macro.  This information is all gleaned from
  the definition (not from the documentation data base) and hence this
  is a definitive way to determine if ~c[fn] is defined as a function or
  macro.

  You might also want to type:
  ~bv[]
  ACL2 !>:pc fn
  ~ev[]
  which will print the ~il[command] which introduced ~c[fn].  You should
  ~pl[command-descriptor] for details on the kinds of input you
  can give the ~c[:]~ilc[pc] command.

  The entire ACL2 documentation data base is user extensible.  That
  is, if you document your function definitions or theorems, then that
  documentation is made available via the data base and its query
  commands.

  The implementation of our online documentation system makes use of
  Common Lisp's ``documentation strings.'' While Common Lisp permits a
  documentation string to be attached to any defined concept, Common
  Lisp assigns no interpretation to these strings.  ACL2 attaches
  special significance to documentation strings that begin with the
  characters ``~c[:doc-section]''.  When such a documentation string is
  seen, it is stored in the data base and may be displayed via ~c[:]~ilc[doc],
  ~c[:]~ilc[more], ~c[:]~ilc[docs], etc.  Such documentation strings must follow rigid
  syntactic rules to permit their processing by our commands.  These
  are spelled out elsewhere; ~pl[doc-string].

  A description of the structure of the documentation data base may
  also be found; ~pl[doc-string].

  Finally: To build the HTML documentation, proceed with the following sequence
  of steps.
  ~bq[]
  1. In the ~c[doc/] subdirectory of the ACL2 distribution, start ACL2 and then
  evaluate ~c[(certify-book \"write-acl2-html\")].

  2. Exit ACL2 and start it up again (or, evaluate ~c[:]~ilc[u]).

  3. Include the documented ~il[books] within your ACL2 loop using
  ~ilc[include-book].

  4. Evaluate ~c[(include-book \"../doc/write-acl2-html\" :dir :system)].

  5. Call macro ~c[write-html-file], following the instructions at the end of
  distributed file ~c[doc/write-acl2-html.lisp].
  ~eq[]~/")

(deflabel history
  :doc
  ":Doc-Section History

  functions that display or change history~/~/

  ACL2 keeps track of the ~il[command]s that you have executed that have
  extended the logic or the rule data base, as by the definition of
  macros, functions, etc.  Using the facilities in this section you
  can review the sequence of ~il[command]s executed so far.  For example,
  you can ask to see the most recently executed ~il[command], or the
  ~il[command] ~c[10] before that, or the ~il[command] that introduced a given
  function symbol.  You can also undo back through some previous
  ~il[command], restoring the logical ~il[world] to what it was before the given
  ~il[command].

  The annotations printed in the margin in response to some of these
  commands (such as `P', `L', and `V') are explained in the
  documentation for ~c[:]~ilc[pc].

  Several technical terms are used in the documentation of the history
  ~il[command]s.  You must understand these terms to use the ~il[command]s.
  These terms are documented via ~c[:]~ilc[doc] entries of their own.
  ~l[command], ~pl[events], ~pl[command-descriptor], and
  ~pl[logical-name].~/")

#+acl2-loop-only
(defmacro first (x)
  ":Doc-Section ACL2::Programming

  first member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car x))

#+acl2-loop-only
(defmacro second (x)
  ":Doc-Section ACL2::Programming

  second member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cadr x))

#+acl2-loop-only
(defmacro third (x)
  ":Doc-Section ACL2::Programming

  third member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'caddr x))

#+acl2-loop-only
(defmacro fourth (x)
  ":Doc-Section ACL2::Programming

  fourth member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cadddr x))

#+acl2-loop-only
(defmacro fifth (x)
  ":Doc-Section ACL2::Programming

  fifth member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cddddr x)))

#+acl2-loop-only
(defmacro sixth (x)
  ":Doc-Section ACL2::Programming

  sixth member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cadr (list 'cddddr x)))

#+acl2-loop-only
(defmacro seventh (x)
  ":Doc-Section ACL2::Programming

  seventh member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'caddr (list 'cddddr x)))

#+acl2-loop-only
(defmacro eighth (x)
  ":Doc-Section ACL2::Programming

  eighth member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cadddr (list 'cddddr x)))

#+acl2-loop-only
(defmacro ninth (x)
  ":Doc-Section ACL2::Programming

  ninth member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'car (list 'cddddr (list 'cddddr x))))

#+acl2-loop-only
(defmacro tenth (x)
  ":Doc-Section ACL2::Programming

  tenth member of the list~/

  See any Common Lisp documentation for details.~/~/"
  (list 'cadr (list 'cddddr (list 'cddddr x))))

#+acl2-loop-only
(defmacro rest (x)

  ":Doc-Section ACL2::Programming

  rest (~ilc[cdr]) of the list~/

  In the logic, ~c[rest] is just a macro for ~ilc[cdr].~/

  ~c[Rest] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (list 'cdr x))

#+acl2-loop-only
(defun identity (x) (declare (xargs :guard t))

  ":Doc-Section ACL2::Programming

  the identity function~/

  ~c[(Identity x)] equals ~c[x]; what else can we say?~/

  ~c[Identity] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  x)

#+acl2-loop-only
(defun revappend (x y)

  ":Doc-Section ACL2::Programming

  concatentate the ~il[reverse] of one list to another~/

  ~c[(Revappend x y)] ~il[concatenate]s the ~il[reverse] of the list ~c[x] to ~c[y],
  which is also typically a list.~/

  The following theorem characterizes this English description.
  ~bv[]
  (equal (revappend x y)
         (append (reverse x) y))
  ~ev[]
  Hint:  This lemma follows immediately from the definition of ~ilc[reverse]
  and the following lemma.
  ~bv[]
  (defthm revappend-append
    (equal (append (revappend x y) z)
           (revappend x (append y z))))
  ~ev[]

  The ~il[guard] for ~c[(revappend x y)] requires that ~c[x] is a true list.

  ~c[Revappend] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (true-listp x)))
  (if (endp x)
      y
    (revappend (cdr x) (cons (car x) y))))

(defthm character-listp-revappend
  (implies (true-listp x)
           (equal (character-listp (revappend x y))
                  (and (character-listp x)
                       (character-listp y))))

; In some versions of ACL2, the following :induct hint hasn't been necessary.

  :hints (("Goal" :induct (revappend x y))))

#+acl2-loop-only
(defun reverse (x)

  ":Doc-Section ACL2::Programming

  reverse a list or string~/

  ~c[(Reverse x)] is the result of reversing the order of the
  elements of the list or string ~c[x].~/

  The ~il[guard] for ~c[reverse] requires that its argument is a true list
  or a string.

  ~c[Reverse] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (or (true-listp x)
                             (stringp x))))
  (cond ((stringp x)
         (coerce (revappend (coerce x 'list) nil) 'string))
        (t (revappend x nil))))

(defdoc switches-parameters-and-modes
  ":Doc-Section switches-parameters-and-modes

  a variety of ways to modify the ACL2 environment~/

  The beginning user might pay special attention to documentation for
  ~ilc[logic] and ~ilc[program].  Other topics in this section can be read as
  one gains familiarity with ACL2.~/~/")

(defconst *valid-output-names*
  '(error warning warning! observation prove proof-checker event expansion
          summary proof-tree))

(defun set-difference-eq (l1 l2)

  ":Doc-Section ACL2::Programming

  elements of one list that are not elements of another~/

  ~c[(Set-difference-eq x y)] is logically equivalent to
  ~ilc[set-difference-equal], except that the ~il[guard] requires not only that
  the two arguments are ~ilc[true-listp]s but also that at least one is a
  ~ilc[symbol-listp].  Also ~pl[set-difference-equal].~/~/"

  (declare (xargs :guard (and (true-listp l1)
                              (true-listp l2)
                              (or (symbol-listp l1)
                                  (symbol-listp l2)))))
  (cond ((endp l1) nil)
        ((member-eq (car l1) l2)
         (set-difference-eq (cdr l1) l2))
        (t (cons (car l1) (set-difference-eq (cdr l1) l2)))))

#+acl2-loop-only
(defun listp (x)

  ":Doc-Section ACL2::Programming

  recognizer for (not necessarily proper) lists~/

  ~c[(listp x)] is true when ~c[x] is either a ~ilc[cons] pair or is
  ~c[nil].~/

  ~c[Listp] has no ~il[guard], i.e., its ~il[guard] is ~c[t].

  ~c[Listp] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :mode :logic :guard t))
  (or (consp x)
      (equal x nil)))

(defconst *summary-types*
  '(header form rules warnings time value))

(defun with-output-fn (ctx args off on gag-mode off-on-p gag-p stack
                           summary summary-p)
  (declare (xargs :mode :program
                  :guard (true-listp args)))
  (cond
   ((endp args) nil)
   ((keywordp (car args))
    (let ((illegal-value-string
           "~x0 is not a legal value for a call of with-output, but has been ~
            supplied for keyword ~x1.  See :DOC with-output."))
      (cond
       ((consp (cdr args))
        (cond
         ((eq (car args) :gag-mode)
          (cond
           ((member-eq
             (cadr args)
             '(t :goals nil)) ; keep this list in sync with set-gag-mode
            (with-output-fn ctx (cddr args) off on (cadr args) off-on-p t
                           stack summary summary-p))
           (t (illegal ctx
                       illegal-value-string
                       (list (cons #\0 (cadr args))
                             (cons #\1 :gag-mode))))))
         ((eq (car args) :stack)
          (cond
           (stack
            (illegal ctx
                     "The keyword :STACK may only be supplied once in a call ~
                      of ~x0."
                     (list (cons #\0 'with-output))))
           ((member-eq (cadr args) '(:push :pop))
            (with-output-fn ctx (cddr args) off on gag-mode off-on-p gag-p
                           (cadr args) summary summary-p))
           (t (illegal ctx
                       illegal-value-string
                       (list (cons #\0 (cadr args))
                             (cons #\1 :stack))))))
         ((eq (car args) :summary)
          (cond (summary-p
                 (illegal ctx
                          "The keyword :SUMMARY may only be supplied once in ~
                           a call of ~x0."
                          (list (cons #\0 'with-output))))
                ((not (or (eq (cadr args) :all)
                          (and (symbol-listp (cadr args))
                               (subsetp-eq (cadr args) *summary-types*))))
                 (illegal ctx
                          "In a call of ~x0, the value of keyword :SUMMARY ~
                           must either be :ALL or a true-list contained in ~
                           the list ~x1."
                          (list (cons #\0 'with-output)
                                (cons #\1 *summary-types*))))
                (t
                 (with-output-fn ctx (cddr args) off on gag-mode off-on-p gag-p
                                 stack (cadr args) t))))
         ((not (member-eq (car args) '(:on :off)))
          (illegal ctx
                   "~x0 is not a legal keyword for a call of with-output.  ~
                    See :DOC with-output."
                   (list (cons #\0 (car args)))))
         (t (let ((syms (cond ((eq (cadr args) :all)
                               :all)
                              ((symbol-listp (cadr args))
                               (cadr args))
                              ((symbolp (cadr args))
                               (list (cadr args))))))
              (cond (syms
                     (cond ((eq (car args) :on)
                            (and (null on)
                                 (with-output-fn ctx (cddr args) off
                                                 (if (eq syms :all)
                                                     :all
                                                   syms)
                                                 gag-mode t gag-p stack summary
                                                 summary-p)))
                           (t ; (eq (car args) :off)
                            (and (null off)
                                 (with-output-fn ctx (cddr args)
                                                 (if (eq syms :all)
                                                     :all
                                                   syms)
                                                 on gag-mode t gag-p stack
                                                 summary summary-p)))))
                    (t (illegal ctx
                                illegal-value-string
                                (list (cons #\0 (cadr args))
                                      (cons #\1 (car args))))))))))
       (t (illegal ctx
                   "A with-output form has terminated with a keyword, ~x0.  ~
                    This is illegal.  See :DOC with-output."
                   (list (cons #\0 (car args))))))))
   ((cdr args)
    (illegal ctx
             "Illegal with-output form.  See :DOC with-output."
             nil))
   ((not (or (eq off :all)
             (subsetp-eq off *valid-output-names*)))
    (illegal ctx
             "The :off argument to with-output-fn must either be :all or a ~
              subset of the list ~X01, but ~x2 contains ~&3."
             (list (cons #\0 *valid-output-names*)
                   (cons #\1 nil)
                   (cons #\2 off)
                   (cons #\3 (set-difference-eq off *valid-output-names*)))))
   ((not (or (eq on :all)
             (subsetp-eq on *valid-output-names*)))
    (illegal ctx
             "The :on argument to with-output-fn must either be :all or a ~
              subset of the list ~X01, but ~x2 contains ~&3."
             (list (cons #\0 *valid-output-names*)
                   (cons #\1 nil)
                   (cons #\2 on)
                   (cons #\3 (set-difference-eq on *valid-output-names*)))))
   (t
    `(state-global-let*
      (,@
       (and gag-p
            `((gag-mode (f-get-global 'gag-mode state)
                        set-gag-mode-fn)))
       ,@
       (and (or off-on-p
                (eq stack :pop))
            '((inhibit-output-lst (f-get-global 'inhibit-output-lst state))))
       ,@
       (and stack
            '((inhibit-output-lst-stack
               (f-get-global 'inhibit-output-lst-stack state))))
       ,@
       (and summary-p
            `((inhibited-summary-types
               ,(if (eq summary :all)
                    nil
                  (list 'quote
                        (set-difference-eq *summary-types* summary)))))))
      (er-progn
       ,@(and gag-p
              `((pprogn (set-gag-mode ,gag-mode)
                        (value nil))))
       ,@(and stack
              `((pprogn ,(if (eq stack :pop)
                             '(pop-inhibit-output-lst-stack state)
                           '(push-inhibit-output-lst-stack state))
                        (value nil))))
       ,@(and off-on-p
              `((set-inhibit-output-lst
                 ,(cond ((eq on :all)
                         (if (eq off :all)
                             '*valid-output-names*
                           `(quote ,off)))
                        ((eq off :all)
                         `(set-difference-eq *valid-output-names* ',on))
                        (t
                         `(union-eq ',off
                                    (set-difference-eq
                                     (f-get-global 'inhibit-output-lst
                                                   state)
                                     ',on)))))))
       ,(car args))))))

#+acl2-loop-only
(defun last (l)

  ":Doc-Section ACL2::Programming

  the last ~ilc[cons] (not element) of a list~/

  ~c[(Last l)] is the last ~ilc[cons] of a list.  Here are examples.
  ~bv[]
  ACL2 !>(last '(a b . c))
  (B . C)
  ACL2 !>(last '(a b c))
  (C)
  ~ev[]
  ~/

  ~c[(Last l)] has a ~il[guard] of ~c[(listp l)]; thus, ~c[l] need not be a
  ~ilc[true-listp].

  ~c[Last] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  Unlike Common Lisp, we do not
  allow an optional second argument for ~c[last].~/"

  (declare (xargs :guard (listp l)))
  (if (atom (cdr l))
      l
    (last (cdr l))))

(defun first-n-ac (i l ac)
  (declare (type (integer 0 *) i)
           (xargs :guard (and (true-listp l)
                              (true-listp ac))))
  (cond ((zp i)
         (reverse ac))
        (t (first-n-ac (1- i) (cdr l) (cons (car l) ac)))))

(defun take (n l)

  ":Doc-Section ACL2::Programming

  initial segment of a list~/

  For any natural number ~c[n] not exceeding the length of ~c[l],
  ~c[(take n l)] collects the first ~c[n] elements of the list ~c[l].~/

  The following is a theorem (though it takes some effort, including
  lemmas, to get ACL2 to prove it):
  ~bv[]
  (equal (length (take n l)) (nfix n))
  ~ev[]
  If ~c[n] is is an integer greater than the length of ~c[l], then
  ~c[take] pads the list with the appropriate number of ~c[nil]
  elements.  Thus, the following is also a theorem.
  ~bv[]
  (implies (and (integerp n)
                (true-listp l)
                (<= (length l) n))
           (equal (take n l)
                  (append l (make-list (- n (length l))))))
  ~ev[]
  For related functions, ~pl[nthcdr] and ~pl[butlast].

  The ~il[guard] for ~c[(take n l)] is that ~c[n] is a nonnegative integer
  and ~c[l] is a true list.~/"

  (declare (xargs :guard
                   (and (integerp n)
                        (not (< n 0))
                        (true-listp l))))
  #-acl2-loop-only
  (when (<= n most-positive-fixnum)
    (return-from take
                 (loop for i fixnum from 1 to n
                       as x in l
                       collect x)))
  (first-n-ac n l nil))

#+acl2-loop-only
(defun butlast (lst n)

  ":Doc-Section ACL2::Programming

  all but a final segment of a list~/

  ~c[(Butlast l n)] is the list obtained by removing the last ~c[n]
  elements from the true list ~c[l].  The following is a theorem
  (though it takes some effort, including lemmas, to get ACL2 to prove
  it).
  ~bv[]
  (implies (and (integerp n)
                (<= 0 n)
                (true-listp l))
           (equal (length (butlast l n))
                  (if (< n (length l))
                      (- (length l) n)
                    0)))
  ~ev[]
  For related functions, ~pl[take] and ~pl[nthcdr].~/

  The ~il[guard] for ~c[(butlast l n)] requires that ~c[n] is a nonnegative
  integer and ~c[lst] is a true list.

  ~c[Butlast] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  Note:  In Common Lisp the
  second argument of ~c[butlast] is optional, but in ACL2 it is
  required.~/"

  (declare (xargs :guard (and (true-listp lst)
                              (integerp n)
                              (<= 0 n))))
  (let ((lng (len lst)))
    (if (<= lng n)
        nil
        (take (- lng n) lst))))

#-acl2-loop-only
(defmacro with-output (&rest args)
  (car (last args)))

#+acl2-loop-only
(defmacro with-output (&rest args)

  ":Doc-Section switches-parameters-and-modes

  suppressing or turning on specified output for an event~/
  ~bv[]
  Examples:

  ; Turn off all output during evaluation of the indicated thm form.
  (with-output
   :off :all 
   :gag-mode nil
   (thm (equal (app (app x y) z) (app x (app y z)))))

  ; Prove the indicated theorem with the event summary turned off and
  ; using the :goals setting for gag-mode.
  (with-output
     :off summary
     :gag-mode :goals
     (defthm app-assoc (equal (app (app x y) z) (app x (app y z)))))

  ; Same effect as just above:
  (with-output
     :on summary
     :summary :all ; equivalently, (header form rules warnings time)
     :gag-mode :goals
     (defthm app-assoc (equal (app (app x y) z) (app x (app y z)))))

  ; Same as just above, but turn off only the indicated parts of the summary.
  (with-output
     :on summary
     :summary (time rules)
     :gag-mode :goals  ; use gag-mode, with goal names printed
     (defthm app-assoc (equal (app (app x y) z) (app x (app y z)))))

  ; Same as specifying :off :all, but showing all output types:
  (with-output
   :off (error warning warning! observation prove proof-checker event expansion
               summary proof-tree)
   :gag-mode nil
   (thm (equal (app (app x y) z) (app x (app y z)))))

  ; Same as above, but :stack :push says to save the current
  ; inhibit-output-lst, which can be restored in a subsidiary with-output call
  ; that specifies :stack :pop.
  (with-output
   :stack :push
   :off :all
   :gag-mode nil
   (thm (equal (app (app x y) z) (app x (app y z)))))~/

  General Form:
  (with-output :key1 val1 ... :keyk valk form)
  ~ev[]
  where each ~c[:keyi] is either ~c[:off], ~c[:on], ~c[:stack],
  ~c[:summary], or ~c[:gag-mode], and ~c[vali] is as follows.  If ~c[:keyi]
  is ~c[:off] or ~c[:on], then ~c[vali] can be ~c[:all], and otherwise is a
  symbol or non-empty list of symbols representing output types that can be
  inhibited; ~pl[set-inhibit-output-lst].  If ~c[:keyi] is ~c[:gag-mode], then
  ~c[vali] is one of the legal values for ~c[:]~ilc[set-gag-mode].
  If ~c[:keyi] is ~c[:summary], then ~c[vali] is either ~c[:all] or a true-list
  of symbols each of which belongs to the list ~c[*summary-types*], i.e., is
  one of ~c[header], ~c[form], ~c[rules], ~c[warnings], ~c[time], or ~c[value].
  Otherwise ~c[:keyi] is ~c[:stack], in which case ~c[:vali] is ~c[:push] or
  ~c[:pop]; for now assume that ~c[:stack] is not specified (we'll return to it
  below).  The result of evaluating the General Form above is to evaluate
  ~c[form], but in an environment where output occurs as follows.  If
  ~c[:on :all] is specified, then every output type is turned on except as
  inhibited by ~c[:off]; else if ~c[:off :all] is specified, then every output
  type is inhibited except as specified by ~c[:on]; and otherwise, the
  currently-inhibited output types are reduced as specified by ~c[:on] and then
  extended as specified by ~c[:off].  But if ~c[:gag-mode] is specified, then
  before modifying how output is inhibited, ~ilc[gag-mode] is set for the
  evaluation of ~c[form] as specified by the value of ~c[:gag-mode];
  ~pl[set-gag-mode].  If ~c[summary] is among the output types that are turned
  on (not inhibited), then if ~c[:summary] is specified, the only parts of the
  summary to be printed will be those specified by the value of ~c[:summary].
  The correspondence should be clear, except perhaps that ~c[header] refers to
  the line containing only the word ~c[Summary], and ~c[value] refers to the
  value of the form printed during evaluation of sequences of events as for
  ~ilc[progn] and ~ilc[encapsulate].

  Note that the handling of the ~c[:stack] argument pays no attention to the
  ~c[:summary] argument.

  Note: When the scope of ~c[with-output] is exited, then all modifications are
  undone, reverting ~c[gag-mode] and the state of output inhibition to those
  which were present before the ~c[with-output] call was entered.

  The ~c[:stack] keyword's effect is illustrated by the following example,
  where ``~c[(encapsulate nil)]'' may replaced by ``~c[(progn]'' without any
  change to the output that is printed.
  ~bv[]
  (with-output
   :stack :push :off :all
   (encapsulate ()
     (defun f1 (x) x)
     (with-output :stack :pop (defun f2 (x) x))
     (defun f3 (x) x)
     (with-output :stack :pop :off warning (in-theory nil))
     (defun f4 (x) x)))
  ~ev[]
  The outer ~c[with-output] call saves the current output settings (as may
  have been modified by earlier calls of ~ilc[set-inhibit-output-lst]), by
  pushing them onto a stack, and then turns off all output.  Each inner
  ~c[with-output] call temporarily pops that stack, restoring the starting
  output settings, until it completes and undoes the effects of that pop.
  Unless ~c[event] output was inhibited at the top level
  (~pl[set-inhibit-output-lst]), the following output is shown:
  ~bv[]
  Since F2 is non-recursive, its admission is trivial.  We observe that
  the type of F2 is described by the theorem (EQUAL (F2 X) X).  
  ~ev[]
  And then, if ~c[summary] output was not inhibited at the top level, we get
  the rest of this output:
  ~bv[]
  Summary
  Form:  ( DEFUN F2 ...)
  Rules: NIL
  Warnings:  None
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)

  Summary
  Form:  ( IN-THEORY NIL)
  Rules: NIL
  Warnings:  None
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)
  ~ev[]
  Note that the use of ~c[:off warning] supresses a ~c[\"Theory\"] warning for
  the ~c[(in-theory nil)] event, and that in no case will output be printed for
  definitions of ~c[f1], ~c[f3], or ~c[f4], or for the ~ilc[encapsulate] event
  itself.

  The following more detailed explanation of ~c[:stack] is intended only for
  advanced users.  After ~c[:gag-mode] is handled (if present) but before
  ~c[:on] or ~c[:off] is handled, the value of ~c[:stack] is handled as
  follows.  If the value is ~c[:push], then ~il[state] global
  ~c[inhibit-output-lst-stack] is modified by pushing the value of ~il[state]
  global ~c[inhibit-output-lst] onto the value of ~il[state] global
  ~c[inhibit-output-lst-stack], which is ~c[nil] at the top level.  If the
  value is ~c[:pop], then ~il[state] global ~c[inhibit-output-lst-stack] is
  modified only if non-~c[nil], in which case its top element is popped and
  becomes the value of of ~il[state] global ~c[inhibit-output-lst].

  Warning: ~c[With-output] has no effect in raw Lisp, and hence is disallowed
  in function bodies.  However, you can probably get the effect you want as
  illustrated below, where ~c[<form>] must return an error triple
  ~c[(mv erp val state)]; ~pl[ld].
  ~bv[]
  Examples avoiding with-output, for use in function definitions:

  ; Inhibit all output:
  (state-global-let*
   ((inhibit-output-lst *valid-output-names*))
   <form>)

  ; Inhibit all warning output:
  (state-global-let*
   ((inhibit-output-lst
     (union-eq (f-get-global 'inhibit-output-lst state)
               '(warning warning!))))
   <form>)
  ~ev[]

  Note that ~c[with-output] is allowed in books.  ~l[embedded-event-form]."

  (let ((val
         (with-output-fn 'with-output args nil nil nil nil nil nil nil nil)))
    (or val
        (illegal 'with-output
                 "Macroexpansion of ~q0 failed."
                 (list (cons #\0 (cons 'with-output args)))))))

; Mutual Recursion

; We are about to need mutual recursion for the first time in axioms.lisp.
; We now define the mutual-recursion macro for the logic.

#-acl2-loop-only
(defun-one-output strip-cdrs1 (x a)

; WARNING: THIS PROGRAM IS DESTRUCTIVE.  DO NOT USE THIS PROGRAM
; UNLESS YOU KNOW WHAT YOU'RE DOING!  IT SMASHES THE CONSES IN THE
; SECOND ARGUMENT!

  (cond ((endp x) (nreverse a))
        (t (strip-cdrs1 (cdr x) (cons (cdr (car x)) a)))))

(defun strip-cdrs (x)

  ":Doc-Section ACL2::Programming

  collect up all second components of pairs in a list~/

  ~c[(strip-cdrs x)] has a ~il[guard] of ~c[(alistp x)], and returns the list
  obtained by walking through the list ~c[x] and collecting up all
  second components (~ilc[cdr]s).  This function is implemented in a
  tail-recursive way, despite its logical definition.~/~/"

  (declare (xargs :guard (alistp x)))

; See the Essay on Strip-cars -- To Tail Recur or not to Tail Recur?
; above.

  #-acl2-loop-only
  (strip-cdrs1 x nil)
  #+acl2-loop-only
  (cond ((endp x) nil)
        (t (cons (cdr (car x))
                 (strip-cdrs (cdr x))))))

(defun mutual-recursion-guardp (rst)
  (declare (xargs :guard t))
  (cond ((atom rst) (equal rst nil))
        (t (and (consp (car rst))
                (true-listp (car rst))
                (true-listp (caddr (car rst))) ; formals
                (member-eq (car (car rst)) '(defun defund defun-nx defund-nx))
                (mutual-recursion-guardp (cdr rst))))))

(defun collect-cadrs-when-car-eq (x alist)
  (declare (xargs :guard (assoc-eq-equal-alistp alist)))
  (cond ((endp alist) nil)
        ((eq x (car (car alist)))
         (cons (cadr (car alist))
               (collect-cadrs-when-car-eq x (cdr alist))))
        (t (collect-cadrs-when-car-eq x (cdr alist)))))

(defmacro value (x)
  `(mv nil ,x state))

(defun value-triple-fn (form on-skip-proofs check)
  (declare (xargs :guard t))
  `(cond ((and ,(not on-skip-proofs)
               (f-get-global 'ld-skip-proofsp state))
          (value :skipped))
         (t ,(let ((form
                    `(let ((check ,check))
                       (cond (check
                              (cond
                               ((check-vars-not-free
                                 (check)
                                 ,form)
                                :passed)
                               ((tilde-@p check)
                                (er hard 'value-triple
                                    "Assertion failed:~%~@0~|"
                                    check))
                               (t
                                (er hard 'value-triple
                                    "Assertion failed on form:~%~x0~|"
                                    ',form))))
                             (t ,form)))))
               `(state-global-let*
                 ((safe-mode t))
                 (value ,form))))))

#+acl2-loop-only
(defmacro value-triple (form &key on-skip-proofs check)

  ":Doc-Section Events

  compute a value, optionally checking that it is not ~c[nil]~/
  ~bv[]
  Examples:
  (value-triple (+ 3 4))
  (value-triple (cw \"hi\") :on-skip-proofs t)
  (value-triple (@ ld-pre-eval-print))
  (value-triple (@ ld-pre-eval-print) :check t)~/

  General Form:
  (value-triple form
                :on-skip-proofs sp ; optional; nil by default
                :check chk         ; optional; nil by default
                )
  ~ev[]

  ~c[Value-triple] provides a convenient way to evaluate a form in an event
  context, including ~ilc[progn] and ~ilc[encapsulate] and in ~il[books];
  ~pl[events].  The form should evaluate to a single, non-~il[stobj] value.

  Calls of ~c[value-triple] are generally skipped when proofs are being
  skipped, in particular when ACL2 is performing the second pass through the
  ~il[events] of an ~ilc[encapsulate] form or during an ~ilc[include-book], or
  indeed any time ~ilc[ld-skip-proofsp] is non-~c[nil].  If you want the call
  evaluated during those times as well, use a non-~c[nil] value for
  ~c[:on-skip-proofs].  Note that the argument to ~c[:on-skip-proofs] is not
  evaluated.

  If you expect the form to evaluate to a non-~c[nil] value and you want an
  error to occur when that is not the case, you can use ~c[:check t].  More
  generally, the argument of ~c[:check] can be a form that evaluates to a
  single, non-~il[stobj] value.  If this value is not ~c[nil], then the
  aforementioned test is made (that the given form is not ~c[nil]).  If an
  error occurs and the value of ~c[:check] is a string or indeed any
  ``message'' suitable for printing by ~ilc[fmt] when supplied as a value for
  tilde-directive ~c[~~@], then that string or message is printed."

  (value-triple-fn form on-skip-proofs check))

(defmacro assert-event (form &key on-skip-proofs msg)

  ":Doc-Section Events

  assert that a given form returns a non-~c[nil] value~/
  ~bv[]
  Examples:
  (assert-event (equal (+ 3 4) 7))
  (assert-event (equal (+ 3 4) 7) :msg (msg \"Error: ~~x0\" 'equal-check))
  (assert-event (equal (+ 3 4) 7) :on-skip-proofs t)~/

  General Forms:
  (assert-event form)
  (assert-event form :on-skip-proofs t)
  ~ev[]

  ~c[Assert-event] takes a ground form, i.e., one with no free variables;
  ~ilc[stobj]s are allowed but only a single non-~ilc[stobj] value can be
  returned.  The form is then evaluated and if the result is ~c[nil], then a
  so-called hard error (~pl[er]) results.  This evaluation is however not done
  if proofs are being skipped, as during ~ilc[include-book] (also
  ~pl[skip-proofs] and ~pl[ld-skip-proofsp]), unless ~c[:on-skip-proofs t] is
  supplied.

  Normally, if an ~c[assert-event] call fails then a generic failure message is
  printed, showing the offending form.  However, if keyword argument ~c[:msg]
  is supplied, then the failure message is printed as with ~ilc[fmt] argument
  ~c[~~@0]; ~pl[fmt].  In particular, ~c[:msg] is typically a string or a call
  ~c[(msg str arg-0 arg-1 ... arg-k)], where ~c[str] is a string and each
  ~c[arg-i] is the value to be associated with ~c[#\i] upon formatted printing
  (as with ~ilc[fmt]) of the string ~c[str].

  This form may be put into a book to be certified (~pl[books]), because
  ~c[assert-event] is a macro whose calls expand to calls of ~c[value-triple]
  (~pl[embedded-event-form]).  When certifying a book, guard-checking is off,
  as though ~c[(set-guard-checking nil)] has been evaluated;
  ~pl[set-guard-checking].  That, together with a ``safe mode,'' guarantees
  that ~c[assert-event] forms are evaluated in the logic without guard
  violations while certifying a book.~/"

  (declare (xargs :guard (booleanp on-skip-proofs)))
  `(value-triple ,form
                 :on-skip-proofs ,on-skip-proofs
                 :check ,(or msg t)))

(defun xd-name (event-type name)
  (declare (xargs :guard (member-eq event-type '(defund defthmd))))
  (cond
   ((eq event-type 'defund)
    (list :defund  name))
   ((eq event-type 'defthmd)
    (list :defthmd name))
   (t (illegal 'xd-name
               "Unexpected event-type for xd-name, ~x0"
               (list (cons #\0 event-type))))))

(defun defund-name-list (defuns acc)
  (declare (xargs :guard (and (mutual-recursion-guardp defuns)
                              (true-listp acc))))
  (cond ((endp defuns) (reverse acc))
        (t (defund-name-list
             (cdr defuns)
             (cons (if (eq (caar defuns) 'defund)
                       (xd-name 'defund (cadar defuns))
                     (cadar defuns))
                   acc)))))

; Begin support for defun-nx.

(defun throw-nonexec-error (fn actuals)
  (declare (xargs :guard t
                  :verify-guards nil)
           #+acl2-loop-only
           (ignore fn actuals))
  #-acl2-loop-only
  (progn

; Keep the following in sync with null-body-er.  The error message printed will
; be a bit ugly since we don't do the trick done with null-body-er+ -- we tried
; that and it didn't work when tracing because some conversion had already been
; done, so we'll leave well enough alone as we don't really expect to call this
; functions anyhow.  After all, this extra attention to causing an error
; (actually, a throw) is merely to prevent the case where include-book loads
; compiled code to overwrite a defun with :non-executable t -- that's why we
; introduced defun-nx and insisted that non-executable functions have a call of
; throw-nonexec-error.

    (throw-raw-ev-fncall
     (list* 'ev-fncall-null-body-er fn actuals))
; Just in case throw-raw-ev-fncall doesn't throw -- though it always should.
    (error "This error is caused by what should be dead code!"))
  #+acl2-loop-only
  nil)

(defun defun-nx-fn (form disabledp)
  (declare (xargs :guard (and (true-listp form)
                              (true-listp (caddr form)))
                  :verify-guards nil))
  (let ((name (cadr form))
        (formals (caddr form))
        (rest (cdddr form))
        (defunx (if disabledp 'defund 'defun)))
    `(,defunx ,name ,formals
       (declare (xargs :non-executable t :mode :logic))
       ,@(butlast rest 1)
       (prog2$ (throw-nonexec-error ',name (list ,@formals))
               ,@(last rest)))))

(defmacro defun-nx (&whole form &rest rest)

  ":Doc-Section acl2::Events

  define a non-executable function symbol~/~/

  The macro ~c[defun-nx] introduces definitions using the ~ilc[defun] macro,
  always in ~c[:]~ilc[logic] mode, such that the calls of the resulting
  function cannot be evaluated.  Such a definition is admitted without the
  syntactic restrictions usually imposed on definitions, as opposed to
  theorems, in particular regarding function signatures and the use of
  single-threaded object names, even though such functions are permitted to
  declare names to be ~c[:]~ilc[stobj]s.

  The syntax is identical to that of ~ilc[defun].  A form
  ~bv[]
  (defun-nx name (x1 ... xk) ... body)
  ~ev[]
  expands to the following form.
  ~bv[]
  (defun name (x1 ... xk)
    (declare (xargs :non-executable t :mode :logic))
    ...
    (prog2$ (throw-nonexec-error 'name (list x1 ... xk))
            body))
  ~ev[]
  Note that because of the insertion of the above call of
  ~c[throw-nonexec-error], no formal is ignored when using ~c[defun-nx].

  If you prefer to avoid the use of ~c[defun-nx] for non-executable function
  definitions in ~c[:]~ilc[logic] mode, you can use an ~ilc[xargs]
  ~c[:non-executable t] ~ilc[declare] form, provided the body has the form
  ~c[(prog2$ (throw-nonexec-error ...) ...)].  The function
  ~c[throw-nonexec-error] is guaranteed to cause an error, though the
  ~c[:non-executable] keyword will lay down code such that an error generally
  occurs when calling ~c[name] without calling ~c[throw-nonexec-error].

  During proofs, the error is silent; it is ``caught'' by the proof mechanism
  and generally results in the introduction of a call of ~ilc[hide] during a
  proof.  If an error message is produced by evaluating a call of the function
  on a list of arguments that includes ~c[state] or user-defined ~ilc[stobj]s,
  these arguments will be shown as symbols such as ~c[|<state>|] in the error
  message.

  It is harmless to include ~c[:non-executable t] in your own ~ilc[xargs]
  ~ilc[declare] form; ~c[defun-nx] will still lay down its own such
  declaration, but ACL2 can tolerate the duplication.

  Note that ~c[defund-nx] is also available.  It has an effect identitcal to
  that of ~c[defun-nx] except that as with ~ilc[defund], it leaves the function
  disabled.

  ~l[defun] for documentation of ~c[defun]."

  (declare (xargs :guard (and (true-listp form)
                              (true-listp (caddr form))))
           (ignore rest))
  (defun-nx-fn form nil))

(defmacro defund-nx (&whole form &rest rest)
  (declare (xargs :guard (and (true-listp form)
                              (true-listp (caddr form))))
           (ignore rest))
  (defun-nx-fn form t))

(defun update-mutual-recursion-for-defun-nx-1 (defs)
  (declare (xargs :guard (mutual-recursion-guardp defs)
                  :verify-guards nil))
  (cond ((endp defs)
         nil)
        ((eq (caar defs) 'defun-nx)
         (cons (defun-nx-fn (car defs) nil)
               (update-mutual-recursion-for-defun-nx-1 (cdr defs))))
        ((eq (caar defs) 'defund-nx)
         (cons (defun-nx-fn (car defs) t)
               (update-mutual-recursion-for-defun-nx-1 (cdr defs))))
        (t
         (cons (car defs)
               (update-mutual-recursion-for-defun-nx-1 (cdr defs))))))

(defun update-mutual-recursion-for-defun-nx (defs)
  (declare (xargs :guard (mutual-recursion-guardp defs)
                  :verify-guards nil))
  (cond ((or (assoc-eq 'defun-nx defs)
             (assoc-eq 'defund-nx defs))
         (update-mutual-recursion-for-defun-nx-1 defs))
        (t defs)))

#+acl2-loop-only
(defmacro mutual-recursion (&whole event-form &rest rst)

  ":Doc-Section Events

  define some mutually recursive functions~/
  ~bv[]
  Example:
  (mutual-recursion
   (defun evenlp (x)
     (if (consp x) (oddlp (cdr x)) t))
   (defun oddlp (x)
     (if (consp x) (evenlp (cdr x)) nil)))~/

  General Form:
  (mutual-recursion def1 ... defn)
  where each defi is a ~ilc[defun] form or a ~ilc[defund] form.
  ~ev[]
  When mutually recursive functions are introduced it is necessary
  to do the termination analysis on the entire clique of definitions.
  Each ~ilc[defun] form specifies its own measure, either with the ~c[:measure]
  keyword ~c[xarg] (~pl[xargs]) or by default to ~ilc[acl2-count].  When a
  function in the clique calls a function in the clique, the measure
  of the callee's actuals must be smaller than the measure of the
  caller's formals ~-[] just as in the case of a simply recursive
  function.  But with mutual recursion, the callee's actuals are
  measured as specified by the callee's ~ilc[defun] while the caller's
  formals are measured as specified by the caller's ~ilc[defun].  These two
  measures may be different but must be comparable in the sense that
  ~ilc[o<] decreases through calls.

  If you want to specify ~c[:]~ilc[hints] or ~c[:guard-hints] (~pl[xargs]), you
  can put them in the ~ilc[xargs] declaration of any of the ~ilc[defun] forms,
  as the ~c[:]~ilc[hints] from each form will be appended together, as will the
  ~ilc[guard-hints] from each form.

  You may find it helpful to use a lexicographic order, the idea being to have
  a measure that returns a list of two arguments, where the first takes
  priority over the second.  Here is an example.
  ~bv[]
  (include-book \"ordinals/lexicographic-ordering\" :dir :system)

  (encapsulate
   ()
   (set-well-founded-relation l<) ; will be treated as LOCAL

   (mutual-recursion
    (defun foo (x)
      (declare (xargs :measure (list (acl2-count x) 1)))
      (bar x))
    (defun bar (y)
      (declare (xargs :measure (list (acl2-count y) 0)))
      (if (zp y) y (foo (1- y))))))
  ~ev[]

  The ~ilc[guard] analysis must also be done for all of the functions at
  the same time.  If any one of the ~ilc[defun]s specifies the
  ~c[:]~ilc[verify-guards] ~c[xarg] to be ~c[nil], then ~il[guard] verification is
  omitted for all of the functions.

  Technical Note: Each ~c[defi] above must be of the form ~c[(defun ...)].  In
  particular, it is not permitted for a ~c[defi] to be a form that will
  macroexpand into a ~ilc[defun] form.  This is because ~c[mutual-recursion] is
  itself a macro, and since macroexpansion occurs from the outside in,
  at the time ~c[(mutual-recursion def1 ... defk)] is expanded the ~c[defi]
  have not yet been.  But ~c[mutual-recursion] must decompose the ~c[defi].
  We therefore insist that they be explicitly presented as ~ilc[defun]s or
  ~ilc[defund]s (or a mixture of these).

  Suppose you have defined your own ~ilc[defun]-like macro and wish to use
  it in a ~c[mutual-recursion] expression.  Well, you can't.  (!)  But you
  can define your own version of ~c[mutual-recursion] that allows your
  ~ilc[defun]-like form.  Here is an example.  Suppose you define
  ~bv[]
  (defmacro my-defun (&rest args) (my-defun-fn args))
  ~ev[]
  where ~c[my-defun-fn] takes the arguments of the ~c[my-defun] form and
  produces from them a ~ilc[defun] form.  As noted above, you are not
  allowed to write ~c[(mutual-recursion (my-defun ...) ...)].  But you can
  define the macro ~c[my-mutual-recursion] so that
  ~bv[]
  (my-mutual-recursion (my-defun ...) ... (my-defun ...))
  ~ev[]
  expands into ~c[(mutual-recursion (defun ...) ... (defun ...))] by
  applying ~c[my-defun-fn] to each of the arguments of
  ~c[my-mutual-recursion].
  ~bv[]
  (defun my-mutual-recursion-fn (lst)
    (declare (xargs :guard (alistp lst)))

  ; Each element of lst must be a consp (whose car, we assume, is always
  ; MY-DEFUN).  We apply my-defun-fn to the arguments of each element and
  ; collect the resulting list of DEFUNs.

    (cond ((atom lst) nil)
          (t (cons (my-defun-fn (cdr (car lst)))
                   (my-mutual-recursion-fn (cdr lst))))))

  (defmacro my-mutual-recursion (&rest lst)

  ; Each element of lst must be a consp (whose car, we assume, is always
  ; MY-DEFUN).  We obtain the DEFUN corresponding to each and list them
  ; all inside a MUTUAL-RECURSION form.

    (declare (xargs :guard (alistp lst)))
    (cons 'mutual-recursion (my-mutual-recursion-fn lst))).
  ~ev[]~/

  :cited-by Programming"

  (declare (xargs :guard (mutual-recursion-guardp rst)))
  (let ((rst (update-mutual-recursion-for-defun-nx rst)))
    (let ((form (list 'defuns-fn
                      (list 'quote (strip-cdrs rst))
                      'state
                      (list 'quote event-form)
                      #+:non-standard-analysis ; std-p
                      nil)))
      (cond
       ((assoc-eq 'defund rst)
        (list 'er-progn
              form
              (list
               'with-output
               :off 'summary
               (list 'in-theory
                     (cons 'disable
                           (collect-cadrs-when-car-eq 'defund rst))))
              (list 'value-triple (list 'quote (defund-name-list rst nil)))))
       (t
        form)))))

; Now we define the weak notion of term that guards metafunctions.

(mutual-recursion

(defun pseudo-termp (x)

  ":Doc-Section Miscellaneous

  a predicate for recognizing term-like s-expressions~/
  ~bv[]
  Example Forms:
  (pseudo-termp '(car (cons x 'nil)))      ; has value t
  (pseudo-termp '(car x y z))              ; also has value t!
  (pseudo-termp '(delta (h x)))            ; has value t
  (pseudo-termp '(delta (h x) . 7))        ; has value nil (not a true-listp)
  (pseudo-termp '((lambda (x) (car x)) b)) ; has value t
  (pseudo-termp '(if x y 123))             ; has value nil (123 is not quoted)
  (pseudo-termp '(if x y '123))            ; has value t
  ~ev[]
  If ~c[x] is the quotation of a term, then ~c[(pseudo-termp x)] is ~c[t].
  However, if ~c[x] is not the quotation of a term it is not necessarily
  the case that ~c[(pseudo-termp x)] is ~c[nil].~/

  ~l[term] for a discussion of the various meanings of the word
  ``term'' in ACL2.  In its most strict sense, a term is either a
  legal variable symbol, a quoted constant, or the application of an
  ~c[n]-ary function symbol or closed ~c[lambda]-expression to ~c[n] terms.  By
  ``legal variable symbol'' we exclude constant symbols, such as ~c[t],
  ~c[nil], and ~c[*ts-rational*].  By ``quoted constants'' we include ~c['t] (aka
  ~c[(quote t)]), ~c['nil], ~c['31], etc., and exclude constant names such as ~c[t],
  ~c[nil] and ~c[*ts-rational*], unquoted constants such as ~c[31] or ~c[1/2], and
  ill-formed ~c[quote] expressions such as ~c[(quote 3 4)].  By ``closed
  lambda expression'' we exclude expressions, such as
  ~c[(lambda (x) (cons x y))], containing free variables in their bodies.
  Terms typed by the user are translated into strict terms for
  internal use in ACL2.

  The predicate ~c[termp] checks this strict sense of ``term'' with
  respect to a given ACL2 logical world; ~l[world].  Many ACL2
  functions, such as the rewriter, require certain of their arguments
  to satisfy ~c[termp].  However, as of this writing, ~c[termp] is in ~c[:]~ilc[program]
  mode and thus cannot be used effectively in conjectures to be
  proved.  Furthermore, if regarded simply from the perspective of an
  effective ~il[guard] for a term-processing function, ~c[termp] checks many
  irrelevant things.  (Does it really matter that the variable symbols
  encountered never start and end with an asterisk?)  For these
  reasons, we have introduced the notion of a ``pseudo-term'' and
  embodied it in the predicate ~c[pseudo-termp], which is easier to
  check, does not require the logical ~il[world] as input, has ~c[:]~ilc[logic]
  mode, and is often perfectly suitable as a ~il[guard] on term-processing
  functions.

  A ~c[pseudo-termp] is either a symbol, a true list of length 2
  beginning with the word ~c[quote], the application of an ~c[n]-ary
  pseudo-~c[lambda] expression to a true list of ~c[n] pseudo-terms, or
  the application of a symbol to a true list of ~c[n] ~c[pseudo-termp]s.
  By an ``~c[n]-ary pseudo-~c[lambda] expression'' we mean an expression
  of the form ~c[(lambda (v1 ... vn) pterm)], where the ~c[vi] are
  symbols (but not necessarily distinct legal variable symbols) and
  ~c[pterm] is a ~c[pseudo-termp].

  Metafunctions may use ~c[pseudo-termp] as a ~il[guard]."

  (declare (xargs :guard t :mode :logic))
  (cond ((atom x) (symbolp x))
        ((eq (car x) 'quote)
         (and (consp (cdr x))
              (null (cdr (cdr x)))))
        ((not (true-listp x)) nil)
        ((not (pseudo-term-listp (cdr x))) nil)
        (t (or (symbolp (car x))

; For most function applications we do not check that the number of
; arguments matches the number of formals.  However, for lambda
; applications we do make that check.  The reason is that the
; constraint on an evaluator dealing with lambda applications must use
; pairlis$ to pair the formals with the actuals and pairlis$ insists on
; the checks below.

               (and (true-listp (car x))
                    (equal (length (car x)) 3)
                    (eq (car (car x)) 'lambda)
                    (symbol-listp (cadr (car x)))
                    (pseudo-termp (caddr (car x)))
                    (equal (length (cadr (car x)))
                           (length (cdr x))))))))

(defun pseudo-term-listp (lst)
  (declare (xargs :guard t))
  (cond ((atom lst) (equal lst nil))
        (t (and (pseudo-termp (car lst))
                (pseudo-term-listp (cdr lst))))))

)

(defthm pseudo-term-listp-forward-to-true-listp
  (implies (pseudo-term-listp x)
           (true-listp x))
  :rule-classes :forward-chaining)

; For the encapsulate of too-many-ifs-post-rewrite-wrapper
(encapsulate
 ()
 (table acl2-defaults-table :defun-mode :logic)
 (verify-guards pseudo-termp))

(defun pseudo-term-list-listp (l)
  (declare (xargs :guard t))
  (if (atom l)
      (equal l nil)
    (and (pseudo-term-listp (car l))
         (pseudo-term-list-listp (cdr l)))))

(verify-guards pseudo-term-list-listp)

(defun add-to-set-eq (x lst)

  ":Doc-Section ACL2::Programming

  add a symbol to a list~/

  For a symbol ~c[x] and a true list ~c[lst], ~c[(add-to-set-eq x lst)]
  is the result of ~ilc[cons]ing ~c[x] on to the front of ~c[lst], unless
  ~c[x] is already a member of ~c[lst], in which case it equals ~c[lst].~/

  ~c[(add-to-set-eq x lst)] has a ~il[guard] that ~c[lst] is a true list and
  moreover, either ~c[x] is a symbol or ~c[lst] is a list of symbols.~/"

  (declare (xargs :guard (if (symbolp x)
                             (true-listp lst)
                           (symbol-listp lst))))
  (cond ((member-eq x lst) lst)
        (t (cons x lst))))

(defun add-to-set-eql (x lst)

  ":Doc-Section ACL2::Programming

  add an object to a list~/

  For an object ~c[x] and a true list ~c[lst], ~c[(add-to-set-eql x lst)]
  is the result of ~ilc[cons]ing ~c[x] on to the front of ~c[lst], unless
  ~c[x] is already a member of ~c[lst], in which case it equals ~c[lst].~/

  ~c[(add-to-set-eql x lst)] has a ~il[guard] that ~c[lst] is a true list and
  moreover, either ~c[x] is ~ilc[eqlablep] or ~c[lst] is an ~ilc[eqlable-listp].~/"

  (declare (xargs :guard (if (eqlablep x)
                             (true-listp lst)
                           (eqlable-listp lst))))
  (cond ((member x lst) lst)
        (t (cons x lst))))

(defmacro variablep (x) (list 'atom x))

(defmacro nvariablep (x) (list 'consp x))

(defmacro fquotep (x) (list 'eq ''quote (list 'car x)))

(defun quotep (x)
  (declare (xargs :guard t))
  (and (consp x)
       (eq (car x) 'quote)))

(defun kwote (x)

  ":Doc-Section ACL2::Programming

  quote an arbitrary object~/

  For any object ~c[x], ~c[(kwote x)] returns the two-element list whose
  elements are the symbol ~c[quote] and the given ~c[x], respectively.
  The guard of ~c[(kwote x)] is ~c[t].~/~/"

  (declare (xargs :guard t))
  (list 'quote x))

(defun kwote-lst (lst)

  ":Doc-Section ACL2::Programming

  quote an arbitrary true list of objects~/

  The function ~c[kwote-lst] applies the function ~c[kwote] to each element of
  a given list.  The guard of ~c[(kwote-lst lst)] is ~c[(true-listp lst)].~/~/"

  (declare (xargs :guard (true-listp lst)))
  (cond ((endp lst) nil)
        (t (cons (kwote (car lst)) (kwote-lst (cdr lst))))))

(defmacro unquote (x) (list 'cadr x))

(defmacro ffn-symb (x) (list 'car x))

(defun fn-symb (x)
  (declare (xargs :guard t))
  (if (and (nvariablep x)
           (not (fquotep x)))
      (car x)
    nil))

(defmacro fargs (x) (list 'cdr x))

(mutual-recursion

(defun all-vars1 (term ans)
  (declare (xargs :guard (and (pseudo-termp term)
                              (symbol-listp ans))
                  :mode :program))
  (cond ((variablep term)
         (add-to-set-eq term ans))
        ((fquotep term) ans)
        (t (all-vars1-lst (fargs term) ans))))

(defun all-vars1-lst (lst ans)
  (declare (xargs :guard (and (pseudo-term-listp lst)
                              (symbol-listp ans))
                  :mode :program))
  (cond ((endp lst) ans)
        (t (all-vars1-lst (cdr lst)
                          (all-vars1 (car lst) ans)))))

)

(verify-termination-boot-strap (all-vars1 (declare (xargs :mode :logic
                                               :verify-guards nil)))
                    (all-vars1-lst (declare (xargs :mode :logic))))

(defun all-vars (term)

; This function collects the variables in term in reverse print order of
; first occurrence.  E.g., all-vars of '(f (g a b) c) is '(c b a).
; This ordering is exploited by, at least, loop-stopper and bad-synp-hyp.

  (declare (xargs :guard (pseudo-termp term)
                  :verify-guards nil))
  (all-vars1 term nil))

; Progn.

; The definition of er-progn-fn below exposes a deficiency in ACL2 not
; present in full Common Lisp, namely ACL2's inability to generate a
; really ``new'' variable the way one can in a Common Lisp macro via
; gensym.  One would like to be sure that in binding the two variables
; er-progn-not-to-be-used-elsewhere-erp
; er-progn-not-to-be-used-elsewhere-val that they were not used
; anywhere in the subsequent macro expansion of lst.  If one had the
; macro expansion of lst at hand, one could manufacture a variable
; that was not free in the expansion with genvars, and that would do.

; As a less than elegant rememdy to the situation, we introduce below
; the macro check-vars-not-free, which takes two arguments, the first
; a not-to-be-evaluated list of variable names and the second an
; expression.  We arrange to return the translation of the expression
; provided none of the variables occur freely in it.  Otherwise, an error
; is caused.  The situation is subtle because we cannot even obtain
; the free vars in an expression until it has been translated.  For
; example, (value x) has the free var STATE in it, thanks to the macro
; expansion of value.  But a macro can't call translate because macros
; can't get their hands on state.

; In an earlier version of this we built check-vars-not-free into
; translate itself.  We defined it with a defmacro that expanded to
; its second arg, but translate did not actually look at the macro
; (raw lisp did) and instead implemented the semantics described
; above.  Of course, if no error was caused the semantics agreed with
; the treatment and if an error was caused, all bets are off anyway.
; The trouble with that approach was that it worked fine as long as
; check-vars-not-free was the only such example we had of needing to
; look at the translated form of something in a macro.  Unfortunately,
; others came along.  So we invented the more general
; translate-and-test and now use it to define check-vars-not-free.

(defmacro translate-and-test (test-fn form)

; Test-fn should be a LAMBDA expression (or function or macro symbol)
; of one non-STATE argument, and form is an arbitrary form.  Logically
; we ignore test-fn and return form.  However, an error is caused by
; TRANSLATE if the translation of form is not "approved" by test-fn.
; By "approved" we mean that when (test-fn 'term) is evaluated, where
; term is the translation of form, (a) the evaluation completes
; without an error and (b) the result is T.  Otherwise, the result is
; treated as an error msg and displayed.  (Actually, test-fn's answer
; is treated as an error msg if it is a stringp or a consp.  Any other
; result, e.g., T or NIL (!), is treated as "approved.")  If test-fn
; approves then the result of translation is the translation of form.

; For example,
; (translate-and-test
;  (lambda (term)
;   (or (subsetp (all-vars term) '(x y z))
;       (msg "~x0 uses variables other than x, y, and z."
;            term)))
;  <form>)
; is just the translation of <form> provided that translation
; only involves the free vars x, y, and z; otherwise an error is
; caused.  By generating calls of this macro other macros can
; ensure that the <form>s they generate satisfy certain tests
; after those <forms>s are translated.

; This macro is actually implemented in translate.  It can't be
; implemented here because translate isn't defined yet.  However the
; semantics is consistent with the definition below, namely, it just
; expands to its second argument (which is, of course, translated).
; It is just that sometimes errors are caused.

; There are two tempting generalizations of this function.  The first
; is that test-fn should be passed STATE so that it can make more
; "semantic" checks on the translation of form and perhaps so that it
; can signal the error itself.  There is, as far as I know,
; nothing wrong with this generalization except that it is hard to
; implement.  In order for TRANSLATE to determine whether test-fn
; approves of the term it must ev an expression.  If that expression
; involved STATE then translated must pass in its STATE in that
; position.  This requires coercing the state to an object, an act
; which is done with some trepidation in trans-eval and which could,
; presumably, be allowed earlier in translate.

; The second tempting generalization is that test-fn should have the
; power to massage the translation and return a new form which should,
; in turn, be translated.  For example, then one could imagine, say, a
; macro that would permit a form to be turned into the quoted constant
; listing the variables that occur freely in the translated form.  If
; the first generalization above has been carried out, then this would
; permit the translation of a form to be state dependent, which is
; illegal.  But this second generalization is problematic anyway.  In
; particular, what is the raw lisp counterpart of the generalized
; macro?  Note that in its current incarnation, the raw lisp
; counterpart of translate-and-test is the same as its logical
; meaning: it just expands to its second arg.  But if the desired
; expansion is computed from the translation of its second arg, then
; raw lisp would have to translate that argument.  But we can't do
; that for a variety of reasons: (a) CLTL macros shouldn't be state
; dependent, (b) we can't call translate during compilation because in
; general the ACL2 world isn't present, etc.

  (declare (ignore test-fn))
  form)

(defun intersectp-eq (x y)

  ":Doc-Section ACL2::Programming

  test whether two lists of symbols intersect~/

  ~l[intersectp-equal], which is logically the same function.~/

  ~c[(Intersectp-eq x y)] has a ~il[guard] that ~c[x] and ~c[y] are lists of
  symbols.~/"

  (declare (xargs :guard (and (symbol-listp x)
                              (symbol-listp y))))
  (cond ((endp x) nil)
        ((member-eq (car x) y) t)
        (t (intersectp-eq (cdr x) y))))

(defun intersectp (x y)

  ":Doc-Section ACL2::Programming

  test whether two lists of ~ilc[eqlablep] objects intersect~/

  ~l[intersectp-equal], which is logically the same function.~/

  ~c[(Intersectp x y)] has a ~il[guard] that ~c[x] and ~c[y] are lists 
  containing only numbers, symbols, and characters.~/"

  (declare (xargs :guard (and (eqlable-listp x)
                              (eqlable-listp y))))
  (cond ((endp x) nil)
        ((member (car x) y) t)
        (t (intersectp (cdr x) y))))

(defun intersectp-equal (x y)

  ":Doc-Section ACL2::Programming

  test whether two lists intersect~/

  ~c[(Intersectp-equal x y)] returns ~c[t] if ~c[x] and ~c[y] have a
  member in common, else it returns ~c[nil.]  Also
  ~pl[intersectp-eq], which is logically the same but can be more
  efficient since it uses ~ilc[eq] instead of ~ilc[equal] to look for
  members common to the two given lists.~/

  ~c[(Intersectp-equal x y)] has a ~il[guard] that ~c[x] and ~c[y] are true lists.~/"

  (declare (xargs :guard (and (true-listp x)
                              (true-listp y))))
  (cond ((endp x) nil)
        ((member-equal (car x) y) t)
        (t (intersectp-equal (cdr x) y))))

(defun make-fmt-bindings (chars forms)
  (declare (xargs :guard (and (true-listp chars)
                              (true-listp forms)
                              (<= (length forms) (length chars)))))
  (cond ((endp forms) nil)
        (t (list 'cons
                 (list 'cons (car chars) (car forms))
                 (make-fmt-bindings (cdr chars) (cdr forms))))))

(defmacro msg (str &rest args)

; This macro returns a pair suitable giving to the fmt directive ~@.  Fmt is
; defined much later.  But we need msg now because several of our macros
; generate calls of msg and thus msg must be a function when terms using those
; macros are translated.

; In any case, suppose that #\0, say, is bound to the value of this function.
; Then the fmt directive ~@0 will print out the string, str, above, in the
; context of the alist in which the successive fmt variables #\0 through
; possibly #\9 are bound to the successive elements of args.

  (declare (xargs :guard (<= (length args) 10)))

  `(cons ,str ,(make-fmt-bindings '(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9) args)))

(defmacro check-vars-not-free (vars form)

; A typical use of this macro is (check-vars-not-free (my-erp my-val)
; ...)  which just expands to the translation of ... provided my-erp
; and my-val do not occur freely in it.

  (declare (xargs :guard (symbol-listp vars)))
  `(translate-and-test
    (lambda (term)
      (let ((vars ',vars))
        (or (not (intersectp-eq vars (all-vars term)))
            (msg "It is forbidden to use ~v0 in ~x1."
                 vars term))))
    ,form))

(defun er-progn-fn (lst)
  (declare (xargs :guard (true-listp lst)))
  (cond ((endp lst) nil)
        ((endp (cdr lst)) (car lst))
        (t (list 'mv-let
                 '(er-progn-not-to-be-used-elsewhere-erp
                   er-progn-not-to-be-used-elsewhere-val
                   state)
                 (car lst)
; Avoid possible warning after optimized compilation:
                 '(declare (ignorable er-progn-not-to-be-used-elsewhere-val))
                 (list 'if
                       'er-progn-not-to-be-used-elsewhere-erp
                       '(mv er-progn-not-to-be-used-elsewhere-erp
                            er-progn-not-to-be-used-elsewhere-val
                            state)
                       (list 'check-vars-not-free
                             '(er-progn-not-to-be-used-elsewhere-erp
                               er-progn-not-to-be-used-elsewhere-val)
                             (er-progn-fn (cdr lst))))))))

(defmacro er-progn (&rest lst)

  ":Doc-Section ACL2::Programming

  perform a sequence of state-changing ``error triples''~/

  ~bv[]
  Example:
  (er-progn (check-good-foo-p (f-get-global 'my-foo state) state)
            (value (* (f-get-global 'my-foo state)
                      (f-get-global 'bar state))))
  ~ev[]

  This sequencing primitive is only useful when programming with
  ~il[state], something that very few users will probably want to do.
  ~l[state].~/

  ~c[Er-progn] is used much the way that ~ilc[progn] is used in Common
  Lisp, except that it expects each form within it to evaluate to an
  ``error triple'' of the form ~c[(mv erp val state)].  The first such
  form, if any, that evaluates to such a triple where ~c[erp] is not
  ~c[nil] yields the error triple returned by the ~c[er-progn].  If
  there is no such form, then the last form returns the value of the
  ~c[er-progn] form.

  We intend to write more about this topic, especially if there are
  requests to do so.~/"

  (declare (xargs :guard (and (true-listp lst)
                              lst)))
  (er-progn-fn lst))

(defun legal-case-clausesp (tl)
  (declare (xargs :guard t))
  (cond ((atom tl)
         (eq tl nil))
        ((and (consp (car tl))
              (or (eqlablep (car (car tl)))
                  (eqlable-listp (car (car tl))))
              (consp (cdr (car tl)))
              (null (cdr (cdr (car tl))))
              (if (or (eq t (car (car tl)))
                      (eq 'otherwise (car (car tl))))
                  (null (cdr tl))
                t))
         (legal-case-clausesp (cdr tl)))
        (t nil)))

(defun case-test (x pat)
  (declare (xargs :guard t))
  (cond ((atom pat) (list 'eql x (list 'quote pat)))
        (t (list 'member x (list 'quote pat)))))

(defun case-list (x l)
  (declare (xargs :guard (legal-case-clausesp l)))
  (cond ((endp l) nil)
        ((or (eq t (car (car l)))
             (eq 'otherwise (car (car l))))
         (list (list 't (car (cdr (car l))))))
        ((null (car (car l)))
         (case-list x (cdr l)))
        (t (cons (list (case-test x (car (car l)))
                       (car (cdr (car l))))
                 (case-list x (cdr l))))))

(defun case-list-check (l)
  (declare (xargs :guard (legal-case-clausesp l)))
  (cond ((endp l) nil)
        ((or (eq t (car (car l)))
             (eq 'otherwise (car (car l))))
         (list (list 't (list 'check-vars-not-free
                              '(case-do-not-use-elsewhere)
                              (car (cdr (car l)))))))
        ((null (car (car l)))
         (case-list-check (cdr l)))
        (t (cons (list (case-test 'case-do-not-use-elsewhere (car (car l)))
                       (list 'check-vars-not-free
                             '(case-do-not-use-elsewhere)
                             (car (cdr (car l)))))
                 (case-list-check (cdr l))))))

#+acl2-loop-only
(defmacro case (&rest l)

  ":Doc-Section ACL2::Programming

  conditional based on if-then-else using ~ilc[eql]~/
  ~bv[]
  Example Form:
  (case typ
    ((:character foo)
     (open file-name :direction :output))
    (bar (open-for-bar file-name))
    (otherwise
     (my-error \"Illegal.\")))
  ~ev[]
  is the same as
  ~bv[]
  (cond ((member typ '(:character foo))
         (open file-name :direction :output))
        ((eql typ 'bar)
         (open-for-bar file-name))
        (t (my-error \"Illegal.\")))
  ~ev[]
  which in turn is the same as
  ~bv[]
  (if (member typ '(:character foo))
      (open file-name :direction :output)
      (if (eql typ 'bar)
          (open-for-bar file-name)
          (my-error \"Illegal.\")))~/
  ~ev[]
  Notice the quotations that appear in the example above:
  ~c['(:character foo)] and ~c['bar].

  ~bv[]
  General Forms:
  (case expr
    (x1 val-1)
    ...
    (xk val-k)
    (otherwise val-k+1))

  (case expr
    (x1 val-1)
    ...
    (xk val-k)
    (t val-k+1))

  (case expr
    (x1 val-1)
    ...
    (xk val-k))
  ~ev[]
  where each ~c[xi] is either ~ilc[eqlablep] or a true list of ~ilc[eqlablep]
  objects.  The final ~c[otherwise] or ~c[t] case is optional.

  ~c[Case] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (consp l)
                              (legal-case-clausesp (cdr l)))))
  (cond ((atom (car l))
         (cons 'cond (case-list (car l) (cdr l))))
        (t `(let ((case-do-not-use-elsewhere ,(car l)))
              (cond ,@(case-list-check (cdr l)))))))

(defun position-equal-ac (item lst acc)
  (declare (xargs :guard (and (true-listp lst)
                              (acl2-numberp acc))))
  (cond
   ((endp lst) nil)
   ((equal item (car lst))
    acc)
   (t (position-equal-ac item (cdr lst) (1+ acc)))))

(defun position-ac (item lst acc)
  (declare (xargs :guard (and (true-listp lst)
                              (or (eqlablep item)
                                  (eqlable-listp lst))
                              (acl2-numberp acc))))
  (cond
   ((endp lst) nil)
   ((eql item (car lst))
    acc)
   (t (position-ac item (cdr lst) (1+ acc)))))

(defun position-equal (item lst)

  ":Doc-Section ACL2::Programming

  position of an item in a string or a list~/

  ~c[(Position item seq)] is the least index (zero-based) of the
  element ~c[item] in the string or list ~c[seq], if in fact ~c[item] is
  an element of ~c[seq].  Otherwise ~c[(position item seq)] is ~c[nil].~/

  ~c[(Position-equal item lst)] has a ~il[guard] of ~c[(true-listp lst)].
  ~c[Position-equal] has the same functionality as the Common Lisp
  function ~ilc[position], except that it uses the ~ilc[equal] function to
  test whether ~c[item] is the same as each successive element of
  ~c[lst].  ~l[position] and ~pl[position-eq].~/"

  (declare (xargs :guard (or (stringp lst) (true-listp lst))))
  #-acl2-loop-only ; for assoc-eq, Jared Davis found native assoc efficient
  (position item lst :test #'equal)
  #+acl2-loop-only
  (if (stringp lst)
      (position-ac item (coerce lst 'list) 0)
    (position-equal-ac item lst 0)))

(defun position-eq-ac (item lst acc)
  (declare (xargs :guard (and (true-listp lst)
                              (or (symbolp item)
                                  (symbol-listp lst))
                              (acl2-numberp acc))))
  (cond
   ((endp lst) nil)
   ((eq item (car lst))
    acc)
   (t (position-eq-ac item (cdr lst) (1+ acc)))))

(defun position-eq (item lst)

  ":Doc-Section ACL2::Programming

  position of an item in a string or a list, using ~ilc[eq] as test~/

  ~c[(Position-eq item seq)] is the least index (zero-based) of the
  element ~c[item] in the list ~c[seq], if in fact ~c[item] is
  an element of ~c[seq].  Otherwise ~c[(position-eq item seq)] is ~c[nil].~/

  ~c[(Position-eq item lst)] is provably the same in the ACL2 logic as
  ~c[(position item lst)] and ~c[(position-equal item lst)] when ~c[lst] is a
  true list, but it has a stronger ~il[guard] because it uses ~ilc[eq] for a
  more efficient test for whether ~c[item] is equal to a given member of
  ~c[lst].  Its ~il[guard] requires that ~c[lst] is a true list, and moreover,
  either ~c[item] is a symbol or ~c[lst] is a list of symbols.
  ~l[position-equal] and ~pl[position], which unlike ~c[position-eq] have
  guards that allow the second argument to be a string.~/"

  (declare (xargs :guard (and (true-listp lst)
                              (or (symbolp item)
                                  (symbol-listp lst)))))

; Should we write the same sort of body that we do for position?  Why would
; anyone deliberately make a call (position-eq item string)?  Such a call has
; to produce a guard violation unless item is a symbol, in which case
; position-eq would return nil in the logic.  On the other hand, we could prove
; equivalence of position and position-eq without a true-listp hypothesis if we
; made the definition below that tests for stringp in analogy to the bodies of
; position and position-equal.

  #-acl2-loop-only ; for assoc-eq, Jared Davis found native assoc efficient
  (position item lst :test #'eq)
  #+acl2-loop-only
  (position-eq-ac item lst 0))

#+acl2-loop-only
(defun position (item lst)

  ":Doc-Section ACL2::Programming

  position of an item in a string or a list, using ~ilc[eql] as test~/

  ~c[(Position item seq)] is the least index (zero-based) of the
  element ~c[item] in the string or list ~c[seq], if in fact ~c[item] is
  an element of ~c[seq].  Otherwise ~c[(position item seq)] is ~c[nil].~/

  ~c[(Position item lst)] is provably the same in the ACL2 logic as
  ~c[(position-equal item lst)].  It has a stronger ~il[guard] than
  ~ilc[position-equal] because uses ~ilc[eql] to test equality of ~c[item]
  with members of ~c[lst].  Its ~il[guard] requires that either ~c[lst] is a
  string, or else ~c[lst] is a true list such that either ~c[(eqlablep item)]
  or all members of ~c[lst] are ~ilc[eqlablep].  ~l[position-equal]
  and ~pl[position-eq].

  ~c[Position] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  Since ACL2 functions cannot
  take keyword arguments (though macros can), the ACL2 functions
  ~ilc[position-equal] and ~ilc[position-eq] are defined to correspond to
  calls of the Common Lisp function ~c[position] whose keyword argument
  ~c[:test] is ~ilc[equal] or ~ilc[eq], respectively.~/"

  (declare (xargs :guard (or (stringp lst)
                             (and (true-listp lst)
                                  (or (eqlablep item)
                                      (eqlable-listp lst))))))
  (if (stringp lst)
      (position-ac item (coerce lst 'list) 0)
    (position-ac item lst 0)))

(defun nonnegative-integer-quotient (i j)

  ":Doc-Section ACL2::Programming

  natural number division function~/
  ~bv[]
  Example Forms:
  (nonnegative-integer-quotient 14 3) ; equals 4
  (nonnegative-integer-quotient 15 3) ; equals 5
  ~ev[]
  ~c[(nonnegative-integer-quotient i j)] returns the integer quotient
  of the integers ~c[i] and (non-zero) ~c[j], i.e., the largest ~c[k]
  such that ~c[(* j k)] is less than or equal to ~c[i].  Also
  ~pl[floor], ~pl[ceiling] and ~pl[truncate], which are
  derived from this function and apply to rational numbers.~/

  The ~il[guard] of ~c[(nonnegative-integer-quotient i j)] requires that
  ~c[i] is a nonnegative integer and ~c[j] is a positive integer.~/"

  (declare (xargs :guard (and (integerp i)
                              (not (< i 0))
                              (integerp j)
                              (< 0 j))))
  #-acl2-loop-only
  (values (floor i j)) ; see books/misc/misc2/misc.lisp for justification
  #+acl2-loop-only
  (if (or (= (nfix j) 0)
          (< (ifix i) j))
      0
    (+ 1 (nonnegative-integer-quotient (- i j) j))))

; Next we develop let* in the logic.

(defun true-list-listp (x)

  ":Doc-Section ACL2::Programming

  recognizer for true (proper) lists of true lists~/

  ~c[True-list-listp] is the function that checks whether its argument
  is a list that ends in, or equals, ~c[nil], and furthermore, all of
  its elements have that property.  Also ~pl[true-listp].~/~/"

  (declare (xargs :guard t))
  (cond ((atom x) (eq x nil))
        (t (and (true-listp (car x))
                (true-list-listp (cdr x))))))

(defthm true-list-listp-forward-to-true-listp
  (implies (true-list-listp x)
           (true-listp x))
  :rule-classes :forward-chaining)

(defun legal-let*-p (bindings ignore-vars ignored-seen top-form)

; We check that no variable declared ignored or ignorable is bound twice.  We
; also check that all ignored-vars are bound.  We could leave it to translate
; to check the resulting LET form instead, but we prefer to do the check here,
; both in order to clarify the problem for the user (the blame will be put on
; the LET* form) and because we are not sure of the Common Lisp treatment of
; such a LET* and could thus be in unknown territory were we ever to relax the
; corresponding restriction on LET.

; Ignored-seen should be nil at the top level.

  (declare (xargs :guard (and top-form ; to avoid irrelevance
                              (symbol-alistp bindings)
                              (symbol-listp ignore-vars)
                              (symbol-listp ignored-seen))))
  (cond ((endp bindings)
         (or (eq ignore-vars nil)
             (hard-error 'let*
                         "All variables declared IGNOREd or IGNORABLE in a ~
                          LET* form must be bound, but ~&0 ~#0~[is~/are~] not ~
                          bound in the form ~x1."
                         (list (cons #\0 ignore-vars)
                               (cons #\1 top-form)))))
        ((member-eq (caar bindings) ignored-seen)
         (hard-error 'let*
                     "A variable bound more than once in a LET* form may not ~
                      be declared IGNOREd or IGNORABLE, but the variable ~x0 ~
                      is bound more than once in form ~x1 and yet is so ~
                      declared."
                     (list (cons #\0 (caar bindings))
                           (cons #\1 top-form))))
        ((member-eq (caar bindings) ignore-vars)
         (legal-let*-p (cdr bindings)
                       (remove (caar bindings) ignore-vars)
                       (cons (caar bindings) ignored-seen)
                       top-form))
        (t (legal-let*-p (cdr bindings) ignore-vars ignored-seen top-form))))

(defun well-formed-type-decls-p (decls vars)

; Decls is a true list of declarations (type tp var1 ... vark).  We check that
; each vari is bound in vars.

  (declare (xargs :guard (and (true-list-listp decls)
                              (symbol-listp vars))))
  (cond ((endp decls) t)
        ((subsetp-eq (cddr (car decls)) vars)
         (well-formed-type-decls-p (cdr decls) vars))
        (t nil)))

(defun symbol-list-listp (x)
  (declare (xargs :guard t))
  (cond ((atom x) (eq x nil))
        (t (and (symbol-listp (car x))
                (symbol-list-listp (cdr x))))))

(defun get-type-decls (var type-decls)
  (declare (xargs :guard (and (symbolp var)
                              (true-list-listp type-decls)
                              (alistp type-decls)
                              (symbol-list-listp (strip-cdrs type-decls)))))
  (cond ((endp type-decls) nil)
        ((member-eq var (cdr (car type-decls)))
         (cons (list 'type (car (car type-decls)) var)
               (get-type-decls var (cdr type-decls))))
        (t (get-type-decls var (cdr type-decls)))))

(defun let*-macro (bindings ignore-vars ignorable-vars type-decls body)
  (declare (xargs :guard (and (symbol-alistp bindings)
                              (symbol-listp ignore-vars)
                              (symbol-listp ignorable-vars)
                              (true-list-listp type-decls)
                              (alistp type-decls)
                              (symbol-list-listp (strip-cdrs type-decls)))))
  (cond ((endp bindings)
         (prog2$ (or (null ignore-vars)
                     (hard-error 'let*-macro
                                 "Implementation error: Ignored variables ~x0 ~
                                  must be bound in superior LET* form!"
                                 ignore-vars))
                 (prog2$ (or (null ignorable-vars)
                             (hard-error 'let*-macro
                                         "Implementation error: Ignorable ~
                                          variables ~x0 must be bound in ~
                                          superior LET* form!"
                                         ignorable-vars))
                         body)))
        (t ; (consp bindings)
         (cons 'let
               (cons (list (car bindings))
                     (let ((rest (let*-macro (cdr bindings)
                                             (remove (caar bindings)
                                                     ignore-vars)
                                             (remove (caar bindings)
                                                     ignorable-vars)
                                             type-decls
                                             body)))
                       (append
                        (and (member-eq (caar bindings) ignore-vars)
                             (list (list 'declare
                                         (list 'ignore (caar bindings)))))
                        (and (member-eq (caar bindings) ignorable-vars)
                             (list (list 'declare
                                         (list 'ignorable (caar bindings)))))
                        (let ((var-type-decls
                               (get-type-decls (caar bindings) type-decls)))
                          (and var-type-decls
                               (list (cons 'declare var-type-decls))))
                        (list rest))))))))

(defun collect-cdrs-when-car-eq (x alist)
  (declare (xargs :guard (and (symbolp x)
                              (true-list-listp alist))))
  (cond ((endp alist) nil)
        ((eq x (car (car alist)))
         (append (cdr (car alist))
                 (collect-cdrs-when-car-eq x (cdr alist))))
        (t (collect-cdrs-when-car-eq x (cdr alist)))))

(defun append-lst (lst)
  (declare (xargs :guard (true-list-listp lst)))
  (cond ((endp lst) nil)
        (t (append (car lst) (append-lst (cdr lst))))))

(defun restrict-alist (keys alist)

; Returns the subsequence of alist whose cars are among keys (without any
; reordering).

  (declare (xargs :guard (and (symbol-listp keys)
                              (alistp alist))))
  (cond
   ((endp alist)
    nil)
   ((member-eq (caar alist) keys)
    (cons (car alist)
          (restrict-alist keys (cdr alist))))
   (t (restrict-alist keys (cdr alist)))))

#+acl2-loop-only
(defmacro let* (&whole form bindings &rest decl-body)

  ":Doc-Section ACL2::Programming

  binding of lexically scoped (local) variables~/
  ~bv[]
  Example LET* Forms:
  (let* ((x (* x x))
         (y (* 2 x)))
   (list x y))

  (let* ((x (* x x))
         (y (* 2 x))
         (x (* x y))
         (a (* x x)))
   (declare (ignore a))
   (list x y))
  ~ev[]
  If the forms above are executed in an environment in which ~c[x] has the
  value ~c[-2], then the respective results are ~c['(4 8)] and ~c['(32 8)].
  ~l[let] for a discussion of both ~ilc[let] and ~c[let*], or read
  on for a briefer discussion.~/

  The difference between ~ilc[let] and ~c[let*] is that the former binds its
  local variables in parallel while the latter binds them
  sequentially.  Thus, in ~c[let*], the term evaluated to produce the
  local value of one of the locally bound variables is permitted to
  reference any locally bound variable occurring earlier in the
  binding list and the value so obtained is the newly computed local
  value of that variable.  ~l[let].

  In ACL2 the only ~ilc[declare] forms allowed for a ~c[let*] form are
  ~c[ignore], ~c[ignorable], and ~c[type].  ~l[declare].  Moreover, no variable
  declared ~c[ignore]d or ~c[ignorable] may be bound more than once.  A
  variable with a type declaration may be bound more than once, in which case
  the type declaration is treated by ACL2 as applying to each binding
  occurrence of that variable.  It seems unclear from the Common Lisp spec
  whether the underlying Lisp implementation is expected to apply such a
  declaration to more than one binding occurrence, however, so performance in
  such cases may depend on the underlying Lisp.

  ~c[Let*] is a Common Lisp macro.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs
            :guard

; We do not check that the variables declared ignored are not free in the body,
; nor do we check that variables bound in bindings that are used in the body
; are not declared ignored.  Those properties will be checked for the expanded
; LET form, as appropriate.

            (and (symbol-alistp bindings)
                 (true-listp decl-body)
                 decl-body
                 (let ((declare-forms (butlast decl-body 1)))
                   (and
                    (alistp declare-forms)
                    (subsetp-eq (strip-cars declare-forms)
                                '(declare))
                    (let ((decls (append-lst (strip-cdrs declare-forms))))
                      (let ((ign-decls (restrict-alist '(ignore ignorable)
                                                       decls))
                            (type-decls (restrict-alist '(type) decls)))
                        (and (symbol-alistp decls)
                             (symbol-list-listp ign-decls)
                             (subsetp-eq (strip-cars decls)
                                         '(ignore ignorable type))
                             (well-formed-type-decls-p type-decls
                                                       (strip-cars bindings))
                             (legal-let*-p
                              bindings
                              (append-lst (strip-cdrs ign-decls))
                              nil
                              form)))))))))
  (declare (ignore form))
  (let ((decls (append-lst (strip-cdrs (butlast decl-body 1))))
        (body (car (last decl-body))))
    (let ((ignore-vars (collect-cdrs-when-car-eq 'ignore decls))
          (ignorable-vars (collect-cdrs-when-car-eq 'ignorable decls))
          (type-decls (strip-cdrs (restrict-alist '(type) decls))))
      (let*-macro bindings ignore-vars ignorable-vars type-decls body))))

#+acl2-loop-only
(defmacro progn (&rest r)

; Warning: See the Important Boot-Strapping Invariants before modifying!

  ":Doc-Section ACL2::Events

  evaluate some ~il[events]~/
  ~bv[]
  Example Form:
  (progn (defun foo (x) x)
         (defmacro my-defun (&rest args)
           (cons 'defun args))
         (my-defun bar (x) (foo x)))

  General form:
  (progn event1 event2 ... eventk)
  ~ev[]
  where ~c[k] >= 0 and each ~c[eventi] is a legal embedded event form
  (~pl[embedded-event-form]).  These events are evaluated in sequence.  A
  utility is provided to assist in debugging failures of such execution;
  ~pl[redo-flat].

  NOTE: If the ~c[eventi] above are not all legal embedded event forms
  (~pl[embedded-event-form]), consider using ~ilc[er-progn] or (with great
  care!) ~ilc[progn!] instead.

  For a related event form that does allow introduction of ~il[constraint]s
  and ~ilc[local] ~il[events], ~pl[encapsulate].

  ACL2 does not allow the use of ~c[progn] in definitions.  Instead, the
  macro ~ilc[er-progn] can be used for sequencing ~il[state]-oriented
  operations; ~pl[er-progn] and ~pl[state].  If you are using single-threaded
  objects (~pl[stobj]) you may wish to define a version of ~ilc[er-progn] that
  cascades the object through successive changes.  ACL2's ~ilc[pprogn] is the
  ~c[state] analogue of such a macro.

  If your goal is simply to execute a sequence of top-level forms, for example
  a sequence of definitions, consider using ~c[ld] instead; ~pl[ld].~/~/"

; Like defun, defmacro, and in-package, progn does not have quite the same
; semantics as the Common Lisp function.  This is useful only for sequences at
; the top level.  It permits us to handle things like type sets and records.

  (list 'progn-fn
        (list 'quote r)
        'state))

#+acl2-loop-only
(progn

;; RAG - This function had to be modified in a major way.  It was
;; originally defined only for rationals, and it used the fact that
;; the floor of "p/q" could be found by repeatedly subtracting "q"
;; from "p" (roughly speaking).  This same trick, sadly, does not work
;; for the reals.  Instead, we need something similar to the
;; archimedean axiom.  Our version thereof is the _undefined_ function
;; "floor1", which takes a single argument and returns an integer
;; equal to it or smaller to it by no more than 1.  Using this
;; function, we can define the more general floor function offered
;; below.

(defdoc real
  ":Doc-Section ACL2::Real

  ACL2(r) support for real numbers~/

  ACL2 supports rational numbers but not real numbers.  However,
  starting with Version  2.5, a variant of ACL2 called ``ACL2(r)''
  supports the real numbers by way of non-standard analysis.  ACL2(r)
  was conceived and first implemented by Ruben Gamboa in his Ph.D.
  dissertation work, supervised by Bob Boyer with active participation
  by Matt Kaufmann.  The Makefile provided with ACL2 has target
  ~c[large-acl2r] for building ACL2(r) images.  To see
  which image you have, see if the prompt includes the string ``~c[(r)]'',
  e.g.:
  ~bv[]
  ACL2(r) !>
  ~ev[]
  Or, look at ~c[(@ acl2-version)] and see if ``~c[(r)]'' is a substring.

  In ACL2 (as opposed to ACL2(r)), when we say ``real'' we mean
  ``rational.''~/

  Caution:  ACL2(r) should be considered experimental as of Version
  2.5: although we (Kaufmann and Moore) have carefully completed
  Gamboa's integration of the reals into the ACL2 source code, our
  primary concern to date has been to ensure unchanged behavior when
  ACL2 is compiled in the default manner, i.e., without the
  non-standard extensions.  As for every release of ACL2, at the time
  of this release we are unaware of soundness bugs in ACL2 Version  2.5.
  We are confident that ACL2(r) will behave much as it does now and will
  ultimately be sound; but we have not yet argued the soundness of
  every detail of the integration.

  There is only limited documentation on the non-standard features of
  ACL2(r).  We hope to provide more documentation for such features in
  future releases.  Please feel free to query the authors if you are
  interested in learning more about ACL2(r).  Gamboa's dissertation
  may also be helpful.~/")

(defun floor (i j)

  ":Doc-Section ACL2::Programming

  division returning an integer by truncating toward negative infinity~/
  ~bv[]
  Example Forms:
  ACL2 !>(floor 14 3)
  4
  ACL2 !>(floor -14 3)
  -5
  ACL2 !>(floor 14 -3)
  -5
  ACL2 !>(floor -14 -3)
  4
  ACL2 !>(floor -15 -3)
  5
  ~ev[]
  ~c[(Floor i j)] returns the result of taking the quotient of ~c[i] and
  ~c[j] and returning the greatest integer not exceeding that quotient.
  For example, the quotient of ~c[-14] by ~c[3] is ~c[-4 2/3], and the largest
  integer not exceeding that rational number is ~c[-5].~/

  The ~il[guard] for ~c[(floor i j)] requires that ~c[i] and ~c[j] are
  rational (~il[real], in ACL2(r)) numbers and ~c[j] is non-zero.

  ~c[Floor] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  However, note that unlike Common Lisp,
  the ACL2 ~c[floor] function returns only a single value, ~/"

  (declare (xargs :guard (and (real/rationalp i)
                              (real/rationalp j)
                              (not (eql j 0)))))
  #+:non-standard-analysis
  (let ((q (* i (/ j))))
    (cond ((integerp q) q)
          ((rationalp q)
           (if (>= q 0)
               (nonnegative-integer-quotient (numerator q) (denominator q))
             (+ (- (nonnegative-integer-quotient (- (numerator q))
                                                 (denominator q)))
                -1)))
          (t (floor1 q))))
  #-:non-standard-analysis
  (let* ((q (* i (/ j)))
         (n (numerator q))
         (d (denominator q)))
    (cond ((= d 1) n)
          ((>= n 0)
           (nonnegative-integer-quotient n d))
          (t (+ (- (nonnegative-integer-quotient (- n) d)) -1))))
  )

;; RAG - This function was also modified to fit in the reals.  It's
;; also defined in terms of the _undefined_ function floor1 (which
;; corresponds to the usual unary floor function).

(defun ceiling (i j)

  ":Doc-Section ACL2::Programming

  division returning an integer by truncating toward positive infinity~/
  ~bv[]
  Example Forms:
  ACL2 !>(ceiling 14 3)
  5
  ACL2 !>(ceiling -14 3)
  -4
  ACL2 !>(ceiling 14 -3)
  -4
  ACL2 !>(ceiling -14 -3)
  5
  ACL2 !>(ceiling -15 -3)
  5
  ~ev[]
  ~c[(Ceiling i j)] is the result of taking the quotient of ~c[i] and
  ~c[j] and returning the smallest integer that is at least as great as
  that quotient.  For example, the quotient of ~c[-14] by ~c[3] is ~c[-4 2/3], and
  the smallest integer at least that great is ~c[-4].~/

  The ~il[guard] for ~c[(ceiling i j)] requires that ~c[i] and ~c[j] are
  rational (~il[real], in ACL2(r)) numbers and ~c[j] is non-zero.

  ~c[Ceiling] is a Common Lisp function.  See any Common Lisp documentation for
  more information.  However, note that unlike Common Lisp, the ACL2
  ~c[ceiling] function returns only a single value, ~/"

  (declare (xargs :guard (and (real/rationalp i)
                              (real/rationalp j)
                              (not (eql j 0)))))
  #+:non-standard-analysis
  (let ((q (* i (/ j))))
    (cond ((integerp q) q)
          ((rationalp q)
           (if (>= q 0)
               (+ (nonnegative-integer-quotient (numerator q)
                                                (denominator q))
                  1)
             (- (nonnegative-integer-quotient (- (numerator q))
                                              (denominator q)))))
          (t (1+ (floor1 q)))))
  #-:non-standard-analysis
  (let* ((q (* i (/ j)))
         (n (numerator q))
         (d (denominator q)))
    (cond ((= d 1) n)
          ((>= n 0)
           (+ (nonnegative-integer-quotient n d) 1))
          (t (- (nonnegative-integer-quotient (- n) d)))))
  )

;; RAG - Another function  modified to fit in the reals, using floor1.

(defun truncate (i j)

  ":Doc-Section ACL2::Programming

  division returning an integer by truncating toward 0~/
  ~bv[]
  Example Forms:
  ACL2 !>(truncate 14 3)
  4
  ACL2 !>(truncate -14 3)
  -4
  ACL2 !>(truncate 14 -3)
  -4
  ACL2 !>(truncate -14 -3)
  4
  ACL2 !>(truncate -15 -3)
  5
  ACL2 !>(truncate 10/4 3/4)
  3
  ~ev[]
  ~c[(Truncate i j)] is the result of taking the quotient of ~c[i] and
  ~c[j] and dropping the fraction.  For example, the quotient of ~c[-14] by
  ~c[3] is ~c[-4 2/3], so dropping the fraction ~c[2/3], we obtain a result for
  ~c[(truncate -14 3)] of ~c[-4].~/

  The ~il[guard] for ~c[(truncate i j)] requires that ~c[i] and ~c[j] are
  rational (~il[real], in ACL2(r)) numbers and ~c[j] is non-zero.

  ~c[Truncate] is a Common Lisp function.  However, note that unlike Common
  Lisp, the ACL2 ~c[truncate] function returns only a single value,  Also
  ~pl[nonnegative-integer-quotient], which is appropriate for integers and may
  simplify reasoning, unless a suitable arithmetic library is loaded, but be
  less efficient for evaluation on concrete objects.~/"

  (declare (xargs :guard (and (real/rationalp i)
                              (real/rationalp j)
                              (not (eql j 0)))))
  #+:non-standard-analysis
  (let ((q (* i (/ j))))
    (cond ((integerp q) q)
          ((rationalp q)
           (if (>= q 0)
               (nonnegative-integer-quotient (numerator q)
                                             (denominator q))
             (- (nonnegative-integer-quotient (- (numerator q))
                                              (denominator q)))))
          (t (if (>= q 0)
                 (floor1 q)
               (- (floor1 (- q)))))))
  #-:non-standard-analysis
  (let* ((q (* i (/ j)))
         (n (numerator q))
         (d (denominator q)))
    (cond ((= d 1) n)
          ((>= n 0)
           (nonnegative-integer-quotient n d))
          (t (- (nonnegative-integer-quotient (- n) d)))))
  )

;; RAG - Another function  modified to fit in the reals, using floor1.

(defun round (i j)

  ":Doc-Section ACL2::Programming

  division returning an integer by rounding off~/
  ~bv[]
  Example Forms:
  ACL2 !>(round 14 3)
  5
  ACL2 !>(round -14 3)
  -5
  ACL2 !>(round 14 -3)
  -5
  ACL2 !>(round -14 -3)
  5
  ACL2 !>(round 13 3)
  4
  ACL2 !>(round -13 3)
  -4
  ACL2 !>(round 13 -3)
  -4
  ACL2 !>(round -13 -3)
  4
  ACL2 !>(round -15 -3)
  5
  ACL2 !>(round 15 -2)
  -8
  ~ev[]
  ~c[(Round i j)] is the result of taking the quotient of ~c[i] and ~c[j]
  and rounding off to the nearest integer.  When the quotient is
  exactly halfway between consecutive integers, it rounds to the even
  one.~/

  The ~il[guard] for ~c[(round i j)] requires that ~c[i] and ~c[j] are
  rational (~il[real], in ACL2(r)) numbers and ~c[j] is non-zero.

  ~c[Round] is a Common Lisp function.  See any Common Lisp documentation for
  more information.  However, note that unlike Common Lisp, the ACL2 ~c[round]
  function returns only a single value, ~/"

  (declare (xargs :guard (and (real/rationalp i)
                              (real/rationalp j)
                              (not (eql j 0)))))
  (let ((q (* i (/ j))))
    (cond ((integerp q) q)
          ((>= q 0)
           (let* ((fl (floor q 1))
                  (remainder (- q fl)))
             (cond ((> remainder 1/2)
                    (+ fl 1))
                   ((< remainder 1/2)
                    fl)
                   (t (cond ((integerp (* fl (/ 2)))
                             fl)
                            (t (+ fl 1)))))))
          (t
           (let* ((cl (ceiling q 1))
                  (remainder (- q cl)))
             (cond ((< (- 1/2) remainder)
                    cl)
                   ((> (- 1/2) remainder)
                    (+ cl -1))
                   (t (cond ((integerp (* cl (/ 2)))
                             cl)
                            (t (+ cl -1)))))))))
  )

;; RAG - I only had to modify the guards here to allow the reals,
;; since this function is defined in terms of the previous ones.

(defun mod (x y)

  ":Doc-Section ACL2::Programming

  remainder using ~ilc[floor]~/
  ~bv[]
  ACL2 !>(mod 14 3)
  2
  ACL2 !>(mod -14 3)
  1
  ACL2 !>(mod 14 -3)
  -1
  ACL2 !>(mod -14 -3)
  -2
  ACL2 !>(mod -15 -3)
  0
  ACL2 !>
  ~ev[]
  ~c[(Mod i j)] is that number ~c[k] that ~c[(* j (floor i j))] added to
  ~c[k] equals ~c[i].~/

  The ~il[guard] for ~c[(mod i j)] requires that ~c[i] and ~c[j] are rational
  (~il[real], in ACL2(r)) numbers and ~c[j] is non-zero.

  ~c[Mod] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (declare (xargs :guard (and (real/rationalp x)
                              (real/rationalp y)
                              (not (eql y 0)))))
  (- x (* (floor x y) y)))

(defun rem (x y)

  ":Doc-Section ACL2::Programming

  remainder using ~ilc[truncate]~/
  ~bv[]
  ACL2 !>(rem 14 3)
  2
  ACL2 !>(rem -14 3)
  -2
  ACL2 !>(rem 14 -3)
  2
  ACL2 !>(rem -14 -3)
  -2
  ACL2 !>(rem -15 -3)
  0
  ACL2 !>
  ~ev[]
  ~c[(Rem i j)] is that number ~c[k] for which ~c[(* j (truncate i j))] added
  to ~c[k] equals ~c[i].~/

  The ~il[guard] for ~c[(rem i j)] requires that ~c[i] and ~c[j] are rational
  (~il[real], in ACL2(r)) numbers and ~c[j] is non-zero.

  ~c[Rem] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (declare (xargs :guard (and (real/rationalp x)
                              (real/rationalp y)
                              (not (eql y 0)))))
  (- x (* (truncate x y) y)))

(defun evenp (x)

  ":Doc-Section ACL2::Programming

  test whether an integer is even~/

  ~c[(evenp x)] is true if and only if the integer ~c[x] is even.
  Actually, in the ACL2 logic ~c[(evenp x)] is defined to be true when
  ~c[x/2] is an integer.~/

  The ~il[guard] for ~c[evenp] requires its argument to be an integer.

  ~c[Evenp] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (integerp x)))
  (integerp (* x (/ 2))))

(defun oddp (x)

  ":Doc-Section ACL2::Programming

  test whether an integer is odd~/

  ~c[(oddp x)] is true if and only if ~c[x] is odd, i.e., not even in
  the sense of ~ilc[evenp].~/

  The ~il[guard] for ~c[oddp] requires its argument to be an integer.

  ~c[Oddp] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (integerp x)))
  (not (evenp x)))

(defun zerop (x)
  (declare (xargs :mode :logic
                  :guard (acl2-numberp x)))

  ":Doc-Section ACL2::Programming

  test an acl2-number against 0~/

  ~c[(zerop x)] is ~c[t] if ~c[x] is ~c[0] and is ~c[nil] otherwise.  Thus,
  it is logically equivalent to ~c[(equal x 0)].~/

  ~c[(Zerop x)] has a ~il[guard] requiring ~c[x] to be numeric and can be
  expected to execute more efficiently than ~c[(equal x 0)] in properly
  ~il[guard]ed compiled code.

  In recursions down the natural numbers, ~c[(zp x)] is preferred over
  ~c[(zerop x)] because the former coerces ~c[x] to a natural and allows
  the termination proof.  In recursions through the integers,
  ~c[(zip x)] is preferred.  ~l[zero-test-idioms].

  ~c[Zerop] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (eql x 0))

;; RAG - Only the guard changed here.

(defun plusp (x)

  ":Doc-Section ACL2::Programming

  test whether a number is positive~/

  ~c[(Plusp x)] is true if and only if ~c[x > 0].~/

  The ~il[guard] of ~c[plusp] requires its argument to be a rational (~il[real], in
  ACL2(r)) number.

  ~c[Plusp] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :mode :logic
                  :guard (real/rationalp x)))
  (> x 0))

;; RAG - Only the guard changed here.

(defun minusp (x)

  ":Doc-Section ACL2::Programming

  test whether a number is negative~/

  ~c[(Minusp x)] is true if and only if ~c[x < 0].~/

  The ~il[guard] of ~c[minusp] requires its argument to be a rational (~il[real], in
  ACL2(r)) number.

  ~c[Minusp] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :mode :logic
                  :guard (real/rationalp x)))
  (< x 0))

;; RAG - Only the guard changed here.

(defun min (x y)

  ":Doc-Section ACL2::Programming

  the smaller of two numbers~/

  ~c[(Min x y)] is the smaller of the numbers ~c[x] and ~c[y].~/

  The ~il[guard] for ~c[min] requires its arguments to be rational (~il[real],
  in ACL2(r)) numbers.

  ~c[Min] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (declare (xargs :guard (and (real/rationalp x)
                              (real/rationalp y))))
  (if (< x y)
      x
    y))

;; RAG - Only the guard changed here.

(defun max (x y)

  ":Doc-Section ACL2::Programming

  the larger of two numbers~/

  ~c[(Max x y)] is the larger of the numbers ~c[x] and ~c[y].~/

  The ~il[guard] for ~c[max] requires its arguments to be rational (~il[real],
  in ACL2(r)) numbers.

  ~c[Max] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (declare (xargs :guard (and (real/rationalp x)
                              (real/rationalp y))))
  (if (> x y)
      x
    y))

;; RAG - Only the guard changed here.  The docstring below says that
;; abs must not be used on complex arguments, since that could result
;; in a non-ACL2 object.

(defun abs (x)

  ":Doc-Section ACL2::Programming

  the absolute value of a real number~/

  ~c[(Abs x)] is ~c[-x] if ~c[x] is negative and is ~c[x] otherwise.~/

  The ~il[guard] for ~c[abs] requires its argument to be a rational (~il[real],
  in ACL2(r)) number.

  ~c[Abs] is a Common Lisp function.  See any Common Lisp documentation
  for more information.

  From ``Common Lisp the Language'' page 205, we must not allow
  complex ~c[x] as an argument to ~c[abs] in ACL2, because if we did we
  would have to return a number that might be a floating point number
  and hence not an ACL2 object.~/"

  (declare (xargs :guard (real/rationalp x)))

  (if (minusp x) (- x) x))

(defun signum (x)

  ":Doc-Section ACL2::Programming

  indicator for positive, negative, or zero~/

  ~c[(Signum x)] is ~c[0] if ~c[x] is ~c[0], ~c[-1] if ~c[x] is negative,
  and is ~c[1] otherwise.~/

  The ~il[guard] for ~c[signum] requires its argument to be rational (~il[real], in
  ACL2(r)) number.

  ~c[Signum] is a Common Lisp function.  See any Common Lisp
  documentation for more information.

  From ``Common Lisp the Language'' page 206, we see a definition of
  ~c[signum] in terms of ~ilc[abs].  As explained elsewhere
  (~pl[abs]), the ~il[guard] for ~ilc[abs] requires its argument to be a
  rational (~il[real], in ACL2(r))  number; hence, we make the same
  restriction for ~c[signum].~/"

  (declare (xargs :guard (real/rationalp x)))

; On CLTL p. 206 one sees the definition

; (if (zerop x) x (* x (/ (abs x)))).

; However, that suffers because it looks to type-set like it returns
; an arbitrary rational when in fact it returns -1, 0, or 1.  So we
; give a more explicit definition.  See the doc string in abs for a
; justification for disallowing complex arguments.

  (if (zerop x) 0
      (if (minusp x) -1 +1)))

(defun lognot (i)

  ":Doc-Section ACL2::Programming

  bitwise not of a two's complement number~/

  ~c[(lognot i)] is the two's complement bitwise ~c[`not'] of the integer ~c[i].~/

  ~c[Lognot] is actually defined by coercing its argument to an integer
  (~pl[ifix]), negating the result, and then subtracting ~c[1].

  The ~il[guard] for ~c[lognot] requires its argument to be an integer.

  ~c[Lognot] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (integerp i)))
  (+ (- (ifix i)) -1))

; This function is introduced now because we need it in the admission of
; logand.  The admission of o-p could be moved up to right
; after the introduction of the "and" macro.

)

(defthm standard-char-p-nth
  (implies (and (standard-char-listp chars)
                (<= 0 i)
                (< i (len chars)))
           (standard-char-p (nth i chars)))
  :hints (("Goal" :in-theory (enable standard-char-listp))))

(verify-termination-boot-strap (string-equal1
                     (declare (xargs :measure (nfix (- maximum (nfix i)))))))
(verify-termination-boot-strap string-equal)
(verify-termination-boot-strap assoc-string-equal)
(verify-termination-boot-strap xxxjoin)

(deflabel proof-of-well-foundedness
  :doc
  ":Doc-Section Miscellaneous

  a proof that ~ilc[o<] is well-founded on ~ilc[o-p]s~/

  The soundness of ACL2 rests in part on the well-foundedness of ~ilc[o<] on
  ~ilc[o-p]s.  This can be taken as obvious if one is willing to grant that
  those concepts are simply encodings of the standard mathematical notions of
  the ordinals below ~c[epsilon-0] and its natural ordering relation.  But it
  is possible to prove that ~ilc[o<] is well-founded on ~ilc[o-p]s without
  having to assert any connection to the ordinals and that is what we do here.
  The book ~c[books/ordinals/proof-of-well-foundedness] carries out the proof
  outlined below in ACL2, using only that the natural numbers are
  well-founded.~/

  Before outlining the above mentioned proof, we note that in the analogous
  documentation page of ACL2 Version_2.7, there is a proof of the
  well-foundedness of ~c[e0-ord-<] on ~c[e0-ordinalp]s, the less-than relation
  and recognizer for the old ordinals (that is, for the ordinals appearing in
  ACL2 up through that version).  Manolios and Vroon have given a proof in ACL2
  Version_2.7 that the current ordinals (based on ~ilc[o<] and ~ilc[o-p]) are
  order-isomorphic to the old ordinals (based on ~c[e0-ord-<] and
  ~c[e0-ordinalp]).  Their proof establishes that switching from the old
  ordinals to the current ordinals preserves the soundness of ACL2.  For
  details see their paper:
  ~bf[]
  Manolios, Panagiotis & Vroon, Daron.
  Ordinal arithmetic in ACL2.
  Kaufmann, Matt, & Moore, J Strother (eds).
  Fourth International Workshop on the ACL2 Theorem
  Prover and Its Applications (ACL2-2003),
  July, 2003.
  See ~url[http://www.cs.utexas.edu/users/moore/acl2/workshop-2003/].
  ~ef[]

  We now give an outline of the above mentioned proof of well-foundedness.  We
  first observe three facts about ~ilc[o<] on ordinals that have been proved by
  ACL2 using only structural induction on lists.  These theorems can be proved
  by hand.
  ~bv[]
  (defthm transitivity-of-o<
    (implies (and (o< x y)
                  (o< y z))
             (o< x z))
    :rule-classes nil)

  (defthm non-circularity-of-o<
    (implies (o< x y)
             (not (o< y x)))
    :rule-classes nil)

  (defthm trichotomy-of-o<
    (implies (and (o-p x)
                  (o-p y))
             (or (equal x y)
                 (o< x y)
                 (o< y x)))
    :rule-classes nil)
  ~ev[]
  These three properties establish that ~ilc[o<] orders the
  ~ilc[o-p]s.  To put such a statement in the most standard
  mathematical nomenclature, we can define the macro:
  ~bv[]
  (defmacro o<= (x y)
    `(not (o< ,y ,x)))
  ~ev[]
  and then establish that ~c[o<=] is a relation that is a simple,
  complete (i.e., total) order on ordinals by the following three
  lemmas, which have been proved:
  ~bv[]
  (defthm antisymmetry-of-o<=
    (implies (and (o-p x)
                  (o-p y)
                  (o<= x y)
                  (o<= y x))
             (equal x y))
    :rule-classes nil
    :hints ((\"Goal\" :use non-circularity-of-o<)))

  (defthm transitivity-of-o<=
    (implies (and (o-p x)
                  (o-p y)
                  (o<= x y)
                  (o<= y z))
             (o<= x z))
    :rule-classes nil
    :hints ((\"Goal\" :use transitivity-of-o<)))

  (defthm trichotomy-of-o<=
    (implies (and (o-p x)
                  (o-p y))
             (or (o<= x y)
                 (o<= y x)))
    :rule-classes nil
    :hints ((\"Goal\" :use trichotomy-of-o<)))
  ~ev[]
  Crucially important to the proof of the well-foundedness of
  ~ilc[o<] on ~ilc[o-p]s is the concept of ordinal-depth,
  abbreviated ~c[od]:
  ~bv[]
  (defun od (l)
    (if (o-finp l)
        0
      (1+ (od (o-first-expt l)))))
  ~ev[]
  If the ~c[od] of an ~ilc[o-p] ~c[x] is smaller than that of an
  ~ilc[o-p] ~c[y], then ~c[x] is ~ilc[o<] ~c[y]:
  ~bv[]
  (defun od-1 (x y)
    (if (o-finp x)
        (list x y)
      (od-1 (o-first-expt x) (o-first-expt y))))

  (defthm od-implies-ordlessp
    (implies (and (o-p x)
                  (< (od x) (od y)))
             (o< x y))
    :hints ((\"Goal\"
             :induct (od-1 x y))))
  ~ev[]
  Remark.  A consequence of this lemma is the fact that if ~c[s = s(1)],
  ~c[s(2)], ... is an infinite, ~ilc[o<] descending sequence of ~ilc[o-p]s, then
  ~c[od(s(1))], ~c[od(s(2))], ... is a ``weakly'' descending sequence of
  non-negative integers: ~c[od(s(i))] is greater than or equal to
  ~c[od(s(i+1))].

  ~em[Lemma Main.]  For each non-negative integer ~c[n], ~ilc[o<] well-orders
  the set of ~ilc[o-p]s with ~c[od] less than or equal to ~c[n] .
  ~bv[]
   Base Case.  n = 0.  The o-ps with 0 od are the non-negative
   integers.  On the non-negative integers, o< is the same as <.

   Induction Step.  n > 0.  We assume that o< well-orders the
   o-ps with od less than n.

     If o< does not well-order the o-ps with od less than or equal to n,
     consider, D, the set of infinite, o< descending sequences of o-ps of od
     less than or equal to n.  The first element of a sequence in D has od n.
     Therefore, the o-first-expt of the first element of a sequence in D has od
     n-1.  Since o<, by IH, well-orders the o-ps with od less than n, the set
     of o-first-expts of first elements of the sequences in D has a minimal
     element, which we denote by B and which has od of n-1.

     Let k be the minimum integer such that for some infinite, o< descending
     sequence s of o-ps with od less than or equal to n, the first element of s
     has an o-first-expt of B and an o-first-coeff of k.  Notice that k is
     positive.

     Having fixed B and k, let s = s(1), s(2), ... be an infinite, o<
     descending sequence of o-ps with od less than or equal to n such that s(1)
     has a o-first-expt of B and an o-first-coeff of k.

     We show that each s(i) has a o-first-expt of B and an o-first-coeff of
     k. For suppose that s(j) is the first member of s either with o-first-expt
     B and o-first-coeff m (m neq k) or with o-first-expt B' and o-first-coeff
     B' (B' neq B). If (o-first-expt s(j)) = B', then B' has od n-1 (otherwise,
     by IH, s would not be infinite) and B' is o< B, contradicting the
     minimality of B. If 0 < m < k, then the fact that the sequence beginning
     at s(j) is infinitely descending contradicts the minimality of k. If m >
     k, then s(j) is greater than its predecessor; but this contradicts the
     fact that s is descending.

     Thus, by the definition of o<, for s to be a decreasing sequence of o-ps,
     (o-rst s(1)), (o-rst s(2)), ... must be a decreasing sequence. We end by
     showing this cannot be the case. Let t = t(1), t(2), ... be an infinite
     sequence of o-ps such that t(i) = (o-rst s(i)). Then t is infinitely
     descending. Furthermore, t(1) begins with an o-p B' that is o< B. Since t
     is in D, t(1) has od n, therefore, B' has od n-1. But this contradicts the
     minimality of B. Q.E.D.
  ~ev[]
  Theorem.  ~ilc[o<] well-orders the ~ilc[o-p]s.  Proof.  Every
  infinite,~c[ o<] descending sequence of ~ilc[o-p]s has the
  property that each member has ~c[od] less than or equal to the
  ~c[od], ~c[n], of the first member of the sequence.  This
  contradicts Lemma Main.
  Q.E.D.")

#+acl2-loop-only
(progn

(defun expt (r i)

  ":Doc-Section ACL2::Programming

  exponential function~/

  ~c[(Expt r i)] is the result of raising the number ~c[r] to the
  integer power ~c[i].~/

  The ~il[guard] for ~c[(expt r i)] is that ~c[r] is a number and ~c[i]
  is an integer, and furthermore, if ~c[r] is ~c[0] then ~c[i] is
  nonnegative.  When the type requirements of the ~il[guard] aren't
  met, ~c[(expt r i)] first coerces ~c[r] to a number and ~c[i] to an
  integer.

  ~c[Expt] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  Note that ~c[r] can be a complex
  number; this is consistent with Common lisp.~/"

; CLtL2 (page 300) allows us to include complex rational arguments.

  (declare (xargs :guard (and (acl2-numberp r)
                              (integerp i)
                              (not (and (eql r 0) (< i 0))))
                  :measure (abs (ifix i))))
  (cond ((zip i) 1)
        ((= (fix r) 0) 0)
        ((> i 0) (* r (expt r (+ i -1))))
        (t (* (/ r) (expt r (+ i +1))))))

(defun logcount (x)

  ":Doc-Section ACL2::Programming

  number of ``on'' bits in a two's complement number~/

  ~c[(Logcount x)] is the number of ``on'' bits in the two's complement
  representation of ~c[x].~/

  ~c[(Logcount x)] has a ~il[guard] of ~c[(integerp x)].

  ~c[Logcount] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (integerp x)))
  (cond
   ((zip x)
    0)
   ((< x 0)
    (logcount (lognot x)))
   ((evenp x)
    (logcount (nonnegative-integer-quotient x 2)))
   (t
    (1+ (logcount (nonnegative-integer-quotient x 2))))))

(defun nthcdr (n l)

  ":Doc-Section ACL2::Programming

  final segment of a list~/

  ~c[(Nthcdr n l)] removes the first ~c[n] elements from the list ~c[l].~/

  The following is a theorem.
  ~bv[]
  (implies (and (integerp n)
                (<= 0 n)
                (true-listp l))
           (equal (length (nthcdr n l))
                  (if (<= n (length l))
                      (- (length l) n)
                    0)))
  ~ev[]
  For related functions, ~pl[take] and ~pl[butlast].

  The ~il[guard] of ~c[(nthcdr n l)] requires that ~c[n] is a nonnegative
  integer and ~c[l] is a true list.

  ~c[Nthcdr] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (integerp n)
                              (<= 0 n)
                              (true-listp l))))
  (if (zp n)
      l
    (nthcdr (+ n -1) (cdr l))))

(defun logbitp (i j)

  ":Doc-Section ACL2::Programming

  the ~c[i]th bit of an integer~/

  For a nonnegative integer ~c[i] and an integer ~c[j], ~c[(logbitp i j)]
  is the value of the ~c[i]th bit in the two's complement
  representation of ~c[j].~/

  ~c[(Logbitp i j)] has a ~il[guard] that ~c[i] is a nonnegative integer and
  ~c[j] is an integer.

  ~c[Logbitp] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (integerp j)
                              (integerp i)
                              (>= i 0))
                  :mode :program))
  (oddp (floor (ifix j) (expt 2 (nfix i)))))

(defun ash (i c)

  ":Doc-Section ACL2::Programming

  arithmetic shift operation~/

  ~c[(ash i c)] is the result of taking the two's complement
  representation of the integer ~c[i] and shifting it by ~c[c] bits:  shifting
  left and padding with ~c[c] ~c[0] bits if ~c[c] is positive, shifting right and
  dropping ~c[(abs c)] bits if ~c[c] is negative, and simply returning ~c[i] if ~c[c]
  is ~c[0].~/

  The ~il[guard] for ~c[ash] requires that its arguments are integers.

  ~c[Ash] is a Common Lisp function.  See any Common Lisp documentation
  for more information.~/"

  (declare (xargs :guard (and (integerp i)
                              (integerp c))
                  :mode :program))
  (floor (* (ifix i) (expt 2 c)) 1))

)

; John Cowles first suggested a version of the following lemma for rationals.

(defthm expt-type-prescription-non-zero-base
  (implies (and (acl2-numberp r)
                (not (equal r 0)))
           (not (equal (expt r i) 0)))
  :rule-classes :type-prescription)

;; RAG - I added the following lemma, similar to the rational case.

#+:non-standard-analysis
(defthm realp-expt-type-prescription
  (implies (realp r)
           (realp (expt r i)))
  :rule-classes :type-prescription)

(defthm rationalp-expt-type-prescription
  (implies (rationalp r)
           (rationalp (expt r i)))
  :rule-classes :type-prescription)

(verify-termination-boot-strap logbitp)

(verify-termination-boot-strap ash)

(deflabel characters
  :doc
  ":Doc-Section ACL2::Programming

  characters in ACL2~/

  ACL2 accepts 256 distinct characters, which are the characters
  obtained by applying the function ~ilc[code-char] to each integer from ~c[0]
  to ~c[255].  Among these, Common Lisp designates certain ones as
  ~em[standard characters], namely those of the form ~c[(code-char n)]
  where ~c[n] is from ~c[33] to ~c[126], together with ~c[#\\Newline] and ~c[#\\Space].  The
  actual standard characters may be viewed by evaluating the
  ~ilc[defconst] ~c[*standard-chars*].~/

  To be more precise, Common Lisp does not specify the precise
  relationship between ~ilc[code-char] and the standard characters.
  However, we check that the underlying Common Lisp implementation
  uses a particular relationship that extends the usual ASCII coding
  of characters.  We also check that Space, Tab, Newline, Page, and
  Rubout correspond to characters with respective ~ilc[char-code]s ~t[32], ~t[9],
  ~t[10], ~t[12], and ~t[127].

  ~ilc[Code-char] has an inverse, ~ilc[char-code].  Thus, when ~ilc[char-code] is
  applied to an ACL2 character, ~c[c], it returns a number ~c[n] between ~c[0] and
  ~c[255] inclusive such that ~c[(code-char n)] = ~c[c].

  The preceding paragraph implies that there is only one ACL2
  character with a given character code.  CLTL allows for
  ``attributes'' for characters, which could allow distinct characters
  with the same code, but ACL2 does not allow this.

  ~em[The Character Reader]

  ACL2 supports the `~c[#\\]' notation for characters provided by Common
  Lisp, with some restrictions.  First of all, for every character ~c[c],
  the notation
  ~bv[]
  #\\c
  ~ev[]
  may be used to denote the character object ~c[c].  That is, the user may
  type in this notation and ACL2 will read it as denoting the
  character object ~c[c].  In this case, the character immediately
  following ~c[c] must be one of the following ``terminating characters'':
  a Tab, a Newline, a Page character, a space, or one of the
  characters:
  ~bv[]
  \"  '  (  )  ;  `  ,
  ~ev[]
  Other than the notation above, ACL2 accepts alternate notation for
  five characters.
  ~bv[]
  #\\Space
  #\\Tab
  #\\Newline
  #\\Page
  #\\Rubout
  ~ev[]

  Again, in each of these cases the next character must be from among
  the set of ``terminating characters'' described in the
  single-character case.  Our implementation is consistent with
  IS0-8859, even though we don't provide ~c[#\\] syntax for entering
  characters other than that described above.

  Finally, we note that it is our intention that any object printed by
  ACL2's top-level-loop may be read back into ACL2.  Please notify the
  implementors if you find a counterexample to this claim.~/")

(defaxiom char-code-linear

; The other properties that we might be tempted to state here,
; (integerp (char-code x)) and (<= 0 (char-code x)), are taken care of by
; type-set-char-code.

  (< (char-code x) 256)
  :rule-classes :linear)

(defaxiom code-char-type
  (characterp (code-char n))
  :rule-classes :type-prescription)

(defaxiom code-char-char-code-is-identity
  (implies (force (characterp c))
           (equal (code-char (char-code c)) c)))

(defaxiom char-code-code-char-is-identity
  (implies (and (force (integerp n))
                (force (<= 0 n))
                (force (< n 256)))
           (equal (char-code (code-char n)) n)))

#+acl2-loop-only
(defun char< (x y)

  ":Doc-Section ACL2::Programming

  less-than test for ~il[characters]~/

  ~c[(char< x y)] is true if and only if the character code of ~c[x] is
  less than that of ~c[y].  ~l[char-code].~/

  The ~il[guard] for ~c[char<] specifies that its arguments are ~il[characters].

  ~c[Char<] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (characterp x) (characterp y))))
  (< (char-code x) (char-code y)))

#+acl2-loop-only
(defun char> (x y)

  ":Doc-Section ACL2::Programming

  greater-than test for ~il[characters]~/

  ~c[(char> x y)] is true if and only if the character code of ~c[x] is
  greater than that of ~c[y].  ~l[char-code].~/

  The ~il[guard] for ~c[char>] specifies that its arguments are ~il[characters].

  ~c[Char>] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (characterp x) (characterp y))))
  (> (char-code x) (char-code y)))

#+acl2-loop-only
(defun char<= (x y)

  ":Doc-Section ACL2::Programming

  less-than-or-equal test for ~il[characters]~/

  ~c[(char<= x y)] is true if and only if the character code of ~c[x] is
  less than or equal to that of ~c[y].  ~l[char-code].~/

  The ~il[guard] for ~c[char<=] specifies that its arguments are ~il[characters].

  ~c[Char<=] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (characterp x) (characterp y))))
  (<= (char-code x) (char-code y)))

#+acl2-loop-only
(defun char>= (x y)

  ":Doc-Section ACL2::Programming

  greater-than-or-equal test for ~il[characters]~/

  ~c[(char>= x y)] is true if and only if the character code of ~c[x] is
  greater than or equal to that of ~c[y].  ~l[char-code].~/

  The ~il[guard] for ~c[char>=] specifies that its arguments are ~il[characters].

  ~c[Char>=] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (characterp x) (characterp y))))
  (>= (char-code x) (char-code y)))

(defun string<-l (l1 l2 i)
  (declare (xargs :guard (and (character-listp l1)
                              (character-listp l2)
                              (integerp i))))
  (cond ((endp l1)
         (cond ((endp l2) nil)
               (t i)))
        ((endp l2) nil)
        ((eql (car l1) (car l2))
         (string<-l (cdr l1) (cdr l2) (+ i 1)))
        ((char< (car l1) (car l2)) i)
        (t nil)))

#+acl2-loop-only
(defun string< (str1 str2)

  ":Doc-Section ACL2::Programming

  less-than test for strings~/

  ~c[(String< str1 str2)] is non-~c[nil] if and only if the string
  ~c[str1] precedes the string ~c[str2] lexicographically, where
  character inequalities are tested using ~ilc[char<].  When non-~c[nil],
  ~c[(string< str1 str2)] is the first position (zero-based) at which
  the strings differ.  Here are some examples.
  ~bv[]
  ACL2 !>(string< \"abcd\" \"abu\")
  2
  ACL2 !>(string< \"abcd\" \"Abu\")
  NIL
  ACL2 !>(string< \"abc\" \"abcde\")
  3
  ACL2 !>(string< \"abcde\" \"abc\")
  NIL
  ~ev[]
  ~/

  The ~il[guard] for ~c[string<] specifies that its arguments are strings.

  ~c[String<] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (stringp str1)
                              (stringp str2))))
  (string<-l (coerce str1 'list)
             (coerce str2 'list)
             0))

#+acl2-loop-only
(defun string> (str1 str2)

  ":Doc-Section ACL2::Programming

  greater-than test for strings~/

  ~c[(String> str1 str2)] is non-~c[nil] if and only if ~c[str2] precedes
  ~c[str1] lexicographically.  When non-~c[nil], ~c[(string> str1 str2)]
  is the first position (zero-based) at which the strings differ.
  ~l[string<].~/

  ~c[String>] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (stringp str1)
                              (stringp str2))))
  (string< str2 str1))

#+acl2-loop-only
(defun string<= (str1 str2)

  ":Doc-Section ACL2::Programming

  less-than-or-equal test for strings~/

  ~c[(String<= str1 str2)] is non-~c[nil] if and only if the string
  ~c[str1] precedes the string ~c[str2] lexicographically or the strings
  are equal.  When non-~c[nil], ~c[(string<= str1 str2)] is the first
  position (zero-based) at which the strings differ, if they differ,
  and otherwise is their common length.  ~l[string<].~/

  The ~il[guard] for ~c[string<=] specifies that its arguments are strings.

  ~c[String<=] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (stringp str1)
                              (stringp str2))))
  (if (equal str1 str2)
      (length str1)
    (string< str1 str2)))

#+acl2-loop-only
(defun string>= (str1 str2)

  ":Doc-Section ACL2::Programming

  less-than-or-equal test for strings~/

  ~c[(String>= str1 str2)] is non-~c[nil] if and only if the string
  ~c[str2] precedes the string ~c[str1] lexicographically or the strings
  are equal.  When non-~c[nil], ~c[(string>= str1 str2)] is the first
  position (zero-based) at which the strings differ, if they differ,
  and otherwise is their common length.  ~l[string>].~/

  The ~il[guard] for ~c[string>=] specifies that its arguments are strings.

  ~c[String>=] is a Common Lisp function.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (stringp str1)
                              (stringp str2))))
  (if (equal str1 str2)
      (length str1)
    (string> str1 str2)))

(defun symbol-< (x y)

  ":Doc-Section ACL2::Programming

  less-than test for symbols~/

  ~c[(symbol-< x y)] is non-~c[nil] if and only if either the
  ~ilc[symbol-name] of the symbol ~c[x] lexicographially precedes the
  ~ilc[symbol-name] of the symbol ~c[y] (in the sense of ~ilc[string<]) or
  else the ~ilc[symbol-name]s are equal and the ~ilc[symbol-package-name] of
  ~c[x] lexicographically precedes that of ~c[y] (in the same sense).
  So for example, ~c[(symbol-< 'abcd 'abce)] and
  ~c[(symbol-< 'acl2::abcd 'foo::abce)] are true.~/

  The ~il[guard] for ~c[symbol] specifies that its arguments are symbols.~/"

  (declare (xargs :guard (and (symbolp x) (symbolp y))))
  (let ((x1 (symbol-name x))
        (y1 (symbol-name y)))
    (or (string< x1 y1)
        (and (equal x1 y1)
             (string< (symbol-package-name x)
                      (symbol-package-name y))))))

(defthm string<-l-irreflexive
  (not (string<-l x x i)))

(defthm string<-irreflexive
  (not (string< s s)))

(defun substitute-ac (new old seq acc)
  (declare (xargs :guard (and (true-listp acc)
                              (true-listp seq)
                              (or (eqlablep old)
                                  (eqlable-listp seq)))))
  (cond
   ((endp seq)
    (reverse acc))
   ((eql old (car seq))
    (substitute-ac new old (cdr seq) (cons new acc)))
   (t
    (substitute-ac new old (cdr seq) (cons (car seq) acc)))))

#+acl2-loop-only
(defun substitute (new old seq)

  ":Doc-Section ACL2::Programming

  substitute into a string or a list, using ~ilc[eql] as test~/

  ~c[(Substitute new old seq)] is the result of replacing each occurrence
  of ~c[old] in ~c[seq], which is a list or a string, with ~c[new].~/

  The guard for ~c[substitute] requires that either ~c[seq] is a string and
  ~c[new] is a character, or else:  ~c[seq] is a ~ilc[true-listp] such that either
  all of its members are ~ilc[eqlablep] or ~c[old] is ~c[eqlablep].

  ~c[Substitute] is a Common Lisp function.  See any Common Lisp
  documentation for more information.  Since ACL2 functions cannot
  take keyword arguments (though macros can), the test used in
  ~c[substitute] is ~c[eql].~/"

  (declare (xargs :guard (or (and (stringp seq)
                                  (characterp new))
                             (and (true-listp seq)
                                  (or (eqlablep old)
                                      (eqlable-listp seq))))

; Wait for state-global-let* to be defined, so that we can provide a
; local lemma.

                  :verify-guards nil))
  (if (stringp seq)
      (coerce (substitute-ac new old (coerce seq 'list) nil)
              'string)
    (substitute-ac new old seq nil)))

#+acl2-loop-only
(defun subsetp (x y)

  ":Doc-Section ACL2::Programming

  test if every ~ilc[member] of one list is a ~ilc[member] of the other~/

  ~c[(Subsetp x y)] is true if and only if every member of the list
  ~c[x] is a member of the list ~c[y].~/

  Membership is tested using the function ~ilc[member].  Thus, the ~il[guard]
  for ~c[subsetp] requires that its arguments are true lists, and
  moreover, at least one satisfies ~ilc[eqlable-listp].  This ~il[guard]
  ensures that the ~il[guard] for ~ilc[member] will be met for each call
  generated by ~c[subsetp].

  ~c[Subsetp] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard
                  (if (eqlable-listp y)
                      (true-listp x)
                    (if (eqlable-listp x)
                        (true-listp y)
                      nil))))
  (cond ((endp x) t)
        (t (and (member (car x) y)
                (subsetp (cdr x) y)))))

#+acl2-loop-only
(defun sublis (alist tree)

  ":Doc-Section ACL2::Programming

  substitute an alist into a tree~/

  ~c[(Sublis alist tree)] is obtained by replacing every leaf of
  ~c[tree] with the result of looking that leaf up in the association
  list ~c[alist].  However, a leaf is left unchanged if it is not found
  as a key in ~c[alist].~/

  Leaves are lookup up using the function ~ilc[assoc].  The ~il[guard] for
  ~c[(sublis alist tree)] requires ~c[(eqlable-alistp alist)].  This
  ~il[guard] ensures that the ~il[guard] for ~ilc[assoc] will be met for each
  lookup generated by ~c[sublis].

  ~c[Sublis] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (eqlable-alistp alist)))
  (cond ((atom tree)
         (let ((pair (assoc tree alist)))
           (cond (pair (cdr pair))
                 (t tree))))
        (t (cons (sublis alist (car tree))
                 (sublis alist (cdr tree))))))

#+acl2-loop-only
(defun subst (new old tree)

  ":Doc-Section ACL2::Programming

  a single substitution into a tree~/

  ~c[(Subst new old tree)] is obtained by substituting ~c[new] for every
  occurence of ~c[old] in the given tree.~/

  Equality to ~c[old] is determined using the function ~ilc[eql].  The
  ~il[guard] for ~c[(subst new old tree)] requires ~c[(eqlablep old)], which
  ensures that the ~il[guard] for ~ilc[eql] will be met for each comparison
  generated by ~c[subst].

  ~c[Subst] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (eqlablep old)))
  (cond ((eql old tree) new)
        ((atom tree) tree)
        (t (cons (subst new old (car tree))
                 (subst new old (cdr tree))))))

(defmacro pprogn (&rest lst)

  ":Doc-Section ACL2::Programming

  evaluate a sequence of forms that return ~il[state]~/
  ~bv[]
  Example Form:
  (pprogn
   (cond ((or (equal (car l) #\\) (equal (car l) slash-char))
          (princ$ #\\ channel state))
         (t state))
   (princ$ (car l) channel state)
   (mv (cdr l) state))
  ~ev[]
  The convention for ~c[pprogn] usage is to give it a non-empty
  sequence of forms, each of which (except possibly for the last)
  returns state (~pl[state]) as its only value.  The ~il[state] returned by
  each but the last is passed on to the next.  The value or values of
  the last form are returned as the value of the ~c[pprogn].

  If you are using single-threaded objects you may wish to define an
  analogue of this function for your own ~il[stobj].~/

  General Form:
  ~bv[]
  (PPROGN form1
          form2
          ...
          formk
          result-form)
  ~ev[]
  This general form is equivalent, via macro expansion, to:
  ~bv[]
  (LET ((STATE form1))
       (LET ((STATE form2))
            ...
            (LET ((STATE formk))
                 result-form)))
  ~ev[]
  ~/"

  (declare (xargs :guard (and lst
                              (true-listp lst))))
  (cond ((endp (cdr lst)) (car lst))
        #-acl2-loop-only

; The next case avoids compiler warnings from (pprogn .... (progn! ...)).  Note
; that progn! in raw Lisp binds state to *the-live-state*, and hence shadows
; superior bindings of state.  We are tempted to check that the last form
; starts with progn!, but of course it could be a macro call that expands to a
; call of progn!, so we make no such check.

        ((endp (cddr lst))
         (list 'let
               (list (list 'STATE (car lst)))
               '(DECLARE (IGNORABLE STATE))
               (cadr lst)))
        (t (list 'let
                 (list (list 'STATE (car lst)))
                 (cons 'pprogn (cdr lst))))))


; The Unwind-Protect Essay

; We wish to define an ACL2 macro form:

; (acl2-unwind-protect "expl" body cleanup1 cleanup2)

; with the following logical semantics

; (mv-let (erp val state)
;         ,body
;         (cond (erp (pprogn ,cleanup1 (mv erp val state)))
;               (t   (pprogn ,cleanup2 (mv erp val state)))))

; The idea is that it returns the 3 results of evaluating body except before
; propagating those results upwards it runs one of the two cleanup forms,
; depending on whether the body signalled an error.  The cleanup forms return
; state.  In typical use the cleanup forms restore the values of state global
; variables that were "temporarily" set by body.  [Note that the "expl"
; is a string and it is always ignored.  Its only use is to tag the elements
; of the stacks in the frames of *acl2-unwind-protect-stack* so that debugging
; is easier.  None of our code actually looks at it.]

; In addition, we want acl2-unwind-protect to handle aborts caused by the user
; during the processing of body and we want ev to handle acl2-unwind-protect
; "properly" in a sense discussed later.

; We deal first with the notion of the "proper" way to handle aborts.  Because
; of the way acl2-unwind-protect is used, namely to "restore" a "temporarily"
; smashed state, aborts during body should not prevent the execution of the
; cleanup code.  Intuitively, the compiled form of an acl2-unwind-protect
; ought to involve a Common Lisp unwind-protect.  In fact, it does not, for
; reasons developed below.  But it is easier to think about the correctness of
; our implementation if we start by thinking in terms of using a raw lisp
; unwind-protect in the macroexpansion of each acl2-unwind-protect.

; The (imagined) unwind-protect is almost always irrelevant because "errors"
; signalled by body are in fact not Lisp errors.  But should the user cause an
; abort during body, the unwind-protect will ensure that cleanup1 is executed.
; This is a logically arbitrary choice; we might have said cleanup2 is
; executed.  By "ensure" we mean not only will the Lisp unwind-protect fire
; the cleanup code even though body was aborted; we mean that the cleanup code
; will be executed without possibility of abort.  Now there is no way to
; disable interrupts in CLTL.  But if we make sufficient assumptions about the
; cleanup forms then we can effectively disable interrupts by executing each
; cleanup form repeatedly until it is executed once without being aborted.  We
; might define "idempotency" to be just the necessary property: the repeated
; (possibly partial) execution of the form, followed by a complete execution
; of the form, produces the same state as a single complete execution.  For
; example, (f-put-global 'foo 'old-val state) is idempotent but (f-put-global
; 'foo (1- (get-global 'foo state)) state) is not.  Cleanup1 should be idempotent
; to ensure that our implementation of unwind protect in the face of aborts is
; correct with respect to the (non-logical) semantics we have described.
; Furthermore, it bears pointing out that cleanup1 might be called upon to undo
; the work of a "partial" execution of cleanup2!  This happens if the body
; completes normally and without signalling an error, cleanup2 is undertaken,
; and then the user aborts.  So the rule is that if an abort occurs during an
; acl2-unwind-protect, cleanup1 is executed without interrupts.

; What, pray, gives us the freedom to give arbitrary semantics to
; acl2-unwind-protect in the face of an abort?  We regard an abort as akin to
; unplugging the machine and plugging it back in.  One should be thankful for
; any reasonable behavior and not quibble over whether it is the "logical" one
; or whether one ought to enforce informal rules like idempotency.  Thus, we
; are not terribly sympathetic to arguments that this operational model is
; inconsistent with ACL2 semantics when the user types "Abort!" or doesn't
; understand unenforced assumptions about his cleanup code.  All logical bets
; are off the moment the user types "Abort!".  This model has the saving grace
; that we can implement it and that it can be used within the ACL2 system code
; to implement what we need during abort recovery.  The operational model of
; an abort is that the machine finds the innermost acl2-unwind-protect, rips
; out of the execution of its body (or its cleanup code), executes the
; cleanup1 code with all aborts disabled and then propagates the abort upward.

; Now unfortunately this operational model cannot be implemented
; entirely locally in the compilation of an acl2-unwind-protect.
; Operationally, (acl2-unwind-protect "expl" body cleanup1
; cleanup2) sort of feels like:

; (unwind-protect ,body
;   (cond (<body was aborted> ,cleanup1 <pass abort up>)
;         (<body signalled erp> ,cleanup1 <pass (mv erp val state') up>)
;         (t ,cleanup2 <pass (mv erp val state') up>)))

; where we do whatever we have to do to detect aborts and to pass aborts up in
; some cases and triples up in others.  This can all be done with a suitable
; local nest of let, catch, unwind-protect, tests, and throw.  But there is a
; problem: if the user is typing "Abort!" then what is to prevent him from
; doing it during the cleanup forms?  Nothing.  So in fact the sketched use of
; unwind-protect doesn't guarantee that the cleanup forms are executed fully.
; We have been unable to find a way to guarantee via locally produced compiled
; code that even idempotent cleanup forms are executed without interruption.

; Therefore, we take a step back and claim that at the top of the system is
; the ACL2 command interpreter.  It will have an unwind-protect in it (quite
; probably the only unwind-protect in the whole system) and it will guarantee
; to execute all the cleanup forms before it prompts the user for the next
; expression to evaluate.  An abort there will rip us out of the command
; interpreter.  We shall arrange for re-entering it to execute the cleanup
; forms before prompting.  If we imagine, again, that each acl2-unwind-protect
; is compiled into an unwind-protect, then since the aborts are passed up and
; the cleanup forms are each executed in turn as we ascend back to the top,
; the cleanup forms are just stacked.  It suffices then for
; acl2-unwind-protect to push the relevant cleanup form (always form 1) on
; this stack before executing body and for the top-level to pop these forms
; and evaluate them one at a time before prompting for the next input.
; Actually, we must push the cleanup form and the current variable bindings in
; order to be able to evaluate the form "out of context."

; The stack in question is called *acl2-unwind-protect-stack*.  It is really a
; stack of "frames".  Each frame on the stack corresponds to a call of the
; general-purpose ACL2 read-eval-print loop.  By so organizing it we can ensure
; that each call of the read-eval-print loop manages its own unwind protection
; (in the normal case) while also insuring that the stack is global and visible
; to all.  This allows each level to clean up after aborted inferiors what
; failed to clean up after themselves.  If however we abort during the last
; cleanup form, we will find ourselves in raw Lisp.  See the comment about this
; case in ld-fn.

; One final observation is in order.  It could be that there is no command
; interpreter because we are running an ACL2 application in raw lisp.  In that
; case, "Abort!" means the machine was unplugged and all bets are off anyway.

#-acl2-loop-only
(defparameter *acl2-unwind-protect-stack* nil)

#-acl2-loop-only
(defvar *lp-ever-entered-p* nil)

#-acl2-loop-only
(defmacro push-car (item place ctx)
  (let ((g (gensym)))
    `(let ((,g ,place))
       (if (consp ,g)
           (push ,item (car ,g))
         (if *lp-ever-entered-p*
             (illegal ,ctx
                      "Apparently you have tried to execute a form in raw Lisp ~
                       that is only intended to be executed inside the ACL2 ~
                       loop.  You should probably abort (e.g., :Q in akcl or ~
                       gcl, :A in Lispworks, :POP in Allegro), then type (LP) ~
                       and try again.  If this explanation seems incorrect, ~
                       then please contact the implementors of ACL2."
                      nil)
           (illegal ,ctx
                    "Please enter the ACL2 loop by typing (LP) <return>."
                    nil))))))

(defmacro acl2-unwind-protect (expl body cleanup1 cleanup2)

; Note: If the names used for the erp and val results are changed in the #+
; code, then change them in the #- code also.  We use the same names (rather
; than using gensym) just because we know they are acceptable if translate
; approves the check-vars-not-free.

; Note: Keep this function in sync with translated-acl2-unwind-protectp4.  That
; function not only knows the precise form of the expression generated below
; but even knows the variable names used!

  #+acl2-loop-only
  (declare (ignore expl))
  #+acl2-loop-only
  `(mv-let (acl2-unwind-protect-erp acl2-unwind-protect-val state)
           (check-vars-not-free
            (acl2-unwind-protect-erp acl2-unwind-protect-val)
            ,body)
           (cond
            (acl2-unwind-protect-erp
             (pprogn (check-vars-not-free
                      (acl2-unwind-protect-erp acl2-unwind-protect-val)
                      ,cleanup1)
                     (mv acl2-unwind-protect-erp
                         acl2-unwind-protect-val
                         state)))
            (t (pprogn (check-vars-not-free
                        (acl2-unwind-protect-erp acl2-unwind-protect-val)
                        ,cleanup2)
                       (mv acl2-unwind-protect-erp
                           acl2-unwind-protect-val
                           state)))))

; The raw code is very similar.  But it starts out by pushing onto the undo
; stack the name of the cleanup function and the values of the arguments.  Note
; however that we do this only if the state is the live state.  That is the
; only state that matters after an abort.  Suppose unwind protected code is
; modifying some state object other than the live one (e.g., we are computing
; some explicit value during a proof).  Suppose an abort occurs.  Consider the
; operational model described: we rip out of the computation, execute the
; cleanup code for the nearest unwind protect, and then pass the abort upwards,
; continuing until we get to the top level.  No state besides the live one is
; relevant because no value is returned from an aborted computation.  The fake
; state cleaned up at each stop on the way up is just wasted time.  So we don't
; push the cleanup code for fake states.  If body concludes without an abort we
; execute the appropriate cleanup form and then we pop the undo stack (if we
; pushed something).  Note that it is possible that body completes without
; error, cleanup2 is started (and begins smashing state) and then (perhaps even
; after the completion of cleanup2 but before the pop) an abort rips us out,
; causing cleanup1 to be executed after cleanup2.  Idempotency is not enough to
; say.

  #-acl2-loop-only
  `(let ((temp (and (live-state-p state)

; We have seen warnings from Lispworks 4.2.7 of this form that appear to be
; related to the present binding, but we do not yet know how to eliminate them:
;
; Eliminating a test of a variable with a declared type : TEMP [type CONS]

                    (cons ,expl (function (lambda nil ,cleanup1))))))

; FUNCTION captures the binding environment in which cleanup1 would
; have been executed.  So by applying the resulting function to no
; arguments we evaluate cleanup1 in the current environment.  We save
; this cons in temp so we can recognize it below.  If we're not
; operating on the live state, temp is nil.

     (cond (temp
            (push-car temp
                      *acl2-unwind-protect-stack*
                      'acl2-unwind-protect)))

     (mv-let (acl2-unwind-protect-erp acl2-unwind-protect-val state)
             ,body

; Roughly speaking, we should execute cleanup1 or cleanup2, as
; appropriate based on acl2-unwind-protect-erp, and then pop the
; stack.  (Indeed, we used to do this.)  However, it is possible that
; the execution of body pushed more forms on the stack and they
; haven't been cleaned off yet because of hard errors.  Therefore, we
; first restore the stack to just after the pushing of temp, if we
; pushed temp.

             (cond (temp (acl2-unwind -1 temp)))

             (cond
              (acl2-unwind-protect-erp
               (pprogn ,cleanup1
                       (cond (temp
                              (pop (car *acl2-unwind-protect-stack*))
                              state)
                             (t state))
                       (mv acl2-unwind-protect-erp
                           acl2-unwind-protect-val
                           state)))
              (t (pprogn ,cleanup2
                         (cond (temp
                                (pop (car *acl2-unwind-protect-stack*))
                                state)
                               (t state))
                         (mv acl2-unwind-protect-erp
                             acl2-unwind-protect-val
                             state)))))))

#-acl2-loop-only
(defun-one-output acl2-unwind (n flg)

; flg = nil, pop until length of stack is n.  Do not mess with new top-most
; frame.

; flg = t, pop until the length of the stack is n and there is
; at most one form in the top-most frame.  This configures the stack
; the way it was when frame n was first built.

; (consp flg), pop until the top-most form in the top frame is eq to
; flg.  We do not execute that form.  Note that n is irrelevant in
; this case.

; In all cases, no form is removed from the stack until the form has been
; executed.  Thus, an interruption in this process will leave the still-undone
; cleanup forms on the stack for continued processing.

; There is a very odd aspect to this function: the value of each cleanup form
; is simply discarded!  What is going on?  To think about this it is clarifying
; first to consider the case of cleanup in the absence of aborts, i.e., to
; think about the logical semantics of unwind protection.  Consider then
; (acl2-unwind-protect "expl" body cleanup1 cleanup2).  Call the initial STATE st.
; Suppose body computes normally but returns (mv t nil st').  That is, body
; signals an error and returns a modified state (e.g., that has the error
; message printed to it).  Then cleanup1 is executed on st' to produce st''
; and then the error triple (mv t nil st'') is propagated upwards.  Note that
; unlike all the other variables in the cleanup form, the STATE used by
; cleanup1 is the post-body value of the variable, not the pre-body value.

; Now reflect on our abort processing.  Before body is executed we captured the
; binding environment in which cleanup1 would have been executed, except that
; that environment contains the pre-body value for STATE.  If an abort occurs
; during body we evaluate the cleanup function on those saved values.
; Technically we should replace the value of STATE by the post-body state, st',
; produced by body before the abort.  Technically we should then pass upward to
; the next cleanup form the state, st'', produced by the just executed cleanup
; form.

; What prevents us from having to do this is the fact that we are always
; cleaning up the live state and only the live state.  The slot holding STATE
; in the environment captured by FUNCTION contains *the-live-state*, which is
; both the pre-body and post-body value of STATE.  The result of the cleanup
; form is guaranteed to be *the-live-state*.  And so it only looks like we are
; ignoring the values of the cleanup forms!

  (cond ((cond
          ((eq flg nil)
           (= (length *acl2-unwind-protect-stack*) n))
          ((eq flg t)
           (and (= (length *acl2-unwind-protect-stack*) n)
                (or (null (car *acl2-unwind-protect-stack*))
                    (null (cdr (car *acl2-unwind-protect-stack*))))))
          (t (eq flg (car (car *acl2-unwind-protect-stack*)))))
         nil)
        ((null (car *acl2-unwind-protect-stack*))
         (pop *acl2-unwind-protect-stack*)
         (acl2-unwind n flg))
        (t (let ((*wormholep* nil))

; We bind *wormholep* to nil so that we do not try to store undo forms
; for the state changes we are about to make.

             (apply (cdr (car (car *acl2-unwind-protect-stack*)))
; The presence of expl requires us to take the cdr!
                    nil))

           (pop (car *acl2-unwind-protect-stack*))
           (acl2-unwind n flg))))

; The above function, acl2-unwind, will be called in the command interpreter
; before any command is read from the user.  Thus, by the time a user command
; is executed we are guaranteed that all cleanup forms from the previous
; command have been completed, regardless of how often it and its cleanup forms
; were interrupted.  This completes our consideration of user-caused aborts
; during the execution of ACL2 source or compiled code by the Common Lisp
; system.  Now we turn to the even more complicated (!) business of the
; "correct" execution acl2-unwind-protect by ACL2's own EV.

; The code for EV is presented several files from here.  But we discuss
; the design issues here while the previous discussion is still fresh.
; By way of foreshadowing, ev is an interpreter for the logic.

; The first problem is that when EV sees an acl2-unwind-protect it doesn't see
; an acl2-unwind-protect at all.  It sees the translation of the macro
; expansion.  To make matters worse, there are two translations of an MV-LET
; expression: one if the expression occurs inside a function definition (or is
; otherwise deemed "executable") and another if it does not.  The functions
; translated-acl2-unwind-protectp and translated-acl2-unwind-protectp4
; recognize and return the relevant parts of a translated acl2-unwind-protect.
; We can't define them here because they use case-match, which isn't yet
; defined.

; So imagine that EV encounters a translated acl2-unwind-protect form, say
; (acl2-unwind-protect "expl" body cleanup1 cleanup2).  Of course, if the
; evaluation is error and abort free, then it is done correctly.  If an abort
; occurs we are free (by the unplugging argument) to do whatever we want.  But
; what should EV do if there is some kind of an evaluation error in body?  For
; example, suppose body calls an undefined function or violates some guard.  A
; simple concrete question is "what should EV return on

; (acl2-unwind-protect "expl"
;                      (mv nil (car 0) state)
;                      (f-put-global 'foo 'error state)
;                      (f-put-global 'foo 'no-error state))?"

; For what it is worth, our answer to this concrete question is:
; (mv t "guard violation msg for car" (f-put-global 'foo 'error state)).
; To discuss this, we have to tip-toe carefully around a variety of "errors."
; Let us review EV's functionality.

; EV returns (mv erp val latches), where val is the value of the given
; form when erp is nil.  If the form returns a multiple value, then val
; is the corresponding list.  Note well: if form returns an error
; triple, then the error flag of that triple is the car of val, not
; erp.  If erp is t, then some sort of "evaluation error" occurred
; (such as a udf, ubv or guard violation) and val is an error message.
; Latches is an alist that contains the current values of all stobjs,
; including one for 'state.  We distinguish "evaluation errors" (erp =
; t) from the "programmed errors" that may be signaled by some bodies.
; A programmed error is signaled by val being a list of the form
; (t nil state), except that the actual state is to be found in the final
; value of the latches, not in val.

; It is useful to draw an analogy between Common Lisp execution of
; ACL2 source code and the EV interpretation of such code.  In that
; analogy, EV's "evaluation errors" correspond to "aborts" and "hard
; errors," while EV's "programmed errors" correspond to "soft errors."
; It is this analogy that guides us in the design of EV.  What does EV
; do if an evaluation error occurs during body?  Consider the analogy:
; if Common Lisp gets a hard error during the evaluation of body, it
; evaluates cleanup1 and then passes the hard error up.  Therefore, if
; EV gets an evaluation error during the evaluation of body, it
; evaluates cleanup1 and then passes the evaluation error up.  In
; particular, if the attempt to eval body produces (mv t "msg"
; latches') then EV returns (mv t "msg" latches''), where latches'' is
; obtained by evaluating cleanup1 with STATE bound to latches'.  This is
; analogous to what Common Lisp does for the live state.  EV can do it
; for any state (live or otherwise) because it is tracking explicitly
; "the last returned state" during the computation, while Common Lisp
; is not.  Furthermore, Common Lisp need not pass non-live states up
; since it is only the cleaned up live state that matters -- no other
; value is returned from aborted computations.  But EV may be called
; by ACL2 code that makes use of the last state returned during the
; computation.

; If we could stop here the situation would be pretty neat.  But there
; is more.  EV must deal with a third kind of error: true aborts.  We
; have just spoken of evaluation errors (i.e., guard violations and
; other errors detected by EV during evaluation) and of programmed
; errors signaled by the code EV is evaluating.  But what if the user
; types "Abort?"  Certainly neither EV nor its caller "catches" the
; abort: we just rip our way up through the unwind protects.  But if
; EV was being used to modify the live state in an unwind protected
; way, those cleanup forms must be evaluated.  This is just another
; way of saying that EV's interpretation of acl2-unwind-protect must
; be phrased in terms of acl2-unwind-protect just so that the live
; state is cleaned up after aborts.  We can't actually do that because
; acl2-unwind-protect is too structured and insists that we deal with
; (mv erp val state) triples when EV is dealing with (mv erp (mv erp
; val state) latches) triples.  But we use the same raw mechanism of
; the *acl2-unwind-protect-stack*.

; Now the question arises, "what gives us the right to design EV by
; analogy?"  The spec for EV is that it returns the correct value when
; it reports no error (returned erp = nil).  When an evaluation error
; is reported then all bets are off, i.e., the plug was pulled, and we
; can pretty much return the latches we want, as long as it, indeed,
; contains the final values of all the stobjs.

; This completes the unwind-protect essay.  There are some additional comments
; in the code for EV.

; It is IMPERATIVE that the following macro, when-logic, is ONLY used when its
; second argument is a form that evaluates to an error triple.  Keep this
; function in sync with boot-translate.

(defmacro when-logic (str x)
  (list 'if
        '(eq (default-defun-mode-from-state state)
             :program)
        (list 'skip-when-logic (list 'quote str) 'state)
        x))

; ---------------------------------------------------------------------------
; The *initial-event-defmacros* Discussion

; Lasciate ogni speranza, voi ch' entrate

; The following sequence of defmacros is critically important during
; boot strapping because they define the macros we have been using all
; this time!  In fact, this very sequence of forms (minus those not
; marked by the Warning message seen repeatedly below) appears
; elsewhere in this system as a quoted list of constants,
; *initial-event-defmacros*.

; We'll present the defmacros first and then explain the rules for
; adding to or changing them.  See also the discussion at
; *initial-event-defmacros*.

#+acl2-loop-only
(defmacro in-package (str)

; Warning: See the Important Boot-Strapping Invariants before modifying!

  ":Doc-Section ACL2::Other

  select current package~/
  ~bv[]
  Example:
  (in-package \"MY-PKG\")~/

  General Form:
  (in-package str)
  ~ev[]
  where ~c[str] is a string that names an existing ACL2 package, i.e.,
  one of the initial packages such as ~c[\"KEYWORD\"] or ~c[\"ACL2\"] or a
  package introduced with ~ilc[defpkg].  For a complete list of the known
  packages created with ~ilc[defpkg], evaluate
  ~bv[]
  (strip-cars (known-package-alist state)).
  ~ev[]
  ~l[defpkg].  An ACL2 book (~pl[books]) must contain a single ~c[in-package]
  form, which must be the first form in that book."

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'in-package-fn (list 'quote str) 'state))

#+acl2-loop-only
(defmacro defpkg (&whole event-form name form &optional doc book-path)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section Events

  define a new symbol package~/
  ~bv[]
  Example:
  (defpkg \"MY-PKG\"
          (union-eq *acl2-exports*
                    *common-lisp-symbols-from-main-lisp-package*))~/

  General Form:
  (defpkg \"name\" term doc-string)
  ~ev[]

  where ~c[\"name\"] is a non-empty string consisting of standard characters
  (~pl[standard-char-p]), none of which is lower case, that names the package
  to be created; ~c[term] is a variable-free expression that evaluates to a
  list of symbols, where no two distinct symbols in the list may have the same
  ~ilc[symbol-name], to be imported into the newly created package; and
  ~ilc[doc-string] is an optional ~il[documentation] string; ~pl[doc-string].
  The name of the new package must be ``new'': the host lisp must not contain
  any package of that name.  There are two exceptions to this newness rule,
  discussed at the end of this documentation.

  (There is actually an additional argument, book-path, that is used for error
  reporting but has no logical content.  Users should generally ignore this
  argument, as well as the rest of this sentence: a book-path will be specified
  for ~ilc[defpkg] events added by ACL2 to the ~il[portcullis] of a book's
  ~il[certificate]; ~pl[hidden-death-package].)

  ~c[Defpkg] forms can be entered at the top-level of the ACL2 ~il[command]
  loop.  They should not occur in ~il[books] (~pl[certify-book]).

  After a successful ~c[defpkg] it is possible to ``intern'' a string
  into the package using ~ilc[intern-in-package-of-symbol].  The result
  is a symbol that is in the indicated package, provided the imports
  allow it.  For example, suppose ~c['my-pkg::abc] is a symbol whose
  ~ilc[symbol-package-name] is ~c[\"MY-PKG\"].  Suppose further that
  the imports specified in the ~c[defpkg] for ~c[\"MY-PKG\"] do not include
  a symbol whose ~ilc[symbol-name] is ~c[\"XYZ\"].  Then
  ~bv[]
  (intern-in-package-of-symbol \"XYZ\" 'my-pkg::abc)
  ~ev[]
  returns a symbol whose ~ilc[symbol-name] is ~c[\"XYZ\"] and whose
  ~ilc[symbol-package-name] is ~c[\"MY-PKG\"].  On the other hand, if
  the imports to the ~c[defpkg] does include a symbol with the name
  ~c[\"XYZ\"], say in the package ~c[\"LISP\"], then
  ~bv[]
  (intern-in-package-of-symbol \"XYZ\" 'my-pkg::abc)
  ~ev[]
  returns that symbol (which is uniquely determined by the restriction
  on the imports list above).  ~l[intern-in-package-of-symbol].

  ~c[Defpkg] is the only means by which an ACL2 user can create a new
  package or specify what it imports.  That is, ACL2 does not support
  the Common Lisp functions ~c[make-package] or ~c[import].  Currently, ACL2
  does not support exporting at all.

  The Common Lisp function ~ilc[intern] is weakly supported by ACL2.
  ~l[intern].

  We now explain the two exceptions to the newness rule for package
  names.  The careful experimenter will note that if a package is
  created with a ~c[defpkg] that is subsequently undone, the host lisp
  system will contain the created package even after the undo.
  Because ACL2 hangs onto ~il[world]s after they have been undone, e.g., to
  implement ~c[:]~ilc[oops] but, more importantly, to implement error recovery,
  we cannot actually destroy a package upon undoing it.  Thus, the
  first exception to the newness rule is that ~c[name] is allowed to be
  the name of an existing package if that package was created by an
  undone ~c[defpkg] and the newly proposed set of imports is identical to the
  old one.  ~l[package-reincarnation-import-restrictions].  This
  exception does not violate the spirit of the newness rule, since one
  is disinclined to believe in the existence of undone packages.  The
  second exception is that ~c[name] is allowed to be the name of an
  existing package if the package was created by a ~c[defpkg] with
  identical set of imports.  That is, it is permissible to execute
  ``redundant'' ~c[defpkg] ~il[command]s.  The redundancy test is based on the
  values of the two import forms (comparing them after sorting and removing
  duplicates), not on the forms themselves.

  Finally, we explain why we require the package name to contain standard
  characters, none of which is lower case.  We have seen at least one
  implementation that handled lower-case package names incorrectly.  Since we
  see no need for lower-case characters in package names, which can lead to
  confusion anyhow (note for example that ~c[foo::bar] is a symbol whose
  ~ilc[symbol-package-name] is ~c[\"FOO\"], not ~c[\"foo\"]), we simply
  disallow them.  Since the notion of ``lower case'' is only well-specified in
  Common Lisp for standard characters, we restrict to these.

  NOTE: Also ~pl[managing-acl2-packages] for contributed documentation on
  managing ACL2 packages.~/

  :cited-by Programming"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'defpkg-fn
        (list 'quote name)
        (list 'quote form)
        'state
        (list 'quote doc)
        (list 'quote book-path)
        (list 'quote event-form)))

(defdoc managing-acl2-packages
  ":Doc-Section defpkg

  user-contributed documentation on packages~/

  Jared Davis has contributed documentation on managing ACL2
  packages.  See
  ~url[http://www.cs.utexas.edu/users/moore/acl2/contrib/managing-acl2-packages.html].~/~/")

(deflabel hidden-defpkg
  :doc
  ":Doc-Section defpkg

  handling defpkg events that are local~/

  ~l[hidden-death-package]~/~/")

(deflabel hidden-death-package
  :doc
  ":Doc-Section defpkg

  handling ~ilc[defpkg] ~il[events] that are ~ilc[local]~/

  This documentation topic explains a little bit about certain errors users may
  see when attempting to evaluate a ~ilc[defpkg] event.  In brief, if you see
  an error that refers you to this topic, you are probably trying to admit a
  ~ilc[defpkg] event, and you should change the name of the package to be
  introduced by that event.

  Recall that ~c[defpkg] events introduce axioms, for example as follows.
  ~bv[]
  ACL2 !>(defpkg \"PKG0\" '(a b))

  Summary
  Form:  ( DEFPKG \"PKG0\" ...)
  Rules: NIL
  Warnings:  None
  Time:  0.01 seconds (prove: 0.00, print: 0.00, other: 0.01)
   \"PKG0\"
  ACL2 !>:pr! \"PKG0\"

  Rune:       (:REWRITE PKG0-PACKAGE)
  Status:     Enabled
  Lhs:        (SYMBOL-PACKAGE-NAME (INTERN-IN-PACKAGE-OF-SYMBOL X Y))
  Rhs:        \"PKG0\"
  Hyps:       (AND (STRINGP X)
                   (NOT (MEMBER-SYMBOL-NAME X '(A B)))
                   (SYMBOLP Y)
                   (EQUAL (SYMBOL-PACKAGE-NAME Y) \"PKG0\"))
  Equiv:      EQUAL
  Backchain-limit-lst:    NIL
  Subclass:   BACKCHAIN
  Loop-stopper: NIL
  ACL2 !>
  ~ev[]
  Now, a ~ilc[defpkg] event may be executed underneath an ~ilc[encapsulate] or
  ~ilc[include-book] form that is marked ~ilc[local].  In that case, traces of
  the added axiom will disappear after the surrounding ~ilc[encapsulate] or
  ~ilc[include-book] form is admitted.  This can cause inconsistencies.  (You
  can take our word for it, or you can look at the example shown in the
  ``Essay on Hidden Packages'' in source file ~c[axioms.lisp].)

  In order to prevent unsoundness, then, ACL2 maintains the following
  invariant.  Let us say that a ~c[defpkg] event is ``hidden'' if it is in
  support of the current logical ~il[world] but is not present in that world as
  an event, because it is ~ilc[local] as indicated above.  We maintain the
  invariant that all ~ilc[defpkg] ~il[events], even if ``hidden'', are tracked
  under-the-hood in the current logical ~il[world].  Sometimes this property
  causes ~ilc[defpkg] events to be written to the ~il[portcullis] of a book's
  ~il[certificate] (~pl[books]).  At any rate, if you then try to define the
  package in a manner inconsistent with the earlier such definition, that is,
  with a different imports list, you will see an error because of the
  above-mentioned tracking.

  (By the way, this topic's name comes from Holly Bell, who heard
  \"hidden death package\" instead of \"hidden defpkg\".  The description
  seemed to fit.  Thanks, Holly!)~/~/")

#+acl2-loop-only
(defmacro defun (&whole event-form &rest def)

; Warning: See the Important Boot-Strapping Invariants before modifying!

  ":Doc-Section acl2::Events

  define a function symbol~/
  ~bv[]
  Examples:
  (defun app (x y)
    (if (consp x)
        (cons (car x) (app (cdr x) y))
        y))

  (defun fact (n)
    (declare (xargs :guard (and (integerp n)
                                (>= n 0))))
    (if (zp n)
        1
        (* n (fact (1- n)))))~/

  General Form:
  (defun fn (var1 ... varn) doc-string dcl ... dcl body),
  ~ev[]
  where ~c[fn] is the symbol you wish to define and is a new symbolic name
  (~pl[name]), ~c[(var1 ... varn)] is its list of formal parameters
  (~pl[name]), and ~c[body] is its body.  The definitional axiom is logically
  admissible provided certain restrictions are met.  These are sketched below.

  Note that ACL2 does not support the use of ~c[lambda-list] keywords (such as
  ~c[&optional]) in the formals list of functions.  We do support some such
  keywords in macros and often you can achieve the desired syntax by defining a
  macro in addition to the general version of your function.  ~l[defmacro].
  The ~il[documentation] string, ~ilc[doc-string], is optional; for a
  description of its form, ~pl[doc-string].

  The ~em[declarations] (~pl[declare]), ~c[dcl], are also optional.  If more
  than one ~c[dcl] form appears, they are effectively grouped together as one.
  Perhaps the most commonly used ACL2 specific declaration is of the form
  ~c[(declare (xargs :guard g :measure m))].  This declaration in the ~c[defun]
  of some function ~c[fn] has the effect of making the ``~il[guard]'' for
  ~c[fn] be the term ~c[g] and the ``measure'' be the term ~c[m].  The notion
  of ``measure'' is crucial to ACL2's definitional principle.  The notion of
  ``guard'' is not, and is discussed elsewhere; ~pl[verify-guards] and
  ~pl[set-verify-guards-eagerness].  Note that the ~c[:measure] is ignored in
  ~c[:]~ilc[program] mode; ~pl[defun-mode].

  We now briefly discuss the ACL2 definitional principle, using the following
  definition form which is offered as a more or less generic example.
  ~bv[]
  (defun fn (x y)
    (declare (xargs :guard (g x y)
                    :measure (m x y)))
    (if (test x y)
        (stop x y)
      (step (fn (d x) y))))
  ~ev[]
  Note that in our generic example, ~c[fn] has just two arguments, ~c[x] and
  ~c[y], the ~il[guard] and measure terms involve both of them, and the body is
  a simple case split on ~c[(test x y)] leading to a ``non-recursive'' branch,
  ~c[(stop x y)], and a ``recursive'' branch.  In the recursive branch, ~c[fn]
  is called after ``decrementing'' ~c[x] to ~c[(d x)] and some step function is
  applied to the result.  Of course, this generic example is quite specific in
  form but is intended to illustrate the more general case.

  Provided this definition is admissible under the logic, as outlined below, it
  adds the following axiom to the logic.
  ~bv[]
  Defining Axiom:
  (fn x y)
    =
  (if (test x y)
      (stop x y)
    (step (fn (d x) y)))
  ~ev[]
  Note that the ~il[guard] of ~c[fn] has no bearing on this logical axiom.

  This defining axiom is actually implemented in the ACL2 system by a
  ~c[:]~ilc[definition] rule, namely
  ~bv[]
  (equal (fn x y)
         (if (test a b)
             (stop a b)
           (step (fn (d a) b)))).
  ~ev[]
  ~l[definition] for a discussion of how definition rules are applied.  Roughly
  speaking, the rule causes certain instances of ~c[(fn x y)] to be replaced by
  the corresponding instances of the body above.  This is called ``opening up''
  ~c[(fn x y)].  The instances of ~c[(fn x y)] opened are chosen primarily by
  heuristics which determine that the recursive calls of ~c[fn] in the opened
  body (after simplification) are more desirable than the unopened call of
  ~c[fn].

  This discussion has assumed that the definition of ~c[fn] was admissible.
  Exactly what does that mean?  First, ~c[fn] must be a previously
  unaxiomatized function symbol (however, ~pl[ld-redefinition-action]).
  Second, the formal parameters must be distinct variable names.  Third, the
  ~il[guard], measure, and body should all be terms and should mention no free
  variables except the formal parameters.  Thus, for example, body may not
  contain references to ``global'' or ``special'' variables; ACL2 constants or
  additional formals should be used instead.

  The final conditions on admissibility concern the termination of the
  recursion.  Roughly put, all applications of ~c[fn] must terminate.  In
  particular, there must exist a binary relation, ~c[rel], and some unary
  predicate ~c[mp] such that ~c[rel] is well-founded on objects satisfying
  ~c[mp], the measure term ~c[m] must always produce something satisfying
  ~c[mp], and the measure term must decrease according to ~c[rel] in each
  recursive call, under the hypothesis that all the tests governing the call
  are satisfied.  By the meaning of well-foundedness, we know there are no
  infinitely descending chains of successively ~c[rel]-smaller ~c[mp]-objects.
  Thus, the recursion must terminate.

  The only primitive well-founded relation in ACL2 is ~ilc[o<] (~pl[o<]), which
  is known to be well-founded on the ~ilc[o-p]s (~pl[o-p]).  For the proof of
  well-foundedness, ~pl[proof-of-well-foundedness].  However it is possible to
  add new well-founded relations.  For details, ~pl[well-founded-relation].  We
  discuss later how to specify which well-founded relation is selected by
  ~c[defun] and in the present discussion we assume, without loss of
  generality, that it is ~ilc[o<] on the ~ilc[o-p]s.

  For example, for our generic definition of ~c[fn] above, with measure term
  ~c[(m x y)], two theorems must be proved.  The first establishes that ~c[m]
  produces an ordinal:
  ~bv[]
  (o-p (m x y)).
  ~ev[]
  The second shows that ~c[m] decreases in the (only) recursive call of ~c[fn]:
  ~bv[]
  (implies (not (test x y))
           (o< (m (d x) y) (m x y))).
  ~ev[]
  Observe that in the latter formula we must show that the ``~c[m]-size'' of
  ~c[(d x)] and ~c[y] is ``smaller than'' the ~c[m]-size of ~c[x] and ~c[y],
  provided the test, ~c[(test x y)], in the body fails, thus leading to the
  recursive call ~c[(fn (d x) y)].

  ~l[o<] for a discussion of this notion of ``smaller than.''  It should be
  noted that the most commonly used ordinals are the natural numbers and that
  on natural numbers, ~ilc[o<] is just the familiar ``less than'' relation
  (~ilc[<]).  Thus, it is very common to use a measure ~c[m] that returns a
  nonnegative integer, for then ~c[(o-p (m x y))] becomes a simple conjecture
  about the type of ~c[m] and the second formula above becomes a conjecture
  about the less-than relationship of nonnegative integer arithmetic.

  The most commonly used measure function is ~ilc[acl2-count], which computes a
  nonnegative integer size for all ACL2 objects.  ~l[acl2-count].

  Probably the most common recursive scheme in Lisp ~il[programming] is when
  some formal is supposed to be a list and in the recursive call it is replaced
  by its ~ilc[cdr].  For example, ~c[(test x y)] might be simply ~c[(atom x)]
  and ~c[(d x)] might be ~c[(cdr x)].  In that case, ~c[(acl2-count x)] is a
  suitable measure because the ~ilc[acl2-count] of a ~ilc[cons] is strictly
  larger than the ~ilc[acl2-count]s of its ~ilc[car] and ~ilc[cdr].  Thus,
  ``recursion by ~ilc[car]'' and ``recursion by ~ilc[cdr]'' are trivially
  admitted if ~ilc[acl2-count] is used as the measure and the definition
  protects every recursive call by a test insuring that the decremented
  argument is a ~ilc[consp].  Similarly, ``recursion by ~ilc[1-]'' in which a
  positive integer formal is decremented by one in recursion, is also trivially
  admissible.  ~l[built-in-clause] to extend the class of trivially admissible
  recursive schemes.

  We now turn to the question of which well-founded relation ~c[defun] uses.
  It should first be observed that ~c[defun] must actually select both a
  relation (e.g., ~ilc[o<]) and a domain predicate (e.g., ~ilc[o-p]) on which
  that relation is known to be well-founded.  But, as noted elsewhere
  (~pl[well-founded-relation]), every known well-founded relation has a unique
  domain predicate associated with it and so it suffices to identify simply the
  relation here.

  The ~ilc[xargs] field of a ~ilc[declare] permits the explicit specification
  of any known well-founded relation with the keyword
  ~c[:]~ilc[well-founded-relation].  An example is given below.  If the
  ~ilc[xargs] for a ~c[defun] specifies a well-founded relation, that relation
  and its associated domain predicate are used in generating the termination
  conditions for the definition.

  If no ~c[:]~ilc[well-founded-relation] is specified, ~c[defun] uses the
  ~c[:]~ilc[well-founded-relation] specified in the ~ilc[acl2-defaults-table].
  ~l[set-well-founded-relation] to see how to set the default well-founded
  relation (and, implicitly, its domain predicate).  The initial default
  well-founded relation is ~ilc[o<] (with domain predicate ~ilc[o-p]).

  This completes the brief sketch of the ACL2 definitional principle.
  Optionally, ~pl[ruler-extenders] for a more detailed discussion of the
  termination analysis and resulting proof obligations for admissibility, as
  well as a discussion of the relation to how ACL2 stores induction schemes.

  On very rare occasions ACL2 will seem to \"hang\" when processing a
  definition, especially if there are many subexpressions of the body whose
  function symbol is ~ilc[if] (or which macroexpand to such an expression).  In
  those cases you may wish to supply the following to ~ilc[xargs]:
  ~c[:normalize nil].  This is an advanced feature that turns off ACL2's usual
  propagation upward of ~c[if] tests.

  The following example illustrates all of the available declarations and most
  hint keywords, but is completely nonsensical.  For documentation, ~pl[xargs]
  and ~pl[hints].
  ~bv[]
  (defun example (x y z a b c i j)
    (declare (ignore a b c)
             (type integer i j)
             (xargs :guard (symbolp x)
                    :measure (- i j)
                    :ruler-extenders :basic
                    :well-founded-relation my-wfr
                    :hints ((\"Goal\"
                             :do-not-induct t
                             :do-not '(generalize fertilize)
                             :expand ((assoc x a) (member y z))
                             :restrict ((<-trans ((x x) (y (foo x)))))
                             :hands-off (length binary-append)
                             :in-theory (set-difference-theories
                                          (current-theory :here)
                                          '(assoc))
                             :induct (and (nth n a) (nth n b))
                             :use ((:instance assoc-of-append
                                              (x a) (y b) (z c))
                                   (:functional-instance
                                     (:instance p-f (x a) (y b))
                                     (p consp)
                                     (f assoc)))))
                    :guard-hints ((\"Subgoal *1/3'\"
                                   :use ((:instance assoc-of-append
                                                    (x a) (y b) (z c)))))
                    :mode :logic
                    :normalize nil
                    :verify-guards nil
                    :non-executable t
                    :otf-flg t))
    (example-body x y z i j))
  ~ev[]~/

  :cited-by Programming"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'defun-fn
        (list 'quote def)
        'state
        (list 'quote event-form)
        #+:non-standard-analysis ; std-p
        nil))

#+acl2-loop-only
(defmacro defund (&rest def)

  ":Doc-Section acl2::Events

  define a function symbol and then disable it~/~/

  Use ~c[defund] instead of ~ilc[defun] when you want to disable a function
  immediately after its definition.  This macro has been provided for users who
  prefer working in a mode where functions are only enabled when explicitly
  directed by ~c[:]~ilc[in-theory].  Specifically, the form
  ~bv[]
  (defund NAME FORMALS ...)
  ~ev[]
  expands to:
  ~bv[]
  (progn
    (defun NAME FORMALS ...)
    (with-output
     :off summary
     (in-theory (disable NAME)))
    (value NAME)).
  ~ev[]
  Only the ~c[:]~ilc[definition] rule (and, for recursively defined functions,
  the ~c[:]~ilc[induction] rule) for the function are disabled, and the summary
  for the ~ilc[in-theory] event is suppressed.

  Note that ~c[defund] commands are never redundant (~pl[redundant-events]).
  If the function has already been defined, then the ~ilc[in-theory] event
  will still be executed.

  ~l[defun] for documentation of ~c[defun]."

  (declare (xargs :guard (and (true-listp def)
                              (symbolp (car def))
                              (symbol-listp (cadr def)))))

  (list 'progn
        (cons 'defun def)
        (list
         'with-output
         :off 'summary
         (list 'in-theory
               (list 'disable (car def))))
        (list 'value-triple (list 'quote (xd-name 'defund (car def))))))

#-acl2-loop-only
(defmacro defund (&rest def)
  (cons 'defun def))

#+(and acl2-loop-only :non-standard-analysis)
(defmacro defun-std (&whole event-form &rest def)

  ":Doc-Section acl2::Events

  define a function symbol~/~/

  ~l[defun] for details.  (More documentation on features
  related to non-standard analysis may be available in the future.)"

  (list 'defun-fn
        (list 'quote def)
        'state
        (list 'quote event-form)
        t))

#+acl2-loop-only
(defmacro defuns (&whole event-form &rest def-lst)

; Warning: See the Important Boot-Strapping Invariants before modifying!

  ":Doc-Section Miscellaneous

  an alternative to ~ilc[mutual-recursion]~/
  ~bv[]
  Example:
  (DEFUNS
   (evenlp (x)
     (if (consp x) (oddlp (cdr x)) t))
   (oddlp (x)
     (if (consp x) (evenlp (cdr x)) nil)))~/

  General Form:
  (DEFUNS defuns-tuple1 ... defuns-tuplen)
  ~ev[]
  is equivalent to
  ~bv[]
  (MUTUAL-RECURSION
    (DEFUN . defuns-tuple1)
    ...
    (DEFUN . defuns-tuplen))
  ~ev[]
  In fact, ~c[defuns] is the more primitive of the two and
  ~ilc[mutual-recursion] is just a macro that expands to a call of ~ilc[defun]
  after stripping off the ~ilc[defun] at the ~ilc[car] of each argument to
  ~ilc[mutual-recursion].  We provide and use ~ilc[mutual-recursion] rather than
  ~c[defuns] because by leaving the ~ilc[defun]s in place, ~ilc[mutual-recursion]
  forms can be processed by the Emacs ~c[tags] program.
  ~l[mutual-recursion]."

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'defuns-fn
        (list 'quote def-lst)
        'state
        (list 'quote event-form)
        #+:non-standard-analysis ; std-p
        nil))

#+(and acl2-loop-only :non-standard-analysis)
(defmacro defuns-std (&whole event-form &rest def-lst)

  ":Doc-Section Miscellaneous

  an alternative to ~ilc[mutual-recursion]~/~/

  ~l[defuns] for details.  (More documentation on features
  related to non-standard analysis may be available in the future.)"

  (list 'defuns-fn
        (list 'quote def-lst)
        'state
        (list 'quote event-form)
        t))

(defmacro verify-termination (&rest lst)

  ":Doc-Section Events

  convert a function from :program mode to :logic mode~/
  ~bv[]
  Example:
  (verify-termination fact)~/

  General Forms:
  (verify-termination fn dcl ... dcl)
  (verify-termination (fn1 dcl ... dcl)
                      (fn2 dcl ... dcl)
                      ...)
  ~ev[]
  where ~c[fn] and the ~c[fni] are function symbols having ~c[:]~ilc[program] mode
  (~pl[defun-mode]) and all of the ~c[dcl]s are either ~ilc[declare]
  forms or ~il[documentation] strings.  The first form above is an
  abbreviation for
  ~bv[]
  (verify-termination (fn dcl ... dcl))
  ~ev[]
  so we limit our discussion to the second form.  Each of the ~c[fni]
  must be in the same clique of mutually recursively defined
  functions, but not every function in the clique need be among the
  ~c[fni].

  ~c[Verify-termination] attempts to establish the admissibility of the
  ~c[fni]. ~c[Verify-termination] retrieves their definitions, creates modified
  definitions using the ~c[dcl]s supplied above, and resubmits these
  definitions.  You could avoid using ~c[verify-termination] by typing the new
  definitions yourself.  So in that sense, ~c[verify-termination] adds no new
  functionality.  But if you have prototyped your system in ~c[:]~ilc[program]
  mode and tested it, you can use ~c[verify-termination] to resubmit your
  definitions and change their ~il[defun-mode]s to ~c[:]~ilc[logic], addings
  ~il[hints] without having to retype or recopy the code.

  The ~ilc[defun] ~il[command] executed by ~c[verify-termination] is obtained
  by retrieving the ~ilc[defun] (or ~ilc[mutual-recursion]) ~il[command] that
  introduced the clique in question and then possibly modifying each definition
  as follows.  Consider a function, ~c[fn], in the clique.  If ~c[fn] is not
  among the ~c[fni] above, its definition is left unmodified other than to add
  ~c[(declare (xargs :mode :logic))].  Otherwise, ~c[fn] is some ~c[fni] and we
  modify its definition by inserting into it the corresponding ~c[dcl]s listed
  with ~c[fni] in the arguments to ~c[verify-termination], as well as
  ~c[(declare (xargs :mode :logic))].  In addition, we throw out from the old
  declarations in ~c[fn] the ~c[:mode] specification and anything that is
  specified in the new ~c[dcl]s.

  For example, suppose that ~c[fact] was introduced with:
  ~bv[]
  (defun fact (n)
    (declare (type integer n)
             (xargs :mode :program))
    (if (zp n) 1 (* n (fact (1- n))))).
  ~ev[]
  Suppose later we do ~c[(verify-termination fact)].  Then the
  following definition is submitted.
  ~bv[]
  (defun fact (n)
    (declare (type integer n))
    (if (zp n) 1 (* n (fact (1- n))))).
  ~ev[]
  Observe that this is the same definition as the original one, except
  the old specification of the ~c[:mode] has been deleted so that the
  ~il[defun-mode] now defaults to ~c[:]~ilc[logic].  Although the termination
  proof succeeds, ACL2 also tries to verify the ~il[guard], because we have
  (implicitly) provided a ~il[guard], namely ~c[(integerp n)], for this
  function.  (~l[guard] for a general discussion of guards, and
  ~pl[type-spec] for a discussion of how type declarations are
  used in guards.)  Unfortunately, the ~il[guard] verification fails,
  because the subterm ~c[(zp n)] requires that ~c[n] be nonnegative, as
  can be seen by invoking ~c[:args zp].  (For a discussion of termination
  issues relating to recursion on the naturals, ~pl[zero-test-idioms].)
  So we might be tempted to submit the following:
  ~bv[]
  (verify-termination
   fact
   (declare (xargs :guard (and (integerp n) (<= 0 n))))).
  ~ev[]
  However, this is considered a changing of the guard (from ~c[(integerp n)]),
  which is illegal.  If we instead change the guard in the earlier ~c[defun]
  after undoing that earlier definition with ~c[:]~ilc[ubt]~c[ fact], then
  ~c[(verify-termination fact)] will succeed.

  ~st[Remark on system functions.]  There may be times when you want to apply
  ~c[verify-termination] (and also, perhaps, ~ilc[verify-guards]) to functions
  that are predefined in ACL2.  It may be necessary in such cases to modify the
  system code first.  See
  ~url[http://www.cs.utexas.edu/users/moore/acl2/open-architecture/] for a
  discussion of the process for contributing updates to the system code and
  ~il[books] with such ~c[verify-termination] or ~ilc[verify-guards]
  ~il[events].  To see which built-in ~c[:]~ilc[program] mode functions have
  already received such treatment, see directory ~c[books/system]; for example,
  use the Unix utility `~c[grep]' to search:
  ~bv[]
  grep '(verify-' books/system/*.lisp
  ~ev[]

  We conclude with a discussion of the use of ~ilc[make-event] to implement
  ~c[verify-termination].  This discussion can be skipped; we include it only
  for the curious.

  Consider the following proof of ~c[nil], which succeeded up through
  Version_3.4 of ACL2.
  ~bv[]
  (encapsulate
   ()
   (defun foo (x y)
     (declare (xargs :mode :program))
     (if (or (zp x) (zp y))
         (list x y)
       (foo (1+ x) (1- y))))
   (local (defun foo (x y)
            (declare (xargs :measure (acl2-count y)))
            (if (or (zp x) (zp y))
                (list x y)
              (foo (1+ x) (1- y)))))
   (verify-termination foo))

  (defthm bad-lemma
    (zp x)
    :hints ((\"Goal\" :induct (foo x 1)))
    :rule-classes nil)
  ~ev[]
  How did this work?  In the first pass of the ~ilc[encapsulate], the second
  ~ilc[defun] of ~c[foo] promoted ~c[foo] from ~c[:program] to ~c[:logic] mode,
  with ~c[y] as the unique measured variable.  The following call to
  ~c[verify-termination] was then redundant.  However, on the second pass of
  the ~ilc[encapsulate], the second (~ilc[local]) definition of ~c[foo] was
  skipped, and the ~c[verify-termination] event then used the first definition
  of ~c[foo] to guess the measure, based (as with all guesses of measures) on a
  purely syntactic criterion.  ACL2 incorrectly chose ~c[(acl2-count x)] as the
  measure, installing ~c[x] as the unique measured variable, which in turn led
  to an unsound induction scheme subsequently used to prove ~c[nil] (lemma
  ~c[bad-lemma], above)

  Now, ~c[verify-termination] is a macro whose calls expand to ~ilc[make-event]
  calls.  So in the first pass above, the ~c[verify-termination] call generated
  a ~c[defun] event identical to the ~ilc[local] ~ilc[defun] of ~c[foo], which
  was correctly identified as redundant.  That expansion was recorded, and on
  the second pass of the ~ilc[encapsulate], the expansion was recalled and used
  in place of the ~c[verify-termination] call (that is how ~ilc[make-event]
  works).  So instead of a measure being guessed for the ~c[verify-termination]
  call on the second pass, the same measure was used as was used on the first
  pass, and a sound induction scheme was stored.  The attempt to prove ~c[nil]
  (lemma ~c[bad-lemma]) then failed."

  `(make-event
    (verify-termination-fn ',lst state)))

#+acl2-loop-only
(defmacro verify-termination-boot-strap (&whole event-form &rest lst)

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'verify-termination-boot-strap-fn
        (list 'quote lst)
        'state
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro verify-guards (&whole event-form name &key hints otf-flg guard-debug
                                doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Note: If you change the default for guard-debug, then consider changing it in
; chk-acceptable-defuns as well, and fix the "Otherwise" message about
; :guard-debug in prove-guard-clauses.

  ":Doc-Section Events

  verify the ~il[guard]s of a function~/
  ~bv[]
  Examples:
  (verify-guards flatten)
  (verify-guards flatten
                 :hints ((\"Goal\" :use (:instance assoc-of-app)))
                 :otf-flg t
                 :guard-debug t ; default = nil
                 :doc \"string\")~/

  General Form:
  (verify-guards name
          :hints        hints
          :otf-flg      otf-flg
          :guard-debug  t ; typically t, but any value is legal
          :doc          doc-string)
  ~ev[]
  ~l[guard] for a general discussion of guards.  In the General Form above,
  ~c[name] is the name of a ~c[:]~ilc[logic] function (~pl[defun-mode]) or of a
  theorem or axiom.  In the most common case ~c[name] is the name of a function
  that has not yet had its ~il[guard]s verified, each subroutine of which has
  had its ~il[guard]s verified. ~ilc[hints], ~ilc[otf-flg], and
  ~ilc[guard-debug] are as described in the corresponding ~il[documentation]
  entries; and ~ilc[doc-string], if supplied, is a string ~st[not] beginning
  with ``~c[:Doc-Section]''.  The four keyword arguments above are all
  optional.  ~c[Verify-guards] will attempt to prove that the ~il[guard] on the
  named function implies the ~il[guard]s of all of the subroutines called in
  the body of the function, and that the guards are satisfied for all function
  calls in the guard itself (under an implicit guard of ~c[t]).  If successful,
  ~c[name] is considered to have had its ~il[guard]s verified.

  ~l[verify-guards-formula] for a utility that lets you view the formula to be
  proved by ~c[verify-guards], but without creating an event.

  If ~c[name] is one of several functions in a mutually recursive clique,
  ~c[verify-guards] will attempt to verify the ~il[guard]s of all of the
  functions.

  If ~c[name] is a theorem or axiom name, ~c[verify-guards] verifies the
  guards of the associated formula.  When a theorem has had its guards
  verified then you know that the theorem will evaluate to non-~c[nil]
  in all Common Lisps, without causing a runtime error (other than possibly
  a resource error).  In particular, you know that the theorem's validity
  does not depend upon ACL2's arbitrary completion of the domains of partial
  Common Lisp functions.

  For example, if ~c[app] is defined as
  ~bv[]
  (defun app (x y)
    (declare (xargs :guard (true-listp x)))
    (if (endp x)
        y
        (cons (car x) (app (cdr x) y))))
  ~ev[]
  then we can verify the guards of ~c[app] and we can prove the theorem:
  ~bv[]
  (defthm assoc-of-app
    (equal (app (app a b) c) (app a (app b c))))
  ~ev[]
  However, if you go into almost any Common Lisp in which ~c[app] is defined
  as shown and evaluate
  ~bv[]
  (equal (app (app 1 2) 3) (app 1 (app 2 3)))
  ~ev[]
  we get an error or, perhaps, something worse like ~c[nil]!  How can
  this happen since the formula is an instance of a theorem?  It is supposed
  to be true!

  It happens because the theorem exploits the fact that ACL2 has completed
  the domains of the partially defined Common Lisp functions like ~ilc[car]
  and ~ilc[cdr], defining them to be ~c[nil] on all non-conses.  The formula
  above violates the guards on ~c[app].  It is therefore ``unreasonable''
  to expect it to be valid in Common Lisp.

  But the following formula is valid in Common Lisp:
  ~bv[]
  (if (and (true-listp a)
           (true-listp b))
      (equal (app (app a b) c) (app a (app b c)))
      t)
  ~ev[]
  That is, no matter what the values of ~c[a], ~c[b] and ~c[c] the formula
  above evaluates to ~c[t] in all Common Lisps (unless the Lisp engine runs out
  of memory or stack computing it).  Furthermore the above formula is a theorem:

  ~bv[]
  (defthm guarded-assoc-of-app
    (if (and (true-listp a)
             (true-listp b))
        (equal (app (app a b) c) (app a (app b c)))
        t))
  ~ev[]
  This formula, ~c[guarded-assoc-of-app], is very easy to prove from
  ~c[assoc-of-app].  So why prove it?  The interesting thing about
  ~c[guarded-assoc-of-app] is that we can verify the guards of the
  formula.  That is, ~c[(verify-guards guarded-assoc-of-app)] succeeds.
  Note that it has to prove that if ~c[a] and ~c[b] are true lists then
  so is ~c[(app a b)] to establish that the guard on the outermost ~c[app]
  on the left is satisfied.  By verifying the guards of the theorem we
  know it will evaluate to true in all Common Lisps.  Put another way,
  we know that the validity of the formula does not depend on ACL2's
  completion of the partial functions or that the formula is ``well-typed.''

  One last complication:  The careful reader might have thought we could
  state ~c[guarded-assoc-of-app] as
  ~bv[]
  (implies (and (true-listp a)
                (true-listp b))
           (equal (app (app a b) c)
                  (app a (app b c))))
  ~ev[]
  rather than using the ~c[if] form of the theorem.  We cannot!  The
  reason is technical:  ~ilc[implies] is defined as a function in ACL2.
  When it is called, both arguments are evaluated and then the obvious truth
  table is checked.  That is, ~c[implies] is not ``lazy.''  Hence, when
  we write the guarded theorem in the ~c[implies] form we have to prove
  the guards on the conclusion without knowing that the hypothesis is true.
  It would have been better had we defined ~c[implies] as a macro that
  expanded to the ~c[if] form, making it lazy.  But we did not and after
  we introduced guards we did not want to make such a basic change.

  Recall however that ~c[verify-guards] is almost always used to verify
  the guards on a function definition rather than a theorem.  We now
  return to that discussion.

  Because ~c[name] is not uniquely associated with the ~c[verify-guards] event
  (it necessarily names a previously defined function) the
  ~il[documentation] string, ~ilc[doc-string], is not stored in the
  ~il[documentation] data base.  Thus, we actually prohibit ~ilc[doc-string]
  from having the form of an ACL2 ~il[documentation] string;
  ~pl[doc-string].

  If the guard on a function is not ~c[t], then guard verification
  requires not only consideration of the body under the assumption
  that the guard is true, but also consideration of the guard itself.
  Thus, for example, guard verification fails in the following
  example, even though there are no proof obligations arising from the
  body, because the guard itself can cause a guard violation when
  evaluated for an arbitrary value of ~c[x]:
  ~bv[]
  (defun foo (x)
    (declare (xargs :guard (car x)))
    x)
  ~ev[]

  ~c[Verify-guards] must often be used when the value of a recursive call
  of a defined function is given as an argument to a subroutine that
  is ~il[guard]ed.  An example of such a situation is given below.  Suppose
  ~c[app] (read ``append'') has a ~il[guard] requiring its first argument to be
  a ~ilc[true-listp].  Consider
  ~bv[]
  (defun rev (x)
    (declare (xargs :guard (true-listp x)))
    (cond ((endp x) nil)
          (t (app (rev (cdr x)) (list (car x))))))
  ~ev[]
  Observe that the value of a recursive call of ~c[rev] is being passed
  into a ~il[guard]ed subroutine, ~c[app].  In order to verify the ~il[guard]s of
  this definition we must show that ~c[(rev (cdr x))] produces a
  ~ilc[true-listp], since that is what the ~il[guard] of ~c[app] requires.  How do we
  know that ~c[(rev (cdr x))] is a ~ilc[true-listp]?  The most elegant argument
  is a two-step one, appealing to the following two lemmas: (1) When ~c[x]
  is a ~ilc[true-listp], ~c[(cdr x)] is a ~ilc[true-listp].  (2) When ~c[z] is a
  ~ilc[true-listp], ~c[(rev z)] is a ~ilc[true-listp].  But the second lemma is a
  generalized property of ~c[rev], the function we are defining.  This
  property could not be stated before ~c[rev] is defined and so is not
  known to the theorem prover when ~c[rev] is defined.

  Therefore, we might break the admission of ~c[rev] into three steps:
  define ~c[rev] without addressing its ~il[guard] verification, prove some
  general properties about ~c[rev], and then verify the ~il[guard]s.  This can
  be done as follows:
  ~bv[]
  (defun rev (x)
    (declare (xargs :guard (true-listp x)
                    :verify-guards nil))    ; Note this additional xarg.
    (cond ((endp x) nil)
          (t (app (rev (cdr x)) (list (car x))))))

  (defthm true-listp-rev
    (implies (true-listp x2)
             (true-listp (rev x2))))

  (verify-guards rev)
  ~ev[]
  The ACL2 system can actually admit the original definition of
  ~c[rev], verifying the ~il[guard]s as part of the ~ilc[defun] event.  The
  reason is that, in this particular case, the system's heuristics
  just happen to hit upon the lemma ~c[true-listp-rev].  But in many
  more complicated functions it is necessary for the user to formulate
  the inductively provable properties before ~il[guard] verification is
  attempted.

  ~st[Remark on computation of guard conjectures and evaluation].  When ACL2
  computes the ~il[guard] conjecture for the body of a function, it
  evaluates any ground subexpressions (those with no free variables), for
  calls of functions whose ~c[:]~ilc[executable-counterpart] ~il[rune]s are
  ~ilc[enable]d.  Note that here, ``enabled'' refers to the current global
  ~il[theory], not to any ~c[:]~ilc[hints] given to the guard verification
  process; after all, the guard conjecture is computed even before its initial
  goal is produced.  Also note that this evaluation is done in an environment
  as though ~c[:set-guard-checking :all] had been executed, so that we can
  trust that this evaluation takes place without guard violations;
  ~pl[set-guard-checking].

  If you want to verify the ~il[guard]s on functions that are built into ACL2,
  you will first need to put them into ~c[:]~ilc[logic] mode.
  ~l[verify-termination], specifically the ``Remark on system functions'' in
  that ~il[documentation]."

; Warning: See the Important Boot-Strapping Invariants before modifying!

 (list 'verify-guards-fn
       (list 'quote name)
       'state
       (list 'quote hints)
       (list 'quote otf-flg)
       (list 'quote guard-debug)
       (list 'quote doc)
       (list 'quote event-form)))

(defdoc defpun

  ":Doc-Section acl2::Events

  define a tail-recursive function symbol~/~/

  ~c[Defpun] is a macro developed by Pete Manolios and J Moore that allows
  tail-recursive definitions.  It is defined in distributed book
  ~c[books/misc/defpun.lisp], so to use it, execute the following event.
  ~bv[]
  (include-book \"misc/defpun\" :dir :system)
  ~ev[]
  Details of defpun are provided by Manolios and Moore in the ``Partial
  Functions in ACL2'' published with the ACL2 2000 workshop; see
  ~url[http://www.cs.utexas.edu/users/moore/acl2/workshop-2000/].  Also see
  ~url[http://www.cs.utexas.edu/users/moore/publications/defpun/index.html].

  A variant, ~c[defp], has been developed by Matt Kaufmann to allow more
  general forms of tail recursion.  If ~c[defpun] doesn't work for you, try
  ~c[defp] by first executing the following event.
  ~bv[]
  (include-book \"misc/defp\" :dir :system)
  ~ev[]

  Sandip Ray has contributed a variant of ~c[defpun], ~c[defpun-exec], that
  supports executability.  See distributed book
  ~c[books/defexec/defpun-exec/defpun-exec.lisp]:
  ~bv[]
  (include-book \"defexec/defpun-exec/defpun-exec\" :dir :system)
  ~ev[]
  He has also contributed book
  ~c[books/misc/misc2/defpun-exec-domain-example.lisp], for functions that are
  uniquely defined in a particular domain.")

#+acl2-loop-only
(defmacro defmacro (&whole event-form &rest mdef)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section acl2::Events

  define a macro~/
  ~bv[]
  Example Defmacros:
  (defmacro xor (x y)
    (list 'if x (list 'not y) y))

  (defmacro git (sym key)
    (list 'getprop sym key nil
          '(quote current-acl2-world)
          '(w state)))

  (defmacro one-of (x &rest rst)
    (declare (xargs :guard (symbol-listp rst)))
    (cond ((null rst) nil)
          (t (list 'or
                   (list 'eq x (list 'quote (car rst)))
                   (list* 'one-of x (cdr rst))))))

  Example Expansions:
  term                    macroexpansion

  (xor a b)              (if a (not b) b)
  (xor a (foo b))        (if a (not (foo b)) (foo b))

  (git 'car 'lemmas)     (getprop 'car 'lemmas nil
                                  'current-acl2-world
                                  (w state))

  (one-of x a b c)       (or (eq x 'a)
                             (or (eq x 'b)
                                 (or (eq x 'c) nil)))

  (one-of x 1 2 3)       ill-formed (guard violation)~/

  General Form:
  (defmacro name macro-args doc-string dcl ... dcl body)
  ~ev[]
  where ~c[name] is a new symbolic name (~pl[name]), ~c[macro-args] specifies
  the formal parameters of the macro, and ~c[body] is a term.  The formal
  parameters can be specified in a much more general way than is allowed by
  ACL2 ~ilc[defun] ~il[events]; ~pl[macro-args] for a description of keyword
  (~c[&key]) and optional (~c[&optional]) parameters as well as other so-called
  ``lambda-list keywords'', ~c[&rest] and ~c[&whole].  ~ilc[Doc-string] is an
  optional ~il[documentation] string; ~pl[doc-string].  Each ~c[dcl] is an
  optional declaration (~pl[declare]) except that the only ~ilc[xargs] keyword
  permitted by ~c[defmacro] is ~c[:]~ilc[guard].

  For compute-intensive applications see the distributed book
  ~c[misc/defmac.lisp], which can speed up macroexpansion by introducing an
  auxiliary ~c[defun].  For more information, evaluate the form
  ~c[(include-book \"misc/defmac\" :dir :system)] and then evaluate
  ~c[:doc defmac].

  Macroexpansion occurs when a form is read in, i.e., before the
  evaluation or proof of that form is undertaken.  To experiment with
  macroexpansion, ~pl[trans].  When a form whose ~ilc[car] is ~c[name]
  arises as the form is read in, the arguments are bound as described
  in CLTL pp. 60 and 145, the ~il[guard] is checked, and then the ~c[body] is
  evaluated.  The result is used in place of the original form.

  In ACL2, macros do not have access to ~ilc[state].  That is, ~ilc[state]
  is not allowed among the formal parameters.  This is in part a
  reflection of CLTL, p. 143, ``More generally, an implementation of
  Common Lisp has great latitude in deciding exactly when to expand
  macro calls with a program. ...  Macros should be written in such a
  way as to depend as little as possible on the execution environment
  to produce a correct expansion.'' In ACL2, the product of
  macroexpansion is independent of the current environment and is
  determined entirely by the macro body and the functions and
  constants it references.  It is possible, however, to define macros
  that produce expansions that refer to ~ilc[state] or other single-threaded
  objects (~pl[stobj]) or variables not among the macro's arguments.
  See the ~c[git] example above.  For a related utility that avoids this
  ~ilc[state] restriction, ~pl[make-event].~/"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'defmacro-fn
        (list 'quote mdef)
        'state
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro defconst (&whole event-form name form &optional doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section acl2::Events

  define a constant~/
  ~bv[]
  Examples:
  (defconst *digits* '(0 1 2 3 4 5 6 7 8 9))
  (defconst *n-digits* (the unsigned-byte (length *digits*)))~/

  General Form:
  (defconst name term doc-string)
  ~ev[]
  where ~c[name] is a symbol beginning and ending with the character ~c[*],
  ~c[term] is a variable-free term that is evaluated to determine the
  value of the constant, and ~ilc[doc-string] is an optional ~il[documentation]
  string (~pl[doc-string]).

  When a constant symbol is used as a ~il[term], ACL2 replaces it by
  its value; ~pl[term].

  Note that ~c[defconst] uses a ``safe mode'' to evaluate its form, in order
  to avoids soundness issues but with an efficiency penalty (perhaps increasing
  the evaluation time by several hundred percent).  If efficiency is a concern,
  consider using the macro ~c[defconst-fast] instead, defined in
  ~c[books/make-event/defconst-fast.lisp], for example:
  ~bv[]
  (defconst-fast *x* (expensive-fn ...))
  ~ev[]
  Also ~il[using-tables-efficiently] for an analogous issue with ~ilc[table]
  events.

  It may be of interest to note that ~c[defconst] is implemented at the
  lisp level using ~c[defparameter], as opposed to ~c[defconstant].
  (Implementation note:  this is important for proper support of
  undoing and redefinition.)~/

  :cited-by Programming"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'defconst-fn
        (list 'quote name)
        (list 'quote form)
        'state
        (list 'quote doc)
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro defthm (&whole event-form
                  name term
                       &key (rule-classes '(:REWRITE))
                       instructions
                       hints
                       otf-flg
                       doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

  ":Doc-Section Events

  prove and name a theorem~/
  ~bv[]
  Examples:
  (defthm assoc-of-app
          (equal (app (app a b) c)
                 (app a (app b c))))
  ~ev[]
  The following nonsensical example illustrates all the optional
  arguments but is illegal because not all combinations are permitted.
  ~l[hints] for a complete list of ~il[hints].
  ~bv[]
  (defthm main
          (implies (hyps x y z) (concl x y z))
         :rule-classes (:REWRITE :GENERALIZE)
         :instructions (induct prove promote (dive 1) x
                               (dive 2) = top (drop 2) prove)
         :hints ((\"Goal\"
                  :do-not '(generalize fertilize)
                  :in-theory (set-difference-theories
                               (current-theory :here)
                               '(assoc))
                  :induct (and (nth n a) (nth n b))
                  :use ((:instance assoc-of-append
                                   (x a) (y b) (z c))
                        (:functional-instance
                          (:instance p-f (x a) (y b))
                          (p consp)
                          (f assoc)))))
         :otf-flg t
         :doc \"#0[one-liner/example/details]\")~/

  General Form:
  (defthm name term
          :rule-classes rule-classes
          :instructions instructions
          :hints        hints
          :otf-flg      otf-flg
          :doc          doc-string)
  ~ev[]
  where ~c[name] is a new symbolic name (~pl[name]), ~c[term] is a
  term alleged to be a theorem, and ~ilc[rule-classes], ~ilc[instructions],
  ~ilc[hints], ~ilc[otf-flg] and ~ilc[doc-string] are as described in their
  respective ~il[documentation].  The five keyword arguments above are
  all optional, however you may not supply both ~c[:]~ilc[instructions]
  and ~c[:]~ilc[hints], since one drives the proof checker and the other
  drives the theorem prover.  If ~c[:]~ilc[rule-classes] is not specified,
  the list ~c[(:rewrite)] is used; if you wish the theorem to generate
  no rules, specify ~c[:]~ilc[rule-classes] ~c[nil].

  When ACL2 processes a ~c[defthm] event, it first tries to prove the
  ~il[term] using the indicated hints (~pl[hints]) or ~il[instructions]
  (~pl[proof-checker]).  If it is successful, it stores the rules
  described by the rule-classes (~pl[rule-classes]), proving the
  necessary corollaries.~/"

  (list 'defthm-fn
        (list 'quote name)
        (list 'quote term)
        'state
        (list 'quote rule-classes)
        (list 'quote instructions)
        (list 'quote hints)
        (list 'quote otf-flg)
        (list 'quote doc)
        (list 'quote event-form)
        #+:non-standard-analysis ; std-p
        nil))

#+acl2-loop-only
(defmacro defthmd (&whole event-form
                          name term
                          &rest rst)

  ":Doc-Section acl2::Events

  prove and name a theorem and then disable it~/~/

  Use ~c[defthmd] instead of ~ilc[defthm] when you want to disable a theorem
  immediately after proving it.  This macro has been provided for users who
  prefer working in a mode where theorems are only enabled when explicitly
  directed by ~c[:]~ilc[in-theory].  Specifically, the form
  ~bv[]
  (defthmd NAME TERM ...)
  ~ev[]
  expands to:
  ~bv[]
  (progn
    (defthmd NAME TERM ...)
    (with-output
     :off summary
     (in-theory (disable NAME)))
    (value NAME)).
  ~ev[]

  Note that ~c[defthmd] commands are never redundant (~pl[redundant-events]).
  Even if the ~c[defthm] event is redundant, then the ~ilc[in-theory] event
  will still be executed.

  The summary for the ~ilc[in-theory] event is suppressed.  ~l[defthm] for
  documentation of ~c[defthm]."

  (declare (xargs :guard t) (ignore term rst))

  (list 'progn
        (cons 'defthm (cdr event-form))
        (list
         'with-output
         :off 'summary
         (list 'in-theory
               (list 'disable name)))
        (list 'value-triple (list 'quote (xd-name 'defthmd name)))))

#+(and acl2-loop-only :non-standard-analysis)
(defmacro defthm-std (&whole event-form
                      name term
                       &key (rule-classes '(:REWRITE))
                       instructions
                       hints
                       otf-flg
                       doc)

  ":Doc-Section Events

  prove and name a theorem~/~/

  ~l[defthm] for details.  (More documentation on features
  related to non-standard analysis may be available in the future.)"

  (list 'defthm-fn
        (list 'quote name)
        (list 'quote term)
        'state
        (list 'quote rule-classes)
        (list 'quote instructions)
        (list 'quote hints)
        (list 'quote otf-flg)
        (list 'quote doc)
        (list 'quote event-form)
        t))

#+acl2-loop-only
(defmacro defaxiom (&whole event-form name term
                    &key (rule-classes '(:REWRITE))
                         doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

  ":Doc-Section Events

  add an axiom~/

  WARNING: We strongly recommend that you not add axioms.  If at all
  possible you should use ~ilc[defun] or ~ilc[mutual-recursion] to define new
  concepts recursively or use ~ilc[encapsulate] to constrain them
  constructively.  If your goal is to defer a proof by using a
  top-down style, consider using ~ilc[skip-proofs]; see the discussion
  on ``Top-Down Proof'' in Section B.1.2 of ``Computer-Aided
  Reasoning: An Approach.''  Adding new axioms frequently renders the
  logic inconsistent.
  ~bv[]
  Example:
  (defaxiom sbar (equal t nil)
            :rule-classes nil
            :doc \":Doc-Section ...\")~/

  General Form:
  (defaxiom name term
           :rule-classes rule-classes
           :doc          doc-string)
  ~ev[]
  where ~c[name] is a new symbolic name (~pl[name]), ~c[term] is a term
  intended to be a new axiom, and ~ilc[rule-classes] and ~ilc[doc-string] are as
  described in the corresponding ~il[documentation] topics .  The two keyword
  arguments are optional.  If ~c[:]~ilc[rule-classes] is not supplied, the list
  ~c[(:rewrite)] is used; if you wish the axiom to generate no rules,
  specify ~c[:]~ilc[rule-classes] ~c[nil].~/"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'defaxiom-fn
        (list 'quote name)
        (list 'quote term)
        'state
        (list 'quote rule-classes)
        (list 'quote doc)
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro deflabel (&whole event-form name &key doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section Events

  build a landmark and/or add a ~il[documentation] topic~/
  ~bv[]
  Examples:
  (deflabel interp-section
     :doc
     \":Doc-Section ...\")~/

  General Form:
  (deflabel name :doc doc-string)
  ~ev[]
  where ~c[name] is a new symbolic name (~pl[name]) and ~ilc[doc-string]
  is an optional ~il[documentation] string (~pl[doc-string]).  This
  event adds the ~il[documentation] string for symbol ~c[name] to the ~c[:]~ilc[doc] data
  base.  By virtue of the fact that ~c[deflabel] is an event, it also
  marks the current ~il[history] with the ~c[name].  Thus, even undocumented
  labels are convenient as landmarks in a proof development.  For
  example, you may wish to undo back through some label or compute a
  theory expression (~pl[theories]) in terms of some labels.
  ~c[Deflabel] ~il[events] are never considered redundant.
  ~l[redundant-events].

  ~l[defdoc] for a means of attaching a ~il[documentation] string to a
  name without marking the current ~il[history] with that name.~/"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'deflabel-fn
        (list 'quote name)
        'state
        (list 'quote doc)
        (list 'quote event-form)))

(deflabel theories
  :doc
  ":Doc-Section Theories

  sets of ~il[rune]s to enable/disable in concert~/
  ~bv[]
  Example: '((:definition app)
             (:executable-counterpart app)
             rv
             (rv)
             assoc-of-app)
  ~ev[]
  See:~/

  A theory is a list of ``runic designators'' as described below.
  Each runic designator denotes a set of ``runes'' (~pl[rune]) and
  by unioning together the runes denoted by each member of a theory we
  define the set of runes corresponding to a theory.  Theories are
  used to control which rules are ``~il[enable]d,'' i.e., available for
  automatic application by the theorem prover.  There is always a
  ``current'' theory.  A rule is ~il[enable]d precisely if its ~il[rune] is an
  element of the set of ~il[rune]s corresponding to the current theory.  At
  the top-level, the current theory is the theory selected by the most
  recent ~ilc[in-theory] event, extended with the rule names introduced
  since then.  Inside the theorem prover, the ~c[:]~ilc[in-theory] hint
  (~pl[hints]) can be used to select a particular theory as
  current during the proof attempt for a particular goal.

  Theories are generally constructed by ``theory expressions.''
  Formally, a theory expression is any term, containing at most the
  single free variable ~ilc[world], that when evaluated with ~ilc[world] bound to
  the current ACL2 world (~pl[world]) produces a theory.  ACL2
  provides various functions for the convenient construction and
  manipulation of theories.  These are called ``theory functions''
  (~pl[theory-functions]).  For example, the theory function
  ~ilc[union-theories] takes two theories and produces their union.  The
  theory function ~ilc[universal-theory] returns the theory containing all
  known rule names as of the introduction of a given logical name.
  But a theory expression can contain constants, e.g.,
  ~bv[]
  '(assoc (assoc) (:rewrite car-cons) car-cdr-elim)
  ~ev[]
  and user-defined functions.  The only important criterion is that a
  theory expression mention no variable freely except ~ilc[world] and
  evaluate to a theory.

  More often than not, theory expressions typed by the user do not
  mention the variable ~ilc[world].  This is because user-typed theory
  expressions are generally composed of applications of ACL2's theory
  functions.  These ``functions'' are actually macros that expand into
  terms in which ~ilc[world] is used freely and appropriately.  Thus, the
  technical definition of ``theory expression'' should not mislead you
  into thinking that interestng theory expressions must mention ~ilc[world];
  they probably do and you just didn't know it!

  One aspect of this arrangement is that theory expressions cannot
  generally be evaluated at the top-level of ACL2, because ~ilc[world] is
  not bound.  To see the value of a theory expression, ~c[expr], at the
  top-level, type
  ~bv[]
  ACL2 !>(LET ((WORLD (W STATE))) expr).
  ~ev[]
  However, because the built-in theories are quite long, you may be
  sorry you printed the value of a theory expression!

  A theory is a true list of runic designators and to each theory
  there corresponds a set of ~il[rune]s, obtained by unioning together the
  sets of ~il[rune]s denoted by each runic designator.  For example, the
  theory constant
  ~bv[]
     '(assoc (assoc) (:rewrite car-cons) car-cdr-elim)
  ~ev[]
  corresponds to the set of ~il[rune]s
  ~bv[]
     {(:definition assoc)
      (:induction assoc)
      (:executable-counterpart assoc)
      (:elim car-cdr-elim)
      (:rewrite car-cons)} .
  ~ev[]
  Observe that the theory contains four elements but its runic
  correspondent contains five.  That is because some designators
  denote sets of several ~il[rune]s.  If the above theory were selected as
  current then the five rules named in its runic counterpart would be
  ~il[enable]d and all other rules would be ~il[disable]d.

  We now precisely define the runic designators and the set of ~il[rune]s
  denoted by each.~bq[]

  o A rune is a runic designator and denotes the singleton set
  containing that rune.

  o If ~c[symb] is a function symbol introduced with a ~ilc[defun] (or ~ilc[defuns])
  event, then ~c[symb] is a runic designator and denotes the set containing
  the runes ~c[(:definition symb)] and ~c[(:induction symb)], omitting the
  latter if no such ~il[induction] rule exists (presumably because the function's
  definition is not singly recursive).

  o If ~c[symb] is a function symbol introduced with a ~ilc[defun] (or ~ilc[defuns])
  event, then ~c[(symb)] is a runic designator and denotes the singleton
  set containing the rune ~c[(:executable-counterpart symb)].

  o If ~c[symb] is the name of a ~ilc[defthm] (or ~ilc[defaxiom]) event that
  introduced at least one rule, then ~c[symb] is a runic designator and
  denotes the set of the names of all rules introduced by the named
  event.

  o If ~c[str] is the string naming some ~ilc[defpkg] event and ~c[symb] is the
  symbol returned by ~c[(intern str \"ACL2\")], then ~c[symb] is a runic
  designator and denotes the singleton set containing ~c[(:rewrite symb)],
  which is the name of the rule stating the conditions under which the
  ~ilc[symbol-package-name] of ~c[(intern x str)] is ~c[str].

  o If ~c[symb] is the name of a ~ilc[deftheory] event, then ~c[symb] is a runic
  designator and denotes the runic theory corresponding to ~c[symb].

  ~eq[]These conventions attempt to implement the Nqthm-1992 treatment of
  theories.  For example, including a function name, e.g., ~ilc[assoc], in
  the current theory ~il[enable]s that function but does not ~il[enable] the
  executable counterpart.  Similarly, including ~c[(assoc)] ~il[enable]s the
  executable counterpart (Nqthm's ~c[*1*assoc]) but not the symbolic
  definition.  And including the name of a proved lemma ~il[enable]s all of
  the rules added by the event.  These conventions are entirely
  consistent with Nqthm usage.  Of course, in ACL2 one can include
  explicitly the ~il[rune]s naming the rules in question and so can avoid
  entirely the use of non-runic elements in theories.

  Because a ~il[rune] is a runic designator denoting the set containing
  that ~il[rune], a list of ~il[rune]s is a theory and denotes itself.  We call
  such theories ``runic theories.''  To every theory there corresponds
  a runic theory obtained by unioning together the sets denoted by
  each designator in the theory.  When a theory is selected as
  ``current'' it is actually its runic correspondent that is
  effectively used.  That is, a ~il[rune] is ~il[enable]d iff it is a member of
  the runic correspondent of the current theory.  The value of a
  theory defined with ~ilc[deftheory] is the runic correspondent of the
  theory computed by the defining theory expression.  The theory
  manipulation functions, e.g., ~ilc[union-theories], actually convert their
  theory arguments to their runic correspondents before performing the
  required set operation.  The manipulation functions always return
  runic theories.  Thus, it is sometimes convenient to think of
  (non-runic) theories as merely abbreviations for their runic
  correspondents, abbreviations which are ``expanded'' at the first
  opportunity by theory manipulation functions and the ``theory
  consumer'' functions such as ~ilc[in-theory] and ~ilc[deftheory].~/")

#+acl2-loop-only
(defmacro deftheory (&whole event-form name expr &key doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section Events

  define a theory (to ~il[enable] or ~il[disable] a set of rules)~/
  ~bv[]
  Example:
  (deftheory interp-theory
             (set-difference-theories
               (universal-theory :here)
               (universal-theory 'interp-section)))~/

  General Form:
  (deftheory name term :doc doc-string)
  ~ev[]
  where ~c[name] is a new symbolic name (~pl[name]), ~c[term] is a term
  that when evaluated will produce a theory (~pl[theories]), and
  ~ilc[doc-string] is an optional ~il[documentation] string
  (~pl[doc-string]).  Except for the variable ~ilc[world], ~c[term] must
  contain no free variables.  ~c[Term] is evaluated with ~ilc[world] bound to
  the current world (~pl[world]) and the resulting theory is then
  converted to a ~em[runic theory] (~pl[theories]) and associated with
  ~c[name].  Henceforth, this runic theory is returned as the value of the
  theory expression ~c[(theory name)].

  The value returned is the length of the resulting theory.  For example, in
  the following, the theory associated with ~c['FOO] has 54 ~il[rune]s:
  ~bv[]
  ACL2 !>(deftheory foo (union-theories '(binary-append)
                                        (theory 'minimal-theory)))

  Summary
  Form:  ( DEFTHEORY FOO ...)
  Rules: NIL
  Warnings:  None
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)
   54
  ACL2 !>
  ~ev[]~/

  :cited-by Theories"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'deftheory-fn
        (list 'quote name)
        (list 'quote expr)
        'state
        (list 'quote doc)
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro defstobj (&whole event-form name &rest args)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations (other than those
; that are always skipped), remove it from the list of exceptions in
; install-event just below its "Comment on irrelevance of skip-proofs".

  ":Doc-Section Events

  define a new single-threaded object ~/

  Note: Novices are advised to avoid ~c[defstobj], perhaps instead using
  distributed book ~c[books/data-structures/structures.lisp].  At the least,
  consider using ~c[(]~ilc[set-verify-guards-eagerness]~c[ 0)] to avoid
  ~il[guard] verification.

  ~bv[]
  Example:
  (defstobj st
            (reg :type (array (unsigned-byte 31) (8))
                 :initially 0)
            (p-c :type (unsigned-byte 31)
                 :initially 555)
            halt                  ; = (halt :type t :initially nil)
            (mem :type (array (unsigned-byte 31) (64))
                 :initially 0 :resizable t))

  General Form:
  (defstobj name
            (field1 :type type1 :initially val1 :resizable b1)
            ...
            (fieldk :type typek :initially valk :resizable bk)
            :renaming alist
            :doc doc-string
            :inline inline-flag)
  ~ev[]
  where ~c[name] is a new symbol, each ~c[fieldi] is a symbol, each ~c[typei]
  is either a ~ilc[type-spec] or ~c[(ARRAY] ~ilc[type-spec] ~c[(max))], each
  ~c[vali] is an object satisfying ~c[typei], and each ~c[bi] is ~c[t] or
  ~c[nil].  Each pair ~c[:initially vali] and ~c[:resizable bi] may be omitted;
  more on this below.  The ~c[alist] argument is optional and allows the user
  to override the default function names introduced by this event.  The
  ~ilc[doc-string] is also optional.  The ~c[inline-flag] Boolean argument is
  also optional and declares to ACL2 that the generated access and update
  functions for the stobj should be implemented as macros under the hood (which
  has the effect of inlining the function calls).  We describe further
  restrictions on the ~c[fieldi], ~c[typei], ~c[vali], and on ~c[alist] below.
  We recommend that you read about single-threaded objects (stobjs) in ACL2
  before proceeding; ~pl[stobj].

  The effect of this event is to introduce a new single-threaded object (i.e.,
  a ``~il[stobj]''), named ~c[name], and the associated recognizers, creator,
  accessors, updaters, constants, and, for fields of ~c[ARRAY] type, length and
  resize functions.
  ~/

  ~em[The Single-Threaded Object Introduced]

  The ~c[defstobj] event effectively introduces a new global variable, named
  ~c[name], which has as its initial logical value a list of ~c[k] elements,
  where ~c[k] is the number of ``field descriptors'' provided.  The elements
  are listed in the same order in which the field descriptors appear.  If the
  ~c[:type] of a field is ~c[(ARRAY type-spec (max))] then the corresponding
  element of the stobj is initially a list of length ~c[max] containing the
  value, ~c[val], specified by ~c[:initially val].  Otherwise, the ~c[:type] of
  the field is a ~ilc[type-spec] and the corresponding element of the stobj is
  the specified initial value ~c[val].  (The actual representation of the stobj
  in the underlying Lisp may be quite different; ~pl[stobj-example-2].  For the
  moment we focus entirely on the logical aspects of the object.)

  In addition, the ~c[defstobj] event introduces functions for recognizing and
  creating the stobj and for recognizing, accessing, and updating its fields.
  For fields of ~c[ARRAY] type, length and resize functions are also
  introduced.  Constants are introduced that correspond to the accessor
  functions.

  ~em[Restrictions on the Field Descriptions in Defstobj]

  Each field descriptor is of the form:
  ~bv[]
  (fieldi :TYPE typei :INITIALLY vali)
  ~ev[]
  Note that the type and initial value are given in ``keyword argument'' format
  and may be given in either order.  The ~c[typei] and ~c[vali] ``arguments''
  are not evaluated.  If omitted, the type defaults to ~c[t] (unrestricted) and
  the initial value defaults to ~c[nil].

  Each ~c[typei] must be either a ~ilc[type-spec] or else a list of the form
  ~c[(ARRAY type-spec (max))].  The latter forms are said to be ``array
  types.''  Examples of legal ~c[typei] are:
  ~bv[]
  (INTEGER 0 31)
  (SIGNED-BYTE 31)
  (ARRAY (SIGNED-BYTE 31) (16))
  ~ev[]

  The ~c[typei] describes the objects which are expected to occupy the given
  field.  Those objects in ~c[fieldi] should satisfy ~c[typei].  We are more
  precise below about what we mean by ``expected.''  We first present the
  restrictions on ~c[typei] and ~c[vali].

  Non-Array Types

  When ~c[typei] is a ~ilc[type-spec] it restricts the contents, ~c[x], of
  ~c[fieldi] according to the ``meaning'' formula given in the table for
  ~ilc[type-spec].  For example, the first ~c[typei] above restricts the field
  to be an integer between 0 and 31, inclusive.  The second restricts the field
  to be an integer between -2^30 and (2^30)-1, inclusive.

  The initial value, ~c[vali], of a field description may be any ACL2 object
  but must satisfy ~c[typei].  Note that ~c[vali] is not a form to be evaluated
  but an object.  A form that evaluates to ~c[vali] could be written ~c['vali],
  but ~c[defstobj] does not expect you to write the quote mark.  For example,
  the field description
  ~bv[]
  (days-off :initially (saturday sunday))
  ~ev[]
  describes a field named ~c[days-off] whose initial value is the list
  consisting of the two symbols ~c[SATURDAY] and ~c[SUNDAY].  In particular,
  the initial value is NOT obtained by applying the function ~c[saturday] to
  the variable ~c[sunday]!  Had we written
  ~bv[]
  (days-off :initially '(saturday sunday))
  ~ev[]
  it would be equivalent to writing
  ~bv[]
  (days-off :initially (quote (saturday sunday)))
  ~ev[]
  which would initialize the field to a list of length two, whose first element
  is the symbol ~c[quote] and whose second element is a list containing the
  symbols ~c[saturday] and ~c[sunday].

  Array Types

  When ~c[typei] is of the form ~c[(ARRAY type-spec (max))], the field is
  supposed to be a list of items, initially of length ~c[max], each of which
  satisfies the indicated ~c[type-spec].  ~c[Max] must be a non-negative
  integer.  Thus,
  ~bv[]
  (ARRAY (SIGNED-BYTE 31) (16))
  ~ev[]
  restricts the field to be a list of integers, initially of length 16, where
  each integer in the list is a ~c[(SIGNED-BYTE 31)].  We sometimes call such a
  list an ``array'' (because it is represented as an array in the underlying
  Common Lisp).  The elements of an array field are indexed by position,
  starting at 0.  Thus, the maximum legal index of an array field is ~c[max]-1.
  Note that ~c[max] must be less than the Common Lisp constant
  ~c[array-dimension-limit], and also (though this presumably follows) less
  than the Common Lisp constant ~c[array-total-size-limit].

  Note also that the ~c[ARRAY] type requires that the ~c[max] be enclosed in
  parentheses.  This makes ACL2's notation consistent with the Common Lisp
  convention of describing the (multi-)dimensionality of arrays.  But ACL2
  currently supports only single dimensional arrays in stobjs.

  For array fields, the initial value ~c[vali] must be an object satisfying the
  ~ilc[type-spec] of the ~c[ARRAY] description.  The initial value of the field
  is a list of ~c[max] repetitions of ~c[vali].

  Array fields can be ``resized,'' that is, their lengths can be changed, if
  ~c[:resizable t] is supplied as shown in the example and General Form above.
  The new length must satisfy the same restriction as does ~c[max], as
  described above.  Each array field in a ~c[defstobj] event gives rise to a
  length function, which gives the length of the field, and a resize function,
  which modifies the length of the field if ~c[:resizable t] was supplied with
  the field when the ~c[defstobj] was introduced and otherwise causes an error.
  If ~c[:resizable t] was supplied and the resize function specifies a new
  length ~c[k], then: if ~c[k] is less than the existing array length, the array
  is shortened simply by dropping elements with index at least ~c[k];
  otherwise, the array is extended to length ~c[k] by mapping the new indices
  to the initial value (supplied by ~c[:initially], else default ~c[nil]).

  Array resizing is relatively slow, so we recommend using it somewhat
  sparingly.

  ~em[The Default Function Names]

  To recap, in
  ~bv[]
  (defstobj name
            (field1 :type type1 :initially val1)
            ...
            (fieldk :type typek :initially valk)
            :renaming alist
            :doc doc-string
            :inline inline-flag)
  ~ev[]
  ~c[name] must be a new symbol, each ~c[fieldi] must be a symbol,
  each ~c[typei] must be a ~ilc[type-spec] or ~c[(ARRAY type-spec (max))],
  and each ~c[vali] must be an object satisfying ~c[typei].

  Roughly speaking, for each ~c[fieldi], a ~c[defstobj] introduces a
  recognizer function, an accessor function, and an updater function.
  The accessor function, for example, takes the stobj and returns the
  indicated component; the updater takes a new component value and the
  stobj and return a new stobj with the component replaced by the new
  value.  But that summary is inaccurate for array fields.

  The accessor function for an array field does not take the stobj
  and return the indicated component array, which is a list of length
  ~c[max].  Instead, it takes an additional index argument and
  returns the indicated element of the array component.  Similarly,
  the updater function for an array field takes an index, a new
  value, and the stobj, and returns a new stobj with the indicated
  element replaced by the new value.

  These functions ~-[] the recognizer, accessor, and updater, and also
  length and resize functions in the case of array fields ~-[] have
  ``default names.''  The default names depend on the field name,
  ~c[fieldi], and on whether the field is an array field or not.  For
  clarity, suppose ~c[fieldi] is named ~c[c]. The default names are
  shown below in calls, which also indicate the arities of the
  functions.  In the expressions, we use ~c[x] as the object to be
  recognized by field recognizers, ~c[i] as an array index, ~c[v] as
  the ``new value'' to be installed by an updater, and ~c[name] as the
  single-threaded object.

  ~bv[]
                   non-array field        array field
  recognizer         (cP x)                (cP x)
  accessor           (c name)              (cI i name)
  updater            (UPDATE-c v name)     (UPDATE-cI i v name)
  length                                   (c-LENGTH name)
  resize                                   (RESIZE-c k name)
  ~ev[]

  Finally, a recognizer and a creator for the entire single-threaded
  object are introduced.  The creator returns the initial stobj, but
  may only be used in limited contexts; ~pl[with-local-stobj].  If
  the single-threaded object is named ~c[name], then the default names
  and arities are as shown below.
  ~bv[]
  top recognizer     (nameP x)
  creator            (CREATE-name)
  ~ev[]

  For example, the event
  ~bv[]
  (DEFSTOBJ $S
    (X :TYPE INTEGER :INITIALLY 0)
    (A :TYPE (ARRAY (INTEGER 0 9) (3)) :INITIALLY 9))
  ~ev[]
  introduces a stobj named ~c[$S].  The stobj has two fields, ~c[X] and
  ~c[A].  The ~c[A] field is an array.  The ~c[X] field contains an
  integer and is initially 0.  The ~c[A] field contains a list of
  integers, each between 0 and 9, inclusively.  Initially, each of the
  three elements of the ~c[A] field is 9.

  This event introduces the following sequence of definitions:
  ~bv[]
  (DEFUN XP (X) ...)               ; recognizer for X field
  (DEFUN AP (X) ...)               ; recognizer of A field
  (DEFUN $SP ($S) ...)             ; top-level recognizer for stobj $S
  (DEFUN CREATE-$S () ...)         ; creator for stobj $S
  (DEFUN X ($S) ...)               ; accessor for X field
  (DEFUN UPDATE-X (V $S) ...)      ; updater for X field
  (DEFUN A-LENGTH ($S) ...)        ; length of A field
  (DEFUN RESIZE-A (K $S) ...)      ; resizer for A field
  (DEFUN AI (I $S) ...)            ; accessor for A field at index I
  (DEFUN UPDATE-AI (I V $S) ...)   ; updater for A field at index I
  ~ev[]

  ~em[Avoiding the Default Function Names]

  If you do not like the default names listed above you may use the
  optional ~c[:renaming] alist to substitute names of your own
  choosing.  Each element of ~c[alist] should be of the form
  ~c[(fn1 fn2)], where ~c[fn1] is a default name and ~c[fn2] is your choice
  for that name.

  For example
  ~bv[]
  (DEFSTOBJ $S
    (X :TYPE INTEGER :INITIALLY 0)
    (A :TYPE (ARRAY (INTEGER 0 9) (3)) :INITIALLY 9)
    :renaming ((X XACCESSOR) (CREATE-$S MAKE$S)))
  ~ev[]
  introduces the following definitions
  ~bv[]
  (DEFUN XP (X) ...)               ; recognizer for X field
  (DEFUN AP (X) ...)               ; recognizer of A field
  (DEFUN $SP ($S) ...)             ; top-level recognizer for stobj $S
  (DEFUN MAKE$S () ...)            ; creator for stobj $S
  (DEFUN XACCESSOR ($S) ...)       ; accessor for X field
  (DEFUN UPDATE-X (V $S) ...)      ; updater for X field
  (DEFUN A-LENGTH ($S) ...)        ; length of A field
  (DEFUN RESIZE-A (K $S) ...)      ; resizer for A field
  (DEFUN AI (I $S) ...)            ; accessor for A field at index I
  (DEFUN UPDATE-AI (I V $S) ...)   ; updater for A field at index I
  ~ev[]
  Note that even though the renaming alist substitutes ``~c[XACCESSOR]''
  for ``~c[X]'' the updater for the ~c[X] field is still called
  ``~c[UPDATE-X].''  That is because the renaming is applied to the
  default function names, not to the field descriptors in the
  event.

  Use of the ~c[:renaming] alist may be necessary to avoid name
  clashes between the default names and and pre-existing function
  symbols.

  ~em[Constants]

  ~c[Defstobj] events also introduce constant definitions
  (~pl[defconst]).  One constant is introduced for each accessor
  function by prefixing and suffixing a `~c[*]' character on the function
  name.  The value of that constant is the position of the field being
  accessed.  For example, if the accessor functions are ~c[a], ~c[b], and ~c[c],
  in that order, then the following constant definitions are introduced.
  ~bv[]
  (defconst *a* 0)
  (defconst *b* 1)
  (defconst *c* 2)
  ~ev[]
  These constants are used for certain calls of ~ilc[nth] and ~ilc[update-nth]
  that are displayed to the user in proof output.  For example, for
  stobj ~c[st] with accessor functions ~c[a], ~c[b], and ~c[c], in that order, the
  term ~c[(nth '2 st)] would be printed during a proof as ~c[(nth *c* st)].
  Also ~pl[term], in particular the discussion there of untranslated
  terms, and ~pl[nth-aliases-table].

  ~em[Inspecting the Effects of a Defstobj]

  Because the stobj functions are introduced as ``sub-events'' of the
  ~c[defstobj] the history commands ~c[:]~ilc[pe] and ~c[:]~ilc[pc]
  will not print the definitions of these functions but will print
  the superior ~c[defstobj] event.  To see the definitions of these
  functions use the history command ~c[:]~ilc[pcb!].

  To see an s-expression containing the definitions what constitute the raw
  Lisp implementation of the event, evaluate the form
  ~bv[]
  (nth 4 (global-val 'cltl-command (w state)))
  ~ev[]
  ~em[immediately after] the ~c[defstobj] event has been processed.

  A ~c[defstobj] is considered redundant only if the name, field descriptors,
  renaming alist, and inline flag are identical to a previously executed
  ~c[defstobj].  Note that a redundant ~c[defstobj] does not reset the
  ~il[stobj] fields to their initial values.

  ~em[Inlining and Performance]

  The ~c[:inline] keyword argument controls whether or not accessor, updater,
  and length functions are inlined (as macros under the hood, in raw Lisp).  If
  ~c[:inline t] is provided then these are inlined; otherwise they are not.
  The advantage of inlining is potentially better performance; there have been
  contrived examples, doing essentially nothing except accessing and updating
  array fields, where inlining reduced the time by a factor of 10 or more; and
  inlining has sped up realistic examples by a factor of at least 2.  Inlining
  may get within a factor of 2 of C execution times for such contrived
  examples, and within a few percent of C execution times on realistic
  examples.

  A drawback to inlining is that redefinition may not work as expected, much as
  redefinition may not work as expected for macros: defined functions that call
  a macro, or inlined stobj function, will not see a subsequent redefinition of
  the macro or inlined function.  Another drawback to inlining is that because
  inlined functions are implemented as macros in raw Lisp, tracing
  (~pl[trace$]) will not show their calls.  These drawbacks are avoided by
  default, but the user who is not concerned about them is advised to specify
  ~c[:inline t].~/"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'defstobj-fn
        (list 'quote name)
        (list 'quote args)
        'state
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro in-theory (&whole event-form expr &key doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section Events

  designate ``current'' theory (enabling its rules)~/
  ~bv[]
  Example:
  (in-theory (set-difference-theories
               (universal-theory :here)
               '(flatten (:executable-counterpart flatten))))~/

  General Form:
  (in-theory term :doc doc-string)
  ~ev[]
  where ~c[term] is a term that when evaluated will produce a theory
  (~pl[theories]), and ~ilc[doc-string] is an optional ~il[documentation]
  string not beginning with ``~c[:doc-section] ...''.  Except for the
  variable ~ilc[world], ~c[term] must contain no free variables.  ~c[Term] is
  evaluated with the variable ~ilc[world] bound to the current ~il[world] to
  obtain a theory and the corresponding runic theory
  (~pl[theories]) is then made the current theory.  Thus,
  immediately after the ~c[in-theory], a rule is ~il[enable]d iff its rule name
  is a member of the runic interpretation (~pl[theories]) of some
  member of the value of ~c[term].  ~l[theory-functions] for a list
  of the commonly used theory manipulation functions.

  Because no unique name is associated with an ~c[in-theory] event, there
  is no way we can store the ~il[documentation] string ~ilc[doc-string] in our
  ~il[documentation] data base.  Hence, we actually prohibit ~ilc[doc-string]
  from having the form of an ACL2 ~il[documentation] string;
  ~pl[doc-string].

  Note that it is often useful to surround ~c[in-theory] ~il[events] with
  ~c[local], that is, to use ~c[(local (in-theory ...))].  This use of
  ~ilc[local] in ~ilc[encapsulate] events and ~il[books] will prevent the
  effect of this theory change from being exported outside the context of that
  ~c[encapsulate] or book.

  Also ~pl[hints] for a discussion of the ~c[:in-theory] hint, including some
  explanation of the important point that an ~c[:in-theory] hint will always be
  evaluated relative to the current ACL2 logical ~il[world], not relative to
  the theory of a previous goal.~/

  :cited-by Theories"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'in-theory-fn
        (list 'quote expr)
        'state
        (list 'quote doc)
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro in-arithmetic-theory (&whole event-form expr &key doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section Events

  designate ``current'' theory for some rewriting done in linear arithmetic~/
  ~bv[]
  Example:
  (in-arithmetic-theory '(lemma1 lemma2))~/

  General Form:
  (in-arithmetic-theory term :doc doc-string)
  ~ev[]
  where ~c[term] is a term that when evaluated will produce a theory
  (~pl[theories]), and ~ilc[doc-string] is an optional ~il[documentation]
  string not beginning with ``~c[:doc-section] ...''.  Except for the
  variable ~ilc[world], ~c[term] must contain no free variables.  ~c[Term] is
  evaluated with the variable ~ilc[world] bound to the current ~il[world] to
  obtain a theory and the corresponding runic theory
  (~pl[theories]) is then made the current theory.  Thus,
  immediately after the ~c[in-arithmetic-theory], a rule is ~il[enable]d
  iff its rule name is a member of the runic interpretation (~pl[theories])
  of some member of the value of ~c[term].

  Warning:  If ~c[term] involves macros such as ~ilc[ENABLE] and ~ilc[DISABLE]
  you will probably not get what you expect!  Those macros are defined
  relative to the ~ilc[CURRENT-THEORY].  But in this context you might
  wish they were defined in terms of the ``~c[CURRENT-ARITHMETIC-THEORY]''
  which is not actually a defined function.  We do not anticipate that users
  will repeatedly modify the arithmetic theory.  We expect ~c[term] most often
  to be a constant list of runes and so have not provided ``arithmetic theory
  manipulation functions'' analogous to ~ilc[CURRENT-THEORY] and ~ilc[ENABLE].

  BECAUSE NO UNIQUE name is associated with an ~c[in-arithmetic-theory] event,
  there is no way we can store the ~il[documentation] string ~ilc[doc-string]
  in our il[documentation] data base.  Hence, we actually prohibit ~ilc[doc-string]
  from having the form of an ACL2 ~il[documentation] string;
  ~pl[doc-string].

  ~l[non-linear-arithmetic].~/"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'in-arithmetic-theory-fn
        (list 'quote expr)
        'state
        (list 'quote doc)
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro push-untouchable (&whole event-form name fn-p &key doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section switches-parameters-and-modes

  add name or list of names to the list of untouchable symbols~/
  ~bv[]
  Examples:
  (push-untouchable my-var nil)
  (push-untouchable set-mem t)~/

  General Form:
  (push-untouchable name{s}  fn-p :doc doc-string)
  ~ev[]
  where ~c[name{s}] is a non-~c[nil] symbol or a non-~c[nil] true list of
  symbols, ~c[fn-p] is any value (but generally ~c[nil] or ~c[t]), and
  ~ilc[doc-string] is an optional ~il[documentation] string not
  beginning with ``~c[:doc-section] ...''.  If ~c[name{s}] is a symbol it
  is treated as the singleton list containing that symbol.  The effect
  of this event is to union the given symbols into the list of
  ``untouchable variables'' in the current world if ~c[fn-p] is
  ~c[nil], else to union the symbols into the list of ``untouchable
  functions''.  This event is redundant if every symbol listed is
  already a member of the appropriate untouchables list (variables or
  functions).

  When a symbol is on the untouchables list it is syntactically
  illegal for any event to call a function or macro of that name, if
  ~c[fn-p] is non-~c[nil], or to change the value of a state global
  variable of that name, if ~c[fn-p] is ~c[nil].  Thus, the effect of
  pushing a function symbol, ~c[name], onto untouchables is to prevent
  any future event from using that symbol as a function or macro, or
  as a state global variable (according to ~c[fn-p]).  This is
  generally done to ``fence off'' some primitive function symbol from
  ``users'' after the developer has used the symbol freely in the
  development of some higher level mechanism.

  Also ~pl[remove-untouchable].~/"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (declare (xargs :guard (and name
                              (or (symbolp name)
                                  (symbol-listp name)))))
  (list 'push-untouchable-fn
        (list 'quote name)
        (list 'quote fn-p)
        'state
        (list 'quote doc)
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro remove-untouchable (&whole event-form name fn-p &key doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section switches-parameters-and-modes

  remove names from lists of untouchable symbols~/

  Also ~pl[push-untouchable].

  This documentation topic is directed at those who build systems on top of
  ACL2.  We first describe a means for removing restrictions related to
  so-called ``untouchables'': functions (or macros) that cannot be called, or
  state global variables that cannot be modified or unbound, without
  intervention that requires an active trust tag (~pl[defttag]).  Then we
  describe the ~c[remove-untouchable] event.

  We begin by discussing untouchable state global variables
  ~c[temp-touchable-vars] and ~c[temp-touchable-fns], which initially have
  value ~c[nil].  These can often be used in place of ~c[remove-untouchable].
  When the value is ~c[t], no variable (respectively, no function or macro) is
  treated as untouchable, regardless of the set of initial untouchables or the
  ~c[remove-untouchable] or ~ilc[push-untouchable] ~il[events] that have been
  admitted.  Otherwise the value of each of these two variables is a
  ~ilc[symbol-listp], and no member of this list is treated as an untouchable
  variable (in the case of ~c[temp-touchable-vars]) or as an untouchable
  function or macro (in the case of ~c[temp-touchable-fns]).  These two state
  global variables can be set by ~c[set-temp-touchable-vars] and
  ~c[set-temp-touchable-fns], respectively, provided there is an active trust
  tag (~pl[defttag]).  Here is an illustrative example.  This macro executes the
  indicated forms in a context where there are no untouchable variables, but
  requires an active trust tag when invoked.
  ~bv[]
  (defmacro with-all-touchable (&rest forms)
    `(progn!
      :state-global-bindings
      ((temp-touchable-vars t set-temp-touchable-vars))
      (progn! ,@forms)))
  ~ev[]
  An equivalent version, which however is not recommended since
  ~c[state-global-let*] may have surprising behavior in raw Lisp, is as
  follows.
  ~bv[]
  (defmacro with-all-touchable (&rest forms)
    `(progn!
      (state-global-let*
       ((temp-touchable-vars t set-temp-touchable-vars))
       (progn! ,@forms))))
  ~ev[]
  Finally, the value ~c[t] for ~c[temp-touchable-vars] removes the requirement
  that built-in state globals cannot be made unbound (with
  ~c[makunbound-global]).~/

  We now turn to the ~c[remove-untouchable] event, in case the approach above
  is for some reason not adequate.  This event is illegal by default, since it
  can be used to provide access to ACL2 internal functions and data structures
  that are intentionally made untouchable for the user.  If you want to call
  it, you must first create an active trust tag; ~pl[defttag].

  ~bv[]
  Examples:
  (remove-untouchable my-var nil)
  (remove-untouchable set-mem t)

  General Form:
  (remove-untouchable name{s}  fn-p :doc doc-string)
  ~ev[]
  where ~c[name{s}] is a non-~c[nil] symbol or a non-~c[nil] true list of symbols,
  ~c[fn-p] is any value (but generally ~c[nil] or ~c[t]), and ~ilc[doc-string]
  is an optional ~il[documentation] string not beginning with
  ``~c[:doc-section] ...''.  If ~c[name{s}] is a symbol it is treated as the
  singleton list containing that symbol.  The effect of this event is to remove
  the given symbols from the list of ``untouchable variables'' in the current
  world if ~c[fn-p] is ~c[nil], else to remove the symbols into the list of
  ``untouchable functions''.  This event is redundant if no symbol listed is a
  member of the appropriate untouchables list (variables or functions).~/"

  (declare (xargs :guard (and name
                              (or (symbolp name)
                                  (symbol-listp name)))))
  `(cond ((not (ttag (w state)))
          (er soft 'remove-untouchable
              "It is illegal to execute remove-untouchable when there is no ~
               active ttag; see :DOC defttag."))
         (t ,(list 'remove-untouchable-fn
                   (list 'quote name)
                   (list 'quote fn-p)
                   'state
                   (list 'quote doc)
                   (list 'quote event-form)))))

#+acl2-loop-only
(defmacro set-body (&whole event-form fn name-or-rune)

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section Events

  set the definition body~/
  ~bv[]
  Examples:
  (set-body foo (:definition foo)) ; restore original definition of foo
  (set-body foo foo) ; same as just above
  (set-body foo my-foo-def) ; use my-foo-def for the body of foo
  (set-body foo (:definition my-foo-def)) ; same as just above
  ~ev[]
  Rules of class ~c[:]~ilc[definition] can install a new definition body, used
  for example by ~c[:expand] ~il[hints].  ~l[definition] and also ~pl[hints]
  for a detailed discussion of the ~c[:install-body] fields of
  ~c[:]~ilc[definition] rules and their role in ~c[:expand] hints.

  There may be several such definitions, but by default, the latest one is used
  by ~c[:expand] hints.  Although the ~c[:with] keyword may be used in
  ~c[:expand] hints to override this behavior locally (~pl[hints]), it may be
  convenient to install a definition for expansion other than the latest one
  ~-[] for example, the original definition.  ~c[Set-body] may be used for this
  purpose.

  ~bv[]
  General Form:
  (set-body function-symbol rule-name)
  ~ev[]
  where ~c[rule-name] is either a ~c[:definition] ~il[rune] or is a function
  symbol, ~c[sym], which represents the rune ~c[(:definition sym)].

  You can view all definitions available for expansion;
  ~pl[show-bodies].~/~/"

  `(set-body-fn ',fn ',name-or-rune state ',event-form))

#+acl2-loop-only
(defmacro table (&whole event-form name &rest args)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section Events

  user-managed tables~/
  ~bv[]
  Examples:
  (table tests 1 '(...))                ; set contents of tests[1] to '(...)
  (table tests 25)                      ; get contents of tests[25]
  (table tests)                         ; return table tests as an alist
  (table tests nil nil :clear)          ; clear table tests
  (table tests nil '((foo . 7)) :clear) ; set table tests to (foo 7)
  (table tests nil nil :guard)          ; fetch the table guard
  (table tests nil nil :guard term)     ; set the table guard~/

  General Form:
  (table table-name key-term value-term op term)
  ~ev[]
  where ~c[table-name] is a symbol that is the name of a (possibly new)
  table, ~c[key-term] and ~c[value-term], if present, are arbitrary terms
  involving (at most) the single variable ~ilc[world], ~c[op], if present, is
  one of the table operations below, and ~c[term], if present, is a term.
  ~c[Table] returns an acl2 ``error triple.'' The effect of ~c[table] on ~ilc[state]
  depends on ~c[op] and how many arguments are presented.  Some
  invocations actually have no effect on the ACL2 ~il[world] and hence an
  invocation of ~c[table] is not always an ``event''.  We explain below,
  after giving some background information.

  ~b[Important Note:]  The ~c[table] forms above are calls of a macro
  that expand to involve the special variable ~ilc[state].  This will
  prevent you from accessing a table from within a hint or theory where
  where you do not have the ~ilc[state] variable.  However, the form
  ~bv[]
  (table-alist 'tests world)
  ~ev[]
  returns the alist representation of the table named ~c[test] in the
  given world.  Often you have access to ~c[world].

  The ACL2 system provides ``tables'' by which the user can associate
  one object with another.  Tables are in essence just conventional
  association lists ~-[] lists of pairs ~-[] but the ACL2 environment
  provides a means of storing these lists in the ``ACL2 world'' of the
  current ~ilc[state].  The ACL2 user could accomplish the same ends by
  using ACL2 ``global variables;'' however, limitations on global
  variable names are imposed to ensure ACL2's soundness.  By
  convention, no table is important to ACL2's soundness, even though
  some features of the system use tables, and the user is invited to
  make free use of tables.  Because tables are stored in the ACL2
  ~il[world] they are restored by ~ilc[include-book] and undone by ~c[:]~ilc[ubt].  Many
  users of Nqthm requested a facility by which user data could be
  saved in Nqthm ``lib files'' and tables are ACL2's answer to that
  request.

  Abstractly, each table is an association list mapping ``keys'' to
  ``values.'' In addition, each table has a ``~c[:guard],'' which is a
  term that must be true of any key and value used.  By setting the
  ~c[:guard] on a table you may enforce an invariant on the objects in the
  table, e.g., that all keys are positive integers and all values are
  symbols.  Each table has a ``name,'' which must be a symbol.  Given
  a table name, the following operations can be performed on the table.

  ~c[:put] ~-[] associate a value with a key (possibly changing the value
  currently associated with that key).

  ~c[:get] ~-[] retrieve the value associated with a key (or nil if no
  value has been associated with that key).

  ~c[:alist] ~-[] return an alist showing all keys and non-nil values in
  the table.

  ~c[:clear] ~-[] clear the table (so that every value is nil), or if val
  is supplied then set table to that value (which must be an alist).

  ~c[:guard] ~-[] fetch or set the :guard of the table.

  When the operations above suggest that the table or its ~c[:guard] are
  modified, what is actually meant is that the current ~il[state] is redefined
  so that in it, the affected table name has the appropriate properties.  in
  such cases, the ~c[table] form is an event (~pl[events]).  In the ~c[:put]
  case, if the key is already in the table and associated with the proposed
  value, then the ~c[table] event is redundant (~pl[redundant-events]).

  ~c[Table] forms are commonly typed by the user while interacting with
  the system.  ~c[:Put] and ~c[:get] forms are especially common.  Therefore,
  we have adopted a positional syntax that is intended to be
  convenient for most applications.  Essentially, some operations
  admit a ``short form'' of invocation.
  ~bv[]
  (table name key-term value-term :put)   ; long form
  (table name key-term value-term)        ; short form
  ~ev[]
  evaluates the key- and value-terms, obtaining two objects that we
  call ~c[key] and ~c[value], checks that the ~c[key] and ~c[value] satisfy the
  ~c[:guard] on the named table and then ``modifies'' the named table
  so that the value associated with ~c[key] is ~c[value].  When used like
  this, ~c[table] is actually an event in the sense that it changes the
  ACL2 ~il[world].  In general, the forms evaluated to obtain the ~c[key] and
  ~c[value] may involve the variable ~ilc[world], which is bound to the
  then-current ~il[world] during the evaluation of the forms.  However, in
  the special case that the table in question is named
  ~ilc[acl2-defaults-table], the ~c[key] and ~c[value] terms may not contain any
  variables.  Essentially, the keys and values used in ~il[events] setting
  the ~ilc[acl2-defaults-table] must be explicitly given constants.
  ~l[acl2-defaults-table].
  ~bv[]
  (table name key-term nil :get)          ; long form
  (table name key-term)                   ; short form
  ~ev[]
  evaluates the key-term (see note below), obtaining an object, ~c[key],
  and returns the value associated with ~c[key] in the named table (or,
  ~c[nil] if there is no value associated with ~c[key]).  When used like this,
  ~c[table] is not an event; the value is simply returned.
  ~bv[]
  (table name nil nil :alist)             ; long form
  (table name)                            ; short form
  ~ev[]
  returns an alist representing the named table; for every key in
  the table with a non-~c[nil] associated value, the alist pairs the key
  and its value.  The order in which the keys are presented is
  unspecified.  When used like this, ~c[table] is not an event; the alist
  is simply returned.
  ~bv[]
  (table name nil val :clear)
  ~ev[]
  sets the named table to the alist ~c[val], making the checks that ~c[:put]
  makes for each key and value of ~c[val].  When used like this, ~c[table] is
  an event because it changes the ACL2 ~il[world].
  ~bv[]
  (table name nil nil :guard)
  ~ev[]
  returns the translated form of the guard of the named table.
  ~bv[]
  (table name nil nil :guard term)
  ~ev[]
  Provided the named table is empty and has not yet been assigned a
  ~c[:guard] and ~c[term] (which is not evaluated) is a term that mentions at
  most the variables ~c[key], ~c[val] and ~ilc[world], this event sets the ~c[:guard] of
  the named table to ~c[term].  Whenever a subsequent ~c[:put] occurs, ~c[term]
  will be evaluated with ~c[key] bound to the key argument of the ~c[:put],
  ~c[val] bound to the ~c[val] argument of the ~c[:put], and ~ilc[world] bound to the
  then current ~il[world].  An error will be caused by the ~c[:put] if the
  result of the evaluation is ~c[nil].

  Note that it is not allowed to change the ~c[:guard] on a table once it
  has been explicitly set.  Before the ~c[:guard] is explicitly set, it is
  effectively just ~c[t].  After it is set it can be changed only by
  undoing the event that set it.  The purpose of this restriction is
  to prevent the user from changing the ~c[:guards] on tables provided by
  other people or the system.

  The intuition behind the ~c[:guard] mechanism on tables is to enforce
  invariants on the keys and values in a table, so that the values,
  say, can be used without run-time checking.  But if the ~c[:guard] of a
  table is sensitive to the ACL2 ~il[world], it may be possible to cause
  some value in the table to cease satisfying the ~c[:guard] without doing
  any operations on the table.  Consider for example the ~c[:guard] ``no
  value in this table is the name of an event.'' As described, that is
  enforced each time a value is stored.  Thus, ~c['bang] can be ~c[:put] in
  the table provided there is no event named ~c[bang].  But once it is in
  the table, there is nothing to prevent the user from defining ~c[bang]
  as a function, causing the table to contain a value that could not
  be ~c[:put] there anymore.  Observe that not all state-sensitive ~c[:guard]s
  suffer this problem.  The ~c[:guard] ``every value is an event name''
  remains invariant, courtesy of the fact that undoing back through an
  event name in the table would necessarily undo the ~c[:put] of the name
  into the table.

  ~c[Table] was designed primarily for convenient top-level use.  Tables
  are not especially efficient.  Each table is represented by an alist
  stored on the property list of the table name.  ~c[:Get] is just a
  ~c[getprop] and ~ilc[assoc-equal].  ~c[:Put] does a ~c[getprop] to the get the table
  alist, a ~c[put-assoc-equal] to record the new association, and a
  ~c[putprop] to store the new table alist ~-[] plus the overhead associated
  with ~c[:guard]s and undoable ~il[events], and checking (for redundancy) if
  the key is already bound to its proposed value.  Note that there are never
  duplicate keys in the resulting ~c[alist]; in particular, when the
  operation ~c[:clear] is used to install new ~c[alist], duplicate keys are
  removed from that alist.

  A table name may be any symbol whatsoever.  Symbols already in use
  as function or theorem names, for example, may be used as table
  names.  Symbols in use only as table names may be defined with
  ~ilc[defun], etc.  Because there are no restrictions on the user's choice
  of table names, table names are not included among the logical
  names.  Thus, ~c[:pe name] will never display a table event (for a
  logical name other than ~c[:here]).  Either ~c[:pe name] will display a
  ``normal'' event such as ~c[(defun name ...)] or ~c[(defthm name ...)] or
  else ~c[:pe name] will cause an error indicating that ~c[name] is not a
  logical name.  This happens even if ~c[name] is in use as a table name.
  Similarly, we do not permit table names to have ~il[documentation]
  strings, since the same name might already have a ~il[documentation]
  string.  If you want to associate a ~il[documentation] string with a
  table name that is being used no other way, define the name as a
  label and use the ~c[:]~ilc[doc] feature of ~ilc[deflabel]
  (~pl[deflabel]); also ~pl[defdoc].~/"

; At one time the table macro expanded to several different forms,
; depending on whether it was really expected to affect world.  That
; was abandoned when it was actually included in the source files
; because of the important invariant that these defmacros be
; translatable by boot-translate.

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'table-fn
        (list 'quote name)
        (list 'quote args)
        'state
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro encapsulate (&whole event-form signatures &rest cmd-lst)

; Warning: See the Important Boot-Strapping Invariants before modifying!

  ":Doc-Section Events

  constrain some functions and/or hide some ~il[events]~/

  The following example illustrates the idea of ~c[encapsulate].
  ~bv[]
  (encapsulate

  ; Introduce a function foo of one argument that returns one value:
   ( ((foo *) => *) )

  ; Introduce a ``witness'' (example) for foo, marked as local so that
  ; it is not exported:
   (local (defun foo (x) x))

  ; Introduce a non-local property to be exported:
   (defthm foo-preserves-consp
     (implies (consp x)
              (consp (foo x))))
  )
  ~ev[]
  The form above introduces a new function symbol, ~c[foo], with the indicated
  property and no definition.  In fact, the output from ACL2 concludes as
  follows.
  ~bv[]

  The following constraint is associated with the function FOO:

  (IMPLIES (CONSP X) (CONSP (FOO X)))
  ~ev[]

  We turn now to more complete documentation.

  ~bv[]
  Other Examples:
  (encapsulate (((an-element *) => *))

  ; The list of signatures above could also be written
  ;            ((an-element (lst) t))

    (local (defun an-element (lst)
             (if (consp lst) (car lst) nil)))
    (local (defthm member-equal-car
              (implies (and lst (true-listp lst))
                       (member-equal (car lst) lst))))
    (defthm thm1
       (implies (null lst) (null (an-element lst))))
    (defthm thm2
       (implies (and (true-listp lst)
                     (not (null lst)))
                (member-equal (an-element lst) lst))))

  (encapsulate
   () ; empty signature: no constrained functions indicated

   (local (defthm hack
            (implies (and (syntaxp (quotep x))
                          (syntaxp (quotep y)))
                     (equal (+ x y z)
                            (+ (+ x y) z)))))

   (defthm nthcdr-add1-conditional
     (implies (not (zp (1+ n)))
              (equal (nthcdr (1+ n) x)
                     (nthcdr n (cdr x))))))~/

  General Form:
  (encapsulate (signature ... signature)
    ev1
    ...
    evn)
  ~ev[]
  where each ~ilc[signature] is a well-formed signature (~pl[signature]), each
  ~c[signature] describes a different function symbol, and each ~c[evi] is an
  embedded event form (~l[embedded-event-form]).  There must be at least one
  ~c[evi].  The ~c[evi] inside ~ilc[local] special forms are called ``local''
  ~il[events] below.  ~il[Events] that are not ~ilc[local] are sometimes said
  to be ``exported'' by the encapsulation.  We make the further restriction
  that no ~ilc[defaxiom] event may be introduced in the scope of an
  ~c[encapsulate] (not even by ~c[encapsulate] or ~ilc[include-book] events
  that are among the ~c[evi]).  Furthermore, no non-~ilc[local]
  ~ilc[include-book] event is permitted in the scope of any ~c[encapsulate]
  with a non-empty list of signatures.

  To be well-formed, an ~c[encapsulate] event must have the properties that
  each event in the body (including the ~ilc[local] ones) can be successfully
  executed in sequence and that in the resulting theory, each function
  mentioned among the ~il[signature]s was introduced via a ~ilc[local] event
  and has the ~il[signature] listed.  (A utility is provided to assist in
  debugging failures of such execution; ~pl[redo-flat].)  In addition, the body
  may contain no ``local incompatibilities'' which, roughly stated, means that
  the ~il[events] that are not ~ilc[local] must not syntactically require
  symbols defined by ~ilc[local] ~ilc[events], except for the functions listed
  in the ~il[signature]s.  ~l[local-incompatibility].  Finally, no
  non-~ilc[local] recursive definition in the body may involve in its suggested
  induction scheme any function symbol listed among the ~il[signature]s.
  ~l[subversive-recursions].

  The result of an ~c[encapsulate] event is an extension of the logic
  in which, roughly speaking, the functions listed in the
  ~il[signature]s are constrained to have the ~il[signature]s listed
  and to satisfy the non-~ilc[local] theorems proved about them.  In
  fact, other functions introduced in the ~c[encapsulate] event may be
  considered to have ``~il[constraint]s'' as well.  (~l[constraint]
  for details, which are only relevant to functional instantiation.)
  Since the ~il[constraint]s were all theorems in the ``ephemeral'' or
  ``local'' theory, we are assured that the extension produced by
  ~c[encapsulate] is sound.  In essence, the ~ilc[local] definitions of
  the constrained functions are just ``witness functions'' that
  establish the consistency of the ~il[constraint]s.  Because those
  definitions are ~ilc[local], they are not present in the theory
  produced by encapsulation.  ~c[Encapsulate] also exports all rules
  generated by its non-~ilc[local] ~il[events], but rules generated by
  ~ilc[local] ~il[events] are not exported.

  The ~il[default-defun-mode] for the first event in an encapsulation is
  the default ~il[defun-mode] ``outside'' the encapsulation.  But since
  ~il[events] changing the ~il[defun-mode] are permitted within the body of an
  ~c[encapsulate], the default ~il[defun-mode] may be changed.  However,
  ~il[defun-mode] changes occurring within the body of the ~c[encapsulate]
  are not exported.  In particular, the ~ilc[acl2-defaults-table] after
  an ~c[encapsulate] is always the same as it was before the
  ~c[encapsulate], even though the ~c[encapsulate] body might contain
  ~il[defun-mode] changing ~il[events], ~c[:]~ilc[program] and ~c[:]~ilc[logic].
  ~l[defun-mode].  More generally, after execution of an
  ~c[encapsulate] event, the value of ~ilc[acl2-defaults-table] is
  restored to what it was immediately before that event was executed.
  ~l[acl2-defaults-table].

  Theorems about the constrained function symbols may then be proved
  ~-[] theorems whose proofs necessarily employ only the ~il[constraint]s.
  Thus, those theorems may be later functionally instantiated, as with
  the ~c[:functional-instance] lemma instance
  (~pl[lemma-instance]), to derive analogous theorems about
  different functions, provided the constraints (~pl[constraint])
  can be proved about the new functions.

  We make some remarks on ~il[guard]s and evaluation.  Calls of functions
  introduced in the ~il[signature]s list cannot be evaluated in the ACL2
  read-eval-print loop.  ~l[defattach] for a way to overcome this limitation.
  Moreover, any ~c[:]~ilc[guard] supplied in the signature is automatically
  associated in the ~il[world] with its corresponding function symbol, with no
  requirement other than that the guard is a legal term all of whose function
  symbols are in ~c[:]~ilc[logic] mode with their ~il[guard]s verified.  In
  particular, there need not be any relationship between a guard in a signature
  and the guard in a ~c[local] witness function.  Finally, note that for
  functions introduced non-~il[local]ly inside an ~c[encapsulate] event,
  ~il[guard] verification is illegal unless ACL2 determines that the proof
  obligations hold outside the ~ilc[encapsulate] event as well.
  ~bv[]
  (encapsulate
   ((f (x) t))
   (local (defun f (x) (declare (xargs :guard t)) (consp x)))
   ;; ERROR!
   (defun g (x)
     (declare (xargs :guard (f x)))
     (car x)))
  ~ev[]

  Observe that if the ~il[signature]s list is empty, ~c[encapsulate] may still
  be useful for deriving theorems to be exported whose proofs require lemmas
  you prefer to hide (i.e., made ~ilc[local]).

  The order of the ~il[events] in the vicinity of an ~c[encapsulate] is
  confusing.  We discuss it in some detail here because when logical names are
  being used with theory functions to compute sets of rules, it is sometimes
  important to know the order in which ~il[events] were executed.
  (~l[logical-name] and ~pl[theory-functions].)  What, for example, is the set
  of function names extant in the middle of an encapsulation?

  If the most recent event is ~c[previous] and then you execute an
  ~c[encapsulate] constraining ~c[an-element] with two non-~ilc[local]
  ~il[events] in its body, ~c[thm1] and ~c[thm2], then the order of the
  ~il[events] after the encapsulation is (reading chronologically forward):
  ~c[previous], ~c[thm1], ~c[thm2], ~c[an-element] (the ~c[encapsulate]
  itself).  Actually, between ~c[previous] and ~c[thm1] certain extensions were
  made to the ~il[world] by the superior ~c[encapsulate], to permit
  ~c[an-element] to be used as a function symbol in ~c[thm1].

  Finally, we note that an ~ilc[encapsulate] event is redundant if and only if
  a syntactically identical ~ilc[encapsulate] has already been executed under
  the same ~ilc[default-defun-mode], ~ilc[default-ruler-extenders], and
  ~ilc[default-verify-guards-eagerness].  ~l[redundant-events].~/"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'encapsulate-fn
        (list 'quote signatures)
        (list 'quote cmd-lst)
        'state
        (list 'quote event-form)))

(defconst *load-compiled-file-values*
  '(t nil :warn :default :comp))

#+acl2-loop-only
(defmacro include-book (&whole event-form user-book-name
                               &key

; Warning:  If you change the defaults below, be sure to change the
; construction of event-form in include-book-fn!

                               (load-compiled-file ':default)
                               (uncertified-okp 't)
                               (defaxioms-okp 't)
                               (skip-proofs-okp 't)
                               (ttags ':default)
                               dir
                               doc)

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (declare (xargs :guard
                  (member-eq load-compiled-file *load-compiled-file-values*)))

  ":Doc-Section Events

  load the ~il[events] in a file~/
  ~bv[]
  Examples:
  (include-book \"my-arith\")
  (include-book \"/home/smith/my-arith\")
  (include-book \"/../../my-arith\")

  General Form:
  (include-book file :load-compiled-file action
                     :uncertified-okp t/nil      ; [default t]
                     :defaxioms-okp t/nil        ; [default t]
                     :skip-proofs-okp t/nil      ; [default t]
                     :ttags ttags                ; [default nil]
                     :dir directory
                     :doc doc-string)
  ~ev[]
  where ~c[file] is a book name.  ~l[books] for general information,
  ~pl[book-name] for information about book names, and ~pl[pathname] for
  information about file names.  ~c[Action] is one of ~c[t], ~c[nil],
  ~c[:default], ~c[:warn], or ~c[:comp]; these values are explained below, and
  the default is ~c[:default].  The three ~c[-okp] keyword arguments, which
  default to ~c[t], determine whether errors or warnings are generated under
  certain conditions explained below; when the argument is ~c[t], warnings are
  generated.  The ~c[dir] argument, if supplied, is a keyword that represents
  an absolute pathname for a directory (~pl[pathname]), to be used instead of
  the current book directory (~pl[cbd]) for resolving the given ~c[file]
  argument to an absolute pathname.  In particular, by default ~c[:dir :system]
  resolves ~c[file] using the distributed ~c[books/] directory of your ACL2
  installation, unless your ACL2 executable was built somewhere other than
  where it currently resides; please see the ``Distributed Books Directory''
  below.  To define other keywords that can be used for ~c[dir],
  ~pl[add-include-book-dir].  ~c[Doc-string] is an optional ~il[documentation]
  string; ~pl[doc-string].  If the book has no ~ilc[certificate], if its
  ~ilc[certificate] is invalid or if the certificate was produced by a
  different ~il[version] of ACL2, a warning is printed and the book is included
  anyway; ~pl[certificate].  This can lead to serious errors;
  ~pl[uncertified-books].  If the portcullis of the ~il[certificate]
  (~pl[portcullis]) cannot be raised in the host logical ~il[world], an error
  is caused and no change occurs to the logic.  Otherwise, the non-~ilc[local]
  ~il[events] in file are assumed.  Then the ~il[keep] of the ~il[certificate]
  is checked to ensure that the correct files were read; ~pl[keep].  A warning
  is printed if uncertified ~il[books] were included.  Even if no warning is
  printed, ~c[include-book] places a burden on you; ~pl[certificate].

  If you use ~il[guard]s, please note ~c[include-book] is executed as though
  ~c[(set-guard-checking nil)] has been evaluated; ~Pl[set-guard-checking].  If
  you want guards checked, please ~pl[ld] and/or ~pl[rebuild].

  The value, ~c[action], of ~c[:load-compiled-file] controls whether a compiled
  file is loaded by ~c[include-book].  If compilation has been suppressed by
  ~c[(set-compiler-enabled nil)], then ~c[action] is coerced to ~c[nil];
  ~pl[compilation].  Otherwise, if ~c[action] is missing or its value is the
  keyword ~c[:default], then it is treated as ~c[:warn].  If ~c[action] is
  ~c[nil], no attempt is made to load the compiled file for the book provided.
  In order to load the compiled file, it must be more recent than the book's
  ~il[certificate] (except in raw mode, where it must be more recent than the
  book itself; ~pl[set-raw-mode]).  For non-~c[nil] values of ~c[action] that
  do not result in a loaded compiled file, ACL2 proceeds as follows.  Note that
  a load of a compiled file or expansion file aborts partway through whenever
  an ~ilc[include-book] form is encountered for which no suitable compiled or
  expansion file can be loaded.  For much more detail, ~pl[book-compiled-file].
  ~bq[]

  ~c[t]: Cause an error if the compiled file is not loaded.  This same
  requirement is imposed on every ~ilc[include-book] form evaluated during the
  course of evaluation of the present ~c[include-book] form, except that for
  those subsidiary ~c[include-book]s that do not themselves specify
  ~c[:load-compiled-file t], it suffices to load the expansion file
  (~pl[book-compiled-file]).

  ~c[:warn]: An attempt is made to load the compiled file, and a warning is
  printed if that load takes place and runs to completion.

  ~c[:comp]: A compiled file is loaded as with value ~c[t], except that if a
  suitable ``expansion file'' exists but the compiled file does not, then the
  compiled file is first created.  ~l[book-compiled-file].~eq[]

  The three ~c[-okp] arguments, ~c[:uncertified-okp], ~c[defaxioms-okp],
  and ~c[skip-proofs-okp], determine the system's behavior when
  the book or any subbook is found to be uncertified, when the book
  or any subbook is found to contain ~ilc[defaxiom] events, and when
  the book or any subbook is found to contain ~ilc[skip-proofs] events,
  respectively.  All three default to ~c[t], which means it is ``ok''
  for the condition to arise.  In this case, a warning is printed but
  the processing to load the book is allowed to proceed.  When one of
  these arguments is ~c[nil] and the corresponding condition arises,
  an error is signaled and processing is aborted.  ~st[Exception]:
  ~c[:uncertified-okp] is ignored if the ~c[include-book] is being
  performed on behalf of a ~ilc[certify-book].

  The keyword argument ~c[:ttags] may normally be omitted.  A few constructs,
  used for example if you are building your own system based on ACL2, may
  require it.  ~l[defttag] for an explanation of this argument.

  ~c[Include-book] is similar in spirit to ~ilc[encapsulate] in that it is
  a single event that ``contains'' other ~il[events], in this case the
  ~il[events] listed in the file named.  ~c[Include-book] processes the
  non-~ilc[local] event forms in the file, assuming that each is
  admissible.  ~ilc[Local] ~il[events] in the file are ignored.  You may
  use ~c[include-book] to load several ~il[books], creating the logical
  ~il[world] that contains the definitions and theorems of all of
  them.

  If any non-~ilc[local] event of the book attempts to define a ~il[name]
  that has already been defined ~-[] and the book's definition is not
  syntactically identical to the existing definition ~-[] the attempt to
  include the book fails, an error message is printed, and no change
  to the logical ~il[world] occurs.  ~l[redundant-events] for the
  details.

  When a book is included, the default ~il[defun-mode]
  (~pl[default-defun-mode]) for the first event is always ~c[:]~ilc[logic].
  That is, the default ~il[defun-mode] ``outside'' the book ~-[] in the
  environment in which ~c[include-book] was called ~-[] is irrelevant to the
  book.  ~il[Events] that change the ~il[defun-mode] are permitted within a
  book (provided they are not in ~ilc[local] forms).  However, such changes
  within a book are not exported, i.e., at the conclusion of an
  ~c[include-book], the ``outside'' default ~il[defun-mode] is always the same
  as it was before the ~c[include-book].

  Unlike every other event in ACL2, ~c[include-book] puts a burden on
  you.  Used improperly, ~c[include-book] can be unsound in the sense
  that it can create an inconsistent extension of a consistent logical
  ~il[world].  A certification mechanism is available to help you
  carry this burden ~-[] but it must be understood up front that even
  certification is no guarantee against inconsistency here.  The
  fundamental problem is one of file system security.
  ~l[certificate] for a discussion of the security issues.

  After execution of an ~c[include-book] form, the value of
  ~ilc[acl2-defaults-table] is restored to what it was immediately before
  that ~c[include-book] form was executed.
  ~l[acl2-defaults-table].

  ~b[Distributed Books Directory.]  We refer to the ``books directory'' of an
  executable image as the full pathname string of the books directory
  associated with ~c[:dir :system] for that image.  This is where the
  distributed books directory should reside.  By default, it is the ~c[books/]
  subdirectory of the directory where the sources reside and the executable
  image is thus built (except for ACL2(r) ~-[] ~pl[real] ~-[], where it is
  ~c[books/nonstd/]).  If those books reside elsewhere, the environment
  variable ~c[ACL2_SYSTEM_BOOKS] can be set to the ~c[books/] directory under
  which they reside (a Unix-style pathname, typically ending in ~c[books/] or
  ~c[books], is permissible).  In most cases, your ACL2 executable is a small
  script in which you can set this environment variable just above the line on
  which the actual ACL2 image is invoked, for example:
  ~bv[]
  export ACL2_SYSTEM_BOOKS
  ACL2_SYSTEM_BOOKS=/home/acl2/4-0/acl2-sources/books
  ~ev[]

  This concludes the guided tour through ~il[books].
  ~l[set-compile-fns] for a subtle point about the interaction
  between ~c[include-book] and on-the-fly ~il[compilation].
  ~l[certify-book] for a discussion of how to certify a book.~/

  :cited-by Programming"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'include-book-fn
        (list 'quote user-book-name)
        'state
        (list 'quote load-compiled-file)
        (list 'quote :none)
        (list 'quote uncertified-okp)
        (list 'quote defaxioms-okp)
        (list 'quote skip-proofs-okp)
        (list 'quote ttags)
        (list 'quote doc)
        (list 'quote dir)
        (list 'quote event-form)))

#+acl2-loop-only
(defmacro make-event (&whole event-form
                             form
                             &key check-expansion on-behalf-of)

; Essay on Make-event

; This essay incorporates by reference :doc make-event and :doc
; make-event-details.  That is, one should start by reading those documentation
; topics.  This is a place to add details that seem of interest only to the
; implementors, not to ACL2 users.

; When we lay down a command landmark for a command for which expansion has
; taken place, we need to record that expansion somehow for subsequent calls of
; certify-book, in order to recover portcullis commands.  Thus,
; add-command-landmark and make-command-tuple have an argument for the
; expansion (which could be nil, indicating that no expansion took place).

; We use record-expansion (as described in :doc make-event-details) in order to
; support redundancy of encapsulate, as implemented by redundant-encapsulatep
; and its subroutines.  Here is a summary of the issue.  Consider: (encapsulate
; ((foo (x) t)) ... (make-event <form>)).  We have several goals.
; + Be able to execute this form a second time and have it be redundant.
; + If this form is redundant yet in a book, it cannot cause a new expansion
;   result for the make-event or the encapsulate, and include-book has to do
;   the right thing even, if possible, in raw mode.
; + We want to store a proper expansion of an encapsulate.
; + We want to recognize redundancy without having to execute the encapsulate.
; + If an encapsulate form is redundant then its stored version is identical
;   to the stored version of the earlier form for which it is redundant.
; The last of these properties is important because otherwise unsoundness could
; result!  Suppose for example that a book bar.lisp contains (local
; (include-book "foo")), where foo.lisp contains an encapsulate that causes a
; later encapsulate in bar.lisp to be redundant.  What should we know at the
; point we see the later encapsulate?  We should know that the event logically
; represented by the encapsulate is the same as the one logically represented
; by the earlier encapsulate, so we certainly do not want to re-do its
; expansion at include-book time.  Thus, when an encapsulate is redundant, we
; store the expanded version of the earlier encapsulate as the expansion of the
; current unexpanded encapsulate, unless the two are identical.  But how do we
; expand a non-redundant encapsulate?  We expand it by replacing every
; sub-event ev by (record-expansion ev exp), when ev has an expansion exp.
; Then, we recognize a subsequent encapsulate as redundant with this one if
; their signatures are equal and each of the subsequent encapsulate's events,
; ev2, is either the same as the corresponding event ev1 of the old encapsulate
; or else ev1 is of the form (record-expansion ev2 ...).

; We elide local forms arising from make-event expansions when writing to book
; certificates, in order to save space.  See elide-locals.

; Note that when :puff (specifically puff-command-block) is applied to an
; include-book form, it uses the expansion-alist from the book's certificate if
; there is an up-to-date certificate.

 ":Doc-Section Events

  evaluate (expand) a given form and then evaluate the result~/

  ~c[Make-event] is a utility for generating ~il[events].  It provides a
  capability not offered by Lisp macros (~pl[defmacro]), as it allows access to
  the ACL2 ~ilc[state] and logical ~il[world].  In essence, the expression
  ~c[(make-event form)] replaces itself with the result of evaluating ~c[form],
  say, ~c[ev], as though one had submitted ~c[ev] instead of the ~c[make-event]
  call.  But the evaluation of ~c[form] may involve ~ilc[state] and even modify
  ~c[state], for example by attempting to admit some definitions and theorems.
  ~c[Make-event] protects the ACL2 logical ~il[world] so that it is restored
  after ~c[form] is evaluated, before ~c[ev] is submitted.~/

  ~bv[]
  Examples:

  ; Trivial example: evaluate (quote (defun foo (x) x)) to obtain
  ; (defun foo (x) x), which is then evaluated.
  (make-event (quote (defun foo (x) x)))

  ; Evaluate (generate-form state) to obtain (mv nil val state), and
  ; then evaluate val.  (Generate-form is not specified here, but
  ; imagine for example that it explores the state and then generates
  ; some desired definition or theorem.)
  (make-event (generate-form state))

  ; As above, but make sure that if this form is in a book, then when
  ; we include the book, the evaluation of (generate-form state)
  ; should return the same value as it did when the book was
  ; certified.
  (make-event (generate-form state)
              :check-expansion t)

  ; As above (where the :check-expansion value can be included or
  ; not), where if there is an error during expansion, then the error
  ; message will explain that expansion was on behalf of the indicated
  ; object, typically specified as the first argument.
  (make-event (generate-form state)
              :on-behalf-of (generate-form state))

  General Form:
  (make-event form :check-expansion chk :on-behalf-of obj)
  ~ev[]
  where ~c[chk] is ~c[nil] (the default), ~c[t], or the intended ``expansion
  result'' from the evaluation of ~c[form] (as explained below); and if
  supplied, ~c[obj] is an arbitrary ACL2 object, used only in reporting errors
  in expansion, i.e., in the evaluation of form.

  We strongly recommend that you look at ~c[books/make-event/Readme.lsp], which
  summarizes and suggests browsing of some ~c[.lisp] files in that directory,
  in order to understand ~c[make-event], perhaps before continuing to read this
  documentation.  For example, ~c[eval.lisp] contains definitions of macros
  ~c[must-succeed] and ~c[must-fail] that are useful for testing and are used
  in many other books in that directory, especially ~c[eval-tests.lisp].  Other
  than the examples, the explanations here should suffice for most users.  If
  you want explanations of subtler details, ~pl[make-event-details].

  ~c[Make-event] is related to Lisp macroexpansion in the sense that its
  argument is evaluated to obtain an expansion result, which is evaluated
  again.  Let us elaborate on each of these notions in turn: ``is evaluated,''
  ``expansion result'', and ``evaluated again.''~bq[]

  ``is evaluated'' ~-[] The argument can be any expression, which is evaluated
  as would be any expression submitted to ACL2's top level loop.  Thus,
  ~ilc[state] and user-defined ~ilc[stobj]s may appear in the form supplied to
  ~c[make-event].  Henceforth, we will refer to this evaluation as
  ``expansion.''  Expansion is actually done in a way that restores ACL2's
  built-in ~ilc[state] global variables, including the logical ~il[world], to
  their pre-expansion values (with a few exceptions ~-[]
  ~pl[make-event-details] ~-[] and where we note that changes to user-defined
  ~ilc[state] global variables (~pl[assign]) are preserved).  So, for example,
  events might be evaluated during expansion, but they will disappear from the
  logical ~il[world] after expansion returns its result.  Moreover, proofs are
  enabled by default at the start of expansion (~pl[ld-skip-proofsp]), because
  an anticipated use of ~c[make-event] is to call the prover to decide which
  event to generate, and that would presumably be necessary even if proofs had
  been disabled.

  ``expansion result'' ~-[] The above expansion may result in an ordinary
  (non-~ilc[state], non-~ilc[stobj]) value, which we call the ``expansion
  result.''  Or, expansion may result in a multiple value of the form
  ~c[(mv erp val state stobj-1 ... stobj-k)], where ~c[k] may be 0; in fact the
  most common case is probably ~c[(mv erp val state)].  In that case, if
  ~c[erp] is not ~c[nil], then there is no expansion result, and the original
  ~c[make-event] evaluates to a soft error.  If however ~c[erp] is ~c[nil],
  then the resulting value is ~c[val].  Moreover, ~c[val] must be an embedded
  event form (~pl[embedded-event-form]); otherwise, the original ~c[make-event]
  evaluates to a soft error.  Note that error messages from expansion are
  printed as described under ``Error Reporting'' below.

  ``evaluated again'' ~-[] the expansion result is evaluated in place of the
  original ~c[make-event].

  ~eq[]Note that the result of expansion can be an ordinary event, but it can
  instead be another call of ~c[make-event], or even of a call of a macro that
  expands to a call of ~c[make-event].  Or, expansion itself can cause
  subsidiary calls of ~c[make-event], for example if expansion uses ~ilc[ld] to
  evaluate some ~c[make-event] forms.  The state global variable
  ~c[make-event-debug] may be set to a non-~c[nil] value in order to see a
  trace of the expansion process, where the level shown (as in ``~c[3>]'')
  indicates the depth of expansions in progress.

  Expansion of a ~c[make-event] call will yield an event that replaces the
  original ~c[make-event] call.  In particular, if you put a ~c[make-event]
  form in a book, then in essence it is replaced by its expansion result,
  created during the proof pass of the ~ilc[certify-book] process.  We now
  elaborate on this idea of keeping the original expansion.

  By default, a ~c[make-event] call in a certified book is replaced (by a
  process hidden from the user, in an ~c[:expansion-alist] field of the book's
  ~il[certificate]) by the expansion result from evaluation of its first
  argument.  Thus, although the book is not textually altered during
  certification, one may imagine a ``book expansion'' corresponding to the
  original book in which all of the events for which expansion took place
  (during the proof phase of certification) have been replaced by their
  expansions.  A subsequent ~ilc[include-book] will then include the book
  expansion corresponding to the indicated book.  When a book is compiled
  during ~ilc[certify-book], it is actually the corresponding book expansion,
  stored as a temporary file, that is compiled instead.  That temporary file is
  deleted after compilation unless one first evaluates the form
  ~c[(assign keep-tmp-files t)].  Note however that all of the original forms
  must still be legal ~il[events] (~pl[embedded-event-form]).  So for example,
  if the first event in a book is ~c[(local (defmacro my-id (x) x))], followed
  by ~c[(my-id (make-event ...))], the final ``~c[include-book]'' pass of
  ~ilc[certify-book] will fail because ~c[my-id] is not defined when the
  ~c[my-id] call is encountered.

  The preceding paragraph begins with ``by default'' because if you specify
  ~c[:check-expansion t], then subsequent evaluation of the same ~c[make-event]
  call ~-[] during the second pass of an ~ilc[encapsulate] or during event
  processing on behalf of ~ilc[include-book] ~-[] will do the expansion again
  and check that the expansion result equals the original expansion result.  In
  the unusual case that you know the expected expansion result, ~c[res], you
  can specify ~c[:check-expansion res].  This will will cause a check that
  every subsequent expansion result for the ~c[make-event] form is ~c[res],
  including the original one.  IMPORTANT NOTE: That expansion check is only
  done when processing events, not during a preliminary load of a book's
  compiled file.  The following paragraph elaborates.

  (Here are details on the point made just above, for those who use the
  ~c[:check-expansion] argument to perform side-effects on the ~il[state].
  When you include a book, ACL2 generally loads a compiled file before
  processing the events in the book; ~pl[book-compiled-file].  While it is true
  that a non-~c[nil] ~c[:check-expansion] argument causes ~ilc[include-book] to
  perform expansion of the ~c[make-event] form during event processing it does
  ~em[not] perform expansion when the compiled file (or expansion file; again,
  ~pl[book-compiled-file]) is loaded.)

  ACL2 performs the following space-saving optimization for book certificates:
  a ~ilc[local] event arising from ~c[make-event] expansion is replaced in that
  expansion by ~c[(local (value-triple :ELIDED))].

  Finally, we note that ACL2 extends the notion of ``make-event expansion'' to
  the case that a call of ~c[make-event] is found in the course of
  macroexpansion.  We illustrate with the following example.
  ~bv[]
  (encapsulate
   ()
   (defmacro my-mac ()
     '(make-event '(defun foo (x) x)))
   (my-mac))
  :pe :here
  ~ev[]
  The above call of ~ilc[pe] shows that the form ~c[(my-mac)] has a
  ~c[make-event] expansion of ~c[(DEFUN FOO (X) X)]:
  ~bv[]
  (ENCAPSULATE NIL
               (DEFMACRO MY-MAC
                         NIL
                         '(MAKE-EVENT '(DEFUN FOO (X) X)))
               (RECORD-EXPANSION (MY-MAC)
                                 (DEFUN FOO (X) X)))
  ~ev[]

  ~st[Error Reporting.]

  Suppose that expansion produces a soft error as described above.  That is,
  suppose that the argument of a ~c[make-event] call evaluates to a multiple
  value ~c[(mv erp val state ...)] where ~c[erp] is not ~c[nil].  If ~c[erp] is
  a string, then that string is printed in the error message.  If ~c[erp] is
  a ~ilc[cons] pair whose ~ilc[car] is a string, then the error prints
  ~c[\"~~@0\"] with ~c[#\\0] bound to that ~c[cons] pair; ~pl[fmt].  Any other
  non-~c[nil] value of ~c[erp] causes a generic error message to be printed.

  ~st[Restriction to the Top Level.]

  Every form enclosing a ~c[make-event] call must be an embedded event form
  (~pl[embedded-event-form]).  This restriction enables ACL2 to track
  expansions produced by ~c[make-event].  For example:
  ~bv[]
  ; Legal:
  (progn (with-output
          :on summary
          (make-event '(defun foo (x) x))))

  ; Illegal:
  (mv-let (erp val state)
          (make-event '(defun foo (x) x))
          (mv erp val state))
  ~ev[]
  Low-level remark, for system implementors.  There is the one exception to
  this restriction: a single ~c[state-global-let*] form immediately under a
  ~c[progn!] call.  For example:
  ~bv[]
  (progn! (state-global-let* <bindings> (make-event ...)))
  ~ev[]
  However, the following form may be preferable (~pl[progn!]):
  ~bv[]
  (progn! :state-global-bindings <bindings> (make-event ...))
  ~ev[]
  ~l[remove-untouchable] for an interesting use of this exception.

  ~st[Examples illustrating how to access state]

  You can modify the ACL2 ~il[state] by doing your state-changing computation
  during the expansion phase, before expansion returns the event that is
  submitted.  Here are some examples.

  First consider the following.  Notice that expansion modifies state global
  ~c[my-global] during ~c[make-event] expansion, and then expansion returns a
  ~ilc[defun] event to be evaluated.
  ~bv[]
  (make-event
    (er-progn (assign my-global (length (w state)))
              (value '(defun foo (x) (cons x x)))))
  ~ev[]
  Then we get:
  ~bv[]
    ACL2 !>(@ my-global)
    72271
    ACL2 !>:pe foo
     L        1:x(MAKE-EVENT (ER-PROGN # #))
                 \
    >L            (DEFUN FOO (X) (CONS X X))
    ACL2 !>
  ~ev[]

  Here's a slightly fancier example, where the computation affects the
  ~ilc[defun].  In a new session, execute:
  ~bv[]
  (make-event
    (er-progn (assign my-global (length (w state)))
              (value `(defun foo (x) (cons x ,(@ my-global))))))
  ~ev[]
  Then:
  ~bv[]
    ACL2 !>(@ my-global)
    72271
    ACL2 !>:pe foo
     L        1:x(MAKE-EVENT (ER-PROGN # #))
                 \
    >L            (DEFUN FOO (X) (CONS X 72271))
    ACL2 !>
  ~ev[]
  Note that ACL2 ~il[table] ~il[events] may avoid the need to use ~il[state]
  globals.  For example, instead of the example above, consider this example in
  a new session.
  ~bv[]
  (make-event
    (let ((world-len (length (w state))))
      `(progn (table my-table :stored-world-length ,world-len)
              (defun foo (x) (cons x ,world-len)))))
  ~ev[]
  Then:
  ~bv[]
    ACL2 !>(table my-table)
     ((:STORED-WORLD-LENGTH . 72271))
    ACL2 !>:pe foo
              1:x(MAKE-EVENT (LET # #))
                 \
    >L            (DEFUN FOO (X) (CONS X 72271))
    ACL2 !>
  ~ev[]

  By the way, most built-in ~il[state] globals revert after expansion.  But
  your own global (like ~c[my-global] above) can be set during expansion, and
  the new value will persist."

  (declare (xargs :guard t))
; Keep this in sync with the -acl2-loop-only definition.
  `(make-event-fn ',form
                  ',check-expansion
                  ',on-behalf-of
                  ',event-form
                  state))

(defdoc make-event-details

  ":Doc-Section Make-event

  details on ~ilc[make-event] expansion~/

  The normal user of ~c[make-event] can probably ignore this section, but we
  include it for completeness.  We assume that the reader has read and
  understood the basic documentation for ~c[make-event] (~pl[make-event]), but
  we begin below with a summary of expansion.~/

  ~st[Introduction]

  Here is a summary of how we handle expansion involving ~c[make-event] forms.

  ~c[(make-event form :check-expansion nil)]

  This shows the ~c[:check-expansion] default of ~c[nil], and is typical user
  input.  We compute the expansion ~c[exp] of ~c[form], which is the expansion
  of the original ~c[make-event] expression and is evaluated in place of that
  expression.

  ~c[(make-event form :check-expansion t)]

  The user presumably wants it checked that the expansion doesn't change in the
  future, in particular during ~ilc[include-book].  If the expansion of
  ~c[form] is ~c[exp], then we will evaluate ~c[exp] to obtain the value as
  before, but this time we record that the expansion of the original
  ~c[make-event] expression is ~c[(make-event form :check-expansion exp)]
  rather than simply ~c[exp].

  ~c[(make-event form :check-expansion exp) ; exp a cons]

  This is generated for the case that ~c[:check-expansion] is ~c[t], as
  explained above.  Evaluation is handled as described in that above case,
  except here we check that the expansion result is the given ~c[exp].
  (Actually, the user is also allowed supply such a form.)  The original
  ~c[make-event] expression does not undergo any expansion (intuitively, it
  expands to itself).

  Now let us take a look at how we expand ~ilc[progn] forms (~ilc[encapsulate]
  is handled similarly).

  ~c[(progn ... (make-event form :check-expansion nil) ...)]

  The expansion is obtained by replacing the ~c[make-event] form as follows.
  Let ~c[exp] be the expansion of ~c[form],  Then replace the above
  ~c[make-event] form, which we denote as ~c[F], by
  ~c[(record-expansion F exp)].  Here, ~c[record-expansion] is a macro that
  returns its second argument.

  ~c[(progn ... (make-event form :check-expansion t) ...)]

  The expansion is of the form ~c[(record-expansion F exp)] as in the ~c[nil]
  case above, except that this time ~c[exp] is
  ~c[(make-event form :check-expansion exp')], where ~c[exp'] is the expansion
  of ~c[form].

  ~c[(progn ... (make-event form :check-expansion exp) ...) ; exp a cons]

  No expansion takes place unless expansion takes place for at least one of the
  other subforms of the ~c[progn], in which case each such form ~c[F] is
  replaced by ~c[(record-expansion F exp)] where ~c[exp] is the expansion of
  ~c[F].

  ~st[Detailed semantics]

  In our explanation of the semantics of ~c[make-event], we assume familiarity
  with the notion of ``embedded event form'' (~pl[embedded-event-form]).

  Let's say that the ``actual embedded event form'' corresponding to a given
  form is the underlying call of an ACL2 event: that is, ~ilc[LOCAL]s are
  dropped when ~c[ld-skip-proofsp] is ~c['include-book], and macros are
  expanded away, thus leaving us with a ~ilc[progn], a ~ilc[make-event], or an
  event form (possibly ~ilc[encapsulate]), any of which might have surrounding
  ~ilc[local], ~ilc[skip-proofs], or ~ilc[with-output] calls.

  Thus, such an actual embedded event form can be viewed as having the form
  ~c[(rebuild-expansion wrappers base-form)] where ~c[base-form] is a
  ~c[progn], a ~c[make-event], or an event form (possibly ~c[encapsulate]), and
  ~c[wrappers] are (as in ACL2 source function ~c[destructure-expansion]) the
  result of successively removing the event form from the result of
  macroexpansion, leaving a sequence of ~c[(local)], ~c[(skip-proofs)], and
  ~c[(with-output ...)] forms.  In this case we say that the form
  ``destructures into'' the indicated ~c[wrappers] and ~c[base-form], and that
  it can be ``rebuilt from'' those ~c[wrappers] and ~c[base-form].

  Elsewhere we define the notion of the ``expansion result'' from an evaluation
  (~pl[make-event]), and we mention that when expansion concludes, the ACL2
  logical ~il[world] and most of the ~c[state] are restored to their
  pre-expansion values.  Specifically, after evaluation of the argument of
  ~c[make-event] (even if it is aborted), the ACL2 logical world is restored to
  its pre-evaluation value, as are all state global variables in the list
  ~c[*protected-system-state-globals*].  Thus, assignments to
  user-defined state globals (~pl[assign]) do persist after expansion, since
  they are not in that list.

  We recursively define the combination of evaluation and expansion of an
  embedded event form, as follows.  We also simultaneously define the notion of
  ``expansion takes place,'' which is assumed to propagate upward (in a sense
  that will be obvious), such that if no expansion takes place, then the
  expansion of the given form is considered to be itself.  It is useful to keep
  in mind a goal that we will consider later: Every ~c[make-event] subterm of
  an expansion result has a ~c[:check-expansion] field that is a ~ilc[consp],
  where for this purpose ~c[make-event] is viewed as a macro that returns its
  ~c[:check-expansion] field.  (Implementation note: The latest expansion of a
  ~ilc[make-event], ~ilc[progn], ~ilc[progn!], or ~ilc[encapsulate] is stored
  in state global ~c['last-make-event-expansion], except that if no expansion
  has taken place for that form then ~c['last-make-event-expansion] has value
  ~c[nil].)~bq[]

  If the given form is not an embedded event form, then simply cause a soft
  error, ~c[(mv erp val state)] where ~c[erp] is not ~c[nil].  Otherwise:

  If the evaluation of the given form does not take place (presumably because
  ~ilc[local] events are being skipped), then no expansion takes place.
  Otherwise:

  Let ~c[x] be the actual embedded event form corresponding to the given
  form, which destructures into wrappers ~c[W] and base-form ~c[B].  Then the
  original form is evaluated by evaluating ~c[x], and its expansion is as
  follows.

  If ~c[B] is ~c[(make-event form :check-expansion val)], then expansion
  takes place if and only if ~c[val] is not a ~c[consp] and no error occurs,
  as now described.  Let ~c[R] be the expansion result from protected
  evaluation of ~c[form], if there is no error.  ~c[R] must be an embedded
  event form, or it is an error.  Then evaluate/expand ~c[R], where if
  ~c[val] is not ~c[nil] then state global ~c['ld-skip-proofsp] is
  initialized to ~c[nil].  (This initialization is important so that
  subsequent expansions are checked in a corresponding environment, i.e.,
  where proofs are turned on in both the original and subsquent
  environments.)  It is an error if this evaluation causes an error.
  Otherwise, the evaluation yields a value, which is the result of evaluation
  of the original ~c[make-event] expression, as well as an expansion,
  ~c[E_R].  Let ~c[E] be rebuilt from ~c[W] and ~c[E_R].  The expansion of
  the original form is ~c[E] if ~c[val] is ~c[nil], and otherwise is the
  result of replacing the original form's ~c[:check-expansion] field with
  ~c[E], with the added requirement that if ~c[val] is not ~c[t] (thus, a
  ~c[consp]) then ~c[E] must equal ~c[val] or else we cause an error.

  If ~c[B] is either ~c[(progn form1 form2 ...)] or
  ~c[(encapsulate sigs form1 form2 ...)], then after evaluating ~c[B], the
  expansion of the original form is the result of rebuilding from ~c[B], with
  wrappers ~c[W], after replacing each ~c[formi] in ~c[B] for which expansion
  takes place by ~c[(record-expansion formi formi')], where ~c[formi'] is the
  expansion of ~c[formi].  Note that these expansions are determined as the
  ~c[formi] are evaluated in sequence (where in the case of ~c[encapsulate],
  this determination occurs only during the first pass).  Except, if no
  expansion takes place for any ~c[formi], then the expansion of the original
  form is itself.

  Otherwise, the expansion of the original form is itself.

  ~eq[]Similarly to the ~ilc[progn] and ~ilc[encapsulate] cases above, book
  certification causes a book to be replaced by its so-called ``book
  expansion.''  There, each event ~c[ev] for which expansion took place during
  the proof pass of certification ~-[] say, producing ~c[ev'] ~-[] is replaced
  by ~c[(record-expansion ev ev')].

  Implementation Note.  The book expansion is actually implemented by way of
  the ~c[:expansion-alist] field of its ~il[certificate], which associates
  0-based positions of top-level forms in the book (not including the initial
  ~ilc[in-package] form) with their expansions.  Thus, the book's source file
  is not overwritten; rather, the certificate's expansion-alist is applied when
  the book is included or compiled.  End of Implementation Note.

  It is straightforward by computational induction to see that for any
  expansion of an embedded event form, every ~c[make-event] sub-event has a
  ~ilc[consp] ~c[:check-expansion] field.  Here, by ``sub-event'' we mean to
  expand macros; and we also mean to traverse ~c[progn] and ~c[encapsulate]
  forms as well as ~c[:check-expansion] fields of ~c[make-event] forms.  Thus,
  we will only see ~c[make-event] forms with ~c[consp] ~c[:check-expansion]
  fields in the course of ~c[include-book] forms, the second pass of
  ~c[encapsulate] forms, and raw Lisp.  This fact guarantees that an event form
  will always be treated as its original expansion.

  ~st[A note on ttags]

  ~l[defttag] for documentation of the notion of ``trust tag'' (``ttag'').  We
  note here that even if an event ~c[(defttag tag-name)] for non-~c[nil]
  ~c[tag-name] is admitted only during the expansion phase of a
  ~ilc[make-event] form, then such expansion will nevertheless still cause
  ~c[tag-name] to be recorded in the logical ~il[world] (assuming that the
  ~c[make-event] form is admitted).  This behavior will avoid surprises
  involving ttags and ~c[make-event] expansion in almost all cases, but we now
  point out a case where one might get such a surprise.

  Below we consider a ~c[make-event] specifying ~c[:check-expansion t], whose
  expansion generates a ~ilc[defttag] event during ~ilc[include-book] but not
  ~ilc[certify-book].  Consider the following book.
  ~bv[]
  (in-package \"ACL2\")
  (make-event
   (er-progn
    (if (@ skip-notify-on-defttag) ; non-nil when including a certified book
        (pprogn
         (fms \"Value of (@ skip-notify-on-defttag): ~~x0~~|\"
              (list (cons #\0 (@ skip-notify-on-defttag)))
              *standard-co* state nil)
         (encapsulate
          ()
          (defttag :foo)
          (value-triple \"Imagine something bad here!\")))
      (value nil))
    (value '(value-triple :some-value)))
   :check-expansion t)
  ~ev[]
  This book certifies successfully without the need for a ~c[:ttags] argument
  for ~ilc[certify-book].  Indeed, the above book's ~il[certificate] does not
  specify ~c[:foo] as a trust tag associated with the book, because no
  ~c[defttag] event was executed during book certification.  Unfortunately, if
  we try to include this book without specifying a value of ~c[:ttags] that
  allows ~c[:foo], book inclusion will be attempted and will only fail when the
  above ~ilc[defttag] event is eventually encountered.~/")

(defdoc using-tables-efficiently
 ":doc-section Table

  Notes on how to use tables efficiently~/

  (Thanks to Jared Davis for contributing this ~il[documentation] topic, to
  which we have made only minor modifications.)

  Suppose your book contains ~ilc[table] ~il[events], or macros that expand
  into ~c[table] events, of the following form:
  ~bv[]
     (table my-table 'my-field <computation>)
  ~ev[]
  Then ~c[<computation>] will be evaluated ~em[twice] during ~ilc[certify-book]
  and ~em[again] every time you include the book with ~ilc[include-book].  In
  some cases this overhead can be avoided using ~ilc[make-event].

  See also ~c[books/make-event/defconst-fast.lisp] for an analogous trick
  involving ~ilc[defconst].~/

  As an example, suppose we want to store numbers in a table only if they
  satisfy some computationally expensive predicate.  We'll introduce a new
  book, ~c[number-table.lisp], and create a table to store these numbers:
  ~bv[]
    (table number-table 'data nil)
  ~ev[]
  Instead of implementing a ``computationally expensive predicate,'' we'll
  write a function that just prints a message when it is called and accepts
  even numbers:
  ~bv[]
  (defun expensive-computation (n)
    (prog2$ (cw \"Expensive computation on ~~x0.~~%\" n)
            (evenp n)))
  ~ev[]
  Now we'll implement a macro, ~c[add-number], which will add its argument to
  the table only if it satisfies the expensive predicate:
  ~bv[]
  (defmacro add-number (n)
    `(table number-table 'data
            (let ((current-data
                   (cdr (assoc-eq 'data (table-alist 'number-table world)))))
              (if (expensive-computation ,n)
                  (cons ,n current-data)
                current-data))))
  ~ev[]
  Finally, we'll call ~c[add-number] a few times to finish the book.
  ~bv[]
  (add-number 1)
  (add-number 2)
  (add-number 3)
  ~ev[]
  When we now invoke ~c[(certify-book \"number-table\")], we see the expensive
  predicate being called twice for each number: once in Step 2, the main pass,
  then again in Step 3, the admissibility check.  Worse, the computation is
  performed again for each number when we use ~ilc[include-book] to load
  ~c[number-table], e.g.,
  ~bv[]
     ACL2 !>(include-book \"number-table\")
     Expensive computation on 1.
     Expensive computation on 2.
     Expensive computation on 3.
  ~ev[]
  To avoid these repeated executions, we can pull the test out of the ~c[table]
  event using ~ilc[make-event].  Here's an alternate implementation of
  ~c[add-number] that won't repeat the computation:
  ~bv[]
  (defmacro add-number (n)
    `(make-event
      (if (expensive-computation ,n)
          '(table number-table 'data
                  (cons ,n (cdr (assoc 'data
                                       (table-alist 'number-table world)))))
        '(value-triple :expensive-computation-failed))))
  ~ev[]
  When we recertify ~c[number-table.lisp], we'll see the expensive computation
  is still called once for each number in Step 2, but is no longer called
  during Step 3.  Similarly, the ~ilc[include-book] no longer shows any calls
  of the expensive computation.~/

  :cite make-event")

(defmacro record-expansion (x y)

; This funny macro simply returns its second argument.  However, we use it in
; the implementation to replace a given embedded event form x by its make-event
; expansion y, while retaining the information that y came from expanding x.

  (declare (ignore x))
  y)


; Essay on Soundness Threats

; Several of ACL2's rich set of features have the potential to compromise
; soundness unless we take suitable care, including:

; * defaxiom
; * hidden defpkg events (known-package-alist)
; * skip-proofs (skip-proofs and set-ld-skip-proofsp)
; * illegal certification world: uncertified books, non-events (including
;   redefinition), trust tags (defttag)
; * acl2-defaults-table
; * local [not yet explained here, but there's lots we could say -- see release
;   notes for related soundness bugs!]

; Here we briefly discuss these soundness threats and how we deal with them,
; pointing to other essays for further details.  Many of these issues are
; caused by LOCAL, which can introduce axioms that ultimately disappear.

; To see the potential problem with defaxiom, imagine an event such as
; (encapsulate () (local (defaxiom temp <formula>)) (defthm foo <formula>)).
; Such an event would leave us in an ACL2 logical world for which <formula> is
; stored under the name foo as through it were a logical consequence of the
; axioms in that logical world, which presumably it is not.  Our solution is to
; disallow defaxiom events in the scope of LOCAL.  This is a bit tricky since
; the LOCAL may not be lexically apparent, as when a defaxiom occurs inside a
; book that is locally included.  We therefore track LOCAL by binding state
; global variable 'in-local-flg to t (see the #+acl2-loop-only definition of
; LOCAL).

; The "hidden defpkg" problem is discussed in the Essay on Hidden Packages and
; is briefly summarized in :doc topic hidden-death-package.  The basic problem
; is that a defpkg event introduces axioms, yet it may be introduced
; temporarily through a local include-book.  The problem is thus similar to the
; defaxiom problem discussed just above, and a solution would be to disallow
; defpkg events in the scope of LOCAL.  But that solution would be harsh: For
; example, books/arithmetic/top.lisp defines packages and yet we would like to
; be able to include this book locally when proving arithmetic facts.  Our
; solution is to store all packages, even such "hidden" packages, in a world
; global 'known-package-alist.  We are careful to track such packages during
; the first pass (proof pass) of encapsulate and certify-book.  In the case of
; certify-book, we write out such defpkg events to the portcullis of the
; certificate so that they are not hidden when executing a subsequent
; corresponding include-book.

; The Essay on Skip-proofs describes our handling of skip-proofs in some
; detail, but here is a summary.  We want to claim correctness for a system of
; books that is validated using certify-book without any keyword parameters.
; We thus want to require a non-nil value of keyword parameter :skip-proofs-okp
; for any book that depends on a skip-proofs event, whether that dependency is
; in the book's certification world, is in the book itself, or is
; (hereditarily) in an included book.  We thus maintain a world global
; 'skip-proofs-seen with value t whenever the world depends on a skip-proofs,
; as explained in the above essay.

; Certification worlds are checked for legality by
; chk-acceptable-certify-book1, which collects uncertified books (using
; collect-uncertified-books) from the existing include-book-alist and which
; checks if any redefinition was done.  We of course miss uncertified
; locally-included books this way, but the only relevance of such books is
; whether they employed skip-proofs, ttags, or defaxioms, and this information
; is ultimately stored in the certificate of a parent book that is non-locally
; included in the certification world.

; The acl2-defaults-table stores the default defun-mode, and hence can affect
; soundness.  However, chk-acceptable-certify-book1 checks that the default
; defun mode is logic at certification time, and we take various measures to
; avoid other potential pitfalls (probably identifiable by tags-searches
; through the source code for acl2-defaults-table and for default-defun-mode).

; When additional, tricky soundness threats are identified, it would be good to
; describe them here, along with how we deal with them.

; End of Essay on Soundness Threats

; Essay on Skip-proofs

; The skip-proofs event allows a modular, top-down style of proof.  Skip-proofs
; differs from defaxiom: skip-proofs is intended for use when proof obligations
; are believed to be theorems but it is convenient to defer their proofs, while
; defaxiom is to be used for extending the first-order theory.  Therefore,
; while we disallow local defaxiom events (which really do not make sense; are
; we extending the theory or not?), it does make sense to allow local
; skip-proofs events.  Indeed, if we were to disallow local skip-proofs events
; then we would be ruling out the top-down, modular style of proof outlined in
; Kaufmann's article in the case studies book.

; But we then must track skip-proofs events in support of our correctness
; story.  Our claim is that when a certified book has an empty portcullis and
; all of :SKIPPED-PROOFSP, :AXIOMSP, and :TTAGS are NIL in its certificate,
; then it is sound to extend a history by including such a book without error.

; In Version_2.5 we did such tracking using world global include-book-alist.
; That tracking proved inadequate, however.  Consider the following books "top"
; and "sub".

#|
 ; book "top"
 (in-package "ACL2")
 (encapsulate
  ()
  (local (include-book "sub"))
  (defthm bad nil
    :rule-classes nil))
|#

#|
 ; book "sub"
 (in-package "ACL2")
 (skip-proofs
  (defthm bad nil
    :rule-classes nil))
|#

; In Version_2.5, if you certify these books in the initial logical world and
; then (include-book "top"), then you will not see a "Skip-proofs" warning when
; you do the include-book, because the value of :SKIPPED-PROOFSP in the
; cert-annotations of the certificate of "foo" is nil.

; Version_2.6 through Version_3.4 more carefully tracked include-books for the
; presence of supporting skip-proofs events, including skip-proofs that are
; local inside an encapsulate, using a state global, 'include-book-alist-state.
; When constructing a book's certificate, the value of
; 'include-book-alist-state was bound to nil initially and then updated by
; include-book, and its final value was used to create the post-alist of the
; certificate.  (We do not have to worry about analogous handling of :AXIOMSP
; because defaxioms are never allowed in a local context.)

; But that approach entailed, at certification time, looking in certificates of
; already-included books for skip-proofs information.  This was inefficient for
; very large certificates such as those found in the work at Centaur
; Technology.  So starting after Version_3.4 we are adopting a different
; approach.  We no longer have state global 'skipped-proofsp.  Instead, we
; focus only on maintaining world global 'skip-proofs-seen, consulting
; 'ld-skip-proofsp when we call install-event.

; We maintain the invariant that skip-proofs-seen is a form evaluated with
; proofs skipped in support of the construction of the current ACL2 logical
; world, if such exists (otherwise skip-proofs-seen is nil).  This "form" can
; be (:include-book full-book-name) if full-book-name logically supports the
; current ACL2 world (perhaps locally) and contains a skip-proofs form.  When
; we install an event, we set world global 'skip-proofs-seen (if it is not
; already set) if the event is evaluated with a non-nil value of state global
; 'ld-skip-proofsp, unless we are inside an include-book or the second pass of
; an encapsulate.  (Note that the certificate of a book already carries the
; information of whether skip-proofs was invoked during cerification, and we
; use that information when including a book.)  We may also avoid setting
; 'skip-proofs-seen if the event has no logical content, for example, a
; deflabel event.  However, we avoid updating 'skip-proofs-seen in the cases of
; encapsulate and include-book, since they manage this global themselves, as
; follows.  Encapsulate checks the value of 'skip-proofs-seen after its first
; pass and installs that value at the end of its second pass.  Include-book
; sets 'skip-proofs-seen based on its certificate (its so-called cert-obj),
; which provides skip-proofs information at the top level and also in its
; post-alist (which is set based on world global include-book-alist-all).  Note
; that certify-book does not set skip-proofs-seen in the resulting world, but
; since certify-book is not a valid embedded event form for a certification
; world, that is not a problem.

; Up through Version_3.4, we updated world globals 'skip-proofs-seen and
; 'redef-seen in maybe-add-command-landmark intead of as indicated above (in
; particular, instead of using install-event).  But with progn!, this is
; misguided -- these should be updated at the event level, not the command
; level -- as the following example shows.

; (progn! (set-ld-redefinition-action '(:doit . :overwrite) state)
;         (defun foo (x) (cons x x))
;         (set-ld-redefinition-action nil state))

; Of course, this isn't exactly a soundness bug, since one needs an active
; trust tag in order to evaluate progn!.  Nevertheless, we would like to avoid
; such a simple way to prove nil whenever there is any active trust tag!

; Finally, we note a related problem with Version_2.5 that was fixed in
; Version_2.6.  Suppose that foo.lisp and bar.lisp both have this unique
; form after (in-package "ACL2"):

#||
(defthm bad nil
  :rule-classes nil)
||#

; Now suppose we do this in a fresh session:

#||
(encapsulate ()
             (local (include-book "foo"))
             (defthm bad nil
               :rule-classes nil))
||#

; Then (certify-book "bar" 1) succeeded in Version_2.5, and in subsequent
; sessions, if we evaluated (include-book "bar"), that succeeded without
; warning or error.

; End of Essay on Skip-proofs

#+acl2-loop-only
(defmacro skip-proofs (x)
  ":Doc-Section Other

  skip proofs for a given form ~-[] a quick way to introduce unsoundness~/
  ~bv[]
  Example Form:
  (skip-proofs
    (defun foo (x)
      (if (atom x) nil (cons (car x) (foo (reverse (cdr x)))))))

  General Form:
  (skip-proofs form)
  ~ev[]
  where ~c[form] is processed as usual except that the proof obligations
  usually generated are merely assumed.

  Normally ~c[form] is an event; ~pl[events].  If you want to put
  ~c[skip-proofs] around more than one event, consider the following
  (~pl[progn]): ~c[(skip-proofs (progn event1 event2 ... eventk))].

  WARNING: ~c[Skip-proofs] allows inconsistent ~il[events] to be admitted to
  the logic.  Use it at your own risk!~/

  Sometimes in the development of a formal model or proof it is convenient to
  skip the proofs required by a given event.  By embedding the event in a
  ~c[skip-proofs] form, you can avoid the proof burdens generated by the event,
  at the risk of introducing unsoundness.  Below we list four illustrative
  situations in which you might find ~c[skip-proofs] useful.

  1. The termination argument for a proposed function definition is
  complicated.  You presume you could admit it, but are not sure that
  your definition has the desired properties.  By embedding the
  ~ilc[defun] event in a ~c[skip-proofs] you can ``admit'' the
  function and experiment with theorems about it before undoing
  (~pl[ubt]) and then paying the price of its admission.  Note however that you
  might still have to supply a measure.  The set of formals used in some valid
  measure, known as the ``measured subset'' of the set of formals, is used by
  ACL2's induction heuristics and therefore needs to be suitably specified.
  You may wish to specify the special measure of ~c[(:? v1 ... vk)], where
  ~c[(v1 ... vk)] enumerates the measured subset.

  2. You intend eventually to verify the ~il[guard]s for a definition but do
  not want to take the time now to pursue that.  By embedding the
  ~ilc[verify-guards] event in a ~c[skip-proofs] you can get the system to
  behave as though the ~il[guard]s were verified.

  3. You are repeatedly recertifying a book while making many experimental
  changes.  A certain ~ilc[defthm] in the book takes a very long time to prove
  and you believe the proof is not affected by the changes you are making.  By
  embedding the ~ilc[defthm] event in a ~c[skip-proofs] you allow the theorem
  to be assumed without proof during the experimental recertifications.

  4. You are constructing a proof top-down and wish to defer the proof of a
  ~ilc[defthm] until you are convinced of its utility.  You can embed the
  ~c[defthm] in a ~c[skip-proofs].  Of course, you may find later (when you
  attempt prove the theorem) that the proposed ~c[defthm] is not a theorem.

  Unsoundness or Lisp errors may result if the presumptions underlying a use of
  ~c[skip-proofs] are incorrect.  Therefore, ~c[skip-proofs] must be considered
  a dangerous (though useful) tool in system development.

  Roughly speaking, a ~ilc[defthm] embedded in a ~c[skip-proofs] is
  essentially a ~ilc[defaxiom], except that it is not noted as an axiom
  for the purposes of functional instantiation
  (~pl[lemma-instance]).  But a skipped ~ilc[defun] is much more subtle since
  not only is the definitional equation being assumed but so are formulas
  relating to termination and type.  The situation is also difficult to
  characterize if the ~c[skip-proofs] ~il[events] are within the scope of an
  ~ilc[encapsulate] in which constrained functions are being introduced.  In
  such contexts no clear logical story is maintained; in particular,
  constraints aren't properly tracked for definitions.  A proof script
  involving ~c[skip-proofs] should be regarded as work-in-progress, not as a
  completed proof with some unproved assumptions.  A ~c[skip-proofs] event
  represents a promise by the author to admit the given event without further
  axioms.  In other words, ~c[skip-proofs] should only be used when the belief
  is that the proof obligations are indeed theorems in the existing ACL2
  logical ~il[world].

  ACL2 allows the certification of ~il[books] containing ~c[skip-proofs]
  ~il[events] by providing the keyword argument ~c[:skip-proofs-okp t] to the
  ~ilc[certify-book] command.  This is contrary to the spirit of certified
  ~il[books], since one is supposedly assured by a valid ~il[certificate] that
  a book has been ``blessed.''  But certification, too, takes the view of
  ~c[skip-proofs] as ``work-in-progress'' and so allows the author of the book
  to promise to finish.  When such ~il[books] are certified, a warning to the
  author is printed, reminding him or her of the incurred obligation.  When
  ~il[books] containing ~c[skip-proofs] are included into a session, a warning
  to the user is printed, reminding the user that the book is in fact
  incomplete and possibly inconsistent.  This warning is in fact an error if
  ~c[:skip-proofs-okp] is ~c[nil] in the ~ilc[include-book] form;
  ~pl[include-book].~/"

  `(state-global-let*
    ((ld-skip-proofsp (or (f-get-global 'ld-skip-proofsp state)
                          t)))
    ,x))

#+acl2-loop-only
(defmacro local (x)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Keep this in sync with chk-embedded-event-form: if we skip the check on x
; there, we should skip evaluation of x here.

  ":Doc-Section Events

  hiding an event in an encapsulation or book~/
  ~bv[]
  Examples:
  (local (defthm hack1
           (implies (and (acl2-numberp x)
                         (acl2-numberp y)
                         (equal (* x y) 1))
                    (equal y (/ x)))))

  (local (defun double-naturals-induction (a b)
           (cond ((and (integerp a) (integerp b) (< 0 a) (< 0 b))
                  (double-naturals-induction (1- a) (1- b)))
                 (t (list a b)))))~/

  General Form:
  (local ev)
  ~ev[]
  where ~c[ev] is an event form.  If the current default ~il[defun-mode]
  (~pl[default-defun-mode]) is ~c[:]~ilc[logic] and ~ilc[ld-skip-proofsp] is
  ~c[nil] or ~c[t], then ~c[(local ev)] is equivalent to ~c[ev].  But if
  the current default ~il[defun-mode] is ~c[:]~ilc[program] or if
  ~ilc[ld-skip-proofsp] is ~c[']~ilc[include-book], then ~c[(local ev)] is a
  ~c[no-op].  Thus, if such forms are in the event list of an
  ~ilc[encapsulate] event or in a book, they are processed when the
  encapsulation or book is checked for admissibility in ~c[:]~ilc[logic] mode
  but are skipped when extending the host ~il[world].  Such ~il[events] are thus
  considered ``local'' to the verification of the encapsulation or
  book.  The non-local ~il[events] are the ones ``exported'' by the
  encapsulation or book.  ~l[encapsulate] for a thorough
  discussion.  Also ~pl[local-incompatibility] for a discussion of
  a commonly encountered problem with such event hiding:  you can't
  make an event local if its presence is required to make sense of a
  non-local one.

  Note that ~il[events] that change the default ~il[defun-mode], and in fact any
  ~il[events] that set the ~ilc[acl2-defaults-table], are disallowed inside
  the scope of ~c[local].  ~l[embedded-event-form]."

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'if
        '(equal (ld-skip-proofsp state) 'include-book)
        '(mv nil nil state)
        (list 'if
              '(equal (ld-skip-proofsp state) 'initialize-acl2)
              '(mv nil nil state)
              (list 'state-global-let*
                    '((in-local-flg t))
                    (list 'when-logic "LOCAL" x)))))

#+acl2-loop-only
(defmacro defchoose (&whole event-form &rest def)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; Warning: If this event ever generates proof obligations, remove it from the
; list of exceptions in install-event just below its "Comment on irrelevance of
; skip-proofs".

  ":Doc-Section Events

  define a Skolem (witnessing) function~/
  ~bv[]
  Examples:
  (defchoose choose-x-for-p1-and-p2 (x) (y z)
    (and (p1 x y z)
         (p2 x y z)))

  (defchoose choose-x-for-p1-and-p2 x (y z) ; equivalent to the above
    (and (p1 x y z)
         (p2 x y z)))

  ; The following is as above, but strengthens the axiom added to pick a sort
  ; of canonical witness, as described below.
  (defchoose choose-x-for-p1-and-p2 x (y z)
    (and (p1 x y z)
         (p2 x y z))
    :strengthen t)

  (defchoose choose-x-and-y-for-p1-and-p2 (x y) (z)
    (and (p1 x y z)
         (p2 x y z)))~/

  General Form:
  (defchoose fn
             (bound-var1 ... bound-varn)
             (free-var1 ... free-vark)
             body
             :doc doc-string
             :strengthen b),
  ~ev[]
  where ~c[fn] is the symbol you wish to define and is a new symbolic
  name (~pl[name]), ~c[(bound-var1 ... bound-varn)] is a list of
  distinct `bound' variables (see below), ~c[(free-var1 ... free-vark)]
  is the list of formal parameters of ~c[fn] and is disjoint from the
  bound variables, and ~c[body] is a term.  The use of ~c[lambda-list]
  keywords (such as ~c[&optional]) is not allowed.  The ~il[documentation]
  string argument, ~c[:doc doc-string], is optional; for a description of the
  form of ~c[doc-string] ~pl[doc-string].  The ~c[:strengthen] keyword argument
  is optional; if supplied, it must be ~c[t] or ~c[nil].

  The system treats ~c[fn] very much as though it were declared in the
  ~il[signature] of an ~ilc[encapsulate] event, with a single axiom exported as
  described below.  If you supply a ~c[:use] hint (~pl[hints]), ~c[:use fn], it
  will refer to that axiom.  No rule (of class ~c[:]~ilc[rewrite] or otherwise;
  ~pl[rule-classes]) is created for ~c[fn].

  ~c[Defchoose] is only executed in ~il[defun-mode] ~c[:]~ilc[logic];
  ~pl[defun-mode].  Also ~pl[defun-sk].

  In the most common case, where there is only one bound variable, it is
  permissible to omit the enclosing parentheses on that variable.  The effect
  is the same whether or not those parentheses are omitted.  We describe this
  case first, where there is only one bound variable, and then address the
  other case.  Both cases are discussed assuming ~c[:strengthen] is ~c[nil],
  which is the default.  We deal with the case ~c[:strengthen t] at the end.

  The effect of the form
  ~bv[]
  (defchoose fn bound-var (free-var1 ... free-vark)
    body)
  ~ev[]
  is to introduce a new function symbol, ~c[fn], with formal parameters
  ~c[(free-var1 ... free-vark)].  Now consider the following axiom, which
  states that ~c[fn] picks a value of ~c[bound-var] so that the body will be
  true, if such a value exists:
  ~bv[]
  (1)   (implies body
                 (let ((bound-var (fn free-var1 ... free-vark)))
                   body))
  ~ev[]
  This axiom is ``clearly conservative'' under the conditions expressed above:
  the function ~c[fn] simply picks out a ``witnessing'' value of ~c[bound-var]
  if there is one.  For a rigorous statement and proof of this conservativity
  claim, ~pl[conservativity-of-defchoose].

  Next consider the case that there is more than one bound variable, i.e.,
  there is more than one bound-var in the following.
  ~bv[]
  (defchoose fn
             (bound-var1 ... bound-varn)
             (free-var1 ... free-vark)
             body)
  ~ev[]
  Then ~c[fn] returns a multiple value with ~c[n] components, and formula (1)
  above is expressed using ~ilc[mv-let] as follows:
  ~bv[]
  (implies body
           (mv-let (bound-var1 ... bound-varn)
                   (fn free-var1 ... free-vark)
                   body))
  ~ev[]

  We now discuss the case that ~c[:strengthen t] is supplied.  For simplicity
  we return to our simplest case, with ~c[defchoose] applied to function
  ~c[fn], a single free variable ~c[y], and a single bound variable
  ~c[bound-var].  The idea is that if we pick the ``smallest'' witnessing
  ~c[bound-var] for two different free variables ~c[y] and ~c[y1], then either
  those two witnesses are the same, or else one is less than the other, in
  which case the smaller one is a witness for its free variable but not for the
  other.  (See comments in source function ~c[defchoose-constraint-extra] for
  more details.)  Below, ~c[body1] is the result of replacing ~c[y] by ~c[y1]
  in ~c[body].
  ~bv[]
  (2)   (or (equal (fn y) (fn y1))
            (let ((bound-var (fn y)))
              (and body
                   (not body1)))
            (let ((bound-var (fn y1)))
              (and body1
                   (not body))))
  ~ev[]
  An important application of this additional axiom is to be able to define a
  ``fixing'' function that picks a canonical representative of each equivalence
  class, for a given equivalence relation.  The following events illustrate
  this point.
  ~bv[]
  (encapsulate
   ((equiv (x y) t))
   (local (defun equiv (x y) (equal x y)))
   (defequiv equiv))

  (defchoose efix (x) (y)
    (equiv x y)
    :strengthen t)

  (defthm equiv-implies-equal-efix-1
    (implies (equiv y y1)
             (equal (efix y) (efix y1)))
    :hints ((\"Goal\" :use efix))
    :rule-classes (:congruence))

  (defthm efix-fixes
    (equiv (efix x) x)
    :hints ((\"Goal\" :use ((:instance efix (y x))))))
  ~ev[]

  If there is more than one bound variable, then (2) is modified in complete
  analogy to (1) to use ~ilc[mv-let] in place of ~ilc[let].

  Comment for logicians:  As we point out in the documentation for
  ~ilc[defun-sk], ~c[defchoose] is ``appropriate,'' by which we mean that
  it is conservative, even in the presence of ~c[epsilon-0] induction.
  For a proof, ~l[conservativity-of-defchoose].~/"

; Warning: See the Important Boot-Strapping Invariants before modifying!

  (list 'defchoose-fn
        (list 'quote def)
        'state
        (list 'quote event-form)))

(deflabel conservativity-of-defchoose
  :doc
  ":Doc-Section defchoose

  proof of conservativity of ~ilc[defchoose]~/

  This documentation topic provides underlying theory.  It is of theoretical
  interest only; it has no relationship to the effective use of ACL2.~/

  The argument below for the conservativity of ~il[defchoose] replaces the
  terse and somewhat misleading reference to a forcing argument in Appendix B
  of the paper by ACL2 authors Kaufmann and Moore, ``Structured Theory
  Development for a Mechanized Logic'' (Journal of Automated Reasoning 26,
  no. 2 (2001), pp. 161-203).

  Our basic idea is to to take a (countable) first-order structure for ACL2, M,
  together with a function symbol, f, introduced by ~il[defchoose], and find a
  way to expand M with an interpretation of f (without changing the universe of
  M) so that e0-induction continues to hold in the expansion.  A remark at the
  end of this documentation topic shows why care is necessary.  A concept
  called ``forcing'', originally introduced by Paul Cohen for set theory, has
  long since been adapted by logicians (in a simplified form) to model theory.
  This simplified model-theoretic forcing provides the means for making our
  careful expansion.

  The forcing argument presented below is intended to be completely
  self-contained for those familiar with basic first-order logic and ACL2.  No
  background in forcing (model-theoretic or otherwise) is expected, though we
  do expect a rudimentary background in first-order logic and familiarity with
  the following.

  Preliminaries.  We write s[p<-p0] to denote the result of extending or
  modifying the assignment s by binding p to p0.  Now let A be a subset of the
  universe U of a first-order structure M.  A is said to be ``first-order
  definable with parameters'' in M if for some formula phi, variable x, and
  assignment s binding the free variables of phi except perhaps for x, A = {a
  \\in U: M |= phi[s[x<-a]].  Note that we are writing ``\\in'' to denote set
  membership.  Finally, we indicate the end of a proof (or of a theorem
  statement, when the proof is omitted) with the symbol ``-|''.

  We gratefully acknowledge very helpful feedback from John Cowles, who found
  several errors in a draft of this note and suggested the exercises.  We also
  thank Ruben Gamboa for helpful feedback, and we thank Jim Schmerl for an
  observation that led us directly to this proof in the first place.

  We are given a consistent first-order theory T, extending the ACL2
  ground-zero theory, that satisfies the e0-induction scheme.  We wish to show
  that the extension of T by the following arbitrary defchoose event is
  conservative, where g is a new function symbol.
  ~bv[]
       (defchoose g <bound-vars> <free-vars> <body>)
  ~ev[]
  Note that by ``the extension of T'' here we mean the extension of T by not
  only the new defchoose axiom displayed just below, but also the addition of
  e0-induction axioms for formulas in the language with the new defchoose
  function symbol, g.
  ~bv[]
       <body> -> (LET <free-vars> = g(<bound-vars>) in <body>)
  ~ev[]
  By definition of conservativity, since proofs are finite, it clearly suffices
  to consider an arbitrary finite subset of T.  Then by the completeness,
  soundness, and downward Lowenheim-Skolem theorems of first-order logic, it
  suffices to show that an arbitrary countable model of T can be expanded
  (i.e., by interpreting the new symbol g without changing the universe of the
  model) to a model of the corresponding defchoose axiom above, in which all
  e0-induction axioms hold in the language of that model.

  Below, we will carry out a so-called ~em[forcing] construction, which
  allows us to expand any countable model M of T to a model M[G] that satisfies
  e0-induction and also satisfies the above axiom generated from the above
  defchoose event.  The ideas in this argument are standard in model theory; no
  novelty is claimed here.

  Fix a countable model M of a theory T that satisfies e0-induction and extends
  the ACL2 ground-zero theory.  Also fix the above defchoose axiom, where g is
  not in the language of T.

  We start by defining a partial order P as follows.  Let Nb and Nf be the
  lengths of <bound-vars> and <free-vars>, respectively.  P consists of all fn in
  M such that the following formula is true in M.  Roughly speaking, it says that
  fn is a finite function witnessing the above requirement for g.
  ~bv[]
         alistp(fn) &
         no-duplicatesp-equal(strip-cars(fn)) &
         (forall <bound-vars>, <free-vars> .
            (member-equal(cons(<bound-vars>,<free-vars>), fn) ->
             (length(<bound-vars>) = Nb &
              length(<free-vars>)  = Nf &
              ((exists <free-vars> . <body>) -> <body>))))
  ~ev[]
  P is ordered by subset, i.e., we say that p2 ~em[extends] p1 if p1 is a
  subset (not necessarily proper) of p2 (more precisely, M |=
  subsetp-equal(p1,p2)).

  Remark.  The original argument in Appendix B of the aforementioned paper can
  essentially be salvaged, as we now show.  The key observation is that the
  particular choice of P is nearly irrelevant for the argument that follows
  below.  In particular, we can instead define P to consist of finite one-one
  functions with domain contained in the set of natural numbers.  More
  precisely, consider the following definitions.
  ~bv[]
       (defun function-p (fn)
         (declare (xargs :guard t))
         (and (alistp fn)
              (no-duplicatesp-equal (strip-cars fn))))

       (defun nat-listp (x)
         (declare (xargs :guard t))
         (cond ((atom x)
                (equal x nil))
               (t (and (natp (car x))
                       (nat-listp (cdr x))))))

       (defun nat-function-p (x)
         (and (function-p x)
              (nat-listp (strip-cars x))))
  ~ev[]
  and define inverse as follows.
  ~bv[]
       (defun inverse (fn)
         (declare (xargs :guard (alistp fn)))
         (if (endp fn)
             nil
           (cons (cons (cdar fn) (caar fn))
                 (inverse (cdr fn)))))
  ~ev[]
  Then P may instead be defined to consist of those fn for which
  nat-function-p(fn) & function-p(inverse(fn)).  With this alternate definition
  of P, the argument below then goes through virtually unchanged, and we get an
  expansion M[G] of M in which there is a definable enumeration of the
  universe.  The conservativity of defchoose then follows easily because the
  function being introduced can be defined explicitly using that enumeration
  (namely, always pick the least witness in the sense of the enumeration).

  End of Remark.

  Next we present the relevant forcing concepts from model theory.

  A ~em[dense] subset of P is a subset D of P such that for every p \\in P,
  there is d \\in D such that d extends p.  A subset G of P is ~em[generic]
  with respect to a collection Ds of dense subsets of P, also written ``G is
  Ds-generic,'' if G is closed under subset (if p2 \\in G and p2 extends p1
  then p1 \\in G), G is pairwise compatible (the union-equal of any two
  elements of G is in G), and every set in Ds has non-empty intersection with
  G.

  For p \\in P, we say that a subset D of P is ~em[dense beyond] p if for all
  p1 extending p there exists p2 extending p1 such that p2 \\in D.  This notion
  makes sense even for D not a subset of P if we treat elements of D not in P
  as nil.

  Proposition 1.  For any partial order P and countable collection Ds of dense
  subsets of P, there is a Ds-generic subset of P.

  Proof.  Let Ds = {D0,D1,D2,...}.  Define a sequence <p_0,p_1,...> such that
  for all i, p_i \\in Di and p_(i+1) extends p_i.  Let G = {p \\in P: for some
  i, pi extends p}.  Then G is Ds-generic. -|

  Note that P is first-order definable (with parameters) in M.  Let Df be the
  set of dense subsets of P that are first-order definable (with parameters) in
  M.  A standard argument shows there are only countably many first-order
  definitions with parameters in a countable model M ~-[] for example, we can
  Goedel number all terms and then all formulas ~-[] hence, Df is countable.

  By Proposition 1, let G be Df-generic.  Notice that for any list x of length
  Nb in M, the set of elements f of P for which x is in the domain of f is
  dense and first-order definable.  We may thus define a function g0 as
  follows: g0(x_1,...,x_Nb) = y if there is some element of G containing the
  pair ((x_1 ... x_Nb) . y).  It is easy to see that g0 is a total function on
  M.  Let L be the language of T and let L[g] be the union of L with a set
  containing a single new function symbol, g.  Let M[G] be the expansion of M
  to L[g] obtained by interpreting g to be g0 (see also Proposition 5 below).

  So now we have fixed M, P, Df, G, and g0, where G is Df-generic.

  Proposition 2.  Let Df be the set of dense subsets of P that are first-order
  definable (with parameters) in M.  Suppose that p \\in G and D \\in Df.  Then for
  some q \\in G extending p, q \\in D.

  Proof.  Let D0 be the set of p' \\in D that either extend p or have no
  extension in D that extends p.  We leave it as a straightforward exercise to
  show that D0 is dense, and D0 is clearly first-order definable (with
  parameters) in M.  So by genericity of G, we may pick q \\in D0 such that q
  \\in G.  Thus q \\in D.  By definition of generic, some extension q1 of both
  p and q belongs to G.  Pick q2 \\in D extending q1; thus q has an extension
  in D that extends p (namely, q2), so by definition of D0, q extends p. -|

  Definition of forcing.  Let phi(x1,...,xk) be a first-order formula in L[g]
  and let p \\in P.  We define a formula of L, denoted ``p ||- phi'' (``p
  forces phi''), by recursion on phi (in the metatheory) as follows.  (Here, we
  view ``or'' and ``forall'' as abbreviations.)

  ~bq[]
    If phi is atomic, then let phi'(A) be the result of replacing, inside-out,
    each subterm of the form g(x_1,...,x_Nb) with the term (cdr (assoc-equal
    (list x_1 ... x_Nb) A)), where A is neither p nor a variable occurring in
    phi.  Then p ||- phi is defined as follows: ``The set {A \\in P: A extends
    p and phi'(A)} is dense beyond p''.  That is, p ||- phi is the following
    formula:
  ~bv[]
      (forall p1 \\in P extending p)
       (exists p2 \\in P extending p1) phi'(p2).
  ~ev[]
    p ||- ~~phi is:  (forall p' \\in P extending p) ~~(p' ||- phi)

    p ||- phi_1 & phi_2 is: (p ||- phi_1) & (p ||- phi_2)

    p ||- (exists x) phi is:  (exists x) (p ||- phi)
  ~eq[]

  We will need the following definition later.

  Definition.  p ||-w phi (p ~em[weakly forces] phi) is an abbreviation for p
  ||- ~~~~phi.

  The following exercises were suggested by John Cowles as a means for gaining
  familiarity with the definition of forcing.

  Exercise 1. Consider the formula (phi_1 OR phi_2) as an abbreviation for
  ~~(~~phi_1 & ~~phi_2), Show that p ||- (phi_1 OR phi_2) is equivalent to the
  following.
  ~bv[]
       (forall p' \\in P extending p) (exists p'' \\in P extending p')
        ((p'' ||- phi_1) OR (p'' ||- phi_2))
  ~ev[]

  Exercise 2. Consider the formula (forall x)phi as an abbreviation for
  ~~(exists x)~~phi, Show that p ||- (forall x)phi is equivalent to the following.
  ~bv[]
       (forall x)
        (forall p1 \\in P extending p)
         (exists p2 \\in P extending p1) (p2 ||- phi).
  ~ev[]

  Exercise 3. Prove that p ||-w phi is equivalent to the following.
  ~bv[]
       (forall p' \\in P extending p)
        (exists p'' \\in P extending p') (p'' ||- phi).
  ~ev[]

  Exercise 4. Let phi be a formula of L[g].  Prove:
       M |= (p ||-  phi)[s[p<-p0]] implies
       M |= (p ||-w phi)[s[p<-p0]].

  Exercise 5. Let phi be a formula of L[g].  Prove:
       M |= (p ||-  ~~phi)[s[p<-p0]] iff
       M |= (p ||-w ~~phi)[s[p<-p0]].

  [End of exercises.]

  The definition of forcing stipulates how to view ``p ||- phi(x1,...,xk)'' as
  a new formula theta(p,x1,...,xk).  That is, ``||-'' transforms formulas, so
  for any first-order formula phi, ``p ||- phi'' is just another first-order
  formula.  That observation shows that a formula such as ((p ||- phi) OR (p
  ||- ~~phi)) is really just another first-order formula.  The following
  proposition thus follows easily.

  Proposition 3. For any formula phi of L[g], {p0: M |= ((p ||- phi) OR (p ||-
  ~~phi))[s[p<-p0]]]} is a dense subset of P, which (since it is first-order
  definable with parameters in M) intersects G. -|

  The following proposition is easily proved by a structural induction on phi,
  and is left to the reader.

  Proposition 4. Let phi be a formula of L[g].  Suppose ~c[p0 \in P],
  ~c[p1 \in P],~nl[]
  M |= (p ||- phi)[s[p<-p0]] and p1 extends p0.  Then~nl[]
  M |= (p ||- phi)[s[p<-p1]]. -|

  We will also need the following.

  Proposition 5. The following is dense for any finite set S of Nb-tuples: {p
  \\in P: for some <x_1 ... x_Nb> \\in S, (list x_1 ... x_Nb) \\in
  strip-cars(p)}.  Thus, the function g0 is a total function. -|

  The next lemma tells us that the sentences true in M[G] are those that are
  forced by an element of G.

  Truth Lemma.  Let phi be a formula in L[g], let s be an assignment to the
  free variables of phi, and let p be a variable not in the domain of s.  Then
  M[G] |= phi[s] iff for some p0 \\in G, M |= (p ||- phi)[s[p<-p0]].

  Proof.  The proof is by induction on the structure of phi.  First suppose phi
  is atomic.  Let D* be the set of elements p0 \\in P such that every
  assoc-equal evaluation from the definition of forcing phi returns a pair when
  A is bound to p0.  (Intuitively, this means that p0 is a sufficiently large
  approximation from any G containing p0 to make sense of phi in M[G].)  We
  make the following claim.
  ~bv[]
  (*)   For all p0 \\in G such that p0 \\in D*,
        M[G] |= phi[s] iff M |= (p ||- phi)[s[p<-p0]].
  ~ev[]

  To prove the claim, fix p0 in both G and D*, and recall the function g0
  constructed from G in the definition of M[G].  Suppose that t_1, ..., t_Nb
  are terms and g(t_1, ..., t_Nb) is a subterm of phi.  Then s assigns a value
  in M to each of the t_i.  Let a_i be the value assigned by s to t_i.  Then
  g0(a_1, ..., a_Nb) = (cdr (assoc-equal (list a_1 ... a_Nb) p0)), as the
  assoc-equal is a pair (since p0 \\in D*) and has the indicated value (because
  p0 \\in G).  It follows by the definition of formula phi' in the definition
  of forcing:
  ~bv[]
       M[G] |= phi[s]  iff  M |= phi'(p)[s[p<-p0]]
  ~ev[]
  Moreover, because p0 \\in D* it is clear that this holds if p0 is replaced by
  an arbitrary extension of p0.  Then (*) easily follows.

  By Proposition 5, D* is dense, so there is some p0 in the intersection of D*
  and G.  The forward direction of the conclusion then follows by (*).  The
  reverse direction is clear from (*) by application of Proposition 2 to D* and
  Proposition 4.

  Next, suppose M[G] |= ~~phi[x].  Then it is not the case that M[G] |= phi, so
  by the inductive hypothesis, there is no p0 \\in G for which M |= (p ||-
  phi)[s[p<-p0]].  By Proposition 3, there is p0 \\in G for which M |= (p ||-
  ~~phi)[s[p<-p0]].  For the other direction, suppose it is not the case that
  M[G] |= ~~phi[s].  So M[G] |= phi[s], and by the inductive hypothesis, there
  is p0 \\in G for which M |= (p ||- phi)[s[p<-p0]].  It follows that there is
  no p1 \\in G for which M |= (p ||- ~~phi)[s[p<-p1]], since from such p1 we can
  find a common extension p2 of p0 and p1 (since G is generic), and since p2
  extends p0 then by Proposition 4, M |= (p ||- phi)[s[p<-p2]], contradicting
  (by definition of forcing) M |= (p ||- ~~phi)[s[p<-p1]] since p2 extends p1.

  The case (phi_1 & phi_2) follows easily from the inductive hypothesis.  For
  the forward direction, apply Proposition 4 and the observation that by
  genericity, if p0 \\in G and p1 \\in G then p0 and p1 they have a common
  extension in G.

  Finally, the case (exists x) phi follows trivially from the inductive
  hypothesis. -|

  Truth Lemma Corollary.  The Truth Lemma holds with ||-w replacing ||-.

  Proof.  This is clear by applying the Truth Lemma to ~~~~phi. -|

  Here is our main theorem.  Recall that all first-order theories in our ACL2
  context satisfy the e0-induction scheme.

  Theorem.  M[G] satisfies e0-induction.

  Proof.  We consider an arbitrary instance of e0-induction in L[g], stated
  using a strict well-founded relation <| and a formula phi.  We write phi(y)
  to indicate that y may be among the free variables of phi, and phi(y<-x) to
  denote the result of substituting x for y in phi.
  ~bv[]
    theta(y):   (forall y) [((forall x <| y) phi(y<-x)) -> phi(y)]
             -> (forall y) phi(y)
  ~ev[]
  Our goal is to prove that theta holds in M[G].

  Below, we abuse notation by leaving assignments implicit and by writing ``p
  ||- phi(y0)'' to signify that the formula (p ||- phi(y)) is true in M under
  the extension of the explicit assignment that binds y to y0.  We believe that
  the intended meaning will be clear.

  Consider the following set D.
  ~bv[]
    D = {p \\in P: either p ||-w phi(y0) for all y0,
                  or else
                  for some y0, p ||- ~~phi(y0) and
                               for all y1 <| y0 p ||-w phi(y1)}.
  ~ev[]
  The set D is clearly first-order definable (with parameters) in M.  We claim
  that D is a dense subset of P.  For suppose p0 \\in P; we find p1 \\in D
  extending p0, as follows.  If p0 ||-w phi(y0) for all y0, then we may take p1
  to be p0.  Otherwise, by definition of ||-w and ||-, there is some y0 such
  that for some extension p0' of p0, p0' ||- ~~phi(y0).  Pick a <|-minimal such
  y0, and correspondingly pick p1 so that p1 extends p0 and p1 ||- ~~phi(y0).
  In order to show that p1 \\in D, it remains to show that for all y1 <| y0,
  p1 ||-w phi(y1), i.e., there is no q extending p1 such that q ||- ~~phi(y1).
  This is indeed the case since otherwise q and y1 would contradict the
  <|-minimality of y0.

  Applying the genericity of G and just-proved density of D, pick p0 \\in G
  such that p0 \\in D.  If p0 ||-w phi(y0) for all y0, then by the Truth Lemma
  Corollary, M[G] |= phi(y0) for all y0, and thus M[G] |= theta.  Otherwise,
  since p0 \\in D we may choose y0 such that p0 ||- ~~phi(y0) and for all y1 <|
  y0, p0 ||-w phi(y1).  By the Truth Lemma and its corollary, since p0 \\in G
  we have:
  ~bv[]
  (1)   M[G] |= ~~phi(y0).
  (2)   For all y1 <| y0, M[G] |= phi(y1).
  ~ev[]
  It follows that the antecedent of theta is false in M[G], as witnessed by y =
  y0; thus M[G] |= theta. -|

  Remark.  We close by returning, as promised above, to the question of why so
  much care is necessary in constructing an expansion of M.  We assume
  familiarity here with the notion of a ``non-standard'' natural number of M,
  i.e., one that is greater than the interpretation of any term that has the
  form (+ 1 1 1 ... 1).  Here is a very simple example that illustrates the
  need for some care.  Consider the following event, which introduces a
  function foo with the following property: for all x, if natp(x) then
  natp(foo(x)).
  ~bv[]
       (defchoose foo (y) (x)
         (implies (natp x) (natp y)))
  ~ev[]
  Certainly we can build a model of the above property from a model M of the
  ground-zero theory, by interpreting foo so that for all x for which M
  satisfies natp(x), foo(x) is also a natp in M.  But suppose we start with a
  non-standard model M of the ground-zero theory, and we happen to define
  foo(x) to be 1 for all non-standard natural numbers x and 0 for all other x.
  The resulting expansion of M will not satisfy the e0-induction scheme or even
  the ordinary natural number induction scheme: foo(0)=0 holds in that
  expansion as does the implication foo(n)=0 => foo(n+1)=0 for every natural
  number n of M, standard or not; and yet foo(k)=0 fails for every non-standard
  natural number k of M.")

#+acl2-loop-only
(defmacro defattach (&whole event-form &rest args)

; Warning: See the Important Boot-Strapping Invariants before modifying!

; See the Essay on Defattach.

  ":Doc-Section Events

  execute constrained functions using corresponding attached functions~/

  This ~il[documentation] topic is organized into the following sections:

  ~st[Introductory example.]~nl[]
  ~st[Syntax and semantics of defattach.]~nl[]
  ~st[Three primary uses of defattach.]~nl[]
  ~st[Miscellaneous remarks, with discussion of possible user errors.]

  Please ~pl[encapsulate] if you intend to use ~c[defattach] but are not
  already familiar with the use of ~c[encapsulate] to introduce constrained
  functions.

  ~st[Introductory example.]

  We begin with a short log illustrating the use of ~c[defattach].  Notice that
  after evaluating the event ~c[(defattach f g)], a call of the constrained
  function ~c[f] is evaluated by instead calling ~c[g] on the arguments.

  ~bv[]
  ACL2 !>(encapsulate
          ((f (x) t :guard (true-listp x)))
          (local (defun f (x) x))
          (defthm f-property
            (implies (consp x) (consp (f x)))))
  [... output omitted ...]
   T
  ACL2 !>(defun g (x)
           (declare (xargs :guard (or (consp x) (null x))))
           (cons 17 (car x)))
  [... output omitted ...]
   G
  ACL2 !>(f '(3 4)) ; undefined function error


  ACL2 Error in TOP-LEVEL:  ACL2 cannot ev the call of undefined function
  F on argument list:

  ((3 4))

  To debug see :DOC print-gv, see :DOC trace, and see :DOC wet.

  ACL2 !>(defattach f g)
  [... output omitted ...]
   :ATTACHMENTS-RECORDED
  ACL2 !>(f '(3 4)) ; f is evaluated using g
  (17 . 3)
  ACL2 !>(trace$ f g)
   ((F) (G))
  ACL2 !>(f '(3 4)) ; f is evaluated using g
  1> (ACL2_*1*_ACL2::F (3 4))
    2> (ACL2_*1*_ACL2::G (3 4))
      3> (G (3 4))
      <3 (G (17 . 3))
    <2 (ACL2_*1*_ACL2::G (17 . 3))
  <1 (ACL2_*1*_ACL2::F (17 . 3))
  (17 . 3)
  ACL2 !>(defattach f nil) ; unattach f (remove its attachment)
  [... output omitted ...]
   :ATTACHMENTS-RECORDED
  ACL2 !>(f '(3 4)) ; undefined function error once again
  1> (ACL2_*1*_ACL2::F (3 4))


  ACL2 Error in TOP-LEVEL:  ACL2 cannot ev the call of undefined function
  F on argument list:

  ((3 4))

  To debug see :DOC print-gv, see :DOC trace, and see :DOC wet.

  ACL2 !>
  ~ev[]

  ~st[Syntax and semantics of defattach.]

  The log above shows that the event ~c[(defattach f g)] allows ~c[g] to be
  used for evaluating calls of ~c[f].  From a logical perspective, the
  evaluation takes place in the addition to the current session of an
  ``attachment equation'' axiom (universally quantified over all ~c[x]) for
  each ~c[defattach] event:
  ~bv[]
  (equal (f x) (g x)) ;;; attachment equation axiom for (defattach f g)
  ~ev[]

  Below we explain ~c[defattach] in some detail.  But it is important to keep
  in mind that evaluation with the attachment equations takes place in an
  extension of the logical theory of the session.  ACL2 guarantees that this
  so-called ``evaluation theory'' remains consistent, assuming that the absence
  of ~ilc[defaxiom] ~il[events] from the user.  A deeper discussion of the
  logical issues is available (but not intended to be read by most users) in a
  comment in the ACL2 source code labeled ``Essay on Defattach.''

  ~bv[]
  Example Forms:
  (defattach f g)   ; call g in place of calling constrained function f
  (defattach (f g)) ; same as just above
  (defattach (f g :hints ((\"Goal\" :in-theory (enable foo)))))
                    ; equivalent to first form above, except with hints for the
                    ; proof that the guard of f implies the guard of g
  (defattach (f g :hints ((\"Goal\" :in-theory (enable foo)))
                  :otf-flg t))
                    ; as above, except with an :otf-flg of t for the proof that
                    ; the guard of f implies the guard of g
  (defattach (f g)
             :hints ((\"Goal\" :use my-thm)))
                    ; equivalent to first form above, except with hints for the
                    ; proof that the constraints on f hold for g
  (defattach (f g)
             :hints ((\"Goal\" :use my-thm))
             :otf-flg t)
                    ; as above, except with an :otf-flg of t for the proof that
                    ; the constraints on f hold for g
  (defattach (f g)
             (h j)) ; Attach g to f and attach j to h
  (defattach (f g :attach nil)
             (h j)) ; Same as just above, including the same proof obligations,
                    ; except for one difference: because of :attach nil, calls
                    ; of f will not be evaluated, i.e., there will be no
                    ; executable attachment of g to f
  (defattach (f nil)
             (h j)) ; Attach j to h and unattach f
  (defattach (f g :hints ((\"Goal\" :in-theory (enable foo))))
             (h j :hints ((\"Goal\" :in-theory (enable bar))))
             :hints ((\"Goal\" :use my-thm)))
                    ; Attach g to f and attach j to h, with hints:
                    ; - For proving that the guard of f implies the guard of g,
                    ;   enable foo;
                    ; - For proving that the guard of h implies the guard of j,
                    ;   enable bar; and
                    ; - For proving that the constraints on f and h hold for
                    ;   g and j (respectively), use theorem my-thm.

  
  (defattach f nil)   ; remove the attachment of f, if any (e.g., g above)
  (defattach (f nil)) ; same as just above~/

  General Forms:
  (defattach f g)   ; single attach or, if g is nil, unattach
  (defattach (f1 g1 :kwd val ...)
             ...
             (fk gk :kwd' val' ...)
             :kwd'' val'' ...)
  ~ev[]
  where each indicated keyword-value pair is optional and each keyword is one
  of ~c[:ATTACH], ~c[:HINTS], ~c[:OTF-FLG], or ~c[:INSTRUCTIONS].  The
  value of each ~c[:ATTACH] keyword is either ~c[t] or ~c[nil], with default
  ~c[t] except that the value of ~c[:ATTACH] at the ``top level,'' after each
  entry ~c[(fi gi ...)], is the default for each ~c[:ATTACH] keyword supplied
  in such an entry.  The associated values for the other keywords have the
  usual meanings for the proof obligations described below: the guard proof
  obligation for keywords within each ~c[(fi gi ...)] entry, and the constraint
  proof obligation for keywords at the top level.  No keyword may occur twice
  in the same context, i.e., within the same ~c[(fi gi ...)] entry or at the
  top level; and ~c[:INSTRUCTIONS] may not occur in the same context with
  ~c[:HINTS] or ~c[:OTF-FLG].

  The first General Form above is simply an abbreviation for the form
  ~c[(defattach (f g))], which is an instance of the second General Form above.
  For the second General Form we say that ~c[gi] is ``attached to'' ~c[fi] (by
  the ~c[defattach] event) if ~c[gi] is not ~c[nil], and otherwise we say that
  ~c[fi] is ``unattached'' (by the ~c[defattach] event).  It is also convenient
  to refer to ~c[<fi,gi>] as an ``attachment pair'' (of the event) if ~c[gi] is
  not ~c[nil].  We may refer to the set of ~c[fi] as the ``attachment nest'' of
  each ~c[fi].

  We start with a brief introduction to the first General Form in the case that
  ~c[g] is not ~c[nil].  This form arranges that during evaluation, with
  exceptions noted below, every call of the constrained function symbol ~c[f]
  will in essence be replaced by a call of the function symbol ~c[g] on the
  same arguments.  We may then refer to ~c[g] as the ``attachment of'' ~c[f],
  or say that ``~c[g] is attached to ~c[f].''  Notable exceptions, where we do
  not use attachments during evaluation, are for macroexpansion, evaluation of
  ~ilc[defconst] and ~ilc[defpkg] terms, evaluation during ~ilc[table] events,
  and especially evaluation of ground terms (terms without free variables)
  during proofs.  Regarding the last of these, consider the following example.
  ~bv[]
  (defstub f (x) t)
  (defun g (x) (+ 3 x))
  (defattach f g)
  ~ev[]
  If the form ~c[(f 2)] is submitted at the ACL2 prompt, the result will be
  ~c[5] because the attachment ~c[g] of ~c[f] is called on the argument,
  ~c[2].  However, during a proof the term ~c[(f 2)] will not be simplified to
  ~c[5], since that would be unsound, as there are no axioms about ~c[f] that
  would justify such a simplification.

  For the case that ~c[g] is ~c[nil] in the first General Form above, the
  result is the removal of the existing attachment to ~c[f], if any.  After
  this removal, calls of ~c[f] will once again cause errors saying that ``ACL2
  cannot ev the call of undefined function ~c[f] ...''.  In this case not only
  is the previous attachment to ~c[f] removed; moreover, for every function
  symbol ~c[f'] in the attachment nest of ~c[f] in the ~c[defattach] event that
  introduced the existing attachment to ~c[f], then ~c[f'] is unattached.  (An
  example near the end of this ~il[documentation] topic shows why this
  unattachment needs to be done.) Such removal takes place before the current
  ~c[defattach] is processed, but is restored if the new event fails to be
  admitted.

  We focus henceforth on the second General Form.  There must be at least one
  attachment, i.e., ~c[i] must be at least 1.  All keywords are optional; their
  role is described below.  The ~c[fi] must be distinct constrained function
  symbols, that is, function symbols all introduced in ~il[signature]s of
  ~ilc[encapsulate] ~il[events] (or macros such as ~ilc[defstub] that generate
  ~ilc[encapsulate] events).  Each non-~c[nil] ~c[gi] is a
  ~c[:]~ilc[logic]-mode function symbol that has had its guards verified, with
  the same ~il[signature] as ~c[fi] (though formal parameters for ~c[fi] and
  ~c[gi] may have different names).  This event generates proof obligations and
  an ordering check, both described below.  The effect of this event is first
  to remove any existing attachments for all the function symbols ~c[fi], as
  described above for the first General Form, and then to attach each ~c[gi] to
  ~c[fi].

  Proof obligations must be checked before making attachments.  For this
  discussion we assume that each ~c[gi] is non-~c[nil] (otherwise first remove
  all attachment pairs ~c[<fi,gi>] for which ~c[gi] is nil).  Let ~c[s] be the
  functional substitution mapping each ~c[fi] to ~c[gi].  For any term ~c[u],
  we write ~c[u\s] for the result of applying ~c[s] to ~c[u]; that is, ~c[u\s]
  is the ``functional instance'' obtained by replacing each ~c[fi] by ~c[gi] in
  ~c[u].  Let ~c[G_fi] and ~c[G_gi] be the guards of ~c[fi] and ~c[gi],
  respectively.  ACL2 first proves, for each ~c[i] (in order), the formula
  ~c[(implies G_fi G_gi)\s].  If this sequence of proofs succeeds, then the
  remaining formula to prove is the functional instance ~c[C\s] of the
  conjunction ~c[C] of the constraints on the symbols ~c[fi]; ~pl[constraint].
  This last proof obligation is thus similar to the one generated by functional
  instantiation (~pl[constraint]).  As with functional instantiation, ACL2
  stores the fact that such proofs have been done so that they are avoided in
  future events (~pl[lemma-instance]).  Thus, you will likely avoid some proofs
  with the sequence
  ~bv[]
  (defattach f g)
  (defattach f nil)
  (defattach f g)
  (defattach f nil)
  ...
  ~ev[]
  rather than the sequence:
  ~bv[]
  (defattach f g)
  :u
  (defattach f g)
  :u
  ...
  ~ev[]

  It remains to describe an ordering check.  We begin with the following
  motivating example.
  ~bv[]
  (defstub f (x) t) ; constrained function with no constraints
  (defun g (x) (declare (xargs :guard t)) (not (f x)))
  (defattach f g) ; ILLEGAL!
  ~ev[]
  Were the above ~c[defattach] event to succeed, the evaluation theory
  (discussed above) would be inconsistent: ~c[(f x)] equals ~c[(g x)] by the
  new attachment equation, which in turn equals ~c[(not (f x))] by definition
  of ~c[g].  The evaluation would therefore be meaningless.  Also, from a
  practical perspective, there would be an infinite loop resulting from any
  call of ~c[f].

  We consider a function symbol ~c[g] to be an ``extended ancestor of'' a
  function symbol ~c[f] if either of the following two criteria is met: (a) ~c[g]
  occurs in the formula that introduces ~c[f] (i.e., definition body or
  constraint) and ~c[g] is introduced by an event different from (earlier than)
  the event introducing ~c[f]; or (b) ~c[g] is attached to ~c[f].  For a
  proposed ~c[defattach] event, we check that the resulting extended ancestor
  relation has no cycles, where for condition (b) we include all attachment
  pairs that would result, including those remaining from earlier ~c[defattach]
  events.

  Of course, a special case is that no function symbol may be attached to
  itself.  Similarly, no function symbol may be attached to any of its
  ``siblings'' ~-[] function symbols introduced by the same event ~-[] as
  siblings are considered equivalent for purposes of the acyclicity check.

  ~st[Three primary uses of defattach.]~nl[]

  We anticipate three uses of ~c[defattach]:

  (1) Constrained function execution

  (2) Sound modification of the ACL2 system

  (3) Program refinement

  We discuss these in turn.

  (1) The example at the beginning of this ~il[documentation] illustrates
  constrained function execution.

  (2) ACL2 is written essentially in itself.  Thus, there is an opportunity to
  attaching to system functions.  This is illustrated by the use of constrained
  function ~c[too-many-ifs-post-rewrite-wrapper] in the ACL2 source code, for
  a heuristic used in the rewriter.  Early in an ACL2 build, this function
  receives a trivial attachment that implements a trivial heuristic.  Then late
  in the ACL2 build, that attachment is replaced, so that the wrapper receives
  a more interesting attachment, one which implements the desired heuristic.

  Over time, we expect to continue replacing ACL2 source code in a similar
  manner.  We invite the ACL2 community to assist in this ``open architecture''
  enterprise; feel free to email the ACL2 implementors if you are interested in
  such activity.

  (3) Recall that for an attachment pair ~c[<f,g>], a proof obligation is
  (speaking informally) that ~c[g] satisfies the constraint on ~c[f].  Yet more
  informally speaking, ~c[g] is ``more defined'' than ~c[f]; we can think of
  ~c[g] as ``refining'' ~c[f].  We can consider attachment as refinement with a
  moreformal perspective: the evaluation theory extends the theory of the ACL2
  session, specifically by the addition of all attachment equations.  For the
  logic-inclined, it may be useful to think model-theoretically: The class of
  models of the evaluation theory is non-empty but is a subset of the class of
  models of the current session theory.

  ~st[Miscellaneous remarks, with discussion of possible user errors.]

  We conclude with remarks on some details.

  A ~c[defattach] event is never redundant (~pl[redundant-events]); in that
  sense it is analogous to ~ilc[in-theory].

  As mentioned above, the use of attachments is disabled for evaluation of
  ground terms during proofs.  However, attachments can be used on code during
  the proof process, essentially when the ``program refinement'' is on theorem
  prover code rather than on functions we are reasoning about.  The attachment
  to ~c[too-many-ifs-post-rewrite-wrapper] described above provides one example
  of such attachments.  Meta functions and clause-processor functions can also
  have attachments, with the restriction that no common ancestor with the
  evaluator can have an attachment; ~pl[evaluator-restrictions].

  For an attachment pair ~c[<f,g>], evaluation of ~c[f] never consults the
  ~il[guard] of ~c[f].  Rather, control passes to ~c[g], whose guard is checked
  if necessary.  The proof obligation related to guards, as described above,
  guarantees that any legal call of ~c[f] is also a legal call of ~c[g].  Thus
  for guard-verified code that results in calls of ~c[f] in raw Lisp, it is
  sound to replace these calls with corresponding calls of ~c[g].

  ~c[Defattach] events are illegal inside any ~ilc[encapsulate] event with a
  non-empty ~il[signature] unless they are ~il[local] to the ~ilc[encapsulate].

  To see all attachments: ~c[(all-attachments (w state))].

  We conclude with an example promised above, showing why it is necessary in
  general to unattach all function symbols in an existing attachment nest when
  unattaching any one of those function symbols.  Consider the following
  example.
  ~bv[]
  (defstub f1 () t)
  (encapsulate ((f2 () t))
    (local (defun f2 () (f1)))
    (defthm f2=f1 (equal (f2) (f1))))
  (encapsulate ((f3 () t))
    (local (defun f3 () (f1)))
    (defthm f3=f1 (equal (f3) (f1))))
  (defun four () (declare (xargs :guard t)) 4)
  (defun five () (declare (xargs :guard t)) 5)
  (defattach (f1 four) (f2 four))
  (defattach (f1 five) (f3 five))
  ~ev[]
  The second ~c[defattach] replaces erases the existing attachment pair
  ~c[<f1,four>] before installing the new attachment pairs ~c[<f1,five>] and
  ~c[<f3,five>].  After the second defattach, both ~c[(f1)] and ~c[(f3)]
  evaluate to 5.  Now suppose that the attachment pair ~c[<f2,four>] were not
  erased.  Then we would have ~c[(f1)] evaluating to 5 and ~c[(f2)] evaluating
  to 4, contradicting the constraint ~c[f2=f1].  The evaluation theory would
  thus be inconsistent, and at a more concrete level, the user might well be
  surprised by evaluation results if the code were written with the assumption
  specified in the constraint ~c[f2=f1].~/"

  (list 'defattach-fn
        (list 'quote args)
        'state
        (list 'quote event-form)))

; Note:  Important Boot-Strapping Invariants

; If any of the above forms are modified, be sure to change the
; setting of *initial-event-defmacros* as described there.  Each of
; the defmacros above (except those excused) is of a rigid form
; recognized by the function primordial-event-macro-and-fn.  For
; example, there are no declarations and the bodies used above are
; simple enough to be translatable by boot-translate before the world
; is created.

; More subtly, except for local, each macro generates a call of a
; corresponding -fn function on some actuals computed from the macros
; args: THE FORMALS OF THE -fn FUNCTIONS CAN BE DETERMINED BY LOOKING
; AT THE ACTUALS!  For example, we can see that the 'formals for
; 'in-theory-fn, whenever it gets defined, will be '(expr state doc
; event-form).  The function primordial-event-macro-and-fn1 computes
; the formals from the actuals.  Don't change the expressions above,
; don't even change the formals to the defmacros, and don't change the
; formals of the -fns unless you understand this!

; End of *initial-event-defmacros* discussion.


; GETPROP - an efficient applicative property list replacement.

; We provide here a property list facility with applicative
; semantics.  The two primitive operations are putprop and
; getprop.  A ``world-alist'' is a list of ``triples'' of the
; form (symbol key . val).  Putprop conses triples on to a given
; world-alist.  Getprop take a symbol and key and looks for the
; first member of the given world-alist with the given symbol and
; key, returning the corresponding val, or a default if no such
; triple is found.

; In the ``usual case'', the cost of a getprop will be no more than
; the cost of a couple of get's in Common Lisp, rather than a search
; linear in the length of the given world-alist.  The efficiency is
; based upon the strange ``world-name'' extra argument of getprop.
; Formally, world-name is to be regarded as a parameter of getprop
; that is simply ignored.  Practically speaking, getprop uses this
; hint to check whether the given world-alist is in fact currently and
; validly represented by a set of properties on property lists.  To do
; this, getprop checks that as the 'acl2-world-pair property of the
; given world-name, there is a pair whose car is (eq) the given
; world-alist.  If this is the case, then the cdr of the pair, say
; world-key, is a gensymed symbol.  The world-key property of any
; given symbol, symb, is an alist containing exactly those pairs (key
; . val) such that (symb key . val) is in world-alist.  That is, to
; find the key property of symb it is sufficient to assoc-eq for key
; up the alist obtained by (get symb world-key).

; For a more thorough description of the issues concerning
; installation of worlds, see the discussion in interface-raw.lisp,
; under the section heading EXTENDING AND RETRACTING PROPERTY LIST
; WORLDS.

; To use getprop and putprop effectively, one must think clearly in
; terms of the usual order of Lisp evaluation.  Getprop is only fast
; on worlds that have been ``installed'' as by extend-world or
; retract-world.

(deflabel worldp) ; reserving this symbol for later use

(defun plist-worldp (alist)
  (declare (xargs :guard t))
  #-acl2-loop-only
  (cond ((eq alist (w *the-live-state*))
         (return-from plist-worldp t)))
  (cond ((atom alist) (eq alist nil))
        (t
         (and (consp (car alist))
              (symbolp (car (car alist)))
              (consp (cdr (car alist)))
              (symbolp (cadr (car alist)))
              (plist-worldp (cdr alist))))))

(defthm plist-worldp-forward-to-assoc-eq-equal-alistp
  (implies (plist-worldp x)
           (assoc-eq-equal-alistp x))
  :rule-classes :forward-chaining)

(defdoc getprop
  ":Doc-Section Programming

  access fast property lists~/

  ~bv[]
  General form:
  (getprop symb key default world-name world-alist)
  ~ev[]

  See ~c[books/misc/getprop.lisp] for an example that illustrates the use
  of ACL2 utilities ~ilc[getprop] and ~c[putprop] to take advantage of
  under-the-hood Lisp (hashed) property lists.~/~/")

(defun putprop (symb key value world-alist)

  ":Doc-Section Programming

  update fast property lists~/

  ~bv[]
  General form:
  (putprop symbol key value world-alist)
  ~ev[]

  See ~c[books/misc/getprop.lisp] for an example that illustrates the use
  of ACL2 utilities ~ilc[getprop] and ~c[putprop] to take advantage of
  under-the-hood Lisp (hashed) property lists.~/~/"

  (declare (xargs :guard (and (symbolp symb)
                              (symbolp key)
                              (plist-worldp world-alist))))
  (cons (cons symb (cons key value)) world-alist))

; Occasionally you will find comments of the form:

; On Metering

; Occasionally in this code you will see forms protected by
; #+acl2-metering.  If you (push :acl2-metering *features*) and then
; recompile the affected forms, you will get some additional printing
; that indicates random performance meters we have found useful.

; The following two definitions support a particularly common style of
; metering we do.  Suppose you have a typical tail recursive fn for
; exploring a big list

; (defun scan (lst)
;   (cond (test
;          finish)
;         (t
;          (scan (cdr lst)))))

; We often meter it with:

; (defun scan (lst)
;   (cond (test
;          #+acl2-metering (meter-maid 'scan 100)
;          finish)
;         (t
;          #+acl2-metering (setq meter-maid-cnt (1+ meter-maid-cnt))
;          (scan (cdr lst)))))

; Where (meter-maid 'scan 100) tests meter-maid-cnt against 100 and if
; it is bigger prints a msg about 'scan.  In any case, meter-maid
; resets cnt to 0.  This style of metering is not very elegant because
; meter-maid-cnt ought to be initialized cleanly to 0 "at the top" and
; protected against error aborts (i.e., by binding it).  But to do
; that we'd have to recode many of our tail recursive functions so
; they had preludes and lets.  With our meter-maid style, we can just
; insert the metering text into the existing text and preserve the
; tail recursion and lack of initialization.  Not often in metered
; runs do we abort (leaving meter-maid-cnt artificially high) and that
; results (at worst) in a spurious report on the next metered call.

#-acl2-loop-only
(defparameter meter-maid-cnt 0)

#-acl2-loop-only
(defun meter-maid (fn maximum &optional arg1 arg2 cnt)
  (cond ((> (or cnt meter-maid-cnt) maximum)
         (cond
          (arg2
           (format t "~%Meter:  ~s on ~s and ~s used ~s cycles.~%"
                   fn arg1 arg2 (or cnt meter-maid-cnt)))
          (arg1
           (format t "~%Meter:  ~s on ~s used ~s cycles.~%"
                   fn arg1 (or cnt meter-maid-cnt)))
          (t (format t "~%Meter:  ~s used ~s cycles.~%"
                     fn (or cnt meter-maid-cnt))))))
  (setq meter-maid-cnt 0))

; If we ever find this value stored under a property, then getprop acts as
; though no value was found.  Thus, this value had better never be stored as a
; "legitimate" value of the property.  To belabor this point:  we have here a
; fundamental difference between our getprop and Lisp's get.

(defconst *acl2-property-unbound* :acl2-property-unbound)

(defun getprop-default (symb key default)
  (declare (xargs :guard t))
  (prog2$
   (and (consp default)
        (eq (car default) :error)
        (consp (cdr default))
        (stringp (cadr default))
        (null (cddr default))
        (hard-error 'getprop
                    "No property was found under symbol ~x0 for key ~x1.  ~@2"
                    (list (cons #\0 symb)
                          (cons #\1 key)
                          (cons #\2 (cadr default)))))
   default))

#-acl2-loop-only
(defun-one-output sgetprop1 (symb key default world-alist inst-world-alist
                                  inst-gensym)
  (do ((tl world-alist (cdr tl)))
      ((null tl)
       (getprop-default symb key default))
      (cond ((eq tl inst-world-alist)
             (return-from
              sgetprop1
              (let ((temp (assoc-eq key (get symb inst-gensym))))
                (cond (temp
                       (cond
                        ((cdr temp)
                         (let ((ans (car (cdr temp))))
                           (if (eq ans *acl2-property-unbound*)
                               (getprop-default symb key default)
                               ans)))
                        (t (getprop-default symb key default))))
                      (t (getprop-default symb key default))))))
            ((and (eq symb (caar tl))
                  (eq key (cadar tl)))
             (return-from
              sgetprop1
              (let ((ans (cddar tl)))
                (if (eq ans *acl2-property-unbound*)
                    (getprop-default symb key default)
                    ans)))))))

; The following code, not generally loaded, is used to augment fgetprop to
; determine the frequency with which we access properties.  See the
; fgetprop-stats comment in fgetprop for a description of how to use
; this code.

#|
(defvar fgetprop-stats nil)

(defvar analyzed-fgetprop-stats nil)

(compile
 (defun update-fgetprop-stats (sym key)
   (let* ((sym-entry (assoc sym fgetprop-stats :test #'eq))
          (key-entry (assoc key (cdr sym-entry) :test #'eq)))
     (cond (key-entry (setf (cdr key-entry) (1+ (cdr key-entry))))
           (sym-entry (setf (cdr sym-entry) (cons (cons key 1) (cdr sym-entry))))
           (t (setq fgetprop-stats
                    (cons (cons sym (list (cons key 1))) fgetprop-stats)))))))

(compile
 (defun analyze-fgetprop-stats nil
   (format t "Properties accessed and access counts:~%")
   (loop
    for x in (sort (let ((prop-alist nil))
                     (loop
                      for pair in fgetprop-stats
                      do
                      (loop
                       for x in (cdr pair)
                       do
                       (let ((temp (assoc (car x) prop-alist :test #'eq)))
                         (cond (temp (setf (cdr temp) (+ (cdr temp) (cdr x))))
                               (t (setq prop-alist
                                        (cons (cons (car x) (cdr x))
                                              prop-alist)))))))
                     prop-alist)
                   #'(lambda (x y) (> (cdr x) (cdr y))))
    do
    (format t "~A~50T~9D~%" (car x) (cdr x)))
   (terpri t)
   (setq analyzed-fgetprop-stats
         (sort
          (loop
           for pair in fgetprop-stats
           collect
           (let* ((other-cutoff 1)
                  (others
                   (loop
                    for x in (cdr pair) when (<= (cdr x) other-cutoff)
                    sum (cdr x))))
             (list* (car pair)
                    (loop for x in (cdr pair) sum (cdr x))
                    (let ((temp
                           (sort (loop
                                  for x in (cdr pair)
                                  when
                                  (or (= others 0)
                                      (= others other-cutoff) ;i.e., just 1 other
                                      (> (cdr x) other-cutoff))
                                  collect x)
                                 #'(lambda (x y)(> (cdr x) (cdr y))))))
                      (if (> others other-cutoff)
                          (append temp
                                  (list (cons "all other" others)))
                        temp)))))
          #'(lambda (x y) (> (cadr x) (cadr y)))))
   (format t "Analyzed fgetprop-stats~%")
   (loop
    for trip in analyzed-fgetprop-stats
    do
    (format t "~S~45T~9D~%" (car trip) (cadr trip))
    (loop
     for pair in (cddr trip)
     do
     (format t " ~A~50T~9D~%" (car pair) (cdr pair))))
   t))

|# ; |

; Note:  In versions before V2.2 the following defvar was in
; interface-raw.lisp.  But it is used earlier than that in the
; initialization process.

; *current-acl2-world-key* is the property used for the current-acl2-
; world world.  We use a defvar here so that it will not get reset
; merely by reloading the sources of this file when debugging.

#-acl2-loop-only
(defvar *current-acl2-world-key* (make-symbol "*CURRENT-ACL2-WORLD-KEY*"))

(defun fgetprop (symb key default world-alist)

; This is getprop's meaning when we know the world name is 'current-acl2-world.
; The invariant maintained for the 'current-acl2-world is the same as that
; maintained for other world names with the additional fact that the installed
; alist itself is the value of the state global variable 'current-acl2-world,
; whose raw lisp counterpart is ACL2_GLOBAL_ACL2::CURRENT-ACL2-WORLD, and the
; gensym under which the property alist is stored for each symbol is also kept
; in the raw lisp global *current-acl2-world-key*.  Put another way, (get
; 'current-acl2-world 'acl2-world-pair) returns a pair equal to (cons
; ACL2_GLOBAL_ACL2::CURRENT-ACL2-WORLD *current-acl2-world-key*).

  (declare (xargs :guard (and (symbolp symb)
                              (symbolp key)
                              (plist-worldp world-alist))))

  #+acl2-loop-only
  (cond ((endp world-alist) default)
        ((and (eq symb (caar world-alist))
              (eq key (cadar world-alist)))
         (let ((ans (cddar world-alist)))
           (if (eq ans *acl2-property-unbound*)
               default
               ans)))
        (t (fgetprop symb key default (cdr world-alist))))
  #-acl2-loop-only
  (declare (special *current-acl2-world-key*
                    ACL2_GLOBAL_ACL2::CURRENT-ACL2-WORLD))

; The following two lines are commented out.  They collect the fgetprop-stats.
; Those stats will tell you, for a given run of the system, which properties
; are accessed, the frequency with which they are accessed, and a breakdown by
; symbol of all the properties accessed.  If you wish to collect the
; fgetprop-stats, then load the code above into raw lisp, remove the two
; semi-colons below, reload this defun of fgetprop, and run some experiments.
; Then use (analyze-fgetprop-stats) to print out the results.  It is generally
; advisable to compile all the defuns just loaded.

; #-acl2-loop-only
; (update-fgetprop-stats symb key)

  #-acl2-loop-only
  (cond
   ((eq world-alist
        ACL2_GLOBAL_ACL2::CURRENT-ACL2-WORLD)
    (let ((temp
           (assoc-eq key
                     (get symb *current-acl2-world-key*))))
      (cond (temp
             (cond
              ((cdr temp)
               (let ((ans (car (cdr temp))))
                 (if (eq ans *acl2-property-unbound*)
                     (getprop-default symb key default)
                     ans)))
              (t (getprop-default symb key default))))
            (t (getprop-default symb key default)))))
   (t (sgetprop1 symb key default world-alist
                 ACL2_GLOBAL_ACL2::CURRENT-ACL2-WORLD
                 *current-acl2-world-key*))))

(defun sgetprop (symb key default world-name world-alist)

; This is getprop's meaning when we don't know the world-name.

  (declare (xargs :guard (and (symbolp symb)
                              (symbolp key)
                              (symbolp world-name)
                              (plist-worldp world-alist))))

; Note that if default has the form '(:error string) where string is a
; stringp, then in raw Lisp we execute a hard error with context
; 'getprop and string string.  Otherwise (and logically in any case),
; default is what we return when there is no key property of symb.

  #+acl2-loop-only
  (cond ((endp world-alist) default)
        ((and (eq symb (caar world-alist))
              (eq key (cadar world-alist)))
         (let ((ans (cddar world-alist)))
           (if (eq ans *acl2-property-unbound*)
               default
             ans)))
        (t (sgetprop symb key default world-name (cdr world-alist))))
  #-acl2-loop-only
  (let ((pair (get world-name 'acl2-world-pair)))
    (cond (pair (sgetprop1 symb key default world-alist (car pair) (cdr pair)))
          (t (do ((tl world-alist (cdr tl)))
                 ((null tl)
                  (getprop-default symb key default))
                 (cond ((and (eq symb (caar tl))
                             (eq key (cadar tl)))
                        (return-from
                         sgetprop
                         (let ((ans (cddar tl)))
                           (if (eq ans *acl2-property-unbound*)
                               (getprop-default symb key default)
                             ans))))))))))

(defun ordered-symbol-alistp (x)

; An ordered-symbol-alist is an alist whose keys are symbols which are
; in the symbol-< order.

  (declare (xargs :guard t))
  (cond ((atom x) (null x))
        ((atom (car x)) nil)
        (t (and (symbolp (caar x))
                (or (atom (cdr x))
                    (and (consp (cadr x))
                         (symbolp (caadr x))
                         (symbol-< (caar x)
                                   (caadr x))))
                (ordered-symbol-alistp (cdr x))))))

(in-theory (disable symbol-<))

(defthm ordered-symbol-alistp-forward-to-symbol-alistp
  (implies (ordered-symbol-alistp x)
           (symbol-alistp x))
  :rule-classes :forward-chaining)

(defun add-pair (key value l)
  (declare (xargs :guard (and (symbolp key)
                              (ordered-symbol-alistp l))))
  (cond ((endp l)
         (list (cons key value)))
        ((eq key (caar l))
         (cons (cons key value) (cdr l)))
        ((symbol-< key (caar l))
         (cons (cons key value) l))
        (t (cons (car l)
                 (add-pair key value (cdr l))))))

(defun remove-first-pair (key l)
  (declare (xargs :guard (and (symbolp key)
                              (symbol-alistp l)
                              (assoc-eq key l))))
  (cond ((endp l) nil)
        ((eq key (caar l)) (cdr l))
        (t (cons (car l)
                 (remove-first-pair key (cdr l))))))

(defun getprops1 (alist)

; Each element of alist is of the form (key val1 ... valk), i.e., key is bound
; to a stack of vali's.  We transform each element to (key . val1), i.e., each
; key is bound to the top-most vali.  An empty stack or a top value of
; *acl2-property-unbound* means there is no binding for key.

  (declare (xargs :guard (true-list-listp alist)))
  (cond ((endp alist) nil)
        ((or (null (cdar alist))
             (eq (car (cdar alist)) *acl2-property-unbound*))
         (getprops1 (cdr alist)))
        (t (cons (cons (caar alist) (cadar alist))
                 (getprops1 (cdr alist))))))

(defun getprops (symb world-name world-alist)

; returns all of the properties of symb in world-alist, as a list of
; key-value pairs, sorted according to ordered-symbol-alistp.  We
; respect the *acl2-property-unbound* convention.

  (declare (xargs :guard (and (symbolp symb)
                              (symbolp world-name)
                              (plist-worldp world-alist))
                  :mode :program))
  #+acl2-metering
  (setq meter-maid-cnt (1+ meter-maid-cnt))
  (cond #-acl2-loop-only
        ((eq world-alist (car (get world-name 'acl2-world-pair)))
         #+acl2-metering
         (meter-maid 'getprops 100 symb)
         (sort (getprops1 (get symb (cdr (get world-name 'acl2-world-pair))))
               #'(lambda (x y)
                   (symbol-< (car x) (car y)))))
        ((endp world-alist)
         #+acl2-metering
         (meter-maid 'getprops 100 symb)
         nil)
        ((eq symb (caar world-alist))
         (let ((alist (getprops symb world-name (cdr world-alist))))
           (if (eq (cddar world-alist) *acl2-property-unbound*)
               (if (assoc-eq (cadar world-alist) alist)
                   (remove-first-pair (cadar world-alist) alist)
                 alist)
             (add-pair (cadar world-alist)
                       (cddar world-alist)
                       alist))))
        (t (getprops symb world-name (cdr world-alist)))))

(verify-termination-boot-strap getprops (declare (xargs :mode :logic
                                             :verify-guards nil)))

; We don't verify the guards for getprops until we have LOCAL, which really
; means, until LOCAL has STATE-GLOBAL-LET*.

; We disable the following function in order to protect people from getting
; burned by string<-l.

(in-theory (disable string<))

(defthm equal-char-code
  (implies (and (characterp x)
                (characterp y))
           (implies (equal (char-code x) (char-code y))
                    (equal x y)))
  :rule-classes nil
  :hints (("Goal" :use
           ((:instance
             code-char-char-code-is-identity
             (c x))
            (:instance
             code-char-char-code-is-identity
             (c y))))))

(defun has-propsp1 (alist exceptions known-unbound)

; This function is only called from raw lisp code in has-propsp.  Alist is the
; alist of ACL2 properties stored on the property list of some symbol.  As
; such, each element of alist is of the form (prop val1 val2 ... valk) where
; val1 is the most recently stored value of the property prop for that symbol.
; We here check that each val1 is *acl2-property-unbound* (unless prop is among
; exceptions or known-unbound).

  (declare (xargs :guard (and (assoc-eq-equal-alistp alist)
                              (true-listp exceptions)
                              (true-listp known-unbound))))

  (cond ((endp alist) nil)
        ((or (null (cdar alist))
             (eq (cadar alist) *acl2-property-unbound*)
             (member-eq (caar alist) exceptions)
             (member-eq (caar alist) known-unbound))
         (has-propsp1 (cdr alist) exceptions known-unbound))
        (t t)))

(defun has-propsp (symb exceptions world-name world-alist known-unbound)

; We return t iff symb has properties other than those listed in exceptions.

  (declare (xargs :guard (and (symbolp symb)
                              (symbolp world-name)
                              (plist-worldp world-alist)
                              (true-listp exceptions)
                              (true-listp known-unbound))))
  #+acl2-metering
  (setq meter-maid-cnt (1+ meter-maid-cnt))
  (cond #-acl2-loop-only
        ((eq world-alist (car (get world-name 'acl2-world-pair)))
         #+acl2-metering
         (meter-maid 'has-propsp 100 symb)
         (has-propsp1 (get symb (cdr (get world-name 'acl2-world-pair)))
                      exceptions
                      known-unbound))
        ((endp world-alist)
         #+acl2-metering
         (meter-maid 'has-propsp 100 symb)
         nil)
        ((or (not (eq symb (caar world-alist)))
             (member-eq (cadar world-alist) exceptions)
             (member-eq (cadar world-alist) known-unbound))
         (has-propsp symb exceptions world-name (cdr world-alist)
                     known-unbound))
        ((eq (cddar world-alist) *acl2-property-unbound*)
         (has-propsp symb exceptions world-name (cdr world-alist)
                     (cons (cadar world-alist) known-unbound)))
        (t t)))

(defun extend-world (name wrld)

; Logically speaking, this function is a no-op that returns wrld.
; Practically speaking, it changes the Lisp property list
; state so that future getprops on name and wrld will be fast.
; However, wrld must be an extension of the current world installed
; under name, or else a hard error occurs.  Finally, if name is
; 'current-acl2-world, then no changes are made, since we do not want
; the user to smash our world.

  #+acl2-loop-only
  (declare (xargs :guard t)
           (ignore name))
  #+acl2-loop-only
  wrld
  #-acl2-loop-only
  (cond ((eq name 'current-acl2-world)
         wrld)
        (t (extend-world1 name wrld))))

(defun retract-world (name wrld)

; Logically speaking, this function is a no-op that returns wrld.
; Practically speaking, it changes the Lisp property list
; state so that future getprops on name and wrld will be fast.
; However, wrld must be a retraction of the current world installed
; under name, or else a hard error occurs.  Finally, if name is
; 'current-acl2-world, then no changes are made, since we do not want
; the user to smash our world.

  #+acl2-loop-only
  (declare (xargs :guard t)
           (ignore name))
  #+acl2-loop-only
  wrld
  #-acl2-loop-only
  (cond ((eq name 'current-acl2-world)
         wrld)
        (t (retract-world1 name wrld))))

(defun global-val (var wrld)

; If you are tempted to access a global variable value with getprop
; directly, so you can specify your own default value, it suggests
; that you have not initialized the global variable.  See the
; discussion in primordial-world-globals.  Follow the discipline of
; always initializing and always accessing with global-val.

  (declare (xargs :guard (and (symbolp var)
                              (plist-worldp wrld))))
  (getprop var 'global-value
           '(:error "GLOBAL-VAL didn't find a value.  Initialize this ~
                     symbol in PRIMORDIAL-WORLD-GLOBALS.")
           'current-acl2-world wrld))

; Declarations.

(defun function-symbolp (sym wrld)

; Sym must be a symbolp.  We return t if sym is a function symbol and
; nil otherwise.  We exploit the fact that every function symbol has a
; formals property.  Of course, the property may be NIL so when we
; seek it we default to t so we can detect the absence of the
; property.  Of course, if someone were to putprop 'formals t we would
; therefore claim the symbol weren't a function-symbolp.  This fact is
; exploited when we prepare the world for the redefinition of a
; symbol.  If for some reason you change the default, you must change
; it there too.  It would be a good idea to search for 'formals t.

  (declare (xargs :guard (and (symbolp sym)
                              (plist-worldp wrld))))
  (not (eq (getprop sym 'formals t 'current-acl2-world wrld) t)))

; We define translate-declaration-to-guard and accompanying functions in
; program mode, including the-fn, simply so that they take up a little less
; space in the image by avoiding the need to store 'def-bodies and
; 'unnormalized-body properties.

(defun translate-declaration-to-guard/integer (lo var hi)
  (declare (xargs :guard t
                  :mode :program))
  (let ((lower-bound
         (cond ((integerp lo) lo)
               ((eq lo '*) '*)
               ((and (consp lo)
                     (integerp (car lo))
                     (null (cdr lo)))
                (1+ (car lo)))
               (t nil)))
        (upper-bound
         (cond ((integerp hi) hi)
               ((eq hi '*) '*)
               ((and (consp hi)
                     (integerp (car hi))
                     (null (cdr hi)))
                (1- (car hi)))
               (t nil))))
    (cond ((and upper-bound lower-bound)
           (cond ((eq lower-bound '*)
                  (cond ((eq upper-bound '*)
                         (list 'integerp var))
                        (t (list 'and
                                 (list 'integerp var)
                                 (list '<= var upper-bound)))))
                 (t (cond ((eq upper-bound '*)
                           (list 'and
                                 (list 'integerp var)
                                 (list '<= lower-bound var)))
                          (t

; It is tempting to use integer-range-p below.  However, integer-range-p was
; introduced in Version_2.7 in support of signed-byte-p and unsigned-byte-p,
; whose definitions were kept similar to those that had been in the ihs library
; for some time.  Hence, integer-range-p is defined in terms of a strict <
; comparison to the upper integer, which does not fit well with our current
; needs.  (It feels wrong to use (< var (1+ upper-bound)), even though not
; unsound.)

                           (list 'and
                                 (list 'integerp var)
                                 (list '<= lower-bound var)
                                 (list '<= var upper-bound)))))))
          (t nil))))

(defun weak-satisfies-type-spec-p (x)
  (declare (xargs :guard t))
  (and (consp x)
       (eq (car x) 'satisfies)
       (true-listp x)
       (equal (length x) 2)
       (symbolp (cadr x))))

;; RAG - I added entries for 'real and 'complex.  Guards with 'complex
;; have CHANGED SEMANTICS!  Yikes!  Before, the moniker 'complex had
;; the semantics of complex-rationalp.  Now, it has the semantics of
;; complexp.  I added a new declaration, 'complex-rational, to stand
;; for the old semantics of 'complex.

(defun translate-declaration-to-guard1 (x var wrld)

; Wrld is either nil or an ACL2 logical world.  We get a pretty good check even
; if wrld is nil, but if wrld is the ACL2 logical world we can do a stronger
; check.

  (declare (xargs :guard (or (null wrld)
                             (plist-worldp wrld))
                  :mode :program))
  (cond ((or (eq x 'integer)
             (eq x 'signed-byte))
         (list 'integerp var))
        ((and (consp x)
              (eq (car x) 'integer)
              (true-listp x)
              (equal (length x) 3))
         (translate-declaration-to-guard/integer (cadr x) var (caddr x)))
        ((eq x 'rational) (list 'rationalp var))
        ((eq x 'real) (list 'real/rationalp var))
        ((eq x 'complex) (list 'complex/complex-rationalp var))
        ((and (consp x)
              (eq (car x) 'rational)
              (true-listp x)
              (equal (length x) 3))
         (let ((lower-bound
                (cond ((rationalp (cadr x)) (cadr x))
                      ((eq (cadr x) '*) '*)
                      ((and (consp (cadr x))
                            (rationalp (car (cadr x)))
                            (null (cdr (cadr x))))
                       (list (car (cadr x))))
                      (t nil)))
               (upper-bound
                (cond ((rationalp (caddr x)) (caddr x))
                      ((eq (caddr x) '*) '*)
                      ((and (consp (caddr x))
                            (rationalp (car (caddr x)))
                            (null (cdr (caddr x))))
                       (list (car (caddr x))))
                      (t nil))))
           (cond
            ((and upper-bound lower-bound)
             (cond
              ((eq lower-bound '*)
               (cond
                ((eq upper-bound '*)
                 (list 'rationalp var))
                (t (list 'and
                         (list 'rationalp var)
                         (cond ((consp upper-bound)
                                (list '< var (car upper-bound)))
                               (t (list '<= var upper-bound)))))))
              (t (cond
                  ((eq upper-bound '*)
                   (list 'and
                         (list 'rationalp var)
                         (cond ((consp lower-bound)
                                (list '< (car lower-bound) var))
                               (t (list '<= lower-bound var)))))
                  (t (list 'and
                           (list 'rationalp var)
                           (cond ((consp lower-bound)
                                  (list '< (car lower-bound) var))
                                 (t (list '<= lower-bound var)))
                           (cond ((consp upper-bound)
                                  (list '> (car upper-bound) var))
                                 (t (list '<= var upper-bound)))))))))
            (t nil))))
        ((and (consp x)
              (eq (car x) 'real)
              (true-listp x)
              (equal (length x) 3))
         (let ((lower-bound
                (cond ((real/rationalp (cadr x)) (cadr x))
                      ((eq (cadr x) '*) '*)
                      ((and (consp (cadr x))
                            (real/rationalp (car (cadr x)))
                            (null (cdr (cadr x))))
                       (list (car (cadr x))))
                      (t nil)))
               (upper-bound
                (cond ((real/rationalp (caddr x)) (caddr x))
                      ((eq (caddr x) '*) '*)
                      ((and (consp (caddr x))
                            (real/rationalp (car (caddr x)))
                            (null (cdr (caddr x))))
                       (list (car (caddr x))))
                      (t nil))))
           (cond
            ((and upper-bound lower-bound)
             (cond
              ((eq lower-bound '*)
               (cond
                ((eq upper-bound '*)
                 (list 'real/rationalp var))
                (t (list 'and
                         (list 'real/rationalp var)
                         (cond ((consp upper-bound)
                                (list '< var (car upper-bound)))
                               (t (list '<= var upper-bound)))))))
              (t (cond
                  ((eq upper-bound '*)
                   (list 'and
                         (list 'real/rationalp var)
                         (cond ((consp lower-bound)
                                (list '< (car lower-bound) var))
                               (t (list '<= lower-bound var)))))
                  (t (list 'and
                           (list 'real/rationalp var)
                           (cond ((consp lower-bound)
                                  (list '< (car lower-bound) var))
                                 (t (list '<= lower-bound var)))
                           (cond ((consp upper-bound)
                                  (list '> (car upper-bound) var))
                                 (t (list '<= var upper-bound)))))))))
            (t nil))))
        ((eq x 'bit) (list 'or
                           (list 'equal var 1)
                           (list 'equal var 0)))
        ((and (consp x)
              (eq (car x) 'mod)
              (true-listp x)
              (equal (length x) 2)
              (integerp (cadr x)))
         (translate-declaration-to-guard/integer 0 var (1- (cadr x))))
        ((and (consp x)
              (eq (car x) 'signed-byte)
              (true-listp x)
              (equal (length x) 2)
              (integerp (cadr x))
              (> (cadr x) 0))
         (list 'signed-byte-p (cadr x) var))
        ((eq x 'unsigned-byte)
         (translate-declaration-to-guard/integer 0 var '*))
        ((and (consp x)
              (eq (car x) 'unsigned-byte)
              (true-listp x)
              (equal (length x) 2)
              (integerp (cadr x))
              (> (cadr x) 0))
         (list 'unsigned-byte-p (cadr x) var))
        ((eq x 'atom) (list 'atom var))
        ((eq x 'character) (list 'characterp var))
        ((eq x 'cons) (list 'consp var))
        ((eq x 'list) (list 'listp var))
        ((eq x 'nil)

; We return a translated nil here instead of just nil so as not to
; look like we're saying "This is an unrecognized declaration."

         ''nil)
        ((eq x 'null) (list 'eq var nil))
        ((eq x 'ratio) (list 'and
                             (list 'rationalp var)
                             (list 'not (list 'integerp var))))
        ((eq x 'standard-char) (list 'standard-charp var))
        ((eq x 'string) (list 'stringp var))
        ((and (consp x)
              (eq (car x) 'string)
              (true-listp x)
              (equal (length x) 2)
              (integerp (cadr x))
              (>= (cadr x) 0))
         (list 'and
               (list 'stringp var)
               (list 'equal
                     (list 'length var)
                     (cadr x))))
        ((eq x 'symbol) (list 'symbolp var))
        ((eq x 't) t)
        ((and (weak-satisfies-type-spec-p x)
              (or (null wrld)
                  (eql (length (getprop (cadr x) 'formals nil
                                        'current-acl2-world wrld))
                       1)))
         (list (cadr x) var))
        ((and (consp x)
              (eq (car x) 'member)
              (eqlable-listp (cdr x)))
         (list 'member var (list 'quote (cdr x))))
        (t nil)))

(mutual-recursion

;; RAG - This was modified to change the moniker 'complex to use
;; complexp instead of complex-rationalp.

(defun translate-declaration-to-guard (x var wrld)

; This function is typically called on the sort of x you might write
; in a TYPE declaration, e.g., (DECLARE (TYPE x var1 ... varn)).
; Thus, x might be something like '(or symbol cons (integer 0 128))
; meaning that var is either a symbolp, a consp, or an integer in the
; given range.  X is taken as a declaration about the variable symbol
; var and is converted into an UNTRANSLATED term about var.

; Wrld is either nil or an ACL2 logical world.  We get a pretty good check even
; if wrld is nil, but if wrld is the ACL2 logical world we can do a stronger
; check.  This extra argument was added after Version_3.0 when Dave Greve
; pointed out that Common Lisp only allows the type-spec (satisfies pred) when
; pred is a unary function symbol, not a macro.  But we allow a value of
; nil for backward compatibility, as some users have called
; translate-declaration-to-guard in a context where the world is not
; available.  In fact, we make such a call in the-fn.  Note that a non-nil wrld
; can only strengthen this function, causing it to return nil in more cases.

; WARNING: This function is assumed elsewhere (e.g., in the generation
; of the recognizers for user declared single-threaded objects) to
; return a pseudo-termp.  We use all-vars on the output of this
; function to determine if the var is mentioned.

  (declare (xargs :guard (or (null wrld)
                             (plist-worldp wrld))
                  :mode :program

; See the comment above translate-declaration-to-guard/integer.

;                  :measure (acl2-count x)
                  ))
  (cond ((atom x) (translate-declaration-to-guard1 x var wrld))
        ((eq (car x) 'not)
         (cond ((and (true-listp x)
                     (equal (length x) 2))
                (let ((term (translate-declaration-to-guard (cadr x)
                                                            var
                                                            wrld)))
                  (and term
                       (list 'not term))))
               (t nil)))
        ((eq (car x) 'and)
         (cond ((true-listp x)
                (cond ((null (cdr x)) t)
                      (t (let ((args (translate-declaration-to-guard-lst
                                      (cdr x)
                                      var
                                      wrld)))
                           (cond (args (cons 'and args))
                                 (t nil))))))
               (t nil)))
        ((eq (car x) 'or)
         (cond ((true-listp x)
                (cond ((null (cdr x)) ''nil)
                      (t (let ((args (translate-declaration-to-guard-lst
                                      (cdr x)
                                      var
                                      wrld)))
                           (cond (args (cons 'or args))
                                 (t nil))))))
               (t nil)))
        ((eq (car x) 'complex)
         (cond ((and (consp (cdr x))
                     (null (cddr x)))
                (list 'and
                      (list 'complex/complex-rationalp var)
                      (translate-declaration-to-guard (cadr x)
                                                      (list 'realpart var)
                                                      wrld)
                      (translate-declaration-to-guard (cadr x)
                                                      (list 'imagpart var)
                                                      wrld)))
               (t nil)))
        (t (translate-declaration-to-guard1 x var wrld))))

(defun translate-declaration-to-guard-lst (l var wrld)

; Wrld is either nil or an ACL2 logical world.  We get a pretty good check even
; if wrld is nil, but if wrld is the ACL2 logical world we can do a stronger
; check.

  (declare (xargs ; :measure (acl2-count l)
                  :guard (and (true-listp l)
                              (consp l)
                              (or (null wrld)
                                  (plist-worldp wrld)))
                  :mode :program))
  (and (consp l)
       (let ((frst (translate-declaration-to-guard (car l) var wrld)))
         (cond ((null frst)
                nil)
               ((endp (cdr l))
                (list frst))
               (t (let ((rst (translate-declaration-to-guard-lst
                              (cdr l)
                              var
                              wrld)))
                    (cond ((null rst) nil)
                          (t (cons frst rst)))))))))

)

(deflabel declare
  :doc
  ":Doc-Section ACL2::Programming

  declarations~/
  ~bv[]
  Examples:
  (declare (ignore x y z))
  (declare (ignorable x y z)
           (type integer i j k)
           (type (satisfies integerp) m1 m2))
  (declare (xargs :guard (and (integerp i)
                              (<= 0 i))
                  :guard-hints ((\"Goal\" :use (:instance lemma3
                                                (x (+ i j)))))))~/

  General Form:
  (declare d1 ... dn)
  where, in ACL2, each di is of one of the following forms:

    (ignore v1 ... vn) -- where each vi is a variable introduced in
    the immediately superior lexical environment.  These variables must not
    occur free in the scope of the declaration.

    (ignorable v1 ... vn) -- where each vi is a variable introduced in
    the immediately superior lexical environment.  These variables need not
    occur free in the scope of the declaration.  This declaration can be useful
    for inhibiting compiler warnings.

    (type type-spec v1 ... vn) -- where each vi is a variable introduced in the
    immediately superior lexical environment and type-spec is a type specifier
    (as described in the documentation for ~il[type-spec]).

    (xargs :key1 val1 ... :keyn valn) -- where the legal values of the keyi's
    and their respective vali's are described in the documentation for
    ~il[xargs].

    (optimize ...) -- for example, ~c[(optimize (safety 3))].  This is allowed
    only at the top level of ~ilc[defun] forms.  See any Common Lisp
    documentation for more information.

  ~ev[]
  Declarations in ACL2 may occur only where ~c[dcl] occurs below:
  ~bv[]
    (DEFUN name args doc-string dcl ... dcl body)
    (DEFMACRO name args doc-string dcl ... dcl body)
    (LET ((v1 t1) ...) dcl ... dcl body)
    (MV-LET (v1 ...) term dcl ... dcl body)
  ~ev[]
  Of course, if a form macroexpands into one of these (as ~ilc[let*] expands
  into nested ~ilc[let]s and our ~c[er-let*] expands into nested ~ilc[mv-let]s)
  then declarations are permitted as handled by the macros involved.

  ~c[Declare] is defined in Common Lisp.  See any Common Lisp documentation for
  more information.~/")

(deflabel type-spec
  :doc
  ":Doc-Section ACL2::Programming

  type specifiers in declarations~/
  ~bv[]
  Examples:
  The symbol INTEGER in (declare (type INTEGER i j k)) is a type-spec.  Other
  type-specs supported by ACL2 include RATIONAL, COMPLEX, (INTEGER 0 127),
  (RATIONAL 1 *), CHARACTER, and ATOM.  ~terminal[Type :more for a complete listing.]
  ~ev[]~/

  The type-specs and their meanings (when applied to the variable ~c[x]
  as in ~c[(declare (type type-spec x))] are given below.
  ~bv[]
  type-spec              meaning
  (AND type1 ... typek)  (AND (p1 X) ... (pk X))
                         where (pj x) is the meaning for type-spec typej
  ATOM                   (ATOM X)
  BIT                    (OR (EQUAL X 1) (EQUAL X 0))
  CHARACTER              (CHARACTERP X)
  COMPLEX                (AND (COMPLEX-RATIONALP X)
                              (RATIONALP (REALPART X))
                              (RATIONALP (IMAGPART X)))
  (COMPLEX RATIONAL)     same as COMPLEX, above
  (COMPLEX type)         (AND (COMPLEX-RATIONALP X)
                              (p (REALPART X))
                              (p (IMAGPART X)))
                         where (p x) is the meaning for type-spec type
  CONS                   (CONSP X)
  INTEGER                (INTEGERP X)
  (INTEGER i j)          (AND (INTEGERP X)   ; See notes below
                              (<= i X)
                              (<= X j))
  (MEMBER x1 ... xn)     (MEMBER X '(x1 ... xn))
  (MOD i)                same as (INTEGER 0 i-1)
  NIL                    NIL
  (NOT type)             (NOT (p X))
                         where (p x) is the meaning for type-spec type
  NULL                   (EQ X NIL)
  (OR type1 ... typek)   (OR (p1 X) ... (pk X))
                         where (pj x) is the meaning for type-spec typej
  RATIO                  (AND (RATIONALP X) (NOT (INTEGERP X)))
  RATIONAL               (RATIONALP X)
  (RATIONAL i j)         (AND (RATIONALP X)  ; See notes below
                              (<= i X)
                              (<= X j))
  REAL                   (RATIONALP X)       ; (REALP X) in ACL2(r)
  (REAL i j)             (AND (RATIONALP X)  ; See notes below
                              (<= i X)
                              (<= X j))
  (SATISFIES pred)       (pred X) ; Lisp requires a unary function, not a macro
  SIGNED-BYTE            (INTEGERP X)
  (SIGNED-BYTE i)        same as (INTEGER k m) where k=-2^(i-1), m=2^(i-1)-1
  STANDARD-CHAR          (STANDARD-CHARP X)
  STRING                 (STRINGP X)
  (STRING max)           (AND (STRINGP X) (EQUAL (LENGTH X) max))
  SYMBOL                 (SYMBOLP X)
  T                      T
  UNSIGNED-BYTE          same as (INTEGER 0 *)
  (UNSIGNED-BYTE i)      same as (INTEGER 0 (2^i)-1)
  ~ev[]
  ~em[Notes:]

  In general, ~c[(integer i j)] means
  ~bv[]
       (AND (INTEGERP X)
            (<= i X)
            (<= X j)).
  ~ev[]
  But if ~c[i] is the symbol ~c[*], the first inequality is omitted.  If ~c[j]
  is the symbol ~c[*], the second inequality is omitted.  If instead of
  being an integer, the second element of the type specification is a
  list containing an integer, ~c[(i)], then the first inequality is made
  strict.  An analogous remark holds for the ~c[(j)] case.  The ~c[RATIONAL]
  and ~c[REAL] type specifiers are similarly generalized.~/")

(defun the-error (x y)
  (declare (xargs :guard
                  (hard-error
                   nil
                   "The object ~xa does not satisfy the declaration ~xb."
                   (list (cons #\a y)
                         (cons #\b x)))))
  (declare (ignore x))
  y)

(defun the-fn (x y)
  (declare (xargs :guard (translate-declaration-to-guard x 'var nil)

; As noted above the definition of translate-declaration-to-guard/integer, we
; are trying to save a little space in the image.

                  :mode :program))
  (let ((guard (translate-declaration-to-guard x 'var nil)))

; Observe that we translate the type expression, x, wrt the variable var
; and then bind var to y below.  It is logically equivalent to translate
; wrt to y instead and then generate the if-expression below instead of the
; let.   Why do we do that?  Because y (or var) is liable to occur many times
; in the guard (and one more time in the if) and if y is a huge expression we
; blow ourselves away there.  A good example of this comes up if one
; translates the expression (the-type-set xxx).  When we translated the
; declaration wrt to 'xxx we got an expression in which 'xxx occurred
; five times.  By generating the let below, it occurs only once.

; We have tried an experiment in which we treat the (symbolp y) case
; specially:  translate wrt to y and just lay down the if-expression
; (if guard y (the-error 'x y)).  The system was able to do an :init, so
; this did not blow us out of the water -- as we know it does if you so
; treat all y's.  But this IF-expressions in the guard are therefore
; turned loose in the surrounding term and contribute to the explosion of
; normalized bodies.  So we have backtracked to this, which has the
; advantage of keeping the normalized sizes just linearly bigger.

    (cond ((null guard)
           (illegal nil
                    "Illegal-type."
                    (list (cons #\0 x))))
          (t
           (list 'let (list (list 'var y))

; The following declaration allows a check at translate time that any part
; (satisfies pred) of x is such that pred is a unary function symbol in the
; current world.  An optimization in dcl-guardian guarantees that this
; declaration won't generate any proof obligations.

; WARNING: Do not change the form of this declaration without visiting the
; corresponding code for the-fn in chk-dcl-lst and dcl-guardian.

                 `(declare (type (or t ,x) var))
                 (list 'if
                       guard 'var
                       (list 'the-error (list 'quote x) 'var)))))))

#+acl2-loop-only
(defmacro the (x y)

  ":Doc-Section ACL2::Programming

  run-time type check~/

  ~c[(The typ val)] checks that ~c[val] satisfies the type specification
  ~c[typ] (~pl[type-spec]).  An error is caused if the check fails,
  and otherwise, ~c[val] is the value of this expression.  Here are
  some examples.
  ~bv[]
  (the integer 3)       ; returns 3
  (the (integer 0 6) 3) ; returns 3
  (the (integer 0 6) 7) ; causes an error
  ~ev[]
  ~l[type-spec] for a discussion of the legal type
  specifications.~/

  ~c[The] is defined in Common Lisp.  See any Common Lisp documentation
  for more information.~/"

  (declare (xargs :guard (translate-declaration-to-guard x 'var nil)))
  (the-fn x y))

; THEORY PROTO-PRIMITIVES

; Thus far it has been impossible to use the :in-theory hint in
; defthm and defun -- unless one wants to quote a theory -- because
; there are no primitives for getting all the names in the world.
; We here define the necessary basic functions, just so we can
; conveniently disable.  See the extended discussion of theories
; in "other-events.lisp" where deftheory is defined.

(defun set-difference-equal (l1 l2)

  ":Doc-Section ACL2::Programming

  elements of one list that are not elements of another~/

  ~c[(Set-difference-equal x y)] equals a list whose members
  (~pl[member-equal]) contains the members of ~c[x] that are not
  members of ~c[y].  More precisely, the resulting list is the same as
  one gets by deleting the members of ~c[y] from ~c[x], leaving the
  remaining elements in the same order as they had in ~c[x].~/

  The ~il[guard] for ~c[set-difference-equal] requires both arguments to be
  true lists.  Essentially, ~c[set-difference-equal] has the same
  functionality as the Common Lisp function ~c[set-difference], except
  that it uses the ~ilc[equal] function to test membership rather than
  ~ilc[eql].  However, we do not include the function ~c[set-difference]
  in ACL2, because the Common Lisp language does not specify the order
  of the elements in the list that it returns.

  Also ~pl[set-difference-eq] for a semantically equivalent function that
  executes more efficiently on lists of symbols.~/"

  (declare (xargs :guard (and (true-listp l1)
                              (true-listp l2))))
  (cond ((endp l1) nil)
        ((member-equal (car l1) l2)
         (set-difference-equal (cdr l1) l2))
        (t (cons (car l1) (set-difference-equal (cdr l1) l2)))))

; ARRAYS - efficient applicative arrays.

; We provide functions for accessing and updating both one and two
; dimensional arrays, with applicative semantics, but good access time
; to the most recently updated copy and usually constant update time.

; We first describe the one dimensional array data type.  From the
; formal point of view, an array is simply an alist, i.e. a list of
; pairs.  With one exception, the key (i.e., the car) of each pair is
; a nonnegative integer.  However each array must have (at least) one
; pair whose car is :header and whose cdr is a keyword list, whose
; keys include :dimensions, :maximum-length, and :default.  Thus, for
; example, the list '((1 . 2) (:header :dimensions (3) :maximum-length
; 7 :default a) (0 . 6)) represents the sequence #s(6 2 7).  In the
; case of a one dimensional array, the dimension is a list of length
; one which is a nonnegative integer one greater than the maximum
; permitted index.  (Other keywords, e.g. :purpose, for
; identification, are permitted and ignored.)  Formally speakign, to
; find the value of a non-negative integer key in such an alist, we
; search the alist (with the function aref1) for the first pair whose
; car matches the key.  If such a pair is found, then aref1 returns
; the cdr of the pair; otherwise aref1 returns the value associated
; with the :default key.  It is illegal to give aref1 an an index
; equal to or greater than the car of the value associated with the
; :dimensions key.  In the normal case, updating happens by simply
; consing a new pair on to the alist with the function aset1.
; However, when the list resulting from such a cons has length greater
; than the value associated with the :maximum-length key, the alist is
; ``compressed'' back to an alist of minimal length, but with the same
; aref1 search semantics.

; For efficiency, the user is asked to call the array functions with
; an additional argument, a symbol, called the ``name'' of the given
; array.  From the point of view of the formal semantics, the name
; argument is simply and completely ignored.  However, as with the
; implementation of property lists described above, the name provides
; a hint about where to find a ``real'' Common Lisp array that may
; currently represent the given alist, in which case an array access
; can go quite quickly because the real array may be accessed
; directly.

; A further requirement for fast access is that the user initially
; alert the implementation to the desire to make fast accesses by
; calling the function compress1 on the array (and the desired name).
; compress1 then associates with the alist (under the name) a ``real''
; array.  Compress1 returns a list that begins with the header and has
; its other elements in key-ascending order unless otherwise indicated
; by the hearder, with aref1-irrelevant pairs deleted.  If the alist
; is already in this normal form, then no consing is done.  If there
; is already an array associated with the given name, and if it
; happens to have the desired length, then no array allocation is done
; but instead that array is ``stolen''.

; In the usual case, whenever an array is updated (with aset1), the
; ``real'' array which acts as its shadow and supports efficient
; access, is set to support the ``new'' array, and no longer supports
; the ``old'' array.  Thus one must, for efficiency's sake, be
; extremely conscious of the usual order of Common Lisp evaluation.

; For two dimensional arrays, the value of the key :dimensions should
; be a list of two positive integers and the aset2 and aref2 function
; take two indices.

; The following constant was originally introduced in order to
; "require that array indices fit into 32 bits so that some compilers
; can lay down faster code.  In the case of two dimensional arrays, we
; require that the product of legal indices fit into 32 bits."  In
; fact, we now make stronger requirements based on the
; array-total-size-limit and array-dimension-limit of the underlying
; Common Lisp implementation, as enforced by make-array$, whose
; definition follows shortly after this.

(defconst *maximum-positive-32-bit-integer*
  (1- (expt 2 31)))

#-acl2-loop-only
(defconst *our-array-total-size-limit*

; GCL 2.3.8 has a bug that defines array-total-size-limit to be a symbol,
; 'ARRAY-DIMENSION-LIMIT.  (Presumably the intention was to define
; array-total-size-limit to be the value of that symbol.)  So we define our own
; version of array-total-size-limit.

  (if (eql array-total-size-limit 'ARRAY-DIMENSION-LIMIT)
      array-dimension-limit
    array-total-size-limit))

#-acl2-loop-only
(defun-one-output chk-make-array$ (dimensions form)
  (or (let* ((dimensions
              (if (integerp dimensions) (list dimensions) dimensions)))
        (and (true-listp dimensions)
             (do ((tl dimensions (cdr tl)))
                 ((null tl) t)
                 (let ((dim (car dimensions)))
                   (or (and (integerp dim)
                            (<= 0 dim)
                            (< dim array-dimension-limit))
                       (return nil))))
             (< (let ((prod 1))
                  (do ((tl dimensions (cdr tl)))
                      ((null tl))
                      (setq prod (* prod (car dimensions))))
                  prod)
                *our-array-total-size-limit*)))
      (illegal 'make-array$
               "The dimensions of an array must obey restrictions of ~
                the underlying Common Lisp:  each must be a ~
                non-negative integer less than the value of ~
                array-dimension-limit (here, ~x0) and their product ~
                must be less than the value of array-total-size-limit ~
                (here, ~x1).  The call ~x2, which has dimensions ~x3, ~
                is thus illegal."
               (list (cons #\0
                           array-dimension-limit)
                     (cons #\1
                           array-total-size-limit)
                     (cons #\2 form)
                     (cons #\3 dimensions)))))

#-acl2-loop-only
(defmacro make-array$ (&whole form dimensions &rest args)

; Common Lisp implementations are supposed to have limits on the dimensions of
; arrays: array-dimension-limit is a strict bound on each dimension, and
; array-total-size-limit is a strict bound on the product of the dimensions.
; But, we do not want to rely on the implementation to signal an error in such
; cases (as opposed to returning garbage or corrupting the image), let alone
; provide a useful error message.  So we provide this function for creation of
; arrays.

; In case we find the following information useful later, here is a summary of
; the above constants in various lisps.  The

; Lisp              array-dimension-limit            array-total-size-limit
; ---------------   ---------------------            ----------------------
; CLISP 2.30          16777216 [2^24]                  16777216 [2^24]
; CMUCL 18e          536870911 [2^29-1]               536870911 [2^29-1]
; SBCL 0.0           536870911 [2^29-1]               536870911 [2^29-1]
; GCL 2.5.0         2147483647 [2^31-1]              2147483647 [2^31-1]
; LISPWORKS 4.2.7      8388607 [2^23-1]                 2096896 [2^21-256]
; Allegro CL 6.2      16777216 [2^24]                  16777216 [2^24]
; MCL 4.2             16777216 [2^24]                  16777216 [2^24]
; OpenMCL Version (Beta: Darwin) 0.13.6 (CCL):
;                     16777216 [2^24]                  16777216 [2^24]

; We go through some effort to find violations at compile time, partly for
; efficiency but mostly in order to provide compile-time feedback when there is
; a problem.

  (declare (ignore args))
  (cond ((integerp dimensions)
         (prog2$ (chk-make-array$ dimensions (kwote form))
                 `(make-array ,@(cdr form))))
        ((and (true-listp dimensions) ; (quote dims)
              (equal (length dimensions) 2)
              (eq (car dimensions) 'quote))
         (prog2$ (chk-make-array$ (cadr dimensions) (kwote form))
                 `(make-array ,@(cdr form))))
        (t `(prog2$ (chk-make-array$ ,dimensions ',form)
                    (make-array ,@(cdr form))))))

; For 1 and 2 dimensional arrays, there may be a property,
; 'acl2-array, stored under a symbol name.  If so, this property has
; is a list of length four, (object actual-array to-go-array header),
; where object is an alist; actual-array, is the current ``real''
; array associated with object under name; to-go-array is an array of
; length one whose content is the number of additional conses that may
; be added before compresses is required; and header is the first pair
; beginning with :header in object. (to-go-array is kept as an array
; rather than as a mere integer in order to avoid number boxing.)

(defun bounded-integer-alistp (l n)

; Check that l is a true-list of pairs, (n . x), where each n is
; either :header or a nonnegative integer less than n.

  (declare (xargs :guard t))
  (cond ((atom l) (null l))
        (t (and (consp (car l))
                (let ((key (caar l)))
                  (and (or (eq key :header)
                           (and (integerp key)
                                (integerp n)
                                (>= key 0)
                                (< key n)))
                       (bounded-integer-alistp (cdr l) n)))))))

(defthm bounded-integer-alistp-forward-to-eqlable-alistp
  (implies (bounded-integer-alistp x n)
           (eqlable-alistp x))
  :rule-classes :forward-chaining)

(defun keyword-value-listp (l)

  ":Doc-Section ACL2::Programming

  recognizer for true lists whose even-position elements are keywords~/

  ~c[(keyword-value-listp l)] is true if and only if ~c[l] is a list of
  even length of the form ~c[(k1 a1 k2 a2 ... kn an)], where each ~c[ki]
  is a keyword.~/~/"

  (declare (xargs :guard t))
  (cond ((atom l) (null l))
        (t (and (keywordp (car l))
                (consp (cdr l))
                (keyword-value-listp (cddr l))))))

(defthm keyword-value-listp-forward-to-true-listp
  (implies (keyword-value-listp x)
           (true-listp x))
  :rule-classes :forward-chaining)

(defun assoc-keyword (key l)

  ":Doc-Section ACL2::Programming

  look up key in a ~ilc[keyword-value-listp]~/

  If ~c[l] is a list of even length of the form ~c[(k1 a1 k2 a2 ... kn an)],
  where each ~c[ki] is a keyword, then ~c[(assoc-keyword key l)] is the
  first tail of ~c[l] starting with ~c[key] if key is some ~c[ki], and is
  ~c[nil] otherwise.~/

  The ~il[guard] for ~c[(assoc-keyword key l)] is ~c[(keyword-value-listp l)].~/"

  (declare (xargs :guard (keyword-value-listp l)))
  (cond ((endp l) nil)
        ((eq key (car l)) l)
        (t (assoc-keyword key (cddr l)))))

; The following seems useful, though at this point its use isn't clear.

(defthm keyword-value-listp-assoc-keyword
  (implies (keyword-value-listp l)
           (keyword-value-listp (assoc-keyword key l)))
  :rule-classes ((:forward-chaining
                  :trigger-terms ((assoc-keyword key l)))))

(defthm consp-assoc-eq
  (implies (alistp l)
           (or (consp (assoc-eq name l))
               (equal (assoc-eq name l) nil)))
  :rule-classes :type-prescription)

(defmacro f-get-global (x st)
  #-acl2-loop-only
  (cond ((and (consp x)
              (eq 'quote (car x))
              (symbolp (cadr x))
              (null (cddr x)))

; The cmulisp complier complains about unreachable code every (perhaps) time
; that f-get-global is called in which st is *the-live-state*.  The following
; optimization is included primarily in order to eliminate those warnings;
; the extra efficiency is pretty minor, though a nice side effect.

         (if (eq st '*the-live-state*)
             `(let ()
                (declare (special ,(global-symbol (cadr x))))
                ,(global-symbol (cadr x)))
           (let ((s (gensym)))
             `(let ((,s ,st))
                (declare (special ,(global-symbol (cadr x))))
                (cond ((live-state-p ,s)
                       ,(global-symbol (cadr x)))
                      (t (get-global ,x ,s)))))))
        (t `(get-global ,x ,st)))
  #+acl2-loop-only
  (list 'get-global x st))

#-acl2-loop-only
(defun-one-output slow-array-warning (fn nm)
  (let ((action (f-get-global 'slow-array-action *the-live-state*)))
    (when action
      (format
       *error-output*
       "~%~%**********************************************************~%~
          Slow Array Access!  A call of ~a on an array named~%~
          ~a is being executed slowly.  See :DOC slow-array-warning~%~
          **********************************************************~%~%"
       fn nm)
      (when (not (eq action :warning))
        (format
         *error-output*
         "To avoid the following break and get only the above warning:~%~s~%"
         '(assign slow-array-action :warning))
        (break$)))))

(deflabel arrays
  :doc
  ":Doc-Section Miscellaneous

  an introduction to ACL2 arrays~/

  Below we begin a detailed presentation of ACL2 arrays.  ACL2's single-threaded
  objects (~pl[stobj]) provide a similar functionality that is generally more
  efficient when there are updates (writes), but is also more restrictive.

  ~/

  ~l[arrays-example] for a brief introduction illustrating the use
  of ACL2 arrays.

  ACL2 provides relatively efficient 1- and 2-dimensional arrays.
  Arrays are awkward to provide efficiently in an applicative language
  because the programmer rightly expects to be able to ``modify'' an
  array object with the effect of changing the behavior of the element
  accessing function on that object.  This, of course, does not make
  any sense in an applicative setting.  The element accessing function
  is, after all, a function, and its behavior on a given object is
  immutable.  To ``modify'' an array object in an applicative setting
  we must actually produce a new array object.  Arranging for this to
  be done efficiently is a challenge to the implementors of the
  language.  In addition, the programmer accustomed to the von Neumann
  view of arrays must learn how to use immutable applicative arrays
  efficiently.

  In this note we explain 1-dimensional arrays.  In particular, we
  explain briefly how to create, access, and ``modify'' them, how they
  are implemented, and how to program with them.  2-dimensional arrays
  are dealt with by analogy.

  ~em[The Logical Description of ACL2 Arrays]

  An ACL2 1-dimensional array is an object that associates arbitrary
  objects with certain integers, called ``indices.'' Every array has a
  dimension, ~c[dim], which is a positive integer.  The indices of an
  array are the consecutive integers from ~c[0] through ~c[dim-1].  To obtain
  the object associated with the index ~c[i] in an array ~c[a], one uses
  ~c[(aref1 name a i)].  ~c[Name] is a symbol that is irrelevant to the
  semantics of ~ilc[aref1] but affects the speed with which it computes.  We
  will talk more about array ``names'' later.  To produce a new array
  object that is like ~c[a] but which associates ~c[val] with index ~c[i], one
  uses ~c[(aset1 name a i val)].

  An ACL2 1-dimensional array is actually an alist.  There is no
  special ACL2 function for creating arrays; they are generally built
  with the standard list processing functions ~ilc[list] and ~ilc[cons].  However,
  there is a special ACL2 function, called ~ilc[compress1], for speeding up
  access to the elements of such an alist.  We discuss ~ilc[compress1]
  later.

  One element of the alist must be the ``header'' of the array.  The
  ~il[header] of a 1-dimensional array with dimension ~c[dim] is of the form:
  ~bv[]
  (:HEADER :DIMENSIONS (dim)
           :MAXIMUM-LENGTH max
           :DEFAULT obj ; optional
           :NAME name   ; optional
           :ORDER order ; optional values are < (the default), >, or :none/nil
           ).
  ~ev[]
  ~c[Obj] may be any object and is called the ``default value'' of the array.
  ~ilc[Max] must be an integer greater than ~c[dim].  ~c[Name] must be a
  symbol.  The ~c[:]~ilc[default] and ~c[:name] entries are optional; if
  ~c[:]~ilc[default] is omitted, the default value is ~c[nil].  The function
  ~ilc[header], when given a name and a 1- or 2-dimensional array, returns the
  ~il[header] of the array.  The functions ~ilc[dimensions],
  ~ilc[maximum-length], and ~ilc[default] are similar and return the
  corresponding fields of the ~il[header] of the array.  The role of the
  ~c[:]~ilc[dimensions] field is obvious: it specifies the legal indices into
  the array.  The roles played by the ~c[:]~ilc[maximum-length] and
  ~c[:]~ilc[default] fields are described below.

  Aside from the ~il[header], the other elements of the alist must each be
  of the form ~c[(i . val)], where ~c[i] is an integer and ~c[0 <= i < dim], and
  ~c[val] is an arbitrary object.

  The ~c[:order] field of the header is ignored for 2-dimensional arrays.  For
  1-dimensional arrays, it specifies the order of keys (~c[i], above) when the
  array is compressed as with ~ilc[compress1], as described below.  An
  ~c[:order] of ~c[:none] or ~c[nil] specifies no reordering of the alist by
  ~ilc[compress1], and an order of ~c[>] specifies reordering by
  ~ilc[compress1] so that keys are in descending order.  Otherwise, the alist
  is reordered by ~ilc[compress1] so that keys are in ascending order.

  ~c[(Aref1 name a i)] is ~il[guard]ed so that ~c[name] must be a symbol, ~c[a] must be
  an array and ~c[i] must be an index into ~c[a].  The value of
  ~c[(aref1 name a i)] is either ~c[(cdr (assoc i a))] or else is the
  default value of ~c[a], depending on whether there is a pair in ~c[a]
  whose ~ilc[car] is ~c[i].  Note that ~c[name] is irrelevant to the value of
  an ~ilc[aref1] expression.  You might ~c[:pe aref1] to see how simple
  the definition is.

  ~c[(Aset1 name a i val)] is ~il[guard]ed analogously to the ~ilc[aref1] expression.
  The value of the ~ilc[aset1] expression is essentially
  ~c[(cons (cons i val) a)].  Again, ~c[name] is irrelevant.  Note
  ~c[(aset1 name a i val)] is an array, ~c[a'], with the property that
  ~c[(aref1 name a' i)] is ~c[val] and, except for index ~c[i], all other
  indices into ~c[a'] produce the same value as in ~c[a].  Note also
  that if ~c[a] is viewed as an alist (which it is) the pair
  ``binding'' ~c[i] to its old value is in ~c[a'] but ``covered up'' by
  the new pair.  Thus, the length of an array grows by one when
  ~ilc[aset1] is done.

  Because ~ilc[aset1] covers old values with new ones, an array produced by
  a sequence of ~ilc[aset1] calls may have many irrelevant pairs in it.  The
  function ~ilc[compress1] can remove these irrelevant pairs.  Thus,
  ~c[(compress1 name a)] returns an array that is equivalent
  (vis-a-vis ~ilc[aref1]) to ~c[a] but which may be shorter.  For technical
  reasons, the alist returned by ~ilc[compress1] may also list the pairs
  in a different order than listed in ~c[a].

  To prevent arrays from growing excessively long due to repeated ~ilc[aset1]
  operations, ~ilc[aset1] actually calls ~ilc[compress1] on the new alist
  whenever the length of the new alist exceeds the ~c[:]~ilc[maximum-length]
  entry, ~ilc[max], in the ~il[header] of the array.  See the definition of
  ~ilc[aset1] (for example by using ~c[:]~ilc[pe]).  This is primarily just a
  mechanism for freeing up ~ilc[cons] space consumed while doing ~ilc[aset1]
  operations.  Note however that this ~ilc[compress1] call is replaced by a
  hard error if the header specifies an ~c[:order] of ~c[:none] or ~c[nil].

  This completes the logical description of 1-dimensional arrays.
  2-dimensional arrays are analogous.  The ~c[:]~ilc[dimensions] entry of the
  ~il[header] of a 2-dimensional array should be ~c[(dim1 dim2)].  A pair of
  indices, ~c[i] and ~c[j], is legal iff ~c[0 <= i < dim1] and ~c[0 <= j < dim2].
  The ~c[:]~ilc[maximum-length] must be greater than ~c[dim1*dim2].  ~ilc[Aref2], ~ilc[aset2],
  and ~ilc[compress2] are like their counterparts but take an additional
  ~c[index] argument.  Finally, the pairs in a 2-dimensional array are of
  the form ~c[((i . j) . val)].

  ~em[The Implementation of ACL2 Arrays]

  Very informally speaking, the function ~ilc[compress1] ``creates'' an
  ACL2 array that provides fast access, while the function ~ilc[aref1]
  ``maintains'' fast access.  We now describe this informal idea more
  carefully.

  ~ilc[Aref1] is essentially ~ilc[assoc].  If ~ilc[aref1] were implemented naively the
  time taken to access an array element would be linear in the
  dimension of the array and the number of ``assignments'' to it (the
  number of ~ilc[aset1] calls done to create the array from the initial
  alist).  This is intolerable; arrays are ``supposed'' to provide
  constant-time access and change.

  The apparently irrelevant names associated with ACL2 arrays allow us
  to provide constant-time access and change when arrays are used in
  ``conventional'' ways.  The implementation of arrays makes it clear
  what we mean by ``conventional.''

  Recall that array names are symbols.  Behind the scenes, ACL2
  associates two objects with each ACL2 array name.  The first object
  is called the ``semantic value'' of the name and is an alist.  The
  second object is called the ``raw lisp array'' and is a Common Lisp
  array.

  When ~c[(compress1 name alist)] builds a new alist, ~c[a'], it sets the
  semantic value of ~c[name] to that new alist.  Furthermore, it creates a
  Common Lisp array and writes into it all of the index/value pairs of
  ~c[a'], initializing unassigned indices with the default value.  This
  array becomes the raw lisp array of ~c[name].  ~ilc[Compress1] then returns
  ~c[a'], the semantic value, as its result, as required by the definition
  of ~ilc[compress1].

  When ~c[(aref1 name a i)] is invoked, ~ilc[aref1] first determines whether the
  semantic value of ~c[name] is ~c[a] (i.e., is ~ilc[eq] to the alist ~c[a]).  If so,
  ~ilc[aref1] can determine the ~c[i]th element of ~c[a] by invoking Common Lisp's
  ~c[aref] function on the raw lisp array associated with name.  Note that
  no linear search of the alist ~c[a] is required; the operation is done
  in constant time and involves retrieval of two global variables, an
  ~ilc[eq] test and ~c[jump], and a raw lisp array access.  In fact, an ACL2
  array access of this sort is about 5 times slower than a C array
  access.  On the other hand, if ~c[name] has no semantic value or if it
  is different from ~c[a], then ~ilc[aref1] determines the answer by linear
  search of ~c[a] as suggested by the ~c[assoc-like] definition of ~ilc[aref1].
  Thus, ~ilc[aref1] always returns the axiomatically specified result.  It
  returns in constant time if the array being accessed is the current
  semantic value of the name used.  The ramifications of this are
  discussed after we deal with ~ilc[aset1].

  When ~c[(aset1 name a i val)] is invoked, ~ilc[aset1] does two ~ilc[cons]es to
  create the new array.  Call that array ~c[a'].  It will be returned as
  the answer.  (In this discussion we ignore the case in which ~ilc[aset1]
  does a ~ilc[compress1].)  However, before returning, ~ilc[aset1] determines if
  ~c[name]'s semantic value is ~c[a].  If so, it makes the new semantic value
  of ~c[name] be ~c[a'] and it smashes the raw lisp array of ~c[name] with ~c[val] at
  index ~c[i], before returning ~c[a'] as the result.  Thus, after doing an
  ~ilc[aset1] and obtaining a new semantic value ~c[a'], all ~ilc[aref1]s on that new
  array will be fast.  Any ~ilc[aref1]s on the old semantic value, ~c[a], will
  be slow.

  To understand the performance implications of this design, consider
  the chronological sequence in which ACL2 (Common Lisp) evaluates
  expressions:  basically inner-most first, left-to-right,
  call-by-value.  An array use, such as ~c[(aref1 name a i)], is ``fast''
  (constant-time) if the alist supplied, ~c[a], is the value returned by
  the most recently executed ~ilc[compress1] or ~ilc[aset1] on the name supplied.
  In the functional expression of ``conventional'' array processing,
  all uses of an array are fast.

  The ~c[:name] field of the ~il[header] of an array is completely irrelevant.
  Our convention is to store in that field the symbol we mean to use
  as the name of the raw lisp array.  But no ACL2 function inspects
  ~c[:name] and its primary value is that it allows the user, by
  inspecting the semantic value of the array ~-[] the alist ~-[] to recall
  the name of the raw array that probably holds that value.  We say
  ``probably'' since there is no enforcement that the alist was
  compressed under the name in the ~il[header] or that all ~c[aset]s used that
  name.  Such enforcement would be inefficient.

  ~em[Some Programming Examples]

  In the following examples we will use ACL2 ``global variables'' to
  hold several arrays.  ~l[@], and ~pl[assign].

  Let the ~ilc[state] global variable ~c[a] be the 1-dimensional compressed
  array of dimension ~c[5] constructed below.
  ~bv[]
  ACL2 !>(assign a (compress1 'demo
                              '((:header :dimensions (5)
                                         :maximum-length 15
                                         :default uninitialized
                                         :name demo)
                                (0 . zero))))
  ~ev[]
  Then ~c[(aref1 'demo (@ a) 0)] is ~c[zero] and ~c[(aref1 'demo (@ a) 1)] is
  ~c[uninitialized].

  Now execute
  ~bv[]
  ACL2 !>(assign b (aset1 'demo (@ a) 1 'one))
  ~ev[]
  Then ~c[(aref1 'demo (@ b) 0)] is ~c[zero] and ~c[(aref1 'demo (@ b) 1)] is
  ~c[one].

  All of the ~ilc[aref1]s done so far have been ``fast.''

  Note that we now have two array objects, one in the global variable
  ~c[a] and one in the global variable ~c[b].  ~c[B] was obtained by assigning to
  ~c[a].  That assignment does not affect the alist ~c[a] because this is an
  applicative language.  Thus, ~c[(aref1 'demo (@ a) 1)] must ~st[still] be
  ~c[uninitialized].  And if you execute that expression in ACL2 you will
  see that indeed it is.  However, a rather ugly comment is printed,
  namely that this array access is ``slow.''  The reason it is slow is
  that the raw lisp array associated with the name ~c[demo] is the array
  we are calling ~c[b].  To access the elements of ~c[a], ~ilc[aref1] must now do a
  linear search.  Any reference to ~c[a] as an array is now
  ``unconventional;'' in a conventional language like Ada or Common
  Lisp it would simply be impossible to refer to the value of the
  array before the assignment that produced our ~c[b].

  Now let us define a function that counts how many times a given
  object, ~c[x], occurs in an array.  For simplicity, we will pass in the
  name and highest index of the array:
  ~bv[]
  ACL2 !>(defun cnt (name a i x)
           (declare (xargs :guard
                           (and (array1p name a)
                                (integerp i)
                                (>= i -1)
                                (< i (car (dimensions name a))))
                           :mode :logic
                           :measure (nfix (+ 1 i))))
           (cond ((zp (1+ i)) 0) ; return 0 if i is at most -1
                 ((equal x (aref1 name a i))
                  (1+ (cnt name a (1- i) x)))
                 (t (cnt name a (1- i) x))))
  ~ev[]
  To determine how many times ~c[zero] appears in ~c[(@ b)] we can execute:
  ~bv[]
  ACL2 !>(cnt 'demo (@ b) 4 'zero)
  ~ev[]
  The answer is ~c[1].  How many times does ~c[uninitialized] appear in
  ~c[(@ b)]?
  ~bv[]
  ACL2 !>(cnt 'demo (@ b) 4 'uninitialized)
  ~ev[]
  The answer is ~c[3], because positions ~c[2], ~c[3] and ~c[4] of the array contain
  that default value.

  Now imagine that we want to assign ~c['two] to index ~c[2] and then count
  how many times the 2nd element of the array occurs in the array.
  This specification is actually ambiguous.  In assigning to ~c[b] we
  produce a new array, which we might call ~c[c].  Do we mean to count the
  occurrences in ~c[c] of the 2nd element of ~c[b] or the 2nd element of ~c[c]?
  That is, do we count the occurrences of ~c[uninitialized] or the
  occurrences of ~c[two]?  If we mean the former the correct answer is ~c[2]
  (positions ~c[3] and ~c[4] are ~c[uninitialized] in ~c[c]); if we mean the latter,
  the correct answer is ~c[1] (there is only one occurrence of ~c[two] in ~c[c]).

  Below are ACL2 renderings of the two meanings, which we call
  ~c[[former~]] and ~c[[latter~]].  (Warning:  Our description of these
  examples, and of an example ~c[[fast former~]] that follows, assumes
  that only one of these three examples is actually executed; for
  example, they are not executed in sequence.  See ``A Word of
  Warning'' below for more about this issue.)
  ~bv[]
  (cnt 'demo (aset1 'demo (@ b) 2 'two) 4 (aref1 'demo (@ b) 2))  ; [former]

  (let ((c (aset1 'demo (@ b) 2 'two)))                           ; [latter]
    (cnt 'demo c 4 (aref1 'demo c 2)))
  ~ev[]
  Note that in ~c[[former~]] we create ~c[c] in the second argument of the
  call to ~c[cnt] (although we do not give it a name) and then refer to ~c[b]
  in the fourth argument.  This is unconventional because the second
  reference to ~c[b] in ~c[[former~]] is no longer the semantic value of ~c[demo].
  While ACL2 computes the correct answer, namely ~c[2], the execution of
  the ~ilc[aref1] expression in ~c[[former~]] is done slowly.

  A conventional rendering with the same meaning is
  ~bv[]
  (let ((x (aref1 'demo (@ b) 2)))                           ; [fast former]
    (cnt 'demo (aset1 'demo (@ b) 2 'two) 4 x))
  ~ev[]
  which fetches the 2nd element of ~c[b] before creating ~c[c] by
  assignment.  It is important to understand that ~c[[former~]] and
  ~c[[fast former~]] mean exactly the same thing: both count the number
  of occurrences of ~c[uninitialized] in ~c[c].  Both are legal ACL2 and
  both compute the same answer, ~c[2].  Indeed, we can symbolically
  transform ~c[[fast former~]] into ~c[[former~]] merely by substituting
  the binding of ~c[x] for ~c[x] in the body of the ~ilc[let].  But ~c[[fast former~]]
  can be evaluated faster than ~c[[former~]] because all of the
  references to ~c[demo] use the then-current semantic value of
  ~c[demo], which is ~c[b] in the first line and ~c[c] throughout the
  execution of the ~c[cnt] in the second line.  ~c[[Fast former~]] is
  the preferred form, both because of its execution speed and its
  clarity.  If you were writing in a conventional language you would
  have to write something like ~c[[fast former~]] because there is no
  way to refer to the 2nd element of the old value of ~c[b] after
  smashing ~c[b] unless it had been saved first.

  We turn now to ~c[[latter~]].  It is both clear and efficient.  It
  creates ~c[c] by assignment to ~c[b] and then it fetches the 2nd element of
  ~c[c], ~c[two], and proceeds to count the number of occurrences in ~c[c].  The
  answer is ~c[1].  ~c[[Latter~]] is a good example of typical ACL2 array
  manipulation: after the assignment to ~c[b] that creates ~c[c], ~c[c] is used
  throughout.

  It takes a while to get used to this because most of us have grown
  accustomed to the peculiar semantics of arrays in conventional
  languages.  For example, in raw lisp we might have written something
  like the following, treating ~c[b] as a ``global variable'':
  ~bv[]
  (cnt 'demo (aset 'demo b 2 'two) 4 (aref 'demo b 2))
  ~ev[]
  which sort of resembles ~c[[former~]] but actually has the semantics of
  ~c[[latter~]] because the ~c[b] from which ~c[aref] fetches the 2nd element is
  not the same ~c[b] used in the ~c[aset]!  The array ~c[b] is destroyed by the
  ~c[aset] and ~c[b] henceforth refers to the array produced by the ~c[aset], as
  written more clearly in ~c[[latter~]].

  A Word of Warning:  Users must exercise care when experimenting with
  ~c[[former~]], ~c[[latter~]] and ~c[[fast former~]].  Suppose you have
  just created ~c[b] with the assignment shown above,
  ~bv[]
  ACL2 !>(assign b (aset1 'demo (@ a) 1 'one))
  ~ev[]
  If you then evaluate ~c[[former~]] in ACL2 it will complain that the
  ~ilc[aref1] is slow and compute the answer, as discussed.  Then suppose
  you evaluate ~c[[latter~]] in ACL2.  From our discussion you might expect
  it to execute fast ~-[] i.e., issue no complaint.  But in fact you
  will find that it complains repeatedly.  The problem is that the
  evaluation of ~c[[former~]] changed the semantic value of ~c[demo] so that it
  is no longer ~c[b].  To try the experiment correctly you must make ~c[b] be
  the semantic value of ~c[demo] again before the next example is
  evaluated.  One way to do that is to execute
  ~bv[]
  ACL2 !>(assign b (compress1 'demo (@ b)))
  ~ev[]
  before each expression.  Because of issues like this it is often
  hard to experiment with ACL2 arrays at the top-level.  We find it
  easier to write functions that use arrays correctly and efficiently
  than to so use them interactively.

  This last assignment also illustrates a very common use of
  ~ilc[compress1].  While it was introduced as a means of removing
  irrelevant pairs from an array built up by repeated assignments, it
  is actually most useful as a way of insuring fast access to the
  elements of an array.

  Many array processing tasks can be divided into two parts.  During
  the first part the array is built.  During the second part the array
  is used extensively but not modified.  If your ~il[programming] task can
  be so divided, it might be appropriate to construct the array
  entirely with list processing, thereby saving the cost of
  maintaining the semantic value of the name while few references are
  being made.  Once the alist has stabilized, it might be worthwhile
  to treat it as an array by calling ~ilc[compress1], thereby gaining
  constant time access to it.

  ACL2's theorem prover uses this technique in connection with its
  implementation of the notion of whether a ~il[rune] is ~il[disable]d or not.
  Associated with every ~il[rune] is a unique integer ~c[index], called its
  ``nume.''  When each rule is stored, the corresponding nume is
  stored as a component of the rule.  ~il[Theories] are lists of ~il[rune]s and
  membership in the ``current theory'' indicates that the
  corresponding rule is ~il[enable]d.  But these lists are very long and
  membership is a linear-time operation.  So just before a proof
  begins we map the list of ~il[rune]s in the current theory into an alist
  that pairs the corresponding numes with ~c[t].  Then we compress this
  alist into an array.  Thus, given a rule we can obtain its nume
  (because it is a component) and then determine in constant time
  whether it is ~il[enable]d.  The array is never modified during the
  proof, i.e., ~ilc[aset1] is never used in this example.  From the logical
  perspective this code looks quite odd:  we have replaced a
  linear-time membership test with an apparently linear-time ~ilc[assoc]
  after going to the trouble of mapping from a list of ~il[rune]s to an
  alist of numes.  But because the alist of numes is an array, the
  ``apparently linear-time ~ilc[assoc]'' is more apparent than real; the
  operation is constant-time.~/

  :cited-by Programming")

(deflabel arrays-example
  :doc

; The transcript below was generated after executing the following two forms:
; (assign fmt-soft-right-margin 55)
; (assign fmt-hard-right-margin 68)

  ":Doc-Section Arrays

  an example illustrating ACL2 arrays~/

  The example below illustrates the use of ACL2 arrays.  It is not, of
  course, a substitute for the detailed explanations provided
  elsewhere (~pl[arrays], including subtopics).~/

  ~bv[]
  ACL2 !>(defun defarray (name size initial-element)
           (compress1 name
                      (cons (list :HEADER
                                  :DIMENSIONS (list size)
                                  :MAXIMUM-LENGTH (1+ size)
                                  :DEFAULT initial-element
                                  :NAME name)
                            nil)))

  Since DEFARRAY is non-recursive, its admission is trivial.  We observe
  that the type of DEFARRAY is described by the theorem
  (AND (CONSP (DEFARRAY NAME SIZE INITIAL-ELEMENT))
       (TRUE-LISTP (DEFARRAY NAME SIZE INITIAL-ELEMENT))).
  We used the :type-prescription rule COMPRESS1.

  Summary
  Form:  ( DEFUN DEFARRAY ...)
  Rules: ((:TYPE-PRESCRIPTION COMPRESS1))
  Warnings:  None
  Time:  0.02 seconds (prove: 0.00, print: 0.02, other: 0.00)
   DEFARRAY
  ACL2 !>(assign my-ar (defarray 'a1 5 17))
   ((:HEADER :DIMENSIONS (5)
             :MAXIMUM-LENGTH 6 :DEFAULT 17 :NAME A1))
  ACL2 !>(aref1 'a1 (@ my-ar) 3)
  17
  ACL2 !>(aref1 'a1 (@ my-ar) 8)


  ACL2 Error in TOP-LEVEL:  The guard for the function symbol AREF1,
  which is
  (AND (ARRAY1P NAME L) (INTEGERP N) (>= N 0) (< N (CAR (DIMENSIONS NAME L)))),
  is violated by the arguments in the call (AREF1 'A1 '(#) 8).

  ACL2 !>(assign my-ar (aset1 'a1 (@ my-ar) 3 'xxx))
   ((3 . XXX)
    (:HEADER :DIMENSIONS (5)
             :MAXIMUM-LENGTH 6 :DEFAULT 17 :NAME A1))
  ACL2 !>(aref1 'a1 (@ my-ar) 3)
  XXX
  ACL2 !>(aset1 'a1 (@ my-ar) 3 'yyy) ; BAD: (@ my-ar) now points to
                                      ;      an old copy of the array!
  ((3 . YYY)
   (3 . XXX)
   (:HEADER :DIMENSIONS (5)
            :MAXIMUM-LENGTH 6 :DEFAULT 17 :NAME A1))
  ACL2 !>(aref1 'a1 (@ my-ar) 3) ; Because of \"BAD\" above, the array
                                 ; access is done using assoc rather
                                 ; than Lisp aref, hence is slower;
                                 ; but the answer is still correct,
                                 ; reflecting the value in (@ my-ar),
                                 ; which was not changed above.


  **********************************************************
  Slow Array Access!  A call of AREF1 on an array named
  A1 is being executed slowly.  See :DOC slow-array-warning
  **********************************************************

  XXX
  ACL2 !>
  ~ev[]")

(deflabel slow-array-warning
  :doc
  ":Doc-Section Arrays

  a warning or error issued when ~il[arrays] are used inefficiently~/

  If you use ACL2 ~il[arrays] you may sometimes see a ~st[slow array] warning.
  We explain below what that warning means and some likely ``mistakes''
  it may signify.

  First, we note that you can control whether or not you get a warning and, if
  so, whether or not a break (error from which you can continue; ~pl[break$])
  ensues:
  ~bv[]
  (assign slow-array-action :warning) ; warn on slow array access (default)
  (assign slow-array-action :break)   ; warn as above, and then call break$
  (assign slow-array-action nil) ; do not warn or break on slow array access
  ~ev[]
  If you are using ACL2 arrays, then you probably care about performance, in
  which case is is probably best to avoid the ~c[nil] setting.  Below we assume
  the default behavior: a warning, but no break.~/

  The discussion in the documentation for ~il[arrays] defines what we
  mean by the semantic value of a name.  As noted there, behind the
  scenes ACL2 maintains the invariant that with some names there is
  associated a pair consisting of an ACL2 array ~c[alist], called the
  semantic value of the name, and an equivalent raw lisp array.
  Access to ACL2 array elements, as in ~c[(aref1 name alist i)], is
  executed in constant time when the array alist is the semantic value
  of the name, because we can just use the corresponding raw lisp
  array to obtain the answer.  ~ilc[Aset1] and ~ilc[compress1] modify the raw lisp
  array appropriately to maintain the invariant.

  If ~ilc[aref1] is called on a name and alist, and the alist is not the
  then-current semantic value of the name, the correct result is
  computed but it requires linear time because the alist must be
  searched.  When this happens, ~ilc[aref1] prints a ~st[slow array] warning
  message to the comment window.  ~ilc[Aset1] behaves similarly because the
  array it returns will cause the ~st[slow array] warning every time it is
  used.

  From the purely logical perspective there is nothing ``wrong'' about
  such use of ~il[arrays] and it may be spurious to print a warning
  message.  But because ~il[arrays] are generally used to achieve
  efficiency, the ~st[slow array] warning often means the user's
  intentions are not being realized.  Sometimes merely performance
  expectations are not met; but the message may mean that the
  functional behavior of the program is different than intended.

  Here are some ``mistakes'' that might cause this behavior.  In the
  following we suppose the message was printed by ~ilc[aset1] about an array
  named ~c[name].  Suppose the alist supplied ~ilc[aset1] is ~c[alist].

  (1) ~ilc[Compress1] was never called on ~c[name] and ~c[alist].  That is, perhaps
  you created an alist that is an ~ilc[array1p] and then proceeded to access
  it with ~ilc[aref1] but never gave ACL2 the chance to create a raw lisp
  array for it.  After creating an alist that is intended for use as
  an array, you must do ~c[(compress1 name alist)] and pass the resulting
  ~c[alist'] as the array.

  (2) ~c[Name] is misspelled.  Perhaps the array was compressed under the
  name ~c['delta-1] but accessed under ~c['delta1]?

  (3) An ~ilc[aset1] was done to modify ~c[alist], producing a new array,
  ~c[alist'], but you subsequently used ~c[alist] as an array.  Inspect all
  ~c[(aset1 name ...)] occurrences and make sure that the alist modified
  is never used subsequently (either in that function or any other).
  It is good practice to adopt the following syntactic style.  Suppose
  the alist you are manipulating is the value of the local variable
  ~c[alist].  Suppose at some point in a function definition you wish to
  modify ~c[alist] with ~ilc[aset1].  Then write
  ~bv[]
  (let ((alist (aset1 name alist i val))) ...)
  ~ev[]
  and make sure that the subsequent function body is entirely within
  the scope of the ~ilc[let].  Any uses of ~c[alist] subsequently will refer
  to the new alist and it is impossible to refer to the old alist.
  Note that if you write
  ~bv[]
   (foo (let ((alist (aset1 name alist i val))) ...)  ; arg 1
        (bar alist))                                  ; arg 2
  ~ev[]
  you have broken the rules, because in ~c[arg 1] you have modified
  ~c[alist] but in ~c[arg 2] you refer to the old value.  An appropriate
  rewriting is to lift the ~ilc[let] out:
  ~bv[]
   (let ((alist (aset1 name alist alist i val)))
     (foo ...                                         ; arg 1
          (bar alist)))                               ; arg 2
  ~ev[]
  Of course, this may not mean the same thing.

  (4) A function which takes ~c[alist] as an argument and modifies it with
  ~ilc[aset1] fails to return the modified version.  This is really the same
  as (3) above, but focuses on function interfaces.  If a function
  takes an array ~c[alist] as an argument and the function uses ~ilc[aset1] (or
  a subfunction uses ~ilc[aset1], etc.), then the function probably
  ``ought'' to return the result produced by ~ilc[aset1].  The reasoning
  is as follows.  If the array is passed into the function, then the
  caller is holding the array.  After the function modifies it, the
  caller's version of the array is obsolete.  If the caller is going
  to make further use of the array, it must obtain the latest version,
  i.e., that produced by the function.")

(defun array1p (name l)

  ":Doc-Section Arrays

  recognize a 1-dimensional array~/
  ~bv[]
  Example Form:
  (array1p 'delta1 a)~/

  General Form:
  (array1p name alist)
  ~ev[]
  where ~c[name] and ~c[alist] are arbitrary objects.  This function
  returns ~c[t] if ~c[alist] is a 1-dimensional ACL2 array.  Otherwise it
  returns ~c[nil].  The function operates in constant time if ~c[alist] is the
  semantic value of ~c[name].  ~l[arrays]."

  (declare (xargs :guard t))
  #-acl2-loop-only
  (cond ((symbolp name)
         (let ((prop (get name 'acl2-array)))
           (cond ((and prop (eq l (car prop)))
                  (return-from array1p (= 1 (array-rank (cadr prop)))))))))

; Note: This function does not use the header, dimensions, and maximum-length
; functions, but obtains their results through duplication of code.  The reason
; is that we want those functions to have array1p or array2p as guards, so they
; can't be introduced before array1p.  The reason we want this function in
; their guards, even though it is overly strong, is as follows.  Users who use
; aref1 guard their functions with arrayp1 and then start proving theorems.
; The theorems talk about dimensions, etc.  If dimensions, etc., are guarded
; with weaker things (like keyword-value-listp) then you find yourself either
; having to open up array1p or forward chain from it.  But array1p is fairly
; hideous.  So we intend to keep it disabled and regard it as the atomic test
; that it is ok to use array processing functions.

  (and (symbolp name)
       (alistp l)
       (let ((header-keyword-list (cdr (assoc-eq :header l))))
         (and (keyword-value-listp header-keyword-list)
              (let ((dimensions (cadr (assoc-keyword :dimensions header-keyword-list)))
                    (maximum-length (cadr (assoc-keyword :maximum-length header-keyword-list))))
                (and (true-listp dimensions)
                     (equal (length dimensions) 1)
                     (integerp (car dimensions))
                     (integerp maximum-length)
                     (< 0 (car dimensions))
                     (< (car dimensions) maximum-length)
                     (<= maximum-length *maximum-positive-32-bit-integer*)
                     (bounded-integer-alistp l (car dimensions))))))))

(defthm array1p-forward
  (implies (array1p name l)
           (and (symbolp name)
                (alistp l)
                (keyword-value-listp (cdr (assoc-eq :header l)))
                (true-listp (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
                (equal (length (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
                       1)
                (integerp (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                (integerp (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l)))))
                (< 0 (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                (< (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
                   (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l)))))
                (<= (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l))))
                    *maximum-positive-32-bit-integer*)
                (bounded-integer-alistp
                 l
                 (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))))
  :rule-classes :forward-chaining)

(defthm array1p-linear
  (implies (array1p name l)
           (and (< 0 (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                (< (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
                   (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l)))))
                (<= (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l))))
                    *maximum-positive-32-bit-integer*)))
  :rule-classes ((:linear :match-free :all)))

(defun bounded-integer-alistp2 (l i j)
  (declare (xargs :guard t))
  (cond ((atom l) (null l))
        (t (and (consp (car l))
                (let ((key (caar l)))
                  (and (or (eq key :header)
                           (and (consp key)
                                (let ((i1 (car key))
                                      (j1 (cdr key)))
                                  (and (integerp i1)
                                       (integerp j1)
                                       (integerp i)
                                       (integerp j)
                                       (>= i1 0)
                                       (< i1 i)
                                       (>= j1 0)
                                       (< j1 j)))))))
                (bounded-integer-alistp2 (cdr l) i j)))))

(defun assoc2 (i j l)
  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (if (atom l)
      nil
    (if (and (consp (car l))
             (consp (caar l))
             (eql i (caaar l))
             (eql j (cdaar l)))
        (car l)
      (assoc2 i j (cdr l)))))

(defun array2p (name l)

  ":Doc-Section Arrays

  recognize a 2-dimensional array~/
  ~bv[]
  Example Form:
  (array2p 'delta1 a)~/

  General Form:
  (array2p name alist)
  ~ev[]
  where ~c[name] and ~c[alist] are arbitrary objects.  This function returns ~c[t] if
  ~c[alist] is a 2-dimensional ACL2 array.  Otherwise it returns ~c[nil].  The function
  operates in constant time if ~c[alist] is the semantic value of ~c[name].  ~l[arrays]."

  (declare (xargs :guard t))
  #-acl2-loop-only
  (cond ((symbolp name)
         (let ((prop (get name 'acl2-array)))
           (cond ((and prop (eq l (car prop))
                       (return-from array2p
                                    (= 2 (array-rank (cadr prop))))))))))
  (and (symbolp name)
       (alistp l)
       (let ((header-keyword-list (cdr (assoc-eq :header l))))
         (and (keyword-value-listp header-keyword-list)
              (let ((dimensions (cadr (assoc-keyword :dimensions header-keyword-list)))
                    (maximum-length (cadr (assoc-keyword :maximum-length header-keyword-list))))
                (and (true-listp dimensions)
                     (equal (length dimensions) 2)
                     (let ((d1 (car dimensions))
                           (d2 (cadr dimensions)))
                       (and (integerp d1)
                            (integerp d2)
                            (integerp maximum-length)
                            (< 0 d1)
                            (< 0 d2)
                            (< (* d1 d2) maximum-length)
                            (<= maximum-length
                                *maximum-positive-32-bit-integer*)
                            (bounded-integer-alistp2 l d1 d2)))))))))

(defthm array2p-forward
  (implies (array2p name l)
           (and (symbolp name)
                (alistp l)
                (keyword-value-listp (cdr (assoc-eq :header l)))
                (true-listp (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
                (equal (length (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))) 2)
                (integerp (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                (integerp (cadr (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                (integerp (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l)))))
                (< 0 (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                (< 0 (cadr (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                (< (* (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
                      (cadr (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                   (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l)))))
                (<= (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l))))
                    *maximum-positive-32-bit-integer*)
                (bounded-integer-alistp2
                 l
                 (car (cadr (assoc-keyword
                             :dimensions
                             (cdr (assoc-eq :header l)))))
                 (cadr (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))))
  :rule-classes :forward-chaining)

(defthm array2p-linear
  (implies (array2p name l)
           (and (< 0 (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                (< 0 (cadr (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                (< (* (car (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l)))))
                      (cadr (cadr (assoc-keyword :dimensions (cdr (assoc-eq :header l))))))
                   (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l)))))
                (<= (cadr (assoc-keyword :maximum-length (cdr (assoc-eq :header l))))
                    *maximum-positive-32-bit-integer*)))
  :rule-classes ((:linear :match-free :all)))

#|
(in-theory (disable array1p array2p))
|#

(defun header (name l)
  (declare (xargs :guard (or (array1p name l) (array2p name l))))

  ":Doc-Section Arrays

  return the header of a 1- or 2-dimensional array~/
  ~bv[]
  Example Form:
  (header 'delta1 a)~/

  General Form:
  (header name alist)
  ~ev[]
  where ~c[name] is arbitrary and ~c[alist] is a 1- or 2-dimensional array.
  This function returns the header of the array ~c[alist].  The function
  operates in virtually constant time if ~c[alist] is the semantic value
  of ~c[name].  ~l[arrays]."

  #+acl2-loop-only
  (prog2$ name ;to avoid warning in *1* function definition
          (assoc-eq :header l))

; In the usual case, this function will take constant time regardless
; of where the header is in the alist.  This makes the related
; functions for getting the fields of the header fast, too.

  #-acl2-loop-only
  (let ((prop (get name 'acl2-array)))
    (cond ((and prop (eq l (car prop)))
           (cadddr prop))
          (t (assoc-eq :header l)))))

(defun dimensions (name l)

  ":Doc-Section Arrays

  return the ~c[:dimensions] from the ~il[header] of a 1- or 2-dimensional array~/
  ~bv[]
  Example Form:
  (dimensions 'delta1 a)~/

  General Form:
  (dimensions name alist)
  ~ev[]
  where ~c[name] is arbitrary and ~c[alist] is a 1- or 2-dimensional array.
  This function returns the dimensions list of the array ~c[alist].  That
  list will either be of the form ~c[(dim1)] or ~c[(dim1 dim2)], depending on
  whether ~c[alist] is a 1- or 2-dimensional array.  ~c[Dim1] and ~c[dim2] will be
  integers and each exceed by 1 the maximum legal corresponding index.
  Thus, if ~c[dimensions] returns, say, ~c['(100)] for an array ~c[a]
  named ~c['delta1], then ~c[(aref1 'delta1 a 99)] is legal but
  ~c[(aref1 'delta1 a 100)] violates the ~il[guard]s on ~ilc[aref1].
  ~c[Dimensions] operates in virtually constant time if ~c[alist] is the
  semantic value of ~c[name].  ~l[arrays]."

  (declare (xargs :guard (or (array1p name l) (array2p name l))))
  (cadr (assoc-keyword :dimensions
                       (cdr (header name l)))))

(defun maximum-length (name l)

  ":Doc-Section Arrays

  return the ~c[:maximum-length] from the ~il[header] of an array~/
  ~bv[]
  Example Form:
  (maximum-length 'delta1 a)~/

  General Form:
  (maximum-length name alist)
  ~ev[]
  where ~c[name] is an arbitrary object and ~c[alist] is a 1- or
  2-dimensional array.  This function returns the contents of the
  ~c[:maximum-length] field of the ~il[header] of ~c[alist].  Whenever an ~ilc[aset1] or
  ~ilc[aset2] would cause the length of the alist to exceed its maximum
  length, a ~ilc[compress1] or ~ilc[compress2] is done automatically to remove
  irrelevant pairs from the array.  ~c[Maximum-length] operates in
  virtually constant time if ~c[alist] is the semantic value of ~c[name].
  ~l[arrays]."

  (declare (xargs :guard (or (array1p name l) (array2p name l))))
  (cadr (assoc-keyword :maximum-length (cdr (header name l)))))

(defun default (name l)

  ":Doc-Section Arrays

  return the ~c[:default] from the ~il[header] of a 1- or 2-dimensional array~/
  ~bv[]
  Example Form:
  (default 'delta1 a)~/

  General Form:
  (default name alist)
  ~ev[]
  where ~c[name] is an arbitrary object and ~c[alist] is a 1- or
  2-dimensional array.  This function returns the contents of the
  ~c[:default] field of the ~il[header] of ~c[alist].  When ~ilc[aref1] or ~ilc[aref2] is used
  to obtain a value for an index (or index pair) not bound in ~c[alist],
  the default value is returned instead.  Thus, the array ~c[alist] may be
  thought of as having been initialized with the default value.
  ~c[default] operates in virtually constant time if ~c[alist] is the semantic
  value of ~c[name].  ~l[arrays]."

  (declare (xargs :guard (or (array1p name l) (array2p name l))))
  (cadr (assoc-keyword :default
                       (cdr (header name l)))))

(defthm consp-assoc
  (implies (alistp l)
           (or (consp (assoc name l))
               (equal (assoc name l) nil)))
  :rule-classes :type-prescription)

(defun aref1 (name l n)

  ":Doc-Section Arrays

  access the elements of a 1-dimensional array~/
  ~bv[]
  Example Form:
  (aref1 'delta1 a (+ i k))~/

  General Form:
  (aref1 name alist index)
  ~ev[]
  where ~c[name] is a symbol, ~c[alist] is a 1-dimensional array and ~c[index]
  is a legal index into ~c[alist].  This function returns the value
  associated with ~c[index] in ~c[alist], or else the default value of the
  array.  ~l[arrays] for details.

  This function executes in virtually constant time if ~c[alist] is in
  fact the ``semantic value'' associated with ~c[name] (~pl[arrays]).
  When it is not, ~c[aref1] must do a linear search through ~c[alist].  In
  that case the correct answer is returned but a ~st[slow array] comment is
  printed to the comment window.  ~l[slow-array-warning]."

  #+acl2-loop-only
  (declare (xargs :guard (and (array1p name l)
                              (integerp n)
                              (>= n 0)
                              (< n (car (dimensions name l))))))
  #+acl2-loop-only
  (let ((x (and (not (eq n :header)) (assoc n l))))
    (cond ((null x) (default name l))
          (t (cdr x))))

; We are entitled to make the following declaration because of the
; guard.

  #-acl2-loop-only
  (declare (type (unsigned-byte 31) n))
  #-acl2-loop-only
  (let ((prop (get name 'acl2-array)))
    (cond ((eq l (car prop))
           (svref (the simple-vector (car (cdr prop)))
                  n))
          (t (slow-array-warning 'aref1 name)
             (let ((x (assoc n l))) ; n is a number, hence not :header
               (cond ((null x) (default name l))
                     (t (cdr x))))))))

(defun compress11 (name l i n default)
  (declare (xargs :guard (and (array1p name l)
                              (integerp i)
                              (integerp n)
                              (<= i n))
                  :measure (nfix (- n i))))
  (cond ((zp (- n i)) nil)
        (t (let ((pair (assoc i l)))
             (cond ((or (null pair)
                        (equal (cdr pair) default))
                    (compress11 name l (+ i 1) n default))
                   (t (cons pair
                            (compress11 name l (+ i 1) n default))))))))

#-acl2-loop-only
(defconstant *invisible-array-mark* 'acl2_invisible::|An Invisible Array Mark|)

(defun array-order (header)
  (declare (xargs :guard (and (consp header)
                              (keyword-value-listp (cdr header)))))
  (let ((orderp (assoc-keyword :order (cdr header))))
    (cond
     ((and orderp (or (eq (cadr orderp) nil)
                      (eq (cadr orderp) :none)))
      nil)
     ((and orderp (eq (cadr orderp) '>))
      '>)
     (t ; default
      '<))))

(defun compress1 (name l)

  ":Doc-Section Arrays

  remove irrelevant pairs from a 1-dimensional array~/
  ~bv[]
  Example Form:
  (compress1 'delta1 a)~/

  General Form:
  (compress1 name alist)
  ~ev[]
  where ~c[name] is a symbol and ~c[alist] is a 1-dimensional array, generally
  named ~c[name].  ~l[arrays] for details.  Logically speaking, this function
  removes irrelevant pairs from ~c[alist], possibly shortening it.  The
  function returns a new array, ~c[alist'], with the same ~ilc[header]
  (including name and dimension) as ~c[alist], that, under ~ilc[aref1], is
  everywhere equal to ~c[alist].  That is, ~c[(aref1 name alist' i)] is
  ~c[(aref1 name alist i)], for all legal indices ~c[i].  ~c[Alist'] may be
  shorter than ~c[alist] and the non-irrelevant pairs may occur in a different
  order than in ~c[alist].

  Practically speaking, this function plays an important role in the efficient
  implementation of ~ilc[aref1].  In addition to creating the new array,
  ~c[alist'], ~c[compress1] makes that array the ``semantic value'' of ~c[name]
  and allocates a raw lisp array to ~c[name].  For each legal index, ~c[i],
  that raw lisp array contains ~c[(aref1 name alist' i)] in slot ~c[i].  Thus,
  subsequent ~ilc[aref1] operations can be executed in virtually constant time
  provided they are given ~c[name] and the ~c[alist'] returned by the most
  recently executed ~c[compress1] or ~ilc[aset1] on ~c[name].  ~l[arrays].

  In general, ~c[compress1] returns an alist whose ~ilc[cdr] is an association
  list whose keys are nonnegative integers in ascending order.  However, if the
  ~ilc[header] specifies an ~c[:order] of ~c[>] then the keys will occur in
  descending order, and if the ~c[:order] is ~c[:none] or ~c[nil] then the keys
  will not be sorted, i.e., ~c[compress1] is logically the identity function
  (though it still attaches an array under the hood).  Note however that a
  ~ilc[compress1] call is replaced by a hard error if the header specifies an
  ~c[:order] of ~c[:none] or ~c[nil] and the array's length exceeds the
  ~ilc[maximum-length] field of its ~ilc[header]."

; The uses of (the (unsigned-byte 31) ...) below rely on the array1p guard,
; which for example guarantees that the dimension is bounded by
; *maximum-positive-32-bit-integer* and that each array index (i.e., each car)
; is less than the dimension.  These declarations probably only assist
; efficiency in GCL, but that may be the Lisp that benefits most from such
; fixnum declarations, anyhow.

  #+acl2-loop-only
  (declare (xargs :guard (array1p name l)))
  #+acl2-loop-only
  (case (array-order (header name l))
    (< (cons (header name l)
             (compress11
              name l 0
              (car (dimensions name l))
              (default name l))))
    (> (cons (header name l)
             (reverse (compress11
                       name l 0
                       (car (dimensions name l))
                       (default name l)))))
    (t
     (prog2$
      (and (> (length l)
              (maximum-length name l))
           (hard-error 'compress1
                       "Attempted to compress a one-dimensional array named ~
                        ~x0 whose header specifies :ORDER ~x1 and whose ~
                        length, ~x2, exceeds its maximum-length, ~x3."
                       (list (cons #\0 name)
                             (cons #\1 nil)
                             (cons #\2 (length l))
                             (cons #\3 (maximum-length name l)))))
      l)))
  #-acl2-loop-only
  (let* ((old (get name 'acl2-array))
         (header (header name l))
         (length (car (cadr (assoc-keyword :dimensions (cdr header)))))
         (maximum-length (cadr (assoc-keyword :maximum-length (cdr header))))
         (default (cadr (assoc-keyword :default (cdr header))))
         (order (array-order header))
         old-car
         ar
         in-order)

    (when (and (null order)
               (> (length l) maximum-length))
      (hard-error 'compress1
                  "Attempted to compress a one-dimensional array named ~x0 ~
                   whose header specifies :ORDER ~x1 and whose length, ~x2, ~
                   exceeds its maximum-length, ~x3."
                  (list (cons #\0 name)
                        (cons #\1 nil)
                        (cons #\2 (length l))
                        (cons #\3 (maximum-length name l)))))

; Get an array that is all filled with the special mark *invisible-array-mark*.

    (cond ((and old
                (= 1 (array-rank (cadr old)))
                (= (length (cadr old)) length))
           (setq old-car (car old))
           (setf (car old) *invisible-array-mark*)
           (setq ar (cadr old))
           (do ((i (1- length) (1- i))) ((< i 0))
               (declare (type (signed-byte 32) i))
               (setf (svref ar i) *invisible-array-mark*)))
          (t (setq ar (make-array$ length :initial-element
                                   *invisible-array-mark*))))

; Store the value of each pair under its key (unless it is covered by
; an earlier pair with the same key).

    (do ((tl l (cdr tl)))
        ((null tl))
        (let ((index (caar tl)))
          (cond ((eq index :header) nil)
                ((eq *invisible-array-mark* (svref ar index))
                 (setf (svref ar index)
                       (cdar tl))))))

; Determine whether l is already is in normal form (header first,
; strictly ascending keys, no default values, no extra header.)

    (setq in-order t)
    (when order
      (cond ((eq (caar l) :header)
             (do ((tl (cdr l) (cdr tl)))
                 (nil)
                 (cond ((or (eq (caar tl) :header)
                            (eq (car (cadr tl)) :header))
                        (setq in-order nil)
                        (return nil))
                       ((equal (cdr (car tl)) default)
                        (setq in-order nil)
                        (return nil))
                       ((null (cdr tl)) (return nil))
                       ((if (eq order '>)
                            (<= (the (unsigned-byte 31) (caar tl))
                                (the (unsigned-byte 31) (car (cadr tl))))
                          (>= (the (unsigned-byte 31) (caar tl))
                              (the (unsigned-byte 31) (car (cadr tl)))))
                        (setq in-order nil)
                        (return nil)))))
            (t (setq in-order nil))))
    (let ((num 1) x max-ar)
      (declare (type (unsigned-byte 31) num))

;  In one pass, set x to the value to be returned, put defaults into the array
;  where the invisible mark still sits, and calculate the length of x.

      (cond (in-order
             (do ((i (1- length) (1- i))) ((< i 0))
                 (declare (type (signed-byte 32) i))
                 (let ((val (svref ar i)))
                   (cond ((eq *invisible-array-mark* val)
                          (setf (svref ar i) default))
                         (t (setq num (the (unsigned-byte 31) (1+ num)))))))
             (setq x l))
            ((eq order '>)
             (do ((i 0 (1+ i))) ((int= i length))
                 (declare (type (unsigned-byte 31) i))
                 (let ((val (svref ar i)))
                   (cond ((eq *invisible-array-mark* val)
                          (setf (svref ar i) default))
                         ((equal val default) nil)
                         (t (push (cons i val) x)
                            (setq num (the (unsigned-byte 31) (1+ num)))))))
             (setq x (cons header x)))
            (t (do ((i (1- length) (1- i))) ((< i 0))
                   (declare (type (signed-byte 32) i))
                   (let ((val (svref ar i)))
                     (cond ((eq *invisible-array-mark* val)
                            (setf (svref ar i) default))
                           ((equal val default) nil)
                           (t (push (cons i val) x)
                              (setq num (the (unsigned-byte 31) (1+ num)))))))
               (setq x (cons header x))))
      (cond (old (setq max-ar (caddr old))
                 (setf (aref (the (array (unsigned-byte 31) (*)) max-ar)
                             0)
                       (the (unsigned-byte 31)
                        (- maximum-length num))))
            (t (setq max-ar
                     (make-array$ 1
                                  :initial-contents
                                  (list (- maximum-length num))
                                  :element-type
                                  '(unsigned-byte 31)))))
      (cond (old
             (setf (cadr old) ar)
             (setf (cadddr old) header)

; We re-use the old value if it is equal to the new value.  The example
; regarding compress1 in :doc note-2-7-other shows why we need to do this.  In
; case that is not enough of a reason, here is a comment from Version_2.6 code,
; which is once again the code in Version_2.8.  (Version_2.7 had a bug from an
; ill-advised attempt to deal with a problem with slow array warnings reported
; in :doc note-2-7-bug-fixes.)

; If the old car is equal to x, then we put the old pointer back into the
; car of the 'acl2-array property rather than the new pointer.
; This has the good effect of preserving the validity of any old
; copies of the array.  It is clear the code below is correct, since
; we are putting down an equal structure in place of a newly consed up
; one.  But why go out of our way?  Why not just (setf (car old) x)?
; In fact, once upon a time, that is what we did.  But it bit us when
; we tried to prove theorems in a post-:init world.

; When ACL2 is loaded the Common Lisp global constant
; *type-set-binary-+-table* is defined by (defconst & (compress2 ...)).
; It is set to some list, here called ptr1, built by compress2 (which
; contains code analogous to that we are documenting here in
; compress1).  When ptr1 is built it is stored as the car of the
; 'acl2-array property of the array name 'type-set-binary-+-table, because at
; the time ACL2 is loaded, there is no old 'acl2-array property on
; that name.  Suppose we then :init, loading the ACL2 source code into
; the current ACL2 world.  That will execute the same defconst, in
; the acl2-loop-only setting.  Compress2 is called and will build a
; new structure, ptr2 (called x in this code).  Upon finishing, it
; will (according to the code here) find that ptr2 is equal to ptr1
; and will put ptr1 into the car of the 'acl2-array property of
; 'type-set-binary-+-table.  It will return ptr1.  That will become the value
; of the 'const getprop of '*type-set-binary-+-table* in the
; current-acl2-world.  When that world is installed, we will note that
; a non-virgin name, *type-set-binary-+-table*, is being defconst'd and so
; we will DO NOTHING, leaving ptr1 as the value of the Common Lisp
; global contant *type-set-binary-+-table*.  So, because of the code below,
; all logical copies of this array are represented by ptr1.

; In the old days, compress2 put ptr2 into the car of the 'acl2-array
; property of 'type-set-binary-+-table.  It returned ptr2, which thus became
; the value of the 'const getprop of '*type-set-binary-+-table*.  When
; that world was installed, we noted that a non-virgin name was being
; defconst'd and we DID NOTHING, leaving ptr1 as the value of the
; global constant *type-set-binary-+-table*.  Subsequent references to
; *type-set-binary-+-table* in our type-set code, e.g., as occurred when one
; tried to prove theorems about + after an :init, provoked the
; slow-array-warning.

; The following historical comment no longer applies to
; 'global-enabled-stucture, but it is still relevant to
; 'global-arithmetic-enabled-structure.

; This preservation (eq) of the old array is also crucial to the way
; recompress-global-enabled-structure works.  That function extracts
; the :theory-array from the current global-enabled-structure -- said
; theory-array having been produced by a past call of compress1 and
; hence guaranteed to be sorted etc.  It calls compress1 on it, which
; side-effects the underlying von Neumann array but returns the very
; same (eq) structure.  We then discard that structure, having only
; wanted the side effect!  Before we exploited this, we had to cons up
; a new global-enabled-structure and rebind 'global-enabled-stucture
; in the world.  This had the bad effect of sometimes putting more
; than one binding of that variable.

             (setf (car old)
                   (cond ((equal old-car x) old-car)
                         (t x)))
             (car old))
            (t (setf (get name 'acl2-array)
                     (list x ar max-ar header))
               x)))))

(defthm array1p-cons
  (implies (and (< n
                   (caadr (assoc-keyword :dimensions
                                         (cdr (assoc-eq :header l)))))
                (not (< n 0))
                (integerp n)
                (array1p name l))
           (array1p name (cons (cons n val) l)))
  :hints (("Goal" :in-theory (enable array1p))))

(defun aset1 (name l n val)

  ":Doc-Section Arrays

  set the elements of a 1-dimensional array~/
  ~bv[]
  Example Form:
  (aset1 'delta1 a (+ i k) 27)~/

  General Form:
  (aset1 name alist index val)
  ~ev[]
  where ~c[name] is a symbol, ~c[alist] is a 1-dimensional array named ~c[name],
  ~c[index] is a legal index into ~c[alist], and ~c[val] is an arbitrary object.
  ~l[arrays] for details.  Roughly speaking this function
  ``modifies'' ~c[alist] so that the value associated with ~c[index] is ~c[val].
  More precisely, it returns a new array, ~c[alist'], of the same name and
  dimension as ~c[alist] that, under ~ilc[aref1], is everywhere equal to ~c[alist]
  except at ~c[index] where the result is ~c[val].  That is,
  ~c[(aref1 name alist' i)] is ~c[(aref1 name alist i)] for all legal
  indices ~c[i] except ~c[index], where ~c[(aref1 name alist' i)] is ~c[val].

  In order to ``modify'' ~c[alist], ~c[aset1] ~ilc[cons]es a new pair onto the
  front.  If the length of the resulting alist exceeds the
  ~c[:]~ilc[maximum-length] entry in the array ~il[header], ~c[aset1] compresses the
  array as with ~ilc[compress1].

  It is generally expected that the ``semantic value'' of ~c[name] will be
  ~c[alist] (~pl[arrays]).  This function operates in virtually
  constant time whether this condition is true or not (unless the
  ~ilc[compress1] operation is required).  But the value returned by this
  function cannot be used efficiently by subsequent ~c[aset1] operations
  unless ~c[alist] is the semantic value of ~c[name] when ~c[aset1] is executed.
  Thus, if the condition is not true, ~c[aset1] prints a ~st[slow array]
  warning to the comment window.  ~l[slow-array-warning]."

  #+acl2-loop-only
  (declare (xargs :guard (and (array1p name l)
                              (integerp n)
                              (>= n 0)
                              (< n (car (dimensions name l))))))
  #+acl2-loop-only
  (let ((l (cons (cons n val) l)))
    (cond ((> (length l)
              (maximum-length name l))
           (compress1 name l))
          (t l)))
  #-acl2-loop-only
  (declare (type (unsigned-byte 31) n))
  #-acl2-loop-only
  (let ((prop (get name 'acl2-array)))
    (cond ((eq l (car prop))
           (let* ((ar (cadr prop))
                  (to-go (aref (the (array (unsigned-byte 31) (*))
                                    (caddr prop))
                               0)))
             (declare (type (unsigned-byte 31) to-go)
                      (simple-vector ar))
             (cond ((eql (the (unsigned-byte 31) to-go) 0)
                    (setf (car prop) *invisible-array-mark*)
                    (setf (aref ar n) val)
                    (let* ((header (cadddr prop))
                           (order (array-order header))
                           (length (car (cadr (assoc-keyword
                                               :dimensions
                                               (cdr header)))))
                           (maximum-length
                            (cadr (assoc-keyword
                                   :maximum-length (cdr header))))
                           (default
                             (cadr (assoc-keyword
                                    :default (cdr header))))
                           (x nil)
                           (num 1))
                      (declare (type (unsigned-byte 31) num length))
                      (declare (type (unsigned-byte 31) maximum-length))
                      (cond ((null order)
; Cause same error as in the logic.
                             (return-from aset1
                                          (compress1 name (cons (cons n val)
                                                                l))))
                            ((eq order '>)
                             (do ((i 0 (1+ i)))
                                 ((int= i length))
                                 (declare (type (unsigned-byte 31) i))
                                 (let ((val (svref ar (the (unsigned-byte 31) i))))
                                   (cond ((equal val default) nil)
                                         (t (push (cons i val) x)
                                            (setq num (the (unsigned-byte 31)
                                                           (1+ num))))))))
                            (t
                             (do ((i (1- length) (1- i)))
                                 ((< i 0))
                                 (declare (type (signed-byte 32) i))
                                 (let ((val (svref ar (the (signed-byte 32) i))))
                                   (cond ((equal val default) nil)
                                         (t (push (cons i val) x)
                                            (setq num (the (unsigned-byte 31)
                                                           (1+ num)))))))))
                      (setq x (cons header x))
                      (setf (aref (the (array (unsigned-byte 31) (*))
                                       (caddr prop)) 0)
                            (the (unsigned-byte 31) (- maximum-length num)))
                      (setf (car prop) x)
                      x))
                   (t (let ((x (cons (cons n val) l)))
                        (setf (car prop) *invisible-array-mark*)
                        (setf (svref (the simple-vector ar) n) val)
                        (setf (aref (the (array (unsigned-byte 31) (*))
                                         (caddr prop))
                                    0)
                              (the (unsigned-byte 31) (1- to-go)))
                        (setf (car prop) x)
                        x)))))
          (t (let ((l (cons (cons n val) l)))
               (slow-array-warning 'aset1 name)
               (cond ((> (length l)
                         (maximum-length name l))
                      (compress1 name l))
                     (t l)))))))

(defun aref2 (name l i j)

  ":Doc-Section Arrays

  access the elements of a 2-dimensional array~/
  ~bv[]
  Example Form:
  (aref2 'delta1 a i j)~/

  General Form:
  (aref2 name alist i j)
  ~ev[]
  where ~c[name] is a symbol, ~c[alist] is a 2-dimensional array and ~c[i] and ~c[j]
  are legal indices into ~c[alist].  This function returns the value
  associated with ~c[(i . j)] in ~c[alist], or else the default value of the
  array.  ~l[arrays] for details.

  This function executes in virtually constant time if ~c[alist] is in
  fact the ``semantic value'' associated with ~c[name] (~pl[arrays]).
  When it is not, ~c[aref2] must do a linear search through ~c[alist].  In
  that case the correct answer is returned but a ~st[slow array] comment is
  printed to the comment window.  ~l[slow-array-warning]."

  #+acl2-loop-only
  (declare (xargs :guard (and (array2p name l)
                              (integerp i)
                              (>= i 0)
                              (< i (car (dimensions name l)))
                              (integerp j)
                              (>= j 0)
                              (< j (cadr (dimensions name l))))))
  #+acl2-loop-only
  (let ((x (assoc2 i j l)))
    (cond ((null x) (default name l))
          (t (cdr x))))
  #-acl2-loop-only
  (declare (type (unsigned-byte 31) i j))
  #-acl2-loop-only
  (let ((prop (get name 'acl2-array)))
    (cond ((eq l (car prop))
           (aref (the (array * (* *)) (car (cdr prop)))
                 i j))
          (t (slow-array-warning 'aref2 name)
             (let ((x (assoc2 i j l)))
               (cond ((null x) (default name l))
                     (t (cdr x))))))))

(defun compress211 (name l i x j default)
  (declare (xargs :guard (and (array2p name l)
                              (integerp x)
                              (integerp i)
                              (integerp j)
                              (<= x j))
                  :measure (nfix (- j x))))
  (cond ((zp (- j x))
         nil)
        (t (let ((pair (assoc2 i x l)))
             (cond ((or (null pair)
                        (equal (cdr pair) default))
                    (compress211 name l i (+ 1 x) j default))
                   (t (cons pair
                            (compress211 name l i (+ 1 x) j default))))))))

(defun compress21 (name l n i j default)
  (declare (xargs :guard (and (array2p name l)
                              (integerp n)
                              (integerp i)
                              (integerp j)
                              (<= n i)
                              (<= 0 j))
                  :measure (nfix (- i n))))

  (cond ((zp (- i n)) nil)
        (t (append (compress211 name l n 0 j default)
                   (compress21 name l (+ n 1) i j default)))))

(defun compress2 (name l)

  ":Doc-Section Arrays

  remove irrelevant pairs from a 2-dimensional array~/
  ~bv[]
  Example Form:
  (compress2 'delta1 a)~/

  General Form:
  (compress2 name alist)
  ~ev[]
  where ~c[name] is a symbol and ~c[alist] is a 2-dimensional array, generally
  named ~c[name].  ~l[arrays] for details.  Logically speaking, this function
  removes irrelevant pairs from ~c[alist], possibly shortening it.  The
  function returns a new array, ~c[alist'], with the same ~ilc[header]
  (including name and dimension) as ~c[alist], that, under ~ilc[aref2], is
  everywhere equal to ~c[alist].  That is, ~c[(aref2 name alist' i j)] is
  ~c[(aref2 name alist i j)], for all legal indices ~c[i] and ~c[j].
  ~c[Alist'] may be shorter than ~c[alist] and the non-irrelevant pairs may
  occur in a different order in ~c[alist'] than in ~c[alist].

  Practically speaking, this function plays an important role in the
  efficient implementation of ~ilc[aref2].  In addition to creating the new
  array, ~c[alist'], ~c[compress2] makes that array the ``semantic value'' of
  ~c[name] and allocates a raw lisp array to ~c[name].  For all legal indices,
  ~c[i] and ~c[j], that raw lisp array contains ~c[(aref2 name alist' i j)] in
  slot ~c[i],~c[j].  Thus, subsequent ~ilc[aref2] operations can be executed in
  virtually constant time provided they are given ~c[name] and the ~c[alist']
  returned by the most recently executed ~c[compress2] or ~ilc[aset2] on ~c[name].
  ~l[arrays]."

  #+acl2-loop-only

; The uses of (the (unsigned-byte 31) ...) below rely on the array2p
; guard, which for example guarantees that each dimension is bounded
; by *maximum-positive-32-bit-integer* and that array indices are
; therefore less than *maximum-positive-32-bit-integer*.  These
; declarations probably only assist efficiency in GCL, but that may be
; the Lisp that benefits most from such fixnum declarations, anyhow.

  (declare (xargs :guard (array2p name l)))
  #+acl2-loop-only
  (cons (header name l)
        (compress21 name l 0
                    (car (dimensions name l))
                    (cadr (dimensions name l))
                    (default name l)))
  #-acl2-loop-only
  (let* ((old (get name 'acl2-array))
         (header (header name l))
         (dimension1 (car (cadr (assoc-keyword :dimensions (cdr header)))))
         (dimension2 (cadr (cadr (assoc-keyword :dimensions (cdr header)))))
         (maximum-length (cadr (assoc-keyword :maximum-length (cdr header))))
         (default (cadr (assoc-keyword :default (cdr header))))
         old-car
         ar
         in-order)

;  Get an array that is filled with the special mark *invisible-array-mark*.

    (cond ((and old
                (= 2 (array-rank (cadr old)))
                (and (= dimension1 (array-dimension (cadr old) 0))
                     (= dimension2 (array-dimension (cadr old) 1))))
           (setq old-car (car old))
           (setf (car old) *invisible-array-mark*)
           (setq ar (cadr old))
           (let ((ar ar))
             (declare (type (array * (* *)) ar))
             (do ((i (1- dimension1) (1- i))) ((< i 0))
                 (declare (type fixnum i))
                 (do ((j (1- dimension2) (1- j))) ((< j 0))
                     (declare (type fixnum j))
                     (setf (aref ar i j) *invisible-array-mark*)))))
          (t (setq ar
                   (make-array$ (list dimension1 dimension2)
                                :initial-element
                                *invisible-array-mark*))))
    (let ((ar ar))
      (declare (type (array * (* *)) ar))

; Store the value of each pair under its key (unless it is covered by
; an earlier pair with the same key).

      (do ((tl l (cdr tl)))
          ((null tl))
          (let ((index (caar tl)))
            (cond ((eq index :header) nil)
                  ((eq *invisible-array-mark*
                       (aref ar
                             (the fixnum (car index))
                             (the fixnum (cdr index))))
                   (setf (aref ar
                               (the fixnum (car index))
                               (the fixnum (cdr index)))
                         (cdar tl))))))

; Determine whether l is already in normal form (header first,
; strictly ascending keys, no default values, n extra header.)

      (setq in-order t)
      (cond ((eq (caar l) :header)
             (do ((tl (cdr l) (cdr tl)))
                 (nil)
                 (cond ((or (eq (caar tl) :header)
                            (eq (car (cadr tl)) :header))
                        (setq in-order nil)
                        (return nil))
                       ((equal (cdr (car tl)) default)
                        (setq in-order nil)
                        (return nil))
                       ((null (cdr tl)) (return nil))
                       ((or (> (the (unsigned-byte 31) (caaar tl))
                               (the (unsigned-byte 31) (caaadr tl)))
                            (and (= (the (unsigned-byte 31) (caaar tl))
                                    (the (unsigned-byte 31) (caaadr tl)))
                                 (> (the (unsigned-byte 31) (cdaar tl))
                                    (the (unsigned-byte 31) (cdaadr tl)))))
                        (setq in-order nil)
                        (return nil)))))
            (t (setq in-order nil)))
      (let ((x nil) (num 1) max-ar)
        (declare (type (unsigned-byte 31) num))

;  In one pass, set x to the value to be returned, put defaults into the array
;  where the invisible mark still sits, and calculate the length of x.

        (cond (in-order
               (do ((i (1- dimension1) (1- i)))
                   ((< i 0))
                   (declare (type fixnum i))
                   (do ((j (1- dimension2) (1- j)))
                       ((< j 0))
                       (declare (type fixnum j))
                       (let ((val (aref ar i j)))
                         (cond ((eq *invisible-array-mark* val)
                                (setf (aref ar i j) default))
                               (t
                                (setq num (the (unsigned-byte 31)
                                           (1+ num))))))))
               (setq x l))
              (t (do ((i (1- dimension1) (1- i)))
                     ((< i 0))
                     (declare (type fixnum i))
                     (do ((j (1- dimension2) (1- j)))
                         ((< j 0))
                         (declare (type fixnum j))
                         (let ((val (aref ar i j)))
                           (cond ((eq *invisible-array-mark* val)
                                  (setf (aref ar i j) default))
                                 ((equal val default) nil)
                                 (t (push (cons (cons i j) val) x)
                                    (setq num (the (unsigned-byte 31)
                                               (1+ num))))))))
                 (setq x (cons header x))))
        (cond (old (setq max-ar (caddr old))
                   (setf (aref (the (array (unsigned-byte 31) (*)) max-ar)
                               0)
                         (the (unsigned-byte 31)
                          (- maximum-length num))))
              (t (setq max-ar
                       (make-array$ 1
                                    :initial-contents
                                    (list (- maximum-length num))
                                    :element-type
                                    '(unsigned-byte 31)))))
        (cond (old
               (setf (cadr old) ar)
               (setf (cadddr old) header)
               (setf (car old)
                     (cond ((equal old-car x) old-car)
                           (t x)))
               (car old))
              (t
               (setf (get name 'acl2-array)
                     (list x ar max-ar header))
               x))))))

(defthm array2p-cons
  (implies (and (< j (cadr (dimensions name l)))
                (not (< j 0))
                (integerp j)
                (< i (car (dimensions name l)))
                (not (< i 0))
                (integerp i)
                (array2p name l))
           (array2p name (cons (cons (cons i j) val) l)))
  :hints (("Goal" :in-theory (enable array2p))))

(defun aset2 (name l i j val)

  ":Doc-Section Arrays

  set the elements of a 2-dimensional array~/
  ~bv[]
  Example Form:
  (aset2 'delta1 a i j 27)~/

  General Form:
  (aset2 name alist i j val)
  ~ev[]
  where ~c[name] is a symbol, ~c[alist] is a 2-dimensional array named ~c[name],
  ~c[i] and ~c[j] are legal indices into ~c[alist], and ~c[val] is an arbitrary
  object.  ~l[arrays] for details.  Roughly speaking this
  function ``modifies'' ~c[alist] so that the value associated with
  ~c[(i . j)] is ~c[val].  More precisely, it returns a new array,
  ~c[alist'], of the same name and dimension as ~c[alist] that, under
  ~ilc[aref2], is everywhere equal to ~c[alist] except at ~c[(i . j)] where
  the result is ~c[val].  That is, ~c[(aref2 name alist' x y)] is
  ~c[(aref2 name alist x y)] for all legal indices ~c[x] ~c[y] except
  ~c[i] and ~c[j] where ~c[(aref2 name alist' i j)] is ~c[val].

  In order to ``modify'' ~c[alist], ~c[aset2] ~ilc[cons]es a new pair onto the
  front.  If the length of the resulting ~c[alist] exceeds the
  ~c[:]~ilc[maximum-length] entry in the array ~il[header], ~c[aset2] compresses the
  array as with ~ilc[compress2].

  It is generally expected that the ``semantic value'' of ~c[name] will be
  ~c[alist] (~pl[arrays]).  This function operates in virtually
  constant time whether this condition is true or not (unless the
  ~ilc[compress2] operation is required).  But the value returned by this
  function cannot be used efficiently by subsequent ~c[aset2] operations
  unless ~c[alist] is the semantic value of ~c[name] when ~c[aset2] is executed.
  Thus, if the condition is not true, ~c[aset2] prints a ~st[slow array]
  warning to the comment window.  ~l[slow-array-warning]."

  #+acl2-loop-only
  (declare (xargs :guard (and (array2p name l)
                              (integerp i)
                              (>= i 0)
                              (< i (car (dimensions name l)))
                              (integerp j)
                              (>= j 0)
                              (< j (cadr (dimensions name l))))))
  #+acl2-loop-only
  (let ((l (cons (cons (cons i j) val) l)))
    (cond ((> (length l)
              (maximum-length name l))
           (compress2 name l))
          (t l)))
  #-acl2-loop-only
  (declare (type (unsigned-byte 31) i j))
  #-acl2-loop-only
  (let ((prop (get name 'acl2-array)))
    (cond
     ((eq l (car prop))
      (let* ((ar (car (cdr prop)))
             (to-go (aref (the (array (unsigned-byte 31) (*))
                           (caddr prop))
                          0)))
        (declare (type (unsigned-byte 31) to-go))
        (declare (type (array * (* *)) ar))
        (cond
         ((eql (the (unsigned-byte 31) to-go) 0)
          (setf (car prop) *invisible-array-mark*)
          (setf (aref ar i j) val)
          (let* ((header (cadddr prop))
                 (d1 (car (cadr (assoc-keyword :dimensions (cdr header)))))
                 (d2 (cadr (cadr (assoc-keyword :dimensions (cdr header)))))
                 (maximum-length
                  (cadr (assoc-keyword
                         :maximum-length (cdr header))))
                 (default (cadr (assoc-keyword :default (cdr header))))
                 (x nil)
                 (num 1))
            (declare (type (unsigned-byte 31) num d1 d2 maximum-length))
            (do ((i (1- d1) (1- i)))
                ((< i 0))
                (declare (type fixnum i))
                (do ((j (1- d2) (1- j)))
                    ((< j 0))
                    (declare (type fixnum j))
                    (let ((val (aref ar
                                     (the fixnum i)
                                     (the fixnum j))))
                      (cond ((equal val default) nil)
                            (t (push (cons (cons i j) val) x)
                               (setq num (the (unsigned-byte 31)
                                          (1+ num))))))))
            (setq x (cons header x))
            (setf (aref (the (array (unsigned-byte 31) (*))
                         (caddr prop))
                        0)
                  (the (unsigned-byte 31) (- maximum-length num)))
            (setf (car prop) x)
            x))
         (t (let ((x (cons (cons (cons i j) val) l)))
              (setf (car prop) *invisible-array-mark*)
              (setf (aref ar i j) val)
              (setf (aref (the (array (unsigned-byte 31) (*))
                           (caddr prop))
                          0)
                    (the (unsigned-byte 31) (1- to-go)))
              (setf (car prop) x)
              x)))))
     (t (let ((l (cons (cons (cons i j) val) l)))
          (slow-array-warning 'aset2 name)
          (cond ((> (length l)
                    (maximum-length name l))
                 (compress2 name l))
                (t l)))))))

(defun flush-compress (name)

  ":Doc-Section Arrays

  flush the under-the-hood array for the given name~/
  ~bv[]
  Example Form:
  (flush-compress 'my-array)~/

  General Form:
  (flush-compress name)
  ~ev[]
  where ~c[name] is a symbol.

  Recall that ~c[(compress1 nm alist)] associates an under-the-hood raw Lisp
  one-dimensional array of name ~c[nm] with the given association list,
  ~c[alist], while ~c[(compress2 nm alist)] is the analogous function for
  two-dimensional arrays; ~pl[compress1] and ~pl[compress2].  The only purpose
  of ~c[flush-compress], which always returns ~c[nil], is to remove the
  association of any under-the-hood array with the given name, thus eliminating
  slow array accesses (~pl[slow-array-warning]).  It is not necessary if the
  return values of ~ilc[compress1] and ~ilc[compress2] are always used as the
  ``current'' copy of the named array, and thus ~c[flush-compress] should
  rarely, if ever, be needed in user applications.

  Nevertheless, we provide the following contrived example to show how
  ~c[flush-compress] can be used to good effect.  Comments have been added to
  this log to provide explanation.
  ~bv[]
  ACL2 !>(assign a (compress1 'demo
                              '((:header :dimensions (5)
                                         :maximum-length 15
                                         :default uninitialized
                                         :name demo)
                                (0 . zero)
                                (1 . one))))
   ((:HEADER :DIMENSIONS (5)
             :MAXIMUM-LENGTH
             15 :DEFAULT UNINITIALIZED :NAME DEMO)
    (0 . ZERO)
    (1 . ONE))
  ACL2 !>(aref1 'demo (@ a) 0)
  ZERO
  ; As expected, the above evaluation did not cause a slow array warning.  Now
  ; we associate a different under-the-hood array with the name 'demo.
  ACL2 !>(compress1 'demo
                    '((:header :dimensions (5)
                               :maximum-length 15
                               :default uninitialized
                               :name demo)
                      (0 . zero)))
  ((:HEADER :DIMENSIONS (5)
            :MAXIMUM-LENGTH
            15 :DEFAULT UNINITIALIZED :NAME DEMO)
   (0 . ZERO))
  ; The following array access produces a slow array warning because (@ a) is
  ; no longer associated under-the-hood with the array name 'demo.
  ACL2 !>(aref1 'demo (@ a) 0)


  **********************************************************
  Slow Array Access!  A call of AREF1 on an array named
  DEMO is being executed slowly.  See :DOC slow-array-warning
  **********************************************************

  ZERO
  ; Now we associate under-the-hood, with array name 'demo, an alist equal to
  ; (@ a).
  ACL2 !>(compress1 'demo
                    '((:header :dimensions (5)
                               :maximum-length 15
                               :default uninitialized
                               :name demo)
                      (0 . zero)
                      (1 . one)))
  ((:HEADER :DIMENSIONS (5)
            :MAXIMUM-LENGTH
            15 :DEFAULT UNINITIALIZED :NAME DEMO)
   (0 . ZERO)
   (1 . ONE))
  ; The following array access is still slow, because the under-the-hood array
  ; is merely associated with a copy of (@ a), not with the actual object
  ; (@ a).
  ACL2 !>(aref1 'demo (@ a) 0)


  **********************************************************
  Slow Array Access!  A call of AREF1 on an array named
  DEMO is being executed slowly.  See :DOC slow-array-warning
  **********************************************************

  ZERO
  ; So we might try to fix the problem by recompressing. But this doesn't
  ; work.  It would work, by the way, if we re-assign a:
  ; (assign a (compress1 'demo (@ a))).  That is why we usually will not need
  ; flush-compress.
  ACL2 !>(compress1 'demo (@ a))
  ((:HEADER :DIMENSIONS (5)
            :MAXIMUM-LENGTH
            15 :DEFAULT UNINITIALIZED :NAME DEMO)
   (0 . ZERO)
   (1 . ONE))
  ACL2 !>(aref1 'demo (@ a) 0)


  **********************************************************
  Slow Array Access!  A call of AREF1 on an array named
  DEMO is being executed slowly.  See :DOC slow-array-warning
  **********************************************************

  ZERO
  ; Finally, we eliminate the warning by calling flush-compress before we call
  ; compress1.  The call of flush-compress removes any under-the-hood
  ; association of an array with the name 'demo.  Then the subsequent call of
  ; compress1 associates the object (@ a) with that name.  (Technical point:
  ; compress1 always associates the indicated name with the value that it
  ; returns.  in this case, what compress1 returns is (@ a), because (@ a) is
  ; already, logically speaking, a compressed array1p (starts with a :header
  ; and the natural number keys are ordered).
  ACL2 !>(flush-compress 'demo)
  NIL
  ACL2 !>(compress1 'demo (@ a))
  ((:HEADER :DIMENSIONS (5)
            :MAXIMUM-LENGTH
            15 :DEFAULT UNINITIALIZED :NAME DEMO)
   (0 . ZERO)
   (1 . ONE))
  ACL2 !>(aref1 'demo (@ a) 0)
  ZERO
  ACL2 !>
  ~ev[]"

  (declare (xargs :guard t))
  #+acl2-loop-only
  (declare (ignore name))
  #+acl2-loop-only
  nil
  #-acl2-loop-only
  (setf (get name 'acl2-array) nil))

; MULTIPLE VALUE returns, done our way, not Common Lisp's way.

; We implement an efficient mechanism for returning a multiple value,
; with an applicative semantics.  Formally, the macro mv is just the
; same as ``list''; one can use it to return a list of arbitrary
; objects.  However, the translator for ACL2 checks that mv is in fact
; only used to return values to mv-let, a special form of let which
; picks out the members of a list but does not hold on to the cdrs of
; the list.  Because mv-let does not hold on to cdrs, we are able to
; implement mv so that the list is never actually consed up.  Instead,
; the elements of the list are passed to mv-let in global locations.

; *number-of-return-values* may be increased (but not reduced) to be
; as high as required to increase the allowed number of ACL2 return
; values.  However, if it is increased, the entire ACL2 system must be
; recompiled.  Currently, the first 10 locations are handled specially
; in releases of AKCL past 206.

#-(or acl2-loop-only acl2-mv-as-values)
(progn

(defparameter *return-values*
  (let (ans)
    (do ((i *number-of-return-values* (1- i))) ((= i 0))
        (push (intern (format nil "*return-value-~a*" i))
              ans))
    ans))

(defmacro declare-return-values ()
  (cons 'progn (declare-return-values1)))

(defun declare-return-values1 ()
  (mapcar #'(lambda (v) `(defvar ,v))
          *return-values*))

(eval-when
 #-cltl2
 (load eval compile)
 #+cltl2
 (:load-toplevel :execute :compile-toplevel)
 (declare-return-values))

(defun in-akcl-with-mv-set-and-ref ()
  (member :akcl-set-mv *features*))

(defconstant *akcl-mv-ref-and-set-inclusive-upper-bound* 9)

(defmacro special-location (i)
  (cond ((or (not (integerp i))
             (< i 1))
         (acl2::interface-er
          "Macro calls of special-location must have an explicit ~
           positive integer argument, which is not the case with ~x0." i))
        ((> i *number-of-return-values*)
         (acl2::interface-er "Not enough built-in return values."))
        (t (nth (1- i) *return-values*))))

(defmacro set-mv (i v)
  (cond ((or (not (integerp i))
             (< i 1))
         (interface-er
          "The first argument to a macro call of set-mv must be ~
           an explicit positive integer, but that is not the case ~
           with ~A." i))
        #+akcl
        ((and (in-akcl-with-mv-set-and-ref)
              (<= i *akcl-mv-ref-and-set-inclusive-upper-bound*))
         `(system::set-mv ,i ,v))
        (t `(setf (special-location ,i) ,v))))

(defmacro mv-ref (i)
  (cond ((or (not (integerp i))
             (< i 1))
         (interface-er
          "The argument to macro calls of mv-ref must be an ~
           explicit positive integer, but that is not the case with ~x0." i))
        #+akcl
        ((and (in-akcl-with-mv-set-and-ref)
              (<= i *akcl-mv-ref-and-set-inclusive-upper-bound*))
         `(system::mv-ref ,i))
        (t `(special-location ,i))))

(defun mv-refs-fn (i)
  (let (ans)
    (do ((k i (1- k)))
        ((= k 0))
        (push `(mv-ref ,k)
              ans))
    ans))

(defmacro mv-refs (i)
  (cond
   ((and (natp i) (< i *number-of-return-values*)) ; optimization
    (cons 'list (mv-refs-fn i)))
   (t
    `(case ,i
       ,@(let (ans)
           (do ((j *number-of-return-values* (1- j)))
               ((= j 0))
               (push
                `(,j (list ,@(mv-refs-fn j)))
                ans))
           ans)
       (otherwise (interface-er "Not enough return values."))))))

)

(defun cdrn (x i)
  (declare (xargs :guard (and (integerp i)
                              (<= 0 i))))
  (cond ((zp i) x)
        (t (cdrn (list 'cdr x) (- i 1)))))

(defun mv-nth (n l)

  ":Doc-Section ACL2::Programming

  the mv-nth element (zero-based) of a list~/

  ~c[(Mv-nth n l)] is the ~c[n]th element of ~c[l], zero-based.  If ~c[n] is
  greater than or equal to the length of ~c[l], then ~c[mv-nth] returns
  ~c[nil].~/

  ~c[(Mv-nth n l)] has a ~il[guard] that ~c[n] is a non-negative integer and
  ~c[l] is a ~ilc[true-listp].

  ~c[Mv-nth] is equivalent to the Common Lisp function ~ilc[nth], but is used by
  ACL2 to access the nth value returned by a multiply valued expression.  For
  an example of the use of ~c[mv-nth], try
  ~bv[]
  ACL2 !>:trans1 (mv-let (erp val state)
                         (read-object ch state)
                         (value (list erp val)))
  ~ev[]
  ~/"

  (declare (xargs :guard (and (integerp n)
                              (>= n 0)
                              (true-listp l))))
  (if (endp l)
      nil
    (if (zp n)
        (car l)
      (mv-nth (- n 1) (cdr l)))))

(defun make-mv-nths (args call i)
  (declare (xargs :guard (and (true-listp args)
                              (integerp i))))
  (cond ((endp args) nil)
        (t (cons (list (car args) (list 'mv-nth i call))
                 (make-mv-nths (cdr args) call (+ i 1))))))

#-(or acl2-loop-only acl2-mv-as-values)
(defun mv-bindings (lst)

; Gensym a var for every element of lst except the last and pair
; that var with its element in a doublet.  Return the list of doublets.

  (cond ((null (cdr lst)) nil)
        (t (cons (list (gensym) (car lst))
                 (mv-bindings (cdr lst))))))

#-(or acl2-loop-only acl2-mv-as-values)
(defun mv-set-mvs (bindings i)
  (cond ((null bindings) nil)
        (t (cons `(set-mv ,i ,(caar bindings))
                 (mv-set-mvs (cdr bindings) (1+ i))))))

(defmacro mv (&rest l)

  ":Doc-Section ACL2::Programming

  returning a multiple value~/

  ~c[Mv] is the mechanism provided by ACL2 for returning two or more
  values.  Logically, ~c[(mv x1 x2 ... xn)] is the same as
  ~c[(list x1 x2 ... xn)], a list of the indicated values.  However,
  ACL2 avoids the cost of building this list structure, with the cost
  that ~c[mv] may only be used in a certain style in definitions:  if a
  function ever returns using ~c[mv] (either directly, or by calling
  another function that returns a multiple value), then this function
  must always return the same number of values.

  For more explanation of the multiple value mechanism,
  ~pl[mv-let].~/

  ACL2 does not support the Common Lisp construct ~c[values], whose
  logical meaning seems difficult to characterize.  ~c[Mv] is the ACL2
  analogue of that construct.~/"

  (declare (xargs :guard (>= (length l) 2)))

  #+acl2-loop-only
  (cons 'list l)
  #+(and (not acl2-loop-only) acl2-mv-as-values)
  (return-from mv (cons 'values l))
  #+(and (not acl2-loop-only) (not acl2-mv-as-values))

; In an earlier version of the mv macro, we had a terrible bug.
; (mv a b ... z) expanded to

; (LET ((#:G1 a))
;   (SET-MV 1 b)
;   ...
;   (SET-MV k z)
;   (SETQ *MOST-RECENT-MULTIPLICITY* 3)
;   #:G1)

; Note that if the evaluation of z uses a multiple value then it
; overwrites the earlier SET-MV.  Now this expansion is safe if there
; are only two values because the only SET-MV is done after the second
; value is computed.  If there are three or more value forms, then
; this expansion is also safe if all but the first two are atomic.
; For example, (mv & & (killer)) is unsafe because (killer) may
; overwrite the SET-MV, but (mv & & STATE) is safe because the
; evaluation of an atomic form is guaranteed not to overwrite SET-MV
; settings.  In general, all forms after the second must be atomic for
; the above expansion to be used.

  (cond ((atom-listp (cddr l))

; We use the old expansion because it is safe and more efficient.

         (let ((v (gensym)))
           `(let ((,v ,(car l)))

; In GCL (at the least), it is possible to avoid boxing fixnums that are the
; first value returned, if we are a bit careful.  In particular, it is useful
; to insert a declaration here when we see (mv (the type expr) ...) where
; type is contained in the set of fixnums.

              ,@(let ((output (macroexpand-till (car l) 'the)))
                  (cond ((and (consp output)
                              (eq 'the (car output)))
                         `((declare (type ,(cadr output) ,v))))
                        (t nil)))
              ,@(let (ans)
                  (do ((tl (cdr l) (cdr tl))
                       (i 1 (1+ i)))
                      ((null tl))
                      (push `(set-mv ,i ,(car tl))
                            ans))
                  (nreverse ans))
              ,v)))
        (t

; We expand (mv a b ... y z) to
; (LET ((#:G1 a)
;       (#:G2 b)
;       ...
;       (#:Gk y))
;  (SET-MV k z)
;  (SET-MV 1 #:G2)
;  ...
;  (SET-MV k-1 #:Gk)
;  #:G1)

         (let ((bindings (mv-bindings l)))
           `(let ,bindings

; See comment above regarding boxing fixnums.

              ,@(let ((output (macroexpand-till (car l) 'the)))
                  (cond ((and (consp output)
                              (eq 'the (car output)))
                         `((declare (type ,(cadr output) ,(caar bindings)))))
                        (t nil)))
              (set-mv ,(1- (length l)) ,(car (last l)))
              ,@(mv-set-mvs (cdr bindings) 1)
              ,(caar bindings))))))

(defmacro mv-let (&rest rst)

; Warning: If the final logical form of a translated mv-let is
; changed, be sure to reconsider translated-acl2-unwind-protectp.

  ":Doc-Section ACL2::Programming

  calling multi-valued ACL2 functions~/
  ~bv[]
  Example Form:
  (mv-let (x y z)              ; local variables
          (mv 1 2 3)           ; multi-valued expression
          (declare (ignore y)) ; optional declarations
          (cons x z))          ; body
  ~ev[]
  The form above binds the three ``local variables,'' ~c[x], ~c[y], and ~c[z],
  to the three results returned by the multi-valued expression and
  then evaluates the body.  The result is ~c['(1 . 3)].  The second local,
  ~c[y], is ~il[declare]d ~c[ignore]d.  The multi-valued expression can be any ACL2
  expression that returns ~c[k] results, where ~c[k] is the number of local
  variables listed.  Often however it is simply the application of a
  ~c[k]-valued function.  ~c[Mv-let] is the standard way to invoke a
  multi-valued function when the caller must manipulate the vector of
  results returned.~/
  ~bv[]
  General Form:
  (mv-let (var1 ... vark)
          term
          body)
  or
  (mv-let (var1 ... vark)
          term
          (declare ...) ... (declare ...)
          body)
  ~ev[]
  where the ~c[vari] are distinct variables, ~c[term] is a term that returns
  ~c[k] results and mentions only variables bound in the environment containing
  the ~c[mv-let] expression, and ~c[body] is a term mentioning only the
  ~c[vari] and variables bound in the environment containing the ~c[mv-let].
  Each ~c[vari] must occur in ~c[body] unless it is ~il[declare]d ~c[ignore]d
  or ~c[ignorable] in one of the optional ~ilc[declare] forms, unless this
  requirement is turned off; ~pl[set-ignore-ok].  The value of the ~c[mv-let]
  term is the result of evaluating ~c[body] in an environment in which the
  ~c[vari] are bound, in order, to the ~c[k] results obtained by evaluating
  ~c[term] in the environment containing the ~c[mv-let].

  Here is an extended example that illustrates both the definition of
  a multi-valued function and the use of ~c[mv-let] to call it.  Consider
  a simple binary tree whose interior nodes are ~ilc[cons]es and whose
  leaves are non-~ilc[cons]es.  Suppose we often need to know the number, ~c[n],
  of interior nodes of such a tree; the list, ~c[syms], of symbols that
  occur as leaves; and the list, ~c[ints], of integers that occur as
  leaves.  (Observe that there may be leaves that are neither symbols
  nor integers.)  Using a multi-valued function we can collect all
  three results in one pass.

  Here is the first of two definitions of the desired function.  This
  definition is ``primitive recursive'' in that it has only one
  argument and that argument is reduced in size on every recursion.
  ~bv[]
  (defun count-and-collect (x)

  ; We return three results, (mv n syms ints) as described above.

    (cond ((atom x)

  ; X is a leaf.  Thus, there are 0 interior nodes, and depending on
  ; whether x is a symbol, an integer, or something else, we return
  ; the list containing x in as the appropriate result.

           (cond ((symbolp x) (mv 0 (list x) nil))
                 ((integerp x)(mv 0 nil      (list x)))
                 (t           (mv 0 nil      nil))))
          (t

  ; X is an interior node.  First we process the car, binding n1, syms1, and
  ; ints1 to the answers.

             (mv-let (n1 syms1 ints1)
                     (count-and-collect (car x))

  ; Next we process the cdr, binding n2, syms2, and ints2.

                     (mv-let (n2 syms2 ints2)
                             (count-and-collect (car x))

  ; Finally, we compute the answer for x from those obtained for its car
  ; and cdr, remembering to increment the node count by one for x itself.

                             (mv (1+ (+ n1 n2))
                                 (append syms1 syms2)
                                 (append ints1 ints2)))))))
  ~ev[]
  This use of a multiple value to ``do several things at once'' is
  very common in ACL2.  However, the function above is inefficient
  because it ~il[append]s ~c[syms1] to ~c[syms2] and ~c[ints1] to ~c[ints2], copying the
  list structures of ~c[syms1] and ~c[ints1] in the process.  By adding
  ``accumulators'' to the function, we can make the code more
  efficient.
  ~bv[]
  (defun count-and-collect1 (x n syms ints)
    (cond ((atom x)
           (cond ((symbolp x) (mv n (cons x syms) ints))
                 ((integerp x) (mv n syms (cons x ints)))
                 (t (mv n syms ints))))
          (t (mv-let (n2 syms2 ints2)
                     (count-and-collect1 (cdr x) (1+ n) syms ints)
                     (count-and-collect1 (car x) n2 syms2 ints2)))))
  ~ev[]
  We claim that ~c[(count-and-collect x)] returns the same triple of
  results as ~c[(count-and-collect1 x 0 nil nil)].  The reader is urged to
  study this claim until convinced that it is true and that the latter
  method of computing the results is more efficient.  One might try
  proving the theorem
  ~bv[]
  (defthm count-and-collect-theorem
    (equal (count-and-collect1 x 0 nil nil) (count-and-collect x))).
  ~ev[]
  Hint:  the inductive proof requires attacking a more general
  theorem.

  ACL2 does not support the Common Lisp construct
  ~c[multiple-value-bind], whose logical meaning seems difficult to
  characterize.  ~c[Mv-let] is the ACL2 analogue of that construct.~/"

  (declare (xargs :guard (and (>= (length rst) 3)
                              (true-listp (car rst))
                              (>= (length (car rst)) 2))))
  #+acl2-loop-only
  (list* 'let
         (make-mv-nths (car rst)
                       (list 'mv-list (length (car rst)) (cadr rst))
                       0)
         (cddr rst))
  #+(and (not acl2-loop-only) acl2-mv-as-values)
  (return-from mv-let (cons 'multiple-value-bind rst))
  #+(and (not acl2-loop-only) (not acl2-mv-as-values))
  (cond ((> (length (car rst)) (+ 1 *number-of-return-values*))
         (interface-er
          "Need more *return-values*.  Increase ~
           *number-of-return-values* and recompile ACL2."))
        (t
         `(let ((,(car (car rst)) ,(cadr rst))
                (,(cadr (car rst)) (mv-ref 1))
                ,@(let (ans)
                    (do ((tl (cddr (car rst)) (cdr tl))
                         (i 2 (1+ i)))
                        ((null tl))
                        (push (list (car tl) `(mv-ref ,i))
                              ans))
                    (nreverse ans)))
            ,@ (cddr rst)))))

#+acl2-loop-only
(defun mv-list (input-arity x)

  ":Doc-Section ACL2::Programming

  converting multiple-valued result to a single-valued list~/
  ~bv[]
  Example Forms:
  ; Returns the list (3 4):
  (mv-list 2 (mv 3 4))

  ; Returns a list containing the three values returned by var-fn-count:
  (mv-list 3 (var-fn-count '(cons (binary-+ x y) z) nil))~/

  General form:
  (mv-list n term)
  ~ev[]

  Logically, ~c[(mv-list n term)] is just ~c[term]; that is, in the logic
  ~c[mv-list] simply returns its second argument.  However, the evaluation of a
  call of ~c[mv-list] on explicit values always results in a single value,
  which is a (null-terminated) list.  For evaluation, the term ~c[n] above (the
  first argument to an ~c[mv-list] call) must ``essentially'' (see below) be an
  integer not less than 2, where that integer is the number of values returned
  by the evaluation of ~c[term] (the second argument to that ~c[mv-list] call).

  We say ``essentially'' above because it suffices that the translation of
  ~c[n] to a term (~pl[trans]) be of the form ~c[(quote k)], where ~c[k] is an
  integer greater than 1.  So for example, if ~c[term] above returns three
  values, then ~c[n] can be the expression ~c[3], or ~c[(quote 3)], or even
  ~c[(mac 3)] if ~c[mac] is a macro defined by ~c[(defmacro mac (x) x)].  But
  ~c[n] cannot be ~c[(+ 1 2)], because even though that expression evaluates to
  ~c[3], nevertheless it translates to ~c[(binary-+ '1 '2)], not to
  ~c[(quote 3)].

  ~c[Mv-list] is the ACL2 analogue of the Common Lisp construct
  ~c[multiple-value-list].~/"

  (declare (xargs :guard t

; WARNING: Do not remove :mode :logic, or we may wind up with the wrong *1*
; definition of mv-list!  See the comment about mv-list in add-trip.

                  :mode :logic)
           (ignore input-arity))
  x)

#+(and (not acl2-loop-only) acl2-mv-as-values)
(defmacro mv-list (input-arity x)
  (declare (ignore input-arity))
  `(multiple-value-list ,x))

#+(and (not acl2-loop-only) (not acl2-mv-as-values))
(defmacro mv-list (input-arity x)
  `(cons ,x (mv-refs (1- ,input-arity))))

(deflabel state
  :doc
  ":Doc-Section Miscellaneous

  the von Neumannesque ACL2 state object~/

  The ACL2 state object is used extensively in programming the ACL2
  system, and has been used in other ACL2 programs as well.  However,
  most users, especially those interested in specification and
  verification (as opposed to programming ~i[per se]), need not be
  aware of the role of the state object in ACL2, and will not write
  functions that use it explicitly.  We say more about this point at
  the end of this documentation topic.

  The ACL2 state object is an example of a single-threaded object or
  ~il[stobj].  ACL2 allows the user to define new single-threaded objects.
  Generally, ACL2 may need to access the ACL2 state but should not
  (cannot) change it except via a certain set of approved functions
  such as ~ilc[defun] and ~ilc[defthm].  If you need a state-like object
  to which you have complete rights, you may want a ~il[stobj].

  Key to the idea of our ~c[state] is the notion of single-threadedness.
  For an explanation, ~pl[stobj].  The upshot of it is that ~c[state]
  is a variable symbol with severe restrictions on its use, so that it
  can be passed into only certain functions in certain slots, and must be
  returned by those functions that ``modify'' it.  Henceforth, we do not
  discuss single-threaded objects in general (which the user can introduce
  with ~ilc[defstobj]) but one in particular, namely ACL2's ~c[state] object.

  The ~i[global table] is perhaps the most visible portion of the state
  object.  Using the interface functions ~c[@] and ~c[assign], a user
  may bind global variables to the results of function evaluations
  (much as an Nqthm user exploits the Nqthm utility ~c[r-loop]).
  ~l[@], and ~pl[assign].~/

  ACL2 supports several facilities of a truly von Neumannesque state
  machine character, including file ~il[io] and global variables.
  Logically speaking, the state is a true list of the 14 components
  described below.  There is a ``current'' state object at the
  top-level of the ACL2 ~il[command] loop.  This object is understood to be
  the value of what would otherwise be the free variable ~c[state]
  appearing in top-level input.  When any ~il[command] returns a state
  object as one of its values, that object becomes the new current
  state.  But ACL2 provides von Neumann style speed for state
  operations by maintaining only one physical (as opposed to logical)
  state object.  Operations on the state are in fact destructive.
  This implementation does not violate the applicative semantics
  because we enforce certain draconian syntactic rules regarding the
  use of state objects.  For example, one cannot ``hold on'' to an old
  state, access the components of a state arbitrarily, or ``modify'' a
  state object without passing it on to subsequent state-sensitive
  functions.

  Every routine that uses the state facilities (e.g. does ~il[io], or calls
  a routine that does ~il[io]), must be passed a ``state object.'' And a
  routine must return a state object if the routine modifies the state
  in any way.  Rigid syntactic rules governing the use of state
  objects are enforced by the function ~c[translate], through which all
  ACL2 user input first passes.  State objects can only be ``held'' in
  the formal parameter ~c[state], never in any other formal parameter and
  never in any structure (excepting a multiple-value return list
  field which is always a state object).  State objects can only be
  accessed with the primitives we specifically permit.  Thus, for
  example, one cannot ask, in code to be executed, for the length of
  ~c[state] or the ~ilc[car] of ~c[state].  In the statement and proof of theorems,
  there are no syntactic rules prohibiting arbitrary treatment of
  state objects.

  Logically speaking, a state object is a true list whose members
  are as follows:~bq[]

  ~c[Open-input-channels], an alist with keys that are symbols in
  package ~c[\"ACL2-INPUT-CHANNEL\"].  The value (~ilc[cdr]) of each pair has
  the form ~c[((:header type file-name open-time) . elements)], where
  ~c[type] is one of ~c[:character], ~c[:byte], or ~c[:object] and ~c[elements] is a
  list of things of the corresponding ~c[type], i.e. characters,
  integers of type ~c[(mod 255)], or lisp objects in our theory.
  ~c[File-name] is a string.  ~c[Open-time] is an integer.  ~l[io].

  ~c[Open-output-channels], an alist with keys that are symbols in
  package ~c[\"ACL2-OUTPUT-CHANNEL\"].  The value of a pair has the form
  ~c[((:header type file-name open-time) .  current-contents)].
  ~l[io].

  ~c[Global-table], an alist associating symbols (to be used as ``global
  variables'') with values.  ~l[@], and ~pl[assign].

  ~c[T-stack], a list of arbitrary objects accessed and changed by the
  functions ~c[aref-t-stack] and ~c[aset-t-stack].

  ~c[32-bit-integer-stack], a list of arbitrary 32-bit-integers accessed
  and changed by the functions ~c[aref-32-bit-integer-stack] and
  ~c[aset-32-bit-integer-stack].

  ~c[Big-clock-entry], an integer, that is used logically to bound the
  amount of effort spent to evaluate a quoted form.

  ~c[Idates], a list of dates and times, used to implement the function
  ~c[print-current-idate], which prints the date and time.

  ~c[Acl2-oracle], a list of objects, used for example to implement the
  functions that let ACL2 report how much time was used, but inaccessible to
  the user.  Also ~pl[with-prover-time-limit].

  ~c[File-clock], an integer that is increased on every file opening and
  closing, and on each call of ~ilc[sys-call], and is used to maintain the
  consistency of the ~ilc[io] primitives.

  ~c[Readable-files], an alist whose keys have the form
  ~c[(string type time)], where ~ilc[string] is a file name and ~c[time] is
  an integer.  The value associated with such a key is a list of
  characters, bytes, or objects, according to ~c[type].  The ~c[time] field
  is used in the following way:  when it comes time to open a file for
  input, we will only look for a file of the specified name and ~c[type]
  whose time field is that of ~c[file-clock].  This permits us to have
  a ``probe-file'' aspect to ~c[open-file]: one can ask for a file,
  find it does not exist, but come back later and find that it does
  now exist.

  ~c[Written-files], an alist whose keys have the form
  ~c[(string type time1 time2)], where ~ilc[string] is a file name,
  ~c[type] is one of ~c[:character], ~c[:byte] or ~c[:object], and
  ~c[time1] and ~c[time2] are integers.  ~c[Time1] and ~c[time2]
  correspond to the ~c[file-clock] time at which the channel for the
  file was opened and closed.  This field is write-only; the only
  operation that affects this field is ~c[close-output-channel], which
  ~ilc[cons]es a new entry on the front.

  ~c[Read-files], a list of the form ~c[(string type time1 time2)], where
  ~ilc[string] is a file name and ~c[time1] and ~c[time2] were the times at which
  the file was opened for reading and closed.  This field is write
  only.

  ~c[Writeable-files], an alist whose keys have the form
  ~c[(string type time)].  To open a file for output, we require that
  the name, type, and time be on this list.

  ~c[List-all-package-names-lst], a list of ~c[true-listps].  Roughly
  speaking, the ~ilc[car] of this list is the list of all package names
  known to this Common Lisp right now and the ~ilc[cdr] of this list is
  the value of this ~c[state] variable after you look at its ~ilc[car].
  The function, ~c[list-all-package-names], which takes the state as an
  argument, returns the ~ilc[car] and ~ilc[cdr]s the list (returning a new state
  too).  This essentially gives ACL2 access to what is provided by
  CLTL's ~c[list-all-packages].  ~ilc[Defpkg] uses this feature to ensure that
  the about-to-be-created package is new in this lisp.  Thus, for
  example, in ~c[akcl] it is impossible to create the package
  ~c[\"COMPILER\"] with ~ilc[defpkg] because it is on the list, while in Lucid
  that package name is not initially on the list.

  ~c[User-stobj-alist], an alist which associates user-defined single-threaded
  objects (~pl[stobj]) with their values.
  ~eq[]

  We recommend avoiding the use of the state object when writing ACL2
  code intended to be used as a formal model of some system, for
  several reasons.  First, the state object is complicated and
  contains many components that are oriented toward implementation and
  are likely to be irrelevant to the model in question.  Second, there
  is currently not much support for reasoning about ACL2 functions
  that manipulate the state object, beyond their logical definitions.
  Third, the documentation about state is not as complete as one might wish.

  User-defined single-threaded objects offer the speed of ~c[state] while
  giving the user complete access to all the fields.  ~l[stobj].~/")

(defun update-nth (key val l)

  ":Doc-Section ACL2::Programming

  modify a list by putting the given value at the given position~/

  ~c[(Update-nth key val l)] returns a list that is the same as the
  list ~c[l], except that the value at the ~c[0]-based position ~c[key]
  (a natural number) is ~c[val].~/

  If ~c[key] is an integer at least as large as the length of ~c[l], then
  ~c[l] will be padded with the appropriate number of ~c[nil] elements,
  as illustrated by the following example.
  ~bv[]
  ACL2 !>(update-nth 8 'z '(a b c d e))
  (A B C D E NIL NIL NIL Z)
  ~ev[]
  We have the following theorem.
  ~bv[]
  (implies (and (true-listp l)
                (integerp key)
                (<= 0 key))
           (equal (length (update-nth key val l))
                  (if (< key (length l))
                      (length l)
                    (+ 1 key))))
  ~ev[]

  The ~il[guard] of ~c[update-nth] requires that its first (position)
  argument is a natural number and its last (list) argument is a true
  list.~/"

  (declare (xargs :guard (true-listp l))
           (type (integer 0 *) key))
  (cond ((zp key)
         (cons val (cdr l)))
        (t (cons (car l)
                 (update-nth (1- key) val (cdr l))))))

; Rockwell Addition:

(defun update-nth-array (j key val l)
  (declare (xargs :guard (and (integerp j)
                              (integerp key)
                              (<= 0 j)
                              (<= 0 key)
                              (true-listp l)
                              (true-listp (nth j l)))))
  (update-nth j (update-nth key val (nth j l)) l))

; The following defmacro forms may speed up 32-bit-integerp a little.

(defmacro maximum-positive-32-bit-integer ()
  *maximum-positive-32-bit-integer*)

(defmacro maximum-positive-32-bit-integer-minus-1 ()
  (+ (- *maximum-positive-32-bit-integer*) -1))

(defun 32-bit-integerp (x)
  (declare (xargs :guard t))
  (and (integerp x)
       (<= x (maximum-positive-32-bit-integer))
       (>= x (maximum-positive-32-bit-integer-minus-1))))

(defthm 32-bit-integerp-forward-to-integerp
  (implies (32-bit-integerp x)
           (integerp x))
  :rule-classes :forward-chaining)

(defun rational-listp (l)

  ":Doc-Section ACL2::Programming

  recognizer for a true list of rational numbers~/

  The predicate ~c[rational-listp] tests whether its argument is a true
  list of rational numbers.~/~/"

  (declare (xargs :guard t))
  (cond ((atom l)
         (eq l nil))
        (t (and (rationalp (car l))
                (rational-listp (cdr l))))))

(defthm rational-listp-forward-to-true-listp
  (implies (rational-listp x)
           (true-listp x))
  :rule-classes :forward-chaining)

;; RAG - This function is analogous to rational-listp.

#+:non-standard-analysis
(defun real-listp (l)
  (declare (xargs :guard t))
  (cond ((atom l)
         (eq l nil))
        (t (and (realp (car l))
                (real-listp (cdr l))))))

(defdoc real-listp
  ":Doc-Section ACL2::Real

  ACL2(r) recognizer for a true list of real numbers~/

  The predicate ~c[real-listp] tests whether its argument is a true
  list of real numbers.  This predicate is only defined in ACL2(r)
  (~pl[real]).~/~/")

;; RAG - Standard forward chaining theorem about <type>-listp.

#+:non-standard-analysis
(defthm real-listp-forward-to-true-listp
  (implies (real-listp x)
           (true-listp x))
  :rule-classes :forward-chaining)

(defun integer-listp (l)

  ":Doc-Section ACL2::Programming

  recognizer for a true list of integers~/

  The predicate ~c[integer-listp] tests whether its argument is a true
  list of integers.~/~/"

  (declare (xargs :guard t))
  (cond ((atom l)
         (equal l nil))
        (t (and (integerp (car l))
                (integer-listp (cdr l))))))

(defthm integer-listp-forward-to-rational-listp
  (implies (integer-listp x)
           (rational-listp x))
  :rule-classes :forward-chaining)

;; RAG - Analogous to the forward rule from integers to rationals.

#+:non-standard-analysis
(defthm rational-listp-forward-to-real-listp
  (implies (rational-listp x)
           (real-listp x))
  :rule-classes :forward-chaining)

(defun 32-bit-integer-listp (l)
  (declare (xargs :guard t))
  (cond ((atom l) (equal l nil))
        (t (and (32-bit-integerp (car l))
                (32-bit-integer-listp (cdr l))))))

(defthm 32-bit-integer-listp-forward-to-integer-listp
  (implies (32-bit-integer-listp x)
           (integer-listp x))
  :rule-classes :forward-chaining)

; Observe that even though we are defining the primitive accessors and
; updaters for states, we do not use the formal parameter STATE as an
; argument.  This is discussed in STATE-STATE below.

(defun open-input-channels (st)
  (declare (xargs :guard (true-listp st)))
  (nth 0 st))

(defun update-open-input-channels (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 0 x st))

(defun open-output-channels (st)
  (declare (xargs :guard (true-listp st)))
  (nth 1 st))

(defun update-open-output-channels (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 1 x st))

(defun global-table (st)
  (declare (xargs :guard (true-listp st)))
  (nth 2 st))

(defun update-global-table (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 2 x st))

(defun t-stack (st)
  (declare (xargs :guard (true-listp st)))
  (nth 3 st))

(defun update-t-stack (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 3 x st))

(defun 32-bit-integer-stack (st)
  (declare (xargs :guard (true-listp st)))
  (nth 4 st))

(defun update-32-bit-integer-stack (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 4 x st))

(defun big-clock-entry (st)
  (declare (xargs :guard (true-listp st)))
  (nth 5 st))

(defun update-big-clock-entry (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 5 x st))

(defun idates (st)
  (declare (xargs :guard (true-listp st)))
  (nth 6 st))

(defun update-idates (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 6 x st))

(defun acl2-oracle (st)
  (declare (xargs :guard (true-listp st)))
  (nth 7 st))

(defun update-acl2-oracle (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 7 x st))

(defun file-clock (st)
  (declare (xargs :guard (true-listp st)))
  (nth 8 st))

(defun update-file-clock (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 8 x st))

(defun readable-files (st)
  (declare (xargs :guard (true-listp st)))
  (nth 9 st))

(defun written-files (st)
  (declare (xargs :guard (true-listp st)))
  (nth 10 st))

(defun update-written-files (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 10 x st))

(defun read-files (st)
  (declare (xargs :guard (true-listp st)))
  (nth 11 st))

(defun update-read-files (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 11 x st))

(defun writeable-files (st)
  (declare (xargs :guard (true-listp st)))
  (nth 12 st))

(defun list-all-package-names-lst (st)
  (declare (xargs :guard (true-listp st)))
  (nth 13 st))

(defun update-list-all-package-names-lst (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 13 x st))

; We use the name ``user-stobj-alist1'' below so that we can reserve the
; name ``user-stobj-alist'' for the same function but which is known to
; take STATE as its argument.  See the discussion of STATE-STATE.

(defun user-stobj-alist1 (st)
  (declare (xargs :guard (true-listp st)))
  (nth 14 st))

(defun update-user-stobj-alist1 (x st)
  (declare (xargs :guard (true-listp st)))
  (update-nth 14 x st))


; Warning:  The following list must satisfy the predicate ordered-symbol-alistp
; above if build-state is to built a state-p.

#-acl2-mv-as-values
(defconst *initial-raw-arity-alist*

; The list below is used for printing raw mode results.  It should include any
; functions that we know have arity 1 (in the sense of mv) but are not in
; *common-lisp-symbols-from-main-lisp-package*.

; The symbol :last means that the number of values returned by the call is the
; number of values returned by the last argument.

  '((er-progn . :last)
    (eval-when . :last) ; needed?
    (let . :last)
    (let* . :last)
    (make-event . 3)
    (mv-let . :last)
    (prog2$ . :last)
    (progn . :last)
    (the . :last) ; needed?
    (time . :last)
    (trace . 1)
    (untrace . 1)
    (set-raw-mode-on . 3)
    (set-raw-mode-off . 3)
    (mv-list . 1)))

(defconst *initial-checkpoint-processors*

; This constant is used in the implementation of proof-trees.

; We have removed preprocess-clause and simplify-clause because they are
; clearly not checkpoint processors; settled-down-clause, because it shouldn't
; come up anyhow; and :forcing-round, which should not be included unless
; special provision is made for forcing rounds that do not start with this
; marker.  Note that :induct is not a real processor, but rather will be a
; marker pointing to the start of the inductive proof of a pushed goal (in
; particular, to the induction scheme).

  '(eliminate-destructors-clause
    fertilize-clause
    generalize-clause
    eliminate-irrelevance-clause
    push-clause
    :induct))

(defconst *primitive-program-fns-with-raw-code*

; This is the list of :program mode functions generated by
; fns-different-wrt-acl2-loop-only in acl2-check.lisp.  We have added comments
; to give a sense of why these functions have #-acl2-loop-only code.

; Functions in this list should be executed only in raw Lisp, hence perhaps not
; in safe-mode.  See the case of 'program-only-er in ev-fncall-msg.

; This list is used in defining state global 'program-fns-with-raw-code, whose
; value has been expanded significantly (from the corresponding old state
; global, 'built-in-program-mode-fns) in Version_3.4.  If errors are caused by
; attempting to call some of these functions in safe-mode, consider adding such
; functions to the list *oneify-primitives*.

  '(relieve-hyp-synp ; *deep-gstack*
    apply-abbrevs-to-lambda-stack1 ; *nth-update-tracingp*
    nth-update-rewriter ; *nth-update-tracingp*
    ev-w-lst ; *the-live-state*
    simplify-clause1 ; dmr-flush
    ev-rec-acl2-unwind-protect ; *acl2-unwind-protect-stack*
    allocate-fixnum-range ; *the-live-state*
    trace$-fn-general ; trace
    ev-fncall! ; apply
    open-trace-file-fn ; *trace-output*
    set-trace-evisc-tuple ; *trace-evisc-tuple*
    ev-fncall-w ; *the-live-state*
    ev-rec ; time
    setup-simplify-clause-pot-lst1 ; dmr-flush
    save-exec ; save-exec-raw, etc.
    cw-gstack-fn ; *deep-gstack*
    recompress-global-enabled-structure ; (get name 'acl2-array)
    ev-w ; *the-live-state*
    verbose-pstack ; *verbose-pstk*
    user-stobj-alist-safe ; chk-user-stobj-alist
    comp-fn ; compile-uncompiled-defuns
    fmt-ppr ; print-infix
    get-memo ; *nu-memos*
    acl2-raw-eval ; eval
    pstack-fn ; *pstk-stack*
    dmr-start-fn ; dmr-start-fn-raw
    memo-exit ; *nu-memos*
    memo-key1 ; *nu-memos*
    sys-call-status ; *last-sys-call-status*
    ev-fncall-meta ; *metafunction-context*
    set-debugger-enable-fn ; lisp::*break-enable* and *debugger-hook*
    ld-loop ; *ld-level*
    print-summary ; dmr-flush
    ev ; *ev-shortcut-okp*
    ev-lst ; *ev-shortcut-okp*
    allegro-allocate-slowly-fn ; sys:gsgc-parameter
    certify-book-fn ; si::sgc-on
    translate11-flet-alist1 ; special-form-or-op-p
    include-book-fn1
    include-book-fn
    fmt1 ; finish-output
    flsz ; flatsize-infix
    set-w ; retract-world1, extend-world1, ...
    prove-loop ; *deep-gstack*
    chk-virgin ; chk-virgin2
    w-of-any-state ; live-state-p
    lambda-abstract ; *lambda-abstractp*
    ld-fn-body ; reset-parallelism-variables, *first-entry-to-ld-fn-body-flg*
    untranslate ; *the-live-state*
    longest-common-tail-length-rec ; eq
    compile-function ; compile
    untranslate-lst ; *the-live-state*
    ev-synp ; *metafunction-context*
    add-polys ; *add-polys-counter*
    dmr-stop-fn ; dmr-stop-fn-raw
    ld-print-results ; print-infix
    apply-abbrevs-to-lambda-stack ; *nth-update-tracingp*
    break$ ; break
    flpr ; print-flat-infix
    close-trace-file-fn ; *trace-output*
    ev-fncall-rec ; raw-ev-fncall
    sys-call ; system-call
    ev-fncall ; live-state-p
    ld-fn0 ; *acl2-unwind-protect-stack*, etc.
    ld-fn  ; unwind-protect
    write-expansion-file ; compile-uncompiled-*1*-defuns
    latch-stobjs1 ; eq
    chk-package-reincarnation-import-restrictions ; [-restrictions2 version]
    untrace$-fn1 ; eval
    bdd-top ; (GCL only) si::sgc-on
    defstobj-field-fns-raw-defs ; call to memoize-flush when #+hons
    expansion-alist-pkg-names
    times-mod-m31 ; gcl has raw code
    print-call-history
    iprint-ar-aref1
    prove ; #+write-arithmetic-goals
    make-event-fn
    oops-warning
    checkpoint-world
    ubt-prehistory-fn
    get-declaim-list
    pathname-unix-to-os
    hcomp-build-from-portcullis
    defconst-val
    ))

(defconst *primitive-logic-fns-with-raw-code*

; This is the list of :logic mode functions generated by
; fns-different-wrt-acl2-loop-only.  We have commented on those functions whose
; #-acl2-loop-only code has side effects.  (Side effects are presumably the
; only issue, since functionally the #-acl2-loop-only code had better implement
; the logic code!)  We use lower-case when we can live with the
; #+acl2-loop-only code and upper case when we can't.

  '(mod-expt ; (GCL only) si::powm
    header
    search-fn
    state-p1 ; LIVE-STATE-P
    aref2 ; slow-array-warning
    aref1 ; slow-array-warning
    mfc-ancestors ; *metafunction-context*
    fgetprop ; EQ, GET, ...
    getenv$ ; GETENV$-RAW
    wormhole-eval ; *WORMHOLE-STATUS-ALIST*
    wormhole1 ; *WORMHOLEP*, ...
    get-wormhole-status ; *WORMHOLE-STATUS-ALIST*
    aset2 ; [seems like we can live with logic code]
    sgetprop ; SGETPROP1
    setenv$ ; SI::SETENV ...
    getprops ; EQ, GET, ...
    compress1 ; [seems like we can live with logic code]
    time-limit4-reached-p ; THROW
    fmt-to-comment-window ; *THE-LIVE-STATE*
    len ; len1
    mfc-clause ; *metafunction-context*
    cpu-core-count ; CORE-COUNT-RAW
    nonnegative-integer-quotient ; floor
    check-print-base ; PRIN1-TO-STRING
    retract-world ; RETRACT-WORLD1
    aset1 ; [seems like we can live with logic code]
    array1p ; get [seems like we can live with logic code]
    boole$ ; boole
    array2p ; [seems like we can live with logic code]
    strip-cdrs ; strip-cdrs1
    compress2 ; [seems like we can live with logic code]
    strip-cars ; strip-cars1
    plist-worldp ; *the-live-state* (huge performance penalty?)
    wormhole-p ; *WORMHOLEP*
    mfc-type-alist ; *metafunction-context*
    may-need-slashes-fn ;*suspiciously-first-numeric-array* ...
    fmt-to-comment-window! ; *THE-LIVE-STATE*
    has-propsp ; EQ, GET, ...
    hard-error ; *HARD-ERROR-RETURNS-NILP*, FUNCALL, ...
    abort! p! ; THROW
    mfc-rdepth ; *metafunction-context*
    flush-compress ; SETF [may be critical for correctness]
    alphorder ; [bad atoms]
    extend-world ; EXTEND-WORLD1

; The following have arguments of state-state, and hence may not be of concern
; since presumably users cannot define these redundantly anyhow.  But we go
; ahead and include them, just to be safe.

    user-stobj-alist read-acl2-oracle update-user-stobj-alist
    decrement-big-clock put-global close-input-channel makunbound-global
    open-input-channel-p1 boundp-global1 global-table-cars1 extend-t-stack
    list-all-package-names close-output-channel write-byte$ shrink-t-stack
    aset-32-bit-integer-stack get-global 32-bit-integer-stack-length1
    extend-32-bit-integer-stack aset-t-stack aref-t-stack read-char$
    aref-32-bit-integer-stack open-output-channel-p1 read-object
    big-clock-negative-p peek-char$ shrink-32-bit-integer-stack read-run-time
    read-byte$ read-idate t-stack-length1 print-object$

    ec-call prog2$ mv-list must-be-equal with-prover-time-limit
    with-guard-checking

; It is tempting to add time$-logic, but it has no raw code!

; The following were discovered after we included functions defined in
; #+acl2-loop-only whose definitions are missing (or defined with
; defun-one-output) in #-acl-loop-only.

    ZPF IDENTITY ENDP NTHCDR LAST REVAPPEND NULL BUTLAST STRING MEMBER NOT
    MOD PLUSP ATOM LISTP ZP FLOOR CEILING TRUNCATE ROUND REM REMOVE
    REMOVE-DUPLICATES LOGBITP ASH LOGCOUNT SIGNUM INTEGER-LENGTH EXPT
    SUBSETP SUBSTITUTE ZEROP MINUSP ODDP EVENP = /= MAX MIN CONJUGATE
    LOGANDC1 LOGANDC2 LOGNAND LOGNOR LOGNOT LOGORC1 LOGORC2 LOGTEST
    POSITION ABS STRING-EQUAL STRING< STRING> STRING<= STRING>=
    STRING-UPCASE STRING-DOWNCASE KEYWORDP EQ EQL CHAR SUBST SUBLIS
    ACONS ASSOC RASSOC NTH SUBSEQ LENGTH REVERSE ZIP STANDARD-CHAR-P
    ALPHA-CHAR-P UPPER-CASE-P LOWER-CASE-P CHAR< CHAR> CHAR<= CHAR>=
    CHAR-EQUAL CHAR-UPCASE CHAR-DOWNCASE
    AND-LIST OR-LIST ; relevant for #+acl2-par

; Might as well add additional ones below:

    random$
    throw-nonexec-error
    gc$-fn
    set-compiler-enabled
    good-bye-fn ; exit-lisp
    assoc-eq assoc-equal
    member-eq member-equal
    subsetp-eq subsetp-equal
    remove-eq remove-equal
    position-eq position-equal
    take
    canonical-pathname
  ))

(defconst *primitive-macros-with-raw-code*

; This list is generated by fns-different-wrt-acl2-loop-only.

  '(mbe ; some Lisps, not all
    theory-invariant
    set-let*-abstractionp defaxiom
    set-bogus-mutual-recursion-ok
    set-ruler-extenders
    delete-include-book-dir certify-book progn!
    f-put-global push-untouchable
    set-backchain-limit set-default-hints! set-override-hints-macro
    deftheory pstk verify-guards defchoose
    set-default-backchain-limit set-state-ok
    set-ignore-ok set-non-linearp with-output
    set-compile-fns add-include-book-dir
    clear-pstk add-custom-keyword-hint
    initial-gstack
    acl2-unwind-protect set-well-founded-relation
    catch-time-limit4 defuns add-default-hints!
    local encapsulate remove-default-hints!
    include-book pprogn set-enforce-redundancy
    set-ignore-doc-string-error
    logic er deflabel mv-let program value-triple
    set-body comp set-bogus-defun-hints-ok
    dmr-stop defpkg set-measure-function
    set-inhibit-warnings defthm mv
    f-big-clock-negative-p reset-prehistory
    mutual-recursion set-rewrite-stack-limit
    add-match-free-override
    set-match-free-default
    the-mv table in-arithmetic-theory
    set-case-split-limitations
    set-irrelevant-formals-ok remove-untouchable
    in-theory with-output-forced dmr-start
    rewrite-entry skip-proofs f-boundp-global
    make-event set-verify-guards-eagerness
    wormhole verify-termination-boot-strap start-proof-tree
    f-decrement-big-clock defstobj defund defttag
    defdoc push-gframe defthmd f-get-global
    set-nu-rewriter-mode

; The following were discovered after we included macros defined in
; #+acl2-loop-only whose definitions are missing in #-acl-loop-only.

    CAAR CADR CDAR CDDR CAAAR CAADR CADAR CADDR CDAAR CDADR CDDAR CDDDR
    CAAAAR CAAADR CAADAR CAADDR CADAAR CADADR CADDAR CADDDR CDAAAR
    CDAADR CDADAR CDADDR CDDAAR CDDADR CDDDAR CDDDDR REST MAKE-LIST
    LIST OR AND * LOGIOR LOGXOR LOGAND SEARCH LOGEQV CONCATENATE LET*
    DEFUN THE > <= >= + - / 1+ 1- PROGN DEFMACRO COND CASE LIST*
    APPEND DEFCONST IN-PACKAGE INTERN FIRST SECOND THIRD FOURTH FIFTH
    SIXTH SEVENTH EIGHTH NINTH TENTH DIGIT-CHAR-P
    UNMEMOIZE HONS-LET MEMOIZE-LET MEMOIZE ; for #+hons
    DEFUNS-STD DEFTHM-STD DEFUN-STD ; for #+:non-standard-analysis
    POR PAND PLET PARGS ; for #+acl2-par

; The following were included after Version_3.4 as ACL2 continued to evolve.

    trace!
    with-live-state
    with-output-object-channel-sharing
    time$
    with-hcomp-bindings
    with-hcomp-ht-bindings
    redef+
    redef-
    bind-acl2-time-limit
    defattach
    count
    ))

(defmacro with-live-state (form)

; Occasionally macros will generate uses of STATE, which is fine in the ACL2
; loop but can cause compiler warnings in raw Lisp.  For example, in v3-4 with
; CCL one found:

;     ? [RAW LISP] (trace$)
;     ;Compiler warnings :
;     ;   In an anonymous lambda form: Undeclared free variable STATE
;     NIL
;     NIL
;     ACL2_INVISIBLE::|The Live State Itself|
;     ? [RAW LISP] 

; The present macro is provided in order to avoid this problem:
; (with-live-state form) binds state to *the-live-state* in form after checking
; that the current value of state really is *the-live-state*, which we expect
; will always be the case.

  ":Doc-Section ACL2::Programming

  allow a reference to ~c[state] in raw Lisp~/
  ~bv[]
  Example Form:
  (with-live-state (assign y 3))

  General form:
  (with-live-state form)
  ~ev[]
  where form is an arbitrary form that mentions ~ilc[state].

  Most users will not need ~c[with-live-state].  If for some reason a form
  that mentions the variable ~ilc[state] might be executed in raw Lisp, outside
  the ACL2 loop, then the use of ~c[with-live-state] is recommended in order to
  avoid potential warnings or (less likely) errors.~/~/"

  #+acl2-loop-only
  form
  #-acl2-loop-only
  `(progn
     (or (eq (symbol-value 'state) *the-live-state*)
         (error "Implementation error:~%~p
                 Illegal use of with-live-state on state that is not live."
                ',form))
     (let ((state *the-live-state*))
       ,form)))

(defun init-iprint-ar (hard-bound enabledp)

; We return an iprint-ar with the given hard-bound.

; As stated in the Essay on Iprinting, we maintain the invariants that the
; dimension of state global 'iprint-ar exceeds the hard bound and that the
; maximum-length of the 'iprint-ar is always at least four times its dimension.

; Therefore, we need to avoid :order nil so that compress can shrink the
; array.

; We write the array ar as we do below so that (equal (compress1 'iprint-ar ar)
; ar) is T.  That probably is not important, but it may come in handy at some
; point to know that compress1 is the identity on this array.

; WARNING: Consider carefully comments in rollover-iprint-ar and
; disable-iprint-ar before changing :ORDER.

  (declare (xargs :guard (natp hard-bound)))
  (let* ((dim (1+ hard-bound)))
    `((:HEADER :DIMENSIONS     (,dim)
               :MAXIMUM-LENGTH ,(* 4 dim)
               :DEFAULT        nil
               :NAME           iprint-ar
               :ORDER          :none)
      (0 . ,(if enabledp 0 (list 0))))))

; The default bounds for iprinting are deliberately rather high, in order to
; minimize the chance that novice users attempt to read stale #@i# values.
; We assume that for those who use ACL2 with large objects, for whom iprinting
; causes a space problem because of these large bounds, will know to reset the
; bounds using set-iprint.
(defconst *iprint-soft-bound-default* 1000)
(defconst *iprint-hard-bound-default* 10000)

(defconst *initial-global-table*

; Keep this list in alphabetic order as per ordered-symbol-alistp.

; When you add a new state global to this table, consider whether to modify
; *protected-system-state-globals*.

; Note that check-state-globals-initialized insists that all state globals that
; are bound by the build are bound in this alist or in
; *initial-ld-special-bindings*.

  `((abbrev-evisc-tuple . :default)
    (accumulated-ttree . nil) ; just what succeeded; tracking the rest is hard
    (accumulated-warnings . nil)
    (acl2-raw-mode-p . nil)

    (acl2-version .

; Keep this value in sync with the value assigned to
; acl2::*copy-of-acl2-version* in file acl2.lisp.

; The reason MCL needs special treatment is that (char-code #\Newline) = 13 in
; MCL, not 10.  See also :DOC version.

; ACL2 Version 4.1

; We put the version number on the line above just to remind ourselves to bump
; the value of state global 'acl2-version, which gets printed out with the
; check-sum info.

; Leave this here.  It is read when loading acl2.lisp.  This constant should be
; a string containing at least one `.'.  The function save-acl2-in-akcl in
; akcl-init.lisp suggests that the user see :doc notexxx, where xxx is the
; substring appearing after the first `.'.

; We have occasion to write fixed version numbers in this code, that is,
; version numbers that are not supposed to be touched when we do ``version
; bump.''  The problem is that version bump tends to replace the standard
; version string with an incremented one, so we need a way to make references
; to versions in some non-standard form.  In Lisp comments we tend to write
; these with an underscore instead of a space before the number.  Thus, `ACL2
; Version_2.5' is a fixed reference to that version.  In :DOC strings we tend
; to write ACL2 Version 2.5.  Note the two spaces.  This is cool because HTML
; etc removes the redundant spaces so the output of this string is perfect.
; Unfortunately, if you use the double space convention in Lisp comments the
; double space is collapsed by ctrl-meta-q when comments are formatted.  They
; are also collapsed by `fill-format-string', so one has to be careful when
; reformatting :DOC comments.

                  ,(concatenate 'string
                                "ACL2 Version 4.1"
                                #+non-standard-analysis
                                "(r)"
                                #+(and mcl (not ccl))
                                "(mcl)"))
    (axiomsp . nil)
    (bddnotes . nil)
    (certify-book-info .

; Certify-Book-Info is non-nil when certifying a book, in which case it is the
; full-book-name of the book being certified, except if the book is known to be
; "tainted" (a certificate of some included book has the wrong version) then it
; is a one-element list containing that full-book-name.

                       nil)
    (checkpoint-forced-goals . nil) ; default in :doc
    (checkpoint-processors . ; avoid unbound var error with collect-checkpoints
                           ,*initial-checkpoint-processors*)
    (checkpoint-summary-limit . (nil . 3))
    (compiler-enabled . nil) ; Lisp-specific; set by initialize-state-globals
    (connected-book-directory . nil)  ; set-cbd couldn't have put this!
    (current-acl2-world . nil)
    (current-package . "ACL2")
    (debugger-enable . nil) ; keep in sync with :doc set-debugger-enable
    (defaxioms-okp-cert . t) ; t when not inside certify-book
    (deferred-ttag-notes . :not-deferred)
    (deferred-ttag-notes-saved . nil)
    (distributed-books-dir . nil) ; set in enter-boot-strap-mode and perhaps lp
    (dmrp . nil)
    (evisc-hitp-without-iprint . nil)
    (eviscerate-hide-terms . nil)
    (fmt-hard-right-margin . 77)
    (fmt-soft-right-margin . 65) ; same as proof-tree-buffer-width
    (gag-mode . nil)
    (gag-state . nil)
    (gag-state-saved . nil) ; saved when gag-state is set to nil
    (global-enabled-structure . nil) ; initialized in enter-boot-strap-mode
    (gstackp . nil)
    (guard-checking-on . t)
    (hons-enabled . nil) ; set in *hons-init-hook*
    (hons-read-p . t) ; only of interest in the #+hons version
    (host-lisp . ; GCL 2.6.7 can fail if instead we do the obvious thing here
               ,(let ()
                  #+gcl :gcl
                  #+ccl :ccl
                  #+sbcl :sbcl
                  #+allegro :allegro
                  #+clisp :clisp
                  #+cmu :cmu
                  #-(or gcl ccl sbcl allegro clisp cmu)
                  (illegal "The underlying host Lisp appears not to support ~
                            ACL2.  Feel free to contact the ACL2 implementors ~
                            to request such support.")))
    (in-local-flg . nil)
    (in-prove-flg . nil)
    (in-verify-flg . nil)
    (infixp . nil)                   ; See the Essay on Infix below
    (inhibit-output-lst . (summary)) ; Without this setting, initialize-acl2
                                     ; will print a summary for each event.
                                     ; Exit-boot-strap-mode sets this list
                                     ; to nil.
    (inhibit-output-lst-stack . nil)
    (inhibited-summary-types . nil)
    (iprint-ar . ,(init-iprint-ar *iprint-hard-bound-default* nil))
    (iprint-hard-bound . ,*iprint-hard-bound-default*)
    (iprint-soft-bound . ,*iprint-soft-bound-default*)
    (keep-tmp-files . nil)
    (last-make-event-expansion . nil)
    (ld-level . 0)
    (ld-redefinition-action . nil)
    (ld-skip-proofsp . nil)
    (logic-fns-with-raw-code . ,*primitive-logic-fns-with-raw-code*)
    (macros-with-raw-code . ,*primitive-macros-with-raw-code*)
    (main-timer . 0)
    (make-event-debug . nil)
    (make-event-debug-depth . 0)
    (match-free-error . nil) ; if t, modify :doc for set-match-free-error
    (modifying-include-book-dir-alist . nil)
    (more-doc-max-lines . 45)
    (more-doc-min-lines . 35)
    (more-doc-state . nil)
    (mswindows-drive . nil) ; for #+mswindows, to be set at start-up
    (parallel-evaluation-enabled . ; GCL 2.6.6 breaks with only 2 lines below
                                 #+acl2-par
                                 t
                                 #-acl2-par
                                 nil)
    (pc-output . nil)
    (ppr-flat-right-margin . 40)
    (print-base . 10)
    (print-case . :upcase)
    (print-circle . nil)
    (print-circle-files . t) ; set to nil for #+gcl in LP
    (print-clause-ids . nil)
    (print-doc-start-column . 15)
    (print-escape . t)
    (print-length . nil)
    (print-level . nil)
    (print-lines . nil)
    (print-pretty . nil) ; default in Common Lisp is implementation dependent
    (print-radix . nil)
    (print-readably . nil)
    (print-right-margin . nil)
    (program-fns-with-raw-code . ,*primitive-program-fns-with-raw-code*)
    (prompt-function . default-print-prompt)
    (prompt-memo . nil)
    (proof-tree . nil)
    (proof-tree-buffer-width . 65) ; same as fmt-soft-right-margin
    (proof-tree-ctx . nil)
    (proof-tree-indent . "|  ")
    (proof-tree-start-printed . nil)
    (proofs-co . acl2-output-channel::standard-character-output-0)
    (raw-arity-alist . nil)
    (raw-include-book-dir-alist . :ignore)
    (raw-proof-format . nil)
    (redo-flat-fail . nil)
    (redo-flat-succ . nil)
    (redundant-with-raw-code-okp . nil)
    (retrace-p . nil)
    (safe-mode . nil)
    (save-expansion-file . nil)
    (saved-output-p . nil)
    (saved-output-reversed . nil)
    (saved-output-token-lst . nil)
    (show-custom-keyword-hint-expansion . nil)
    (skip-notify-on-defttag . nil)
    (skip-proofs-by-system . nil)
    (skip-proofs-okp-cert . t) ; t when not inside certify-book
    (slow-array-action . :break) ; set to :warning in exit-boot-strap-mode
    (standard-co . acl2-output-channel::standard-character-output-0)
    (standard-oi . acl2-output-channel::standard-object-input-0)
    (tainted-okp . nil)
    (temp-touchable-fns . nil)
    (temp-touchable-vars . nil)
    (term-evisc-tuple . :default)
    (timer-alist . nil)
    (tmp-dir . nil) ; set by lp; user-settable but not much advertised.
    (trace-co . acl2-output-channel::standard-character-output-0)
    (trace-level . 0)
    (trace-specs . nil)
    (triple-print-prefix . " ")
    (ttags-allowed . :all)
    (undone-worlds-kill-ring . (nil nil nil))

; By making the above list of nils be of length n you can arrange for ACL2 to
; save n worlds for undoing undos.  If n is 0, no undoing of undos is possible.
; If n is 1, the last undo can be undone.

    (user-home-dir . nil) ; set first time entering lp
    (verbose-theory-warning . t)
    (window-interface-postlude
     . "#>\\>#<\\<e(acl2-window-postlude ?~sw ~xt ~xp)#>\\>")
    (window-interface-prelude
     . "~%#<\\<e(acl2-window-prelude ?~sw ~xc)#>\\>#<\\<~sw")
    (window-interfacep . nil)
    (wormhole-name . nil)
    (wormhole-status . nil)
    (writes-okp . t)))

#+acl2-loop-only ; not during compilation
(value ; avoid value-triple, as state-global-let* is not yet defined in pass 1
 (or (ordered-symbol-alistp *initial-global-table*)
     (illegal 'top-level
              "*initial-global-table* is not an ordered-symbol-alistp!"
              nil)))

; Essay on Infix

; ACL2 has a hook for providing a different syntax.  We call this different
; syntax "infix" but it could be anything.  If the state global variable
; infixp is nil, ACL2 only supports CLTL syntax.  If infixp is non-nil
; then infix syntax may be used, depending on the context and the value of
; infixp.

; First, what is the "infix" syntax supported?  The answer is "a really stupid
; one."  In the built-in infix syntax, a well-formed expression is a dollar
; sign followed by a CLTL s-expression.  E.g., if infixp is t one must
; write $ (car (cdr '(a b c))) instead of just (car (cdr '(a b c))).  If
; infixp is t, the prover prints formulas by preceding them with a dollar
; sign.  This stupid syntax allows the ACL2 developers to test the infix
; hooks without having to invent and implement an new syntax.  Such testing
; has helped us identify places where, for example, we printed or read in
; one syntax when the other was expected by the user.

; However, we anticipate that users will redefine the infix primitives so as to
; implement interesting alternative syntax.  This note explains the primitives
; which must be redefined.  But first we discuss the values of the state
; global variable infixp.

; In addition to nil, infixp can be :in, :out or t (meaning both).  As noted,
; if infixp is nil, we use Common Lisp s-expression syntax.  If infixp is
; non-nil the syntax used depends on both infixp and on the context.  On
; printing, we use infix if infixp is t or :out.  On reading from the terminal,
; we expect infix if infixp is :in or t.  When reading from files (as in
; include-book) with infixp non-nil, we peek at the file and if it begins with

; (IN-PACKAGE "...

; optionally preceded by Lisp-style comments and whitespace, we use lisp
; syntax, otherwise infix.  The check is made with the raw Lisp function
; lisp-book-syntaxp.

; By allowing the :in and :out settings we allow users to type one and see the
; other.  We think this might help some users learn the other syntax.

; The following raw Lisp variable and functions should be redefined to
; implement an alternative infix syntax.
; 
; (defparameter *parse* ...)
; (defun parse-infix-from-terminal (eof) ...)
; (defun print-infix (x termp width rpc col file eviscp) ...)
; (defun print-flat-infix (x termp file eviscp) ...)
; (defun flatsize-infix (x termp j max eviscp) ...)

; We document each of these when we define them for the silly $ syntax.

(defun all-boundp (alist1 alist2)
  (declare (xargs :guard (and (eqlable-alistp alist1)
                              (eqlable-alistp alist2))))
  (cond ((endp alist1) t)
        ((assoc (caar alist1) alist2)
         (all-boundp (cdr alist1) alist2))
        (t nil)))

(defun known-package-alistp (x)

; Keep this in sync with make-package-entry.

  (declare (xargs :guard t))
  (cond ((atom x) (null x))
        (t (and (true-listp (car x)) ; "final cdr" of book-path is a true-listp
                (stringp (car (car x)))         ; name
                (symbol-listp (cadr (car x)))   ; imports
                (known-package-alistp (cdr x))))))

(defthm known-package-alistp-forward-to-true-list-listp-and-alistp
  (implies (known-package-alistp x)
           (and (true-list-listp x)
                (alistp x)))
  :rule-classes :forward-chaining)

(defun timer-alistp (x)

; A timer-alistp is an alist binding symbols to lists of rationals.

  (declare (xargs :guard t))
  (cond ((atom x) (equal x nil))
        ((and (consp (car x))
              (symbolp (caar x))
              (rational-listp (cdar x)))
         (timer-alistp (cdr x)))
        (t nil)))

(defthm timer-alistp-forward-to-true-list-listp-and-symbol-alistp
  (implies (timer-alistp x)
           (and (true-list-listp x)
                (symbol-alistp x)))
  :rule-classes :forward-chaining)

(defun typed-io-listp (l typ)
  (declare (xargs :guard t))
  (cond ((atom l) (equal l nil))
        (t (and (case typ
                      (:character (characterp (car l)))
                      (:byte (and (integerp (car l))
                                  (<= 0 (car l))
                                  (< (car l) 256)))
                      (:object t)
                      (otherwise nil))
                (typed-io-listp (cdr l) typ)))))

(defthm typed-io-listp-forward-to-true-listp
  (implies (typed-io-listp x typ)
           (true-listp x))
  :rule-classes :forward-chaining)

(defconst *file-types* '(:character :byte :object))

(defun open-channel1 (l)
  (declare (xargs :guard t))
  (and (true-listp l)
       (consp l)
       (let ((header (car l)))
         (and
          (true-listp header)
          (equal (length header) 4)
          (eq (car header) :header)
          (member-eq (cadr header) *file-types*)
          (stringp (caddr header))
          (integerp (cadddr header))
          (typed-io-listp (cdr l) (cadr header))))))

(defthm open-channel1-forward-to-true-listp-and-consp
  (implies (open-channel1 x)
           (and (true-listp x)
                (consp x)))
  :rule-classes :forward-chaining)

(defun open-channel-listp (l)

; The following guard seems reasonable (and is certainly necessary, or at least
; some guard is) since open-channels-p will tell us that we're looking at an
; ordered-symbol-alistp.

  (declare (xargs :guard (alistp l)))

  (if (endp l)
      t
    (and (open-channel1 (cdr (car l)))
         (open-channel-listp (cdr l)))))

(defun open-channels-p (x)
  (declare (xargs :guard t))
  (and (ordered-symbol-alistp x)
       (open-channel-listp x)))

(defthm open-channels-p-forward
  (implies (open-channels-p x)
           (and (ordered-symbol-alistp x)
                (true-list-listp x)))
  :rule-classes :forward-chaining)

(defun file-clock-p (x)
  (declare (xargs :guard t))
  (natp x))

(defthm file-clock-p-forward-to-integerp
  (implies (file-clock-p x)
           (natp x))
  :rule-classes :forward-chaining)

(defun readable-file (x)
  (declare (xargs :guard t))
  (and (true-listp x)
       (consp x)
       (let ((key (car x)))
         (and (true-listp key)
              (equal (length key) 3)
              (stringp (car key))
              (member (cadr key) *file-types*)
              (integerp (caddr key))
              (typed-io-listp (cdr x) (cadr key))))))

(defthm readable-file-forward-to-true-listp-and-consp
  (implies (readable-file x)
           (and (true-listp x)
                (consp x)))
  :rule-classes :forward-chaining)

(defun readable-files-listp (x)
  (declare (xargs :guard t))
  (cond ((atom x) (equal x nil))
        (t (and (readable-file (car x))
                (readable-files-listp (cdr x))))))

(defthm readable-files-listp-forward-to-true-list-listp-and-alistp
  (implies (readable-files-listp x)
           (and (true-list-listp x)
                (alistp x)))
  :rule-classes :forward-chaining)

(defun readable-files-p (x)
  (declare (xargs :guard t))
  (readable-files-listp x))

(defthm readable-files-p-forward-to-readable-files-listp
  (implies (readable-files-p x)
           (readable-files-listp x))
  :rule-classes :forward-chaining)

(defun written-file (x)
  (declare (xargs :guard t))
  (and (true-listp x)
       (consp x)
       (let ((key (car x)))
         (and (true-listp key)
              (equal (length key) 4)
              (stringp (car key))
              (integerp (caddr key))
              (integerp (cadddr key))
              (member (cadr key) *file-types*)
              (typed-io-listp (cdr x) (cadr key))))))

(defthm written-file-forward-to-true-listp-and-consp
  (implies (written-file x)
           (and (true-listp x)
                (consp x)))
  :rule-classes :forward-chaining)

(defun written-file-listp (x)
  (declare (xargs :guard t))
  (cond ((atom x) (equal x nil))
        (t (and (written-file (car x))
                (written-file-listp (cdr x))))))

(defthm written-file-listp-forward-to-true-list-listp-and-alistp
  (implies (written-file-listp x)
           (and (true-list-listp x)
                (alistp x)))
  :rule-classes :forward-chaining)

(defun written-files-p (x)
  (declare (xargs :guard t))
  (written-file-listp x))

(defthm written-files-p-forward-to-written-file-listp
  (implies (written-files-p x)
           (written-file-listp x))
  :rule-classes :forward-chaining)

(defun read-file-listp1 (x)
  (declare (xargs :guard t))
  (and (true-listp x)
       (equal (length x) 4)
       (stringp (car x))
       (member (cadr x) *file-types*)
       (integerp (caddr x))
       (integerp (cadddr x))))

(defthm read-file-listp1-forward-to-true-listp-and-consp
  (implies (read-file-listp1 x)
           (and (true-listp x)
                (consp x)))
  :rule-classes :forward-chaining)

(defun read-file-listp (x)
  (declare (xargs :guard t))
  (cond ((atom x) (equal x nil))
        (t (and (read-file-listp1 (car x))
                (read-file-listp (cdr x))))))

(defthm read-file-listp-forward-to-true-list-listp
  (implies (read-file-listp x)
           (true-list-listp x))
  :rule-classes :forward-chaining)

(defun read-files-p (x)
  (declare (xargs :guard t))
  (read-file-listp x))

(defthm read-files-p-forward-to-read-file-listp
  (implies (read-files-p x)
           (read-file-listp x))
  :rule-classes :forward-chaining)

(defun writable-file-listp1 (x)
  (declare (xargs :guard t))
  (and (true-listp x)
       (equal (length x) 3)
       (stringp (car x))
       (member (cadr x) *file-types*)
       (integerp (caddr x))))

(defthm writable-file-listp1-forward-to-true-listp-and-consp
  (implies (writable-file-listp1 x)
           (and (true-listp x)
                (consp x)))
  :rule-classes :forward-chaining)

(defun writable-file-listp (x)
  (declare (xargs :guard t))
  (cond ((atom x) (equal x nil))
        (t (and (writable-file-listp1 (car x))
                (writable-file-listp (cdr x))))))

(defthm writable-file-listp-forward-to-true-list-listp
  (implies (writable-file-listp x)
           (true-list-listp x))
  :rule-classes :forward-chaining)

(defun writeable-files-p (x)
  (declare (xargs :guard t))
  (writable-file-listp x))

(defthm writeable-files-p-forward-to-writable-file-listp
  (implies (writeable-files-p x)
           (writable-file-listp x))
  :rule-classes :forward-chaining)

(defun state-p1 (x)
  (declare (xargs :guard t))
  #-acl2-loop-only
  (cond ((live-state-p x)
         (return-from state-p1 t)))
  (and (true-listp x)
       (equal (length x) 15)
       (open-channels-p (open-input-channels x))
       (open-channels-p (open-output-channels x))
       (ordered-symbol-alistp (global-table x))
       (all-boundp *initial-global-table*
                   (global-table x))
       (plist-worldp (cdr (assoc 'current-acl2-world (global-table x))))
       (symbol-alistp
        (getprop 'acl2-defaults-table 'table-alist nil
                 'current-acl2-world
                 (cdr (assoc 'current-acl2-world (global-table x)))))
       (timer-alistp (cdr (assoc 'timer-alist (global-table x))))
       (known-package-alistp
        (getprop 'known-package-alist 'global-value nil
                 'current-acl2-world
                 (cdr (assoc 'current-acl2-world (global-table x)))))
       (true-listp (t-stack x))
       (32-bit-integer-listp (32-bit-integer-stack x))
       (integerp (big-clock-entry x))
       (integer-listp (idates x))
       (true-listp (acl2-oracle x))
       (file-clock-p (file-clock x))
       (readable-files-p (readable-files x))
       (written-files-p (written-files x))
       (read-files-p (read-files x))
       (writeable-files-p (writeable-files x))
       (true-list-listp (list-all-package-names-lst x))
       (symbol-alistp (user-stobj-alist1 x))))

(defthm state-p1-forward
  (implies (state-p1 x)
           (and
            (true-listp x)
            (equal (length x) 15)
            (open-channels-p (nth 0 x))
            (open-channels-p (nth 1 x))
            (ordered-symbol-alistp (nth 2 x))
            (all-boundp *initial-global-table*
                        (nth 2 x))
            (plist-worldp (cdr (assoc 'current-acl2-world (nth 2 x))))
            (symbol-alistp
             (getprop 'acl2-defaults-table 'table-alist nil
                      'current-acl2-world
                      (cdr (assoc 'current-acl2-world (nth 2 x)))))
            (timer-alistp (cdr (assoc 'timer-alist (nth 2 x))))
            (known-package-alistp
             (getprop 'known-package-alist 'global-value nil
                      'current-acl2-world
                      (cdr (assoc 'current-acl2-world (nth 2 x)))))
            (true-listp (nth 3 x))
            (32-bit-integer-listp (nth 4 x))
            (integerp (nth 5 x))
            (integer-listp (nth 6 x))
            (true-listp (nth 7 x))
            (file-clock-p (nth 8 x))
            (readable-files-p (nth 9 x))
            (written-files-p (nth 10 x))
            (read-files-p (nth 11 x))
            (writeable-files-p (nth 12 x))
            (true-list-listp (nth 13 x))
            (symbol-alistp (nth 14 x))))
  :rule-classes :forward-chaining
  ;; The hints can speed us up from over 40 seconds to less than 2.
  :hints (("Goal" :in-theory
           (disable nth length open-channels-p ordered-symbol-alistp
                    all-boundp plist-worldp assoc timer-alistp
                    known-package-alistp true-listp
                    32-bit-integer-listp integer-listp rational-listp
                    file-clock-p readable-files-p written-files-p
                    read-files-p writeable-files-p true-list-listp
                    symbol-alistp))))

(defun state-p (state-state)
  (declare (xargs :guard t))
  (state-p1 state-state))

; Let us use state-p1 in our theorem-proving.
(in-theory (disable state-p1))

; The following could conceivably be useful before rewriting a literal
; containing state-p.

(defthm state-p-implies-and-forward-to-state-p1
  (implies (state-p state-state)
           (state-p1 state-state))
  :rule-classes (:forward-chaining :rewrite))

; On STATE-STATE

; No one should imagine calling any of the state accessors or updaters
; in executable code.  These fields are all ``magic'' in some sense,
; in that they don't actually exist -- or, to put it more accurately,
; we do not represent them concretely as the ACL2 objects we alleged
; them to be in the axioms.  In some cases, we might have gone to the
; trouble of supporting these things, at considerable cost, e.g.
; keeping a giant list of all characters printed this year or code to
; recover the logical value of written-files (which shows the times at
; which channels to files were opened and closed) from the actual file
; system.  In other cases, such as big-clock-entry, the cost of
; support would have been intuitively equivalent to infinite: no ACL2.

; The user should be grateful that he can even indirectly access these
; fields at all in executable code, and should expect to study the
; means of access with excruciating pain and care.  Although the
; fields of states may be THOUGHT of as ordinary logical objects (e.g.
; in theorems), the severe restriction on runtime access to them is
; the PRICE ONE PAYS for (a) high efficiency and (b) real-time
; effects.

; How do we prevent the user from applying, say, written-files, to the
; live state?  Well, that is pretty subtle.  We simply make the formal
; parameter to written-files be ST rather than STATE.  Translate
; enforces the rule that a function may receive STATE only in a slot
; whose STOBJS-IN flag is STATE.  And, with only one exception, the
; STOBJS-IN setting is always calculated by noting which formal is
; called STATE.  So by giving written-files ST and never reseting its
; STOBJS-IN, we prevent it from being fed the live state (or any
; state) in code (such as defuns and top-level commands) where we are
; checking the use of state.  (In theorems, anything goes.)  As noted,
; this is the price one pays.

; So what is the exception to the rule that (the STATE flag in)
; STOBJS-IN is determined by STATE's position?  The exception is
; managed by super-defun-wart and is intimately tied up with the use
; of STATE-STATE.  The problem is that even though we don't permit
; written-files to be called by the user, we wish to support some
; functions (like close-output-channel) which do take state as an
; argument, which may be called by the user and which -- logically
; speaking -- are defined in terms of written-files.

; So consider close-output-channel.  We would like to make its second
; parameter be STATE.  But it must pass that parameter down to
; written-files in the logical code that defines close-output-channel.
; If that happened, we would get a translate error upon trying to
; define close-output-channel, because we would be passing STATE into
; a place (namely ST) where no state was allowed.  So we use
; STATE-STATE instead.  But while that lets close-output-channel be
; defined, it doesn't let the user apply it to state.  However, after
; the definitional principle has translated the body and during the
; course of its storage of the many properties of the newly defined
; function, it calls super-defun-wart which asks "is this one of the
; special functions I was warned about?"  If so, it sets STOBJS-IN and
; STOBJS-OUT for the function properly.  A fixed number of functions
; are so built into super-defun-wart, which knows the location of the
; state-like argument and value for each of them.  Once
; super-defun-wart has done its job, state must be supplied to
; close-output-channel, where expected.

; "But," you ask, "if state is supplied doesn't it find its way down
; to written-files and then cause trouble because written files isn't
; expecting the live state?"  Yes, it would cause trouble if it ever
; got there, but it doesn't.  Because for each of the functions that
; use STATE-STATE and are known to super-defun-wart, we provide raw
; lisp code to do the real work.  That is, there are two definitions
; of close-output-channel.  One, the logical one, is read in
; #+acl2-loop-only mode and presents the prissy logical definition in
; terms of written-files.  This definition gets processed during our
; system initialization and generates the usual properties about a
; defined function that allow us to do theorem proving about the
; function.  The other, in #-acl2-loop-only, is raw Lisp that knows
; how to close a channel when its given one in the live state.

; So the convention is that those functions (all defined in
; axioms.lisp) which (a) the user is permitted to call with real
; states but which (b) can only be logically defined in terms of calls
; to the primitive state accessors and updaters are (i) defined with
; STATE-STATE as a formal parameter, (ii) have their property list
; smashed appropriately for STOBJS-IN and STOBJS-OUT right right after
; their admission, to reflect their true state character, and (iii)
; are operationally defined with raw lisp at some level between the
; defun and the use of the primitive state accessors and updaters.

;  We need the following theorem to make sure that we cannot introduce
;  via build-state something that fails to be a state.

(defmacro build-state
  (&key open-input-channels open-output-channels global-table t-stack
        32-bit-integer-stack (big-clock '4000000) idates acl2-oracle
        (file-clock '1) readable-files written-files
        read-files writeable-files list-all-package-names-lst
        user-stobj-alist)
  (list 'build-state1
        (list 'quote open-input-channels)
        (list 'quote open-output-channels)
        (list 'quote (or global-table
                         *initial-global-table*))
        (list 'quote t-stack)
        (list 'quote 32-bit-integer-stack)
        (list 'quote big-clock)
        (list 'quote idates)
        (list 'quote acl2-oracle)
        (list 'quote file-clock)
        (list 'quote readable-files)
        (list 'quote written-files)
        (list 'quote read-files)
        (list 'quote writeable-files)
        (list 'quote list-all-package-names-lst)
        (list 'quote user-stobj-alist)))

(defconst *default-state*
  (list nil nil
        *initial-global-table*
        nil nil 4000000 nil nil 1 nil nil nil nil nil nil))

(defun build-state1 (open-input-channels
   open-output-channels global-table t-stack 32-bit-integer-stack big-clock
   idates acl2-oracle file-clock readable-files written-files
   read-files writeable-files list-all-package-names-lst user-stobj-alist)
  (declare (xargs :guard (state-p1 (list open-input-channels
   open-output-channels global-table t-stack 32-bit-integer-stack big-clock
   idates acl2-oracle file-clock readable-files written-files
   read-files writeable-files list-all-package-names-lst
   user-stobj-alist))))

; The purpose of this function is to provide a means for constructing
; a state other than the live state.

  (let ((s
         (list open-input-channels open-output-channels global-table
               t-stack 32-bit-integer-stack big-clock idates acl2-oracle
               file-clock readable-files written-files
               read-files writeable-files list-all-package-names-lst
               user-stobj-alist)))
    (cond ((state-p1 s)
           s)
          (t *default-state*))))

; Although the two following functions are only identity functions
; from the logical point of view, in the von Neumann machinery
; implementation they are potentially dangerous and should not
; be used anywhere besides trans-eval.

(defun coerce-state-to-object (x)
  (declare (xargs :guard t))
  x)

(defun coerce-object-to-state (x)
  (declare (xargs :guard t))
  x)


;                              GLOBALS

#-acl2-loop-only
(defun-one-output strip-numeric-postfix (sym)
  (coerce
   (reverse (do ((x (reverse (coerce (symbol-name sym) 'list)) (cdr x)))
                ((or (null x)
                     (eq (car x) #\-))
                 (cdr x))))
   'string))

(defun global-table-cars1 (state-state)

; Wart: We use state-state instead of state because of a bootstrap problem.

  (declare (xargs :guard (state-p1 state-state)))
  #-acl2-loop-only
  (cond ((live-state-p state-state)
         (return-from
          global-table-cars1
          (let (ans)
            (dolist (package-entry
                     (global-val 'known-package-alist (w *the-live-state*)))
                    (do-symbols (sym (find-package
                                      (concatenate 'string
                                                   *global-package-prefix*
                                                   (package-entry-name
                                                    package-entry))))
                                (cond ((boundp sym)
                                       (push (intern (symbol-name sym)
                                                     (package-entry-name
                                                      package-entry))
                                             ans)))))
            (sort ans (function symbol-<))))))
  (strip-cars (global-table state-state)))

(defun global-table-cars (state-state)

; Wart: We use state-state instead of state because of a bootstrap problem.

  (declare (xargs :guard (state-p1 state-state)))
  (global-table-cars1 state-state))

(defun boundp-global1 (x state-state)
  (declare (xargs :guard (and (symbolp x)
                              (state-p1 state-state))))
  #-acl2-loop-only
  (cond ((live-state-p state-state)
         (return-from boundp-global1 (boundp (global-symbol x)))))
  (cond ((assoc x (global-table state-state)) t)
        (t nil)))

(defun boundp-global (x state-state)

; Wart: We use state-state instead of state because of a bootstrap problem.

  (declare (xargs :guard (and (symbolp x)
                              (state-p1 state-state))))
  (boundp-global1 x state-state))

(defmacro f-boundp-global (x st)
  #-acl2-loop-only
  (cond ((and (consp x)
              (eq 'quote (car x))
              (symbolp (cadr x))
              (null (cddr x)))
         (let ((s (gensym)))
           `(let ((,s ,st))
              (declare (special ,(global-symbol (cadr x))))
              (cond ((eq ,s *the-live-state*)
                     (boundp ',(global-symbol (cadr x))))
                    (t (boundp-global ,x ,s))))))
        (t `(boundp-global ,x ,st)))
  #+acl2-loop-only
  (list 'boundp-global x st))

(defun delete-pair (x l)
  (declare (xargs :guard (and (symbolp x)
                              (eqlable-alistp l))))
  (cond ((endp l) nil)
        ((eq x (caar l))
         (cdr l))
        (t (cons (car l) (delete-pair x (cdr l))))))

(defun makunbound-global (x state-state)

; Wart: We use state-state instead of state because of a bootstrap problem.

; This function is not very fast because it calls global-symbol.  A
; faster version could easily be created.

  (declare (xargs :guard (and (symbolp x)
                              (state-p1 state-state))))
  #-acl2-loop-only
  (cond ((live-state-p state-state)
         (cond (*wormholep*
                (cond
                 ((boundp-global1 x state-state)

; If the variable is not bound, then the makunbound below doesn't do
; anything and we don't have to save undo information.  (Furthermore,
; there is nothing to save.)

                  (push-wormhole-undo-formi 'put-global x
                                            (get-global x state-state))))))
         (makunbound (global-symbol x))
         (return-from makunbound-global *the-live-state*)))
  (update-global-table (delete-pair
                        x
                        (global-table state-state))
                       state-state))

(defun get-global (x state-state)

; Wart: We use state-state instead of state because of a bootstrap problem.

  (declare (xargs :guard (and (symbolp x)
                              (state-p1 state-state)
                              (boundp-global1 x state-state))))
  #-acl2-loop-only
  (cond ((live-state-p state-state)
         (return-from get-global
                      (symbol-value (the symbol (global-symbol x))))))
  (cdr (assoc x (global-table state-state))))

(defun put-global (key value state-state)

; Wart: We use state-state instead of state because of a bootstrap problem.

  (declare (xargs :guard (and (symbolp key)
                              (state-p1 state-state))))
  #-acl2-loop-only
  (cond ((live-state-p state-state)
         (cond (*wormholep*
                (cond ((boundp-global1 key state-state)
                       (push-wormhole-undo-formi 'put-global key
                                                 (get-global key state-state)))
                      (t
                       (push-wormhole-undo-formi 'makunbound-global key nil)))))
         (setf (symbol-value (global-symbol key)) value)
         (return-from put-global state-state)))
  (update-global-table
   (add-pair key value
             (global-table state-state))
   state-state))

(defmacro f-put-global (key value st)
  #-acl2-loop-only
  (cond ((and (consp key)
              (eq 'quote (car key))
              (symbolp (cadr key))
              (null (cddr key)))
         (let ((v (gensym))
               (s (gensym)))
           `(let ((,v ,value)
                  (,s ,st))
              (cond ((live-state-p ,s)
                     (cond
                      (*wormholep*
                       (cond
                        ((boundp-global1 ,key ,s)
                         (push-wormhole-undo-formi 'put-global ,key
                                                   (get-global ,key ,s)))
                        (t
                         (push-wormhole-undo-formi 'makunbound-global
                                                   ,key
                                                   nil)))))
                     (let ()
                       (declare (special ,(global-symbol (cadr key))))
                       ,@(when (eq (cadr key) 'acl2-raw-mode-p)
                           `((observe-raw-mode-setting ,v ,s)))
                       (setq ,(global-symbol (cadr key))
                             ,v)
                       ,s))
                    (t (put-global ,key ,v ,s))))))
        (t `(put-global ,key ,value ,st)))
  #+acl2-loop-only
  (list 'put-global key value st))

; We now define state-global-let*, which lets us "bind" state
; globals.

(defun symbol-doublet-listp (lst)

; This function returns t iff lst is a true-list and each element is
; a doublet of the form (symbolp anything).

  (declare (xargs :guard t))
  (cond ((atom lst) (eq lst nil))
        (t (and (consp (car lst))
                (symbolp (caar lst))
                (consp (cdar lst))
                (null (cddar lst))
                (symbol-doublet-listp (cdr lst))))))

(defconst *initial-ld-special-bindings*

; This alist is used by initialize-acl2 to set the initial values of the LD
; specials.  It is assumed by reset-ld-specials that the first three are the
; channels.

  `((standard-oi . ,*standard-oi*)
    (standard-co . ,*standard-co*)
    (proofs-co . ,*standard-co*)
    (ld-skip-proofsp . nil)
    (ld-redefinition-action . nil)
    (ld-prompt . t)
    (ld-keyword-aliases . nil)
    (ld-pre-eval-filter . :all)
    (ld-pre-eval-print . nil)
    (ld-post-eval-print . :command-conventions)
    (ld-evisc-tuple . nil)
    (ld-error-triples . t)
    (ld-error-action . :continue)
    (ld-query-control-alist . nil)
    (ld-verbose . "~sv.  Level ~Fl.  Cbd ~xc.~|Distributed books ~
                   directory ~xb.~|Type :help for help.~%Type (good-bye) to ~
                   quit completely out of ACL2.~|~%")))

(defun always-boundp-global (x)
  (declare (xargs :guard (symbolp x)))
  (or (assoc-eq x
                *initial-global-table*)
      (assoc-eq x
                *initial-ld-special-bindings*)))

(defun state-global-let*-bindings-p (lst)

; This function returns t iff lst is a true-list and each element is
; a doublet of the form (symbolp anything) or a triplet of the form (symbolp
; anything symbolp).

  (declare (xargs :guard t))
  (cond ((atom lst) (eq lst nil))
        (t (and (consp (car lst))
                (symbolp (caar lst))
                (consp (cdar lst))
                (or (null (cddar lst))
                    (and (consp (cddar lst))
                         (symbolp (car (cddar lst)))
                         (null (cdr (cddar lst)))))
                (state-global-let*-bindings-p (cdr lst))))))

(defun state-global-let*-get-globals (bindings)

; This function is used to generate code for the macroexpansion of
; state-global-let*.  Roughly speaking, it returns a list, lst, of f-get-global
; forms that fetch the values of the variables we are about to smash.  The
; expansion of state-global-let* will start with (LET ((temp (LIST ,@lst)))
; ...) and we will use the value of temp to restore the globals after the
; execution of the body.

; Now there is a subtlety.  Some of the vars we are to "bind" might NOT be
; already bound in state.  So we don't want to call f-get-global on them until
; we know they are bound, and for those that are not, "restoring" their old
; values means making them unbound again.  So a careful specification of the
; value of temp (i.e., the value of (LIST ,@lst) where lst is what we are
; producing here) is that it is a list in 1:1 correspondence with the vars
; bound in bindings such that the element corresponding to the var x is nil if
; x is unbound in the pre-body state and is otherwise a singleton list
; containing the value of x in the pre-body state.

  (declare (xargs :guard (state-global-let*-bindings-p bindings)))
  (cond ((endp bindings) nil)
        (t (cons
            (if (always-boundp-global (caar bindings))
                `(list (f-get-global ',(caar bindings) state))
              `(if (f-boundp-global ',(caar bindings) state)
                   (list (f-get-global ',(caar bindings) state))
                 nil))
            (state-global-let*-get-globals (cdr bindings))))))

(defun state-global-let*-put-globals (bindings)

; This function is used to generate code for the macroexpansion of
; state-global-let*.  It generates a list of f-put-globals that will set the
; bound variables in bindings to their desired local values, except that
; ``setters'' are used instead where provided (see the discussion of setters in
; state-global-let*).  We insist that those initialization forms not mention
; the temporary variable state-global-let* uses to hang onto the restoration
; values.

  (declare (xargs :guard (state-global-let*-bindings-p bindings)))
  (cond ((endp bindings) nil)
        (t (cons (let ((val-form `(check-vars-not-free
                                   (state-global-let*-cleanup-lst)
                                   ,(cadar bindings))))
                   (cond ((cddr (car bindings))
                          `(if (f-boundp-global ',(caar bindings) state)
                               (,(caddr (car bindings)) ; setter
                                ,val-form
                                state)
                             (prog2$
                              (er hard 'state-global-let*
                                  "It is illegal to bind an unbound variable ~
                                   in state-global-let*, in this case, ~x0, ~
                                   when a setter function is supplied."
                                  ',(caar bindings))
                              state)))
                         (t
                          `(f-put-global ',(caar bindings)
                                         ,val-form
                                         state))))
                 (state-global-let*-put-globals (cdr bindings))))))

(defun state-global-let*-cleanup (bindings index)

; This function is used to generate code for the macroexpansion of
; state-global-let*.  We generate a list of forms that when executed will
; restore the "bound" variables to their original values, using the list of
; restoration values.  Recall that each restoration value is either a nil,
; indicating the variable was unbound, or a singleton listing the original
; value.  We are generating that code.  Index is the number of CDRs to be taken
; of the restoration values list that begins with the value for the first
; variable in bindings.  It is initially 0, to represent the temporary variable
; used by state-global-let*, and is incremented by 1 on each call so that the
; restoration values list is symbolically CDRd ever time we recurse here.

; Note: Once upon a time we used a recursive function to do the cleanup.  It
; essentially swept through the names of the state globals as it swept through
; the list of their initial values and did an f-put-global on each (here
; ignoring the unbound variable problem).  That was dangerous because it
; violated the rules that f-put-global was only called on a quoted var.  Those
; rules allow translate to enforce untouchables.  To get away with it, we had
; to exempt that function from translate's restrictions on f-put-global.  We
; thought we could regain security by then putting that function name on
; untouchables.  But since calls to that function were laid down in macros, it
; can't be untouchable if the user is to use the macros.  So we did it this
; way, which makes each f-put-global explicit and needs no special treatment.

; Finally, note that we use setters in place of f-put-global, when they are
; provided; see the discussion of setters in state-global-let*.

  (declare (xargs :guard (and (state-global-let*-bindings-p bindings)
                              (natp index))))
  (let ((cdr-expr 'state-global-let*-cleanup-lst))
    (cond ((endp bindings) nil)
          (t (cons (cond
                    ((cddr (car bindings))
                     `(,(caddr (car bindings))
                       (car (nth ,index ,cdr-expr))
                       state))
                    ((always-boundp-global (caar bindings))
                     `(f-put-global ',(caar bindings)
                                    (car (nth ,index ,cdr-expr))
                                    state))
                    (t
                     `(if (nth ,index ,cdr-expr)
                          (f-put-global ',(caar bindings)
                                        (car (nth ,index ,cdr-expr))
                                        state)
                        (makunbound-global ',(caar bindings) state))))
                   (state-global-let*-cleanup (cdr bindings)
                                              (1+ index)))))))

(defmacro state-global-let* (bindings body)

; NOTE: In April 2010 we discussed the possibility that we could simplify the
; raw-Lisp code for state-global-let* to avoid acl2-unwind-protect, in favor of
; let*-binding the state globals under the hood so that clean-up is done
; automatically by Lisp; after all, state globals are special variables.  We
; see no reason why this can't work, but we prefer not to mess with this very
; stable code unless/until there is a reason.  (Note that we however do not
; have in mind any potential change to the logic code for state-global-let*.)

; A typical use is (state-global-let* ((<var1> <form1>) ...  (<vark> <formk>))
; <body>).  Bindings thus are in the style of let* (but see the discussion of
; setters below).  Body must return an error triple.  The meaning of this form
; is to smash the global values of the "bound" variables with f-put-global,
; execute body, restore the values to their previous values, and return the
; triple produced by body (with its state as modified by the restoration).
; Because we use acl2-unwind-protect, the restoration is guaranteed even in the
; face of aborts.  The "bound" variables may initially be unbound in state and
; restoration means to make them unbound again.

; However, if any of the <vari> are untouchable then this won't work, because
; the generated calls of f-put-global are illegal.  This is sad if there is a
; ``setter'' function of the form (set-vari val state), equivalent to
; (f-put-global '<vari> val state) except that set-vari is not untouchable (as
; a function, of course).  If there is such a function symbol set-vari, then we
; can use the ``binding'' (<vari> <formi> set-vari) in place of (<vari>
; <formi>), in order to get the behavior described above even if <vari> is
; untouchable.

; Note: This function is a generalization of the now obsolete
; WITH-STATE-GLOBAL-BOUND.

  (declare (xargs :guard (and (state-global-let*-bindings-p bindings)
                              (no-duplicatesp-equal (strip-cars bindings)))))

  `(let ((state-global-let*-cleanup-lst
          (list ,@(state-global-let*-get-globals bindings))))
     ,@(and (null bindings)
            '((declare (ignore state-global-let*-cleanup-lst))))
     (acl2-unwind-protect
      "state-global-let*"
      (pprogn ,@(state-global-let*-put-globals bindings)
              (check-vars-not-free (state-global-let*-cleanup-lst) ,body))
      (pprogn
       ,@(state-global-let*-cleanup bindings 0)
       state)
      (pprogn
       ,@(state-global-let*-cleanup bindings 0)
       state))))

; With state-global-let* defined, we may now define a few more primitives and
; finish some unfinished business.

; We start by introducing functions that support type declarations.  We had to
; delay these because we use local in our proof, and local uses
; state-global-let*.  Bootstrapping is tough.  We could presumably do this
; earlier in the file and defer guard verification (which is why we need
; local), but since types are involved with guards, that seems dicey -- so we
; just wait till here.

(defun integer-range-p (lower upper x)

; Notice the strict inequality for upper.  This function was introduced in
; Version_2.7 in support of signed-byte-p and unsigned-byte-p, whose
; definitions were kept similar to those that had been in the ihs library for
; some time.

  (declare (xargs :guard (and (integerp lower) (integerp upper))))
  (and (integerp x)
       (<= lower x)
       (< x upper)))

(local (defthm natp-expt
         (implies (and (integerp base)
                       (integerp n)
                       (<= 0 n))
                  (integerp (expt base n)))
         :rule-classes :type-prescription))

; For the definitions of signed-byte-p and unsigned-byte-p, we were tempted to
; put (integerp n) and (< 0 n) [or for unsigned-byte-p, (<= 0 n)] in the guards.
; However, instead we follow the approach already used for some time in
; books/ihs/logops-definitions.lisp, namely to include these as conjuncts in
; the bodies of the definitions, an approach that seems at least as
; reasonable.

(defun signed-byte-p (bits x)
  (declare (xargs :guard t))
  (and (integerp bits)
       (< 0 bits)
       (integer-range-p (- (expt 2 (1- bits)))
                        (expt 2 (1- bits))
                        x)))

(defun unsigned-byte-p (bits x)
  (declare (xargs :guard t))
  (and (integerp bits)
       (<= 0 bits)
       (integer-range-p 0
                        (expt 2 bits)
                        x)))

; The following rules help built-in-clausep to succeed when guards are
; generated from type declarations.

(defthm integer-range-p-forward
  (implies (and (integer-range-p lower (1+ upper-1) x)
                (integerp upper-1))
           (and (integerp x)
                (<= lower x)
                (<= x upper-1)))
  :rule-classes :forward-chaining)

(defthm signed-byte-p-forward-to-integerp
  (implies (signed-byte-p n x)
           (integerp x))
  :rule-classes :forward-chaining)

(defthm unsigned-byte-p-forward-to-nonnegative-integerp
  (implies (unsigned-byte-p n x)
           (and (integerp x)
                (<= 0 x)))
  :rule-classes :forward-chaining)

; The logic-only definition of zpf needs to come after expt and integer-range-p.

(defmacro the-fixnum (n)
  (list 'the '(signed-byte 30) n))

#-acl2-loop-only
(defun-one-output zpf (x)
  (declare (type (unsigned-byte 29) x))
  (eql (the-fixnum x) 0))
#+acl2-loop-only
(defun zpf (x)
  (declare (type (unsigned-byte 29) x))
  ":Doc-Section ACL2::Programming

  testing a nonnegative fixnum against 0~/

  ~c[Zpf] is exactly the same as ~ilc[zp], except that ~c[zpf] is intended for,
  and faster for, fixnum arguments.  Its guard is specified with a type
  declaration, ~c[(type (unsigned-byte 29) x)].  (~l[declare] and
  ~pl[type-spec].)  Also ~pl[zp].~/~/"

  (if (integerp x)
      (<= x 0)
    t))

; We continue by proving the guards on substitute, all-vars1 and all-vars.

(local
 (defthm character-listp-substitute-ac
   (implies (and (characterp new)
                 (character-listp x)
                 (character-listp acc))
            (character-listp (substitute-ac new old x acc)))))

(verify-guards substitute)

(local
 (encapsulate
  ()

; We wish to prove symbol-listp-all-vars1, below, so that we can verify the
; guards on all-vars1.  But it is in a mutually recursive clique.  Our strategy
; is simple: (1) define the flagged version of the clique, (2) prove that it is
; equal to the given pair of official functions, (3) prove that it has the
; desired property and (4) then obtain the desired property of the official
; function by instantiation of the theorem proved in step 3, using the theorem
; proved in step 2 to rewrite the flagged flagged calls in that instance to the
; official ones.

; Note: It would probably be better to make all-vars1/all-vars1-lst local,
; since it's really not of any interest outside the guard verification of
; all-vars1.  However, since we are passing through this file more than once,
; that does not seem to be an option.

  (local
   (defun all-vars1/all-vars1-lst (flg lst ans)
     (if (eq flg 'all-vars1)
         (cond ((variablep lst) (add-to-set-eq lst ans))
               ((fquotep lst) ans)
               (t (all-vars1/all-vars1-lst 'all-vars-lst1 (cdr lst) ans)))
         (cond ((endp lst) ans)
               (t (all-vars1/all-vars1-lst 'all-vars-lst1 (cdr lst)
                                           (all-vars1/all-vars1-lst 'all-vars1 (car lst) ans)))))))

  (local
   (defthm step-1-lemma
     (equal (all-vars1/all-vars1-lst flg lst ans)
            (if (equal flg 'all-vars1) (all-vars1 lst ans) (all-vars1-lst lst ans)))))

  (local
   (defthm step-2-lemma
     (implies (and (symbol-listp ans)
                   (if (equal flg 'all-vars1)
                       (pseudo-termp lst)
                       (pseudo-term-listp lst)))
              (symbol-listp (all-vars1/all-vars1-lst flg lst ans)))))

  (defthm symbol-listp-all-vars1
    (implies (and (symbol-listp ans)
                  (pseudo-termp lst))
             (symbol-listp (all-vars1 lst ans)))
    :hints (("Goal" :use (:instance step-2-lemma (flg 'all-vars1)))))))

(verify-guards all-vars1)

(verify-guards all-vars)

; Next, we verify the guards of getprops, which we delayed for the same
; reasons.

(encapsulate
 ()

 (local
  (defthm member-equal-is-member
    (implies (eqlable-listp x)
             (equal (member-equal a x)
                    (member a x))))
  )

 (defthm string<-l-asymmetric
   (implies (and (eqlable-listp x1)
                 (eqlable-listp x2)
                 (integerp i)
                 (string<-l x1 x2 i))
            (not (string<-l x2 x1 i)))
   :hints (("Goal" :in-theory (disable member))))

 (defthm symbol-<-asymmetric
   (implies (and (symbolp sym1)
                 (symbolp sym2)
                 (symbol-< sym1 sym2))
            (not (symbol-< sym2 sym1)))
   :hints (("Goal" :in-theory
            (set-difference-theories
             (enable string< symbol-<)
             '(string<-l)))))

 (defthm string<-l-transitive
   (implies (and (string<-l x y i)
                 (string<-l y z j)
                 (integerp i)
                 (integerp j)
                 (integerp k)
                 (character-listp x)
                 (character-listp y)
                 (character-listp z))
            (string<-l x z k))
   :rule-classes ((:rewrite :match-free :all))
   :hints (("Goal" :induct t
            :in-theory (disable member))))

 (in-theory (disable string<-l))

 (defthm symbol-<-transitive
   (implies (and (symbol-< x y)
                 (symbol-< y z)
                 (symbolp x)
                 (symbolp y)
                 (symbolp z))
            (symbol-< x z))
   :rule-classes ((:rewrite :match-free :all))
   :hints (("Goal" :in-theory (enable symbol-< string<))))

 (local
  (defthm equal-char-code-rewrite
    (implies (and (characterp x)
                  (characterp y))
             (implies (equal (char-code x) (char-code y))
                      (equal (equal x y) t)))
    :hints (("Goal" :use equal-char-code))))

 (defthm string<-l-trichotomy
   (implies (and (not (string<-l x y i))
                 (integerp i)
                 (integerp j)
                 (character-listp x)
                 (character-listp y))
            (iff (string<-l y x j)
                 (not (equal x y))))
   :rule-classes ((:rewrite :match-free :all))
   :hints (("Goal" :in-theory
            (set-difference-theories
             (enable string<-l)
             '(member))
            :induct t)))

 (local
  (defthm equal-coerce
    (implies (and (stringp x)
                  (stringp y))
             (equal (equal (coerce x 'list)
                           (coerce y 'list))
                    (equal x y)))
    :hints (("Goal" :use
             ((:instance coerce-inverse-2 (x x))
              (:instance coerce-inverse-2 (x y)))
             :in-theory (disable coerce-inverse-2)))))

 (local
  (defthm symbol-equality-rewrite
    (implies (and (symbolp s1)
                  (symbolp s2)
                  (equal (symbol-name s1)
                         (symbol-name s2))
                  (equal (symbol-package-name s1)
                         (symbol-package-name s2)))
             (equal (equal s1 s2) t))
    :hints (("Goal" :use symbol-equality))))

 (defthm symbol-<-trichotomy
   (implies (and (symbolp x)
                 (symbolp y)
                 (not (symbol-< x y)))
            (iff (symbol-< y x)
                 (not (equal x y))))
   :hints (("Goal" :in-theory (enable symbol-< string<))))

 (defthm ordered-symbol-alistp-remove-first-pair
   (implies (and (ordered-symbol-alistp l)
                 (symbolp key)
                 (assoc-eq key l))
            (ordered-symbol-alistp (remove-first-pair key l))))

 (defthm symbol-<-irreflexive
   (implies (symbolp x)
            (not (symbol-< x x)))
   :hints (("Goal" :use
            ((:instance symbol-<-asymmetric
                        (sym1 x) (sym2 x)))
            :in-theory (disable symbol-<-asymmetric))))

 (defthm ordered-symbol-alistp-add-pair
   (implies (and (ordered-symbol-alistp gs)
                 (symbolp w5))
            (ordered-symbol-alistp (add-pair w5 w6 gs))))

 (defthm ordered-symbol-alistp-getprops
   (implies (and (plist-worldp w)
                 (symbolp world-name)
                 (symbolp key))
            (ordered-symbol-alistp (getprops key world-name w)))
   :hints (("Goal" :in-theory (enable symbol-<))))

 (local (defthm ordered-symbol-alistp-implies-symbol-alistp
          (implies (ordered-symbol-alistp x)
                   (symbol-alistp x))))

 (local (defthm symbol-alistp-implies-alistp
          (implies (symbol-alistp x)
                   (alistp x))))

 (verify-guards getprops)

 )

; Functions such as logand require significant arithmetic to prove.  Therefore
; part of the proofs for their "warming" will be deferred.

; Bishop Brock has contributed the lemma justify-integer-floor-recursion that
; follows.  Although he has proved this lemma as part of a larger proof effort,
; we are not yet in a hurry to isolate its proof just now.

(local
 (skip-proofs
  (defthm justify-integer-floor-recursion

; To use this, be sure to disable acl2-count and floor.  If you leave
; acl2-count enabled, then prove a version of this appropriate to that setting.

    (implies
     (and (integerp i)
          (integerp j)
          (not (equal i 0))
          (not (equal i -1))
          (> j 1))
     (< (acl2-count (floor i j)) (acl2-count i)))
    :rule-classes :linear)))

#+acl2-loop-only
(defmacro logand (&rest args)

  ":Doc-Section ACL2::Programming

  bitwise logical `and' of zero or more integers~/

  When integers are viewed in their two's complement representation,
  ~c[logand] returns their bitwise logical `and'.  In ACL2 ~c[logand] is a
  macro that expands into calls of the binary function ~c[binary-logand],
  except that ~c[(logand)] expands to ~c[-1] and ~c[(logand x)] expands to ~c[x].~/

  The ~il[guard] for ~c[binary-logand] requires its arguments to be integers.
  ~c[Logand] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (cond
   ((null args)
    -1)
   ((null (cdr args))
    (car args))
   (t (xxxjoin 'binary-logand args))))

#+acl2-loop-only
(defmacro logeqv (&rest args)

  ":Doc-Section ACL2::Programming

  bitwise logical equivalence of zero or more integers~/

  When integers are viewed in their two's complement representation,
  ~c[logeqv] returns their bitwise logical equivalence.  In ACL2 ~c[logeqv] is a
  macro that expands into calls of the binary function ~c[binary-logeqv],
  except that ~c[(logeqv)] expands to ~c[-1] and ~c[(logeqv x)] expands to ~c[x].~/

  The ~il[guard] for ~c[binary-logeqv] requires its arguments to be integers.
  ~c[Logeqv] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (cond
   ((null args)
    -1)
   ((null (cdr args))
    (car args))
   (t (xxxjoin 'binary-logeqv args))))

#+acl2-loop-only
(defmacro logior (&rest args)

  ":Doc-Section ACL2::Programming

  bitwise logical inclusive or of zero or more integers~/

  When integers are viewed in their two's complement representation,
  ~c[logior] returns their bitwise logical inclusive or.  In ACL2 ~c[logior] is a
  macro that expands into calls of the binary function ~c[binary-logior],
  except that ~c[(logior)] expands to ~c[0] and ~c[(logior x)] expands to ~c[x].~/

  The ~il[guard] for ~c[binary-logior] requires its arguments to be integers.
  ~c[Logior] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (cond
   ((null args)
    0)
   ((null (cdr args))
    (car args))
   (t (xxxjoin 'binary-logior args))))

#+acl2-loop-only
(defmacro logxor (&rest args)

  ":Doc-Section ACL2::Programming

  bitwise logical exclusive or of zero or more integers~/

  When integers are viewed in their two's complement representation,
  ~c[logxor] returns their bitwise logical exclusive or.  In ACL2 ~c[logxor] is a
  macro that expands into calls of the binary function ~c[binary-logxor],
  except that ~c[(logxor)] expands to ~c[0] and ~c[(logxor x)] expands to ~c[x].~/

  The ~il[guard] for ~c[binary-logxor] requires its arguments to be integers.
  ~c[Logxor] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (cond
   ((null args)
    0)
   ((null (cdr args))
    (car args))
   (t (xxxjoin 'binary-logxor args))))

#+acl2-loop-only
(defun integer-length (i)

; Bishop Brock contributed the following definition.  We believe it to be
; equivalent to one on p. 361 of CLtL2:
; (log2 (if (< x 0) (- x) (1+ x))).

  ":Doc-Section ACL2::Programming

  number of bits in two's complement integer representation~/

  For non-negative integers, ~c[(integer-length i)] is the minimum number
  of bits needed to represent the integer.  Any integer can be
  represented as a signed two's complement field with a minimum of
  ~c[(+ (integer-length i) 1)] bits.~/

  The ~il[guard] for ~c[integer-length] requires its argument to be an
  integer.  ~c[Integer-length] is defined in Common Lisp.  See any
  Common Lisp documentation for more information.~/"

  (declare (xargs :guard (integerp i)
                  :hints (("Goal" :in-theory (disable acl2-count floor)))))
  (if (zip i)
      0
    (if (= i -1)
        0
      (+ 1 (integer-length (floor i 2))))))

(defun binary-logand (i j)
  (declare (xargs :guard (and (integerp i)
                              (integerp j))
                  :hints (("Goal" :in-theory (disable acl2-count floor)))))
  (cond ((zip i) 0)
        ((zip j) 0)
        ((eql i -1) j)
        ((eql j -1) i)
        (t (let ((x (* 2 (logand (floor i 2) (floor j 2)))))
             (+ x (cond ((evenp i) 0)
                        ((evenp j) 0)
                        (t 1)))))))

#+acl2-loop-only
(defun lognand (i j)

  ":Doc-Section ACL2::Programming

  bitwise logical `nand' of two integers~/

  When integers are viewed in their two's complement representation,
  ~c[lognand] returns their bitwise logical `nand'.~/

  The ~il[guard] for ~c[lognand] requires its arguments to be integers.
  ~c[Lognand] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (lognot (logand i j)))

(defun binary-logior (i j)
  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (lognot (logand (lognot i) (lognot j))))

#+acl2-loop-only
(defun logorc1 (i j)

  ":Doc-Section ACL2::Programming

  bitwise logical inclusive or of two ints, complementing the first~/

  When integers are viewed in their two's complement representation,
  ~c[logorc1] returns the bitwise logical inclusive or of the second
  with the bitwise logical `not' of the first.~/

  The ~il[guard] for ~c[logorc1] requires its arguments to be integers.
  ~c[Logorc1] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (logior (lognot i) j))

#+acl2-loop-only
(defun logorc2 (i j)

  ":Doc-Section ACL2::Programming

  bitwise logical inclusive or of two ints, complementing the second~/

  When integers are viewed in their two's complement representation,
  ~c[logorc2] returns the bitwise logical inclusive or of the first
  with the bitwise logical `not' of the second.~/

  The ~il[guard] for ~c[logorc2] requires its arguments to be integers.
  ~c[Logorc2] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (logior i (lognot j)))

#+acl2-loop-only
(defun logandc1 (i j)

  ":Doc-Section ACL2::Programming

  bitwise logical `and' of two ints, complementing the first~/

  When integers are viewed in their two's complement representation,
  ~c[logandc1] returns the bitwise logical `and' of the second with the
  bitwise logical `not' of the first.~/

  The ~il[guard] for ~c[logandc1] requires its arguments to be integers.
  ~c[Logandc1] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (logand (lognot i) j))

#+acl2-loop-only
(defun logandc2 (i j)

  ":Doc-Section ACL2::Programming

  bitwise logical `and' of two ints, complementing the second~/

  When integers are viewed in their two's complement representation,
  ~c[logandc2] returns the bitwise logical `and' of the first with the
  bitwise logical `not' of the second.~/

  The ~il[guard] for ~c[logandc2] requires its arguments to be integers.
  ~c[Logandc2] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (logand i (lognot j)))

(defun binary-logeqv (i j)
  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (logand (logorc1 i j)
          (logorc1 j i)))

(defun binary-logxor (i j)
  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (lognot (logeqv i j)))

#+acl2-loop-only
(defun lognor (i j)

  ":Doc-Section ACL2::Programming

  bitwise logical `nor' of two integers~/

  When integers are viewed in their two's complement representation,
  ~c[lognor] returns the bitwise logical `nor' of the first with the
  second.~/

  The ~il[guard] for ~c[lognor] requires its arguments to be integers.
  ~c[Lognor] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (integerp i)
                              (integerp j))))
  (lognot (logior i j)))

#+acl2-loop-only
(defun logtest (x y)

; p. 360 of CLtL2

  ":Doc-Section ACL2::Programming

  test if two integers share a `1' bit~/

  When integers ~c[x] and ~c[y] are viewed in their two's complement
  representation, ~c[(logtest x y)] is true if and only if there is
  some position for which both ~c[x] and ~c[y] have a `1' bit in that
  position.~/

  The ~il[guard] for ~c[logtest] requires its arguments to be integers.
  ~c[Logtest] is defined in Common Lisp.  See any Common Lisp
  documentation for more information.~/"

  (declare (xargs :guard (and (integerp x) (integerp y))))
  (not (zerop (logand x y))))

; Warning: Keep the following defconst forms in sync wi