/***************************************************************************
 *   Copyright (C) 2003-2005 by Kevin Hessels                              *
 *   khessels@shaw.ca                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qdatetime.h>
#include <qregexp.h>
#include <qstringlist.h>
 
#include <kdebug.h>
#include <kiconloader.h>
#include <klistview.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kprocess.h>
 
#include "kfoldingqueuewidget.h"


kfoldingQueueItem::kfoldingQueueItem( QListView* parent, const QStringList& lines ) : KListViewItem( parent )
{
	parse( lines );
		
	setPixmap( 0, _icon );
	setText( 0, _index );	
	setText( 1, _projectName );
	setText( 2, _unit );
	setText( 3, _pph );
	setText( 4, _speed );
	setText( 5, _status );
	setText( 6, _benchmark );
	setText( 7, _begin.toString() );
	setText( 8, _due.toString() );
	return;
} // kfoldingQueueItem ctor


kfoldingQueueItem::~kfoldingQueueItem()
{
	return;
} // kfoldingQueueItem dtor


int kfoldingQueueItem::compare( QListViewItem *i, int col, bool ascending ) const
{
	kfoldingQueueItem *other = 0L;
	
	switch( col ) {
		case 0:
			return KListViewItem::compare( i, 1, ascending );
			break;
		case 7:
		{
			other = static_cast<kfoldingQueueItem*>(i);
			if ( other->_begin.isValid() ) 
				return 1;
			if ( this->_begin == other->_begin ) 
				return 0;
			return ( ascending ? 1 : -1 ) * ( this->_begin > other->_begin ? 1 : -1 );
			break;
		}
		default:
			break;
	} // switch
		
	return KListViewItem::compare( i, col, ascending );
} // compare 


void kfoldingQueueItem::parse( const QStringList& lines )
{
	QRegExp rxQueueItemStart( "^\\sIndex\\s(\\d):\\s(.*)$" );
	QRegExp rxStatus( "^(\\D*)((\\d+\\.\\d{2})\\spts\\s\\((\\d+\\.\\d+).*\\)\\s(\\d+\\.\\d+))?.*$" );
	QRegExp rxProject( "^.*project:\\s(\\d+)(,\\s\\\"(\\w+)\\\")?$" );
	QRegExp rxUnit( "^\\s{2}Folding:\\s(.*);\\sbenchmark\\s(\\d+)(.*)$" );
	QRegExp rxStart( "^.*begin:\\s(.*)$" );
	QRegExp rxDue( "^.*due:\\s(.*)\\s\\(.*$" );
	
	
	for ( QStringList::const_iterator it = lines.begin(); it != lines.end(); ++it ) {
		if ( rxQueueItemStart.search( *it ) != -1 ) {
			_index = rxQueueItemStart.cap( 1 );
			if ( rxQueueItemStart.cap( 2 ) == QString::fromLatin1( "empty" ) ) {
				_status =  i18n( "Empty" );
				_icon = SmallIcon( "misc" );
				break;
			} else {
				if ( rxStatus.search( rxQueueItemStart.cap( 2 ) ) != -1 ) {
					QString captured = rxStatus.cap( 1 ).stripWhiteSpace();
					if ( captured == QString::fromLatin1( "finished" ) ) {
						_status = i18n( "Finished" );
						_icon = SmallIcon( "button_ok" );
					} 
					else if ( captured == QString::fromLatin1( "deleted" ) ) {
						_status = i18n( "Deleted" );
 						_icon = SmallIcon( "edittrash" );
					}
					else if ( captured == QString::fromLatin1( "garbage" ) ) {
						_status = i18n( "Garbage" );
						_icon = SmallIcon( "edittrash" );
					}
					else if ( captured == QString::fromLatin1( "queued for processing" ) ) {
						_status = i18n( "Queued" );
						_icon = SmallIcon( "redo" );
					}
					else if ( captured == QString::fromLatin1( "ready for upload" ) ) {
						_status = i18n( "Ready for Upload" );
						_icon = SmallIcon( "button_ok" );
					}
					else if ( captured == QString::fromLatin1( "fetching from server" ) ) {
						_status = i18n( "Fetching from Server" );
						_icon = SmallIcon( "down" );
					}
					else if ( captured == QString::fromLatin1( "folding now" ) ) {
						_status = i18n( "Folding Now" );
						_icon = SmallIcon( "redo" );
					}
					else {
						_status = i18n( "Unknown" );
						_icon = SmallIcon( "misc" );
					}
					//_pts = rxStatus.cap( 3 );
					_pph = rxStatus.cap( 4 );
					_speed = rxStatus.cap( 5 );
				}
				continue;
			} // else
		} // if
		
		if ( rxProject.search( *it ) != -1 ) {
			_projectName = rxProject.cap( 1 );
			//if ( !rxProject.cap( 3 ).isEmpty() ) {
			//	_projectName += " " + rxProject.cap( 3 );
			//}
			continue;
		}
		
		if ( rxUnit.search( *it ) != -1 ) {
			_unit = rxUnit.cap( 1 );
			_benchmark = rxUnit.cap( 2 );
			continue;
		}

		if ( rxStart.search( *it ) != -1 ) {
			QDateTime begin = QDateTime::fromString( rxStart.cap( 1 ) );
			if ( begin.isValid() )
				_begin = begin;
			continue;
		}

		if ( rxDue.search( *it ) != -1 ) {
			QDateTime due = QDateTime::fromString( rxDue.cap( 1 ) );
			if ( due.isValid() ) 
				_due = due;
			continue;
		}
	} // for
	return;
} // parse



//////////////////////////////////////////////////////////////////////////////////////////////////////////////////


kfoldingQueueWidget::kfoldingQueueWidget( QWidget *parent, const char *name, const QString& workingDir )
 : KListView( parent, name ),
   _qdProcess( new KProcess() ), _populated( false )
{
	addColumn( "", -10 );
	addColumn( i18n( "Project" ) );
	addColumn( i18n( "Unit" ) );
	addColumn( i18n( "Points/hr" ) );
	addColumn( i18n( "Speed" ) );
	addColumn( i18n( "Status" ) );
	addColumn( i18n( "Benchmark" ) );
	addColumn( i18n( "Begin" ) );
	addColumn( i18n( "Due" ) );

	setColumnAlignment( 1, Qt::AlignCenter );
	setColumnAlignment( 3, Qt::AlignCenter );
	setColumnAlignment( 4, Qt::AlignCenter );
	setColumnAlignment( 6, Qt::AlignCenter );

	*_qdProcess << "qd";
	*_qdProcess << "-f" << workingDir;

	connect( _qdProcess, SIGNAL( receivedStdout( KProcess*, char*, int ) ), this, SLOT( slotReadQD( KProcess*, char*, int ) ) );
	connect( _qdProcess, SIGNAL( processExited( KProcess* ) ), this, SLOT( slotQDExited( KProcess* ) ) );
	_qdProcess->start( KProcess::NotifyOnExit, KProcess::Stdout );

	return;
} // kfoldingQueueWidget ctor


kfoldingQueueWidget::~kfoldingQueueWidget()
{
	delete _qdProcess;
	return;
} // kfoldingQueueWidget dtor

/*
int kfoldingQueueWidget::width() const
{
	if ( !_populated )
		return 0;

	return KListView::width();
} // width


int kfoldingQueueWidget::height() const
{
	if ( !_populated )
		return 0;

	return KListView::height();
} // height


QSize kfoldingQueueWidget::sizeHint() const
{
	return QSize( width(), height() );
} // sizeHint
*/


void kfoldingQueueWidget::populate()
{
	QStringList qdOutput = QStringList::split( "\n", _outputBuffer );
	
	QRegExp rxQueueItemStart( "^\\sIndex\\s(\\d+):\\s(.*)$" );
	for ( QStringList::Iterator it = qdOutput.begin(); it != qdOutput.end(); ++it ) {
		if ( rxQueueItemStart.search( *it ) != -1 ) {
			QStringList queueItemData;
			if ( rxQueueItemStart.cap( 2 ) == QString::fromLatin1( "empty" ) ) {
				queueItemData.append( *it );
			} else {
				for ( int i = 0; i < 10; ++i, ++it ) {
					kdDebug() << *it << endl;
					queueItemData.append( *it );
				} // for
				it--;
			} // else
			insertItem( new kfoldingQueueItem( this, queueItemData ) );
		} // if
	} // for
	setSorting( 7, true );
	_populated = true;

	emit populated();
	return;
} // populate


int kfoldingQueueWidget::calculateHeight()
{
	int height = 0;

	QListViewItemIterator it( this );
	while ( it.current() ) {
		kfoldingQueueItem *item = static_cast<kfoldingQueueItem *>( it.current() );
		height += item->height();
		++it;
	}
	return height;
} // calculateHeight


void kfoldingQueueWidget::slotReadQD( KProcess*, char* buffer, int buflen )
{
	QString myBuf = QString::fromLatin1( buffer, buflen );
	_outputBuffer.append( myBuf );
	return;
} // slotReadQD


void kfoldingQueueWidget::slotQDExited( KProcess* )
{
	populate();
	return;
} // slotQDExited


#include "kfoldingqueuewidget.moc"
