/*****************************************************************************
 *
 * utouch-frame - Touch Frame Library
 *
 * Copyright (C) 2010-2012 Canonical Ltd.
 *
 * This library is free software: you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License version 3
 * as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranties of 
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

/**
 * @file utouch/frame.h
 * Definitions of the main and platform-generic API
 */

#ifndef UTOUCH_FRAME_UTOUCH_FRAME_H_
#define UTOUCH_FRAME_UTOUCH_FRAME_H_

/* Macros that set symbol visibilities in shared libraries properly.
 * Adapted from http://gcc.gnu.org/wiki/Visibility
 */

#if defined _WIN32 || defined __CYGWIN__
  #ifdef BUILDING_FRAME
    #define FRAME_PUBLIC __declspec(dllexport)
  #else
    #define FRAME_PUBLIC __declspec(dllimport)
  #endif
#else
  #if defined __GNUC__
    #define FRAME_PUBLIC __attribute__ ((visibility("default")))
  #else
    #pragma message ("Compiler does not support symbol visibility.")
    #define FRAME_PUBLIC
  #endif
#endif

/* Clang provides __has_extension, but GCC does not */
#ifdef __has_extension
#if __has_extension(c_generic_selections)
#define HAS_C_GENERIC_SELECTIONS
#endif // __has_extension
#endif // __has_extension(c_generic_selections)

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>

/**
 * @defgroup v1 uTouch-Frame 1.x
 * @{
 * @internal
 */

#define UTOUCH_FRAME_VERSION	0x00001010

/**
 * struct utouch_surface - device surface details
 * @needs_pointer: device needs a screen pointer to function
 * @is_direct: surface is a direct device (e.g. touchscreen)
 * @is_buttonpad: surface has button(s) under it
 * @is_semi_mt: surface detects bounding rectangle only
 * @use_touch_major: device uses major axis for contact modulation
 * @use_touch_minor: device uses minor axis for contact modulation
 * @use_width_major: device uses approaching major axis for contact modulation
 * @use_width_minor: device uses approaching minor axis for contact modulation
 * @use_orientation: device uses ellipse orientation for contact modulation
 * @use_pressure: device uses pressure for contact modulation
 * @use_distance: device uses hover distance
 * @phys_width: physical width in millimeters (mm)
 * @phys_height: physical height in millimeters (mm)
 * @phys_pressure: maximal physical pressure (N/cm^2)
 * @min_x: minimum horizontal device coordinate
 * @min_y: minimum vertical device coordinate
 * @max_x: maximum horizontal device coordinate
 * @max_y: maximum vertical device coordinate
 * @max_pressure: maximum pressure device coordinate
 * @max_orient: maximum orientation device coordinate
 * @mapped_min_x: minimum horizontal mapped coordinate
 * @mapped_min_y: minimum vertical mapped coordinate
 * @mapped_max_x: maximum horizontal mapped coordinate
 * @mapped_max_y: maximum vertical mapped coordinate
 * @mapped_max_pressure: maximum pressure mapped coordinate
 *
 * The mutable contact given by utouch_frame_get_current_slot() should
 * be set in device coordinates. The contact data is subsequently
 * transformed to mapped (e.g., screen) coordinates in
 * utouch_frame_sync(). To a frame user, all data will appear in
 * mapped coordinates.
 *
 * Device properties and touch surface details. Later versions of this
 * struct may grow in size, but will remain binary compatible with
 * older versions.
 */
struct utouch_surface {
	int needs_pointer;
	int is_direct;
	int is_buttonpad;
	int is_semi_mt;
	int use_touch_major;
	int use_touch_minor;
	int use_width_major;
	int use_width_minor;
	int use_orientation;
	int use_pressure;
	int use_distance;
	float phys_width;
	float phys_height;
	float phys_pressure;
	float min_x;
	float min_y;
	float max_x;
	float max_y;
	float max_pressure;
	float max_orient;
	float mapped_min_x;
	float mapped_min_y;
	float mapped_max_x;
	float mapped_max_y;
	float mapped_max_pressure;
	unsigned int min_id;
	unsigned int max_id;
};

#define UTOUCH_TOOL_FINGER	0
#define UTOUCH_TOOL_PEN		1

/**
 * struct utouch_contact - surface contact details
 * @prev: pointer to same slot of previous frame
 * @active: currently in use
 * @slot: slot occupied by this contact
 * @id: unique id of this contact
 * @tool_type: the tool type of this contact
 * @x: horizontal center position coordinate (surface units)
 * @y: vertical center position coordinate (surface units)
 * @touch_major: major axis of contact (surface units)
 * @touch_minor: minor axis of contact (surface units)
 * @width_major: major axis of approaching contact (surface units)
 * @width_minor: minor axis of approaching contact (surface units)
 * @orientation: direction of ellipse (left: -Pi/2, up: 0, right: Pi/2)
 * @pressure: pressure of contact (pressure units)
 * @distance: distance of contact (surface units)
 * @vx: horizontal velocity coordinate (units / millisecond)
 * @vy: vertical velocity coordinate (units / millisecond)
 *
 * Surface contact details. Later versions of this struct may grow in
 * size, but will remain binary compatible with older versions.
 *
 * Contact structures are connected into one ring per slot. The
 * previous contact pointers are ABI agnostic, owned by the engine,
 * and have engine scope.
 */
struct utouch_contact {
	const struct utouch_contact *prev;
	int active;
	int slot;
	unsigned int id;
	int tool_type;
	float x;
	float y;
	float touch_major;
	float touch_minor;
	float width_major;
	float width_minor;
	float orientation;
	float pressure;
	float distance;
	float vx;
	float vy;
};

/* time in milliseconds */
typedef uint64_t utouch_frame_time_t;

/* the frame engine handle */
typedef struct utouch_frame_engine *utouch_frame_handle;

/**
 * struct utouch_frame - emitted frame details
 * @prev: pointer to previous frame
 * @sequence_id: frame sequence number
 * @revision: changes whenever the contact count changes
 * @slot_revision: changes whenever the slot id array change
 * @num_active: the number of contacts in the active array
 * @time: time of frame completion (ms)
 * @mod_time: time of last contact count change (ms)
 * @slot_mod_time: time of last slot id array change (ms)
 * @active: the array of active contacts
 *
 * Contact frame details. Later versions of this struct may grow in
 * size, but will remain binary compatible with older versions.
 *
 * Frames are connected into a ring. The previous frame pointer is ABI
 * agnostic, owned by the engine, and has engine scope.
 */
struct utouch_frame {
	const struct utouch_frame *prev;
	unsigned int sequence_id;
	unsigned int revision;
	unsigned int slot_revision;
	unsigned int num_active;
	utouch_frame_time_t time;
	utouch_frame_time_t mod_time;
	utouch_frame_time_t slot_mod_time;
	struct utouch_contact **active;
	struct utouch_contact **slots;
};

/**
 * utouch_frame_get_version - get library abi version
 *
 * Returns the version of the library, which may be different
 * from the api version of the compiled user program.
 */
FRAME_PUBLIC
unsigned int utouch_frame_get_version(void);

/**
 * utouch_frame_get_num_frames - get number of supported frames
 *
 * Returns the number of frames supported by this engine.
 */
FRAME_PUBLIC
unsigned int utouch_frame_get_num_frames(utouch_frame_handle fh);

/**
 * utouch_frame_get_num_slots - get number of supported slots
 *
 * Returns the number of simultaneous contacts supported by this engine.
 */
FRAME_PUBLIC
unsigned int utouch_frame_get_num_slots(utouch_frame_handle fh);

FRAME_PUBLIC
utouch_frame_handle utouch_frame_new_engine_raw(unsigned int num_frames,
						unsigned int num_slots,
						unsigned int frame_rate,
						unsigned int version,
						unsigned int surface_size,
						unsigned int frame_size,
						unsigned int slot_size);

/**
 * utouch_frame_new_engine - allocate a new frame engine
 * @num_frames: number of frames in cyclic buffer
 * @num_slots: maximum number of slots per frame
 * @frame_rate: maximum frame rate (frames/s)
 *
 * Allocates memory, initializes the internal engine and returns a
 * handle to it. A rate of 100 frames per second is normal.
 */
#define utouch_frame_new_engine(num_frames, num_slots, frame_rate)	\
	utouch_frame_new_engine_raw(num_frames,				\
				    num_slots,				\
				    frame_rate,				\
				    UTOUCH_FRAME_VERSION,		\
				    sizeof(struct utouch_surface),	\
				    sizeof(struct utouch_frame),	\
				    sizeof(struct utouch_contact))

/**
 * utouch_frame_delete_engine - deallocate a frame engine
 * @fh: frame engine in use
 *
 * Deallocates all memory associated with the engine.
 */
FRAME_PUBLIC
void utouch_frame_delete_engine(utouch_frame_handle fh);

/**
 * utouch_frame_get_surface - get the mutable device surface information
 * @fh: the frame engine in use
 *
 * Returns a pointer to the mutable device surface information. It is
 * preferrably set up by one of the input handlers. The pointer is ABI
 * agnostic, has frame engine scope, and is owned by the engine.
 */
FRAME_PUBLIC
struct utouch_surface *utouch_frame_get_surface(utouch_frame_handle fh);

/**
 * utouch_frame_get_current_slot - get the current mutable slot contact
 * @fh: the frame engine in use
 *
 * Returns a pointer to the contact current being modified. The
 * pointer is ABI agnostic, has frame engine scope, and is owned by
 * the engine.
 */
FRAME_PUBLIC
struct utouch_contact *utouch_frame_get_current_slot(utouch_frame_handle fh);

/**
 * utouch_frame_set_current_slot - set the current slot number
 * @fh: the frame engine in use
 * @slot: the slot number
 *
 * Sets the slot currently being modified. Returns zero if successful,
 * negative error otherwise.
 */
FRAME_PUBLIC
int utouch_frame_set_current_slot(utouch_frame_handle fh, int slot);

/**
 * utouch_frame_set_current_id - set the current slot by touch id
 * @fh: the frame engine in use
 * @id: the touch id
 *
 * Sets the slot currently being modified, by touch id. If the id is
 * not currently in use (does not have an active slot), a new slot is
 * assigned. Returns zero if successful, negative error otherwise.
 */
FRAME_PUBLIC
int utouch_frame_set_current_id(utouch_frame_handle fh, int id);

/**
 * utouch_frame_sync - synchronize and return new frame
 * @fh: the frame engine in use
 * @time: the frame synchronization time (ms)
 *
 * Scans through the updates, and in case the changes make up a new
 * frame, returns the updated frame.
 *
 * If time is zero, a time-of-receipt will be used instead.
 *
 * The frame returned is always the next in the cyclic list, and
 * always points back at the previous frame returned by this function.
 *
 * The returned pointer is ABI agnostic and owned by the frame
 * engine. It may very well be zero if there is nothing to report or
 * if the frame rate is limited.
 */
FRAME_PUBLIC
const struct utouch_frame *utouch_frame_sync(utouch_frame_handle fh,
					     utouch_frame_time_t time);

/**
 * utouch_coordinate_transform_cb - user-definable callback that transforms a touch point
 * @x: pointer to an x coordinate (in, out)
 * @y: pointer to a y coordinate (in, out)
 * @user_data: opaque pointer, passed by the user when setting the callback
 */
typedef void (*utouch_coordinate_transform_cb)(float *x, float *y, void *user_data);

/**
 * utouch_frame_set_coordinate_transform_callback - set a callback to obtain a transformation
 *   to apply to every touch point
 * @fh: the frame engine in use
 * @callback: the callback that transforms x and y
 * @user_data: opaque pointer to user data for the callback
 */
FRAME_PUBLIC
void utouch_frame_set_coordinate_transform_callback(utouch_frame_handle fh,
						    utouch_coordinate_transform_cb callback,
						    void *user_data);
/** @} */

/**
 * @defgroup v2 uTouch-Frame 2.x
 * @{
 */

/** An object for the context of the uTouch Frame instance */
typedef struct UFHandle_* UFHandle;
/** An object for an event */
typedef struct UFEvent_* UFEvent;
/** An object for a frame of touches */
typedef struct UFFrame_* UFFrame;
/** An object for a touch */ 
typedef struct UFTouch_* UFTouch;
/** An object for a device */
typedef struct UFDevice_* UFDevice;
/** An object for a device axis */
typedef struct UFAxis_* UFAxis;
/** An object for a window ID */
typedef uint64_t UFWindowId;
/** An object for a touch ID */
typedef uint64_t UFTouchId;

/** The status code denoting the result of a function call */
typedef enum UFStatus {
  UFStatusSuccess = 0, /**< The call was successful */
  UFStatusErrorGeneric, /**< A platform-dependent error occurred */
  UFStatusErrorResources, /**< An error occurred due to insufficient resources */
  UFStatusErrorNoEvent, /**< No events were available to get */
  UFStatusErrorUnknownProperty, /**< The requested property value was not set */
  UFStatusErrorInvalidTouch, /**< The requested touch does not exist */
  UFStatusErrorInvalidAxis, /**< The requested axis does not exist */
  UFStatusErrorUnsupported, /**< The requested function is not supported by the
                                 window server */
  UFStatusErrorInvalidType, /**< The variable type passed as a void pointer into
                                 a property getter is invalid for the property
                             */
} UFStatus;

/** Properties of a device */
typedef enum UFDeviceProperty {
  /**
   * The name of the device
   *
   * Value type: const char *
   *
   * The uTouch frame library owns the string. The string is valid until an
   * event notifying removal of the device is released.
   */
  UFDevicePropertyName = 0,
  /**
   * Whether the device is a direct touch device
   *
   * Value type: int with boolean semantics
   *
   * A direct touch device is a device where there is a direct transformation
   * from the touch location to the event location on the screen. An indirect
   * touch device is a device where the touch has meaning relative to a position
   * on the screen, such as the location of a cursor. A touchscreens is an
   * example of a direct device, and a trackpad is an example of an indirect
   * device.
   */
  UFDevicePropertyDirect,
  /**
   * Whether the device is an independent touch device
   *
   * Value type: int with boolean semantics
   *
   * An independent device is an indirect device whose cursor moves
   * independently of the touches on the device. A mouse with a touch area for
   * gestures is an example of an independent device, and a trackpad is an
   * example of a dependent device.
   */
  UFDevicePropertyIndependent,
  /**
   * Whether the device is a semi-multitouch device
   *
   * Value type: int with boolean semantics
   *
   * A semi-multitouch device provides a bounding box of some touches on the
   * touch surface. In contrast, a full-multitouch device provides accurate
   * locations of each individual touch.
   */
  UFDevicePropertySemiMT,
  /**
   * The maximum number of touches supported by the device
   *
   * Value type: unsigned int
   */
  UFDevicePropertyMaxTouches,
  /**
   * The number of touch axes provided by the device
   *
   * Value type: unsigned int
   */
  UFDevicePropertyNumAxes,
  /**
   * The resolution of the window coordinates of the device in the X axis
   *
   * Value type: float
   *
   * The resolution is provided in pixels per meter.
   */
  UFDevicePropertyWindowResolutionX,
  /**
   * The resolution of the window coordinates of the device in the Y axis
   *
   * Value type: float
   *
   * The resolution is provided in pixels per meter.
   */
  UFDevicePropertyWindowResolutionY,
} UFDeviceProperty;

/** Device touch axis types */
typedef enum UFAxisType {
  UFAxisTypeX = 0, /**< X coordinate */
  UFAxisTypeY, /**< Y coordinate */
  UFAxisTypeTouchMajor, /**< Width along major axis of contact area of touch */
  UFAxisTypeTouchMinor, /**< Width along minor axis of contact area of touch */
  UFAxisTypeWidthMajor, /**< Width along major axis of touch tool */
  UFAxisTypeWidthMinor, /**< Width along minor axis of touch tool */
  UFAxisTypeOrientation, /**< Orientation of major axis of contact ellipse */
  UFAxisTypeTool, /**< Tool type */
  UFAxisTypeBlobId, /**< Blob ID of group of touches */
  UFAxisTypeTrackingId, /**< Tracking ID */
  UFAxisTypePressure, /**< Pressure */
  UFAxisTypeDistance, /**< Hover distance */
} UFAxisType;

/** Event properties */
typedef enum UFEventProperty {
  /**
   * Type of event
   *
   * Value type: UFEventType
   */
  UFEventPropertyType = 0,
  /**
   * Device added or removed
   *
   * Value type: UFDevice
   *
   * This property is set only when the event type is UFEventTypeDeviceAdded
   * or UFEventTypeDeviceRemoved. The object is owned by the library and is
   * valid until an event notifying removal of the device is released.
   */
  UFEventPropertyDevice,
  /**
   * Touch frame
   *
   * Value type: UFFrame
   *
   * This property is set only when the event type is UFEventTypeFrame. The
   * object is owned by the library and is valid until the event is released.
   */
  UFEventPropertyFrame,
  /**
   * Event time
   *
   * Value type: 64-bit unsigned int
   *
   * This property holds the time the event occurred in display server
   * timespace. The time is provided in milliseconds (ms). If the event, such as
   * device addition, occurred before the uTouch Frame context was created, the
   * value will be 0.
   */
  UFEventPropertyTime,
} UFEventProperty;

/** Event types */
typedef enum UFEventType {
  UFEventTypeDeviceAdded = 0, /**< A new device has been added */
  UFEventTypeDeviceRemoved, /**< An existing device has been removed */
  UFEventTypeFrame, /**< The state of one or more touches has changed */
} UFEventType;

/** Touch frame properties */
typedef enum UFFrameProperty {
  /**
   * The device for the frame
   *
   * Value type: UFDevice
   */
  UFFramePropertyDevice = 0,
  /**
   * The window server ID of the window for the frame
   *
   * Value type: UFWindowId
   */
  UFFramePropertyWindowId,
  /**
   * Number of touches in the frame
   *
   * Value type: unsigned int
   *
   * Some devices can track more touches than they can report data for. Only
   * touches with X and Y position are provided in the frame.
   */
  UFFramePropertyNumTouches,
  /**
   * Total number of active touches on the device
   *
   * Value type: unsigned int
   *
   * Some devices can track more touches than they can report data for. This
   * value includes the number of reported and unreported touches.
   */
  UFFramePropertyActiveTouches,
} UFFrameProperty;

/** State of an individual touch */
typedef enum UFTouchState {
  UFTouchStateBegin = 0, /**< The touch began */
  UFTouchStateUpdate, /**< A value or property of the touch changed */
  UFTouchStateEnd, /**< The touch ended */
} UFTouchState;

/** Touch properties */
typedef enum UFTouchProperty {
  /**
   * Window server ID of the touch
   *
   * Value type: UFTouchId
   */
  UFTouchPropertyId = 0,
  /**
   * State of the touch
   *
   * Value type: UFTouchState
   */
  UFTouchPropertyState,
  /**
   * Location along X axis of touch relative to event window
   *
   * Value type: float
   *
   * The window server may provide touch location in window coordinate space.
   * This property will be set where available.
   */
  UFTouchPropertyWindowX,
  /**
   * Location along Y axis of touch relative to event window
   *
   * Value type: float
   *
   * The window server may provide touch location in window coordinate space.
   * This property will be set where available.
   */
  UFTouchPropertyWindowY,
  /**
   * Time of last touch state change
   *
   * Value type: 64-bit unsigned int
   *
   * See UFEventPropertyTime for the semantics of the value. If the touch has
   * not changed during this frame, the value of this property will be less than
   * the value of the UFEventPropertyTime event property for this frame.
   */
  UFTouchPropertyTime,
  /**
   * Start time of touch
   *
   * Value type: 64-bit unsigned int
   *
   * See UFEventPropertyTime for the semantics of the value.
   */
  UFTouchPropertyStartTime,
  /**
   * Whether the touch is owned by the client
   *
   * Value type: int with boolean semantics
   *
   * Some window servers have the concept of touch ownership. This property
   * is only valid when the server supports touch ownership.
   */
  UFTouchPropertyOwned,
  /**
   * Whether the touch has physically ended before the touch sequence has ended
   *
   * Value type: int with boolean semantics
   *
   * Some window servers have the concept of touch ownership. If a touch has
   * ended before the client receives ownership, this property will be set to
   * true. The property will also be set to true when the touch has ended before
   * the client has accepted or rejected ownership of the touch sequence.
   */
  UFTouchPropertyPendingEnd,
} UFTouchProperty;

/**
 * Get the event file descriptor for the uTouch Frame context
 *
 * @param [in] handle The uTouch Frame context object
 * @return A file descriptor for the context
 *
 * When events are available for processing, the file descriptor will be
 * readable. Perform an 8-byte read from the file descriptor to clear the state.
 * Refer to the EVENTFD(2) man page for more details.
 */
FRAME_PUBLIC
int frame_get_fd(UFHandle handle);

/**
 * Get an event from the uTouch Frame context
 *
 * @param [in] handle The context object
 * @param [out] event The retrieved event
 * @return UFStatusSuccess or UFStatusErrorNoEvent
 *
 * The reference count of the returned event is implicity incremented once.
 */
FRAME_PUBLIC
UFStatus frame_get_event(UFHandle handle, UFEvent *event);

/**
 * Get the value of a property of a device
 *
 * @param [in] device The device object (const)
 * @param [in] property The property to retrieve a value for
 * @param [out] value The value retrieved
 * @return UFStatusSuccess or UFStatusErrorUnknownProperty
 */
#ifndef HAS_C_GENERIC_SELECTIONS /* See frame_internal.h */
FRAME_PUBLIC
UFStatus frame_device_get_property(UFDevice device, UFDeviceProperty property,
                                   void *value);
#endif

/**
 * Get a device touch axis by index
 *
 * @param [in] device The device object (const)
 * @param [in] index The index of the axis to get
 * @param [out] axis The axis retrieved
 * @return UFStatusSuccess or UFStatusErrorInvalidAxis
 *
 * The index value must be greater than or equal to 0 and less than the number
 * axes of the device.
 */
FRAME_PUBLIC
UFStatus frame_device_get_axis_by_index(UFDevice device, unsigned int index,
                                        UFAxis *axis);

/**
 * Get a device touch axis by axis type
 *
 * @param [in] device The device object (const)
 * @param [in] type The axis type
 * @param [out] axis The axis retrieved
 * @return UFStatusSuccess or UFStatusErrorInvalidAxis
 *
 * UFStatusErrorInvalidAxis is returned if the device does not have an axis of
 * the type requested.
 */
FRAME_PUBLIC
UFStatus frame_device_get_axis_by_type(UFDevice device, UFAxisType type,
                                       UFAxis *axis);

/**
 * Get the type of a touch device axis
 *
 * @param [in] axis The touch device axis (const)
 * @return The type of the axis
 */
FRAME_PUBLIC
UFAxisType frame_axis_get_type(UFAxis axis);

/**
 * Get the minimum value of a touch device axis
 *
 * @param [in] axis The touch device axis (const)
 * @return The minimum value of the axis
 */
FRAME_PUBLIC
float frame_axis_get_minimum(UFAxis axis);

/**
 * Get the maximum value of a touch device axis
 *
 * @param [in] axis The touch device axis (const)
 * @return The maximum value of the axis
 */
FRAME_PUBLIC
float frame_axis_get_maximum(UFAxis axis);

/**
 * Get the resolution of a touch device axis
 *
 * @param [in] axis The touch device axis (const)
 * @return The resolution of the axis
 */
FRAME_PUBLIC
float frame_axis_get_resolution(UFAxis axis);

/**
 * Increment the reference count of an event
 *
 * @param [in] event The event object
 */
FRAME_PUBLIC
void frame_event_ref(UFEvent event);

/**
 * Decrement the reference count of an event
 *
 * @param [in] event The event object
 *
 * When the reference count reaches zero, the event is freed.
 */
FRAME_PUBLIC
void frame_event_unref(UFEvent event);

/**
 * Get the value of a property of an event
 *
 * @param [in] event The event object (const)
 * @param [in] property The property to retrieve a value for
 * @param [out] value The value retrieved
 * @return UFStatusSuccess or UFStatusErrorUnknownProperty
 */
#ifndef HAS_C_GENERIC_SELECTIONS /* See frame_internal.h */
FRAME_PUBLIC
UFStatus frame_event_get_property(UFEvent event, UFEventProperty property,
                                  void *value);
#endif

/**
 * Get the value of a property of a frame
 *
 * @param [in] frame The frame object (const)
 * @param [in] property The property to retrieve a value for
 * @param [out] value The value retrieved
 * @return UFStatusSuccess or UFStatusErrorUnknownProperty
 */
#ifndef HAS_C_GENERIC_SELECTIONS /* See frame_internal.h */
FRAME_PUBLIC
UFStatus frame_frame_get_property(UFFrame frame, UFFrameProperty property,
                                  void *value);
#endif

/**
 * Get a touch of a frame by index
 *
 * @param [in] frame The frame object (const)
 * @param [in] index The index of the touch to get
 * @param [out] touch The touch retrieved
 * @return UFStatusSuccess or UFStatusErrorInvalidTouch
 *
 * The index value must be greater than or equal to 0 and less than the number
 * touches reported in the frame.
 */
FRAME_PUBLIC
UFStatus frame_frame_get_touch_by_index(UFFrame frame, unsigned int index,
                                        UFTouch *touch);

/**
 * Get a touch from a frame by the window server ID
 *
 * @param [in] frame The frame object (const)
 * @param [in] touch_id The window server ID of the touch
 *             The value type of the touch ID is window server dependent. See
 *             UFTouchPropertyId for more details.
 * @param [out] touch The touch object
 * @return UFStatusSuccess or UFStatusErrorInvalidTouch
 */
FRAME_PUBLIC
UFStatus frame_frame_get_touch_by_id(UFFrame frame, UFTouchId touch_id,
                                     UFTouch* touch);

/**
 * Get the previous value of a property of a touch
 *
 * @param [in] frame The current frame object (const)
 * @param [in] touch The current touch object (const)
 * @param [in] property The property to retrieve a value for
 * @param [out] value The value retrieved
 * @return UFStatusSuccess, UFStatusErrorUnknownProperty, or
 *         UFStatusErrorInvalidTouch
 *
 * The previous value is the value of the property in the previous frame.
 * UFStatusErrorInvalidTouch is returned if the touch did not exist in the
 * previous frame.
 */
FRAME_PUBLIC
UFStatus frame_frame_get_previous_touch_property(UFFrame frame, UFTouch touch,
                                                 UFTouchProperty property,
                                                 void *value);

/**
 * Get the previous value of an axis of a touch
 *
 * @param [in] frame The current frame object (const)
 * @param [in] touch The current touch object (const)
 * @param [in] type The axis to retrieve a value for
 * @param [out] value The value retrieved
 * @return UFStatusSuccess, UFStatusErrorInvalidAxis, or
 *         UFStatusErrorInvalidTouch
 *
 * The previous value is the value of the axis in the previous frame.
 * UFStatusErrorInvalidTouch is returned if the touch did not exist in the
 * previous frame.
 */
FRAME_PUBLIC
UFStatus frame_frame_get_previous_touch_value(UFFrame frame, UFTouch touch,
                                              UFAxisType type, float* value);

/**
 * Get the value of a property of a touch
 *
 * @param [in] touch The touch object (const)
 * @param [in] property The property to retrieve a value for
 * @param [out] value The value retrieved
 * @return UFStatusSuccess or UFStatusErrorUnknownProperty
 */
#ifndef HAS_C_GENERIC_SELECTIONS /* See frame_internal.h */
FRAME_PUBLIC
UFStatus frame_touch_get_property(UFTouch touch, UFTouchProperty property,
                                  void *value);
#endif

/**
 * Get the value of an axis of a touch
 *
 * @param [in] touch The touch object (const)
 * @param [in] type The axis to retrieve a value for
 * @param [out] value The value retrieved
 * @return UFStatusSuccess or UFStatusErrorInvalidAxis
 */
FRAME_PUBLIC
UFStatus frame_touch_get_value(UFTouch touch, UFAxisType type, float *value);

/**
 * @defgroup v2-helpers Helper Functions
 * These helper functions may be used in place of the generic property getters.
 * They are limited to properties that are guaranteed to exist in all instances
 * of the objects.
 * @{
 */

/**
 * Get the type of an event
 *
 * @param [in] event The event object (const)
 * @return The type of the event
 */
FRAME_PUBLIC
UFEventType frame_event_get_type(UFEvent event);

/**
 * Get the time of an event
 *
 * @param [in] event The event object (const)
 * @return The time of the event
 */
FRAME_PUBLIC
uint64_t frame_event_get_time(UFEvent event);

/**
 * Get the number of axes of a device
 *
 * @param [in] device The device object (const)
 * @return The number of axes
 */
FRAME_PUBLIC
unsigned int frame_device_get_num_axes(UFDevice device);

/**
 * Get The resolution of the window coordinates of the device in the X axis
 *
 * @param [in] device The device object (const)
 * @return The resolution in pixels per meter
 */
FRAME_PUBLIC
float frame_device_get_window_resolution_x(UFDevice device);

/**
 * Get The resolution of the window coordinates of the device in the Y axis
 *
 * @param [in] device The device object (const)
 * @return The resolution in pixels per meter
 */
FRAME_PUBLIC
float frame_device_get_window_resolution_y(UFDevice device);

/**
 * Get the number of touches in the frame
 *
 * @param [in] frame The frame object (const)
 * @return The number of touches
 */
FRAME_PUBLIC
uint32_t frame_frame_get_num_touches(UFFrame frame);

/**
 * Get the device of a frame
 *
 * @param [in] frame The frame context object (const)
 * return The device of the window context
 */
FRAME_PUBLIC
UFDevice frame_frame_get_device(UFFrame frame);

/**
 * Get the window ID of a frame
 *
 * @param [in] frame The frame context object (const)
 * @return The window server ID of the window of the frame
 */
FRAME_PUBLIC
UFWindowId frame_frame_get_window_id(UFFrame frame);

/**
 * Get the window server ID of a touch
 *
 * @param [in] touch The touch context object (const)
 * @return The window server ID of the touch
 */
FRAME_PUBLIC
UFTouchId frame_touch_get_id(UFTouch touch);

/**
 * Get the state of a touch
 *
 * @param [in] touch The touch object (const)
 * @return The state of the touch
 */
FRAME_PUBLIC
UFTouchState frame_touch_get_state(UFTouch touch);

/**
 * Get the X window coordinate of a touch
 *
 * @param [in] touch The touch object (const)
 * @return The X window coordinate of the touch
 */
FRAME_PUBLIC
float frame_touch_get_window_x(UFTouch touch);

/**
 * Get the Y window coordinate of a touch
 *
 * @param [in] touch The touch object (const)
 * @return The Y window coordinate of the touch
 */
FRAME_PUBLIC
float frame_touch_get_window_y(UFTouch touch);

/**
 * Get the X device coordinate of a touch
 *
 * @param [in] touch The touch object (const)
 * @return The X device coordinate of the touch
 */
FRAME_PUBLIC
float frame_touch_get_device_x(UFTouch touch);

/**
 * Get the Y device coordinate of a touch
 *
 * @param [in] touch The touch object (const)
 * @return The Y device coordinate of the touch
 */
FRAME_PUBLIC
float frame_touch_get_device_y(UFTouch touch);

/**
 * Get the time of a touch state change
 *
 * @param [in] touch The touch object (const)
 * @return The time of the touch state change
 */
FRAME_PUBLIC
uint64_t frame_touch_get_time(UFTouch touch);

/**
 * Get the start time of a touch
 *
 * @param [in] touch The touch object (const)
 * @return The start time of the touch
 */
FRAME_PUBLIC
uint64_t frame_touch_get_start_time(UFTouch touch);

/** @} */

/** @} */

#include <utouch/frame_internal.h>

#ifdef __cplusplus
}
#endif

#undef HAS_C_GENERIC_SELECTIONS

#endif // UTOUCH_FRAME_UTOUCH_FRAME_H_
