# tests.platform.linux.test_messaging
#
# Author: Eric Casteleijn <eric.casteleijn@canonical.com>
#
# Copyright 2010-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""
Test the messaging on linux. These tests are kind of stupid, but at
least they ensure 100% coverage and hence no silly/syntax errors.
"""

import sys

from mocker import Mocker, ANY
from twisted.internet import defer
from twisted.trial.unittest import TestCase
from ubuntuone.devtools.testcases import skipIf
from ubuntuone.platform.messaging import linux
from ubuntuone.platform.messaging.linux import (
    Messaging,
    _server_callback,
    indicate,
)

FAKE_TIME = 123
FAKE_TIME2 = 456
FAKE_COUNT = 11
FAKE_SENDER = "Mom"
FAKE_ICON = object()


def callback(indicator, message_time=None):
    """Dummy callback."""
    pass


@skipIf(indicate is None, "No valid Indicate found.")
class MessagingTestCase(TestCase):
    """Test the Messaging API."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(MessagingTestCase, self).setUp()
        self.mocker = Mocker()

    @defer.inlineCallbacks
    def tearDown(self):
        yield super(MessagingTestCase, self).tearDown()
        self.mocker.restore()
        self.mocker.verify()

    # pylint: disable=R0913
    def _show_message_setup(self, message_time=None, message_count=None,
                            icon=None, update_count=None, real_callback=False):
        """Set up the mocker expectations for show_method."""
        mock_server = self.mocker.mock()
        if 'gobject' in sys.modules and sys.modules['gobject'] is not None:
            indicate = self.mocker.replace("indicate")
            indicate.indicate_server_ref_default()
        else:
            indicate = self.mocker.replace('gi.repository.Indicate')
            indicate.Server.ref_default()
        self.mocker.result(mock_server)
        mock_server.connect('server-display', _server_callback)
        mock_server.set_type("message.u1")
        mock_server.set_desktop_file(
            "/usr/share/applications/ubuntuone-installer.desktop")
        mock_server.show()
        mock_indicator = self.mocker.mock()
        indicate.Indicator()
        self.mocker.result(mock_indicator)
        mock_indicator.set_property("subtype", "u1")
        mock_indicator.set_property("name", FAKE_SENDER)
        mock_indicator.set_property("sender", FAKE_SENDER)
        mock_indicator.connect("user-display", ANY)
        if icon is not None:
            mock_indicator.set_property_icon("icon", icon)
        if message_count is not None:
            mock_indicator.set_property("count", str(message_count))
        else:
            if message_time is not None:
                mock_indicator.set_property_time("time", message_time)
        mock_indicator.set_property("draw-attention", "true")
        if update_count:
            mock_indicator.get_property("count")
            self.mocker.result("1200")
            mock_indicator.set_property("count", '2500')
        mock_indicator.show()
        if real_callback:
            open_volumes = self.mocker.replace(
                "ubuntuone.platform.messaging.linux.open_volumes")
            open_volumes()
            mock_indicator.set_property('draw-attention', 'false')
            mock_indicator.hide()
        self.mocker.replay()
    # pylint: enable=R0913

    def test_show_message(self):
        """On message, libnotify receives the proper calls."""
        self._show_message_setup()
        messaging = Messaging()
        messaging.show_message(FAKE_SENDER, callback)

    def test_show_message_with_time(self):
        """On message with time, libnotify receives the proper calls."""
        self._show_message_setup(FAKE_TIME)
        messaging = Messaging()
        messaging.show_message(FAKE_SENDER, callback, message_time=FAKE_TIME)

    def test_show_message_with_icon(self):
        """On message with icon, libnotify receives the proper calls."""
        self._show_message_setup(icon=FAKE_ICON)
        messaging = Messaging()
        messaging.show_message(FAKE_SENDER, callback, icon=FAKE_ICON)

    def test_show_message_with_count(self):
        """On message with count, libnotify receives the proper calls."""
        self._show_message_setup(message_count='1200')
        messaging = Messaging()
        messaging.show_message(FAKE_SENDER, callback, message_count=1200)

    def test_update_count(self):
        """On message count update, libnotify receives the proper calls."""
        self._show_message_setup(message_count='1200', update_count='1300')
        messaging = Messaging()
        messaging.show_message(
            FAKE_SENDER, callback, message_count=1200)
        messaging.update_count(messaging.indicators[-1], 1300)

    def test_callback(self):
        """When the callback fires, libnotify receives the proper calls."""
        self._show_message_setup(real_callback=True)
        messaging = Messaging()
        messaging.show_message(FAKE_SENDER)
        self.assertEquals(1, len(messaging.indicators))
        actual_callback = messaging.create_callback()
        actual_callback(messaging.indicators[-1])
        self.assertEquals(0, len(messaging.indicators))

    def test_open_u1(self):
        """Check that the proper action is executed."""
        data = []

        self.patch(linux.glib, "spawn_command_line_async", data.append)
        _server_callback(None)
        self.assertEqual(data, ['ubuntuone-control-panel-qt'])
