# -*- coding: utf-8 -*-
#
# Authors: Manuel de la Pena <manuel@canonical.com>
#          Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""IPC tests on windows."""

from collections import namedtuple

from mocker import MockerTestCase, Mocker, ANY

from twisted.trial.unittest import TestCase
from twisted.internet import defer
from twisted.spread.pb import (
    DeadReferenceError,
    NoSuchMethod,
    PBServerFactory,
    PBClientFactory,
    Root,
    RemoteReference
)

from ubuntuone.platform.windows import ipc
from ubuntuone.platform.windows.ipc import (
    AllEventsSender,
    Config,
    Events,
    Folders,
    FileSystem,
    IPCInterface,
    IPCRoot,
    PublicFiles,
    RemoteMeta,
    Shares,
    SignalBroadcaster,
    Status,
    SyncDaemon,
    ipc_client_connect,
    ipc_server_listen,
)

from ubuntuone.platform.windows.ipc_client import (
    signal,
    ConfigClient,
    EventsClient,
    FoldersClient,
    FileSystemClient,
    PublicFilesClient,
    RemoteClient,
    StatusClient,
    SyncDaemonClient,
    SharesClient,
)

TEST_PORT = 40404


class RandomException(Exception):
    """A random exception."""


class FakeActivationClient(object):
    """A fake ActivationClient."""

    def __init__(self, config):
        """Initialize this fake instance."""
        self.config = config

    def get_active_port(self):
        """Return the port where the pb server is running."""
        return defer.succeed(self.config.port)


class FakeReactor(object):
    """A fake reactor."""

    def __init__(self):
        """Initialize this faker."""
        self.connection_class = namedtuple("Connection", "host port factory")
        self.connections = []

    def connectTCP(self, host, port, factory):
        """Store the connected factory."""
        connection = self.connection_class(host, port, factory)
        self.connections.append(connection)


class BaseIPCTestCase(TestCase):
    """Set the ipc to a random port for this instance."""

    def setUp(self):
        """Initialize this test instance."""
        self.patch(ipc, "get_sd_pb_port", lambda: TEST_PORT)
        self.patch(ipc, "ActivationClient", FakeActivationClient)


class SaveProtocolServerFactory(PBServerFactory):
    """A PBServerFactory that saves the latest connected client."""

    protocolInstance = None

    def clientConnectionMade(self, protocol):
        """Keep track of the given protocol."""
        self.protocolInstance = protocol


class FakeDecoratedObject(object):
    """An object that has decorators."""

    def __init__(self):
        """Create a new instance."""
        super(FakeDecoratedObject, self).__init__()

    @signal
    def on_no_args(self):
        """Get no args passwed."""

    @signal
    def on_just_args(self, *args):
        """Just get args."""

    @signal
    def on_just_kwargs(self, **kwargs):
        """Just get kwargs."""

    @signal
    def on_both_args(self, *args, **kwargs):
        """Both args."""


class SignalTestCase(MockerTestCase):
    """Test the signal decorator."""

    def setUp(self):
        """Set the tests."""
        super(SignalTestCase, self).setUp()
        self.fake_object = FakeDecoratedObject()
        self.cb = self.mocker.mock()

    def test_no_args(self):
        """Test when the cb should have no args."""
        self.fake_object.on_no_args_cb = self.cb
        self.cb()
        self.mocker.replay()
        self.fake_object.on_no_args()

    def test_just_args(self):
        """Test when the cb just has *args"""
        first = 'first'
        second = 'second'
        self.fake_object.on_just_args_cb = self.cb
        self.cb(first, second)
        self.mocker.replay()
        self.fake_object.on_just_args(first, second)

    def test_just_kwargs(self):
        """Test when the cb just has kwargs."""
        first = 'first'
        second = 'second'
        self.fake_object.on_just_kwargs_cb = self.cb
        self.cb(first=first, second=second)
        self.mocker.replay()
        self.fake_object.on_just_kwargs(first=first, second=second)

    def test_just_kwargs_empty(self):
        """Test when the cb just has kwargs."""
        self.fake_object.on_just_kwargs_cb = self.cb
        self.cb()
        self.mocker.replay()
        self.fake_object.on_just_kwargs()

    def test_both_args(self):
        """Test with args and kwargs."""
        first = 'first'
        second = 'second'
        self.fake_object.on_both_args_cb = self.cb
        self.cb(first, second, first=first, second=second)
        self.mocker.replay()
        self.fake_object.on_both_args(first, second, first=first,
                                      second=second)

    def test_both_args_no_kwargs(self):
        """Test with args and kwargs."""
        first = 'first'
        second = 'second'
        self.fake_object.on_both_args_cb = self.cb
        self.cb(first, second)
        self.mocker.replay()
        self.fake_object.on_both_args(first, second)

    def test_both_args_no_args(self):
        """Test with args and kwargs."""
        first = 'first'
        second = 'second'
        self.fake_object.on_both_args_cb = self.cb
        self.cb(first=first, second=second)
        self.mocker.replay()
        self.fake_object.on_both_args(first=first, second=second)


class PerspectiveBrokerTestCase(TestCase):
    """Base test case for the IPC used on Windows."""

    def setUp(self):
        """Setup tests."""
        super(PerspectiveBrokerTestCase, self).setUp()
        self.config = Config(None, None)
        self.status = Status(None, None, None)
        self.events = Events(None)
        self.sync = SyncDaemon(None, None, None, None)
        self.shares = Shares(None, None)
        self.folders = Folders(None, None)
        self.public_files = PublicFiles(None, None)
        self.fs = FileSystem(None, None)


class FakeRoot(Root):
    """Root object that exposes the diff referenceable objects."""

    def __init__(self, status, events, sync_daemon, file_system, shares,
                 config, folders, public_files):
        """Create a new instance that will expose the objects."""
        self._status = status
        self._events = events
        self._sync_daemon = sync_daemon
        self._file_system = file_system
        self._shares = shares
        self._config = config
        self._folders = folders
        self._public_files = public_files

    def remote_get_status(self):
        """Return the status remote object."""
        return self._status

    def remote_get_events(self):
        """Return the events remote object."""
        return self._events

    def remote_get_sync_daemon(self):
        """Return the sync daemon remote object."""
        return self._sync_daemon

    def remote_get_file_system(self):
        """Return the file system remote object."""
        return self._file_system

    def remote_get_shares(self):
        """Return the shares remote object."""
        return self._shares

    def remote_get_config(self):
        """Return the config remote object."""
        return self._config

    def remote_get_folders(self):
        """Return the folders remote object."""
        return self._folders

    def remote_get_public_files(self):
        """Return the public files remote object."""
        return self._public_files


class TestSignalBroadcaster(MockerTestCase):
    """Test the signal broadcaster code."""

    def setUp(self):
        super(TestSignalBroadcaster, self).setUp()
        self.client = self.mocker.mock()
        self.broad_caster = SignalBroadcaster()

    def test_remote_register_to_signals(self):
        """Assert that the client was added."""
        self.mocker.replay()
        signals = ["demo_signal1", "demo_signal2"]
        self.broad_caster.remote_register_to_signals(self.client, signals)
        for signal in signals:
            clients = self.broad_caster.clients_per_signal[signal]
            self.assertTrue(self.client in clients)

    def test_emit_signal(self):
        """Assert that the client method was called."""
        first = 1
        second = 2
        word = 'word'
        signal_name = 'on_test'
        deferred = self.mocker.mock()
        self.client.callRemote(signal_name, first, second, word=word)
        self.mocker.result(deferred)
        deferred.addErrback(ANY, ANY, ANY)
        deferred.addErrback(ANY, ANY, ANY)
        self.mocker.replay()
        signals = [signal_name]
        self.broad_caster.remote_register_to_signals(self.client, signals)
        self.broad_caster.emit_signal(signal_name, first, second, word=word)

    def test_emit_signal_dead_reference(self):
        """Test dead reference while emitting a signal."""
        sample_signal = "sample_signal"
        fake_remote_client = self.mocker.mock()
        fake_remote_client.callRemote(sample_signal)
        self.mocker.throw(DeadReferenceError())
        self.mocker.replay()

        sb = SignalBroadcaster()
        sb.remote_register_to_signals(fake_remote_client, [sample_signal])
        self.assertIn(fake_remote_client, sb.clients_per_signal[sample_signal])
        sb.emit_signal(sample_signal)
        self.assertNotIn(fake_remote_client,
                         sb.clients_per_signal[sample_signal])

    def test_emit_signal_some_dead_some_not(self):
        """Test a clean reference after a dead one."""
        sample_signal = "sample_signal"
        fake_dead_remote = self.mocker.mock()
        fake_alive_remote = self.mocker.mock()

        fake_dead_remote.callRemote(sample_signal)
        self.mocker.throw(DeadReferenceError())
        fake_alive_remote.callRemote(sample_signal)
        self.mocker.result(defer.succeed(None))
        self.mocker.replay()

        sb = SignalBroadcaster()
        sb.remote_register_to_signals(fake_dead_remote, [sample_signal])
        sb.remote_register_to_signals(fake_alive_remote, [sample_signal])
        sb.emit_signal(sample_signal)


class FakeRemoteClient(object):
    """A fake RemoteClient."""

    missing_signal = "missing"
    failing_signal = "failing"
    random_exception = RandomException()

    def callRemote(self, signal_name):
        """Fake a call to a given remote method."""
        if signal_name == self.missing_signal:
            return defer.fail(NoSuchMethod())
        if signal_name == self.failing_signal:
            return defer.fail(self.random_exception)
        raise ValueError("not a valid fake signal name")


class SignalBroadcasterFailuresTestCase(TestCase):
    """Test some signal broadcaster failures."""

    def test_emit_signal_ignore_missing_handlers(self):
        """A missing signal handler should just log a debug line."""
        debugs = []
        self.patch(ipc.logger, "debug", lambda *args: debugs.append(args))

        fake_remote_client = FakeRemoteClient()

        sb = SignalBroadcaster()
        signals = [fake_remote_client.missing_signal]
        sb.remote_register_to_signals(fake_remote_client, signals)
        sb_clients = sb.clients_per_signal[fake_remote_client.missing_signal]
        self.assertIn(fake_remote_client, sb_clients)
        sb.emit_signal(fake_remote_client.missing_signal)

        expected = (
            SignalBroadcaster.MSG_NO_SIGNAL_HANDLER,
            fake_remote_client.missing_signal,
            fake_remote_client,
        )
        self.assertIn(expected, debugs)

    def test_emit_signal_log_other_errors(self):
        """Other errors should be logged as warnings."""
        warnings = []
        self.patch(ipc.logger, "warning", lambda *args: warnings.append(args))

        fake_remote_client = FakeRemoteClient()

        sb = SignalBroadcaster()
        signals = [fake_remote_client.failing_signal]
        sb.remote_register_to_signals(fake_remote_client, signals)
        sb_clients = sb.clients_per_signal[fake_remote_client.failing_signal]
        self.assertIn(fake_remote_client, sb_clients)
        sb.emit_signal(fake_remote_client.failing_signal)

        expected = (
            SignalBroadcaster.MSG_COULD_NOT_EMIT_SIGNAL,
            fake_remote_client.failing_signal,
            fake_remote_client,
            fake_remote_client.random_exception,
        )
        self.assertIn(expected, warnings)


class FakeRemoteObject(object):
    """A test helper."""

    def __init__(self):
        """Initialize this test helper."""
        self.called = []

    def callRemote(self, *args):
        """A remote call to this object."""
        self.called.append(args)


class RemoteClientTestCase(TestCase):
    """Tests for the RemoteClient class."""

    def test_register_to_signals(self):
        """Test the register_to_signals method."""
        fake_remote_object = FakeRemoteObject()
        client = RemoteClient(fake_remote_object)
        client.signal_handlers = ["on_abc"]
        client.register_to_signals()
        expected = [
            ("register_to_signals", client, client.signal_handlers)
        ]
        self.assertEqual(fake_remote_object.called, expected)


class TestStatusEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Tests that the emit methods are correct."""

    def setUp(self):
        """Setup tests."""
        super(TestStatusEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.status.syncdaemon_status = self.mocker.mock()
        self.status.emit_signal = self.signal_method

    def test_emit_content_queue_changed(self):
        """Emit ContentQueueChanged."""
        self.signal_method('on_content_queue_changed')
        self.mocker.replay()
        self.status.emit_content_queue_changed()

    def test_emit_invalid_name(self):
        """Emit InvalidName."""
        dirname = 'dirname'
        filename = 'filename'
        self.signal_method('on_invalid_name', unicode(dirname), str(filename))
        self.mocker.replay()
        self.status.emit_invalid_name(dirname, filename)

    def test_emit_broken_node(self):
        """Emit BrokenNode."""
        volume_id = 'volume_id'
        node_id = 'node_id'
        mdid = 'mdid'
        path = 'path'
        self.signal_method('on_broken_node', volume_id, node_id, mdid,
                           path.decode('utf8'))
        self.mocker.replay()
        self.status.emit_broken_node(volume_id, node_id, mdid, path)

    def test_emit_status_changed(self):
        """Emit StatusChanged."""
        status = 'status'
        self.status.syncdaemon_status.current_status()
        self.mocker.result(status)
        self.signal_method('on_status_changed', status)
        self.mocker.replay()
        self.status.emit_status_changed(status)

    def test_emit_download_started(self):
        """Emit DownloadStarted."""
        download = 'download'
        self.signal_method('on_download_started', download)
        self.mocker.replay()
        self.status.emit_download_started(download)

    def test_emit_download_file_progress(self):
        """Emit DownloadFileProgress."""
        download = 'download'
        string_info = {'test':'2', 'name':'3'}
        self.signal_method('on_download_file_progress', download, string_info)
        self.mocker.replay()
        self.status.emit_download_file_progress(download, test=2, name=3)

    def test_emit_download_finished(self):
        """Emit DownloadFinished."""
        download = 'download'
        string_info = {'test':'2', 'name':'3'}
        self.signal_method('on_download_finished', download, string_info)
        self.mocker.replay()
        self.status.emit_download_finished(download, test=2, name=3)

    def test_emit_upload_started(self):
        """Emit UploadStarted."""
        upload = 'upload'
        self.signal_method('on_upload_started', upload)
        self.mocker.replay()
        self.status.emit_upload_started(upload)

    def test_emit_upload_file_progress(self):
        """Emit UploadFileProgress."""
        upload = 'upload'
        string_info = {'test':'2', 'name':'3'}
        self.signal_method('on_upload_file_progress', upload, string_info)
        self.mocker.replay()
        self.status.emit_upload_file_progress(upload, test=2, name=3)

    def test_emit_upload_finished(self):
        """Emit UploadFinished."""
        upload = 'upload'
        string_info = {'test':'2', 'name':'3'}
        self.signal_method('on_upload_finished', upload, string_info)
        self.mocker.replay()
        self.status.emit_upload_finished(upload, test=2, name=3)

    def test_emit_account_changed(self):
        """Emit AccountChanged."""
        bytes = '5'
        account_info = self.mocker.mock()
        account_info.purchased_bytes
        self.mocker.result(bytes)
        info_dict = {'purchased_bytes' : unicode('5')}
        self.signal_method('on_account_changed', info_dict)
        self.mocker.replay()
        self.status.emit_account_changed(account_info)

    def test_emit_metaqueue_changed(self):
        """Emit MetaQueueChanged."""
        self.signal_method('on_metaqueue_changed')
        self.mocker.replay()
        self.status.emit_metaqueue_changed()


class TestEventsEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    def setUp(self):
        """Setup tests."""
        super(TestEventsEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.events.emit_signal = self.signal_method

    def test_emit_event(self):
        """Test the Event signal."""
        items = {1:2}
        event = self.mocker.mock()
        event.iteritems()
        self.mocker.result(items.iteritems())
        self.signal_method('on_event', {'1':'2'})
        self.mocker.replay()
        self.events.emit_event(event)


class TestSyncDaemonEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    def setUp(self):
        """Setup tests."""
        super(TestSyncDaemonEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.sync.emit_signal = self.signal_method

    def test_emit_root_mismatch(self):
        """Emit RootMismatch signal."""
        root_id = 'root_id'
        new_root_id = 'new_root_id'
        self.signal_method('on_root_mismatch', root_id, new_root_id)
        self.mocker.replay()
        self.sync.emit_root_mismatch(root_id, new_root_id)

    def test_emit_quota_exceeded(self):
        """Emit QuotaExceeded signal."""
        volume_dict = 'volume_dict'
        self.signal_method('on_quota_exceeded', volume_dict)
        self.mocker.replay()
        self.sync.emit_quota_exceeded(volume_dict)


class TestSharesEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    def setUp(self):
        """Setup tests."""   
        super(TestSharesEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.shares.emit_signal = self.signal_method
        self.shares.syncdaemon_shares = self.mocker.mock()
        self.get_share_dict = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.get_share_dict')

    def test_emit_share_changed_deleted(self):
        """Test emit share changed on a delete event."""
        message = 'deleted'
        share = 'share'
        share_dict = 'share_dict'
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_deleted', share_dict)
        self.mocker.replay()
        self.shares.emit_share_changed(message, share)

    def test_emit_share_changed_changed(self):
        """Test emit share changed on a changed event."""
        message = 'changed'
        share = 'share'
        share_dict = 'share_dict'
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_changed', share_dict)
        self.mocker.replay()
        self.shares.emit_share_changed(message, share)

    def test_emit_share_delete_error(self):
        """Emit ShareDeleteError signal."""
        share = 'share'
        raised_error = 'error'
        share_dict = 'share_dict'
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_delete_error', share_dict, raised_error)
        self.mocker.replay()
        self.shares.emit_share_delete_error(share, raised_error)

    def test_emit_free_space(self):
        """Emit ShareChanged when free space changes """
        free_bytes = '0'
        share_dict = shares = {'1': 'share', 'free_bytes':'0' }
        share = 'share'
        share_id = '1'
        self.shares.syncdaemon_shares.shares
        self.mocker.result(shares)
        self.shares.syncdaemon_shares.shares
        self.mocker.result(shares)
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_changed', share_dict)
        self.mocker.replay()
        self.shares.emit_free_space(share_id, free_bytes)

    def test_emit_share_created(self):
        """Emit ShareCreated signal """
        share_info = 'info'
        self.signal_method('on_share_created', share_info)
        self.mocker.replay()
        self.shares.emit_share_created(share_info)

    def test_emit_share_create_error(self):
        """Emit ShareCreateError signal."""
        share_info = 'id'
        raised_error = 'error'
        info = 'info'
        self.shares.syncdaemon_shares.get_create_error_share_info(share_info)
        self.mocker.result(info)
        self.signal_method('on_share_create_error', info, raised_error)
        self.mocker.replay()
        self.shares.emit_share_create_error(share_info, raised_error)

    def test_emit_share_answer_response(self):
        """Emits ShareAnswerResponse signal."""
        share_id = 'id'
        answer = 'yes'
        raised_error = 'boom'
        answer_info = dict(volume_id=share_id, answer=answer, error=raised_error)
        self.signal_method('on_share_answer_response', answer_info)
        self.mocker.replay()
        self.shares.emit_share_answer_response(share_id, answer, raised_error)

    def test_emit_new_share(self):
        """Emits NewShare signal."""
        share_id = 'id'
        share = 'share'
        share_dict = {'share':'id'}
        self.shares.syncdaemon_shares.get_volume(share_id)
        self.mocker.result(share)
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_new_share', share_dict)
        self.mocker.replay()
        self.shares.emit_new_share(share_id)

    def test_emit_share_subscribed(self):
        """Emit the ShareSubscribed signal"""
        share = 'share'
        share_dict = {'share' : 'id'}
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_subscribed', share_dict)
        self.mocker.replay()
        self.shares.emit_share_subscribed(share)

    def test_emit_share_subscribe_error(self):
        """Emit the ShareSubscribeError signal"""
        share_id = 'id'
        raised_error = 'error'
        self.signal_method('on_share_subscribe_error',
                           {'id': share_id}, str(raised_error))
        self.mocker.replay()
        self.shares.emit_share_subscribe_error(share_id, raised_error)

    def test_emit_share_unsubscribed(self):
        """Emit the ShareUnSubscribed signal"""
        share = 'share'
        share_dict = {'share':'id'}
        self.get_share_dict(share)
        self.mocker.result(share_dict) 
        self.signal_method('on_share_unsubscribed', share_dict)
        self.mocker.replay()
        self.shares.emit_share_unsubscribed(share)

    def test_emit_share_unsubscribe_error(self):
        """Emit the ShareUnSubscribeError signal"""
        share_id = 'id'
        raised_error = 'error'
        self.signal_method('on_share_unsubscribe_error',{'id': share_id}, str(raised_error))
        self.mocker.replay()
        self.shares.emit_share_unsubscribe_error(share_id, raised_error)


class TestFoldersEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    def setUp(self):
        """Setup tests."""
        super(TestFoldersEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.folders.emit_signal = self.signal_method
        self.get_udf_dict = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.get_udf_dict')

    def test_emit_folder_deleted(self):
        """Emit the FolderCreated signal"""
        folder = 'folder'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.signal_method('on_folder_deleted', udf_dict)
        self.mocker.replay()
        self.folders.emit_folder_deleted(folder)

    def test_emit_folder_delete_error(self):
        """Emit the FolderCreateError signal"""
        folder = 'folder'
        raised_error = 'error'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.signal_method('on_folder_delete_error', udf_dict, str(raised_error))
        self.mocker.replay()
        self.folders.emit_folder_delete_error(folder, raised_error)

    def test_emit_folder_subscribed(self):
        """Emit the FolderSubscribed signal"""
        folder = 'folder'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.signal_method('on_folder_subscribed', udf_dict)
        self.mocker.replay()
        self.folders.emit_folder_subscribed(folder)

    def test_emit_folder_subscribe_error(self):
        """Emit the FolderSubscribeError signal"""
        folder_id = 'id'
        raised_error = 'error'
        self.signal_method('on_folder_subscribe_error',
                           {'id':folder_id}, str(raised_error))
        self.mocker.replay()
        self.folders.emit_folder_subscribe_error(folder_id, raised_error)

    def test_emit_folder_unsubscribed(self):
        """Emit the FolderUnSubscribed signal"""
        folder = 'folder'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.signal_method('on_folder_unsubscribed', udf_dict)
        self.mocker.replay()
        self.folders.emit_folder_unsubscribed(folder)

    def test_emit_folder_unsubscribe_error(self):
        """Emit the FolderUnSubscribeError signal"""
        folder_id = 'id'
        raised_error = 'error'
        self.signal_method('on_folder_unsubscribe_error',
                           {'id':folder_id}, str(raised_error))
        self.mocker.replay()
        self.folders.emit_folder_unsubscribe_error(folder_id, raised_error)


class TestPublicFilesEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    def setUp(self):
        """Setup tests."""
        super(TestPublicFilesEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.public_files.emit_signal = self.signal_method
        self.public_files.syncdaemon_public_files = self.mocker.mock()
        self.bool_str = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.bool_str')

    def test_emit_public_access_changed(self):
        """Emit the PublicAccessChanged signal."""
        share_id = 'share_id'
        node_id = 'node_id'
        path = 'path'
        is_public = True
        public_url = 'url'
        self.public_files.syncdaemon_public_files.get_path(share_id, node_id)
        self.mocker.result(path)
        self.bool_str(is_public)
        self.mocker.result('True')
        self.signal_method('on_public_access_changed',
                           dict(share_id=share_id, node_id=node_id,
                                is_public='True', public_url=public_url,
                                path=path))
        self.mocker.replay()
        self.public_files.emit_public_access_changed(share_id, node_id, 
                                                     is_public, public_url)

    def test_emit_public_access_change_error(self):
        """Emit the PublicAccessChangeError signal."""
        share_id = 'share_id'
        node_id = 'node_id'
        raised_error = 'error'
        path = 'path'
        self.public_files.syncdaemon_public_files.get_path(share_id, node_id)
        self.mocker.result(path)
        self.signal_method('on_public_access_change_error',
                           dict(share_id=share_id, node_id=node_id, path=path),
                           raised_error)
        self.mocker.replay()
        self.public_files.emit_public_access_change_error(share_id, node_id,
                                                          raised_error)

    def test_emit_public_files_list(self):
        """Emit the PublicFilesList signal."""
        volume_id = 'volume_id'
        node_id = 'node_id'
        public_url = 'url'
        path = 'path'
        public_files = [dict(volume_id=volume_id, node_id=node_id,
                             public_url=public_url)]
        files = [dict(volume_id=volume_id, node_id=node_id,
                      public_url=public_url, path=path)]
        self.public_files.syncdaemon_public_files.get_path(volume_id, node_id)
        self.mocker.result(path)
        self.signal_method('on_public_files_list',files)
        self.mocker.replay()
        self.public_files.emit_public_files_list(public_files)

    def test_emit_public_files_list_error(self):
        """Emit the PublicFilesListError signal."""
        raised_error = 'error'
        self.signal_method('on_public_files_list_error', raised_error)
        self.mocker.replay()
        self.public_files.emit_public_files_list_error(raised_error)


class IPCFunctionsTestCase(BaseIPCTestCase):
    """Test the functions in the ubuntuone.platform.windows.ipc module."""
    timeout = 5

    @defer.inlineCallbacks
    def test_connect(self):
        """Test the ipc_client_connect function."""
        fake_reactor = FakeReactor()
        self.patch(ipc, "reactor", fake_reactor)
        fake_factory = object()
        yield ipc_client_connect(fake_factory)
        self.assertEqual(len(fake_reactor.connections), 1)
        c = fake_reactor.connections[0]
        self.assertEqual(c.port, TEST_PORT)
        self.assertEqual(c.factory, fake_factory)


class TestStatusClient(BaseIPCTestCase):
    """Test the status client class."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        super(TestStatusClient, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.status = Status(None, None, None)
        self.status.syncdaemon_status = self.root
        # start pb
        self.syncdaemon_root = FakeRoot(self.status, None, None, None,
                                        None, None, None, None)
        self.server_factory = SaveProtocolServerFactory(self.syncdaemon_root)
        self.listener = ipc_server_listen(self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = yield ipc_client_connect(self.client_factory)

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_status')
        client = StatusClient(remote)
        # set the cb
        callback_names = [signal + "_cb" for signal in client.signal_handlers]
        yield client.register_to_signals()
        for callback_name in callback_names:
            setattr(client, callback_name, self.mocker.mock())
        defer.returnValue(client)

    def test_current_status(self):
        """Test the current status method."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.current_status()
            self.mocker.replay()
            yield client.current_status()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_current_downloads(self):
        """Return a list of files with a download in progress."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.current_downloads()
            self.mocker.replay()
            yield client.current_downloads()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_waiting(self):
        """Test the waiting method."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.waiting()
            self.mocker.result({})
            self.mocker.replay()
            yield client.waiting()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_waiting_metadata(self):
        """Test the waiting metadata method."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.waiting_metadata()
            self.mocker.replay()
            yield client.waiting_metadata()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_waiting_content(self):
        """Test the waiting content method."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.waiting_content()
            self.mocker.replay()
            yield client.waiting_content()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_current_uploads(self):
        """Test the current uploads method."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.current_uploads()
            self.mocker.replay()
            yield client.current_uploads()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_emit_content_queue_changed(self):
        """Test that the cb was called."""

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_content_queue_changed_cb()
            self.mocker.replay()
            self.status.emit_content_queue_changed()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_invalid_name(self):
        """Emit InvalidName."""
        dirname = 'dirname'
        filename = 'filename'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_invalid_name_cb(dirname, filename)
            self.mocker.replay()
            self.status.emit_invalid_name(dirname, filename)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_broken_node(self):
        """Emit BrokenNode."""
        volume_id = 'volume_id'
        node_id = 'node_id'
        mdid = 'mdid'
        path = 'path'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_broken_node_cb(volume_id, node_id, mdid, path)
            self.mocker.replay()
            self.status.emit_broken_node(volume_id, node_id, mdid, path)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_status_changed(self):
        """Emit StatusChanged."""
        state = 'state'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.status.current_status()
            self.mocker.result(state)
            client.on_status_changed_cb(state)
            self.mocker.replay()
            self.status.emit_status_changed(state)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_download_started(self):
        """Emit DownloadStarted."""
        download = 'download'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_download_started_cb(download)
            self.mocker.replay()
            self.status.emit_download_started(download)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_download_file_progress(self):
        """Emit DownloadFileProgress."""
        download = 'download'
        info = dict(name='name', info='info')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_download_file_progress_cb(download, info)
            self.mocker.replay()
            self.status.emit_download_file_progress(download, **info)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_download_finished(self):
        """Emit DownloadFinished."""
        download = 'download'
        info = dict(name='name', info='info')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_download_finished_cb(download, info)
            self.mocker.replay()
            self.status.emit_download_finished(download, **info)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_upload_started(self):
        """Emit UploadStarted."""
        upload = 'upload'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_upload_started_cb(upload)
            self.mocker.replay()
            self.status.emit_upload_started(upload)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_upload_file_progress(self):
        """Emit UploadFileProgress."""
        upload = 'upload'
        info = dict(name='name', info='info')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_upload_file_progress_cb(upload, info)
            self.mocker.replay()
            self.status.emit_upload_file_progress(upload, **info)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_upload_finished(self):
        """Emit UploadFinished."""
        upload = 'upload'
        info = dict(name='name', info='info')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_upload_finished_cb(upload, info)
            self.mocker.replay()
            self.status.emit_upload_finished(upload, **info)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_account_changed(self):
        """Emit AccountChanged."""
        info = self.mocker.mock()
        info_dict = dict(purchased_bytes='10')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            info.purchased_bytes
            self.mocker.result(info_dict['purchased_bytes'])
            client.on_account_changed_cb(info_dict)
            self.mocker.replay()
            self.status.emit_account_changed(info)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_metaqueue_changed(self):
        """Emit MetaQueueChanged."""

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_metaqueue_changed_cb()
            self.mocker.replay()
            self.status.emit_metaqueue_changed()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d


class TestEventsClient(BaseIPCTestCase):
    """Test the status client class."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        super(TestEventsClient, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.events = Events(None)
        self.events.events = self.root
        # start pb
        self.syncdaemon_root = FakeRoot(None, self.events, None, None,
                                        None, None, None, None)
        self.server_factory = SaveProtocolServerFactory(self.syncdaemon_root)
        self.listener = ipc_server_listen(self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = yield ipc_client_connect(self.client_factory)

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_events')
        client = EventsClient(remote)
        yield client.register_to_signals()
        # set the cb
        client.on_event_cb = self.mocker.mock()
        defer.returnValue(client)

    def test_push_event(self):
        """Push an event to the event queue."""
        event_name = 'Name'
        args = ('first', 'second')

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.push_event(event_name, args)
            self.mocker.replay()
            yield client.push_event(event_name, args)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_emit_event(self):
        """Emit on event."""
        event = dict(name='name', info='info')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_event_cb(event)
            self.mocker.replay()
            self.events.emit_event(event)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d


class TestSyncDaemonClient(BaseIPCTestCase):
    """Test the status client class."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        super(TestSyncDaemonClient, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.daemon = SyncDaemon(None, None, None, None)
        self.daemon.service = self.root
        # start pb
        self.syncdaemon_root = FakeRoot(None, None, self.daemon, None,
                                        None, None, None, None)
        self.server_factory = SaveProtocolServerFactory(self.syncdaemon_root)
        self.listener = ipc_server_listen(self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = yield ipc_client_connect(self.client_factory)

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_sync_daemon')
        client = SyncDaemonClient(remote)
        yield client.register_to_signals()
        # set the cb
        for signal in ['on_root_mismatch_cb',
                       'on_quota_exceeded_cb']:
            setattr(client, signal, self.mocker.mock())
        defer.returnValue(client)

    def test_connect(self):
        """Connect to the server."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.connect()
            self.mocker.replay()
            yield client.connect()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_disconnect(self):
        """Disconnect from the server. """

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.disconnect()
            self.mocker.replay()
            yield client.disconnect()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_rootdir(self):
        """Returns the root dir/mount point. """

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_rootdir()
            self.mocker.replay()
            yield client.get_rootdir()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_sharesdir(self):
        """Returns the shares dir/mount point. """

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_sharesdir()
            self.mocker.replay()
            yield client.get_sharesdir()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_sharesdir_link(self):
        """Returns the shares dir/mount point. """

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_sharesdir_link()
            self.mocker.replay()
            yield client.get_sharesdir_link()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_wait_for_nirvana(self):
        """Call the reply handler when there are no more
        events or transfers.
        """
        last_event_interval = 'interval'
        reply_handler = lambda: None
        error_handler = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.wait_for_nirvana(last_event_interval, ANY, ANY)
            self.mocker.replay()
            yield client.wait_for_nirvana(last_event_interval, reply_handler,
                                          error_handler)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_quit(self):
        """Shutdown the syncdaemon. """

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.quit(None, None)
            self.mocker.replay()
            yield client.quit()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_rescan_from_scratch(self):
        """Request a rescan from scratch of the volume with volume_id."""
        volume_id = 'volume_id'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.rescan_from_scratch(volume_id)
            self.mocker.replay()
            yield client.rescan_from_scratch(volume_id)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_emit_root_mismatch(self):
        """Emit RootMismatch signal."""
        root_id = 'root_id'
        new_root_id = 'new_root_id'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_root_mismatch_cb(root_id, new_root_id)
            self.mocker.replay()
            self.daemon.emit_root_mismatch(root_id, new_root_id)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_emit_quota_exceeded(self):
        """Emit QuotaExceeded signal."""
        volume_dict = dict(id='id', data='data')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_quota_exceeded_cb(volume_dict)
            self.mocker.replay()
            self.daemon.emit_quota_exceeded(volume_dict)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d


class TestFileSystemClient(BaseIPCTestCase):
    """Test the status client class."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        super(TestFileSystemClient, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.fs = FileSystem(None, None)
        self.fs.syncdaemon_filesystem  = self.root
        # start pb
        self.syncdaemon_root = FakeRoot(None, None, None, self.fs,
                                        None, None, None, None)
        self.server_factory = SaveProtocolServerFactory(self.syncdaemon_root)
        self.listener = ipc_server_listen(self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = yield ipc_client_connect(self.client_factory)

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_file_system')
        client = FileSystemClient(remote)
        defer.returnValue(client)

    def test_get_metadata(self):
        """Return the metadata (as a dict) for the specified path."""
        path = 'path'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_metadata(path)
            self.mocker.replay()
            yield client.get_metadata(path)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_metadata_by_node(self):
        """Return the metadata (as a dict) for the specified share/node."""
        share_id = 'share_id'
        node_id = 'node_id'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_metadata_by_node(share_id, node_id)
            self.mocker.replay()
            yield client.get_metadata_by_node(share_id, node_id)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_metadata_and_quick_tree_synced(self):
        """Return the dict with the attributes of the metadata for the path."""
        path = 'path'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_metadata_and_quick_tree_synced(path)
            self.mocker.replay()
            yield client.get_metadata_and_quick_tree_synced(path)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_dirty_nodes(self):
        """Return a list of dirty nodes.""" 

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_dirty_nodes()
            self.mocker.replay()
            yield client.get_dirty_nodes()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

class TestSyncDaemonClient(BaseIPCTestCase):
    """Test the status client class."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        super(TestSyncDaemonClient, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.get_share_dict = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.get_share_dict')
        self.shares = Shares(None, None)
        self.shares.syncdaemon_shares = self.root
        # start pb
        self.syncdaemon_root = FakeRoot(None, None, None, None,
                                        self.shares, None, None, None)
        self.server_factory = SaveProtocolServerFactory(self.syncdaemon_root)
        self.listener = ipc_server_listen(self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = yield ipc_client_connect(self.client_factory)

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_shares')
        client = SharesClient(remote)
        yield client.register_to_signals()
        # set the cb
        for signal in ['on_share_deleted_cb',
                       'on_share_changed_cb',
                       'on_share_delete_error_cb',
                       'on_share_created_cb',
                       'on_share_create_error_cb',
                       'on_share_answer_response_cb',
                       'on_new_share_cb',
                       'on_share_subscribed_cb',
                       'on_share_subscribe_error_cb',
                       'on_share_unsubscribed_cb',
                       'on_share_unsubscribe_error_cb']:
            setattr(client, signal, self.mocker.mock())
        defer.returnValue(client)

    def test_get_shares(self):
        """Return a list of dicts, each dict represents a share."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_shares()
            self.mocker.replay()
            yield client.get_shares()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_accept_share(self):
        """Accept a share."""
        share_id = 'share_id'
        reply_handler = lambda: None
        error_handler = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.accept_share(share_id, ANY, ANY)
            self.mocker.replay()
            yield client.accept_share(share_id, reply_handler, error_handler)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_reject_share(self):
        """Reject a share."""
        share_id = 'share_id'
        reply_handler = lambda: None
        error_handler = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.reject_share(share_id, ANY, ANY)
            self.mocker.replay()
            yield client.reject_share(share_id, reply_handler, error_handler)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_delete_share(self):
        """Delete a Share, both kinds: "to me" and "from me"."""
        share_id = 'share_id'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.delete_share(share_id)
            self.mocker.replay()
            yield client.delete_share(share_id)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_subscribe(self):
        """Subscribe to the specified share."""
        share_id = 'share_id'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.subscribe(share_id)
            self.mocker.replay()
            yield client.subscribe(share_id)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_unsubscribe(self):
        """Unsubscribe from the specified share."""
        share_id = 'share_id'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.unsubscribe(share_id)
            self.mocker.replay()
            yield client.unsubscribe(share_id)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_create_share(self):
        """Share a subtree to the user identified by username."""
        path = 'path'
        username = 'username'
        name = 'name'
        access_level = 'level'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.create_share(path, username, name, access_level)
            self.mocker.replay()
            yield client.create_share(path, username, name, access_level)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_create_shares(self):
        """Share a subtree with several users at once."""
        path = 'path'
        usernames = ['username','other']
        name = 'name'
        access_level = 'level'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            for user in usernames:
                self.root.create_share(path, user, name, access_level)
            self.mocker.replay()
            yield client.create_shares(path, usernames, name, access_level)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_refresh_shares(self):
        """Refresh the share list, requesting it to the server. """

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.refresh_shares()
            self.mocker.replay()
            yield client.refresh_shares()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_shared(self):
        """Test get shared."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_shared()
            self.mocker.replay()
            yield client.get_shared()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_on_share_deleted(self):
        """Emit ShareDeleted."""
        share_dict = dict(name='name',user='user')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.get_share_dict(ANY)
            self.mocker.result(share_dict)
            client.on_share_deleted_cb(share_dict)
            self.mocker.replay()
            self.shares.emit_share_changed('deleted', share_dict)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_share_changed(self):
        """Emit ShareChanged."""
        share_dict = dict(name='name',user='user')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.get_share_dict(ANY)
            self.mocker.result(share_dict)
            client.on_share_changed_cb(share_dict)
            self.mocker.replay()
            self.shares.emit_share_changed('changed', share_dict)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_share_delete_error(self):
        """Emits ShareDeleteError signal."""
        share = 'share'
        error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.get_share_dict(share)
            self.mocker.result(share)
            client.on_share_delete_error_cb(share, error)
            self.mocker.replay()
            self.shares.emit_share_delete_error(share, error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d


    def test_on_share_created(self):
        """Emits ShareCreated signal """
        share_dict = dict(name='name',user='user')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_share_created_cb(share_dict)
            self.mocker.replay()
            self.shares.emit_share_created(share_dict)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_share_create_error(self):
        """Emit ShareCreateError signal."""
        share_dict = dict(name='name',user='user')
        error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.root.get_create_error_share_info(share_dict)
            self.mocker.result(share_dict)
            client.on_share_create_error_cb(share_dict, error)
            self.mocker.replay()
            self.shares.emit_share_create_error(share_dict, error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_share_answer_response(self):
        """Emits ShareAnswerResponse signal."""
        share_id = 'share_id'
        answer = 'yes'
        answer_info = dict(volume_id=share_id, answer=answer)

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_share_answer_response_cb(answer_info)
            self.mocker.replay()
            self.shares.emit_share_answer_response(share_id, answer)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_new_share(self):
        """Emits NewShare signal."""
        share_dict = dict(share='share')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.get_share_dict(share_dict)
            self.mocker.result(share_dict)
            self.root.get_volume(share_dict)
            self.mocker.result(share_dict)
            client.on_new_share_cb(share_dict)
            self.mocker.replay()
            self.shares.emit_new_share(share_dict)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_share_subscribed(self):
        """Emit the ShareSubscribed signal"""
        share = 'share'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.get_share_dict(share)
            self.mocker.result(share)
            client.on_share_subscribed_cb(share)
            self.mocker.replay()
            self.shares.emit_share_subscribed(share)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_share_subscribe_error(self):
        """Emit the ShareSubscribeError signal"""
        share_id = 'share_id'
        error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_share_subscribe_error_cb(dict(id=share_id), error)
            self.mocker.replay()
            self.shares.emit_share_subscribe_error(share_id, error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_share_unsubscribed(self):
        """Emit the ShareUnSubscribed signal"""
        share = 'share'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.get_share_dict(share)
            self.mocker.result(share)
            client.on_share_unsubscribed_cb(share)
            self.mocker.replay()
            self.shares.emit_share_unsubscribed(share)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_share_unsubscribe_error(self):
        """Emit the ShareUnSubscribeError signal""" 
        share_id = 'share_id'
        error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_share_subscribe_error_cb(dict(id=share_id), error)
            self.mocker.replay()
            self.shares.emit_share_subscribe_error(share_id, error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d


class TestConfigClient(BaseIPCTestCase):
    """Test the status client class."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        super(TestConfigClient, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.config = Config(None, None)
        self.config.syncdaemon_config = self.root
        # start pb
        self.syncdaemon_root = FakeRoot(None, None, None, None,
                                        None, self.config, None, None)
        self.server_factory = SaveProtocolServerFactory(self.syncdaemon_root)
        self.listener = ipc_server_listen(self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = yield ipc_client_connect(self.client_factory)

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_config')
        client = ConfigClient(remote)
        defer.returnValue(client)

    def test_get_throttling_limits(self):
        """Get the read/write limit from AQ and return a dict.
        Returns a dict(download=int, upload=int), if int is -1 the value isn't
        configured.
        The values are bytes/second
        """
        reply_handler = lambda: None
        error_handler = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_throttling_limits(ANY, ANY)
            self.mocker.replay()
            yield client.get_throttling_limits(reply_handler, error_handler)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_set_throttling_limits(self):
        """Set the read and write limits. The expected values are bytes/sec."""
        download = 'download'
        upload = 'upload'
        reply_handler = lambda: None
        error_handler = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.set_throttling_limits(download, upload, ANY, ANY)
            self.mocker.replay()
            yield client.set_throttling_limits(download, upload, reply_handler,
                                               error_handler)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_bandwidth_throttling(self):
        """Enable bandwidth throttling."""
        reply_handler = lambda: None
        error_handler = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.enable_bandwidth_throttling(ANY, ANY)
            self.mocker.replay()
            yield client.enable_bandwidth_throttling(reply_handler,
                                                     error_handler)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_disable_bandwidth_throttling(self):
        """Disable bandwidth throttling."""
        reply_handler = lambda: None
        error_handler = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.disable_bandwidth_throttling(ANY, ANY)
            self.mocker.replay()
            yield client.disable_bandwidth_throttling(reply_handler,
                                                      error_handler )
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_bandwidth_throttling_enabled(self):
        """Returns True (actually 1) if bandwidth throttling is enabled and
        False (0) otherwise.
        """
        reply_handler = lambda: None
        error_handler = lambda: None

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.bandwidth_throttling_enabled(ANY, ANY)
            self.mocker.replay()
            yield client.bandwidth_throttling_enabled(reply_handler,
                                                      error_handler)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_udf_autosubscribe_enabled(self):
        """Return the udf_autosubscribe config value."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.udf_autosubscribe_enabled()
            self.mocker.replay()
            yield client.udf_autosubscribe_enabled()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_udf_autosubscribe(self):
        """Enable UDF autosubscribe."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.enable_udf_autosubscribe()
            self.mocker.replay()
            yield client.enable_udf_autosubscribe()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_disable_udf_autosubscribe(self):
        """Enable UDF autosubscribe."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.disable_udf_autosubscribe()
            self.mocker.replay()
            yield client.disable_udf_autosubscribe()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_share_autosubscribe_enabled(self):
        """Return the share_autosubscribe config value."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.share_autosubscribe_enabled()
            self.mocker.replay()
            yield client.share_autosubscribe_enabled()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_share_autosubscribe(self):
        """Enable UDF autosubscribe."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.enable_share_autosubscribe()
            self.mocker.replay()
            yield client.enable_share_autosubscribe()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_disable_share_autosubscribe(self):
        """Enable UDF autosubscribe."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.disable_share_autosubscribe()
            self.mocker.replay()
            yield client.disable_share_autosubscribe()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_set_files_sync_enabled(self):
        """Enable/disable file sync service."""
        enabled = True

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.set_files_sync_enabled(enabled)
            self.mocker.replay()
            yield client.set_files_sync_enabled(enabled)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_files_sync_enabled(self):
        """Return the files_sync_enabled config value."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.files_sync_enabled()
            self.mocker.replay()
            yield client.files_sync_enabled()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_autoconnect_enabled(self):
        """Return the autoconnect config value."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.autoconnect_enabled()
            self.mocker.replay()
            yield client.autoconnect_enabled()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_set_autoconnect_enabled(self):
        """Enable syncdaemon autoconnect."""
        enabled = False

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.set_autoconnect_enabled(enabled)
            self.mocker.replay()
            yield client.set_autoconnect_enabled(enabled)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_show_all_notifications_enabled(self):
        """Return the show_all_notifications config value."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.show_all_notifications_enabled()
            self.mocker.replay()
            yield client.show_all_notifications_enabled()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_show_all_notifications(self):
        """Enable showing all notifications."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.enable_show_all_notifications()
            self.mocker.replay()
            yield client.enable_show_all_notifications()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_disable_show_all_notifications(self):
        """Disable showing all notifications."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.disable_show_all_notifications()
            self.mocker.replay()
            yield client.disable_show_all_notifications()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

class TestFoldersClient(BaseIPCTestCase):
    """Test the status client class."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        super(TestFoldersClient, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.folders = Folders(None, None)
        self.folders.syncdaemon_folders = self.root
        # start pb
        self.syncdaemon_root = FakeRoot(None, None, None, None,
                                        None, None, self.folders, None)
        self.server_factory = SaveProtocolServerFactory(self.syncdaemon_root)
        self.listener = ipc_server_listen(self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = yield ipc_client_connect(self.client_factory)

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_folders')
        client = FoldersClient(remote)
        yield client.register_to_signals()
        # set the cb
        for signal in ['on_folder_created_cb',
                       'on_folder_create_error_cb',
                       'on_folder_deleted_cb',
                       'on_folder_delete_error_cb',
                       'on_folder_subscribed_cb',
                       'on_folder_subscribe_error_cb',
                       'on_folder_unsubscribed_cb',
                       'on_folder_unsubscribe_error_cb']:
            setattr(client, signal, self.mocker.mock())
        defer.returnValue(client)

    def test_create(self):
        """Create a user defined folder in the specified path."""
        path = 'path'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.create(path)
            self.mocker.replay()
            yield client.create(path)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_delete(self):
        """Delete the folder specified by folder_id"""
        folder_id = 'folder_id'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.delete(folder_id)
            self.mocker.replay()
            yield client.delete(folder_id)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_folders(self):
        """Return the list of folders (a list of dicts)"""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_folders()
            self.mocker.replay()
            yield client.get_folders()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_subscribe(self):
        """Subscribe to the specified folder"""
        folder_id = 'folder_id' 

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.subscribe(folder_id )
            self.mocker.replay()
            yield client.subscribe(folder_id )
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_unsubscribe(self):
        """Unsubscribe from the specified folder"""
        folder_id = 'folder_id'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.unsubscribe(folder_id)
            self.mocker.replay()
            yield client.unsubscribe(folder_id)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d


    def test_get_info(self):
        """Returns a dict containing the folder information."""
        path = 'path'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_info(path)
            self.mocker.replay()
            yield client.get_info(path)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_refresh_volumes(self):
        """Refresh the volumes list, requesting it to the server."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.refresh_volumes()
            self.mocker.replay()
            yield client.refresh_volumes()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_on_folder_created(self):
        """Emit the FolderCreated signal"""
        folder = self.mocker.mock()
        folder_dict = dict(folder='folder', id='id')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            folder.__dict__
            self.mocker.result(folder_dict)
            client.on_folder_created_cb(folder_dict)
            self.mocker.replay()
            self.folders.emit_folder_created(folder)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_folder_create_error(self):
        """Emit the FolderCreateError signal"""
        path = 'path'
        error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_folder_create_error_cb(dict(path=path), error)
            self.mocker.replay()
            self.folders.emit_folder_create_error(path, error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_folder_deleted(self):
        """Emit the FolderCreated signal"""
        folder = self.mocker.mock()
        folder_dict = dict(folder='folder', id='id')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            folder.__dict__
            self.mocker.result(folder_dict)
            client.on_folder_deleted_cb(folder_dict)
            self.mocker.replay()
            self.folders.emit_folder_deleted(folder)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_folder_delete_error(self):
        """Emit the FolderCreateError signal"""
        folder = self.mocker.mock()
        folder_dict = dict(folder='folder', id='id')
        error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            folder.__dict__
            self.mocker.result(folder_dict)
            client.on_folder_delete_error_cb(folder_dict, error)
            self.mocker.replay()
            self.folders.emit_folder_delete_error(folder, error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_folder_subscribed(self):
        """Emit the FolderSubscribed signal"""
        folder = self.mocker.mock()
        folder_dict = dict(folder='folder', id='id')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            folder.__dict__
            self.mocker.result(folder_dict)
            client.on_folder_subscribed_cb(folder_dict)
            self.mocker.replay()
            self.folders.emit_folder_subscribed(folder)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_folder_subscribe_error(self):
        """Emit the FolderSubscribeError signal"""
        folder_id = 'id'
        error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_folder_subscribe_error_cb(dict(id=folder_id), error)
            self.mocker.replay()
            self.folders.emit_folder_subscribe_error(folder_id, error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_folder_unsubscribed(self):
        """Emit the FolderUnSubscribed signal"""
        folder = self.mocker.mock()
        folder_dict = dict(folder='folder', id='id')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            folder.__dict__
            self.mocker.result(folder_dict)
            client.on_folder_unsubscribed_cb(folder_dict)
            self.mocker.replay()
            self.folders.emit_folder_unsubscribed(folder)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_folder_unsubscribe_error(self):
        """Emit the FolderUnSubscribeError signal"""
        folder_id = 'id'
        error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_folder_unsubscribe_error_cb(dict(id=folder_id), error)
            self.mocker.replay()
            self.folders.emit_folder_unsubscribe_error(folder_id, error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d


class TestPublicFilesClient(BaseIPCTestCase):
    """Test the status client class."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        super(TestPublicFilesClient, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.files = PublicFiles(None, None)
        self.files.syncdaemon_public_files = self.root
        # start pb
        self.syncdaemon_root = FakeRoot(None, None, None, None,
                                        None, None, None, self.files)
        self.server_factory = SaveProtocolServerFactory(self.syncdaemon_root)
        self.listener = ipc_server_listen(self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = yield ipc_client_connect(self.client_factory)

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        remote = yield root.callRemote('get_public_files')
        client = PublicFilesClient(remote)
        yield client.register_to_signals()
        # set the cb
        for signal in ['on_public_access_changed_cb',
                       'on_public_access_change_error_cb',
                       'on_public_files_list_cb',
                       'on_public_files_list_error_cb']:
            setattr(client, signal, self.mocker.mock())
        defer.returnValue(client)

    def test_change_public_access(self):
        """Change the public access of a file."""
        share_id = 'share_id'
        node_id = 'node_id'
        is_public = False

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.change_public_access(share_id, node_id, is_public)
            self.mocker.replay()
            yield client.change_public_access(share_id, node_id, is_public)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_public_files(self):
        """Request the list of public files to the server."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_public_files()
            self.mocker.replay()
            yield client.get_public_files()
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_on_public_access_changed(self):
        """Emit the PublicAccessChanged signal."""
        share_id = 'share_id'
        node_id = 'node_id'
        is_public = False
        public_url = 'public_url'
        path = 'path'
        info_dict = dict(path=path, public_url=public_url, share_id=share_id,
                         node_id=node_id, is_public='')

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.root.get_path(share_id, node_id)
            self.mocker.result(path)
            client.on_public_access_changed_cb(info_dict)
            self.mocker.replay()
            self.files.emit_public_access_changed(share_id, node_id, is_public,
                                                  public_url)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_public_access_change_error(self):
        """Emit the PublicAccessChangeError signal."""
        share_id = 'share_id'
        node_id = 'node_id'
        error = 'error'
        path = 'path'
        info = dict(share_id=share_id, node_id=node_id, path=path)

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            self.root.get_path(share_id, node_id)
            self.mocker.result(path)
            client.on_public_access_change_error_cb(info, error)
            self.mocker.replay()
            self.files.emit_public_access_change_error(share_id, node_id, error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_public_files_list(self):
        """Emit the PublicFilesList signal."""
        public_files = []

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_public_files_list_cb(public_files)
            self.mocker.replay()
            self.files.emit_public_files_list(public_files)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d

    def test_on_public_files_list_error(self):
        """Emit the PublicFilesListError signal."""
        error = 'error'

        @defer.inlineCallbacks
        def test_emit(client):
            """Actual test."""
            client.on_public_files_list_error_cb(error)
            self.mocker.replay()
            self.files.emit_public_files_list_error(error)
            yield client.unregister_to_signals()
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_emit)
        # pylint: enable=E1101
        return d


class TestAllEventsSender(MockerTestCase):
    """Test the status client class."""

    def setUp(self):
        """Setup tests."""
        super(TestAllEventsSender, self).setUp()
        self.events = self.mocker.mock()
        self.sender = AllEventsSender(self.events)

    def test_handle_default(self):
        """Test that the events are indeed sent."""
        event_name = 'event_name'
        kwargs = dict(foo='foo', bar='bar')
        expected = dict(event_name=event_name, foo='foo', bar='bar')
        self.events.emit_event(expected)
        self.mocker.replay()
        self.sender.handle_default(event_name, **kwargs)


class RemoteMetaTestCase(TestCase):
    """Tests for the RemoteMeta metaclass."""

    def test_remote_calls_renamed(self):
        """The remote_calls are renamed."""
        test_token = object()

        class TestClass(object):
            """A class for testing."""

            __metaclass__ = RemoteMeta

            remote_calls = ['test_method']

            def test_method(self):
                """Fake call."""
                return test_token

        tc = TestClass()
        self.assertEquals(tc.test_method(), test_token)
        self.assertEquals(tc.remote_test_method(), test_token)

    def test_signal_handlers_renamed(self):
        """The signal_handlers are renamed."""
        test_token = object()

        class TestClass(object):
            """A class for testing."""

            __metaclass__ = RemoteMeta

            signal_handlers = ['test_signal_handler']

            def test_signal_handler(self):
                """Fake call."""
                return test_token

        tc = TestClass()
        self.assertEquals(tc.test_signal_handler(), test_token)
        self.assertEquals(tc.remote_test_signal_handler(), test_token)


class TestIPCRoot(BaseIPCTestCase):
    """Ensure that the IPCRoot works as expected."""

    @defer.inlineCallbacks
    def setUp(self):
        super(TestIPCRoot, self).setUp()
        self.mocker = Mocker()
        self.main = self.mocker.mock()
        # diff objects that will be exposed by root
        self.event_queue = self.mocker.mock()
        self.action_queue = self.mocker.mock()
        self.fs_manager = self.mocker.mock()
        self.volume_manager = self.mocker.mock()
        self.ipc_interface = self.mocker.mock()
        self.ipc_root = IPCRoot(self.ipc_interface, self.main)
        self.server_factory = SaveProtocolServerFactory(self.ipc_root)
        self.listener = ipc_server_listen(self.server_factory)
        self.client_factory = PBClientFactory()
        self.connector = yield ipc_client_connect(self.client_factory)
        self.mocker.reset()

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.connector.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _get_client(self):
        """Get the client."""
        # request the remote object and create a client
        root = yield self.client_factory.getRootObject()
        defer.returnValue(root)

    def test_get_status(self):
        """Ensure that a reference object is returned."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.mocker.replay()
            status = yield client.callRemote('get_status')
            self.assertNot(status is None,
                           'Status remote object should not be None')
            self.assertIsInstance(status, RemoteReference)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_events(self):
        """Ensure that a reference object is returned."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.mocker.replay()
            status = yield client.callRemote('get_events')
            self.assertNot(status is None,
                           'Status remote object should not be None')
            self.assertIsInstance(status, RemoteReference)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_sync_daemon(self):
        """Ensure that a reference object is returned."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.mocker.replay()
            status = yield client.callRemote('get_sync_daemon')
            self.assertNot(status is None,
                           'Status remote object should not be None')
            self.assertIsInstance(status, RemoteReference)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_file_system(self):
        """Ensure that a reference object is returned."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.mocker.replay()
            status = yield client.callRemote('get_file_system')
            self.assertNot(status is None,
                           'Status remote object should not be None')
            self.assertIsInstance(status, RemoteReference)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_shares(self):
        """Ensure that a reference object is returned."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.mocker.replay()
            status = yield client.callRemote('get_shares')
            self.assertNot(status is None,
                           'Status remote object should not be None')
            self.assertIsInstance(status, RemoteReference)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_config(self):
        """Ensure that a reference object is returned."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.mocker.replay()
            status = yield client.callRemote('get_config')
            self.assertNot(status is None,
                           'Status remote object should not be None')
            self.assertIsInstance(status, RemoteReference)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_folders(self):
        """Ensure that a reference object is returned."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.mocker.replay()
            status = yield client.callRemote('get_folders')
            self.assertNot(status is None,
                           'Status remote object should not be None')
            self.assertIsInstance(status, RemoteReference)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_public_files(self):
        """Ensure that a reference object is returned."""

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.mocker.replay()
            status = yield client.callRemote('get_public_files')
            self.assertNot(status is None,
                           'Status remote object should not be None')
            self.assertIsInstance(status, RemoteReference)
            self.mocker.verify()

        d = self._get_client()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d


class FakeIPCRoot(object, ipc.Root):
    """A Fake IPC Root."""

    def __init__(self, ipc_interface, *args, **kwargs):
        """Initialize this fake instance."""
        # just save the ipc_interface
        self.ipc_interface = ipc_interface


class FakeCredentialsManagementTool(object):
    """A Fake CredentialsManagementTool."""

    fake_credentials = {
        "consumer_key": "sample consumer key",
        "consumer_secret": "sample consumer secret",
        "token": "sample token",
        "token_secret": "sample token secret",
    }
    should_fail = False

    def find_credentials(self):
        """Find the credentials."""
        if self.should_fail:
            return defer.fail(RandomException())
        return defer.succeed(self.fake_credentials)

    def register(self, window_id=0):
        """Register a new user."""
        assert isinstance(window_id, (str, int))
        if self.should_fail:
            return defer.fail(RandomException())
        return defer.succeed(self.fake_credentials)


class EmptyCredentialsManagementTool(FakeCredentialsManagementTool):
    """A Fake CredentialsManagementTool that returns empty credentials."""

    fake_credentials = {}


class FakeCredentialsManagementToolFails(FakeCredentialsManagementTool):
    """A Fake CredentialsManagementTool that consistently fails."""

    should_fail = True


class FakeEventQueue(object):
    """A fake event queue."""

    def __init__(self):
        """Initialize this fake instance."""
        self.pushed_events = []

    def push(self, event_name, **kwargs):
        """Store the pushed event."""
        event = (event_name, kwargs)
        self.pushed_events.append(event)


class FakeMain(object):
    """A fake main."""

    def __init__(self):
        """Initialize this fake instance."""
        self.event_q = FakeEventQueue()


class IPCInterfaceTestCaseBase(TestCase):
    """Base tests for IPCInterface."""

    cmt_class = None

    def setUp(self):
        """Initialize this test instance."""
        if self.cmt_class is None: return
        self.patch(ipc, "CredentialsManagementTool", self.cmt_class)
        self.patch(ipc, "IPCRoot", FakeIPCRoot)
        no_op = lambda *args: None
        self.patch(ipc, "ipc_server_listen", no_op)
        self.patch(IPCInterface, "network_connected", no_op)
        self.ipc = IPCInterface(FakeMain())


class IPCInterfaceTestCase(IPCInterfaceTestCaseBase):
    """Tests for the IPCInterface when everything goes dandy."""

    cmt_class = FakeCredentialsManagementTool

    def test_ipc_root_created_ok(self):
        """The IPCRoot is passed the IPCInterface."""
        self.assertEqual(self.ipc.root.ipc_interface, self.ipc)

    @defer.inlineCallbacks
    def test_connect(self):
        """Test the connect method."""
        expected_token = FakeCredentialsManagementTool.fake_credentials
        yield self.ipc.connect()
        token = self.ipc.main.event_q.pushed_events[0][1]["access_token"]
        self.assertEqual(token, expected_token)

    @defer.inlineCallbacks
    def test_connect_autoconnecting(self):
        """Test the connect method when autoconnecting."""
        expected_token = FakeCredentialsManagementTool.fake_credentials
        yield self.ipc.connect(autoconnecting=True)
        token = self.ipc.main.event_q.pushed_events[0][1]["access_token"]
        self.assertEqual(token, expected_token)

    def test_daemonized(self):
        """Check that the NM thread won't block the process."""
        self.assertTrue(self.ipc.network_manager_thread.daemon)


class IPCInterfaceEmptyCredentialsTestCase(IPCInterfaceTestCaseBase):
    """Tests for the IPCInterface with empty credentials."""

    cmt_class = EmptyCredentialsManagementTool

    @defer.inlineCallbacks
    def test_connect(self):
        """Test the connect method fails."""
        yield self.assertFailure(self.ipc.connect(), ipc.NoAccessToken)

    @defer.inlineCallbacks
    def test_connect_autoconnecting(self):
        """Test the connect method when autoconnecting fails."""
        d = self.ipc.connect(autoconnecting=True)
        yield self.assertFailure(d, ipc.NoAccessToken)


class IPCInterfaceFailingTestCase(IPCInterfaceTestCaseBase):
    """Tests for the IPCInterface on a rainy day."""

    cmt_class = FakeCredentialsManagementToolFails

    @defer.inlineCallbacks
    def test_connect(self):
        """Test the connect method fails."""
        yield self.assertFailure(self.ipc.connect(), ipc.NoAccessToken)

    @defer.inlineCallbacks
    def test_connect_autoconnecting(self):
        """Test the connect method when autoconnecting fails."""
        d = self.ipc.connect(autoconnecting=True)
        yield self.assertFailure(d, ipc.NoAccessToken)


class IPCPortTestCase(TestCase):
    """Tests for the ipc port setup."""

    def test_get_sd_pb_port(self):
        """A test for the get_sd_pb_port function."""
        sso_port = 50001
        self.patch(ipc, "get_sso_pb_port", lambda: sso_port)
        result = ipc.get_sd_pb_port()
        expected = sso_port + ipc.SD_SSO_PORT_OFFSET
        self.assertEqual(result, expected)

    @defer.inlineCallbacks
    def test_is_already_running_no(self):
        """Test the is_already_running function."""
        self.patch(ipc.ActivationInstance, "get_port",
                   lambda _: defer.succeed(TEST_PORT))
        is_running = yield ipc.is_already_running()
        self.assertFalse(is_running, "Should not be running.")

    @defer.inlineCallbacks
    def test_is_already_running_yes(self):
        """Test the is_already_running function."""
        self.patch(ipc.ActivationInstance, "get_port",
               lambda _: defer.fail(ipc.AlreadyStartedError()))
        is_running = yield ipc.is_already_running()
        self.assertTrue(is_running, "Should be running by now.")
