/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "config.h"
#ifdef HAVE_GDBUS
#else
#include <dbus/dbus-glib.h>
#endif
#include "syncdaemon-events-interface.h"

G_DEFINE_TYPE(SyncdaemonEventsInterface, syncdaemon_events_interface, SYNCDAEMON_TYPE_INTERFACE)

struct _SyncdaemonEventsInterfacePrivate {
	GObject *proxy;
};

static void
syncdaemon_events_interface_finalize (GObject *object)
{
	SyncdaemonEventsInterface *interface = SYNCDAEMON_EVENTS_INTERFACE (object);

	if (interface->priv != NULL) {
		g_free (interface->priv);
	}

	G_OBJECT_CLASS (syncdaemon_events_interface_parent_class)->finalize (object);
}

static void
syncdaemon_events_interface_class_init (SyncdaemonEventsInterfaceClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_events_interface_finalize;
}

static void
event_cb (DBusGProxy *proxy, GHashTable *event_dict, gpointer user_data)
{
	SyncdaemonDaemon *daemon = NULL;
	SyncdaemonEventsInterface *interface = SYNCDAEMON_EVENTS_INTERFACE (user_data);

	g_object_get (G_OBJECT (interface), "daemon", &daemon, NULL);
	if (daemon != NULL)
		g_signal_emit_by_name (daemon, "event", event_dict);
}

static void
syncdaemon_events_interface_init (SyncdaemonEventsInterface *interface)
{
	interface->priv = g_new0 (SyncdaemonEventsInterfacePrivate, 1);

	/* Setup DBus proxy */
	interface->priv->proxy = syncdaemon_interface_setup_proxy (SYNCDAEMON_INTERFACE (interface),
								   "com.ubuntuone.SyncDaemon",
								   "/events", "com.ubuntuone.SyncDaemon.Events");
	if (interface->priv->proxy != NULL) {
		dbus_g_proxy_add_signal (DBUS_G_PROXY (interface->priv->proxy), "Event",
					 dbus_g_type_get_map ("GHashTable",
							      G_TYPE_STRING,
							      G_TYPE_STRING),
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (DBUS_G_PROXY (interface->priv->proxy), "Event",
					     G_CALLBACK (event_cb), interface, NULL);
	}
}

/**
 * syncdaemon_events_interface_new:
 */
SyncdaemonEventsInterface *
syncdaemon_events_interface_new (SyncdaemonDaemon *daemon)
{
	g_return_val_if_fail (SYNCDAEMON_IS_DAEMON (daemon), NULL);

	return g_object_new (SYNCDAEMON_TYPE_EVENTS_INTERFACE, "daemon", daemon, NULL);
}

/**
 * syncdaemon_events_interface_push_event:
 */
void
syncdaemon_events_interface_push_event (SyncdaemonEventsInterface *interface,
					const gchar *event_name,
					const gchar **args)
{
	GError *error = NULL;

	g_return_if_fail (SYNCDAEMON_IS_EVENTS_INTERFACE (interface));

	if (!dbus_g_proxy_call (DBUS_G_PROXY (interface->priv->proxy), "push_event", &error,
				G_TYPE_STRING, event_name,
				G_TYPE_STRV, args,
				G_TYPE_INVALID,
				G_TYPE_INVALID)) {
		g_warning ("Failed calling push_event: %s", error->message);
		g_error_free (error);
	}
}
