# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the DBUS interface."""

import os
import uuid
import logging

import dbus
from dbus import exceptions
from mocker import MockerTestCase
from twisted.internet import defer, reactor
from twisted.python import failure

from contrib.testing.testcase import (
    BaseTwistedTestCase, FakeMain, FAKED_CREDENTIALS)

from ubuntuone.storageprotocol.sharersp import NotifyShareHolder
from ubuntuone.syncdaemon import states, config
from ubuntuone.syncdaemon.interaction_interfaces import (
    get_share_dict,
    get_udf_dict,
    SyncdaemonEventListener
)
from ubuntuone.platform.linux.dbus_interface import (
    AllEventsSender,
    bool_str,
    DBusExposedObject,
    DBUS_IFACE_STATUS_NAME,
    DBUS_IFACE_EVENTS_NAME,
    DBUS_IFACE_FS_NAME,
    DBUS_IFACE_SYNC_NAME,
    DBUS_IFACE_SHARES_NAME,
    DBUS_IFACE_CONFIG_NAME,
    DBUS_IFACE_FOLDERS_NAME,
    DBUS_IFACE_PUBLIC_FILES_NAME,
    DBUS_IFACE_LAUNCHER_NAME,
    DBUS_CREDENTIALS_IFACE,
    DBusInterface,
    logger,
    NM_STATE_CONNECTED_GLOBAL,
    NM_STATE_DISCONNECTED,
    NoAccessToken,
)
from ubuntuone.platform.linux import ExternalInterface
from ubuntuone.syncdaemon import action_queue

from ubuntuone.syncdaemon.volume_manager import (
    get_udf_path,
    Share,
    Shared,
    UDF,
    VolumeDoesNotExist,
)
from ubuntuone.platform.linux.tools import DBusClient
from ubuntuone.syncdaemon.marker import MDMarker

from ubuntuone.devtools.handlers import MementoHandler
from ubuntuone.devtools.testcase import DBusTestCase


class FakeDBusInterface(object):
    """A fake DBusInterface..."""

    def shutdown(self, with_restart=False):
        """...that only knows how to go away"""

    def _request_token(self, *args, **kwargs):
        """Return a token which is a fixed set of credentials."""
        return FAKED_CREDENTIALS


class DBusTwistedTestCase(BaseTwistedTestCase, DBusTestCase):
    """ Test the DBus event handling """

    @defer.inlineCallbacks
    def setUp(self):
        """ Setup the infrastructure fo the test (dbus service). """
        yield super(DBusTwistedTestCase, self).setUp()
        self.log = logging.getLogger("ubuntuone.SyncDaemon.TEST")
        self.log.info("starting test %s.%s", self.__class__.__name__,
                      self._testMethodName)
        self.timeout = 2
        self.data_dir = self.mktemp('data_dir')
        self.partials_dir = self.mktemp('partials')
        self.root_dir = self.mktemp('root_dir')
        self.shares_dir = self.mktemp('shares_dir')
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.fs_manager = self.main.fs
        self.event_q = self.main.event_q
        self.action_q = self.main.action_q
        self.nm = FakeNetworkManager(self.bus)
        self.dbus_iface = DBusInterface(self.bus, self.main,
                                        system_bus=self.bus)
        # expose the inner object so that other tests can access them
        self.status = self.dbus_iface.status
        self.events = self.dbus_iface.events
        self.sync = self.dbus_iface.sync
        self.fs = self.dbus_iface.fs
        self.shares = self.dbus_iface.shares
        self.config = self.dbus_iface.config
        self.folders = self.dbus_iface.folders
        self.public_files = self.dbus_iface.public_files
        # token is a fixed set of credentials
        self.dbus_iface._request_token = lambda *a, **kw: FAKED_CREDENTIALS

        self.main.external = ExternalInterface(self.main, False, False,
                                               dbus_iface=self.dbus_iface)
        self.busName = self.dbus_iface.busName
        self.dbus_iface.connect()
        self.event_q.push('SYS_INIT_DONE')

        self.handler = MementoHandler()
        self._logger = logging.getLogger("ubuntuone.SyncDaemon")
        self._logger.addHandler(self.handler)

    @defer.inlineCallbacks
    def tearDown(self):
        """ Cleanup the test. """
        self._logger.removeHandler(self.handler)
        self._tearDown()
        yield super(DBusTwistedTestCase, self).tearDown()

    def _tearDown(self, *args):
        """ shutdown """
        self.main.shutdown()
        self.nm.shutdown()
        self.dbus_iface.bus.flush()
        self.log.info("finished test %s.%s", self.__class__.__name__,
                      self._testMethodName)

    def error_handler(self, error):
        """ default error handler for DBus calls. """
        if isinstance(error, failure.Failure):
            self.fail(error.getErrorMessage())
        else:
            self.fail(str(error))


class FakeNetworkManager(DBusExposedObject):
    """ A fake NetworkManager that only emits StatusChanged signal. """

    State = 3

    def __init__(self, bus):
        """ Creates the instance. """
        self.path = '/org/freedesktop/NetworkManager'
        self.bus = bus
        self.bus.request_name('org.freedesktop.NetworkManager',
                              flags=dbus.bus.NAME_FLAG_REPLACE_EXISTING | \
                              dbus.bus.NAME_FLAG_DO_NOT_QUEUE | \
                              dbus.bus.NAME_FLAG_ALLOW_REPLACEMENT)
        self.busName = dbus.service.BusName('org.freedesktop.NetworkManager',
                                        bus=self.bus)
        DBusExposedObject.__init__(self, bus_name=self.busName,
                                   path=self.path)

    def shutdown(self):
        """ Shutdown the fake NetworkManager """
        self.busName.get_bus().release_name(self.busName.get_name())
        self.remove_from_connection()

    @dbus.service.signal('org.freedesktop.NetworkManager', signature='i')
    # pylint: disable-msg=C0103
    def StateChanged(self, state):
        """ Fire DBus signal StatusChanged. """
        pass

    def emit_connected(self):
        """ Emits the signal StateCganged(3). """
        self.StateChanged(NM_STATE_CONNECTED_GLOBAL)

    def emit_disconnected(self):
        """ Emits the signal StateCganged(4). """
        self.StateChanged(NM_STATE_DISCONNECTED)

    @dbus.service.method(dbus.PROPERTIES_IFACE,
                         in_signature='ss', out_signature='v',
                         async_callbacks=('reply_handler', 'error_handler'))
    # pylint: disable-msg=C0103
    def Get(self, interface, propname, reply_handler=None, error_handler=None):
        """
        Fake dbus's Get method to get at the State property
        """
        try:
            reply_handler(getattr(self, propname, None))
        except Exception, e: # pylint: disable-msg=W0703
            error_handler(e)


class FakeCommand(object):
    """A fake command."""

    uniqueness = None
    running = True

    def __init__(self, share_id, node_id, other='', path=None):
        self.share_id = share_id
        self.node_id = node_id
        self.other = other
        self.path = path

    def is_runnable(self):
        """Always runnable."""
        return True

    def run(self):
        """Just succeed."""
        return defer.succeed(None)

    def to_dict(self):
        """Just send both values."""
        d = dict(share_id=self.share_id, node_id=self.node_id,
                 other=self.other, running=self.running)
        # some commands have path, others don't
        if self.path is not None:
            d['path'] = self.path
        return d


class FakeUpload(FakeCommand, action_queue.Upload):
    """Fake command that inherits from Upload."""
    def __init__(self, *args):
        FakeCommand.__init__(self, *args)
        self.path = 'upload_path'


class FakeDownload(FakeCommand, action_queue.Download):
    """Fake command that inherits from Download."""
    def __init__(self, *args):
        FakeCommand.__init__(self, *args)
        self.path = 'download_path'


class EmptyCommand(FakeCommand):
    """A command without any attributes."""

    def __init__(self):
        """__init__ that doesn't set any attributes."""

    def to_dict(innerself):
        """We have no attributes, return an empty dict."""
        return {}


class FakedSSOBackend(object):
    """Fake a SSO Backend (acts as a dbus.Interface as well)."""

    def __init__(self, *args, **kwargs):
        self._args = args
        self._kwargs = kwargs
        self._called = {}

    def register(self, *args, **kwargs):
        """Fake register."""
        self._called['register'] = (args, kwargs)

    def find_credentials(self, *args, **kwargs):
        """Fake find_credentials."""
        self._called['find_credentials'] = (args, kwargs)


class FakedBusName(object):
    """Fake a dbus name."""

    def __init__(self, name, bus):
        """Init."""
        self._name = name
        self.get_bus = lambda: bus


class FakedDbusObject(object):
    """Fake a dbus."""

    def __init__(self, *args, **kwargs):
        """Init."""
        self._args = args
        self._kwargs = kwargs


class FakedSessionBus(object):
    """Fake the session bus."""

    def __init__(self):
        """Init."""
        self.obj = None
        self.callbacks = {}

    def add_signal_receiver(self, method, signal_name=None, path=None,
                            dbus_interface=None, bus_name=None, **keywords):
        """Add a signal receiver."""
        self.callbacks[(dbus_interface, signal_name)] = method

    def remove_signal_receiver(self, match, signal_name=None, path=None,
                               dbus_interface=None, bus_name=None, **keywords):
        """Remove the signal receiver."""
        assert (dbus_interface, signal_name) in self.callbacks
        del self.callbacks[(dbus_interface, signal_name)]

    def get_object(self, bus_name, object_path, introspect=True,
                   follow_name_owner_changes=False, **kwargs):
        """Return a faked proxy for the given remote object."""
        assert self.obj is None
        kwargs = dict(object_path=object_path,
                      bus_name=bus_name, follow_name_owner_changes=True)
        self.obj = FakedDbusObject(**kwargs)
        return self.obj

    def _register_object_path(self, *args, **kwargs):
        """Faked _register_object_path."""


class DBusInterfaceTests(DBusTwistedTestCase):
    """Basic tests to the objects exposed with D-Bus."""

    def test_status(self):
        """Test the Status Object.

        Registering it to the session bus, and calling current_uploads method.
        """
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def waiting_handler(result):
            """Waiting handler."""
            self.assertEquals(result, [])
            d.callback(True)

        def status_handler(result):
            """Reply handler."""
            state = states.StateManager.QUEUE_MANAGER
            self.assertEquals(state.name, result['name'])
            self.assertEquals(state.description, result['description'])
            self.assertEquals(state.is_error, bool(result['is_error']))
            self.assertEquals(state.is_connected, bool(result['is_connected']))
            self.assertEquals(state.is_online, bool(result['is_online']))
            client.call_method('waiting_content',
                               reply_handler=waiting_handler,
                               error_handler=self.error_handler)

        def downloads_handler(result):
            """Reply handler."""
            self.assertEquals(0, len(result))
            client.call_method('current_status',
                               reply_handler=status_handler,
                               error_handler=self.error_handler)

        def uploads_handler(result):
            """Reply handler."""
            self.assertEquals(0, len(result))
            client.call_method('current_downloads',
                               reply_handler=downloads_handler,
                               error_handler=self.error_handler)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_syncDaemon(self):
        """Test the SyncDaemon object.

        Registering it to the session bus, and calling connect and disconnect.
        """
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        def disconnect_handler(result):
            """Reply handler."""
            self.assertEquals(None, result)
            d.callback(True)

        def connect_handler(result):
            """Reply handler."""
            self.assertEquals(None, result)
            client.call_method("disconnect", reply_handler=disconnect_handler,
                               error_handler=self.error_handler)

        client.call_method("connect", reply_handler=connect_handler,
                           error_handler=self.error_handler)

        return d

    @defer.inlineCallbacks
    def test_get_metadata_path(self):
        """Test the FileSystem Object, getting MD by path."""
        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")
        d = defer.Deferred()
        def handler(metadata):
            """Reply handler."""
            d.callback(metadata)

        def callback(result):
            """Callback to check the result."""
            self.assertEquals(path, str(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata', path, reply_handler=handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_get_metadata_path_symlink(self):
        """Test the FileSystem Object, getting MD by path in a symlink."""
        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, "foo")
        symlink_path = os.path.join(self.shares_dir, "share_symlink")
        share_context = self.main.fs._enable_share_write('share',
                                                        share_path)
        with share_context:
            os.makedirs(share_path)
            os.symlink(share_path, symlink_path)
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")
        d = defer.Deferred()
        def handler(metadata):
            """Reply handler."""
            d.callback(metadata)

        def callback(result):
            """Callback to check the result."""
            self.assertEquals(os.path.join(symlink_path, 'foo'),
                              str(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata', os.path.join(symlink_path, 'foo'),
                           reply_handler=handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_get_metadata_share_node(self):
        """Test the FileSystem Object, getting MD by share and node."""
        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")
        d = defer.Deferred()
        def handler(metadata):
            """Reply handler."""
            d.callback(metadata)

        def callback(result):
            """Callback to check the result."""
            self.assertEquals(path, str(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata_by_node', 'share', 'node_id',
                           reply_handler=handler,
                           error_handler=self.error_handler)
        yield d

    def test_push_event(self):
        """Test the exposed method: push_event."""
        push_deferred = defer.Deferred()
        queue_deferred = defer.Deferred()
        push_deferred.addCallback(lambda _: queue_deferred)
        class Listener(object):
            """A basic listener to handle the pushed event."""

            def handle_FS_FILE_CREATE(self, path):
                """FS_FILE_CREATE handling."""
                self.assertEquals('bar', path)

        listener = Listener()
        self.event_q.subscribe(listener)
        self.event_q.unsubscribe(self.dbus_iface.event_listener)

        def empty_queue_cb():
            """Called when the event queue empties."""
            queue_deferred.callback(True)

        def default(path):
            """Default callback to assert the pushed event."""
            self.event_q.unsubscribe(listener)
            self.event_q.subscribe(self.dbus_iface.event_listener)
            self.event_q.remove_empty_event_queue_callback(empty_queue_cb)

        queue_deferred.addCallback(default)

        client = DBusClient(self.bus, '/events', DBUS_IFACE_EVENTS_NAME)
        client.call_method(
            'push_event', 'FS_FILE_CREATE', {'path':'bar'},
            reply_handler=lambda *_: push_deferred.callback(True),
            error_handler=self.error_handler)
        self.event_q.add_empty_event_queue_callback(empty_queue_cb)
        return push_deferred

    @defer.inlineCallbacks
    def test_free_space(self):
        """Test Status.waiting, for all commands."""
        # fake share
        share_path = os.path.join(self.main.shares_dir, 'share')
        share = Share(path=share_path, volume_id='vol_id')
        yield self.main.vm.add_share(share)

        self.main.vm.update_free_space('vol_id', 12345)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def waiting_handler(result):
            """Waiting reply handler."""
            self.assertEqual(result, 12345)
            d.callback(True)

        client.call_method('free_space', 'vol_id',
                           reply_handler=waiting_handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_waiting_simple(self):
        """Test Status.waiting, for all commands."""
        # inject the fake data
        c1 = FakeCommand("share_id", "node_id_a", other=123)
        c2 = FakeCommand("share_id", "node_id_b", other=None)
        c2.running = False
        c3 = FakeCommand("share_id", "node_id_c", other=MDMarker('bar'))
        self.action_q.queue.waiting.extend([c1, c2, c3])

        # testing time
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def waiting_handler(result):
            """Waiting reply handler."""
            node_a, node_b, node_c = result

            should = dict(share_id='share_id', node_id='node_id_a',
                          running='True', other='123')
            self.assertEqual(node_a, ('FakeCommand', str(id(c1)), should))

            should = dict(share_id='share_id', node_id='node_id_b',
                          running='', other='None')
            self.assertEqual(node_b, ('FakeCommand', str(id(c2)), should))

            should = dict(share_id='share_id', node_id='node_id_c',
                          running='True', other='marker:bar')
            self.assertEqual(node_c, ('FakeCommand', str(id(c3)), should))

            # OK, we're done
            d.callback(True)

        client.call_method('waiting', reply_handler=waiting_handler,
                                      error_handler=self.error_handler)
        yield d


    @defer.inlineCallbacks
    def test_waiting_content(self):
        """Test Status.waiting_content."""
        # inject the fake data
        self.action_q.queue.waiting.extend([
            FakeUpload("share_id", "node_id_b"),
            FakeDownload("share_id", "node_id_c"),
        ])

        # testing time
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def waiting_handler(result):
            """Waiting_content reply handler."""
            self.assertEqual(2, len(result))
            node_b, node_c = result

            self.assertEqual('upload_path', str(node_b['path']))
            self.assertEqual('share_id', str(node_b['share']))
            self.assertEqual('node_id_b', str(node_b['node']))

            self.assertEqual('download_path', str(node_c['path']))
            self.assertEqual('share_id', str(node_c['share']))
            self.assertEqual('node_id_c', str(node_c['node']))

            # OK, we're done
            d.callback(True)

        client.call_method('waiting_content',
                           reply_handler=waiting_handler,
                           error_handler=self.error_handler)
        yield d
        self.assertTrue(self.handler.check_warning('deprecated'))

    @defer.inlineCallbacks
    def test_waiting_metadata(self):
        """Test Status.waiting_metadata with fake data in the AQ."""
        # inject the fake data
        self.action_q.queue.waiting.extend([
            FakeCommand("share_id", "node_id_b", u"moño"),
            FakeCommand("share_id", "node_id_c", path='/some/path'),
            FakeCommand("share_id", "node_id_d"),
        ])

        # testing time
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def waiting_handler(result):
            """Reply handler."""
            self.assertEqual(len(result), 3)

            pl = dict(share_id='share_id', node_id='node_id_b',
                      other=u'moño', running='True')
            self.assertEqual(result[0], ('FakeCommand', pl))

            pl = dict(share_id='share_id', node_id='node_id_c',
                      other='', path='/some/path', running='True')
            self.assertEqual(result[1], ('FakeCommand', pl))

            pl = dict(share_id='share_id', node_id='node_id_d',
                      other='', running='True')
            self.assertEqual(result[2], ('FakeCommand', pl))
            d.callback(True)
        client.call_method('waiting_metadata',
                           reply_handler=waiting_handler,
                           error_handler=self.error_handler)
        yield d
        self.assertTrue(self.handler.check_warning('deprecated'))

    @defer.inlineCallbacks
    def test_get_metadata_and_quick_tree_not_synced(self):
        """Test Status.get_metadata_and_quick_tree_status.

        Fake data in the AQ is used.
        """
        share_path = os.path.join(self.shares_dir, 'share')
        share = Share(path=share_path, volume_id='share')
        yield self.main.vm.add_share(share)
        path = os.path.join(share_path, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")

        # inject fake data
        self.action_q.queue.waiting.append(
            FakeCommand("share", u"node_id"))

        d = defer.Deferred()
        def handler(metadata):
            """Reply handler."""
            d.callback(metadata)

        def callback(result):
            """Callback to check the result."""
            self.assertEquals(path.decode('utf-8'), result['path'])
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])
            self.assertEquals('', result['quick_tree_synced'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata_and_quick_tree_synced',
                           path, reply_handler=handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_get_metadata_and_quick_tree_no_blow_up_kthxbye(self):
        """Test Status.get_metadata_and_quick_tree_status.

        Fake data in the AQ is used.
        """
        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")

        # inject fake data
        self.action_q.queue.waiting.append(
            FakeCommand("this share id no longer exists",
                        "neither does this path id"))

        d = defer.Deferred()
        def handler(metadata):
            """Reply handler."""
            d.callback(metadata)

        def callback(result):
            """Callback to check the result."""
            self.assertEquals(path.decode('utf-8'), unicode(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])
            self.assertEquals('synced', result['quick_tree_synced'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata_and_quick_tree_synced',
                           path, reply_handler=handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_get_metadata_and_quick_tree_not_synced_2(self):
        """Test Status.get_metadata_and_quick_tree_status.

        Fake data in the AQ is used.
        """
        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")

        # inject fake data
        self.action_q.queue.waiting.append(
            FakeCommand("share", u"node_id"))

        d = defer.Deferred()
        def handler(metadata):
            """Reply handler."""
            d.callback(metadata)

        def callback(result):
            """Callback to check the result."""
            self.assertEquals(path.decode('utf-8'), unicode(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])
            self.assertEquals('', result['quick_tree_synced'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata_and_quick_tree_synced',
                           path, reply_handler=handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_get_metadata_and_quick_tree_synced(self):
        """Test Status.get_metadata_and_quick_tree_status
        without fake data in the AQ."""
        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")

        d = defer.Deferred()
        def handler(metadata):
            """Reply handler."""
            d.callback(metadata)

        def callback(result):
            """Callback to check the result."""
            self.assertEquals(path.decode('utf-8'), result['path'])
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])
            self.assertEquals('synced', result['quick_tree_synced'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata_and_quick_tree_synced',
                           path, reply_handler=handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_get_metadata_and_quick_tree_synced_symlink(self):
        """Test Status.get_metadata_and_quick_tree_status
        without fake data in the AQ."""
        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")
        symlink_path = os.path.join(self.shares_dir, "share_symlink")
        share_context = self.main.fs._enable_share_write('share',
                                                        share_path)
        with share_context:
            os.makedirs(share_path)
            os.symlink(share_path, symlink_path)
        expected_path = os.path.join(symlink_path, os.path.basename(path))

        d = defer.Deferred()
        def handler(metadata):
            """Reply handler."""
            d.callback(metadata)

        def callback(result):
            """Callback to check the result."""
            self.assertEquals(expected_path.decode('utf-8'), unicode(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])
            self.assertEquals('synced', result['quick_tree_synced'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata_and_quick_tree_synced',
                           expected_path, reply_handler=handler,
                           error_handler=self.error_handler)
        yield d

    def test_content_queue_added(self):
        """Test the signal because a command was added to the queue."""
        class FakeCommand(action_queue.Upload):
            """Fake to don't need to instantiate everything for a test."""
            def __init__(self):
                """Nothing needed :)"""

        d = defer.Deferred()
        receiver = self.bus.add_signal_receiver(lambda: d.callback(True),
                                            signal_name='ContentQueueChanged')
        self.signal_receivers.add(receiver)
        self.main.event_q.push('SYS_QUEUE_ADDED', command=FakeCommand())
        return d

    def test_content_queue_removed(self):
        """Test the signal because a command was removed from the queue."""
        class FakeCommand(action_queue.Upload):
            """Fake to don't need to instantiate everything for a test."""
            def __init__(self):
                """Nothing needed :)"""

        d = defer.Deferred()
        receiver = self.bus.add_signal_receiver(lambda: d.callback(True),
                                            signal_name='ContentQueueChanged')
        self.signal_receivers.add(receiver)
        self.main.event_q.push('SYS_QUEUE_REMOVED', command=FakeCommand())
        return d

    @defer.inlineCallbacks
    def test_current_downloads(self):
        """Test Status.current_downloads with fake data in the AQ."""
        fake_download = FakeDownload('share_id', 'down_node_id')
        fake_download.deflated_size = 10
        fake_download.n_bytes_read = 1
        fake_download.path = "down_path"
        self.action_q.queue.waiting.append(fake_download)

        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def downloads_handler(result):
            """Reply handler."""
            self.assertEqual(1, len(result))
            self.assertEqual("down_path", str(result[0]['path']))
            self.assertEqual('10', str(result[0]['deflated_size']))
            self.assertEqual('1', str(result[0]['n_bytes_read']))
            d.callback(True)

        client.call_method('current_downloads',
                           reply_handler=downloads_handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_current_uploads(self):
        """Test Status.current_uploads with fake data in the AQ."""
        fake_upload = FakeUpload('share_id', 'node_id')
        fake_upload.deflated_size = 100
        fake_upload.n_bytes_written = 10
        fake_upload.path = "up_path"
        self.action_q.queue.waiting.append(fake_upload)

        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """Reply handler."""
            self.assertEqual(1, len(result))
            self.assertEqual("up_path", str(result[0]['path']))
            self.assertEqual('100', str(result[0]['deflated_size']))
            self.assertEqual('10', str(result[0]['n_bytes_written']))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_two_current_downloads(self):
        """Test Status.current_downloads with fake data in the AQ."""
        fake_download = FakeDownload('share_id', 'node_id')
        fake_download.deflated_size = 10
        fake_download.n_bytes_read = 8
        fake_download.path = "down_path"
        self.action_q.queue.waiting.append(fake_download)

        fake_download = FakeDownload('share_id_1', 'node_id_1')
        fake_download.deflated_size = 10
        fake_download.n_bytes_read = 5
        fake_download.path = "down_path_1"
        self.action_q.queue.waiting.append(fake_download)

        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def downloads_handler(result):
            """Reply handler."""
            self.assertEqual(2, len(result))
            self.assertEqual('down_path', str(result[0]['path']))
            self.assertEqual('10', str(result[0]['deflated_size']))
            self.assertEqual('8', str(result[0]['n_bytes_read']))
            self.assertEqual('down_path_1', str(result[1]['path']))
            self.assertEqual('10', str(result[1]['deflated_size']))
            self.assertEqual('5', str(result[1]['n_bytes_read']))
            d.callback(True)

        client.call_method('current_downloads',
                           reply_handler=downloads_handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_two_current_uploads(self):
        """Test Status.current_uploads with fake data in the AQ."""
        fake_upload = FakeUpload('share_id', 'node_id')
        fake_upload.deflated_size = 100
        fake_upload.n_bytes_written = 10
        fake_upload.path = "up_path"
        self.action_q.queue.waiting.append(fake_upload)

        fake_upload = FakeUpload('share_id_1', 'node_id_1')
        fake_upload.deflated_size = 80
        fake_upload.n_bytes_written = 20
        fake_upload.path = "up_path_1"
        self.action_q.queue.waiting.append(fake_upload)

        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """Reply handler."""
            self.assertEqual(2, len(result))
            self.assertEqual('up_path', str(result[0]['path']))
            self.assertEqual('100', str(result[0]['deflated_size']))
            self.assertEqual('10', str(result[0]['n_bytes_written']))
            self.assertEqual('up_path_1', str(result[1]['path']))
            self.assertEqual('80', str(result[1]['deflated_size']))
            self.assertEqual('20', str(result[1]['n_bytes_written']))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_current_downloads_deflated_size_NA(self):
        """Test Status.current_downloads with fake data in the AQ."""
        fake_download = FakeDownload('share_id', 'down_node_id')
        fake_download.deflated_size = None
        fake_download.n_bytes_read = 0
        fake_download.path = "down_path"
        self.action_q.queue.waiting.append(fake_download)

        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def downloads_handler(result):
            """Reply handler."""
            self.assertEqual(1, len(result))
            self.assertEqual('down_path', str(result[0]['path']))
            self.assertNotIn('deflated_size', result[0])
            self.assertEqual('0', str(result[0]['n_bytes_read']))
            d.callback(True)

        client.call_method('current_downloads',
                           reply_handler=downloads_handler,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_current_uploads_deflated_size_NA(self):
        """Test Status.current_uploads with fake data in the AQ."""
        fake_upload = FakeUpload('share_id', 'node_id')
        fake_upload.deflated_size = None
        fake_upload.n_bytes_written = 0
        fake_upload.path = "up_path"
        self.action_q.queue.waiting.append(fake_upload)

        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """Reply handler."""
            self.assertEquals(1, len(result))
            self.assertEquals("up_path", str(result[0]['path']))
            self.assertNotIn('deflated_size', result[0])
            self.assertEquals('0', str(result[0]['n_bytes_written']))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        yield d

    def test_nm_signals(self):
        """Test that NM signals are received and handled properly."""
        result = None
        d = defer.Deferred()
        # helper class, pylint: disable-msg=C0111
        # class-closure, cannot use self, pylint: disable-msg=E0213
        class Listener(object):
            def handle_SYS_NET_CONNECTED(innerself):
                self.nm.emit_disconnected()
            def handle_SYS_NET_DISCONNECTED(innerself):
                self.assertTrue(result)

        listener = Listener()
        self.event_q.subscribe(listener)

        def empty_queue_cb():
            d.callback(True)

        def callback(result):
            self.event_q.unsubscribe(listener)
            self.event_q.remove_empty_event_queue_callback(empty_queue_cb)
        d.addCallback(callback)

        self.nm.emit_connected()
        self.event_q.add_empty_event_queue_callback(empty_queue_cb)
        return d

    @defer.inlineCallbacks
    def test_get_shares(self):
        """Test Shares.get_shares method."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        share = Share(path=share_path, volume_id='share_id',
                      access_level='Read', accepted=False)
        yield self.main.vm.add_share(share)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        def check(shares):
            """Handle get_shares reply."""
            self.assertEquals(1, len(shares))
            for share in shares:
                if share['volume_id'] == '':
                    self.assertEquals('', str(share['volume_id']))
                    self.assertEquals(self.root_dir, str(share['path']))
                    self.assertEquals('Modify', str(share['access_level']))
                    self.assertEquals('False', str(share['accepted']))
                else:
                    self.assertEquals('share_id', str(share['volume_id']))
                    self.assertEquals(share_path, str(share['path']))
                    self.assertEquals('Read', str(share['access_level']))
                    self.assertEquals('', str(share['accepted']))

        def shares_handler(shares):
            d.callback(shares)

        client.call_method('get_shares', reply_handler=shares_handler,
                           error_handler=self.error_handler)
        d.addCallback(check)
        yield d

    @defer.inlineCallbacks
    def test_accept_share(self):
        """Test the accept_share method in dbus_interface.Share."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share_id',
                                     access_level='Read', accepted=False,
                                     node_id="node_id"))
        self.assertEquals(False, self.main.vm.shares['share_id'].accepted)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)

        d = defer.Deferred()
        match = self.bus.add_signal_receiver(d.callback,
                                             signal_name='ShareAnswerResponse')
        self.signal_receivers.add(match)
        client.call_method('accept_share', 'share_id',
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)
        def check(result):
            """The async checker."""
            self.assertEquals('Yes', result['answer'])
            self.assertEquals('share_id', result['volume_id'])
            self.assertEquals(True, self.main.vm.shares['share_id'].accepted)

        d.addCallback(check)
        yield d

    @defer.inlineCallbacks
    def test_reject_share(self):
        """Test the reject_share method in dbus_interface.Share."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share_id',
                                     access_level='Read', accepted=False))
        self.assertEquals(False, self.main.vm.shares['share_id'].accepted)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)

        d = defer.Deferred()
        match = self.bus.add_signal_receiver(d.callback,
                                             signal_name='ShareAnswerResponse')
        self.signal_receivers.add(match)
        client.call_method('reject_share', 'share_id',
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)
        def check(result):
            """The async checker."""
            self.assertEquals('No', result['answer'])
            self.assertEquals('share_id', result['volume_id'])
            self.assertEquals(False, self.main.vm.shares['share_id'].accepted)

        d.addCallback(check)
        yield d

    def test_get_rootdir(self):
        """Check SycnDaemon.get_rootdir exposed method."""
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        def reply_handler(result):
            """Handle get_rootdir reply."""
            current_root = self.main.get_rootdir()
            self.assertEquals(current_root, str(result))
            d.callback(True)
        client.call_method('get_rootdir',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_get_sharesdir(self):
        """Check SycnDaemon.get_sharesdir exposed method."""
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()

        def reply_handler(result):
            """Handle get_sharesdir reply."""
            result = self.main.get_sharesdir()
            self.assertEquals(result, str(result))
            d.callback(True)

        client.call_method('get_sharesdir',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_get_sharesdir_link(self):
        """Check SycnDaemon.get_sharesdir_link exposed method."""
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()

        def reply_handler(result):
            """Handle get_sharesdir_link reply."""
            result = self.main.get_sharesdir_link()
            self.assertEquals(result, str(result))
            d.callback(True)

        client.call_method('get_sharesdir_link',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_create_share(self):
        """Test share offering."""
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def fake_create_share(*result):
            # pylint: disable-msg=W0612
            node_id, username, name, access_level, marker, path = result
            self.assertEquals('node_id', node_id)
            mdobj = self.fs_manager.get_by_path(a_dir)
            self.assertEquals(self.fs_manager.get_abspath("", mdobj.path),
                              os.path.join(self.main.root_dir, a_dir))
            self.assertEquals(u'test_user', username)
            self.assertEquals(u'share_a_dir', name)
            self.assertEquals('View', access_level)

        self.action_q.create_share = fake_create_share

        def reply_handler(result):
            d.callback(result)

        client.call_method('create_share',
                           a_dir, 'test_user',
                           'share_a_dir', 'View',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_create_shares(self):
        """Test share offering to multiple users at once."""
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def fake_create_share(*result):
            # pylint: disable-msg=W0612
            node_id, username, name, access_level, marker, path = result
            self.assertEquals('node_id', node_id)
            mdobj = self.fs_manager.get_by_path(a_dir)
            self.assertEquals(self.fs_manager.get_abspath("", mdobj.path),
                              os.path.join(self.main.root_dir, a_dir))
            self.assertEquals(u'test_user', username[:-1])
            self.assertEquals(u'share_a_dir', name)
            self.assertEquals('View', access_level)

        self.action_q.create_share = fake_create_share

        def reply_handler(result):
            d.callback(result)

        client.call_method('create_shares',
                           a_dir, ['test_user1', 'test_user2', 'test_user3'],
                           'share_a_dir', 'View',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_get_shared(self):
        """Test Shares.get_shared API."""
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        # helper functions, pylint: disable-msg=C0111
        def aq_create_share(*args):
            self.main.event_q.push('AQ_CREATE_SHARE_OK',
                                   share_id='share_id', marker=args[-2])
        self.main.action_q.create_share = aq_create_share
        self.main.vm.create_share(a_dir, 0, 'share_a_dir', 'View')
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        def reply_handler(results):
            """Handle get_shared reply."""
            self.assertEqual(1, len(results))
            shared = results[0]
            self.assertEqual(a_dir, str(shared['path']))
            self.assertEqual('node_id', str(shared['node_id']))
            self.assertEqual('share_id', str(shared['volume_id']))
            self.assertEqual('View', str(shared['access_level']))
            d.callback(True)
        client.call_method('get_shared',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_get_shared_missing_path(self):
        """Test Shares.get_shared, but without having the path."""
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        # helper functions, pylint: disable-msg=C0111
        def aq_create_share(*args):
            self.main.event_q.push('AQ_CREATE_SHARE_OK',
                                   share_id='share_id', marker=args[-2])
        self.main.action_q.create_share = aq_create_share
        self.main.vm.create_share(a_dir, 0, 'share_a_dir', 'View')
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        # remove the md of the subtree from fsm
        self.fs_manager.delete_file(a_dir)
        # set the path to None
        share = self.main.vm.shared['share_id']
        share.path = None
        self.main.vm.shared['share_id'] = share
        d = defer.Deferred()
        def reply_handler(results):
            """Handle get_shared reply."""
            self.assertEqual(1, len(results))
            shared = results[0]
            self.assertEqual('', str(shared['path']))
            self.assertEqual('node_id', str(shared['node_id']))
            self.assertEqual('share_id', str(shared['volume_id']))
            self.assertEqual('View', str(shared['access_level']))
            d.callback(True)
        client.call_method('get_shared',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_refresh_shares(self):
        """Just check that refresh_shares method API works."""
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        def reply_handler(result):
            """Handle refresh_shares reply."""
            d.callback(True)
        client.call_method('refresh_shares',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_quit(self):
        """Test the quit exposed method."""
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        # helper functions, we need to call quit but don't quit
        # pylint: disable-msg=C0111,W0601,W0602
        def fake_quit():
            pass
        self.main.quit = fake_quit
        def handler(result):
            d.callback(True)
        d.addCallback(self.assertTrue)
        client.call_method('quit',
                           reply_handler=handler,
                           error_handler=self.error_handler)
        return d

    def test_change_public_access(self):
        """Test the change_public_access exposed method."""
        client = DBusClient(
            self.bus, '/publicfiles', DBUS_IFACE_PUBLIC_FILES_NAME)
        d = defer.Deferred()
        def handler(result):
            d.callback(True)
        d.addCallback(self.assertTrue)
        share_id = str(uuid.uuid4())
        node_id = str(uuid.uuid4())
        client.call_method('change_public_access', share_id, node_id, True,
                           reply_handler=handler,
                           error_handler=self.error_handler)
        return d

    def test_get_public_files(self):
        """Test the get_public_files exposed method."""
        public_files = []
        expected = []
        udf_id = str(uuid.uuid4())
        suggested_path = u'~/foo/bar'
        udf_path = get_udf_path(suggested_path)
        udf = UDF(udf_id, 'udf_node_id', suggested_path, udf_path, True)
        self.main.vm.udfs[udf_id] = udf
        for i in xrange(5):
            if i % 2:
                volume_id = udf_id
                path = os.path.join(udf.path, "foo_%d" % i)
            else:
                volume_id = ''
                path = os.path.join(self.root_dir, "foo_%d" % i)
            node_id = str(uuid.uuid4())
            public_url = 'http://example.com/%d' % i
            self.fs_manager.create(path, volume_id)
            self.fs_manager.set_node_id(path, node_id)
            public_files.append(dict(volume_id=volume_id, node_id=node_id,
                                     public_url=public_url))
            expected.append(dict(volume_id=volume_id, node_id=node_id,
                                     public_url=public_url, path=path))
        def fake_get_public_files():
            self.main.event_q.push('AQ_PUBLIC_FILES_LIST_OK',
                                   public_files=public_files)
        self.main.action_q.get_public_files = fake_get_public_files

        d = defer.Deferred()
        def public_files_list_handler(files):
            """Handler for PublicFilesList signal."""
            d.callback(files)
        match = self.bus.add_signal_receiver(public_files_list_handler,
                                             signal_name='PublicFilesList')
        self.signal_receivers.add(match)
        def check(files):
            self.assertEquals(len(public_files), len(files))
            self.assertEquals(expected, files)
        d.addCallback(check)
        client = DBusClient(
            self.bus, '/publicfiles', DBUS_IFACE_PUBLIC_FILES_NAME)
        client.call_method('get_public_files',
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)

        return d

    @defer.inlineCallbacks
    def test_rescan_from_scratch(self):
        """Test for rescan_from_scratch."""
        # create a volume
        share_path = os.path.join(self.main.shares_dir, 'share')
        share = Share(path=share_path, volume_id=str(uuid.uuid4()),
                      access_level='Modify', accepted=True,
                      node_id=str(uuid.uuid4()))
        yield self.main.vm.add_share(share)
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)

        rescan_d = defer.Deferred()
        self.main.action_q.rescan_from_scratch = rescan_d.callback

        d = defer.Deferred()
        client.call_method('rescan_from_scratch', share.volume_id,
                           reply_handler=d.callback,
                           error_handler=d.errback)

        yield d

        vol_id = yield rescan_d
        self.assertEqual(vol_id, share.volume_id)

    @defer.inlineCallbacks
    def test_rescan_from_scratch_missing_volume(self):
        """Test for rescan_from_scratch with a non-existing volume."""
        volume_id = str(uuid.uuid4())
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        d.addCallback(self.assertTrue)
        client.call_method('rescan_from_scratch', volume_id,
                           reply_handler=d.callback,
                           error_handler=d.errback)
        try:
            yield d
        except exceptions.DBusException, e:
            self.assertEqual('org.freedesktop.DBus.Python.ubuntuone'
                             '.syncdaemon.volume_manager.VolumeDoesNotExist',
                             e.get_dbus_name())
        else:
            self.fail('Should fail with a non-existing volume.')

    @defer.inlineCallbacks
    def test_get_dirty_nodes(self):
        """Test the FileSystem.get_dirty_nodes exposed method."""
        # create some nodes
        path1 = os.path.join(self.root_dir, u'ñoño-1'.encode('utf-8'))
        mdid1 = self.main.fs.create(path1, "")
        path2 = os.path.join(self.root_dir, u'ñoño-2'.encode('utf-8'))
        mdid2 = self.main.fs.create(path2, "")
        path3 = os.path.join(self.root_dir, "path3")
        mdid3 = self.main.fs.create(path3, "")
        path4 = os.path.join(self.root_dir, "path4")
        mdid4 = self.main.fs.create(path4, "")

        # dirty some
        self.main.fs.set_by_mdid(mdid2, dirty=True)
        self.main.fs.set_by_mdid(mdid4, dirty=True)

        d = defer.Deferred()
        def handler(dirty):
            """Reply handler."""
            d.callback(dirty)

        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_dirty_nodes', reply_handler=handler,
                           error_handler=d.errback)
        all_dirty = yield d
        dirty_mdids = dict((n['mdid'], n) for n in all_dirty)
        self.assertEqual(len(all_dirty), 2)
        self.assertIn(mdid2, dirty_mdids)
        self.assertIn(mdid4, dirty_mdids)
        self.assertNotIn(mdid1, dirty_mdids)
        self.assertNotIn(mdid3, dirty_mdids)
        # check that path de/encoding is done correctly
        self.assertEquals(repr(self.main.fs.get_by_mdid(mdid2).path),
                          repr(dirty_mdids[mdid2]['path'].encode('utf-8')))


class DBusInterfaceUnicodeTests(DBusTwistedTestCase):
    """Unicode variant of basic tests to the objects exposed with D-Bus."""

    @defer.inlineCallbacks
    def test_filesystem_unicode(self):
        """Test the FileSystem Object, registering it to the session bus.

        Excercise the API.
        """
        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")
        d = defer.Deferred()
        def handler(metadata):
            """Reply handler."""
            d.callback(metadata)

        def callback(result):
            """Callback to check the result."""
            self.assertEquals(path.decode('utf-8'), unicode(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata', path, reply_handler=handler,
                           error_handler=self.error_handler)
        yield d

    def test_create_share_unicode(self):
        """Test share offering."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def fake_create_share(*result):
            # pylint: disable-msg=W0612
            node_id, username, name, access_level, marker, path = result
            self.assertEquals('node_id', node_id)
            mdobj = self.fs_manager.get_by_path(a_dir)
            self.assertEquals(self.fs_manager.get_abspath("", mdobj.path),
                              os.path.join(self.root_dir, a_dir))
            self.assertEquals(u'test_user', username)
            self.assertEquals(u'share_ñoño', name)
            self.assertEquals('View', access_level)

        self.action_q.create_share = fake_create_share

        def reply_handler(result):
            d.callback(result)

        client.call_method('create_share',
                           a_dir, u'test_user',
                           u'share_ñoño', 'View',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    @defer.inlineCallbacks
    def test_get_shared_unicode(self):
        """Test that list_shared method behaves with unicode data."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        share = Shared(path=a_dir, volume_id='shared_id', name=u'ñoño_shared',
                      access_level='View', other_username=u'test_username',
                      node_id='node_id')
        yield self.main.vm.add_shared(share)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()

        def check(results):
            """Check the returned values."""
            self.assertEquals(1, len(results))
            shared = results[0]
            self.assertEquals(a_dir, shared['path'].encode('utf-8'))
            self.assertEquals('node_id', str(shared['node_id']))
            self.assertEquals('shared_id', str(shared['volume_id']))
            self.assertEquals('View', str(shared['access_level']))

        d.addCallback(check)
        client.call_method('get_shared',
                           reply_handler=d.callback, error_handler=d.errback)
        yield d


class DBusSignalTest(DBusTwistedTestCase):
    """Test the DBus signals."""

    def test_event_signal(self):
        """Test the Event signal."""
        client = DBusClient(self.bus, '/events', DBUS_IFACE_EVENTS_NAME)
        d = defer.Deferred()
        def handler(event_dict):
            """Signal handler."""
            self.assertEquals({'path':'foo', 'name':'FS_FILE_CREATE'},
                              event_dict)
            d.callback(True)

        match = self.bus.add_signal_receiver(handler, signal_name='Event')
        self.signal_receivers.add(match)
        client.send_signal('Event', {'name':'FS_FILE_CREATE', 'path':'foo'})

        return d

    def test_download_started(self):
        """Test the DBus signals in Status."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path):
            """Handler for DownloadStarted signal."""
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadStarted')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_STARTED', share_id='',
                               node_id='node_id', server_hash='')
        return d

    def test_download_file_progress(self):
        """Test the DBus signals in Status."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path, info):
            """Handler for DownloadFileProgress signal."""
            self.assertEqual(a_dir, path.encode('utf-8'))
            self.assertEqual(info, {'n_bytes_read': '10',
                                    'deflated_size': '20'})
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadFileProgress')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_FILE_PROGRESS',
                               share_id='',
                               node_id='node_id',
                               n_bytes_read=10,
                               deflated_size=20)
        return d

    def test_download_finished(self):
        """Test the DBus signals in Status."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path, info):
            """Handler for DownloadFinished signal."""
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_FINISHED', share_id='',
                               node_id='node_id', server_hash='')
        return d

    def test_download_error(self):
        """Test the DBus signals in Status."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path, info):
            """Handler for DownloadFinished signal."""
            self.assertEquals(a_dir, path.encode('utf-8'))
            self.assertEquals('AN_ERROR', info['error'])
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_ERROR', share_id='',
                               node_id='node_id', server_hash='',
                               error='AN_ERROR')
        return d

    def test_upload_started(self):
        """Test the DBus signals in Status."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path):
            """Handler for UploadStarted signal."""
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadStarted')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_STARTED', share_id='',
                               node_id='node_id', hash='hash')
        return d

    def test_upload_file_progress(self):
        """Test the DBus signals in Status."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path, info):
            """Handler for UploadFileProgress signal."""
            self.assertEqual(a_dir, path.encode('utf-8'))
            self.assertEqual(info, {'n_bytes_written': '10',
                                     'deflated_size': '20'})
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadFileProgress')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_FILE_PROGRESS',
                               share_id='',
                               node_id='node_id',
                               n_bytes_written=10,
                               deflated_size=20)
        return d

    def test_upload_finished(self):
        """Test the DBus signals in Status."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path, info):
            """Handler for UploadFinished signal."""
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_FINISHED', share_id='',
                               node_id='node_id', hash='',
                               new_generation='new_generation')
        return d

    def test_upload_error(self):
        """Test the DBus signals in Status."""
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path, info):
            """Handler for UploadFinished signal."""
            self.assertEquals(a_dir, path.encode('utf-8'))
            self.assertEquals('AN_ERROR', info['error'])
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_ERROR', share_id='',
                               node_id='node_id',
                               error='AN_ERROR', hash='hash')
        return d

    def test_meta_queue_added(self):
        """Test the signal because a command was added to the queue."""
        d = defer.Deferred()
        receiver = self.bus.add_signal_receiver(lambda: d.callback(True),
                                                signal_name='MetaQueueChanged')
        self.signal_receivers.add(receiver)
        cmd = action_queue.ListShares   # any "meta" command
        self.main.event_q.push('SYS_QUEUE_ADDED', command=cmd)
        return d

    def test_meta_queue_changed(self):
        """Test the signal because a command was removed from the queue."""
        d = defer.Deferred()
        receiver = self.bus.add_signal_receiver(lambda: d.callback(True),
                                                signal_name='MetaQueueChanged')
        self.signal_receivers.add(receiver)
        cmd = action_queue.ListShares   # any "meta" command
        self.main.event_q.push('SYS_QUEUE_REMOVED', command=cmd)
        return d

    def test_sys_queue_added(self):
        """Test the signal because a command was removed from the queue."""
        d = defer.Deferred()
        def check(op_name, op_id, data):
            self.assertEqual(op_name, 'FakeCommand')
            self.assertEqual(op_id, str(id(cmd)))
            should = dict(share_id='share', node_id='node',
                          running='True', other='123')
            self.assertEqual(data, should)
            d.callback(True)
        rec = self.bus.add_signal_receiver(check,
                                           signal_name='RequestQueueAdded')
        self.signal_receivers.add(rec)
        cmd = FakeCommand('share', 'node', other=123)
        self.main.event_q.push('SYS_QUEUE_ADDED', command=cmd)
        return d

    def test_sys_queue_removed(self):
        """Test the signal because a command was removed from the queue."""
        d = defer.Deferred()
        def check(op_name, op_id, data):
            self.assertEqual(op_name, 'FakeCommand')
            self.assertEqual(op_id, str(id(cmd)))
            should = dict(share_id='share', node_id='node',
                          running='True', other='marker:foo')
            self.assertEqual(data, should)
            d.callback(True)
        rec = self.bus.add_signal_receiver(check,
                                           signal_name='RequestQueueRemoved')
        self.signal_receivers.add(rec)
        cmd = FakeCommand('share', 'node', other=MDMarker('foo'))
        self.main.event_q.push('SYS_QUEUE_REMOVED', command=cmd)
        return d

    def test_status_changed(self):
        """Test the DBus signals in Status."""
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        def start(result):
            """Start the test."""
            match = self.bus.add_signal_receiver(status_handler,
                                     signal_name='StatusChanged')
            self.signal_receivers.add(match)
            state = states.StateManager.READY
            state_dict = {'name':state.name,
                          'description':state.description,
                          'is_error':bool_str(state.is_error),
                          'is_connected':bool_str(state.is_connected),
                          'is_online':bool_str(state.is_online)}
            client.send_signal('StatusChanged', state_dict)
        ready = self.main.wait_for_nirvana()
        ready.addCallback(start)
        d = defer.Deferred()
        def status_handler(result):
            """Handler for StatusChanged signal."""
            self.log.debug('status_handler, received: %r' % result)
            state = states.StateManager.READY
            self.assertEquals(state.name, result['name'])
            self.assertEquals(state.description, result['description'])
            self.assertEquals(state.is_error, bool(result['is_error']))
            self.assertEquals(state.is_connected, bool(result['is_connected']))
            self.assertEquals(state.is_online, bool(result['is_online']))
            d.callback(True)

        return d

    def test_invalid_filename(self):
        """Test the FS_INVALID_NAME signal."""
        d = defer.Deferred()

        def handler(dirname, filename):
            """Handler for InvalidName signal."""
            self.assertTrue(isinstance(dirname, unicode))
            self.assertTrue(isinstance(filename, str))
            d.callback(True)

        match = self.bus.add_signal_receiver(handler,
                                             signal_name='InvalidName',
                                             byte_arrays=True)
        self.signal_receivers.add(match)
        self.main.event_q.push('FS_INVALID_NAME',
                               dirname=u'test/dir', filename='testpath')
        return d

    def test_broken_node_all_data(self):
        """Test the SYS_BROKEN_NODE signal with all data."""
        d = defer.Deferred()

        def handler(volume_id, node_id, mdid, path):
            """Handler for BrokenNode signal."""
            self.assertEqual(volume_id, 'volume')
            self.assertEqual(node_id, 'node')
            self.assertEqual(mdid, 'mdid')
            self.assertEqual(path, 'somepath')
            self.assertTrue(isinstance(path, unicode))
            d.callback(True)

        match = self.bus.add_signal_receiver(handler,
                                             signal_name='BrokenNode',
                                             byte_arrays=True)
        self.signal_receivers.add(match)
        self.main.event_q.push('SYS_BROKEN_NODE', volume_id='volume',
                               node_id='node', path='somepath', mdid='mdid')
        return d

    def test_broken_node_partial(self):
        """Test the SYS_BROKEN_NODE signal with some data."""
        d = defer.Deferred()

        def handler(volume_id, node_id, mdid, path):
            """Handler for BrokenNode signal."""
            self.assertEqual(volume_id, 'volume')
            self.assertEqual(node_id, 'node')
            self.assertEqual(mdid, '')
            self.assertEqual(path, u'')
            d.callback(True)

        match = self.bus.add_signal_receiver(handler,
                                             signal_name='BrokenNode',
                                             byte_arrays=True)
        self.signal_receivers.add(match)
        self.main.event_q.push('SYS_BROKEN_NODE', volume_id='volume',
                               node_id='node', path=None, mdid=None)
        return d

    @defer.inlineCallbacks
    def test_share_changed(self):
        """Test the ShareChanged signal."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        share_holder = NotifyShareHolder.from_params(uuid.uuid4(), uuid.uuid4(),
                                                     u'fake_share',
                                                     u'test_username',
                                                     u'visible_name', 'Write')

        yield self.main.vm.add_share(Share(path=share_path,
                                     volume_id=str(share_holder.share_id),
                                     node_id=str(share_holder.subtree),
                                     access_level='Read', accepted=False))
        d = defer.Deferred()
        def check(share):
            """Handler for ShareChanged signal."""
            self.assertEquals(str(share_holder.share_id), str(share['volume_id']))
            self.assertEquals(str(share_holder.subtree), str(share['node_id']))
            self.assertEquals(share_path, str(share['path']))
            self.assertEquals('Write', str(share['access_level']))
            self.assertEquals('', str(share['accepted']))
        d.addCallback(check)
        def share_handler(result):
            d.callback(result)
        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareChanged')
        self.signal_receivers.add(match)
        self.main.event_q.push('SV_SHARE_CHANGED', info=share_holder)
        yield d

    @defer.inlineCallbacks
    def test_share_deleted(self):
        """Test the ShareDeleted signal."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        share_holder = NotifyShareHolder.from_params(uuid.uuid4(), uuid.uuid4(),
                                                     u'fake_share',
                                                     u'test_username',
                                                     u'visible_name', 'Read')
        share = Share.from_notify_holder(share_holder, share_path)
        yield self.main.vm.add_share(share)
        d = defer.Deferred()
        match = self.bus.add_signal_receiver(d.callback,
                                             signal_name='ShareDeleted')
        self.signal_receivers.add(match)

        def check(share_dict):
            """Check the result."""
            expected_dict = dict(volume_id=str(share_holder.share_id),
                                 node_id=str(share_holder.subtree),
                                 name=u'fake_share',
                                 other_username=u'test_username',
                                 other_visible_name=u'visible_name',
                                 free_bytes='',
                                 path=share_path,
                                 accepted='',
                                 subscribed='',
                                 access_level='Read',
                                 generation='')
            expected_dict['type'] = 'Share'
            for k, v in share_dict.items():
                self.assertEquals(expected_dict[str(k)], str(v))
        d.addCallback(check)

        self.main.event_q.push('SV_SHARE_DELETED',
                               share_id=share_holder.share_id)
        yield d

    def test_share_created(self):
        """Test the ShareCreated signal."""
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        mdobj = self.fs_manager.get_by_node_id("", "node_id")
        mdid = mdobj.mdid
        marker = MDMarker(mdid)
        share_id = uuid.uuid4()
        d = defer.Deferred()
        def share_handler(result):
            """Handler for ShareCreated signal."""
            self.assertEquals(str(share_id), str(result['volume_id']))
            d.callback(True)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareCreated')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_CREATE_SHARE_OK',
                               share_id=share_id, marker=marker)
        return d

    def test_share_created_error(self):
        """Test the ShareCreated signal."""
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        mdobj = self.fs_manager.get_by_node_id("", "node_id")
        mdid = mdobj.mdid
        marker = MDMarker(mdid)
        error_msg = 'a error message'
        d = defer.Deferred()
        def share_handler(info, error):
            """Handler for ShareCreated signal."""
            self.assertEquals(str(marker), str(info['marker']))
            self.assertEquals(error, error_msg)
            d.callback(True)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareCreateError')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_CREATE_SHARE_ERROR',
                               marker=marker, error=error_msg)
        return d

    def test_signal_error(self):
        """Test sending SignalError."""
        d = defer.Deferred()
        def error_handler(type, args):
            """Error signal handler."""
            self.assertEquals('node_id', args['node_id'].decode('utf-8'))
            self.assertEquals('', args['share_id'].decode('utf-8'))
            d.callback(True)

        error_match = self.bus.add_signal_receiver(error_handler,
                                     signal_name='SignalError',
                                     dbus_interface=DBUS_IFACE_SYNC_NAME)
        self.signal_receivers.add(error_match)
        # force a invalid AQ_UPLOAD_FINISHED
        self.main.event_q.push('AQ_UPLOAD_FINISHED', share_id='',
                               node_id='node_id', hash='hash',
                               new_generation='new_generation')
        return d

    @defer.inlineCallbacks
    def test_new_share(self):
        """Test the NewShare signal."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        share_holder = NotifyShareHolder.from_params(uuid.uuid4(), uuid.uuid4(),
                                                     u'fake_share',
                                                     u'test_username',
                                                     u'visible_name', 'Read')
        share = Share.from_notify_holder(share_holder, share_path)
        # mark the share as accepted
        share.accepted = True
        d = defer.Deferred()
        def share_handler(share_dict):
            """Handler for NewShare signal."""
            d.callback(share_dict)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='NewShare')
        self.signal_receivers.add(match)

        def check(share_dict):
            """Check the result."""
            expected_dict = dict(volume_id=share.volume_id,
                                 node_id=share.node_id,
                                 name=u'fake_share',
                                 other_username=u'test_username',
                                 other_visible_name=u'visible_name',
                                 free_bytes='',
                                 path=share_path,
                                 accepted='True',
                                 subscribed='',
                                 access_level='Read',
                                 generation='')
            expected_dict['type'] = 'Share'
            for k, v in share_dict.items():
                self.assertEquals(expected_dict[str(k)], str(v))
        d.addCallback(check)
        yield self.main.vm.add_share(share)
        yield d

    @defer.inlineCallbacks
    def test_public_access_changed(self):
        """Test the PublicAccessChanged signal."""
        d = defer.Deferred()
        def access_changed_handler(file_info):
            """Handler for PublicAccessChanged signal."""
            d.callback(file_info)

        match = self.bus.add_signal_receiver(access_changed_handler,
                                             signal_name='PublicAccessChanged')
        self.signal_receivers.add(match)

        share_id = "share"
        node_id = "node_id"
        is_public = True
        public_url = 'http://example.com'

        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, str(share_id))
        self.fs_manager.set_node_id(path, str(node_id))

        def check(file_info):
            """Check the result."""
            expected_dict = dict(share_id=str(share_id),
                                 node_id=str(node_id),
                                 is_public=str(is_public),
                                 public_url=public_url,
                                 path=path)
            self.assertEquals(expected_dict, file_info)
        d.addCallback(check)
        self.event_q.push('AQ_CHANGE_PUBLIC_ACCESS_OK',
                          share_id=share_id, node_id=node_id,
                          is_public=is_public, public_url=public_url)
        yield d

    @defer.inlineCallbacks
    def test_public_access_change_error(self):
        """Test the PublicAccessChangeError signal."""
        d = defer.Deferred()
        def access_change_error_handler(file_info, error):
            """Handler for PublicAccessChangeError signal."""
            d.callback((file_info, error))

        match = self.bus.add_signal_receiver(
            access_change_error_handler, signal_name='PublicAccessChangeError')
        self.signal_receivers.add(match)

        share_id = "share"
        node_id = "node_id"
        expected_error = 'error message'

        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share'))
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, str(share_id))
        self.fs_manager.set_node_id(path, str(node_id))

        def check((file_info, error)):
            """Check the result."""
            expected_dict = dict(share_id=str(share_id),
                                 node_id=str(node_id),
                                 path=path)
            self.assertEquals(expected_dict, file_info)
            self.assertEquals(expected_error, error)
        d.addCallback(check)
        self.event_q.push('AQ_CHANGE_PUBLIC_ACCESS_ERROR',
                          share_id=share_id, node_id=node_id,
                          error=expected_error)
        yield d

    def test_root_mismatch(self):
        """Test RootMismatch signal."""
        d = defer.Deferred()
        def root_mismatch_handler(root_id, new_root_id):
            """Handler for RootMismatch signal."""
            d.callback((root_id, new_root_id))

        match = self.bus.add_signal_receiver(root_mismatch_handler,
                                             signal_name='RootMismatch')
        self.signal_receivers.add(match)

        def check((root_id, new_root_id)):
            """Check the result."""
            self.assertEquals('root_id', root_id)
            self.assertEquals('another_root_id', new_root_id)
        d.addCallback(check)
        self.main.vm._got_root('root_id')
        self.main.vm._got_root('another_root_id')
        return d

    @defer.inlineCallbacks
    def test_public_files_list(self):
        """Test the PublicAccessChanged signal."""
        d = defer.Deferred()
        def public_files_list_handler(files):
            """Handler for PublicFilesList signal."""
            d.callback(files)

        match = self.bus.add_signal_receiver(public_files_list_handler,
                                             signal_name='PublicFilesList')
        self.signal_receivers.add(match)

        volume_id = "share"
        node_id = "node_id"
        public_url = 'http://example.com'

        share_path = os.path.join(self.shares_dir, 'share')
        share = Share(path=share_path, volume_id=volume_id)
        yield self.main.vm.add_share(share)
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, str(volume_id))
        self.fs_manager.set_node_id(path, str(node_id))

        def check(files):
            """Check the result."""
            expected_list = [dict(volume_id=str(volume_id),
                                 node_id=str(node_id),
                                 public_url=public_url,
                                 path=path)]
            self.assertEquals(expected_list, files)
        d.addCallback(check)
        pf = dict(volume_id=volume_id, node_id=node_id, public_url=public_url)
        self.event_q.push('AQ_PUBLIC_FILES_LIST_OK', public_files=[pf])
        yield d

    def test_public_files_list_error(self):
        """Test the PublicFilesListError signal."""
        d = defer.Deferred()
        def access_change_error_handler(error):
            """Handler for PublicFilesListError signal."""
            d.callback(error)

        match = self.bus.add_signal_receiver(
            access_change_error_handler, signal_name='PublicFilesListError')
        self.signal_receivers.add(match)

        expected_error = 'error message'

        def check(error):
            """Check the result."""
            self.assertEquals(expected_error, error)
        d.addCallback(check)
        self.event_q.push('AQ_PUBLIC_FILES_LIST_ERROR',
                          error=expected_error)
        return d

    def assert_quota_exceeded(self, volume_id, expected_volume_dict):
        """Check correct signaling of QuotaExceeded."""
        d = defer.Deferred()

        def quota_exceeded_handler(volume_dict):
            """Handler for QuotaExceeded signal."""
            # as we fix the free bytes with signal info, fix the expected dict
            expected_volume_dict['free_bytes'] = '123'
            if volume_dict == expected_volume_dict:
                d.callback(volume_dict)
            else:
                d.errback('volume_dict must be %s (got %s instead)' %
                          (expected_volume_dict, volume_dict))

        match = self.bus.add_signal_receiver(quota_exceeded_handler,
                                             signal_name='QuotaExceeded')
        self.signal_receivers.add(match)
        self.event_q.push('SYS_QUOTA_EXCEEDED',
                          volume_id=volume_id, free_bytes=123)
        return d

    @defer.inlineCallbacks
    def test_quota_exceeded_for_share(self):
        """Test QuotaExceeded signal for share."""
        volume_id = 'test this please'
        path = 'test/path/bla'
        name = 'Beatiful Stranger'
        share = Share(volume_id=volume_id, node_id=None, path=path, name=name,
                      other_username=None, other_visible_name=None,
                      accepted=False, access_level='View', free_bytes=None)
        yield self.main.vm.add_share(share)
        yield self.assert_quota_exceeded(volume_id, get_share_dict(share))

    def test_quota_exceeded_for_udf(self):
        """Test QuotaExceeded signal for UDF."""
        volume_id = 'test this please'
        path = 'test/path/bla'
        a_udf = UDF(volume_id=volume_id, node_id='test', suggested_path=None,
                    path=path, subscribed=True)
        d = self.main.vm.add_udf(a_udf)
        test = self.assert_quota_exceeded(volume_id, get_udf_dict(a_udf))
        d.addCallback(lambda _: test)
        return d

    def test_quota_exceeded_for_root(self):
        """Test QuotaExceeded signal for root."""
        root = self.main.vm.root
        return self.assert_quota_exceeded(root.volume_id, get_share_dict(root))


class TestSendingAllEvents(BaseTwistedTestCase):
    """Test the machinery to send absolutely all the events."""

    timeout = 2

    @defer.inlineCallbacks
    def setUp(self):
        """Init."""
        yield super(TestSendingAllEvents, self).setUp()
        self.patch(DBusInterface, 'test', True)
        self.patch(dbus, 'Interface', FakedSSOBackend)
        self.patch(dbus.service, 'BusName', FakedBusName)
        self.bus = FakedSessionBus()

        self.data_dir = self.mktemp('data_dir')
        self.partials_dir = self.mktemp('partials')
        self.root_dir = self.mktemp('root_dir')
        self.shares_dir = self.mktemp('shares_dir')
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)

    @defer.inlineCallbacks
    def tearDown(self, *args):
        """Shutdown."""
        self.main.shutdown()
        yield super(TestSendingAllEvents, self).tearDown()

    def test_not_active(self):
        """All event listener is not subscribed by default."""
        subscribed = []
        self.main.event_q.subscribe = lambda l: subscribed.append(l)
        DBusInterface(self.bus, self.main, system_bus=self.bus)

        # normal event listener is subscribed but not the all events one
        self.assertTrue(any(isinstance(x, SyncdaemonEventListener)
                        for x in subscribed))
        self.assertFalse(any(isinstance(x, AllEventsSender)
                        for x in subscribed))

    def test_active(self):
        """All event listener is subscribed if indicated."""
        subscribed = []
        self.main.event_q.subscribe = lambda l: subscribed.append(l)
        DBusInterface(self.bus, self.main,
                      system_bus=self.bus, send_events=True)

        # both should be subscribed
        self.assertTrue(any(isinstance(x, SyncdaemonEventListener)
                        for x in subscribed))
        self.assertTrue(any(isinstance(x, AllEventsSender)
                        for x in subscribed))

    def test_events_are_sent(self):
        """Test that event information is sent to dbus."""
        dbus_iface = DBusInterface(self.bus, self.main,
                                   system_bus=self.bus, send_events=True)

        # patch the DBus emitter to register what is being sent
        class FakeEmitter(object):
            def emit_event(self, data):
                self.emitted = data
        fe = FakeEmitter()
        dbus_iface.events = fe

        # test with some method
        dbus_iface.all_events_sender.handle_default('FS_FILE_CREATE', path='x')
        self.assertEqual(fe.emitted,
                         dict(event_name='FS_FILE_CREATE', path='x'))


class TestDBusRestart(DBusTwistedTestCase):
    """Test main's restart method (and its interaction with dbus)."""

    def test_restart(self):
        """Start things up, then fire a restart, check it tries to restart."""
        d = defer.Deferred()
        def _handler(*a):
            """Async helper."""
            d.callback(True)
        # shutdown will fail when trying to restart because of our
        # half-backed dbus. That's OK, we don't actually want it
        # restarted :)
        self.main.external.shutdown = d.callback
        try:
            self.main.restart()
        except SystemExit:
            pass
        return d
    test_restart.skip = "leaves dbus stuff around, need to cleanup"


class ConfigTests(DBusTwistedTestCase):
    """Basic tests to the Config object exposed via D-Bus."""

    def get_client(self):
        """Return a Config DBusClient."""
        return DBusClient(self.bus, '/config', DBUS_IFACE_CONFIG_NAME)

    def test_get_throttling_limits(self):
        """Test get_throttling_limits exposed method."""
        client = self.get_client()
        d = defer.Deferred()
        aq = self.main.action_q
        def reply_handler(result):
            """Handle the reply."""
            self.assertEquals(aq.readLimit, result['download'])
            self.assertEquals(aq.writeLimit, result['upload'])
            self.assertEquals(100, result['download'])
            self.assertEquals(200, result['upload'])
            d.callback(True)
        def reply_handler_None(result):
            """Handle the reply."""
            self.assertEquals(-1, result['download'])
            self.assertEquals(-1, result['upload'])
            aq.readLimit = 100
            aq.writeLimit = 200
            client.call_method('get_throttling_limits',
                               reply_handler=reply_handler,
                               error_handler=self.error_handler)
        client.call_method('get_throttling_limits',
                           reply_handler=reply_handler_None,
                           error_handler=self.error_handler)
        return d


    def test_set_throttling_limits(self):
        """Test set_throttling_limits exposed method."""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler(_):
            """Handle the reply."""
            aq = self.main.action_q
            self.assertEquals(aq.readLimit, 100)
            self.assertEquals(aq.writeLimit, 500)
            d.callback(True)
        client.call_method('set_throttling_limits', 100, 500,
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_enable_bandwidth_throttling(self):
        """Test enable_bandwidth_throttling exposed method."""
        client = self.get_client()
        d = defer.Deferred()
        aq = self.main.action_q
        aq.throttling = False
        def reply_handler(_):
            """Handle the reply."""
            self.assertTrue(aq.throttling_enabled)
            d.callback(True)
        client.call_method('enable_bandwidth_throttling',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_disable_bandwidth_throttling(self):
        """Test disable_bandwidth_throttling exposed method."""
        client = self.get_client()
        d = defer.Deferred()
        aq = self.main.action_q
        aq.throttling = True
        def reply_handler(_):
            """Handle the reply."""
            self.assertFalse(aq.throttling_enabled)
            d.callback(True)
        client.call_method('disable_bandwidth_throttling',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_bandwidth_throttling_enabled(self):
        """Test bandwidth_throttling_enabled exposed method."""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler_enabled(result):
            """Handle the reply."""
            self.assertEquals(1, result)
            d.callback(True)

        def reply_handler_disabled(result):
            """Handle the reply."""
            self.assertEquals(0, result)
            self.main.action_q.throttling_enabled = True
            client.call_method('bandwidth_throttling_enabled',
                               reply_handler=reply_handler_enabled,
                               error_handler=self.error_handler)
        client.call_method('bandwidth_throttling_enabled',
                           reply_handler=reply_handler_disabled,
                           error_handler=self.error_handler)
        return d

    def test_udf_autosubscribe_enabled(self):
        """Test for Config.udf_autosubscribe_enabled."""
        client = self.get_client()
        d = defer.Deferred()

        def reply_handler_disabled(result):
            """Handle the reply."""
            self.assertFalse(result)
            config.get_user_config().set_udf_autosubscribe(not result)
            client.call_method('udf_autosubscribe_enabled',
                               reply_handler=reply_handler_enabled,
                               error_handler=self.error_handler)

        def reply_handler_enabled(result):
            """Handle the reply."""
            self.assertTrue(result)
            d.callback(True)

        client.call_method('udf_autosubscribe_enabled',
                           reply_handler=reply_handler_disabled,
                           error_handler=self.error_handler)
        return d

    def test_enable_udf_autosubscribe(self):
        """Test for Config.enable_udf_autosubscribe."""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler(_):
            """Handle the reply."""
            self.assertTrue(config.get_user_config().get_udf_autosubscribe())
            d.callback(True)
        client.call_method('enable_udf_autosubscribe',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_disable_udf_autosubscribe(self):
        """Test for Config.disable_udf_autosubscribe."""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler(_):
            """Handle the reply."""
            self.assertFalse(config.get_user_config().get_udf_autosubscribe())
            d.callback(True)
        client.call_method('disable_udf_autosubscribe',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_share_autosubscribe_enabled(self):
        """Test for Config.share_autosubscribe_enabled."""
        client = self.get_client()
        d = defer.Deferred()

        def reply_handler_disabled(result):
            """Handle the reply."""
            self.assertFalse(result)
            config.get_user_config().set_share_autosubscribe(not result)
            client.call_method('share_autosubscribe_enabled',
                               reply_handler=reply_handler_enabled,
                               error_handler=self.error_handler)

        def reply_handler_enabled(result):
            """Handle the reply."""
            self.assertTrue(result)
            d.callback(True)

        client.call_method('share_autosubscribe_enabled',
                           reply_handler=reply_handler_disabled,
                           error_handler=self.error_handler)
        return d

    def test_enable_share_autosubscribe(self):
        """Test for Config.enable_share_autosubscribe."""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler(_):
            """Handle the reply."""
            self.assertTrue(config.get_user_config().get_share_autosubscribe())
            d.callback(True)
        client.call_method('enable_share_autosubscribe',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_disable_share_autosubscribe(self):
        """Test for Config.disable_share_autosubscribe."""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler(_):
            """Handle the reply."""
            self.assertFalse(config.get_user_config().get_share_autosubscribe())
            d.callback(True)
        client.call_method('disable_share_autosubscribe',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_autoconnect_enabled(self):
        """Test for Config.autoconnect_enabled."""
        client = self.get_client()
        d = defer.Deferred()

        def reply_handler_disabled(result):
            """Handle the reply."""
            self.assertFalse(result)
            config.get_user_config().set_autoconnect(not result)
            d.callback(True)

        def reply_handler_enabled(result):
            """Handle the reply."""
            self.assertTrue(result)
            config.get_user_config().set_autoconnect(not result)
            client.call_method('autoconnect_enabled',
                               reply_handler=reply_handler_disabled,
                               error_handler=self.error_handler)

        client.call_method('autoconnect_enabled',
                           reply_handler=reply_handler_enabled,
                           error_handler=self.error_handler)
        return d

    def test_enable_autoconnect(self):
        """Test for Config.enable_autoconnect."""
        client = self.get_client()
        d = defer.Deferred()
        orig = config.get_user_config().get_autoconnect()
        # restore autoconnect to original value
        self.addCleanup(config.get_user_config().set_autoconnect, orig)

        def reply_handler(_):
            """Handle the reply."""
            self.assertTrue(config.get_user_config().get_autoconnect())
            d.callback(True)

        client.call_method('enable_autoconnect',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_disable_autoconnect(self):
        """Test for Config.disable_autoconnect."""
        client = self.get_client()
        d = defer.Deferred()
        orig = config.get_user_config().get_autoconnect()
        # restore autoconnect to original value
        self.addCleanup(config.get_user_config().set_autoconnect, orig)

        def reply_handler(_):
            """Handle the reply."""
            self.assertFalse(config.get_user_config().get_autoconnect())
            d.callback(True)

        client.call_method('disable_autoconnect',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_set_autoconnect_enabled(self):
        """Test for Config.set_autoconnect_enabled.

        DEPRECATED.

        """
        client = self.get_client()
        d = defer.Deferred()
        orig = config.get_user_config().get_autoconnect()
        # restore autoconnect to original value
        self.addCleanup(config.get_user_config().set_autoconnect, orig)

        def reply_handler(_):
            """Handle the reply."""
            self.assertEqual(config.get_user_config().get_autoconnect(),
                             not orig)
            d.callback(True)

        client.call_method('set_autoconnect_enabled', not orig,
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_show_all_notifications_enabled(self):
        """Test for Config.show_all_notifications_enabled."""
        client = self.get_client()
        d = defer.Deferred()

        def reply_handler_disabled(result):
            """Handle the reply."""
            self.assertTrue(result)
            config.get_user_config().set_show_all_notifications(not result)
            client.call_method('show_all_notifications_enabled',
                               reply_handler=reply_handler_enabled,
                               error_handler=self.error_handler)

        def reply_handler_enabled(result):
            """Handle the reply."""
            self.assertFalse(result)
            d.callback(True)

        client.call_method('show_all_notifications_enabled',
                           reply_handler=reply_handler_disabled,
                           error_handler=self.error_handler)
        return d

    def test_enable_show_all_notifications(self):
        """Test for Config.enable_show_all_notifications."""
        client = self.get_client()
        d = defer.Deferred()
        self.main.status_listener.show_all_notifications = False
        def reply_handler(_):
            """Handle the reply."""
            user_config = config.get_user_config()
            self.assertTrue(user_config.get_show_all_notifications())
            self.assertTrue(self.main.status_listener.show_all_notifications)
            d.callback(True)
        client.call_method('enable_show_all_notifications',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_disable_show_all_notifications(self):
        """Test for Config.disable_show_all_notifications."""
        client = self.get_client()
        d = defer.Deferred()
        self.main.status_listener.show_all_notifications = True
        def reply_handler(_):
            """Handle the reply."""
            user_config = config.get_user_config()
            self.assertFalse(user_config.get_show_all_notifications())
            self.assertFalse(self.main.status_listener.show_all_notifications)
            d.callback(True)
        client.call_method('disable_show_all_notifications',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d


class DBusOAuthTestCase(BaseTwistedTestCase):
    """Tests the interaction between dbus_interface and credentials.

    Check conditions when autconnecting is False.

    """

    timeout = 2
    method = 'register'
    autoconnecting = False

    @defer.inlineCallbacks
    def setUp(self):
        """Init."""
        yield super(DBusOAuthTestCase, self).setUp()
        self.events = []
        self.patch(DBusInterface, 'test', True)
        self.patch(dbus, 'Interface', FakedSSOBackend)
        self.patch(dbus.service, 'BusName', FakedBusName)
        self.bus = FakedSessionBus()

        self.data_dir = self.mktemp('data_dir')
        self.partials_dir = self.mktemp('partials')
        self.root_dir = self.mktemp('root_dir')
        self.shares_dir = self.mktemp('shares_dir')
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.dbus_iface = DBusInterface(bus=self.bus, main=self.main,
                                        system_bus=None, send_events=False)
        self.dbus_iface.event_queue.push = lambda name, **kw: \
                                           self.events.append((name, kw))
        self.memento = MementoHandler()
        logger.addHandler(self.memento)
        self.addCleanup(logger.removeHandler, self.memento)

    @defer.inlineCallbacks
    def tearDown(self, *args):
        """Shutdown."""
        self.main.shutdown()
        yield super(DBusOAuthTestCase, self).tearDown()

    @defer.inlineCallbacks
    def test_signals_are_connected(self):
        """Dbus signals have connected handlers."""

        def f(*a, **kw):
            """Just succeed."""
            self.assertIn((DBUS_CREDENTIALS_IFACE, None), self.bus.callbacks)
            cb = self.bus.callbacks[(DBUS_CREDENTIALS_IFACE, None)]
            self.assertEqual(self.dbus_iface._signal_handler, cb)
            self.dbus_iface._deferred.callback(None)

        self.patch(FakedSSOBackend, self.method, f)
        yield self.dbus_iface.connect(autoconnecting=self.autoconnecting)

    @defer.inlineCallbacks
    def test_signals_are_removed_after_connection(self):
        """Dbus signals are removed after connection."""

        def f(*a, **kw):
            """Just succeed."""
            self.dbus_iface._deferred.callback(None)

        self.patch(FakedSSOBackend, self.method, f)
        yield self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        self.assertNotIn((DBUS_CREDENTIALS_IFACE, None), self.bus.callbacks)

    @defer.inlineCallbacks
    def test_connect_pushes_SYS_USER_CONNECT_with_the_token(self):
        """On connect, the event SYS_USER_CONNECT is pushed."""

        def f(*a, **kw):
            """Receive credentials."""
            self.dbus_iface._signal_handler(FAKED_CREDENTIALS,
                                            member='CredentialsFound')

        self.patch(FakedSSOBackend, self.method, f)
        yield self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        self.assertEqual(self.events, [('SYS_USER_CONNECT',
                                       {'access_token': FAKED_CREDENTIALS})])

    @defer.inlineCallbacks
    def test_connect_raises_NoAccessToken_if_no_token(self):
        """If no credentials, NoAccessToken is raised."""

        def f(*a, **kw):
            """Receive error signal."""
            self.dbus_iface._signal_handler({'error_type': 'Error description',
                                             'error_detail': 'Detailed error'},
                                            member='CredentialsError')

        self.patch(FakedSSOBackend, self.method, f)
        d = self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        d.addErrback(lambda failure: self.assertEqual(NoAccessToken,
                                                      failure.type))
        yield d

    @defer.inlineCallbacks
    def test_connect_raises_NoAccessToken_if_auth_denied(self):
        """If no credentials, NoAccessToken if user denied auth."""

        def f(*a, **kw):
            """Receive error signal."""
            self.dbus_iface._signal_handler(member='AuthorizationDenied')

        self.patch(FakedSSOBackend, self.method, f)
        d = self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        d.addErrback(lambda failure: self.assertEqual(NoAccessToken,
                                                      failure.type))
        yield d

    @defer.inlineCallbacks
    def test_connect_raises_NoAccessToken_if_no_creds(self):
        """If no credentials, NoAccessToken if no credentials."""

        def f(*a, **kw):
            """Receive error signal."""
            self.dbus_iface._signal_handler(member='CredentialsNotFound')

        self.patch(FakedSSOBackend, self.method, f)
        d = self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        d.addErrback(lambda failure: self.assertEqual(NoAccessToken,
                                                      failure.type))
        yield d

    @defer.inlineCallbacks
    def test_dbus_exceptions_are_handled(self):
        """Every DbusException is handled."""
        expected = exceptions.DBusException('test me')

        def f(*a, **kw):
            """Just fail."""
            raise expected

        self.patch(FakedSSOBackend, self.method, f)
        d = self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        d.addErrback(lambda failure: self.assertEqual(expected, failure.value))
        yield d

    @defer.inlineCallbacks
    def test_other_exceptions_are_logged_and_re_raised(self):
        """Every other Exception is logged and re raised."""
        expected = TypeError('test me')

        def f(*a, **kw):
            """Just fail."""
            raise expected

        self.patch(FakedSSOBackend, self.method, f)
        d = self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        d.addErrback(lambda failure: self.assertEqual(expected, failure.value))
        yield d
        self.assertTrue(len(self.memento.records) > 0)
        record = self.memento.records[1]
        msg = record.message
        self.assertIn('connect failed while getting the token', msg)
        self.assertIn(expected, record.exc_info)

    def test_oauth_credentials_are_none_at_startup(self):
        """If the oauth_credentials are not passed as param, they are None."""
        self.assertTrue(self.dbus_iface.oauth_credentials is None)

    @defer.inlineCallbacks
    def test_oauth_credentials_are_used_to_connect(self):
        """If present, the oauth_credentials are used to connect."""
        expected = {'consumer_key': 'ubuntuone',
                    'consumer_secret': 'hammertime',
                    'token': 'faked_token',
                    'token_secret': 'faked_token_secret'}
        self.dbus_iface.oauth_credentials = (expected['token'],
                                             expected['token_secret'])
        yield self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        self.assertEqual(self.events, [('SYS_USER_CONNECT',
                                       {'access_token': expected})])

    @defer.inlineCallbacks
    def test_oauth_credentials_can_be_a_four_uple(self):
        """If present, the oauth_credentials are used to connect."""
        expected = {'consumer_key': 'faked_consumer_key',
                    'consumer_secret': 'faked_consumer_secret',
                    'token': 'faked_token',
                    'token_secret': 'faked_token_secret'}
        self.dbus_iface.oauth_credentials = (expected['consumer_key'],
                                             expected['consumer_secret'],
                                             expected['token'],
                                             expected['token_secret'])
        yield self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        self.assertEqual(self.events, [('SYS_USER_CONNECT',
                                       {'access_token': expected})])

    @defer.inlineCallbacks
    def test_log_warning_if_oauth_credentials_len_is_useless(self):
        """Log a warning and return if the oauth_credentials are useless."""
        self.dbus_iface.oauth_credentials = ('consumer_key',
                                             'consumer_secret',
                                             'token_secret')
        yield self.dbus_iface.connect(autoconnecting=self.autoconnecting)
        self.assertEqual(self.events, [])
        msgs = (str(self.dbus_iface.oauth_credentials), 'useless')
        self.assertTrue(self.memento.check_warning(*msgs))

    def test_signal_handler_remains_generic(self):
        """The signal handler function should be generic."""
        self.dbus_iface._signal_handler()
        # no failure
        self.assertTrue(self.memento.check_debug('member: None'))

        self.dbus_iface._signal_handler(no_member_kwarg='Test')
        # no failure
        self.assertTrue(self.memento.check_debug('member: None'))


class DBusOAuthTestCaseRegister(DBusOAuthTestCase):
    """Tests the interaction between dbus_interface and credentials.

    Check conditions when autconnecting is True.

    """

    method = 'find_credentials'
    autoconnecting = True


class FolderTests(DBusTwistedTestCase):
    """Tests for the Folder object exposed via dbus."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup the test."""
        yield super(FolderTests, self).setUp()
        self.home_dir = self.mktemp('ubuntuonehacker')
        self.folders_client = DBusClient(self.bus, '/folders',
                                         DBUS_IFACE_FOLDERS_NAME)
        self._old_home = os.environ['HOME']
        os.environ['HOME'] = self.home_dir

    def tearDown(self):
        os.environ['HOME'] = self._old_home
        return DBusTwistedTestCase.tearDown(self)

    def _create_udf(self, id, node_id, suggested_path, subscribed=True):
        """Create an UDF and returns it and the volume."""
        path = get_udf_path(suggested_path)
        # make sure suggested_path is unicode
        if isinstance(suggested_path, str):
            suggested_path = suggested_path.decode('utf-8')
        udf = UDF(str(id), str(node_id), suggested_path, path, subscribed)
        return udf

    def testget_udf_dict(self):
        """Test for Folders.get_udf_dict."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=False)
        udf_dict = get_udf_dict(udf)
        # check the path it's unicode
        self.assertEquals(udf_dict['path'], udf.path.decode('utf-8'))
        self.assertEquals(udf_dict['volume_id'], udf.id)
        self.assertEquals(udf_dict['suggested_path'], udf.suggested_path)
        self.assertEquals(udf_dict['node_id'], udf.node_id)
        self.assertFalse(udf_dict['subscribed'])

    def testget_udf_dict_bad_encoding(self):
        """Test for Folders.get_udf_dict."""
        suggested_path = u'~/Música'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=False)
        udf.suggested_path = udf.suggested_path.encode('utf-8')
        udf_dict = get_udf_dict(udf)
        # check the path it's unicode
        self.assertEquals(udf_dict['path'], udf.path.decode('utf-8'))
        self.assertEquals(udf_dict['volume_id'], udf.id)
        self.assertEquals(repr(udf_dict['suggested_path']),
                          repr(udf.suggested_path.decode('utf-8')))
        self.assertEquals(udf_dict['node_id'], udf.node_id)
        self.assertFalse(udf_dict['subscribed'])

    @defer.inlineCallbacks
    def test_get_folders(self):
        """Test for Folders.get_folders."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path)
        d = defer.Deferred()
        self.folders_client.call_method('get_folders',
                                        reply_handler=d.callback,
                                        error_handler=self.error_handler)
        info = yield d
        self.assertEquals(len(info), 0)
        # add a udf
        yield self.main.vm.add_udf(udf)
        d2 = defer.Deferred()
        self.folders_client.call_method('get_folders',
                                        reply_handler=d2.callback,
                                        error_handler=self.error_handler)
        info = yield d2
        udf_dict = get_udf_dict(self.main.vm.get_volume(udf.volume_id))
        self.assertEquals(1, len(info))
        for key, value in udf_dict.items():
            self.assertEquals(info[0][key], value)

    def test_get_info(self):
        """Test for Folders.get_info."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path)
        d = defer.Deferred()
        self.folders_client.call_method('get_info', udf.path,
                                        reply_handler=self.error_handler,
                                        error_handler=d.callback)
        def check_error(f):
            """Check we get an error."""
            # check the error type
            self.assertEquals('org.freedesktop.DBus.Python.KeyError',
                              f._dbus_error_name)
            # add a udf
            add_deferred = self.main.vm.add_udf(udf)
            def get_info(_):
                """Call get_info once we created the udf."""
                d = defer.Deferred()
                self.folders_client.call_method('get_info', udf.path,
                                            reply_handler=d.callback,
                                            error_handler=self.error_handler)
                return d
            add_deferred.addCallback(get_info)
            return add_deferred
        def check(info):
            """Check we get the udf info."""
            udf_dict = get_udf_dict(self.main.vm.get_volume(udf.volume_id))
            self.assertEquals(info, udf_dict)
        d.addCallback(check_error)
        d.addCallback(check)
        return d

    def test_create(self):
        """Test for Folders.create."""
        path = os.path.join(self.home_dir, u'ñoño/'.encode('utf-8'))
        id = uuid.uuid4()
        node_id = uuid.uuid4()
        d = defer.Deferred()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf."""
            # check that the marker it's the full path to the udf
            udf_path = os.path.join(os.path.expanduser(path.encode('utf-8')), name.encode('utf-8'))
            if str(marker) != udf_path:
                d.errback(ValueError("marker != path - "
                                     "marker: %r path: %r" % (marker, udf_path)))
            self.main.event_q.push("AQ_CREATE_UDF_OK", volume_id=id,
                                   node_id=node_id, marker=marker)
        self.main.action_q.create_udf = create_udf
        def created_handler(info):
            """FolderCreated handler."""
            d.callback(info)
        match = self.bus.add_signal_receiver(created_handler,
                                             signal_name='FolderCreated')
        self.signal_receivers.add(match)
        self.folders_client.call_method('create', path,
                                        reply_handler=lambda *agrs: None,
                                        error_handler=self.error_handler)
        def check(info):
            """Check the FolderCreated info."""
            self.assertTrue(os.path.exists(info['path'].encode('utf-8')),
                            info['path'].encode('utf-8'))
            self.assertEquals(info['path'].encode('utf-8'),
                              os.path.normpath(path))
            mdobj = self.main.fs.get_by_path(path)
            udf = self.main.vm.get_volume(mdobj.share_id)
            self.assertNotEqual(None, udf)
            self.assertEquals(udf.path, os.path.normpath(path))
            udf_dict = get_udf_dict(udf)
            self.assertEquals(info, udf_dict)
            self.main.vm.udf_deleted(udf.volume_id)

        d.addCallback(check)
        d.addErrback(self.error_handler)
        return d

    def test_create_server_error(self):
        """Test for Folders.create."""
        path = os.path.join(self.home_dir, u'ñoño')
        d = defer.Deferred()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf, that fails."""
            self.main.event_q.push("AQ_CREATE_UDF_ERROR",
                                   marker=marker, error="Oops, error!")
        self.main.action_q.create_udf = create_udf
        def create_handler(info, error):
            """FolderCreateError handler."""
            d.callback((info, error))
        match = self.bus.add_signal_receiver(create_handler,
                                             signal_name='FolderCreateError')
        self.signal_receivers.add(match)
        self.folders_client.call_method('create', path,
                                        reply_handler=lambda *agrs: None,
                                        error_handler=self.error_handler)
        def check(result):
            """Check the result."""
            info, error = result
            self.assertEquals(info['path'], path)
            self.assertEquals(error, 'Oops, error!')
        d.addCallback(check)
        return d

    def test_create_client_error(self):
        """Test for Folders.create."""
        path = os.path.join(self.home_dir, u'ñoño')
        d = defer.Deferred()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf, that fails."""
            raise ValueError("I'm broken.")
        self.main.action_q.create_udf = create_udf
        def create_handler(info, error):
            """FolderCreateError handler."""
            d.callback((info, error))
        match = self.bus.add_signal_receiver(create_handler,
                                             signal_name='FolderCreateError')
        self.signal_receivers.add(match)
        self.folders_client.call_method('create', path,
                                        reply_handler=lambda *agrs: None,
                                        error_handler=self.error_handler)
        def check(result):
            """Check the result."""
            info, error = result
            self.assertEquals(info['path'], path)
            self.assertEquals(error, "UNKNOWN_ERROR: I'm broken.")
        d.addCallback(check)
        return d

    def test_create_error_signal(self):
        """Test for FolderCreateError."""
        path = os.path.join(self.home_dir, u'ñoño'.encode('utf-8'))
        d = defer.Deferred()
        def create_error_handler(info, error):
            """FolderCreateError handler"""
            self.assertEquals(info['path'], path.decode('utf-8'))
            self.assertEquals(error, "I'm broken")
            d.callback(True)
        match = self.bus.add_signal_receiver(create_error_handler,
                                             signal_name='FolderCreateError')
        self.signal_receivers.add(match)
        # TODO: once create_udf is implemented remove this callLater
        reactor.callLater(0, self.main.event_q.push,
                          'VM_UDF_CREATE_ERROR', path=path, error="I'm broken")
        return d

    @defer.inlineCallbacks
    def test_delete(self):
        """Test for Folders.delete."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path)
        yield self.main.vm.add_udf(udf)
        d = defer.Deferred()
        def delete_volume(volume_id, path):
            """Fake delete_volume."""
            self.main.event_q.push("AQ_DELETE_VOLUME_OK", volume_id=udf.id)
        self.main.action_q.delete_volume = delete_volume
        def deleted_handler(info):
            """FolderDeleted handler."""
            self.assertRaises(KeyError, self.main.fs.get_by_path,
                              info['path'].encode('utf-8'))
            self.assertRaises(VolumeDoesNotExist,
                              self.main.vm.get_volume, str(info['volume_id']))
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_handler,
                                             signal_name='FolderDeleted')
        self.signal_receivers.add(match)
        def check_deleted(info):
            """The callback."""
            self.assertNotIn(udf.volume_id, self.main.vm.udfs)
            self.assertRaises(KeyError, self.main.fs.get_by_path, udf.path)
            self.assertRaises(VolumeDoesNotExist,
                              self.main.vm.get_volume, udf.volume_id)
        self.folders_client.call_method('delete', udf.volume_id,
                                        reply_handler=check_deleted,
                                        error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_delete_error_signal(self):
        """Test for FolderDeleteError."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path)
        yield self.main.vm.add_udf(udf)
        d = defer.Deferred()
        # patch delete_volume to fail
        def delete_volume(volume_id, path):
            """Fake delete_volume."""
            self.main.event_q.push("AQ_DELETE_VOLUME_ERROR",
                                   volume_id=udf.volume_id, error="I'm broken")
        self.main.action_q.delete_volume = delete_volume
        def deleted_error_handler(info, error):
            """FolderDeleteError handler."""
            self.assertEquals(info['volume_id'], udf.volume_id)
            self.assertEquals(error, "I'm broken")
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_error_handler,
                                             signal_name='FolderDeleteError')
        self.signal_receivers.add(match)
        self.folders_client.call_method('delete', udf.volume_id,
                                        reply_handler=lambda *args: None,
                                        error_handler=self.error_handler)
        yield d
        self.main.vm.udf_deleted(udf.volume_id)

    @defer.inlineCallbacks
    def test_delete_error_signal_folder_id(self):
        """Test for FolderDeleteError for a volume that doesn't exists."""
        udf_id = 'foobar'
        d = defer.Deferred()
        def deleted_error_handler(info, error):
            """FolderDeleteError handler."""
            d.callback((info, error))
        match = self.bus.add_signal_receiver(deleted_error_handler,
                                             signal_name='FolderDeleteError')
        self.signal_receivers.add(match)
        self.folders_client.call_method('delete', udf_id,
                                        reply_handler=lambda *args: None,
                                        error_handler=d.errback)
        info, error = yield d
        self.assertEquals(info['volume_id'], udf_id)
        self.assertEquals(error, "DOES_NOT_EXIST")

    @defer.inlineCallbacks
    def test_subscribe(self):
        """Test for Folders.subscribe and that it fires a dbus signal."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=False)
        yield self.main.vm.add_udf(udf)
        d = defer.Deferred()
        def subscribe_handler(info):
            """FolderSubscribed handler."""
            d.callback(info)
        match = self.bus.add_signal_receiver(subscribe_handler,
                                             signal_name='FolderSubscribed')
        self.signal_receivers.add(match)
        self.folders_client.call_method('subscribe', udf.volume_id,
                                        reply_handler=lambda x: None,
                                        error_handler=self.error_handler)
        def check(info):
            """Check that the folder is subscribed."""
            self.assertTrue(info['subscribed'],
                            "UDF %s isn't subscribed" % udf.volume_id)
        d.addCallback(check)
        d.addCallback(lambda _: self.main.vm.udf_deleted(udf.volume_id))
        yield d

    @defer.inlineCallbacks
    def test_unsubscribe(self):
        """Test for Folders.unsubscribe."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=True)
        yield self.main.vm.add_udf(udf)
        d = defer.Deferred()
        self.folders_client.call_method('unsubscribe', udf.volume_id,
                                        reply_handler=d.callback,
                                        error_handler=self.error_handler)
        def check(r):
            """Check that the folder it's not subscribed."""
            self.assertFalse(self.main.vm.udfs[udf.volume_id].subscribed,
                            "UDF %s is subscribed" % udf.volume_id)
        d.addCallback(check)
        yield d

    @defer.inlineCallbacks
    def test_unsubscribe_signal(self):
        """Test for Folders.unsubscribe fired dbus signal."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=True)
        yield self.main.vm.add_udf(udf)
        signal_deferred = defer.Deferred()
        d = defer.Deferred()
        def unsubscribe_handler(info):
            """FolderUnSubscribed handler."""
            self.assertFalse(info['subscribed'],
                             "UDF %s is subscribed" % udf.volume_id)
            signal_deferred.callback(info)
        match = self.bus.add_signal_receiver(unsubscribe_handler,
                                             signal_name='FolderUnSubscribed')
        self.signal_receivers.add(match)
        self.folders_client.call_method('unsubscribe', udf.volume_id,
                                        reply_handler=d.callback,
                                        error_handler=self.error_handler)
        def check(r):
            """Check that the folder it's not subscribed."""
            a_udf = self.main.vm.udfs[udf.volume_id]
            self.assertFalse(a_udf.subscribed,
                            "UDF %s is subscribed" % a_udf.volume_id)
            return signal_deferred
        d.addCallback(check)
        d.addCallback(lambda _: self.main.vm.udf_deleted(udf.volume_id))
        yield d

    @defer.inlineCallbacks
    def test_refresh_volumes(self):
        """Just check that refresh_volumes method works."""
        client = DBusClient(self.bus, '/folders', DBUS_IFACE_FOLDERS_NAME)
        list_volumes_d = defer.Deferred()
        self.main.action_q.list_volumes = lambda: list_volumes_d.callback(True)

        d = defer.Deferred()
        client.call_method('refresh_volumes',
                           reply_handler=d.callback,
                           error_handler=self.error_handler)
        yield list_volumes_d
        yield d


class LauncherTests(DBusTwistedTestCase, MockerTestCase):
    """Tests for the launcher Dbus Interface."""

    @defer.inlineCallbacks
    def test_unset_urgency(self):
        """Calling remove_urgency removes the urgency hint."""
        launcher = self.mocker.replace(
            "ubuntuone.platform.linux.launcher.UbuntuOneLauncher")
        launcher()
        launcher_instance = self.mocker.mock()
        self.mocker.result(launcher_instance)
        launcher_instance.set_urgent(False)
        self.mocker.replay()
        client = DBusClient(self.bus, '/launcher', DBUS_IFACE_LAUNCHER_NAME)
        d = defer.Deferred()
        client.call_method('unset_urgency',
                           reply_handler=d.callback,
                           error_handler=self.error_handler)
        yield d


class ShareTests(DBusTwistedTestCase):
    """Share specific tests."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup the test."""
        yield super(ShareTests, self).setUp()
        self.client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)

    def _create_share(self, volume_id=None, node_id=None, access_level='View',
                      accepted=True, subscribed=False):
        """Create a Share and return it."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        if volume_id is None:
            volume_id = str(uuid.uuid4())
        if node_id is None:
            node_id = str(uuid.uuid4())
        share = Share(path=share_path, volume_id=volume_id, node_id=node_id,
                      accepted=accepted, access_level=access_level,
                      subscribed=subscribed)
        return share

    @defer.inlineCallbacks
    def test_delete_share(self):
        """Test for Shares.delete_share."""
        share = self._create_share(accepted=True)
        yield self.main.vm.add_share(share)
        d = defer.Deferred()
        def delete_volume(volume_id, path):
            """Fake delete_volume."""
            self.main.event_q.push("AQ_DELETE_VOLUME_OK", volume_id=volume_id)
        self.main.action_q.delete_volume = delete_volume
        def deleted_handler(info):
            """ShareDeleted handler."""
            self.assertRaises(KeyError, self.main.fs.get_by_path,
                              info['path'].decode('utf-8'))
            self.assertRaises(VolumeDoesNotExist,
                              self.main.vm.get_volume, str(info['volume_id']))
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_handler,
                                             signal_name='ShareDeleted')
        self.signal_receivers.add(match)
        def check_deleted(info):
            """The callback."""
            self.assertNotIn(share.volume_id, self.main.vm.shares)
            self.assertRaises(KeyError, self.main.fs.get_by_path, share.path)
            self.assertRaises(VolumeDoesNotExist,
                              self.main.vm.get_volume, share.volume_id)
        self.client.call_method('delete_share', share.volume_id,
                           reply_handler=check_deleted,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_delete_share_from_me(self):
        """Test for Shares.delete_share with share from_me."""
        share = self._create_share(accepted=True)
        yield self.main.vm.add_shared(share)
        d = defer.Deferred()
        def delete_share(volume_id):
            """Fake delete_volume."""
            self.main.event_q.push("AQ_DELETE_SHARE_OK", share_id=volume_id)
        self.patch(self.main.action_q, 'delete_share', delete_share)
        def deleted_handler(info):
            """ShareDeleted handler."""
            self.assertRaises(KeyError,
                              self.main.vm.shared.__getitem__, str(info['volume_id']))
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_handler,
                                             signal_name='ShareDeleted')
        self.signal_receivers.add(match)
        self.client.call_method('delete_share', share.volume_id,
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_delete_share_error_signal(self):
        """Test for Shares.delete_share with an error."""
        share = self._create_share(accepted=True)
        yield self.main.vm.add_share(share)
        d = defer.Deferred()
        # patch delete_volume to fail
        def delete_volume(volume_id, path):
            """Fake delete_volume."""
            self.main.event_q.push("AQ_DELETE_VOLUME_ERROR",
                                   volume_id=volume_id, error="I'm broken")
        self.main.action_q.delete_volume = delete_volume
        def deleted_error_handler(info, error):
            """ShareDeleteError handler."""
            self.assertEquals(info['volume_id'], share.volume_id)
            self.assertEquals(error, "I'm broken")
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_error_handler,
                                             signal_name='ShareDeleteError')
        self.signal_receivers.add(match)
        self.client.call_method('delete_share', share.volume_id,
                           reply_handler=lambda *args: None,
                           error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_delete_share_from_me_error(self):
        """Test failure of Shares.delete_share with a share from_me."""
        share = self._create_share(accepted=True)
        yield self.main.vm.add_shared(share)
        d = defer.Deferred()
        # patch delete_share to fail
        def delete_share(share_id):
            """Fake delete_share."""
            self.main.event_q.push("AQ_DELETE_SHARE_ERROR",
                                   share_id=share_id, error="I'm broken")
        self.patch(self.main.action_q, 'delete_share', delete_share)
        def deleted_error_handler(info, error):
            """ShareDeleteError handler."""
            self.assertEquals(info['volume_id'], share.volume_id)
            self.assertEquals(error, "I'm broken")
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_error_handler,
                                             signal_name='ShareDeleteError')
        self.signal_receivers.add(match)
        self.client.call_method('delete_share', share.volume_id,
                           reply_handler=lambda *args: None,
                           error_handler=self.error_handler)
        yield d

    def test_delete_share_from_me_doesnotexist(self):
        """Test failure of Shares.delete_share with a share from_me."""
        d = defer.Deferred()
        # patch delete_share to fail
        def deleted_error_handler(info, error):
            """ShareDeleteError handler."""
            self.assertEquals(info['volume_id'], 'missing_share_id')
            self.assertEquals(error, "DOES_NOT_EXIST")
            d.callback(True)
        match = self.bus.add_signal_receiver(deleted_error_handler,
                                             signal_name='ShareDeleteError')
        self.signal_receivers.add(match)
        self.client.call_method('delete_share', 'missing_share_id',
                           reply_handler=lambda *args: None,
                           error_handler=self.error_handler)
        return d

    @defer.inlineCallbacks
    def test_subscribe(self):
        """Test for Shares.subscribe and that it fires a dbus signal."""
        share = self._create_share(accepted=True, subscribed=False)
        yield self.main.vm.add_share(share)

        d = defer.Deferred()
        match = self.bus.add_signal_receiver(d.callback,
                                             signal_name='ShareSubscribed')
        self.signal_receivers.add(match)

        match = self.bus.add_signal_receiver(lambda _, error: d.errback(error),
                                             signal_name='ShareSubscribeError')
        self.signal_receivers.add(match)

        self.client.call_method('subscribe', share.volume_id,
                                reply_handler=lambda x: None,
                                error_handler=self.error_handler)
        info = yield d
        self.assertTrue(bool(info['subscribed']),
                        "share %r should be subscribed" % share)
        yield self.main.vm.share_deleted(share.volume_id)

    @defer.inlineCallbacks
    def test_subscribe_error(self):
        """Test for Shares.subscribe when there is an error."""
        # do not add a share to have an error

        d = defer.Deferred()

        match = self.bus.add_signal_receiver(lambda sid, _: d.callback(sid),
                                             signal_name='ShareSubscribeError')
        self.signal_receivers.add(match)

        match = self.bus.add_signal_receiver(d.errback,
                                             signal_name='ShareSubscribed')
        self.signal_receivers.add(match)

        self.client.call_method('subscribe', 'invalid_id',
                                reply_handler=lambda x: None,
                                error_handler=self.error_handler)
        yield d

    @defer.inlineCallbacks
    def test_unsubscribe(self):
        """Test for Shares.unsubscribe."""
        share = self._create_share(accepted=True, subscribed=True)
        yield self.main.vm.add_share(share)

        d = defer.Deferred()
        match = self.bus.add_signal_receiver(d.callback,
                                             signal_name='ShareUnSubscribed')
        self.signal_receivers.add(match)

        match = self.bus.add_signal_receiver(lambda _, error: d.errback(error),
                                           signal_name='ShareUnSubscribeError')
        self.signal_receivers.add(match)

        self.client.call_method('unsubscribe', share.volume_id,
                                reply_handler=lambda x: None,
                                error_handler=self.error_handler)
        info = yield d

        self.assertFalse(info['subscribed'],
                         "share %r should be subscribed" % share)

        self.main.vm.share_deleted(share.volume_id)

    @defer.inlineCallbacks
    def test_unsubscribe_error(self):
        """Test for Shares.unsubscribe when there is an error."""
        # do not add a share to have an error

        d = defer.Deferred()
        match = self.bus.add_signal_receiver(lambda sid, _: d.callback(sid),
                                           signal_name='ShareUnSubscribeError')
        self.signal_receivers.add(match)

        match = self.bus.add_signal_receiver(d.errback,
                                             signal_name='ShareUnSubscribed')
        self.signal_receivers.add(match)

        self.client.call_method('unsubscribe', 'invalid_id',
                                reply_handler=lambda x: None,
                                error_handler=self.error_handler)
        yield d


class TestStatusEmitSignals(DBusTwistedTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        yield super(TestStatusEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()

    def test_emit_content_queue_changed(self):
        """Emit ContentQueueChanged."""
        self.status.ContentQueueChanged = self.signal_method
        self.signal_method()
        self.mocker.replay()
        # will assert that the signal method was called
        self.status.emit_content_queue_changed()

    def test_emit_requestqueue_removed(self):
        """Emit RequestQueueRemoved."""
        self.status.RequestQueueRemoved = self.signal_method
        self.signal_method('name', 'id', {})
        self.mocker.replay()
        # will assert that the signal method was called
        self.status.emit_requestqueue_removed('name', 'id', {})

    def test_emit_requestqueue_added(self):
        """Emit RequestQueueAdded."""
        self.status.RequestQueueAdded = self.signal_method
        self.signal_method('name', 'id', {})
        self.mocker.replay()
        # will assert that the signal method was called
        self.status.emit_requestqueue_added('name', 'id', {})

    def test_emit_invalid_name(self):
        """Emit InvalidName."""
        dirname = 'dirname'
        filename = 'filename'
        self.status.InvalidName = self.signal_method
        self.signal_method(unicode(dirname), str(filename))
        self.mocker.replay()
        self.status.emit_invalid_name(dirname, filename)

    def test_emit_broken_node(self):
        """Emit BrokenNode."""
        volume_id = 'volume_id'
        node_id = 'node_id'
        mdid = 'mdid'
        path = 'path'
        self.status.BrokenNode = self.signal_method
        self.signal_method(volume_id, node_id, mdid, path.decode('utf8'))
        self.mocker.replay()
        self.status.emit_broken_node(volume_id, node_id, mdid,
                                     path.decode('utf8'))

    def test_emit_status_changed(self):
        """Emit StatusChanged."""
        status = 'status'
        status_mock = self.mocker.mock()
        self.status.syncdaemon_status = status_mock
        status_mock.current_status()
        self.mocker.result(status)
        self.status.StatusChanged = self.signal_method
        self.signal_method(status)
        self.mocker.replay()
        self.status.emit_status_changed(status)

    def test_emit_download_started(self):
        """Emit DownloadStarted."""
        download = 'download'
        self.status.DownloadStarted = self.signal_method
        self.signal_method(download)
        self.mocker.replay()
        self.status.emit_download_started(download)

    def test_emit_download_file_progress(self):
        """Emit DownloadFileProgress."""
        download = 'download'
        string_info = {'blah':'3', 'do':'4'}
        self.status.DownloadFileProgress = self.signal_method
        self.signal_method(download, string_info)
        self.mocker.replay()
        self.status.emit_download_file_progress(download, blah=3, do=4)

    def test_emit_download_finished(self):
        """Emit DownloadFinished."""
        download = 'download'
        string_info = {'blah':'3', 'do':'4'}
        self.status.DownloadFinished = self.signal_method
        self.signal_method(download, string_info)
        self.mocker.replay()
        self.status.emit_download_finished(download, blah=3, do=4)

    def test_emit_upload_started(self):
        """Emit UploadStarted."""
        upload = 'upload'
        self.status.UploadStarted = self.signal_method
        self.signal_method(upload)
        self.mocker.replay()
        self.status.emit_upload_started(upload)

    def test_emit_upload_file_progress(self):
        """Emit UploadFileProgress."""
        upload = 'upload'
        string_info = {'blah':'3', 'do':'4'}
        self.status.UploadFileProgress = self.signal_method
        self.signal_method(upload , string_info)
        self.mocker.replay()
        self.status.emit_upload_file_progress(upload , blah=3, do=4)

    def test_emit_upload_finished(self):
        """Emit UploadFinished."""
        upload = 'upload'
        string_info = {'blah':'3', 'do':'4'}
        self.status.UploadFinished= self.signal_method
        self.signal_method(upload , string_info)
        self.mocker.replay()
        self.status.emit_upload_finished(upload, blah=3, do=4)

    def test_emit_account_changed(self):
        """Emit AccountChanged."""
        info_dict = {'purchased_bytes': unicode('34')}
        account_info = self.mocker.mock()
        account_info.purchased_bytes
        self.mocker.result('34')
        self.status.AccountChanged = self.signal_method
        self.signal_method(info_dict)
        self.mocker.replay()
        self.status.emit_account_changed(account_info)

    def test_emit_metaqueue_changed(self):
        """Emit MetaQueueChanged."""
        self.status.MetaQueueChanged = self.signal_method
        self.signal_method()
        self.mocker.replay()
        self.status.emit_metaqueue_changed()


class TestEventsEmitSignals(DBusTwistedTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        yield super(TestEventsEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()

    def test_emit_event(self):
        """Test the Event signal."""
        items = {1:2}
        event = self.mocker.mock()
        event.iteritems()
        self.mocker.result(items.iteritems())
        self.events.Event = self.signal_method
        self.signal_method({'1':'2'})
        self.mocker.replay()
        self.events.emit_event(event)


class TestSyncDaemonEmitSignals(DBusTwistedTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        yield super(TestSyncDaemonEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()


    def test_emit_root_mismatch(self):
        """Emit RootMismatch signal."""
        root_id = 'root_id'
        new_root_id = 'new_root_id'
        self.sync.RootMismatch = self.signal_method
        self.signal_method(root_id, new_root_id)
        self.mocker.replay()
        self.sync.emit_root_mismatch(root_id, new_root_id)

    def test_emit_quota_exceeded(self):
        """Emit QuotaExceeded signal."""
        volume_dict = 'volume_dict'
        self.sync.QuotaExceeded = self.signal_method
        self.signal_method(volume_dict)
        self.mocker.replay()
        self.sync.emit_quota_exceeded(volume_dict)


class TestSharesEmitSignals(DBusTwistedTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        yield super(TestSharesEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.shares.syncdaemon_shares = self.mocker.mock()
        self.get_share_dict = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.get_share_dict')

    def test_emit_share_changed_deleted(self):
        share = 'share'
        message = 'deleted'
        share_dict = {'share':'id'}
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.shares.ShareDeleted = self.signal_method
        self.signal_method(share_dict)
        self.mocker.replay()
        self.shares.emit_share_changed(message, share)

    def test_emit_share_changed_changed(self):
        share = 'share'
        message = 'changed'
        share_dict = {'share':'id'}
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.shares.ShareChanged = self.signal_method
        self.signal_method(share_dict)
        self.mocker.replay()
        self.shares.emit_share_changed(message, share)

    def test_emit_share_delete_error(self):
        """Emit ShareDeleteError signal."""
        share = 'share'
        error = 'error'
        share_dict = {'share':'id'}
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.shares.ShareDeleteError = self.signal_method
        self.signal_method(share_dict, error)
        self.mocker.replay()
        self.shares.emit_share_delete_error(share, error)

    def test_emit_free_space(self):
        """Emit ShareChanged when free space changes """
        free_bytes = '0'
        share_dict = shares = {'1': 'share', 'free_bytes':'0' }
        share = 'share'
        share_id = '1'
        self.shares.syncdaemon_shares.shares
        self.mocker.result(shares)
        self.shares.syncdaemon_shares.shares
        self.mocker.result(shares)
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.shares.ShareChanged = self.signal_method
        self.signal_method(share_dict)
        self.mocker.replay()
        self.shares.emit_free_space(share_id, free_bytes)

    def test_emit_share_created(self):
        """Emit ShareCreated signal """
        share_info = 'info'
        self.shares.ShareCreated = self.signal_method
        self.signal_method(share_info)
        self.mocker.replay()
        self.shares.emit_share_created(share_info)

    def test_emit_share_create_error(self):
        """Emit ShareCreateError signal."""
        share_info = 'id'
        error = 'error'
        info = 'info'
        self.shares.syncdaemon_shares.get_create_error_share_info(share_info)
        self.mocker.result(info)
        self.shares.ShareCreateError = self.signal_method
        self.signal_method(info, error)
        self.mocker.replay()
        self.shares.emit_share_create_error(share_info, error)

    def test_emit_share_answer_response(self):
        """Emits ShareAnswerResponse signal."""
        share_id = 'id'
        answer = 'yes'
        error = 'boom'
        answer_info = dict(volume_id=share_id, answer=answer, error=error)
        self.shares.ShareAnswerResponse = self.signal_method
        self.signal_method(answer_info)
        self.mocker.replay()
        self.shares.emit_share_answer_response(share_id, answer, error)

    def test_emit_new_share(self):
        """Emits NewShare signal."""
        share_id = 'id'
        share = 'share'
        share_dict = {'share':'id'}
        self.shares.syncdaemon_shares.get_volume(share_id)
        self.mocker.result(share)
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.shares.NewShare = self.signal_method
        self.signal_method(share_dict)
        self.mocker.replay()
        self.shares.emit_new_share(share_id)

    def test_emit_share_subscribed(self):
        """Emit the ShareSubscribed signal"""
        share = 'share'
        share_dict = {'share' : 'id'}
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.shares.ShareSubscribed = self.signal_method
        self.signal_method(share_dict)
        self.mocker.replay()
        self.shares.emit_share_subscribed(share)

    def test_emit_share_subscribe_error(self):
        """Emit the ShareSubscribeError signal"""
        share_id = 'id'
        error = 'error'
        self.shares.ShareSubscribeError = self.signal_method
        self.signal_method({'id': share_id}, str(error))
        self.mocker.replay()
        self.shares.emit_share_subscribe_error(share_id, error)

    def test_emit_share_unsubscribed(self):
        """Emit the ShareUnSubscribed signal"""
        share = 'share'
        share_dict = {'share':'id'}
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.shares.ShareUnSubscribed = self.signal_method
        self.signal_method(share_dict)
        self.mocker.replay()
        self.shares.emit_share_unsubscribed(share)

    def test_emit_share_unsubscribe_error(self):
        """Emit the ShareUnSubscribeError signal"""
        share_id = 'id'
        error = 'error'
        self.shares.ShareUnSubscribeError = self.signal_method
        self.signal_method({'id': share_id}, str(error))
        self.mocker.replay()
        self.shares.emit_share_unsubscribe_error(share_id, error)


class TestFoldersEmitSignals(DBusTwistedTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        yield super(TestFoldersEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.get_udf_dict = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.get_udf_dict')

    def test_emit_folder_deleted(self):
        """Emit the FolderCreated signal"""
        folder = 'folder'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.folders.FolderDeleted = self.signal_method
        self.signal_method(udf_dict)
        self.mocker.replay()
        self.folders.emit_folder_deleted(folder)

    def test_emit_folder_delete_error(self):
        """Emit the FolderCreateError signal"""
        folder = 'folder'
        error = 'error'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.folders.FolderDeleteError = self.signal_method
        self.signal_method(udf_dict, str(error))
        self.mocker.replay()
        self.folders.emit_folder_delete_error(folder, error)

    def test_emit_folder_subscribed(self):
        """Emit the FolderSubscribed signal"""
        folder = 'folder'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.folders.FolderSubscribed = self.signal_method
        self.signal_method(udf_dict)
        self.mocker.replay()
        self.folders.emit_folder_subscribed(folder)

    def test_emit_folder_subscribe_error(self):
        """Emit the FolderSubscribeError signal"""
        folder_id = 'id'
        error = 'error'
        self.folders.FolderSubscribeError = self.signal_method
        self.signal_method({'id':folder_id}, str(error))
        self.mocker.replay()
        self.folders.emit_folder_subscribe_error(folder_id, error)

    def test_emit_folder_unsubscribed(self):
        """Emit the FolderUnSubscribed signal"""
        folder = 'folder'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.folders.FolderUnSubscribed = self.signal_method
        self.signal_method(udf_dict)
        self.mocker.replay()
        self.folders.emit_folder_unsubscribed(folder)

    def test_emit_folder_unsubscribe_error(self):
        """Emit the FolderUnSubscribeError signal"""
        folder_id = 'id'
        error = 'error'
        self.folders.FolderUnSubscribeError= self.signal_method
        self.signal_method({'id':folder_id}, str(error))
        self.mocker.replay()
        self.folders.emit_folder_unsubscribe_error(folder_id, error)


class TestPublicFilesEmitSignals(DBusTwistedTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    @defer.inlineCallbacks
    def setUp(self):
        """Setup tests."""
        yield super(TestPublicFilesEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.public_files.syncdaemon_public_files = self.mocker.mock()
        self.bool_str = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.bool_str')

    def test_emit_public_access_changed(self):
        """Emit the PublicAccessChanged signal."""
        share_id = 'share_id'
        node_id = 'node_id'
        path = 'path'
        is_public = True
        public_url = 'url'
        self.public_files.syncdaemon_public_files.get_path(share_id, node_id)
        self.mocker.result(path)
        self.bool_str(is_public)
        self.mocker.result('True')
        self.public_files.PublicAccessChanged = self.signal_method
        self.signal_method(dict(share_id=share_id, node_id=node_id,
                                is_public='True', public_url=public_url,
                                path=path))
        self.mocker.replay()
        self.public_files.emit_public_access_changed(share_id, node_id,
                                                     is_public, public_url)

    def test_emit_public_access_change_error(self):
        """Emit the PublicAccessChangeError signal."""
        share_id = 'share_id'
        node_id = 'node_id'
        error = 'error'
        path = 'path'
        self.public_files.syncdaemon_public_files.get_path(share_id, node_id)
        self.mocker.result(path)
        self.public_files.PublicAccessChangeError = self.signal_method
        self.signal_method(dict(share_id=share_id, node_id=node_id, path=path),
                           error)
        self.mocker.replay()
        self.public_files.emit_public_access_change_error(share_id, node_id, error)

    def test_emit_public_files_list(self):
        """Emit the PublicFilesList signal."""
        volume_id = 'volume_id'
        node_id = 'node_id'
        public_url = 'url'
        path = 'path'
        public_files = [dict(volume_id=volume_id, node_id=node_id,
                             public_url=public_url)]
        files = [dict(volume_id=volume_id, node_id=node_id,
                      public_url=public_url, path=path)]
        self.public_files.syncdaemon_public_files.get_path(volume_id, node_id)
        self.mocker.result(path)
        self.public_files.PublicFilesList = self.signal_method
        self.signal_method(files)
        self.mocker.replay()
        self.public_files.emit_public_files_list(public_files)

    def test_emit_public_files_list_error(self):
        """Emit the PublicFilesListError signal."""
        error = 'error'
        self.public_files.PublicFilesListError = self.signal_method
        self.signal_method(error)
        self.mocker.replay()
        self.public_files.emit_public_files_list_error(error)
