# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Tests for main.Main class """

import logging
import os
import shutil

from twisted.internet import defer, reactor

from ubuntuone.syncdaemon import main as main_mod
from ubuntuone.clientdefs import VERSION

from contrib.testing.testcase import (
    BaseTwistedTestCase, FAKED_CREDENTIALS,
)
from tests.platform import setup_main_test, get_main_params

from ubuntuone.devtools.handlers import MementoHandler


def _get_main_common_params(testcase):
    """Return the parameters used by the all platforms."""
    return dict(root_dir=testcase.root,
                shares_dir=testcase.shares,
                data_dir=testcase.data,
                partials_dir=testcase.partials_dir,
                host='localhost', port=0,
                dns_srv=False, ssl=False,
                mark_interval=60,
                handshake_timeout=2,
                oauth_credentials=FAKED_CREDENTIALS)


class FakeListener(object):
    """Just an object that will listen something."""

    def handle_default(self, *a):
        """Something! :)"""


class MainTests(BaseTwistedTestCase):
    """ Basic tests to check main.Main """

    def setUp(self):
        """ Sets up a test. """
        BaseTwistedTestCase.setUp(self)
        self.root = self.mktemp('root')
        self.shares = self.mktemp('shares')
        self.data = self.mktemp('data')
        self.partials_dir = self.mktemp('partials_dir')
        # perform the extra setup steps according to the platform
        setup_main_test(self)

        self.handler = MementoHandler()
        self.handler.setLevel(logging.DEBUG)
        self._logger = logging.getLogger('ubuntuone.SyncDaemon')
        self._logger.addHandler(self.handler)

    def build_main(self, **kwargs):
        """
        Build and return a Main object, using reasonable defaults for
        the tests, plus whatever extra kwargs are passed in.
        """
        # get the params using the platform code to ensure they are correct
        params =  get_main_params(self, _get_main_common_params(self))
        params.update(kwargs)
        m = main_mod.Main(**params)
        m.local_rescan = lambda *_: m.event_q.push('SYS_LOCAL_RESCAN_DONE')
        return m

    def tearDown(self):
        """Clean up the tests."""
        self._logger.removeHandler(self.handler)
        shutil.rmtree(self.root)
        shutil.rmtree(self.shares)
        shutil.rmtree(self.data)
        BaseTwistedTestCase.tearDown(self)

    def test_main_initialization(self):
        """test that creating a Main instance works as expected."""
        main = self.build_main()
        main.shutdown()

    def test_main_start(self):
        """Test that Main.start works."""
        main = self.build_main()
        main.start()
        main.shutdown()

    def test_main_restarts_on_critical_error(self):
        """Test that Main restarts when syncdaemon gets into UNKNOWN_ERROR."""
        self.restarted = False
        main = self.build_main()
        main.restart = lambda: setattr(self, 'restarted', True)
        main.start()
        main.event_q.push('SYS_UNKNOWN_ERROR')
        self.assertTrue(self.restarted)
        main.shutdown()

    def test_shutdown_pushes_sys_quit(self):
        """When shutting down, the SYS_QUIT event is pushed."""
        main = self.build_main()
        events = []
        self.patch(main.event_q, 'push',
                   lambda *a, **kw: events.append((a, kw)))

        main.shutdown()
        expected = [(('SYS_USER_DISCONNECT',), {}), (('SYS_QUIT',), {})]
        self.assertEqual(expected, events)

    def test_handshake_timeout(self):
        """Check connecting times out."""
        d0 = defer.Deferred()

        class Handler:
            """Trivial event handler."""
            def handle_SYS_HANDSHAKE_TIMEOUT(self):
                """Pass the test when we get this event."""
                reactor.callLater(0, d0.callback, None)

        main = self.build_main(handshake_timeout=0)

        def fake_connect(*a):
            """Only connect when States told so."""
            main.event_q.push('SYS_CONNECTION_MADE')
            return defer.Deferred()
        main.action_q.connect = fake_connect

        # fake the following to not be executed
        main.get_root = lambda *_: defer.Deferred()
        main.action_q.check_version = lambda *_: defer.Deferred()

        main.event_q.subscribe(Handler())
        main.start()
        main.event_q.push('SYS_NET_CONNECTED')
        main.event_q.push('SYS_USER_CONNECT', access_token='')
        d0.addCallback(lambda _: main.shutdown())
        return d0

    def test_create_dirs_already_exists_dirs(self):
        """test that creating a Main instance works as expected."""
        link = os.path.join(self.root, 'Shared With Me')
        self.assertFalse(os.path.exists(link))
        self.assertTrue(os.path.exists(self.shares))
        self.assertTrue(os.path.exists(self.root))
        main = self.build_main()
        self.assertTrue(os.path.exists(main.shares_dir_link))
        # check that the shares link is actually a link
        self.assertTrue(os.path.islink(main.shares_dir_link))
        self.assertEquals(link, main.shares_dir_link)
        main.shutdown()

    def test_create_dirs_already_exists_symlink_too(self):
        """test that creating a Main instance works as expected."""
        link = os.path.join(self.root, 'Shared With Me')
        os.symlink(self.shares, link)
        self.assertTrue(os.path.exists(link))
        self.assertTrue(os.path.islink(link))
        self.assertTrue(os.path.exists(self.shares))
        self.assertTrue(os.path.exists(self.root))
        main = self.build_main()
        # check that the shares link is actually a link
        self.assertTrue(os.path.islink(main.shares_dir_link))
        main.shutdown()

    def test_create_dirs_already_exists_but_not_symlink(self):
        """test that creating a Main instance works as expected."""
        link = os.path.join(self.root, 'Shared With Me')
        os.makedirs(link)
        self.assertTrue(os.path.exists(link))
        self.assertFalse(os.path.islink(link))
        self.assertTrue(os.path.exists(self.shares))
        self.assertTrue(os.path.exists(self.root))
        main = self.build_main()
        # check that the shares link is actually a link
        self.assertEquals(main.shares_dir_link, link)
        self.assertFalse(os.path.islink(main.shares_dir_link))
        main.shutdown()

    def test_create_dirs_none_exists(self):
        """test that creating a Main instance works as expected."""
        link = os.path.join(self.root, 'Shared With Me')
        # remove the existing dirs
        os.rmdir(self.root)
        os.rmdir(self.shares)
        main = self.build_main()
        # check that the shares link is actually a link
        self.assertTrue(os.path.exists(link))
        self.assertTrue(os.path.islink(main.shares_dir_link))
        self.assertTrue(os.path.exists(self.shares))
        self.assertTrue(os.path.exists(self.root))
        main.shutdown()

    def test_connect_if_autoconnect_is_enabled(self):
        """If autoconnect option is enabled, connect the syncdaemon."""
        user_config = main_mod.config.get_user_config()
        orig = user_config.get_autoconnect()
        user_config.set_autoconnect(True)
        self.addCleanup(lambda: user_config.set_autoconnect(orig))

        self.connect_called = False
        self.patch(main_mod.ubuntuone.platform.ExternalInterface, 'connect',
                   lambda *a, **kw: setattr(self, 'connect_called', (a, kw)))

        main = self.build_main()
        self.addCleanup(lambda: main.shutdown())

        self.assertEqual(self.connect_called, ((main.external,),
                                               {'autoconnecting': True}))

    def test_dont_connect_if_autoconnect_is_disabled(self):
        """If autoconnect option is disabled, do not connect the syncdaemon."""
        user_config = main_mod.config.get_user_config()
        orig = user_config.get_autoconnect()
        user_config.set_autoconnect(False)
        self.addCleanup(lambda: user_config.set_autoconnect(orig))

        self.connect_called = False
        self.patch(main_mod.ubuntuone.platform.ExternalInterface, 'connect',
                   lambda *a, **kw: setattr(self, 'connect_called', True))

        main = self.build_main()
        self.addCleanup(lambda: main.shutdown())

        self.assertFalse(self.connect_called)

    def _get_listeners(self, main):
        """Return the subscribed objects."""
        s = set()
        for listener in main.event_q.listener_map.values():
            for x in listener:
                s.add(x)
        return s

    def test_event_logger_starts_if_available(self):
        """The event logger is started if available."""
        self.patch(main_mod.status_listener,
                   "get_listener", lambda *a: FakeListener())
        main = self.build_main()
        self.addCleanup(lambda: main.shutdown())
        self.assertIn(main.eventlog_listener, self._get_listeners(main))

    def test_event_logger_not_started_if_not_available(self):
        """The event logger is not started if it's not available."""
        self.patch(main_mod.event_logging, "get_listener", lambda *a: None)
        main = self.build_main()
        self.addCleanup(lambda: main.shutdown())
        self.assertNotIn(main.eventlog_listener, self._get_listeners(main))

    def test_status_listener_is_installed(self):
        """The status listener is installed if needed."""
        self.patch(main_mod.status_listener,
                   "get_listener", lambda *a: FakeListener())
        main = self.build_main()
        self.addCleanup(lambda: main.shutdown())
        self.assertIn(main.status_listener, self._get_listeners(main))

    def test_status_listener_not_installed_when_disabled(self):
        """The status listener is not started if it's not available."""
        self.patch(main_mod.status_listener, "get_listener", lambda *a: None)
        main = self.build_main()
        self.addCleanup(lambda: main.shutdown())
        self.assertNotIn(main.status_listener, self._get_listeners(main))

    def test_get_rootdir(self):
        """The get_rootdir returns the root dir."""
        expected = '~/Ubuntu Test One'
        main = self.build_main(root_dir=expected)
        self.addCleanup(lambda: main.shutdown())

        self.assertEqual(main.get_rootdir(), expected)

    def test_get_sharesdir(self):
        """The get_sharesdir returns the shares dir."""
        expected = '~/Share it to Me'
        main = self.build_main(shares_dir=expected)
        self.addCleanup(lambda: main.shutdown())

        self.assertEqual(main.get_sharesdir(), expected)

    def test_get_sharesdirlink(self):
        """The get_sharesdirlink returns the shares dir link."""
        expected = 'Share it to Me'
        main = self.build_main(shares_symlink_name=expected)
        self.addCleanup(lambda: main.shutdown())

        self.assertEqual(main.get_sharesdir_link(),
                         os.path.join(main.get_rootdir(), expected))

    def test_version_is_logged(self):
        """Test that the client version is logged."""
        main = self.build_main()
        self.addCleanup(lambda: main.shutdown())
        self.assertTrue(self.handler.check_info("client version", VERSION))
