#
# Author: Facundo Batista <facundo@canonical.com>
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Event Queue."""

import logging
import os
import unittest

from twisted.internet import defer

from ubuntuone.syncdaemon import (
    event_queue,
    filesystem_manager,
)
from contrib.testing import testcase
from ubuntuone.devtools.handlers import MementoHandler
from ubuntuone.syncdaemon import volume_manager
from ubuntuone.syncdaemon.tritcask import Tritcask


class BaseFSMonitorTestCase(testcase.BaseTwistedTestCase):
    """Test the structures where we have the path/watch."""

    timeout = 3

    def setUp(self):
        """Set up."""
        testcase.BaseTwistedTestCase.setUp(self)
        fsmdir = self.mktemp('fsmdir')
        partials_dir = self.mktemp('partials_dir')
        self.root_dir = self.mktemp('root_dir')
        self.home_dir = self.mktemp('home_dir')
        self.vm = testcase.FakeVolumeManager(self.root_dir)
        self.tritcask_dir = self.mktemp("tritcask_dir")
        self.db = Tritcask(self.tritcask_dir)
        self.fs = filesystem_manager.FileSystemManager(fsmdir, partials_dir,
                                                       self.vm, self.db)
        self.fs.create(path=self.root_dir, share_id='', is_dir=True)
        self.fs.set_by_path(path=self.root_dir,
                            local_hash=None, server_hash=None)
        eq = event_queue.EventQueue(self.fs)

        self.deferred = deferred = defer.Deferred()

        class HitMe(object):
            # class-closure, cannot use self, pylint: disable-msg=E0213
            def handle_default(innerself, event, **args):
                deferred.callback(True)

        eq.subscribe(HitMe())
        self.monitor = eq.monitor
        self.log_handler = MementoHandler()
        self.log_handler.setLevel(logging.DEBUG)
        self.monitor.log.addHandler(self.log_handler)

    def tearDown(self):
        """Clean up the tests."""
        self.monitor.shutdown()
        self.rmtree(self.tmpdir)
        self.monitor.log.removeHandler(self.log_handler)
        testcase.BaseTwistedTestCase.tearDown(self)


class WatchManagerTests(BaseFSMonitorTestCase):
    """Test the structures where we have the path/watch."""

    def test_watch_updated_when_deleting_dir(self):
        """Internal data structures are fixed when deleting the dir."""
        path = os.path.join(self.root_dir, "path")
        os.mkdir(path)
        self.monitor.add_watch(self.root_dir)
        self.monitor.add_watch(path)

        # we have the watch, remove the dir, the watch should be gone
        self.assertTrue(self.monitor.has_watch(path))
        os.rmdir(path)

        def check(_):
            """Check state after the event."""
            self.assertFalse(self.monitor.has_watch(path))

        self.deferred.addCallback(check)
        return self.deferred

    def test_watch_updated_when_renaming_dir(self):
        """Internal data structures are fixed when renaming the dir."""
        path1 = os.path.join(self.root_dir, "path1")
        path2 = os.path.join(self.root_dir, "path2")
        os.mkdir(path1)
        self.monitor.add_watch(self.root_dir)
        self.monitor.add_watch(path1)

        # we have the watch, rename the dir, new name should have the watch,
        # old name should not
        self.assertTrue(self.monitor.has_watch(path1))
        os.rename(path1, path2)

        def check(_):
            """Check state after the event."""
            self.assertTrue(self.monitor.has_watch(path2))
            self.assertFalse(self.monitor.has_watch(path1))

        self.deferred.addCallback(check)
        return self.deferred

    def test_watch_updated_when_movingout_dir(self):
        """Internal data structures are fixed when moving out the dir."""
        notu1 = os.path.join(self.root_dir, "notu1")
        path1 = os.path.join(self.root_dir, "path1")
        path2 = os.path.join(notu1, "path2")
        os.mkdir(notu1)
        os.mkdir(path1)
        self.monitor.add_watch(self.root_dir)
        self.monitor.add_watch(path1)

        # we have the watch, move it outside watched structure, no more watches
        self.assertTrue(self.monitor.has_watch(path1))
        os.rename(path1, path2)

        def check(_):
            """Check state after the event."""
            self.assertFalse(self.monitor.has_watch(path1))
            self.assertFalse(self.monitor.has_watch(path2))

        self.deferred.addCallback(check)
        return self.deferred

    def test_fix_path_not_there(self):
        """Try to fix path but it's not there."""
        self.monitor._general_watchs = {}
        self.monitor._ancestors_watchs = {}
        self.monitor.inotify_watch_fix("not-there", "new-one")
        self.assertTrue(self.log_handler.check_warning("Tried to fix",
                                                       "not-there"))

    def test_fix_path_general(self):
        """Fixing path in general watches."""
        self.monitor._general_watchs = {'/path1/foo': 1, '/other': 2}
        self.monitor._ancestors_watchs = {'/foo': 3}
        self.monitor.inotify_watch_fix('/path1/foo', '/path1/new')
        self.assertEqual(self.monitor._general_watchs, {'/path1/new': 1, '/other': 2})
        self.assertEqual(self.monitor._ancestors_watchs, {'/foo': 3})

    def test_fix_path_ancestors(self):
        """Fixing path in ancestors watches."""
        self.monitor._general_watchs = {'/bar': 3}
        self.monitor._ancestors_watchs = {'/oth': 1, '/other': 2}
        self.monitor.inotify_watch_fix('/oth', '/baz')
        self.assertEqual(self.monitor._general_watchs, {'/bar': 3})
        self.assertEqual(self.monitor._ancestors_watchs, {'/baz': 1, '/other': 2})


class DynamicHitMe(object):
    """Helper class to test a sequence of signals."""

    def __init__(self, should_events, test_machinery):
        self.should_events = []
        for i, info in enumerate(should_events):
            self.should_events.append((i,) + info)
        if self.should_events:
            self.final_step = self.should_events[-1][0]
            self.should_events.reverse()
        self.test_machinery = test_machinery

    def __getattr__(self, name):
        """typical method faker"""
        if not name.startswith("handle_"):
            return

        asked_event = name[7:]

        # to what we should match
        test_info = self.should_events.pop()
        step = test_info[0]
        should_evtname = test_info[1]
        should_args = test_info[2:]

        def to_check(*asked_args):
            """the function that actually be called"""
            if asked_args != should_args:
                self.test_machinery.finished_error(
                    "In step %d received wrong args (%r)" % (step, asked_args))
            else:
                if step == self.final_step:
                    self.test_machinery.finished_ok()

        if should_evtname != asked_event:
            msg = "Event %r asked in bad order (%d)" % (asked_event, step)
            self.test_machinery.finished_error(msg)
        else:
            return to_check


class BaseTwisted(BaseFSMonitorTestCase):
    """Base class for twisted tests."""

    # this timeout must be bigger than the one used in event_queue
    timeout = 2

    def setUp(self):
        """Setup the test."""
        BaseFSMonitorTestCase.setUp(self)

        # create the deferred for the tests
        self._deferred = defer.Deferred()

    def finished_ok(self):
        """Called to indicate that the tests finished ok."""
        self._deferred.callback(True)

    def finished_error(self, msg):
        """Called to indicate that the tests finished badly."""
        self._deferred.errback(Exception(msg))

    def failUnlessEqual(self, first, second, msg=''):
        """Fail the test if C{first} and C{second} are not equal.

        @param msg: A string describing the failure that's included in the
            exception.

        """
        if not first == second:
            if msg is None:
                msg = ''
            if len(msg) > 0:
                msg += '\n'
            exception = self.failureException(
                '%snot equal:\na = %s\nb = %s\n'
                % (msg, repr(first), repr(second)))
            self.finished_error(exception)
            raise exception
        return first
    assertEqual = assertEquals = failUnlessEquals = failUnlessEqual


class WatchTests(BaseFSMonitorTestCase):
    """Test the EQ API to add and remove watchs."""

    @defer.inlineCallbacks
    def _create_udf(self, path):
        """Create an UDF and returns it and the volume"""
        os.makedirs(path)
        udf = volume_manager.UDF("vol_id", "node_id", path.decode('utf-8'),
                                 path, True)
        yield self.vm.add_udf(udf)

    def test_add_general_watch(self):
        """Test that general watchs can be added."""
        # we should have what we asked for
        self.monitor.add_watch(self.root_dir)

        # check only added dir in watchs
        # pylint: disable-msg=W0212
        self.assertTrue(self.root_dir in self.monitor._general_watchs)
        self.assertTrue("not-added-dir" not in self.monitor._general_watchs)

        # nothing in the udf ancestors watch
        self.assertEqual(self.monitor._ancestors_watchs, {})

    @defer.inlineCallbacks
    def test_add_watch_on_udf_ancestor(self):
        """Test that ancestors watchs can be added."""
        # create the udf and add the watch
        path_udf = os.path.join(self.home_dir, "path/to/UDF")
        yield self._create_udf(path_udf)
        path_ancestor = os.path.join(self.home_dir, "path")
        self.monitor.add_watch(path_ancestor)

        # check only added dir in watchs
        # pylint: disable-msg=W0212
        self.assertTrue(path_ancestor in self.monitor._ancestors_watchs)
        self.assertTrue("not-added-dir" not in self.monitor._ancestors_watchs)

        # nothing in the general watch
        self.assertEqual(self.monitor._general_watchs, {})

    @defer.inlineCallbacks
    def test_add_watch_on_udf_exact(self):
        """Test adding the watch exactly on UDF."""
        # create the udf and add the watch
        path_udf = os.path.join(self.home_dir, "path/to/UDF")
        yield self._create_udf(path_udf)
        self.monitor.add_watch(path_udf)

        # pylint: disable-msg=W0212
        self.assertTrue(path_udf in self.monitor._general_watchs)
        self.assertEqual(self.monitor._ancestors_watchs, {})

    @defer.inlineCallbacks
    def test_add_watch_on_udf_child(self):
        """Test adding the watch inside UDF."""
        # create the udf and add the watch
        path_udf = os.path.join(self.home_dir, "path/to/UDF")
        yield self._create_udf(path_udf)
        path_ancestor = os.path.join(self.home_dir, "path/to/UDF/inside")
        os.mkdir(path_ancestor)
        self.monitor.add_watch(path_ancestor)

        # pylint: disable-msg=W0212
        self.assertTrue(path_ancestor in self.monitor._general_watchs)
        self.assertEqual(self.monitor._ancestors_watchs, {})

    def test_rm_watch_not_dir_anymore(self):
        """Test that a watch can be removed even not having the dir anymore.

        This is the case where the directory is deleted from the filesystem,
        the watch is automatically removed in pyinotify but we need to take
        care of it from our own data structures.
        """
        not_existing_dir = "not-added-dir"
        self.monitor.add_watch(not_existing_dir)
        self.assertTrue(self.monitor.has_watch(not_existing_dir))
        self.monitor.rm_watch(not_existing_dir)
        self.assertFalse(self.monitor.has_watch(not_existing_dir))

    @defer.inlineCallbacks
    def test_rm_watch_wrong(self):
        """Test that general watchs can be removed."""
        # add two types of watchs
        self.monitor.add_watch(self.root_dir)
        path_udf = os.path.join(self.home_dir, "path/to/UDF")
        yield self._create_udf(path_udf)
        path_ancestor = os.path.join(self.home_dir, "path")
        self.monitor.add_watch(path_ancestor)

        # remove different stuff
        self.monitor.rm_watch("not-added-dir")
        self.assertTrue(self.log_handler.check_warning('remove', 'watch',
                                                       'not-added-dir'))

    def test_rm_watch_general(self):
        """Test that general watchs can be removed."""
        # remove what we added
        self.monitor.add_watch(self.root_dir)
        self.monitor.rm_watch(self.root_dir)

        # pylint: disable-msg=W0212
        self.assertEqual(self.monitor._general_watchs, {})
        self.assertEqual(self.monitor._ancestors_watchs, {})

    @defer.inlineCallbacks
    def test_rm_watch_ancestor(self):
        """Test that ancestor watchs can be removed."""
        # create the udf and add the watch
        path_udf = os.path.join(self.home_dir, "path/to/UDF")
        yield self._create_udf(path_udf)
        path_ancestor = os.path.join(self.home_dir, "path")
        self.monitor.add_watch(path_ancestor)

        # remove what we added
        self.monitor.rm_watch(path_ancestor)
        # pylint: disable-msg=W0212
        self.assertEqual(self.monitor._general_watchs, {})
        self.assertEqual(self.monitor._ancestors_watchs, {})

    def test_has_watch_general(self):
        """Test that a general path is watched."""
        self.assertFalse(self.monitor.has_watch(self.root_dir))

        # add
        self.monitor.add_watch(self.root_dir)
        self.assertTrue(self.monitor.has_watch(self.root_dir))

        # remove
        self.monitor.rm_watch(self.root_dir)
        self.assertFalse(self.monitor.has_watch(self.root_dir))

    @defer.inlineCallbacks
    def test_has_watch_ancestor(self):
        """Test that an ancestor path is watched."""
        path_udf = os.path.join(self.home_dir, "path/to/UDF")
        yield self._create_udf(path_udf)
        path_ancestor = os.path.join(self.home_dir, "path")

        self.assertFalse(self.monitor.has_watch(path_ancestor))

        # add
        # create the udf and add the watch
        self.monitor.add_watch(path_ancestor)
        self.assertTrue(self.monitor.has_watch(path_ancestor))

        # remove
        self.monitor.rm_watch(path_ancestor)
        self.assertFalse(self.monitor.has_watch(path_ancestor))


def test_suite():
    """Collect the tests."""
    import sys
    if sys.platform == 'linux2':
        tests = unittest.TestLoader().loadTestsFromName(__name__)
    else:
        tests = []
    return tests

if __name__ == "__main__":
    unittest.main()
