#
# Author: Facundo Batista <facundo@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests the Hashs Queue."""

from __future__ import with_statement

import os
import unittest
import random
import time
import logging
import threading

from StringIO import StringIO
from twisted.trial.unittest import TestCase as TwistedTestCase
from twisted.internet import defer, reactor

from contrib.testing import testcase
from ubuntuone.syncdaemon import hash_queue
from ubuntuone.storageprotocol.hash import \
            content_hash_factory, crc32


# this is a unit test we need to access protected members
# pylint: disable-msg=W0212
class HasherTests(testcase.BaseTwistedTestCase):
    """Test the whole stuff to receive signals."""

    def setUp(self):
        """Setup the test."""
        testcase.BaseTwistedTestCase.setUp(self)
        self.test_dir = self.mktemp('test_dir')
        self.timeout = 2

    def tearDown(self):
        """Clean up the tests."""
        testcase.BaseTwistedTestCase.tearDown(self)

    def test_live_process(self):
        """Check that the hasher lives and dies."""
        # create the hasher
        mark = object()
        queue = hash_queue.UniqueQueue()
        class Helper(object):
            """helper class"""
            def push(self, *args):
                """callback"""
        receiver = Helper()
        hasher = hash_queue._Hasher(queue, mark, receiver)
        hasher.start()

        # it's aliveeeeeeee!
        self.assertTrue(hasher.isAlive())

        # stop it, and release the processor to let the other thread run
        hasher.stop()
        time.sleep(.1)

        # "I see dead threads"
        self.assertFalse(hasher.isAlive())
        hasher.join(timeout=5)

    def test_called_back_ok(self):
        """Tests that the hasher produces correct info."""
        # create the hasher
        mark = object()
        queue = hash_queue.UniqueQueue()
        d = defer.Deferred()
        class Helper(object):
            """helper class"""
            def push(self, *args):
                """callback"""
                d.callback(args)
        receiver = Helper()
        hasher = hash_queue._Hasher(queue, mark, receiver)
        hasher.start()

        # send what to hash
        testfile = os.path.join(self.test_dir, "testfile")
        with open(testfile, "w") as fh:
            fh.write("foobar")
        queue.put((testfile, "mdid"))

        def check_info(args):
            """check the info pushed by the hasher"""
            # pylint: disable-msg=W0612
            hasher.stop()
            hasher.join(timeout=5)
            event, path, hash, crc, size, stat = args
            self.assertEqual(event, "HQ_HASH_NEW")
            # calculate what we should receive
            realh = content_hash_factory()
            realh.hash_object.update("foobar")
            should_be = realh.content_hash()
            curr_stat = os.stat(testfile)
            self.assertEquals(should_be, hash)
            for attr in ('st_mode', 'st_ino', 'st_dev', 'st_nlink', 'st_uid',
                         'st_gid', 'st_size', 'st_ctime', 'st_mtime'):
                self.assertEquals(getattr(curr_stat, attr),
                                  getattr(stat, attr))


        d.addCallback(check_info)
        # release the processor and check
        return d

    def test_called_back_error(self):
        """Tests that the hasher signals error when no file."""
        # create the hasher
        mark = object()
        queue = hash_queue.UniqueQueue()
        d = defer.Deferred()
        class Helper(object):
            """helper class"""
            def push(self, *args):
                """callback"""
                d.callback(args)
        receiver = Helper()
        hasher = hash_queue._Hasher(queue, mark, receiver)
        hasher.start()

        # send what to hash
        queue.put(("not_to_be_found", "foo"))

        def check_info(args):
            """check the info pushed by the hasher"""
            hasher.stop()
            hasher.join(timeout=5)
            event, mdid = args
            self.assertEqual(event, "HQ_HASH_ERROR")
            self.assertEqual(mdid, "foo")


        d.addCallback(check_info)
        # release the processor and check
        return d

    def test_order(self):
        """The hasher should return in order."""
        # calculate what we should receive
        should_be = []
        for i in range(10):
            hasher = content_hash_factory()
            text = "supercalifragilistico"+str(i)
            hasher.hash_object.update(text)
            tfile = os.path.join(self.test_dir, "tfile"+str(i))
            with open(tfile, "w") as fh:
                fh.write("supercalifragilistico"+str(i))
            should_be.append(("HQ_HASH_NEW", tfile, hasher.content_hash(),
                              crc32(text), len(text), os.stat(tfile)))

        # create the hasher
        mark = object()
        queue = hash_queue.UniqueQueue()
        d = defer.Deferred()

        class Helper(object):
            """helper class"""
            # class-closure, cannot use self, pylint: disable-msg=E0213
            def __init__(innerself):
                innerself.store = []
            def push(innerself, *args):
                """callback"""
                innerself.store.append(args)
                if len(innerself.store) == 10:
                    hasher.stop()
                    hasher.join(timeout=5)
                    if innerself.store == should_be:
                        d.callback(True)
                    else:
                        d.errback(Exception("are different!"))
        receiver = Helper()

        hasher = hash_queue._Hasher(queue, mark, receiver)
        hasher.start()

        # send what to hash
        for i in range(10):
            tfile = os.path.join(self.test_dir, "tfile"+str(i))
            queue.put((tfile, "mdid"))
        return d

    def test_large_content(self):
        """The hasher works ok for a lot of info."""
        # calculate what we should receive
        testinfo = "".join(chr(random.randint(0, 255)) for i in range(100000))
        hasher = content_hash_factory()
        hasher.hash_object.update(testinfo)
        testfile = os.path.join(self.test_dir, "testfile")
        testhash = hasher.content_hash()

        # create the hasher
        mark = object()
        queue = hash_queue.UniqueQueue()

        d = defer.Deferred()

        class Helper(object):
            """helper class"""
            def push(self, event, path, newhash, crc, size, stat):
                """callback"""
                hasher.stop()
                hasher.join(timeout=5)
                if event != "HQ_HASH_NEW":
                    d.errback(Exception("envent is not HQ_HASH_NEW"))
                elif path != testfile:
                    d.errback(Exception("path is not the original one"))
                elif newhash != testhash:
                    d.errback(Exception("the hashes are different!"))
                else:
                    d.callback(True)
        receiver = Helper()
        hasher = hash_queue._Hasher(queue, mark, receiver)
        hasher.start()
        # send what to hash
        with open(testfile, "w") as fh:
            fh.write(testinfo)
        queue.put((testfile, "mdid"))
        return d


class HashQueueTests(testcase.BaseTwistedTestCase):
    """Test the whole stuff to receive signals."""

    def setUp(self):
        """Setup the test."""
        testcase.BaseTwistedTestCase.setUp(self)
        self.test_dir = self.mktemp('test_dir')
        self.timeout = 2
        self.log = logging.getLogger("ubuntuone.SyncDaemon.TEST")
        self.log.info("starting test %s.%s", self.__class__.__name__,
                      self._testMethodName)

    def tearDown(self):
        """Clean up the tests."""
        self.log.info("finished test %s.%s", self.__class__.__name__,
                      self._testMethodName)
        testcase.BaseTwistedTestCase.tearDown(self)

    def test_called_back_ok(self):
        """Tests that the hasher produces correct info."""
        # create the hasher
        d = defer.Deferred()
        class Helper(object):
            """helper class"""
            def push(self, *args):
                """callback"""
                d.callback(args)
        receiver = Helper()
        hq = hash_queue.HashQueue(receiver)
        self.addCleanup(hq.shutdown)

        # send what to hash
        testfile = os.path.join(self.test_dir, "testfile")
        with open(testfile, "w") as fh:
            fh.write("foobar")
        hq.insert(testfile, "mdid")

        def check_info(args):
            """check the info pushed by the hasher"""
            # pylint: disable-msg=W0612
            event, path, hash, crc, size, stat = args
            self.assertEqual(event, "HQ_HASH_NEW")
            # calculate what we should receive
            realh = content_hash_factory()
            realh.hash_object.update("foobar")
            should_be = realh.content_hash()
            curr_stat = os.stat(testfile)
            self.assertEquals(should_be, hash)
            for attr in ('st_mode', 'st_ino', 'st_dev', 'st_nlink', 'st_uid',
                         'st_gid', 'st_size', 'st_ctime', 'st_mtime'):
                self.assertEquals(getattr(curr_stat, attr),
                                  getattr(stat, attr))

        d.addCallback(check_info)
        return d

    def test_called_back_error(self):
        """Tests that the hasher generates an error when no file."""
        # create the hasher
        d = defer.Deferred()
        class Helper(object):
            """helper class"""
            def push(self, *args):
                """callback"""
                d.callback(args)
        receiver = Helper()
        hq = hash_queue.HashQueue(receiver)
        self.addCleanup(hq.shutdown)

        # send what to hash
        hq.insert("not_to_be_found", "foo")

        def check_info(args):
            """check the info pushed by the hasher"""
            event, mdid = args
            self.assertEqual(event, "HQ_HASH_ERROR")
            self.assertEqual(mdid, "foo")

        d.addCallback(check_info)
        return d

    def test_being_hashed(self):
        """Tell if something is being hashed."""
        tfile1 = os.path.join(self.test_dir, "tfile1")
        open(tfile1, "w").close()
        tfile2 = os.path.join(self.test_dir, "tfile2")
        open(tfile2, "w").close()

        class C(object):
            """bogus"""
            def push(self, *a):
                """none"""
        hq = hash_queue.HashQueue(C())
        self.addCleanup(hq.shutdown)

        event = threading.Event()
        original_hash = hash_queue._Hasher._hash

        def f(*a):
            event.wait()
            return "foo"
        hash_queue._Hasher._hash = f

        try:
            # nothing yet
            self.assertFalse(hq.is_hashing(tfile1, "mdid"))

            # push something, test for it and for other stuff
            hq.insert(tfile1, "mdid")
            self.assertTrue(hq.is_hashing(tfile1, "mdid"))
            self.assertFalse(hq.is_hashing(tfile2, "mdid"))

            # push tfile2, that gets queued, and check again
            hq.insert(tfile2, "mdid")
            self.assertTrue(hq.is_hashing(tfile2, "mdid"))
        finally:
            event.set()
            hash_queue._Hasher._hash = original_hash


    def test_order(self):
        """The hasher should return in order."""
        # calculate what we should receive
        should_be = []
        for i in range(10):
            hasher = content_hash_factory()
            text = "supercalifragilistico"+str(i)
            hasher.hash_object.update(text)
            tfile = os.path.join(self.test_dir, "tfile"+str(i))
            with open(tfile, "w") as fh:
                fh.write("supercalifragilistico"+str(i))
            should_be.append(("HQ_HASH_NEW", tfile, hasher.content_hash(),
                              crc32(text), len(text), os.stat(tfile)))

        d = defer.Deferred()
        class Helper(object):
            """helper class"""
            # class-closure, cannot use self, pylint: disable-msg=E0213
            def __init__(innerself):
                innerself.store = []
            def push(innerself, *args):
                """callback"""
                innerself.store.append(args)
                if len(innerself.store) == 10:
                    if innerself.store[:-1] == should_be[:-1]:
                        d.callback(True)
                    else:
                        d.errback(Exception("are different! "))
        receiver = Helper()

        hq = hash_queue.HashQueue(receiver)
        self.addCleanup(hq.shutdown)

        # send what to hash
        for i in range(10):
            tfile = os.path.join(self.test_dir, "tfile"+str(i))
            hq.insert(tfile, "mdid")
        return d

    def test_unique(self):
        """The hasher should return in order."""
        # calculate what we should receive
        should_be = []
        for i in range(10):
            hasher = content_hash_factory()
            text = "supercalifragilistico"+str(i)
            hasher.hash_object.update(text)
            tfile = os.path.join(self.test_dir, "tfile"+str(i))
            with open(tfile, "w") as fh:
                fh.write("supercalifragilistico"+str(i))
            should_be.append(("HQ_HASH_NEW", tfile, hasher.content_hash(),
                              crc32(text), len(text), os.stat(tfile)))

        d = defer.Deferred()
        class Helper(object):
            """helper class"""
            # class-closure, cannot use self, pylint: disable-msg=E0213
            def __init__(innerself):
                innerself.store = []
            def push(innerself, *args):
                """callback"""
                innerself.store.append(args)
                if len(innerself.store) == 10:
                    if len(innerself.store[:-1]) == len(should_be[:-1]):
                        stored = set(innerself.store[:-1])
                        expected = set(should_be[:-1])
                        if len(stored.symmetric_difference(expected)) == 0:
                            d.callback(True)
                    else:
                        d.errback(Exception("are different!"))

        receiver = Helper()

        hq = hash_queue.HashQueue(receiver)
        self.addCleanup(hq.shutdown)
        # stop the hasher so we can test the unique items in the queue
        hq.hasher.stop()
        self.log.debug('Hasher stopped (forced)')
        # allow the hasher to fully stop
        time.sleep(0.1)
        # create a new hasher just like the HashQueue creates it
        hq.hasher = hash_queue._Hasher(hq._queue, hq._end_mark, receiver)
        hq.hasher.setDaemon(True)

        # send to hash twice
        for i in range(10):
            tfile = os.path.join(self.test_dir, "tfile"+str(i))
            hq.insert(tfile, "mdid")
            hq.insert(tfile, "mdid")
        # start the hasher
        self.log.debug('Hasher started (forced)')
        hq.hasher.start()
        # insert the last item to check the unqieness in the queue while
        # the hasher is running
        for i in range(9, 10):
            tfile = os.path.join(self.test_dir, "tfile"+str(i))
            hq.insert(tfile, "mdid")
        return d

    def test_interrupt_current(self):
        """ Test that the hasher correctly interrupts a inprogress task."""
        # calculate what we should receive
        testinfo = os.urandom(1000)
        hasher = content_hash_factory()
        hasher.hash_object.update(testinfo)
        testfile = os.path.join(self.test_dir, "testfile")
        testhash = hasher.content_hash()
        # send what to hash
        with open(testfile, "w") as fh:
            fh.write(testinfo)

        d = defer.Deferred()

        class Helper(object):
            """helper class"""
            def push(self, event, path, newhash, crc, size, stat):
                """callback"""
                if event != "HQ_HASH_NEW":
                    d.errback(Exception("envent is not HQ_HASH_NEW"))
                elif path != testfile:
                    d.errback(Exception("path is not the original one"))
                elif newhash != testhash:
                    d.errback(Exception("the hashes are different!"))
                else:
                    d.callback(True)

        class FakeFile(StringIO):
            """An endless file"""
            def read(self, size=10):
                """return random bytes"""
                return os.urandom(size)
            # context manager API
            def __enter__(self):
                return self
            def __exit__(self, *args):
                pass

        # patch __builtin__.open so we don't have to try to interrupt a hashing
        bi = __import__('__builtin__')
        old_open = bi.open

        class OpenFaker(object):
            """A class to fake open for specific paths"""
            def __init__(self, paths):
                self.paths = paths
                self.done = False
            def __call__(self, path, mode='r'):
                """the custom open implementation"""
                if self.done or path not in self.paths:
                    return old_open(path, mode)
                else:
                    self.done = True
                    return FakeFile()
        open_faker = OpenFaker([testfile])
        bi.open = open_faker

        receiver = Helper()
        hq = hash_queue.HashQueue(receiver)
        def cleanup():
            """cleanup the mess"""
            bi.open = old_open
            hq.shutdown()
        self.addCleanup(cleanup)
        hq.insert(testfile, "mdid")

        # insert it again, to cancel the first one
        reactor.callLater(0.1, hq.insert, testfile, "mdid")
        return d

    def test_shutdown(self):
        """Test that the HashQueue shutdown """
        class Helper(object):
            """helper class"""
            def push(self, *args):
                """callback"""
        receiver = Helper()
        hq = hash_queue.HashQueue(receiver)
        hq.shutdown()
        self.assertTrue(hq._stopped)

    def test_shutdown_while_hashing(self):
        """Test that the HashQueue is shutdown ASAP while it's hashing."""
        # create large data in order to test
        testinfo = os.urandom(500000)
        hasher = content_hash_factory()
        hasher.hash_object.update(testinfo)
        testfile = os.path.join(self.test_dir, "testfile")
        # send what to hash
        with open(testfile, "w") as fh:
            fh.write(testinfo)

        class Helper(object):
            """helper class"""
            def push(self, *args):
                """callback"""
        receiver = Helper()
        hq = hash_queue.HashQueue(receiver)
        self.addCleanup(hq.shutdown)
        # read in small chunks, so we have more iterations
        hq.hasher.chunk_size = 2**10
        hq.insert(testfile, "mdid")
        time.sleep(0.1)
        hq.shutdown()
        # block until the hash is stopped and the queue is empty
        # a shutdown clears the queue
        hq._queue.join()
        self.assertFalse(hq.hasher.hashing)
        self.assertTrue(hq.hasher._stopped)
        #self.assertFalse(hq.hasher.isAlive())
        self.assertTrue(hq._queue.empty())

    def test_insert_post_shutdown(self):
        """test inserting a path after the shutdown"""
        class Helper(object):
            """helper class"""
            def push(self, *args):
                """callback"""
        receiver = Helper()
        hq = hash_queue.HashQueue(receiver)
        hq.shutdown()
        hq.insert('foo', 'mdid')
        self.assertFalse(hq.is_hashing('foo', 'mdid'))


class UniqueQueueTests(TwistedTestCase):
    """Tests for hash_queue.UniqueQueue"""

    def test_unique_elements(self):
        """Test that the queue actually holds unique elements."""
        queue = hash_queue.UniqueQueue()
        queue.put(('item1', "mdid"))
        queue.put(('item1', "mdid"))
        self.assertEquals(1, queue.qsize())
        self.assertEquals(1, len(queue._set))
        queue.get()
        self.assertEquals(0, queue.qsize())
        self.assertEquals(0, len(queue._set))
        queue.put(('item1', "mdid"))
        queue.put(('item2', "mdid"))
        queue.put(('item1', "mdid"))
        queue.put(('item2', "mdid"))
        self.assertEquals(2, queue.qsize())
        self.assertEquals(2, len(queue._set))
        queue.get()
        queue.get()
        self.assertEquals(0, queue.qsize())
        self.assertEquals(0, len(queue._set))

    def test_contains(self):
        """test contains functionality"""
        queue = hash_queue.UniqueQueue()

        # nothing in it
        self.assertFalse(("item1", "mdid") in queue)

        # put one and check
        queue.put(('item1', "mdid"))
        self.assertTrue(("item1", "mdid") in queue)
        self.assertFalse(("item2", "mdid") in queue)

        # put second and check
        queue.put(('item2', "mdid"))
        self.assertTrue(("item1", "mdid") in queue)
        self.assertTrue(("item2", "mdid") in queue)

    def test_clear(self):
        """test clear method"""
        queue = hash_queue.UniqueQueue()
        queue.put(('item1', "mdid"))
        queue.put(('item2', "mdid"))
        self.assertEquals(2, queue.qsize())
        # check that queue.clear actually clear the queue
        queue.clear()
        self.assertEquals(0, queue.qsize())
        queue.put(('item3', "mdid"))
        queue.put(('item4', "mdid"))
        queue.get()
        self.assertEquals(2, queue.unfinished_tasks)
        self.assertEquals(1, queue.qsize())
        # check that queue.clear also cleanup unfinished_tasks
        queue.clear()
        self.assertEquals(0, queue.unfinished_tasks)
        self.assertEquals(0, queue.qsize())

    def test_clear_unfinished_tasks(self):
        """test the clear wakeup waiting threads."""
        queue = hash_queue.UniqueQueue()
        d = defer.Deferred()
        # helper function, pylint: disable-msg=C0111
        def consumer(queue, d):
            # wait util unfinished_tasks == 0
            queue.join()
            reactor.callFromThread(d.callback, True)

        # helper function, pylint: disable-msg=C0111
        def check(result):
            self.assertTrue(result)

        d.addCallback(check)
        t = threading.Thread(target=consumer, args=(queue, d))
        t.setDaemon(True)
        queue.put(('item1', "mdid"))
        t.start()
        reactor.callLater(0.1, queue.clear)
        return d

def test_suite():
    # pylint: disable-msg=C0111
    return unittest.TestLoader().loadTestsFromName(__name__)

if __name__ == "__main__":
    unittest.main()
