/*
 * ubuntuone-nautilus.c - Nautilus extensions for Ubuntu One
 *
 * Authors: Tim Cole <tim.cole@canonical.com>
 *          Rodney Dawes <rodney.dawes@canonical.com>
 *          Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n-lib.h>

#include <dbus/dbus-glib.h>

#include <libnautilus-extension/nautilus-extension-types.h>
#include <libnautilus-extension/nautilus-file-info.h>
#include <libnautilus-extension/nautilus-info-provider.h>
#include <libnautilus-extension/nautilus-menu-provider.h>
#include <libnautilus-extension/nautilus-location-widget-provider.h>
#include "u1-contacts-picker.h"

/* We need to do this explicitly because older versions of nautilus
 * don't seem to do it for us
 */
#include <gtk/gtk.h>

#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

/* The header is generated from ubuntuone-marshallers.list */
#include "ubuntuone-marshallers.h"

#define UBUNTUONE_TYPE_NAUTILUS  (ubuntuone_nautilus_get_type ())
#define UBUNTUONE_NAUTILUS(o)    (G_TYPE_CHECK_INSTANCE_CAST ((o), UBUNTUONE_TYPE_NAUTILUS, UbuntuOneNautilus))

/* This is used when a change is pending */
#define UPDATE_PENDING "pending"

typedef struct _CBData CBData;

typedef struct {
  GObject parent_slot;

  DBusGConnection * bus;
  DBusGProxy * u1_proxy;
  DBusGProxy * u1_status;
  DBusGProxy * u1_shares;
  DBusGProxy * u1_fs;
  DBusGProxy * u1_folders;
  DBusGProxy * u1_public;

  /* Are we connected? */
  gboolean connected;

  /* The managed directory root */
  gchar * managed;

  /* Avoid calling get_rootdir and get_folders lots of times */
  gboolean gotroot;
  gboolean gotudfs;
  gboolean gotpubs;

  /* Lists of ul/dl/shares for setting emblems */
  GHashTable * uploads;
  GHashTable * downloads;
  GHashTable * shares;

  /* Lists of public files and user defined folders */
  GHashTable * public;
  GHashTable * udfs;
  
  /* Lists of sync/unsync'd files */
  GHashTable * updated;
  GHashTable * needsupdating;

  /* List of files that are observed for changes */
  GHashTable * observed;

  /* Extra data we need to free on finalization */
  CBData * cb_data;
} UbuntuOneNautilus;

typedef struct {
  GObjectClass parent_slot;
} UbuntuOneNautilusClass;

static void ubuntuone_nautilus_finalize(GObject * object);

static GType ubuntuone_nautilus_get_type (void);
static void ubuntuone_nautilus_register_type (GTypeModule * module);

/* Utility functions */
static void ubuntuone_nautilus_reset_emblem (UbuntuOneNautilus * uon,
					     const char * path);

static void ubuntuone_nautilus_observed_file_unref (gpointer user_data,
                                                    GObject *where_the_object_was);

static void ubuntuone_nautilus_observed_file_foreach_unref (gpointer key,
							    gpointer value,
							    gpointer user_data);

/* DBus signal and async method call handlers */
static void ubuntuone_nautilus_update_meta (DBusGProxy * proxy,
					    DBusGProxyCall * call_id,
					    gpointer user_data);
static void ubuntuone_nautilus_end_dbus_call (DBusGProxy * proxy,
					      DBusGProxyCall * call_id,
					      gpointer user_data);
static void ubuntuone_nautilus_got_root (DBusGProxy * proxy,
					 DBusGProxyCall * call_id,
					 gpointer user_data);
static void ubuntuone_nautilus_got_shared (DBusGProxy * proxy,
					   DBusGProxyCall * call_id,
					   gpointer user_data);
static void ubuntuone_nautilus_status_changed (DBusGProxy * proxy,
					       GHashTable * hash,
					       gpointer user_data);
static void ubuntuone_nautilus_upload_started (DBusGProxy * proxy,
					       gchar * path,
					       gpointer user_data);
static void ubuntuone_nautilus_upload_finished (DBusGProxy * proxy,
						 gchar * path, GHashTable * info,
						 gpointer user_data);
static void ubuntuone_nautilus_download_started (DBusGProxy * proxy,
						 gchar * path,
						 gpointer user_data);
static void ubuntuone_nautilus_download_finished (DBusGProxy * proxy,
						  gchar * path, GHashTable * info,
						  gpointer user_data);
static void ubuntuone_nautilus_udf_created (DBusGProxy * proxy,
					    GHashTable * hash,
					    gpointer user_data);
static void ubuntuone_nautilus_udf_deleted (DBusGProxy * proxy,
					    GHashTable * hash,
					    gpointer user_data);
static void ubuntuone_nautilus_udf_error (DBusGProxy * proxy,
										  GHashTable * hash,
										  gchar * error,
										  gpointer user_data);
static void ubuntuone_nautilus_file_published (DBusGProxy * proxy,
					       GHashTable * hash,
					       gpointer user_data);
static void ubuntuone_nautilus_publish_error (DBusGProxy * proxy,
					      GHashTable * hash,
					      gchar * error,
					      gpointer user_data);
static void ubuntuone_nautilus_got_public_files (DBusGProxy * proxy,
                                                 GPtrArray * files,
                                                 gpointer user_data);
static void ubuntuone_nautilus_public_files_list_error (DBusGProxy * proxy,
                                                        gchar * error,
                                                        gpointer user_data);
static void ubuntuone_nautilus_share_created (DBusGProxy * proxy,
					      GHashTable * hash,
					      gpointer user_data);
static void ubuntuone_nautilus_sharing_error (DBusGProxy * proxy,
					      GHashTable * hash,
					      gchar * error,
					      gpointer user_data);

static GObjectClass * parent_class = NULL;

static void ubuntuone_nautilus_reset_emblem (UbuntuOneNautilus * uon,
					     const char * path) {

  NautilusFileInfo *file = g_hash_table_lookup (uon->observed, path);
  if (file)
    nautilus_file_info_invalidate_extension_info (file);
}

/* Called when NautilusFileInfo is unreferenced */
static void ubuntuone_nautilus_observed_file_unref (gpointer user_data,
                                                    GObject *where_the_object_was) {
  UbuntuOneNautilus * uon;
  NautilusFileInfo * file;
  gchar * path = NULL;

  uon  = UBUNTUONE_NAUTILUS(user_data);
  file = NAUTILUS_FILE_INFO(where_the_object_was);
  path = g_filename_from_uri (nautilus_file_info_get_uri (file), NULL, NULL);

  if (g_hash_table_lookup (uon->observed, path ))
    g_hash_table_remove (uon->observed, path);

}

/* Cleanup routine for weak reference callbacks */
static void ubuntuone_nautilus_observed_file_foreach_unref (gpointer key,
							    gpointer value,
							    gpointer user_data) {
  UbuntuOneNautilus * uon;
  NautilusFileInfo * file;

  uon = UBUNTUONE_NAUTILUS(user_data);
  file = NAUTILUS_FILE_INFO(value);

  g_object_weak_unref (G_OBJECT(file),
		       (GWeakNotify)ubuntuone_nautilus_observed_file_unref,
		       uon);

}

/* Are we in an Ubuntu One managed (root or udf) directory */
static gboolean ubuntuone_is_storagefs (UbuntuOneNautilus * uon,
					const char * path) {
  gboolean managed = FALSE;
  gchar * dirpath;
  gchar * shared;
  GList * udfs, * l;

  if (!uon->managed)
    return FALSE;

  if (!path)
    return FALSE;

  if (strcmp (path, uon->managed) == 0)
    return TRUE;

  shared = g_build_filename (uon->managed, "Shared With Me", NULL);
  if (strncmp (path, shared, strlen (shared)) == 0) {
    managed = FALSE;
    goto fsdone;
  }

  dirpath = g_strdup_printf ("%s/", uon->managed);
  if (strncmp (path, dirpath, strlen (dirpath)) == 0)
    managed = TRUE;

  g_free (dirpath);

  udfs = g_hash_table_get_keys (uon->udfs);
  for (l = udfs; l != NULL && l->data != NULL; l = l->next) {
    dirpath = g_strdup_printf ("%s/", (char *) l->data);
    if (strncmp (path, dirpath, strlen (dirpath)) == 0)
      managed = TRUE;

    g_free (dirpath);
  }
  g_list_free (udfs);

 fsdone:
  g_free (shared);
  return managed;
}

static void ubuntuone_nautilus_add_observed (UbuntuOneNautilus * uon,
					     gchar * path,
					     NautilusFileInfo * file) {
  g_object_weak_ref (G_OBJECT(file),
		     (GWeakNotify)ubuntuone_nautilus_observed_file_unref,
		     uon);
  g_hash_table_insert (uon->observed, g_strdup (path), file);
}

/* Update file info provider */
static NautilusOperationResult ubuntuone_nautilus_update_file_info (NautilusInfoProvider * provider,
								    NautilusFileInfo * file,
								    GClosure * update_complete,
								    NautilusOperationHandle ** handle) {
  UbuntuOneNautilus * uon;
  char * path = NULL;

  uon = UBUNTUONE_NAUTILUS(provider);

  path = g_filename_from_uri (nautilus_file_info_get_uri (file), NULL, NULL);

  if (path == NULL)
    return NAUTILUS_OPERATION_COMPLETE;

  if (!g_hash_table_lookup (uon->updated, path) &&
      !g_hash_table_lookup (uon->needsupdating, path) &&
      ubuntuone_is_storagefs (uon, path)) {
    /* Add the unsynchronized emblem anyway, and update later */
    ubuntuone_nautilus_add_observed (uon, path, file);
    nautilus_file_info_add_emblem (file, "ubuntuone-unsynchronized");
    dbus_g_proxy_begin_call (uon->u1_fs, "get_metadata",
			     ubuntuone_nautilus_update_meta, uon,
			     NULL, G_TYPE_STRING, path, G_TYPE_INVALID);
    goto updating_meta;
  }

  if (g_hash_table_lookup (uon->uploads, path) ||
      g_hash_table_lookup (uon->downloads, path)) {
    nautilus_file_info_add_emblem (file, "ubuntuone-updating");
    ubuntuone_nautilus_add_observed (uon, path, file);
  } else if (g_hash_table_lookup (uon->updated, path)) {
    nautilus_file_info_add_emblem (file, "ubuntuone-synchronized");
    ubuntuone_nautilus_add_observed (uon, path, file);
  } else if (g_hash_table_lookup (uon->needsupdating, path)) {
    nautilus_file_info_add_emblem (file, "ubuntuone-unsynchronized");
    ubuntuone_nautilus_add_observed (uon, path, file);
  }

  if (g_hash_table_lookup (uon->udfs, path)) {
    nautilus_file_info_add_emblem (file, "ubuntuone-enabled");
    ubuntuone_nautilus_add_observed (uon, path, file);
  }

  if (g_hash_table_lookup (uon->public, path)) {
    nautilus_file_info_add_emblem (file, "ubuntuone-public");
    ubuntuone_nautilus_add_observed (uon, path, file);
  }

  if (g_hash_table_lookup (uon->shares, path)) {
    nautilus_file_info_add_emblem (file, "shared");
    ubuntuone_nautilus_add_observed (uon, path, file);
  }

 updating_meta:
  g_free (path);

  return NAUTILUS_OPERATION_COMPLETE;
}

static void ubuntuone_nautilus_info_provider_iface_init (NautilusInfoProviderIface * iface) {
  iface->update_file_info = ubuntuone_nautilus_update_file_info;
}

/* LocationWidget provider */
static GtkWidget * ubuntuone_nautilus_get_location_widget(NautilusLocationWidgetProvider * provider,
							  const char * uri,
							  GtkWidget * parent) {
  UbuntuOneNautilus * uon;
  gchar * path;
  GtkWidget * hbox = NULL;
  GtkWidget * label;
  gchar * labeltext;

  uon = UBUNTUONE_NAUTILUS (provider);

  path = g_filename_from_uri (uri, NULL, NULL);

  if (!path || !ubuntuone_is_storagefs (uon, path))
    goto location_done;

  /* Update the list of shared folders - ok to call dbus now */
  dbus_g_proxy_begin_call (uon->u1_shares, "get_shared",
			   ubuntuone_nautilus_got_shared, uon,
			   NULL, G_TYPE_INVALID);

  hbox = gtk_hbox_new (FALSE, 6);
  labeltext = g_strdup_printf ("<b>Ubuntu One</b> %s", _("File Sharing"));
  label = gtk_label_new (labeltext);
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  gtk_widget_show (hbox);

 location_done:
  g_free (path);

  return hbox;
}

static void ubuntuone_nautilus_bar_provider_iface_init (NautilusLocationWidgetProviderIface * iface) {
  iface->get_widget = ubuntuone_nautilus_get_location_widget;
}


/* Magical struct for passing data in callbacks */
struct _CBData {
  UbuntuOneNautilus * uon;
  gchar * path;
  GtkWidget * parent;

  /* Share dialog widgets */
  GtkWidget * user_picker;
  GtkWidget * name_entry;
  GtkWidget * allow_mods;

  /* Whether to make a file public or private */
  gboolean make_public;
};

static void __cb_data_free (struct _CBData * data) {
  if (!data)
    return;

  data->uon = NULL;
  data->parent = NULL;

  data->user_picker = NULL;
  data->name_entry = NULL;
  data->allow_mods = NULL;

  if (data->path) {
    g_free (data->path);
    data->path = NULL;
  }

  g_free (data);

  data = NULL;
}

static void
picker_selection_changed_cb (U1ContactsPicker *picker, gpointer user_data)
{
  GSList *selection;
  GtkWidget * dialog = (GtkWidget *) user_data;

  selection = u1_contacts_picker_get_selected_emails (picker);
  if (selection != NULL) {
    gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT, TRUE);
    u1_contacts_picker_free_selection_list (selection);
  } else
    gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT, FALSE);
}

/* Share on Ubuntu One dialog constructor */
static GtkWidget * ubuntuone_nautilus_share_dialog_construct (struct _CBData * data) {
  GtkWidget * dialog;
  GtkWidget * area, * table, * label;

  dialog = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Share on Ubuntu One"));
  gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (data->parent));
  gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
			  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			  (_("Share")), GTK_RESPONSE_ACCEPT,
			  NULL);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog),
				   GTK_RESPONSE_ACCEPT);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT, FALSE);
  gtk_window_set_icon_name (GTK_WINDOW (dialog), "ubuntuone");

  area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));

  table = gtk_table_new (4, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 12);
  gtk_table_set_col_spacings (GTK_TABLE (table), 6);
  gtk_container_set_border_width (GTK_CONTAINER (table), 7);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (area), table);

  label = gtk_label_new (_("Share _with:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_label_set_use_underline (GTK_LABEL (label), TRUE);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  gtk_widget_show (label);

  data->user_picker = u1_contacts_picker_new ();
  g_signal_connect (G_OBJECT (data->user_picker), "selection-changed",
		    G_CALLBACK (picker_selection_changed_cb), dialog);
  gtk_table_attach (GTK_TABLE (table), data->user_picker, 0, 2, 1, 2,
		    GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		    GTK_FILL | GTK_EXPAND | GTK_SHRINK, 3, 3);
  gtk_widget_show (data->user_picker);

  label = gtk_label_new (_("Share _Name:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_label_set_use_underline (GTK_LABEL (label), TRUE);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
  gtk_widget_show (label);

  data->name_entry = gtk_entry_new ();
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), data->name_entry);
  gtk_table_attach (GTK_TABLE (table), data->name_entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
  gtk_widget_show (data->name_entry);

  /* Default to the directory name */
  gtk_entry_set_text (GTK_ENTRY (data->name_entry),
		      g_path_get_basename (data->path));

  data->allow_mods = gtk_check_button_new_with_mnemonic (_("_Allow Modification"));
  gtk_table_attach (GTK_TABLE (table), data->allow_mods, 0, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
  gtk_widget_show (data->allow_mods);

  gtk_widget_set_size_request (dialog, 500, 450);

  return dialog;
}

/* Menu callbacks */
static void ubuntuone_nautilus_share_dialog_response (GtkDialog * dialog,
						      gint response,
						      gpointer user_data) {
  struct _CBData * data = (struct _CBData *) user_data;

  switch (response) {
  case GTK_RESPONSE_ACCEPT: {
    gchar * name, * modify;
    GSList *emails, *l;
    gchar **emails_array;
    gint i;
    gboolean allow_mods = FALSE;

    emails = u1_contacts_picker_get_selected_emails (U1_CONTACTS_PICKER (data->user_picker));
    if (emails == NULL) {
      GtkWidget *error_dialog;

      error_dialog = gtk_message_dialog_new (GTK_WINDOW (dialog),
					     GTK_DIALOG_DESTROY_WITH_PARENT,
					     GTK_MESSAGE_ERROR,
					     GTK_BUTTONS_CLOSE,
					     _("You need to select at least one contact to share this folder with"));
      gtk_dialog_run (GTK_DIALOG (error_dialog));
      gtk_widget_destroy (error_dialog);

      return;
    }

    emails_array = g_new0 (gchar *, g_slist_length (emails));
    for (l = emails, i = 0; l != NULL; l = l->next, i++)
       emails_array[i] = g_strdup (l->data);

    name = g_strdup (gtk_entry_get_text (GTK_ENTRY (data->name_entry)));
    allow_mods = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->allow_mods));
    modify = g_strdup_printf ("%d", allow_mods);

    dbus_g_proxy_begin_call (data->uon->u1_shares, "create_shares",
			     ubuntuone_nautilus_end_dbus_call, NULL,
			     NULL,
			     G_TYPE_STRING, data->path,
			     G_TYPE_STRV, emails_array,
			     G_TYPE_STRING, name,
			     G_TYPE_STRING, modify,
			     G_TYPE_INVALID);
 
    g_strfreev (emails_array);
    u1_contacts_picker_free_selection_list (emails);
    g_free (name);
    g_free (modify);
  }
  default:
    gtk_widget_destroy (GTK_WIDGET (dialog));
    break;
  }
}

static void ubuntuone_nautilus_share_dialog_closed (GtkDialog * dialog,
						    gpointer user_data) {
  ubuntuone_nautilus_share_dialog_response (dialog, GTK_RESPONSE_CANCEL,
					    user_data);
}

static void ubuntuone_nautilus_public_meta (DBusGProxy * proxy,
					    DBusGProxyCall * call_id,
					    gpointer user_data) {
  struct _CBData * data = (struct _CBData *) user_data;
  GHashTable * metadata;
  GError * error = NULL;
  gchar * share_id, * node_id;

  g_return_if_fail (proxy != NULL);

  if (!dbus_g_proxy_end_call (proxy, call_id, &error,
			      dbus_g_type_get_map ("GHashTable",
						   G_TYPE_STRING,
						   G_TYPE_STRING),
			      &metadata,
			      G_TYPE_INVALID)) {
    g_warning ("ERROR: %s", error->message);
    return;
  }

  share_id = g_hash_table_lookup (metadata, "share_id");
  node_id = g_hash_table_lookup (metadata, "node_id");

  dbus_g_proxy_begin_call (data->uon->u1_public, "change_public_access",
			   ubuntuone_nautilus_end_dbus_call, NULL,
			   NULL,
			   G_TYPE_STRING, share_id,
			   G_TYPE_STRING, node_id,
			   G_TYPE_BOOLEAN, data->make_public,
			   G_TYPE_INVALID);
}

static void ubuntuone_nautilus_copy_public_url (NautilusMenuItem * item,
						gpointer user_data) {
  struct _CBData * data = (struct _CBData *) user_data;
  gchar * url;

  url = g_hash_table_lookup (data->uon->public, data->path);
  gtk_clipboard_set_text (gtk_clipboard_get(GDK_SELECTION_CLIPBOARD),
			  url, strlen (url));
  gtk_clipboard_store (gtk_clipboard_get(GDK_SELECTION_CLIPBOARD));
}

static void ubuntuone_nautilus_toggle_publicity (NautilusMenuItem * item,
						 gpointer user_data) {
  struct _CBData * data = (struct _CBData *) user_data;

  dbus_g_proxy_begin_call (data->uon->u1_fs, "get_metadata",
			   ubuntuone_nautilus_public_meta, data,
			   NULL,
			   G_TYPE_STRING, data->path,
			   G_TYPE_INVALID);

  g_hash_table_replace (data->uon->public, g_strdup (data->path), UPDATE_PENDING);
  ubuntuone_nautilus_reset_emblem (data->uon, data->path);
}

static void ubuntuone_nautilus_share_folder (NautilusMenuItem * item,
					     gpointer * user_data) {
  struct _CBData * data = (struct _CBData *) user_data;
  GtkWidget * dialog;

  dialog = ubuntuone_nautilus_share_dialog_construct (data);
  g_signal_connect (dialog, "close",
		    G_CALLBACK (ubuntuone_nautilus_share_dialog_closed), data);
  g_signal_connect (dialog, "response",
		    G_CALLBACK (ubuntuone_nautilus_share_dialog_response), data);

  gtk_widget_show (dialog);
}

static void ubuntuone_nautilus_unshare_folder (NautilusMenuItem * item,
					       gpointer * user_data) {
  struct _CBData * data = (struct _CBData *) user_data;
  gchar * share_id;

  share_id = g_hash_table_lookup (data->uon->shares, data->path);
  dbus_g_proxy_begin_call (data->uon->u1_shares, "delete_share",
			   ubuntuone_nautilus_end_dbus_call, NULL,
			   NULL,
			   G_TYPE_STRING, share_id,
			   G_TYPE_INVALID);
}

static void ubuntuone_nautilus_synchronize_folder (NautilusMenuItem * item,
						   gpointer * user_data) {
  struct _CBData * data = (struct _CBData *) user_data;

  dbus_g_proxy_begin_call (data->uon->u1_folders, "create",
			   ubuntuone_nautilus_end_dbus_call,
			   NULL, NULL,
			   G_TYPE_STRING, data->path,
			   G_TYPE_INVALID);

  g_hash_table_replace (data->uon->udfs, g_strdup (data->path), UPDATE_PENDING);
  ubuntuone_nautilus_reset_emblem (data->uon, data->path);
}

static void ubuntuone_nautilus_delete_folder (NautilusMenuItem * item,
						   gpointer * user_data) {
  struct _CBData * data = (struct _CBData *) user_data;
  gchar * udf_id;

  udf_id = g_hash_table_lookup (data->uon->udfs, data->path);

  dbus_g_proxy_begin_call (data->uon->u1_folders, "delete",
			   ubuntuone_nautilus_end_dbus_call,
			   NULL, NULL,
			   G_TYPE_STRING, udf_id,
			   G_TYPE_INVALID);

  g_hash_table_replace (data->uon->udfs, g_strdup (data->path), UPDATE_PENDING);
  ubuntuone_nautilus_reset_emblem (data->uon, data->path);
}

/* Menu provider */
static GList * ubuntuone_nautilus_get_menu_items (NautilusMenuProvider * provider,
						   GtkWidget * window,
						   GList * files) {
  UbuntuOneNautilus * uon;
  NautilusFileInfo * file;
  GList * items = NULL;
  gchar * path;
  gchar * item;
  gchar * homedir_path;
  gboolean is_managed, is_root, is_udf, is_public, is_shared, is_pending;
  gboolean is_inhome;
  struct _CBData * cb_data;

  is_managed = is_root = is_udf = is_public = is_shared = is_pending = FALSE;
  is_inhome = FALSE;

  if (g_list_length (files) != 1)
    return NULL;

  uon = UBUNTUONE_NAUTILUS (provider);

  file = g_list_nth_data (files, 0);
  path = g_filename_from_uri (nautilus_file_info_get_uri (file), NULL, NULL);

  if (path == NULL)
    goto done;

  if (strcmp (path, uon->managed) == 0)
    is_root = TRUE;

  if (ubuntuone_is_storagefs (uon, path))
    is_managed = TRUE;

  homedir_path = g_strdup_printf ("%s/", g_get_home_dir());
  if (strncmp (path, homedir_path, strlen (homedir_path)) == 0)
	is_inhome = TRUE;
  g_free (homedir_path);

  if ((item = g_hash_table_lookup (uon->udfs, path)) != NULL) {
    is_udf = TRUE;
	if (strcmp (item, UPDATE_PENDING) == 0)
	  is_pending = TRUE;
  } else if ((item = g_hash_table_lookup (uon->public, path)) != NULL) {
    is_public = TRUE;
	if (strcmp (item, UPDATE_PENDING) == 0)
	  is_pending = TRUE;
  }
  if (g_hash_table_lookup (uon->shares, path) != NULL)
    is_shared = TRUE;

  if (uon->cb_data)
    __cb_data_free (uon->cb_data);

  cb_data = g_new0 (struct _CBData, 1);
  cb_data->uon = uon;
  cb_data->parent = window;
  cb_data->path = g_strdup (path);

  uon->cb_data = cb_data;

  if ((is_managed || is_udf) && !is_root && nautilus_file_info_is_directory (file)) {
    NautilusMenuItem * item;

    item = nautilus_menu_item_new ("ubuntuone-share",
				   _("Share on Ubuntu One..."),
				   _("Share this folder on Ubuntu One"),
				   "ubuntuone");
	if (is_pending)
	  g_object_set (item, "sensitive", FALSE, NULL);

    g_signal_connect (item, "activate",
		      G_CALLBACK (ubuntuone_nautilus_share_folder),
		      cb_data);
    items = g_list_prepend (items, item);
  }

  if ((is_managed && is_shared) && !is_root &&  nautilus_file_info_is_directory (file)) {
    NautilusMenuItem * item;

    item = nautilus_menu_item_new ("ubuntuone-unshare",
				   _("Stop sharing on Ubuntu One..."),
				   _("Stop sharing this folder on Ubuntu One"),
				   "ubuntuone");
	if (is_pending)
	  g_object_set (item, "sensitive", FALSE, NULL);

    g_signal_connect (item, "activate",
		      G_CALLBACK (ubuntuone_nautilus_unshare_folder),
		      cb_data);
    items = g_list_prepend (items, item);
  }

  if (is_udf && nautilus_file_info_is_directory (file)) {
    NautilusMenuItem * item;

    item = nautilus_menu_item_new ("ubuntuone-udf-delete",
				   _("Stop synchronizing on Ubuntu One"),
				   _("Stop synchronizing this folder with Ubuntu One."),
				   "ubuntuone");
	if (is_pending)
	  g_object_set (item, "sensitive", FALSE, NULL);

    g_signal_connect (item, "activate",
		      G_CALLBACK (ubuntuone_nautilus_delete_folder),
		      cb_data);
    items = g_list_append (items, item);
  }

  if (is_inhome && !is_managed && !is_udf &&
	  nautilus_file_info_is_directory (file) &&
	  (nautilus_file_info_get_file_type (file) != G_FILE_TYPE_SYMBOLIC_LINK)) {
    NautilusMenuItem * item;

    item = nautilus_menu_item_new ("ubuntuone-udf-create",
				   _("Synchronize on Ubuntu One"),
				   _("Synchronize this folder with Ubuntu One."),
				   "ubuntuone");
    g_signal_connect (item, "activate",
		      G_CALLBACK (ubuntuone_nautilus_synchronize_folder),
		      cb_data);
    items = g_list_append (items, item);
  }
    
  if (is_managed && (nautilus_file_info_get_file_type (file) == G_FILE_TYPE_REGULAR)) {
    NautilusMenuItem * item, * urlitem;

    if (is_public) {
      urlitem = nautilus_menu_item_new ("ubuntuone-geturl",
					_("Copy Ubuntu One public URL"),
					_("Copy the Ubuntu One public URL for this file to the clipboard."),
					"ubuntuone");
	  if (is_pending)
		g_object_set (urlitem, "sensitive", FALSE, NULL);

      g_signal_connect (urlitem, "activate",
			G_CALLBACK (ubuntuone_nautilus_copy_public_url),
			cb_data);
      items = g_list_append (items, urlitem);
      item = nautilus_menu_item_new ("ubuntuone-unpublish",
                                     _("Stop publishing via Ubuntu One"),
                                     _("No longer share this file with everyone."),
                                     "ubuntuone");
	  if (is_pending)
		g_object_set (item, "sensitive", FALSE, NULL);

      cb_data->make_public = FALSE;
    } else {
      item = nautilus_menu_item_new ("ubuntuone-publish",
                                     _("Publish via Ubuntu One"),
                                     _("Make this file available to anyone."),
                                     "ubuntuone");
      cb_data->make_public = TRUE;
    }
    g_signal_connect (item, "activate",
                      G_CALLBACK (ubuntuone_nautilus_toggle_publicity),
                      cb_data);
    items = g_list_append (items, item);
  }

 done:
  g_free (path);

  return items;
}

static GList * ubuntuone_nautilus_get_bg_menu_items (NautilusMenuProvider * provider,
						     GtkWidget * window,
						     NautilusFileInfo * folder) {
  GList * files = NULL;
  GList * items = NULL;

  files = g_list_prepend (files, folder);
  items = ubuntuone_nautilus_get_menu_items (provider, window, files);
  files = g_list_remove (files, folder);
  g_list_free (files);

  return items;
}

static void ubuntuone_nautilus_menu_provider_iface_init (NautilusMenuProviderIface * iface) {
  iface->get_file_items = ubuntuone_nautilus_get_menu_items;
  iface->get_background_items = ubuntuone_nautilus_get_bg_menu_items;
}

/* GType and nautilus module stuff */
static GType un_type = 0;

static void ubuntuone_nautilus_instance_init (UbuntuOneNautilus * uon) {
  uon->connected = FALSE;
  uon->uploads = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->downloads = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->shares = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->updated = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->needsupdating = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->observed = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->udfs = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
  uon->public = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);

  uon->bus = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
  if (!uon->bus) {
    g_warning ("Failed to get session bus.");
    return;
  }

  uon->u1_proxy = dbus_g_proxy_new_for_name (uon->bus,
					     "com.ubuntuone.SyncDaemon",
					     "/",
					     "com.ubuntuone.SyncDaemon.SyncDaemon");
  uon->u1_status = dbus_g_proxy_new_for_name (uon->bus,
					      "com.ubuntuone.SyncDaemon",
					      "/status",
					      "com.ubuntuone.SyncDaemon.Status");
  uon->u1_shares = dbus_g_proxy_new_for_name (uon->bus,
					      "com.ubuntuone.SyncDaemon",
					      "/shares",
					      "com.ubuntuone.SyncDaemon.Shares");
  uon->u1_fs     = dbus_g_proxy_new_for_name (uon->bus,
					      "com.ubuntuone.SyncDaemon",
					      "/filesystem",
					      "com.ubuntuone.SyncDaemon.FileSystem");

  uon->u1_folders = dbus_g_proxy_new_for_name (uon->bus,
					       "com.ubuntuone.SyncDaemon",
					       "/folders",
					       "com.ubuntuone.SyncDaemon.Folders");

  uon->u1_public = dbus_g_proxy_new_for_name (uon->bus,
					      "com.ubuntuone.SyncDaemon",
					      "/publicfiles",
					      "com.ubuntuone.SyncDaemon.PublicFiles");

  /* Default to ~/Ubuntu One for now, as it's all we really support */
  uon->managed = g_build_filename (g_get_home_dir (), "Ubuntu One", NULL);
  uon->gotroot = FALSE;
  uon->gotudfs = FALSE;
  uon->gotpubs = FALSE;

#if 0
  /*
   * These DBus calls are temporarily disabled, to improve performance
   * when logging in by avoiding start-up of ubuntuone-syncdaemon which
   * provides the interface
   */
  dbus_g_proxy_begin_call (uon->u1_proxy, "get_rootdir",
			   ubuntuone_nautilus_got_root, uon,
			   NULL, G_TYPE_INVALID);
  dbus_g_proxy_begin_call (uon->u1_shares, "get_shared",
			   ubuntuone_nautilus_got_shared, uon,
			   NULL, G_TYPE_INVALID);
#endif

  /* Marshaller for UploadFinished and DownloadFinished signals */
  dbus_g_object_register_marshaller (ubuntuone_nautilus_marshal_VOID__STRING_OBJECT,
				     G_TYPE_NONE,
				     G_TYPE_STRING,
				     dbus_g_type_get_map ("GHashTable",
					G_TYPE_STRING,
					G_TYPE_STRING),
				     G_TYPE_INVALID);

  /* Marshaller for ShareCreateError signal */
  dbus_g_object_register_marshaller (ubuntuone_nautilus_marshal_VOID__OBJECT_STRING,
				     G_TYPE_NONE,
				     dbus_g_type_get_map ("GHashTable",
					G_TYPE_STRING,
					G_TYPE_STRING),
				     G_TYPE_STRING,
				     G_TYPE_INVALID);

  dbus_g_proxy_add_signal (uon->u1_status, "StatusChanged",
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_status, "StatusChanged",
			       G_CALLBACK(ubuntuone_nautilus_status_changed),
			       uon, NULL);

  dbus_g_proxy_add_signal (uon->u1_status, "UploadStarted",
			   G_TYPE_STRING,
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_status, "UploadStarted",
			       G_CALLBACK (ubuntuone_nautilus_upload_started),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_status, "UploadFinished",
			   G_TYPE_STRING,
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_status, "UploadFinished",
			       G_CALLBACK (ubuntuone_nautilus_upload_finished),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_status, "DownloadStarted",
			   G_TYPE_STRING,
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_status, "DownloadStarted",
			       G_CALLBACK (ubuntuone_nautilus_download_started),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_status, "DownloadFinished",
			   G_TYPE_STRING,
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_status, "DownloadFinished",
			       G_CALLBACK (ubuntuone_nautilus_download_finished),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_shares, "ShareCreated",
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_shares, "ShareCreated",
			       G_CALLBACK (ubuntuone_nautilus_share_created),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_shares, "ShareCreateError",
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_STRING,
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_shares, "ShareCreateError",
			       G_CALLBACK (ubuntuone_nautilus_sharing_error),
			       uon, NULL);

  dbus_g_proxy_add_signal (uon->u1_folders, "FolderCreated",
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_folders, "FolderCreated",
			       G_CALLBACK (ubuntuone_nautilus_udf_created),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_folders, "FolderSubscribed",
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_folders, "FolderSubscribed",
			       G_CALLBACK (ubuntuone_nautilus_udf_created),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_folders, "FolderCreateError",
						   dbus_g_type_get_map ("GHashTable",
												G_TYPE_STRING,
												G_TYPE_STRING),
						   G_TYPE_STRING,
						   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_folders, "FolderCreateError",
							   G_CALLBACK (ubuntuone_nautilus_udf_error),
							   uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_folders, "FolderSubscribeError",
						   dbus_g_type_get_map ("GHashTable",
												G_TYPE_STRING,
												G_TYPE_STRING),
						   G_TYPE_STRING,
						   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_folders, "FolderSubscribeError",
							   G_CALLBACK (ubuntuone_nautilus_udf_error),
							   uon, NULL);

  dbus_g_proxy_add_signal (uon->u1_folders, "FolderDeleted",
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_folders, "FolderDeleted",
			       G_CALLBACK (ubuntuone_nautilus_udf_deleted),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_folders, "FolderUnSubscribed",
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_folders, "FolderUnSubscribed",
			       G_CALLBACK (ubuntuone_nautilus_udf_deleted),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_folders, "FolderDeleteError",
						   dbus_g_type_get_map ("GHashTable",
												G_TYPE_STRING,
												G_TYPE_STRING),
						   G_TYPE_STRING,
						   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_folders, "FolderDeleteError",
							   G_CALLBACK (ubuntuone_nautilus_udf_error),
							   uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_folders, "FolderUnSubscribeError",
						   dbus_g_type_get_map ("GHashTable",
												G_TYPE_STRING,
												G_TYPE_STRING),
						   G_TYPE_STRING,
						   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_folders, "FolderUnSubscribeError",
							   G_CALLBACK (ubuntuone_nautilus_udf_error),
							   uon, NULL);


  dbus_g_proxy_add_signal (uon->u1_public, "PublicAccessChanged",
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_public, "PublicAccessChanged",
			       G_CALLBACK (ubuntuone_nautilus_file_published),
			       uon, NULL);
  dbus_g_proxy_add_signal (uon->u1_public, "PublicAccessChangeError",
			   dbus_g_type_get_map ("GHashTable",
						G_TYPE_STRING,
						G_TYPE_STRING),
			   G_TYPE_STRING,
			   G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_public, "PublicAccessChangeError",
			       G_CALLBACK (ubuntuone_nautilus_publish_error),
			       uon, NULL);

  dbus_g_proxy_add_signal (uon->u1_public, "PublicFilesList",
                           dbus_g_type_get_collection ("GPtrArray",
                                                       dbus_g_type_get_map
                                                       ("GHashTable",
                                                        G_TYPE_STRING,
                                                        G_TYPE_STRING)),
                           G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_public, "PublicFilesList",
                               G_CALLBACK (ubuntuone_nautilus_got_public_files),
                               uon, NULL);

  dbus_g_proxy_add_signal (uon->u1_public, "PublicFilesListError",
                           G_TYPE_STRING,
                           G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (uon->u1_public, "PublicFilesListError",
                               G_CALLBACK (ubuntuone_nautilus_public_files_list_error),
                               uon, NULL);

}

static void ubuntuone_nautilus_class_init (UbuntuOneNautilusClass * klass) {
  parent_class = g_type_class_peek_parent (klass);

  G_OBJECT_CLASS(klass)->finalize = ubuntuone_nautilus_finalize;
}

static void ubuntuone_nautilus_finalize(GObject * object) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS(object);

  __cb_data_free (uon->cb_data);

  if (uon->u1_proxy)
    g_object_unref (uon->u1_proxy);

  if (uon->u1_status)
    g_object_unref (uon->u1_status);

  if (uon->u1_shares)
    g_object_unref (uon->u1_shares);

  if (uon->u1_fs)
    g_object_unref (uon->u1_fs);

  if (uon->u1_folders)
    g_object_unref (uon->u1_folders);

  if (uon->u1_public)
    g_object_unref (uon->u1_public);

  if (uon->bus)
    dbus_g_connection_unref (uon->bus);

  g_hash_table_destroy (uon->uploads);
  uon->uploads = NULL;

  g_hash_table_destroy (uon->downloads);
  uon->downloads = NULL;

  g_hash_table_destroy (uon->shares);
  uon->shares = NULL;

  g_hash_table_destroy (uon->udfs);
  uon->udfs = NULL;

  g_hash_table_destroy (uon->public);
  uon->public = NULL;

  g_hash_table_destroy (uon->updated);
  uon->updated = NULL;

  g_hash_table_destroy (uon->needsupdating);
  uon->needsupdating = NULL;

  /* We need to remove all weak reference callbacks */
  g_hash_table_foreach (uon->observed, 
			(GHFunc)ubuntuone_nautilus_observed_file_foreach_unref,
			uon);

  g_hash_table_destroy (uon->observed);
  uon->observed = NULL;
}

static GType ubuntuone_nautilus_get_type (void) {
  return un_type;
}

static void ubuntuone_nautilus_register_type (GTypeModule * module) {
  static const GTypeInfo info = {
    sizeof (UbuntuOneNautilusClass),
    (GBaseInitFunc) NULL,
    (GBaseFinalizeFunc) NULL,
    (GClassInitFunc) ubuntuone_nautilus_class_init,
    NULL,
    NULL,
    sizeof (UbuntuOneNautilus),
    0,
    (GInstanceInitFunc) ubuntuone_nautilus_instance_init,
  };

  static const GInterfaceInfo info_provider_iface_info = {
    (GInterfaceInitFunc) ubuntuone_nautilus_info_provider_iface_init,
    NULL,
    NULL
  };

  static const GInterfaceInfo bar_provider_iface_info = {
    (GInterfaceInitFunc) ubuntuone_nautilus_bar_provider_iface_init,
    NULL,
    NULL
  };

  static const GInterfaceInfo menu_provider_iface_info = {
    (GInterfaceInitFunc) ubuntuone_nautilus_menu_provider_iface_init,
    NULL,
    NULL
  };

  un_type = g_type_module_register_type (module, 
					 G_TYPE_OBJECT,
					 "UbuntuOneNautilus",
					 &info, 0);
  
  g_type_module_add_interface (module,
			       un_type,
			       NAUTILUS_TYPE_INFO_PROVIDER,
			       &info_provider_iface_info);

  g_type_module_add_interface (module,
			       un_type,
			       NAUTILUS_TYPE_LOCATION_WIDGET_PROVIDER,
			       &bar_provider_iface_info);

  /* XXX Need to sign a request via DBus */
  g_type_module_add_interface (module,
			       un_type,
			       NAUTILUS_TYPE_MENU_PROVIDER,
			       &menu_provider_iface_info);
}


/* DBus signal handlers and async method call handlers */
static void ubuntuone_nautilus_update_meta (DBusGProxy * proxy,
					    DBusGProxyCall * call_id,
					    gpointer user_data) {
  UbuntuOneNautilus * uon;
  GHashTable * metadata;
  GError * error = NULL;
  gchar * local, * server, * path, * new_path;

  g_return_if_fail (proxy != NULL);

  if (!dbus_g_proxy_end_call (proxy, call_id, &error,
			      dbus_g_type_get_map ("GHashTable",
						   G_TYPE_STRING,
						   G_TYPE_STRING),
			      &metadata,
			      G_TYPE_INVALID)) {
    g_warning ("ERROR: %s", error->message);
    return;
  }

  uon = UBUNTUONE_NAUTILUS (user_data);

  path = g_hash_table_lookup (metadata, "path");
  local = g_hash_table_lookup (metadata, "local_hash");
  server = g_hash_table_lookup (metadata, "server_hash");

  new_path = g_strdup (path);

  gboolean is_updated = local && server &&
			strlen (local) > 0 &&
			strlen (server) > 0 &&
			strcmp (local, server) == 0;

  if (is_updated)
    g_hash_table_replace (uon->updated, new_path, new_path);
  else
    g_hash_table_replace (uon->needsupdating, new_path, new_path);
  ubuntuone_nautilus_reset_emblem (uon, new_path);
}

static void ubuntuone_nautilus_end_dbus_call (DBusGProxy * proxy,
					      DBusGProxyCall * call_id,
					      gpointer user_data) {
  g_return_if_fail (proxy != NULL);

  dbus_g_proxy_end_call (proxy, call_id, NULL, G_TYPE_INVALID);
}

static void ubuntuone_nautilus_got_root (DBusGProxy * proxy,
					 DBusGProxyCall * call_id,
					 gpointer user_data) {
  UbuntuOneNautilus * uon;
  gchar * managed;
  GError * error = NULL;

  g_return_if_fail (proxy != NULL);

  if (!dbus_g_proxy_end_call (proxy, call_id, &error,
			      G_TYPE_STRING, &managed,
			      G_TYPE_INVALID)) {
    g_warning ("ERROR: %s", error->message);
    return;
  }

  uon = UBUNTUONE_NAUTILUS (user_data);

  uon->managed = g_strdup (managed);
  uon->gotroot = TRUE;

  g_free (managed);
}

static void ubuntuone_nautilus_got_udfs(DBusGProxy * proxy,
					DBusGProxyCall * call_id,
					gpointer user_data) {
  UbuntuOneNautilus * uon;
  GSList * data, *l;
  GError * error = NULL;

  g_return_if_fail (proxy != NULL);

  if (!dbus_g_proxy_end_call (proxy, call_id, &error,
			      dbus_g_type_get_collection ("GSList",
							  dbus_g_type_get_map
							  ("GHashTable",
							   G_TYPE_STRING,
							   G_TYPE_STRING)),
			      &data,
			      G_TYPE_INVALID)) {
    g_warning ("ERROR: %s", error->message);
    return;
  }

  uon = UBUNTUONE_NAUTILUS (user_data);
  for (l = data; l != NULL && l->data != NULL; l = l->next) {
    GHashTable * hash = l->data;
    ubuntuone_nautilus_udf_created (proxy, hash, user_data);
  }
  g_slist_free (data);

  uon->gotudfs = TRUE;
}

static void ubuntuone_nautilus_got_shared (DBusGProxy * proxy,
					   DBusGProxyCall * call_id,
					   gpointer user_data) {
  UbuntuOneNautilus * uon;
  GSList * data, * l;
  GError * error = NULL;

  g_return_if_fail (proxy != NULL);

  if (!dbus_g_proxy_end_call (proxy, call_id, &error,
			      dbus_g_type_get_collection ("GSList",
							  dbus_g_type_get_map
							  ("GHashTable",
							   G_TYPE_STRING,
							   G_TYPE_STRING)),
			      &data,
			      G_TYPE_INVALID)) {
    g_warning ("ERROR: %s", error->message);
    return;
  }

  uon = UBUNTUONE_NAUTILUS (user_data);
  for (l = data; l != NULL && l->data != NULL; l = l->next) {
    GHashTable * hash = l->data;
    ubuntuone_nautilus_share_created (proxy, hash, user_data);
  }
  g_slist_free (data);
}

static void ubuntuone_nautilus_status_changed (DBusGProxy * proxy,
					       GHashTable * hash,
					       gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * status;
  gboolean is_online;

  is_online = g_strcmp0(g_hash_table_lookup (hash, "is_online"), "True") == 0;
  status = g_hash_table_lookup (hash, "name");

  /* Get the root when we get a status change signal, if we haven't yet */
  if (!uon->gotroot)
    dbus_g_proxy_begin_call (uon->u1_proxy, "get_rootdir",
			     ubuntuone_nautilus_got_root, uon,
			     NULL, G_TYPE_INVALID);

  /* Get the list of UDFs if we haven't already */
  if (!uon->gotudfs)
    dbus_g_proxy_begin_call (uon->u1_folders, "get_folders",
			     ubuntuone_nautilus_got_udfs, uon,
			     NULL, G_TYPE_INVALID);

  /* Get the list of public files if we haven't already */
  if (is_online && !uon->gotpubs) {
    dbus_g_proxy_begin_call (uon->u1_public, "get_public_files",
			     ubuntuone_nautilus_end_dbus_call, uon,
			     NULL, G_TYPE_INVALID);
    uon->gotpubs = TRUE;
  }
}

static void ubuntuone_nautilus_upload_started (DBusGProxy * proxy,
					       gchar * path,
					       gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);

  if (!g_hash_table_lookup (uon->uploads, path)) {
    gchar *new_path = g_strdup (path);
    g_hash_table_insert (uon->uploads, new_path, new_path);
    ubuntuone_nautilus_reset_emblem (uon, path);
  }
}

static void ubuntuone_nautilus_upload_finished (DBusGProxy * proxy,
						gchar * path, GHashTable * info,
						gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * new_path;

  g_hash_table_remove (uon->uploads, path);
  g_hash_table_remove (uon->needsupdating, path);

  new_path = g_strdup (path);
  g_hash_table_replace (uon->updated, new_path, new_path);

  ubuntuone_nautilus_reset_emblem (uon, path);
}

static void ubuntuone_nautilus_download_started (DBusGProxy * proxy,
						 gchar * path,
						 gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);

  if (!g_hash_table_lookup (uon->downloads, path)) {
    gchar *new_path = g_strdup (path);
    g_hash_table_insert (uon->downloads, new_path, new_path);
    ubuntuone_nautilus_reset_emblem (uon, path);
  }
}

static void ubuntuone_nautilus_download_finished (DBusGProxy * proxy,
						  gchar * path,
						  GHashTable * info,
						  gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * new_path;

  g_hash_table_remove (uon->downloads, path);
  g_hash_table_remove (uon->needsupdating, path);

  new_path = g_strdup (path);
  g_hash_table_replace (uon->updated, new_path, new_path);

  ubuntuone_nautilus_reset_emblem (uon, path);
}

static void ubuntuone_nautilus_udf_created (DBusGProxy * proxy,
					    GHashTable * hash,
					    gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * path, * id, * subbed;
  gboolean subscribed;

  path = g_hash_table_lookup (hash, "path");
  id = g_hash_table_lookup (hash, "volume_id");
  subbed = g_hash_table_lookup (hash, "subscribed");
  subscribed = (subbed != NULL && strlen (subbed) != 0);
  if (!g_hash_table_lookup (uon->udfs, path)) {
    if (subscribed) {
      g_hash_table_replace (uon->udfs, g_strdup (path), g_strdup (id));
      ubuntuone_nautilus_reset_emblem (uon, path);
    }
  }
}

static void ubuntuone_nautilus_udf_deleted (DBusGProxy * proxy,
					   GHashTable * hash,
					   gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * path;

  path = g_hash_table_lookup (hash, "path");
  if (path != NULL) {
	g_hash_table_remove (uon->udfs, path);
	ubuntuone_nautilus_reset_emblem (uon, path);
  }
}

static void ubuntuone_nautilus_udf_error (DBusGProxy * proxy,
										  GHashTable * hash,
										  gchar * error,
										  gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * path, * id, * subbed;
  gboolean subscribed;

  path = g_hash_table_lookup (hash, "path");
  id = g_hash_table_lookup (hash, "volume_id");
  subbed = g_hash_table_lookup (hash, "subscribed");
  subscribed = (subbed != NULL && strlen (subbed) != 0);

  if (id != NULL && subscribed)
	ubuntuone_nautilus_udf_created (proxy, hash, user_data);
  else
	ubuntuone_nautilus_udf_deleted (proxy, hash, user_data);
}

static void ubuntuone_nautilus_file_published (DBusGProxy * proxy,
					       GHashTable * hash,
					       gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * path, * url, * public;
  gboolean is_public;

  path = g_hash_table_lookup (hash, "path");
  url = g_hash_table_lookup (hash, "public_url");
  public = g_hash_table_lookup (hash, "is_public");
  is_public = (public != NULL && strlen (public) != 0);

  if (!is_public && g_hash_table_lookup (uon->public, path))
    g_hash_table_remove (uon->public, path);

  if (is_public)
    g_hash_table_replace (uon->public, g_strdup (path), g_strdup (url));

  ubuntuone_nautilus_reset_emblem (uon, path);
  uon->gotpubs = TRUE;
}

static void ubuntuone_nautilus_publish_error (DBusGProxy * proxy,
					      GHashTable * hash,
					      gchar * error,
					      gpointer user_data) {
  ubuntuone_nautilus_file_published (proxy, hash, user_data);
}

static void ubuntuone_nautilus_got_public_files (DBusGProxy * proxy,
                                                 GPtrArray * files,
                                                 gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  int i;

  g_hash_table_remove_all (uon->public);
  for (i = 0; i < files->len; i++)
    g_hash_table_insert (uon->public,
                         g_strdup (g_hash_table_lookup (files->pdata[i], "path")),
                         g_strdup (g_hash_table_lookup (files->pdata[i], "public_url")));
}

static void ubuntuone_nautilus_public_files_list_error (DBusGProxy * proxy,
                                                        gchar * error,
                                                        gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  /* the get_public_files call failed. We unset gotpubs, so the call
  is retried when the syncdaemon next tells us it's online */
  uon->gotpubs = FALSE;
}

static void ubuntuone_nautilus_share_created (DBusGProxy * proxy,
					      GHashTable * hash,
					      gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * path, * id;

  path = g_hash_table_lookup (hash, "path");
  id = g_hash_table_lookup (hash, "volume_id");

  if (!g_hash_table_lookup (uon->shares, path)) {
    g_hash_table_insert (uon->shares, g_strdup (path), g_strdup (id));
    ubuntuone_nautilus_reset_emblem (uon, path);
  }
}

static void ubuntuone_nautilus_sharing_error (DBusGProxy * proxy,
					      GHashTable * hash,
					      gchar * error,
					      gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * path, * message;
  GtkWidget * dialog;

  path = g_hash_table_lookup (hash, "path");

  dialog = gtk_message_dialog_new (GTK_WINDOW (uon->cb_data->parent),
				   GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_ERROR,
				   GTK_BUTTONS_CLOSE,
				   _("Error creating share."));
  message = g_strdup_printf (_("There was an error sharing the folder '%s':\n%s"),
			     path,
			     error);
  gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
					    "%s", message);
  g_free (message);
  g_signal_connect_swapped (G_OBJECT (dialog), "response",
			    G_CALLBACK (gtk_widget_destroy), dialog);
  gtk_widget_show (dialog);
}

/* Required Nautilus module handling methods */
void nautilus_module_initialize (GTypeModule * module) {
#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  ubuntuone_nautilus_register_type (module);
}

void nautilus_module_shutdown (void) {
}


void nautilus_module_list_types (const GType ** types,
				 int * num_types) {
  static GType type_list[1];
  
  type_list[0] = UBUNTUONE_TYPE_NAUTILUS;

  *types = type_list;
  *num_types = 1;
}
