# ubuntuone.syncdaemon.sync - sync module
#
# Author: Lucio Torre <lucio.torre@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""This is the magic"""
from __future__ import with_statement

import os
import logging
from cStringIO import StringIO
import sys

from ubuntuone.syncdaemon.marker import MDMarker
from ubuntuone.storageprotocol.dircontent_pb2 import DIRECTORY
from ubuntuone.storageprotocol import dircontent
from ubuntuone.syncdaemon.fsm.fsm import \
    StateMachineRunner, StateMachine
from ubuntuone.syncdaemon.interfaces import IMarker
from ubuntuone.syncdaemon import u1fsfsm
from ubuntuone.syncdaemon.logger import DebugCapture
from ubuntuone.syncdaemon.filesystem_manager import \
    InconsistencyError
empty_hash = ""

class FSKey(object):
    """This encapsulates the problem of getting the metadata with different
    keys."""
    __slots__ = ('fs', 'keys', 'mdid', '_changes')

    def __init__(self, fs, **keys):
        """create"""
        self.fs = fs
        self.keys = keys
        self.mdid = None
        self._changes = {}

    def get_mdid(self):
        """Get the metadata id."""
        if self.mdid is not None:
            return self.mdid
        if len(self.keys) == 1 and "path" in self.keys:
            # pylint: disable-msg=W0212
            mdid = self.fs._idx_path[self.keys["path"]]
        elif len(self.keys) == 1 and "mdid" in self.keys:
            mdid = self.keys["mdid"]
        elif len(self.keys) == 2 and "node_id" in self.keys \
                    and "share_id" in self.keys:
            # pylint: disable-msg=W0212
            mdid = self.fs._idx_node_id[self.keys["share_id"],
                                                        self.keys["node_id"]]
        else:
            raise KeyError("Incorrect keys: %s" % self.keys)
        if mdid is None:
            raise KeyError("cant find mdid")
        self.mdid = mdid
        return mdid

    def get(self, key):
        """Get the value for key."""
        mdid = self.get_mdid()
        if key == 'path':
            mdobj = self.fs.get_by_mdid(mdid)
            return self.fs.get_abspath(mdobj.share_id, mdobj.path)
        elif key == 'node_id':
            mdobj = self.fs.get_by_mdid(mdid)
            if mdobj.node_id is None:
                return MDMarker(mdid)
            else:
                return mdobj.node_id
        elif key == 'parent_id':
            mdobj = self.fs.get_by_mdid(mdid)
            path = self.fs.get_abspath(mdobj.share_id, mdobj.path)
            parent_path = os.path.dirname(path)
            parent = self.fs.get_by_path(parent_path)
            return parent.node_id or MDMarker(parent.mdid)
        else:
            return getattr(self.fs.get_by_mdid(mdid), key, None)

    def __getitem__(self, key):
        """Get the value for key."""
        return self.get(key)

    def set(self, **kwargs):
        """Set the values for kwargs."""
        self._changes.update(kwargs)

    def sync(self):
        """sync the changes back to FSM"""
        if self._changes and self.has_metadata():
            self.fs.set_by_mdid(self.get_mdid(), **self._changes)
            self._changes = {}

    def has_metadata(self):
        """The State Machine value version of has_metadata."""
        try:
            return str(self.fs.has_metadata(**self.keys))[0]
        except (KeyError, TypeError):
            return 'NA'

    def is_directory(self):
        """The State Machine value version of is_dir.

        This is a string like "T" or "F", not useful as a bool.
        """
        try:
            return str(self.fs.is_dir(**self.keys))[0]
        except KeyError:
            return 'NA'

    def is_dir(self):
        """If the node is a directory or not.

        This is a direct wrapper around FSM.is_dir().
        """
        return self.fs.is_dir(**self.keys)

    def changed(self):
        """The State Machine value version of changed."""
        try:
            return self.fs.changed(**self.keys)
        except KeyError:
            return 'NA'

    def open_file(self):
        """get the file object for reading"""
        mdid = self.get_mdid()
        try:
            fo = self.fs.open_file(mdid)
        except IOError:
            # this is a HUGE cheat
            # the state expectes to start a download
            # but the file is gone. so to keep the transitions correct
            # we return an empty file. we will later receive the FS_DELETE
            return StringIO()
        return fo

    def upload_finished(self, server_hash):
        """signal that we have uploaded the file"""
        mdid = self.get_mdid()
        self.fs.upload_finished(mdid, server_hash)

    def delete_file(self):
        """delete the file and metadata"""
        path = self["path"]
        self.fs.delete_file(path)
        self.mdid = None
        self._changes = {}

    def delete_to_trash(self):
        """Move the node to trash"""
        self.fs.delete_to_trash(self.get_mdid(), self["parent_id"])

    def remove_from_trash(self, share_id, node_id):
        """Remove the node from trash"""
        self.fs.remove_from_trash(share_id, node_id)

    def delete_metadata(self):
        """delete the metadata"""
        path = self["path"]
        self.fs.delete_metadata(path)
        self.mdid = None
        self._changes = {}

    def move_file(self, new_share_id, new_parent_id, new_name):
        """get the stuff we need to move the file."""
        source_path = self['path']
        parent_path = self.fs.get_by_node_id(new_share_id, new_parent_id).path
        dest_path = os.path.join(
            self.fs.get_abspath(new_share_id, parent_path),
            new_name)
        self.fs.move_file(new_share_id, source_path, dest_path)

    def moved(self, new_share_id, path_to):
        """change the metadata of a moved file."""
        self.fs.moved(new_share_id, self['path'], path_to)
        if "path" in self.keys:
            self.keys["path"] = path_to

    def remove_partial(self):
        """remove a partial file"""
        # pylint: disable-msg=W0704
        try:
            self.fs.remove_partial(self["node_id"], self["share_id"])
        except ValueError:
            # we had no partial, ignore
            pass

    def move_to_conflict(self):
        """Move file to conflict"""
        self.fs.move_to_conflict(self.get_mdid())

    def refresh_stat(self):
        """refresh the stat"""
        path = self["path"]
        # pylint: disable-msg=W0704
        try:
            self.fs.refresh_stat(path)
        except OSError:
            # no file to stat, nothing to do
            pass

    def safe_get(self, key, default='^_^'):
        """ safe version of self.get, to be used in the FileLogger. """
        # catch all errors as we are here to help logging
        # pylint: disable-msg=W0703
        try:
            return self.get(key)
        except Exception:
            return default


def loglevel(lvl):
    """Make a function that logs at lvl log level."""
    def level_log(self, message, *args, **kwargs):
        """inner."""
        self.log(lvl, message, *args, **kwargs)
    return level_log


class FileLogger(object):
    """A logger that knows about the file and its state."""
    __slots__ = ('logger', 'key')

    def __init__(self, logger, key):
        """Create a logger for this guy"""
        self.logger = logger
        self.key = key

    def log(self, lvl, message, *args, **kwargs):
        """Log."""
        format = "%(hasmd)s:%(changed)s:%(isdir)s %(mdid)s "\
                 "[%(share_id)r::%(node_id)r] '%(path)r' | %(message)s"
        exc_info = sys.exc_info
        if self.key.has_metadata() == "T":
            # catch all errors as we are logging, pylint: disable-msg=W0703
            try:
                # pylint: disable-msg=W0212
                base = os.path.split(self.key.fs._get_share(
                    self.key['share_id']).path)[1]
                # pylint: disable-msg=W0212
                path = os.path.join(base, self.key.fs._share_relative_path(
                    self.key['share_id'], self.key['path']))
            except Exception:
                # error while getting the path
                self.logger.exception("Error in logger while building the "
                                      "relpath of: %r", self.key['path'])
                path = self.key.safe_get('path')
            extra = dict(message=message,
                         mdid=self.key.safe_get("mdid"),
                         path=path,
                         share_id=self.key.safe_get("share_id") or 'root',
                         node_id=self.key.safe_get("node_id"),
                         hasmd=self.key.has_metadata(),
                         isdir=self.key.is_directory(),
                         changed=self.key.changed())
        else:
            extra = dict(message=message, mdid="-",
                         path="-",
                         share_id="-",
                         node_id="-",
                         hasmd="-",
                         isdir="-",
                         changed="-")
            extra.update(self.key.keys)
        message = format % extra
        if lvl == -1:
            kwargs.update({'exc_info':exc_info})
            self.logger.error(message, *args, **kwargs)
        else:
            self.logger.log(lvl, message, *args, **kwargs)

    critical = loglevel(logging.CRITICAL)
    error = loglevel(logging.ERROR)
    warning = loglevel(logging.WARNING)
    info = loglevel(logging.INFO)
    debug = loglevel(logging.DEBUG)
    exception = loglevel(-1)

class SyncStateMachineRunner(StateMachineRunner):
    """This is where all the state machine methods are."""

    def __init__(self, fsm, main, key, logger=None):
        """Create the runner."""
        super(SyncStateMachineRunner, self).__init__(fsm, logger)
        self.m = main
        self.key = key

    def on_event(self, *args, **kwargs):
        """Override on_event to capture the debug log"""
        in_state = '%(hasmd)s:%(changed)s:%(isdir)s' % \
                dict(hasmd=self.key.has_metadata(),
                     isdir=self.key.is_directory(),
                     changed=self.key.changed())
        is_debug = self.log.logger.isEnabledFor(logging.DEBUG)
        with DebugCapture(self.log.logger):
            func_name = super(SyncStateMachineRunner, self).on_event(*args,
                                                                     **kwargs)
            if not is_debug:
                self.log.info("Called %s (In: %s)" % (func_name, in_state))

    def signal_event_with_hash(self, event, hash, *args):
        """An event that takes a hash ocurred, build the params and signal."""
        self.on_event(event, self.build_hash_eq(hash), hash, *args)

    def validate_actual_data(self, path, oldstat):
        """Validates that the received info is not obsolete."""
        newstat = os.stat(path)
        if newstat.st_ino != oldstat.st_ino or \
           newstat.st_size != oldstat.st_size or \
           newstat.st_mtime != oldstat.st_mtime:
            m = "The received information is obsolete! New stat: %s"
            self.log.debug(m, newstat)
            return False
        return True

    def build_hash_eq(self, hash):
        """Build the event params."""
        try:
            sh = str(self.key["server_hash"] == hash)[0]
            lh = str(self.key["local_hash"] == hash)[0]
        except KeyError:
            sh = lh = "NA"
        return dict(hash_eq_server_hash=sh, hash_eq_local_hash=lh)

    def signal_event_with_error_and_hash(self, event, error, hash, *args):
        """An event that takes a hash ocurred, build the params and signal."""
        params = self.build_error_eq(error)
        params.update(self.build_hash_eq(hash))
        self.on_event(event, params, error, hash, *args)

    def signal_event_with_error(self, event, error, *args):
        """An event returned with error."""
        params = self.build_error_eq(error)
        self.on_event(event, params, error, *args)

    def build_error_eq(self, error):
        """Get the error state."""
        not_available = str(error == 'NOT_AVAILABLE')[0]
        not_authorized = str(error == 'NOT_AUTHORIZED')[0]
        return dict(not_available=not_available, not_authorized=not_authorized)

    def get_state_values(self):
        """Get the values for the current state."""
        return dict(
            has_metadata=self.key.has_metadata(),
            changed=self.key.changed(),
            is_directory=self.key.is_directory(),
        )

    # EVENT HANDLERS

    def nothing(self, event, params, *args):
        """pass"""
        pass

    def new_dir(self, event, params, share_id, node_id, parent_id, name):
        """create a local file."""
        mdobj = self.m.fs.get_by_node_id(share_id, parent_id)
        path = os.path.join(self.m.fs.get_abspath(share_id, mdobj.path), name)
        self.m.fs.create(path=path, share_id=share_id, node_id=node_id,
                         is_dir=True)
        self.m.action_q.query([(share_id, node_id, "")])
        # pylint: disable-msg=W0704
        # this should be provided by FSM, fix!!
        try:
            # pylint: disable-msg=W0212
            with self.m.fs._enable_share_write(share_id, self.key['path']):
                os.mkdir(self.key['path'])
        except OSError, e:
            if not e.errno == 17: #already exists
                raise
        else:
            # just add the watch
            # we hope the user wont have time to add a file just after
            # *we* created the directory
            # this is until we can solve and issue with LR and
            # new dirs and fast downloads
            # see bug #373940
            self.m.event_q.inotify_add_watch(path)
            #self.m.lr.scan_dir(path)


    def new_dir_on_server_with_local_file(self, event, params, share_id,
                                          node_id, parent_id, name):
        """New dir on server and we have local file."""
        self.key.move_to_conflict()
        self.key.delete_metadata()
        self.new_dir(event, params, share_id, node_id, parent_id, name)

    def new_dir_on_server_with_local_dir(self, event, params, share_id,
                                         node_id, parent_id, name):
        """New dir on server and we have local dir: re-get it to converge."""
        self.m.fs.set_node_id(self.key['path'], node_id)
        self.reget_dir(event, params)

    def reget_dir(self, event, params, hash=None):
        """Reget the directory."""
        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.remove_partial()
        self.m.action_q.query([(self.key['share_id'],
                                self.key['node_id'],
                                self.key['local_hash'] or "")])
        self.key.set(server_hash=self.key['local_hash'])
        self.key.sync()

    def get_dir(self, event, params, hash):
        """Get the directory."""
        self.key.set(server_hash=hash)
        self.key.sync()
        self.m.fs.create_partial(node_id=self.key['node_id'],
                                 share_id=self.key['share_id'])
        self.m.action_q.listdir(
            self.key['share_id'], self.key['node_id'], hash,
            lambda : self.m.fs.get_partial_for_writing(
                node_id=self.key['node_id'],
                share_id=self.key['share_id'])
            )

    def file_conflict(self, event, params, hash, crc32, size, stat):
        """This file is in conflict."""
        self.key.move_to_conflict()

    def local_file_conflict(self, event, params, hash):
        """This file is in conflict."""
        self.key.move_to_conflict()
        self.m.action_q.cancel_upload(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.get_file(event, params, hash)

    def merge_directory(self, event, params, hash):
        """Merge the server directory with the local one."""
        new_files = []
        new_dirs = []
        deleted_filedirs = []
        moved = set()

        try:
            fd = self.m.fs.get_partial(node_id=self.key['node_id'],
                            share_id=self.key['share_id'])
        except InconsistencyError:
            self.key.remove_partial()
            self.key.set(server_hash=self.key['local_hash'])
            self.key.sync()
            self.m.action_q.query([
                (self.key["share_id"], self.key["node_id"], "")])
            # we dont perform the merge, we try to re get it
            return


        items = dircontent.parse_dir_content(fd)
        server_dir = [ (o.utf8_name, o.node_type == DIRECTORY, o.uuid)
                        for o in items ]
        client_dir = self.m.fs.dir_content(self.key['path'])
        # XXX: lucio.torre: with huge dirs, this could take a while

        share = self.key['share_id']
        for name, isdir, uuid in server_dir:
            # we took the name as bytes already encoded in utf8
            # directly from dircontent!
            try:
                md = self.m.fs.get_by_node_id(share, uuid)
            except KeyError:
                # not there, a new thing
                if isdir:
                    new_dirs.append((share, uuid, name))
                else:
                    new_files.append((share, uuid, name))
                continue
            mdpath = self.m.fs.get_abspath(md.share_id, md.path)
            if mdpath != os.path.join(self.key['path'], name):
                # this was moved, or maybe the server still didn't receive
                # the move that happened here
                if not self.m.action_q.node_is_with_queued_move(share, uuid):
                    # mark as moved
                    moved.add(uuid)
                    # signal moved
                    self.m.event_q.push("SV_MOVED",
                        share_id=md.share_id, node_id=uuid,
                        new_share_id=share, new_parent_id=self.key['node_id'],
                        new_name=name)


        for name, isdir, uuid in client_dir:
            if uuid is None:
                continue

            if not (name, isdir, uuid) in server_dir:
                # not there, a its gone on the server
                if uuid in moved:
                    # this was a move, dont delete
                    continue
                deleted_filedirs.append((share, uuid))

        # these nodes are in process of being deleted from the server, so they
        # are not exactly new locally (were already deleted here)
        trash = set((share_id, node_id) for (share_id, node_id, _)
                                        in self.m.fs.get_iter_trash())

        parent_uuid = self.key['node_id']
        for share, uuid in deleted_filedirs:
            self.m.event_q.push("SV_FILE_DELETED",
                                node_id=uuid, share_id=share)
        for share, uuid, name in new_files:
            if (share, uuid) not in trash:
                self.m.event_q.push("SV_FILE_NEW", parent_id=parent_uuid,
                                    node_id=uuid, share_id=share, name=name)
        for share, uuid, name in new_dirs:
            if (share, uuid) not in trash:
                self.m.event_q.push("SV_DIR_NEW", parent_id=parent_uuid,
                                    node_id=uuid, share_id=share, name=name)

        self.key.remove_partial()
        self.key.set(local_hash=hash)
        self.key.sync()

    def new_file(self, event, params, share_id, node_id, parent_id, name):
        """create a local file."""
        mdobj = self.m.fs.get_by_node_id(share_id, parent_id)
        path = os.path.join(self.m.fs.get_abspath(share_id, mdobj.path), name)
        self.m.fs.create(path=path, share_id=share_id, node_id=node_id,
                         is_dir=False)
        self.key.set(server_hash="")
        self.key.set(local_hash="")
        self.key.sync()
        self.m.action_q.query([(share_id, node_id, "")])

    def new_file_on_server_with_local(self, event, params, share_id,
                                      node_id, parent_id, name):
        """move local file to conflict and re create"""
        self.key.move_to_conflict()
        self.key.delete_metadata()
        self.new_file(event, params, share_id, node_id, parent_id, name)

    def get_file(self, event, params, hash):
        """Get the contents for the file."""
        self.key.set(server_hash=hash)
        self.key.sync()
        self.m.fs.create_partial(node_id=self.key['node_id'],
                                 share_id=self.key['share_id'])
        self.m.action_q.download(
            share_id=self.key['share_id'], node_id=self.key['node_id'],
            server_hash=hash,
            fileobj_factory=lambda: self.m.fs.get_partial_for_writing(
                node_id=self.key['node_id'],
                share_id=self.key['share_id'])
            )

    def reget_file(self, event, params, hash):
        """cancel and reget this download."""
        self.key.set(server_hash=hash)
        self.key.sync()
        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.remove_partial()
        self.get_file(event, params, hash)

    def client_moved(self, event, params, path_from, path_to):
        """the client moved a file"""
        parent_path = os.path.dirname(path_from)
        old_parent = FSKey(self.m.fs, path=parent_path)
        old_parent_id = old_parent['node_id']
        new_path = os.path.dirname(path_to)
        new_name = os.path.basename(path_to)
        new_parent = FSKey(self.m.fs, path=new_path)
        new_parent_id = new_parent['node_id']

        self.m.action_q.move(share_id=self.key['share_id'],
            node_id=self.key['node_id'], old_parent_id=old_parent_id,
            new_parent_id=new_parent_id, new_name=new_name)
        self.key.moved(self.key['share_id'], path_to)

        # this is cheating, we change the state of another node
        if not IMarker.providedBy(old_parent_id):
            share_id = self.key['share_id']
            self.m.action_q.cancel_download(share_id, old_parent_id)
            old_parent.remove_partial()
            self.m.fs.set_by_node_id(old_parent_id, share_id,
                                     server_hash="", local_hash="")
            self.m.action_q.query([(share_id, old_parent_id, "")])

        # we only hash it if we're a file, not a directory
        if not self.key.is_dir():
            self.m.hash_q.insert(self.key['path'], self.key['mdid'])


    def server_file_changed_back(self, event, params, hash):
        """cancel and dont reget this download."""
        self.key.set(server_hash=hash)
        self.key.sync()
        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.remove_partial()

    def commit_file(self, event, params, hash):
        """commit the new content."""
        try:
            self.m.fs.commit_partial(
                            self.key['node_id'], self.key['share_id'], hash)
        except InconsistencyError:
            # someone or something broke out partials.
            # start work to go to a good state
            self.key.remove_partial()
            self.key.set(server_hash=self.key['local_hash'])
            self.key.sync()
            self.m.action_q.query([
                (self.key["share_id"], self.key["node_id"], "")])

    def new_local_file(self, event, parms, path):
        """a new local file was created"""
        # XXX: lucio.torre: we should use markers here
        parent_path = os.path.dirname(path)
        parent = self.m.fs.get_by_path(parent_path)
        parent_id = parent.node_id or MDMarker(parent.mdid)
        share_id = parent.share_id
        self.m.fs.create(path=path, share_id=share_id, is_dir=False)
        self.key.set(local_hash=empty_hash)
        self.key.set(server_hash=empty_hash)
        self.key.sync()
        name = os.path.basename(path)
        marker = MDMarker(self.key.get_mdid())
        self.m.action_q.make_file(share_id, parent_id, name, marker)

    def new_local_file_created(self, event, parms, new_id):
        """we got the server answer for the file creation."""
        self.m.fs.set_node_id(self.key['path'], new_id)


    def new_local_dir(self, event, parms, path):
        """a new local dir was created"""
        # XXX: lucio.torre: we should use markers here
        parent_path = os.path.dirname(path)
        parent = self.m.fs.get_by_path(parent_path)
        parent_id = parent.node_id or MDMarker(parent.mdid)
        share_id = parent.share_id
        self.m.fs.create(path=path, share_id=share_id, is_dir=True)
        name = os.path.basename(path)
        mdid = self.key.get_mdid()
        marker = MDMarker(mdid)
        self.m.action_q.make_dir(share_id, parent_id, name, marker)
        self.m.lr.scan_dir(mdid, path)

    def new_local_dir_created(self, event, parms, new_id):
        """Server answered that dir creation was ok."""
        self.m.fs.set_node_id(self.key['path'], new_id)

        # query to get any updates in case dir was already there in server
        self.m.action_q.query([(self.key['share_id'], self.key['node_id'],
                                self.key['local_hash'])])

    def calculate_hash(self, event, params):
        """calculate the hash of this."""
        self.m.hash_q.insert(self.key['path'], self.key['mdid'])

    def rescan_dir(self, event, parms, udfmode):
        """Starts the scan again on a dir."""
        self.m.lr.scan_dir(self.key['mdid'], self.key['path'], udfmode)

    def put_file(self, event, params, hash, crc32, size, stat):
        """upload the file to the server."""
        previous_hash = self.key['server_hash']
        self.key.set(local_hash=hash, stat=stat)
        self.key.sync()

        self.m.action_q.upload(share_id=self.key['share_id'],
            node_id=self.key['node_id'], previous_hash=previous_hash,
            hash=hash, crc32=crc32, size=size,
            fileobj_factory=self.key.open_file)

    def converges_to_server(self, event, params, hash, crc32, size, stat):
        """the local changes now match the server"""
        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                    node_id=self.key['node_id'])
        self.key.remove_partial()
        self.key.set(local_hash=hash, stat=stat)
        self.key.sync()

    def reput_file_from_ok(self, event, param, hash):
        """put the file again, mark upload as ok"""
        self.m.action_q.cancel_upload(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.set(local_hash=hash)
        self.key.set(server_hash=hash)
        self.key.sync()
        self.m.hash_q.insert(self.key['path'], self.key['mdid'])


    def reput_file(self, event, param, hash, crc32, size, stat):
        """put the file again."""
        self.m.action_q.cancel_upload(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        previous_hash = self.key['server_hash']

        self.key.set(local_hash=hash, stat=stat)
        self.key.sync()
        self.m.action_q.upload(share_id=self.key['share_id'],
            node_id=self.key['node_id'], previous_hash=previous_hash,
            hash=hash, crc32=crc32, size=size,
            fileobj_factory=self.key.open_file)

    def server_file_now_matches(self, event, params, hash):
        """We got a server hash that matches local hash"""
        self.m.action_q.cancel_upload(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.set(server_hash=hash)
        self.key.sync()

    def commit_upload(self, event, params, hash):
        """Finish an upload."""
        self.key.upload_finished(hash)

    def cancel_and_commit(self, event, params, hash):
        """Finish an upload."""
        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.remove_partial()
        self.key.upload_finished(hash)

    def delete_file(self, event, params, *args, **kwargs):
        """server file was deleted."""
        try:
            self.key.delete_file()
        except OSError, e:
            if e.errno == 39:
                # if directory not empty
                self.key.move_to_conflict()
                self.key.delete_metadata()
            elif e.errno == 2:
                # file gone
                pass
            else:
                raise e

    def conflict_and_delete(self, event, params, *args, **kwargs):
        """move to conflict and delete file."""
        self.key.move_to_conflict()
        self.key.delete_metadata()

    def file_gone_wile_downloading(self, event, params):
        """a file we were downloading is gone."""
        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.remove_partial()
        self.delete_file(event, params)

    def file_not_created_remove(self, event, params, error):
        """kill it"""
        self.key.move_to_conflict()
        self.key.delete_metadata()

    def dir_not_created_in_server(self, event, params, error):
        """Re-get the dir if it was because already exist."""
        if error == "ALREADY_EXISTS":
            # delete metadata and query the father to converge
            parent = self.m.fs.get_by_node_id(self.key['share_id'],
                                              self.key['parent_id'])
            self.key.delete_metadata()
            self.m.action_q.query([(parent.share_id,
                                    parent.node_id, parent.local_hash)])
        else:
            self.key.move_to_conflict()
            self.key.delete_metadata()

    def delete_on_server(self, event, params, path):
        """local file was deleted."""
        self.m.action_q.unlink(self.key['share_id'],
                               self.key['parent_id'],
                               self.key['node_id'])
        self.key.delete_to_trash()

    def deleted_dir_while_downloading(self, event, params, path):
        """kill it"""
        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                                        node_id=self.key['node_id'])
        self.key.remove_partial()
        self.m.action_q.unlink(self.key['share_id'],
                               self.key['parent_id'],
                               self.key['node_id'])
        self.key.delete_to_trash()

    def cancel_download_and_delete_on_server(self, event, params, path):
        """cancel_download_and_delete_on_server"""
        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                                        node_id=self.key['node_id'])
        self.key.remove_partial()
        self.m.action_q.unlink(self.key['share_id'],
                               self.key['parent_id'],
                               self.key['node_id'])
        self.key.delete_to_trash()

    def cancel_upload_and_delete_on_server(self, event, params, path):
        """cancel_download_and_delete_on_server"""
        self.m.action_q.cancel_upload(share_id=self.key['share_id'],
                                      node_id=self.key['node_id'])
        self.m.action_q.unlink(self.key['share_id'],
                               self.key['parent_id'],
                               self.key['node_id'])
        self.key.delete_to_trash()

    def remove_trash(self, event, params, share_id, node_id):
        """Remove the node from trash."""
        self.key.remove_from_trash(share_id, node_id)

    def server_moved(self, event, params, share_id, node_id,
                     new_share_id, new_parent_id, new_name):
        """file was moved on the server"""
        self.key.move_file(new_share_id, new_parent_id, new_name)

    def server_moved_dirty(self, event, params, share_id, node_id,
                     new_share_id, new_parent_id, new_name):
        """file was moved on the server while downloading it"""
        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.remove_partial()
        self.key.move_file(new_share_id, new_parent_id, new_name)
        self.get_file(event, params, self.key['server_hash'])

    def moved_dirty_local(self, event, params, path_from, path_to):
        """file was moved while uploading it"""
        self.m.action_q.cancel_upload(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.set(local_hash=self.key['server_hash'])
        self.key.sync()
        self.client_moved(event, params, path_from, path_to)

    def moved_dirty_server(self, event, params, path_from, path_to):
        """file was moved while downloading it"""
        self.client_moved(event, params, path_from, path_to)

        self.m.action_q.cancel_download(share_id=self.key['share_id'],
                            node_id=self.key['node_id'])
        self.key.remove_partial()
        self.key.set(server_hash=self.key['local_hash'])
        self.key.sync()
        self.m.action_q.query([(self.key['share_id'],
                                self.key['node_id'],
                                self.key['local_hash'] or "")])

    # pylint: disable-msg=C0103
    def DESPAIR(self, event, params, *args, **kwargs):
        """if we got here, we are in trouble"""
        self.log.error("DESPAIR on event=%s params=%s args=%s kwargs=%s",
                                                event, params, args, kwargs)

    def save_stat(self, event, params, hash, crc32, size, stat):
        """Save the stat"""
        self.key.set(stat=stat)
        self.key.sync()

    def remove_partial(self, event, params, error=None, server_hash=''):
        """remove the .partial file"""
        self.key.remove_partial()
        local_hash = self.key['local_hash']
        self.key.set(server_hash=local_hash, local_hash=local_hash)
        self.key.sync()


class Sync(object):
    """Translates from EQ events into state machine events."""
    # XXX: lucio.torre:
    # this will need some refactoring once we handle more events

    def __init__(self, main):
        """create"""
        # XXX: verterok: add a custom Logger to lazyily build the LogRecord,
        # now that the DebugCapture is enabled
        self.logger = logging.getLogger('ubuntuone.SyncDaemon.sync')
        self.fsm = StateMachine(u1fsfsm.state_machine)
        self.m = main
        self.m.event_q.subscribe(self)

    def handle_SV_HASH_NEW(self, share_id, node_id, hash):
        """on SV_HASH_NEW"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.signal_event_with_hash("SV_HASH_NEW", hash)

    def handle_SV_FILE_NEW(self, share_id, node_id, parent_id, name):
        """on SV_FILE_NEW"""
        parent = FSKey(self.m.fs, share_id=share_id, node_id=parent_id)
        path = os.path.join(parent["path"], name)
        key = FSKey(self.m.fs, path=path)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("SV_FILE_NEW", {}, share_id, node_id, parent_id, name)

    def handle_SV_DIR_NEW(self, share_id, node_id, parent_id, name):
        """on SV_DIR_NEW"""
        parent = FSKey(self.m.fs, share_id=share_id, node_id=parent_id)
        path = os.path.join(parent["path"], name)
        key = FSKey(self.m.fs, path=path)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("SV_DIR_NEW", {}, share_id, node_id, parent_id, name)

    def handle_SV_FILE_DELETED(self, share_id, node_id):
        """on SV_FILE_DELETED"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("SV_FILE_DELETED", {})

    def handle_AQ_DOWNLOAD_FINISHED(self, share_id, node_id, server_hash):
        """on AQ_DOWNLOAD_FINISHED"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.signal_event_with_hash("AQ_DOWNLOAD_FINISHED", server_hash)

    def handle_AQ_DOWNLOAD_ERROR(self, share_id, node_id, server_hash, error):
        """on AQ_DOWNLOAD_ERROR"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.signal_event_with_error_and_hash("AQ_DOWNLOAD_ERROR", error,
                                              server_hash)

    def handle_AQ_DOWNLOAD_CANCELLED(self, share_id, node_id, server_hash):
        """on AQ_DOWNLOAD_CANCELLED"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.signal_event_with_hash("AQ_DOWNLOAD_CANCELLED", server_hash)

    def handle_AQ_DOWNLOAD_DOES_NOT_EXIST(self, share_id, node_id):
        """on AQ_DOWNLOAD_DOES_NOT_EXIST."""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("AQ_DOWNLOAD_DOES_NOT_EXIST", {}, share_id, node_id)

    def handle_FS_FILE_CREATE(self, path):
        """on FS_FILE_CREATE"""
        key = FSKey(self.m.fs, path=path)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("FS_FILE_CREATE", {}, path)

    def handle_FS_DIR_CREATE(self, path):
        """on FS_DIR_CREATE"""
        key = FSKey(self.m.fs, path=path)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("FS_DIR_CREATE", {}, path)

    def handle_FS_FILE_DELETE(self, path):
        """on FS_FILE_DELETE"""
        key = FSKey(self.m.fs, path=path)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("FS_FILE_DELETE", {}, path)

    def handle_FS_DIR_DELETE(self, path):
        """on FS_DIR_DELETE"""
        key = FSKey(self.m.fs, path=path)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("FS_DIR_DELETE", {}, path)

    def handle_FS_FILE_MOVE(self, path_from, path_to):
        """on FS_FILE_MOVE"""
        key = FSKey(self.m.fs, path=path_from)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("FS_FILE_MOVE", {}, path_from, path_to)

    def handle_FS_DIR_MOVE(self, path_from, path_to):
        """on FS_DIR_MOVE"""
        key = FSKey(self.m.fs, path=path_from)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("FS_DIR_MOVE", {}, path_from, path_to)

    def handle_AQ_FILE_NEW_OK(self, marker, new_id):
        """on AQ_FILE_NEW_OK"""
        key = FSKey(self.m.fs, mdid=marker)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("AQ_FILE_NEW_OK", {}, new_id)

    def handle_AQ_FILE_NEW_ERROR(self, marker, error):
        """on AQ_FILE_NEW_ERROR"""
        key = FSKey(self.m.fs, mdid=marker)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.signal_event_with_error("AQ_FILE_NEW_ERROR", error)

    def handle_AQ_DIR_NEW_ERROR(self, marker, error):
        """on AQ_DIR_NEW_ERROR"""
        key = FSKey(self.m.fs, mdid=marker)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.signal_event_with_error("AQ_DIR_NEW_ERROR", error)

    def handle_AQ_DIR_NEW_OK(self, marker, new_id):
        """on AQ_DIR_NEW_OK"""
        key = FSKey(self.m.fs, mdid=marker)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("AQ_DIR_NEW_OK", {}, new_id)

    def handle_FS_FILE_CLOSE_WRITE(self, path):
        """on FS_FILE_CLOSE_WRITE"""
        key = FSKey(self.m.fs, path=path)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event('FS_FILE_CLOSE_WRITE', {})

    def handle_LR_SCAN_ERROR(self, mdid, udfmode):
        """on LR_SCAN_ERROR"""
        key = FSKey(self.m.fs, mdid=mdid)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event('LR_SCAN_ERROR', {}, udfmode)

    def handle_HQ_HASH_ERROR(self, mdid):
        """on HQ_HASH_ERROR"""
        key = FSKey(self.m.fs, mdid=mdid)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event('HQ_HASH_ERROR', {})

    def handle_HQ_HASH_NEW(self, path, hash, crc32, size, stat):
        """on HQ_HASH_NEW"""
        key = FSKey(self.m.fs, path=path)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        try:
            data_ok = ssmr.validate_actual_data(path, stat)
        except OSError, e:
            # the file went away between the moment HQ finished and now, we
            # discard the info, but needs to send to rehash.
            log.debug("Changing HQ_HASH_NEW to ERROR in %r: %s", path, e)
            ssmr.on_event('HQ_HASH_ERROR', {})
        else:
            if data_ok:
                ssmr.signal_event_with_hash("HQ_HASH_NEW", hash,
                                            crc32, size, stat)

    def handle_AQ_UPLOAD_FINISHED(self, share_id, node_id, hash):
        """on AQ_UPLOAD_FINISHED"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.signal_event_with_hash("AQ_UPLOAD_FINISHED", hash)

    def handle_AQ_UPLOAD_ERROR(self, share_id, node_id, error, hash):
        """on AQ_UPLOAD_ERROR"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.signal_event_with_error_and_hash("AQ_UPLOAD_ERROR", error, hash)

    def handle_SV_MOVED(self, share_id, node_id, new_share_id, new_parent_id,
                        new_name):
        """on SV_MOVED"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("SV_MOVED", {}, share_id, node_id, new_share_id,
                      new_parent_id, new_name)

    def handle_AQ_UNLINK_OK(self, share_id, parent_id, node_id):
        """on AQ_UNLINK_OK"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("AQ_UNLINK_OK", {}, share_id, node_id)

    def handle_AQ_UNLINK_ERROR(self, share_id, parent_id, node_id, error):
        """on AQ_UNLINK_ERROR"""
        key = FSKey(self.m.fs, share_id=share_id, node_id=node_id)
        log = FileLogger(self.logger, key)
        ssmr = SyncStateMachineRunner(self.fsm, self.m, key, log)
        ssmr.on_event("AQ_UNLINK_ERROR", {}, share_id, node_id)

