# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""tests for the syncdaemon tools module """
import os
import uuid

from ubuntuone.syncdaemon import (
    dbus_interface,
    event_queue,
    tools,
    volume_manager,
    states,
)
from tests.syncdaemon.test_dbus import FakeCommand
from contrib.testing.testcase import (
    DBusTwistedTestCase,
)

from twisted.internet import defer, reactor


class TestToolsBase(DBusTwistedTestCase):
    """ Base test case for SyncDaemonTool tests"""

    def setUp(self):
        DBusTwistedTestCase.setUp(self)
        self.tool = tools.SyncDaemonTool(self.bus)
        self.home_dir = self.mktemp('ubuntuonehacker')
        self._old_home = os.environ['HOME']
        os.environ['HOME'] = self.home_dir
        return self.main.wait_for_nirvana()

    def tearDown(self):
        os.environ['HOME'] = self._old_home
        self.rmtree(self.home_dir)
        return DBusTwistedTestCase.tearDown(self)

    def create_file(self, path):
        """ creates a test file in fsm """
        share_path = os.path.join(self.shares_dir, 'share_tools')
        self.main.vm.add_share(volume_manager.Share(share_path,
                                                    volume_id='tools_share_id'))
        self.fs_manager.create(path, "tools_share_id")
        self.fs_manager.set_node_id(path, "node_id")
        return 'tools_share_id', 'node_id'


class TestToolsBasic(TestToolsBase):
    """ Baic test of SyncDaemonTool """

    def test_wait_connected(self):
        """ test wait_connected """
        self.action_q.connect()
        d = self.tool.wait_connected()
        # test callback, pylint: disable-msg=C0111
        def connected(result):
            self.assertEquals(True, result)
        d.addBoth(connected)
        return d

    def test_wait_no_more_events(self):
        """ test wait_no_more_events """
        d = self.tool.wait_no_more_events(last_event_interval=.1)
        # test callback, pylint: disable-msg=C0111
        event_names = event_queue.EVENTS.keys()
        def events(result):
            self.assertEquals(True, result)

        def fire_events():
            # unsubscribe the vm subscribed in FakeMain as we are going to push
            # fake events
            self.event_q.unsubscribe(self.main.vm)
            for event_name in event_names:
                args = event_queue.EVENTS[event_name]
                self.event_q.push(event_name, *args)

        reactor.callLater(0, self.action_q.connect)
        reactor.callLater(0, fire_events)
        d.addBoth(events)
        d.addCallback(lambda _: self.event_q.subscribe(self.main.vm))
        return d

    def test_all_downloads(self):
        """ test wait_all_downloads """
        d = self.tool.wait_all_downloads()

        # test callback, pylint: disable-msg=C0111
        def downloads(result):
            self.assertEquals(True, result)
        d.addBoth(downloads)
        return d

    def test_all_uploads(self):
        """ test wait_all_uploads """
        d = self.tool.wait_all_uploads()

        # test callback, pylint: disable-msg=C0111
        def uploads(result):
            self.assertEquals(True, result)
        d.addBoth(uploads)
        return d

    def test_wait_for_nirvana(self):
        """ test wait_for_nirvana """
        # setup States to be in Nirvana condition
        self.main.state_manager.state = states.StateManager.QUEUE_MANAGER
        self.main.state_manager.queues.state = states.QueueManager.IDLE

        # unsubscribe VM and States to not receive everything
        self.event_q.unsubscribe(self.main.vm)
        self.event_q.unsubscribe(self.main.state_manager)
        d = self.tool.wait_for_nirvana(last_event_interval=.1)

        # test callback, pylint: disable-msg=C0111
        def callback(result):
            self.assertEquals(True, result)
        d.addBoth(callback)

        # clear downloading
        reactor.callLater(0, self.action_q.connect)

        def fire_events():
            for event_name in event_queue.EVENTS.keys():
                args = event_queue.EVENTS[event_name]
                self.event_q.push(event_name, *args)

        # fire fake events to keep the deferred running
        reactor.callLater(0, fire_events)
        # 1 sec later, clear the download queue, and wait to reach nirvana
        d.addCallback(lambda _: self.event_q.subscribe(self.main.vm))
        return d

    def test_get_metadata(self):
        """ check that get_metadata works as expected """
        path = os.path.join(self.root_dir, "foo")
        self.fs_manager.create(path, "")
        self.fs_manager.set_node_id(path, "node_id")
        d = self.tool.get_metadata(path)
        # the callback, pylint: disable-msg=C0111
        def callback(result):
            self.assertEquals(path, str(result['path']))
            self.assertEquals('', str(result['share_id']))
            self.assertEquals('node_id', result['node_id'])
        d.addCallback(callback)
        return d

    def test_quit(self):
        """test the quit method."""
        # helper functions, we need to call quit but don't quit
        # pylint: disable-msg=C0111,W0601,W0602
        quitted = False
        def fake_quit():
            global quitted
            quitted = True
        self.main.quit = fake_quit
        d = self.tool.quit()
        # test callback, pylint: disable-msg=C0111
        def check(result):
            global quitted
            self.assertTrue(quitted)
        d.addBoth(check)
        return d

    def test_accept_share(self):
        """Test accept_share method"""
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(volume_manager.Share(path=share_path,
                                     volume_id='share_id', access_level='Read',
                                     accepted=False, node_id="node_id"))
        self.assertEquals(False, self.main.vm.shares['share_id'].accepted)
        d = self.tool.accept_share('share_id')
        def check(result):
            """do the asserts"""
            self.assertEquals('Yes', result['answer'])
            self.assertEquals('share_id', result['volume_id'])
            self.assertEquals(True, self.main.vm.shares['share_id'].accepted)

        d.addCallback(check)
        return d

    def test_reject_share(self):
        """Test the reject_share method"""
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(volume_manager.Share(path=share_path,
                                     volume_id='share_id', access_level='Read',
                                     accepted=False))
        self.assertEquals(False, self.main.vm.shares['share_id'].accepted)
        d = self.tool.reject_share('share_id')
        def check(result):
            """do the asserts"""
            self.assertEquals('No', result['answer'])
            self.assertEquals('share_id', result['volume_id'])
            self.assertEquals(False, self.main.vm.shares['share_id'].accepted)

        d.addCallback(check)
        return d

    def test_wait_for_signal(self):
        """Test wait_for_signal method"""
        d = self.tool.wait_for_signal('Foo', lambda _: True)
        def check(result):
            """do the assert"""
            self.assertEquals(True, result)
        d.addCallback(check)
        client = tools.DBusClient(self.bus, '/events',
                                  dbus_interface.DBUS_IFACE_EVENTS_NAME)
        client.send_signal('Foo', True)
        return d

    def test_wait_for_signal_failure(self):
        """Test (with error) wait_for_signal method"""
        def filter(value):
            """broken filter"""
            raise ValueError('DIE!!!!')
        d = self.tool.wait_for_signal('Foo', filter)
        def check(result):
            """do the assert"""
            self.assertEquals('DIE!!!!', result.getErrorMessage())
            self.assertEquals(ValueError, result.type)
        d.addErrback(check)
        client = tools.DBusClient(self.bus, '/events',
                                  dbus_interface.DBUS_IFACE_EVENTS_NAME)
        client.send_signal('Foo', True)
        return d

    def test_wait_for_signals_ok(self):
        """Test wait_for_signals method"""
        d = self.tool.wait_for_signals('FooOk','BarError')
        def check(result):
            """do the assert"""
            self.assertEquals(True, result[0])
        d.addCallback(check)
        client = tools.DBusClient(self.bus, '/folders',
                                  dbus_interface.DBUS_IFACE_FOLDERS_NAME)
        client.send_signal('FooOk', True)
        return d

    def test_wait_for_signals_error(self):
        """Test (with error) wait_for_signals method"""
        d = self.tool.wait_for_signals('FooOk', 'BarError')
        def handle_error(error):
            """do the assert"""
            errorname, errorargs = error.value.args
            self.assertEquals('BarError', errorname)
        d.addCallbacks(self.fail, handle_error)
        client = tools.DBusClient(self.bus, '/folders',
                                  dbus_interface.DBUS_IFACE_FOLDERS_NAME)
        client.send_signal('BarError', True)
        return d

    def test_connect(self):
        """Test the connect method"""
        self.assertEquals(self.main.state_manager.state,
                          states.StateManager.QUEUE_MANAGER)
        d = self.main.wait_for('SYS_USER_DISCONNECT')
        self.main.dbus_iface.disconnect()
        def connect(r):
            d = self.main.wait_for('SYS_USER_CONNECT')
            self.tool.connect()
            d.addCallbacks(lambda x: x, self.fail)
            return d
        d.addCallbacks(connect, self.fail)
        return d

    def test_disconnect(self):
        """Test the disconnect method"""
        self.assertEquals(self.main.state_manager.state,
                          states.StateManager.QUEUE_MANAGER)
        d = self.main.wait_for('SYS_USER_DISCONNECT')
        self.tool.disconnect()
        d.addCallbacks(self.assertFalse, self.fail)
        return d

    def test_get_status(self):
        """Test the status method"""
        d = self.tool.get_status()
        def handler(result):
            state = states.StateManager.QUEUE_MANAGER
            self.assertEquals(state.name, result['name'])
            self.assertEquals(state.description, result['description'])
            self.assertEquals(state.is_error, bool(result['is_error']))
            self.assertEquals(state.is_connected, bool(result['is_connected']))
            self.assertEquals(state.is_online, bool(result['is_online']))
        d.addCallbacks(handler, self.fail)
        return d

    def test_waiting_metadata(self):
        """Test SyncDaemonTool.waiting_metadata."""
        # inject the fake data
        self.action_q.meta_queue.waiting.extend([
                FakeCommand("node_a_foo", "node_a_bar"),
                FakeCommand("node_b_foo", "node_b_bar")])
        d = self.tool.waiting_metadata()
        def check(result):
            """waiting_metadata reply handler."""
            self.assertEquals(2, len(result))
            # the second time we're called, the result should be reversed
            node_a, node_b = result
            self.assertEquals(str(FakeCommand("node_a_foo", "node_a_bar")),
                              node_a)
            self.assertEquals(str(FakeCommand("node_b_foo", "node_b_bar")),
                              node_b)
        return d

    def test_waiting_content_schedule_next(self):
        """Test waiting_content and schedule_next"""
        path = os.path.join(self.root_dir, "foo")
        self.fs_manager.create(path, "")
        self.fs_manager.set_node_id(path, "node_id")
        path1 = os.path.join(self.root_dir, "bar")
        self.fs_manager.create(path1, "")
        self.fs_manager.set_node_id(path1, "node_id_1")
        # inject the fake data
        self.action_q.content_queue.waiting.extend([
                FakeCommand("", "node_id"),
                FakeCommand("", "node_id_1")])

        d = self.tool.waiting_content()
        self.second = False
        def handler(result):
            if self.second:
                node_1, node = result
            else:
                node, node_1 = result
            self.assertEquals(path, str(node['path']))
            self.assertEquals(path1, str(node_1['path']))
            self.assertEquals('', str(node['share']))
            self.assertEquals('', str(node_1['share']))
            self.assertEquals('node_id', str(node['node']))
            self.assertEquals('node_id_1', str(node_1['node']))
            self.assertTrue(result)
        d.addCallbacks(handler, self.fail)
        def second(_):
            self.second = True
        d.addCallback(second)
        d.addCallback(lambda _: self.tool.schedule_next('', 'node_id_1'))
        d.addCallbacks(lambda _: self.tool.waiting_content(), self.fail)
        d.addCallbacks(handler, self.fail)
        return d

    def test_start(self):
        """Test start method"""
        # first quit the currently running client
        d = self.tool.start()
        d.addCallbacks(lambda r: self.assertEquals(r, None), self.fail)
        return d

    def test_create_folder(self):
        """Test for Folders.create."""
        path = os.path.join(self.home_dir, u'ñoño')
        id = uuid.uuid4()
        node_id = uuid.uuid4()
        d = defer.Deferred()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf"""
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=id,
                                                       node_id=node_id,
                                                       marker=marker))
        self.main.action_q.create_udf = create_udf

        d = self.tool.create_folder(path)
        return d

    @defer.inlineCallbacks
    def test_delete_folder(self):
        """Test for Folders.delete."""
        path = os.path.join(self.home_dir, 'ñoño')
        # patch AQ.delete_volume
        def delete_volume(volume_id):
            """Fake delete_volume"""
            self.main.event_q.push("AQ_DELETE_VOLUME_OK", volume_id="folder_id")
        self.main.action_q.delete_volume = delete_volume

        # create sample folder
        suggested_path = os.path.join("~", 'ñoño')
        udf = volume_manager.UDF("folder_id", "node_id", suggested_path, path,
                                 subscribed=True)
        yield self.main.vm.add_udf(udf)

        d = self.tool.delete_folder("folder_id")
        yield d

    def _create_udf(self, id, node_id, suggested_path, subscribed=True):
        """Create an UDF and returns it and the volume"""
        path = self.main.vm._build_udf_path(suggested_path)
        udf = volume_manager.UDF(str(id), str(node_id), suggested_path, path,
              subscribed)
        return udf

    def test_subscribe_folder(self):
        """Test for Folders.subscribe and that it fires a dbus signal."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=False)
        self.main.vm.add_udf(udf)
        d = self.tool.subscribe_folder(udf.id)

        def check(r):
            """Check that the folder is subscribed."""
            self.assertTrue(self.main.vm.udfs[udf.id].subscribed,
                            "UDF %s isn't subscribed" % udf.id)
        d.addCallback(check)
        return d

    def test_unsubscribe_folder(self):
        """Test for Folders.unsubscribe."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=True)
        self.main.vm.add_udf(udf)
        d = self.tool.unsubscribe_folder(udf.id)

        def check(r):
            """Check that the folder is not subscribed."""
            self.assertFalse(self.main.vm.udfs[udf.id].subscribed,
                            "UDF %s is subscribed" % udf.id)
        d.addCallback(check)
        return d

    def test_change_public_access(self):
        """Test change_public_access."""
        node_id = str(uuid.uuid4())
        share_id = ""
        path = os.path.join(self.root_dir, "foo")
        self.fs_manager.create(path, "")
        self.fs_manager.set_node_id(path, node_id)

        def change_public_access(share_id, node_id, is_public):
            """Fake change_public_access"""
            self.main.event_q.push("AQ_CHANGE_PUBLIC_ACCESS_OK",
                                   share_id=share_id, node_id=node_id,
                                   is_public=True,
                                   public_url='http://example.com')
        self.main.action_q.change_public_access = change_public_access

        d = self.tool.change_public_access(path, True)
        def check(file_info):
            """Check the returned file info dictionary."""
            self.assertEqual(share_id, file_info['share_id'])
            self.assertEqual(node_id, file_info['node_id'])
            self.assertEqual('True', file_info['is_public'])
            self.assertEqual('http://example.com', file_info['public_url'])
        d.addCallback(check)
        return d

    @defer.inlineCallbacks
    def test_get_throttling_limits(self):
        """Test for get_throttling_limits."""
        limits = yield self.tool.get_throttling_limits()
        self.assertEquals(-1, limits[u'download'])
        self.assertEquals(-1, limits[u'upload'])

    @defer.inlineCallbacks
    def test_set_throttling_limits(self):
        """Test for set_throttling_limits."""
        yield self.tool.set_throttling_limits(10, 20)
        limits = yield self.tool.get_throttling_limits()
        self.assertEquals(10, limits[u'download'])
        self.assertEquals(20, limits[u'upload'])

    @defer.inlineCallbacks
    def test_is_throttling_enabled(self):
        """Test for is_throttling_enabled."""
        enabled = yield self.tool.is_throttling_enabled()
        self.assertFalse(enabled)
        self.main.action_q.enable_throttling()
        enabled = yield self.tool.is_throttling_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_enable_throttling(self):
        """Test for enable_throttling."""
        yield self.tool.enable_throttling(True)
        enabled = self.main.action_q.throttling_enabled
        self.assertTrue(enabled)
        yield self.tool.enable_throttling(False)
        enabled = self.main.action_q.throttling_enabled
        self.assertFalse(enabled)

