# test_key_acls - tests for ubuntuone.oauthdesktop.key_acls
#
# Author: Stuart Langridge <stuart.langridge@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the ACL setting code for ubuntuone-oauth-login."""

import os, StringIO
import xdg.BaseDirectory
from contrib.mocker import MockerTestCase, IN, ANY
from ubuntuone.oauthdesktop.key_acls import (
  get_privileged_config_folder, get_acl_preregistrations, set_all_key_acls,
  set_single_acl)

class Loader(MockerTestCase):
    """Confirm only the /etc/xdg config files are loaded"""

    def test_etc_found(self):
        """Is the /etc/xdg folder found if present?"""
        
        ETC_FOLDER = "/etc/xdg/ubuntuone"
        osp = self.mocker.replace("os.path")
        osp.exists(ETC_FOLDER)
        self.mocker.result(True)
        self.mocker.replay()
        
        self.assertEqual(ETC_FOLDER,
          get_privileged_config_folder(use_source_tree_folder=False))

    def test_etc_found_despite_home(self):
        """Is the /etc/xdg folder found even if the $HOME folder is present?"""
        
        ETC_FOLDER = "/etc/xdg/ubuntuone"
        HOME_FOLDER = os.path.join(xdg.BaseDirectory.xdg_config_home, 
          "ubuntuone")
        osp = self.mocker.replace("os.path")
        osp.exists(ETC_FOLDER)
        self.mocker.result(True)
        osp.exists(HOME_FOLDER)
        self.mocker.result(True)
        self.mocker.replay()
        
        self.assertEqual(ETC_FOLDER,
          get_privileged_config_folder(use_source_tree_folder=False))

    def test_etc_files_found(self):
        """Are files in /etc found?"""
        ETC_FOLDER = "/etc/xdg/ubuntuone"
        ETC_FILES_FOLDER = "/etc/xdg/ubuntuone/oauth_registration.d"
        FILE_LIST = ["a", "b"]
        osp = self.mocker.replace("os.path")
        osp.exists(ETC_FOLDER)
        self.mocker.result(True)
        osp.isdir(ETC_FILES_FOLDER)
        self.mocker.result(True)
        listdir = self.mocker.replace("os.listdir")
        listdir(ETC_FILES_FOLDER)
        self.mocker.result(FILE_LIST)
        self.mocker.replay()
        
        our_list = sorted(
          [os.path.join(ETC_FILES_FOLDER, x) for x in FILE_LIST])
        their_list = sorted(
          get_acl_preregistrations(use_source_tree_folder=False))
        self.assertEqual(our_list, their_list)
        
    def test_set_single_acl(self):
        """Does set_single_acl work?"""

        giifr = self.mocker.replace(
          "ubuntuone.oauthdesktop.key_acls.get_item_ids_for_realm")
        igas = self.mocker.replace("gnomekeyring.item_get_acl_sync")
        isas = self.mocker.replace("gnomekeyring.item_set_acl_sync")
        acon = self.mocker.replace("gnomekeyring.AccessControl")
        self.expect(giifr("realm", "consumer_key")).result([999])
        self.expect(igas(None, 999)).result([])
        ac1 = self.mocker.mock()
        self.expect(acon(ANY, ANY)).result(ac1)
        ac1.set_display_name("application_name")
        ac1.set_path_name("/tmp/exe_path")
        self.expect(isas(None, 999, [ac1])).result(None)
        self.expect(giifr("realm2", "consumer_key2")).result([9999])
        self.expect(igas(None, 9999)).result([])
        ac2 = self.mocker.mock()
        self.expect(acon(ANY, ANY)).result(ac2)
        ac2.set_display_name("application_name2")
        ac2.set_path_name("/tmp/exe_path2")
        self.expect(isas(None, 9999, [ac2])).result(None)
        self.mocker.replay()

        set_single_acl([
          ("realm", "consumer_key", "/tmp/exe_path", "application_name"),
          ("realm2", "consumer_key2", "/tmp/exe_path2", "application_name2"),
        ])
        
        
    def test_etc_files_parsed(self):
        """Are files in /etc parsed correctly?"""
        
        ETC_FOLDER = "/etc/xdg/ubuntuone"
        ETC_FILES_FOLDER = "/etc/xdg/ubuntuone/oauth_registration.d"
        FILE_LIST = ["a", "b"]
        osp = self.mocker.replace("os.path")
        osp.exists(ETC_FOLDER)
        self.mocker.result(True)
        osp.isdir(ETC_FILES_FOLDER)
        self.mocker.result(True)
        listdir = self.mocker.replace("os.listdir")
        listdir(ETC_FILES_FOLDER)
        self.mocker.result(FILE_LIST)

        sio1 = StringIO.StringIO("""[app1]
realm = https://realm.example.com/
consumer_key = example_key
exe_path = /nowhere/executable/path
application_name = example_app_name

[app2]
realm = https://other.example.com/
consumer_key = example_key
exe_path = /nowhere/executable/path
application_name = example_app_name

""")
        sio2 = StringIO.StringIO("""[app3]
exe_path = /nowhere/path/2
application_name = example_app_name2
consumer_key = example_key2
realm = https://realm2.example.com/
""")
        mock_open = self.mocker.replace(open)
        mock_open(os.path.join(ETC_FILES_FOLDER, "a"))
        self.mocker.result(sio1)
        mock_open(os.path.join(ETC_FILES_FOLDER, "b"))
        self.mocker.result(sio2)
        
        ssa = self.mocker.replace(
          "ubuntuone.oauthdesktop.key_acls.set_single_acl")
        # list may come up in any order
        ssa(IN([
              [
                ("https://realm.example.com/", "example_key", 
                "/nowhere/executable/path", "example_app_name"),
                ("https://other.example.com/", "example_key", 
                "/nowhere/executable/path", "example_app_name"),
              ],
              [
                ("https://other.example.com/", "example_key", 
                "/nowhere/executable/path", "example_app_name"),
                ("https://realm.example.com/", "example_key", 
                "/nowhere/executable/path", "example_app_name"),
              ],
            ]), specific_item_id=None
        )
        self.mocker.result(None)
        ssa([
            ("https://realm2.example.com/", "example_key2", 
            "/nowhere/path/2", "example_app_name2")
            ], specific_item_id=None)
        self.mocker.result(None)
        self.mocker.replay()
        
        set_all_key_acls(use_source_tree_folder=False)
    
    
    
    
    
    
