/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *  @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#include "config.h"

#include <QDebug>
#include "QtTapioca/Handle"

using namespace QtTapioca;

Handle::Handle(org::freedesktop::Telepathy::Connection *conn, uint type, const QString &name, QObject *parent)
    : QObject(parent),
      m_conn(conn),
      m_type(type),
      m_id(request(name)),
      m_isValid(!((m_type != 0) && (m_id == 0)))
{
    /* no need to hold the handle as we are requesting it */
}

Handle::Handle(org::freedesktop::Telepathy::Connection *conn, uint type, uint id, QObject *parent)
    : QObject(parent),
      m_conn(conn),
      m_type(type),
      m_id(id),
      m_isValid(!((m_type != 0) && (m_id == 0)))
{
    hold();
}

Handle::~Handle()
{
    release();
}

bool Handle::isValid() const
{
    return m_isValid;
}

QString Handle::inspect()
{
    if (!m_isValid)
       return QString("Unknown@localhost");

    QList<uint> handleList;
    handleList << m_id;

    if (m_conn.isNull()) { return QString(); }

    QDBusReply<QStringList> reply = m_conn->InspectHandles(static_cast<uint>(m_type), handleList);

    if (!reply.isValid()) {
        qDebug() << "error inspecting handles:" << reply.error().message();
        return QString();
    }

    return (reply.value().isEmpty() ? "" : reply.value().first());
}

uint Handle::id() const
{
    return m_id;
}

uint Handle::type() const
{
    return m_type;
}

void Handle::hold()
{
    if (!m_isValid || m_conn.isNull())
        return;

    QList<uint> handleList;
    handleList << m_id;
    QDBusReply<void> reply = m_conn->HoldHandles(static_cast<uint>(m_type), handleList);

    if (!reply.isValid())
        qDebug() << "error holding handles:" << reply.error().message();
}

void Handle::release()
{
    if (!m_isValid || m_conn.isNull())
        return;

    QList<uint> handleList;
    handleList << m_id;
    QDBusReply<void> reply = m_conn->ReleaseHandles(static_cast<uint>(m_type), handleList);

    if(!reply.isValid())
        qDebug() << "error releasing handles:" << reply.error().message();
}

uint Handle::request(const QString &name)
{
    uint handle = 0;
    if (m_conn.isNull()) { return 0; }

    QStringList names(name);
    QDBusReply<QList<uint> > handleList = m_conn->RequestHandles(static_cast<uint>(m_type), names);
    if (!handleList.isValid()) {
        qDebug() << "error requesting handles:" << handleList.error().message();
        return 0;
    }

    if (static_cast<QList<uint> >(handleList).size() > 0)
        handle = handleList.value().first();

    return handle;
}

bool Handle::operator==(const Handle &handle) const
{
    return ((m_id == handle.id()) && (m_type == handle.type()));
}

