/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: markarr.cxx,v $
 *
 *  $Revision: 1.9 $
 *
 *  last change: $Author: vg $ $Date: 2007/02/27 12:06:54 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_sc.hxx"



// INCLUDE ---------------------------------------------------------------

#include <tools/debug.hxx>

#include "markarr.hxx"
#include "global.hxx"
#include "address.hxx"

// STATIC DATA -----------------------------------------------------------

//------------------------------------------------------------------------

ScMarkArray::ScMarkArray()
{
    nCount = nLimit = 1;
	pData = new ScMarkEntry[1];
	if (pData)
	{
		pData[0].nRow = MAXROW;
		pData[0].bMarked = FALSE;
	}
}

//------------------------------------------------------------------------

ScMarkArray::~ScMarkArray()
{
	if (pData)
		delete[] pData;
}

//------------------------------------------------------------------------

void ScMarkArray::Reset( BOOL bMarked )
{
	if (pData)
	{
		delete[] pData;

        nCount = nLimit = 1;
		pData = new ScMarkEntry[1];
		if (pData)
		{
			pData[0].nRow = MAXROW;
			pData[0].bMarked = bMarked;
		}
	}
}

//------------------------------------------------------------------------

BOOL ScMarkArray::Search( SCROW nRow, SCSIZE& nIndex ) const
{
	long	nLo 		= 0;
	long	nHi 		= static_cast<long>(nCount) - 1;
	long	nStartRow	= 0;
	long	nEndRow 	= 0;
	long	i			= 0;
	BOOL	bFound		= (nCount == 1);
	if (pData)
	{
		while ( !bFound && nLo <= nHi )
		{
			i = (nLo + nHi) / 2;
			if (i > 0)
				nStartRow = (long) pData[i - 1].nRow;
			else
				nStartRow = -1;
			nEndRow = (long) pData[i].nRow;
			if (nEndRow < (long) nRow)
				nLo = ++i;
			else
				if (nStartRow >= (long) nRow)
					nHi = --i;
				else
					bFound = TRUE;
		}
	}
	else
		bFound = FALSE;

	if (bFound)
		nIndex=(SCSIZE)i;
	else
		nIndex=0;
	return bFound;
}

BOOL ScMarkArray::GetMark( SCROW nRow ) const
{
	SCSIZE i;
	if (Search( nRow, i ))
		return pData[i].bMarked;
	else
		return FALSE;

}

//------------------------------------------------------------------------

void ScMarkArray::SetMark( SCROW nRow, BOOL bMarked )
{
	SetMarkArea( nRow, nRow, bMarked );
}

void ScMarkArray::SetMarkArea( SCROW nStartRow, SCROW nEndRow, BOOL bMarked )
{
	if (ValidRow(nStartRow) && ValidRow(nEndRow))
	{
		if ((nStartRow == 0) && (nEndRow == MAXROW))
		{
			Reset(bMarked);
		}
		else
		{
            SCSIZE nNeeded = nCount + 2;
            if ( nLimit < nNeeded )
            {
                nLimit += SC_MARKARRAY_DELTA;
                if ( nLimit < nNeeded )
                    nLimit = nNeeded;
                ScMarkEntry* pNewData = new ScMarkEntry[nLimit];
                memcpy( pNewData, pData, nCount*sizeof(ScMarkEntry) );
				delete[] pData;
				pData = pNewData;
            }

            SCSIZE ni;          // number of entries in beginning
            SCSIZE nInsert;     // insert position (MAXROW+1 := no insert)
            BOOL bCombined = FALSE;
            BOOL bSplit = FALSE;
            if ( nStartRow > 0 )
            {
                // skip beginning
                SCSIZE nIndex;
                Search( nStartRow, nIndex );
                ni = nIndex;

                nInsert = MAXROWCOUNT;
                if ( pData[ni].bMarked != bMarked )
                {
                    if ( ni == 0 || (pData[ni-1].nRow < nStartRow - 1) )
                    {   // may be a split or a simple insert or just a shrink,
                        // row adjustment is done further down
                        if ( pData[ni].nRow > nEndRow )
                            bSplit = TRUE;
                        ni++;
                        nInsert = ni;
                    }
                    else if ( ni > 0 && pData[ni-1].nRow == nStartRow - 1 )
                        nInsert = ni;
                }
                if ( ni > 0 && pData[ni-1].bMarked == bMarked )
                {   // combine
                    pData[ni-1].nRow = nEndRow;
                    nInsert = MAXROWCOUNT;
                    bCombined = TRUE;
                }
            }
            else
	    {
                nInsert = 0;
                ni = 0;
	    }

            SCSIZE nj = ni;     // stop position of range to replace
            while ( nj < nCount && pData[nj].nRow <= nEndRow )
                nj++;
            if ( !bSplit )
            {
                if ( nj < nCount && pData[nj].bMarked == bMarked )
                {   // combine
                    if ( ni > 0 )
                    {
                        if ( pData[ni-1].bMarked == bMarked )
                        {   // adjacent entries
                            pData[ni-1].nRow = pData[nj].nRow;
                            nj++;
                        }
                        else if ( ni == nInsert )
                            pData[ni-1].nRow = nStartRow - 1;   // shrink
                    }
                    nInsert = MAXROWCOUNT;
                    bCombined = TRUE;
                }
                else if ( ni > 0 && ni == nInsert )
                    pData[ni-1].nRow = nStartRow - 1;   // shrink
            }
            if ( ni < nj )
            {   // remove middle entries
                if ( !bCombined )
                {   // replace one entry
                    pData[ni].nRow = nEndRow;
                    pData[ni].bMarked = bMarked;
                    ni++;
                    nInsert = MAXROWCOUNT;
                }
                if ( ni < nj )
                {   // remove entries
                    memmove( pData + ni, pData + nj, (nCount - nj) * sizeof(ScMarkEntry) );
                    nCount -= nj - ni;
                }
            }

            if ( nInsert < sal::static_int_cast<SCSIZE>(MAXROWCOUNT) )
            {   // insert or append new entry
                if ( nInsert <= nCount )
                {
                    if ( !bSplit )
                        memmove( pData + nInsert + 1, pData + nInsert,
                            (nCount - nInsert) * sizeof(ScMarkEntry) );
                    else
                    {
                        memmove( pData + nInsert + 2, pData + nInsert,
                            (nCount - nInsert) * sizeof(ScMarkEntry) );
                        pData[nInsert+1] = pData[nInsert-1];
                        nCount++;
                    }
                }
                if ( nInsert )
                    pData[nInsert-1].nRow = nStartRow - 1;
                pData[nInsert].nRow = nEndRow;
                pData[nInsert].bMarked = bMarked;
                nCount++;
            }
		}
	}
//	InfoBox(0, String(nCount) + String(" Eintraege") ).Execute();
}

void ScMarkArray::DeleteArea(SCROW nStartRow, SCROW nEndRow)
{
	SetMarkArea(nStartRow, nEndRow, FALSE);
}

BOOL ScMarkArray::IsAllMarked( SCROW nStartRow, SCROW nEndRow ) const
{
	SCSIZE nStartIndex;
	SCSIZE nEndIndex;

	if (Search( nStartRow, nStartIndex ))
		if (pData[nStartIndex].bMarked)
			if (Search( nEndRow, nEndIndex ))
				if (nEndIndex==nStartIndex)
					return TRUE;

	return FALSE;
}

BOOL ScMarkArray::HasOneMark( SCROW& rStartRow, SCROW& rEndRow ) const
{
	BOOL bRet = FALSE;
	if ( nCount == 1 )
	{
		if ( pData[0].bMarked )
		{
			rStartRow = 0;
			rEndRow = MAXROW;
			bRet = TRUE;
		}
	}
	else if ( nCount == 2 )
	{
		if ( pData[0].bMarked )
		{
			rStartRow = 0;
			rEndRow = pData[0].nRow;
		}
		else
		{
			rStartRow = pData[0].nRow + 1;
			rEndRow = MAXROW;
		}
		bRet = TRUE;
	}
	else if ( nCount == 3 )
	{
		if ( pData[1].bMarked )
		{
			rStartRow = pData[0].nRow + 1;
			rEndRow = pData[1].nRow;
			bRet = TRUE;
		}
	}
	return bRet;
}

BOOL ScMarkArray::HasMarks() const
{
	return ( nCount > 1 || pData[0].bMarked );
}

void ScMarkArray::SwapCol(ScMarkArray& rMarkArray)
{
	SCSIZE nTemp = rMarkArray.nCount;
	rMarkArray.nCount = nCount;
	nCount = nTemp;

    nTemp = rMarkArray.nLimit;
    rMarkArray.nLimit = nLimit;
    nLimit = nTemp;

	ScMarkEntry* pTemp = rMarkArray.pData;
	rMarkArray.pData = pData;
	pData = pTemp;
}

void ScMarkArray::MoveTo(SCROW nStartRow, SCROW nEndRow, ScMarkArray& rMarkArray)
{
	SCROW nStart = nStartRow;
	for (SCSIZE i = 0; i < nCount; i++)
	{
		if ((pData[i].nRow >= nStartRow) && ((i==0) ? TRUE : pData[i-1].nRow < nEndRow))
		{
			rMarkArray.SetMarkArea(nStart, Min(pData[i].nRow,nEndRow), pData[i].bMarked);
		}
		nStart = Max((SCROW)nStart, (SCROW)(pData[i].nRow + 1) );
	}
	DeleteArea(nStartRow, nEndRow);
}

void ScMarkArray::CopyMarksTo( ScMarkArray& rDestMarkArray ) const
{
	delete[] rDestMarkArray.pData;

	if (pData)
	{
		rDestMarkArray.pData = new ScMarkEntry[nCount];
		memmove( rDestMarkArray.pData, pData, nCount * sizeof(ScMarkEntry) );
	}
	else
		rDestMarkArray.pData = NULL;

    rDestMarkArray.nCount = rDestMarkArray.nLimit = nCount;
}

SCsROW ScMarkArray::GetNextMarked( SCsROW nRow, BOOL bUp ) const
{
	SCsROW nRet = nRow;
	if (VALIDROW(nRow))
	{
		SCSIZE nIndex;
		Search(nRow, nIndex);
		if (!pData[nIndex].bMarked)
		{
			if (bUp)
			{
				if (nIndex>0)
					nRet = pData[nIndex-1].nRow;
				else
					nRet = -1;
			}
			else
				nRet = pData[nIndex].nRow + 1;
		}
	}
	return nRet;
}

SCROW ScMarkArray::GetMarkEnd( SCROW nRow, BOOL bUp ) const
{
	SCROW nRet;
	SCSIZE nIndex;
	Search(nRow, nIndex);
	DBG_ASSERT( pData[nIndex].bMarked, "GetMarkEnd ohne bMarked" );
	if (bUp)
	{
		if (nIndex>0)
			nRet = pData[nIndex-1].nRow + 1;
		else
			nRet = 0;
	}
	else
		nRet = pData[nIndex].nRow;

	return nRet;
}

//
//	-------------- Iterator ----------------------------------------------
//

ScMarkArrayIter::ScMarkArrayIter( const ScMarkArray* pNewArray ) :
	pArray( pNewArray ),
	nPos( 0 )
{
}

ScMarkArrayIter::~ScMarkArrayIter()
{
}

BOOL ScMarkArrayIter::Next( SCROW& rTop, SCROW& rBottom )
{
	if ( nPos >= pArray->nCount )
		return FALSE;
	while (!pArray->pData[nPos].bMarked)
	{
		++nPos;
		if ( nPos >= pArray->nCount )
			return FALSE;
	}
	rBottom = pArray->pData[nPos].nRow;
	if (nPos==0)
		rTop = 0;
	else
		rTop = pArray->pData[nPos-1].nRow + 1;
	++nPos;
	return TRUE;
}





