/*
 * Copyright 2007-2008 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Random;
import java.io.IOException;

public class TestUtil {
    private TestUtil() {
    }

    public static Path createTemporaryDirectory() throws IOException {
        Path tmpdir = Paths.get(System.getProperty("java.io.tmpdir"));
        Random r = new Random();

        Path dir;
        do {
            dir = tmpdir.resolve("name" + r.nextInt());
        } while (dir.exists());
        return dir.createDirectory();
    }

    static void removeAll(Path dir) {
        Files.walkFileTree(dir, new FileVisitor<Path>() {

            public FileVisitResult preVisitDirectory(Path dir) {
                return FileVisitResult.CONTINUE;
            }

            public FileVisitResult preVisitDirectoryFailed(Path dir, IOException exc) {
                System.err.format("Error occured accessing directory %s\n", dir, exc);
                return FileVisitResult.CONTINUE;
            }

            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                try {
                    file.delete(false);
                } catch (IOException x) {
                    System.err.format("Unable to delete %s: %s\n", file, x);
                }
                return FileVisitResult.CONTINUE;
            }

            public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                try {
                    dir.delete(false);
                } catch (IOException x) {
                    System.err.format("Unable to delete %s: %s\n", dir, x);
                }
                return FileVisitResult.CONTINUE;
            }

            public FileVisitResult visitFileFailed(Path file, IOException exc) {
                System.err.format("Unable to visit %s: %s\n", file, exc);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    /**
     * Creates a directory tree in the given directory so that the total
     * size of the path is more than 2k in size. This is used for long
     * path tests on Windows.
     */
    static Path createDirectoryWithLongPath(Path dir)
        throws IOException
    {
        StringBuilder sb = new StringBuilder();
        for (int i=0; i<240; i++) {
            sb.append('A');
        }
        String name = sb.toString();
        do {
            dir = dir.resolve(name).resolve(".");
            dir.createDirectory();
        } while (dir.toString().length() < 2048);
        return dir;
    }

    /**
     * Returns true if symbolic links are supported
     */
    static boolean supportsLinks(Path dir) {
        Path link = dir.resolve("testlink");
        Path target = dir.resolve("testtarget");
        try {
            link.createSymbolicLink(target);
            target.delete(false);
            return true;
        } catch (UnsupportedOperationException x) {
            return false;
        } catch (IOException x) {
            return false;
        }
    }
}
