/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Michal Hruby <michal.hruby@canonical.com>
 *
 */

namespace Unity.Tester
{
  namespace Options
  {
    public static string lens_dbus_name;
    public static string lens_dbus_path;
    public static string lens_file;
    public static string search_string;
    public static int search_type;
    public static bool common_tests;
    public static bool no_search_reply;
    public static bool dump_results;
    public static bool dump_filters;

    public static bool test_server_mode;
    public static string[] test_cases;
  }

  namespace TestRunner
  {
    public static string[] test_cases;
    public static int test_index;
  }

  public errordomain TesterError
  {
    INVALID_ARGS
  }

  public struct LensInfo
  {
    public string dbus_path;
    public bool search_in_global;
    public bool visible;
    public string search_hint;
    public string private_connection_name;
    public string results_model_name;
    public string global_results_model_name;
    public string categories_model_name;
    public string filters_model_name;
    public HashTable<string, Variant> hints;
  }

  const OptionEntry[] options =
  {
    {
      "dbus-name", 'n', 0, OptionArg.STRING, out Options.lens_dbus_name,
      "Unique dbus name of the tested lens", null
    },
    {
      "dbus-path", 'p', 0, OptionArg.STRING, out Options.lens_dbus_path,
      "Object path of the lens", null
    },
    {
      "lens-file", 'l', 0, OptionArg.STRING, out Options.lens_file,
      "Path to the lens file (to read out dbus name and path)", null
    },
    {
      "common-tests", 'c', 0, OptionArg.NONE, out Options.common_tests,
      "Perform common tests each lens should conform to", null
    },
    {
      "search", 's', 0, OptionArg.STRING, out Options.search_string,
      "Search string to send to the lens", null
    },
    {
      "search-type", 't', 0, OptionArg.INT, out Options.search_type,
      "Type of the search (value from Unity.SearchType enum)", null
    },
    {
      "dump-results", 'r', 0, OptionArg.NONE, out Options.dump_results,
      "Output the results model on stdout", null
    },
    {
      "dump-filters", 'f', 0, OptionArg.NONE, out Options.dump_filters,
      "Output the filter model on stdout", null
    },
    {
      "no-search-reply", 0, 0, OptionArg.NONE, out Options.no_search_reply,
      "Don't output reply of the search call", null
    },
    {
      "test-server-mode", 0, 0, OptionArg.NONE, out Options.test_server_mode,
      "Run a collection of test scripts", null
    },
    {
      "", 0, 0, OptionArg.FILENAME_ARRAY, out Options.test_cases,
      "Invididual test cases", "<test-scripts>"
    },
    {
      null
    }
  };

  public static void warn (string format, ...)
  {
    var args = va_list ();
    logv ("unity-tool", LogLevelFlags.LEVEL_WARNING, format, args);
  }

  public static int main (string[] args)
  {
    Environment.set_prgname ("unity-tool");
    var opt_context = new OptionContext (" - Unity tool");
    opt_context.add_main_entries (options, null);

    try
    {
      if (args.length <= 1)
      {
        print ("%s\n", opt_context.get_help (true, null));
        return 0;
      }

      opt_context.parse (ref args);
      if (Options.test_server_mode)
      {
        if (Options.test_cases == null ||
          (Options.test_cases.length=(int)strv_length(Options.test_cases)) == 0)
        {
          throw new TesterError.INVALID_ARGS ("No test cases specified");
        }

        // special mode where we just run test scripts inside a directory
        string[] test_scripts = get_test_cases ();
        TestRunner.test_cases = test_scripts;

        Test.init (ref args);
        foreach (unowned string test_case in test_scripts)
        {
          Test.add_data_func ("/Integration/LensTest/" + 
                              Path.get_basename (test_case),
                              () =>
                              {
                                string test = TestRunner.test_cases[TestRunner.test_index++];
                                int status;
                                try
                                {
                                  Process.spawn_command_line_sync (test,
                                                                   null,
                                                                   null,
                                                                   out status);
                                }
                                catch (Error e)
                                {
                                  warn ("%s", e.message);
                                  status = -1;
                                }
                                assert (status == 0);
                              });
        }
        return Test.run ();
      }
      else
      {
        // read dbus name and path from the lens file
        if (Options.lens_file != null)
        {
          var keyfile = new KeyFile ();
          keyfile.load_from_file (Options.lens_file, 0);

          Options.lens_dbus_name = keyfile.get_string ("Lens", "DBusName");
          Options.lens_dbus_path = keyfile.get_string ("Lens", "DBusPath");
        }

        // check that we have dbus names
        if (Options.lens_dbus_name == null || Options.lens_dbus_path == null)
        {
          throw new TesterError.INVALID_ARGS ("Lens DBus name and path not specified!");
        }

        if (Options.common_tests)
        {
          int status = run_common_tests ();
          assert (status == 0);
        }

        // Performing search
        if (Options.search_string != null)
        {
          var ml = new MainLoop ();

          call_lens_search (Options.search_string,
                            Options.search_type,
                            (result) =>
          {
            if (!Options.no_search_reply)
              print ("%s\n", result.print (true));
            ml.quit ();
          });

          assert (run_with_timeout (ml, 15000));
        }

        // Dumping models
        if (Options.dump_results || Options.dump_filters)
        {
          LensInfo li = get_lens_info ();

          if (Options.dump_results)
          {
            var model_name = Options.search_type == 0 ?
              li.results_model_name : li.global_results_model_name;
            var model = new Dee.SharedModel (model_name);
            sync_model (model);

            dump_results_model (model);
          }

          if (Options.dump_filters)
          {
            var model = new Dee.SharedModel (li.filters_model_name);
            sync_model (model);

            dump_filters_model (model);
          }
        }
      }
    }
    catch (Error err)
    {
      warn ("%s", err.message);
      return 1;
    }


    return 0;
  }

  public static string[] get_test_cases ()
  {
    string[] results = {};
    foreach (string path in Options.test_cases)
    {
      if (FileUtils.test (path, FileTest.IS_REGULAR) &&
          FileUtils.test (path, FileTest.IS_EXECUTABLE))
      {
        results += path;
      }
      else if (FileUtils.test (path, FileTest.IS_DIR))
      {
        try
        {
          var dir = Dir.open (path);
          unowned string name = dir.read_name ();
          while (name != null)
          {
            var child_path = Path.build_filename (path, name, null);
            if (FileUtils.test (child_path, FileTest.IS_REGULAR) &&
                FileUtils.test (child_path, FileTest.IS_EXECUTABLE))
            {
              results += child_path;
            }

            name = dir.read_name ();
          }
        } catch (Error e) { warn ("%s", e.message); }
      }
    }

    return results;
  }

  public static bool run_with_timeout (MainLoop ml, uint timeout_ms)
  {
    bool timeout_reached = false;
    var t_id = Timeout.add (timeout_ms, () => 
    {
      timeout_reached = true;
      debug ("Timeout reached");
      ml.quit ();
      return false;
    });

    ml.run ();

    if (!timeout_reached) Source.remove (t_id);

    return !timeout_reached;
  }

  private static int run_common_tests ()
  {
    string[] args = { "./unity-tool" };
    unowned string[] dummy = args;

    Test.init (ref dummy);

    // checks that lens emits finished signal for every search type
    // (and both empty and non-empty searches)
    Test.add_data_func ("/Integration/LensTest/DefaultSearch/Empty", () =>
    {
      var ml = new MainLoop ();
      call_lens_search ("", 0, () => { ml.quit (); });
      if (!run_with_timeout (ml, 20000)) warn ("Lens didn't respond");
    });

    Test.add_data_func ("/Integration/LensTest/DefaultSearch/NonEmpty", () =>
    {
      var ml = new MainLoop ();
      call_lens_search ("a", 0, () => { ml.quit (); });
      if (!run_with_timeout (ml, 20000)) warn ("Lens didn't respond");
    });

    // check also non-empty -> empty search
    Test.add_data_func ("/Integration/LensTest/DefaultSearch/Empty2", () =>
    {
      var ml = new MainLoop ();
      call_lens_search ("", 0, () => { ml.quit (); });
      if (!run_with_timeout (ml, 20000)) warn ("Lens didn't respond");
    });

    Test.add_data_func ("/Integration/LensTest/GlobalSearch/Empty", () =>
    {
      var ml = new MainLoop ();
      call_lens_search ("", 1, () => { ml.quit (); });
      if (!run_with_timeout (ml, 20000)) warn ("Lens didn't respond");
    });

    Test.add_data_func ("/Integration/LensTest/GlobalSearch/NonEmpty", () =>
    {
      var ml = new MainLoop ();
      call_lens_search ("a", 1, () => { ml.quit (); });
      if (!run_with_timeout (ml, 20000)) warn ("Lens didn't respond");
    });

    // check also non-empty -> empty search
    Test.add_data_func ("/Integration/LensTest/GlobalSearch/Empty2", () =>
    {
      var ml = new MainLoop ();
      call_lens_search ("", 1, () => { ml.quit (); });
      if (!run_with_timeout (ml, 20000)) warn ("Lens didn't respond");
    });

    return Test.run ();
  }

  private static void call_lens_search (string search_string,
                                        int search_type,
                                        Func<Variant?>? cb = null)
  {
    var vb = new VariantBuilder (new VariantType ("(sa{sv})"));
    vb.add ("s", search_string);
    vb.open (new VariantType ("a{sv}"));
    vb.close ();

    try
    {
      var bus = Bus.get_sync (BusType.SESSION);

      bus.call.begin (Options.lens_dbus_name,
                      Options.lens_dbus_path,
                      "com.canonical.Unity.Lens",
                      search_type == 0 ? "Search" : "GlobalSearch",
                      vb.end (),
                      null,
                      0,
                      -1,
                      null,
                      (obj, res) =>
      {
        try
        {
          var reply = bus.call.end (res);
          if (cb != null) cb (reply);
        }
        catch (Error e)
        {
          warn ("%s", e.message);
        }
      });
    }
    catch (Error e) { warn ("%s", e.message); }
  }

  private static LensInfo get_lens_info ()
  {

    var ml = new MainLoop ();
    LensInfo info = LensInfo ();

    try
    {
      var bus = Bus.get_sync (BusType.SESSION);
      var sig_id = bus.signal_subscribe (null,
                                         "com.canonical.Unity.Lens",
                                         "Changed",
                                         Options.lens_dbus_path,
                                         null,
                                         0,
                                         (conn, sender, obj_path, ifc_name, sig_name, parameters) =>
      {
        info = (LensInfo) parameters.get_child_value (0);
        ml.quit ();
      });
      bus.call.begin (Options.lens_dbus_name,
                      Options.lens_dbus_path,
                      "com.canonical.Unity.Lens",
                      "InfoRequest",
                      null,
                      null,
                      0,
                      -1,
                      null,
                      null);

      if (!run_with_timeout (ml, 5000)) warn ("Unable to get LensInfo!");

      bus.signal_unsubscribe (sig_id);
    }
    catch (Error e) { warn ("%s", e.message); }

    return info;
  }

  private void sync_model (Dee.SharedModel model)
  {
    while (!model.synchronized)
    {
      var ml = new MainLoop ();
      var sig_id = model.notify["synchronized"].connect (
        () => { ml.quit (); }
      );
      ml.run ();
      SignalHandler.disconnect (model, sig_id);
    }
  }

  private void dump_results_model (Dee.Model model)
  {
    var iter = model.get_first_iter ();
    var last_iter = model.get_last_iter ();

    while (iter != last_iter)
    {
      var row = model.get_row (iter);
      print ("%s\t%s\t%u\t%s\t%s\t%s\t%s\n",
             row[0].get_string (),
             row[1].get_string (),
             row[2].get_uint32 (),
             row[3].get_string (),
             row[4].get_string (),
             row[5].get_string (),
             row[6].get_string ()
             );

      iter = model.next (iter);
    }
  }

  private void dump_filters_model (Dee.Model model)
  {
    var iter = model.get_first_iter ();
    var last_iter = model.get_last_iter ();

    while (iter != last_iter)
    {
      var row = model.get_row (iter);
      print ("%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
             row[0].get_string (),
             row[1].get_string (),
             row[2].get_string (),
             row[3].get_string (),
             row[4].print (true),
             row[5].get_boolean ().to_string (),
             row[6].get_boolean ().to_string (),
             row[7].get_boolean ().to_string ()
             );

      iter = model.next (iter);
    }
  }
}
