/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

/*
 * IMPLEMENTATION NOTE:
 * It may appear this code it is bit more bloated than it needs to be
 * (considering the pure number of classes and indirections), but this has
 * some good reasons.
 *
 * Firstly we want to hide away Vala's internal DBus marshalling which would
 * expose raw structs in the API. These structs are hidden away in _PlaceRendererInfo,
 * and _PlaceEntryInfo. We wrap these in handy GObjects with properties and what not.
 * In fact we want to hide all DBusisms, which is also why the DBus interfaces
 * are declared private.
 *
 * Secondly we want the generatedd C API to be nice and not too Vala-ish. We
 * must anticipate that place daemons consuming libunity will be written in
 * both Vala and C.
 *
 */

using GLib;
using Dee;

namespace Unity {

  /**
   * UnityPlace_PlaceRendererInfo:
   *
   * Private helper struct used for marshalling a PlaceRendererInfo object over DBus
   */
  private struct _PlaceRendererInfo {
  	public string default_renderer;
  	public string groups_model;
  	public string results_model;
  	public HashTable<string,string> hints;
  }

  /**
   * SECTION:unity-place-renderer-info
   * @short_description: Encapsulates all place entry metadata the Unity shell needs in order to render this model
   * @include: unity.h
   *
   *
   */
  public class PlaceRendererInfo : GLib.Object {
    
    /* We use a pointer to the _PlaceRendererInfo to avoid Vala duping the struct
     * values. The actual struct pointed to is owned by the parent PlaceEntryInfo */
    private _PlaceRendererInfo* info;
    private Dee.Model _groups_model;
    private Dee.Model _results_model;

    /*
     * Properties
     */
    public string default_renderer {
      get { return info->default_renderer; }
      set { info->default_renderer = value; }
    }

    public Dee.Model groups_model {
      get { return _groups_model; }
      set {
        _groups_model = value;
        if (value is Dee.SharedModel)
        {
          Dee.SharedModel model = value as Dee.SharedModel;
          info->groups_model = model.get_swarm_name();
        }
        else
          info->groups_model = "__local__";
      }
    }

    public Dee.Model results_model {
      get { return _results_model; }
      set {
        _results_model = value;
        if (value is Dee.SharedModel)
        {
          Dee.SharedModel model = value as Dee.SharedModel;
          info->results_model = model.get_swarm_name();
        }
        else
          info->results_model = "__local__";
      }
    }

    /*
     * Constructors
     */

    internal PlaceRendererInfo (_PlaceRendererInfo* info)
    {
      this.info = info;
    }

    /*
     * Public API
     */

    public void set_hint (string hint, string val)
    {
      info->hints.insert (hint, val);
    }

    public string? get_hint (string hint)
    {
      return info->hints.lookup (hint);
    }

    public void clear_hint (string hint)
    {
      info->hints.remove (hint);
    }

    public void clear_hints ()
    {
      info->hints.remove_all ();
    }

    public uint num_hints ()
    {
      return info->hints.size ();
    }
    
    internal _PlaceRendererInfo get_raw ()
    {
      return *info;
    }
  }

  /**
   *
   */
  public class PlaceSearch : InitiallyUnowned {

    private string search;
    private HashTable<string,string> hints;
    private weak PlaceEntryInfo? parent_entry;
    private uint section;

    /*
     * Public API
     */

    public PlaceSearch (string search, HashTable<string,string> hints)
    {
      GLib.Object();
      this.search = search;
      this.hints = hints;
      parent_entry = null;
      section = 0;
    }

    public string get_search_string ()
    {
      return search;
    }

    public List<unowned string> get_hints ()
    {
      return hints.get_keys ();
    }

    public void set_hint (string hint, string val)
    {
      hints.insert (hint, val);
    }

    public string? get_hint (string hint)
    {
      return hints.lookup (hint);
    }

    public void clear_hint (string hint)
    {
      hints.remove (hint);
    }

    public void clear_hints ()
    {
      hints.remove_all ();
    }

    public uint num_hints ()
    {
      return hints.size ();
    }
    
    /* Returns true if search strings and all hints match */
    public bool equals (PlaceSearch? other)
    {
      if (other == null)
        return false;
        
      if (get_search_string() != other.get_search_string ())
        return false;
      
      if (num_hints () != other.num_hints())
        return false;
        
      foreach (var hint in get_hints ())
      {
        if (other.get_hint (hint) != get_hint (hint))
          return false;
      }
      
      return true;
    }
    
    public void set_section (uint section)
    {
      this.section = section;
    }
    
    public uint get_section ()
    {
      return section;
    }
    
    public void set_parent_entry (PlaceEntryInfo entry)
    {
      parent_entry = entry;
    }
    
    public unowned PlaceEntryInfo? get_parent_entry ()
    {
      return parent_entry;
    }
    
    public void finished ()
    {
      if (parent_entry == null)
        return;
  
      try {
        Variant vhints = hints;
        var params = new Variant ("(us@a{ss})", section, search, vhints);
        var conn = Bus.get_sync(BusType.SESSION);
        
        /* Make sure we push any changes to the models *before* sending
         * the SearchFinished signal */
        parent_entry.flush_all_models ();
        
        conn.emit_signal (null,                                 /* dest name */
                          parent_entry.dbus_path,               /* orig path */
                          "com.canonical.Unity.PlaceEntry",     /* iface */
                          "SearchFinished",                     /* member */
                          params);                              /* params */
      } catch (IOError e) {
        warning ("Failed to emit com.canonical.Unity.PlaceEntry.SearchFinished: %s",
                 e.message);
      }
    }
  }

  /**
   * Unity_PlaceEntryInfo:
   *
   * Private helper struct used for marshalling an PlaceEntryInfo object over DBus
   */
  private struct _PlaceEntryInfo {
  	public string   dbus_path;
    public string   display_name;
    public string   icon;
    public uint     position;
    public string[] mimetypes;
    public bool     sensitive;
    public string   sections_model;
    public HashTable<string,string> hints;
    public _PlaceRendererInfo entry_renderer_info;
    public _PlaceRendererInfo global_renderer_info;
  }

  /**
   * Unity_PlaceEntryInfoData:
   *
   * Private helper struct used for marshalling the PlaceEntryInfo data without
   * the RenderingInfo data over the bus
   */
  private struct _PlaceEntryInfoData {
  	public string   dbus_path;
    public string   display_name;
    public string   icon;
    public uint     position;
    public string[] mimetypes;
    public bool     sensitive;
    public string   sections_model;
    public HashTable<string,string> hints;
  }

  public class PlaceEntryInfo : GLib.Object {

    /* The _PlaceEntryInfo needs to be set before we set properties, so it's
     * paramount we do it here */
    private _PlaceEntryInfo info = _PlaceEntryInfo();
    private PlaceRendererInfo _entry_renderer_info;
    private PlaceRendererInfo _global_renderer_info;
    private Dee.Model _sections_model;
    private bool _active = false;
    private uint _active_section = 0;

    /*
     * Properties
     */
    public PlaceRendererInfo entry_renderer_info {
      get { return _entry_renderer_info; }
    }

    public PlaceRendererInfo global_renderer_info {
      get { return _global_renderer_info; }
    }

    public string dbus_path {
      get { return info.dbus_path; }
      construct set { info.dbus_path = value; }
    }

    public string display_name {
      get { return info.display_name; }
      construct set { info.display_name = value; }
    }

    public string icon {
      get { return info.icon; }
      construct set { info.icon = value; }
    }

    public uint position {
      get { return info.position; }
      construct set { info.position = value; }
    }

    public string[] mimetypes {
      get { return info.mimetypes; }
      construct set { info.mimetypes = value; }
    }

    public bool sensitive {
      get { return info.sensitive; }
      construct set { info.sensitive = value; }
    }

    public Dee.Model sections_model {
      get { return _sections_model; }
      construct set {
        _sections_model = value;
        if (value is Dee.SharedModel)
        {
          Dee.SharedModel model = value as Dee.SharedModel;
          info.sections_model = model.get_swarm_name();
        }
        else
          info.sections_model = "__local__";
      }
    }

    public bool active {
      get { return _active; }
      set { _active = value; }
    }

    public uint active_section {
      get { return _active_section; }
      set { _active_section = value; }
    }

    public PlaceSearch active_search { get; set; }
    public PlaceSearch active_global_search { get; set; }

    /* The browser will automagically be exported/unexported on the bus
     * when this property is set or unset. Setting the browser also sets the
     * UnityPlaceBrowserPath and UnitySectionStyle hints accordingly. */
    private Browser? _browser = null;
    public Browser? browser {
      get { return _browser; }
      set {
        _browser = value;
        if (value != null)
          {
            set_hint ("UnityPlaceBrowserPath", value.dbus_path);
            set_hint ("UnitySectionStyle", "breadcrumb");
          }
        else
          {
            clear_hint ("UnityPlaceBrowserPath");
            clear_hint ("UnitySectionStyle");
          }
      }
    }

    /*
     * Constructors
     */

    construct {
      if (info.dbus_path == null)
        {
          critical ("""No DBus path set for PlaceEntryInfo.
'dbus-path' property in the UnityPlaceEntryInfo constructor""");
          info.dbus_path = "";
        }
      if (info.display_name == null)
        info.display_name = "";
      if (info.icon == null)
        info.icon = "";
      info.position = 0;
      if (info.mimetypes == null)
        info.mimetypes = new string[0];
      info.sensitive = true;
      if (info.sections_model == null)
        info.sections_model = "";
      info.hints = new HashTable<string,string>(str_hash, str_equal);

      info.entry_renderer_info.default_renderer = "";
      info.entry_renderer_info.groups_model = "";
      info.entry_renderer_info.results_model = "";
      info.entry_renderer_info.hints = new HashTable<string, string> (str_hash, str_equal);

      info.global_renderer_info.default_renderer = "";
      info.global_renderer_info.groups_model = "";
      info.global_renderer_info.results_model = "";
      info.global_renderer_info.hints = new HashTable<string, string> (str_hash, str_equal);

      /* Pass in the address of the _PlaceRendererInfos as they are embedded
       * by value inside the _PlaceEntryInfo struct. The PlaceRendererInfo will then
       * only access the pointer to the struct to avoid memory duping which
       * would give incorrect return values for our DBus methods */
      _entry_renderer_info = new PlaceRendererInfo (&(info.entry_renderer_info));
      _global_renderer_info = new PlaceRendererInfo (&(info.global_renderer_info));
    }

    public PlaceEntryInfo (string dbus_path) {
      /* We need the _empty hack here to avoid a bug in valac, otherwise
       * valac will set it to NULL somehow and cause a g_critical in
       * g_strv_length() */
      var _empty = new string[0];
      GLib.Object(dbus_path : dbus_path,
                  mimetypes : _empty);
    }

    /*
     * Public API
     */

    public void set_hint (string hint, string val)
    {
      info.hints.insert (hint, val);
    }

    public string? get_hint (string hint)
    {
      return info.hints.lookup (hint);
    }

    public void clear_hint (string hint)
    {
      info.hints.remove (hint);
    }

    public void clear_hints ()
    {
      info.hints.remove_all ();
    }

    public uint num_hints ()
    {
      return info.hints.size ();
    }

    public void flush_all_models ()
    {
      /* Important note: If the revision queues of the models are empty
       *                 these calls are just no-ops */
      if (sections_model is Dee.SharedModel)
        {
          (sections_model as Dee.SharedModel).flush_revision_queue ();
        }
      
      if (entry_renderer_info.results_model is Dee.SharedModel)
        {
          (entry_renderer_info.results_model as Dee.SharedModel).flush_revision_queue ();
        }
      if (entry_renderer_info.groups_model is Dee.SharedModel)
        {
          (entry_renderer_info.groups_model as Dee.SharedModel).flush_revision_queue ();
        }
      
      if (global_renderer_info.results_model is Dee.SharedModel)
        {
          (entry_renderer_info.results_model as Dee.SharedModel).flush_revision_queue ();
        }
      if (global_renderer_info.groups_model is Dee.SharedModel)
        {
          (global_renderer_info.groups_model as Dee.SharedModel).flush_revision_queue ();
        }
    }

    /*
     * Internal API
     */
     internal _PlaceEntryInfo get_raw ()
     {
       return info;
     }
  }

  /**
   * UnityPlaceService:
   *
   * DBus interface exported by a place daemon
   */
  [DBus (name = "com.canonical.Unity.Place")]
  private interface PlaceService : GLib.Object
  {
    public abstract _PlaceEntryInfo[] get_entries () throws IOError;

    public signal void entry_added (_PlaceEntryInfo entry);

    public signal void entry_removed (string entry_dbus_path);
  }

  /**
   * UnityPlaceEntryService:
   *
   * DBus interface for a given place entry exported by a place daemon
   */
  [DBus (name = "com.canonical.Unity.PlaceEntry")]
  private interface PlaceEntryService : GLib.Object
  {
    public abstract void set_global_search (string search,
                                            HashTable<string,string> hints) throws IOError;

    public abstract void set_search (string search,
                                     HashTable<string,string> hints) throws IOError;
    
    public abstract void set_active (bool is_active) throws IOError;

    public abstract void set_active_section (uint section_id) throws IOError;

    public signal void entry_renderer_info_changed (_PlaceRendererInfo renderer_info);
    
    public signal void global_renderer_info_changed (_PlaceRendererInfo renderer_info);
    
    public signal void place_entry_info_changed (_PlaceEntryInfoData entry_info_data);
    
    public signal void search_finished (uint section, string search, HashTable<string,string> hints);
  }

  /**
   * UnityPlaceServiceImpl:
   *
   * Private helper class to shield of DBus details and ugly
   * internal structs used for marshalling
   */
  private class PlaceServiceImpl : GLib.Object, PlaceService, Activation
  {
    private string _dbus_path;
    private HashTable<string,PlaceEntryServiceImpl> entries;
    private uint _service_dbus_id = 0;
    private uint _activation_dbus_id = 0;

    /*
     * Properties
     */

    [Property(nick = "DBus object path", blurb = "The DBus path this object is exported under")]
    public string dbus_path {
      get { return _dbus_path; }
      construct { _dbus_path = value; }
    }

    public bool exported {
      get { return _service_dbus_id != 0; }
    }
    
    public Activation? activation { get; set; default = null; }

    /*
     * Constructors
     */

    construct {
      entries = new HashTable<string,PlaceEntryServiceImpl> (str_hash, str_equal);
    }

    public PlaceServiceImpl (string dbus_path)
    {
      GLib.Object (dbus_path : dbus_path);
    }

    /*
     * DBus API
     */

    public _PlaceEntryInfo[] get_entries ()
    {
      _PlaceEntryInfo[] result = new _PlaceEntryInfo[entries.size ()];

      int i = 0;
      foreach (var entry in entries.get_values ())
      {
        result[i] = entry.entry_info.get_raw();
        i++;
      }

      return result;
    }

    /*
     * Internal API
     */

    public void add_entry (PlaceEntryInfo entry_info)
    {
      if (entries.lookup (entry_info.dbus_path) != null)
        return;

      var entry = new PlaceEntryServiceImpl(entry_info);
      entries.insert (entry_info.dbus_path, entry);
      
      /* If the place is exported then also export the new entry */
      if (exported)
        {
          try {
            entry.export ();
          } catch (IOError e) {
            critical ("Failed to export entry '%s': %s",
                      entry.entry_info.dbus_path, e.message);
          }
        }
      entry_added (entry_info.get_raw ());
    }

    public PlaceEntryInfo? get_entry (string dbus_path)
    {
      var entry = entries.lookup (dbus_path);
      if (entry != null)
        return entry.entry_info;
      else
        return null;
    }

    public PlaceEntryServiceImpl? get_entry_service (string dbus_path)
    {
      var entry = entries.lookup (dbus_path);
      if (entry != null)
        return entry;
      else
        return null;
    }

    public uint num_entries ()
    {
      return entries.size ();
    }

    public string[] get_entry_paths ()
    {
      string[] result = new string[entries.size ()];

      int i = 0;
      foreach (var entry in entries.get_values ())
      {
        result[i] = entry.entry_info.dbus_path;
        i++;
      }

      return result;
    }

    public void remove_entry (string dbus_path)
    {
      var entry = entries.lookup (dbus_path);

      if (entry == null)
        return;

      entry_removed (dbus_path);
      if (exported)
        {
          try {
            entry.unexport ();
          } catch (IOError e) {
            critical ("Failed to unexport '%s': %s",
                      entry.entry_info.dbus_path, e.message);
          }
        }

      entries.remove (dbus_path);
    }

    public void export () throws IOError
    {
      var conn = Bus.get_sync(BusType.SESSION);
      _service_dbus_id = conn.register_object (_dbus_path, this as PlaceService);
      _activation_dbus_id = conn.register_object (_dbus_path, this as Activation);

      foreach (var entry in entries.get_values ())
      {
        entry.export ();
      }

      notify_property("exported");
    }

    public void unexport () throws IOError
    {
      foreach (var entry in entries.get_values ())
      {
        entry.unexport ();
      }
      var conn = Bus.get_sync (BusType.SESSION);
      conn.unregister_object (_service_dbus_id);
      conn.unregister_object (_activation_dbus_id);

      _service_dbus_id = 0;
      _activation_dbus_id = 0;
      notify_property("exported");
    }
    
    /* Default impl of the Unity.Place.Activation interface,
     * delegates to the installed Activation impl if there is one */
    public async uint32 activate (string uri)
    {
      if (activation == null )
        return ActivationStatus.NOT_ACTIVATED;
      
      try {
        uint32 activated = yield activation.activate (uri);
        return activated;
      } catch (IOError e) {
        return ActivationStatus.NOT_ACTIVATED;
      }
    }
    
  } /* End: PlaceServiceImpl */

  /**
   * UnityPlaceEntryServiceImpl:
   *
   * Private helper class to shield of DBus details and ugly
   * internal structs used for marshalling
   */
  private class PlaceEntryServiceImpl : GLib.Object, PlaceEntryService
  {
    private uint _dbus_id = 0;
    private PlaceEntryInfo _entry_info;
    
    /* Queue up change signals in order not to flood the bus on rapid changes */
    private uint place_entry_info_changed_signal_source = 0;
    
    /* Keep a ref to the previous browser to properly handle it leaving and
     * entering the bus */
    private Browser? _browser = null;
    
    /* DBus registration id set if we've registered the browser on the bus */
    private uint _browser_dbus_id = 0;

    /*
     * Properties
     */
    public PlaceEntryInfo entry_info {
      get { return _entry_info; }
      construct { _entry_info = value; }
    }

    public bool exported {
      get { return _dbus_id != 0; }
    }

    /*
     * Constructors
     */
    public PlaceEntryServiceImpl (PlaceEntryInfo entry_info)
    {
      GLib.Object(entry_info : entry_info);
      _browser = entry_info.browser;
      
      entry_info.notify["browser"].connect (on_browser_changed);
    }

    /*
     * DBus API
     */

    public void set_global_search (string search,
                                   HashTable<string,string> hints)
    {
      var s = new PlaceSearch (search, hints);
      s.set_parent_entry (_entry_info);
      
      if (!s.equals (this._entry_info.active_global_search))
        this._entry_info.active_global_search = s;
    }

    public void set_search (string search,
                            HashTable<string,string> hints)
    {
      var s = new PlaceSearch (search, hints);
      s.set_section (_entry_info.active_section);
      s.set_parent_entry (_entry_info);
      
      if (!s.equals (this._entry_info.active_search))
        this._entry_info.active_search = s;
    }

    public void set_active (bool is_active)
    {
      this._entry_info.active = is_active;
    }

    public void set_active_section (uint section_id)
    {
      this._entry_info.active_section = section_id;
    }

    /*
     * Internal API
     */

    public void export () throws IOError
    {
      var conn = Bus.get_sync (BusType.SESSION);
      _dbus_id = conn.register_object (_entry_info.dbus_path, this as PlaceEntryService);

      if (_entry_info.browser != null && _browser_dbus_id == 0)
        {
          debug ("Exporting browser at '%s'", _entry_info.browser.dbus_path);
          _browser_dbus_id = conn.register_object (_entry_info.browser.dbus_path,
                                                   _entry_info.browser.get_service ());

        }
      else
        debug ("No browser to export");
      
      notify_property("exported");
    }

    public void unexport () throws IOError
    {
      var conn = Bus.get_sync (BusType.SESSION);
      conn.unregister_object (_dbus_id);

      if (_entry_info.browser != null && _browser_dbus_id != 0)
        {
          debug ("Unexporting browser '%s'", _entry_info.browser.dbus_path);
          conn.unregister_object (_browser_dbus_id);
          _browser_dbus_id = 0;
        }

      _dbus_id = 0;
      notify_property("exported");
    }
    
    private void on_browser_changed (GLib.Object obj, ParamSpec pspec)
    {
      DBusConnection conn;
      try {
        conn = Bus.get_sync (BusType.SESSION);
      } catch (IOError e) {
        warning ("Unable to connect to session bus: %s", e.message);
        return;
      }

      /* If nothing changed, don't do anything */
      if (_browser == entry_info.browser)
        return;

      /* Unexport previous browser if relevant */
      if (_browser != null && _browser_dbus_id != 0)
        {
          debug ("Unexporting browser '%s'", _browser.dbus_path);
          conn.unregister_object (_browser_dbus_id);          
        }

      _browser = entry_info.browser;
      _browser_dbus_id = 0;

      /* Export the new one if any */
      if (_browser != null)
        {
          debug ("Exporting browser '%s'", _browser.dbus_path);
          try {
            _browser_dbus_id = conn.register_object (_browser.dbus_path,
                                                     _browser.get_service ());
          } catch (IOError e) {
            critical ("failed to export browser '%s': %s",
                      _browser.dbus_path, e.message);
          }
        }
    }
    
    /* Prepare emission of a PlaceEntryInfoChanged signal in next idle call,
     * unless one is already scheduled */
    public void queue_place_entry_info_changed_signal ()
    {
      if (place_entry_info_changed_signal_source == 0)
        {
          place_entry_info_changed_signal_source =
                          Idle.add (emit_place_entry_info_changed_signal);
        }
    }
    
    /* Idle handler for emitting PlaceEntryInfoChanged on the bus */
    private bool emit_place_entry_info_changed_signal ()
    {
      var entry_data = _PlaceEntryInfoData();
      
      var _entry = _entry_info.get_raw ();
      entry_data.dbus_path = _entry.dbus_path;
      entry_data.display_name = _entry.display_name;
      entry_data.icon = _entry.icon;
      entry_data.position = _entry.position;
      entry_data.mimetypes = _entry.mimetypes;
      entry_data.sensitive = _entry.sensitive;  
      entry_data.sections_model = _entry.sections_model;
      entry_data.hints = _entry.hints;
      
      /* Emit the signal */
      place_entry_info_changed (entry_data);
      
      /* Clear the queue */
      place_entry_info_changed_signal_source = 0;
      
      return false;
    }
    
  } /* End: PlaceEntryServiceImpl */

  /*
   * Private helper struct used to keep track of the installed signal handlers
   * for an entry added to a PlaceController
   */
  internal struct _PlaceEntrySignals
  {
    ulong place_entry_info_changed_id;
    ulong entry_renderer_info_changed_id;
    ulong global_renderer_info_changed_id;
  }

  /**
   * UnityPlaceController:
   *
   * Main handle for controlling the place entries managed by a place daemon
   */
  public class PlaceController : GLib.Object
  {
    private PlaceServiceImpl service;
    private string _dbus_path;
    private bool _exported = false;
    private HashTable<string, _PlaceEntrySignals?> entry_signals;
    private Gee.Set<string> ignore_remote_notify_props;

    /*
     * Properties
     */

    [Property(nick = "DBus object path", blurb = "The DBus path this object is exported under")]
    public string dbus_path {
      get { return _dbus_path; }
      construct { _dbus_path = value; }
    }

    /* Whether or not this controller exports its installed entries and
     * activation service on the bus */
    public bool exported {
      get { return _exported; }
    }
    
    /* Defer URI activation to this instance. If there is no activation set
     * for the controller it will always decline any request for activation */
    public Activation? activation {
      get { return service.activation; }
      set { service.activation = value; }
    }

    /*
     * Constructors
     */

    construct {
      service = new PlaceServiceImpl (_dbus_path);
      entry_signals = new HashTable<string, _PlaceEntrySignals?>(str_hash, str_equal);
      
      ignore_remote_notify_props = new Gee.HashSet<string> ();
      ignore_remote_notify_props.add ("active-search");
      ignore_remote_notify_props.add ("active-global-search");
      ignore_remote_notify_props.add ("active");
      ignore_remote_notify_props.add ("active-section");
    }

    public PlaceController (string dbus_path)
    {
      GLib.Object (dbus_path : dbus_path);
    }

    /*
     * Public API
     */

    public void add_entry (PlaceEntryInfo entry)
    {
      service.add_entry (entry);
      
      var signals = _PlaceEntrySignals ();
      
      signals.place_entry_info_changed_id = entry.notify.connect (on_entry_changed);
      
      /* We use a closure here because we need to capture the entry in order
       * to look up the PlaceEntryServiceImpl */
      signals.entry_renderer_info_changed_id = entry.entry_renderer_info.notify.connect (
        (obj, pspec) => {
          var renderer_info = (obj as PlaceRendererInfo);
          var entry_service = service.get_entry_service (entry.dbus_path);
          if (entry_service == null)
            {
              warning ("PlaceEntry renderer info changed for unknown entry '%s'", entry.dbus_path);
            }
          else
            entry_service.entry_renderer_info_changed (renderer_info.get_raw ());
          
        }
      );
      
      /* We use a closure here because we need to capture the entry in order
       * to look up the PlaceEntryServiceImpl */
      signals.global_renderer_info_changed_id = entry.global_renderer_info.notify.connect (
        (obj, pspec) => {
          var renderer_info = (obj as PlaceRendererInfo);
          var entry_service = service.get_entry_service (entry.dbus_path);
          if (entry_service == null)
            {
              warning ("Global renderer info changed for unknown entry '%s'", entry.dbus_path);
            }
          else
            entry_service.global_renderer_info_changed (renderer_info.get_raw ());
          
        }
      );
      
      /* Store all signal handler ids so we can remove them later */
      entry_signals.insert (entry.dbus_path, signals);
    }

    public PlaceEntryInfo? get_entry (string dbus_path)
    {
      return service.get_entry (dbus_path);
    }

    public void remove_entry (string dbus_path)
    {
      /* Disconnect all signals on the entry before we remove
       * it from the PlaceServiceImpl */
      var signals = entry_signals.lookup (dbus_path);
      if (signals == null)
        {
          warning ("No signals connected for unknown entry '%s'",
                   dbus_path);
          service.remove_entry (dbus_path);
          return;
        }
      
      var entry = service.get_entry (dbus_path);
      if (entry == null)
        {
          warning ("Can not disconnect signals for unknown entry '%s'",
                   dbus_path);
          entry_signals.remove (dbus_path);
          return;
        }
      
      entry.disconnect (signals.place_entry_info_changed_id);
      entry.entry_renderer_info.disconnect (signals.entry_renderer_info_changed_id);
      entry.global_renderer_info.disconnect (signals.global_renderer_info_changed_id);
      
      service.remove_entry (dbus_path);
    }

    public uint num_entries ()
    {
      return service.num_entries ();
    }

    public string[] get_entry_paths ()
    {
      return service.get_entry_paths ();
    }

    public PlaceEntryInfo[] get_entries ()
    {
      uint len = num_entries();
      var result = new PlaceEntryInfo[len];
      var entry_paths = get_entry_paths ();

      int i = 0;
      for (i = 0; i < len; i++)
        {
          result[i] = get_entry (entry_paths[i]);
        }

      return result;
    }

    public void export () throws IOError
    {
      /* Export the Place and Activation insterfaces */
      service.export ();
      
      _exported = true;
      notify_property("exported");
    }

    public void unexport () throws IOError
    {
      service.unexport ();
      _exported = false;
      notify_property("exported");
    }
    
    /* Callback for when an entry property changes */
    private void on_entry_changed (GLib.Object obj, ParamSpec pspec)
    {
      var entry = (obj as PlaceEntryInfo);
      var entry_service = service.get_entry_service (entry.dbus_path);
      
      if (entry_service == null)
        {
          warning ("Got change signal from unknown entry service '%s'",
                   entry.dbus_path);
          return;
        }
      
      /* Don't emit signals on the bus that are strictly local metadata */
      if (pspec.get_name () in ignore_remote_notify_props)
        return;
      
      entry_service.queue_place_entry_info_changed_signal ();
    }
    
  } /* End: PlaceController class */

} /* namespace */
