/***************************************************************************
 *   Copyright (C) 2005 by Joris Guisson                                   *
 *   joris.guisson@gmail.com                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.             *
 ***************************************************************************/
#include <QBoxLayout>
#include <kglobal.h>
#include <kconfig.h>
#include <klocale.h>
#include <qapplication.h>
#include "logviewer.h"
#include "logflags.h"
#include "logviewerpluginsettings.h"
#include <QMenu>
#include <KIcon>


namespace kt
{

	LogViewer::LogViewer(LogFlags* flags,QWidget *parent) : Activity(i18n("Log"),"utilities-log-viewer",100,parent),use_rich_text(true),flags(flags),suspended(false),menu(0),max_block_count(200)
	{
		setToolTip(i18n("View the logging output generated by KTorrent")); 
		QVBoxLayout* layout = new QVBoxLayout(this);
		output = new QTextBrowser(this);
		layout->setMargin(0);
		layout->setSpacing(0);
		layout->addWidget(output);
		output->document()->setMaximumBlockCount(max_block_count);
		output->setContextMenuPolicy(Qt::CustomContextMenu);
		connect(output,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(showMenu(QPoint)));
		
		suspend_action = new QAction(KIcon("media-playback-pause"),i18n("Suspend Output"),this);
		suspend_action->setCheckable(true);
		connect(suspend_action,SIGNAL(toggled(bool)),this,SLOT(suspend(bool)));
	}


	LogViewer::~LogViewer()
	{
	}


	void LogViewer::message(const QString& line, unsigned int arg)
	{
		/*
			IMPORTANT: because QTextBrowser is not thread safe, we must use the Qt event mechanism 
			to add strings to it, this will ensure that strings will only be added in the main application
			thread.
		*/
		if (arg==0x00 || flags->checkFlags(arg))
		{
			QMutexLocker lock(&mutex);
			if (use_rich_text)
			{
				pending.append(flags->getFormattedMessage(arg, line));
			}
			else
			{
				pending.append(line);
			}
			
			while (pending.size() > max_block_count)
				pending.pop_front();
		}
	}
	
	void LogViewer::processPending()
	{
		QMutexLocker lock(&mutex);
		foreach (const QString & line,pending)
		{
			QTextCharFormat fm = output->currentCharFormat();
			output->append(line);
			output->setCurrentCharFormat(fm);
		}
		
		pending.clear();
	}
	
	void LogViewer::setRichText(bool val)
	{
		use_rich_text = val;
	}
	
	void LogViewer::setMaxBlockCount(int max)
	{
		max_block_count = max;
		output->document()->setMaximumBlockCount(max);
	}

		
	void LogViewer::showMenu(const QPoint& pos)
	{
		if (!menu)
		{
			menu = output->createStandardContextMenu();
			QAction* first = menu->actions().at(0);
			QAction* sep = menu->insertSeparator(first);
			menu->insertAction(sep,suspend_action);
		}
		menu->popup(output->viewport()->mapToGlobal(pos));
	}
	
	void LogViewer::suspend(bool on)
	{
		suspended = on;
		QTextCharFormat fm = output->currentCharFormat();
		if (on)
			output->append(i18n("<font color=\"#FF0000\">Logging output suspended</font>"));
		else
			output->append(i18n("<font color=\"#00FF00\">Logging output resumed</font>"));
		output->setCurrentCharFormat(fm);
	}


}
#include "logviewer.moc"
