# -*- encoding: utf-8 -*-
#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import datetime
import logging
import hashlib
import base64
import hmac
import copy
import time
import sys

from twisted.web import server, resource
from twisted.internet import reactor

try:
    import json
except ImportError:
    import simplejson as json


# DISCLAIMER: This is an UGLY HACK to test the UI logic.  This code will be
#             TRASHED and replaced by a descent implementation at the earliest
#             chance.


CLIENT_ID = "gn0r6yROWhShJjGxcyBeOtV6XiYqiRwZvd3hbW"
SECRET_KEY = "Wbh3dvZwRiqYiX6VtOeBycxGjJhShWORy6r0ng"
CREDENTIALS = {CLIENT_ID: SECRET_KEY}

SUPPORTED_API_VERSIONS = ["2009-10-01"]


ALFRESCO_IMAGE = {
    "uri": "http://imagestore.canonical.com/api/images/b8b4680a-f218-4e85-8671-002705e38e34",
    "title": u"Alfresco Server (áéíóú)",
    "version": "1.0",
    "summary": "Open source content management system",
    "size-in-mb": 640,
    "description-html":
        "<p>Open source enterprise content management system including "
        "document management, web content management, SharePoint "
        "alternative and content repository.</p>"
        "<ul><li>This<li>is<li>html!</ul>",
    "provider": {
        "uri": "http://www.alfresco.com",
        "title": "Alfresco"
        },
#    "files": [
#        {"url": "http://....",
#         "size-in-bytes": 123456,
#         "kind": "kernel",
#         "sha256": "abcdefgh"},
#        {"url": "http://....",
#         "size-in-bytes": 123456,
#         "kind": "image",
#         "sha256": "abcdefgh"},
#        {"url": "http://....",
#         "size-in-bytes": 123456,
#         "kind": "ramdisk",
#         "sha256": "abcdefgh"},
#    ],
    }

WEB_SERVER_IMAGE = {
    "uri": "http://imagestore.canonical.com/api/images/fd82787c-f346-4426-aa2b-c75b7d5544a7",
    "title": "Ubuntu Apache Web Server",
    "version": "1.5",
    "summary": "Basic apache-based appliance",
    "tags": ["ubuntu", "apache", "web", "server"],
    "size-in-mb": 520,
    "description-html":
        "Basic appliance based on Apache.",
    "provider": {
        "uri": "http://www.canonical.com",
        "title": "Canonical"
        },
    }

TOMCAT_IMAGE = {
    "uri": "http://imagestore.canonical.com/api/images/f2644bd5-d598-471c-9e14-d910b1a47c50",
    "title": "Tomcat",
    "version": "1.5",
    "summary": "Java application server",
    "size-in-mb": 700,
    "description-html":
        "Robust Java application server backed by the Apache Foundation.",
    "provider": {
        "uri": "http://www.apache.org",
        "title": "Apache Foundation"
        },
    }

DJANGO_IMAGE = {
    "uri": "http://imagestore.canonical.com/api/images/0893b94e-436c-49d2-bce8-07c189128b15",
    "title": "Django",
    "version": "0.9",
    "summary": "Django web framework",
    "description-html":
        "Python-based web framework.",
    }

KARMIC_IMAGE = {
    "uri": "http://imagestore.canonical.com/api/images/619447e9-b746-43d7-91c5-4a49efe40a64",
    "title": "Ubuntu Karmic",
    "version": "1.0",
    "size-in-mb": 310,
    "summary": "Basic Ubuntu Karmic image",
    "description-html":
        "Yet another excellent Ubuntu image for your pleasure.",
    }


IMAGES = [
    TOMCAT_IMAGE,
    ALFRESCO_IMAGE,
    WEB_SERVER_IMAGE,
    DJANGO_IMAGE,
    KARMIC_IMAGE,
    ]

STATES = {
    TOMCAT_IMAGE["uri"]: {
        "is-upgrade": True,
        },
    ALFRESCO_IMAGE["uri"]: {
        "status": "installed",
        "emi": "emi-123456789",
        },
    }

DASHBOARD_SECTIONS = [
    {"title": "Upgrades",
     "summary": "The following recent versions of appliances you have installed are available:",
     "image-uris": [
         TOMCAT_IMAGE["uri"],
         ],
    },
    {"title": "Highlights",
     "summary": "Fantastic images brought by Ubuntu and Eucalyptus to you:",
     "image-uris": [
         ALFRESCO_IMAGE["uri"],
         WEB_SERVER_IMAGE["uri"],
         ],
    },
    {"title": "New",
     "summary": "Appliances recently made available:",
     "image-uris": [
         DJANGO_IMAGE["uri"],
         ALFRESCO_IMAGE["uri"],
         ],
    },
    ]

SEARCH_SECTIONS = [
    {"title": "Search results",
     "summary": "The following results were found:",
     "image-uris": [],
    },
    ]


def log(*args):
    print datetime.datetime.now(), " ".join(str(x) for x in args)


class StateTracker(object):

    def __init__(self, images, states, erroring_image_uris=()):
        self._states = {}
        self._installed_time = {}
        self._last_advancement = time.time()
        self._erroring_image_uris = set(erroring_image_uris)
        for image in images:
            image_uri = image["uri"]
            state = copy.deepcopy(states.get(image_uri, {}))
            state["image-uri"] = image_uri
            if "status" not in state:
                state["status"] = "uninstalled"
            self._add_action(state, "install")
            self._states[image_uri] = state

    def _add_action(self, state, action):
        encoded_image_uri = base64.urlsafe_b64encode(state["image-uri"])
        state.setdefault("actions", {})[action] = \
            "http://localhost:52780/api/images/%s/%s" % (encoded_image_uri, action)

    def _remove_action(self, state, action):
        state.setdefault("actions", {}).pop(action, None)

    def get_state(self, uri):
        self.advance_progress()
        return self._states[uri]

    def get_all_states(self):
        return self._states.values()

    def install(self, uri):
        self.advance_progress()
        state = self._states[uri]
        if state["status"] == "uninstalled":
            state["status"] = "downloading"
            state["progress-percentage"] = 0
            self._remove_action(state, "install")
            self._add_action(state, "cancel")

    def cancel(self, uri):
        self.advance_progress()
        state = self._states[uri]
        if state["status"] == "downloading":
            state["status"] = "uninstalled"
            del state["progress-percentage"]
            self._remove_action(state, "cancel")
            self._add_action(state, "install")

    def clear_error(self, uri):
        self.advance_progress()
        state = self._states[uri]
        state.pop("error-message", None)
        self._remove_action(state, "clear-error")

    def set_error(self, uri, message):
        state = self._states[uri]
        state["error-message"] = message
        self._add_action(state, "clear-error")

    def advance_progress(self):
        # We move 1% forward every 0.5 seconds.
        now = time.time()
        delta = (now - self._last_advancement)
        delta //= 0.5 
        if delta > 0:
            self._last_advancement = now
            for uri, installed_time in list(self._installed_time.iteritems()):
                # Reset the installation status every 5 minutes to enable
                # people to test it continuously.
                if now - installed_time > 60 * 5:
                    del self._installed_time[uri]
                    state = self._states[uri]
                    state["status"] = "uninstalled"
                    del state["emi"]
                    self._add_action(state, "install")
            for state in self._states.itervalues():
                percentage = state.get("progress-percentage")
                if percentage is not None:
                    percentage += delta
                    if percentage <= 100:
                        if (percentage > 20 and
                            state["image-uri"] in self._erroring_image_uris):
                            del state["progress-percentage"]
                            state["status"] = "uninstalled"
                            self.set_error(state["image-uri"],
                                           "If this was real you'd see a "
                                           "nice error message here.")
                            self._add_action(state, "install")
                            self._remove_action(state, "cancel")
                        else:
                            state["progress-percentage"] = percentage
                    else:
                        if state["status"] == "downloading":
                            self._remove_action(state, "cancel")
                            percentage -= 100
                            state["status"] = "installing"
                            state["progress-percentage"] = percentage
                        if percentage > 100:
                            state["status"] = "installed"
                            del state["progress-percentage"]
                            self._installed_time[state["image-uri"]] = now
                            state["emi"] = "emi-" + str(int(time.time()))


STATE_TRACKER = StateTracker(IMAGES, STATES,
                             erroring_image_uris=[DJANGO_IMAGE["uri"]])


def build_answer(sections):
    answer = {}
    answer["sections"] = sections
    answer["images"] = images = []
    answer["states"] = states = []
    done = set()
    for section in sections:
        for image_uri in section["image-uris"]:
            if image_uri not in done:
                done.add(image_uri)
                for image in IMAGES:
                    if image["uri"] == image_uri:
                        images.append(image)
                        states.append(STATE_TRACKER.get_state(image_uri))
                        break
    for state in STATE_TRACKER.get_all_states():
        if (state["image-uri"] not in done and
            ("error-message" in state or
             state["status"] in ("installing", "downloading"))):
            states.append(state)
            for image in IMAGES:
                if image["uri"] == state["image-uri"]:
                    images.append(image)
    return answer


RFC3986_UNRESERVED = dict.fromkeys("ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                                   "abcdefghijklmnopqrstuvwxyz"
                                   "01234567890-_.~", True)
def _encode(value):
    return "".join((c in RFC3986_UNRESERVED and c or ("%%%02X" % ord(c)))
                   for c in value)

def generate_signature(secret_key, method, host, path, params):
    if params.get("SignatureVersion") != ["2"]:
        raise AssertionError("Unknown SignatureVersion: %s" %
                             params.get("SignatureVersion"))
    if params.get("SignatureMethod") != ["HmacSHA256"]:
        raise AssertionError("Unknown SignatureMethod: %s" %
                             params.get("SignatureMethod"))
    encoded_params = []
    for key, value in params.iteritems():
        encoded_key = _encode(key)
        if isinstance(value, list):
            for item in value:
                encoded_params.append((encoded_key, _encode(item)))
        else:
            encoded_params.append((encoded_key, _encode(value)))
    encoded_params.sort()
    canonical_params = "&".join("%s=%s" % pair for pair in encoded_params)
    canonical_payload = "%s\n%s\n%s\n%s" % (
        method, host, path, canonical_params)
    print "SIGNATURE PAYLOAD -----------------------------"
    print canonical_payload
    print "-----------------------------------------------"
    return base64.b64encode(hmac.HMAC(secret_key, canonical_payload,
                                      hashlib.sha256).digest())


class AuthenticationError(Exception): pass
class SignatureError(AuthenticationError): pass
class UnsupportedAPIVersionError(AuthenticationError): pass


def check_request_signature(request, secret_key, supported_api_versions):
    method = request.method
    host = request.getHeader("Host")
    path = request.path
    params = dict(request.args)
    signature_param = params.pop("Signature", None)
    signature = generate_signature(secret_key, method, host, path, params)

    if signature_param:
        signature_param = signature_param[0]
    if (signature_param != signature and
        signature_param != signature.rstrip("=")):
        raise SignatureError("Invalid authentication signature provided.")
    
    api_version = params.get("Version", None)
    if api_version:
        api_version = api_version[0]
    if api_version not in supported_api_versions:
        raise UnsupportedAPIVersionError("Provided API version %s "
                                         "is not supported." % (api_version,))

    # XXX Validate Expire header!



class DashboardResource(resource.Resource):

    isLeaf = True

    def render_GET(self, request):
        log("Loading dashboard")
        return json.dumps(build_answer(DASHBOARD_SECTIONS))


class SearchResource(resource.Resource):

    isLeaf = True

    def render_GET(self, request):
        params = dict(request.args)
        search_param = params.get("q", [None])[0]
        log("Searching for", search_param)

        sections = copy.deepcopy(SEARCH_SECTIONS)
        found_image_uris = sections[0]["image-uris"] 

        if search_param:
            search_param = search_param.lower()
            for image in IMAGES:
                if (search_param in image["title"].lower() or
                    search_param in image["summary"].lower() or
                    search_param in image["description-html"].lower()):
                    found_image_uris.append(image["uri"])

        if not found_image_uris:
            sections[0]["summary"] = "No results were found."

        time.sleep(1) # Simulate the handling of this request.

        return json.dumps(build_answer(sections))


class Root(resource.Resource):
    
    prefix = ["api"]

    isLeaf = False

    def getChild(self, path, request):
        request.prepath.pop()
        request.postpath.insert(0, path)

        if request.postpath[:len(self.prefix)] == self.prefix:
            del request.postpath[:len(self.prefix)]
            request.prepath.extend(self.prefix)
            return CloudRoot()
        return NotFound()


class NotFound(resource.Resource):

    isLeaf = False

    def render_GET(self, request):
        pass


class URIResourceBase(resource.Resource):

    def __init__(self, uri):
        resource.Resource.__init__(self)
        self._uri = uri


class StateResource(URIResourceBase):

    def render_GET(self, request):
        return json.dumps({"state": STATE_TRACKER.get_state(self._uri)})


class AuthenticatedResource(URIResourceBase):

    def render_authenticated(self, request):
        pass

    def render_POST(self, request):
        client_id = request.args.get("ClientId")
        try:
            if not client_id:
                raise AuthenticationError("ClientId was not provided")
            secret_key = CREDENTIALS.get(client_id[0])
            if secret_key is None:
                raise AuthenticationError("Provided ClientId %s is not known."
                                          % (client_id,))
            check_request_signature(request, secret_key, SUPPORTED_API_VERSIONS)
        except AuthenticationError, error:
            return json.dumps({"error-message": str(error)})
        return self.render_authenticated(request)


class InstallHandlerResource(AuthenticatedResource):

    def render_authenticated(self, request):
        time.sleep(2) # Simulate the handling of this request.
        STATE_TRACKER.install(self._uri)
        return StateResource(self._uri).render_GET(request)


class ClearErrorHandlerResource(AuthenticatedResource):

    def render_authenticated(self, request):
        time.sleep(1) # Simulate the handling of this request.
        STATE_TRACKER.clear_error(self._uri)
        return StateResource(self._uri).render_GET(request)


class CancelHandlerResource(AuthenticatedResource):

    def render_authenticated(self, request):
        time.sleep(2) # Simulate the handling of this request.
        STATE_TRACKER.cancel(self._uri)
        return StateResource(self._uri).render_GET(request)


class StatesResource(resource.Resource):

    def render_POST(self, request):
        params = dict(request.args)
        image_uris = params.get("image-uri", [])
        states = []
        for image_uri in image_uris:
            state = STATE_TRACKER.get_state(image_uri)
            if state:
                states.append(state)
        log("Returning %d states for %d image URIs" %
            (len(states), len(image_uris)))
        return json.dumps({"states": states})


class ImageResource(resource.Resource):

    def __init__(self, uri):
        resource.Resource.__init__(self)
        self._uri = uri

    def getChild(self, name, request):
        if name == "install":
            return InstallHandlerResource(self._uri)
        elif name == "clear-error":
            return ClearErrorHandlerResource(self._uri)
        elif name == "cancel":
            return CancelHandlerResource(self._uri)
        elif name == "state":
            return StateResource(self._uri)
        return NotFound()

    def render_GET(self, request):
        assert False, "Implement this."


class ImagesResource(resource.Resource):

    isLeaf = False

    def getChild(self, name, request):
        return ImageResource(base64.urlsafe_b64decode(name))


class CloudRoot(resource.Resource):

    def __init__(self):
        resource.Resource.__init__(self)
        self.putChild("dashboard", DashboardResource())
        self.putChild("search", SearchResource())
        self.putChild("images", ImagesResource())
        self.putChild("states", StatesResource())


if __name__ == "__main__":
    site = server.Site(Root())
    reactor.listenTCP(52780, site)
    reactor.run()
