/* -*- mode: C; c-basic-offset: 4; indent-tabs-mode: nil; -*- */
/* Copyright 2008 LiTL, LLC. All Rights Reserved. */

/* This file should be considered to be under the same terms as the
 * main upstream source code.
 */

#include <config.h>

#include "Clutter-custom.h"

ClutterColor*
clutter_color_new (void)
{
    return g_slice_new0(ClutterColor);
}

/**
 * clutter_color_get_components:
 * @color: A #ClutterColor
 *
 * Return value: (element-type int) (transfer container): A list
 * with the color components. Must be freed with g_list_free.
 */
GList *
clutter_color_get_components(ClutterColor *color)
{
    GList *list;

    list = NULL;
    list = g_list_prepend(list, GINT_TO_POINTER((gint)color->alpha));
    list = g_list_prepend(list, GINT_TO_POINTER((gint)color->blue));
    list = g_list_prepend(list, GINT_TO_POINTER((gint)color->green));
    list = g_list_prepend(list, GINT_TO_POINTER((gint)color->red));

    return list;
}

/**
 * clutter_color_set_components:
 * @color: A #ClutterColor
 * @components: (element-type int) (transfer none): A list of compoents
 */
void
clutter_color_set_components(ClutterColor *color,
                             GList *components)
{
    g_return_if_fail(g_list_length(components) == 4);

    color->red = GPOINTER_TO_INT(components->data);
    components = g_list_next(components);
    color->green = GPOINTER_TO_INT(components->data);
    components = g_list_next(components);
    color->blue = GPOINTER_TO_INT(components->data);
    components = g_list_next(components);
    color->alpha = GPOINTER_TO_INT(components->data);
}

#ifdef HAVE_CLUTTER_GDKPIXBUF
/**
 * clutter_texture_set_from_pixbuf:
 * @texture: A #ClutterTexture
 * @pixbuf: (transfer none): A #GdkPixbuf
 */
gboolean
clutter_texture_set_from_pixbuf (ClutterTexture *texture,
                                 GdkPixbuf      *pixbuf)
{
    return clutter_texture_set_from_rgb_data (texture,
                                              gdk_pixbuf_get_pixels (pixbuf),
                                              gdk_pixbuf_get_has_alpha (pixbuf),
                                              gdk_pixbuf_get_width (pixbuf),
                                              gdk_pixbuf_get_height (pixbuf),
                                              gdk_pixbuf_get_rowstride (pixbuf),
                                              gdk_pixbuf_get_has_alpha (pixbuf)
                                              ? 4 : 3,
                                              0, NULL);
}
#endif

ClutterGeometry *
clutter_geometry_new(void)
{
    return g_slice_new0(ClutterGeometry);
}

gint
clutter_geometry_get_x(ClutterGeometry *geometry)
{
    return geometry->x;
}

gint
clutter_geometry_get_y(ClutterGeometry *geometry)
{
    return geometry->y;
}

guint
clutter_geometry_get_width(ClutterGeometry *geometry)
{
    return geometry->width;
}

guint
clutter_geometry_get_height(ClutterGeometry *geometry)
{
    return geometry->height;
}

ClutterActorBox *
clutter_actor_box_new(void)
{
    return g_slice_new0(ClutterActorBox);
}

/* return is ClutterUnit, but gint32 in gidl, so we want a warning here if that's ever wrong */
gint32
clutter_actor_box_get_x1(ClutterActorBox *actor_box)
{
    return actor_box->x1;
}

gint32
clutter_actor_box_get_y1(ClutterActorBox *actor_box)
{
    return actor_box->y1;
}

gint32
clutter_actor_box_get_x2(ClutterActorBox *actor_box)
{
    return actor_box->x2;
}

gint32
clutter_actor_box_get_y2(ClutterActorBox *actor_box)
{
    return actor_box->y2;
}

/* we assume in the IDL that ClutterUnit is gint32, so use gint32 here to warn
 * if that ever breaks
 */
gint
clutter_units_to_device(gint32 units)
{
    return CLUTTER_UNITS_TO_DEVICE(units);
}

gint32
clutter_device_to_units(gint device)
{
    return CLUTTER_UNITS_FROM_DEVICE(device);
}

gint
clutter_event_get_x(ClutterEvent *event)
{
    int x;

    clutter_event_get_coords(event, &x, NULL);

    return x;
}

gint
clutter_event_get_y(ClutterEvent *event)
{
    int y;

    clutter_event_get_coords(event, NULL, &y);

    return y;
}

guint16
clutter_event_get_code(ClutterEvent *event)
{
    g_return_val_if_fail(event->type == CLUTTER_KEY_PRESS ||
                         event->type == CLUTTER_KEY_RELEASE,
                         0);

    return event->key.hardware_keycode;
}

guint
clutter_event_get_symbol(ClutterEvent *event)
{
    g_return_val_if_fail(event->type == CLUTTER_KEY_PRESS ||
                         event->type == CLUTTER_KEY_RELEASE,
                         0);

    return event->key.keyval;
}

guint32
clutter_event_get_unicode(ClutterEvent *event)
{
    g_return_val_if_fail(event->type == CLUTTER_KEY_PRESS ||
                         event->type == CLUTTER_KEY_RELEASE,
                         0);

    return clutter_key_event_unicode((ClutterKeyEvent *)event);
}

ClutterVertex*
clutter_vertex_new(void)
{
    return g_slice_new0(ClutterVertex);
}

void
clutter_vertex_set_values(ClutterVertex *vertex,
                          int            x,
                          int            y,
                          int            z)
{
    vertex->x = x;
    vertex->y = y;
    vertex->z = z;
}
