# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Benjamin Kampmann <benjamin@fluendo.com>'

# the widgets
from raval_widgets.popup import PopupButton, Popup
import pgm

# the view
from elisa.plugins.pigment.pigment_view import PigmentView

# the animation framework
from pgm.timing.implicit import AnimatedObject

# the signal
from elisa.core.utils.signal import Signal

class PopupView(PigmentView, Popup):
    """
    A Popup with a OK-Button in the middle.

    @ivar ok_signal: gets emited, when the ok-button got pressed
    @type ok_signal: L{elisa.core.utils.signal.Signal}
    """
    supported_controllers = ('raval:popup_controller',)

    current = None

    bindings = (('label', 'label'),
                ('left', 'left'), # FIXME: passthrough hacks!
                ('right', 'right'),
                ('enter', 'enter'),
                ('signal', 'signal'))

    def __init__(self):
        self.animated = AnimatedObject(self)
        super(PopupView, self).__init__()

    def left__set(self, value):
        self.prev()

    def right__set(self, value):
        self.next()

    def enter__set(self, value):
        self.trigger()

    def create_widgets(self):
        """
        Loads the images from the theme:
            * popup_background for the background of the popup
            * yes_selected for the selected ok-button
            * yes_unselected for teh unselected ok-button
        """
        canvas = self.frontend.context.viewport_handle.get_canvas()
        self.canvas = canvas

        self.position = (canvas.width/4.0, canvas.height/3.0, +1.0)
        self.size = (canvas.width/2., canvas.height/3.)
        self.background.bg_color = (0, 0, 0, 0)
        self.load_from_theme('popup_background', self.background)
        self.background.layout = pgm.IMAGE_FILLED

        # add the buttons
        yes_btn = PopupButton()
        self.load_from_theme('yes_selected', yes_btn.selected_image)
        self.load_from_theme('yes_unselected', yes_btn.unselected_image)
        yes_btn.connect('clicked', self._emit_yes_signal)
        self.append_button(yes_btn)

        no_btn = PopupButton()
        self.load_from_theme('no_selected', no_btn.selected_image)
        self.load_from_theme('no_unselected', no_btn.unselected_image)
        no_btn.connect('clicked', self._emit_no_signal)
        self.append_button(no_btn)

        self.text.fg_color = (233, 233, 233, 255)
        self.text.z = +1.1
        self.lines = 2
        self.visible = False
    
    def label__set(self, text):
        self.text.label = self.frontend.translate_text(text)

    def visible__set(self, value):
        if value:
            super(PopupView, self).visible__set(value)
            self.animated.opacity = 255
        else:
            self.animated.opacity = 0

    def _emit_yes_signal(self, *args):
        if self.signal:
            self.signal.emit(1)

    def _emit_no_signal(self, *args):
        if self.signal:
            self.signal.emit(0)

