// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include "main.h"

template< typename T, int Size, typename MatrixType, typename VectorType >
void helper_checkImplicitLU( const char *strType )
{
    MatrixType mat1(Size), mat2(Size), mat3(Size);
    VectorType vec1(Size), vec2(Size);
    do pickRandomMatrix( mat1 ); while( ! mat1.isInvertible() );
    mat2.loadIdentity();
    mat1.computeInverse( & mat3 );
    QVERIFY( compareMatrices( mat2, mat1 * mat3, strType ) );
    mat3 = mat1.inverse();
    QVERIFY( compareMatrices( mat2, mat1 * mat3, strType ) );
    QVERIFY( Size == mat3.rank() );
    QVERIFY( 0 == mat3.dimKer() );
    do pickRandomMatrix( mat2 ); while( ! mat2.isInvertible() );
    QVERIFY( test_isApprox_relax( (mat1 * mat2).determinant(), mat1.determinant() * mat2.determinant() ) );
    bool invertible;
    mat3.loadIdentity();
    mat2.computeInverseSafely( & mat1, & invertible );
    QVERIFY( invertible );
    QVERIFY( compareMatrices( mat1*mat2, mat3, strType ) );
    mat2.loadIdentity();
    mat2(0,0) = static_cast<T>(0);
    QVERIFY( ! mat2.isInvertible() );
    mat2.computeInverseSafely( & mat1, & invertible );
    QVERIFY( ! invertible );
}

void MainTest::checkImplicitLU()
{
#define HELPER_CHECKIMPLICITLU_FIX( T, Size ) \
    helper_checkImplicitLU< T, Size, \
        Matrix< T, Size >, Vector< T, Size > >\
            ("fixed<"__STRING(T)","__STRING(Size)">")

#define HELPER_CHECKIMPLICITLU_DYN( T, Size ) \
    helper_checkImplicitLU< T, Size, \
        MatrixX<T>, VectorX<T> >\
            ("dynamic<"__STRING(T)">("__STRING(Size)")")

    for( int repeat = 0; repeat < REPEAT; repeat++ )
    {
        HELPER_CHECKIMPLICITLU_FIX( float, 1 );
        HELPER_CHECKIMPLICITLU_FIX( double, 2 );
        HELPER_CHECKIMPLICITLU_FIX( float, 3 );
        HELPER_CHECKIMPLICITLU_FIX( double, 4 );
        HELPER_CHECKIMPLICITLU_FIX( double, 7 );
        HELPER_CHECKIMPLICITLU_FIX( float, 8 );
        HELPER_CHECKIMPLICITLU_FIX( double, 12 );
        HELPER_CHECKIMPLICITLU_FIX( complex<double>, 7 );
        HELPER_CHECKIMPLICITLU_FIX( complex<float>, 8 );
        HELPER_CHECKIMPLICITLU_FIX( complex<double>, 12 );
    
        HELPER_CHECKIMPLICITLU_DYN( float, 1 );
        HELPER_CHECKIMPLICITLU_DYN( double, 2 );
        HELPER_CHECKIMPLICITLU_DYN( float, 3 );
        HELPER_CHECKIMPLICITLU_DYN( double, 4 );
        HELPER_CHECKIMPLICITLU_DYN( double, 7 );
        HELPER_CHECKIMPLICITLU_DYN( float, 8 );
        HELPER_CHECKIMPLICITLU_DYN( double, 12 );
        HELPER_CHECKIMPLICITLU_DYN( complex<double>, 7 );
        HELPER_CHECKIMPLICITLU_DYN( complex<float>, 8 );
        HELPER_CHECKIMPLICITLU_DYN( complex<double>, 12 );
    }
}

