"""testing desktopcouch/local_files.py module"""

import os
import shutil
import tempfile

import testtools
import desktopcouch.tests as test_environment
import desktopcouch

import gnomekeyring


class TestLocalFiles(testtools.TestCase):
    """Testing that local files returns the right things"""

    def setUp(self):
        super(TestLocalFiles, self).setUp()
        cf = test_environment.test_context.configuration
        # Test loading from file.
        cf._fill_from_file(test_environment.test_context.file_ini)

    def test_all_files_returned(self):
        "Does local_files list all the files that it needs to?"
        import desktopcouch.local_files
        for required in [
                "file_log", "file_ini", "file_pid", "file_stdout",
                "file_stderr", "db_dir"]:
            #"couch_exe", "couch_exec_command"
            self.assertTrue(required in dir(test_environment.test_context))

    def test_xdg_overwrite_works(self):
        # this should really check that it's in os.environ["TMP"]
        self.assertTrue(
            test_environment.test_context.file_ini.startswith("/tmp"))

    def test_couch_chain_ini_files(self):
        "Is compulsory-auth.ini picked up by the ini file finder?"
        import desktopcouch.local_files
        ok = [x for x
                in test_environment.test_context.couch_chain_ini_files().split()
                if x.endswith("compulsory-auth.ini")]
        self.assertTrue(len(ok) > 0)

    def test_bind_address(self):
        old = desktopcouch.local_files.get_bind_address()
        octets = old.split(".")
        octets[2] = str((int(octets[2]) + 128) % 256)
        new = ".".join(octets)
        desktopcouch.local_files.set_bind_address(new)

        self.assertEquals(desktopcouch.local_files.get_bind_address(), new)
        desktopcouch.local_files.set_bind_address(old)
        self.assertEquals(desktopcouch.local_files.get_bind_address(), old)


class TestKeyringIntegration(testtools.TestCase):

    def setUp(self):
        super(TestKeyringIntegration, self).setUp()
        self.tempdir = tempfile.mkdtemp()
        self.addCleanup(shutil.rmtree, self.tempdir)
        self.cache_dir = os.path.join(self.tempdir, 'cache')
        self.data_dir = os.path.join(self.tempdir, 'data')
        self.config_dir = os.path.join(self.tempdir, 'config')

    def get_context(self, keyring=None):
        return desktopcouch.local_files.Context(self.cache_dir, self.data_dir,
                self.config_dir, keyring=keyring)

    def test_no_keyring_wanted(self):
        """Check that nothing crashes when we don't want a keyring."""
        context = self.get_context()
        # also check that credentials are still created
        desktopcouch.local_files.get_admin_credentials(ctx=context)
        desktopcouch.local_files.get_oauth_tokens(ctx=context)

    def test_no_keyring_present(self):
        """Check that it doesn't crash if the keyring isn't present."""
        keyring = test_environment.TestKeyring(appear_absent=True)
        context = self.get_context(keyring=keyring)
        # but the credentials are still created
        desktopcouch.local_files.get_admin_credentials(ctx=context)
        desktopcouch.local_files.get_oauth_tokens(ctx=context)

    def test_admin_creds_already_present(self):
        """Test that the admin credentials are reused."""
        keyring = test_environment.TestKeyring()
        keyring.item_create_sync(None, gnomekeyring.ITEM_GENERIC_SECRET,
                'Desktop Couch user authentication',
                {'desktopcouch': 'basic'},
                'foo:bar', True)
        context = self.get_context(keyring=keyring)
        # Check that what is written to the file is what we put in gnome-keyring
        admin_creds = desktopcouch.local_files.get_admin_credentials(
                ctx=context)
        self.assertEqual(('foo', 'bar'), admin_creds)
        # And check that the oauth creds are stored
        self.check_oauth_creds(context)

    def check_oauth_creds(self, context):
        """Check that the oauth creds in the keyring and on disk match."""
        oauth_creds = context.keyring.find_items_sync(
                gnomekeyring.ITEM_GENERIC_SECRET,
                {'desktopcouch': 'oauth'})[0].secret.split(':')
        keyring_creds = {
            'token': oauth_creds[2],
            'token_secret': oauth_creds[3],
            'consumer_key': oauth_creds[0],
            'consumer_secret': oauth_creds[1],
        }
        file_creds = desktopcouch.local_files.get_oauth_tokens(ctx=context)
        self.assertEqual(file_creds, keyring_creds)

    def test_admin_creds_saved(self):
        """Test that the admin credentials are saved in the keyring."""
        keyring = test_environment.TestKeyring()
        context = self.get_context(keyring=keyring)
        # Check that what is written to the file is also in gnome-keyring
        file_creds = desktopcouch.local_files.get_admin_credentials(
                ctx=context)
        keyring_creds = keyring.find_items_sync(
                gnomekeyring.ITEM_GENERIC_SECRET,
                {'desktopcouch': 'basic'})[0].secret
        self.assertEqual(keyring_creds, ":".join(file_creds))
        self.check_oauth_creds(context)
