/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by:
 *               Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

/**
 * SECTION:dee-versioned-model
 * @short_description: Abstract base class for easing implementations of
 *                     #DeeModel<!-- -->s providing a unique version number
 *                     for each row
 * @include: dee.h
 *
 * #DeeSerializableModel is an abstract base class to ease implementation of
 * #DeeModel<!-- -->s providing rows versioned by a
 * <emphasis>sequence number</emphasis>.
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <memory.h>
#include <time.h>
#include <unistd.h>

#include "dee-model.h"
#include "dee-serializable-model.h"
#include "dee-serializable.h"
#include "dee-marshal.h"
#include "trace-log.h"

static void     dee_serializable_model_model_iface_init (DeeModelIface *iface);
static void     dee_serializable_model_serializable_iface_init (DeeSerializableIface *iface);
static GObject* dee_serializable_model_parse_serialized (GVariant *data);
G_DEFINE_ABSTRACT_TYPE_WITH_CODE (DeeSerializableModel,
                                  dee_serializable_model,
                                  G_TYPE_OBJECT,
                                  G_IMPLEMENT_INTERFACE (DEE_TYPE_MODEL,
                                                         dee_serializable_model_model_iface_init)
                                  G_IMPLEMENT_INTERFACE (DEE_TYPE_SERIALIZABLE,
                                                         dee_serializable_model_serializable_iface_init));

#define DEE_SERIALIZABLE_MODEL_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, DEE_TYPE_SERIALIZABLE_MODEL, DeeSerializableModelPrivate))

/**
 * DeeSerializableModelPrivate:
 *
 * Ignore this structure.
 */
struct _DeeSerializableModelPrivate
{
  /* Seqnum tracking */
  guint64     seqnum;

  /* Column type info */
  guint       n_columns;
  gchar     **column_schemas; // NULL terminated
  
};

/*
 * Public overridable DeeSerializableModel methods
 */
static guint64   dee_serializable_model_get_seqnum_real  (DeeModel     *self);

static void      dee_serializable_model_set_seqnum_real  (DeeModel     *self,
                                                       guint64       seqnum);

static guint64   dee_serializable_model_inc_seqnum_real  (DeeModel     *self);

/*
 * DeeModel forward declarations
 */
static const gchar* const*   dee_serializable_model_get_schema (DeeModel *self,
                                                             guint    *num_columns);

static const gchar*   dee_serializable_model_get_column_schema (DeeModel *self,
                                                             guint     column);

static void           dee_serializable_model_set_schema_full  (DeeModel           *self,
                                                            const gchar* const *column_schemas,
                                                            guint               num_columns);

static guint          dee_serializable_model_get_n_columns  (DeeModel *self);

static guint          dee_serializable_model_get_n_rows     (DeeModel *self);

static void           dee_serializable_model_clear          (DeeModel *self);

static DeeModelIter*  dee_serializable_model_append_row  (DeeModel  *self,
                                                       GVariant **row_members);

static DeeModelIter*  dee_serializable_model_prepend_row  (DeeModel  *self,
                                                        GVariant **row_members);

static DeeModelIter*  dee_serializable_model_insert_row  (DeeModel  *self,
                                                       guint      pos,
                                                       GVariant **row_members);

static DeeModelIter*  dee_serializable_model_insert_row_before (DeeModel      *self,
                                                             DeeModelIter  *iter,
                                                             GVariant     **row_members);

static void           dee_serializable_model_remove         (DeeModel     *self,
                                                          DeeModelIter *iter);

static void           dee_serializable_model_set_value      (DeeModel       *self,
                                                          DeeModelIter   *iter,
                                                          guint           column,
                                                          GVariant       *value);

static void           dee_serializable_model_set_row            (DeeModel       *self,
                                                              DeeModelIter   *iter,
                                                              GVariant      **row_members);

static GVariant*     dee_serializable_model_get_value      (DeeModel     *self,
                                                          DeeModelIter *iter,
                                                          guint         column);

static DeeModelIter* dee_serializable_model_get_first_iter  (DeeModel     *self);

static DeeModelIter* dee_serializable_model_get_last_iter   (DeeModel     *self);

static DeeModelIter* dee_serializable_model_get_iter_at_row (DeeModel     *self,
                                                         guint          row);

static gboolean       dee_serializable_model_get_bool       (DeeModel    *self,
                                                         DeeModelIter *iter,
                                                         guint         column);

static guchar         dee_serializable_model_get_uchar      (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static gint32         dee_serializable_model_get_int32     (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static guint32        dee_serializable_model_get_uint32    (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static gint64         dee_serializable_model_get_int64      (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static guint64        dee_serializable_model_get_uint64     (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static gdouble        dee_serializable_model_get_double     (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static const gchar*   dee_serializable_model_get_string     (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static DeeModelIter* dee_serializable_model_next            (DeeModel     *self,
                                                         DeeModelIter *iter);

static DeeModelIter* dee_serializable_model_prev            (DeeModel     *self,
                                                         DeeModelIter *iter);

static gboolean       dee_serializable_model_is_first       (DeeModel     *self,
                                                         DeeModelIter *iter);

static gboolean       dee_serializable_model_is_last        (DeeModel     *self,
                                                         DeeModelIter *iter);

/* GObject Init */
static void
dee_serializable_model_finalize (GObject *object)
{
  DeeSerializableModelPrivate *priv = DEE_SERIALIZABLE_MODEL (object)->priv;

  priv->n_columns = 0;
  priv->seqnum = 0;

  if (priv->column_schemas != NULL)
    {
      g_strfreev (priv->column_schemas);
      priv->column_schemas = NULL;
    }

  G_OBJECT_CLASS (dee_serializable_model_parent_class)->finalize (object);
}

static void
dee_serializable_model_set_property (GObject      *object,
                                 guint         id,
                                 const GValue *value,
                                 GParamSpec   *pspec)
{
  switch (id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_serializable_model_get_property (GObject    *object,
                                 guint       id,
                                 GValue     *value,
                                 GParamSpec *pspec)
{
  switch (id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_serializable_model_class_init (DeeSerializableModelClass *klass)
{
  GObjectClass  *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize     = dee_serializable_model_finalize;  
  obj_class->set_property = dee_serializable_model_set_property;
  obj_class->get_property = dee_serializable_model_get_property;

  klass->get_seqnum = dee_serializable_model_get_seqnum_real;
  klass->set_seqnum = dee_serializable_model_set_seqnum_real;
  klass->inc_seqnum = dee_serializable_model_inc_seqnum_real;

  /* Add private data */
  g_type_class_add_private (obj_class, sizeof (DeeSerializableModelPrivate));
}

static void
dee_serializable_model_init (DeeSerializableModel *model)
{
  DeeSerializableModelPrivate *priv;

  priv = model->priv = DEE_SERIALIZABLE_MODEL_GET_PRIVATE (model);

  priv->seqnum = 0;

  priv->n_columns = 0;
  priv->column_schemas = NULL;
  
  
}

static guint64
dee_serializable_model_get_seqnum_real (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);
  
  return DEE_SERIALIZABLE_MODEL (self)->priv->seqnum;
}
                                                          
guint64
dee_serializable_model_get_seqnum (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);

  return DEE_SERIALIZABLE_MODEL_GET_CLASS (self)->get_seqnum (self);
}

static void
dee_serializable_model_set_seqnum_real (DeeModel *self,
                                          guint64   seqnum)
{
  g_return_if_fail (DEE_IS_SERIALIZABLE_MODEL (self));
    
  DEE_SERIALIZABLE_MODEL (self)->priv->seqnum = seqnum;
}
                                                          
void
dee_serializable_model_set_seqnum (DeeModel *self,
                                     guint64   seqnum)
{
  g_return_if_fail (DEE_IS_SERIALIZABLE_MODEL (self));

  DEE_SERIALIZABLE_MODEL_GET_CLASS (self)->set_seqnum (self, seqnum);
}

static guint64
dee_serializable_model_inc_seqnum_real (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);

  return ++DEE_SERIALIZABLE_MODEL (self)->priv->seqnum;
}

guint64
dee_serializable_model_inc_seqnum (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);

  return DEE_SERIALIZABLE_MODEL_GET_CLASS (self)->inc_seqnum (self);
}

/*
 * DeeModel API
 */

static void
dee_serializable_model_set_schema_full  (DeeModel           *self,
                                      const gchar* const *column_schemas,
                                      guint               num_columns)
{
  DeeSerializableModelPrivate  *priv;
  gchar                    **column_schemas_copy;
  guint                      i;
  
  g_return_if_fail (DEE_IS_SERIALIZABLE_MODEL (self));
  g_return_if_fail (column_schemas != NULL);

  priv = DEE_SERIALIZABLE_MODEL (self)->priv;

  if (priv->column_schemas != NULL)
    {
      g_critical ("The DeeModel %p already has a schema", self);
      return;
    }

  /* Allocate space to store the column schemas. We NULL terminate it
   * in order to play well with g_strfreev() */
  column_schemas_copy = g_new0 (gchar*, num_columns + 1);

  /* Validate the type strings and copy the schema for our selves */
  for (i = 0; i < num_columns; i++)
    {
      if (!g_variant_type_string_is_valid (column_schemas[i]))
        {
          g_critical ("When setting schema for DeeModel %p: '%s' is not a "
                      "valid type string", self, column_schemas[i]);
          return;
        }
      column_schemas_copy[i] = g_strdup (column_schemas[i]);
    }

  /* Register the validated types as our column types */
  priv->column_schemas = column_schemas_copy; // steal
  priv->n_columns = num_columns;

#ifdef ENABLE_TRACE_LOG
  gchar* schema = g_strjoinv (", ", priv->column_schemas);
  trace_object (self, "Set schema: (%s)", schema);
  g_free (schema);
#endif
}

static const gchar* const*
dee_serializable_model_get_schema (DeeModel *self,
                                guint    *n_columns)
{
  DeeSerializableModelPrivate *priv;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), NULL);

  priv = DEE_SERIALIZABLE_MODEL (self)->priv;

  if (n_columns != NULL)
    *n_columns = priv->n_columns;

  return (const gchar**) priv->column_schemas;
}

static const gchar*
dee_serializable_model_get_column_schema (DeeModel *self,
                                       guint     column)
{
  DeeSerializableModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), NULL);

  priv = DEE_SERIALIZABLE_MODEL (self)->priv;
  g_return_val_if_fail (column < priv->n_columns, NULL);

  return priv->column_schemas[column];
}

static guint
dee_serializable_model_get_n_columns (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);

  return DEE_SERIALIZABLE_MODEL (self)->priv->n_columns;
}

static guint
dee_serializable_model_get_n_rows (DeeModel *self)
{
  DeeModelIter *iter;
  guint         count;

  count = 0;
  iter = dee_model_get_first_iter (self);
  while (!dee_model_is_last (self, iter))
    {
      iter = dee_model_next (self, iter);
      count++;
    }

  return count;
}

static void
dee_serializable_model_clear (DeeModel *self)
{
  DeeModelIter            *iter;

  g_return_if_fail (DEE_IS_SERIALIZABLE_MODEL (self));

  iter = dee_model_get_first_iter (self);

  while (!dee_model_is_last (self, iter))
    {
      dee_model_remove (self, iter);

      iter = dee_model_get_first_iter (self);      
    }
}

static DeeModelIter*
dee_serializable_model_prepend_row (DeeModel  *self,
                                 GVariant **row_members)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static DeeModelIter*
dee_serializable_model_append_row (DeeModel  *self,
                                GVariant **row_members)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static DeeModelIter*
dee_serializable_model_insert_row (DeeModel  *self,
                                guint      pos,
                                GVariant **row_members)
{
  DeeModelIter *iter;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);

  if (pos > 0)
    {
      iter = dee_model_get_iter_at_row (self, pos);
      return dee_model_insert_row_before (self, iter, row_members);
    }
  else if (pos == 0)
    return dee_model_prepend_row (self, row_members);
  else
    return dee_model_append_row (self, row_members);
}

static DeeModelIter*
dee_serializable_model_insert_row_before (DeeModel      *self,
                                       DeeModelIter  *iter,
                                       GVariant     **row_members)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static void
dee_serializable_model_remove (DeeModel     *self,
                            DeeModelIter *iter_)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

static void
dee_serializable_model_set_value (DeeModel       *self,
                               DeeModelIter   *iter,
                               guint           column,
                               GVariant       *value)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

static void
dee_serializable_model_set_row (DeeModel       *self,
                             DeeModelIter   *iter,
                             GVariant      **row_members)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

static GVariant*
dee_serializable_model_get_value (DeeModel     *self,
                               DeeModelIter *iter,
                               guint         column)
{
  g_critical ("%s not implemented", G_STRFUNC);

  return NULL;
}

static gboolean
dee_serializable_model_get_bool (DeeModel     *self,
                                 DeeModelIter *iter,
                                 guint         column)
{
  GVariant *value;
  gboolean  b;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), FALSE);

  value = dee_model_get_value (self, iter, column);

  if (G_UNLIKELY (value == NULL))
    {
      g_critical ("Failed to retrieve bool from row %u column %u in %s@%p",
                  dee_model_get_position (self, iter), column,
                  G_OBJECT_TYPE_NAME (self), self);
      return FALSE;
    }

  b = g_variant_get_boolean (value);
  g_variant_unref (value);

  return b;
}

static guchar
dee_serializable_model_get_uchar (DeeModel      *self,
                                  DeeModelIter  *iter,
                                  guint          column)
{
  GVariant *value;
  guchar    u;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), '\0');

  value = dee_model_get_value (self, iter, column);

  if (G_UNLIKELY (value == NULL))
    {
      g_critical ("Failed to retrieve uchar from row %u column %u in %s@%p",
                  dee_model_get_position (self, iter), column,
                  G_OBJECT_TYPE_NAME (self), self);
      return '\0';
    }

  u = g_variant_get_byte(value);
  g_variant_unref (value);

  return u;
}

static gint32
dee_serializable_model_get_int32 (DeeModel        *self,
                                  DeeModelIter    *iter,
                                  guint            column)
{
  GVariant *value;
  gint32    i;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);

  value = dee_model_get_value (self, iter, column);

  if (G_UNLIKELY (value == NULL))
    {
      g_critical ("Failed to retrieve int64 from row %u column %u in %s@%p",
                  dee_model_get_position (self, iter), column,
                  G_OBJECT_TYPE_NAME (self), self);
      return 0;
    }

  i = g_variant_get_int32 (value);
  g_variant_unref (value);

  return i;
}

static guint32
dee_serializable_model_get_uint32 (DeeModel      *self,
                                   DeeModelIter  *iter,
                                   guint          column)
{
  GVariant *value;
  guint32    u;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);

  value = dee_model_get_value (self, iter, column);

  if (G_UNLIKELY (value == NULL))
    {
      g_critical ("Failed to retrieve uint32 from row %u column %u in %s@%p",
                  dee_model_get_position (self, iter), column,
                  G_OBJECT_TYPE_NAME (self), self);
      return 0;
    }

  u = g_variant_get_uint32 (value);
  g_variant_unref (value);

  return u;
}


static gint64
dee_serializable_model_get_int64 (DeeModel      *self,
                                  DeeModelIter  *iter,
                                  guint          column)
{
  GVariant *value;
  gint64    i;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self),
                        G_GINT64_CONSTANT (0));

  value = dee_model_get_value (self, iter, column);

  if (G_UNLIKELY (value == NULL))
    {
      g_critical ("Failed to retrieve int64 from row %u column %u in %s@%p",
                  dee_model_get_position (self, iter), column,
                  G_OBJECT_TYPE_NAME (self), self);
      return G_GINT64_CONSTANT (0);
    }

  i = g_variant_get_int64 (value);
  g_variant_unref (value);

  return i;
}


static guint64
dee_serializable_model_get_uint64 (DeeModel      *self,
                                   DeeModelIter  *iter,
                                   guint          column)
{
  GVariant *value;
  guint64   u;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self),
                        G_GUINT64_CONSTANT (0));

  value = dee_model_get_value (self, iter, column);

  if (G_UNLIKELY (value == NULL))
    {
      g_critical ("Failed to retrieve uint64 from row %u column %u in %s@%p",
                  dee_model_get_position (self, iter), column,
                  G_OBJECT_TYPE_NAME (self), self);
      return G_GUINT64_CONSTANT (0);
    }

  u = g_variant_get_uint64 (value);
  g_variant_unref (value);

  return u;
}

static gdouble
dee_serializable_model_get_double (DeeModel      *self,
                                   DeeModelIter  *iter,
                                   guint          column)
{
  GVariant *value;
  gdouble   d;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);

  value = dee_model_get_value (self, iter, column);

  if (G_UNLIKELY (value == NULL))
    {
      g_critical ("Failed to retrieve double from row %u column %u in %s@%p",
                  dee_model_get_position (self, iter), column,
                  G_OBJECT_TYPE_NAME (self), self);
      return 0;
    }

  d = g_variant_get_double (value);
  g_variant_unref (value);

  return d;
}

static const gchar*
dee_serializable_model_get_string (DeeModel      *self,
                                   DeeModelIter  *iter,
                                   guint          column)
{
  GVariant    *value;
  const gchar *s;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), NULL);

  value = dee_model_get_value (self, iter, column);

  if (G_UNLIKELY (value == NULL))
    {
      g_critical ("Failed to retrieve string from row %u column %u in %s@%p",
                  dee_model_get_position (self, iter), column,
                  G_OBJECT_TYPE_NAME (self), self);
      return NULL;
    }

  s = g_variant_get_string (value, NULL);
  g_variant_unref (value);

  return s;
}

static DeeModelIter*
dee_serializable_model_get_first_iter (DeeModel     *self)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static DeeModelIter*
dee_serializable_model_get_last_iter (DeeModel *self)
{
  DeeModelIter *iter;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), NULL);

  iter = dee_model_get_first_iter (self);
  while (!dee_model_is_last (self, iter))
    iter = dee_model_next (self, iter);

  return iter;
}

static DeeModelIter*
dee_serializable_model_get_iter_at_row (DeeModel *self,
                                     guint     row)
{
  DeeModelIter *iter;
  guint         pos;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), NULL);

  pos = 0;
  iter = dee_model_get_first_iter (self);
  while (!dee_model_is_last (self, iter) && pos < row)
    iter = dee_model_next (self, iter);
    pos++;

  if (dee_model_is_last (self, iter))
    {
      g_critical ("Index %u is out of bounds in model of size %u",
                  row, pos);
    }

  return iter;
}

static DeeModelIter*
dee_serializable_model_next (DeeModel     *self,
                          DeeModelIter *iter)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static DeeModelIter*
dee_serializable_model_prev (DeeModel     *self,
                          DeeModelIter *iter)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static gboolean
dee_serializable_model_is_first (DeeModel     *self,
                              DeeModelIter *iter)
{
  DeeModelIter *first;
  
  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), FALSE);

  first = dee_model_get_first_iter (self);
  return first == iter;
}

static gboolean
dee_serializable_model_is_last (DeeModel     *self,
                             DeeModelIter *iter)
{
  DeeModelIter *last;
  
  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), FALSE);

  last = dee_model_get_last_iter (self);
  return last == iter;
}

static guint
dee_serializable_model_get_position (DeeModel     *self,
                                  DeeModelIter *iter)
{
  DeeModelIter *_iter;
  guint          pos;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), 0);

  pos = 0;
  _iter = dee_model_get_first_iter (self);
  while (!dee_model_is_last (self, iter) && iter != _iter)
    _iter = dee_model_next (self, _iter);
    pos++;

  if (iter == _iter)
    return pos;
  else
    {
      g_critical ("Can not find position of unknown iter %p", iter);
      return -1;
    }
}

static DeeModelTag*
dee_serializable_model_register_tag    (DeeModel       *self,
                                        GDestroyNotify  tag_destroy)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static gpointer
dee_serializable_model_get_tag (DeeModel       *self,
                            DeeModelIter   *iter,
                            DeeModelTag    *tag)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static void
dee_serializable_model_set_tag (DeeModel       *self,
                            DeeModelIter   *iter,
                            DeeModelTag    *tag,
                            gpointer        value)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

/* Build a '(sasaavauay(tt))' suitable for sending in a Clone response */
static GVariant*
dee_serializable_model_serialize (DeeSerializable *self)
{
  DeeModel               *_self;
  GVariantBuilder         aav, clone;
  GVariant               *val, *tt, *schema;
  DeeModelIter           *iter;
  guint                   i, j, n_columns;
  guint64                 last_seqnum;

  g_return_val_if_fail (DEE_IS_SERIALIZABLE_MODEL (self), FALSE);

  trace_object (self, "Building clone");

  _self = DEE_MODEL (self);
  n_columns = dee_model_get_n_columns (_self);

  g_variant_builder_init (&aav, G_VARIANT_TYPE ("aav"));

  /* Clone the rows */
  i = 0;
  iter = dee_model_get_first_iter (_self);
  while (!dee_model_is_last (_self, iter))
    {
      g_variant_builder_open (&aav, G_VARIANT_TYPE ("av"));
      for (j = 0; j < n_columns; j++)
        {
          val = dee_model_get_value (_self, iter, j);
          g_variant_builder_add_value (&aav, g_variant_new_variant (val));
          g_variant_unref (val);
        }
      g_variant_builder_close (&aav);

      iter = dee_model_next (_self, iter);
      i++;
    }

  /* Collect the schema */
  schema = g_variant_new_strv (dee_model_get_schema(_self, NULL), -1);

  /* Collect the seqnum */
  last_seqnum = dee_serializable_model_get_seqnum (_self);
  tt = g_variant_new ("(tt)", last_seqnum - i, last_seqnum);

  /* Build the final clone */
  g_variant_builder_init (&clone, G_VARIANT_TYPE ("(asaav(tt))"));
  g_variant_builder_add_value (&clone, schema);
  g_variant_builder_add_value (&clone, g_variant_builder_end (&aav));
  g_variant_builder_add_value (&clone, tt);

  trace_object (self, "Serialized with %i rows", dee_model_get_n_rows (_self));

  return g_variant_builder_end (&clone);
}

static GObject*
dee_serializable_model_parse_serialized (GVariant *data)
{
  DeeModel      *model;
  GVariant      *schemav, *seqnumsv, *col;
  GVariantIter  *row_iter, *col_iter;
  GVariant     **row;
  const gchar  **schemas;
  gsize          n_cols, i, j;
  guint64        seqnum_start, seqnum_end;
  static GType   default_model_type = G_TYPE_INVALID;

  if (default_model_type == G_TYPE_INVALID)
    {
      default_model_type = g_type_from_name ("DeeSequenceModel");
      if (default_model_type == 0)
        {
          g_critical ("Unable to look up default DeeModel type, DeeSequenceModel, for deserialization");
          return NULL;
        }
    }

  g_variant_get (data, "(@asaav@(tt))", &schemav, &row_iter, &seqnumsv);
  schemas = g_variant_get_strv (schemav, &n_cols);
  g_variant_get (seqnumsv, "(tt)", &seqnum_start, &seqnum_end);

  model = DEE_MODEL (g_object_new (default_model_type, NULL));
  dee_model_set_schema_full (model, schemas, n_cols);
  dee_serializable_model_set_seqnum (model, seqnum_start);

  /* Note: The 'row' variable is stack allocated. No need to free it */
  row = g_alloca (n_cols * sizeof (GVariant *));

  i = 0;
  while (g_variant_iter_next (row_iter, "av", &col_iter))
    {
      if (g_variant_iter_n_children (col_iter) != n_cols)
        {
          g_warning ("Row %"G_GSIZE_FORMAT" of serialized DeeSerializableModel "
                     "data has illegal length %"G_GSIZE_FORMAT". Expected %"
                     G_GSIZE_FORMAT, i, g_variant_iter_n_children (col_iter),
                     n_cols);
          /* Just skip this row - parsers for DeeSerializable should
           * generally never return NULL */
          continue;
        }

      j = 0;
      while (g_variant_iter_loop (col_iter, "v", &col))
        {
          row[j] = g_variant_ref (col);
          j++;
        }

      dee_model_append_row (model, row);

      for (j = 0; j < n_cols; j++)
        {
          g_variant_unref (row[j]);
        }

      i++;
      g_variant_iter_free (col_iter);
    }
  g_variant_iter_free (row_iter);

  g_variant_unref (schemav);
  g_variant_unref (seqnumsv);
  g_free (schemas);

  return (GObject *) model;
}

static void
dee_serializable_model_model_iface_init (DeeModelIface *iface)
{
  iface->set_schema_full      = dee_serializable_model_set_schema_full;
  iface->get_schema           = dee_serializable_model_get_schema;
  iface->get_column_schema    = dee_serializable_model_get_column_schema;
  iface->get_n_columns        = dee_serializable_model_get_n_columns;
  iface->get_n_rows           = dee_serializable_model_get_n_rows;
  iface->append_row           = dee_serializable_model_append_row;
  iface->prepend_row          = dee_serializable_model_prepend_row;
  iface->insert_row           = dee_serializable_model_insert_row;
  iface->insert_row_before    = dee_serializable_model_insert_row_before;
  iface->remove               = dee_serializable_model_remove;
  iface->clear                = dee_serializable_model_clear;
  iface->set_value            = dee_serializable_model_set_value;
  iface->set_row              = dee_serializable_model_set_row;
  iface->get_value            = dee_serializable_model_get_value;
  iface->get_first_iter       = dee_serializable_model_get_first_iter;
  iface->get_last_iter        = dee_serializable_model_get_last_iter;
  iface->get_iter_at_row      = dee_serializable_model_get_iter_at_row;
  iface->get_bool             = dee_serializable_model_get_bool;
  iface->get_uchar            = dee_serializable_model_get_uchar;
  iface->get_int32            = dee_serializable_model_get_int32;
  iface->get_uint32           = dee_serializable_model_get_uint32;
  iface->get_int64            = dee_serializable_model_get_int64;
  iface->get_uint64           = dee_serializable_model_get_uint64;
  iface->get_double           = dee_serializable_model_get_double;
  iface->get_string           = dee_serializable_model_get_string;
  iface->next                 = dee_serializable_model_next;
  iface->prev                 = dee_serializable_model_prev;
  iface->is_first             = dee_serializable_model_is_first;
  iface->is_last              = dee_serializable_model_is_last;
  iface->get_position         = dee_serializable_model_get_position;
  iface->register_tag         = dee_serializable_model_register_tag;
  iface->get_tag              = dee_serializable_model_get_tag;
  iface->set_tag              = dee_serializable_model_set_tag;
}

static void
dee_serializable_model_serializable_iface_init (DeeSerializableIface *iface)
{
  iface->serialize      = dee_serializable_model_serialize;

  dee_serializable_register_parser (DEE_TYPE_SERIALIZABLE_MODEL,
                                    G_VARIANT_TYPE ("(asaav(tt))"),
                                    dee_serializable_model_parse_serialized);
}
