/*
 * transition_gconf.cpp
 *
 * A small executable to be run at first launch which moves
 * old settings paths to new ones so that the user's settings
 * are preserved
 *
 * Copyright (C) 2010 Canonical Ltd
 *
 * GConf Backend code taken from compizconfig-backend-gconf:
 * Copyright (c) 2007 Danny Baumann <maniac@opencompositing.org>
 *
 * Parts of this code are taken from libberylsettings
 * gconf backend, written by:
 *
 * Copyright (c) 2006 Robert Carr <racarr@opencompositing.org>
 * Copyright (c) 2007 Dennis Kasprzyk <onestone@opencompositing.org>
 *
 * Licence Terms:
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Sam Spilsbury <sam.spilsbury@canonical.com>
 */

#include <iostream>
#include <fstream>
#include <sstream>
#include <boost/algorithm/string.hpp>
#include <boost/foreach.hpp>
#include <cstring>
#include <cstdio>
#include <list>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <gconf/gconf-value.h>

#include <gio/gio.h>

using namespace std;

void move_dir_recursively (GFile *from, GFile *to)
{
    GFileEnumerator *enumerator;
    GFileInfo	    *current_info;
    GFileQueryInfoFlags q_flags;
    GFileCopyFlags	c_flags;

    memset (&q_flags, sizeof (GFileCopyFlags), 0);
    memset (&c_flags, sizeof (GFileCopyFlags), 0);

    /* First of all ensure that the dir that we want to copy to actually exists */

    g_file_make_directory_with_parents (to, NULL, NULL);

    enumerator = g_file_enumerate_children (from, "*", q_flags, NULL, NULL);
    current_info = g_file_enumerator_next_file (enumerator, NULL, NULL);

    while (current_info)
    {
	GFile *nfrom = g_file_new_for_path (string (string (g_file_get_path (from)) + "/" + g_file_info_get_edit_name (current_info)).c_str ());
	GFile *nto = g_file_new_for_path (string (string (g_file_get_path (to)) + "/" + g_file_info_get_edit_name (current_info)).c_str ());
	/* Recursively copy subdirectories */
	if (g_file_info_get_file_type (current_info) == G_FILE_TYPE_DIRECTORY)
	{
	    cout << "[LOG]: Copying subdirectory from " << string (g_file_get_path (nfrom)) << " to " << string (g_file_get_path (nto)) << endl;

	    move_dir_recursively (nfrom, nto);
	    g_file_delete (nfrom, NULL, NULL);

	    current_info = g_file_enumerator_next_file (enumerator, NULL, NULL);
	    continue;
	}

	cout << "[LOG]: Copied file " << string (g_file_get_path (nfrom)) << " to " << string (g_file_get_path (nto)) << endl;

	g_file_copy (nfrom, nto, c_flags, NULL, NULL, NULL, NULL);
	g_file_delete (nfrom, NULL, NULL);

	g_object_unref (current_info);
	g_object_unref (nfrom);
	g_object_unref (nto);

	current_info = g_file_enumerator_next_file (enumerator, NULL, NULL);
    }

    g_object_unref (enumerator);
}

bool move_internal_dirs (void)
{
    char *homedir = getenv ("HOME");
    char old_dir[512], new_dir[512];
    GFile *old_dir_p, *new_dir_p;

    snprintf (old_dir, sizeof (old_dir), "%s/.compiz", homedir);
    snprintf (new_dir, sizeof (new_dir), "%s/.compiz-1", homedir);

    old_dir_p = g_file_new_for_path (old_dir);
    new_dir_p = g_file_new_for_path (new_dir);

    move_dir_recursively (old_dir_p, new_dir_p);
    g_file_delete (old_dir_p, NULL, NULL);

    g_object_unref (old_dir_p);
    g_object_unref (new_dir_p);

    return true;
}

static void
copy_gconf_values (GConfEngine *engine,
		   const gchar *from,
		   const gchar *to,
		   bool        associate,
		   const gchar *schemaPath)
{
    GSList *values, *tmp;
    GError *err = NULL;

    values = gconf_engine_all_entries (engine, from, &err);
    tmp = values;

    while (tmp)
    {
	GConfEntry *entry = (GConfEntry *) tmp->data;
	GConfValue *value;
	const char *key = gconf_entry_get_key (entry);
	char       *name, *newKey, *newSchema = NULL;

	name = strdup (strrchr (key, '/'));
	if (!name)
	    continue;

	if (to)
	{
	    asprintf (&newKey, "%s/%s", to, name + 1);

	    if (associate && schemaPath)
		asprintf (&newSchema, "%s/%s", schemaPath, name + 1);

	    if (newKey && newSchema)
		gconf_engine_associate_schema (engine, newKey, newSchema, NULL);

	    if (newKey)
	    {
		value = gconf_engine_get (engine, key, NULL);
		if (value)
		{
		    gconf_engine_set (engine, newKey, value, NULL);
		    gconf_value_free (value);
		}
	    }

	    if (newSchema)
		free (newSchema);
	    if (newKey)
		free (newKey);
	}
	else
	{
	    if (associate)
		gconf_engine_associate_schema (engine, key, NULL, NULL);
	    gconf_engine_unset (engine, key, NULL);
	}

	gconf_entry_unref (entry);
	tmp = g_slist_next (tmp);
    }

    if (values)
	g_slist_free (values);
}

void
copy_gconf_recursively (GConfEngine *engine,
			GSList      *subdirs,
			const gchar *to,
			bool        associate,
			const gchar *schemaPath)
{
    GSList* tmp;

    tmp = subdirs;

    while (tmp)
    {
 	gchar *path = (gchar *) tmp->data;
	char  *newKey, *newSchema = NULL, *name;

	name = strrchr (path, '/');
	if (name)
	{
  	    if (to)
		asprintf (&newKey, "%s/%s", to, name + 1);
	    else
		newKey = NULL;

	    if (associate && schemaPath)
		asprintf (&newSchema, "%s/%s", schemaPath, name + 1);

	    copy_gconf_values (engine, path, newKey, associate, newSchema);
	    copy_gconf_recursively (engine, gconf_engine_all_dirs (engine, path, NULL),
				    newKey, associate, newSchema);

	    if (newSchema)
		free (newSchema);

	    if (newKey)
		free (newKey);

	    if (!to)
		gconf_engine_remove_dir (engine, path, NULL);
	}

	g_free (path);
	tmp = g_slist_next (tmp);
    }

    if (subdirs)
	g_slist_free (subdirs);
}

void
copy_gconf_tree (GConfClient **client,
		 GConfEngine *engine,
		 const gchar *from,
		 const gchar *to,
		 bool        associate,
		 const gchar *schemaPath)
{
    GSList* subdirs;

    /* We can't have the GConfClient object open at the same time as the engine
     * so close the client for now */

    gconf_client_suggest_sync (*client, NULL);

    g_object_unref (*client);
    *client = NULL;

    subdirs = gconf_engine_all_dirs (engine, from, NULL);
    gconf_engine_suggest_sync (engine, NULL);

    copy_gconf_recursively (engine, subdirs, to, associate, schemaPath);

    gconf_engine_suggest_sync (engine, NULL);

    *client = gconf_client_get_for_engine (engine);
}

bool rename_gconf_path (GConfClient **client,
			GConfEngine *engine,
			const gchar *old_path,
			const gchar *new_path,
			const gchar *schema_path)
{
    GConfUnsetFlags flags;
    gboolean should_associate = schema_path ? TRUE : FALSE;
    memset (&flags, 0, sizeof (GConfUnsetFlags));
    cout << "[LOG]: Moving " << string (old_path) << " to " << string (new_path) << endl;

    copy_gconf_tree (client, engine, old_path, new_path, should_associate, schema_path);

    /* reset tree */
    gconf_client_recursive_unset (*client, old_path, flags, NULL);

    return true;
}


GSList * rename_gconf_plugin (GSList *value_list,
			      const gchar *old_plugin_name,
			      const gchar *new_plugin_name)
{
    GSList *tmp = value_list, *insert_pos = NULL;

    while (tmp)
    {
	if (string ((gchar *) tmp->data).find (old_plugin_name) != string::npos)
	{
	    insert_pos = tmp;
	    break;
	}

	tmp = g_slist_next (tmp);
    }

    if (insert_pos)
	insert_pos = g_slist_next (insert_pos);

    if (insert_pos)
    {
	gchar *data;

	data = strdup (new_plugin_name);
	value_list = g_slist_insert_before (value_list, insert_pos, (gpointer) data);
    }
    else
    {
	cout << "[LOG]: Could not find plugin "<<  string (old_plugin_name) << " in loaded plugin data" << endl;
    }

    tmp = value_list;

    while (tmp)
    {
	if (string ((gchar *) tmp->data).find (old_plugin_name) != string::npos)
	{
	    insert_pos = tmp;
	    break;
	}


	tmp = g_slist_next (tmp);
    }

    if (insert_pos)
    {
	value_list = g_slist_delete_link (value_list, insert_pos);
    }

    return value_list;
}

void move_gconf_setting (GConfClient **client,
			 GConfEngine *engine,
			 const gchar *setting,
			 const gchar *prefix,
			 const gchar *old_plugin,
			 const gchar *new_plugin)
{
    GConfValue *value;
    string     new_key_name, old_key_name;

    old_key_name = string (string (prefix) + string (old_plugin) + "/screen0/options/" + string (setting)).c_str ();

    value = gconf_client_get (*client, old_key_name.c_str (), NULL);

    cout << "[LOG]: Transitioning setting " << string (setting) << " from " << string (old_plugin) << " to " << string (new_plugin) << endl;

    if (value)
    {
	new_key_name = string (string (prefix) + string (new_plugin) + "/screen0/options/" + string (setting));

	switch (value->type)
	{
	    case GCONF_VALUE_STRING:
		gconf_client_set_string (*client, new_key_name.c_str (),
					 gconf_value_get_string (value), NULL);
		break;

	    case GCONF_VALUE_INT:
		gconf_client_set_int (*client, new_key_name.c_str (),
				      gconf_value_get_int (value), NULL);
		break;
	    case GCONF_VALUE_FLOAT:
		gconf_client_set_float (*client, new_key_name.c_str (),
					gconf_value_get_float (value), NULL);
		break;
	    case GCONF_VALUE_BOOL:
		gconf_client_set_bool (*client, new_key_name.c_str (),
				       gconf_value_get_bool (value), NULL);
		break;
	    case GCONF_VALUE_SCHEMA:
		gconf_client_set_schema (*client, new_key_name.c_str (),
					 gconf_value_get_schema (value), NULL);
		break;
	    case GCONF_VALUE_LIST:
		gconf_client_set_list (*client, new_key_name.c_str (), gconf_value_get_list_type (value),
				       gconf_value_get_list (value), NULL);
		break;
	    case GCONF_VALUE_PAIR:
	    case GCONF_VALUE_INVALID:
	    default:
		cout << "[WARNING]: Invalid value given" << endl;
	}
    }
}

bool convert_gconf_tree (GConfClient **client,
			 GConfEngine *engine,
			 const gchar *path,
			 const gchar *schema_path)
{
    string cold_path = string (path);
    string cnew_path = string (path);
    GSList *subdirs, *tmp, *plugins, *insert_pos = NULL;
    gboolean   should_associate = schema_path ? TRUE : FALSE;

    GConfUnsetFlags flags;
    memset (&flags, 0, sizeof (GConfUnsetFlags));

    cout << "[LOG]: Converting tree " << string (path) << endl;

    /* First, rename the plugins as needed */
    cold_path = string (path) + "/plugins/decoration";
    cnew_path = string (path) + "/plugins/decor";

    cout << "[LOG]: Renaming plugin with path " << cold_path << " to " << cnew_path << endl;

    /* decoration -> decor */
    rename_gconf_path (client, engine, cold_path.c_str (), cnew_path.c_str (), schema_path);

    cold_path = string (path) + "/plugins/png";
    cnew_path = string (path) + "/plugins/imgpng";

    cout << "[LOG]: Renaming plugin with path " << cold_path << " to " << cnew_path << endl;

    /* png -> imgpng */
    rename_gconf_path (client, engine, cold_path.c_str (), cnew_path.c_str (), schema_path);

    cold_path = string (path) + "/plugins/svg";
    cnew_path = string (path) + "/plugins/imgsvg";

    cout << "[LOG]: Renaming plugin with path " << cold_path << " to " << cnew_path << endl;

    /* svg -> imgsvg */
    rename_gconf_path (client, engine, cold_path.c_str (), cnew_path.c_str (), schema_path);

    cold_path = string (path) + "/plugins/jpeg";
    cnew_path = string (path) + "/plugins/imgjpeg";

    cout << "[LOG]: Renaming plugin with path " << cold_path << " to " << cnew_path << endl;

    /* jpeg -> imgjpeg */
    rename_gconf_path (client, engine, cold_path.c_str (), cnew_path.c_str (), schema_path);

    /* Now convert allscreens -> screen0 */

    cold_path = string (path) + "/general/allscreens";
    cnew_path = string (path) + "/general/screen0";

    /* core */

    cout << "[LOG]: Moving display/screen settings from " << cold_path.c_str () << " to " << cnew_path.c_str () << endl;

    copy_gconf_tree (client, engine, cold_path.c_str (), cnew_path.c_str (), should_associate, schema_path);
    gconf_client_recursive_unset (*client, cold_path.c_str (), flags, NULL);

    /* plugins:
     * We can't have the GConfClient object open at the same time as the engine
     * so close the client for now */

    gconf_client_suggest_sync (*client, NULL);

    g_object_unref (*client);
    *client = NULL;

    cold_path = string (path) + "/plugins";

    cout << "[LOG]: Traversing directory " << cold_path << endl;

    tmp = subdirs = gconf_engine_all_dirs (engine, cold_path.c_str (), NULL);

    *client = gconf_client_get_for_engine (engine);

    while (tmp)
    {
	gchar *path = (gchar *) tmp->data;

	cnew_path = string (path) + "/screen0";
	cold_path = string (path) + "/allscreens";

	cout << "[LOG]: Moving display/screen settings from " << cold_path.c_str () << " to " << cnew_path.c_str () << endl;

	copy_gconf_tree (client, engine, cold_path.c_str (), cnew_path.c_str (), should_associate, schema_path);
	gconf_client_recursive_unset (*client, cold_path.c_str (), flags, NULL);

	g_free (path);
	tmp = g_slist_next (tmp);
    }

    if (subdirs)
	g_slist_free (subdirs);

    /* Now append the plugins we need into core */

    cout << "[LOG]: Appending new plugins to " << string (string (path) + "/general/screen0/options/active_plugins") << endl;

    tmp = plugins = gconf_client_get_list (*client, string (string (path) + "/general/screen0/options/active_plugins").c_str (), GCONF_VALUE_STRING, NULL);

    while (tmp)
    {
	if (string ((gchar *) tmp->data).find ("core") != string::npos)
	{
	    insert_pos = tmp;
	    break;
	}

	tmp = g_slist_next (tmp);
    }

    if (insert_pos)
	insert_pos = g_slist_next (insert_pos);

    if (!insert_pos)
	insert_pos = g_slist_last (plugins);

    if (insert_pos)
    {
	char *data;

	data = strdup ("bailer");
	plugins = g_slist_insert_before (plugins, insert_pos, (gpointer) data);

	data = strdup ("detection");
	plugins = g_slist_insert_before (plugins, insert_pos, (gpointer) data);

	data = strdup ("composite");
	plugins = g_slist_insert_before (plugins, insert_pos, (gpointer) data);

	data = strdup ("opengl");
	plugins = g_slist_insert_before (plugins, insert_pos, (gpointer) data);

	data = strdup ("compiztoolbox");
	plugins = g_slist_insert_before (plugins, insert_pos, (gpointer) data);
    }
    else
    {
	cout << "[WARNING]: Could not find ccp plugin in loaded plugin data" << endl;
    }

    plugins = rename_gconf_plugin (plugins, "decoration", "decor");
    plugins = rename_gconf_plugin (plugins, "svg", "imgsvg");
    plugins = rename_gconf_plugin (plugins, "png", "imgpng");
    plugins = rename_gconf_plugin (plugins, "jpeg", "imgjpeg");

    /* Remove the glib plugin */

    tmp = plugins;

    while (tmp)
    {
	if (string ((gchar *) tmp->data).find ("glib") != string::npos)
	{
	    insert_pos = tmp;
	    break;
	}

	tmp = g_slist_next (tmp);
    }

    if (insert_pos)
    {
	plugins = g_slist_delete_link (plugins, insert_pos);
    }

    tmp = plugins;

    cout << "[LOG]: New plugins list is [";

    tmp = plugins;

    gconf_client_set_list (*client, string (string (path) + "/general/screen0/options/active_plugins").c_str (), GCONF_VALUE_STRING, plugins, NULL);

    while (tmp)
    {
	gchar *plugin = (gchar *) tmp->data;

	cout << string (plugin) << ", ";
	if (tmp->data)
	    g_free (tmp->data);

	tmp = g_slist_next (tmp);
    }

    cout << "]" << endl;

    if (plugins)
	g_slist_free (plugins);

    /* Now move settings around */

    move_gconf_setting (client, engine, "slow_animations_key", path, "/general", "/plugins/composite");
    move_gconf_setting (client, engine, "detect_refresh_rate", path, "/general", "/plugins/composite");
    move_gconf_setting (client, engine, "refresh_rate", path, "/general", "/plugins/composite");
    move_gconf_setting (client, engine, "unredirect_fullscreen_windows", path, "/general", "/plugins/composite");
    move_gconf_setting (client, engine, "force_independent_output_painting", path, "/general", "/plugins/composite");
    move_gconf_setting (client, engine, "texture_compression", path, "/general", "/plugins/opengl");
    move_gconf_setting (client, engine, "texture_filter", path, "/general", "/plugins/opengl");
    move_gconf_setting (client, engine, "lighting", path, "/general", "/plugins/opengl");
    move_gconf_setting (client, engine, "sync_to_vblank", path, "/general", "/plugins/opengl");

    return true;
}


bool convert_gconf_settings ()
{
    GConfEngine *engine = gconf_engine_get_default ();
    GConfClient *client = gconf_client_get_for_engine (engine);
    GSList *dir_list;
    GSList *tmp;

    cout << "[LOG]: Copying GConf settings from /apps/compiz to /apps/compiz-1" << endl;

    copy_gconf_tree (&client, engine, "/apps/compiz", "/apps/compiz-1", true, "/schemas/apps/compiz-1");

    cout << "[LOG]: Copying GConf settings from /apps/compizconfig to /apps/compizconfig-1" << endl;

    copy_gconf_tree (&client, engine, "/apps/compizconfig", "/apps/compizconfig-1", false, NULL);

    cout << "[LOG]: Converting settings in /apps/compiz-1" << endl;

    if (convert_gconf_tree (&client, engine, "/apps/compiz-1", "/schemas/apps/compiz-1"))
	cout << "[LOG]: Successfully converted settings in /apps/compiz-1" << endl;
    else
	cout << "[WARNING]: Unable to convert setttings in /apps/compiz-1, some settings may be lost!" << endl;

    /* We can't have the GConfClient object open at the same time as the engine
     * so close the client for now */

    gconf_client_suggest_sync (client, NULL);

    g_object_unref (client);
    client = NULL;

    tmp = dir_list = gconf_engine_all_dirs (engine, "/apps/compizconfig-1/profiles", NULL);

    client = gconf_client_get_for_engine (engine);

    while (tmp)
    {
	gchar *path = (gchar *) tmp->data;

	cout << "[LOG]: Converting profile stored in " << string (path) << endl;

	if (convert_gconf_tree (&client, engine, path, (string ("/schemas") + string (path)).c_str ()))
	    cout << "[LOG]: Successfully converted settings in " << string (path) << endl;
	else
	    cout << "[WARNING]: Unable to convert setttings in " << string (path) << ", some settings may be lost!" << endl;

	if (path)
	    g_free (path);

	tmp = g_slist_next (tmp);
    }

    if (dir_list)
	g_slist_free (dir_list);

    gconf_client_suggest_sync (client, NULL);

    g_object_unref (client);
    client = NULL;

    return true;
}

void transition_settings ()
{
    char     *homedir = getenv ("HOME");
    GFile    *old_home_internal_p  = g_file_new_for_path (string (string (homedir) + string ("/.compiz")).c_str ());
    GConfEngine *engine = gconf_engine_get_default ();

    cout << "Checking if settings need to be migrated ...";

    if (!gconf_engine_dir_exists (engine, "/apps/compiz-1", NULL) &&
	!gconf_engine_dir_exists (engine, "/apps/compizconfig-1", NULL))
    {
	cout << "yes" << endl;
	cout << "Compiz Migration Script for Ubuntu 11.04" << endl;
	cout << "Moving settings from Compiz 0.8.6 to Compiz 0.9.4" << endl;

	cout << "[LOG]: Moving and rewriting GConf Settings" << endl;

	gconf_engine_suggest_sync (engine, NULL);

	if (convert_gconf_settings ())
	    cout << "[LOG]: Successfully converted gconf settings" << endl;
	else
	    cout << "[WARNING]: Could not convert all gconf settings, your settings may be reset" << endl;
    }
    else
    {
	gconf_engine_suggest_sync (engine, NULL);

	cout << "no" << endl;
    }

    cout << "Checking if internal files need to be migrated ...";

    if (g_file_query_exists (old_home_internal_p, NULL))
    {
	cout << "yes" << endl;

	cout << "[LOG]: Moving Internal Files" << endl;

	if (move_internal_dirs ())
	    cout << "[LOG]: Successfully moved internal files" << endl;
	else
	    cout << "[WARNING]: Could not move internal files" << endl;
    }
    else
	cout << "no" << endl;
}

extern "C"
{
    void do_transition ()
    {
	transition_settings ();
    }
}
