# Copyright (C) 2008 Aaron Bentley
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os

from bzrlib.transform import TreeTransform
from bzrlib.tests import TestCaseWithTransport

from bzrlib.plugins.bzrtools.link_tree import link_tree


class TestLinkTreeBase(object):

    def setUp(self):
        TestCaseWithTransport.setUp(self)
        self.parent_tree = self.make_branch_and_tree('parent')
        self.parent_tree.lock_write()
        self.addCleanup(self.parent_tree.unlock)
        self.build_tree_contents([('parent/foo', 'bar')])
        self.parent_tree.add('foo', 'foo-id')
        self.parent_tree.commit('added foo')
        child_bzrdir = self.parent_tree.bzrdir.sprout('child')
        self.child_tree = child_bzrdir.open_workingtree()

    def hardlinked(self):
        parent_stat = os.lstat(self.parent_tree.abspath('foo'))
        child_stat = os.lstat(self.child_tree.abspath('foo'))
        return parent_stat.st_ino == child_stat.st_ino


class TestLinkTree(TestLinkTreeBase, TestCaseWithTransport):

    def test_link_fails_if_modified(self):
        """If the file to be linked has modified text, don't link."""
        self.build_tree_contents([('child/foo', 'baz')])
        link_tree(self.child_tree, self.parent_tree)
        self.assertFalse(self.hardlinked())

    def test_link_fails_if_execute_bit_changed(self):
        """If the file to be linked has modified execute bit, don't link."""
        tt = TreeTransform(self.child_tree)
        try:
            trans_id = tt.trans_id_tree_file_id('foo-id')
            tt.set_executability(True, trans_id)
            tt.apply()
        finally:
            tt.finalize()
        link_tree(self.child_tree, self.parent_tree)
        self.assertFalse(self.hardlinked())

    def test_link_succeeds_if_unmodified(self):
        """If the file to be linked is unmodified, link"""
        link_tree(self.child_tree, self.parent_tree)
        self.assertTrue(self.hardlinked())


class TestLinkTreeCommand(TestLinkTreeBase, TestCaseWithTransport):

    def test_link_tree(self):
        """Ensure the command works as intended"""
        os.chdir('child')
        self.parent_tree.unlock()
        self.run_bzr('link-tree ../parent')
        self.assertTrue(self.hardlinked())
        # want teh addCleanup to work properly
        self.parent_tree.lock_write()
