/*
 *   Copyright (c) 2010 - 2012
 *   Canonical Ltd. (All rights reserved)
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of version 2 of the GNU General Public
 *   License published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, contact Novell, Inc. or Canonical,
 *   Ltd.
 */
#include <stdlib.h>
#include <stdarg.h>
#include <libintl.h>
#include <locale.h>
#define _(s) gettext(s)
#include "parser.h"

/* Policy versioning is determined by a combination of 3 values:
 * policy_version:     version of txt policy
 * parser_abi_version: version of abi revision of policy generated by parser
 * kernel_abi_version: version of abi revision for the kernel
 *
 * The version info is stored in a single 32 bit version field in the
 * header portion of each binary policy file.
 *
 * policy_version:
 *   a gross revision number indicating what features and semantics are
 *   expected by the text policy. This does not necessarily map directly
 *   to a feature set as a kernel may not have all the supported features
 *   patched/builtin.
 *
 *   policy_version is not supported by kernels that only support v5
 *   kernel abi, so it will not be written when creating policy for
 *   those kernels.
 *
 * kernel_abi_version:
 *   should be set to the highest version supported by both the parser and
 *   the kernel.
 *   This allows new kernels to detect old userspaces, and new parsers
 *   to support old kernels and policies semantics.
 *
 * parser_abi_version:
 *   should be bumped when a compiler error or some other event happens
 *   and policy cache needs to be forced to be recomputed, when the
 *   policy_version or kernel version has not changed.
 *
 *   parser_abi_version is not supported by kernels that only support
 *   v5 kernel abi so it will not be written when creating policy for those
 *   kernels.
 *
 * Default values set to v5 kernel abi before the different versioning
 * numbers where supported.
 */
uint32_t policy_version = 2;
uint32_t parser_abi_version = 1;
uint32_t kernel_abi_version = 5;

int force_complain = 0;
int perms_create = 0;                   /* perms contain create flag */
int net_af_max_override = -1;           /* use kernel to determine af_max */
int kernel_load = 1;
int kernel_supports_setload = 0;	/* kernel supports atomic set loads */
int kernel_supports_network = 0;        /* kernel supports network rules */
int kernel_supports_policydb = 0;	/* kernel supports new policydb */
int kernel_supports_mount = 0;	        /* kernel supports mount rules */
int kernel_supports_dbus = 0;		/* kernel supports dbus rules */
int kernel_supports_diff_encode = 0;	/* kernel supports diff_encode */
int kernel_supports_signal = 0;		/* kernel supports signal rules */
int kernel_supports_ptrace = 0;		/* kernel supports ptrace rules */
int conf_verbose = 0;
int conf_quiet = 0;
int names_only = 0;
int current_lineno = 1;
int option = OPTION_ADD;

dfaflags_t dfaflags = (dfaflags_t)(DFA_CONTROL_TREE_NORMAL | DFA_CONTROL_TREE_SIMPLE | DFA_CONTROL_MINIMIZE | DFA_CONTROL_DIFF_ENCODE);

char *subdomainbase = NULL;
const char *progname = __FILE__;
char *profile_ns = NULL;
char *profilename = NULL;
char *current_filename = NULL;

FILE *ofile = NULL;

#ifdef FORCE_READ_IMPLIES_EXEC
int read_implies_exec = 1;
#else
int read_implies_exec = 0;
#endif

void pwarn(const char *fmt, ...)
{
        va_list arg;
        char *newfmt;

        if (conf_quiet || names_only || option == OPTION_REMOVE)
                return;

        if (asprintf(&newfmt, _("Warning from %s (%s%sline %d): %s"),
		     profilename ? profilename : "stdin",
		     current_filename ? current_filename : "",
		     current_filename ? " " : "",
		     current_lineno,
		     fmt) == -1)
                return;

        va_start(arg, fmt);
        vfprintf(stderr, newfmt, arg);
        va_end(arg);

        free(newfmt);
}
