/* cmd-replay.c:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */




#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/fs/cwd.h"
#include "hackerlab/vu/vu-dash.h"
#include "tla/libfsutils/tmp-files.h"
#include "tla/libfsutils/rmrf.h"
#include "tla/libfsutils/copy-file.h"
#include "tla/libfsutils/dir-as-cwd.h"
#include "tla/libarch/namespace.h"
#include "tla/libarch/project-tree.h"
#include "tla/libarch/my.h"
#include "tla/libarch/whats-missing.h"
#include "tla/libarch/whats-new.h"
#include "tla/libarch/invent.h"
#include "tla/libarch/archive.h"
#include "tla/libarch/pristines.h"
#include "tla/libarch/build-revision.h"
#include "tla/libarch/replay.h"
#include "tla/libarch/cmd-replay.h"



static t_uchar * usage = "[options] dir [newdir] [version/revision]";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
                                   "\n"
                                   "Copyright 2003 Tom Lord\n"
                                   "\n"
                                   "This is free software; see the source for copying conditions.\n"
                                   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
                                   "PARTICULAR PURPOSE.\n"
                                   "\n"
                                   "Report bugs to " cfg__tla_bug_mail ".\n"
                                   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.") \
  OP (opt_archive, "A", "archive", 1, \
      "Override `my-default-archive'") \
  OP (opt_exact, 0, "exact", 0, \
      "apply a specific patch") \
  OP (opt_in_place, 0, "in-place", 0, \
      "modify DIR directly") \
  OP (opt_list, 0, "list FILE", 1, \
      "read a list of patches to apply") \
  OP (opt_new, 0, "new", 0, \
      "replay only new patches") \
  OP (opt_reverse, 0, "reverse", 0, \
      "reverse the named patch")


t_uchar arch_cmd_replay_help[] = ("apply revision changesets to a project tree\n"
                                  "Construct a new project tree NEW-DIR for the working directory\n"
                                  "containing OLD-DIR.\n"
                                  "\n"
                                  "The new project tree is formed by applying patches in the latest\n"
                                  "revision of VERSION (or the default version of OLD-DIR) to the old working\n"
                                  "directory, stopping after the first patch that causes conflicts.\n"
                                  "\n"
                                  "With the --exact option, the VERSION argument is mandatory, and must\n"
                                  "explicitly specify a patch level.  That patch set, and no other will\n"
                                  "be applied.\n"
                                  "\n"
                                  "With the --in-place option, only one directory is needed -- that directory\n"
                                  "is modified rather than making a copy.\n"
                                  "\n"
                                  "With the --list option, read a list of patches to apply from FILE (- for\n"
                                  "standard input).  Complete revision names should be listed, one per line.\n"
                                  "replay will stop at the first patch in the list that causes a merge conflict,\n"
                                  "leaving behind files with names of the form:\n"
                                  "\n"
                                  "    ,,replay.conflicts-in --    the name of the patch that\n"
                                  "                                  caused conflicts\n"
                                  "\n"
                                  "    ,,replay.remaining    --    the list of patches not yet\n"
                                  "                                  applied\n"
                                  "\n"
                                  "Options --list and --exact cancel each other.\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



enum op
{
  arch_replay_op_missing,
  arch_replay_op_exact,
  arch_replay_op_list,
  arch_replay_op_new,
};

int
arch_cmd_replay (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  t_uchar * list_file = 0;
  int in_place = 0;
  int reverse = 0;
  enum op action = arch_replay_op_missing;
  int exit_status = 0;

  vu_push_dash_handler (0);
  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, arch_cmd_replay_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            lim_free (0, default_archive);
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_exact:
          {
            action = arch_replay_op_exact;
            break;
          }

        case opt_list:
          {
            action = arch_replay_op_list;
            lim_free (0, list_file);
            list_file = str_save (0, option->arg_string);
            break;
          }

        case opt_new:
          {
            action = arch_replay_op_new;
            break;
          }

        case opt_in_place:
          {
            in_place = 1;
            break;
          }

        case opt_reverse:
          {
            reverse = 1;
            break;
          }
        }
    }

  if (in_place)
    {
      if ((argc < 2) || (argc > 3))
        goto usage_error;
    }
  else
    {
      if ((argc < 3) || (argc > 4))
        goto usage_error;
    }

  if (default_archive && !arch_valid_archive_name (default_archive))
    {
      safe_printfmt (2, "replay: invalid archive name (%s)\n", default_archive);
      exit (1);
    }

  {
    t_uchar * tree_root = 0;
    t_uchar * dest = 0;
    t_uchar * src_spec = 0;
    int a;

    tree_root = arch_tree_root (0, argv[1], 0);

    if (in_place)
      {
        a = 2;
      }
    else
      {
        t_uchar * dest_dir_spec = 0;
        t_uchar * dest_dir = 0;
        t_uchar * dest_tail = 0;

        a = 3;

        dest_dir_spec = file_name_directory_file (0, argv[2]);
        if (!dest_dir_spec)
          dest_dir_spec = str_save (-0, ".");
        dest_dir = directory_as_cwd (dest_dir_spec);
        dest_tail = file_name_tail (0, argv[2]);

        dest = file_name_in_vicinity (0, dest_dir, dest_tail);

        lim_free (0, dest_dir_spec);
        lim_free (0, dest_dir);
        lim_free (0, dest_tail);
      }

    if (action != arch_replay_op_list)
      {
        if (argc > a)
          src_spec = str_save (0, argv[a]);
        else
          {
            src_spec = arch_tree_version (tree_root);
          }
      }


    if (!in_place)
      {
        rel_table to_copy = 0;

        safe_printfmt (1, "* copying target tree\n");
        safe_flush (1);

        to_copy = arch_source_inventory (tree_root, 1, 1, 0);
        safe_mkdir (dest, 0777);
        copy_file_list (dest, tree_root, to_copy);

        rel_free_table (to_copy);
      }
    else
      {
        dest = str_save (0, tree_root);
      }

    switch (action)
      {
      default:
        {
          panic ("internal error -- unrecognized `op'");
          break;                /* notreached */
        }

      case arch_replay_op_exact:
        {
          t_uchar * archive = 0;
          struct arch_archive * arch = 0;
          t_uchar * revspec = 0;
          t_uchar * revision = 0;

          if (!arch_valid_package_name (src_spec, arch_maybe_archive, arch_req_version, 1))
            {
              safe_printfmt (2, "replay: invalid revision spec (%s)\n", src_spec);
              exit (1);
            }

          archive = arch_parse_package_name (arch_ret_archive, default_archive, src_spec);
          arch = arch_archive_connect (archive, 0);

          revspec = arch_parse_package_name (arch_ret_non_archive, 0, src_spec);

          if (arch_valid_package_name (revspec, arch_no_archive, arch_req_patch_level, 0))
            {
              revision = str_save (0, revspec);
            }
          else
            {
              rel_table levels = 0;

              levels = arch_archive_revisions (arch, revspec, 0);
              if (!levels)
                {
                  safe_printfmt (2, "replay: version has no revisions (%s/%s)\n", archive, revspec);
                  exit (1);
                }

              revision = str_alloc_cat_many (0, revspec, "--", levels[rel_n_records(levels) - 1], str_end);
              rel_free_table (levels);
            }

          exit_status = arch_replay_exact (1, dest, arch, revision, reverse);

          lim_free (0, archive);
          arch_archive_close (arch);
          lim_free (0, revspec);
          lim_free (0, revision);
          break;
        }

      case arch_replay_op_missing:
        {
          t_uchar * archive = 0;
          struct arch_archive * arch = 0;
          t_uchar * version = 0;
          rel_table revisions = 0;
          int x;

          if (reverse)
            {
              safe_printfmt (2, "replay: --reverse requires --exact or --list\n");
              exit (1);
            }

          if (!arch_valid_package_name (src_spec, arch_maybe_archive, arch_req_version, 0))
            {
              safe_printfmt (2, "replay: invalid version name (%s)\n", src_spec);
              exit (1);
            }

          archive = arch_parse_package_name (arch_ret_archive, default_archive, src_spec);
          arch = arch_archive_connect (archive, 0);

          version = arch_parse_package_name (arch_ret_non_archive, 0, src_spec);

          revisions = arch_whats_missing (tree_root, arch, version);

          for (x = 0; x < rel_n_records (revisions); ++x)
            {
              t_uchar * t = 0;

              t = revisions[x][0];
              revisions[x][0] = str_alloc_cat_many (0, version, "--", t, str_end);

              lim_free (0, t);
            }

          exit_status = arch_replay_list (1, dest, arch, revisions, 0);

          lim_free (0, archive);
          arch_archive_close (arch);
          lim_free (0, version);
          rel_free_table (revisions);
          break;
        }


      case arch_replay_op_new:
        {
          t_uchar * archive = 0;
          struct arch_archive * arch = 0;
          t_uchar * version = 0;
          rel_table revisions = 0;
          int x;

          if (reverse)
            {
              safe_printfmt (2, "replay: --reverse requires --exact or --list\n");
              exit (1);
            }

          if (!arch_valid_package_name (src_spec, arch_maybe_archive, arch_req_version, 0))
            {
              safe_printfmt (2, "replay: invalid version name (%s)\n", src_spec);
              exit (1);
            }

          archive = arch_parse_package_name (arch_ret_archive, default_archive, src_spec);
          arch = arch_archive_connect (archive, 0);

          version = arch_parse_package_name (arch_ret_non_archive, 0, src_spec);

          revisions = arch_whats_new (tree_root, arch, version);

          for (x = 0; x < rel_n_records (revisions); ++x)
            {
              t_uchar * t = 0;

              t = revisions[x][0];
              revisions[x][0] = str_alloc_cat_many (0, version, "--", t, str_end);

              lim_free (0, t);
            }

          exit_status = arch_replay_list (1, dest, arch, revisions, 0);

          lim_free (0, archive);
          arch_archive_close (arch);
          lim_free (0, version);
          rel_free_table (revisions);
          break;
        }

      case arch_replay_op_list:
        {
          int in_fd = -1;
          rel_table revisions = 0;
          int x;

          in_fd = safe_open (list_file, O_RDONLY, 0);
          revisions = rel_read_table (in_fd, 1, "replay", list_file);

          for (x = 0; x < rel_n_records (revisions); ++x)
            {
              if (!arch_valid_package_name (revisions[x][0], arch_maybe_archive, arch_req_patch_level, 0))
                {
                  safe_printfmt (2, "replay: invalid revision name in input list (%s)\n", revisions[x][0]);
                  exit (1);
                }
            }

          exit_status = arch_replay_fqlist (1, dest, default_archive, revisions, reverse);

          safe_close (in_fd);
          rel_free_table (revisions);
          break;
        }
      }


    lim_free (0, tree_root);
    lim_free (0, dest);
    lim_free (0, src_spec);
  }



  lim_free (0, default_archive);
  lim_free (0, list_file);

  if (exit_status)
    {
      safe_printfmt (2, "\nreplay: conflicts occured during replay\n");
    }

  return exit_status;
}




/* tag: Tom Lord Mon Jun  2 16:50:36 2003 (cmd-replay.c)
 */
