/*
 * MainBoard.cs - Handle the main game board layout.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Frogger
{

using System;
using Curses;

public class MainBoard
{
	// Internal state.
	private Window window;
	private DrawingContext dc;
	private int topBank;
	private int bottomBank;
	private Frog frog;
	private LogLine[] logLines;
	private int numLogLines;

	// Constructor.
	public MainBoard(Window window)
			{
				this.window = window;
				dc = window.DrawingContext;

				// Find the top and bottom banks.
				topBank = Frog.FrogHeight;
				bottomBank = window.Height - Frog.FrogHeight - 1;

				// Create the board layout.
				CreateBoard();
			}

	// Create the board layout.
	private void CreateBoard()
			{
				// Create the frog at its initial position.
				frog = new Frog(this, dc, (window.Width - Frog.FrogWidth) / 2,
								bottomBank + 1);

				// Determine how many lines of logs can fit on the screen.
				numLogLines = (bottomBank - topBank) / (LogLine.LogHeight + 1);

				// Create the log lines.
				int line, y;
				y = topBank + 1;
				double sign = 1.0;
				logLines = new LogLine [numLogLines];
				for(line = 0; line < numLogLines; ++line)
				{
					logLines[line] = new LogLine
						(this, dc, y, -3 + 8 * line, 12, 10, 0.2 * sign);
					y += LogLine.LogHeight + 1;
					sign = -sign;
				}
			}

	// Clear the screen and draw the entire board.
	public void Draw()
			{
				dc.Clear();
				dc.Foreground = Color.Cyan;
				dc.Background = Color.Black;
				dc.HLine(0, window.Width - 1, topBank);
				dc.HLine(0, window.Width - 1, bottomBank);
				frog.Draw();
				frog.UpdateTime();
				int line;
				for(line = 0; line < numLogLines; ++line)
				{
					logLines[line].Draw();
				}
			}

	// Stop everything because the frog has died or we
	// will be starting a new level.
	public void StopEvents()
			{
				int line;
				for(line = 0; line < numLogLines; ++line)
				{
					logLines[line].StopEvents();
				}
				frog.StopEvents();
			}

	// Start a new level.
	public void NewLevel()
			{
				CreateBoard();
				Draw();
			}

	// Get the frog object.
	public Frog Frog
			{
				get
				{
					return frog;
				}
			}

	// Get the number of log lines on the board.
	public int NumLogLines
			{
				get
				{
					return numLogLines;
				}
			}

	// Get a specific log line from the board.
	public LogLine GetLogLine(int num)
			{
				return logLines[num];
			}

} // class MainBoard

} // namespace Frogger
