/*
* (C) 2000 David O'Toole $Date: 2001/03/24 22:30:45 $
* 
* ENGINE.C
* Contains implementations for intrinsic engine functions. There are additional
* functions in mixer.c. 
*
* $Revision: 1.2 $
*
* This software is distributed under the terms of the
* GNU General Public License (GPL). Read the included file
* COPYING for more information. 
*
*/
static const char rcsid[]="$Id: engine.c,v 1.2 2001/03/24 22:30:45 dto Exp $";

#include "engine.h"
#include <assert.h>
#include <string.h>
#include <stdio.h>

#define DEFAULT_BLOCK_SIZE (512)
#define DEFAULT_TPB (4)
#define DEFAULT_BPM (120)


// 
// PUBLIC FUNCTIONS
// Synchronization is the responsibility of the caller
//

engine* en_create(void) 
{
	int i, j;
	engine* e;
 
	e = ox_alloc(sizeof(engine));
	if (!e) ox_die("en_create: couldn't allocate memory");

	// init machine table

	for (i=0; i<max_machines; i++) {   
		e->machines[i] = 0;
		for(j=0; j<max_machines; j++)
			e->signals[i][j] = en_nosignal; 
	}  

	e->last_machine = 1; 	// we always have master in [0], and this field is the index of AFTER   
	e->frame = 0; 
	e->tick = 0;
	e->tickpos = 0;
	en_set_timings(e, DEFAULT_BLOCK_SIZE, DEFAULT_TPB, DEFAULT_BPM); 
	e->program = 0;

	// now create a dummy master machine  
  
	e->machines[master] = (machine*) malloc(sizeof(machine)); 
	e->machines[master]->x = 100; 
	e->machines[master]->y = 100; 
	strcpy(e->machines[master]->instance_name, "master");

	// create mutex

	pthread_mutex_init(&e->mutex, NULL);

	return e; 
}


machine* en_get_machine(engine* e, int m) 
{
	return(e->machines[m]);
}


int en_is_present(engine* e, int which)
{
	return (e->machines[which] != 0);
}


int en_is_connected(engine* e, int source, int sink)
{
	return (e->signals[source][sink]); 
}


// deep copy
void en_rep(engine* dest, engine *src)
{
	assert(0);
//	int i; 
//	engine *ne = malloc(sizeof(engine));
//  
//	memcpy(ne, e, sizeof(engine));
//	for (i=0; i<max_machines; i++) 
//		if (e->machines[i]) {
//			ne->machines[i] = malloc(sizeof(machine));
//			memcpy(ne->machines[i], e->machines[i], sizeof(machine));
//   	}
//
//	return ne; 
}


void en_reset_program(engine *en) 
{
	en->program = mx_destroy(en->program);
	en->program = mx_compile_digraph(en, 0, en->program);
	mx_print_program(en->program);
}


machine_type *en_get_type_from_index(engine *e, int ix)
{
	if (e->machines[ix]) return e->machines[ix]->type;
	else return 0;
}




//
// INTERFACE STUFF THAT THE VIEW CLIENTS WILL WANT TO USE
//


void en_position (engine *self, int mid, int x, int y)
{
	machine *m = en_get_machine(self, mid);

	if (!m) ox_die("en_position: bad machine number!");

	m->x = x; 
	m->y = y; 
}


int en_connect(engine *self, int signaltype, int source, int sink) 
{
	assert(source < self->last_machine);
	assert(sink < self->last_machine);
	assert(self->machines[source] && self->machines[sink]);

	// machines exist. now validate connection

	// FIXME: VALIDATE CONNECTION

	// let's connect them and reanalyze flow network
	
	self->signals[source][sink] = signaltype;
	en_reset_program(self);

	return 1;
}


void en_name(engine *self, int mid, const char *newname)
{
	printf("EC_NAME:: %s\n", newname);
	strncpy(self->machines[mid]->instance_name, newname, MAX_NAME);
}

void en_pan(engine *self, int source, int sink, param value)
{

}


void en_amp(engine *self, int source, int sink, param value)
{

}


void en_set_timings(engine *self, int block_size, int tpb, int bpm)
{
	float temp;			

	self->block_size = block_size; 
  	self->tpb = tpb;
  	self->bpm = bpm; 
  	temp = (double)((self->bpm) * (self->tpb))/60.0;	// this is ticks-per-minute / 60 
  	temp = (double)OX_SAMPLING_RATE/temp;				// this gives samples-per-tick 
  	self->tick_size = (int)temp;					 	// loses precision!
}	


int en_add(engine *self, machine *m)
{
	int slot;

	// find first open space in table
	for (slot=1; (self->machines[slot] && slot < max_machines); slot++);
	
	assert(slot < max_machines);
	self->machines[slot] = m;
	if (slot == self->last_machine) self->last_machine++;		// only if there were no holes

	printf("EN_ADD:: Added machine at slot number %d. Type pointer is %p \n", slot, m->type);
	
	return slot;
}	


void en_remove(engine *self, int a)
{
	machine *m = self->machines[a];
	int i;

	printf("EC_REMOVE:: %d %p\n", a, m);
	assert(m);		
	self->machines[a] = 0;

	// now clean out the signal matrix. nobody gets signals from A,
	// nobody sends signals to A.

	for (i = 0; i < self->last_machine; i++) {
		self->signals[a][i] = 0;
		self->signals[i][a] = 0;
	}	

	// now recompile the flow network 
	en_reset_program(self);

}


void en_event		(engine *self, int mid, int chan, int param_ix, param value)
{
	machine *m = self->machines[mid];

	if ((m->num_channels <= chan)  ||  (m->type->num_params <= param_ix))
		printf("en_event: wacky info recieved: channel %d param %d\n", chan, param_ix);
	else mc_event(m, chan, param_ix, value);	
}



