/*
 * jitc_conv.c - Coder implementation for JIT conversions.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef IL_JITC_CODE

/*
 * Make sure that the given value is signed or unsigned.
 * Does a conversion if necessairy and returns the converted item
 * or returns the item as it is.
 */
static ILJitValue ConvertSign(ILJITCoder *jitCoder, ILJitValue value,
							  int isUnsigned, int checkOverflow)
{
	ILJitType type = jit_value_get_type(value);
	
	type = jit_type_normalize(type);

	if(isUnsigned)
	{
		if(type == _IL_JIT_TYPE_INT32)
		{
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_UINT32,
									checkOverflow);
		}
		else if(type == _IL_JIT_TYPE_INT64)
		{
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_UINT64,
									checkOverflow);
		}
	}
	else
	{
		if(type == _IL_JIT_TYPE_UINT32)
		{
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_INT32,
									checkOverflow);
		}
		else if(type == _IL_JIT_TYPE_UINT64)
		{
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_INT64,
									checkOverflow);
		}
	}
	return value;
}

/*
 * Convert the topmost stackitem according to the given opcode.
 * When unsigned is 0 then a sign extension will be done.
 * When checkOberflow is not 0 then an overflow check will be done.
 * The converted value will be returned.
 */
static ILJitValue ConvertValue(ILJITCoder *jitCoder, int opcode, int isUnsigned,
															int checkOverflow)
{
	ILJitValue value = jitCoder->jitStack[jitCoder->stackTop - 1];

	value = ConvertSign(jitCoder, value, isUnsigned, checkOverflow);
	switch(opcode)
	{
		case IL_OP_CONV_I1:
		{
			/* Convert to "int8" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_SBYTE,
									checkOverflow);
		}
		break;

		case IL_OP_CONV_I2:
		{
			/* Convert to "int16" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_INT16,
									checkOverflow);
		}
		break;

		case IL_OP_CONV_I4:
	#ifdef IL_NATIVE_INT32
		case IL_OP_CONV_I:
	#endif
		{
			/* Convert to "int32" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_INT32,
									checkOverflow);

		}
		break;

		case IL_OP_CONV_I8:
	#ifdef IL_NATIVE_INT64
		case IL_OP_CONV_I:
	#endif
		{
			/* Convert to "int64" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_INT64,
									checkOverflow);
		}
		break;

		case IL_OP_CONV_R4:
		{
			/* Convert to "float32" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_SINGLE,
									checkOverflow);
		}
		break;

		case IL_OP_CONV_R8:
		{
			/* Convert to "float64" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_DOUBLE,
									checkOverflow);
		}
		break;

		case IL_OP_CONV_U1:
		{
			/* Convert to "unsigned int8" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_BYTE,
									checkOverflow);
		}
		break;

		case IL_OP_CONV_U2:
		{
			/* Convert to "unsigned int16" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_UINT16,
									checkOverflow);
		}
		break;

		case IL_OP_CONV_U4:
	#ifdef IL_NATIVE_INT32
		case IL_OP_CONV_U:
	#endif
		{
			/* Convert to "unsigned int32" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_UINT32,
									checkOverflow);
		}
		break;

		case IL_OP_CONV_U8:
	#ifdef IL_NATIVE_INT64
		case IL_OP_CONV_U:
	#endif
		{
			/* Convert to "unsigned int64" */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_UINT64,
									checkOverflow);
		}
		break;

		case IL_OP_CONV_R_UN:
		{
			/* Convert to "native float" with unsigned input */
			return jit_insn_convert(jitCoder->jitFunction,
									value,
									_IL_JIT_TYPE_NFLOAT,
									checkOverflow);
		}
		break;
	}

	/* Just to keep gcc happy. */
	return value;
}

/*
 * Handle a conversion opcode.
 */
static void JITCoder_Conv(ILCoder *coder, int opcode, ILEngineType type)
{
	ILJITCoder *jitCoder = _ILCoderToILJITCoder(coder);
	ILJitValue value;

	/* Determine how to convert the value */
	switch(opcode)
	{
		case IL_OP_CONV_I1:
		{
			/* Convert to "int8" */
			value = ConvertValue(jitCoder, IL_OP_CONV_I1, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_INT32,
									 0);
		}
		break;

		case IL_OP_CONV_I2:
		{
			/* Convert to "int16" */
			value = ConvertValue(jitCoder, IL_OP_CONV_I2, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_INT32,
									 0);
		}
		break;

		case IL_OP_CONV_I4:
	#ifdef IL_NATIVE_INT32
		case IL_OP_CONV_I:
	#endif
		{
			/* Convert to "int32" */
			value = ConvertValue(jitCoder, IL_OP_CONV_I4, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_I8:
	#ifdef IL_NATIVE_INT64
		case IL_OP_CONV_I:
	#endif
		{
			/* Convert to "int64" */
			value = ConvertValue(jitCoder, IL_OP_CONV_I8, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_R4:
		{
			/* Convert to "float32" */
			value = ConvertValue(jitCoder, IL_OP_CONV_R4, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_NFLOAT,
									 0);
		}
		break;

		case IL_OP_CONV_R8:
		{
			/* Convert to "float64" */
			value = ConvertValue(jitCoder, IL_OP_CONV_R8, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_NFLOAT,
									 0);
		}
		break;

		case IL_OP_CONV_U1:
		{
			/* Convert to "unsigned int8" */
			value = ConvertValue(jitCoder, IL_OP_CONV_U1, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_UINT32,
									 0);
		}
		break;

		case IL_OP_CONV_U2:
		{
			/* Convert to "unsigned int16" */
			value = ConvertValue(jitCoder, IL_OP_CONV_U2, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_UINT32,
									 0);
		}
		break;

		case IL_OP_CONV_U4:
	#ifdef IL_NATIVE_INT32
		case IL_OP_CONV_U:
	#endif
		{
			/* Convert to "unsigned int32" */
			value = ConvertValue(jitCoder, IL_OP_CONV_U8, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_U8:
	#ifdef IL_NATIVE_INT64
		case IL_OP_CONV_U:
	#endif
		{
			/* Convert to "unsigned int64" */
			value = ConvertValue(jitCoder, IL_OP_CONV_U8, 0, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_R_UN:
		{
			/* Convert to "native float" with unsigned input */
			value = ConvertValue(jitCoder, IL_OP_CONV_R_UN, 1, 0);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_OVF_I1_UN:
		{
			/* Convert to "int8" with unsigned input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_I1, 1, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_INT32,
									 0);
		}
		break;

		case IL_OP_CONV_OVF_I2_UN:
		{
			/* Convert to "int16" with unsigned input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_I2, 1, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_INT32,
									 0);
		}
		break;

		case IL_OP_CONV_OVF_I4_UN:
	#ifdef IL_NATIVE_INT32
		case IL_OP_CONV_OVF_I_UN:
	#endif
		{
			/* Convert to "int32" with unsigned input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_I4, 1, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_OVF_I8_UN:
	#ifdef IL_NATIVE_INT64
		case IL_OP_CONV_OVF_I_UN:
	#endif
		{
			/* Convert to "int64" with unsigned input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_I8, 1, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_OVF_U1_UN:
		{
			/* Convert to "unsigned int8" with unsigned input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_U1, 1, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_UINT32,
									 0);
		}
		break;

		case IL_OP_CONV_OVF_U2_UN:
		{
			/* Convert to "unsigned int16" with unsigned input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_U2, 1, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_UINT32,
									 0);
		}
		break;

		case IL_OP_CONV_OVF_U4_UN:
	#ifdef IL_NATIVE_INT32
		case IL_OP_CONV_OVF_U_UN:
	#endif
		{
			/* Convert to "unsigned int32" with unsigned input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_U4, 1, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_OVF_U8_UN:
	#ifdef IL_NATIVE_INT64
		case IL_OP_CONV_OVF_U_UN:
	#endif
		{
			/* Convert to "unsigned int64" with unsigned input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_U8, 1, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_OVF_I1:
		{
			/* Convert to "int8" with signed input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_I1, 0, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_INT32,
									 0);
		}
		break;

		case IL_OP_CONV_OVF_U1:
		{
			/* Convert to "unsigned int8" with signed input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_U1, 0, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_UINT32,
									 0);
		}
		break;

		case IL_OP_CONV_OVF_I2:
		{
			/* Convert to "int16" with signed input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_I2, 0, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_INT32,
									 0);
		}
		break;

		case IL_OP_CONV_OVF_U2:
		{
			/* Convert to "unsigned int16" with signed input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_U2, 0, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] =
					jit_insn_convert(jitCoder->jitFunction,
									 value,
									 _IL_JIT_TYPE_UINT32,
									 0);
		}
		break;

		case IL_OP_CONV_OVF_I4:
	#ifdef IL_NATIVE_INT32
		case IL_OP_CONV_OVF_I:
	#endif
		{
			/* Convert to "int32" with signed input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_I4, 0, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_OVF_U4:
	#ifdef IL_NATIVE_INT32
		case IL_OP_CONV_OVF_U:
	#endif
		{
			/* Convert to "unsigned int32" with signed input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_U4, 0, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_OVF_I8:
	#ifdef IL_NATIVE_INT64
		case IL_OP_CONV_OVF_I:
	#endif
		{
			/* Convert to "int64" with signed input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_I8, 0, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;

		case IL_OP_CONV_OVF_U8:
	#ifdef IL_NATIVE_INT64
		case IL_OP_CONV_OVF_U:
	#endif
		{
			/* Convert to "unsigned int64" with signed input and overflow */
			value = ConvertValue(jitCoder, IL_OP_CONV_U8, 0, 1);
			jitCoder->jitStack[jitCoder->stackTop - 1] = value;
		}
		break;
	}
}

/*
 * Convert an I or I4 integer into a pointer.
 */
static void JITCoder_ToPointer(ILCoder *coder, ILEngineType type1,
							   ILEngineStackItem *type2)
{
	ILJITCoder *jitCoder = _ILCoderToILJITCoder(coder);
	ILJitValue *value;

	if(!type2)
	{
		value = &(jitCoder->jitStack[jitCoder->stackTop - 1]);
	}
	else
	{
		value = &(jitCoder->jitStack[jitCoder->stackTop - 2]);
	}
	*value = jit_insn_convert(jitCoder->jitFunction,
							  *value, _IL_JIT_TYPE_VPTR, 0);
}

/*
 * Output an instruction to convert the top of stack according
 * to a PInvoke marshalling rule.
 */
static void JITCoder_Convert(ILCoder *coder, int opcode)
{
}

/*
 * Output an instruction to convert the top of stack according
 * to a custom marshalling rule.
 */
static void JITCoder_ConvertCustom(ILCoder *coder, int opcode,
						    	   ILUInt32 customNameLen,
								   ILUInt32 customCookieLen,
						    	   void *customName, void *customCookie)
{
}

#endif	/* IL_JITC_CODE */
