/* --------------------------------------------------------------------------
 * ``Open Sound System (OSS)'' specific audio driver interface.
 * --------------------------------------------------------------------------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <unistd.h>

#if defined(HAVE_LINUX_SOUNDCARD_H)
  #include <linux/soundcard.h>
#elif defined(HAVE_SYS_SOUNDCARD_H)
  #include <sys/soundcard.h>
#elif defined(HAVE_MACHINE_SOUNDCARD_H)
  #include <machine/soundcard.h>
#elif defined(HAVE_SOUNDCARD_H)
  #include <soundcard.h>
#else
  #error Audio driver not supported.
#endif

#ifdef SID_WITH_SIDPLAY2
#include <sidplay/sidconfig.h>
#else
#include <sidplay/compconf.h>
#endif

#include "AudioDriverOSS.h"

#if defined(HAVE_NETBSD)
const char *AudioDriverOSS::AUDIODEVICE[] = {
    "/dev/sound",
    "/dev/audio",
    0
};
#else
const char *AudioDriverOSS::AUDIODEVICE[] = {
    "/dev/dsp",
    "/dev/sound/dsp",
    0
};
#endif

AudioDriverOSS::AudioDriverOSS()
{
	outOfOrder();
	swapEndian = false;
}

AudioDriverOSS::~AudioDriverOSS()
{
	reset();
	close();
}

void AudioDriverOSS::outOfOrder()
{
	// Reset everything.
	errorString = "None";
	audioHd = (-1);
}

bool AudioDriverOSS::isThere()
{
  for(unsigned int i=0; i<sizeof(AUDIODEVICE)/sizeof(const char*); i++)
    if(access(AUDIODEVICE[i], W_OK)==0)
      return true;
  return false;
}

bool AudioDriverOSS::reset()
{
	if (audioHd != (-1))
		return (ioctl(audioHd,SNDCTL_DSP_RESET,0)!=(-1));
	else
		return false;
}

bool AudioDriverOSS::open(const AudioConfig& inConfig)
{
  unsigned int i;

	// Copy input parameters. May later be replaced with driver defaults.
	config = inConfig;

	for(i=0; i<sizeof(AUDIODEVICE)/sizeof(const char *); i++)
    {
        if((audioHd=::open(AUDIODEVICE[i], O_WRONLY, 0)) >= 0)
            break;
		perror(AUDIODEVICE[i]);
    }
	if (audioHd == (-1))
	{
		errorString = "ERROR: Could not open audio device.\n	   See standard error output.";
		return false;
	}
	
	const char *audiodev=AUDIODEVICE[i];

	if (config.maxFrags!=0 && config.fragSize!=0)
	{
		int maxFrags = config.maxFrags;
		if (maxFrags!=0 && (maxFrags<2))
			maxFrags = 2;  // demand double-buffering at least
		
		int fragSizeExp = 0;
		int fragSize = config.fragSize;
		while (fragSize > 1)
		{
			fragSize >>= 1;
			fragSizeExp++;
		};
		// OSS pguide recommends not to check range.
		// if (fragSizeExp < 7)
		//	fragSizeExp = 7;  // 2^7=128 at least
		
		// N fragments of size (2 ^ S) bytes
		//			   NNNNSSSS
		// e.g. frag = 0x0004000e;
		// fragments should be out of [2,3,..,255]
		// fragSizeExp should be out of [7,8,...,17]
		// depending on the kernel audio driver buffer size
		int val = (maxFrags << 16) | fragSizeExp;
		if (ioctl(audioHd,SNDCTL_DSP_SETFRAGMENT,&val) == (-1))
		{
			perror(audiodev);
			errorString = "ERROR: Could not set fragmentation parameters.\n	   See standard error output.";
			return false;
		}
	}
	
	int mask;
	// Query supported sample formats.
	if (ioctl(audioHd,SNDCTL_DSP_GETFMTS,&mask) == (-1))
	{
		perror(audiodev);
		errorString = "AUDIO: Could not get sample formats.";
		return false;
	}

	// Assume CPU and soundcard have same endianess.
	swapEndian = false;
	
	int wantedFormat;
	// Set sample precision and type of encoding.
	if (config.precision == AudioConfig::BITS_16)
	{
#if defined(SID_WORDS_BIGENDIAN) || defined(WORDS_BIGENDIAN)
		if (mask & AFMT_S16_BE)
		{
			wantedFormat = AFMT_S16_BE;
			config.encoding = AudioConfig::SIGNED_PCM;
		}
		else if (mask & AFMT_U16_BE)
		{
			wantedFormat = AFMT_U16_BE;
			config.encoding = AudioConfig::UNSIGNED_PCM;
		}
		else if (mask & AFMT_S16_LE)
		{
			wantedFormat = AFMT_S16_LE;
			config.encoding = AudioConfig::SIGNED_PCM;
			swapEndian = true;
		}
		else if (mask & AFMT_U16_LE)
		{
			wantedFormat = AFMT_U16_LE;
			config.encoding = AudioConfig::UNSIGNED_PCM;
			swapEndian = true;
		}
#else
		if (mask & AFMT_S16_LE)
		{
			wantedFormat = AFMT_S16_LE;
			config.encoding = AudioConfig::SIGNED_PCM;
		}
		else if (mask & AFMT_U16_LE)
		{
			wantedFormat = AFMT_U16_LE;
			config.encoding = AudioConfig::UNSIGNED_PCM;
		}
		else if (mask & AFMT_S16_BE)
		{
			wantedFormat = AFMT_S16_BE;
			config.encoding = AudioConfig::SIGNED_PCM;
			swapEndian = true;
		}
		else if (mask & AFMT_U16_BE)
		{
			wantedFormat = AFMT_U16_BE;
			config.encoding = AudioConfig::UNSIGNED_PCM;
			swapEndian = true;
		}
#endif
		else  // 16-bit not supported
		{
			wantedFormat = AFMT_U8;
			config.precision = AudioConfig::BITS_8;
			config.encoding = AudioConfig::UNSIGNED_PCM;
		}
	}
	else  // if (precision == AudioConfig::BITS_8)
	{
		wantedFormat = AFMT_U8;
		config.precision = AudioConfig::BITS_8;
		config.encoding = AudioConfig::UNSIGNED_PCM;
	}

	if ( !(mask&wantedFormat) )
	{
		errorString = "AUDIO: Desired sample encoding not supported.";
		return false;
	}
	
	int format = wantedFormat;
	if (ioctl(audioHd,SNDCTL_DSP_SETFMT,&format) == (-1))
	{
		perror(audiodev);
		errorString = "AUDIO: Could not set sample format.\n	   See standard error output.";
		return false;
	}
	if (format != wantedFormat)  
	{
		errorString = "AUDIO: Audio driver did not accept sample format.";
		return false;
	}
	
	// Set mono/stereo.
	int dsp_stereo;
	if (config.channels == AudioConfig::STEREO)
		dsp_stereo = 1;
	else  // if (config.channels == AudioConfig::MONO)
		dsp_stereo = 0;
	if (ioctl(audioHd,SNDCTL_DSP_STEREO,&dsp_stereo) == (-1))
	{
		perror(audiodev);
		errorString = "AUDIO: Could not set mono/stereo.\n	   See standard error output.";
		return false;
	}
	// Verify and accept the number of channels the driver accepted.
	if (dsp_stereo == 1)
		config.channels = AudioConfig::STEREO;
	else if (dsp_stereo == 0)
		config.channels = AudioConfig::MONO;
	else
	{
		// Can this ever happen?
		errorString = "AUDIO: This error should never happen.";
		return false;
	}
	
	// Set frequency.
	int dsp_speed = config.frequency;
	if (ioctl(audioHd,SNDCTL_DSP_SPEED,&dsp_speed) == (-1))
	{
		perror(audiodev);
		errorString = "AUDIO: Could not set frequency.\n	   See standard error output.";
		return false;
	}
	// Accept the frequency the driver accepted.
	config.frequency = dsp_speed;

	// Get driver fragment size.
	ioctl(audioHd,SNDCTL_DSP_GETBLKSIZE,&config.blockSize);

// Not used yet.
//
//	audio_buf_info myAudInfo;
//	if (ioctl(audioHd,SNDCTL_DSP_GETOSPACE,&myAudInfo) == (-1))
//	{
//		perror(AUDIODEVICE);
//		errorString = "AUDIO: Could not get audio_buf_info.\n	   See standard error output.";
//		return false;
//	}
	
	errorString = "OK";
	return true;
}

void AudioDriverOSS::close()
{
	if (audioHd != (-1))
	{
		::close(audioHd);
		outOfOrder();
	}
}

void AudioDriverOSS::unload()
{
    close();
}

void AudioDriverOSS::play(void* buffer, unsigned long int bufferSize)
{
	if (audioHd != (-1))
	{
		if (swapEndian)
		{
			unsigned char* pBuffer = (unsigned char*)buffer;
			for (unsigned long int n=0; n<bufferSize; n+=2)
			{
				unsigned char tmp = pBuffer[n+0];
				pBuffer[n+0] = pBuffer[n+1];
				pBuffer[n+1] = tmp;
			}
		}
		write(audioHd,(char*)buffer,bufferSize);
	}
}
