/**
 * @(#)sidid.cpp 2007/11/04
 *
 * @author Stefano Tognon (ice00)
 * @version 1.00
 *
 * Note: This code is based onto Cadaver sidid:
 * SIDId V1.07 - Quick & dirty HVSC playroutine identity scanner
 * Cadaver 9/2007 (loorni@gmail.com)
 * 
 * Copyright (C) 2006-2007 by the author & contributors. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ''AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * This file is part of XSidplay2.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef SIDID
#define SIDID

#include "sidid.h"
#include <stdio.h>
#include <iostream>

#define MAX_SIGSIZE 4096
#define MAX_PATHNAME 256
#define END -1
#define ANY -2
#define AND -3
#define NAME -4

const char * const SidId::m_versionString = "SIDId V1.07 by Cadaver (C) 2007";

/**
 * Constructor
 */
SidId::SidId() {
 ///std::cerr << "constuctor ok\n";
}

/**
 * Desctructor
 */
SidId::~SidId() { 
 ///std::cerr << "destructor ok\n";
}

/**
 * Get the number of players recognized
 *
 * @return the number of players recognized
 */ 
unsigned int SidId::getNumberOfPlayers() {
  return m_sidIdList.size();
}

/**
 * Get the number of patterns used
 *
 * @return the number of patterns used
 */ 
unsigned int SidId::getNumberOfPatterns() {
  unsigned int n=0;

  for (sidIdList::iterator it = m_sidIdList.begin(); it != m_sidIdList.end(); ++it) {
    n+=it->list.size();
  }
  return n;
}

/**
 * Read the patterns id file
 *
 * @param name the name (with path) of the file to read
 * @return true if read is ok
 */
bool SidId::readConfig(const char *name) {
  char tokenstr[MAX_PATHNAME];
  int temp[MAX_SIGSIZE];
  int sigsize = 0;

  FILE *in = fopen(name, "rt");
  if (!in) return false;

  m_sidIdList.clear();    // clear the list as we can read a config more time

  try {
    for (;;) {
      int len;

      tokenstr[0] = 0;
      fscanf(in, "%s", tokenstr);
      len = strlen(tokenstr);

      if (len) {
        int token = NAME;  // suppose this is a NAME declaration

        if (!strcmp("??", tokenstr)) token = ANY;
        if ((!strcmp("end", tokenstr)) || (!strcmp("END", tokenstr))) token = END;
        if ((!strcmp("and", tokenstr)) || (!strcmp("AND", tokenstr))) token = AND;
        if ((len == 2) && (isHex(tokenstr[0])) && (isHex(tokenstr[1]))) {
          token = getHex(tokenstr[0]) * 16 + getHex(tokenstr[1]);
        }

        switch (token) {
          // name declaration
          case NAME: {
            sidIdRecord newid; 
            newid.name = tokenstr;///strdup(tokenstr);

            m_sidIdList.push_back(newid);
            sigsize = 0;
          }
          break;

          case END:
            if (sigsize >= MAX_SIGSIZE) throw "Maximum signature size exceeded!\n";
            temp[sigsize++] = END;
            if (sigsize > 1) {
              int c;

              sidSigRecord newsig;
              int *newbytes = new int[sigsize];
              if (!newbytes) throw ("Out of memory!\n");

              newsig.bytes = newbytes;
              for (c = 0; c < sigsize; c++) {
                newsig.bytes[c] = temp[c];
              }

              if (m_sidIdList.empty()) throw "No playername defined before signature!\n";

              m_sidIdList.back().list.push_back(newsig);
            }
            sigsize = 0;
            break;

          default:
            if (sigsize >= MAX_SIGSIZE) throw "Maximum signature size exceeded!\n";
            temp[sigsize++] = token;
            break;
        }
      }
      else break;
    }
  } catch (...) {
      fclose(in);
      return false;
    }

  fclose(in);
  return true;
}

/**
 * Identify the IDs of the given buffer
 *
 * @param buffer the buffed with the data to identify
 * @param length length of the buffer
 * @return the identified engines as string
 */
std::string SidId::identifyBuffer(const unsigned char *buffer, int length) {
  std::string res="";
  // scan all the engines
  for (sidIdList::iterator it = m_sidIdList.begin(); it != m_sidIdList.end(); ++it) {
    // scan all signature for one engine
    for (sidSigList::iterator it2 = it->list.begin(); it2 != it->list.end(); ++it2) {
      if (identifyBytes(it2->bytes, buffer, length)) {
        res+=it->name+" ";
        break;
      }
    }
  }
  return res;
}

/**
 * Identify the bytes of ID and buffer according to the pattern rules
 *
 * @param bytes the ID bytes pattern
 * @param buffer the SID file buffer
 * @param length length of the buffer
 * @return true if ID is matching 
 */
bool SidId::identifyBytes(int *bytes, const unsigned char *buffer, int length) {
  int c = 0, d = 0, rc = 0, rd = 0;

  while (c < length) {
    if (d == rd) {
      if (buffer[c] == bytes[d]) {
        rc = c+1;
        d++;
      }
      c++;
    } else {
  	if (bytes[d] == END) return true;
  	if (bytes[d] == AND) {
          d++;
          while (c < length) {
          if (buffer[c] == bytes[d]) {
            rc = c+1;
            rd = d;
            break;
          }
          c++;
        }
        if (c >= length) return false;
  		}
        if ((bytes[d] != ANY) && (buffer[c] != bytes[d])) {
          c = rc;
          d = rd;
        } else {
            c++;
            d++;
          }
      }
  }
  return false;
}


/**
 * Determine if the passed char is an exe number
 * 
 * @param c the char to test
 * @return true if the char is an hex number
 */
int SidId::isHex(char c) {
  if ((c >= '0') && (c <= '9')) return 1;
  if ((c >= 'a') && (c <= 'f')) return 1;
  if ((c >= 'A') && (c <= 'F')) return 1;
  return 0;
}

/**
 * Get the decimal number of the passed exe number char
 * 
 * @param c the char to use
 * @return the decimal number of the hex char passed
 */
int SidId::getHex(char c) {
  if ((c >= '0') && (c <= '9')) return c - '0';
  if ((c >= 'a') && (c <= 'f')) return c - 'a' + 10;
  if ((c >= 'A') && (c <= 'F')) return c - 'A' + 10;
  return -1;
}

#endif  /* SIDID */

