/************************************************************************
 *
 *  ConverterFactory.java
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright: 2002-2007 by Henrik Just
 *
 *  All Rights Reserved.
 * 
 *  Version 0.5 (2007-10-15)
 *
 */
 
package writer2latex.api;

import java.io.InputStream;

import writer2latex.api.OutputFile;
import writer2latex.office.MIMETypes;
import writer2latex.util.Config;
import writer2latex.xhtml.XhtmlDocument;
import writer2latex.xmerge.EmbeddedObject;

/** <p>This is a factory class to create converters for documents in
 *  OpenDocument/OOo 1.x format into a specific MIME type.</p>
 */
public class ConverterFactory {

    // Version information
    private static final String VERSION = "0.5";
    private static final String DATE = "2007-10-15";
	
    /** <p>Return version information</p>
     *  @return the Writer2LaTeX version
     */
    public static String getVersion() { return VERSION; }

    /** <p>Return date information</p>
     *  @return the date for this Writer2LaTeX version
     */
    public static String getDate() { return DATE; }


    private Config config = null;
    private GraphicConverter gc = null;
    
    /** <p>Define the configuration to use when converting. If a configuration
     * is specified, it will be used by converters which are later created using
     * the <code>createConverter</code> method</p>
     * <p>Note: This method will change in version 0.6</p>
     *
     *  @param config the configuration to use
     */
    public void setConfig(Config config) { this.config = config; }

    /** <p>Define a <code>GraphicConverter</code> implementation to use for
     *  conversion of graphic files.  If a converter
     * is specified, it will be used by converters which are later created using
     * the <code>createConverter</code> method</p>
     *
     *  @param gc the <code>GraphicConverter</code> to use
     */
    public void setGraphicConverter(GraphicConverter gc) { this.gc = gc; }

    /** <p>Create a <code>Converter</code> implementation which supports
     *  conversion into the specified MIME type</p>
     *  <p>Currently supported MIME types are:</p>
     *  <ul>
     *  <li><code>application/x-latex</code> for LaTeX format</li>
     *  <li><code>application/x-bibtex</code> for BibTeX format</li>
     *  <li><code>text/html</code> for XHTML 1.0 strict format</li>
     *  <li><code>application/xhtml+xml</code> for XHTML+MathML</li>
     *  <li><code>application/xml</code> for XHTML+MathML using stylesheets from w3c's
     *  math working group</li>
     *  </ul>
     *  @param sMIME the MIME type of the target format
     *  @return the required <code>Converter</code> or null if a converter for
     *  the requested MIME type could not be created
     */
    public Converter createConverter(String sMIME) {
        Converter converter = null;
        if (MIMETypes.LATEX.equals(sMIME)) {
            converter = new writer2latex.latex.ConverterPalette();
        }
        else if (MIMETypes.BIBTEX.equals(sMIME)) {
            converter = new writer2latex.bibtex.Converter();
        }
        else if (MIMETypes.XHTML.equals(sMIME)) {
            converter = new writer2latex.xhtml.Converter(XhtmlDocument.XHTML10);
        }
        else if (MIMETypes.XHTML_MATHML.equals(sMIME)) {
            converter = new writer2latex.xhtml.Converter(XhtmlDocument.XHTML_MATHML);
        }
        else if (MIMETypes.XHTML_MATHML_XSL.equals(sMIME)) {
            converter = new writer2latex.xhtml.Converter(XhtmlDocument.XHTML_MATHML_XSL);
        }
        else {
            return null;
        }
        if (config!=null) { converter.setConfig(config); }
        if (gc!=null) { converter.setGraphicConverter(gc); }
        return converter;
    }

}



