#ifndef TAGCOLL_OPTIONS_H
#define TAGCOLL_OPTIONS_H

/*
 * Commandline parser for tagcoll
 *
 * Copyright (C) 2003,2004,2005,2006  Enrico Zini
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <wibble/commandline/parser.h>

namespace wibble {
namespace commandline {

class TagcollParser : public StandardParserWithMandatoryCommand
{
public:
	// Input
	ExistingFileOption* in_derived;
	ExistingFileOption* in_extimpl;
	ExistingFileOption* in_rename;
	ExistingFileOption* in_patch;
	BoolOption* in_rmunfaceted;
	StringOption* in_rmtags;

	// Output
	BoolOption* out_group;
	BoolOption* out_redundant;
	BoolOption* out_itemsOnly;
	BoolOption* out_count;

	// Hierarchy
	IntOption* hie_flatten;
	IntOption* hie_filter;

	// Other options
	StringOption* misc_untaggedTag;
	IntOption* misc_distance;
	BoolOption* misc_invert;
	BoolOption* misc_quiet;

	// Commands
	Engine* copy;
	Engine* reverse;
	Engine* diff;
	Engine* related;
	Engine* implications;
	Engine* hierarchy;
	Engine* cleanhierarchy;
	Engine* dischierarchy;
	Engine* findspecials;
	Engine* grep;
	Engine* test;

protected:
	// Create the input options group
	OptionGroup* inputOptions()
	{
		OptionGroup* group = createGroup("Options controlling transformations of input data");
		in_derived = group->add<ExistingFileOption>(
					"derived-tags-from", 0, "derived", ""
					"expand derived tags using the given list");
		in_extimpl = group->add<ExistingFileOption>(
					"extimpl", 0, "implications-from", ""
					"use an external list of implications");
		in_rename = group->add<ExistingFileOption>(
					"rename", 0, "rename-from", ""
					"rename tags using the given mapping list");
		in_patch = group->add<ExistingFileOption>(
					"patch", 'p', "patch-with", ""
					"apply patches from the given tag patch file");
		in_patch->addAlias("patch");
		in_rmunfaceted	= group->add<BoolOption>(
					"rmunfaceted", 0, "remove-unfaceted", ""
					"while parsing, remove all tags with no facet part");
		in_rmtags		= group->add<StringOption>(
					"rmtags", 0, "remove-tags", "<expression>",
					"while parsing, remove all tags matching the given tag expression");
		return group;
	}

	// Create the output options group
	OptionGroup* outputOptions()
	{
		OptionGroup* group = createGroup("Options controlling transformations of output data");
		out_group	  = group->add<BoolOption>("group", 'g', "group", "",
						"group items with the same tagset in the output collection");
		out_group->addAlias("group-items");
		out_redundant = group->add<BoolOption>("redundant", 0, "redundant", "",
						"when implications are provided, expand them explicitly in the output");
		out_itemsOnly = group->add<BoolOption>("items", 'i', "items", "",
						"output only the names of the items, without the tags");
		out_count = group->add<BoolOption>("count", 'c', "count", "",
						"output the count of tags instead of the tags");
		return group;
	}

	// Create the hierarchy options group
	OptionGroup* hierarchyOptions()
	{
		OptionGroup* group = createGroup("Options controlling generation of tag hierarchies");
		hie_flatten = group->add<IntOption>("flatten", 0, "flatten-threshold", "",
						"set the number of total items below which a branch is flattened when "
						"using the \"hierarchy\" command (defaults to 0, meaning "
						"\"don't flatten\")");
		hie_filter =  group->add<IntOption>("filter", 'f', "filter", "",
						"filter out the tags with cardinality less than the given value "
						"(defaults to not filter; currently only works when building hierarchies)");
		return group;
	}

public:

	TagcollParser()
		: StandardParserWithMandatoryCommand("tagcoll", VERSION, 1, "enrico@enricozini.org")
	{
		usage = "<command> [options and arguments]";
		description = "Perform various operations on a tagged collection";
		
		OptionGroup* inputOpts = inputOptions();
		OptionGroup* outputOpts = outputOptions();
		OptionGroup* hierarchyOpts = hierarchyOptions();


		// 'copy' command
		copy = addEngine("copy", "[files...]",
				"output the collection",
				"Output the normalized collection on standard output, applying transformations "
				"if requested.  This is the default action if no other switches are provided.  "
				"A normalized collection is a collection in which an item appears in just one "
				"line.");
		copy->aliases.push_back("cat");
		copy->add(inputOpts);
		copy->add(outputOpts);

		// 'reverse' command
		reverse = addEngine("reverse", "[files...]",
				"\"reverse\" the collection, outputting one with items associated to tags",
				"Output the inbound collection \"reversed\" from the tags point of view, that is, "
				"associating to each tag the list of items associated to it in the input.\n"
				"The --untagged-tag switch can be used to provide a name to which untagged "
				"items will be associated in the output.");
		misc_untaggedTag = reverse->add<StringOption>("untagged-tag", 0, "untagged-tag", "<tag>",
				"set item name to use for associating untagged items when using the \"reverse\" "
				"command.  If not specified, untagged items are not included in the output");
		reverse->add(inputOpts);	
		reverse->add(outputOpts);	

		// 'diff' command
		diff = addEngine("diff", "<file1> <file2>",
				"output a tag patch file with the differences between two files",
				"Output a tag patch file with the differences between two files (requires two "
				"file arguments).\n"
				"The output tag patch file can then be applied when reading a collection with "
				"the --patch-with option.");
	
		// 'related' command
		related = addEngine("related", "<item> [files...]",
				"print a list of items related to the given one",
				"Output a list of the items that are related to the given item or list of items.  "
				"If more than one items are to be specified, separate them with commas.\n"
				"The --distance option can be used to control how closely related the output "
				"items shold be from the item(s) specified.");
		related->examples = "tagcoll related mutt,mozilla-browser -";
		misc_distance = related->add<IntOption>("distance", 'd', "distance", "",
				"set the maximum distance to use for the \"related\" command (defaults to 0)");
		related->add(inputOpts);

		// 'implications' command
		implications = addEngine("implications", "[files...]",
				"compute a list of tag implications",
				"Output a list of all implicit implications between tags contained in the "
				"hierarchy.  Implication is defined such that tag A implies tag B if every item "
				"tagged with A is also tagged with B.\n"
				"Implications can be used to discover implicit hierarchical relationships "
				"between tags.\n"
				"The output is one line per tag, with just tags that have implications, with the "
				"name of the package, a colon and a comma-separated list of all implied tags.");
		implications->examples =
				"C:devel,languages\n"
				"ada:devel,languages\n"
				"apachemodules:net,servers,web\n"
				"browsers:net,web\n";
		implications->add(inputOpts);

		// 'hierarchy' command
		hierarchy = addEngine("hierarchy", "[files...]",
				"build a smart hierarchy with the collection data",
				"Organize the collection in an intuitively navigable hierarchy.  The "
				"output is one line per package, with the package name prepended by the "
				"path to the item in the hierarchy.\n"
				"A detailed description of the hierarchy generation algorithm is found in the "
				"tagbk-draft.pdf draft paper available in this package; if you want to "
				"understand what are the goals of the algorithm and how it works, please give it "
				"a read.");
		hierarchy->examples =
				"/net/clients/mail: mutt\n"
				"/net/filters/mail: procmail\n";
		hierarchy->add(inputOpts);
		hierarchy->add(hierarchyOpts);

		// 'cleanhierarchy' command
		cleanhierarchy = addEngine("cleanhierarchy", "[files...]",
				"build a cleaned smart hierarchy with the collection data",
				"Like hiearchy, but in every node it merges tags which are attached to the "
				"same set of items.");
		cleanhierarchy->add(inputOpts);
		cleanhierarchy->add(hierarchyOpts);

		// 'cleanhierarchy' command
		dischierarchy = addEngine("dischierarchy", "[files...]",
				"build a hierarchy using discriminance properties of tags",
				"The tree starts with an empty tag set, and branches with the most "
				"discriminant tags.");
		dischierarchy->add(inputOpts);
		dischierarchy->add(hierarchyOpts);

		// 'findspecials' command
		findspecials = addEngine("findspecials", "[files...]",
				"generate a smart hierarchy and print, for each toplevel tag, "
				"what are the items that make it toplevel instead of going below "
				"another tag");
		findspecials->add(inputOpts);
		findspecials->add(hierarchyOpts);

		// 'grep' group
		grep = addEngine("grep", "<expression> [files...]",
			"output the collection of tags that match the given tag expression");
		grep->add(inputOpts);
		grep->add(outputOpts);
		misc_invert = grep->add<BoolOption>("invert", 'v', "invert-match", "",
				"invert the sense of matching, to select non-matching lines");
		misc_quiet = grep->add<BoolOption>("quiet", 'q', "quiet", "",
				"do not write anything to standard output, but exit with 0 if any match is found");

		// 'test' group
		test = addEngine("test", "[files...]",
			"perform internal tests and timings");
		test->add(inputOpts);
	}
};

#if 0
struct TagcollOptions : public MainParser<CommandParser>
{
	struct HelpGroup : public OptionGroup
	{
		BoolOption* help;
		BoolOption* version;

		HelpGroup()
		{
			add(help = new BoolOption("help", 'h', "help"));
			add(version = new BoolOption("version", 'V', "version"));
			help->shortNames.push_back('?');
			help->description = "print an help message and exit";
			version->description = "print the program version and exit";
			description = "Help options";
		}
		~HelpGroup()
		{
			delete help; delete version;
		}
	} helpGroup;


	struct Generic : public OptionParser
	{
		Generic(TagcollOptions* cp) : OptionParser("")
		{
			add(&cp->helpGroup);
		}
	} generic;
	struct Help : public OptionParser
	{
		Help(TagcollOptions* cp) : OptionParser("help")
		{
			usage = "[command]";
			description = "print help informations";
		}
	} help;

#if 0
		//opts.add("verbose", 'v', "verbose", "enable verbose output");
		//opts.add("debug", 0, "debug", "enable debugging output (including verbose output)");
#endif
};
#endif

}
}

// vim:set ts=4 sw=4:
#endif
