/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

namespace Postler {
    public enum AccountType {
        MAILDIR,
        IMAP,
        SEARCH
    }

    public enum FolderType {
        INBOX,
        SENT,
        DRAFTS,
        QUEUE,
        TRASH,
        ARCHIVE,
        JUNK,
        GENERIC,
        MAX;
        public string get_stock_id () {
            return localized_folders [this].stock_id;
        }
        public string get_label () {
            return _(localized_folders [this].label);
        }
    }
    public struct MailFolder {
        public string? role;
        public string? stock_id;
        public string label;
        string[]? localized;
        int n_localized;
        public FolderType type;
        public int i_type;
    }

    const string[] sent_folders = {
        "[Gmail]~-Sent Mail", "[Gmail]~-Gesendet", "Gesendet"
    };
    const string[] drafts_folders = {
        "[Gmail]~-Drafts", "[Gmail]~-Entw&APw-rfe", "Entwurf"
    };
    const string[] queue_folders = {
        "Postausgang"
    };
    const string[] trash_folders = {
        "[Gmail]~-Trash", "[Gmail]~-Papierkorb", "Papierkorb"
    };
    const string[] archive_folders = {
        "[Gmail]~-All Mail", "[Gmail]~-Alle Nachrichten"
    };
    const string[] junk_folders = {
        "[Gmail]~-Spam"
    };

    /* The length values must be in the array to work around Vala mistakenly
       optimizing the value away when using arrays indirectly with MailFolder */
    /* The index at the end is required for FolderType to work, somehow the
       index is taken in place of the enum value */
    const MailFolder[] localized_folders = {
        { "INBOX", STOCK_INBOX, N_("Inbox"), null, 0, FolderType.INBOX, 0 },
        { "Sent", STOCK_SENT_MAIL, N_("Sent"), sent_folders, sent_folders.length, FolderType.SENT, 1 },
        { "Drafts", null, N_("Drafts"), drafts_folders, drafts_folders.length, FolderType.DRAFTS, 2 },
        { "Queue", STOCK_OUTBOX, N_("Outbox"), queue_folders, queue_folders.length, FolderType.QUEUE, 3 },
        { "Trash", STOCK_USER_TRASH, N_("Trash"), trash_folders, trash_folders.length, FolderType.TRASH, 4 },
        { "Archive", STOCK_ARCHIVE, N_("Archive"), archive_folders, archive_folders.length, FolderType.ARCHIVE, 5 },
        /* i18n: Junk, unsolicited bulk mail, spam */
        { null, null, N_("Junk"), junk_folders, junk_folders.length, FolderType.JUNK, 6 },
        { null, Gtk.STOCK_DIRECTORY, null, null, 0, FolderType.GENERIC, 7 }
    };

    const string hide_folder_default = "Apple Mail To Do";

    public class AccountInfo : GLib.Object {
        public string name;
        public string display_name;
        public AccountType type;
        public string realname;
        public string address;
        public string receive;
        public string send;
        public int send_port;
        public string username;
        public string password;
        public string prefix;
        public string path;
        public string certificate;
        public bool unverified;
        public string reply;
        public string organization;
        public string? signature;
        public string hide = hide_folder_default;
        public string[]? folders = { null, null, null, null, null, null, null, null };

        public unowned string? get_folder (FolderType type) {
            return folders[type];
        }

        public bool can_delete () {
            return type == AccountType.SEARCH || folders[FolderType.TRASH] != null;
        }

      public void set_folder (FolderType type, string folder) {
            folders[type] = folder;
        }

        public unowned MailFolder get_localized_folder (string folder_name) {
            for (int type = 0; type < FolderType.MAX; type++)
                if (this.folders[type] == folder_name)
                    return localized_folders[type];
            string sane_name = folder_name.replace ("[Google Mail]", "[Gmail]");
            for (int type = 0; type < FolderType.MAX; type++) {
                if (this.folders[type] == null
                 && localized_folders[type].localized != null
                 && sane_name in localized_folders[type].localized) {
                    this.folders[type] = folder_name;
                    unowned string? role = localized_folders[type].role;
                    if (role != null && !this.hide.contains (role))
                        this.hide += "," + role;
                    return localized_folders[type];
                }
            }
            return localized_folders[FolderType.GENERIC];
        }

        public delegate void MailFolderCallback (int type, MailFolder folder);
        public void verify_folders (MailFolderCallback callback) {
            for (int type = 0; type < FolderType.MAX; type++)
                if (this.folders[type] == null && localized_folders[type].role != null)
                    if (type != FolderType.GENERIC)
                        callback (type, localized_folders[type]);
        }
    }
}

public class Postler.Accounts : GLib.Object {
    string config_path;
    string account_file;
    string data_path;
    string? certificate_file;
    GLib.KeyFile keyfile;
    GLib.List<AccountInfo> infos;

    const string[] root_certificate_files = {
        "/etc/pki/tls/certs/ca-bundle.crt",
        "/etc/ssl/certs/ca-certificates.crt",
        "/etc/ssl/certs/ca-bundle.crt",
        "/usr/local/share/certs/ca-root-nss.crt"
    };

    public Accounts () {
        unowned string config_dir = Environment.get_user_config_dir ();
        config_path = config_dir + "/" + Config.PACKAGE_NAME + "/";
        account_file = config_path + "accountrc";

        unowned string data_dir = Environment.get_user_data_dir ();
        data_path = data_dir + "/" + Config.PACKAGE_NAME + "/mail/";

        foreach (unowned string file in root_certificate_files) {
            if (FileUtils.test (file, FileTest.EXISTS))
                certificate_file = file;
        }
        if (certificate_file == null)
            GLib.warning (_("Failed to find a root certificate file."));

        reload ();
    }

    void update_display_names () {
        /* A single account is "Inbox", a unique provider is "gmail" */
        uint num_accounts = 0;
        foreach (var info in infos) {
            if (info.type != AccountType.SEARCH)
                num_accounts++;
            string? domain = (info.address ?? info.name).rstr ("@");
            if (domain != null) {
                foreach (var account_info in infos) {
                    unowned string address = account_info.address ?? account_info.name;
                    if (account_info != info && domain in address) {
                        domain = null;
                        break;
                    }
                }
                domain = domain != null ? domain.next_char () : null;
                if (domain != null)
                    domain = domain.split (".", 2)[0];
            }
            info.display_name = domain ?? info.name;
        }
        if (num_accounts == 1)
            foreach (var info in infos)
                if (info.type != AccountType.SEARCH)
                    info.display_name = _("Inbox");
    }

    void write_keyfile () throws Error {
        if (DirUtils.create_with_parents (config_path, 0700) != 0)
            throw new GLib.FileError.FAILED (_("Config folder couldn't be created."));
        FileUtils.set_contents (account_file, keyfile.to_data ());

        unowned string config_dir = Environment.get_user_config_dir ();
        unowned string indicator_dir = "/indicators/messages/applications";
        string indicator_path = config_dir + indicator_dir;
        string filename = indicator_path + "/postler.desktop";
        bool exists = FileUtils.test (filename, FileTest.EXISTS);
        if (infos.length () != 0 && !exists) {
            DirUtils.create_with_parents (indicator_path, 0700);
            FileUtils.set_contents (filename,
                Config.DATADIR + "/applications/postler.desktop");
        }
        else if (infos.length () == 0)
            FileUtils.unlink (filename);

        string autostart_path = config_dir + "/autostart";
        filename = autostart_path + "/postler.desktop";
        exists = FileUtils.test (filename, FileTest.EXISTS);
        if (infos.length () != 0 && !exists) {
            DirUtils.create_with_parents (autostart_path, 0700);
            FileUtils.set_contents (filename,
                "[Desktop Entry]\nVersion=1.0\nType=Application\nName=Postler\n"
              + "Exec=postler -m service\nIcon=internet-mail");
        }
        else if (infos.length () == 0)
            FileUtils.unlink (filename);

        update_display_names ();
    }

    public void update () {
        try {
            foreach (AccountInfo account_info in infos) {
                string? group = account_info.name;
                return_if_fail (group != null);

                if (account_info.type == AccountType.MAILDIR) {
                    keyfile.set_string (group, "type", "maildir");
                    if (account_info.path != null
                     && !account_info.path.contains (data_path))
                        keyfile.set_string (group, "path", account_info.path);
                    if (account_info.hide != null
                     && account_info.hide != hide_folder_default)
                        keyfile.set_string (group, "hide", account_info.hide);
                }
                else if (account_info.type == AccountType.IMAP) {
                    keyfile.set_string (group, "type", "imap");
                    if (account_info.realname != null)
                        keyfile.set_string (group, "realname", account_info.realname);
                    if (account_info.address != null)
                        keyfile.set_string (group, "address", account_info.address);
                    if (account_info.username != null)
                        keyfile.set_string (group, "username", account_info.username);
                    if (account_info.password != null)
                        keyfile.set_string (group, "password", account_info.password);
                    if (account_info.prefix != null)
                        keyfile.set_string (group, "prefix", account_info.prefix);
                    if (account_info.receive != null)
                        keyfile.set_string (group, "receive", account_info.receive);
                    if (account_info.send != null)
                        keyfile.set_string (group, "send", account_info.send);
                    if (account_info.send_port != 0)
                        keyfile.set_integer (group, "send-port", account_info.send_port);
                    if (account_info.certificate != null)
                        keyfile.set_string (group, "certificate",
                                            account_info.certificate);
                    if (account_info.unverified)
                        keyfile.set_boolean (group, "unverified", true);
                    if (account_info.reply != null)
                        keyfile.set_string (group, "reply", account_info.reply);
                    if (account_info.organization != null)
                        keyfile.set_string (group, "organization",
                                            account_info.organization);
                    if (account_info.signature != null)
                        keyfile.set_string (group, "signature",
                                            account_info.signature);
                    if (account_info.hide != null
                     && account_info.hide != hide_folder_default)
                        keyfile.set_string (group, "hide", account_info.hide);
                    for (int type = 0; type < FolderType.MAX; type++)
                        if (account_info.folders[type] != null) {
                            keyfile.set_string (group,
                                "localized" + type.to_string (),
                                account_info.folders[type]);
                        }
                }
                else if (account_info.type == AccountType.SEARCH) {
                    keyfile.set_string (group, "type", "search");
                    keyfile.set_string (group, "path", account_info.path);
                }
                else
                    assert_not_reached ();
            }

            write_keyfile ();
        }
        catch (GLib.Error error) {
            GLib.critical ("Failed to save \"%s\": %s", account_file, error.message);
        }
    }

    bool remove_path (string path) throws GLib.FileError {
        Dir dir;
        try {
            dir = Dir.open (path, 0);
        }
        catch (Error error) {
            return FileUtils.remove (path) == 0;
        }
        string? name;
        while ((name = dir.read_name ()) != null) {
            string subpath = Path.build_filename (path, name);
            if (!remove_path (subpath))
                return false;
        }
        return FileUtils.remove (path) == 0;
    }

    void delete_account (AccountInfo info)
        requires (info.name != null) {
        try {
            if (info.type == AccountType.IMAP) {
                if (!remove_path (info.path))
                    throw new GLib.FileError.FAILED (
                        _("Failed to remove \"%s\"").printf (info.path));
            }
            keyfile.remove_group (info.name);
            write_keyfile ();
        }
        catch (GLib.Error error) {
            GLib.critical ("Failed to save \"%s\": %s", account_file, error.message);
        }
    }

    void reload () {
        infos = new GLib.List<AccountInfo> ();
        keyfile = new GLib.KeyFile ();
        try {
            keyfile.load_from_file (account_file, 0);
        }
        catch (GLib.Error error) {
            GLib.debug ("Failed to load \"%s\": %s", account_file, error.message);
        }

        foreach (string group in keyfile.get_groups ()) {
            try {
                string type = keyfile.get_string (group, "type");
                var info = new AccountInfo ();
                info.name = group;
                if (type == "maildir") {
                    info.type = AccountType.MAILDIR;
                    if (keyfile.has_key (group, "path"))
                        info.path = keyfile.get_string (group, "path");
                    else
                        info.path = data_path + info.name;
                    if (keyfile.has_key (group, "hide"))
                        info.hide = keyfile.get_string (group, "hide");
                }
                else if (type == "imap") {
                    info.type = AccountType.IMAP;
                    info.path = data_path + info.name;
                    if (keyfile.has_key (group, "realname"))
                        info.realname = keyfile.get_string (group, "realname");
                    if (keyfile.has_key (group, "address"))
                        info.address = keyfile.get_string (group, "address");
                    if (keyfile.has_key (group, "username"))
                        info.username = keyfile.get_string (group, "username");
                    if (keyfile.has_key (group, "password"))
                        info.password = keyfile.get_string (group, "password");
                    if (keyfile.has_key (group, "prefix"))
                        info.prefix = keyfile.get_string (group, "prefix");
                    if (keyfile.has_key (group, "receive"))
                        info.receive = keyfile.get_string (group, "receive");
                    if (keyfile.has_key (group, "send"))
                        info.send = keyfile.get_string (group, "send");
                    if (keyfile.has_key (group, "send-port"))
                        info.send_port = keyfile.get_integer (group, "send-port");
                    if (keyfile.has_key (group, "certificate"))
                        info.certificate = keyfile.get_string (group, "certificate");
                    if (keyfile.has_key (group, "unverified"))
                        info.unverified = keyfile.get_boolean (group, "unverified");
                    if (keyfile.has_key (group, "reply"))
                        info.reply = keyfile.get_string (group, "reply");
                    if (keyfile.has_key (group, "organization"))
                        info.organization = keyfile.get_string (group, "organization");
                    if (keyfile.has_key (group, "signature"))
                        info.signature = keyfile.get_string (group, "signature");
                    if (keyfile.has_key (group, "hide"))
                        info.hide = keyfile.get_string (group, "hide");
                    for (int ftype = 0; ftype < FolderType.MAX; ftype++) {
                        string keyname = "localized" + ftype.to_string ();
                        if (keyfile.has_key (group, keyname))
                            info.folders[ftype] = keyfile.get_string (group, keyname);
                    }
                }
                else if (type == "search") {
                    info.type = AccountType.SEARCH;
                    info.path = keyfile.get_string (group, "path");
                }
                else
                    throw new GLib.FileError.FAILED (
                        _("Invalid type \"%s\"").printf (type));
                infos.prepend (info);
            } catch (GLib.Error error) {
                /* i18n: File was found but can't contains invalid values */
                GLib.critical (_("Failed to parse account in \"%s\": %s"),
                                 account_file, error.message);
            }
        }
        infos.reverse ();
        update_display_names ();
    }

    public string[] get_folders () {
        string[] folders = {};
        foreach (unowned AccountInfo info in infos) {
            if (info.type != AccountType.SEARCH)
                folders += info.path;
        }
        return folders;
    }

    public unowned List<AccountInfo> get_infos () {
        return infos;
    }

    public virtual signal void add_info (AccountInfo info) {
        if (info.type == AccountType.IMAP)
            info.path = data_path + info.name;
        infos.append (info);
        update ();
    }

    public virtual signal void remove_info (AccountInfo info) {
        infos.remove (info);
        delete_account (info);
    }

    string? get_tool_configfile (string? send, AccountInfo info) throws GLib.FileError {
        unowned string cache_dir = Environment.get_user_cache_dir ();
        string cache_path = cache_dir + "/postler/mail";
        if (DirUtils.create_with_parents (cache_path, 0700) != 0)
            throw new GLib.FileError.FAILED (_("Cache folder couldn't be created."));

        if (send != null)
            return "%s/%s.msmtprc".printf (cache_path, info.name);

        switch (info.type) {
        case AccountType.IMAP:
            return "%s/%s.mbsyncrc".printf (cache_path, info.name);
        case AccountType.MAILDIR:
        case AccountType.SEARCH:
            throw new GLib.FileError.FAILED (_("This type can't receive mail."));
        }
        return null;
    }

    string get_tool_configuration (string? send, AccountInfo info) throws GLib.FileError {
        string filename = get_tool_configfile (send, info);

        /* No need to update if the file exists and is older than accountrc */
        Posix.Stat account_file_status;
        Posix.stat (account_file, out account_file_status);
        Posix.Stat filename_status;
        Posix.stat (filename, out filename_status);
        if (account_file_status.st_mtime < filename_status.st_mtime)
            return filename;

        if (certificate_file == null)
            throw new GLib.FileError.FAILED (_("No SSL certificates available"));

        if (info.address != null) {
            string[] address_parts = info.address.split_set ("@,", 3);
            if (address_parts[0] == null || address_parts[1] == null)
                throw new GLib.FileError.FAILED (_("Invalid address"));
            if (send == null && info.receive == null)
                info.receive = "imap." + address_parts[1];
            if (send != null && info.send == null)
                info.send = "smtp." + address_parts[1];
            if (info.username == null)
                info.username = address_parts[0];
        }

        if (send == null && info.receive == null)
            throw new GLib.FileError.FAILED (_("Hostname is missing"));
        if (send != null && info.send == null)
            throw new GLib.FileError.FAILED (_("Hostname is missing"));
        if (info.username == null)
            throw new GLib.FileError.FAILED (_("Username is missing"));
        if (info.password == null)
            throw new GLib.FileError.FAILED (_("Password is missing"));
        string certificate = info.certificate;
        if (certificate != null && !Path.is_absolute (certificate))
            certificate = info.path + "/" + certificate;

        /* Exclude hidden folders from fetching */
        string[] hidden_tmp = info.hide.split (",");
        string hidden_folders = "";
        foreach (string hidden in hidden_tmp) {
            if (hidden.contains (" "))
                hidden_folders += " \"!" + hidden.replace ("~-", "/") + "\"";
            else
                hidden_folders += " !" + hidden;
        }

        if (send != null) {
            string tls_options;
            if (info.unverified)
                tls_options = "tls_certcheck off";
            else
                tls_options = "tls_trust_file %s\ntls_trust_file %s".printf (
                    certificate_file,
                    certificate != null ? certificate : certificate_file);
            if (info.send_port == 465)
                tls_options += "\ntls_starttls off"; /* SMTPS */
            string msmtprc = """
                defaults
                auth on
                tls on
                %s
                auto_from on
                maildomain %s
                account postler
                host %s
                from %s
                user %s
                password %s
                port %d
                timeout 90
                account default : postler
                """.
                printf (
                    tls_options,
                    info.address.split (",")[0].split ("@")[1],
                    info.send,
                    info.address.split (",")[0],
                    info.username,
                    info.password,
                    info.send_port != 0 ? info.send_port : 25
                    );
            FileUtils.set_contents (filename, msmtprc, -1);
            FileUtils.chmod (filename, 0600);
            return filename;
        }

        switch (info.type) {
        case AccountType.IMAP:
            string mbsyncrc = """
               Sync Pull
               Create Slave
               SyncState *

               IMAPStore remote
                   Host %s
                   User %s
                   Pass "%s"
                   Path "%s"
                   UseIMAPS yes
                   CertificateFile %s
                   %s %s

               MaildirStore local
                   Path %s/
                   Inbox %s/INBOX

               Channel local-remote
                   Master :remote:
                   Slave :local:
                   Patterns %% * !INBOX.*
                   Patterns %% %s
                   MaxMessages 500

               Group mirror
                   Channel local-remote:INBOX
                   Channel local-remote
               """.
               printf (
                   info.receive,
                   info.username,
                   info.password,
                   info.prefix != null ? info.prefix + "." : "",
                   certificate_file,
                   certificate != null ? "CertificateFile" : "",
                   certificate != null ? certificate : "",
                   info.path,
                   info.path,
                   hidden_folders
                   );
            FileUtils.set_contents (filename, mbsyncrc, -1);
            FileUtils.chmod (filename, 0700);
            break;
        case AccountType.MAILDIR:
        case AccountType.SEARCH:
            throw new GLib.FileError.FAILED (
                _("Account \"%s\" can't receive mail."), info.name);
        }
        return filename;
    }

    public string get_receive_command (AccountInfo info) throws FileError {
        string command;
        
        if (info.type == AccountType.IMAP) {
            unowned string? mbsync = Environment.get_variable ("POSTLER_MBSYNC");
            if (mbsync == null || mbsync == "")
                mbsync = "postler-mbsync";
            string filename = get_tool_configuration (null, info);
            command = "%s -c %s --pull%s mirror".printf (
                mbsync, filename, " --push --expunge");
        }
        else
            throw new GLib.FileError.FAILED (
                _("Account \"%s\" can't receive mail."), info.name);

        if (DirUtils.create_with_parents (info.path, 0700) != 0)
            throw new GLib.FileError.FAILED (_("Mail folder couldn't be created."));
        return command;
    }

    public string get_fetch_command (AccountInfo info) throws FileError {
        string command;

        if (info.type == AccountType.IMAP) {
            unowned string? mbsync = Environment.get_variable ("POSTLER_MBSYNC");
            if (mbsync == null || mbsync == "")
                mbsync = "postler-mbsync";
            string filename = get_tool_configuration (null, info);
            command = "%s --list -c %s mirror".printf (
                mbsync, filename);
        }
        else
            throw new GLib.FileError.FAILED (
                _("Account \"%s\" can't receive mail."), info.name);

        if (DirUtils.create_with_parents (info.path, 0700) != 0)
            throw new GLib.FileError.FAILED (_("Mail folder couldn't be created."));
        return command;
    }

    public string get_send_command (AccountInfo info, string send) throws FileError {
        string filename = get_tool_configuration (send, info);
        string command = "%s -c 'cat \"%s\" | msmtp -C %s -t'".printf (
            Environment.get_variable ("SHELL"), send, filename);
        return command;
    }
}

