//
// Timer.cpp
//
// $Id: //poco/1.3/Foundation/src/Timer.cpp#2 $
//
// Library: Foundation
// Package: Threading
// Module:  Timer
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Timer.h"
#include "Poco/ThreadPool.h"
#include "Poco/Exception.h"
#include "Poco/ErrorHandler.h"


namespace Poco {


Timer::Timer(long startInterval, long periodicInterval): 
	_startInterval(startInterval), 
	_periodicInterval(periodicInterval),
	_pCallback(0)
{
	poco_assert (startInterval >= 0 && periodicInterval >= 0);
}


Timer::~Timer()
{
	stop();
}


void Timer::start(const AbstractTimerCallback& method)
{
	start(method, Thread::PRIO_NORMAL, ThreadPool::defaultPool());
}


void Timer::start(const AbstractTimerCallback& method, Thread::Priority priority)
{
	start(method, priority, ThreadPool::defaultPool());
}


void Timer::start(const AbstractTimerCallback& method, ThreadPool& threadPool)
{
	start(method, Thread::PRIO_NORMAL, threadPool);
}


void Timer::start(const AbstractTimerCallback& method, Thread::Priority priority, ThreadPool& threadPool)
{
	poco_assert (!_pCallback);

	FastMutex::ScopedLock lock(_mutex);	
	_pCallback = method.clone();
	_wakeUp.reset();
	threadPool.startWithPriority(priority, *this);
}


void Timer::stop()
{
	FastMutex::ScopedLock lock(_mutex);
	if (_pCallback)
	{
		_periodicInterval = 0;
		_mutex.unlock();
		_wakeUp.set();
		_done.wait(); // warning: deadlock if called from timer callback
		_mutex.lock();
		delete _pCallback;
		_pCallback = 0;
	}
}


void Timer::restart()
{
	FastMutex::ScopedLock lock(_mutex);
	if (_pCallback)
	{
		_wakeUp.set();
	}
}


void Timer::restart(long milliseconds)
{
	poco_assert (milliseconds >= 0);
	FastMutex::ScopedLock lock(_mutex);
	if (_pCallback)
	{
		_periodicInterval = milliseconds;
		_wakeUp.set();
	}
}


long Timer::getStartInterval() const
{
	FastMutex::ScopedLock lock(_mutex);
	return _startInterval;
}


void Timer::setStartInterval(long milliseconds)
{
	poco_assert (milliseconds >= 0);
	FastMutex::ScopedLock lock(_mutex);
	_startInterval = milliseconds;
}


long Timer::getPeriodicInterval() const
{
	FastMutex::ScopedLock lock(_mutex);
	return _periodicInterval;
}


void Timer::setPeriodicInterval(long milliseconds)
{
	poco_assert (milliseconds >= 0);
	FastMutex::ScopedLock lock(_mutex);
	_periodicInterval = milliseconds;
}


void Timer::run()
{
	long interval;
	{
		FastMutex::ScopedLock lock(_mutex);
		interval = _startInterval;
	}
	do
	{
		if (_wakeUp.tryWait(interval))
		{
			FastMutex::ScopedLock lock(_mutex);
			interval = _periodicInterval;
		}
		else
		{
			try
			{
				_pCallback->invoke(*this);
			}
			catch (Exception& exc)
			{
				ErrorHandler::handle(exc);
			}
			catch (std::exception& exc)
			{
				ErrorHandler::handle(exc);
			}
			catch (...)
			{
				ErrorHandler::handle();
			}
			{
				FastMutex::ScopedLock lock(_mutex);
				interval = _periodicInterval;
			}
		}
	}
	while (interval > 0);
	_done.set();
}


AbstractTimerCallback::AbstractTimerCallback()
{
}


AbstractTimerCallback::AbstractTimerCallback(const AbstractTimerCallback& callback)
{
}


AbstractTimerCallback::~AbstractTimerCallback()
{
}


AbstractTimerCallback& AbstractTimerCallback::operator = (const AbstractTimerCallback& callback)
{
	return *this;
}


} // namespace Poco
