//
// ByteOrder.h
//
// $Id: //poco/1.3/Foundation/include/Poco/ByteOrder.h#1 $
//
// Library: Foundation
// Package: Core
// Module:  ByteOrder
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Foundation_ByteOrder_INCLUDED
#define Foundation_ByteOrder_INCLUDED


#include "Poco/Foundation.h"
#include "Poco/Types.h"


namespace Poco {


class Foundation_API ByteOrder
	/// This class contains a number of static methods
	/// to convert between big-endian and little-endian
	/// integers of various sizes.
{
public:
	static Int16 flipBytes(Int16 value);
	static UInt16 flipBytes(UInt16 value);
	static Int32 flipBytes(Int32 value);
	static UInt32 flipBytes(UInt32 value);
#if defined(POCO_HAVE_INT64)
	static Int64 flipBytes(Int64 value);
	static UInt64 flipBytes(UInt64 value);
#endif

	static Int16 toBigEndian(Int16 value);
	static UInt16 toBigEndian (UInt16 value);
	static Int32 toBigEndian(Int32 value);
	static UInt32 toBigEndian (UInt32 value);
#if defined(POCO_HAVE_INT64)
	static Int64 toBigEndian(Int64 value);
	static UInt64 toBigEndian (UInt64 value);
#endif

	static Int16 fromBigEndian(Int16 value);
	static UInt16 fromBigEndian (UInt16 value);
	static Int32 fromBigEndian(Int32 value);
	static UInt32 fromBigEndian (UInt32 value);
#if defined(POCO_HAVE_INT64)
	static Int64 fromBigEndian(Int64 value);
	static UInt64 fromBigEndian (UInt64 value);
#endif

	static Int16 toLittleEndian(Int16 value);
	static UInt16 toLittleEndian (UInt16 value);
	static Int32 toLittleEndian(Int32 value);
	static UInt32 toLittleEndian (UInt32 value);
#if defined(POCO_HAVE_INT64)
	static Int64 toLittleEndian(Int64 value);
	static UInt64 toLittleEndian (UInt64 value);
#endif

	static Int16 fromLittleEndian(Int16 value);
	static UInt16 fromLittleEndian (UInt16 value);
	static Int32 fromLittleEndian(Int32 value);
	static UInt32 fromLittleEndian (UInt32 value);
#if defined(POCO_HAVE_INT64)
	static Int64 fromLittleEndian(Int64 value);
	static UInt64 fromLittleEndian (UInt64 value);
#endif

	static Int16 toNetwork(Int16 value);
	static UInt16 toNetwork (UInt16 value);
	static Int32 toNetwork(Int32 value);
	static UInt32 toNetwork (UInt32 value);
#if defined(POCO_HAVE_INT64)
	static Int64 toNetwork(Int64 value);
	static UInt64 toNetwork (UInt64 value);
#endif

	static Int16 fromNetwork(Int16 value);
	static UInt16 fromNetwork (UInt16 value);
	static Int32 fromNetwork(Int32 value);
	static UInt32 fromNetwork (UInt32 value);
#if defined(POCO_HAVE_INT64)
	static Int64 fromNetwork(Int64 value);
	static UInt64 fromNetwork (UInt64 value);
#endif
};


//
// inlines
//
inline UInt16 ByteOrder::flipBytes(UInt16 value)
{
	return ((value >> 8) & 0x00FF) | ((value << 8) & 0xFF00);
}


inline Int16 ByteOrder::flipBytes(Int16 value)
{
	return Int16(flipBytes(UInt16(value)));
}


inline UInt32 ByteOrder::flipBytes(UInt32 value)
{
	return ((value >> 24) & 0x000000FF) | ((value >> 8) & 0x0000FF00)
	     | ((value << 8) & 0x00FF0000) | ((value << 24) & 0xFF000000);
}


inline Int32 ByteOrder::flipBytes(Int32 value)
{
	return Int32(flipBytes(UInt32(value)));
}


#if defined(POCO_HAVE_INT64)
inline UInt64 ByteOrder::flipBytes(UInt64 value)
{
	UInt32 hi = UInt32(value >> 32);
	UInt32 lo = UInt32(value & 0xFFFFFFFF);
	return UInt64(flipBytes(hi)) | (UInt64(flipBytes(lo)) << 32);
}


inline Int64 ByteOrder::flipBytes(Int64 value)
{
	return Int64(flipBytes(UInt64(value)));
}
#endif // POCO_HAVE_INT64


//
// some macro trickery to automate the method implementation
//
#define POCO_IMPLEMENT_BYTEORDER_NOOP_(op, type) \
	inline type ByteOrder::op(type value)		\
	{											\
		return value;							\
	}
#define POCO_IMPLEMENT_BYTEORDER_FLIP_(op, type) \
	inline type ByteOrder::op(type value)		\
	{											\
		return flipBytes(value);				\
	}


#if defined(POCO_HAVE_INT64)
	#define POCO_IMPLEMENT_BYTEORDER_NOOP(op) \
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, Int16)	\
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, UInt16)	\
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, Int32)	\
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, UInt32)	\
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, Int64)	\
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, UInt64)
	#define POCO_IMPLEMENT_BYTEORDER_FLIP(op) \
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, Int16)	\
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, UInt16)	\
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, Int32)	\
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, UInt32)	\
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, Int64)	\
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, UInt64)
#else
	#define POCO_IMPLEMENT_BYTEORDER_NOOP(op) \
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, Int16)	\
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, UInt16)	\
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, Int32)	\
		POCO_IMPLEMENT_BYTEORDER_NOOP_(op, UInt32)
	#define POCO_IMPLEMENT_BYTEORDER_FLIP(op) \
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, Int16)	\
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, UInt16)	\
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, Int32)	\
		POCO_IMPLEMENT_BYTEORDER_FLIP_(op, UInt32)
#endif


#if defined(POCO_ARCH_BIG_ENDIAN)
	#define POCO_IMPLEMENT_BYTEORDER_BIG POCO_IMPLEMENT_BYTEORDER_NOOP
	#define POCO_IMPLEMENT_BYTEORDER_LIT POCO_IMPLEMENT_BYTEORDER_FLIP
#else
	#define POCO_IMPLEMENT_BYTEORDER_BIG POCO_IMPLEMENT_BYTEORDER_FLIP
	#define POCO_IMPLEMENT_BYTEORDER_LIT POCO_IMPLEMENT_BYTEORDER_NOOP
#endif


POCO_IMPLEMENT_BYTEORDER_BIG(toBigEndian)
POCO_IMPLEMENT_BYTEORDER_BIG(fromBigEndian)
POCO_IMPLEMENT_BYTEORDER_BIG(toNetwork)
POCO_IMPLEMENT_BYTEORDER_BIG(fromNetwork)
POCO_IMPLEMENT_BYTEORDER_LIT(toLittleEndian)
POCO_IMPLEMENT_BYTEORDER_LIT(fromLittleEndian)


} // namespace Poco


#endif // Foundation_ByteOrder_INCLUDED
