/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file timer_kill_players.cpp
 * \brief Implementation of the bear::timer_kill_players class.
 * \author Sbastien Angibaud
 */
#include "ptb/item/timer_kill_players.hpp"

#include "universe/derived_item_handle.hpp"
#include "ptb/item/plee/plee.hpp"
#include "ptb/player_action.hpp"
#include "ptb/level_variables.hpp"
#include "ptb/game_variables.hpp"

BASE_ITEM_EXPORT( timer_kill_players, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to a timer_kill_players.
 * \param that The timer_kill_players to which we apply the message.
 */
bool ptb::timer_kill_players::msg_get_value::apply_to
( timer_kill_players& that )
{
  value = that.get_time();
  countdown = that.get_countdown();
  return true;
} // timer_kill_playersm::msg_get_value::apply_to()


/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::timer_kill_players::timer_kill_players()  
: m_time(0), m_save_time(0), m_countdown(false)
{
  set_phantom(true);
  set_can_move_items(false);
  set_global(true);
} // timer_kill_players::timer_kill_players()


/*----------------------------------------------------------------------------*/
/**
 * \brief Do post creation actions.
 */
void ptb::timer_kill_players::build()
{
  super::build();
 
  game_variables::set_current_level_is_finished(false);
} // ptb::timer_kill_players::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 */
void ptb::timer_kill_players::progress
( bear::universe::time_type elapsed_time )
{
  if ( ! game_variables::get_current_level_is_finished() ) 
    {
      search_players();

      bool marionette;
      if( m_first_player.get() )
	marionette = m_first_player->is_a_marionette();
      
      if ( marionette ) 
	if( m_second_player.get() )
	  marionette = m_second_player->is_a_marionette();
      
      if ( ! marionette ) 
	{
	  if ( m_countdown )
	    {
	      m_time -= elapsed_time;
	      if ( m_time <= 0 )
		{
		  m_time = m_save_time;
		  finish_countdown();
		}
	    }
	  else
	    m_time += elapsed_time;
	}
    }
} // timer_kill_players::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <bool>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::timer_kill_players::set_bool_field
( const std::string& name,bool value )
{
  bool result = true;

  if ( name == "timer_kill_players.countdown" )
    m_countdown = value;
  else
    result = super::set_bool_field( name, value );

  return result;
} // timer_kill_players::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type unsigned int.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::timer_kill_players::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool ok = true;

  if (name == "timer_kill_players.time")
    {
      m_time = (bear::universe::time_type)value;
      m_save_time = m_time;
    }
  else
    ok = super::set_u_integer_field( name, value );

  return ok;
} // timer_kill_players::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is well initialised.
 */
bool ptb::timer_kill_players::is_valid() const
{
  return !get_name().empty() && super::is_valid();
} // timer_kill_players::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the time.
 */
bear::universe::time_type ptb::timer_kill_players::get_time() const
{
  return m_time;
} // timer_kill_players::get_time()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return if the timer is a countdown.
 */
bool ptb::timer_kill_players::get_countdown() const
{
  return m_countdown;
} // timer_kill_players::get_countdown()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform that the countdown is finish.
 */
void ptb::timer_kill_players::finish_countdown()
{
  level_variables::set_time_exceeded( get_level() );
  plee::get_instance_message msg;
  get_level_globals().send_message(plee::player_name(1),msg);
  bear::universe::derived_item_handle_maker<plee>::handle_type
    p1(msg.get_instance());
  
  get_level_globals().send_message(plee::player_name(2),msg);
  bear::universe::derived_item_handle_maker<plee>::handle_type
    p2(msg.get_instance());
  
  if (p1.get() != NULL)
    p1->start_action(player_action::die);
  
  if (p2.get() != NULL)
    p2->start_action(player_action::die);
} // timer_kill_players::finish_countdown()

/*----------------------------------------------------------------------------*/
/**
 * \brief Search players.
 */
void ptb::timer_kill_players::search_players()
{
  plee::get_instance_message msg;
  
  if ( m_first_player.get() == NULL )
    {
      get_level_globals().send_message(plee::player_name(1),msg);
      m_first_player = msg.get_instance();
    }
  
  if ( m_second_player.get() == NULL )
    {
      msg.clear();
      get_level_globals().send_message(plee::player_name(2),msg);
      m_second_player = msg.get_instance();
    }
} // timer_kill_players::search_players()
