/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file link_on_players.cpp
 * \brief Implementation of the ptb::link_on_players class.
 * \author Julien Jorge
 */
#include "ptb/item/link_on_players.hpp"

#include "universe/link/link.hpp"

#include "engine/export.hpp"

#include <limits>

BASE_ITEM_EXPORT( link_on_players, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::link_on_players::link_on_players()
  : m_strength
  (std::numeric_limits<bear::universe::coordinate_type>::infinity()),
    m_minimal_length(0),
    m_maximal_length
  (std::numeric_limits<bear::universe::coordinate_type>::infinity()),
    m_current_sprite(&m_off), m_link_visual(NULL)
{

} // link_on_players::link_on_players()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The instance to copy from.
 */
ptb::link_on_players::link_on_players( const link_on_players& that )
  : super(that), m_strength(that.m_strength),
    m_minimal_length(that.m_minimal_length),
    m_maximal_length(that.m_maximal_length), m_off(that.m_off),
    m_single(that.m_single), m_all(that.m_all), m_current_sprite(&m_off),
    m_link_visual(NULL)
{
  if (that.m_link_visual != NULL)
    m_link_visual = that.m_link_visual->clone();
} // link_on_players::link_on_players()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::link_on_players::~link_on_players()
{
  delete m_link_visual;
} // link_on_players::~link_on_players()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the resources needed by this item.
 */
void ptb::link_on_players::pre_cache()
{
  super::pre_cache();
  get_level_globals().load_image("gfx/coop/linker.png");
} // link_on_players::pre_cache()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the item.
 */
void ptb::link_on_players::build()
{
  super::build();

  m_off = get_level_globals().auto_sprite("gfx/coop/linker.png", "off");
  m_single = get_level_globals().auto_sprite("gfx/coop/linker.png", "single");
  m_all = get_level_globals().auto_sprite("gfx/coop/linker.png", "all");

  m_current_sprite = &m_off;
} // link_on_players::pre_cache()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one step in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::link_on_players::progress( bear::universe::time_type elapsed_time )
{
  m_current_sprite = &m_off;

  super::progress(elapsed_time);
} // link_on_players::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void ptb::link_on_players::get_visual
( std::list<bear::engine::scene_visual>& visuals ) const
{
  super::get_visual( visuals );

  add_visual( *m_current_sprite, visuals );
} // link_on_players::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <real>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::link_on_players::set_real_field
( const std::string& name, double value )
{
  bool result = true;

  if ( name == "link_on_players.strength" )
    m_strength = value;
  else if ( name == "link_on_players.length.minimal" )
    m_minimal_length = value;
  else if ( name == "link_on_players.length.maximal" )
    m_maximal_length = value;
  else
    result = super::set_real_field( name, value );

  return result;
} // link_on_players::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <item>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::link_on_players::set_item_field
( const std::string& name, bear::engine::base_item* value )
{
  bool result = true;

  if ( name == "link_on_players.link_visual" )
    {
      bear::base_link_visual* item
        ( dynamic_cast<bear::base_link_visual*>(value) );

      if ( item == NULL )
        {
          result = false;
          claw::logger << claw::log_warning
                       << "link_on_players.link_visual: item is not of type "
                       << "base_link_visual." << std::endl;
        }
      else
        {
          m_link_visual = item->clone();
          item->kill();
        }
    }
  else
    result = super::set_item_field( name, value );

  return result;
} // link_on_players::set_item_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief One player is colliding with the item.
 * \param p The player.
 */
void ptb::link_on_players::on_one_player( plee* p )
{
  m_current_sprite = &m_single;
} // link_on_players::on_one_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief All players are colliding with the item.
 * \param p1 The first player.
 * \param p2 The second player.
 */
void ptb::link_on_players::on_all_players( plee* p1, plee* p2 )
{
  // all players does not mean two players in single player game.
  if ( (p1 != NULL) && (p2 != NULL) )
    {
      m_current_sprite = &m_all;

      if ( !p1->is_linked_to(*p2) )
        {
          bear::universe::link* link
            ( new bear::universe::link
              ( *p1, *p2, m_strength, m_minimal_length, m_maximal_length ) );

          if (m_link_visual != NULL)
            {
              bear::base_link_visual* vis(m_link_visual->clone());
              vis->set_items(p1, p2, link->get_id());
              new_item(*vis);
            }
        }
    }
} // link_on_players::on_all_players()
