/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file monster_item.tpp
 * \brief Implementation of the ptb::monster_item class.
 * \author Sbastien Angibaud
 */
#include "ptb/item/stone/stone.hpp"
#include "ptb/monster.hpp"

#include <claw/logger.hpp>
#include <sstream>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
template<class Base>
ptb::monster_item<Base>::monster_item()
  : m_opacity_injured(1), m_opacity_inc(-0.02)
{
  this->set_can_move_items(false);
} // monster_item::monster_item()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
template<class Base>
void ptb::monster_item<Base>::progress( bear::universe::time_type elapsed_time )
{
  super::progress( elapsed_time );

  if ( m_is_injured )
    {
      m_injured_time += elapsed_time;

      if ( m_injured_time >= 2 )
        finish_injure();
      else
        {
          m_opacity_injured += m_opacity_inc;

          if ( m_opacity_injured <= 0.3 )
            m_opacity_inc = 0.02;
          else if ( m_opacity_injured >= 1 )
            {
              m_opacity_inc = -0.02;
              m_opacity_injured = 1;
            }

          this->get_rendering_attributes().set_opacity(m_opacity_injured);
        }
    }
} // monster_item::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c unsigned \c integer.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
template<class Base>
bool ptb::monster_item<Base>::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool result = true;

  if ( name == "monster_item.offensive_strength" )
    m_offensive_force = value;
  else if ( name == "monster_item.offensive_coefficient.normal" )
    set_offensive_coefficient(normal_attack, value);
  else if ( name == "monster_item.offensive_coefficient.air" )
    set_offensive_coefficient(air_attack, value);
  else if ( name == "monster_item.offensive_coefficient.fire" )
    set_offensive_coefficient(fire_attack, value);
  else if ( name == "monster_item.offensive_coefficient.water" )
    set_offensive_coefficient(water_attack, value);
  else if ( name == "monster_item.offensive_coefficient.indefensible" )
    set_offensive_coefficient(indefensible_attack, value);
  else
    result = super::set_u_integer_field(name,value);

  return result;
} // monster_item::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c real.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
template<class Base>
bool ptb::monster_item<Base>::set_real_field
( const std::string& name, double value )
{
  bool ok = true;

  if (name == "monster_item.energy")
    set_energy(value);
  else
    ok = super::set_real_field(name, value);

  return ok;
} //  monster_item::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c bool.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
template<class Base>
bool ptb::monster_item<Base>::set_bool_field
( const std::string& name, bool value )
{
  bool ok = true;

  if (name == "monster_item.defensive_power.normal")
    set_defensive_power(normal_attack,value);
  else if (name == "monster_item.defensive_power.air")
    set_defensive_power(air_attack,value);
  else if (name == "monster_item.defensive_power.fire")
    set_defensive_power(fire_attack,value);
  else if (name == "monster_item.defensive_power.water")
    set_defensive_power(water_attack,value);
  else
    ok = super::set_bool_field(name,value);

  return ok;
} // monster_item::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c string.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
template<class Base>
bool ptb::monster_item<Base>::set_string_field
( const std::string& name, const std::string& value )
{
  bool result = true;

  if ( name == "monster_item.monster_type" )
    {
      if ( value == "player_1" )
        m_monster_type = player_1_monster;
      else if ( value == "player_2" )
        m_monster_type = player_2_monster;
      else if ( value == "enemy" )
        m_monster_type = enemy_monster;
      else if ( value == "stone_1" )
        m_monster_type = stone_1_monster;
      else if ( value == "stone_2" )
        m_monster_type = stone_2_monster;
      else if ( value == "nature" )
        m_monster_type = nature_monster;
      else
        {
          claw::logger << claw::log_warning
                       << "monster_item::set_string_field(): '" << name
                       << "' does not accept " << value << std::endl;
          result = false;
        }
    }
  else
    result = super::set_string_field(name,value);

  return result;
} // monster_item::set_string_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
template<class Base>
void ptb::monster_item<Base>::to_string( std::string& str ) const
{
  super::to_string(str);
  monster::to_string(str);
} // monster_item::to_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Process a collision.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
template<class Base>
void ptb::monster_item<Base>::collision
( bear::engine::base_item& that, bear::universe::collision_info& info )
{
  collision_and_attack(that,info);
} // monster_item::collision()

/*----------------------------------------------------------------------------*/
/**
 * \brief Process a collision and attack.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
template<class Base>
bool ptb::monster_item<Base>::collision_and_attack
( bear::engine::base_item& that, bear::universe::collision_info& info )
{
 bool result = false;
 ptb::monster* other = dynamic_cast<ptb::monster*>(&that);

 if (other != NULL)
   {
     if ( m_offensive_phase || m_invincible )
       result = other->receive_an_attack(*this,info);
     else
       super::collision(that,info);
   }
 else
   super::collision(that,info);

 return result;
} // monster_item::collision_and_attack()

/*----------------------------------------------------------------------------*/
/**
 * \brief We remove some energy of the item.
 * \param attacker The attacker monster.
 * \param energy The quantity of energy removed.
 */
template<class Base>
void ptb::monster_item<Base>::remove_energy
(const monster& attacker, double energy )
{
  if ( energy >= m_energy )
    m_energy = 0;
  else
    m_energy -= energy;
  
  if ( m_energy == 0)
    inform_no_energy(attacker);
} // monster_item::remove_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the item that he have no energy now.
 * \param attacker The attacker monster.
 */
template<class Base>
void ptb::monster_item<Base>::inform_no_energy(const monster& attacker)
{
  this->kill();
} // monster_item::inform_no_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief The monster is injure.
 * \param attacker The monster attacking me.
 */
template<class Base>
void ptb::monster_item<Base>::injure
(const monster& attacker,const bear::universe::collision_info& info)
{
  monster::injure(attacker,info);

  m_opacity_inc = -0.02;
} // ptb::monster_item::injure()

/*----------------------------------------------------------------------------*/
/**
 * \brief The monster isn't injure any more.
 */
template<class Base>
void ptb::monster_item<Base>::finish_injure()
{
  monster::finish_injure();

  this->get_rendering_attributes().set_opacity(1);
} // ptb::monster_item::finish_injure()
