/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame_game_options.cpp
 * \brief Implementation of the ptb::frame_game_options class.
 * \author Julien Jorge
 */
#include "ptb/frame/frame_game_options.hpp"

#include "ptb/game_variables.hpp"
#include "ptb/config_file.hpp"

#include <libintl.h>

/*----------------------------------------------------------------------------*/
const unsigned int ptb::frame_game_options::s_margin = 10;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owning_layer The layer owning the window.
 */
ptb::frame_game_options::frame_game_options( windows_layer* owning_layer )
  : frame(owning_layer), m_cursor_position(cursor_cancel)
{
  create_controls();
} // frame_game_options::frame_game_options()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a key has been pressed.
 * \param key The code of the key.
 */
bool ptb::frame_game_options::on_key_press( const bear::input::key_info& key )
{
  bool result = true;

  if ( key.is_escape() )
    on_cancel();
  else if ( key.is_enter() )
    validate();
  else if ( key.is_left() )
    result = on_left();
  else if ( key.is_right() )
    result = on_right();
  else if ( key.is_up() )
    result = on_up();
  else if ( key.is_down() )
    result = on_down();
  else
    result = false;

  return result;
} // frame_game_options::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a joystick button has been pressed.
 * \param button The code of the button.
 * \param joy_index The index of the joytick.
 */
bool ptb::frame_game_options::on_button_press
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;

  switch( button )
    {
    case bear::input::joystick::jc_button_1:
    case bear::input::joystick::jc_button_2:
    case bear::input::joystick::jc_button_3:
    case bear::input::joystick::jc_button_4:
    case bear::input::joystick::jc_button_5:
    case bear::input::joystick::jc_button_6:
    case bear::input::joystick::jc_button_7:
    case bear::input::joystick::jc_button_8:
    case bear::input::joystick::jc_button_9:
    case bear::input::joystick::jc_button_10:
    case bear::input::joystick::jc_button_11:
    case bear::input::joystick::jc_button_12:
    case bear::input::joystick::jc_button_13:
    case bear::input::joystick::jc_button_14:
    case bear::input::joystick::jc_button_15:
    case bear::input::joystick::jc_button_16:
      validate();
      break;
    case bear::input::joystick::jc_axis_left:  result = on_left(); break;
    case bear::input::joystick::jc_axis_right: result = on_right(); break;
    case bear::input::joystick::jc_axis_up:    result = on_up(); break;
    case bear::input::joystick::jc_axis_down:  result = on_down(); break;
    default:
      result = false;
    }

  return result;
} // frame_game_options::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a mouse button has been pressed.
 * \param key The code of the key.
 * \param pos The current position of the cursor.
 */
bool ptb::frame_game_options::on_mouse_press
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = true;

  switch( key )
    {
    case bear::input::mouse::mc_left_button:
    case bear::input::mouse::mc_middle_button:
    case bear::input::mouse::mc_right_button:
      validate();
      break;
    default:
      result = false;
    }

  return result;
} // frame_game_options::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when a mouse moves over the frame.
 * \param pos The position of the mouse.
 */
bool ptb::frame_game_options::on_mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = true;

  if ( m_ok->get_rectangle().includes(pos) )
    m_cursor_position = cursor_ok;
  else if ( m_cancel->get_rectangle().includes(pos) )
    m_cursor_position = cursor_cancel;
  else if ( m_friendly_fire->get_rectangle().includes(pos) )
    m_cursor_position = cursor_friendly_fire;
  else
    result = false;

  if ( result )
    position_cursor();

  return result;
} // frame_game_options::on_mouse_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Validate the selected action.
 */
void ptb::frame_game_options::validate()
{
  switch ( m_cursor_position )
    {
    case cursor_friendly_fire:
      m_friendly_fire->toggle_value();
      break;
    case cursor_ok:
      on_ok();
      break;
    case cursor_cancel:
      on_cancel();
      break;
    }
} // frame_game_options::validate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the static text components.
 */
void ptb::frame_game_options::create_controls()
{
  m_cursor = new bear::gui::picture
    ( this, new bear::visual::sprite(frame::get_cursor()) );

  bear::gui::static_text::font_type font = frame::get_font();

  create_checkbox( font );
  create_ok_cancel( font );

  set_size( m_cancel->right() + s_margin, m_cancel->bottom() + s_margin );
  position_cursor();
} // frame_game_options::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the "friendly_fire" checkbox.
 * \param f The font to use for the text.
 */
void ptb::frame_game_options::create_checkbox
( bear::gui::static_text::font_type f )
{
  m_friendly_fire =
    new bear::gui::checkbox
    ( this, frame::get_checkbox_off(), frame::get_checkbox_on(), f );

  m_friendly_fire->set_text( gettext("Friendly fire") );

  m_friendly_fire->set_position
    ( s_margin + m_cursor->width() + s_margin, s_margin );

  m_friendly_fire->check( game_variables::get_friendly_fire() );
} // frame_game_options::create_checkbox()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the "ok" and "cancel" buttons.
 * \param f The font to use for the texts.
 */
void ptb::frame_game_options::create_ok_cancel
( bear::gui::static_text::font_type f )
{
  m_ok =
    new button_with_text
    ( *this, std::mem_fun(&frame_game_options::on_ok), this );

  m_ok->set_font( f );
  m_ok->set_auto_size( true );
  m_ok->set_text( gettext("Ok") );
  m_ok->set_position
    ( s_margin + m_cursor->width() + s_margin,
      m_friendly_fire->bottom() + s_margin );

  m_cancel = new button_with_text
    ( *this, std::mem_fun(&frame_game_options::on_cancel), this );

  m_cancel->set_font( f );
  m_cancel->set_auto_size( true );
  m_cancel->set_text( gettext("Cancel") );

  unsigned int x = m_ok->right() + s_margin + m_cursor->width() + s_margin;
  unsigned int r = m_friendly_fire->right();

  if ( x + m_cancel->width() > r )
    r = x + m_cancel->width();

  m_cancel->set_position
    ( r - m_cancel->width(), m_friendly_fire->bottom() + s_margin );
} // frame_language::create_ok_cancel()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the game_options configuration.
 */
void ptb::frame_game_options::save()
{
  game_variables::set_friendly_fire( m_friendly_fire->checked() );

  config_file config;
  config.set_friendly_fire( m_friendly_fire->checked() );
  config.save();
} // frame_game_options::save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the configuration.
 */
bool ptb::frame_game_options::on_ok()
{
  save();
  close_window();

  return true;
} // frame_game_options::on_ok()

/*----------------------------------------------------------------------------*/
/**
 * \brief Exit without saving.
 */
bool ptb::frame_game_options::on_cancel()
{
  close_window();
  return true;
} // frame_game_options::on_cancel()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor up.
 */
bool ptb::frame_game_options::on_up()
{
  if ( m_cursor_position == cursor_friendly_fire )
    m_cursor_position = cursor_cancel;
  else
    m_cursor_position = cursor_friendly_fire;

  position_cursor();

  return true;
} // frame_game_options::on_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor down.
 */
bool ptb::frame_game_options::on_down()
{
  if ( m_cursor_position == cursor_friendly_fire )
    m_cursor_position = cursor_cancel;
  else
    m_cursor_position = cursor_friendly_fire;

  position_cursor();

  return true;
} // frame_game_options::on_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor left.
 */
bool ptb::frame_game_options::on_left()
{
  bool result = true;

  if ( m_cursor_position == cursor_ok )
    m_cursor_position = cursor_cancel;
  else if ( m_cursor_position == cursor_cancel )
    m_cursor_position = cursor_ok;
  else
    result = false;

  position_cursor();

  return result;
} // frame_game_options::on_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor right.
 */
bool ptb::frame_game_options::on_right()
{
  bool result = true;

  if ( m_cursor_position == cursor_ok )
    m_cursor_position = cursor_cancel;
  else if ( m_cursor_position == cursor_cancel )
    m_cursor_position = cursor_ok;
  else
    result = false;

  position_cursor();

  return result;
} // frame_game_options::on_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Position the cursor near the selected item.
 */
void ptb::frame_game_options::position_cursor()
{
  unsigned int x;
  unsigned int y;
  unsigned int h;

  switch(m_cursor_position)
    {
    case cursor_ok:
      x = m_ok->get_position().x;
      y = m_ok->get_position().y;
      h = m_ok->height();
      break;
    case cursor_cancel:
      x = m_cancel->get_position().x;
      y = m_cancel->get_position().y;
      h = m_cancel->height();
      break;
    case cursor_friendly_fire:
      x = m_friendly_fire->get_position().x;
      y = m_friendly_fire->get_position().y;
      h = m_friendly_fire->height();
      break;
    }

  x -= m_cursor->width() + s_margin;
  y += (int)(h - m_cursor->height()) / 2;

  m_cursor->set_position(x, y);
} // frame_game_options::position_cursor()
