/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file action_file_player.cpp
 * \brief Implementation of the ptb::action_file_player class.
 * \author Sbastien Angibaud
 */
#include "ptb/action_file_player.hpp"

#include <claw/string_algorithm.hpp>
#include <algorithm>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::action_file_player::action_file_player()
  : m_date(0), m_player_index(1)
{

} // action_file_player::action_file_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 */
ptb::action_file_player::action_file_player
( const action_file_player& that )
  : m_date(that.m_date), m_player_index(that.m_player_index)
{

} // action_file_player::action_file_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the action file.
 */
void ptb::action_file_player::set_action_file
(bear::engine::resource_pool& resources, const std::string& filename)
{
  std::stringstream f;
  resources.get_file(filename, f);

  if ( !f ) 
    claw::logger << claw::log_error << "can not open file '" << filename << "'."
                 << std::endl;
  else
    {
      std::string s;
      while ( claw::text::getline(f,s) )
	if ( !s.empty() )
          if ( s[0] != '#' ) 
            {
              std::istringstream iss(s);
              action_information a;
              std::string act;
		
              if ( iss >> a.date >> act )
                {
                  a.action = player_action::from_string(act);
		
                  if ( a.action == player_action::say ) 
                    {
                      claw::text::getline(iss, a.speech);
                      claw::text::trim(a.speech);
                      a.duration = 0;
                    }
                  else
                    {
                      iss >> a.duration;
                      a.speech.clear();
                    }

                  m_actions.push_back(a);
                }
            }
    }
} // action_file_player::set_action_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the player index.
 */
void ptb::action_file_player::set_player_index(unsigned int index)
{
  m_player_index = index;
} // set_player_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the player index.
 */
unsigned int ptb::action_file_player::get_player_index() const
{
  return m_player_index;
} // get_player_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is well initialized.
 */
bool ptb::action_file_player::is_valid() const
{
  return ( (m_player_index == 1) || (m_player_index == 2) );
} // base_bonus::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Search player.
 * \param glob Level globals in which we take the resources.
 */
void ptb::action_file_player::search_player(bear::engine::level_globals& glob)
{
  plee::get_instance_message msg;
  
  if ( m_player.get() == NULL )
    {
      glob.send_message(plee::player_name(m_player_index),msg);
      m_player = msg.get_instance();

      if ( m_player.get() != NULL ) 
        set_marionette(true);
    }
} // action_file_player::search_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the player.
 */
bear::universe::derived_item_handle_maker<ptb::plee>::handle_type& 
ptb::action_file_player::get_player()
{
  return m_player;
} // action_file_player::get_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start new actions.
 * \param new_actions Set of new actions to do.
 */
void ptb::action_file_player::start_actions
( std::set<player_action::value_type>& new_actions)
{
  bool stop(false);

  while ( !m_actions.empty() && !stop )  
    if ( m_actions.front().date - m_date > 0.0001 ) // epsilon
      stop = true;
    else
      {
        if ( m_actions.front().action == player_action::say ) 
          {
            std::string speech;
            std::vector<std::string> text;
            controller_config cfg;
            const controller_layout& layout
              ( cfg.get_layout( m_player->get_index() ) );
	    
            layout.escape_action_sequence(m_actions.front().speech, text);
            
            for (std::size_t i=0; i!=text.size(); ++i)
              speech += gettext(text[i].c_str());
	    
            m_player->add_speech(speech);
            m_player->speak(m_player->get_level_globals());
          }
        else
          {
            m_player->start_action(m_actions.front().action); 

            new_actions.insert(m_actions.front().action);
            m_current_actions[m_actions.front().action] = 
              m_actions.front().duration;
          }

        m_actions.pop_front();
      }
} // action_file_player::start_actions

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the action file.
 * \param glob Level globals in which we take the resources.
 * \param elapsed_time The duration of the progression.
 */
void ptb::action_file_player::progress_action_file
( bear::engine::level_globals& glob, bear::universe::time_type elapsed_time )
{
  m_date += elapsed_time;
  
  search_player(glob);

  if ( m_player != (plee*)NULL )
    {
      current_actions_map::iterator it;
      std::list<player_action::value_type> finished_actions;
      std::list<player_action::value_type>::iterator it2;
      std::set<player_action::value_type> new_actions;
      
      //start actions
      start_actions(new_actions);      
      
      // continue actions
      for ( it=m_current_actions.begin();  it!=m_current_actions.end(); ++it )
        if ( new_actions.find(it->first) == new_actions.end()) 
          {
            if ( it->second <= elapsed_time ) 
              {
                //m_player->do_action(it->second, it->first);
                it->second = 0;
                finished_actions.push_back(it->first);
              }
            else
              {
                m_player->do_action(elapsed_time, it->first);
                it->second -= elapsed_time;
              }
          }
      
      // stop actions
      for ( it2=finished_actions.begin(); it2!=finished_actions.end(); ++it2 )
        {
          m_player->stop_action(*it2); 
          m_current_actions.erase(*it2);
        }
      
      if ( m_current_actions.empty() && m_actions.empty() )
        finish();
    }
} // action_file_player::progress_action_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the marionette state.
 * \param b The new state.
 */
void ptb::action_file_player::set_marionette( bool b )
{
  m_player->set_marionette(b);
} // action_file_player::set_marionette()

/*----------------------------------------------------------------------------*/
/**
 * \brief Finish the action file.
 */
void ptb::action_file_player::finish()
{
  set_marionette(false);
} // action_file_player::finish()

