/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/level_file_xml_writer.cpp
 * \brief Implementation of the bf::level_file_xml_writer class.
 * \author Julien Jorge
 */
#include "bf/level_file_xml_writer.hpp"

#include "bf/item_class_pool.hpp"
#include "bf/item_comparator.hpp"
#include "bf/level.hpp"
#include "bf/xml/item_instance_fields_node.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Save a level.
 * \param os The stream in which we write.
 * \param lvl The level to save.
 */
void bf::level_file_xml_writer::save( std::ostream& os, const level& lvl ) const
{
  os << "<level width='" << lvl.get_width() << "' height='" << lvl.get_height()
     << "' music='" << lvl.get_music() << "'>\n";

  for (unsigned int i=0; i!=lvl.layers_count(); ++i)
    save_layer( os, lvl.get_layer(i) );

  os << "</level>\n";
} // level_file_xml_writer::save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save a layer.
 * \param os The stream in which we write.
 * \param the_layer The layer to save.
 */
void bf::level_file_xml_writer::save_layer
( std::ostream& os, const layer& the_layer ) const
{
  os << "  <layer class_name='" << the_layer.get_class_name() << "' width='"
     << the_layer.get_width() << "' height='" << the_layer.get_height()
     << "'>\n\n";

  layer::const_item_iterator it;
  std::list<item_instance> items(the_layer.item_begin(), the_layer.item_end());

  items.sort( item_comparator::by_place() );

  for ( ; !items.empty(); items.pop_front() )
    save_item( os, items.front() );

  os << "  </layer><!-- " << the_layer.get_class_name() << " -->\n\n";
} // level_file_xml_writer::save_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save an item.
 * \param os The stream in which we write.
 * \param item The item to save.
 */
void bf::level_file_xml_writer::save_item
( std::ostream& os, const item_instance& item ) const
{
  os << "    <item class_name='" << item.get_class().get_class_name()
     << "' fixed='";

  if ( item.get_fixed() )
    os << "true'";
  else
    os << "false'";

  if ( !item.get_id().empty() )
    os << " id='" << item.get_id() << "'";

  os << ">\n";

  xml::item_instance_fields_node field_node;
  field_node.write(item, os);

  os << "    </item><!-- " << item.get_class().get_class_name() << " -->\n\n";
} // level_file_xml_writer::save_item()
